; SV - Symbolic Vector Hardware Analysis Framework
; Copyright (C) 2014-2015 Centaur Technology
;
; Contact:
;   Centaur Technology Formal Verification Group
;   7600-C N. Capital of Texas Highway, Suite 300, Austin, TX 78731, USA.
;   http://www.centtech.com/
;
; License: (An MIT/X11-style license)
;
;   Permission is hereby granted, free of charge, to any person obtaining a
;   copy of this software and associated documentation files (the "Software"),
;   to deal in the Software without restriction, including without limitation
;   the rights to use, copy, modify, merge, publish, distribute, sublicense,
;   and/or sell copies of the Software, and to permit persons to whom the
;   Software is furnished to do so, subject to the following conditions:
;
;   The above copyright notice and this permission notice shall be included in
;   all copies or substantial portions of the Software.
;
;   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
;   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
;   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
;   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
;   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
;   FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
;   DEALINGS IN THE SOFTWARE.
;
; Original author: Sol Swords <sswords@centtech.com>

(in-package "SV")
(include-book "eval")
(include-book "a4vec-ops")
(include-book "rewrite")
(include-book "centaur/gl/gl-mbe" :dir :system)
(include-book "centaur/gl/def-gl-rewrite" :dir :system)
(local (include-book "arithmetic/top-with-meta" :dir :system))
(local (include-book "centaur/bitops/ihsext-basics" :dir :system))
(local (include-book "std/alists/alist-keys" :dir :System))
(local (include-book "centaur/bitops/equal-by-logbitp" :dir :system))
(local (include-book "clause-processors/just-expand" :dir :system))
(local (std::add-default-post-define-hook :fix))

(local (std::deflist svarlist-p (x)
         (svar-p x)
         :true-listp t
         :elementp-of-nil nil))

(local (defthm true-listp-nthcdr
         (implies (true-listp x)
                  (true-listp (nthcdr n x)))
         :hints(("Goal" :in-theory (e/d (nthcdr)
                                        (acl2::cdr-nthcdr))
                 :induct (nthcdr n x)))
         :rule-classes :type-prescription))

(local (defthm nthcdr-of-append-equal-len
         (implies (equal (nfix n) (len x))
                  (equal (nthcdr n (append x y))
                         y))
         :hints(("Goal" :in-theory (e/d (nthcdr)
                                        (acl2::cdr-nthcdr))
                 :induct (nthcdr n x)))))

(local (defthm take-of-append-equal-len
         (implies (equal (nfix n) (len x))
                  (equal (take n (append x y))
                         (list-fix x)))
         :hints(("Goal" :in-theory (e/d (acl2::take-redefinition))
                 :induct (nthcdr n x)))))

(local (in-theory (disable double-containment)))

(local (defthm 3vec-p-of-4vec-mask
         (implies (3vec-p x)
                  (3vec-p (4vec-mask mask x)))
         :hints(("Goal" :in-theory (enable 4vec-mask 3vec-p))
                (acl2::logbitp-reasoning))))

(local (defthm true-listp-of-scdr
         (implies (true-listp x)
                  (true-listp (gl::scdr x)))
         :hints(("Goal" :in-theory (enable gl::scdr)))
         :rule-classes :type-prescription))

(local (in-theory (disable gl::s-endp-of-bfr-scons
                           aig-list->s)))



(local (defthm aig-list->s-open-quote
         (implies (syntaxp (quotep x))
                  (equal (aig-list->s x env)
                         (B* (((MV FIRST REST GL::END)
                               (GL::FIRST/REST/END X)))
                           (IF GL::END
                               (GL::BOOL->SIGN (AIG-EVAL FIRST ENV))
                               (BITOPS::LOGCONS (BOOL->BIT (AIG-EVAL FIRST ENV))
                                                (AIG-LIST->S REST ENV))))))
         :hints(("Goal" :in-theory (enable aig-list->s)))))


(local (defthm aig-list->s-of-bfr-snorm
         (equal (aig-list->s (gl::bfr-snorm x) env)
                (aig-list->s x env))
         :hints(("Goal" :in-theory (enable aig-list->s gl::bfr-snorm)))))

(local (defthm aig-list->s-of-bfr-scons
         (equal (aig-list->s (gl::bfr-scons a b) env)
                (bitops::logcons (bool->bit (aig-eval a env))
                                 (aig-list->s b env)))
         :hints(("Goal" :expand ((aig-list->s (gl::bfr-scons a b) env)
                                 (aig-list->s b env))
                 :in-theory (enable gl::s-endp-of-bfr-scons)
                 :do-not-induct t))))




(defxdoc bit-blasting
  :parents (expressions)
  :short "We implement an efficient translation from @(see svex) expressions
into @(see acl2::aig)s, to support symbolic simulation with @(see acl2::gl).")

(local (xdoc::set-default-parents bit-blasting))



(defalist svex-a4vec-env
  :key-type svar
  :val-type a4vec)

(define svex-a4vec-env-eval ((x svex-a4vec-env-p) env)
  :returns (xx svex-env-p)
  :measure (len (svex-a4vec-env-fix x))
  (b* ((x (svex-a4vec-env-fix x)))
    (if (atom x)
        nil
      (cons (cons (svar-fix (caar x))
                  (a4vec-eval (cdar x) env))
            (svex-a4vec-env-eval (cdr x) env))))
  ///
  (defret alist-keys-of-svex-a4vec-env-eval
    (equal (alist-keys xx)
           (alist-keys (svex-a4vec-env-fix x)))
    :hints(("Goal" :in-theory (enable svex-a4vec-env-fix alist-keys)))))




(define a4veclist-nth ((n natp) (x a4veclist-p))
  :returns (elt a4vec-p)
  :guard-hints (("goal" :in-theory (enable nth a4veclist-p)))
  (mbe :logic (if (< (nfix n) (len x))
                  (a4vec-fix (nth n x))
                (a4vec-x))
       :exec (or (nth n x) (a4vec-x)))
  ///
  (defthm a4veclist-nth-out-of-bounds
    (implies (<= (len x) (nfix n))
             (equal (a4veclist-nth n x) (a4vec-x))))
  (defthm a4veclist-nth-in-of-bounds
    (implies (< (nfix n) (len x))
             (equal (a4veclist-nth n x) (a4vec-fix (nth n x))))))

(define svexlist-nth ((n natp) (x svexlist-p))
  :returns (elt svex-p)
  :guard-hints (("goal" :in-theory (enable nth svexlist-p)))
  (mbe :logic (if (< (nfix n) (len x))
                  (svex-fix (nth n x))
                (svex-x))
       :exec (or (nth n x) (svex-x)))
  ///
  (defthm svexlist-nth-out-of-bounds
    (implies (<= (len x) (nfix n))
             (equal (svexlist-nth n x) (svex-x))))
  (defthm svexlist-nth-in-of-bounds
    (implies (< (nfix n) (len x))
             (equal (svexlist-nth n x) (svex-fix (nth n x))))))

(local (defthm nth-of-svexlist-eval
         (equal (nth n (svexlist-eval x env))
                (and (< (nfix n) (len x))
                     (svex-eval (nth n x) env)))
         :hints(("Goal" :in-theory (enable nth svexlist-eval)
                 :induct (nth n x)))))

(local (defthm nth-of-a4veclist-eval
         (equal (nth n (a4veclist-eval x env))
                (and (< (nfix n) (len x))
                     (a4vec-eval (nth n x) env)))
         :hints(("Goal" :in-theory (enable nth a4veclist-eval)
                 :induct (nth n x)))))

(define maybe-a3vec-fix ((v (a4vec-p v)) (x svex-p))
  :returns (vv a4vec-p)
  (if (3valued-syntaxp (svex-fix x))
      (a4vec-fix v)
    (a3vec-fix v))
  ///
  (local (defthm nth-under-iff-when-a4veclist-p
           (implies (a4veclist-p x)
                    (iff (nth n x)
                         (< (nfix n) (len x))))
           :hints(("Goal" :in-theory (enable a4veclist-p nth)))))

  (local (defthm nth-out-of-bounds
           (implies (<= (len x) (nfix n))
                    (not (nth n x)))
           :hints(("Goal" :in-theory (enable nth)))))

  (defthm maybe-a3vec-fix-when-implies
    (implies (case-split (implies (3valued-syntaxp x)
                                  (3vec-p (a4vec-eval v env))))
             (equal (a4vec-eval (maybe-a3vec-fix v x) env)
                    (3vec-fix (a4vec-eval v env)))))

  (defthm maybe-a3vec-fix-of-nths
    (implies (equal (a4veclist-eval vals env)
                    (svexlist-eval x (svex-a4vec-env-eval a4env env)))
             (equal (a4vec-eval (maybe-a3vec-fix (nth n vals) (nth n x)) env)
                    (3vec-fix (a4vec-eval (nth n vals) env))))
    :hints(("Goal" :in-theory (e/d (a4veclist-nth)
                                   (nth-of-svexlist-eval)
                                   (nth-of-a4veclist-eval))
            :use ((:instance nth-of-svexlist-eval
                   (env (svex-a4vec-env-eval a4env env)))
                  (:instance nth-of-a4veclist-eval)))))

  (defthm maybe-a3vec-fix-of-a3vec
    (implies (a4vec-syntactic-3vec-p v)
             (equal (maybe-a3vec-fix v x)
                    (a4vec-fix v)))
    :hints(("Goal" :in-theory (enable a3vec-fix)))))










;; (define 4vmask-nth ((n natp) (x 4vmasklist-p))
;;   :returns (mask 4vmask-p :rule-classes (:rewrite :type-prescription))
;;   (b* ((x (4vmasklist-fix x)))
;;     (if (< (lnfix n) (len x))
;;         (4vmask-fix (nth n x))
;;       -1)))


;; (define maybe-a4vec-fix ((v (or (a4vec-p v) (not v))))
;;   :returns (vv a4vec-p)
;;   (if v (a4vec-fix v) (a4vec-x)))

(defconst *svex-aig-op-table*
  ;; fn name, non-3vec-fixing function, args (with notation for 3vec-fixed ones and masks)
  '((id        a4vec-fix            (x)                         "identity function")
    (bitsel    a4vec-bit-extract    (index x)                   "bit select")
    (unfloat   a4vec-fix            ((3v x))                    "change Z bits to Xes")
    (bitnot    a3vec-bitnot         ((3v x))                    "bitwise negation")
    (onp       a4vec-onset          (x)                         "bitwise onset")
    (offp      a4vec-offset         (x)                         "bitwise offset")
    (bitand    a3vec-bitand         ((3v x) (3v y))             "bitwise AND")
    (bitor     a3vec-bitor          ((3v x) (3v y))             "bitwise OR")
    (bitxor    a3vec-bitxor         ((3v x) (3v y))             "bitwise XOR")
    (res       a4vec-res            (x y)                       "resolve (short together)")
    (resand    a4vec-resand         (x y)                       "resolve wired AND")
    (resor     a4vec-resor          (x y)                       "resolve wired OR")
    (override  a4vec-override       (x y)                       "resolve different strengths")
    (uand      a3vec-reduction-and  ((3v x))                    "unary (reduction) AND")
    (uor       a3vec-reduction-or   ((3v x))                    "unary (reduction) OR")
    (uxor      a4vec-parity         (x)                         "reduction XOR, i.e. parity")
    (zerox     a4vec-zero-ext       (width x (mask m))          "zero extend")
    (signx     a4vec-sign-ext       (width x (mask m))          "sign extend")
    (concat    a4vec-concat         (width x y (mask m))        "concatenate at a given bit width")
    (partsel   a4vec-part-select    (lsb width in (mask m))     "part select")
    (partinst  a4vec-part-install   (lsb width in val (mask m)) "part install")
    (blkrev    a4vec-rev-blocks     (width blksz x)             "reverse block order")
    (rsh       a4vec-rsh            (shift x (mask m))          "right shift")
    (lsh       a4vec-lsh            (shift x (mask m))          "left shift")
    (+         a4vec-plus           (x y)                       "addition")
    (b-        a4vec-minus          (x y)                       "subtraction")
    (u-        a4vec-uminus         (x)                         "unary minus")
    (xdet      a4vec-xdet           (x)                         "x detect")
    (countones a4vec-countones      (x)                         "count of set bits")
    (onehot    a4vec-onehot         (x)                         "one-hot check")
    (onehot0   a4vec-onehot0        (x)                         "one-hot check (zero-hot allowed)")
    (*         a4vec-times          (x y)                       "multiplication")
    (/         a4vec-quotient       (x y)                       "division")
    (%         a4vec-remainder      (x y)                       "modulus")
    (<         a4vec-<              (x y)                       "less than")
    (clog2     a4vec-clog2          (x)                         "ceiling of log2")
    (pow       a4vec-pow            (x y)                       "exponentiation")
    (==        a3vec-==             ((3v x) (3v y))             "equality")
    (===       a4vec-===            (x y)                       "case equality")
    (==?       a4vec-wildeq         (x y)                       "wildcard equality")
    (safer-==? a4vec-wildeq-safe    (x y)                       "wildcard equality (monotonic version)")
    (==??      a4vec-symwildeq      (x y)                       "symmetric wildcard equality")
    (?         a3vec-?              ((3v test) (3vp then) (3vp else)) "if-then-else")
    (?*        a3vec-?*             ((3v test) (3vp then) (3vp else)) "if-then-else")
    (bit?      a3vec-bit?           ((3v test) (3vp then) (3vp else)) "bitwise if-then-else")))

#||
(loop for lst in sv::*svex-aig-op-table* do
      (let ((fn (cadr lst)))
        (unless (eq fn 'sv::a4vec-fix) (profile-fn fn))))
||#


(defun svex-apply-aig-collect-args (n restargs argsvar svvar maskvar ;; argmasks-var
                                      )
  (let* ((n (nfix n)))
    (if (atom restargs)
        nil
      (append (if (consp (car restargs))
                  (case (caar restargs)
                    (3v
                     `((maybe-a3vec-fix ;; (a4vec-mask (4vmask-nth ,n ,argmasks-var)
                        (a4veclist-nth ,n ,argsvar)
                        (svexlist-nth ,n ,svvar))))
                    (3vp
                     `(;; (a4vec-mask (4vmask-nth ,n ,argmasks-var)
                       (a4veclist-nth ,n ,argsvar)
                       (3valued-syntaxp (svexlist-nth ,n ,svvar))))
                    (mask
                     `(,maskvar))
                    (t (prog2$
                        (er hard? 'svex-apply-aig-collect-args "bad formal expr")
                        `((a4veclist-nth ,n ,argsvar)))))
                `((a4veclist-nth ,n ,argsvar)))
              (svex-apply-aig-collect-args (+ 1 n) (cdr restargs) argsvar svvar maskvar ;; argmasks-var
                                           )))))


;; (defun svex-apply-aig-uses-argmasks (args)
;;   (if (atom args)
;;       nil
;;     (or (and (consp (car args))
;;              (or (eq (caar args) '3v)
;;                  (eq (caar args) '3vp)))
;;         (svex-apply-aig-uses-argmasks (cdr args)))))

(defun svex-apply-aig-cases-fn (argsvar svvar maskvar optable)
  (b* (((when (atom optable)) '((otherwise (a4vec-x))))
       ((list sym fn args) (car optable))
       (acc-args (svex-apply-aig-collect-args 0 args argsvar svvar maskvar ;; 'tmp-argmasks
                                              ))
       (call `(,fn . ,acc-args))
       (full ;; (if (svex-apply-aig-uses-argmasks args)
             ;;     `(let ((tmp-argmasks (svex-argmasks ,maskvar ',sym ,svvar)))
             ;;        ,call)
               call))
    (cons `(,sym ,full)
          (svex-apply-aig-cases-fn argsvar svvar maskvar (cdr optable)))))

(defmacro svex-apply-aig-cases (fn args svex mask)
  `(case ,fn
     . ,(svex-apply-aig-cases-fn args svex mask *svex-aig-op-table*)))


(defthm svex-p-when-nth
  (implies (and (svexlist-p x)
                (nth n x))
           (svex-p (nth n x)))
  :hints(("Goal" :in-theory (enable nth svexlist-p))))

(defthm a4vec-p-when-nth
  (implies (and (a4veclist-p x)
                (nth n x))
           (a4vec-p (nth n x)))
  :hints(("Goal" :in-theory (enable nth svexlist-p))))

;; (defthm a4vec-eval-of-maybe-a4vec-fix-nth-out-of-bounds
;;   (implies (<= (len x) (nfix n))
;;            (equal (a4vec-eval (maybe-a4vec-fix (nth n x)) env)
;;                   (4vec-x)))
;;   :hints(("Goal" :in-theory (enable maybe-a4vec-fix nth))))


(local (in-theory (disable nth)))









(define svex-apply-aig ((fn fnsym-p) (args a4veclist-p) (terms svexlist-p) (mask 4vmask-p))
  :prepwork ((local (Defthm 4veclist-nth-safe-of-a4veclist-eval
                      (equal (a4vec-eval (a4veclist-nth n x) aigenv)
                             (4veclist-nth-safe n (a4veclist-eval x aigenv)))
                      :hints(("Goal" :in-theory (enable a4veclist-eval a4veclist-nth 4veclist-nth-safe)))))
             (local (defun ind (n vals x)
                      (if (zp n)
                          (list vals x)
                        (ind (1- n) (cdr vals) (cdr x)))))
             (local (defthm 3vec-p-when-3valued-syntaxp-nth
                      (implies (and (EQUAL (A4VECLIST-EVAL VALS AIGENV)
                                           (SVEXLIST-EVAL X (SVEX-A4VEC-ENV-EVAL A4ENV AIGENV)))
                                    (3valued-syntaxp (svexlist-nth n x)))
                               (3vec-p (a4vec-eval (nth n vals) aigenv)))
                      :hints(("Goal" :in-theory (enable a4veclist-eval
                                                        svexlist-eval
                                                        svexlist-nth
                                                        nth)
                              :induct (ind n vals x)
                              :expand ((a4veclist-eval vals aigenv)
                                       (:free (env) (svexlist-eval x env))))
                             (and stable-under-simplificationp
                                  '(:use ((:instance 3vec-p-of-eval-when-3valued-syntaxp
                                           (x (car x))
                                           (env (svex-a4vec-env-eval a4env aigenv)))))))))
             (local (encapsulate nil
                      (local (defun ind2 (n masks vals x)
                               (if (zp n)
                                   (list masks vals x)
                                 (ind2 (1- n) (cdr masks) (cdr vals) (cdr x)))))

                      ;; BOZO do we need this?
                      ;; (defthm 4vmask-of-nths
                      ;;   (implies (equal (len masks) (len vecs))
                      ;;            (equal (4vec-mask (4vmask-nth n masks)
                      ;;                              (4veclist-nth-safe n vecs))
                      ;;                   (4veclist-nth-safe n (4veclist-mask masks vecs))))
                      ;;   :hints(("Goal" :in-theory (enable 4vmask-nth 4veclist-nth-safe 4veclist-mask nth
                      ;;                                     4vmasklist-fix)
                      ;;           :induct (ind2 n masks vecs nil))))

                      (defthm svex-eval-of-nth-rev
                        (equal (svex-eval (nth n x) env)
                               (4veclist-nth-safe n (svexlist-eval x env))))

                      (in-theory (disable svex-eval-of-nth
                                          4veclist-nth-safe-of-svexlist-eval))

                      (local (defthm 3vec-p-of-eval-by-equal
                               (implies (and (equal x (svex-eval y env))
                                             (3valued-syntaxp y))
                                        (3vec-p x))))

                      (local (defthm 3vec-p-of-eval-by-equal-with-mask
                               (implies (and (equal x (4vec-mask mask (svex-eval y env)))
                                             (3valued-syntaxp y))
                                        (3vec-p x))))

                      (defthm 4veclist-masked-idempotent
                        (implies (equal x (4veclist-mask masks y))
                                 (equal (4veclist-mask masks x) x)))


                      (defthm dumb
                        (implies (and (EQUAL (A4VECLIST-EVAL VALS AIGENV)
                                             (4VECLIST-MASK masks
                                                            (SVEXLIST-EVAL X (SVEX-A4VEC-ENV-EVAL A4ENV AIGENV))))
                                      (3valued-syntaxp (svexlist-nth n x)))
                                 (3vec-p (4veclist-nth-safe n (a4veclist-eval vals aigenv)))
                                 ;; (3vec-p (4vec-mask (4vmask-nth n masks)
                                 ;;                    (a4vec-eval (a4veclist-nth n vals) aigenv)))
                                 )
                        :hints (("goal" :in-theory (e/d (nth len 4veclist-nth-safe a4veclist-eval
                                                             4veclist-mask svexlist-eval
                                                             a4veclist-eval svexlist-nth)
                                                        (4veclist-nth-safe-of-a4veclist-eval))
                                 :expand ((:free (env) (svexlist-eval x env))
                                          (a4veclist-eval vals aigenv)
                                          (:free (a b) (4veclist-mask masks (cons a b))))
                                 :induct (ind2 n masks vals x))))
                      )))
  :verbosep t
  :guard-debug t
  :returns (res a4vec-p)
  (b* ((fn (fnsym-fix fn))
       (args (a4veclist-fix args))
       (res (svex-apply-aig-cases fn args terms mask)))
    ;; ;; This cleverly masks out any bits of the result that we don't care about,
    ;; ;; replacing them with Xes.  This might be a great way to get a lot more
    ;; ;; constant propagation...
    (a4vec-mask mask res))
  ///

  (defthm svex-apply-aig-correct
    (implies (and (fnsym-p fn)
                  (bind-free '((a4env . env)) (a4env))
                  (equal (a4veclist-eval vals aigenv)
                         (4veclist-mask argmasks
                                         (svexlist-eval x (svex-a4vec-env-eval a4env aigenv))))
                  (svex-argmasks-okp (svex-call fn x) mask argmasks))
             (equal (a4vec-eval (svex-apply-aig fn vals x mask) aigenv)
                    (4vec-mask mask
                              (svex-apply fn (svexlist-eval x (svex-a4vec-env-eval a4env aigenv))))))
    :hints(("Goal" :in-theory (disable len-of-4veclist-mask
                                      svex-apply-aig)
                  ;; Establish that (len vals) = (len x).
                  :use ((:instance len-of-4veclist-mask
                         (masks (svex-argmasks mask fn x))
                         (values (a4veclist-eval vals aigenv)))
                        (:instance len-of-4veclist-mask
                         (masks (svex-argmasks mask fn x))
                         (values (svexlist-eval x (svex-a4vec-env-eval a4env aigenv))))))
           (and stable-under-simplificationp
                '(
                  :in-theory (e/d (svex-apply svexlist-eval ;; 4veclist-nth-safe ;; 4veclist-mask
                                              4veclist-mask?
                                              4vec-bitnot
                                              4vec-bitand
                                              4vec-bitor
                                              4vec-bitxor-redef
                                              4vec-reduction-and
                                              4vec-reduction-or
                                              4vec-?
                                              4vec-?*
                                              4vec-bit?
                                              4vec-==)
                                  (;; len-of-svexlist-eval
                                   ;; len-of-a4veclist-eval
                                   ;; len-of-4veclist-mask
                                   svex-argmasks-correct
                                   svex-argmasks-remove-mask))
                  :use ;; ((:instance len-of-svexlist-eval
                  ;;   (env (svex-a4vec-env-eval a4env aigenv)))
                  ;;  (:instance len-of-a4veclist-eval
                  ;;   (x vals) (env aigenv)))
                  ((:instance svex-argmasks-okp-necc
                    (x (svex-call fn x))
                    (vals (a4veclist-eval vals aigenv))
                    (env (svex-a4vec-env-eval a4env aigenv)))
                   ;; (:instance svex-argmasks-remove-mask
                   ;;  (fn fn)
                   ;;  (args x)
                   ;;  (env (svex-a4vec-env-eval a4env aigenv)))


                   )
                  :do-not-induct t
                  :do-not '(fertilize generalize eliminate-destructors)
                  )))
    :otf-flg t))


(defalist svex-aig-memotable :key-type svex :val-type a4vec)

(defthm a4vec-p-of-svex-a4vec-env-lookup
  (implies (and (svex-a4vec-env-p x)
                (hons-assoc-equal k x))
           (a4vec-p (cdr (hons-assoc-equal k x)))))

(defthm a4vec-p-of-svex-aig-memotable-lookup
  (implies (and (svex-aig-memotable-p x)
                (hons-assoc-equal k x))
           (a4vec-p (cdr (hons-assoc-equal k x)))))

;; (SVEX->A4VEC
;;  '(RSH (? (< 0 (* 32 (B- (CONCAT 16 CNST 0) 0))) (* 32 (B- (CONCAT 16 CNST 0) 0)) 0) '(-71265535176078871931497435759850128999 . 269016831744859591531877171671918082457))
;;  (make-fast-alist `((cnst ,(acl2::numlist 0 2 16) . ,(acl2::numlist 1 2 16))))
;;  nil)

(define svex-is-const-concat ((x svex-p))
  :returns (is-concat)
  :guard-hints (("goal" :in-theory (enable nth)))
  (svex-case x
    :call (and (eq x.fn 'concat)
               (eql (len x.args) 3)
               (let ((arg1 (mbe :logic (nth 0 x.args)
                                :exec (car x.args))))
                 (svex-case arg1 :quote)))
    :otherwise nil))

(define svex-const-concat-args ((x svex-p))
  :guard (svex-is-const-concat x)
  :guard-hints (("goal" :expand ((:free (n) (nth n (svex-call->args x)))
                                 (:free (n) (nth n (cdr (svex-call->args x))))
                                 (:free (n) (nth n (cddr (svex-call->args x)))))))
  :prepwork ((local (in-theory (enable svex-is-const-concat))))
  :returns (mv (width 4vec-p)
               (lsbs svex-p)
               (msbs svex-p))
  (b* (((svex-call x)))
    (mv (svex-quote->val (mbe :logic (svex-fix (nth 0 x.args))
                              :exec (first x.args)))
        (mbe :logic (svex-fix (nth 1 x.args))
             :exec (second x.args))
        (mbe :logic (svex-fix (nth 2 x.args))
             :exec (third x.args))))
  ///
  (local (defthm nth-when-n-too-big
           (implies (<= (len x) (nfix n))
                    (equal (nth n x) nil))
           :hints(("Goal" :in-theory (enable nth)))))


  (local (defthm 4vec-zero-ext-is-concat
           (equal (4vec-zero-ext n x)
                  (4vec-concat n x 0))
           :hints(("Goal" :in-theory (enable 4vec-zero-ext 4vec-concat)))))

  (defretd svex-const-concat-args-correct-rw
    (implies (svex-is-const-concat x)
             (equal (svex-eval x env)
                    (4vec-concat width (svex-eval lsbs env) (svex-eval msbs env))))
    :hints(("Goal" :in-theory (enable svex-apply svexlist-eval))))

  (local (defthm svex-count-of-nth
           (<= (svex-count (nth n x)) (svexlist-count x))
           :hints(("Goal" :in-theory (enable nth svexlist-count)))
           :rule-classes :linear))

  (local (defthm svex-count-of-svexlist-nth
           (<= (svex-count (svexlist-nth n x)) (svexlist-count x))
           :hints(("Goal" :in-theory (enable svexlist-nth)))
           :rule-classes :linear))

  (defret svex-count-of-svex-const-concat-args-lsbs
    (implies (svex-is-const-concat x)
             (< (svex-count lsbs) (svex-count x)))
    :rule-classes :linear)

  (defret svex-count-of-svex-const-concat-args-msbs
    (implies (svex-is-const-concat x)
             (< (svex-count msbs) (svex-count x)))
    :hints ((and stable-under-simplificationp
                 '(:expand ((svex-count x)))))
    :rule-classes :linear))



(progn
  (local (defthm 4vec-zero-ext-of-4vec-mask?
           (equal (4vec-zero-ext w (4vec-mask? mask x y))
                  (4vec-mask? (if (and (2vec-p w) (<= 0 (2vec->val w)))
                                  (loghead (2vec->val w) (4vmask-fix mask))
                                mask)
                              (4vec-zero-ext w x)
                              (4vec-zero-ext w y)))
           :hints(("Goal" :in-theory (enable 4vec-mask? 4vec-bit? 3vec-bit?
                                             4vec-zero-ext))
                  (logbitp-reasoning))))

  (local (defthm 4vec-zero-ext-of-equal-4vec-mask?
           (implies (equal z (4vec-mask? mask x y))
                    (equal (4vec-zero-ext w z)
                           (4vec-mask? (if (and (2vec-p w) (<= 0 (2vec->val w)))
                                           (loghead (2vec->val w) (4vmask-fix mask))
                                         mask)
                                       (4vec-zero-ext w x)
                                       (4vec-zero-ext w y))))))


  (local (defthm 4vec-zero-ext-of-zero-ext
           (equal (4vec-zero-ext w (4vec-zero-ext w x))
                  (4vec-zero-ext w x))
           :hints(("Goal" :in-theory (enable 4vec-zero-ext))))))


(defines svex->a4vec
  ;; Self-memoized version of svex-eval, for GL
  :verify-guards nil
  :ruler-extenders :all
  (define svex->a4vec ((x svex-p)
                       (env svex-a4vec-env-p)
                       (masks svex-mask-alist-p)
                       (memo svex-aig-memotable-p))
    :returns (mv (res a4vec-p)
                 (memo1 svex-aig-memotable-p))
    :measure (two-nats-measure (svex-count x) 1)
    (b* ((memo (svex-aig-memotable-fix memo))
         (env (svex-a4vec-env-fix env)))
      (svex-case x
        :quote (b* ((mask (svex-mask-lookup x masks)))
                 (mv (4vec->a4vec (4vec-mask mask x.val)) memo))
        :var (mv (let ((look (hons-get x.name env))
                       (mask (svex-mask-lookup x masks)))
                   (a4vec-mask mask (if look (cdr look) (a4vec-x))))
                 memo)
        :call (b* ((x (svex-fix x))
                   (look (hons-get x memo))
                   ((when look) (mv (cdr look) memo))
                   ((mv res memo)
                    (b* (((when (svex-is-const-concat x))
                          (b* (((mv upper lower memo)
                                (svex-concat->a4vec x env masks memo))
                               (mask (svex-mask-lookup x masks)))
                            (mv (a4vec-mask mask (a4vec upper lower))
                                memo)))
                         ((mv args memo) (svexlist->a4vec x.args env masks memo))
                         (mask (svex-mask-lookup x masks))
                         (res (svex-apply-aig x.fn args x.args mask)))
                      (mv res memo)))
                   (memo (hons-acons x res memo)))
                (mv res memo)))))
  (define svexlist->a4vec ((x svexlist-p)
                           (env svex-a4vec-env-p)
                           (masks svex-mask-alist-p)
                           (memo svex-aig-memotable-p))
    :returns (mv (res a4veclist-p)
                 (memo1 svex-aig-memotable-p))
    :measure (two-nats-measure (svexlist-count x) 0)
    (b* (((when (atom x)) (mv nil (svex-aig-memotable-fix memo)))
         ((mv first memo) (svex->a4vec (car x) env masks memo))
         ((mv rest memo) (svexlist->a4vec (cdr x) env masks memo)))
      (mv (cons first rest) memo)))

  (define svex-concat->a4vec ((x svex-p)
                              (env svex-a4vec-env-p)
                              (masks svex-mask-alist-p)
                              (memo svex-aig-memotable-p))
    :returns (mv (upper true-listp)
                 (lower true-listp)
                 (memo1 svex-aig-memotable-p))
    :measure (two-nats-measure (svex-count x)
                               (if (svex-is-const-concat x) 0 2))
    (b* (((unless (svex-is-const-concat x))
          (b* (((mv res memo) (svex->a4vec x env masks memo)))
            (mv (a4vec->upper res) (a4vec->lower res) memo)))
         ((mv width lsbs msbs)
          (svex-const-concat-args x))
         ((unless (and (2vec-p width) (natp (2vec->val width))))
          (mv (a4vec->upper (a4vec-x)) (a4vec->lower (a4vec-x))
              (svex-aig-memotable-fix memo)))
         (width (2vec->val width))
         (mask (svex-mask-lookup x masks))
         ((when (eql 0 (logtail width mask)))
          (svex-concat->a4vec lsbs env masks memo))
         ((mv upper2 lower2 memo)
          (svex-concat->a4vec msbs env masks memo)))
      (svex-concat->a4vec-lower lsbs width upper2 lower2 env masks memo)))

  (define svex-concat->a4vec-lower ((x svex-p)
                                    (width natp)
                                    (upper-acc true-listp)
                                    (lower-acc true-listp)
                                    (env svex-a4vec-env-p)
                                    (masks svex-mask-alist-p)
                                    (memo svex-aig-memotable-p))
    :returns (mv (upper true-listp)
                 (lower true-listp)
                 (memo1 svex-aig-memotable-p))
    :measure (two-nats-measure (svex-count x)
                               (if (svex-is-const-concat x) 0 2))
    (b* ((upper-acc (llist-fix upper-acc))
         (lower-acc (llist-fix lower-acc))
         (memo (svex-aig-memotable-fix memo))
         ((When (zp width))
          (mv upper-acc lower-acc memo))
         ((unless (svex-is-const-concat x))
          (b* (((mv res memo) (svex->a4vec x env masks memo)))
            (mv (aig-logapp-nss width (a4vec->upper res) upper-acc)
                (aig-logapp-nss width (a4vec->lower res) lower-acc)
                memo)))
         ((mv sub-width lsbs msbs)
          (svex-const-concat-args x))
         ((unless (and (2vec-p sub-width) (natp (2vec->val sub-width))))
          (mv (aig-logapp-nss width (a4vec->upper (a4vec-x)) upper-acc)
              (aig-logapp-nss width (a4vec->lower (a4vec-x)) lower-acc)
              (svex-aig-memotable-fix memo)))
         (sub-width (2vec->val sub-width))
         (lsbs-width (min width sub-width))
         (msbs-width (- width lsbs-width))
         ((mv upper-acc lower-acc memo)
          (svex-concat->a4vec-lower msbs msbs-width upper-acc lower-acc env masks memo)))
      (svex-concat->a4vec-lower lsbs lsbs-width upper-acc lower-acc env masks memo)))


  ///
  (verify-guards svex->a4vec)

  (defun-sk svex->a4vec-table-ok (memo env masks aigenv)
    (forall x
            (let* ((memo (svex-aig-memotable-fix memo))
                   (mask (svex-mask-lookup x masks)))
              (implies (hons-assoc-equal (svex-fix x) memo)
                       (equal (a4vec-eval (cdr (hons-assoc-equal (svex-fix x) memo)) aigenv)
                              (4vec-mask mask
                                         (svex-eval x (svex-a4vec-env-eval env aigenv)))))))
    :rewrite :direct)

  (in-theory (disable svex->a4vec-table-ok
                      svex->a4vec-table-ok-necc))
  (local (in-theory (enable svex->a4vec-table-ok-necc)))

  ;; (defthm svex->a4vec-table-ok-necc-rw2
  ;;   (implies (svex->a4vec-table-ok memo env masks aigenv)
  ;;            (let* ((memo (svex-aig-memotable-fix memo))
  ;;                   (mask (svex-mask-lookup x masks)))
  ;;              (implies (and (svex-p x)
  ;;                            (hons-assoc-equal x memo))
  ;;                       (equal (a4vec-eval (cdr (hons-assoc-equal x memo)) aigenv)
  ;;                              (4vec-mask mask (svex-eval x (svex-a4vec-env-eval env aigenv)))))))
  ;;   :hints (("goal" :use svex->a4vec-table-ok-necc
  ;;            :in-theory (disable svex->a4vec-table-ok-necc))))

  (defthm svex->a4vec-table-ok-empty
    (svex->a4vec-table-ok nil env masks aigenv)
    :hints(("Goal" :in-theory (enable svex->a4vec-table-ok))))


  (defthm svex->a4vec-table-ok-extend
    (implies (and (svex->a4vec-table-ok memo env masks aigenv)
                  (equal (a4vec-eval val aigenv)
                         (4vec-mask (svex-mask-lookup x masks)
                                    (svex-eval x (svex-a4vec-env-eval env aigenv)))))
             (svex->a4vec-table-ok
              (cons (cons x val) memo) env masks aigenv))
    :hints (("goal" :expand ((svex->a4vec-table-ok
                              (cons (cons x val) memo) env masks aigenv)
                             (:free (x) (hide x)))
             :in-theory (disable 4vec-equal svex->a4vec-table-ok-necc))
            (and stable-under-simplificationp
                 '(:expand nil
                   :in-theory (disable 4vec-equal) ))))

  ;; (defthm svex->a4vec-table-ok-extend2
  ;;   (implies (and (svex->a4vec-table-ok memo env masks aigenv)
  ;;                 (equal (a4vec-eval val aigenv)
  ;;                        (4vec-mask? (svex-mask-lookup x masks)
  ;;                                    (svex-eval x (svex-a4vec-env-eval env aigenv))
  ;;                                    any)))
  ;;            (svex->a4vec-table-ok
  ;;             (cons (cons x val) memo) env masks aigenv))
  ;;   :hints (("goal" :expand ((svex->a4vec-table-ok
  ;;                             (cons (cons x val) memo) env masks aigenv)
  ;;                            (:free (x) (hide x)))
  ;;            :in-theory (disable 4vec-equal svex->a4vec-table-ok-necc))
  ;;           (and stable-under-simplificationp
  ;;                '(:expand nil
  ;;                  :in-theory (disable 4vec-equal) ))))

  ;; (defthm svex->a4vec-table-ok-extend3
  ;;   (implies (and (svex->a4vec-table-ok memo env masks aigenv)
  ;;                 (equal res (a4vec-eval val aigenv))
  ;;                 (bind-free (and (consp res)
  ;;                                 (eq (car res) '4vec-mask?)
  ;;                                 `((any . ,(fourth res))))
  ;;                            (any))
  ;;                 (equal res
  ;;                        (4vec-mask? (svex-mask-lookup x masks)
  ;;                                    (svex-eval x (svex-a4vec-env-eval env aigenv))
  ;;                                    any)))
  ;;            (svex->a4vec-table-ok
  ;;             (cons (cons x val) memo) env masks aigenv))
  ;;   :hints (("goal" :use svex->a4vec-table-ok-extend2
  ;;            :in-theory (disable svex->a4vec-table-ok-extend2))))

  (defthm svex->a4vec-table-ok-memotable-fix
    (iff (svex->a4vec-table-ok (svex-aig-memotable-fix memo) env masks aigenv)
         (svex->a4vec-table-ok memo env masks aigenv))
    :hints ((and stable-under-simplificationp
                 (if (eq (caar clause) 'not)
                     `(:expand (,(car (last clause)))
                       :in-theory (disable svex->a4vec-table-ok-necc)
                       :use ((:instance svex->a4vec-table-ok-necc
                              (x (svex->a4vec-table-ok-witness memo env masks aigenv))
                              (memo (svex-aig-memotable-fix memo)))))
                   `(:expand (,(car clause))
                     :in-theory (disable svex->a4vec-table-ok-necc)
                     :use ((:instance svex->a4vec-table-ok-necc
                            (x (svex->a4vec-table-ok-witness
                                (svex-aig-memotable-fix memo) env masks aigenv)))))))))

  (defthm svex->a4vec-table-ok-env-fix
    (iff (svex->a4vec-table-ok memo (svex-a4vec-env-fix env) masks aigenv)
         (svex->a4vec-table-ok memo env masks aigenv))
    :hints ((and stable-under-simplificationp
                 (if (eq (caar clause) 'not)
                     `(:expand (,(car (last clause)))
                       :in-theory (disable svex->a4vec-table-ok-necc)
                       :use ((:instance svex->a4vec-table-ok-necc
                              (x (svex->a4vec-table-ok-witness memo env masks aigenv))
                              (env (svex-a4vec-env-fix env)))))
                   `(:expand (,(car clause))
                     :in-theory (disable svex->a4vec-table-ok-necc)
                     :use ((:instance svex->a4vec-table-ok-necc
                            (x (svex->a4vec-table-ok-witness
                                memo (svex-a4vec-env-fix env) masks aigenv)))))))))

  (local (in-theory (disable svex->a4vec svexlist->a4vec)))

  (encapsulate nil
    (local (defthm lookup-in-svex-a4vec-env-eval-lemma
             (implies (svex-a4vec-env-p env)
                      (equal (hons-assoc-equal k (svex-a4vec-env-eval env aigenv))
                             (and (hons-assoc-equal k env)
                                  (cons k (a4vec-eval (cdr (hons-assoc-equal k env))
                                                      aigenv)))))
             :hints(("Goal" :in-theory (enable svex-a4vec-env-eval
                                               svex-a4vec-env-p)
                     :induct (svex-a4vec-env-eval env aigenv)
                     :do-not-induct t))
             :rule-classes nil))

    (defthm lookup-in-svex-a4vec-env-eval
      (equal (hons-assoc-equal k (svex-a4vec-env-eval env aigenv))
             (and (hons-assoc-equal k (svex-a4vec-env-fix env))
                  (cons k (a4vec-eval (cdr (hons-assoc-equal k (svex-a4vec-env-fix env)))
                                      aigenv))))
      :hints(("Goal" :use ((:instance lookup-in-svex-a4vec-env-eval-lemma
                            (env (svex-a4vec-env-fix env))))))))

  (defthm svex-env-lookup-in-svex-a4vec-env-eval
    (equal (svex-env-lookup k (svex-a4vec-env-eval env aigenv))
           (if (hons-assoc-equal (svar-fix k) (svex-a4vec-env-fix env))
               (a4vec-eval (cdr (hons-assoc-equal (svar-fix k) (svex-a4vec-env-fix env)))
                           aigenv)
             (4vec-x)))
    :hints(("Goal" :in-theory (enable svex-env-lookup))))

  ;; (local (defthm svex-apply-aig-correct-rw
  ;;        (implies (and (fnsym-p fn)
  ;;                (bind-free '((a4env . env)) (a4env))
  ;;                (equal (a4veclist-eval vals aigenv)
  ;;                       (4veclist-mask argmasks (svexlist-eval x (svex-a4vec-env-eval a4env aigenv))))
  ;;                (svex-argmasks-okp (svex-call fn x) mask argmasks))
  ;;           (equal (a4vec-eval (svex-apply-aig fn vals x mask) aigenv)
  ;;                  (4vec-mask mask (svex-apply fn (svexlist-eval x (svex-a4vec-env-eval a4env aigenv))))))


  (local (in-theory (enable svex-mask-alist-complete-necc)))


  (local (defthm 4vec-concat-when-not-2vec
           (implies (not (equal (4vec->upper width)
                                (4vec->lower width)))
                    (equal (4vec-concat width x y) (4vec-x)))
           :hints(("Goal" :in-theory (enable 4vec-concat)))))

  (local (defthm 4vec-concat-when-not-natp
           (implies (< (4vec->lower width) 0)
                    (equal (4vec-concat width x y) (4vec-x)))
           :hints(("Goal" :in-theory (enable 4vec-concat)))))

  (local (defthm 4vec-mask-idempotence-rw
           (implies (equal y (4vec-mask mask x))
                    (equal (4vec-mask mask y)
                           y))))

  (local (defthm 4vec-concat-when-width-0
           (implies (and (equal 0 (4vec->upper width))
                         (equal 0 (4vec->lower width)))
                    (equal (4vec-concat width x y) (4vec-fix y)))
           :hints(("Goal" :in-theory (enable 4vec-concat)))))


  ;; (local (defthm svex-argmasks-okp-implies-nths-for-concat
  ;;          (implies (and (equal (svex-call->fn x) 'concat)
  ;;                        (svex-case x :call)
  ;;                        (equal (len (svex-call->args x)) 3))
  ;;                   (implies (svex-argmasks-okp x xmask (svex-argmasks-lookup
  ;;                                                        (svex-call->args x) masks))
  ;;                            (svex-argmasks-okp x xmask (svex-argmasks-lookup
  ;;                                                        (list (nth 0 (svex-call->args x))
  ;;                                                              (nth 1 (svex-call->args x))
  ;;                                                              (nth 2 (svex-call->args x)))
  ;;                                                        masks))))
  ;;          :hints(("Goal" :in-theory (enable 4veclist-mask svex-apply
  ;;                                            svexlist-eval
  ;;                                            svex-argmasks-lookup
  ;;                                            4veclist-nth-safe
  ;;                                            nth))
  ;;                 (acl2::witness))))

  

  ;; (local (defthmd argmasks-okp-for-const-concat
  ;;          (b* (((mv width lsbs msbs) (svex-const-concat-args x)))
  ;;            (implies (and (svex-is-const-concat x)
  ;;                          (svex-mask-alist-complete masks)
  ;;                          (equal (4vec-mask (svex-mask-lookup lsbs masks)
  ;;                                            (svex-eval lsbs env))
  ;;                                 (4vec-mask (svex-mask-lookup lsbs masks) lsbs-val))
  ;;                          (equal (4vec-mask (svex-mask-lookup msbs masks)
  ;;                                            (svex-eval msbs env))
  ;;                                 (4vec-mask (svex-mask-lookup msbs masks) msbs-val)))
  ;;                     (equal (4vec-mask (svex-mask-lookup x masks)
  ;;                                       (svex-eval x env))
  ;;                            (4vec-mask (svex-mask-lookup x masks)
  ;;                                       (4vec-concat width lsbs-val msbs-val)))))
  ;;          :hints (("goal" :use ((:instance svex-mask-alist-complete-necc
  ;;                                 (y x) (mask-al masks))
  ;;                                (:instance svex-argmasks-okp-necc
  ;;                                 (vals (list (mv-nth 0 (svex-const-concat-args x))
  ;;                                             lsbs-val msbs-val))
  ;;                                 (mask (svex-mask-lookup x masks))
  ;;                                 (argmasks (svex-argmasks-lookup (svex-call->args x) masks))))
  ;;                   :in-theory (enable svex-argmasks-lookup
  ;;                                      svex-apply
  ;;                                      svexlist-eval
  ;;                                      4veclist-mask
  ;;                                      len nth)
  ;;                   :expand ((svex-is-const-concat x)
  ;;                            (svex-const-concat-args x)
  ;;                            (len (svex-call->args x))
  ;;                            (len (cdr (svex-call->args x)))
  ;;                            (len (cddr (svex-call->args x)))
  ;;                            (len (cdddr (svex-call->args x)))
  ;;                            (:free (n) (nth n (svex-call->args x)))
  ;;                            (:free (n) (nth n (cdr (svex-call->args x))))
  ;;                            (:free (n) (nth n (cddr (svex-call->args x))))
  ;;                            (:free (n) (nth n (cdddr (svex-call->args x)))))))
  ;;          :otf-flg t))

  (local (define 4vec-change-all-bits ((x 4vec-p))
           :returns (new-x 4vec-p)
           (4vec (lognot (4vec->upper x))
                 (lognot (4vec->lower x)))))


  
  (local
   (encapsulate nil
     (local (in-theory (disable* (:rules-of-class :linear :here)
                                 bitops::logand-natp-type-2
                                 bitops::logand-natp-type-1
                                 bitops::logior-natp-type
                                 bitops::lognot-negp
                                 bitops::lognot-natp
                                 member-svex-mask-alist-keys
                                 acl2::loghead-identity)))
     (defthm mask-of-bit?-logand-lognot
       (implies (4vmask-p mask)
                (equal (4vec-mask mask (4vec-bit? (2vec (logand a (lognot mask))) x y))
                       (4vec-mask mask y)))
       :hints(("Goal" :in-theory (e/d* (4vec-mask 4vec-bit? 3vec-bit?)))
              (logbitp-reasoning)))

     (defthm mask-of-bit?-logand-lognot2
       (implies (4vmask-p mask)
                (equal (4vec-mask mask (4vec-bit? (2vec (logand (lognot mask) a)) x y))
                       (4vec-mask mask y)))
       :hints(("Goal" :in-theory (e/d* (4vec-mask 4vec-bit? 3vec-bit?)))
              (logbitp-reasoning)))

     (defthm argmasks-okp-for-const-concat-implies-lsb-mask-subsumes-outer
       (b* (((mv width lsbs msbs) (svex-const-concat-args x)))
         (implies (and (svex-is-const-concat x)
                       (svex-mask-alist-complete masks)
                       (2vec-p width)
                       (<= (nfix w) (2vec->val width)))
                  (4vmask-subsumes (loghead w (svex-mask-lookup lsbs masks))
                                   (loghead w (svex-mask-lookup x masks)))))
       :hints (("goal" :use ((:instance svex-mask-alist-complete-necc
                              (y x) (mask-al masks))
                             (:instance svex-argmasks-okp-necc
                              (vals (b* (((mv width lsbs msbs) (svex-const-concat-args x)))
                                      (list width
                                            (4vec-bit?
                                             (2vec (logandc1 (svex-mask-lookup lsbs masks)
                                                             (svex-mask-lookup x masks)))
                                             (4vec-change-all-bits (svex-eval lsbs env))
                                             (svex-eval lsbs env))
                                            (svex-eval msbs env))))
                              (mask (svex-mask-lookup x masks))
                              (argmasks (svex-argmasks-lookup (svex-call->args x) masks))))
                :in-theory (enable svex-argmasks-lookup
                                   svex-apply
                                   4vmask-subsumes
                                   svexlist-eval
                                   4veclist-mask
                                   len nth)
                :expand ((svex-is-const-concat x)
                         (svex-const-concat-args x)
                         (len (svex-call->args x))
                         (len (cdr (svex-call->args x)))
                         (len (cddr (svex-call->args x)))
                         (len (cdddr (svex-call->args x)))
                         (:free (n) (nth n (svex-call->args x)))
                         (:free (n) (nth n (cdr (svex-call->args x))))
                         (:free (n) (nth n (cddr (svex-call->args x))))
                         (:free (n) (nth n (cdddr (svex-call->args x))))))
               (and stable-under-simplificationp
                    '(:in-theory (enable 4vec-mask 4vec-bit? 3vec-bit? 4vec-change-all-bits 4vec-concat)))
               (logbitp-reasoning
                :add-hints (:in-theory (enable* logbitp-case-splits))
                :simp-hint (:in-theory (enable* logbitp-case-splits)))
               )
       :otf-flg t)

     (defthm argmasks-okp-for-const-concat-implies-msb-mask-subsumes-outer
       (b* (((mv width lsbs msbs) (svex-const-concat-args x)))
         (implies (and (svex-is-const-concat x)
                       (svex-mask-alist-complete masks)
                       (2vec-p width)
                       (<= 0 (2vec->val width)))
                  (4vmask-subsumes (svex-mask-lookup msbs masks)
                                   (logtail (4vec->lower width) (svex-mask-lookup x masks)))))
       :hints (("goal" :use ((:instance svex-mask-alist-complete-necc
                              (y x) (mask-al masks))
                             (:instance svex-argmasks-okp-necc
                              (vals (b* (((mv width lsbs msbs) (svex-const-concat-args x)))
                                      (list width
                                            (svex-eval lsbs env)
                                            (4vec-bit?
                                             (2vec (logandc1 (svex-mask-lookup msbs masks)
                                                             (logtail (2vec->val width)
                                                                      (svex-mask-lookup x masks))))
                                             (4vec-change-all-bits (svex-eval msbs env))
                                             (svex-eval msbs env)))))
                              (mask (svex-mask-lookup x masks))
                              (argmasks (svex-argmasks-lookup (svex-call->args x) masks))))
                :in-theory (enable svex-argmasks-lookup
                                   svex-apply
                                   4vmask-subsumes
                                   svexlist-eval
                                   4veclist-mask
                                   len nth)
                :expand ((svex-is-const-concat x)
                         (svex-const-concat-args x)
                         (len (svex-call->args x))
                         (len (cdr (svex-call->args x)))
                         (len (cddr (svex-call->args x)))
                         (len (cdddr (svex-call->args x)))
                         (:free (n) (nth n (svex-call->args x)))
                         (:free (n) (nth n (cdr (svex-call->args x))))
                         (:free (n) (nth n (cddr (svex-call->args x))))
                         (:free (n) (nth n (cdddr (svex-call->args x))))))
               (and stable-under-simplificationp
                    '(:in-theory (enable 4vec-mask 4vec-bit? 3vec-bit? 4vec-change-all-bits 4vec-concat)))
               (logbitp-reasoning
                :add-hints (:in-theory (enable* logbitp-case-splits))
                :simp-hint (:in-theory (enable* logbitp-case-splits)))
               )
       :otf-flg t)


     (defthm argmasks-okp-for-const-concat-implies-lsb-mask-subsumes-outer-when-tail-0
       (b* (((mv width lsbs msbs) (svex-const-concat-args x)))
         (implies (and (svex-is-const-concat x)
                       (svex-mask-alist-complete masks)
                       (2vec-p width) (<= 0 (2vec->val width))
                       (equal 0 (logtail (2vec->val width) (svex-mask-lookup x masks))))
                  (4vmask-subsumes (svex-mask-lookup lsbs masks)
                                   (svex-mask-lookup x masks))))
       :hints (("goal" :use ((:instance argmasks-okp-for-const-concat-implies-lsb-mask-subsumes-outer (w (2vec->val (mv-nth 0 (svex-const-concat-args x))))))
                :in-theory (e/d (4vmask-subsumes)
                                (argmasks-okp-for-const-concat-implies-lsb-mask-subsumes-outer)))
               (logbitp-reasoning :prune-examples nil)
               )
       :otf-flg t)))


  (local (defthmd 4vec-mask-over-4vec-zero-ext
           (implies (and (2vec-p width)
                         (<= 0 (2vec->val width)))
                    (equal (4vec-mask mask (4vec-zero-ext width x))
                           (4vec-concat width (4vec-mask mask x)
                                        (4vec-mask (logtail (2vec->val width) (4vmask-fix mask))
                                                   0))))
           :hints(("Goal" :in-theory (enable 4vec-mask 4vec-zero-ext 4vec-concat))
                  (logbitp-reasoning :prune-examples nil))))

  (local (defthmd 4vec-mask-over-4vec-concat
           (implies (and (2vec-p width)
                         (<= 0 (2vec->val width)))
                    (equal (4vec-mask mask (4vec-concat width x y))
                           (4vec-concat width
                                        (4vec-mask mask x)
                                        (4vec-mask (logtail (2vec->val width) (4vmask-fix mask))
                                                   y))))
           :hints(("Goal" :in-theory (enable 4vec-mask 4vec-concat))
                  (logbitp-reasoning :prune-examples nil))))

  (local (defthmd 4vec-concat-of-4vec-mask-identity
           (implies (and (equal mask1 (4vmask-fix mask))
                         (2vec-p w)
                         (<= 0 (2vec->val w))
                         (equal w1 (2vec->val w)))
                    (equal (4vec-concat w
                                        (4vec-mask mask x)
                                        (4vec-mask (logtail w1 mask1)
                                                   (4vec-rsh w x)))
                           (4vec-mask mask x)))
           :hints(("Goal" :in-theory (enable 4vec-mask 4vec-concat 4vec-rsh 4vec-shift-core))
                  (logbitp-reasoning :prune-examples nil))))

  (local (defthmd 4vec-concat-of-equal-4vec-conct
           (implies (and (equal x (4vec-concat w a b))
                         (2vec-p w)
                         (<= 0 (2vec->val w)))
                    (equal (4vec-concat w x y)
                           (4vec-concat w a y)))))

  (local (defthm min-gte-zero
           (implies (and (<= 0 x)
                         (<= 0 y))
                    (<= 0 (min x y)))))

  (local (defthm 2vec-of-4vec-acc
           (implies (2vec-p x)
                    (and (equal (2vec (4vec->lower x)) (4vec-fix x))
                         (equal (2vec (4vec->upper x)) (4vec-fix x))))
           :hints(("Goal" :in-theory (enable 2vec)))))

  (local (defthmd 4vec-concat-identity
           (implies (and (2vec-p w) (<= 0 (2vec->val w)))
                    (equal (4vec-concat w x (4vec-rsh w x))
                           (4vec-fix x)))
           :hints(("Goal" :in-theory (enable 4vec-concat 4vec-rsh 4vec-shift-core))
                  (logbitp-reasoning :prune-examples nil))))

  (local (defthmd equal-of-4vec-concat2
           (implies (and (2vec-p w) (<= 0 (2vec->val w)))
                    (equal (equal (4vec-concat w x y) z)
                           (and (4vec-p z)
                                (equal (4vec-zero-ext w x) (4vec-zero-ext w z))
                                (equal (4vec-fix y) (4vec-rsh w z)))))
           :hints(("Goal" :in-theory (enable 4vec-concat 4vec-zero-ext 4vec-rsh 4vec-shift-core))
                  (logbitp-reasoning))
           :otf-flg t))

  ;; (local (defthm equal-of-4vec-concat2-same-w
  ;;          (implies (and (2vec-p w) (<= 0 (2vec->val w)))
  ;;                   (equal (equal (4vec-concat w x y) (4vec-concat w a b))
  ;;                          (and (equal (4vec-zero-ext w x) (4vec-zero-ext w a))
  ;;                               (equal (4vec-fix y) (4vec-fix b)))))
  ;;          :hints(("Goal" :in-theory (enable equal-of-4vec-concat2))
  ;;                 (and stable-under-simplificationp
  ;;                      '(:in-theory (enable 4vec-concat 4vec-zero-ext))))
  ;;          :otf-flg t))

  ;; (local (defthm equal-of-4vec-concat2-same-head
  ;;          (implies (and (2vec-p w) (<= 0 (2vec->val w))
  ;;                        (equal (4vec-zero-ext w x) (4vec-zero-ext w z)))
  ;;                   (equal (equal (4vec-concat w x y) z)
  ;;                          (and (4vec-p z) 
  ;;                               (equal (4vec-fix y) (4vec-rsh w z)))))
  ;;          :hints(("Goal" :in-theory (enable equal-of-4vec-concat2))
  ;;                 (and stable-under-simplificationp
  ;;                      '(:in-theory (enable 4vec-concat 4vec-zero-ext))))
  ;;          :otf-flg t))
  
  (local (defthm 4vec-zero-ext-of-concat
           (implies (and (2vec-p w1) (2vec-p w2)
                         (<= 0 (2vec->val w1))
                         (<= (2vec->val w1) (2vec->val w2)))
                    (equal (4vec-zero-ext w1 (4vec-concat w2 x y))
                           (4vec-zero-ext w1 x)))
           :hints(("Goal" :in-theory (enable 4vec-zero-ext 4vec-concat)))))

  (local (defthm 4vec-rsh-of-4vec-mask
           (implies (and (2vec-p shift) (<= 0 (2vec->val shift)))
                    (equal (4vec-rsh shift (4vec-mask mask x))
                           (4vec-mask (logtail (2vec->val shift) (4vmask-fix mask))
                                      (4vec-rsh shift x))))
           :hints(("Goal" :in-theory (enable 4vec-mask 4vec-rsh 4vec-shift-core)))))

  (local (defthm equal-zero-ext-of-mask
           (implies (and (2vec-p w)
                         (<= 0 (2vec->val w)))
                    (equal (equal (4vec-zero-ext w (4vec-mask m x))
                                  (4vec-zero-ext w (4vec-mask m y)))
                           (equal (4vec-mask (loghead (2vec->val w) (4vmask-fix m)) x)
                                  (4vec-mask (loghead (2vec->val w) (4vmask-fix m)) y))))
           :hints(("Goal" :in-theory (enable 4vec-zero-ext 4vec-mask))
                  (logbitp-reasoning))))


  ;; (local (defthmd argmasks-okp-for-const-concat-w
  ;;          (b* (((mv width lsbs msbs) (svex-const-concat-args x))
  ;;               (lsbs-width (min (2vec->val width) (2vec->val w)))
  ;;               (msbs-width (- (2vec->val w) lsbs-width)))
  ;;            (implies (and (svex-is-const-concat x)
  ;;                          (svex-mask-alist-complete masks)
  ;;                          (2vec-p width) (<= 0 (2vec->val width))
  ;;                          (2vec-p w) (<= 0 (2vec->val w))
  ;;                          (equal (4vec-mask (svex-mask-lookup lsbs masks)
  ;;                                            (4vec-zero-ext (2vec lsbs-width) (svex-eval lsbs env)))
  ;;                                 (4vec-mask (svex-mask-lookup lsbs masks) lsbs-val))
  ;;                          (equal (4vec-mask (svex-mask-lookup msbs masks)
  ;;                                            (4vec-zero-ext (2vec msbs-width) (svex-eval msbs env)))
  ;;                                 (4vec-mask (svex-mask-lookup msbs masks) msbs-val)))
  ;;                     (equal (4vec-mask (svex-mask-lookup x masks)
  ;;                                       (4vec-zero-ext w (svex-eval x env)))
  ;;                            (4vec-mask (svex-mask-lookup x masks)
  ;;                                       (4vec-concat width lsbs-val msbs-val)))))
  ;;          :hints (("goal" :use ((:instance svex-mask-alist-complete-necc
  ;;                                 (y x) (mask-al masks))
  ;;                                (:instance svex-argmasks-okp-necc
  ;;                                 (vals (b* (((mv width lsbs msbs) (svex-const-concat-args x))
  ;;                                            (lsbs-width (min (2vec->val width) (2vec->val w)))
  ;;                                            (msbs-width (- (2vec->val w) lsbs-width)))
  ;;                                         (list width
  ;;                                               (4vec-concat
  ;;                                                (2vec lsbs-width)
  ;;                                                lsbs-val
  ;;                                                (4vec-rsh
  ;;                                                 (2vec lsbs-width)
  ;;                                                 (svex-eval lsbs env)))
  ;;                                               (4vec-concat
  ;;                                                (2vec msbs-width)
  ;;                                                msbs-val
  ;;                                                (4vec-rsh
  ;;                                                 (2vec msbs-width)
  ;;                                                 (svex-eval msbs env))))))
  ;;                                 (mask (svex-mask-lookup x masks))
  ;;                                 (argmasks (svex-argmasks-lookup (svex-call->args x) masks))))
  ;;                   :in-theory (e/d (svex-argmasks-lookup
  ;;                                    svex-apply
  ;;                                    svexlist-eval
  ;;                                    4veclist-mask
  ;;                                    len nth
  ;;                                    4vec-mask-over-4vec-concat
  ;;                                    4vec-mask-over-4vec-zero-ext
  ;;                                    4vec-concat-of-equal-4vec-conct
  ;;                                    4vec-concat-of-4vec-mask-identity)
  ;;                                   ())
  ;;                   :expand ((svex-is-const-concat x)
  ;;                            (svex-const-concat-args x)
  ;;                            (len (svex-call->args x))
  ;;                            (len (cdr (svex-call->args x)))
  ;;                            (len (cddr (svex-call->args x)))
  ;;                            (len (cdddr (svex-call->args x)))
  ;;                            (:free (n) (nth n (svex-call->args x)))
  ;;                            (:free (n) (nth n (cdr (svex-call->args x))))
  ;;                            (:free (n) (nth n (cddr (svex-call->args x))))
  ;;                            (:free (n) (nth n (cdddr (svex-call->args x)))))
  ;;                   :do-not-induct t)
  ;;                  (and stable-under-simplificationp
  ;;                       '(:in-theory (enable equal-of-4vec-concat2))))))


  ;; (local (defthmd argmasks-okp-for-const-concat-zero
  ;;          (b* (((mv width ?lsbs msbs) (svex-const-concat-args x)))
  ;;            (implies (and (svex-is-const-concat x)
  ;;                          (2vec-p width)
  ;;                          (eql (2vec->val width) 0)
  ;;                          (svex-mask-alist-complete masks)
  ;;                          (equal (4vec-mask (svex-mask-lookup msbs masks)
  ;;                                            (svex-eval msbs env))
  ;;                                 (4vec-mask (svex-mask-lookup msbs masks) msbs-val)))
  ;;                     (equal (4vec-mask (svex-mask-lookup x masks)
  ;;                                       (svex-eval x env))
  ;;                            (4vec-mask (svex-mask-lookup x masks)
  ;;                                       msbs-val))))
  ;;          :hints (("goal" :use ((:instance svex-mask-alist-complete-necc
  ;;                                 (y x) (mask-al masks))
  ;;                                (:instance svex-argmasks-okp-necc
  ;;                                 (vals (list (mv-nth 0 (svex-const-concat-args x))
  ;;                                             (svex-eval (mv-nth 1 (svex-const-concat-args x)) env) msbs-val))
  ;;                                 (mask (svex-mask-lookup x masks))
  ;;                                 (argmasks (svex-argmasks-lookup (svex-call->args x) masks))))
  ;;                   :in-theory (enable svex-argmasks-lookup
  ;;                                      svex-apply
  ;;                                      svexlist-eval
  ;;                                      4veclist-mask
  ;;                                      len nth)
  ;;                   :expand ((svex-is-const-concat x)
  ;;                            (svex-const-concat-args x)
  ;;                            (len (svex-call->args x))
  ;;                            (len (cdr (svex-call->args x)))
  ;;                            (len (cddr (svex-call->args x)))
  ;;                            (len (cdddr (svex-call->args x)))
  ;;                            (:free (n) (nth n (svex-call->args x)))
  ;;                            (:free (n) (nth n (cdr (svex-call->args x))))
  ;;                            (:free (n) (nth n (cddr (svex-call->args x))))
  ;;                            (:free (n) (nth n (cdddr (svex-call->args x)))))))
  ;;          :otf-flg t))



  ;; (local (defthmd argmasks-okp-for-const-concat-zero-w
  ;;          (b* (((mv width ?lsbs msbs) (svex-const-concat-args x)))
  ;;            (implies (and (svex-is-const-concat x)
  ;;                          (2vec-p width)
  ;;                          (eql (2vec->val width) 0)
  ;;                          (svex-mask-alist-complete masks)
  ;;                          (2vec-p w)
  ;;                          (<= 0 (2vec->val w))
  ;;                          (equal (4vec-mask (svex-mask-lookup msbs masks)
  ;;                                            (4vec-zero-ext w (svex-eval msbs env)))
  ;;                                 (4vec-mask (svex-mask-lookup msbs masks)
  ;;                                            msbs-val)))
  ;;                     (equal (4vec-mask (svex-mask-lookup x masks)
  ;;                                       (4vec-zero-ext w (svex-eval x env)))
  ;;                            (4vec-mask (svex-mask-lookup x masks)
  ;;                                       msbs-val))))
  ;;          :hints (("goal" :use ((:instance svex-mask-alist-complete-necc
  ;;                                 (y x) (mask-al masks))
  ;;                                (:instance svex-argmasks-okp-necc
  ;;                                 (vals (list (mv-nth 0 (svex-const-concat-args x))
  ;;                                             (svex-eval (mv-nth 1 (svex-const-concat-args x)) env)
  ;;                                             (4vec-concat w msbs-val
  ;;                                                          (4vec-rsh w (svex-eval (mv-nth 2 (svex-const-concat-args x)) env)))))
  ;;                                 (mask (svex-mask-lookup x masks))
  ;;                                 (argmasks (svex-argmasks-lookup (svex-call->args x) masks))))
  ;;                   :in-theory (enable svex-argmasks-lookup
  ;;                                      svex-apply
  ;;                                      svexlist-eval
  ;;                                      4veclist-mask
  ;;                                      len nth
  ;;                                      4vec-mask-over-4vec-zero-ext
  ;;                                      4vec-mask-over-4vec-concat
  ;;                                      4vec-concat-of-4vec-mask-identity
  ;;                                      4vec-concat-of-equal-4vec-conct)
  ;;                   :expand ((svex-is-const-concat x)
  ;;                            (svex-const-concat-args x)
  ;;                            (len (svex-call->args x))
  ;;                            (len (cdr (svex-call->args x)))
  ;;                            (len (cddr (svex-call->args x)))
  ;;                            (len (cdddr (svex-call->args x)))
  ;;                            (:free (n) (nth n (svex-call->args x)))
  ;;                            (:free (n) (nth n (cdr (svex-call->args x))))
  ;;                            (:free (n) (nth n (cddr (svex-call->args x))))
  ;;                            (:free (n) (nth n (cdddr (svex-call->args x)))))))
  ;;          :otf-flg t))

  (local (defthmd 4vec-mask-of-4vec-concat-under-mask
           (implies (and (equal (logtail (2vec->val width) (4vmask-fix mask))
                                0)
                         (2vec-p width)
                         (<= 0 (2vec->val width)))
                    (equal (4vec-mask mask (4vec-concat width x y))
                           (4vec-mask mask x)))
           :hints(("Goal" :in-theory (enable 4vec-mask 4vec-concat))
                  (logbitp-reasoning :prune-examples nil))))

  ;; (local (defthmd argmasks-okp-for-const-concat-lower
  ;;          (b* (((mv width lsbs ?msbs) (svex-const-concat-args x)))
  ;;            (implies (and (svex-is-const-concat x)
  ;;                          (svex-mask-alist-complete masks)
  ;;                          (2vec-p width)
  ;;                          (<= 0 (2vec->val width))
  ;;                          (equal (logtail (2vec->val width) (svex-mask-lookup x masks))
  ;;                                 0)
  ;;                          (equal (4vec-mask (svex-mask-lookup lsbs masks)
  ;;                                            (svex-eval lsbs env))
  ;;                                 (4vec-mask (svex-mask-lookup lsbs masks) lsbs-val)))
  ;;                     (equal (4vec-mask (svex-mask-lookup x masks)
  ;;                                       (svex-eval x env))
  ;;                            (4vec-mask (svex-mask-lookup x masks)
  ;;                                       lsbs-val))))
  ;;          :hints (("goal" :use ((:instance svex-mask-alist-complete-necc
  ;;                                 (y x) (mask-al masks))
  ;;                                (:instance svex-argmasks-okp-necc
  ;;                                 (vals (list (mv-nth 0 (svex-const-concat-args x))
  ;;                                             lsbs-val
  ;;                                             (svex-eval (mv-nth 2 (svex-const-concat-args x)) env)))
  ;;                                 (mask (svex-mask-lookup x masks))
  ;;                                 (argmasks (svex-argmasks-lookup (svex-call->args x) masks))))
  ;;                   :in-theory (enable svex-argmasks-lookup
  ;;                                      svex-apply
  ;;                                      svexlist-eval
  ;;                                      4veclist-mask
  ;;                                      len nth
  ;;                                      4vec-mask-of-4vec-concat-under-mask)
  ;;                   :expand ((svex-is-const-concat x)
  ;;                            (svex-const-concat-args x)
  ;;                            (len (svex-call->args x))
  ;;                            (len (cdr (svex-call->args x)))
  ;;                            (len (cddr (svex-call->args x)))
  ;;                            (len (cdddr (svex-call->args x)))
  ;;                            (:free (n) (nth n (svex-call->args x)))
  ;;                            (:free (n) (nth n (cdr (svex-call->args x))))
  ;;                            (:free (n) (nth n (cddr (svex-call->args x))))
  ;;                            (:free (n) (nth n (cdddr (svex-call->args x)))))))
  ;;          :otf-flg t))

  ;; (local (defthmd argmasks-okp-for-const-concat-all
  ;;          (b* (((mv width lsbs msbs) (svex-const-concat-args x)))
  ;;            (implies (and (svex-is-const-concat x)
  ;;                          (svex-mask-alist-complete masks))
  ;;                     (and (implies (and (2vec-p width)
  ;;                                        (<= 0 (2vec->val width))
  ;;                                        (equal (logtail (2vec->val width) (svex-mask-lookup x masks))
  ;;                                               0)
  ;;                                        (equal (4vec-mask (svex-mask-lookup lsbs masks)
  ;;                                                          (svex-eval lsbs env))
  ;;                                               (4vec-mask (svex-mask-lookup lsbs masks) lsbs-val)))
  ;;                                   (equal (equal (4vec-mask (svex-mask-lookup x masks)
  ;;                                                            (svex-eval x env))
  ;;                                                 (4vec-mask (svex-mask-lookup x masks)
  ;;                                                            lsbs-val))
  ;;                                          t))
  ;;                          (implies (and (2vec-p width)
  ;;                                        (eql (2vec->val width) 0)
  ;;                                        (equal (4vec-mask (svex-mask-lookup msbs masks)
  ;;                                                          (svex-eval msbs env))
  ;;                                               (4vec-mask (svex-mask-lookup msbs masks) msbs-val)))
  ;;                                   (equal (equal (4vec-mask (svex-mask-lookup x masks)
  ;;                                                            (svex-eval x env))
  ;;                                                 (4vec-mask (svex-mask-lookup x masks)
  ;;                                                            msbs-val))
  ;;                                          t))
  ;;                          (implies (and (equal (4vec-mask (svex-mask-lookup lsbs masks)
  ;;                                                          (svex-eval lsbs env))
  ;;                                               (4vec-mask (svex-mask-lookup lsbs masks) lsbs-val))
  ;;                                        (equal (4vec-mask (svex-mask-lookup msbs masks)
  ;;                                                          (svex-eval msbs env))
  ;;                                               (4vec-mask (svex-mask-lookup msbs masks) msbs-val)))
  ;;                                   (equal (equal (4vec-mask (svex-mask-lookup x masks)
  ;;                                                            (svex-eval x env))
  ;;                                                 (4vec-mask (svex-mask-lookup x masks)
  ;;                                                            (4vec-concat width lsbs-val msbs-val)))
  ;;                                          t)))))
  ;;          :hints (("goal" :use (argmasks-okp-for-const-concat-lower
  ;;                                argmasks-okp-for-const-concat-zero
  ;;                                argmasks-okp-for-const-concat)))))

  (local (defthmd 4vec-concat-of-4vec
           (equal (4vec (logapp w a b)
                        (logapp w c d))
                  (4vec-concat (2vec (nfix w)) (4vec a c) (4vec b d)))
           :hints(("Goal" :in-theory (enable 4vec-concat)))))

  (local (defthmd 4vec-zero-ext-of-4vec
           (equal (4vec (loghead w a)
                        (loghead w c))
                  (4vec-zero-ext (2vec (nfix w)) (4vec a c)))
           :hints(("Goal" :in-theory (enable 4vec-zero-ext)))))

  (local (defthmd 4vec-zero-ext-of-4vec/0
           (equal (4vec (loghead w a) 0)
                  (4vec-zero-ext (2vec (nfix w)) (4vec a 0)))
           :hints(("Goal" :in-theory (enable 4vec-zero-ext)))))

  (local (defthmd 4vec-of-aig-list->s-of-upper/lower
           (equal (4vec (aig-list->s (a4vec->upper x) env)
                        (aig-list->s (a4vec->lower x) env))
                  (a4vec-eval x env))
           :hints(("Goal" :in-theory (enable a4vec-eval)))))


  (local (defthm aig-list->s-of-list-fix
           (equal (aig-list->s (list-fix x) env)
                  (aig-list->s x env))
           :hints(("Goal" :in-theory (enable aig-list->s)))))


  ;; (local (defthmd 4vec-mask-over-4vec-concat
  ;;          (implies (and (2vec-p width)
  ;;                        (<= 0 (2vec->val width)))
  ;;                   (equal (4vec-mask mask (4vec-concat width x y))
  ;;                          (4vec-concat width
  ;;                                       (4vec-mask (loghead (2vec->val width) (4vmask-fix mask)) x)
  ;;                                       (4vec-mask (logtail (2vec->val width) (4vmask-fix mask)) y))))
  ;;          :hints(("Goal" :in-theory (enable 4vec-mask 4vec-concat))
  ;;                 (logbitp-reasoning :prune-examples nil))))

  

  (local (defthm 4vec-mask-of-loghead-4vec-mask
           (implies (equal mask1 (4vmask-fix mask))
                    (Equal (4vec-mask (loghead n mask1) (4vec-mask mask x))
                           (4vec-mask (loghead n mask1) x)))
           :hints(("Goal" :in-theory (enable 4vec-mask))
                  (logbitp-reasoning))))


  (local (defun aig-logapp-nss-ind (w1 w2 x y)
           (declare (xargs :measure (nfix w1)))
           (if (or (zp w1) (zp w2))
               (list x y)
             (aig-logapp-nss-ind (1- w1) (1- w2) (gl::scdr x) y))))

  (local (defthm aig-logapp-nss-of-nfix
           (Equal (aig-logapp-nss (nfix w) x y)
                  (aig-logapp-nss w x y))
           :hints(("Goal" :in-theory (enable aig-logapp-nss)))))

  (local (defthm aig-logapp-nss-of-list-fix
           (Equal (aig-logapp-nss w (list-fix x) y)
                  (aig-logapp-nss w x y))
           :hints(("Goal" :in-theory (enable aig-logapp-nss)))))

  (local (defthm aig-logapp-nss-of-snorm
           (Equal (aig-logapp-nss w (gl::bfr-snorm x) y)
                  (aig-logapp-nss w x y))
           :hints(("Goal" :in-theory (enable aig-logapp-nss)))))

  (local (defthm aig-logapp-nss-of-aig-logapp-nss
           (equal (aig-logapp-nss w1 (aig-logapp-nss w2 x y) z)
                  (let* ((wa (min (nfix w1) (nfix w2)))
                         (wb (- (nfix w1) wa)))
                    (aig-logapp-nss wa x (aig-logapp-nss wb y z))))
           :hints(("Goal" :in-theory (enable aig-logapp-nss)
                   :induct (aig-logapp-nss-ind w1 w2 x y)
                   :expand ((:free (x y) (aig-logapp-nss w1 x y))
                            (:free (x y) (aig-logapp-nss w2 x y)))))))

  (local (defthm aig-logapp-nss-when-zp
           (implies (zp w)
                    (equal (aig-logapp-nss w x y) (list-fix y)))
           :hints(("Goal" :in-theory (enable aig-logapp-nss)))))

  (define svex-concat->a4vec-lower-zero ((x svex-p)
                                         (width natp)
                                         (env svex-a4vec-env-p)
                                         (masks svex-mask-alist-p)
                                         (memo svex-aig-memotable-p))
    :returns (mv (upper true-listp)
                 (lower true-listp)
                 (memo1 svex-aig-memotable-p))
    :measure (svex-count x)
    :hooks nil
    :verify-guards nil
    (b* ((memo (svex-aig-memotable-fix memo))
         ((When (zp width))
          (mv nil nil memo))
         ((unless (svex-is-const-concat x))
          (b* (((mv res memo) (svex->a4vec x env masks memo)))
            (mv (aig-logapp-nss width (a4vec->upper res) nil)
                (aig-logapp-nss width (a4vec->lower res) nil)
                memo)))
         ((mv sub-width lsbs msbs)
          (svex-const-concat-args x))
         ((unless (and (2vec-p sub-width) (natp (2vec->val sub-width))))
          (mv (aig-logapp-nss width (a4vec->upper (a4vec-x)) nil)
              (aig-logapp-nss width (a4vec->lower (a4vec-x)) nil)
              (svex-aig-memotable-fix memo)))
         (sub-width (2vec->val sub-width))
         (lsbs-width (min width sub-width))
         (msbs-width (- width lsbs-width))
         ((mv upper2 lower2 memo)
          (svex-concat->a4vec-lower-zero msbs msbs-width env masks memo))
         ((mv upper1 lower1 memo)
          (svex-concat->a4vec-lower-zero lsbs lsbs-width env masks memo)))
      (mv (aig-logapp-nss lsbs-width upper1 upper2)
          (aig-logapp-nss lsbs-width lower1 lower2)
          memo))
    ///
    (defthm-svex->a4vec-flag
      (defthm svex-concat->a4vec-lower-in-terms-of-zero
        (b* (((mv upper-impl lower-impl memo-impl)
              (svex-concat->a4vec-lower x width upper-acc lower-acc env masks memo))
             ((mv upper-spec lower-spec memo-spec)
              (svex-concat->a4vec-lower-zero x width env masks memo)))
          (and (equal memo-impl memo-spec)
               (equal upper-impl (aig-logapp-nss width upper-spec (list-fix upper-acc)))
               (equal lower-impl (aig-logapp-nss width lower-spec (list-fix lower-acc)))))
        :flag svex-concat->a4vec-lower
        :hints ('(:expand ((svex-concat->a4vec-lower x width upper-acc lower-acc env masks memo)
                           (svex-concat->a4vec-lower-zero x width env masks memo)))))
      :skip-others t))


  (local (defthm 4vec-mask-of-loghead-zero-ext
           (implies (natp width)
                    (equal (4vec-mask (loghead width mask) (4vec-zero-ext (2vec width) x))
                           (4vec-mask (loghead width mask) x)))
           :hints(("Goal" :in-theory (enable 4vec-mask 4vec-zero-ext))
                  (logbitp-reasoning))))

  (local (defthm 4vec-mask-zero-ext-mask
           (equal (4vec-mask mask (4vec-zero-ext w (4vec-mask mask x)))
                  (4vec-mask mask (4vec-zero-ext w x)))
           :hints(("Goal" :in-theory (enable 4vec-mask 4vec-zero-ext))
                  (logbitp-reasoning))))

  
  (local (defthm 4vec-zero-ext-0
           (equal (4vec-zero-ext 0 x) 0)
           :hints(("Goal" :in-theory (enable 4vec-zero-ext)))))

  

  ;; (local (Defthm 4vec-zero-ext-of-zp
  ;;                   (equal (4vec-zero-ext (2vec width) x) 0))
  ;;          :hints(("Goal" :in-theory (enable* 4vec-zero-ext
  ;;                                             ihsext-recursive-redefs))))
  
  (local (defthm 4vec-concat-0
           (equal (4vec-concat width x 0)
                  (4vec-zero-ext width x))
           :hints(("Goal" :in-theory (enable 4vec-concat 4vec-zero-ext)))))

  (local (defthm 4vec-of-equal-upper-lower
           (implies (and (equal a (4vec->upper x))
                         (equal b (4vec->lower x)))
                    (equal (4vec a b) (4vec-fix x)))))



  (local (defthm 4vec-mask?-of-4vec-mask?-when-subsumes
           (implies (4vmask-subsumes m1 m2)
                    (equal (4vec-mask? m2 a (4vec-mask? m1 a b))
                           (4vec-mask? m1 a b)))
           :hints(("Goal" :in-theory (enable 4vec-mask? 4vec-bit? 3vec-bit? 4vmask-subsumes))
                  (logbitp-reasoning))))
             
  (local (defthm zero-ext-of-zero-ext
           (implies (and (2vec-p a) (2vec-p b)
                         (<= 0 (2vec->val a))
                         (<= (2vec->val a) (2vec->val b)))
                    (and (equal (4vec-zero-ext a (4vec-zero-ext b x))
                                (4vec-zero-ext a x))
                         (equal (4vec-zero-ext b (4vec-zero-ext a x))
                                (4vec-zero-ext a x))))
           :hints(("Goal" :in-theory (enable 4vec-zero-ext)))))


  (local (Defthm 4vec-rsh-of-4vec-mask?
           (implies (and (2vec-p sh) (<= 0 (2vec->val sh)))
                    (equal (4vec-rsh sh (4vec-mask? mask a b))
                           (4vec-mask? (logtail (2vec->val sh) (4vmask-fix mask))
                                       (4vec-rsh sh a)
                                       (4vec-rsh sh b))))
           :hints(("Goal" :in-theory (enable 4vec-rsh 4vec-shift-core 4vec-mask?
                                             4vec-bit? 3vec-bit?)))))

  (local (defthm 4vec-rsh-of-4vec-zero-ext
           (implies (and (2vec-p a) (2vec-p b)
                         (<= 0 (2vec->val a))
                         (<= (2vec->val a) (2vec->val b)))
                    (and (equal (4vec-rsh a (4vec-zero-ext b x))
                                (4vec-zero-ext (2vec (- (2vec->val b) (2vec->val a)))
                                               (4vec-rsh a x)))
                         (equal (4vec-rsh b (4vec-zero-ext a x)) 0)))
           :hints(("Goal" :in-theory (enable 4vec-rsh 4vec-shift-core 4vec-zero-ext))
                  (and stable-under-simplificationp
                       '(:in-theory (enable* ihsext-recursive-redefs))))))

  (local (defthm 4vmask-subsumes-of-loghead
           (implies (and (4vmask-subsumes a b)
                         (4vmask-p a) (4vmask-p b))
                    (4vmask-subsumes (loghead w a) (loghead w b)))
           :hints(("Goal" :in-theory (enable 4vmask-subsumes))
                  (logbitp-reasoning))))

  (local (defthm 4vec-mask?-of-concat-same
           (implies (and (2vec-p w) (<= 0 (2vec->val w)))
                    (equal (4vec-mask? mask (4vec-concat w a b) (4vec-concat w c d))
                           (4vec-concat w
                                        (4vec-mask? (loghead (2vec->val w) (4vmask-fix mask))
                                                    (4vec-zero-ext w a)
                                                    (4vec-zero-ext w c))
                                        (4vec-mask? (logtail (2vec->val w) (4vmask-fix mask)) b d))))
           :hints(("Goal" :in-theory (enable 4vec-mask? 4vec-bit? 3vec-bit?
                                             4vec-concat 4vec-zero-ext))
                  (logbitp-reasoning))))

  (local (defthm 4vec-mask?-of-concat-when-logtail-0
           (implies (and (equal 0 (logtail (2vec->val w) (4vmask-fix mask)))
                         (2vec-p w) (<= 0 (2vec->val w)))
                    (equal (4vec-mask? mask (4vec-concat w a b) c)
                           (4vec-mask? mask a c)))
           :hints(("Goal" :in-theory (enable 4vec-mask? 4vec-bit? 3vec-bit? 4vec-concat))
                  (logbitp-reasoning :prune-examples nil))))

  (local (defthm 4veclist-mask?-of-conses
           (equal (4veclist-mask? (cons c1 c2) (cons t1 t2) (cons f1 f2))
                  (cons (4vec-mask? c1 t1 f1)
                        (4veclist-mask? c2 t2 f2)))
           :hints(("Goal" :in-theory (enable 4veclist-mask?)))))


  (defthm-svex->a4vec-flag
    (defthm svex->a4vec-correct
      (b* (((mv res memo1) (svex->a4vec x env masks memo)))
        (implies (and (svex->a4vec-table-ok memo env masks aigenv)
                      (svex-mask-alist-complete masks))
                 (and (svex->a4vec-table-ok memo1 env masks aigenv)
                      (equal (a4vec-eval res aigenv)
                             (4vec-mask (svex-mask-lookup x masks)
                                        (svex-eval x (svex-a4vec-env-eval env aigenv)))))))
      :hints ('(:expand ((svex->a4vec x env masks memo)
                         (:free (env) (svex-eval x env)))))
      :flag svex->a4vec)
    (defthm svexlist->a4vec-correct
      (b* (((mv res memo1) (svexlist->a4vec x env masks memo)))
        (implies (and (svex->a4vec-table-ok memo env masks aigenv)
                      (svex-mask-alist-complete masks))
                 (and (svex->a4vec-table-ok memo1 env masks aigenv)
                      (equal (a4veclist-eval res aigenv)
                             (4veclist-mask (svex-argmasks-lookup x masks)
                                            (svexlist-eval x (svex-a4vec-env-eval env aigenv)))))))
      :hints ('(:expand ((svexlist->a4vec x env masks memo)
                         (:free (env) (svexlist-eval x env))
                         (a4veclist-eval nil aigenv)
                         (svex-argmasks-lookup x masks)
                         (:free (a b) (a4veclist-eval (cons a b) aigenv)))
                :in-theory (enable 4veclist-mask)))
      :flag svexlist->a4vec)
    (defthm svex-concat->a4vec-correct
      (b* (((mv upper lower memo1) (svex-concat->a4vec x env masks memo)))
        (implies (and (svex->a4vec-table-ok memo env masks aigenv)
                      (svex-mask-alist-complete masks))
                 (and (svex->a4vec-table-ok memo1 env masks aigenv)
                      (4vec-mask-equiv (a4vec-eval (a4vec upper lower) aigenv)
                                       (svex-eval x (svex-a4vec-env-eval env aigenv))
                                       (svex-mask-lookup x masks)))))
      :hints ((acl2::just-expand ((:free (x) (hide x))) :last-only t :lambdasp t)
              '(:expand ((svex-concat->a4vec x env masks memo))
                ;; :use ((:instance argmasks-okp-for-const-concat-all
                ;;        (lsbs-val (4vec (aig-list->s (mv-nth 0 (svex-concat->a4vec (mv-nth 1 (svex-const-concat-args x)) env masks memo)) aigenv)
                ;;                        (aig-list->s (mv-nth 1 (svex-concat->a4vec (mv-nth 1 (svex-const-concat-args x)) env masks memo)) aigenv)))
                ;;        (msbs-val (let ((memo (if (eql 0 (2vec->val (mv-nth 0 (svex-const-concat-args x))))
                ;;                                  memo
                ;;                                (mv-nth 2 (svex-concat->a4vec (mv-nth 1 (svex-const-concat-args x))
                ;;                                                              env masks memo)))))
                ;;                    (4vec (aig-list->s (mv-nth 0 (SVEX-CONCAT->A4VEC
                ;;                                                  (MV-NTH 2 (SVEX-CONST-CONCAT-ARGS X))
                ;;                                                  ENV MASKS
                ;;                                                  memo))
                ;;                                       aigenv)
                ;;                          (aig-list->s (mv-nth 1 (SVEX-CONCAT->A4VEC
                ;;                                                  (MV-NTH 2 (SVEX-CONST-CONCAT-ARGS X))
                ;;                                                  ENV MASKS
                ;;                                                  memo))
                ;;                                       aigenv))))
                ;;        (env (SVEX-A4VEC-ENV-EVAL ENV AIGENV))))
                :in-theory (e/d (svex-const-concat-args-correct-rw
                                 4vec-concat-of-4vec))))
      :flag svex-concat->a4vec)

    (defthm svex-concat->a4vec-lower-correct
      (b* (((mv upper lower memo1) (svex-concat->a4vec-lower-zero x width env masks memo)))
        (implies (and (svex->a4vec-table-ok memo env masks aigenv)
                      (svex-mask-alist-complete masks))
                 (and (svex->a4vec-table-ok memo1 env masks aigenv)
                      (let ((ans (a4vec-eval (a4vec upper lower) aigenv)))
                        (equal ans
                               (4vec-zero-ext
                                (2vec (nfix width))
                                (4vec-mask? (svex-mask-lookup x masks)
                                            (svex-eval x (svex-a4vec-env-eval env aigenv))
                                            (hide ans))))))))
      :hints ((acl2::just-expand ((:free (x) (hide x))) :last-only t :lambdasp t)
              '(:expand ((svex-concat->a4vec-lower-zero x width env masks memo))
                ;; :use ((:instance argmasks-okp-for-const-concat-all
                ;;        (lsbs-val (b* ((memo (svex-aig-memotable-fix memo))
                ;;                       ((mv sub-width lsbs msbs)
                ;;                        (svex-const-concat-args x))
                ;;                       (sub-width (2vec->val sub-width))
                ;;                       (lsbs-width (min width sub-width))
                ;;                       (msbs-width (- width lsbs-width))
                ;;                       ((mv upper2 lower2 memo)
                ;;                        (svex-concat->a4vec-lower-zero msbs msbs-width env masks memo))
                ;;                       ((mv upper1 lower1 ?memo)
                ;;                        (svex-concat->a4vec-lower-zero lsbs lsbs-width env masks memo)))
                ;;                    (4vec (aig-list->s upper1 aigenv)
                ;;                          (aig-list->s lower1 aigenv))))
                ;;        (msbs-val (b* ((memo (svex-aig-memotable-fix memo))
                ;;                       ((mv sub-width lsbs msbs)
                ;;                        (svex-const-concat-args x))
                ;;                       (sub-width (2vec->val sub-width))
                ;;                       (lsbs-width (min width sub-width))
                ;;                       (msbs-width (- width lsbs-width))
                ;;                       ((mv upper2 lower2 ?memo)
                ;;                        (svex-concat->a4vec-lower-zero msbs msbs-width env masks memo))
                ;;                       ;; ((mv upper1 lower1 memo)
                ;;                       ;;  (svex-concat->a4vec-lower-zero lsbs lsbs-width env masks memo))
                ;;                       )
                ;;                    (4vec (aig-list->s upper2 aigenv)
                ;;                          (aig-list->s lower2 aigenv))))
                ;;        (env (SVEX-A4VEC-ENV-EVAL ENV AIGENV))))
                :in-theory (e/d (svex-const-concat-args-correct-rw
                                 4vec-concat-of-4vec
                                 4vec-zero-ext-of-4vec
                                 4vec-zero-ext-of-4vec/0
                                 4vec-of-aig-list->s-of-upper/lower
                                 ;; 4vec-mask-over-4vec-concat
                                 ;; 4vec-mask-over-4vec-zero-ext
                                 )
                                (BITOPS::LOGAPP-OF-I-0
                                 a4vec-eval-of-var)))
              (and stable-under-simplificationp
                   '(:in-theory (e/d (equal-of-4vec-concat2)
                                     (a4vec-eval-of-var)))))
      :flag svex-concat->a4vec-lower)
    )
  

  (defthm-svex->a4vec-flag
    (defthm svexlist->a4vec-true-listp
      (true-listp (mv-nth 0 (svexlist->a4vec x env masks memo)))
      :hints ('(:expand ((svexlist->a4vec x env masks memo))))
      :flag svexlist->a4vec)
    :skip-others t)


  (deffixequiv-mutual svex->a4vec))

(define svex->a4vec-top ((x svex-p) (env svex-a4vec-env-p) (masks svex-mask-alist-p))
  :returns (res a4vec-p)
  (b* (((mv res memo)
        (svex->a4vec x (make-fast-alist env) masks nil)))
    (fast-alist-free memo)
    res)
  ///
  (defthm svex->a4vec-top-correct
    (implies (svex-mask-alist-complete masks)
             (equal (a4vec-eval (svex->a4vec-top x env masks) aigenv)
                    (4vec-mask (svex-mask-lookup x masks)
                               (svex-eval x (svex-a4vec-env-eval env aigenv)))))))


(define svexlist->a4vec-nrev ((x svexlist-p)
                              (env svex-a4vec-env-p)
                              (masks svex-mask-alist-p)
                              (memo svex-aig-memotable-p)
                              (acl2::nrev))
  :returns (mv (new-nrev)
               (memo1 svex-aig-memotable-p))
  (if (atom x)
      (b* ((acl2::nrev (acl2::nrev-fix acl2::nrev)))
        (mv acl2::nrev (svex-aig-memotable-fix memo)))
    (b* (((mv first memo) (svex->a4vec (car x) env masks memo))
         (acl2::nrev (acl2::nrev-push first acl2::nrev)))
      (svexlist->a4vec-nrev (cdr x) env masks memo acl2::nrev)))
  ///
  (defret svexlist->a4vec-nrev-removal
    (b* (((mv out-spec memo1-spec)
          (svexlist->a4vec x env masks memo)))
      (and (equal new-nrev
                  (append acl2::nrev out-spec))
           (equal memo1 memo1-spec)))
    :hints(("Goal" :induct t
            :expand ((svexlist->a4vec x env masks memo))))))

(define svexlist->a4vec-top ((x svexlist-p) (env svex-a4vec-env-p) (masks svex-mask-alist-p))
  ;; note: env must be fast
  :prepwork ((local (defthm svexlist->a4vec-decomp
                      (equal (list (mv-nth 0 (svexlist->a4vec x env masks memo))
                                   (mv-nth 1 (svexlist->a4vec x env masks memo)))
                             (svexlist->a4vec x env masks memo))
                      :hints (("goal" :expand ((svexlist->a4vec x env masks memo)))))))
  :returns (res a4veclist-p)
  (b* (((mv res memo)
        (mbe :logic (svexlist->a4vec x env masks nil)
             :exec (with-local-stobj acl2::nrev
                     (mv-let (res memo acl2::nrev)
                       (b* (((mv acl2::nrev memo)
                             (svexlist->a4vec-nrev x env masks nil acl2::nrev))
                            ((mv res acl2::nrev) (acl2::nrev-finish acl2::nrev)))
                         (mv res memo acl2::nrev))
                       (mv res memo))))))
    (fast-alist-free memo)
    res)
  ///
  (defthm svexlist->a4vec-top-correct
    (implies (svex-mask-alist-complete masks)
             (equal (a4veclist-eval (svexlist->a4vec-top x env masks) aigenv)
                    (4veclist-mask (svex-argmasks-lookup x masks)
                                   (svexlist-eval x (svex-a4vec-env-eval env aigenv)))))))


;; There are a few possible approaches to generating the a4vec-env to use in
;; building AIGs.  Basically, we're going to generate a pair of Boolean
;; variables for each bit that matters (mask-wise) in x; the masks for the
;; variables of x must be finite so that we can determine which bits matter.
;; (The don't-care bits will be assigned X.)
;; But there are still a few options:

;; 1. Always assign AIG variables to every care bit of every variable.  This
;; has the advantage that the input and output to svexlist->a4vec is always
;; identical for a given svexlist, so we can memoize, which may improve
;; performance if we run several simulations with the same svex but different
;; environments.  But the complete set of variables might be overkill,
;; e.g. svtvs have lots of variables for don't-care inputs and initial
;; states.

;; 2. Assign AIG variables to the care bits of all the variables bound in the
;; environment.  This could still allow good memoization as long as the same
;; variables are going to be assigned basically all the time.  We can extract
;; and sort the variables so that the order of the bindings doesn't matter.

;; 3. In between 1 and 2: Take a list of variables as an extra argument,
;; ignored in the logic, which should be a superset of the variables in the
;; environment; only generate AIG vars for these variables.  This care list
;; could be provided e.g. by the SVTV.  This has the advantage over 2 that
;; memoization still works if different subsets of the care variables are
;; used in different runs.

;; The implementations of 1,2,3 would be quite similar and could basically
;; all be based on 3.

;; 4. More extreme than 2, more difficult to implement, and little chance of
;; memoization working: in the symbolic environment provided, we only really
;; need AIG variables for non-constant bits.  So ignore masks and just
;; produce an a4vec environment that replicates the constants assigned in env
;; and generates AIG variables for the non-constant bits.  This would mean
;; we'd need to get a hold of the symbolic environment, which probably would
;; mean we'd need a symbolic counterpart function, not just an alternate
;; definition.


;; The following implements option 3 above.  We provide the svexlist and the
;; list of care vars.  The function returns the a4veclist and the a4vec-env.

(define nat-bool-listp (x)
  (if (atom x)
      (eq x nil)
    (and (or (natp (car x))
             (booleanp (car x)))
         (nat-bool-listp (cdr x))))
  ///
  (defthm nat-bool-listp-of-aig-sterm
    (implies (or (booleanp x) (natp x))
             (nat-bool-listp (aig-sterm x)))
    :hints(("Goal" :in-theory (enable gl::bfr-sterm))))

  (defthm nat-bool-listp-of-aig-scons
    (implies (and (or (booleanp x) (natp x))
                  (nat-bool-listp y))
             (nat-bool-listp (aig-scons x y)))
    :hints(("Goal" :in-theory (enable gl::bfr-scons))))

  (defthm nat-bool-listp-of-list-fix
    (implies (nat-bool-listp x)
             (nat-bool-listp (list-fix x)))))

(define nat-bool-list-nats ((x nat-bool-listp))
  :prepwork ((local (in-theory (enable nat-bool-listp))))
  (if (atom x)
      nil
    (if (booleanp (car x))
        (nat-bool-list-nats (cdr x))
      (cons (lnfix (car x))
            (nat-bool-list-nats (cdr x)))))
  ///
  (defthm nat-bool-list-nats-of-list-fix
    (equal (nat-bool-list-nats (list-fix x))
           (nat-bool-list-nats x)))

  (defthm nat-bool-list-nats-of-aig-sterm
    (equal (nat-bool-list-nats (aig-sterm x))
           (if (booleanp x)
               nil
             (list (nfix x))))
    :hints(("Goal" :in-theory (enable gl::bfr-sterm))))

  (defthm nat-bool-list-nats-of-aig-scons-bool
    (implies (booleanp x)
             (equal (nat-bool-list-nats (aig-scons x y))
                    (nat-bool-list-nats y)))
    :hints(("Goal" :in-theory (enable gl::bfr-scons))))

  (defthm nat-bool-list-nats-of-aig-scons-nat
    (implies (and (natp x)
                  (not (member x (nat-bool-list-nats y))))
             (equal (nat-bool-list-nats (aig-scons x y))
                    (cons x (nat-bool-list-nats y))))
    :hints(("Goal" :in-theory (enable gl::bfr-scons)))))

(define nat-bool-list-lower-boundp ((bound natp) (x nat-bool-listp))
  :prepwork ((local (in-theory (enable nat-bool-listp))))
  (if (atom x)
      t
    (and (or (booleanp (car x))
             (<= (lnfix bound) (lnfix (car x))))
         (nat-bool-list-lower-boundp bound (cdr x))))
  ///
  (defthm nat-bool-list-lower-boundp-of-list-fix
    (equal (nat-bool-list-lower-boundp bound (list-fix x))
           (nat-bool-list-lower-boundp bound x)))

  (defthm nat-bool-list-nonmember-by-lower-bound
    (implies (and (nat-bool-list-lower-boundp bound x)
                  (< v (nfix bound))
                  (nat-bool-listp x))
             (not (member v (nat-bool-list-nats x))))
    :hints(("Goal" :in-theory (enable nat-bool-list-nats))))

  (Defthm nat-bool-list-lower-boundp-lower
    (implies (and (nat-bool-list-lower-boundp bound x)
                  (<= (nfix n) (nfix bound)))
             (nat-bool-list-lower-boundp n x))))

(define nat-bool-list-upper-boundp ((bound natp) (x nat-bool-listp))
  :prepwork ((local (in-theory (enable nat-bool-listp))))
  (if (atom x)
      t
    (and (or (booleanp (car x))
             (< (lnfix (car x)) (lnfix bound)))
         (nat-bool-list-upper-boundp bound (cdr x))))
  ///
  (defthm nat-bool-list-upper-boundp-of-list-fix
    (equal (nat-bool-list-upper-boundp bound (list-fix x))
           (nat-bool-list-upper-boundp bound x)))

  (defthm nat-bool-list-nonmember-by-upper-bound
    (implies (and (nat-bool-list-upper-boundp bound x)
                  (<= (nfix bound) v)
                  (nat-bool-listp x))
             (not (member v (nat-bool-list-nats x))))
    :hints(("Goal" :in-theory (enable nat-bool-list-nats))))

  (defthm nat-bool-list-no-intersection-by-bounds
    (implies (and (nat-bool-list-upper-boundp bound0 x0)
                  (nat-bool-list-lower-boundp bound1 x1)
                  (<= (nfix bound0) (nfix bound1))
                  (nat-bool-listp x0)
                  (nat-bool-listp x1))
             (not (intersectp (nat-bool-list-nats x0)
                              (nat-bool-list-nats x1))))
    :hints(("Goal" :in-theory (e/d (intersectp-equal
                                    nat-bool-listp
                                    nat-bool-list-nats)
                                   (acl2::intersectp-equal-commute)))))

  (Defthm nat-bool-list-upper-boundp-higher
    (implies (and (nat-bool-list-upper-boundp bound x)
                  (<= (nfix bound) (nfix n)))
             (nat-bool-list-upper-boundp n x))))


(define nat-bool-a4vec-p ((x a4vec-p))
  (b* (((a4vec x) x))
    (and (nat-bool-listp x.upper)
         (nat-bool-listp x.lower)))
  ///
  (deffixtype nba4vec :pred nat-bool-a4vec-p! :fix a4vec-fix :equiv a4vec-equiv))

(defmacro nat-bool-a4vec-p! (x)
  `(and (a4vec-p ,x)
        (nat-bool-a4vec-p ,x)))

(define nat-bool-a4vec-vars ((x nat-bool-a4vec-p!))
  :prepwork ((local (in-theory (enable nat-bool-a4vec-p))))
  (b* (((a4vec x) x))
    (append (nat-bool-list-nats x.upper)
            (nat-bool-list-nats x.lower))))

(define nat-bool-a4vec-lower-boundp ((bound natp) (x nat-bool-a4vec-p!))
  :prepwork ((local (in-theory (enable nat-bool-a4vec-p))))
  (b* (((a4vec x) x))
    (and (nat-bool-list-lower-boundp bound x.upper)
         (nat-bool-list-lower-boundp bound x.lower)))
  ///
  (defthm nat-bool-a4vec-vars-nonmember-by-lower-bound
    (implies (and (nat-bool-a4vec-lower-boundp bound x)
                  (< v (nfix bound))
                  (nat-bool-a4vec-p x))
             (not (member v (nat-bool-a4vec-vars x))))
    :hints(("Goal" :in-theory (enable nat-bool-a4vec-vars
                                      nat-bool-a4vec-p
                                      nat-bool-list-nonmember-by-lower-bound))))

  (Defthm nat-bool-a4vec-lower-boundp-lower
    (implies (and (nat-bool-a4vec-lower-boundp bound x)
                  (<= (nfix n) (nfix bound)))
             (nat-bool-a4vec-lower-boundp n x))))

(define nat-bool-a4vec-upper-boundp ((bound natp) (x nat-bool-a4vec-p!))
  :prepwork ((local (in-theory (enable nat-bool-a4vec-p))))
  (b* (((a4vec x) x))
    (and (nat-bool-list-upper-boundp bound x.upper)
         (nat-bool-list-upper-boundp bound x.lower)))
  ///
  (defthm nat-bool-a4vec-vars-nonmember-by-upper-bound
    (implies (and (nat-bool-a4vec-upper-boundp bound x)
                  (<= (nfix bound) v)
                  (nat-bool-a4vec-p x))
             (not (member v (nat-bool-a4vec-vars x))))
    :hints(("Goal" :in-theory (enable nat-bool-a4vec-vars
                                      nat-bool-a4vec-p
                                      nat-bool-list-nonmember-by-upper-bound))))

  (defthm nat-bool-a4vec-no-intersection-by-bounds
    (implies (and (nat-bool-a4vec-upper-boundp bound0 x0)
                  (nat-bool-a4vec-lower-boundp bound1 x1)
                  (<= (nfix bound0) (nfix bound1))
                  (nat-bool-a4vec-p x0)
                  (nat-bool-a4vec-p x1))
             (not (intersectp (nat-bool-a4vec-vars x0)
                              (nat-bool-a4vec-vars x1))))
    :hints(("Goal" :in-theory (e/d (intersectp-equal
                                    nat-bool-a4vec-p
                                    nat-bool-a4vec-vars
                                    nat-bool-a4vec-upper-boundp
                                    nat-bool-a4vec-lower-boundp)
                                   (acl2::intersectp-equal-commute)))))

  (Defthm nat-bool-a4vec-upper-boundp-higher
    (implies (and (nat-bool-a4vec-upper-boundp bound x)
                  (<= (nfix bound) (nfix n)))
             (nat-bool-a4vec-upper-boundp n x))))


(define nat-bool-a4env-p ((x svex-a4vec-env-p))
  :prepwork ((local (in-theory (enable svex-a4vec-env-fix))))
  (if (atom x)
      t
    (and (if (mbt (consp (car x)))
             (nat-bool-a4vec-p (cdar x))
           t)
         (nat-bool-a4env-p (cdr x))))
  ///
  (deffixtype nba4env :pred nat-bool-a4env-p! :fix svex-a4vec-env-fix :equiv svex-a4vec-env-equiv))

(defmacro nat-bool-a4env-p! (x)
  `(and (svex-a4vec-env-p ,x)
        (nat-bool-a4env-p ,x)))

(define nat-bool-a4env-vars ((x nat-bool-a4env-p!))
  :prepwork ((local (in-theory (enable nat-bool-a4env-p
                                       svex-a4vec-env-fix))))
  (if (atom x)
      nil
    (append (and (mbt (consp (car x)))
                 (nat-bool-a4vec-vars (cdar x)))
            (nat-bool-a4env-vars (cdr x)))))

(define nat-bool-a4env-lower-boundp ((bound natp) (x nat-bool-a4env-p!))
  :prepwork ((local (in-theory (enable nat-bool-a4env-p
                                       svex-a4vec-env-fix))))
  (if (atom x)
      t
    (and (if (mbt (consp (car x)))
             (nat-bool-a4vec-lower-boundp bound (cdar x))
           t)
         (nat-bool-a4env-lower-boundp bound (cdr x))))
  ///
  (defthm nat-bool-a4env-vars-nonmember-by-lower-bound
    (implies (and (nat-bool-a4env-lower-boundp bound x)
                  (< v (nfix bound))
                  (nat-bool-a4env-p x))
             (not (member v (nat-bool-a4env-vars x))))
    :hints(("Goal" :in-theory (enable nat-bool-a4env-vars
                                      nat-bool-a4env-p
                                      nat-bool-list-nonmember-by-lower-bound))))

  (Defthm nat-bool-a4env-lower-boundp-lower
    (implies (and (nat-bool-a4env-lower-boundp bound x)
                  (<= (nfix n) (nfix bound)))
             (nat-bool-a4env-lower-boundp n x))))

(define nat-bool-a4env-upper-boundp ((bound natp) (x nat-bool-a4env-p!))
  :prepwork ((local (in-theory (enable nat-bool-a4env-p
                                       svex-a4vec-env-fix))))
  (if (atom x)
      t
    (and (if (mbt (consp (car x)))
             (nat-bool-a4vec-upper-boundp bound (cdar x))
           t)
         (nat-bool-a4env-upper-boundp bound (cdr x))))
  ///
  (defthm nat-bool-a4env-vars-nonmember-by-upper-bound
    (implies (and (nat-bool-a4env-upper-boundp bound x)
                  (<= (nfix bound) v)
                  (nat-bool-a4env-p x))
             (not (member v (nat-bool-a4env-vars x))))
    :hints(("Goal" :in-theory (enable nat-bool-a4env-vars
                                      nat-bool-a4env-p
                                      nat-bool-list-nonmember-by-upper-bound))))

  (defthm nat-bool-a4vec/env-no-intersection-by-bounds
    (implies (and (nat-bool-a4vec-upper-boundp bound0 x0)
                  (nat-bool-a4env-lower-boundp bound1 x1)
                  (<= (nfix bound0) (nfix bound1))
                  (nat-bool-a4vec-p x0)
                  (nat-bool-a4env-p x1))
             (not (intersectp (nat-bool-a4vec-vars x0)
                              (nat-bool-a4env-vars x1))))
    :hints(("Goal" :in-theory (e/d (intersectp-equal
                                    nat-bool-a4env-p
                                    nat-bool-a4env-lower-boundp)
                                   (acl2::intersectp-equal-commute))
            :expand ((nat-bool-a4env-vars x1))
            :induct (nat-bool-a4env-lower-boundp bound1 x1))))

  (defthm nat-bool-a4env-no-intersection-by-bounds
    (implies (and (nat-bool-a4env-upper-boundp bound0 x0)
                  (nat-bool-a4env-lower-boundp bound1 x1)
                  (<= (nfix bound0) (nfix bound1))
                  (nat-bool-a4env-p x0)
                  (nat-bool-a4env-p x1))
             (not (intersectp (nat-bool-a4env-vars x0)
                              (nat-bool-a4env-vars x1))))
    :hints(("Goal" :in-theory (e/d (intersectp-equal
                                    nat-bool-a4env-p
                                    nat-bool-a4env-vars
                                    nat-bool-a4env-upper-boundp
                                    nat-bool-a4env-lower-boundp)
                                   (acl2::intersectp-equal-commute)))))

  (defthm nat-bool-a4env-a4vec-no-intersection-by-bounds
    (implies (and (nat-bool-a4env-upper-boundp bound0 x0)
                  (nat-bool-a4vec-lower-boundp bound1 x1)
                  (<= (nfix bound0) (nfix bound1))
                  (nat-bool-a4env-p x0)
                  (nat-bool-a4vec-p x1))
             (not (intersectp (nat-bool-a4env-vars x0)
                              (nat-bool-a4vec-vars x1))))
    :hints(("Goal" :in-theory (e/d (intersectp-equal
                                    nat-bool-a4env-p
                                    nat-bool-a4env-vars
                                    nat-bool-a4env-upper-boundp
                                    nat-bool-a4env-lower-boundp)
                                   (acl2::intersectp-equal-commute)))))

  (Defthm nat-bool-a4env-upper-boundp-higher
    (implies (and (nat-bool-a4env-upper-boundp bound x)
                  (<= (nfix bound) (nfix n)))
             (nat-bool-a4env-upper-boundp n x))))




(local (defthm logcount-of-logand
         (implies (natp y)
                  (<= (logcount (logand x y))
                      (logcount y)))
         :hints(("Goal" :in-theory (e/d* (bitops::logcount**
                                          bitops::logand**
                                          bitops::ihsext-inductions))))
         :rule-classes :linear))

(define 4vmask-to-a4vec-varcount ((mask natp) (boolmask integerp))
  :returns (count natp :rule-classes :type-prescription)
  (b* ((mask (lnfix mask)))
    (- (* 2 (logcount mask))
       (logcount (logand mask (lifix boolmask))))))

(define 4vmask-to-a4vec-rec ((mask natp) (boolmask integerp) (nextvar natp))
  :returns (mv (upper nat-bool-listp)
               (lower nat-bool-listp))
  :measure (integer-length mask)
  :hints(("Goal" :in-theory (enable bitops::integer-length**
                                    4vmask-fix)))
  (b* ((mask (lnfix mask))
       (nextvar (lnfix nextvar))
       ((when (eql mask 0))
        (mv (aig-sterm t) (aig-sterm nil)))
       ((mv ubit0 ubit1 nextvar)
        (if (logbitp 0 mask)
            (if (logbitp 0 boolmask)
                (mv nextvar nextvar (+ 1 nextvar))
              (mv nextvar (1+ nextvar) (+ 2 nextvar)))
          (mv t nil nextvar)))
       ((mv rest-upper rest-lower)
        (4vmask-to-a4vec-rec (logcdr mask) (logcdr boolmask) nextvar)))
    (mv (aig-scons ubit0 rest-upper)
        (aig-scons ubit1 rest-lower)))
  ///
  ;; (defthm 4vmask-to-a4vec-rec-nextvar
  ;;   (equal (mv-nth 2 (4vmask-to-a4vec-rec mask nextvar))
  ;;          (+ (* 2 (logcount (nfix mask))) (nfix nextvar)))
  ;;   :hints(("Goal" :in-theory (enable bitops::logcount**))))

  (defthm 4vmask-to-a4vec-rec-lower-bounds
    (and (nat-bool-list-lower-boundp nextvar (mv-nth 0 (4vmask-to-a4vec-rec mask boolmask nextvar)))
         (nat-bool-list-lower-boundp nextvar (mv-nth 1 (4vmask-to-a4vec-rec mask boolmask nextvar))))
    :hints(("Goal" :in-theory (enable nat-bool-list-lower-boundp gl::bfr-scons)))
    :rule-classes ((:forward-chaining :trigger-terms ((4vmask-to-a4vec-rec mask boolmask nextvar)))))

  (defthm 4vmask-to-a4vec-rec-upper-bounds
    (and (nat-bool-list-upper-boundp (+ (nfix nextvar)
                                        (4vmask-to-a4vec-varcount mask boolmask))
                                     (mv-nth 0 (4vmask-to-a4vec-rec mask boolmask nextvar)))
         (nat-bool-list-upper-boundp (+ (nfix nextvar)
                                        (4vmask-to-a4vec-varcount mask boolmask))
                                     (mv-nth 1 (4vmask-to-a4vec-rec mask boolmask nextvar))))
    :hints(("Goal" :in-theory (enable gl::bfr-scons
                                      4vmask-to-a4vec-varcount
                                      bitops::logand**
                                      bitops::logbitp**
                                      bitops::logcount**)
            :induct (4vmask-to-a4vec-rec mask boolmask nextvar)
            :expand ((:free (bound a b) (nat-bool-list-upper-boundp bound (cons a b)))
                     (:free (bound) (nat-bool-list-upper-boundp bound nil)))
            :do-not-induct t))
    :rule-classes ((:forward-chaining :trigger-terms ((4vmask-to-a4vec-rec mask boolmask nextvar)))))

  ;; (defthm 4vmask-to-a4vec-rec-no-duplicate-vars
  ;;   (b* (((mv upper lower)
  ;;         (4vmask-to-a4vec-rec mask boolmask nextvar)))
  ;;     (and (no-duplicatesp (nat-bool-list-nats upper))
  ;;          (no-duplicatesp (nat-bool-list-nats lower))
  ;;          (not (intersectp (nat-bool-list-nats upper)
  ;;                           (nat-bool-list-nats lower)))))
  ;;   :hints(("Goal" :in-theory (enable nat-bool-list-nats
  ;;                                     intersectp-equal))))

  (defthm member-4vmask-to-a4vec-rec-vars
    ;; not a good rewrite rule but useful for the next phase
    (iff (member v (append (nat-bool-list-nats (mv-nth 0 (4vmask-to-a4vec-rec mask boolmask nextvar)))
                           (nat-bool-list-nats (mv-nth 1 (4vmask-to-a4vec-rec mask boolmask nextvar)))))
         (and (natp v)
              (<= (nfix nextvar) v)
              (< v (+ (nfix nextvar)
                      (4vmask-to-a4vec-varcount mask boolmask)))))
    :hints(("Goal" :in-theory (enable nat-bool-list-nats
                                      4vmask-to-a4vec-varcount
                                      bitops::logand**
                                      bitops::logcount**
                                      bitops::logbitp**)))
    :rule-classes nil))

  ;; (defthm eval-4vmask-to-a4vec-rec-cons-greater
  ;;   (b* (((mv ?err upper lower nextvar1)
  ;;         (4vmask-to-a4vec-rec mask nextvar)))
  ;;     (implies (<= nextvar1 var)
  ;;              (and (equal (aig-list->s upper (cons (cons var val) env))
  ;;                          (aig-list->s upper env))
  ;;                   (equal (aig-list->s lower (cons (cons var val) env))
  ;;                          (aig-list->s lower env))))))

  ;; (defthm eval-4vmask-to-a4vec-rec-cons-lesser
  ;;   (b* (((mv ?err upper lower ?nextvar1)
  ;;         (4vmask-to-a4vec-rec mask nextvar)))
  ;;     (implies (< var (nfix nextvar))
  ;;              (and (equal (aig-list->s upper (cons (cons var val) env))
  ;;                          (aig-list->s upper env))
  ;;                   (equal (aig-list->s lower (cons (cons var val) env))
  ;;                          (aig-list->s lower env)))))))

(define 4vmask-to-a4vec-rec-env ((mask natp)
                                 (boolmask integerp)
                                 (upper integerp)
                                 (lower integerp)
                                 (nextvar natp))
  :returns (env "environment for the resulting 4vmask")
  :measure (integer-length mask)
  :hints(("Goal" :in-theory (enable bitops::integer-length**
                                    4vmask-fix)))
  (b* ((mask (lnfix mask))
       (nextvar (lnfix nextvar))
       ((when (eql mask 0)) nil)
       (rest-env
        (4vmask-to-a4vec-rec-env (logcdr mask)
                                 (logcdr boolmask)
                                 (logcdr upper)
                                 (logcdr lower)
                                 (if (logbitp 0 mask)
                                     (if (logbitp 0 boolmask)
                                         (+ 1 nextvar)
                                       (+ 2 nextvar))
                                   nextvar))))
    (if (logbitp 0 mask)
        (cons (cons nextvar (logbitp 0 upper))
              (if (logbitp 0 boolmask)
                  rest-env
                (cons (cons (1+ nextvar) (logbitp 0 lower))
                      rest-env)))
      rest-env))
  ///

  (defthm key-exists-in-4vmask-to-a4vec-rec-env
    (iff (hons-assoc-equal v (4vmask-to-a4vec-rec-env mask boolmask upper lower nextvar))
         (and (natp v)
              (<= (nfix nextvar) v)
              (< v (+ (nfix nextvar)
                      (4vmask-to-a4vec-varcount mask boolmask)))))
    :hints(("Goal" :in-theory (enable bitops::logcount**
                                      bitops::logbitp**
                                      4vmask-to-a4vec-varcount
                                      bitops::logand**)
            :do-not-induct t
            :induct (4vmask-to-a4vec-rec-env mask boolmask upper lower nextvar))))

  (defthm nat-bool-aig-list->s-of-cons-nonmember
    (implies (and (nat-bool-listp x)
                  (not (member n (nat-bool-list-nats x))))
             (equal (aig-list->s x (cons (cons n v) env))
                    (aig-list->s x env)))
    :hints(("Goal" :in-theory (enable aig-list->s nat-bool-listp
                                      nat-bool-list-nats
                                      gl::scdr
                                      gl::s-endp)
            :induct (aig-list->s x env)
            :expand ((:Free (env) (aig-list->s x env))))))

  (local (defthm equal-nfix-plus-1
           (not (equal x (+ 1 (nfix x))))
           :hints(("Goal" :in-theory (enable nfix)))))

  (defthm eval-4vmask-to-a4vec-rec-with-env
    (b* (((mv uppera lowera)
          (4vmask-to-a4vec-rec mask boolmask nextvar))
         (env
          (4vmask-to-a4vec-rec-env mask boolmask upper lower nextvar)))
      (and (equal (logand (nfix mask) (aig-list->s uppera env))
                  (logand (nfix mask) upper))
           (implies (eql 0 (logand boolmask (logxor upper lower)))
                    (equal (logand (nfix mask) (aig-list->s lowera env))
                           (logand (nfix mask) lower)))))
    :hints(("Goal" :in-theory (enable 4vmask-to-a4vec-rec
                                      bitops::logand**
                                      bitops::logxor**
                                      bitops::logbitp**
                                      4vmask-fix)
            :induct (4vmask-to-a4vec-rec-env mask boolmask upper lower nextvar))
           (and stable-under-simplificationp
                '(:in-theory (enable bitops::b-xor))))))

(define 4vec-boolmaskp ((x 4vec-p) (mask integerp))
  (b* (((4vec x) x))
    (eql 0 (logand mask (logxor x.upper x.lower)))))

(define 4vmask-to-a4vec ((mask natp) (boolmask integerp) (nextvar natp))
  :returns (res nat-bool-a4vec-p!
                :hints(("Goal" :in-theory (enable nat-bool-a4vec-p))))
  :prepwork ((local (defthm true-listp-when-nat-bool-listp
                      (implies (nat-bool-listp x)
                               (true-listp x))
                      :hints(("Goal" :in-theory (enable nat-bool-listp))))))
  (b* (((mv upper lower)
        (4vmask-to-a4vec-rec mask boolmask nextvar)))
    (a4vec upper lower))
  ///
  ;; (defthm 4vmask-to-a4vec-nextvar
  ;;   (equal (mv-nth 1 (4vmask-to-a4vec mask nextvar))
  ;;          (+ (* 2 (logcount (nfix mask))) (nfix nextvar))))

  (defthm member-vars-of-4vmask-to-a4vec
    (iff (member v (nat-bool-a4vec-vars (4vmask-to-a4vec mask boolmask nextvar)))
         (and (natp v)
              (<= (nfix nextvar) v)
              (< v (+ (nfix nextvar) (4vmask-to-a4vec-varcount mask boolmask)))))
    :hints (("goal" :use member-4vmask-to-a4vec-rec-vars
             :in-theory (enable nat-bool-a4vec-vars))))

  (defthm 4vmask-to-a4vec-lower-bounds
    (nat-bool-a4vec-lower-boundp nextvar (4vmask-to-a4vec mask boolmask nextvar))
    :hints(("Goal" :in-theory (enable nat-bool-a4vec-lower-boundp)))
    :rule-classes ((:forward-chaining :trigger-terms ((4vmask-to-a4vec mask boolmask nextvar)))))

  (defthm 4vmask-to-a4vec-upper-bounds
    (nat-bool-a4vec-upper-boundp (+ (nfix nextvar) (4vmask-to-a4vec-varcount mask boolmask))
                                 (4vmask-to-a4vec mask boolmask nextvar))
    :hints(("Goal" :in-theory (enable nat-bool-a4vec-upper-boundp)))
    :rule-classes ((:forward-chaining :trigger-terms ((4vmask-to-a4vec mask boolmask nextvar))))))


(define 4vmask-to-a4vec-env ((mask natp) (boolmask integerp) (val 4vec-p) (nextvar natp))
  :returns env
  :prepwork ((local (defthm true-listp-when-nat-bool-listp
                      (implies (nat-bool-listp x)
                               (true-listp x))
                      :hints(("Goal" :in-theory (enable nat-bool-listp))))))
  (4vmask-to-a4vec-rec-env mask boolmask (4vec->upper val) (4vec->lower val) nextvar)
  ///

  (defthm key-exists-in-4vmask-to-a4vec-env
    (iff (hons-assoc-equal v (4vmask-to-a4vec-env mask boolmask val nextvar))
         (and (natp v)
              (<= (nfix nextvar) v)
              (< v (+ (nfix nextvar) (4vmask-to-a4vec-varcount mask boolmask))))))

  (local (defthm mask-lemma
           (IMPLIES
            (AND
             (EQUAL (LOGAND mask a)
                    (LOGAND b mask)))
            (EQUAL (LOGIOR b (lognot mask))
                   (LOGIOR (LOGNOT mask) a)))
     :hints ((bitops::logbitp-reasoning))))

  (defthm eval-4vmask-to-a4vec-with-env
    (b* ((vala (4vmask-to-a4vec mask boolmask nextvar))
         (env (4vmask-to-a4vec-env mask boolmask val nextvar)))
      (implies (4vec-boolmaskp val boolmask)
               (equal (4vec-mask (nfix mask) (a4vec-eval vala env))
                      (4vec-mask (nfix mask) val))))
    :hints(("Goal" :in-theory (e/d (4vmask-to-a4vec
                                    4vec-boolmaskp
                                      4vmask-fix
                                      4vec-mask
                                      a4vec-eval)
                                   (eval-4vmask-to-a4vec-rec-with-env))
            :use ((:instance eval-4vmask-to-a4vec-rec-with-env
                   (upper (4vec->upper val))
                   (lower (4vec->lower val)))))
           (bitops::logbitp-reasoning)
           ;; (and stable-under-simplificationp
           ;;      '(:bdd (:vars nil)))
           ))

  (defthm eval-4vmask-to-a4vec-with-env-mask-natp
    (b* ((vala (4vmask-to-a4vec mask boolmask nextvar))
         (env (4vmask-to-a4vec-env mask boolmask val nextvar)))
      (implies (and (natp mask)
                    (4vec-boolmaskp val boolmask))
               (equal (4vec-mask mask (a4vec-eval vala env))
                      (4vec-mask mask val))))
    :hints (("Goal" :use eval-4vmask-to-a4vec-with-env
             :in-theory (disable eval-4vmask-to-a4vec-with-env)))))




(local (in-theory (disable PICK-A-POINT-SUBSET-STRATEGY)))



(define svex-maskbits-for-vars ((vars svarlist-p)
                                (masks svex-mask-alist-p)
                                (boolmasks svar-boolmasks-p))
  :prepwork ((local (in-theory (enable svarlist-p svarlist-fix))))
  :returns (incr natp :rule-classes :type-prescription)
  (b* (((when (atom vars)) 0)
       (mask (svex-mask-lookup (svex-var (car vars)) masks))
       (boolmask (svar-boolmasks-lookup (car vars) boolmasks))
       ((when (< mask 0)) 0))
    (+ (4vmask-to-a4vec-varcount mask boolmask)
       (svex-maskbits-for-vars (cdr vars) masks boolmasks))))

(define svex-maskbits-ok ((vars svarlist-p)
                          (masks svex-mask-alist-p))
  :prepwork ((local (in-theory (enable svarlist-p svarlist-fix))))
  (b* (((when (atom vars)) t)
       (mask (svex-mask-lookup (svex-var (car vars)) masks))
       ((when (< mask 0)) nil))
    (svex-maskbits-ok (cdr vars) masks)))

(define svex-varmasks->a4env-rec ((vars svarlist-p)
                                  (masks svex-mask-alist-p)
                                  (boolmasks svar-boolmasks-p)
                                  (nextvar natp)
                                  (acc nat-bool-a4env-p!))
  :prepwork ((local (in-theory (enable svarlist-p svarlist-fix
                                       nat-bool-a4env-p
                                       svex-mask-alist-fix
                                       svex-a4vec-env-fix))))
  :returns (mv (err "some mask was negative"
                    (iff err (not (svex-maskbits-ok vars masks)))
                    :hints(("Goal" :in-theory (enable svex-maskbits-ok))))
               (a4env nat-bool-a4env-p! :hyp (nat-bool-a4env-p! acc))
               (nextvar1 (equal nextvar1 (+ (nfix nextvar)
                                            (svex-maskbits-for-vars vars masks boolmasks)))

                         :hints(("Goal" :in-theory (enable svex-maskbits-for-vars)))))
  (b* ((acc (svex-a4vec-env-fix acc))
       ((when (atom vars))
        (mv nil acc (lnfix nextvar)))
       (mask (svex-mask-lookup (svex-var (car vars)) masks))
       ((when (< mask 0))
        (mv (msg "Negative mask: ~x0~%" (svar-fix (car vars))) acc (lnfix nextvar)))
       (boolmask (svar-boolmasks-lookup (car vars) boolmasks))
       (a4vec (4vmask-to-a4vec mask boolmask nextvar))
       (nextvar (+ (lnfix nextvar)
                   (4vmask-to-a4vec-varcount mask boolmask))))
    (svex-varmasks->a4env-rec
     (cdr vars) masks boolmasks nextvar (cons (cons (svar-fix (car vars)) a4vec)
                                    acc)))
  ///

  (local (defun-nx svex-varmasks->a4env-rec-accumulator-elim-ind
           (vars masks boolmasks nextvar acc)
           (b* ((acc (svex-a4vec-env-fix acc))
                ((when (atom vars))
                 (list nil acc (lnfix nextvar)))
                (mask (svex-mask-lookup (svex-var (car vars)) masks))
                ((when (< mask 0))
                 (list (msg "Negative mask: ~x0~%" (svar-fix (car vars))) acc (lnfix nextvar)))
                (boolmask (svar-boolmasks-lookup (car vars) boolmasks))
                (a4vec (4vmask-to-a4vec mask boolmask nextvar))
                (nextvar (+ (lnfix nextvar)
                            (4vmask-to-a4vec-varcount mask boolmask))))
             (list 
              (svex-varmasks->a4env-rec-accumulator-elim-ind
               (cdr vars) masks boolmasks nextvar (cons (cons (svar-fix (car vars)) a4vec)
                                                        acc))
              (svex-varmasks->a4env-rec-accumulator-elim-ind
               (cdr vars) masks boolmasks nextvar (cons (cons (svar-fix (car vars)) a4vec) nil))))))

  (defthmd svex-varmasks->a4env-rec-accumulator-elim
    (implies (syntaxp (not (equal acc ''nil)))
             (b* (((mv err1 a4env1 nextvar1)
                   (svex-varmasks->a4env-rec vars masks boolmasks nextvar acc))
                  ((mv err2 a4env2 nextvar2)
                   (svex-varmasks->a4env-rec vars masks boolmasks nextvar nil)))
               (and (equal err1 err2)
                    (equal a4env1 (append a4env2 (svex-a4vec-env-fix acc)))
                    (equal nextvar1 nextvar2))))
    :hints (("goal" :induct (svex-varmasks->a4env-rec-accumulator-elim-ind
                             vars masks boolmasks nextvar acc)
             :expand ((:free (acc) (svex-varmasks->a4env-rec vars masks boolmasks nextvar acc))))))

  (defthm member-vars-of-svex-varmasks->a4env-rec
    (iff (member v (nat-bool-a4env-vars
                    (mv-nth 1 (svex-varmasks->a4env-rec vars masks boolmasks nextvar acc))))
         (or (member v (nat-bool-a4env-vars acc))
             (and (natp v)
                  (<= (nfix nextvar) v)
                  (< v (+ (nfix nextvar) (svex-maskbits-for-vars vars masks boolmasks))))))
    :hints(("Goal" :in-theory (enable svex-maskbits-for-vars
                                      nat-bool-a4env-vars))))

  (defthm svex-varmasks->a4env-rec-lower-bounds
    (implies (and (nat-bool-a4env-lower-boundp bound acc)
                  (<= (nfix bound) (nfix nextvar)))
             (nat-bool-a4env-lower-boundp bound (mv-nth 1 (svex-varmasks->a4env-rec vars masks boolmasks nextvar acc))))
    :hints(("Goal" :in-theory (enable nat-bool-a4env-lower-boundp))))

  (defthm svex-varmasks->a4env-rec-upper-bounds
    (implies (and (nat-bool-a4env-upper-boundp bound acc)
                  (<= (+ (nfix nextvar) (svex-maskbits-for-vars vars masks boolmasks))
                      (nfix bound)))
             (nat-bool-a4env-upper-boundp bound (mv-nth 1 (svex-varmasks->a4env-rec vars masks boolmasks nextvar acc))))
    :hints(("Goal" :in-theory (enable nat-bool-a4env-upper-boundp
                                      svex-maskbits-for-vars))))

  (defret key-exists-in-svex-varmasks->a4env-rec
    (implies (not err)
             (iff (hons-assoc-equal v a4env)
                  (or (member v (svarlist-fix vars))
                      (hons-assoc-equal v (svex-a4vec-env-fix acc))))))

  (defret alist-keys-of-svex-varmasks->a4env-rec
    (implies (not err)
             (equal (alist-keys a4env)
                    (append (rev (svarlist-fix vars))
                            (alist-keys (svex-a4vec-env-fix acc)))))
    :hints(("Goal" :in-theory (enable svex-maskbits-ok)))))

(defsection svex-envs-masks-partly-equiv

  (defquant svex-envs-masks-partly-equiv (vars masks env1 env2)
    (forall v
            (implies (not (member (svar-fix v) (svarlist-fix vars)))
                     (equal (4vec-mask (svex-mask-lookup (svex-var v) masks)
                                       (svex-env-lookup v env1))
                            (4vec-mask (svex-mask-lookup (svex-var v) masks)
                                       (svex-env-lookup v env2)))))
    :rewrite :direct)

  (defexample svex-envs-masks-partly-equiv-example
    :pattern (equal (4vec-mask (svex-mask-lookup (svex-var v) masks)
                               val1)
                    (4vec-mask mask2 val2))
    :templates (v)
    :instance-rulename svex-envs-masks-partly-equiv-instancing))


(defsection svex-envs-mask-equiv-on-vars

  (defquant svex-envs-mask-equiv-on-vars (vars masks env1 env2)
    (forall v
            (implies (member (svar-fix v) (svarlist-fix vars))
                     (equal (4vec-mask (svex-mask-lookup (svex-var v) masks)
                                       (svex-env-lookup v env1))
                            (4vec-mask (svex-mask-lookup (svex-var v) masks)
                                       (svex-env-lookup v env2)))))
    :rewrite :direct)

  (defexample svex-envs-mask-equiv-on-vars-example
    :pattern (equal (4vec-mask (svex-mask-lookup (svex-var v) masks)
                               val1)
                    (4vec-mask mask2 val2))
    :templates (v)
    :instance-rulename svex-envs-mask-equiv-on-vars-instancing)
  
  (local (defexample svex-envs-mask-equiv-on-vars-mask-look-example
           :pattern (svex-mask-lookup (svex-var var) masks)
           :templates (var)
           :instance-rulename svex-envs-mask-equiv-on-vars-instancing))

  (local (defexample svex-envs-mask-equiv-on-vars-env-look-example
           :pattern (svex-env-lookup var env)
           :templates (var)
           :instance-rulename svex-envs-mask-equiv-on-vars-instancing))

  (local (defexample svex-argmasks-okp-example
           :pattern (equal (4vec-mask mask (svex-apply fn (svexlist-eval args env1)))
                           (4vec-mask mask (svex-apply fn (svexlist-eval args env2))))
           :templates (env1 (svexlist-eval args env2))
           :instance-rulename svex-argmasks-okp-instancing))

  (local (acl2::def-witness-ruleset svex-mask-alist-reasoning
           '(svex-mask-alist-complete-witnessing
             svex-mask-alist-complete-instancing
             svex-mask-alist-complete-example
             svex-mask-alist-partly-complete-witnessing
             svex-mask-alist-partly-complete-instancing
             svex-mask-alist-partly-complete-example)))

  (local (acl2::def-witness-ruleset svex-env-reasoning
           '(svex-envs-mask-equiv-on-vars-instancing
             svex-envs-mask-equiv-on-vars-witnessing
             svex-envs-mask-equiv-on-vars-mask-look-example
             svex-envs-mask-equiv-on-vars-env-look-example
             svex-mask-alist-reasoning
             SVEX-ARGMASKS-OKP-WITNESSING
             SVEX-ARGMASKS-OKP-INSTANCING
             SVEX-ARGMASKS-OKP-EXAMPLE
             )))


  (defthm-svex-eval-flag
    (defthm svex-eval-of-mask-equiv-on-vars-envs
      (implies (and (svex-mask-alist-complete masks)
                    (svex-envs-mask-equiv-on-vars vars masks env1 env2)
                    (subsetp-equal (intersection-equal (svex-vars x)
                                                       (union-equal (alist-keys (svex-env-fix env1))
                                                                    (alist-keys (svex-env-fix env2))))
                                   (svarlist-fix vars)))
               (equal (equal (4vec-mask (svex-mask-lookup x masks)
                                        (svex-eval x env1))
                             (4vec-mask (svex-mask-lookup x masks)
                                        (svex-eval x env2)))
                      t))
      :hints ('(:expand ((:free (env) (svex-eval x env))
                         (svex-vars x))
                :do-not-induct t)
              (witness :ruleset svex-env-reasoning)
              (witness :ruleset svex-env-reasoning)
              (set-reasoning)
              (and stable-under-simplificationp
                   '(:in-theory (enable svex-env-lookup)))
              ;; (and stable-under-simplificationp
              ;;      '(:use ((:instance svex-argmasks-okp-necc
              ;;               (mask (svex-mask-lookup x masks))
              ;;               (argmasks (svex-argmasks-lookup
              ;;                          (svex-call->args x) masks))
              ;;               (env env1)
              ;;               (vals (svexlist-eval (svex-call->args x) env2))))))
              )
      :flag expr)
    (defthm svexlist-eval-of-mask-equiv-on-vars-envs
      (implies (and (svex-mask-alist-complete masks)
                    (svex-envs-mask-equiv-on-vars vars masks env1 env2)
                    (subsetp-equal (intersection-equal (svexlist-vars x)
                                                       (union-equal (alist-keys (svex-env-fix env1))
                                                                    (alist-keys (svex-env-fix env2))))
                                   (svarlist-fix vars)))
               (equal (equal (4veclist-mask (svex-argmasks-lookup x masks)
                                            (svexlist-eval x env1))
                             (4veclist-mask (svex-argmasks-lookup x masks)
                                            (svexlist-eval x env2)))
                      t))
      :hints ('(:expand ((:free (env) (svexlist-eval x env))
                         (svexlist-vars x)
                         (svex-argmasks-lookup x masks))))
      :flag list))

  )

(local (defthm hons-assoc-equal-of-append
         (equal (hons-assoc-equal k (append a b))
                (or (hons-assoc-equal k a)
                    (hons-assoc-equal k b)))))

(defthm aig-list->s-of-append-when-first-superset
  (implies (and (nat-bool-listp x)
                (subsetp (nat-bool-list-nats x)
                         (alist-keys env)))
           (equal (aig-list->s x (append env rest))
                  (aig-list->s x env)))
  :hints(("Goal" :in-theory (e/d* (aig-list->s
                                   nat-bool-listp
                                   nat-bool-list-nats
                                   gl::scdr
                                   gl::s-endp)
                                 ((:rules-of-class :type-prescription :here)))
          :induct (aig-list->s x env)
          :expand ((:free (env) (aig-list->s x env))))))

(defthm a4vec-eval-of-append-when-first-superset
  (implies (and (nat-bool-a4vec-p x)
                (subsetp (nat-bool-a4vec-vars x)
                         (alist-keys env)))
           (equal (a4vec-eval x (append env rest))
                  (a4vec-eval x env)))
  :hints(("Goal" :in-theory (enable a4vec-eval
                                    nat-bool-a4vec-p
                                    nat-bool-a4vec-vars))))

(defthm aig-list->s-of-append-when-first-not-intersect
  (implies (and (nat-bool-listp x)
                (not (intersectp (nat-bool-list-nats x)
                                 (alist-keys prev))))
           (equal (aig-list->s x (append prev env))
                  (aig-list->s x env)))
  :hints(("Goal" :in-theory (e/d* (aig-list->s
                                   nat-bool-listp
                                   nat-bool-list-nats
                                   gl::scdr
                                   gl::s-endp)
                                 ((:rules-of-class :type-prescription :here)))
          :induct (aig-list->s x env)
          :expand ((:free (env) (aig-list->s x env))))))


(defthm a4vec-eval-of-append-when-first-not-intersect
  (implies (and (nat-bool-a4vec-p x)
                (not (intersectp (nat-bool-a4vec-vars x)
                                 (alist-keys prev))))
           (equal (a4vec-eval x (append prev env))
                  (a4vec-eval x env)))
  :hints(("Goal" :in-theory (enable a4vec-eval
                                    nat-bool-a4vec-p
                                    nat-bool-a4vec-vars))))

(defthm nat-bool-a4vec-p-of-nat-bool-a4env
  (implies (and (nat-bool-a4env-p x)
                (hons-assoc-equal k x))
           (nat-bool-a4vec-p (cdr (hons-assoc-equal k x))))
  :hints(("Goal" :in-theory (enable nat-bool-a4env-p))))


;; (define svex-env-lookup-nofix (x env)
;;   (or (cdr (hons-get x env)) (4vec-x))
;;   ///
;;   (defthm svex-env-lookup-nofix-when-right-types
;;     (implies (and (svar-p x)
;;                   (svex-env-p env))
;;              (equal (svex-env-lookup-nofix x env)
;;                     (svex-env-lookup x env)))
;;     :hints(("Goal" :in-theory (enable svex-env-lookup
;;                                       svex-env-p))))

;;   ;; (local (defthm lookup-in-svex-env-fix-when-svar-p
;;   ;;          (implies (and (svar-p x)
;;   ;;                        (not (cdr (hons-assoc-equal x env))))
;;   ;;                   (4vec-equiv (cdr (hons-assoc-equal x (svex-env-fix env)))
;;   ;;                               (4vec-x)))
;;   ;;          :hints(("Goal" :in-theory (enable svex-env-fix)))))

;;   (defthm svex-env-lookup-nofix-when-right-types-weak
;;     (implies (svar-p x)
;;              (4vec-equiv (svex-env-lookup-nofix x env)
;;                          (svex-env-lookup x env)))
;;     :hints(("Goal" :in-theory (enable svex-env-lookup svex-env-fix)))))







(define svex-mask-alist-extract-vars ((x svex-mask-alist-p))
  :returns (new-x svex-mask-alist-p)
  :hooks nil
  (if (atom x)
      nil
    (if (and (mbt (svex-p (caar x)))
             (eq (svex-kind (caar x)) :var))
        (hons-acons (caar x) (4vmask-fix (cdar x))
                    (svex-mask-alist-extract-vars (cdr x)))
      (svex-mask-alist-extract-vars (cdr x))))
  ///
  (defret lookup-in-svex-mask-alist-extract-vars
    (equal (hons-assoc-equal s new-x)
           (and (svex-p s)
                (equal (svex-kind s) :var)
                (hons-assoc-equal s (svex-mask-alist-fix x)))))

  (local (defthm assoc-in-svex-mask-alist-p
           (implies (svex-mask-alist-p x)
                    (equal (assoc k x)
                           (hons-assoc-equal k x)))
           :hints(("Goal" :in-theory (enable svex-mask-alist-p)))))

  (defret svex-mask-lookup-in-svex-mask-alist-extract-vars
    (equal (svex-mask-lookup s new-x)
           (if (equal (svex-kind s) :var)
               (svex-mask-lookup s x)
             0))
    :hints(("Goal" :in-theory (enable svex-mask-lookup))))

  (defret svex-maskbits-ok-of-svex-mask-alist-extract-vars
    (iff (svex-maskbits-ok vars new-x)
         (svex-maskbits-ok vars x))
    :hints(("Goal" :in-theory (enable svex-maskbits-ok)
            :induct (svex-maskbits-ok vars x)
            :do-not-induct t)))

  (defret svex-maskbits-for-vars-of-svex-mask-alist-extract-vars
    (equal (svex-maskbits-for-vars vars new-x boolmasks)
           (svex-maskbits-for-vars vars x boolmasks))
    :hints(("Goal" :in-theory (e/d (svex-maskbits-for-vars)
                                   (svex-mask-alist-extract-vars)))))

  (deffixequiv svex-mask-alist-extract-vars :hints(("Goal" :in-theory (enable svex-mask-alist-fix)))))



(define svex-varmasks/env->aig-env-rec ((vars svarlist-p)
                                        (masks svex-mask-alist-p)
                                        (boolmasks svar-boolmasks-p)
                                        (env svex-env-p "look up variables in env to get 4vecs to assign -- symbolic")
                                        (nextvar natp)
                                        (acc "aig environment accumulator"))
  :prepwork ((local (in-theory (enable svarlist-p svarlist-fix))))
  :returns (mv (err "some mask was negative"
                    (implies (svex-mask-alist-p masks)
                             (iff err (not (svex-maskbits-ok vars masks))))
                    :hints(("Goal" :in-theory (enable svex-maskbits-ok))))
               (env) ;; binds AIG vars to Boolean values
               (nextvar1
                (implies (and (svex-mask-alist-p masks)
                              (svar-boolmasks-p boolmasks))
                         (equal nextvar1
                                (+ (nfix nextvar)
                                   (svex-maskbits-for-vars vars masks boolmasks))))
                :hints(("Goal" :in-theory (enable svex-maskbits-for-vars)))))
  ;; :hooks ((:fix :args (vars nextvar)))
  (b* (((when (atom vars))
        (mv nil acc (lnfix nextvar)))
       (mask (svex-mask-lookup (svex-var (car vars)) masks))
       ((when (< mask 0))
        (mv (msg "Negative mask: ~x0~%" (svar-fix (car vars)))
            acc (lnfix nextvar)))
       (boolmask (svar-boolmasks-lookup (car vars) boolmasks))
       (4vec (4vec-fix (svex-env-lookup (svar-fix (car vars)) env)))
       (env-part
        (4vmask-to-a4vec-env mask boolmask 4vec nextvar))
       (nextvar (+ (lnfix nextvar)
                   (4vmask-to-a4vec-varcount mask boolmask))))
    (svex-varmasks/env->aig-env-rec
     (cdr vars) masks boolmasks env nextvar (append env-part acc)))
  ///

  (defthm key-exists-in-svex-varmasks/env->aig-env-rec
    (implies (and (svex-mask-alist-p masks)
                  (svar-boolmasks-p boolmasks))
             (iff (hons-assoc-equal v (mv-nth 1 (svex-varmasks/env->aig-env-rec
                                                 vars masks boolmasks env nextvar acc)))
                  (or (hons-assoc-equal v acc)
                      (and (natp v)
                           (<= (nfix nextvar) v)
                           (< v (+ (nfix nextvar) (svex-maskbits-for-vars vars masks boolmasks)))))))
    :hints(("Goal" :in-theory (enable svex-maskbits-for-vars))))

  (local (defun svex-varmasks/env->aig-env-accumulator-elim-ind
           (vars masks boolmasks env nextvar acc)
           (b* (((when (atom vars))
                 (list nil acc (lnfix nextvar)))
                (mask (svex-mask-lookup (svex-var (car vars)) masks))
                ((when (< mask 0))
                 (list (msg "Negative mask: ~x0~%" (svar-fix (car vars)))
                       acc (lnfix nextvar)))
                (boolmask (svar-boolmasks-lookup (car vars) boolmasks))
                (4vec (svex-env-lookup (svar-fix (car vars)) env))
                (env-part
                 (4vmask-to-a4vec-env mask boolmask 4vec nextvar))
                (nextvar (+ (lnfix nextvar)
                            (4vmask-to-a4vec-varcount mask boolmask))))
             (list (svex-varmasks/env->aig-env-accumulator-elim-ind
                    (cdr vars) masks boolmasks env nextvar (append env-part acc))
                   (svex-varmasks/env->aig-env-accumulator-elim-ind
                    (cdr vars) masks boolmasks env nextvar env-part)))))
           
  (local (defthm hide-not
           (equal (hide (not x)) (not (hide x)))
           :hints (("goal" :expand ((:free (x) (hide x)))))))

  (defthm svex-varmasks/env->aig-env-accumulator-elim
    (implies (syntaxp (not (equal acc ''nil)))
             (equal (mv-nth 1 (svex-varmasks/env->aig-env-rec
                               vars masks boolmasks env nextvar acc))
                    (append (mv-nth 1 (svex-varmasks/env->aig-env-rec
                                       vars masks boolmasks env nextvar nil))
                            acc)))
    :hints (("goal" :induct (svex-varmasks/env->aig-env-accumulator-elim-ind
                             vars masks boolmasks env nextvar acc)
             :expand ((:free (acc) (svex-varmasks/env->aig-env-rec
                                       vars masks boolmasks env nextvar acc))))))

  (local
   (defun svex-varmasks->a4env-rec-induct (vars masks boolmasks nextvar a4acc goalenv envacc)
     (declare (ignorable vars masks boolmasks nextvar a4acc goalenv envacc))
     (b* (((when (atom vars)) nil)
          (mask (svex-mask-lookup (svex-var (car vars)) masks))
          ((when (< mask 0)) nil)
          (boolmask (svar-boolmasks-lookup (car vars) boolmasks))
          (4vec (svex-env-lookup (car vars) goalenv))
          (env-part
           (4vmask-to-a4vec-env mask boolmask 4vec nextvar))
          (a4vec (4vmask-to-a4vec mask boolmask nextvar))
          (nextvar (+ (lnfix nextvar)
                      (4vmask-to-a4vec-varcount mask boolmask))))
       (svex-varmasks->a4env-rec-induct
        (cdr vars) masks boolmasks nextvar
        (cons (cons (svar-fix (car vars)) a4vec)
              a4acc)
        goalenv (append env-part envacc)))))

  (defthm 4vmask-to-a4vec-vars-subset-of-keys
    (SUBSETP-EQUAL
     (NAT-BOOL-A4VEC-VARS
      (4vmask-to-a4vec mask boolmask nextvar))
     (ALIST-KEYS
      (4vmask-to-a4vec-env mask boolmask val nextvar)))
    :hints ((acl2::set-reasoning)))

  (defthm member-nat-bool-a4vec-vars-of-lookup-when-upper-bounded
    (implies (and (nat-bool-a4env-p a4acc)
                  (nat-bool-a4env-upper-boundp nextvar a4acc)
                  (<= (nfix nextvar) k))
             (not (member k (nat-bool-a4vec-vars (cdr (hons-assoc-equal v a4acc))))))
    :hints(("Goal" :in-theory (enable nat-bool-a4env-p
                                      nat-bool-a4env-upper-boundp))))

  (defthm 4vmask-to-a4vec-env-vars-not-intersect-when-upper-bounded
    (implies (and (nat-bool-a4env-p a4acc)
                  (double-rewrite (nat-bool-a4env-upper-boundp nextvar a4acc)))
             (not (intersectp (nat-bool-a4vec-vars
                               (cdr (hons-assoc-equal v a4acc)))
                              (alist-keys (4vmask-to-a4vec-env mask boolmask val nextvar)))))
    :hints ((acl2::set-reasoning)))

  (local (defthm 4vmask-to-a4vec-vars-not-intersect-svex-varmsks/env->aig-env-rec-keys
           (implies (and (svex-mask-alist-p masks)
                         (svar-boolmasks-p boolmasks))
                    (NOT
                     (INTERSECTP-EQUAL
                      (NAT-BOOL-A4VEC-VARS
                       (4VMASK-TO-A4VEC mask
                                        boolmask
                                        NEXTVAR))
                      (ALIST-KEYS
                       (MV-NTH
                        1
                        (SVEX-VARMASKS/ENV->AIG-ENV-REC
                         vars masks BOOLMASKS GOALENV
                         (+ (NFIX NEXTVAR)
                            (4VMASK-TO-A4VEC-VARCOUNT mask boolmask))
                         NIL))))))
           :hints ((set-reasoning))))

  (local (defthm not-member-a4vec-vars-lookup-when-not-member-a4env-vars
           (implies (not (member v (nat-bool-a4env-vars a4env)))
                    (not (member v (nat-bool-a4vec-vars (cdr (hons-assoc-equal v0 a4env))))))
           :hints(("Goal" :in-theory (enable nat-bool-a4env-vars)))))

  (local (defthm 4vmask-to-a4vec-vars-subset-svex-varmsks/env->aig-env-rec-keys-2
           (implies (and (svex-mask-alist-p masks)
                         (svar-boolmasks-p boolmasks))
                    (SUBSETP-EQUAL
                     (NAT-BOOL-A4VEC-VARS
                      (CDR
                       (HONS-ASSOC-EQUAL
                        v0
                        (MV-NTH
                         1
                         (SVEX-VARMASKS->A4ENV-REC
                          vars
                          MASKS BOOLMASKS
                          nextvar
                          NIL)))))
                     (ALIST-KEYS
                      (MV-NTH
                       1
                       (SVEX-VARMASKS/ENV->AIG-ENV-REC
                        vars
                        (SVEX-MASK-ALIST-EXTRACT-VARS MASKS)
                        BOOLMASKS GOALENV
                        nextvar
                        NIL)))))
           :hints ((set-reasoning))))

  (acl2::defquant svex-env-boolmasks-ok (env boolmasks)
    (forall v
            (4vec-boolmaskp (svex-env-lookup v env)
                            (svar-boolmasks-lookup v boolmasks)))
    :rewrite :direct)

  (local (defthm svex-env-lookup-of-cons
           (equal (svex-env-lookup k (cons (cons k0 v0) rest))
                  (if (and (svar-p k0) (equal (svar-fix k) k0))
                      (4vec-fix v0)
                    (svex-env-lookup k rest)))
           :hints(("Goal" :in-theory (enable svex-env-lookup
                                             svex-env-fix)))))

  (local (in-theory (enable svex-env-boolmasks-ok-necc
                            svex-varmasks->a4env-rec-accumulator-elim)))

  (defthm eval-svex-varmasks->a4env-rec-with-env
    (b* (((mv err a4env ?nextvar1)
          ;; Assigns AIG variable numbers to each SVEX var. Ignores goalenv (does
          ;; not need to know anything about the values of the svex vars to do
          ;; this, just their caremasks/boolmasks).
          (svex-varmasks->a4env-rec vars masks boolmasks nextvar a4acc))
         ((mv ?err1 env ?nextvar1)
          ;; Binds AIG variable numbers to (symbolic) bits extracted from the goalenv.
          (svex-varmasks/env->aig-env-rec
           vars (svex-mask-alist-extract-vars masks) boolmasks goalenv nextvar envacc)))
      (implies (and (not err)
                    (nat-bool-a4env-p a4acc)
                    (nat-bool-a4env-upper-boundp nextvar a4acc)
                    (svex-mask-alist-p masks)
                    (svar-boolmasks-p boolmasks)
                    (svex-env-boolmasks-ok goalenv boolmasks)
                    ;; (svex-env-p goalenv)
                    ;; (svex-envs-masks-partly-equiv
                    ;;  vars masks
                    ;;  (svex-a4vec-env-eval a4acc envacc)
                    ;;  goalenv)
                    ;; (subsetp (alist-keys (svex-env-fix goalenv))
                    ;;          (append (svarlist-fix vars)
                    ;;                  (alist-keys (svex-a4vec-env-fix a4acc))))
                    )
               (svex-envs-mask-equiv-on-vars
                vars masks
                (svex-a4vec-env-eval a4env env)
                goalenv)))
    :hints(("Goal" :in-theory (enable svex-varmasks->a4env-rec
                                      svarlist-fix
                                      svex-maskbits-ok
                                      nat-bool-a4env-p
                                      nat-bool-a4env-upper-boundp
                                      svex-a4vec-env-fix
                                      svex-a4vec-env-eval
                                      alist-keys
                                      svex-alist-keys)
            :induct (svex-varmasks->a4env-rec-induct
                     vars masks boolmasks nextvar a4acc goalenv envacc)
            :expand (svex-varmasks/env->aig-env-rec
                     vars masks boolmasks goalenv nextvar envacc))
           (and stable-under-simplificationp
                (cond ((assoc 'subsetp-equal clause) ;; has a (not (subsetp-equal... lit
                       (acl2::set-reasoning))
                      ;; ((assoc 'svex-envs-masks-partly-equiv clause)
                      ;;  '(:computed-hint-replacement
                      ;;    ((acl2::witness :ruleset (svex-envs-masks-partly-equiv-witnessing))
                      ;;     (acl2::witness :ruleset (svex-envs-masks-partly-equiv-example)))
                      ;;    :in-theory (enable ;; svex-env-lookup
                      ;;                       svex-env-fix)))
                      ;; (t '(:computed-hint-replacement
                      ;;      ((acl2::witness :ruleset (svex-envs-mask-equiv-witnessing))
                      ;;       (acl2::witness :ruleset (svex-envs-masks-partly-equiv-example)))
                      ;;      :no-op t))
                      (t '(:computed-hint-replacement
                           ((acl2::witness :ruleset (svex-envs-mask-equiv-on-vars-witnessing))
                            (acl2::witness :ruleset (svex-envs-mask-equiv-on-vars-example)))
                           :no-op t))
                      )))))


(define svex-varmasks->a4env ((vars svarlist-p)
                              (masks svex-mask-alist-p)
                              (boolmasks svar-boolmasks-p))
  :returns (mv (err "some mask was negative"
                    (iff err (not (svex-maskbits-ok vars masks))))
               (a4env nat-bool-a4env-p!))
  (b* (((mv err res &)
        (svex-varmasks->a4env-rec vars masks boolmasks 0 nil)))
    (mv err res))
  ///
  (defret alist-keys-of-svex-varmasks->a4env
    (implies (not err)
             (equal (alist-keys a4env)
                    (rev (svarlist-fix vars))))))


(define svex-varmasks/env->aig-env ((vars svarlist-p)
                                    (masks svex-mask-alist-p)
                                    (boolmasks svar-boolmasks-p)
                                    (env svex-env-p "look up variables in env to get 4vecs to assign"))
  :returns (mv (err "some mask was negative"
                    (implies (svex-mask-alist-p masks)
                             (iff err (not (svex-maskbits-ok vars masks))))
                    :hints(("Goal" :in-theory (enable svex-maskbits-ok))))
               (env "binds AIG vars to Boolean values"))
    :hooks ((:fix :args (vars)))
  (b* (((mv err res &)
        (svex-varmasks/env->aig-env-rec vars masks boolmasks env 0 nil)))
    (mv err res))
  ///
  (defthm eval-svex-varmasks->a4env-with-env
    (b* (((mv err a4env)
          (svex-varmasks->a4env vars masks boolmasks))
         ((mv ?err1 env)
          (svex-varmasks/env->aig-env
           vars (svex-mask-alist-extract-vars masks) boolmasks goalenv)))
      (implies (and (not err)
                    (svex-mask-alist-p masks)
                    (svar-boolmasks-p boolmasks)
                    ;; (svex-env-p goalenv)
                    (svex-env-boolmasks-ok goalenv boolmasks))
               (svex-envs-mask-equiv-on-vars vars masks
                                             (svex-a4vec-env-eval a4env env)
                                             goalenv)))
    :hints (("goal" :use ((:instance eval-svex-varmasks->a4env-rec-with-env
                           (nextvar 0)
                           (a4acc nil)
                           (envacc nil)))
             :in-theory (e/d (svex-varmasks->a4env
                              svex-env-lookup
                              svex-lookup)
                             (eval-svex-varmasks->a4env-rec-with-env)))
            ;; (acl2::witness :ruleset (svex-envs-mask-equiv-on-vars))
            (acl2::set-reasoning))))

(define svex-env-check-boolmasks ((boolmasks svar-boolmasks-p)
                                  (env svex-env-p))
  :prepwork ((local (in-theory (enable svar-boolmasks-p svar-boolmasks-fix))))
  ;; :hooks nil
  (b* (((when (atom boolmasks)) t)
       ((unless (mbt (svar-p (caar boolmasks))))
        (svex-env-check-boolmasks (cdr boolmasks) env))
       ((cons var mask) (car boolmasks))
       (val (svex-env-lookup var env))
       (ok (4vec-boolmaskp val mask))
       (?ign (and (not ok)
                  (cw "not 4vec-boolmaskp: ~x0~%" var))))
    (and (svex-env-check-boolmasks (cdr boolmasks) env)
         ok))
  ///
  (acl2::defexample svex-env-boolmasks-ok-example
    :pattern (svex-env-lookup v env)
    :templates (v)
    :instance-rulename svex-env-boolmasks-ok-instancing)

  (defthm svex-env-check-boolmasks-correct
    (implies (and (svex-env-check-boolmasks boolmasks env)
                  ;; (svex-env-p env)
                  (svar-boolmasks-p boolmasks))
             (svex-env-boolmasks-ok env boolmasks))
    :hints (("goal" :induct (svex-env-check-boolmasks boolmasks env))
            (acl2::witness :ruleset (svex-env-boolmasks-ok-witnessing
                                     svex-env-boolmasks-ok-example))
            (and stable-under-simplificationp
                 '(:in-theory (enable svar-boolmasks-lookup)
                   :expand ((:free (x) (4vec-boolmaskp x 0))))))))

(define svexlist-mask-alist-memo ((x svexlist-p))
  :enabled t
  (svexlist-mask-alist x)
  ///
  (memoize 'svexlist-mask-alist-memo))

(define svexlist-vars-memo ((x svexlist-p))
  :enabled t
  (svexlist-collect-vars x)
  ///
  (memoize 'svexlist-vars-memo))


(define svexlist->a4vecs-for-varlist ((x svexlist-p)
                                      (vars svarlist-p)
                                      (boolmasks svar-boolmasks-p))
  :returns (mv (err (iff err (not (svex-maskbits-ok vars (svexlist-mask-alist x)))))
               (a4vecs a4veclist-p))
  :short "Creates a symbolic bit-level representation for x, assuming that vars
          are the only vars relevant to x and that the bits of vars given in boolmasks
          are Boolean-valued."
  :long "<p>Steps: First creates a symbolic environment mapping the variables
to a4vec structures, each bit of which is a free variable.  (For bits
constrained to be Boolean by boolmasks, the same variable is shared for
upper/lower.)  Then uses @('svexlist->a4vec-top') to generate a4vecs corresponding
to the svexes.</p>"

  (b* (;; (- (sneaky-push 'svexlist x))
       (masks (svexlist-mask-alist-memo x))
       ((mv err a4env) (svex-varmasks->a4env vars masks boolmasks))
       ((when err) (mv err nil))
       (a4env (make-fast-alist a4env))
       (res (svexlist->a4vec-top x a4env masks))
       (?ign (fast-alist-free a4env)))
    (mv nil res))
  ///
  (memoize 'svexlist->a4vecs-for-varlist))




(define svexlist-variable-mask-alist ((x svexlist-p))
  ;; We've seen problems in GL where we get a stack overflow in
  ;; gobject-hierarchy-lite traversing the full masks inside
  ;; svexlist->a4vec-aig-env-for-varlist.  But we don't need the full set of
  ;; masks there, only those for the variables.  So to work around this
  ;; problem, this function extracts only the variables from the mask alist,
  ;; producing a much smaller alist.
  :returns (varmasks svex-mask-alist-p)
  :enabled t
  (b* ((masks-full (svexlist-mask-alist-memo x)))
    (svex-mask-alist-extract-vars masks-full)))


(define svexlist->a4vec-aig-env-for-varlist ((x svexlist-p)
                                             (vars svarlist-p)
                                             (boolmasks svar-boolmasks-p)
                                             (env svex-env-p))
  :returns (mv (err (iff err (not (svex-maskbits-ok vars (svexlist-mask-alist x)))))
               (aig-env))
  :hooks ((:fix :args (x vars)))
  ;; We use svexlist-variable-mask-alist here rather than
  ;; svexlist-mask-alist-memo so that GL won't have to traverse the full mask
  ;; alist with gobject-hierarchy-lite, which we've seen cause stack overflows.
  (b* ((masks (svexlist-variable-mask-alist x)))
    (svex-varmasks/env->aig-env vars masks boolmasks env))
  ///
  ;; (local (defthm svex-envs-mask-equiv-lemma
  ;;          (iff (svex-envs-mask-equiv
  ;;                (svexlist-mask-alist x) y z)
  ;;               (svex-envs-mask-equiv
  ;;                (svexlist-variable-mask-alist x) y z))
  ;;          :hints ((witness))))


  (defthm svexlist->a4vec-for-varlist-correct
    (b* (((mv err a4vecs) (svexlist->a4vecs-for-varlist x vars boolmasks))
         ((mv ?err1 aig-env) (svexlist->a4vec-aig-env-for-varlist x vars boolmasks env)))
      (implies (and (not err)
                    ;; (svex-env-p env)
                    (svar-boolmasks-p boolmasks)
                    (svex-env-boolmasks-ok env boolmasks)
                    (subsetp (intersection-equal (svexlist-vars x)
                                                 (alist-keys (svex-env-fix env)))
                             (svarlist-fix vars)))
               (equal (a4veclist-eval a4vecs aig-env)
                      (svexlist-eval x env))))
    :hints(("Goal" :in-theory (e/d (svexlist->a4vecs-for-varlist)
                                   (svexlist-eval-of-mask-equiv-on-vars-envs
                                    ;; svexlist-eval-of-mask-equiv-envs
                                    ))
            ;; :use ((:instance svexlist-eval-of-mask-equiv-envs
            ;;        (masks (svexlist-mask-alist x))
            ;;        (env1 (SVEX-A4VEC-ENV-EVAL
            ;;               (MV-NTH 1
            ;;                       (SVEX-VARMASKS->A4ENV VARS (SVEXLIST-MASK-ALIST X) boolmasks))
            ;;               (MV-NTH 1
            ;;                       (SVEX-VARMASKS/ENV->AIG-ENV VARS (SVEXLIST-variable-MASK-ALIST X)
            ;;                                                   boolmasks ENV))))
            ;;        (env2 env)))
            :use ((:instance svexlist-eval-of-mask-equiv-on-vars-envs
                   (masks (svexlist-mask-alist x))
                   (env1 (SVEX-A4VEC-ENV-EVAL
                          (MV-NTH 1
                                  (SVEX-VARMASKS->A4ENV VARS (SVEXLIST-MASK-ALIST X) boolmasks))
                          (MV-NTH 1
                                  (SVEX-VARMASKS/ENV->AIG-ENV VARS (SVEXLIST-variable-MASK-ALIST X)
                                                              boolmasks ENV))))
                   (env2 env)))

            )
           (set-reasoning))
    :otf-flg t))


(local (defthm subset-of-mergesorts-is-subsetp
         (iff (subset (mergesort a) (mergesort b))
              (subsetp a b))
         :hints(("Goal" :in-theory (enable* set::definitions)))))


(define svexlist-rewrite-fixpoint-memo ((x svexlist-p))
  :enabled t
  (time$ (svexlist-rewrite-fixpoint x :verbosep t)
         :msg "; svex rewriting: ~st sec, ~sa bytes.~%")
  ///
  (memoize 'svexlist-rewrite-fixpoint-memo))

(define maybe-svexlist-rewrite-fixpoint ((x svexlist-p) (do-it))
  :returns (new-x svexlist-p)
  (if do-it
      (svexlist-rewrite-fixpoint-memo x)
    (hons-copy (svexlist-fix x)))
  ///
  (defret maybe-svexlist-rewrite-fixpoint-correct
    (equal (svexlist-eval new-x env)
           (svexlist-eval x env)))
  (defret maybe-svexlist-rewrite-fixpoint-len
    (equal (len new-x)
           (len x)))

  (defret vars-of-maybe-svexlist-rewrite-fixpoint
    (implies (not (member v (svexlist-vars x)))
             (not (member v (svexlist-vars new-x))))))


(local (defthm svarlist-p-of-alist-keys-when-svex-env-p
         (implies (svex-env-p env)
                  (svarlist-p (alist-keys env)))
         :hints(("Goal" :in-theory (enable svex-env-p svarlist-p alist-keys)))))


(define svexlist-vars-for-symbolic-eval ((x svexlist-p)
                                         (env svex-env-p)
                                         (symbolic-params alistp))
  :returns (vars svarlist-p :hyp :guard)
  :guard-hints (("goal" :in-theory (e/d (SET::UNION-WITH-SUBSET-LEFT
                                         double-containment
                                         set::subset-to-subsetp)
                                        (SUBSET-OF-MERGESORTS-IS-SUBSETP))))
  (b* ((allvars (assoc :allvars symbolic-params))
       (vars (if allvars
                 (svexlist-vars-memo x)
               (ec-call (svarlist-fix (cdr (assoc :vars symbolic-params))))))
       (svars (mbe :logic (set::mergesort vars)
                   :exec (if (set::setp vars) vars (set::mergesort vars))))
       ((when allvars) (hons-copy svars))
       (keys (svarlist-filter (alist-keys env)))
       (keys (mbe :logic (set::mergesort keys)
                  :exec (if (set::setp keys) keys (set::mergesort keys)))))
    (hons-copy
     (mbe :logic (union keys svars)
          :exec (if (set::subset keys svars)
                    svars
                  (if (eq svars nil)
                      keys
                    (union keys svars))))))
  ///
  (local (defthm alist-keys-of-svex-env-fix
           (equal (alist-keys (svex-env-fix env))
                  (svarlist-filter (alist-keys env)))
           :hints(("Goal" :in-theory (enable svex-env-fix svarlist-filter)))))

  (defret svexlist-vars-for-symbolic-eval-sufficient
    (subsetp (intersection-equal (svexlist-vars x)
                                 (alist-keys (svex-env-fix env)))
             (svarlist-fix vars))
    :hints ((set-reasoning))))


(define svexlist-eval-gl
  ((x svexlist-p     "Svex expressions to evaluate.")
   (env svex-env-p   "Bindings of variables to @(see 4vec) values.")
   (symbolic-params alistp
                    "Alist giving symbolic execution parameters; see below."))
  :short "Equivalent of svexlist-eval intended to work well under GL symbolic execution."
  :long "

<p>This function is provably equivalent to @(see svexlist-eval), but is
tailored to perform well under symbolic execution.  For symbolic execution, we
assume that the inputs to this function other than @('env') are fully concrete,
and that @('env') is symbolic only in its values, not its keys or its shape.</p>


<p>The @('symbolic-params') input is logically irrelevant, but allows important
optimizations for symbolic execution performance, discussed further below.  It
is safe (but not necessarily optimal) to call this with symbolic-params equal
@('NIL').</p>

<h4>Behavior under Symbolic Execution</h4>

<ol>

<li>Applies rewriting to the supplied svex expressions, if @(':SIMPLIFY') is
bound to a non-nil value in the @('symbolic-params') input -- see @(see
svexlist-rewrite-fixpoint).</li>

<li>If @(':boolmasks') is bound in the symbolic-params, compares the given
@('env') with the bound value, which should be an alist.  If there is a pair
@('(name . mask)') in the boolmasks alist for which the binding for @('name')
in env is not Boolean-valued on the bits set to 1 in @('mask'), then fail out
of symbolic simulation.  (In AIG mode, the masked bits must be
<i>syntactically</i> Boolean-valued -- practically speaking, this means the
upper/lower parts should result from the same computation.)</li>

<li>If @(':VARS') is bound in symbolic-params, it should be bound to a list of
input variables of the SVTV.  Unions this list with the variables bound in
@('env') to obtain the full list of variables to bind as inputs to the SVTV.
Or if @(':ALLVARS') is bound in symbolic params, all the variables in the svex
expressions are used instead.</li>

<li>Compiles the svex list @('x') into @(see a4vec) objects, a symbolic
analogue of @(see 4vec) but with each bit an AIG -- see @(see
svexlist->a4vecs-for-varlist).  This computation uses the assumptions, checked
in the two steps above, that only the variables in @('vars') are non-X, and
that the masked bits in @('boolmasks') are Boolean-valued. These assumptions
can reduce the complexity of the generated AIGs.  (Note everything used in this
computation is concrete -- the @('env') isn't involved.)</li>

<li>Creates an alist binding the AIG variables used in the above step to the
appropriate symbolic bits from @('env').</li>

<li>Symbolically evaluates each of the a4vec objects from step 4 under the
bindings from step 5 using GL's symbolic simulator of @(see acl2::aig-eval).
This results in GL-native symbolic 4vec objects, which is the result we
want.</li>

</ol>

<h4>Optimization using the Extra Arguments</h4>

<p>Performance of symbolic execution (and SAT solving, when in AIG mode) is
related to the size of the AIGs produced by the svex to AIG
transformation (step 4, above).  Two ways to decrease that size are (1) to turn
certain variables into constant Xes, if it is known that they're irrelevant,
and (2) to assume certain bits of some variables are Boolean-valued, which
means it can be represented by just one AIG variable rather than two.</p>

<p>Another performance consideration is that the transformation to AIGs is
itself sometimes significant.  Especially for theorems proved by
case-splitting, it is important not to need to repeat this transformation for
each case.  The function that does the transformation is memoized, but it is
important in this case that it always be called with the same arguments.</p>

<p>The @('vars') list pertains to optimization (1): if not present in the list,
a variable in the svex expressions will just be replaced with an X.  Therefore,
in general it's best to use exactly the set of variables bound in the
environment.  However, it may not be worth it to redo the AIG conversion each
time the environment's bound variables changes, so we take @('vars')
separately.</p>

<p>The @('boolmasks') allows optimization (2).  It is best for symbolic
execution performance to bind every variable in @('vars') to -1, but this may
fail if the @('env') is not constructed in such a way that the values are
obviously 2-vectors.</p>"
  :guard-hints (("goal" :in-theory (e/d (SET::UNION-WITH-SUBSET-LEFT)
                                        (SUBSET-OF-MERGESORTS-IS-SUBSETP))))
  (b* ((env (make-fast-alist (svex-env-fix env)))
       (x (maybe-svexlist-rewrite-fixpoint x (cdr (assoc :simplify symbolic-params)))) 
       (svars (svexlist-vars-for-symbolic-eval x env symbolic-params))
       (boolmasks (make-fast-alist
                   (hons-copy
                    (ec-call
                     (svar-boolmasks-fix (cdr (assoc :boolmasks symbolic-params)))))))
       ((unless (svex-env-check-boolmasks boolmasks env))
        (b* ((?ign (cw "ERROR: some bits assumed to be Boolean were not~%"))
             (?ign (gl::gl-error 'boolcheck-failed)))
          (gl::gl-hide (svexlist-eval x env))))
       ;; (?ign (cw "Boolmasks: ~x0~%" boolmasks))
       ;; (?ign (bitops::sneaky-push 'boolmasks boolmasks))
       ;; (?ign (bitops::sneaky-push 'vars vars))
       ;; (?ign (bitops::sneaky-push 'x x))
       ((mv err a4vecs) (time$ (svexlist->a4vecs-for-varlist x svars boolmasks)
                               :msg "; svex->aigs: ~st sec, ~sa bytes.~%"))
       ((when err)
        (b* ((?ign (cw "ERROR gathering AIG bits for variables: ~@0~%" err))
             (?ign (gl::gl-error 'a4env-failed)))
          (gl::gl-hide (svexlist-eval x env))))
       ((mv ?err aig-env)
        ;; ignore the error; it can't exist if the above doesn't
        (time$ (svexlist->a4vec-aig-env-for-varlist x svars boolmasks env)
               :msg "; env -> aig env: ~st sec, ~sa bytes.~%"))
       (?ign (fast-alist-free env)))
    (a4veclist-eval a4vecs aig-env))
  ///
  (defthm svexlist-eval-gl-is-svexlist-eval
    (equal (svexlist-eval-gl x env symbolic-params)
           (svexlist-eval x env))
    :hints (("goal" :use ((:instance svexlist->a4vec-for-varlist-correct
                           (boolmasks
                            (svar-boolmasks-fix (cdr (assoc :boolmasks symbolic-params))))
                           (vars (svexlist-vars-for-symbolic-eval
                                  (maybe-svexlist-rewrite-fixpoint x (cdr (assoc :simplify symbolic-params)))
                                  env symbolic-params))
                           (x (maybe-svexlist-rewrite-fixpoint x (cdr (assoc :simplify symbolic-params))))
                           (env (svex-env-fix env))))
             :in-theory (disable svexlist->a4vec-for-varlist-correct
                                 SVEXLIST->A4VECS-FOR-VARLIST-SVAR-BOOLMASKS-EQUIV-CONGRUENCE-ON-BOOLMASKS))))

  (gl::def-gl-rewrite svexlist-eval-for-symbolic-redef
    (equal (svexlist-eval-for-symbolic x env symbolic-params)
           (svexlist-eval-gl x env symbolic-params))))



;;; Now rework a4veclist-eval to phrase it in terms of a single call to aig-eval-list

(define a4vec->aiglist ((x a4vec-p))
  :returns (lst true-listp :rule-classes :type-prescription)
  (b* (((a4vec x) x))
    (append x.upper x.lower)))



(define v2i-alt ((v true-listp))
  :returns (v2i (equal v2i (gl::v2i v))
                :hints(("Goal" :in-theory (enable gl::scdr gl::s-endp))))
  :hooks nil
  (if (atom (cdr v))
      (gl::bool->sign (car v))
    (logcons (acl2::bool->bit (car v))
             (v2i-alt (cdr v)))))
  

(local (defthm v2i-of-aig-eval-list
         (equal (gl::v2i (aig-eval-list x env))
                (aig-list->s x env))
         :hints(("Goal" :in-theory (enable (:i aig-list->s) gl::v2i gl::scdr gl::s-endp)
                 :induct (aig-list->s x env)
                 :expand ((aig-list->s x env)
                          (aig-eval-list x env)
                          (:free (A b) (gl::v2i (cons a b))))))))

(define 4vec-from-bitlist ((upper-len natp) (lower-len natp) (bits true-listp))
  :hooks ((:fix :omit (bits)))
  :returns (mv (vec 4vec-p)
               (rest true-listp
                     :hyp (true-listp bits)
                     :rule-classes :type-prescription))
  ;; note: list-fixing bits is bad here because it's not even linear in the
  ;; number of bits we're operating on
  (b* ((upper-bits (take upper-len bits))
       (rest (nthcdr upper-len bits))
       (lower-bits (take lower-len rest))
       (rest (nthcdr lower-len rest)))
    (mv (4vec (v2i-alt upper-bits)
              (v2i-alt lower-bits))
        rest))
  ///
  (defthm 4vec-from-bitlist-correct
    (b* (((a4vec x) x))
      (equal (4vec-from-bitlist (len x.upper) (len x.lower)
                                (append (aig-eval-list (a4vec->aiglist x) env)
                                        rest))
             (mv (a4vec-eval x env) rest)))
    :hints(("Goal" :in-theory (enable a4vec->aiglist)))))

(define a4veclist->aiglist ((x a4veclist-p))
  :returns (aigs true-listp :rule-classes :type-prescription)
  (if (atom x)
      nil
    (append (a4vec->aiglist (car x))
            (a4veclist->aiglist (cdr x)))))

(define 4veclist-from-bitlist ((origs a4veclist-p) (bits true-listp))
  :returns (4vecs 4veclist-p)
  :hooks ((:fix :omit (bits)))
  (b* (((when (atom origs)) nil)
       ((a4vec x) (car origs))
       ((mv first restbits)
        (4vec-from-bitlist (len x.upper) (len x.lower) bits)))
    (cons first (4veclist-from-bitlist (cdr origs) restbits)))
  ///
  (defthm 4veclist-from-bitlist-correct
    (equal (4veclist-from-bitlist x (aig-eval-list (a4veclist->aiglist x) env))
           (a4veclist-eval x env))
    :hints(("Goal" :in-theory (enable a4veclist-eval
                                      a4veclist->aiglist)))))


(define a4veclist-eval-gl ((x a4veclist-p) (env))
  :returns (res 4veclist-p)
  (b* ((aiglist (time$ (a4veclist->aiglist x)
                       :msg "; SV bit-blasting: a4veclist->aiglist: ~st sec, ~sa bytes.~%"))
       (bitlist (time$ (aig-eval-list aiglist env)
                       :msg "; SV bit-blasting: aig-eval-list: ~st sec, ~sa bytes.~%")))
    (time$ (4veclist-from-bitlist x bitlist)
           :msg "; bits->4vecs: ~st sec, ~sa bytes.~%"))
  ///
  (defthm a4veclist-eval-gl-correct
    (equal (a4veclist-eval-gl x env)
           (a4veclist-eval x env)))

  (gl::def-gl-rewrite a4veclist-eval-redef
    (equal (a4veclist-eval x env)
           (a4veclist-eval-gl x env))))





(gl::def-gl-rewrite svex-alist-eval-gl-rewrite
    (equal (svex-alist-eval x env)
           (pairlis$ (svex-alist-keys x)
                     (svexlist-eval-for-symbolic
                      (svex-alist-vals x) env nil)))
    :hints(("Goal" :in-theory (enable svex-alist-eval pairlis$ svex-alist-keys
                                      svex-alist-vals svexlist-eval))))

(gl::def-gl-rewrite svex-eval-gl-rewrite
  (equal (svex-eval x env)
         (car (svexlist-eval-for-symbolic (list x) env nil))))
