; Copyright (C) 2016, ForrestHunt, Inc.
; Written by Matt Kaufmann and J Moore
; License: A 3-clause BSD license.  See the LICENSE file distributed with ACL2.

; Portcullis:
; (include-book "apply")

; Sample User Book for which We Will Establish All of the Warrants

; We define a variety of tame and mapping functions and prove a few
; illustrative theorems about a few of them just to show that we can.

; But the real motivation of this book is to develop a construction method for
; defining attachments for badge-userfn and apply$-userfn that make valid all
; the warrants issued in a user book built on apply.lisp.  We take this as a
; ``typical'' such book.

; With a focus on developing an attachment construction method (aka
; doppelganger construction) we defun$ functions that illustrate and exercise
; the various nuances in our construction.  Every time we've realized a problem
; with with our construction we've introduced a function here to show that
; we've figured out how to handle the problem.  So these mapping functions are
; pretty useless except as exemplars of the doppelganger construction.  That's
; why we don't really prove many theorems about them here.

; The complete list of functions defined with defun$ in this file is given
; below.  The list is in the same order in which the functions are defined
; below and was collected mechanically with an Emacs keyboard macro.  We
; assemble this list here in this comment so we can more easily confirm that
; doppelgangers.lisp deals with each of these functions.

#||
ap
rev
palindromify-list
list-of-true-listsp
list-of-list-of-true-listsp
expt-2-and-expt-3
expt-5
ok-fnp
collect
sumlist
sumlist-with-params
filter
all
collect-on
collect-tips
apply$2
apply$2x
apply$2xx
russell
foldr
foldl
collect-from-to
collect*
collect2
recur-by-collect
prow
prow*
fn-2-and-fn-3
fn-5
map-fn-5
sumlist-expr
collect-rev
sum-of-products
||#

(in-package "ACL2")

(include-book "misc/eval" :dir :system) ; for testing only

; -----------------------------------------------------------------
; Definitions

; For the sake of later modeling, we exhibit all of our defuns together, even
; though some are only used to state theorems about others.  We group them
; into: Group 1 (tame functions independent of apply$), Group 3 (mapping
; functions), and Group 2 (tame functions not independent of apply$).  The
; groups are discussed in doppelgangers.lisp.

; ---
; Group 1 (tame functions independent of apply$)

(defun$ ap (x y)
  (if (consp x)
      (cons (car x) (ap (cdr x) y))
      y))

(defun$ rev (x)
  (if (consp x)
      (ap (rev (cdr x)) (cons (car x) nil))
      nil))

(defun$ palindromify-list (lst)
  (cond ((endp lst) nil)
        (t (cons (ap (car lst) (rev (car lst)))
                 (palindromify-list (cdr lst))))))

(defun$ list-of-true-listsp (lst)
  (cond ((atom lst) (equal lst nil))
        (t (and (true-listp (car lst))
                (list-of-true-listsp (cdr lst))))))

(defun$ list-of-list-of-true-listsp (lst)
  (cond ((atom lst) (equal lst nil))
        (t (and (list-of-true-listsp (car lst))
                (list-of-list-of-true-listsp (cdr lst))))))

; This next pair illustrate the idea that a function, e.g., expt-2-and-expt-3,
; can have a badge in the badge-table without having a warrant, and then be
; used in a function with a warrant, e.g., expt-5.

(defun$ expt-2-and-expt-3 (x)
  (let ((x2 (* x x)))
    (mv x2 (* x x2))))

(defun$ expt-5 (x)
  (mv-let (x2 x3)
    (expt-2-and-expt-3 x)
    (* x2 x3)))

(defun$ ok-fnp (fn)
  (and (not (equal fn 'QUOTE))
       (not (equal fn 'IF))
       (tamep `(,fn X))))

; ---
; Group 3 (mapping functions)

(defun$ collect (lst fn)
  (cond ((endp lst) nil)
        (t (cons (apply$ fn (list (car lst)))
                 (collect (cdr lst) fn)))))

(defun$ sumlist (lst fn)
  (cond ((endp lst) 0)
        (t (+ (apply$ fn (list (car lst)))
              (sumlist (cdr lst) fn)))))

(defun$ sumlist-with-params (lst fn params)
  (cond ((endp lst) 0)
        (t (+ (apply$ fn (cons (car lst) params))
              (sumlist-with-params (cdr lst) fn params)))))

(defun$ filter (lst fn)
  (cond ((endp lst) nil)
        ((apply$ fn (list (car lst)))
         (cons (car lst) (filter (cdr lst) fn)))
        (t (filter (cdr lst) fn))))

(defun$ all (lst fn)
  (cond ((endp lst) t)
        (t (and (apply$ fn (list (car lst)))
                (all (cdr lst) fn)))))

(defun$ collect-on (lst fn)
  (cond ((endp lst) nil)
        (t (cons (apply$ fn (list lst))
                 (collect-on (cdr lst) fn)))))

(defun$ collect-tips (x fn)
  (cond ((atom x) (apply$ fn (list x)))
        (t (cons (collect-tips (car x) fn)
                 (collect-tips (cdr x) fn)))))

(defun$ apply$2 (fn x y)
  (apply$ fn (list x y)))

; These two functions illustrate getting further away from apply$.

(defun$ apply$2x (fn x y)
  (apply$2 fn x y))

(defun$ apply$2xx (fn x y)
  (apply$2x fn x y))

; A Russell-like function: The classic russell function would be

; ( defun$ russell (fn)
;   (not (apply$ fn (list fn))))

; But this abuses our classification system because fn is used both in a
; functional slot and a vanilla slot.  However, the following function raises
; the same problems as Russell's and is admissible here.

(defun$ russell (fn x)
  (not (apply$ fn (list x x))))

; Of interest, of course, is (russell 'russell 'russell) because the
; naive apply$ property would give us:
; (russell 'russell 'russell)                            {def russell}
; = (not (apply$ 'russell (list 'russell 'russell)))     {naive apply$}
; = (not (russell 'russell 'russell))

(defun$ foldr (lst fn init)
  (if (endp lst)
      init
      (apply$ fn
              (list (car lst)
                    (foldr (cdr lst) fn init)))))

(defun$ foldl (lst fn ans)
  (if (endp lst)
      ans
      (foldl (cdr lst)
             fn
             (apply$ fn (list (car lst) ans)))))

(defun$ collect-from-to (i max fn)
  (declare (xargs :measure (nfix (- (+ 1 (ifix max)) (ifix i)))))
  (let ((i (ifix i))
        (max (ifix max)))
    (cond
     ((> i max)
      nil)
     (t (cons (apply$ fn (list i))
              (collect-from-to (+ i 1) max fn))))))

(defun$ collect* (lst fn)
  (if (endp lst)
      nil
      (cons (apply$ fn (car lst))
            (collect* (cdr lst) fn))))

(defun$ collect2 (lst fn1 fn2)
  (if (endp lst)
      nil
      (cons (cons (apply$ fn1 (list (car lst)))
                  (apply$ fn2 (list (car lst))))
            (collect2 (cdr lst) fn1 fn2))))

(defun$ recur-by-collect (lst fn)
  (declare (xargs :measure (len lst)))
  (if (endp lst)
      nil
      (cons (car lst)
	    (recur-by-collect (collect (cdr lst) fn) fn))))

(defun$ prow (lst fn)
  (cond ((or (endp lst) (endp (cdr lst)))
         nil)
        (t (cons (apply$ fn (list (car lst) (cadr lst)))
                 (prow (cdr lst) fn)))))

(defun$ prow* (lst fn)
  (declare (xargs :measure (len lst)))
  (cond ((or (endp lst)
             (endp (cdr lst)))
         (apply$ fn (list lst lst)))
        (t (prow* (prow lst fn) fn))))

; These are nonrecursive mapping functions, the first of which
; is un-warranted because it returns multiple values.
 
(defun$ fn-2-and-fn-3 (fn x)
; Return (mv (fn x x) (fn x (fn x x)))
  (let ((x2 (apply$ fn (list x x))))
    (mv x2 (apply$ fn (list x x2)))))

(defun$ fn-5 (fn x)
  (mv-let (x2 x3)
    (fn-2-and-fn-3 fn x)
    (apply$ fn (list x2 x3))))

(defun$ map-fn-5 (lst fn)
  (if (endp lst)
      nil
      (cons (fn-5 fn (car lst))
            (map-fn-5 (cdr lst) fn))))

(defun$ sumlist-expr (lst expr alist)
  (cond ((endp lst) 0)
        (t (+ (ev$ expr (cons (cons 'x (car lst)) alist))
              (sumlist-expr (cdr lst) expr alist)))))

; ---
; Group 2 (tame functions not independent of apply$)

(defun$ collect-rev (lst)
  (collect lst 'REV))

(defun$ sum-of-products (lst)
  (sumlist lst
           '(LAMBDA (X)
                    (FOLDR X
                           '(LAMBDA (I A)
                                    (BINARY-* I A))
                           '1))))

; -----------------------------------------------------------------
; Some Sample Theorems about Mapping Functions

(defthm recur-by-collect-example
  (equal (recur-by-collect '(1 1 1) '(lambda (x) (binary-+ '1 x)))
         '(1 2 3))
  :rule-classes nil)

(defthm collect-ap
  (equal (collect (ap a b) fn)
         (ap (collect a fn)
             (collect b fn))))

(must-fail
 (with-prover-step-limit
  10000
  (defthm theorem-about-collect-ap-rev
    (equal (collect lst '(lambda (e) (ap e (rev e))))
           (palindromify-list lst))
    :rule-classes nil)))

; But this succeeds in 1671 steps.
(defthm theorem-about-collect-ap-rev
  (implies (warrant AP REV)
           (equal (collect lst '(lambda (e) (ap e (rev e))))
                  (palindromify-list lst)))
  :rule-classes nil)

; Here is another theorem.

; Of course, this concept is the same as a nest of all, and we can prove
; that (but we don't make it a rule).

; By the way, this theorem needs no warrants because it doesn't use any user
; defined functions.

(defthm list-of-list-of-true-listsp-expessed-as-all
  (implies (warrant all)
           (equal (list-of-list-of-true-listsp lst)
                  (and (true-listp lst)
                       (all lst
                                '(lambda (x)
                                   (if (true-listp x)
                                       (all x 'true-listp)
                                       'nil))))))
  :rule-classes nil)

; Note: We have to use IF instead of AND above because AND is a macro and
; apply$ doesn't know how to interpret it.

; We prove three versions of the theorem.  The first is about a composition of
; two collects, each of which map with another collect:

(must-fail ; failed to supply warrants
 (defthm theorem-about-collect-collect-rev-twice-version1
   (implies (list-of-list-of-true-listsp lst)
            (equal (collect
                    (collect lst '(lambda (x) (collect x 'rev)))
                    '(lambda (x) (collect x 'rev)))
                   lst))
;  :hints (("Goal" :do-not '(eliminate-irrelevance))) ; This hint needed in Version_7.2
   )) ; see the next attempt

(defthm theorem-about-collect-collect-rev-twice-version1
  (implies (and (warrant collect rev)
                (list-of-list-of-true-listsp lst))
           (equal (collect
                   (collect lst '(lambda (x) (collect x 'rev)))
                   '(lambda (x) (collect x 'rev)))
                  lst))
; :hints (("Goal" :do-not '(eliminate-irrelevance)))  ; This hint needed in Version_7.2
  )

; Note: Through ACL2 Version_7.2, ACL2 would discard the 0-ary functions as
; irrelevant prior to a second induction.  The :do-not hint prevents it.  This
; was an oversight in the heuristic: quantified hypotheses sharing no variables
; with the rest of the clause might still be relevant!  The heuristic was
; improved after Version_7.2.

; The second version is about a composition of two collect-revs.  One might
; think that we must provide a warrant for COLLECT-REV.  But that's wrong.  We
; must also provide one for REV because after (APPLY$ 'COLLECT-REV ...) expands
; to (collect-rev ...), that expands to (collect ... 'REV) and hence introduces
; (APPLY$ 'REV ...).  It is possible to detect this requirement by looking at
; the forced subgoals.

(defthm theorem-about-collect-collect-rev-twice-version2
  (implies (and (warrant collect-rev rev)
                (list-of-list-of-true-listsp lst))
           (equal (collect
                   (collect lst 'collect-rev)
                   'collect-rev)
                  lst))
 ;:hints (("Goal" :do-not '(eliminate-irrelevance)))
  )

; Here is a version with the hypothesis phrased with ALL:

(defthm theorem-about-collect-collect-rev-twice-version3
  (implies (and (warrant all collect rev)
                (true-listp lst)
                (all lst
                     '(lambda (x)
                        (if (true-listp x)
                            (all x 'true-listp)
                            'nil))))
           (equal (collect
                   (collect lst '(lambda (x) (collect x 'rev)))
                   '(lambda (x) (collect x 'rev)))
                  lst))
; :hints (("Goal" :do-not '(eliminate-irrelevance)))
  )

; A few theorems manipulating mapping functions and the functions they
; apply:

(defthm sumlist-ap
  (equal (sumlist (ap a b) u)
         (+ (sumlist a u)
            (sumlist b u))))

; Here is a way to move a constant factor out of a sumlist regardless of the
; names of the variables.

(defthm sumlist-binary-*-constant
  (implies (tamep body)
           (equal (sumlist lst (lamb (list v) (list 'binary-* (list 'quote const) body)))
                  (* const (sumlist lst (lamb (list v) body))))))

(defthm lamb-x-x-is-identity
  (implies (symbolp x)
           (fn-equal (lamb (list x) x) 'identity))
  :hints (("Goal" :in-theory (enable fn-equal))))

; The hint below is only provided to show that the theorem is proved by
; rewriting, not induction.

(defthm example-of-loop$-rewriting
  (equal (sumlist (ap aaa bbb) (lamb '(x) '(binary-* '2 x)))
         (+ (* 2 (sumlist aaa 'identity))
            (* 2 (sumlist bbb 'identity))))
  :hints (("Goal" :do-not-induct t))
  :rule-classes nil)

; A theorem showing that functions can be data, i.e., we can apply mapping functions
; to mapping functions.

(defthm collect*-collect-example
  (implies (warrant collect)
           (equal (collect* '(((1 2 3) (lambda (x) (cons 'a x)))
                              ((4 5 6 7) (lambda (z) (cons 'b z)))
                              ((8 9) (lambda (y) (cons 'c y))))
                            'collect)
                  '(((a . 1)(a . 2)(a . 3))
                    ((b . 4) (b . 5) (b . 6) (b . 7))
                    ((c . 8) (c . 9)))))
  :hints
  (("Goal"
    :in-theory
    (disable (:executable-counterpart collect)
             (:executable-counterpart collect*)))))

; A couple of nice foldr theorems

(defthm foldr-is-ap
  (equal (foldr x 'cons y) (ap x y)))

(defthm foldr-is-rev
  (implies (warrant foldr)
           (equal (foldr x
                         '(lambda (x y)
                                  (foldr y 'cons (cons x 'nil)))
                         nil)
                  (rev x))))

(defthm collect-is-a-foldr
  (implies (force (ok-fnp fn))
           (equal (collect lst fn)
                  (foldr lst
                         `(LAMBDA (X Y)
                                  (CONS (,fn X) Y))
                         nil))))

(defthm all-is-a-foldr
  (implies (force (ok-fnp fn))
           (equal (all lst fn)
                  (foldr lst
                         `(LAMBDA (X Y)
                                  (IF (,fn X) Y 'NIL))
                         t))))

(defthm sumlist-is-a-foldr
  (implies (force (ok-fnp fn))
           (equal (sumlist lst fn)
                  (foldr lst
                         `(LAMBDA (X Y)
                                  (BINARY-+ (,fn X) Y))
                         0))))

