/*
 *  ACM - 3-D draw utilities
 *  Copyright (C) 2008  Umberto Salsi
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 dated June, 1991.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program;  if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave., Cambridge, MA 02139, USA.
 */

/**
 * This module implements collections of "tracks", every track being
 * a collection of two or more VPoint. 3-D paths can then be transformed
 * with a matrix and projected on the x-y plane of the screen.
 * @file
 */

#ifndef _VPATH_H
#define _VPATH_H

#include "../V/Alib.h"
#include "../V/Vlib.h"

#ifdef vpath_IMPORT
	#define EXTERN
#else
	#define EXTERN extern
#endif

typedef struct vpath_Type vpath_Type;

/**
 * Create new empty VPath. The drawing pen is located by default to the
 * origin point (0,0,0). Can be released with memory_dispose().
 */
EXTERN vpath_Type * vpath_new(void);

/**
 * Move the drawing pen to the given absolute point, beginning a new
 * track.
 */
EXTERN void vpath_moveTo(vpath_Type *p, VPoint *to);

/**
 * Move the pen from the current pen position to the given destination
 * point, drawing a segment.
 */
EXTERN void vpath_lineTo(vpath_Type *p, VPoint *to);

/**
 * Starting from the current pen position as set by vpath_moveTo() or
 * vpath_lineTo(), draw an arc of ellipse having the given center and the
 * final point. n>=1 is the number of intermediate segments, higher values
 * give a smoother line.
 */
EXTERN void vpath_arc(vpath_Type *p, VPoint *center, VPoint *final, int n);

/**
 * Draw the string on the xy plane starting from the origin and
 * proceeding along the x axis as baseline. Characters have the
 * dimension 1.0x1.0.  m is the transformation matrix.  For example,
 * this code will add to the VPath the string "Hello" centered around
 * the point (h,v) of the screen:
 * 
 * VMatrix m;
 * double fw, fh;
 * char *s = "Hello";
 * 
 * fw = fh = 10.0;
 * VIdentMatrix(&m);
 * VScaleMatrix(&m, fw, fh, 1.0);
 * VTranslate(&m, h - fw*strlen(s)/2, v + fh/2, 0.0);
 * vpath_draw_string(path, s, strlen(s), m);
 */
EXTERN void vpath_draw_string(vpath_Type *path, char *str, int len, VMatrix *m);

/**
 * Translate the 3-D VPath into 2-D segments with a parallel projection
 * from (x,y,z) 3-D coords to (x,y) 2-D coords in the screen frame
 * (x=right, y=down, z=toward the screen). Every point get transformed as
 * per the provided matrix. Segments are clipped to the z < 0 space (=
 * front of the screen) so to ensure that only the wanted part of the
 * drawing be visible either translate along the z axis or extrude setting
 * z scale negative.
 */
EXTERN void vpath_stroke(vpath_Type *p, VMatrix *m, Alib_Window *w, Alib_Pixel color);

/**
 * Perspective projection of the vpath on the screen. The matrix m is
 * applied to every point of the vpath to obtain the point q=m*p to
 * project on the given viewport, current clipping region. The unit of q
 * is the meter, so q=(0,0,1) is the point exactly one meter in front to
 * the eye of the observer, and q=(1,0,0) is one meter to the right.
 */
EXTERN void vpath_perspective_stroke(vpath_Type *p, VMatrix *m, Viewport *v, Alib_Pixel color);

#undef EXTERN
#endif
