// ---------------------------------------------------------------------------
// - Lexical.hpp                                                             -
// - afnix engine - lexical name class definition                            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2020 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_LEXICAL_HPP
#define  AFNIX_LEXICAL_HPP

#ifndef  AFNIX_STRING_HPP
#include "String.hpp"
#endif

namespace afnix {

  /// The Lexical class is a simple class which holds a lexical name. Such
  /// object is generated by the afnix lexical analyzer. The object is
  /// constructed by name. Static methods are available to check for the
  /// name character syntax.
  /// @author amaury darsch

  class Lexical : public Literal {
  public:
    /// @return true if the character is a valid lexical one
    static bool valid (const t_quad c);

    /// @return true if the lexical name is a valid one
    static bool valid (const String& name);

  private:
    /// the lexical name
    String d_name;
    /// the lexical quark
    long   d_quark;
    /// the line number
    long   d_lnum;

  public:
    /// create an empty lexical
    Lexical (void);

    /// create a new lexical with a name
    /// @param name the lexical name
    Lexical (const String& name);

    /// create a new lexical with a name and a line number
    /// @param name the lexical name
    /// @param line the line number
    Lexical (const String& name, const long lnum);

    /// copy construct this lexical
    /// @param that the lexical to copy
    Lexical (const Lexical& that);

    /// @return the class name
    String repr (void) const;

    /// @return a clone of this object
    Object* clone (void) const;

    /// clear this lexical name
    void clear (void);

    /// @return a literal representation of this lexical
    String toliteral (void) const;

    /// @return a string representation of this lexical
    String tostring (void) const;

    /// @return the serial did
    t_word getdid (void) const;
    
    /// @return the serial sid
    t_word getsid (void) const;

    /// serialize this lexical to an output stream
    /// @param os the output stream to write
    void wrstream (class OutputStream& os) const;

    /// deserialize a lexical from an input stream
    /// @param is the input steam to read in
    void rdstream (class InputStream& is);

    /// @return the lexical line number
    long getlnum (void) const;

    /// @return the lexical quark
    long toquark (void) const;
    
    /// @return true if the lexical is the nil string
    bool isnil (void) const;

  private:
    // make the assignment operator private
    Lexical& operator = (const Lexical&);
    
  public:
    /// create a new object in a generic way
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// set this object as a const object
    /// @param zobj   the current evaluable
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* cdef (Evaluable* zobj, Nameset* nset, Object* object);

    /// set this object with an object
    /// @param zobj   the current evaluable
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Evaluable* zobj, Nameset* nset, Object* object);

    /// unreference this object
    /// @param zobj   the current evaluable
    /// @param nset   the current nameset
    Object* udef (Evaluable* zobj, Nameset* nset);

    /// evaluate this object
    /// @param zobj  the current evaluable
    /// @param nset  the current nameset
    Object* eval (Evaluable* zobj, Nameset* nset);

    /// apply this object with a set of arguments and a quark
    /// @param zobj  the current evaluable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Evaluable* zobj, Nameset* nset, const long quark,
		   Vector* argv);
  };
}

#endif
