#include <iostream>

#include <gtkmm/messagedialog.h>
#include "../gtkmm-compat.h"

#include "../strategy/actions/iterationaction.h"
#include "../strategy/actions/strategy.h"

#include "../strategy/control/strategyreader.h"
#include "../strategy/control/strategywriter.h"
#include "../strategy/control/defaultstrategy.h"

#include "addstrategyactionmenu.h"
#include "editstrategywindow.h"
#include "interfaces.h"
#include "strategywizardwindow.h"

#include "controllers/rfiguicontroller.h"

#include "strategyframes/absthresholdframe.h"
#include "strategyframes/baselineselectionframe.h"
#include "strategyframes/changeresolutionframe.h"
#include "strategyframes/cutareaframe.h"
#include "strategyframes/foreachbaselineframe.h"
#include "strategyframes/foreachmsframe.h"
#include "strategyframes/foreachpolarisationframe.h"
#include "strategyframes/foreachcomplexcomponentframe.h"
#include "strategyframes/frequencyconvolutionframe.h"
#include "strategyframes/frequencyselectionframe.h"
#include "strategyframes/fringestoppingframe.h"
#include "strategyframes/highpassfilterframe.h"
#include "strategyframes/iterationframe.h"
#include "strategyframes/plotframe.h"
#include "strategyframes/resamplingframe.h"
#include "strategyframes/setflaggingframe.h"
#include "strategyframes/setimageframe.h"
#include "strategyframes/slidingwindowfitframe.h"
#include "strategyframes/statisticalflaggingframe.h"
#include "strategyframes/svdframe.h"
#include "strategyframes/sumthresholdframe.h"
#include "strategyframes/timeconvolutionframe.h"
#include "strategyframes/timeselectionframe.h"

using namespace rfiStrategy;

EditStrategyWindow::EditStrategyWindow(RFIGuiController& guiController, StrategyController &strategyController)
 : Gtk::Window(), 
	_strategyController(strategyController),
	_addActionButton("Add"),
	_removeActionButton("_Remove", true),
	_moveUpButton("_Up", true),
	_moveDownButton("_Down", true),
	_loadEmptyButton("_New", true),
	_wizardButton("_Wizard...", true),
	_saveButton("_Save As...", true),
	_openButton("_Open", true),
	_disableUpdates(false),
	_guiController(guiController),
	_rightFrame(nullptr),
	_wizardWindow()
{
	_addActionButton.set_icon_name("list-add");
	gtkmm_set_image_from_icon_name(_removeActionButton, "edit-delete");
	gtkmm_set_image_from_icon_name(_moveUpButton, "go-up");
	gtkmm_set_image_from_icon_name(_moveDownButton, "go-down");
	gtkmm_set_image_from_icon_name(_loadEmptyButton, "document-new");
	gtkmm_set_image_from_icon_name(_saveButton, "document-save-as");
	gtkmm_set_image_from_icon_name(_openButton, "document-open");
	
	_strategyController.SignalOnStrategyChanged().connect(sigc::mem_fun(*this, &EditStrategyWindow::onStrategyChanged));
	
	_store = Gtk::TreeStore::create(_columns);
	_view.set_model(_store);
	_view.append_column("Description", _columns.description);
	_viewScrollWindow.add(_view);
	_view.get_selection()->signal_changed().connect(
		sigc::mem_fun(*this, &EditStrategyWindow::onSelectionChanged));
	
	_viewScrollWindow.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	_viewScrollWindow.set_size_request(100, 400);
	_strategyBox.pack_start(_viewScrollWindow);

	initEditButtons();

	initLoadDefaultsButtons();
	
	_paned.add1(_strategyBox);

	add(_paned);
	
	show_all();
	
	_strategy = &_strategyController.Strategy();
	fillStore();
}

void EditStrategyWindow::initEditButtons()
{
	_strategyEditButtonBox.pack_start(_addActionButton);
	_addActionButton.set_sensitive(false);
	_addMenu.reset( new AddStrategyActionMenu(*this) );
	_addActionButton.set_menu(*_addMenu);

	_strategyEditButtonBox.pack_start(_moveUpButton);
	_moveUpButton.set_sensitive(false);
	_moveUpButton.signal_clicked().connect(sigc::mem_fun(*this, &EditStrategyWindow::onMoveUpClicked));

	_strategyEditButtonBox.pack_start(_moveDownButton);
	_moveDownButton.set_sensitive(false);
	_moveDownButton.signal_clicked().connect(sigc::mem_fun(*this, &EditStrategyWindow::onMoveDownClicked));

	_strategyEditButtonBox.pack_start(_removeActionButton);
	_removeActionButton.set_sensitive(false);
	_removeActionButton.signal_clicked().connect(sigc::mem_fun(*this, &EditStrategyWindow::onRemoveActionClicked));

	_strategyBox.pack_start(_strategyEditButtonBox, Gtk::PACK_SHRINK, 0);

	_strategyFileButtonBox.pack_start(_saveButton);
	_saveButton.signal_clicked().connect(sigc::mem_fun(*this, &EditStrategyWindow::onSaveClicked));

	_strategyFileButtonBox.pack_start(_openButton);
	_openButton.signal_clicked().connect(sigc::mem_fun(*this, &EditStrategyWindow::onOpenClicked));

	_strategyBox.pack_start(_strategyFileButtonBox, Gtk::PACK_SHRINK, 0);
}

void EditStrategyWindow::initLoadDefaultsButtons()
{
	_strategyLoadDefaultsButtonBox.pack_start(_loadEmptyButton);
	_loadEmptyButton.signal_clicked().connect(sigc::mem_fun(*this, &EditStrategyWindow::onLoadEmptyClicked));

	_strategyLoadDefaultsButtonBox.pack_start(_wizardButton);
	_wizardButton.signal_clicked().connect(sigc::mem_fun(*this, &EditStrategyWindow::onWizardClicked));

	_strategyBox.pack_start(_strategyLoadDefaultsButtonBox, Gtk::PACK_SHRINK, 0);
}

void EditStrategyWindow::onStrategyChanged()
{
	_disableUpdates = true;
	_store->clear();
	_disableUpdates = false;
	_strategy = &_strategyController.Strategy();
	clearRightFrame();
	fillStore();
	onSelectionChanged();
}

void EditStrategyWindow::fillStore()
{
	Gtk::TreeModel::iterator iter = _store->append();
	Gtk::TreeModel::Row row = *iter;
	row[_columns.action] = _strategy;
	row[_columns.description] = _strategy->Description();
	row[_columns.childIndex] = 0;
	for(size_t i = 0;i<_strategy->GetChildCount();++i)
	{
		fillStore(row, _strategy->GetChild(i), i);
	}
	_view.expand_all();
}

void EditStrategyWindow::fillStore(Gtk::TreeModel::Row &row, Action &action, size_t childIndex)
{
	Gtk::TreeModel::iterator iter = _store->append(row.children());
	Gtk::TreeModel::Row newRow = *iter;
	newRow[_columns.action] = &action;
	newRow[_columns.description] = action.Description();
	newRow[_columns.childIndex] = childIndex;
	ActionContainer *container = dynamic_cast<ActionContainer*>(&action);
	if(container != 0)
	{
		for(size_t i = 0;i<container->GetChildCount();++i)
		{
			fillStore(newRow, container->GetChild(i), i);
		}
	}
}

void EditStrategyWindow::onRemoveActionClicked()
{
	_disableUpdates = true;
	clearRightFrame();
	Action *action = GetSelectedAction();
	if(action != nullptr && action->Parent() != nullptr)
	{
		action->Parent()->RemoveAndDelete(action);
		_store->clear();
		fillStore();
		_view.get_selection()->unselect_all();
	}
	_disableUpdates = false;
	onSelectionChanged();
}

void EditStrategyWindow::onMoveUpClicked()
{
	Action *action = GetSelectedAction();
	if(action != 0 && action->Parent() != 0)
	{
		ActionContainer *parent = static_cast<ActionContainer*>(action->Parent());
		size_t index = GetSelectedActionChildIndex();
		parent->MoveChildUp(index);
		_store->clear();
		fillStore();
		selectAction(action);
	}
}

void EditStrategyWindow::onMoveDownClicked()
{
	Action *action = GetSelectedAction();
	if(action != 0 && action->Parent() != 0)
	{
		ActionContainer *parent = static_cast<ActionContainer*>(action->Parent());
		size_t index = GetSelectedActionChildIndex();
		parent->MoveChildDown(index);
		_store->clear();
		fillStore();
		selectAction(action);
	}
}

void EditStrategyWindow::onSelectionChanged()
{
	if(!_disableUpdates)
	{
		Action *selectedAction = GetSelectedAction();
		if(selectedAction != 0)
		{
			clearRightFrame();
			
			_moveDownButton.set_sensitive(true);
			_moveUpButton.set_sensitive(true);
			_removeActionButton.set_sensitive(true);
			ActionContainer *container = dynamic_cast<rfiStrategy::ActionContainer*>(selectedAction);
			if(container != 0)
			{
				_addActionButton.set_sensitive(true);
			} else {
				_addActionButton.set_sensitive(false);
			}

			switch(selectedAction->Type())
			{
				case AbsThresholdActionType:
					showRight(new AbsThresholdFrame(*static_cast<rfiStrategy::AbsThresholdAction*>(selectedAction), *this));
					break;
				case BaselineSelectionActionType:
					showRight(new BaselineSelectionFrame(*static_cast<rfiStrategy::BaselineSelectionAction*>(selectedAction), *this));
					break;
				case ChangeResolutionActionType:
					showRight(new ChangeResolutionFrame(*static_cast<rfiStrategy::ChangeResolutionAction*>(selectedAction), *this));
					break;
				case CutAreaActionType:
					showRight(new CutAreaFrame(*static_cast<rfiStrategy::CutAreaAction*>(selectedAction), *this));
					break;
				case FringeStopActionType:
					showRight(new FringeStoppingFrame(*static_cast<rfiStrategy::FringeStopAction*>(selectedAction), *this));
					break;
				case IterationBlockType:
					showRight(new IterationFrame(*static_cast<rfiStrategy::IterationBlock*>(selectedAction), *this));
					break;
				case SlidingWindowFitActionType:
					showRight(new SlidingWindowFitFrame(*static_cast<rfiStrategy::SlidingWindowFitAction*>(selectedAction), *this));
					break;
				case SVDActionType:
					showRight(new SVDFrame(*static_cast<rfiStrategy::SVDAction*>(selectedAction), *this));
					break;
				case ForEachBaselineActionType:
					showRight(new ForEachBaselineFrame(*static_cast<rfiStrategy::ForEachBaselineAction*>(selectedAction), *this));
					break;
				case ForEachComplexComponentActionType:
					showRight(new ForEachComplexComponentFrame(*static_cast<rfiStrategy::ForEachComplexComponentAction*>(selectedAction), *this));
					break;
				case ForEachMSActionType:
					showRight(new ForEachMSFrame(*static_cast<rfiStrategy::ForEachMSAction*>(selectedAction), *this));
					break;
				case ForEachPolarisationBlockType:
					showRight(new ForEachPolarisationFrame(*static_cast<rfiStrategy::ForEachPolarisationBlock*>(selectedAction), *this));
					break;
				case FrequencyConvolutionActionType:
					showRight(new FrequencyConvolutionFrame(*static_cast<rfiStrategy::FrequencyConvolutionAction*>(selectedAction), *this));
					break;
				case FrequencySelectionActionType:
					showRight(new FrequencySelectionFrame(*static_cast<rfiStrategy::FrequencySelectionAction*>(selectedAction), *this));
					break;
				case HighPassFilterActionType:
					showRight(new HighPassFilterFrame(*static_cast<rfiStrategy::HighPassFilterAction*>(selectedAction), *this));
					break;
				case PlotActionType:
					showRight(new StrategyPlotFrame(*static_cast<rfiStrategy::PlotAction*>(selectedAction), *this));
					break;
				case ResamplingActionType:
					showRight(new ResamplingFrame(*static_cast<rfiStrategy::ResamplingAction*>(selectedAction), *this));
					break;
				case SetImageActionType:
					showRight(new SetImageFrame(*static_cast<rfiStrategy::SetImageAction*>(selectedAction), *this));
					break;
				case SetFlaggingActionType:
					showRight(new SetFlaggingFrame(*static_cast<rfiStrategy::SetFlaggingAction*>(selectedAction), *this));
					break;
				case StatisticalFlagActionType:
					showRight(new MorphologicalFlaggingFrame(*static_cast<rfiStrategy::MorphologicalFlagAction*>(selectedAction), *this));
					break;
				case SumThresholdActionType:
					showRight(new SumThresholdFrame(*static_cast<rfiStrategy::SumThresholdAction*>(selectedAction), *this));
					break;
				case TimeConvolutionActionType:
					showRight(new TimeConvolutionFrame(*static_cast<rfiStrategy::TimeConvolutionAction*>(selectedAction), *this));
					break;
				case TimeSelectionActionType:
					showRight(new TimeSelectionFrame(*static_cast<rfiStrategy::TimeSelectionAction*>(selectedAction), *this));
					break;
				default:
					break;
			}
		} else {
			_addActionButton.set_sensitive(false);
			_moveDownButton.set_sensitive(false);
			_moveUpButton.set_sensitive(false);
			_removeActionButton.set_sensitive(false);
		}
		//resize window
		int prefWidth, natWidth;
		_paned.get_preferred_width(prefWidth, natWidth);
		prefWidth = std::max(prefWidth, natWidth);
		if(get_width() < prefWidth)
			resize(prefWidth, get_height());
	}
}

void EditStrategyWindow::clearRightFrame()
{
	if(_rightFrame != 0)
	{
		delete _rightFrame;
		_rightFrame = 0;
	}
}

rfiStrategy::Action *EditStrategyWindow::GetSelectedAction()
{
	Gtk::TreeModel::iterator iter = _view.get_selection()->get_selected();
	if(iter) //If anything is selected
	{
		Gtk::TreeModel::Row row = *iter;
		return row[_columns.action];
	}
	else return 0;
}

size_t EditStrategyWindow::GetSelectedActionChildIndex()
{
	Gtk::TreeModel::iterator iter = _view.get_selection()->get_selected();
	if(iter) //If anything is selected
	{
		Gtk::TreeModel::Row row = *iter;
		return row[_columns.childIndex];
	}
	else return 0;
}

void EditStrategyWindow::AddAction(std::unique_ptr<rfiStrategy::Action> newAction)
{
	Action *action = GetSelectedAction();
	if(action != 0)
	{
		rfiStrategy::ActionContainer *container = dynamic_cast<rfiStrategy::ActionContainer*>(action);
		if(container != 0)
		{
			rfiStrategy::Action* actionPtr = newAction.get();
			container->Add(std::move(newAction));
			_store->clear();
			fillStore();
			_view.get_selection()->unselect_all();
			selectAction(actionPtr);
		}
	}
}

void EditStrategyWindow::selectAction(rfiStrategy::Action *action)
{
	_view.get_selection()->select(findActionRow(action));
}

void EditStrategyWindow::UpdateAction(Action *action)
{
	if(action != 0)
	{
		Gtk::TreeModel::Row row = findActionRow(action);
		row[_columns.description] = action->Description();
	}
}

Gtk::TreeModel::Row EditStrategyWindow::findActionRow(rfiStrategy::Action *action)
{
	std::deque<Gtk::TreeModel::Row> rows;
	Gtk::TreeNodeChildren children = _store->children();
	for(Gtk::TreeModel::const_iterator iter = children.begin();iter!=children.end();++iter)
	{
		const Gtk::TreeModel::Row &row = (*iter);
		rows.push_back(row);
	}
	while(!rows.empty())
	{
		Gtk::TreeModel::Row row = rows.front();
		rows.pop_front();
		if(row[_columns.action] == action)
		{
			return row;
		}
		Gtk::TreeNodeChildren rowChildren = row.children();
		for(Gtk::TreeModel::const_iterator iter = rowChildren.begin();iter != rowChildren.end();++iter)
		{
			Gtk::TreeModel::Row childRow = *iter;
			rows.push_back(childRow);
		}
	}
	throw BadUsageException("Could not find row in view");
}

void EditStrategyWindow::onLoadEmptyClicked()
{
	_store->clear();
	_strategy->RemoveAll();
	fillStore();
}

void EditStrategyWindow::onSaveClicked()
{
  Gtk::FileChooserDialog dialog(*this, "Save strategy", Gtk::FILE_CHOOSER_ACTION_SAVE);

  dialog.add_button("_Cancel", Gtk::RESPONSE_CANCEL);
  dialog.add_button("_Save", Gtk::RESPONSE_OK);

  Glib::RefPtr<Gtk::FileFilter> filter = Gtk::FileFilter::create();
  filter->set_name("RFI strategies");
  filter->add_pattern("*.rfis");
  filter->add_mime_type("text/rfistrategy+xml");
  dialog.add_filter(filter);

  int result = dialog.run();
  if(result == Gtk::RESPONSE_OK)
	{
		rfiStrategy::StrategyWriter writer;
		std::string filename(dialog.get_filename());
		if(filename.find('.') == std::string::npos)
			filename += ".rfis";
		writer.WriteToFile(*_strategy, filename);
		
	}
}

void EditStrategyWindow::onOpenClicked()
{
  Gtk::FileChooserDialog dialog(*this, "OPEN strategy", Gtk::FILE_CHOOSER_ACTION_OPEN);

  dialog.add_button("_Cancel", Gtk::RESPONSE_CANCEL);
  dialog.add_button("_Open", Gtk::RESPONSE_OK);

  Glib::RefPtr<Gtk::FileFilter> filter = Gtk::FileFilter::create();
  filter->set_name("RFI strategies");
  filter->add_pattern("*.rfis");
  filter->add_mime_type("text/rfistrategy+xml");
  dialog.add_filter(filter);

  int result = dialog.run();
  if(result == Gtk::RESPONSE_OK)
	{
		StrategyReader reader;
		std::string filename(dialog.get_filename());
		try {
			_store->clear();
			std::unique_ptr<Strategy> s = reader.CreateStrategyFromFile(filename);
			_strategy = s.get();
			_strategyController.SetStrategy(std::move(s));
			fillStore();
		} catch(std::exception &e)
		{
			Gtk::MessageDialog dialog(*this, e.what(), false, Gtk::MESSAGE_ERROR);
			dialog.run();
		}
	}
}

/*void EditStrategyWindow::addContainerBetween(rfiStrategy::ActionContainer &root, std::unique_ptr<rfiStrategy::ActionContainer> newContainer)
{
	while(root.GetChildCount() > 0)
	{
		newContainer->Add(root.RemoveAndAcquire(&root.GetFirstChild()));
	}
	root.Add(std::move(newContainer));
}*/

void EditStrategyWindow::onWizardClicked()
{
	if(_wizardWindow)
	{
		_wizardWindow->show();
		_wizardWindow->raise();
	} else {
		_wizardWindow.reset(new StrategyWizardWindow(_guiController, _strategyController));
		_wizardWindow->show();
	}
}
