/*
 * Copyright (c) 2018 ARM Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifndef __ARM_COMPUTE_NEUPSAMPLELAYERKERNEL_H__
#define __ARM_COMPUTE_NEUPSAMPLELAYERKERNEL_H__

#include "arm_compute/core/NEON/INEKernel.h"

namespace arm_compute
{
class ITensor;

/** Interface for the Upsample layer kernel.*/
class NEUpsampleLayerKernel : public INEKernel
{
public:
    const char *name() const override
    {
        return "NEUpsampleLayerKernel";
    }
    /** Default constructor */
    NEUpsampleLayerKernel();
    /** Prevent instances of this class from being copied (As this class contains pointers) */
    NEUpsampleLayerKernel(const NEUpsampleLayerKernel &) = delete;
    /** Prevent instances of this class from being copied (As this class contains pointers) */
    NEUpsampleLayerKernel &operator=(const NEUpsampleLayerKernel &) = delete;
    /** Default Move Constructor. */
    NEUpsampleLayerKernel(NEUpsampleLayerKernel &&) = default;
    /** Default move assignment operator */
    NEUpsampleLayerKernel &operator=(NEUpsampleLayerKernel &&) = default;
    /** Default destructor */
    ~NEUpsampleLayerKernel() = default;
    /** Set the input output tensors.
     *
     * @param[in]  input  Source tensor. Data types supported: QASYMM8/F16/F32.
     * @param[out] output Destination tensor. Data types supported: same as @p input.
     * @param[in]  info   Contains stride information described in @ref Size2D.
     * @param[in]  policy Defines the policy to fill the intermediate pixels.
     *
     */
    void configure(const ITensor *input, ITensor *output, const Size2D &info, const InterpolationPolicy policy);
    /** Static function to check if given info will lead to a valid configuration of @ref NEUpsampleLayerKernel
     *
     * @param[in] input  Source tensor info. Data types supported: QASYMM8/F16/F32.
     * @param[in] output Destination tensor info. Data types supported: same as @p input.
     * @param[in] info   Contains stride information described in @ref Size2D.
     * @param[in] policy Defines the policy to fill the intermediate pixels.
     *
     * @return a status
     */
    static Status validate(const ITensorInfo *input, const ITensorInfo *output, const Size2D &info, const InterpolationPolicy policy);

    // Inherited methods overridden:
    void run(const Window &window, const ThreadInfo &info) override;

private:
    /** Function to run upsample layer for FP32 (NCHW)
     *
     * @param[in] window Region on which to execute the kernel. (Must be a valid region of the window returned by window()).
     */
    void upsample_f32_nchw(const Window &window);
    /** Function to run upsample layer for FP32 (NHWC)
     *
     * @param[in] window Region on which to execute the kernel. (Must be a valid region of the window returned by window()).
     */
    void upsample_f32_nhwc(const Window &window);
    /** Function to run upsample layer for FP16 (NCHW)
     *
     * @param[in] window Region on which to execute the kernel. (Must be a valid region of the window returned by window()).
     */
    void upsample_f16_nchw(const Window &window);
    /** Function to run upsample layer for FP16 (NHWC)
     *
     * @param[in] window Region on which to execute the kernel. (Must be a valid region of the window returned by window()).
     */
    void upsample_f16_nhwc(const Window &window);
    /** Function to run upsample layer for QASYMM8 (NCHW)
     *
     * @param[in] window Region on which to execute the kernel. (Must be a valid region of the window returned by window()).
     */
    void upsample_qasymm8_nchw(const Window &window);
    /** Function to run upsample layer for QASYMM8 (NHWC)
     *
     * @param[in] window Region on which to execute the kernel. (Must be a valid region of the window returned by window()).
     */
    void upsample_qasymm8_nhwc(const Window &window);
    /** Common signature for all the upsample layer functions
     *
     * @param[in] window Region on which to execute the kernel.
     */
    using UpsampleFunctionPtr = void (NEUpsampleLayerKernel::*)(const Window &window);

private:
    UpsampleFunctionPtr _func;
    const ITensor      *_input;
    ITensor            *_output;
    Size2D              _info;
    unsigned int        _num_elems_processed_per_iteration_x;
};
} // namespace arm_compute
#endif /*__ARM_COMPUTE_NEUPSAMPLELAYERKERNEL_H__ */
