/*
 * Copyright (c) 2018-2019 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __ARM_FEATURE_SVE

#include <algorithm>

#include "arm_gemm.hpp"

#include "../../asmlib.hpp"
#include "../../utils.hpp"

namespace arm_gemm {

void sve_hybrid_fp16_mla_4VLx4(const __fp16 *A, int lda, const __fp16 *B, __fp16 *C, int ldc, int M, int N, int K, const __fp16 *bias, Activation act, bool append) {
    const int K_stride = K;
    const long loops_count = ((K + 8) / 16) - 1;
    K -= loops_count * 16;
    const long regs_count = (K / 8) - 1;
    K -= (regs_count + 1) * 8;
    const long leftovers = K;
    __fp16 nullbias[512];
    if (!append && !bias) {
        memset(nullbias, 0, (4 * get_vector_length<__fp16>() * sizeof(__fp16)));
    }
    __fp16 minval = - static_cast<__fp16>(std::numeric_limits<float>::infinity());
    __fp16 maxval =   static_cast<__fp16>(std::numeric_limits<float>::infinity());
    const __fp16 * const minptr = &minval;
    const __fp16 * const maxptr = &maxval;

    switch(act.type)
    {
        default:
        case Activation::Type::None:
            break;
        case Activation::Type::BoundedReLU:
            maxval = static_cast<__fp16>(act.param1);
            /* fall through */
        case Activation::Type::ReLU:
            minval = 0.0f;
            break;
    }

    for (int y=0; y<M; y+=4) {
        const __fp16 * const a_ptr0_base = A + (y * lda);
        const unsigned long ldab = lda * sizeof(__fp16);

        __fp16 *c_ptr0 = C + (y * ldc);

        for (int x0=0; x0<N; x0+=(4 * get_vector_length<__fp16>())) {
            const long width = std::min((unsigned long)N-x0, (4 * get_vector_length<__fp16>()));
            long loops = loops_count;
            long regs = regs_count;
            long temp = 0;
            long blocks = leftovers;
            const __fp16 *a_ptr0 = a_ptr0_base;
            const __fp16 *b_ptr0 = B + (K_stride * x0);
            const unsigned long ldcb = ldc * sizeof(__fp16);
            const __fp16 *biasptr = bias ? bias+x0 : nullbias;

            switch(M-y) {
                case 1:
                    __asm __volatile (
                        "whilelt p6.h, %[temp], %[leftovers]\n"
                        "whilelt p0.h, %[temp], %[width]\n"
                        "inch %[temp], all, mul #1\n"
                        "ptrue p7.h\n"
                        "whilelt p1.h, %[temp], %[width]\n"
                        "inch %[temp], all, mul #1\n"
                        "whilelt p2.h, %[temp], %[width]\n"
                        "inch %[temp], all, mul #1\n"
                        "whilelt p3.h, %[temp], %[width]\n"
                        "cbnz %[append], 1f\n"
                        "ld1h z16.h, p0/z, [%[biasptr]]\n"
                        "ld1h z17.h, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "ld1h z18.h, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "ld1h z19.h, p3/z, [%[biasptr], #3, MUL VL]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1h z16.h, p0/z, [%[c_ptr0]]\n"
                        "ld1h z17.h, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1h z18.h, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1h z19.h, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[7]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z17.h, z13.h, z0.h[7]\n"
                        "fmla z18.h, z14.h, z0.h[7]\n"
                        "fmla z19.h, z15.h, z0.h[7]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[0]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z9.h, z4.h[0]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[0]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[1]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0], #-0x10]\n"
                        "fmla z17.h, z13.h, z4.h[1]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z18.h, z14.h, z4.h[1]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[1]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[2]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z9.h, z4.h[2]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[2]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[2]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[3]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z17.h, z13.h, z4.h[3]\n"
                        "fmla z18.h, z14.h, z4.h[3]\n"
                        "fmla z19.h, z15.h, z4.h[3]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[4]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z9.h, z4.h[4]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[4]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[4]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[5]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z13.h, z4.h[5]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[5]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[5]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[6]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z9.h, z4.h[6]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[6]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[6]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[7]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z17.h, z13.h, z4.h[7]\n"
                        "fmla z18.h, z14.h, z4.h[7]\n"
                        "fmla z19.h, z15.h, z4.h[7]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[7]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z17.h, z13.h, z0.h[7]\n"
                        "fmla z18.h, z14.h, z0.h[7]\n"
                        "fmla z19.h, z15.h, z0.h[7]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[0]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z9.h, z4.h[0]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[0]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[1]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z13.h, z4.h[1]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[1]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[1]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[2]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z9.h, z4.h[2]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[2]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[2]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[3]\n"
                        "ld1rqh z0.h, p6/z, [%[a_ptr0], #0x10]\n"
                        "fmla z17.h, z13.h, z4.h[3]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z18.h, z14.h, z4.h[3]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        "fmla z19.h, z15.h, z4.h[3]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[4]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z9.h, z4.h[4]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[4]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[4]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[5]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z13.h, z4.h[5]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[5]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[5]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[6]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z9.h, z4.h[6]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[6]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[6]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[7]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z17.h, z13.h, z4.h[7]\n"
                        "fmla z18.h, z14.h, z4.h[7]\n"
                        "fmla z19.h, z15.h, z4.h[7]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "b.eq 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "b 5f\n"
                        "4:\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "ld1rqh z4.h, p6/z, [%[a_ptr0]]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[7]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z17.h, z13.h, z0.h[7]\n"
                        "fmla z18.h, z14.h, z0.h[7]\n"
                        "fmla z19.h, z15.h, z0.h[7]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z17.h, z9.h, z4.h[0]\n"
                        "fmla z18.h, z10.h, z4.h[0]\n"
                        "fmla z19.h, z11.h, z4.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[1]\n"
                        "fmla z18.h, z14.h, z4.h[1]\n"
                        "fmla z19.h, z15.h, z4.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[2]\n"
                        "fmla z17.h, z9.h, z4.h[2]\n"
                        "fmla z18.h, z10.h, z4.h[2]\n"
                        "fmla z19.h, z11.h, z4.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[3]\n"
                        "fmla z18.h, z14.h, z4.h[3]\n"
                        "fmla z19.h, z15.h, z4.h[3]\n"
                        "b.eq 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z17.h, z9.h, z4.h[4]\n"
                        "fmla z18.h, z10.h, z4.h[4]\n"
                        "fmla z19.h, z11.h, z4.h[4]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[5]\n"
                        "fmla z18.h, z14.h, z4.h[5]\n"
                        "fmla z19.h, z15.h, z4.h[5]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[6]\n"
                        "fmla z17.h, z9.h, z4.h[6]\n"
                        "fmla z18.h, z10.h, z4.h[6]\n"
                        "fmla z19.h, z11.h, z4.h[6]\n"
                        "5:\n"
                        "ld1rh z14.h, p7/z, [%[minptr]]\n"
                        "ld1rh z15.h, p7/z, [%[maxptr]]\n"
                        "fmax z16.h, p7/m, z16.h, z14.h\n"
                        "fmax z17.h, p7/m, z17.h, z14.h\n"
                        "fmax z18.h, p7/m, z18.h, z14.h\n"
                        "fmax z19.h, p7/m, z19.h, z14.h\n"
                        "fmin z16.h, p7/m, z16.h, z15.h\n"
                        "fmin z17.h, p7/m, z17.h, z15.h\n"
                        "fmin z18.h, p7/m, z18.h, z15.h\n"
                        "fmin z19.h, p7/m, z19.h, z15.h\n"
                        "st1h z16.h, p0, [%[c_ptr0]]\n"
                        "st1h z17.h, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "st1h z18.h, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "st1h z19.h, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [append] "r" (static_cast<uint64_t>(append)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                    );
                    break;
                case 2:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "c_ptr1 .req X1\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "whilelt p6.h, %[temp], %[leftovers]\n"
                        "whilelt p0.h, %[temp], %[width]\n"
                        "inch %[temp], all, mul #1\n"
                        "ptrue p7.h\n"
                        "whilelt p1.h, %[temp], %[width]\n"
                        "inch %[temp], all, mul #1\n"
                        "whilelt p2.h, %[temp], %[width]\n"
                        "inch %[temp], all, mul #1\n"
                        "whilelt p3.h, %[temp], %[width]\n"
                        "cbnz %[append], 1f\n"
                        "ld1h z16.h, p0/z, [%[biasptr]]\n"
                        "ld1h z17.h, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "ld1h z18.h, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "ld1h z19.h, p3/z, [%[biasptr], #3, MUL VL]\n"
                        "mov z20.d, z16.d\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "mov z21.d, z17.d\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "mov z22.d, z18.d\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "mov z23.d, z19.d\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1h z16.h, p0/z, [%[c_ptr0]]\n"
                        "ld1h z17.h, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1h z18.h, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1h z19.h, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1h z20.h, p0/z, [c_ptr1]\n"
                        "ld1h z21.h, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1h z22.h, p2/z, [c_ptr1, #2, MUL VL]\n"
                        "ld1h z23.h, p3/z, [c_ptr1, #3, MUL VL]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.h, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "ld1rqh z5.h, p7/z, [a_ptr1]\n"
                        "fmla z21.h, z9.h, z1.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z22.h, z10.h, z1.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "fmla z23.h, z11.h, z1.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        "fmla z20.h, z12.h, z1.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        "fmla z21.h, z13.h, z1.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "fmla z22.h, z14.h, z1.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "fmla z23.h, z15.h, z1.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z1.h[2]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "fmla z21.h, z9.h, z1.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z22.h, z10.h, z1.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "fmla z23.h, z11.h, z1.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "fmla z20.h, z12.h, z1.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "fmla z21.h, z13.h, z1.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "fmla z22.h, z14.h, z1.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "fmla z23.h, z15.h, z1.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "fmla z20.h, z8.h, z1.h[4]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "fmla z21.h, z9.h, z1.h[4]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "fmla z22.h, z10.h, z1.h[4]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "fmla z23.h, z11.h, z1.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "fmla z20.h, z12.h, z1.h[5]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "fmla z21.h, z13.h, z1.h[5]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "fmla z22.h, z14.h, z1.h[5]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "fmla z23.h, z15.h, z1.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z1.h[6]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "fmla z21.h, z9.h, z1.h[6]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z22.h, z10.h, z1.h[6]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "fmla z23.h, z11.h, z1.h[6]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[7]\n"
                        "fmla z20.h, z12.h, z1.h[7]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[7]\n"
                        "fmla z21.h, z13.h, z1.h[7]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[7]\n"
                        "fmla z22.h, z14.h, z1.h[7]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[7]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0], #-0x10]\n"
                        "fmla z23.h, z15.h, z1.h[7]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[0]\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1, #-0x10]\n"
                        "fmla z20.h, z8.h, z5.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z4.h[0]\n"
                        "fmla z21.h, z9.h, z5.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[0]\n"
                        "fmla z22.h, z10.h, z5.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[0]\n"
                        "fmla z23.h, z11.h, z5.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[1]\n"
                        "fmla z20.h, z12.h, z5.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[1]\n"
                        "fmla z21.h, z13.h, z5.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[1]\n"
                        "fmla z22.h, z14.h, z5.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[1]\n"
                        "fmla z23.h, z15.h, z5.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z5.h[2]\n"
                        "fmla z17.h, z9.h, z4.h[2]\n"
                        "fmla z21.h, z9.h, z5.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z22.h, z10.h, z5.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[2]\n"
                        "fmla z23.h, z11.h, z5.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[3]\n"
                        "fmla z20.h, z12.h, z5.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[3]\n"
                        "fmla z21.h, z13.h, z5.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[3]\n"
                        "fmla z22.h, z14.h, z5.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[3]\n"
                        "fmla z23.h, z15.h, z5.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[4]\n"
                        "fmla z20.h, z8.h, z5.h[4]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z4.h[4]\n"
                        "fmla z21.h, z9.h, z5.h[4]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[4]\n"
                        "fmla z22.h, z10.h, z5.h[4]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[4]\n"
                        "fmla z23.h, z11.h, z5.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[5]\n"
                        "fmla z20.h, z12.h, z5.h[5]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[5]\n"
                        "fmla z21.h, z13.h, z5.h[5]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[5]\n"
                        "fmla z22.h, z14.h, z5.h[5]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[5]\n"
                        "fmla z23.h, z15.h, z5.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[6]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z5.h[6]\n"
                        "fmla z17.h, z9.h, z4.h[6]\n"
                        "fmla z21.h, z9.h, z5.h[6]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[6]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z22.h, z10.h, z5.h[6]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[6]\n"
                        "fmla z23.h, z11.h, z5.h[6]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[7]\n"
                        "fmla z20.h, z12.h, z5.h[7]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[7]\n"
                        "fmla z21.h, z13.h, z5.h[7]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[7]\n"
                        "fmla z22.h, z14.h, z5.h[7]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[7]\n"
                        "fmla z23.h, z15.h, z5.h[7]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.h, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "ld1rqh z5.h, p7/z, [a_ptr1]\n"
                        "fmla z21.h, z9.h, z1.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z22.h, z10.h, z1.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "fmla z23.h, z11.h, z1.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "fmla z20.h, z12.h, z1.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "fmla z21.h, z13.h, z1.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "fmla z22.h, z14.h, z1.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "fmla z23.h, z15.h, z1.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z1.h[2]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "fmla z21.h, z9.h, z1.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z22.h, z10.h, z1.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "fmla z23.h, z11.h, z1.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "fmla z20.h, z12.h, z1.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "fmla z21.h, z13.h, z1.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "fmla z22.h, z14.h, z1.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "fmla z23.h, z15.h, z1.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "fmla z20.h, z8.h, z1.h[4]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "fmla z21.h, z9.h, z1.h[4]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "fmla z22.h, z10.h, z1.h[4]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "fmla z23.h, z11.h, z1.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "fmla z20.h, z12.h, z1.h[5]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "fmla z21.h, z13.h, z1.h[5]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "fmla z22.h, z14.h, z1.h[5]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "fmla z23.h, z15.h, z1.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z1.h[6]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "fmla z21.h, z9.h, z1.h[6]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z22.h, z10.h, z1.h[6]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "fmla z23.h, z11.h, z1.h[6]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[7]\n"
                        "fmla z20.h, z12.h, z1.h[7]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[7]\n"
                        "fmla z21.h, z13.h, z1.h[7]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[7]\n"
                        "fmla z22.h, z14.h, z1.h[7]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[7]\n"
                        "ld1rqh z0.h, p6/z, [%[a_ptr0], #0x10]\n"
                        "fmla z23.h, z15.h, z1.h[7]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[0]\n"
                        "ld1rqh z1.h, p6/z, [a_ptr1, #0x10]\n"
                        "fmla z20.h, z8.h, z5.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z4.h[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        "fmla z21.h, z9.h, z5.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[0]\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        "fmla z22.h, z10.h, z5.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[0]\n"
                        "fmla z23.h, z11.h, z5.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[1]\n"
                        "fmla z20.h, z12.h, z5.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[1]\n"
                        "fmla z21.h, z13.h, z5.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[1]\n"
                        "fmla z22.h, z14.h, z5.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[1]\n"
                        "fmla z23.h, z15.h, z5.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z5.h[2]\n"
                        "fmla z17.h, z9.h, z4.h[2]\n"
                        "fmla z21.h, z9.h, z5.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z22.h, z10.h, z5.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[2]\n"
                        "fmla z23.h, z11.h, z5.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[3]\n"
                        "fmla z20.h, z12.h, z5.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[3]\n"
                        "fmla z21.h, z13.h, z5.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[3]\n"
                        "fmla z22.h, z14.h, z5.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[3]\n"
                        "fmla z23.h, z15.h, z5.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[4]\n"
                        "fmla z20.h, z8.h, z5.h[4]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z4.h[4]\n"
                        "fmla z21.h, z9.h, z5.h[4]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[4]\n"
                        "fmla z22.h, z10.h, z5.h[4]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[4]\n"
                        "fmla z23.h, z11.h, z5.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[5]\n"
                        "fmla z20.h, z12.h, z5.h[5]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[5]\n"
                        "fmla z21.h, z13.h, z5.h[5]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[5]\n"
                        "fmla z22.h, z14.h, z5.h[5]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[5]\n"
                        "fmla z23.h, z15.h, z5.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[6]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z20.h, z8.h, z5.h[6]\n"
                        "fmla z17.h, z9.h, z4.h[6]\n"
                        "fmla z21.h, z9.h, z5.h[6]\n"
                        "fmla z18.h, z10.h, z4.h[6]\n"
                        "fmla z22.h, z10.h, z5.h[6]\n"
                        "fmla z19.h, z11.h, z4.h[6]\n"
                        "fmla z23.h, z11.h, z5.h[6]\n"
                        "fmla z16.h, z12.h, z4.h[7]\n"
                        "fmla z20.h, z12.h, z5.h[7]\n"
                        "fmla z17.h, z13.h, z4.h[7]\n"
                        "fmla z21.h, z13.h, z5.h[7]\n"
                        "fmla z18.h, z14.h, z4.h[7]\n"
                        "fmla z22.h, z14.h, z5.h[7]\n"
                        "fmla z19.h, z15.h, z4.h[7]\n"
                        "fmla z23.h, z15.h, z5.h[7]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.h, z8.h, z1.h[0]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "fmla z21.h, z9.h, z1.h[0]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "fmla z22.h, z10.h, z1.h[0]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "fmla z23.h, z11.h, z1.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.h, z12.h, z1.h[1]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "fmla z21.h, z13.h, z1.h[1]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "fmla z22.h, z14.h, z1.h[1]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "fmla z23.h, z15.h, z1.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "fmla z20.h, z8.h, z1.h[2]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "fmla z21.h, z9.h, z1.h[2]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "fmla z22.h, z10.h, z1.h[2]\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "fmla z23.h, z11.h, z1.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.h, z12.h, z1.h[3]\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "fmla z21.h, z13.h, z1.h[3]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "fmla z22.h, z14.h, z1.h[3]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "fmla z23.h, z15.h, z1.h[3]\n"
                        "b.eq 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.h, z8.h, z1.h[4]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "fmla z21.h, z9.h, z1.h[4]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "fmla z22.h, z10.h, z1.h[4]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "fmla z23.h, z11.h, z1.h[4]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.h, z12.h, z1.h[5]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "fmla z21.h, z13.h, z1.h[5]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "fmla z22.h, z14.h, z1.h[5]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "fmla z23.h, z15.h, z1.h[5]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "fmla z20.h, z8.h, z1.h[6]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "fmla z21.h, z9.h, z1.h[6]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "fmla z22.h, z10.h, z1.h[6]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "fmla z23.h, z11.h, z1.h[6]\n"
                        "b 5f\n"
                        "4:\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.h, z8.h, z1.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "ld1rqh z4.h, p6/z, [%[a_ptr0]]\n"
                        "fmla z21.h, z9.h, z1.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "ld1rqh z5.h, p6/z, [a_ptr1]\n"
                        "fmla z22.h, z10.h, z1.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        "fmla z23.h, z11.h, z1.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        "fmla z20.h, z12.h, z1.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "fmla z21.h, z13.h, z1.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "fmla z22.h, z14.h, z1.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "fmla z23.h, z15.h, z1.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z1.h[2]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "fmla z21.h, z9.h, z1.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z22.h, z10.h, z1.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "fmla z23.h, z11.h, z1.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "fmla z20.h, z12.h, z1.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "fmla z21.h, z13.h, z1.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "fmla z22.h, z14.h, z1.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "fmla z23.h, z15.h, z1.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "fmla z20.h, z8.h, z1.h[4]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "fmla z21.h, z9.h, z1.h[4]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "fmla z22.h, z10.h, z1.h[4]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "fmla z23.h, z11.h, z1.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "fmla z20.h, z12.h, z1.h[5]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "fmla z21.h, z13.h, z1.h[5]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "fmla z22.h, z14.h, z1.h[5]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "fmla z23.h, z15.h, z1.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z20.h, z8.h, z1.h[6]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "fmla z21.h, z9.h, z1.h[6]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "fmla z22.h, z10.h, z1.h[6]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "fmla z23.h, z11.h, z1.h[6]\n"
                        "fmla z16.h, z12.h, z0.h[7]\n"
                        "fmla z20.h, z12.h, z1.h[7]\n"
                        "fmla z17.h, z13.h, z0.h[7]\n"
                        "fmla z21.h, z13.h, z1.h[7]\n"
                        "fmla z18.h, z14.h, z0.h[7]\n"
                        "fmla z22.h, z14.h, z1.h[7]\n"
                        "fmla z19.h, z15.h, z0.h[7]\n"
                        "fmla z23.h, z15.h, z1.h[7]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.h, z8.h, z5.h[0]\n"
                        "fmla z17.h, z9.h, z4.h[0]\n"
                        "fmla z21.h, z9.h, z5.h[0]\n"
                        "fmla z18.h, z10.h, z4.h[0]\n"
                        "fmla z22.h, z10.h, z5.h[0]\n"
                        "fmla z19.h, z11.h, z4.h[0]\n"
                        "fmla z23.h, z11.h, z5.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.h, z12.h, z5.h[1]\n"
                        "fmla z17.h, z13.h, z4.h[1]\n"
                        "fmla z21.h, z13.h, z5.h[1]\n"
                        "fmla z18.h, z14.h, z4.h[1]\n"
                        "fmla z22.h, z14.h, z5.h[1]\n"
                        "fmla z19.h, z15.h, z4.h[1]\n"
                        "fmla z23.h, z15.h, z5.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[2]\n"
                        "fmla z20.h, z8.h, z5.h[2]\n"
                        "fmla z17.h, z9.h, z4.h[2]\n"
                        "fmla z21.h, z9.h, z5.h[2]\n"
                        "fmla z18.h, z10.h, z4.h[2]\n"
                        "fmla z22.h, z10.h, z5.h[2]\n"
                        "fmla z19.h, z11.h, z4.h[2]\n"
                        "fmla z23.h, z11.h, z5.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.h, z12.h, z5.h[3]\n"
                        "fmla z17.h, z13.h, z4.h[3]\n"
                        "fmla z21.h, z13.h, z5.h[3]\n"
                        "fmla z18.h, z14.h, z4.h[3]\n"
                        "fmla z22.h, z14.h, z5.h[3]\n"
                        "fmla z19.h, z15.h, z4.h[3]\n"
                        "fmla z23.h, z15.h, z5.h[3]\n"
                        "b.eq 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.h, z8.h, z5.h[4]\n"
                        "fmla z17.h, z9.h, z4.h[4]\n"
                        "fmla z21.h, z9.h, z5.h[4]\n"
                        "fmla z18.h, z10.h, z4.h[4]\n"
                        "fmla z22.h, z10.h, z5.h[4]\n"
                        "fmla z19.h, z11.h, z4.h[4]\n"
                        "fmla z23.h, z11.h, z5.h[4]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.h, z12.h, z5.h[5]\n"
                        "fmla z17.h, z13.h, z4.h[5]\n"
                        "fmla z21.h, z13.h, z5.h[5]\n"
                        "fmla z18.h, z14.h, z4.h[5]\n"
                        "fmla z22.h, z14.h, z5.h[5]\n"
                        "fmla z19.h, z15.h, z4.h[5]\n"
                        "fmla z23.h, z15.h, z5.h[5]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[6]\n"
                        "fmla z20.h, z8.h, z5.h[6]\n"
                        "fmla z17.h, z9.h, z4.h[6]\n"
                        "fmla z21.h, z9.h, z5.h[6]\n"
                        "fmla z18.h, z10.h, z4.h[6]\n"
                        "fmla z22.h, z10.h, z5.h[6]\n"
                        "fmla z19.h, z11.h, z4.h[6]\n"
                        "fmla z23.h, z11.h, z5.h[6]\n"
                        "5:\n"
                        "ld1rh z14.h, p7/z, [%[minptr]]\n"
                        "ld1rh z15.h, p7/z, [%[maxptr]]\n"
                        "fmax z16.h, p7/m, z16.h, z14.h\n"
                        "fmax z17.h, p7/m, z17.h, z14.h\n"
                        "fmax z18.h, p7/m, z18.h, z14.h\n"
                        "fmax z19.h, p7/m, z19.h, z14.h\n"
                        "fmin z16.h, p7/m, z16.h, z15.h\n"
                        "fmin z17.h, p7/m, z17.h, z15.h\n"
                        "fmin z18.h, p7/m, z18.h, z15.h\n"
                        "fmin z19.h, p7/m, z19.h, z15.h\n"
                        "st1h z16.h, p0, [%[c_ptr0]]\n"
                        "fmax z20.h, p7/m, z20.h, z14.h\n"
                        "fmax z21.h, p7/m, z21.h, z14.h\n"
                        "fmax z22.h, p7/m, z22.h, z14.h\n"
                        "st1h z17.h, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmax z23.h, p7/m, z23.h, z14.h\n"
                        "fmin z20.h, p7/m, z20.h, z15.h\n"
                        "fmin z21.h, p7/m, z21.h, z15.h\n"
                        "st1h z18.h, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "fmin z22.h, p7/m, z22.h, z15.h\n"
                        "fmin z23.h, p7/m, z23.h, z15.h\n"
                        "st1h z19.h, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        "st1h z20.h, p0, [c_ptr1]\n"
                        "st1h z21.h, p1, [c_ptr1, #1, MUL VL]\n"
                        "st1h z22.h, p2, [c_ptr1, #2, MUL VL]\n"
                        "st1h z23.h, p3, [c_ptr1, #3, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq c_ptr1\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [append] "r" (static_cast<uint64_t>(append)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "cc", "memory"
                    );
                    break;
                case 3:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "c_ptr1 .req X2\n"
                        "c_ptr2 .req X3\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "whilelt p6.h, %[temp], %[leftovers]\n"
                        "whilelt p0.h, %[temp], %[width]\n"
                        "inch %[temp], all, mul #1\n"
                        "ptrue p7.h\n"
                        "whilelt p1.h, %[temp], %[width]\n"
                        "inch %[temp], all, mul #1\n"
                        "whilelt p2.h, %[temp], %[width]\n"
                        "inch %[temp], all, mul #1\n"
                        "whilelt p3.h, %[temp], %[width]\n"
                        "cbnz %[append], 1f\n"
                        "ld1h z16.h, p0/z, [%[biasptr]]\n"
                        "ld1h z17.h, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "ld1h z18.h, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "ld1h z19.h, p3/z, [%[biasptr], #3, MUL VL]\n"
                        "mov z20.d, z16.d\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "mov z21.d, z17.d\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "mov z22.d, z18.d\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2]\n"
                        "mov z23.d, z19.d\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "mov z24.d, z16.d\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "mov z25.d, z17.d\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "mov z26.d, z18.d\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z27.d, z19.d\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1h z16.h, p0/z, [%[c_ptr0]]\n"
                        "ld1h z17.h, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1h z18.h, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1h z19.h, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1h z20.h, p0/z, [c_ptr1]\n"
                        "ld1h z21.h, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1h z22.h, p2/z, [c_ptr1, #2, MUL VL]\n"
                        "ld1h z23.h, p3/z, [c_ptr1, #3, MUL VL]\n"
                        "ld1h z24.h, p0/z, [c_ptr2]\n"
                        "ld1h z25.h, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "ld1h z26.h, p2/z, [c_ptr2, #2, MUL VL]\n"
                        "ld1h z27.h, p3/z, [c_ptr2, #3, MUL VL]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.h, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        "fmla z24.h, z8.h, z2.h[0]\n"
                        "ld1rqh z5.h, p7/z, [a_ptr1]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "ld1rqh z6.h, p7/z, [a_ptr2]\n"
                        "fmla z21.h, z9.h, z1.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z25.h, z9.h, z2.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "fmla z22.h, z10.h, z1.h[0]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        "fmla z26.h, z10.h, z2.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        "fmla z23.h, z11.h, z1.h[0]\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        "fmla z27.h, z11.h, z2.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "fmla z20.h, z12.h, z1.h[1]\n"
                        "fmla z24.h, z12.h, z2.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "fmla z21.h, z13.h, z1.h[1]\n"
                        "fmla z25.h, z13.h, z2.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "fmla z22.h, z14.h, z1.h[1]\n"
                        "fmla z26.h, z14.h, z2.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "fmla z23.h, z15.h, z1.h[1]\n"
                        "fmla z27.h, z15.h, z2.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z1.h[2]\n"
                        "fmla z24.h, z8.h, z2.h[2]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z21.h, z9.h, z1.h[2]\n"
                        "fmla z25.h, z9.h, z2.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "fmla z22.h, z10.h, z1.h[2]\n"
                        "fmla z26.h, z10.h, z2.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "fmla z23.h, z11.h, z1.h[2]\n"
                        "fmla z27.h, z11.h, z2.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "fmla z20.h, z12.h, z1.h[3]\n"
                        "fmla z24.h, z12.h, z2.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "fmla z21.h, z13.h, z1.h[3]\n"
                        "fmla z25.h, z13.h, z2.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "fmla z22.h, z14.h, z1.h[3]\n"
                        "fmla z26.h, z14.h, z2.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "fmla z23.h, z15.h, z1.h[3]\n"
                        "fmla z27.h, z15.h, z2.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "fmla z20.h, z8.h, z1.h[4]\n"
                        "fmla z24.h, z8.h, z2.h[4]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "fmla z21.h, z9.h, z1.h[4]\n"
                        "fmla z25.h, z9.h, z2.h[4]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "fmla z22.h, z10.h, z1.h[4]\n"
                        "fmla z26.h, z10.h, z2.h[4]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "fmla z23.h, z11.h, z1.h[4]\n"
                        "fmla z27.h, z11.h, z2.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "fmla z20.h, z12.h, z1.h[5]\n"
                        "fmla z24.h, z12.h, z2.h[5]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "fmla z21.h, z13.h, z1.h[5]\n"
                        "fmla z25.h, z13.h, z2.h[5]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "fmla z22.h, z14.h, z1.h[5]\n"
                        "fmla z26.h, z14.h, z2.h[5]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "fmla z23.h, z15.h, z1.h[5]\n"
                        "fmla z27.h, z15.h, z2.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z1.h[6]\n"
                        "fmla z24.h, z8.h, z2.h[6]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z21.h, z9.h, z1.h[6]\n"
                        "fmla z25.h, z9.h, z2.h[6]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "fmla z22.h, z10.h, z1.h[6]\n"
                        "fmla z26.h, z10.h, z2.h[6]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "fmla z23.h, z11.h, z1.h[6]\n"
                        "fmla z27.h, z11.h, z2.h[6]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[7]\n"
                        "fmla z20.h, z12.h, z1.h[7]\n"
                        "fmla z24.h, z12.h, z2.h[7]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[7]\n"
                        "fmla z21.h, z13.h, z1.h[7]\n"
                        "fmla z25.h, z13.h, z2.h[7]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[7]\n"
                        "fmla z22.h, z14.h, z1.h[7]\n"
                        "fmla z26.h, z14.h, z2.h[7]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[7]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0], #-0x10]\n"
                        "fmla z23.h, z15.h, z1.h[7]\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1, #-0x10]\n"
                        "fmla z27.h, z15.h, z2.h[7]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[0]\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2, #-0x10]\n"
                        "fmla z20.h, z8.h, z5.h[0]\n"
                        "fmla z24.h, z8.h, z6.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z4.h[0]\n"
                        "fmla z21.h, z9.h, z5.h[0]\n"
                        "fmla z25.h, z9.h, z6.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[0]\n"
                        "fmla z22.h, z10.h, z5.h[0]\n"
                        "fmla z26.h, z10.h, z6.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[0]\n"
                        "fmla z23.h, z11.h, z5.h[0]\n"
                        "fmla z27.h, z11.h, z6.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[1]\n"
                        "fmla z20.h, z12.h, z5.h[1]\n"
                        "fmla z24.h, z12.h, z6.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[1]\n"
                        "fmla z21.h, z13.h, z5.h[1]\n"
                        "fmla z25.h, z13.h, z6.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[1]\n"
                        "fmla z22.h, z14.h, z5.h[1]\n"
                        "fmla z26.h, z14.h, z6.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[1]\n"
                        "fmla z23.h, z15.h, z5.h[1]\n"
                        "fmla z27.h, z15.h, z6.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z5.h[2]\n"
                        "fmla z24.h, z8.h, z6.h[2]\n"
                        "fmla z17.h, z9.h, z4.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z21.h, z9.h, z5.h[2]\n"
                        "fmla z25.h, z9.h, z6.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[2]\n"
                        "fmla z22.h, z10.h, z5.h[2]\n"
                        "fmla z26.h, z10.h, z6.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[2]\n"
                        "fmla z23.h, z11.h, z5.h[2]\n"
                        "fmla z27.h, z11.h, z6.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[3]\n"
                        "fmla z20.h, z12.h, z5.h[3]\n"
                        "fmla z24.h, z12.h, z6.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[3]\n"
                        "fmla z21.h, z13.h, z5.h[3]\n"
                        "fmla z25.h, z13.h, z6.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[3]\n"
                        "fmla z22.h, z14.h, z5.h[3]\n"
                        "fmla z26.h, z14.h, z6.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[3]\n"
                        "fmla z23.h, z15.h, z5.h[3]\n"
                        "fmla z27.h, z15.h, z6.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[4]\n"
                        "fmla z20.h, z8.h, z5.h[4]\n"
                        "fmla z24.h, z8.h, z6.h[4]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z4.h[4]\n"
                        "fmla z21.h, z9.h, z5.h[4]\n"
                        "fmla z25.h, z9.h, z6.h[4]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[4]\n"
                        "fmla z22.h, z10.h, z5.h[4]\n"
                        "fmla z26.h, z10.h, z6.h[4]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[4]\n"
                        "fmla z23.h, z11.h, z5.h[4]\n"
                        "fmla z27.h, z11.h, z6.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[5]\n"
                        "fmla z20.h, z12.h, z5.h[5]\n"
                        "fmla z24.h, z12.h, z6.h[5]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[5]\n"
                        "fmla z21.h, z13.h, z5.h[5]\n"
                        "fmla z25.h, z13.h, z6.h[5]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[5]\n"
                        "fmla z22.h, z14.h, z5.h[5]\n"
                        "fmla z26.h, z14.h, z6.h[5]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[5]\n"
                        "fmla z23.h, z15.h, z5.h[5]\n"
                        "fmla z27.h, z15.h, z6.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[6]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z5.h[6]\n"
                        "fmla z24.h, z8.h, z6.h[6]\n"
                        "fmla z17.h, z9.h, z4.h[6]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z21.h, z9.h, z5.h[6]\n"
                        "fmla z25.h, z9.h, z6.h[6]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[6]\n"
                        "fmla z22.h, z10.h, z5.h[6]\n"
                        "fmla z26.h, z10.h, z6.h[6]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[6]\n"
                        "fmla z23.h, z11.h, z5.h[6]\n"
                        "fmla z27.h, z11.h, z6.h[6]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[7]\n"
                        "fmla z20.h, z12.h, z5.h[7]\n"
                        "fmla z24.h, z12.h, z6.h[7]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[7]\n"
                        "fmla z21.h, z13.h, z5.h[7]\n"
                        "fmla z25.h, z13.h, z6.h[7]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[7]\n"
                        "fmla z22.h, z14.h, z5.h[7]\n"
                        "fmla z26.h, z14.h, z6.h[7]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[7]\n"
                        "fmla z23.h, z15.h, z5.h[7]\n"
                        "fmla z27.h, z15.h, z6.h[7]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.h, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        "fmla z24.h, z8.h, z2.h[0]\n"
                        "ld1rqh z5.h, p7/z, [a_ptr1]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "ld1rqh z6.h, p7/z, [a_ptr2]\n"
                        "fmla z21.h, z9.h, z1.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z25.h, z9.h, z2.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "fmla z22.h, z10.h, z1.h[0]\n"
                        "fmla z26.h, z10.h, z2.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "fmla z23.h, z11.h, z1.h[0]\n"
                        "fmla z27.h, z11.h, z2.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "fmla z20.h, z12.h, z1.h[1]\n"
                        "fmla z24.h, z12.h, z2.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "fmla z21.h, z13.h, z1.h[1]\n"
                        "fmla z25.h, z13.h, z2.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "fmla z22.h, z14.h, z1.h[1]\n"
                        "fmla z26.h, z14.h, z2.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "fmla z23.h, z15.h, z1.h[1]\n"
                        "fmla z27.h, z15.h, z2.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z1.h[2]\n"
                        "fmla z24.h, z8.h, z2.h[2]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z21.h, z9.h, z1.h[2]\n"
                        "fmla z25.h, z9.h, z2.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "fmla z22.h, z10.h, z1.h[2]\n"
                        "fmla z26.h, z10.h, z2.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "fmla z23.h, z11.h, z1.h[2]\n"
                        "fmla z27.h, z11.h, z2.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "fmla z20.h, z12.h, z1.h[3]\n"
                        "fmla z24.h, z12.h, z2.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "fmla z21.h, z13.h, z1.h[3]\n"
                        "fmla z25.h, z13.h, z2.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "fmla z22.h, z14.h, z1.h[3]\n"
                        "fmla z26.h, z14.h, z2.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "fmla z23.h, z15.h, z1.h[3]\n"
                        "fmla z27.h, z15.h, z2.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "fmla z20.h, z8.h, z1.h[4]\n"
                        "fmla z24.h, z8.h, z2.h[4]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "fmla z21.h, z9.h, z1.h[4]\n"
                        "fmla z25.h, z9.h, z2.h[4]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "fmla z22.h, z10.h, z1.h[4]\n"
                        "fmla z26.h, z10.h, z2.h[4]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "fmla z23.h, z11.h, z1.h[4]\n"
                        "fmla z27.h, z11.h, z2.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "fmla z20.h, z12.h, z1.h[5]\n"
                        "fmla z24.h, z12.h, z2.h[5]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "fmla z21.h, z13.h, z1.h[5]\n"
                        "fmla z25.h, z13.h, z2.h[5]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "fmla z22.h, z14.h, z1.h[5]\n"
                        "fmla z26.h, z14.h, z2.h[5]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "fmla z23.h, z15.h, z1.h[5]\n"
                        "fmla z27.h, z15.h, z2.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z1.h[6]\n"
                        "fmla z24.h, z8.h, z2.h[6]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z21.h, z9.h, z1.h[6]\n"
                        "fmla z25.h, z9.h, z2.h[6]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "fmla z22.h, z10.h, z1.h[6]\n"
                        "fmla z26.h, z10.h, z2.h[6]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "fmla z23.h, z11.h, z1.h[6]\n"
                        "fmla z27.h, z11.h, z2.h[6]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[7]\n"
                        "fmla z20.h, z12.h, z1.h[7]\n"
                        "fmla z24.h, z12.h, z2.h[7]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[7]\n"
                        "fmla z21.h, z13.h, z1.h[7]\n"
                        "fmla z25.h, z13.h, z2.h[7]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[7]\n"
                        "fmla z22.h, z14.h, z1.h[7]\n"
                        "fmla z26.h, z14.h, z2.h[7]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[7]\n"
                        "ld1rqh z0.h, p6/z, [%[a_ptr0], #0x10]\n"
                        "fmla z23.h, z15.h, z1.h[7]\n"
                        "ld1rqh z1.h, p6/z, [a_ptr1, #0x10]\n"
                        "fmla z27.h, z15.h, z2.h[7]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[0]\n"
                        "ld1rqh z2.h, p6/z, [a_ptr2, #0x10]\n"
                        "fmla z20.h, z8.h, z5.h[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        "fmla z24.h, z8.h, z6.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z4.h[0]\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        "fmla z21.h, z9.h, z5.h[0]\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        "fmla z25.h, z9.h, z6.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[0]\n"
                        "fmla z22.h, z10.h, z5.h[0]\n"
                        "fmla z26.h, z10.h, z6.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[0]\n"
                        "fmla z23.h, z11.h, z5.h[0]\n"
                        "fmla z27.h, z11.h, z6.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[1]\n"
                        "fmla z20.h, z12.h, z5.h[1]\n"
                        "fmla z24.h, z12.h, z6.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[1]\n"
                        "fmla z21.h, z13.h, z5.h[1]\n"
                        "fmla z25.h, z13.h, z6.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[1]\n"
                        "fmla z22.h, z14.h, z5.h[1]\n"
                        "fmla z26.h, z14.h, z6.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[1]\n"
                        "fmla z23.h, z15.h, z5.h[1]\n"
                        "fmla z27.h, z15.h, z6.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z5.h[2]\n"
                        "fmla z24.h, z8.h, z6.h[2]\n"
                        "fmla z17.h, z9.h, z4.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z21.h, z9.h, z5.h[2]\n"
                        "fmla z25.h, z9.h, z6.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[2]\n"
                        "fmla z22.h, z10.h, z5.h[2]\n"
                        "fmla z26.h, z10.h, z6.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[2]\n"
                        "fmla z23.h, z11.h, z5.h[2]\n"
                        "fmla z27.h, z11.h, z6.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[3]\n"
                        "fmla z20.h, z12.h, z5.h[3]\n"
                        "fmla z24.h, z12.h, z6.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[3]\n"
                        "fmla z21.h, z13.h, z5.h[3]\n"
                        "fmla z25.h, z13.h, z6.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[3]\n"
                        "fmla z22.h, z14.h, z5.h[3]\n"
                        "fmla z26.h, z14.h, z6.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[3]\n"
                        "fmla z23.h, z15.h, z5.h[3]\n"
                        "fmla z27.h, z15.h, z6.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[4]\n"
                        "fmla z20.h, z8.h, z5.h[4]\n"
                        "fmla z24.h, z8.h, z6.h[4]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z4.h[4]\n"
                        "fmla z21.h, z9.h, z5.h[4]\n"
                        "fmla z25.h, z9.h, z6.h[4]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[4]\n"
                        "fmla z22.h, z10.h, z5.h[4]\n"
                        "fmla z26.h, z10.h, z6.h[4]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[4]\n"
                        "fmla z23.h, z11.h, z5.h[4]\n"
                        "fmla z27.h, z11.h, z6.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[5]\n"
                        "fmla z20.h, z12.h, z5.h[5]\n"
                        "fmla z24.h, z12.h, z6.h[5]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[5]\n"
                        "fmla z21.h, z13.h, z5.h[5]\n"
                        "fmla z25.h, z13.h, z6.h[5]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[5]\n"
                        "fmla z22.h, z14.h, z5.h[5]\n"
                        "fmla z26.h, z14.h, z6.h[5]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[5]\n"
                        "fmla z23.h, z15.h, z5.h[5]\n"
                        "fmla z27.h, z15.h, z6.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[6]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z20.h, z8.h, z5.h[6]\n"
                        "fmla z24.h, z8.h, z6.h[6]\n"
                        "fmla z17.h, z9.h, z4.h[6]\n"
                        "fmla z21.h, z9.h, z5.h[6]\n"
                        "fmla z25.h, z9.h, z6.h[6]\n"
                        "fmla z18.h, z10.h, z4.h[6]\n"
                        "fmla z22.h, z10.h, z5.h[6]\n"
                        "fmla z26.h, z10.h, z6.h[6]\n"
                        "fmla z19.h, z11.h, z4.h[6]\n"
                        "fmla z23.h, z11.h, z5.h[6]\n"
                        "fmla z27.h, z11.h, z6.h[6]\n"
                        "fmla z16.h, z12.h, z4.h[7]\n"
                        "fmla z20.h, z12.h, z5.h[7]\n"
                        "fmla z24.h, z12.h, z6.h[7]\n"
                        "fmla z17.h, z13.h, z4.h[7]\n"
                        "fmla z21.h, z13.h, z5.h[7]\n"
                        "fmla z25.h, z13.h, z6.h[7]\n"
                        "fmla z18.h, z14.h, z4.h[7]\n"
                        "fmla z22.h, z14.h, z5.h[7]\n"
                        "fmla z26.h, z14.h, z6.h[7]\n"
                        "fmla z19.h, z15.h, z4.h[7]\n"
                        "fmla z23.h, z15.h, z5.h[7]\n"
                        "fmla z27.h, z15.h, z6.h[7]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.h, z8.h, z1.h[0]\n"
                        "fmla z24.h, z8.h, z2.h[0]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "fmla z21.h, z9.h, z1.h[0]\n"
                        "fmla z25.h, z9.h, z2.h[0]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "fmla z22.h, z10.h, z1.h[0]\n"
                        "fmla z26.h, z10.h, z2.h[0]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "fmla z23.h, z11.h, z1.h[0]\n"
                        "fmla z27.h, z11.h, z2.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.h, z12.h, z1.h[1]\n"
                        "fmla z24.h, z12.h, z2.h[1]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "fmla z21.h, z13.h, z1.h[1]\n"
                        "fmla z25.h, z13.h, z2.h[1]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "fmla z22.h, z14.h, z1.h[1]\n"
                        "fmla z26.h, z14.h, z2.h[1]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "fmla z23.h, z15.h, z1.h[1]\n"
                        "fmla z27.h, z15.h, z2.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "fmla z20.h, z8.h, z1.h[2]\n"
                        "fmla z24.h, z8.h, z2.h[2]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "fmla z21.h, z9.h, z1.h[2]\n"
                        "fmla z25.h, z9.h, z2.h[2]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "fmla z22.h, z10.h, z1.h[2]\n"
                        "fmla z26.h, z10.h, z2.h[2]\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "fmla z23.h, z11.h, z1.h[2]\n"
                        "fmla z27.h, z11.h, z2.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.h, z12.h, z1.h[3]\n"
                        "fmla z24.h, z12.h, z2.h[3]\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "fmla z21.h, z13.h, z1.h[3]\n"
                        "fmla z25.h, z13.h, z2.h[3]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "fmla z22.h, z14.h, z1.h[3]\n"
                        "fmla z26.h, z14.h, z2.h[3]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "fmla z23.h, z15.h, z1.h[3]\n"
                        "fmla z27.h, z15.h, z2.h[3]\n"
                        "b.eq 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.h, z8.h, z1.h[4]\n"
                        "fmla z24.h, z8.h, z2.h[4]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "fmla z21.h, z9.h, z1.h[4]\n"
                        "fmla z25.h, z9.h, z2.h[4]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "fmla z22.h, z10.h, z1.h[4]\n"
                        "fmla z26.h, z10.h, z2.h[4]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "fmla z23.h, z11.h, z1.h[4]\n"
                        "fmla z27.h, z11.h, z2.h[4]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.h, z12.h, z1.h[5]\n"
                        "fmla z24.h, z12.h, z2.h[5]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "fmla z21.h, z13.h, z1.h[5]\n"
                        "fmla z25.h, z13.h, z2.h[5]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "fmla z22.h, z14.h, z1.h[5]\n"
                        "fmla z26.h, z14.h, z2.h[5]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "fmla z23.h, z15.h, z1.h[5]\n"
                        "fmla z27.h, z15.h, z2.h[5]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "fmla z20.h, z8.h, z1.h[6]\n"
                        "fmla z24.h, z8.h, z2.h[6]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "fmla z21.h, z9.h, z1.h[6]\n"
                        "fmla z25.h, z9.h, z2.h[6]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "fmla z22.h, z10.h, z1.h[6]\n"
                        "fmla z26.h, z10.h, z2.h[6]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "fmla z23.h, z11.h, z1.h[6]\n"
                        "fmla z27.h, z11.h, z2.h[6]\n"
                        "b 5f\n"
                        "4:\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.h, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p6/z, [%[a_ptr0]]\n"
                        "fmla z24.h, z8.h, z2.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "ld1rqh z5.h, p6/z, [a_ptr1]\n"
                        "fmla z21.h, z9.h, z1.h[0]\n"
                        "ld1rqh z6.h, p6/z, [a_ptr2]\n"
                        "fmla z25.h, z9.h, z2.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        "fmla z22.h, z10.h, z1.h[0]\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        "fmla z26.h, z10.h, z2.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        "fmla z23.h, z11.h, z1.h[0]\n"
                        "fmla z27.h, z11.h, z2.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "fmla z20.h, z12.h, z1.h[1]\n"
                        "fmla z24.h, z12.h, z2.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "fmla z21.h, z13.h, z1.h[1]\n"
                        "fmla z25.h, z13.h, z2.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "fmla z22.h, z14.h, z1.h[1]\n"
                        "fmla z26.h, z14.h, z2.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "fmla z23.h, z15.h, z1.h[1]\n"
                        "fmla z27.h, z15.h, z2.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z1.h[2]\n"
                        "fmla z24.h, z8.h, z2.h[2]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z21.h, z9.h, z1.h[2]\n"
                        "fmla z25.h, z9.h, z2.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "fmla z22.h, z10.h, z1.h[2]\n"
                        "fmla z26.h, z10.h, z2.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "fmla z23.h, z11.h, z1.h[2]\n"
                        "fmla z27.h, z11.h, z2.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "fmla z20.h, z12.h, z1.h[3]\n"
                        "fmla z24.h, z12.h, z2.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "fmla z21.h, z13.h, z1.h[3]\n"
                        "fmla z25.h, z13.h, z2.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "fmla z22.h, z14.h, z1.h[3]\n"
                        "fmla z26.h, z14.h, z2.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "fmla z23.h, z15.h, z1.h[3]\n"
                        "fmla z27.h, z15.h, z2.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "fmla z20.h, z8.h, z1.h[4]\n"
                        "fmla z24.h, z8.h, z2.h[4]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "fmla z21.h, z9.h, z1.h[4]\n"
                        "fmla z25.h, z9.h, z2.h[4]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "fmla z22.h, z10.h, z1.h[4]\n"
                        "fmla z26.h, z10.h, z2.h[4]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "fmla z23.h, z11.h, z1.h[4]\n"
                        "fmla z27.h, z11.h, z2.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "fmla z20.h, z12.h, z1.h[5]\n"
                        "fmla z24.h, z12.h, z2.h[5]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "fmla z21.h, z13.h, z1.h[5]\n"
                        "fmla z25.h, z13.h, z2.h[5]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "fmla z22.h, z14.h, z1.h[5]\n"
                        "fmla z26.h, z14.h, z2.h[5]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "fmla z23.h, z15.h, z1.h[5]\n"
                        "fmla z27.h, z15.h, z2.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z20.h, z8.h, z1.h[6]\n"
                        "fmla z24.h, z8.h, z2.h[6]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "fmla z21.h, z9.h, z1.h[6]\n"
                        "fmla z25.h, z9.h, z2.h[6]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "fmla z22.h, z10.h, z1.h[6]\n"
                        "fmla z26.h, z10.h, z2.h[6]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "fmla z23.h, z11.h, z1.h[6]\n"
                        "fmla z27.h, z11.h, z2.h[6]\n"
                        "fmla z16.h, z12.h, z0.h[7]\n"
                        "fmla z20.h, z12.h, z1.h[7]\n"
                        "fmla z24.h, z12.h, z2.h[7]\n"
                        "fmla z17.h, z13.h, z0.h[7]\n"
                        "fmla z21.h, z13.h, z1.h[7]\n"
                        "fmla z25.h, z13.h, z2.h[7]\n"
                        "fmla z18.h, z14.h, z0.h[7]\n"
                        "fmla z22.h, z14.h, z1.h[7]\n"
                        "fmla z26.h, z14.h, z2.h[7]\n"
                        "fmla z19.h, z15.h, z0.h[7]\n"
                        "fmla z23.h, z15.h, z1.h[7]\n"
                        "fmla z27.h, z15.h, z2.h[7]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.h, z8.h, z5.h[0]\n"
                        "fmla z24.h, z8.h, z6.h[0]\n"
                        "fmla z17.h, z9.h, z4.h[0]\n"
                        "fmla z21.h, z9.h, z5.h[0]\n"
                        "fmla z25.h, z9.h, z6.h[0]\n"
                        "fmla z18.h, z10.h, z4.h[0]\n"
                        "fmla z22.h, z10.h, z5.h[0]\n"
                        "fmla z26.h, z10.h, z6.h[0]\n"
                        "fmla z19.h, z11.h, z4.h[0]\n"
                        "fmla z23.h, z11.h, z5.h[0]\n"
                        "fmla z27.h, z11.h, z6.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.h, z12.h, z5.h[1]\n"
                        "fmla z24.h, z12.h, z6.h[1]\n"
                        "fmla z17.h, z13.h, z4.h[1]\n"
                        "fmla z21.h, z13.h, z5.h[1]\n"
                        "fmla z25.h, z13.h, z6.h[1]\n"
                        "fmla z18.h, z14.h, z4.h[1]\n"
                        "fmla z22.h, z14.h, z5.h[1]\n"
                        "fmla z26.h, z14.h, z6.h[1]\n"
                        "fmla z19.h, z15.h, z4.h[1]\n"
                        "fmla z23.h, z15.h, z5.h[1]\n"
                        "fmla z27.h, z15.h, z6.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[2]\n"
                        "fmla z20.h, z8.h, z5.h[2]\n"
                        "fmla z24.h, z8.h, z6.h[2]\n"
                        "fmla z17.h, z9.h, z4.h[2]\n"
                        "fmla z21.h, z9.h, z5.h[2]\n"
                        "fmla z25.h, z9.h, z6.h[2]\n"
                        "fmla z18.h, z10.h, z4.h[2]\n"
                        "fmla z22.h, z10.h, z5.h[2]\n"
                        "fmla z26.h, z10.h, z6.h[2]\n"
                        "fmla z19.h, z11.h, z4.h[2]\n"
                        "fmla z23.h, z11.h, z5.h[2]\n"
                        "fmla z27.h, z11.h, z6.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.h, z12.h, z5.h[3]\n"
                        "fmla z24.h, z12.h, z6.h[3]\n"
                        "fmla z17.h, z13.h, z4.h[3]\n"
                        "fmla z21.h, z13.h, z5.h[3]\n"
                        "fmla z25.h, z13.h, z6.h[3]\n"
                        "fmla z18.h, z14.h, z4.h[3]\n"
                        "fmla z22.h, z14.h, z5.h[3]\n"
                        "fmla z26.h, z14.h, z6.h[3]\n"
                        "fmla z19.h, z15.h, z4.h[3]\n"
                        "fmla z23.h, z15.h, z5.h[3]\n"
                        "fmla z27.h, z15.h, z6.h[3]\n"
                        "b.eq 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.h, z8.h, z5.h[4]\n"
                        "fmla z24.h, z8.h, z6.h[4]\n"
                        "fmla z17.h, z9.h, z4.h[4]\n"
                        "fmla z21.h, z9.h, z5.h[4]\n"
                        "fmla z25.h, z9.h, z6.h[4]\n"
                        "fmla z18.h, z10.h, z4.h[4]\n"
                        "fmla z22.h, z10.h, z5.h[4]\n"
                        "fmla z26.h, z10.h, z6.h[4]\n"
                        "fmla z19.h, z11.h, z4.h[4]\n"
                        "fmla z23.h, z11.h, z5.h[4]\n"
                        "fmla z27.h, z11.h, z6.h[4]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.h, z12.h, z5.h[5]\n"
                        "fmla z24.h, z12.h, z6.h[5]\n"
                        "fmla z17.h, z13.h, z4.h[5]\n"
                        "fmla z21.h, z13.h, z5.h[5]\n"
                        "fmla z25.h, z13.h, z6.h[5]\n"
                        "fmla z18.h, z14.h, z4.h[5]\n"
                        "fmla z22.h, z14.h, z5.h[5]\n"
                        "fmla z26.h, z14.h, z6.h[5]\n"
                        "fmla z19.h, z15.h, z4.h[5]\n"
                        "fmla z23.h, z15.h, z5.h[5]\n"
                        "fmla z27.h, z15.h, z6.h[5]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[6]\n"
                        "fmla z20.h, z8.h, z5.h[6]\n"
                        "fmla z24.h, z8.h, z6.h[6]\n"
                        "fmla z17.h, z9.h, z4.h[6]\n"
                        "fmla z21.h, z9.h, z5.h[6]\n"
                        "fmla z25.h, z9.h, z6.h[6]\n"
                        "fmla z18.h, z10.h, z4.h[6]\n"
                        "fmla z22.h, z10.h, z5.h[6]\n"
                        "fmla z26.h, z10.h, z6.h[6]\n"
                        "fmla z19.h, z11.h, z4.h[6]\n"
                        "fmla z23.h, z11.h, z5.h[6]\n"
                        "fmla z27.h, z11.h, z6.h[6]\n"
                        "5:\n"
                        "ld1rh z14.h, p7/z, [%[minptr]]\n"
                        "ld1rh z15.h, p7/z, [%[maxptr]]\n"
                        "fmax z16.h, p7/m, z16.h, z14.h\n"
                        "fmax z17.h, p7/m, z17.h, z14.h\n"
                        "fmax z18.h, p7/m, z18.h, z14.h\n"
                        "fmax z19.h, p7/m, z19.h, z14.h\n"
                        "fmin z16.h, p7/m, z16.h, z15.h\n"
                        "fmin z17.h, p7/m, z17.h, z15.h\n"
                        "fmin z18.h, p7/m, z18.h, z15.h\n"
                        "fmin z19.h, p7/m, z19.h, z15.h\n"
                        "st1h z16.h, p0, [%[c_ptr0]]\n"
                        "fmax z20.h, p7/m, z20.h, z14.h\n"
                        "fmax z21.h, p7/m, z21.h, z14.h\n"
                        "fmax z22.h, p7/m, z22.h, z14.h\n"
                        "st1h z17.h, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmax z23.h, p7/m, z23.h, z14.h\n"
                        "fmin z20.h, p7/m, z20.h, z15.h\n"
                        "fmin z21.h, p7/m, z21.h, z15.h\n"
                        "st1h z18.h, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "fmin z22.h, p7/m, z22.h, z15.h\n"
                        "fmin z23.h, p7/m, z23.h, z15.h\n"
                        "fmax z24.h, p7/m, z24.h, z14.h\n"
                        "st1h z19.h, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "fmax z25.h, p7/m, z25.h, z14.h\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        "fmax z26.h, p7/m, z26.h, z14.h\n"
                        "st1h z20.h, p0, [c_ptr1]\n"
                        "fmin z24.h, p7/m, z24.h, z15.h\n"
                        "fmin z25.h, p7/m, z25.h, z15.h\n"
                        "fmax z27.h, p7/m, z27.h, z14.h\n"
                        "st1h z21.h, p1, [c_ptr1, #1, MUL VL]\n"
                        "fmin z26.h, p7/m, z26.h, z15.h\n"
                        "fmin z27.h, p7/m, z27.h, z15.h\n"
                        "st1h z22.h, p2, [c_ptr1, #2, MUL VL]\n"
                        "st1h z23.h, p3, [c_ptr1, #3, MUL VL]\n"
                        "st1h z24.h, p0, [c_ptr2]\n"
                        "st1h z25.h, p1, [c_ptr2, #1, MUL VL]\n"
                        "st1h z26.h, p2, [c_ptr2, #2, MUL VL]\n"
                        "st1h z27.h, p3, [c_ptr2, #3, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [append] "r" (static_cast<uint64_t>(append)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "cc", "memory"
                    );
                    break;
                default:
                case 4:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "a_ptr3 .req X2\n"
                        "c_ptr1 .req X3\n"
                        "c_ptr2 .req X4\n"
                        "c_ptr3 .req X5\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "add a_ptr3, a_ptr2, %[lda]\n"
                        "add c_ptr3, c_ptr2, %[ldc]\n"
                        "whilelt p6.h, %[temp], %[leftovers]\n"
                        "whilelt p0.h, %[temp], %[width]\n"
                        "inch %[temp], all, mul #1\n"
                        "ptrue p7.h\n"
                        "whilelt p1.h, %[temp], %[width]\n"
                        "inch %[temp], all, mul #1\n"
                        "whilelt p2.h, %[temp], %[width]\n"
                        "inch %[temp], all, mul #1\n"
                        "whilelt p3.h, %[temp], %[width]\n"
                        "cbnz %[append], 1f\n"
                        "ld1h z16.h, p0/z, [%[biasptr]]\n"
                        "ld1h z17.h, p1/z, [%[biasptr], #1, MUL VL]\n"
                        "ld1h z18.h, p2/z, [%[biasptr], #2, MUL VL]\n"
                        "ld1h z19.h, p3/z, [%[biasptr], #3, MUL VL]\n"
                        "mov z20.d, z16.d\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "mov z21.d, z17.d\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "mov z22.d, z18.d\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2]\n"
                        "mov z23.d, z19.d\n"
                        "ld1rqh z3.h, p7/z, [a_ptr3]\n"
                        "mov z24.d, z16.d\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "mov z25.d, z17.d\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "mov z26.d, z18.d\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "mov z27.d, z19.d\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z28.d, z16.d\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "mov z29.d, z17.d\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "mov z30.d, z18.d\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "mov z31.d, z19.d\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1h z16.h, p0/z, [%[c_ptr0]]\n"
                        "ld1h z17.h, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1h z18.h, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1h z19.h, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1h z20.h, p0/z, [c_ptr1]\n"
                        "ld1h z21.h, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1h z22.h, p2/z, [c_ptr1, #2, MUL VL]\n"
                        "ld1h z23.h, p3/z, [c_ptr1, #3, MUL VL]\n"
                        "ld1h z24.h, p0/z, [c_ptr2]\n"
                        "ld1h z25.h, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "ld1h z26.h, p2/z, [c_ptr2, #2, MUL VL]\n"
                        "ld1h z27.h, p3/z, [c_ptr2, #3, MUL VL]\n"
                        "ld1h z28.h, p0/z, [c_ptr3]\n"
                        "ld1h z29.h, p1/z, [c_ptr3, #1, MUL VL]\n"
                        "ld1h z30.h, p2/z, [c_ptr3, #2, MUL VL]\n"
                        "ld1h z31.h, p3/z, [c_ptr3, #3, MUL VL]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ld1rqh z3.h, p7/z, [a_ptr3]\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.h, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        "fmla z24.h, z8.h, z2.h[0]\n"
                        "ld1rqh z5.h, p7/z, [a_ptr1]\n"
                        "fmla z28.h, z8.h, z3.h[0]\n"
                        "ld1rqh z6.h, p7/z, [a_ptr2]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "ld1rqh z7.h, p7/z, [a_ptr3]\n"
                        "fmla z21.h, z9.h, z1.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z25.h, z9.h, z2.h[0]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "fmla z29.h, z9.h, z3.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        "fmla z22.h, z10.h, z1.h[0]\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        "fmla z26.h, z10.h, z2.h[0]\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        "fmla z30.h, z10.h, z3.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "add a_ptr3, a_ptr3, #0x20\n"
                        "fmla z23.h, z11.h, z1.h[0]\n"
                        "fmla z27.h, z11.h, z2.h[0]\n"
                        "fmla z31.h, z11.h, z3.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "fmla z20.h, z12.h, z1.h[1]\n"
                        "fmla z24.h, z12.h, z2.h[1]\n"
                        "fmla z28.h, z12.h, z3.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "fmla z21.h, z13.h, z1.h[1]\n"
                        "fmla z25.h, z13.h, z2.h[1]\n"
                        "fmla z29.h, z13.h, z3.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "fmla z22.h, z14.h, z1.h[1]\n"
                        "fmla z26.h, z14.h, z2.h[1]\n"
                        "fmla z30.h, z14.h, z3.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "fmla z23.h, z15.h, z1.h[1]\n"
                        "fmla z27.h, z15.h, z2.h[1]\n"
                        "fmla z31.h, z15.h, z3.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z1.h[2]\n"
                        "fmla z24.h, z8.h, z2.h[2]\n"
                        "fmla z28.h, z8.h, z3.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "fmla z21.h, z9.h, z1.h[2]\n"
                        "fmla z25.h, z9.h, z2.h[2]\n"
                        "fmla z29.h, z9.h, z3.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "fmla z22.h, z10.h, z1.h[2]\n"
                        "fmla z26.h, z10.h, z2.h[2]\n"
                        "fmla z30.h, z10.h, z3.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "fmla z23.h, z11.h, z1.h[2]\n"
                        "fmla z27.h, z11.h, z2.h[2]\n"
                        "fmla z31.h, z11.h, z3.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "fmla z20.h, z12.h, z1.h[3]\n"
                        "fmla z24.h, z12.h, z2.h[3]\n"
                        "fmla z28.h, z12.h, z3.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "fmla z21.h, z13.h, z1.h[3]\n"
                        "fmla z25.h, z13.h, z2.h[3]\n"
                        "fmla z29.h, z13.h, z3.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "fmla z22.h, z14.h, z1.h[3]\n"
                        "fmla z26.h, z14.h, z2.h[3]\n"
                        "fmla z30.h, z14.h, z3.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "fmla z23.h, z15.h, z1.h[3]\n"
                        "fmla z27.h, z15.h, z2.h[3]\n"
                        "fmla z31.h, z15.h, z3.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "fmla z20.h, z8.h, z1.h[4]\n"
                        "fmla z24.h, z8.h, z2.h[4]\n"
                        "fmla z28.h, z8.h, z3.h[4]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "fmla z21.h, z9.h, z1.h[4]\n"
                        "fmla z25.h, z9.h, z2.h[4]\n"
                        "fmla z29.h, z9.h, z3.h[4]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "fmla z22.h, z10.h, z1.h[4]\n"
                        "fmla z26.h, z10.h, z2.h[4]\n"
                        "fmla z30.h, z10.h, z3.h[4]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "fmla z23.h, z11.h, z1.h[4]\n"
                        "fmla z27.h, z11.h, z2.h[4]\n"
                        "fmla z31.h, z11.h, z3.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "fmla z20.h, z12.h, z1.h[5]\n"
                        "fmla z24.h, z12.h, z2.h[5]\n"
                        "fmla z28.h, z12.h, z3.h[5]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "fmla z21.h, z13.h, z1.h[5]\n"
                        "fmla z25.h, z13.h, z2.h[5]\n"
                        "fmla z29.h, z13.h, z3.h[5]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "fmla z22.h, z14.h, z1.h[5]\n"
                        "fmla z26.h, z14.h, z2.h[5]\n"
                        "fmla z30.h, z14.h, z3.h[5]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "fmla z23.h, z15.h, z1.h[5]\n"
                        "fmla z27.h, z15.h, z2.h[5]\n"
                        "fmla z31.h, z15.h, z3.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z1.h[6]\n"
                        "fmla z24.h, z8.h, z2.h[6]\n"
                        "fmla z28.h, z8.h, z3.h[6]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "fmla z21.h, z9.h, z1.h[6]\n"
                        "fmla z25.h, z9.h, z2.h[6]\n"
                        "fmla z29.h, z9.h, z3.h[6]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "fmla z22.h, z10.h, z1.h[6]\n"
                        "fmla z26.h, z10.h, z2.h[6]\n"
                        "fmla z30.h, z10.h, z3.h[6]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "fmla z23.h, z11.h, z1.h[6]\n"
                        "fmla z27.h, z11.h, z2.h[6]\n"
                        "fmla z31.h, z11.h, z3.h[6]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[7]\n"
                        "fmla z20.h, z12.h, z1.h[7]\n"
                        "fmla z24.h, z12.h, z2.h[7]\n"
                        "fmla z28.h, z12.h, z3.h[7]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[7]\n"
                        "fmla z21.h, z13.h, z1.h[7]\n"
                        "fmla z25.h, z13.h, z2.h[7]\n"
                        "fmla z29.h, z13.h, z3.h[7]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[7]\n"
                        "fmla z22.h, z14.h, z1.h[7]\n"
                        "fmla z26.h, z14.h, z2.h[7]\n"
                        "fmla z30.h, z14.h, z3.h[7]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[7]\n"
                        "ld1rqh z0.h, p7/z, [%[a_ptr0], #-0x10]\n"
                        "fmla z23.h, z15.h, z1.h[7]\n"
                        "ld1rqh z1.h, p7/z, [a_ptr1, #-0x10]\n"
                        "fmla z27.h, z15.h, z2.h[7]\n"
                        "ld1rqh z2.h, p7/z, [a_ptr2, #-0x10]\n"
                        "fmla z31.h, z15.h, z3.h[7]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[0]\n"
                        "ld1rqh z3.h, p7/z, [a_ptr3, #-0x10]\n"
                        "fmla z20.h, z8.h, z5.h[0]\n"
                        "fmla z24.h, z8.h, z6.h[0]\n"
                        "fmla z28.h, z8.h, z7.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z4.h[0]\n"
                        "fmla z21.h, z9.h, z5.h[0]\n"
                        "fmla z25.h, z9.h, z6.h[0]\n"
                        "fmla z29.h, z9.h, z7.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[0]\n"
                        "fmla z22.h, z10.h, z5.h[0]\n"
                        "fmla z26.h, z10.h, z6.h[0]\n"
                        "fmla z30.h, z10.h, z7.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[0]\n"
                        "fmla z23.h, z11.h, z5.h[0]\n"
                        "fmla z27.h, z11.h, z6.h[0]\n"
                        "fmla z31.h, z11.h, z7.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[1]\n"
                        "fmla z20.h, z12.h, z5.h[1]\n"
                        "fmla z24.h, z12.h, z6.h[1]\n"
                        "fmla z28.h, z12.h, z7.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[1]\n"
                        "fmla z21.h, z13.h, z5.h[1]\n"
                        "fmla z25.h, z13.h, z6.h[1]\n"
                        "fmla z29.h, z13.h, z7.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[1]\n"
                        "fmla z22.h, z14.h, z5.h[1]\n"
                        "fmla z26.h, z14.h, z6.h[1]\n"
                        "fmla z30.h, z14.h, z7.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[1]\n"
                        "fmla z23.h, z15.h, z5.h[1]\n"
                        "fmla z27.h, z15.h, z6.h[1]\n"
                        "fmla z31.h, z15.h, z7.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z5.h[2]\n"
                        "fmla z24.h, z8.h, z6.h[2]\n"
                        "fmla z28.h, z8.h, z7.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z17.h, z9.h, z4.h[2]\n"
                        "fmla z21.h, z9.h, z5.h[2]\n"
                        "fmla z25.h, z9.h, z6.h[2]\n"
                        "fmla z29.h, z9.h, z7.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[2]\n"
                        "fmla z22.h, z10.h, z5.h[2]\n"
                        "fmla z26.h, z10.h, z6.h[2]\n"
                        "fmla z30.h, z10.h, z7.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[2]\n"
                        "fmla z23.h, z11.h, z5.h[2]\n"
                        "fmla z27.h, z11.h, z6.h[2]\n"
                        "fmla z31.h, z11.h, z7.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[3]\n"
                        "fmla z20.h, z12.h, z5.h[3]\n"
                        "fmla z24.h, z12.h, z6.h[3]\n"
                        "fmla z28.h, z12.h, z7.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[3]\n"
                        "fmla z21.h, z13.h, z5.h[3]\n"
                        "fmla z25.h, z13.h, z6.h[3]\n"
                        "fmla z29.h, z13.h, z7.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[3]\n"
                        "fmla z22.h, z14.h, z5.h[3]\n"
                        "fmla z26.h, z14.h, z6.h[3]\n"
                        "fmla z30.h, z14.h, z7.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[3]\n"
                        "fmla z23.h, z15.h, z5.h[3]\n"
                        "fmla z27.h, z15.h, z6.h[3]\n"
                        "fmla z31.h, z15.h, z7.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[4]\n"
                        "fmla z20.h, z8.h, z5.h[4]\n"
                        "fmla z24.h, z8.h, z6.h[4]\n"
                        "fmla z28.h, z8.h, z7.h[4]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z4.h[4]\n"
                        "fmla z21.h, z9.h, z5.h[4]\n"
                        "fmla z25.h, z9.h, z6.h[4]\n"
                        "fmla z29.h, z9.h, z7.h[4]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[4]\n"
                        "fmla z22.h, z10.h, z5.h[4]\n"
                        "fmla z26.h, z10.h, z6.h[4]\n"
                        "fmla z30.h, z10.h, z7.h[4]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[4]\n"
                        "fmla z23.h, z11.h, z5.h[4]\n"
                        "fmla z27.h, z11.h, z6.h[4]\n"
                        "fmla z31.h, z11.h, z7.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[5]\n"
                        "fmla z20.h, z12.h, z5.h[5]\n"
                        "fmla z24.h, z12.h, z6.h[5]\n"
                        "fmla z28.h, z12.h, z7.h[5]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[5]\n"
                        "fmla z21.h, z13.h, z5.h[5]\n"
                        "fmla z25.h, z13.h, z6.h[5]\n"
                        "fmla z29.h, z13.h, z7.h[5]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[5]\n"
                        "fmla z22.h, z14.h, z5.h[5]\n"
                        "fmla z26.h, z14.h, z6.h[5]\n"
                        "fmla z30.h, z14.h, z7.h[5]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[5]\n"
                        "fmla z23.h, z15.h, z5.h[5]\n"
                        "fmla z27.h, z15.h, z6.h[5]\n"
                        "fmla z31.h, z15.h, z7.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[6]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z5.h[6]\n"
                        "fmla z24.h, z8.h, z6.h[6]\n"
                        "fmla z28.h, z8.h, z7.h[6]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z17.h, z9.h, z4.h[6]\n"
                        "fmla z21.h, z9.h, z5.h[6]\n"
                        "fmla z25.h, z9.h, z6.h[6]\n"
                        "fmla z29.h, z9.h, z7.h[6]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[6]\n"
                        "fmla z22.h, z10.h, z5.h[6]\n"
                        "fmla z26.h, z10.h, z6.h[6]\n"
                        "fmla z30.h, z10.h, z7.h[6]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[6]\n"
                        "fmla z23.h, z11.h, z5.h[6]\n"
                        "fmla z27.h, z11.h, z6.h[6]\n"
                        "fmla z31.h, z11.h, z7.h[6]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[7]\n"
                        "fmla z20.h, z12.h, z5.h[7]\n"
                        "fmla z24.h, z12.h, z6.h[7]\n"
                        "fmla z28.h, z12.h, z7.h[7]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[7]\n"
                        "fmla z21.h, z13.h, z5.h[7]\n"
                        "fmla z25.h, z13.h, z6.h[7]\n"
                        "fmla z29.h, z13.h, z7.h[7]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[7]\n"
                        "fmla z22.h, z14.h, z5.h[7]\n"
                        "fmla z26.h, z14.h, z6.h[7]\n"
                        "fmla z30.h, z14.h, z7.h[7]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[7]\n"
                        "fmla z23.h, z15.h, z5.h[7]\n"
                        "fmla z27.h, z15.h, z6.h[7]\n"
                        "fmla z31.h, z15.h, z7.h[7]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.h, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p7/z, [%[a_ptr0]]\n"
                        "fmla z24.h, z8.h, z2.h[0]\n"
                        "ld1rqh z5.h, p7/z, [a_ptr1]\n"
                        "fmla z28.h, z8.h, z3.h[0]\n"
                        "ld1rqh z6.h, p7/z, [a_ptr2]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "ld1rqh z7.h, p7/z, [a_ptr3]\n"
                        "fmla z21.h, z9.h, z1.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z25.h, z9.h, z2.h[0]\n"
                        "fmla z29.h, z9.h, z3.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "fmla z22.h, z10.h, z1.h[0]\n"
                        "fmla z26.h, z10.h, z2.h[0]\n"
                        "fmla z30.h, z10.h, z3.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "fmla z23.h, z11.h, z1.h[0]\n"
                        "fmla z27.h, z11.h, z2.h[0]\n"
                        "fmla z31.h, z11.h, z3.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "fmla z20.h, z12.h, z1.h[1]\n"
                        "fmla z24.h, z12.h, z2.h[1]\n"
                        "fmla z28.h, z12.h, z3.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "fmla z21.h, z13.h, z1.h[1]\n"
                        "fmla z25.h, z13.h, z2.h[1]\n"
                        "fmla z29.h, z13.h, z3.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "fmla z22.h, z14.h, z1.h[1]\n"
                        "fmla z26.h, z14.h, z2.h[1]\n"
                        "fmla z30.h, z14.h, z3.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "fmla z23.h, z15.h, z1.h[1]\n"
                        "fmla z27.h, z15.h, z2.h[1]\n"
                        "fmla z31.h, z15.h, z3.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z1.h[2]\n"
                        "fmla z24.h, z8.h, z2.h[2]\n"
                        "fmla z28.h, z8.h, z3.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "fmla z21.h, z9.h, z1.h[2]\n"
                        "fmla z25.h, z9.h, z2.h[2]\n"
                        "fmla z29.h, z9.h, z3.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "fmla z22.h, z10.h, z1.h[2]\n"
                        "fmla z26.h, z10.h, z2.h[2]\n"
                        "fmla z30.h, z10.h, z3.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "fmla z23.h, z11.h, z1.h[2]\n"
                        "fmla z27.h, z11.h, z2.h[2]\n"
                        "fmla z31.h, z11.h, z3.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "fmla z20.h, z12.h, z1.h[3]\n"
                        "fmla z24.h, z12.h, z2.h[3]\n"
                        "fmla z28.h, z12.h, z3.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "fmla z21.h, z13.h, z1.h[3]\n"
                        "fmla z25.h, z13.h, z2.h[3]\n"
                        "fmla z29.h, z13.h, z3.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "fmla z22.h, z14.h, z1.h[3]\n"
                        "fmla z26.h, z14.h, z2.h[3]\n"
                        "fmla z30.h, z14.h, z3.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "fmla z23.h, z15.h, z1.h[3]\n"
                        "fmla z27.h, z15.h, z2.h[3]\n"
                        "fmla z31.h, z15.h, z3.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "fmla z20.h, z8.h, z1.h[4]\n"
                        "fmla z24.h, z8.h, z2.h[4]\n"
                        "fmla z28.h, z8.h, z3.h[4]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "fmla z21.h, z9.h, z1.h[4]\n"
                        "fmla z25.h, z9.h, z2.h[4]\n"
                        "fmla z29.h, z9.h, z3.h[4]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "fmla z22.h, z10.h, z1.h[4]\n"
                        "fmla z26.h, z10.h, z2.h[4]\n"
                        "fmla z30.h, z10.h, z3.h[4]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "fmla z23.h, z11.h, z1.h[4]\n"
                        "fmla z27.h, z11.h, z2.h[4]\n"
                        "fmla z31.h, z11.h, z3.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "fmla z20.h, z12.h, z1.h[5]\n"
                        "fmla z24.h, z12.h, z2.h[5]\n"
                        "fmla z28.h, z12.h, z3.h[5]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "fmla z21.h, z13.h, z1.h[5]\n"
                        "fmla z25.h, z13.h, z2.h[5]\n"
                        "fmla z29.h, z13.h, z3.h[5]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "fmla z22.h, z14.h, z1.h[5]\n"
                        "fmla z26.h, z14.h, z2.h[5]\n"
                        "fmla z30.h, z14.h, z3.h[5]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "fmla z23.h, z15.h, z1.h[5]\n"
                        "fmla z27.h, z15.h, z2.h[5]\n"
                        "fmla z31.h, z15.h, z3.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z1.h[6]\n"
                        "fmla z24.h, z8.h, z2.h[6]\n"
                        "fmla z28.h, z8.h, z3.h[6]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "fmla z21.h, z9.h, z1.h[6]\n"
                        "fmla z25.h, z9.h, z2.h[6]\n"
                        "fmla z29.h, z9.h, z3.h[6]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "fmla z22.h, z10.h, z1.h[6]\n"
                        "fmla z26.h, z10.h, z2.h[6]\n"
                        "fmla z30.h, z10.h, z3.h[6]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "fmla z23.h, z11.h, z1.h[6]\n"
                        "fmla z27.h, z11.h, z2.h[6]\n"
                        "fmla z31.h, z11.h, z3.h[6]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[7]\n"
                        "fmla z20.h, z12.h, z1.h[7]\n"
                        "fmla z24.h, z12.h, z2.h[7]\n"
                        "fmla z28.h, z12.h, z3.h[7]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[7]\n"
                        "fmla z21.h, z13.h, z1.h[7]\n"
                        "fmla z25.h, z13.h, z2.h[7]\n"
                        "fmla z29.h, z13.h, z3.h[7]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[7]\n"
                        "fmla z22.h, z14.h, z1.h[7]\n"
                        "fmla z26.h, z14.h, z2.h[7]\n"
                        "fmla z30.h, z14.h, z3.h[7]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[7]\n"
                        "ld1rqh z0.h, p6/z, [%[a_ptr0], #0x10]\n"
                        "fmla z23.h, z15.h, z1.h[7]\n"
                        "ld1rqh z1.h, p6/z, [a_ptr1, #0x10]\n"
                        "fmla z27.h, z15.h, z2.h[7]\n"
                        "ld1rqh z2.h, p6/z, [a_ptr2, #0x10]\n"
                        "fmla z31.h, z15.h, z3.h[7]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[0]\n"
                        "ld1rqh z3.h, p6/z, [a_ptr3, #0x10]\n"
                        "fmla z20.h, z8.h, z5.h[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        "fmla z24.h, z8.h, z6.h[0]\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        "fmla z28.h, z8.h, z7.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z4.h[0]\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        "fmla z21.h, z9.h, z5.h[0]\n"
                        "addvl a_ptr3, a_ptr3, #2\n"
                        "fmla z25.h, z9.h, z6.h[0]\n"
                        "fmla z29.h, z9.h, z7.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[0]\n"
                        "fmla z22.h, z10.h, z5.h[0]\n"
                        "fmla z26.h, z10.h, z6.h[0]\n"
                        "fmla z30.h, z10.h, z7.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[0]\n"
                        "fmla z23.h, z11.h, z5.h[0]\n"
                        "fmla z27.h, z11.h, z6.h[0]\n"
                        "fmla z31.h, z11.h, z7.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[1]\n"
                        "fmla z20.h, z12.h, z5.h[1]\n"
                        "fmla z24.h, z12.h, z6.h[1]\n"
                        "fmla z28.h, z12.h, z7.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[1]\n"
                        "fmla z21.h, z13.h, z5.h[1]\n"
                        "fmla z25.h, z13.h, z6.h[1]\n"
                        "fmla z29.h, z13.h, z7.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[1]\n"
                        "fmla z22.h, z14.h, z5.h[1]\n"
                        "fmla z26.h, z14.h, z6.h[1]\n"
                        "fmla z30.h, z14.h, z7.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[1]\n"
                        "fmla z23.h, z15.h, z5.h[1]\n"
                        "fmla z27.h, z15.h, z6.h[1]\n"
                        "fmla z31.h, z15.h, z7.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z5.h[2]\n"
                        "fmla z24.h, z8.h, z6.h[2]\n"
                        "fmla z28.h, z8.h, z7.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z17.h, z9.h, z4.h[2]\n"
                        "fmla z21.h, z9.h, z5.h[2]\n"
                        "fmla z25.h, z9.h, z6.h[2]\n"
                        "fmla z29.h, z9.h, z7.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[2]\n"
                        "fmla z22.h, z10.h, z5.h[2]\n"
                        "fmla z26.h, z10.h, z6.h[2]\n"
                        "fmla z30.h, z10.h, z7.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[2]\n"
                        "fmla z23.h, z11.h, z5.h[2]\n"
                        "fmla z27.h, z11.h, z6.h[2]\n"
                        "fmla z31.h, z11.h, z7.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[3]\n"
                        "fmla z20.h, z12.h, z5.h[3]\n"
                        "fmla z24.h, z12.h, z6.h[3]\n"
                        "fmla z28.h, z12.h, z7.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[3]\n"
                        "fmla z21.h, z13.h, z5.h[3]\n"
                        "fmla z25.h, z13.h, z6.h[3]\n"
                        "fmla z29.h, z13.h, z7.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[3]\n"
                        "fmla z22.h, z14.h, z5.h[3]\n"
                        "fmla z26.h, z14.h, z6.h[3]\n"
                        "fmla z30.h, z14.h, z7.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[3]\n"
                        "fmla z23.h, z15.h, z5.h[3]\n"
                        "fmla z27.h, z15.h, z6.h[3]\n"
                        "fmla z31.h, z15.h, z7.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[4]\n"
                        "fmla z20.h, z8.h, z5.h[4]\n"
                        "fmla z24.h, z8.h, z6.h[4]\n"
                        "fmla z28.h, z8.h, z7.h[4]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z4.h[4]\n"
                        "fmla z21.h, z9.h, z5.h[4]\n"
                        "fmla z25.h, z9.h, z6.h[4]\n"
                        "fmla z29.h, z9.h, z7.h[4]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z4.h[4]\n"
                        "fmla z22.h, z10.h, z5.h[4]\n"
                        "fmla z26.h, z10.h, z6.h[4]\n"
                        "fmla z30.h, z10.h, z7.h[4]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z4.h[4]\n"
                        "fmla z23.h, z11.h, z5.h[4]\n"
                        "fmla z27.h, z11.h, z6.h[4]\n"
                        "fmla z31.h, z11.h, z7.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[5]\n"
                        "fmla z20.h, z12.h, z5.h[5]\n"
                        "fmla z24.h, z12.h, z6.h[5]\n"
                        "fmla z28.h, z12.h, z7.h[5]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z4.h[5]\n"
                        "fmla z21.h, z13.h, z5.h[5]\n"
                        "fmla z25.h, z13.h, z6.h[5]\n"
                        "fmla z29.h, z13.h, z7.h[5]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z4.h[5]\n"
                        "fmla z22.h, z14.h, z5.h[5]\n"
                        "fmla z26.h, z14.h, z6.h[5]\n"
                        "fmla z30.h, z14.h, z7.h[5]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z4.h[5]\n"
                        "fmla z23.h, z15.h, z5.h[5]\n"
                        "fmla z27.h, z15.h, z6.h[5]\n"
                        "fmla z31.h, z15.h, z7.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[6]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z20.h, z8.h, z5.h[6]\n"
                        "fmla z24.h, z8.h, z6.h[6]\n"
                        "fmla z28.h, z8.h, z7.h[6]\n"
                        "fmla z17.h, z9.h, z4.h[6]\n"
                        "fmla z21.h, z9.h, z5.h[6]\n"
                        "fmla z25.h, z9.h, z6.h[6]\n"
                        "fmla z29.h, z9.h, z7.h[6]\n"
                        "fmla z18.h, z10.h, z4.h[6]\n"
                        "fmla z22.h, z10.h, z5.h[6]\n"
                        "fmla z26.h, z10.h, z6.h[6]\n"
                        "fmla z30.h, z10.h, z7.h[6]\n"
                        "fmla z19.h, z11.h, z4.h[6]\n"
                        "fmla z23.h, z11.h, z5.h[6]\n"
                        "fmla z27.h, z11.h, z6.h[6]\n"
                        "fmla z31.h, z11.h, z7.h[6]\n"
                        "fmla z16.h, z12.h, z4.h[7]\n"
                        "fmla z20.h, z12.h, z5.h[7]\n"
                        "fmla z24.h, z12.h, z6.h[7]\n"
                        "fmla z28.h, z12.h, z7.h[7]\n"
                        "fmla z17.h, z13.h, z4.h[7]\n"
                        "fmla z21.h, z13.h, z5.h[7]\n"
                        "fmla z25.h, z13.h, z6.h[7]\n"
                        "fmla z29.h, z13.h, z7.h[7]\n"
                        "fmla z18.h, z14.h, z4.h[7]\n"
                        "fmla z22.h, z14.h, z5.h[7]\n"
                        "fmla z26.h, z14.h, z6.h[7]\n"
                        "fmla z30.h, z14.h, z7.h[7]\n"
                        "fmla z19.h, z15.h, z4.h[7]\n"
                        "fmla z23.h, z15.h, z5.h[7]\n"
                        "fmla z27.h, z15.h, z6.h[7]\n"
                        "fmla z31.h, z15.h, z7.h[7]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.h, z8.h, z1.h[0]\n"
                        "fmla z24.h, z8.h, z2.h[0]\n"
                        "fmla z28.h, z8.h, z3.h[0]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "fmla z21.h, z9.h, z1.h[0]\n"
                        "fmla z25.h, z9.h, z2.h[0]\n"
                        "fmla z29.h, z9.h, z3.h[0]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "fmla z22.h, z10.h, z1.h[0]\n"
                        "fmla z26.h, z10.h, z2.h[0]\n"
                        "fmla z30.h, z10.h, z3.h[0]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "fmla z23.h, z11.h, z1.h[0]\n"
                        "fmla z27.h, z11.h, z2.h[0]\n"
                        "fmla z31.h, z11.h, z3.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.h, z12.h, z1.h[1]\n"
                        "fmla z24.h, z12.h, z2.h[1]\n"
                        "fmla z28.h, z12.h, z3.h[1]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "fmla z21.h, z13.h, z1.h[1]\n"
                        "fmla z25.h, z13.h, z2.h[1]\n"
                        "fmla z29.h, z13.h, z3.h[1]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "fmla z22.h, z14.h, z1.h[1]\n"
                        "fmla z26.h, z14.h, z2.h[1]\n"
                        "fmla z30.h, z14.h, z3.h[1]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "fmla z23.h, z15.h, z1.h[1]\n"
                        "fmla z27.h, z15.h, z2.h[1]\n"
                        "fmla z31.h, z15.h, z3.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "fmla z20.h, z8.h, z1.h[2]\n"
                        "fmla z24.h, z8.h, z2.h[2]\n"
                        "fmla z28.h, z8.h, z3.h[2]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "fmla z21.h, z9.h, z1.h[2]\n"
                        "fmla z25.h, z9.h, z2.h[2]\n"
                        "fmla z29.h, z9.h, z3.h[2]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "fmla z22.h, z10.h, z1.h[2]\n"
                        "fmla z26.h, z10.h, z2.h[2]\n"
                        "fmla z30.h, z10.h, z3.h[2]\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "fmla z23.h, z11.h, z1.h[2]\n"
                        "fmla z27.h, z11.h, z2.h[2]\n"
                        "fmla z31.h, z11.h, z3.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.h, z12.h, z1.h[3]\n"
                        "fmla z24.h, z12.h, z2.h[3]\n"
                        "fmla z28.h, z12.h, z3.h[3]\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "fmla z21.h, z13.h, z1.h[3]\n"
                        "fmla z25.h, z13.h, z2.h[3]\n"
                        "fmla z29.h, z13.h, z3.h[3]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "fmla z22.h, z14.h, z1.h[3]\n"
                        "fmla z26.h, z14.h, z2.h[3]\n"
                        "fmla z30.h, z14.h, z3.h[3]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "fmla z23.h, z15.h, z1.h[3]\n"
                        "fmla z27.h, z15.h, z2.h[3]\n"
                        "fmla z31.h, z15.h, z3.h[3]\n"
                        "b.eq 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.h, z8.h, z1.h[4]\n"
                        "fmla z24.h, z8.h, z2.h[4]\n"
                        "fmla z28.h, z8.h, z3.h[4]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "fmla z21.h, z9.h, z1.h[4]\n"
                        "fmla z25.h, z9.h, z2.h[4]\n"
                        "fmla z29.h, z9.h, z3.h[4]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "fmla z22.h, z10.h, z1.h[4]\n"
                        "fmla z26.h, z10.h, z2.h[4]\n"
                        "fmla z30.h, z10.h, z3.h[4]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "fmla z23.h, z11.h, z1.h[4]\n"
                        "fmla z27.h, z11.h, z2.h[4]\n"
                        "fmla z31.h, z11.h, z3.h[4]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.h, z12.h, z1.h[5]\n"
                        "fmla z24.h, z12.h, z2.h[5]\n"
                        "fmla z28.h, z12.h, z3.h[5]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "fmla z21.h, z13.h, z1.h[5]\n"
                        "fmla z25.h, z13.h, z2.h[5]\n"
                        "fmla z29.h, z13.h, z3.h[5]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "fmla z22.h, z14.h, z1.h[5]\n"
                        "fmla z26.h, z14.h, z2.h[5]\n"
                        "fmla z30.h, z14.h, z3.h[5]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "fmla z23.h, z15.h, z1.h[5]\n"
                        "fmla z27.h, z15.h, z2.h[5]\n"
                        "fmla z31.h, z15.h, z3.h[5]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "fmla z20.h, z8.h, z1.h[6]\n"
                        "fmla z24.h, z8.h, z2.h[6]\n"
                        "fmla z28.h, z8.h, z3.h[6]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "fmla z21.h, z9.h, z1.h[6]\n"
                        "fmla z25.h, z9.h, z2.h[6]\n"
                        "fmla z29.h, z9.h, z3.h[6]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "fmla z22.h, z10.h, z1.h[6]\n"
                        "fmla z26.h, z10.h, z2.h[6]\n"
                        "fmla z30.h, z10.h, z3.h[6]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "fmla z23.h, z11.h, z1.h[6]\n"
                        "fmla z27.h, z11.h, z2.h[6]\n"
                        "fmla z31.h, z11.h, z3.h[6]\n"
                        "b 5f\n"
                        "4:\n"
                        "fmla z16.h, z8.h, z0.h[0]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.h, z8.h, z1.h[0]\n"
                        "ld1rqh z4.h, p6/z, [%[a_ptr0]]\n"
                        "fmla z24.h, z8.h, z2.h[0]\n"
                        "ld1rqh z5.h, p6/z, [a_ptr1]\n"
                        "fmla z28.h, z8.h, z3.h[0]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z0.h[0]\n"
                        "ld1rqh z6.h, p6/z, [a_ptr2]\n"
                        "fmla z21.h, z9.h, z1.h[0]\n"
                        "ld1rqh z7.h, p6/z, [a_ptr3]\n"
                        "fmla z25.h, z9.h, z2.h[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        "fmla z29.h, z9.h, z3.h[0]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[0]\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        "fmla z22.h, z10.h, z1.h[0]\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        "fmla z26.h, z10.h, z2.h[0]\n"
                        "addvl a_ptr3, a_ptr3, #1\n"
                        "fmla z30.h, z10.h, z3.h[0]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[0]\n"
                        "fmla z23.h, z11.h, z1.h[0]\n"
                        "fmla z27.h, z11.h, z2.h[0]\n"
                        "fmla z31.h, z11.h, z3.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[1]\n"
                        "fmla z20.h, z12.h, z1.h[1]\n"
                        "fmla z24.h, z12.h, z2.h[1]\n"
                        "fmla z28.h, z12.h, z3.h[1]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[1]\n"
                        "fmla z21.h, z13.h, z1.h[1]\n"
                        "fmla z25.h, z13.h, z2.h[1]\n"
                        "fmla z29.h, z13.h, z3.h[1]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[1]\n"
                        "fmla z22.h, z14.h, z1.h[1]\n"
                        "fmla z26.h, z14.h, z2.h[1]\n"
                        "fmla z30.h, z14.h, z3.h[1]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[1]\n"
                        "fmla z23.h, z15.h, z1.h[1]\n"
                        "fmla z27.h, z15.h, z2.h[1]\n"
                        "fmla z31.h, z15.h, z3.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "fmla z20.h, z8.h, z1.h[2]\n"
                        "fmla z24.h, z8.h, z2.h[2]\n"
                        "fmla z28.h, z8.h, z3.h[2]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "fmla z17.h, z9.h, z0.h[2]\n"
                        "fmla z21.h, z9.h, z1.h[2]\n"
                        "fmla z25.h, z9.h, z2.h[2]\n"
                        "fmla z29.h, z9.h, z3.h[2]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[2]\n"
                        "fmla z22.h, z10.h, z1.h[2]\n"
                        "fmla z26.h, z10.h, z2.h[2]\n"
                        "fmla z30.h, z10.h, z3.h[2]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[2]\n"
                        "fmla z23.h, z11.h, z1.h[2]\n"
                        "fmla z27.h, z11.h, z2.h[2]\n"
                        "fmla z31.h, z11.h, z3.h[2]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[3]\n"
                        "fmla z20.h, z12.h, z1.h[3]\n"
                        "fmla z24.h, z12.h, z2.h[3]\n"
                        "fmla z28.h, z12.h, z3.h[3]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[3]\n"
                        "fmla z21.h, z13.h, z1.h[3]\n"
                        "fmla z25.h, z13.h, z2.h[3]\n"
                        "fmla z29.h, z13.h, z3.h[3]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[3]\n"
                        "fmla z22.h, z14.h, z1.h[3]\n"
                        "fmla z26.h, z14.h, z2.h[3]\n"
                        "fmla z30.h, z14.h, z3.h[3]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[3]\n"
                        "fmla z23.h, z15.h, z1.h[3]\n"
                        "fmla z27.h, z15.h, z2.h[3]\n"
                        "fmla z31.h, z15.h, z3.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[4]\n"
                        "fmla z20.h, z8.h, z1.h[4]\n"
                        "fmla z24.h, z8.h, z2.h[4]\n"
                        "fmla z28.h, z8.h, z3.h[4]\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "fmla z17.h, z9.h, z0.h[4]\n"
                        "fmla z21.h, z9.h, z1.h[4]\n"
                        "fmla z25.h, z9.h, z2.h[4]\n"
                        "fmla z29.h, z9.h, z3.h[4]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "fmla z18.h, z10.h, z0.h[4]\n"
                        "fmla z22.h, z10.h, z1.h[4]\n"
                        "fmla z26.h, z10.h, z2.h[4]\n"
                        "fmla z30.h, z10.h, z3.h[4]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z19.h, z11.h, z0.h[4]\n"
                        "fmla z23.h, z11.h, z1.h[4]\n"
                        "fmla z27.h, z11.h, z2.h[4]\n"
                        "fmla z31.h, z11.h, z3.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z16.h, z12.h, z0.h[5]\n"
                        "fmla z20.h, z12.h, z1.h[5]\n"
                        "fmla z24.h, z12.h, z2.h[5]\n"
                        "fmla z28.h, z12.h, z3.h[5]\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "fmla z17.h, z13.h, z0.h[5]\n"
                        "fmla z21.h, z13.h, z1.h[5]\n"
                        "fmla z25.h, z13.h, z2.h[5]\n"
                        "fmla z29.h, z13.h, z3.h[5]\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "fmla z18.h, z14.h, z0.h[5]\n"
                        "fmla z22.h, z14.h, z1.h[5]\n"
                        "fmla z26.h, z14.h, z2.h[5]\n"
                        "fmla z30.h, z14.h, z3.h[5]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z19.h, z15.h, z0.h[5]\n"
                        "fmla z23.h, z15.h, z1.h[5]\n"
                        "fmla z27.h, z15.h, z2.h[5]\n"
                        "fmla z31.h, z15.h, z3.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z16.h, z8.h, z0.h[6]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "fmla z20.h, z8.h, z1.h[6]\n"
                        "fmla z24.h, z8.h, z2.h[6]\n"
                        "fmla z28.h, z8.h, z3.h[6]\n"
                        "fmla z17.h, z9.h, z0.h[6]\n"
                        "fmla z21.h, z9.h, z1.h[6]\n"
                        "fmla z25.h, z9.h, z2.h[6]\n"
                        "fmla z29.h, z9.h, z3.h[6]\n"
                        "fmla z18.h, z10.h, z0.h[6]\n"
                        "fmla z22.h, z10.h, z1.h[6]\n"
                        "fmla z26.h, z10.h, z2.h[6]\n"
                        "fmla z30.h, z10.h, z3.h[6]\n"
                        "fmla z19.h, z11.h, z0.h[6]\n"
                        "fmla z23.h, z11.h, z1.h[6]\n"
                        "fmla z27.h, z11.h, z2.h[6]\n"
                        "fmla z31.h, z11.h, z3.h[6]\n"
                        "fmla z16.h, z12.h, z0.h[7]\n"
                        "fmla z20.h, z12.h, z1.h[7]\n"
                        "fmla z24.h, z12.h, z2.h[7]\n"
                        "fmla z28.h, z12.h, z3.h[7]\n"
                        "fmla z17.h, z13.h, z0.h[7]\n"
                        "fmla z21.h, z13.h, z1.h[7]\n"
                        "fmla z25.h, z13.h, z2.h[7]\n"
                        "fmla z29.h, z13.h, z3.h[7]\n"
                        "fmla z18.h, z14.h, z0.h[7]\n"
                        "fmla z22.h, z14.h, z1.h[7]\n"
                        "fmla z26.h, z14.h, z2.h[7]\n"
                        "fmla z30.h, z14.h, z3.h[7]\n"
                        "fmla z19.h, z15.h, z0.h[7]\n"
                        "fmla z23.h, z15.h, z1.h[7]\n"
                        "fmla z27.h, z15.h, z2.h[7]\n"
                        "fmla z31.h, z15.h, z3.h[7]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[0]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.h, z8.h, z5.h[0]\n"
                        "fmla z24.h, z8.h, z6.h[0]\n"
                        "fmla z28.h, z8.h, z7.h[0]\n"
                        "fmla z17.h, z9.h, z4.h[0]\n"
                        "fmla z21.h, z9.h, z5.h[0]\n"
                        "fmla z25.h, z9.h, z6.h[0]\n"
                        "fmla z29.h, z9.h, z7.h[0]\n"
                        "fmla z18.h, z10.h, z4.h[0]\n"
                        "fmla z22.h, z10.h, z5.h[0]\n"
                        "fmla z26.h, z10.h, z6.h[0]\n"
                        "fmla z30.h, z10.h, z7.h[0]\n"
                        "fmla z19.h, z11.h, z4.h[0]\n"
                        "fmla z23.h, z11.h, z5.h[0]\n"
                        "fmla z27.h, z11.h, z6.h[0]\n"
                        "fmla z31.h, z11.h, z7.h[0]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[1]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.h, z12.h, z5.h[1]\n"
                        "fmla z24.h, z12.h, z6.h[1]\n"
                        "fmla z28.h, z12.h, z7.h[1]\n"
                        "fmla z17.h, z13.h, z4.h[1]\n"
                        "fmla z21.h, z13.h, z5.h[1]\n"
                        "fmla z25.h, z13.h, z6.h[1]\n"
                        "fmla z29.h, z13.h, z7.h[1]\n"
                        "fmla z18.h, z14.h, z4.h[1]\n"
                        "fmla z22.h, z14.h, z5.h[1]\n"
                        "fmla z26.h, z14.h, z6.h[1]\n"
                        "fmla z30.h, z14.h, z7.h[1]\n"
                        "fmla z19.h, z15.h, z4.h[1]\n"
                        "fmla z23.h, z15.h, z5.h[1]\n"
                        "fmla z27.h, z15.h, z6.h[1]\n"
                        "fmla z31.h, z15.h, z7.h[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[2]\n"
                        "fmla z20.h, z8.h, z5.h[2]\n"
                        "fmla z24.h, z8.h, z6.h[2]\n"
                        "fmla z28.h, z8.h, z7.h[2]\n"
                        "fmla z17.h, z9.h, z4.h[2]\n"
                        "fmla z21.h, z9.h, z5.h[2]\n"
                        "fmla z25.h, z9.h, z6.h[2]\n"
                        "fmla z29.h, z9.h, z7.h[2]\n"
                        "fmla z18.h, z10.h, z4.h[2]\n"
                        "fmla z22.h, z10.h, z5.h[2]\n"
                        "fmla z26.h, z10.h, z6.h[2]\n"
                        "fmla z30.h, z10.h, z7.h[2]\n"
                        "fmla z19.h, z11.h, z4.h[2]\n"
                        "fmla z23.h, z11.h, z5.h[2]\n"
                        "fmla z27.h, z11.h, z6.h[2]\n"
                        "fmla z31.h, z11.h, z7.h[2]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[3]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "fmla z20.h, z12.h, z5.h[3]\n"
                        "fmla z24.h, z12.h, z6.h[3]\n"
                        "fmla z28.h, z12.h, z7.h[3]\n"
                        "fmla z17.h, z13.h, z4.h[3]\n"
                        "fmla z21.h, z13.h, z5.h[3]\n"
                        "fmla z25.h, z13.h, z6.h[3]\n"
                        "fmla z29.h, z13.h, z7.h[3]\n"
                        "fmla z18.h, z14.h, z4.h[3]\n"
                        "fmla z22.h, z14.h, z5.h[3]\n"
                        "fmla z26.h, z14.h, z6.h[3]\n"
                        "fmla z30.h, z14.h, z7.h[3]\n"
                        "fmla z19.h, z15.h, z4.h[3]\n"
                        "fmla z23.h, z15.h, z5.h[3]\n"
                        "fmla z27.h, z15.h, z6.h[3]\n"
                        "fmla z31.h, z15.h, z7.h[3]\n"
                        "b.eq 5f\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[4]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "fmla z20.h, z8.h, z5.h[4]\n"
                        "fmla z24.h, z8.h, z6.h[4]\n"
                        "fmla z28.h, z8.h, z7.h[4]\n"
                        "fmla z17.h, z9.h, z4.h[4]\n"
                        "fmla z21.h, z9.h, z5.h[4]\n"
                        "fmla z25.h, z9.h, z6.h[4]\n"
                        "fmla z29.h, z9.h, z7.h[4]\n"
                        "fmla z18.h, z10.h, z4.h[4]\n"
                        "fmla z22.h, z10.h, z5.h[4]\n"
                        "fmla z26.h, z10.h, z6.h[4]\n"
                        "fmla z30.h, z10.h, z7.h[4]\n"
                        "fmla z19.h, z11.h, z4.h[4]\n"
                        "fmla z23.h, z11.h, z5.h[4]\n"
                        "fmla z27.h, z11.h, z6.h[4]\n"
                        "fmla z31.h, z11.h, z7.h[4]\n"
                        "b.eq 5f\n"
                        "ld1h z12.h, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1h z13.h, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1h z14.h, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "fmla z16.h, z12.h, z4.h[5]\n"
                        "ld1h z15.h, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "fmla z20.h, z12.h, z5.h[5]\n"
                        "fmla z24.h, z12.h, z6.h[5]\n"
                        "fmla z28.h, z12.h, z7.h[5]\n"
                        "fmla z17.h, z13.h, z4.h[5]\n"
                        "fmla z21.h, z13.h, z5.h[5]\n"
                        "fmla z25.h, z13.h, z6.h[5]\n"
                        "fmla z29.h, z13.h, z7.h[5]\n"
                        "fmla z18.h, z14.h, z4.h[5]\n"
                        "fmla z22.h, z14.h, z5.h[5]\n"
                        "fmla z26.h, z14.h, z6.h[5]\n"
                        "fmla z30.h, z14.h, z7.h[5]\n"
                        "fmla z19.h, z15.h, z4.h[5]\n"
                        "fmla z23.h, z15.h, z5.h[5]\n"
                        "fmla z27.h, z15.h, z6.h[5]\n"
                        "fmla z31.h, z15.h, z7.h[5]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "ld1h z8.h, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1h z9.h, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1h z10.h, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1h z11.h, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "fmla z16.h, z8.h, z4.h[6]\n"
                        "fmla z20.h, z8.h, z5.h[6]\n"
                        "fmla z24.h, z8.h, z6.h[6]\n"
                        "fmla z28.h, z8.h, z7.h[6]\n"
                        "fmla z17.h, z9.h, z4.h[6]\n"
                        "fmla z21.h, z9.h, z5.h[6]\n"
                        "fmla z25.h, z9.h, z6.h[6]\n"
                        "fmla z29.h, z9.h, z7.h[6]\n"
                        "fmla z18.h, z10.h, z4.h[6]\n"
                        "fmla z22.h, z10.h, z5.h[6]\n"
                        "fmla z26.h, z10.h, z6.h[6]\n"
                        "fmla z30.h, z10.h, z7.h[6]\n"
                        "fmla z19.h, z11.h, z4.h[6]\n"
                        "fmla z23.h, z11.h, z5.h[6]\n"
                        "fmla z27.h, z11.h, z6.h[6]\n"
                        "fmla z31.h, z11.h, z7.h[6]\n"
                        "5:\n"
                        "ld1rh z14.h, p7/z, [%[minptr]]\n"
                        "ld1rh z15.h, p7/z, [%[maxptr]]\n"
                        "fmax z16.h, p7/m, z16.h, z14.h\n"
                        "fmax z17.h, p7/m, z17.h, z14.h\n"
                        "fmax z18.h, p7/m, z18.h, z14.h\n"
                        "fmax z19.h, p7/m, z19.h, z14.h\n"
                        "fmin z16.h, p7/m, z16.h, z15.h\n"
                        "fmin z17.h, p7/m, z17.h, z15.h\n"
                        "fmin z18.h, p7/m, z18.h, z15.h\n"
                        "fmin z19.h, p7/m, z19.h, z15.h\n"
                        "st1h z16.h, p0, [%[c_ptr0]]\n"
                        "fmax z20.h, p7/m, z20.h, z14.h\n"
                        "fmax z21.h, p7/m, z21.h, z14.h\n"
                        "fmax z22.h, p7/m, z22.h, z14.h\n"
                        "st1h z17.h, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "fmax z23.h, p7/m, z23.h, z14.h\n"
                        "fmin z20.h, p7/m, z20.h, z15.h\n"
                        "fmin z21.h, p7/m, z21.h, z15.h\n"
                        "st1h z18.h, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "fmin z22.h, p7/m, z22.h, z15.h\n"
                        "fmin z23.h, p7/m, z23.h, z15.h\n"
                        "fmax z24.h, p7/m, z24.h, z14.h\n"
                        "st1h z19.h, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "fmax z25.h, p7/m, z25.h, z14.h\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        "fmax z26.h, p7/m, z26.h, z14.h\n"
                        "st1h z20.h, p0, [c_ptr1]\n"
                        "fmin z24.h, p7/m, z24.h, z15.h\n"
                        "fmin z25.h, p7/m, z25.h, z15.h\n"
                        "fmax z27.h, p7/m, z27.h, z14.h\n"
                        "st1h z21.h, p1, [c_ptr1, #1, MUL VL]\n"
                        "fmin z26.h, p7/m, z26.h, z15.h\n"
                        "fmax z28.h, p7/m, z28.h, z14.h\n"
                        "fmax z29.h, p7/m, z29.h, z14.h\n"
                        "st1h z22.h, p2, [c_ptr1, #2, MUL VL]\n"
                        "fmin z27.h, p7/m, z27.h, z15.h\n"
                        "fmax z30.h, p7/m, z30.h, z14.h\n"
                        "fmin z28.h, p7/m, z28.h, z15.h\n"
                        "st1h z23.h, p3, [c_ptr1, #3, MUL VL]\n"
                        "fmin z29.h, p7/m, z29.h, z15.h\n"
                        "fmax z31.h, p7/m, z31.h, z14.h\n"
                        "fmin z30.h, p7/m, z30.h, z15.h\n"
                        "st1h z24.h, p0, [c_ptr2]\n"
                        "fmin z31.h, p7/m, z31.h, z15.h\n"
                        "st1h z25.h, p1, [c_ptr2, #1, MUL VL]\n"
                        "st1h z26.h, p2, [c_ptr2, #2, MUL VL]\n"
                        "st1h z27.h, p3, [c_ptr2, #3, MUL VL]\n"
                        "st1h z28.h, p0, [c_ptr3]\n"
                        "st1h z29.h, p1, [c_ptr3, #1, MUL VL]\n"
                        "st1h z30.h, p2, [c_ptr3, #2, MUL VL]\n"
                        "st1h z31.h, p3, [c_ptr3, #3, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq a_ptr3\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        ".unreq c_ptr3\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [append] "r" (static_cast<uint64_t>(append)), [lda] "r" (ldab), [ldc] "r" (ldcb), [biasptr] "r" (biasptr), [minptr] "r" (minptr), [maxptr] "r" (maxptr), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "x4", "x5", "cc", "memory"
                    );
                    break;
            }

        }
    }
}

} // namespace arm_gemm

#endif // __ARM_FEATURE_SVE
