/*
 * Copyright (c) 2018-2019 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef __ARM_FEATURE_SVE

#include <algorithm>

#include "arm_gemm.hpp"
#include <cstdint>
#include "../../asmlib.hpp"
#include "../../utils.hpp"

namespace arm_gemm {

void sve_hybrid_s8s32_dot_4VLx4(const int8_t *A, int lda, const int8_t *B, int32_t *C, int ldc, int M, int N, int K, const int32_t *bias, Activation act, bool append) {
    const int K_stride = ((K + 3) / 4) * 4;
    const long loops_count = ((K + 16) / 32) - 1;
    K -= loops_count * 32;
    const long regs_count = (K / 16) - 1;
    K -= (regs_count + 1) * 16;
    const long leftovers = K;
    const long blocks_count = (K + 3) / 4;

    for (int y=0; y<M; y+=4) {
        const int8_t * const a_ptr0_base = A + (y * lda);
        const unsigned long ldab = lda * sizeof(int8_t);

        int32_t *c_ptr0 = C + (y * ldc);

        for (int x0=0; x0<N; x0+=(4 * get_vector_length<int32_t>())) {
            const long width = std::min((unsigned long)N-x0, (4 * get_vector_length<int32_t>()));
            long loops = loops_count;
            long regs = regs_count;
            long temp = 0;
            long blocks = blocks_count;
            const int8_t *a_ptr0 = a_ptr0_base;
            const int8_t *b_ptr0 = B + (K_stride * x0);
            const unsigned long ldcb = ldc * sizeof(int32_t);

            switch(M-y) {
                case 1:
                    __asm __volatile (
                        "whilelt p6.b, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.b\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p3.s, %[temp], %[width]\n"
                        "cbnz %[append], 1f\n"
                        "mov z16.s, #0\n"
                        "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                        "mov z17.s, #0\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "mov z18.s, #0\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "mov z19.s, #0\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z16.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "ld1rqb z4.b, p7/z, [%[a_ptr0]]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[0]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "sdot z17.s, z9.b, z4.b[0]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "sdot z18.s, z10.b, z4.b[0]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "sdot z19.s, z11.b, z4.b[0]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[1]\n"
                        "ld1rqb z0.b, p7/z, [%[a_ptr0], #-0x10]\n"
                        "sdot z17.s, z13.b, z4.b[1]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z18.s, z14.b, z4.b[1]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z19.s, z15.b, z4.b[1]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[2]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z9.b, z4.b[2]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z10.b, z4.b[2]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z11.b, z4.b[2]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[3]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "sdot z17.s, z13.b, z4.b[3]\n"
                        "sdot z18.s, z14.b, z4.b[3]\n"
                        "sdot z19.s, z15.b, z4.b[3]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "ld1rqb z4.b, p7/z, [%[a_ptr0]]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[0]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "sdot z17.s, z9.b, z4.b[0]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "sdot z18.s, z10.b, z4.b[0]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "sdot z19.s, z11.b, z4.b[0]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[1]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z17.s, z13.b, z4.b[1]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z18.s, z14.b, z4.b[1]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z15.b, z4.b[1]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[2]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z9.b, z4.b[2]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z10.b, z4.b[2]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z11.b, z4.b[2]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[3]\n"
                        "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                        "sdot z17.s, z13.b, z4.b[3]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "sdot z18.s, z14.b, z4.b[3]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        "sdot z19.s, z15.b, z4.b[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "b 5f\n"
                        "4:\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "ld1rqb z4.b, p6/z, [%[a_ptr0]]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z17.s, z9.b, z4.b[0]\n"
                        "sdot z18.s, z10.b, z4.b[0]\n"
                        "sdot z19.s, z11.b, z4.b[0]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z17.s, z13.b, z4.b[1]\n"
                        "sdot z18.s, z14.b, z4.b[1]\n"
                        "sdot z19.s, z15.b, z4.b[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[2]\n"
                        "sdot z17.s, z9.b, z4.b[2]\n"
                        "sdot z18.s, z10.b, z4.b[2]\n"
                        "sdot z19.s, z11.b, z4.b[2]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[3]\n"
                        "sdot z17.s, z13.b, z4.b[3]\n"
                        "sdot z18.s, z14.b, z4.b[3]\n"
                        "sdot z19.s, z15.b, z4.b[3]\n"
                        "5:\n"
                        "st1w z16.s, p0, [%[c_ptr0]]\n"
                        "st1w z17.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "st1w z18.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "st1w z19.s, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [append] "r" (static_cast<uint64_t>(append)), [lda] "r" (ldab), [ldc] "r" (ldcb), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "cc", "memory"
                    );
                    break;
                case 2:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "c_ptr1 .req X1\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "whilelt p6.b, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.b\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p3.s, %[temp], %[width]\n"
                        "cbnz %[append], 1f\n"
                        "mov z16.s, #0\n"
                        "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                        "mov z17.s, #0\n"
                        "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                        "mov z18.s, #0\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "mov z19.s, #0\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "mov z20.s, #0\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "mov z21.s, #0\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z22.s, #0\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "mov z23.s, #0\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z16.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1w z20.s, p0/z, [c_ptr1]\n"
                        "ld1w z21.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1w z22.s, p2/z, [c_ptr1, #2, MUL VL]\n"
                        "ld1w z23.s, p3/z, [c_ptr1, #3, MUL VL]\n"
                        "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z20.s, z8.b, z1.b[0]\n"
                        "ld1rqb z4.b, p7/z, [%[a_ptr0]]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "ld1rqb z5.b, p7/z, [a_ptr1]\n"
                        "sdot z21.s, z9.b, z1.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z22.s, z10.b, z1.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "sdot z23.s, z11.b, z1.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        "sdot z20.s, z12.b, z1.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        "sdot z21.s, z13.b, z1.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "sdot z22.s, z14.b, z1.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "sdot z23.s, z15.b, z1.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "sdot z20.s, z8.b, z1.b[2]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "sdot z21.s, z9.b, z1.b[2]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "sdot z22.s, z10.b, z1.b[2]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "sdot z23.s, z11.b, z1.b[2]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "sdot z20.s, z12.b, z1.b[3]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "sdot z21.s, z13.b, z1.b[3]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "sdot z22.s, z14.b, z1.b[3]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "ld1rqb z0.b, p7/z, [%[a_ptr0], #-0x10]\n"
                        "sdot z23.s, z15.b, z1.b[3]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[0]\n"
                        "ld1rqb z1.b, p7/z, [a_ptr1, #-0x10]\n"
                        "sdot z20.s, z8.b, z5.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z17.s, z9.b, z4.b[0]\n"
                        "sdot z21.s, z9.b, z5.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z18.s, z10.b, z4.b[0]\n"
                        "sdot z22.s, z10.b, z5.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z11.b, z4.b[0]\n"
                        "sdot z23.s, z11.b, z5.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[1]\n"
                        "sdot z20.s, z12.b, z5.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z13.b, z4.b[1]\n"
                        "sdot z21.s, z13.b, z5.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z14.b, z4.b[1]\n"
                        "sdot z22.s, z14.b, z5.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z15.b, z4.b[1]\n"
                        "sdot z23.s, z15.b, z5.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "sdot z20.s, z8.b, z5.b[2]\n"
                        "sdot z17.s, z9.b, z4.b[2]\n"
                        "sdot z21.s, z9.b, z5.b[2]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "sdot z18.s, z10.b, z4.b[2]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "sdot z22.s, z10.b, z5.b[2]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "sdot z19.s, z11.b, z4.b[2]\n"
                        "sdot z23.s, z11.b, z5.b[2]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[3]\n"
                        "sdot z20.s, z12.b, z5.b[3]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "sdot z17.s, z13.b, z4.b[3]\n"
                        "sdot z21.s, z13.b, z5.b[3]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "sdot z18.s, z14.b, z4.b[3]\n"
                        "sdot z22.s, z14.b, z5.b[3]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "sdot z19.s, z15.b, z4.b[3]\n"
                        "sdot z23.s, z15.b, z5.b[3]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z20.s, z8.b, z1.b[0]\n"
                        "ld1rqb z4.b, p7/z, [%[a_ptr0]]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "ld1rqb z5.b, p7/z, [a_ptr1]\n"
                        "sdot z21.s, z9.b, z1.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z22.s, z10.b, z1.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "sdot z23.s, z11.b, z1.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "sdot z20.s, z12.b, z1.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "sdot z21.s, z13.b, z1.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "sdot z22.s, z14.b, z1.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "sdot z23.s, z15.b, z1.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "sdot z20.s, z8.b, z1.b[2]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "sdot z21.s, z9.b, z1.b[2]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "sdot z22.s, z10.b, z1.b[2]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "sdot z23.s, z11.b, z1.b[2]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "sdot z20.s, z12.b, z1.b[3]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "sdot z21.s, z13.b, z1.b[3]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "sdot z22.s, z14.b, z1.b[3]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                        "sdot z23.s, z15.b, z1.b[3]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[0]\n"
                        "ld1rqb z1.b, p6/z, [a_ptr1, #0x10]\n"
                        "sdot z20.s, z8.b, z5.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z17.s, z9.b, z4.b[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        "sdot z21.s, z9.b, z5.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z18.s, z10.b, z4.b[0]\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        "sdot z22.s, z10.b, z5.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z11.b, z4.b[0]\n"
                        "sdot z23.s, z11.b, z5.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[1]\n"
                        "sdot z20.s, z12.b, z5.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z13.b, z4.b[1]\n"
                        "sdot z21.s, z13.b, z5.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z14.b, z4.b[1]\n"
                        "sdot z22.s, z14.b, z5.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z15.b, z4.b[1]\n"
                        "sdot z23.s, z15.b, z5.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "sdot z20.s, z8.b, z5.b[2]\n"
                        "sdot z17.s, z9.b, z4.b[2]\n"
                        "sdot z21.s, z9.b, z5.b[2]\n"
                        "sdot z18.s, z10.b, z4.b[2]\n"
                        "sdot z22.s, z10.b, z5.b[2]\n"
                        "sdot z19.s, z11.b, z4.b[2]\n"
                        "sdot z23.s, z11.b, z5.b[2]\n"
                        "sdot z16.s, z12.b, z4.b[3]\n"
                        "sdot z20.s, z12.b, z5.b[3]\n"
                        "sdot z17.s, z13.b, z4.b[3]\n"
                        "sdot z21.s, z13.b, z5.b[3]\n"
                        "sdot z18.s, z14.b, z4.b[3]\n"
                        "sdot z22.s, z14.b, z5.b[3]\n"
                        "sdot z19.s, z15.b, z4.b[3]\n"
                        "sdot z23.s, z15.b, z5.b[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z20.s, z8.b, z1.b[0]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "sdot z21.s, z9.b, z1.b[0]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "sdot z22.s, z10.b, z1.b[0]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "sdot z23.s, z11.b, z1.b[0]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z20.s, z12.b, z1.b[1]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "sdot z21.s, z13.b, z1.b[1]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "sdot z22.s, z14.b, z1.b[1]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "sdot z23.s, z15.b, z1.b[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "sdot z20.s, z8.b, z1.b[2]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "sdot z21.s, z9.b, z1.b[2]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "sdot z22.s, z10.b, z1.b[2]\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "sdot z23.s, z11.b, z1.b[2]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "sdot z20.s, z12.b, z1.b[3]\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "sdot z21.s, z13.b, z1.b[3]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "sdot z22.s, z14.b, z1.b[3]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "sdot z23.s, z15.b, z1.b[3]\n"
                        "b 5f\n"
                        "4:\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z20.s, z8.b, z1.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "ld1rqb z4.b, p6/z, [%[a_ptr0]]\n"
                        "sdot z21.s, z9.b, z1.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "ld1rqb z5.b, p6/z, [a_ptr1]\n"
                        "sdot z22.s, z10.b, z1.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        "sdot z23.s, z11.b, z1.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        "sdot z20.s, z12.b, z1.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "sdot z21.s, z13.b, z1.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "sdot z22.s, z14.b, z1.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "sdot z23.s, z15.b, z1.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "sdot z20.s, z8.b, z1.b[2]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "sdot z21.s, z9.b, z1.b[2]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "sdot z22.s, z10.b, z1.b[2]\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "sdot z23.s, z11.b, z1.b[2]\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "sdot z20.s, z12.b, z1.b[3]\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "sdot z21.s, z13.b, z1.b[3]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "sdot z22.s, z14.b, z1.b[3]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "sdot z23.s, z15.b, z1.b[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z20.s, z8.b, z5.b[0]\n"
                        "sdot z17.s, z9.b, z4.b[0]\n"
                        "sdot z21.s, z9.b, z5.b[0]\n"
                        "sdot z18.s, z10.b, z4.b[0]\n"
                        "sdot z22.s, z10.b, z5.b[0]\n"
                        "sdot z19.s, z11.b, z4.b[0]\n"
                        "sdot z23.s, z11.b, z5.b[0]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z20.s, z12.b, z5.b[1]\n"
                        "sdot z17.s, z13.b, z4.b[1]\n"
                        "sdot z21.s, z13.b, z5.b[1]\n"
                        "sdot z18.s, z14.b, z4.b[1]\n"
                        "sdot z22.s, z14.b, z5.b[1]\n"
                        "sdot z19.s, z15.b, z4.b[1]\n"
                        "sdot z23.s, z15.b, z5.b[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[2]\n"
                        "sdot z20.s, z8.b, z5.b[2]\n"
                        "sdot z17.s, z9.b, z4.b[2]\n"
                        "sdot z21.s, z9.b, z5.b[2]\n"
                        "sdot z18.s, z10.b, z4.b[2]\n"
                        "sdot z22.s, z10.b, z5.b[2]\n"
                        "sdot z19.s, z11.b, z4.b[2]\n"
                        "sdot z23.s, z11.b, z5.b[2]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[3]\n"
                        "sdot z20.s, z12.b, z5.b[3]\n"
                        "sdot z17.s, z13.b, z4.b[3]\n"
                        "sdot z21.s, z13.b, z5.b[3]\n"
                        "sdot z18.s, z14.b, z4.b[3]\n"
                        "sdot z22.s, z14.b, z5.b[3]\n"
                        "sdot z19.s, z15.b, z4.b[3]\n"
                        "sdot z23.s, z15.b, z5.b[3]\n"
                        "5:\n"
                        "st1w z16.s, p0, [%[c_ptr0]]\n"
                        "st1w z17.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "st1w z18.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "st1w z19.s, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        "st1w z20.s, p0, [c_ptr1]\n"
                        "st1w z21.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "st1w z22.s, p2, [c_ptr1, #2, MUL VL]\n"
                        "st1w z23.s, p3, [c_ptr1, #3, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq c_ptr1\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [append] "r" (static_cast<uint64_t>(append)), [lda] "r" (ldab), [ldc] "r" (ldcb), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "cc", "memory"
                    );
                    break;
                case 3:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "c_ptr1 .req X2\n"
                        "c_ptr2 .req X3\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "whilelt p6.b, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.b\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p3.s, %[temp], %[width]\n"
                        "cbnz %[append], 1f\n"
                        "mov z16.s, #0\n"
                        "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                        "mov z17.s, #0\n"
                        "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                        "mov z18.s, #0\n"
                        "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                        "mov z19.s, #0\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "mov z20.s, #0\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "mov z21.s, #0\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "mov z22.s, #0\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z23.s, #0\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "mov z24.s, #0\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "mov z25.s, #0\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "mov z26.s, #0\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "mov z27.s, #0\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z16.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1w z20.s, p0/z, [c_ptr1]\n"
                        "ld1w z21.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1w z22.s, p2/z, [c_ptr1, #2, MUL VL]\n"
                        "ld1w z23.s, p3/z, [c_ptr1, #3, MUL VL]\n"
                        "ld1w z24.s, p0/z, [c_ptr2]\n"
                        "ld1w z25.s, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "ld1w z26.s, p2/z, [c_ptr2, #2, MUL VL]\n"
                        "ld1w z27.s, p3/z, [c_ptr2, #3, MUL VL]\n"
                        "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z20.s, z8.b, z1.b[0]\n"
                        "ld1rqb z4.b, p7/z, [%[a_ptr0]]\n"
                        "sdot z24.s, z8.b, z2.b[0]\n"
                        "ld1rqb z5.b, p7/z, [a_ptr1]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "ld1rqb z6.b, p7/z, [a_ptr2]\n"
                        "sdot z21.s, z9.b, z1.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z25.s, z9.b, z2.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "sdot z22.s, z10.b, z1.b[0]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        "sdot z26.s, z10.b, z2.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        "sdot z23.s, z11.b, z1.b[0]\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        "sdot z27.s, z11.b, z2.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "sdot z20.s, z12.b, z1.b[1]\n"
                        "sdot z24.s, z12.b, z2.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "sdot z21.s, z13.b, z1.b[1]\n"
                        "sdot z25.s, z13.b, z2.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "sdot z22.s, z14.b, z1.b[1]\n"
                        "sdot z26.s, z14.b, z2.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "sdot z23.s, z15.b, z1.b[1]\n"
                        "sdot z27.s, z15.b, z2.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "sdot z20.s, z8.b, z1.b[2]\n"
                        "sdot z24.s, z8.b, z2.b[2]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "sdot z21.s, z9.b, z1.b[2]\n"
                        "sdot z25.s, z9.b, z2.b[2]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "sdot z22.s, z10.b, z1.b[2]\n"
                        "sdot z26.s, z10.b, z2.b[2]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "sdot z23.s, z11.b, z1.b[2]\n"
                        "sdot z27.s, z11.b, z2.b[2]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "sdot z20.s, z12.b, z1.b[3]\n"
                        "sdot z24.s, z12.b, z2.b[3]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "sdot z21.s, z13.b, z1.b[3]\n"
                        "sdot z25.s, z13.b, z2.b[3]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "sdot z22.s, z14.b, z1.b[3]\n"
                        "sdot z26.s, z14.b, z2.b[3]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "ld1rqb z0.b, p7/z, [%[a_ptr0], #-0x10]\n"
                        "sdot z23.s, z15.b, z1.b[3]\n"
                        "ld1rqb z1.b, p7/z, [a_ptr1, #-0x10]\n"
                        "sdot z27.s, z15.b, z2.b[3]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[0]\n"
                        "ld1rqb z2.b, p7/z, [a_ptr2, #-0x10]\n"
                        "sdot z20.s, z8.b, z5.b[0]\n"
                        "sdot z24.s, z8.b, z6.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z17.s, z9.b, z4.b[0]\n"
                        "sdot z21.s, z9.b, z5.b[0]\n"
                        "sdot z25.s, z9.b, z6.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z18.s, z10.b, z4.b[0]\n"
                        "sdot z22.s, z10.b, z5.b[0]\n"
                        "sdot z26.s, z10.b, z6.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z11.b, z4.b[0]\n"
                        "sdot z23.s, z11.b, z5.b[0]\n"
                        "sdot z27.s, z11.b, z6.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[1]\n"
                        "sdot z20.s, z12.b, z5.b[1]\n"
                        "sdot z24.s, z12.b, z6.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z13.b, z4.b[1]\n"
                        "sdot z21.s, z13.b, z5.b[1]\n"
                        "sdot z25.s, z13.b, z6.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z14.b, z4.b[1]\n"
                        "sdot z22.s, z14.b, z5.b[1]\n"
                        "sdot z26.s, z14.b, z6.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z15.b, z4.b[1]\n"
                        "sdot z23.s, z15.b, z5.b[1]\n"
                        "sdot z27.s, z15.b, z6.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "sdot z20.s, z8.b, z5.b[2]\n"
                        "sdot z24.s, z8.b, z6.b[2]\n"
                        "sdot z17.s, z9.b, z4.b[2]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "sdot z21.s, z9.b, z5.b[2]\n"
                        "sdot z25.s, z9.b, z6.b[2]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "sdot z18.s, z10.b, z4.b[2]\n"
                        "sdot z22.s, z10.b, z5.b[2]\n"
                        "sdot z26.s, z10.b, z6.b[2]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "sdot z19.s, z11.b, z4.b[2]\n"
                        "sdot z23.s, z11.b, z5.b[2]\n"
                        "sdot z27.s, z11.b, z6.b[2]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[3]\n"
                        "sdot z20.s, z12.b, z5.b[3]\n"
                        "sdot z24.s, z12.b, z6.b[3]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "sdot z17.s, z13.b, z4.b[3]\n"
                        "sdot z21.s, z13.b, z5.b[3]\n"
                        "sdot z25.s, z13.b, z6.b[3]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "sdot z18.s, z14.b, z4.b[3]\n"
                        "sdot z22.s, z14.b, z5.b[3]\n"
                        "sdot z26.s, z14.b, z6.b[3]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "sdot z19.s, z15.b, z4.b[3]\n"
                        "sdot z23.s, z15.b, z5.b[3]\n"
                        "sdot z27.s, z15.b, z6.b[3]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z20.s, z8.b, z1.b[0]\n"
                        "ld1rqb z4.b, p7/z, [%[a_ptr0]]\n"
                        "sdot z24.s, z8.b, z2.b[0]\n"
                        "ld1rqb z5.b, p7/z, [a_ptr1]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "ld1rqb z6.b, p7/z, [a_ptr2]\n"
                        "sdot z21.s, z9.b, z1.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z25.s, z9.b, z2.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "sdot z22.s, z10.b, z1.b[0]\n"
                        "sdot z26.s, z10.b, z2.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "sdot z23.s, z11.b, z1.b[0]\n"
                        "sdot z27.s, z11.b, z2.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "sdot z20.s, z12.b, z1.b[1]\n"
                        "sdot z24.s, z12.b, z2.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "sdot z21.s, z13.b, z1.b[1]\n"
                        "sdot z25.s, z13.b, z2.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "sdot z22.s, z14.b, z1.b[1]\n"
                        "sdot z26.s, z14.b, z2.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "sdot z23.s, z15.b, z1.b[1]\n"
                        "sdot z27.s, z15.b, z2.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "sdot z20.s, z8.b, z1.b[2]\n"
                        "sdot z24.s, z8.b, z2.b[2]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "sdot z21.s, z9.b, z1.b[2]\n"
                        "sdot z25.s, z9.b, z2.b[2]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "sdot z22.s, z10.b, z1.b[2]\n"
                        "sdot z26.s, z10.b, z2.b[2]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "sdot z23.s, z11.b, z1.b[2]\n"
                        "sdot z27.s, z11.b, z2.b[2]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "sdot z20.s, z12.b, z1.b[3]\n"
                        "sdot z24.s, z12.b, z2.b[3]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "sdot z21.s, z13.b, z1.b[3]\n"
                        "sdot z25.s, z13.b, z2.b[3]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "sdot z22.s, z14.b, z1.b[3]\n"
                        "sdot z26.s, z14.b, z2.b[3]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                        "sdot z23.s, z15.b, z1.b[3]\n"
                        "ld1rqb z1.b, p6/z, [a_ptr1, #0x10]\n"
                        "sdot z27.s, z15.b, z2.b[3]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[0]\n"
                        "ld1rqb z2.b, p6/z, [a_ptr2, #0x10]\n"
                        "sdot z20.s, z8.b, z5.b[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        "sdot z24.s, z8.b, z6.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z17.s, z9.b, z4.b[0]\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        "sdot z21.s, z9.b, z5.b[0]\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        "sdot z25.s, z9.b, z6.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z18.s, z10.b, z4.b[0]\n"
                        "sdot z22.s, z10.b, z5.b[0]\n"
                        "sdot z26.s, z10.b, z6.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z11.b, z4.b[0]\n"
                        "sdot z23.s, z11.b, z5.b[0]\n"
                        "sdot z27.s, z11.b, z6.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[1]\n"
                        "sdot z20.s, z12.b, z5.b[1]\n"
                        "sdot z24.s, z12.b, z6.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z13.b, z4.b[1]\n"
                        "sdot z21.s, z13.b, z5.b[1]\n"
                        "sdot z25.s, z13.b, z6.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z14.b, z4.b[1]\n"
                        "sdot z22.s, z14.b, z5.b[1]\n"
                        "sdot z26.s, z14.b, z6.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z15.b, z4.b[1]\n"
                        "sdot z23.s, z15.b, z5.b[1]\n"
                        "sdot z27.s, z15.b, z6.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "sdot z20.s, z8.b, z5.b[2]\n"
                        "sdot z24.s, z8.b, z6.b[2]\n"
                        "sdot z17.s, z9.b, z4.b[2]\n"
                        "sdot z21.s, z9.b, z5.b[2]\n"
                        "sdot z25.s, z9.b, z6.b[2]\n"
                        "sdot z18.s, z10.b, z4.b[2]\n"
                        "sdot z22.s, z10.b, z5.b[2]\n"
                        "sdot z26.s, z10.b, z6.b[2]\n"
                        "sdot z19.s, z11.b, z4.b[2]\n"
                        "sdot z23.s, z11.b, z5.b[2]\n"
                        "sdot z27.s, z11.b, z6.b[2]\n"
                        "sdot z16.s, z12.b, z4.b[3]\n"
                        "sdot z20.s, z12.b, z5.b[3]\n"
                        "sdot z24.s, z12.b, z6.b[3]\n"
                        "sdot z17.s, z13.b, z4.b[3]\n"
                        "sdot z21.s, z13.b, z5.b[3]\n"
                        "sdot z25.s, z13.b, z6.b[3]\n"
                        "sdot z18.s, z14.b, z4.b[3]\n"
                        "sdot z22.s, z14.b, z5.b[3]\n"
                        "sdot z26.s, z14.b, z6.b[3]\n"
                        "sdot z19.s, z15.b, z4.b[3]\n"
                        "sdot z23.s, z15.b, z5.b[3]\n"
                        "sdot z27.s, z15.b, z6.b[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z20.s, z8.b, z1.b[0]\n"
                        "sdot z24.s, z8.b, z2.b[0]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "sdot z21.s, z9.b, z1.b[0]\n"
                        "sdot z25.s, z9.b, z2.b[0]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "sdot z22.s, z10.b, z1.b[0]\n"
                        "sdot z26.s, z10.b, z2.b[0]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "sdot z23.s, z11.b, z1.b[0]\n"
                        "sdot z27.s, z11.b, z2.b[0]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z20.s, z12.b, z1.b[1]\n"
                        "sdot z24.s, z12.b, z2.b[1]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "sdot z21.s, z13.b, z1.b[1]\n"
                        "sdot z25.s, z13.b, z2.b[1]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "sdot z22.s, z14.b, z1.b[1]\n"
                        "sdot z26.s, z14.b, z2.b[1]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "sdot z23.s, z15.b, z1.b[1]\n"
                        "sdot z27.s, z15.b, z2.b[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "sdot z20.s, z8.b, z1.b[2]\n"
                        "sdot z24.s, z8.b, z2.b[2]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "sdot z21.s, z9.b, z1.b[2]\n"
                        "sdot z25.s, z9.b, z2.b[2]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "sdot z22.s, z10.b, z1.b[2]\n"
                        "sdot z26.s, z10.b, z2.b[2]\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "sdot z23.s, z11.b, z1.b[2]\n"
                        "sdot z27.s, z11.b, z2.b[2]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "sdot z20.s, z12.b, z1.b[3]\n"
                        "sdot z24.s, z12.b, z2.b[3]\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "sdot z21.s, z13.b, z1.b[3]\n"
                        "sdot z25.s, z13.b, z2.b[3]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "sdot z22.s, z14.b, z1.b[3]\n"
                        "sdot z26.s, z14.b, z2.b[3]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "sdot z23.s, z15.b, z1.b[3]\n"
                        "sdot z27.s, z15.b, z2.b[3]\n"
                        "b 5f\n"
                        "4:\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z20.s, z8.b, z1.b[0]\n"
                        "ld1rqb z4.b, p6/z, [%[a_ptr0]]\n"
                        "sdot z24.s, z8.b, z2.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "ld1rqb z5.b, p6/z, [a_ptr1]\n"
                        "sdot z21.s, z9.b, z1.b[0]\n"
                        "ld1rqb z6.b, p6/z, [a_ptr2]\n"
                        "sdot z25.s, z9.b, z2.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        "sdot z22.s, z10.b, z1.b[0]\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        "sdot z26.s, z10.b, z2.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        "sdot z23.s, z11.b, z1.b[0]\n"
                        "sdot z27.s, z11.b, z2.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "sdot z20.s, z12.b, z1.b[1]\n"
                        "sdot z24.s, z12.b, z2.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "sdot z21.s, z13.b, z1.b[1]\n"
                        "sdot z25.s, z13.b, z2.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "sdot z22.s, z14.b, z1.b[1]\n"
                        "sdot z26.s, z14.b, z2.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "sdot z23.s, z15.b, z1.b[1]\n"
                        "sdot z27.s, z15.b, z2.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "sdot z20.s, z8.b, z1.b[2]\n"
                        "sdot z24.s, z8.b, z2.b[2]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "sdot z21.s, z9.b, z1.b[2]\n"
                        "sdot z25.s, z9.b, z2.b[2]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "sdot z22.s, z10.b, z1.b[2]\n"
                        "sdot z26.s, z10.b, z2.b[2]\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "sdot z23.s, z11.b, z1.b[2]\n"
                        "sdot z27.s, z11.b, z2.b[2]\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "sdot z20.s, z12.b, z1.b[3]\n"
                        "sdot z24.s, z12.b, z2.b[3]\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "sdot z21.s, z13.b, z1.b[3]\n"
                        "sdot z25.s, z13.b, z2.b[3]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "sdot z22.s, z14.b, z1.b[3]\n"
                        "sdot z26.s, z14.b, z2.b[3]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "sdot z23.s, z15.b, z1.b[3]\n"
                        "sdot z27.s, z15.b, z2.b[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z20.s, z8.b, z5.b[0]\n"
                        "sdot z24.s, z8.b, z6.b[0]\n"
                        "sdot z17.s, z9.b, z4.b[0]\n"
                        "sdot z21.s, z9.b, z5.b[0]\n"
                        "sdot z25.s, z9.b, z6.b[0]\n"
                        "sdot z18.s, z10.b, z4.b[0]\n"
                        "sdot z22.s, z10.b, z5.b[0]\n"
                        "sdot z26.s, z10.b, z6.b[0]\n"
                        "sdot z19.s, z11.b, z4.b[0]\n"
                        "sdot z23.s, z11.b, z5.b[0]\n"
                        "sdot z27.s, z11.b, z6.b[0]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z20.s, z12.b, z5.b[1]\n"
                        "sdot z24.s, z12.b, z6.b[1]\n"
                        "sdot z17.s, z13.b, z4.b[1]\n"
                        "sdot z21.s, z13.b, z5.b[1]\n"
                        "sdot z25.s, z13.b, z6.b[1]\n"
                        "sdot z18.s, z14.b, z4.b[1]\n"
                        "sdot z22.s, z14.b, z5.b[1]\n"
                        "sdot z26.s, z14.b, z6.b[1]\n"
                        "sdot z19.s, z15.b, z4.b[1]\n"
                        "sdot z23.s, z15.b, z5.b[1]\n"
                        "sdot z27.s, z15.b, z6.b[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[2]\n"
                        "sdot z20.s, z8.b, z5.b[2]\n"
                        "sdot z24.s, z8.b, z6.b[2]\n"
                        "sdot z17.s, z9.b, z4.b[2]\n"
                        "sdot z21.s, z9.b, z5.b[2]\n"
                        "sdot z25.s, z9.b, z6.b[2]\n"
                        "sdot z18.s, z10.b, z4.b[2]\n"
                        "sdot z22.s, z10.b, z5.b[2]\n"
                        "sdot z26.s, z10.b, z6.b[2]\n"
                        "sdot z19.s, z11.b, z4.b[2]\n"
                        "sdot z23.s, z11.b, z5.b[2]\n"
                        "sdot z27.s, z11.b, z6.b[2]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[3]\n"
                        "sdot z20.s, z12.b, z5.b[3]\n"
                        "sdot z24.s, z12.b, z6.b[3]\n"
                        "sdot z17.s, z13.b, z4.b[3]\n"
                        "sdot z21.s, z13.b, z5.b[3]\n"
                        "sdot z25.s, z13.b, z6.b[3]\n"
                        "sdot z18.s, z14.b, z4.b[3]\n"
                        "sdot z22.s, z14.b, z5.b[3]\n"
                        "sdot z26.s, z14.b, z6.b[3]\n"
                        "sdot z19.s, z15.b, z4.b[3]\n"
                        "sdot z23.s, z15.b, z5.b[3]\n"
                        "sdot z27.s, z15.b, z6.b[3]\n"
                        "5:\n"
                        "st1w z16.s, p0, [%[c_ptr0]]\n"
                        "st1w z17.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "st1w z18.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "st1w z19.s, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        "st1w z20.s, p0, [c_ptr1]\n"
                        "st1w z21.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "st1w z22.s, p2, [c_ptr1, #2, MUL VL]\n"
                        "st1w z23.s, p3, [c_ptr1, #3, MUL VL]\n"
                        "st1w z24.s, p0, [c_ptr2]\n"
                        "st1w z25.s, p1, [c_ptr2, #1, MUL VL]\n"
                        "st1w z26.s, p2, [c_ptr2, #2, MUL VL]\n"
                        "st1w z27.s, p3, [c_ptr2, #3, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [append] "r" (static_cast<uint64_t>(append)), [lda] "r" (ldab), [ldc] "r" (ldcb), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "cc", "memory"
                    );
                    break;
                default:
                case 4:
                    __asm __volatile (
                        "a_ptr1 .req X0\n"
                        "a_ptr2 .req X1\n"
                        "a_ptr3 .req X2\n"
                        "c_ptr1 .req X3\n"
                        "c_ptr2 .req X4\n"
                        "c_ptr3 .req X5\n"
                        "add a_ptr1, %[a_ptr0], %[lda]\n"
                        "add c_ptr1, %[c_ptr0], %[ldc]\n"
                        "add a_ptr2, a_ptr1, %[lda]\n"
                        "add c_ptr2, c_ptr1, %[ldc]\n"
                        "add a_ptr3, a_ptr2, %[lda]\n"
                        "add c_ptr3, c_ptr2, %[ldc]\n"
                        "whilelt p6.b, %[temp], %[leftovers]\n"
                        "whilelt p0.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "ptrue p7.b\n"
                        "whilelt p1.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p2.s, %[temp], %[width]\n"
                        "incw %[temp], all, mul #1\n"
                        "whilelt p3.s, %[temp], %[width]\n"
                        "cbnz %[append], 1f\n"
                        "mov z16.s, #0\n"
                        "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                        "mov z17.s, #0\n"
                        "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                        "mov z18.s, #0\n"
                        "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                        "mov z19.s, #0\n"
                        "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                        "mov z20.s, #0\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "mov z21.s, #0\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "mov z22.s, #0\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "mov z23.s, #0\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "mov z24.s, #0\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "mov z25.s, #0\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "mov z26.s, #0\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "mov z27.s, #0\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "mov z28.s, #0\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "mov z29.s, #0\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "mov z30.s, #0\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "mov z31.s, #0\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "b 3f\n"
                        "1:\n"
                        "ld1w z16.s, p0/z, [%[c_ptr0]]\n"
                        "ld1w z17.s, p1/z, [%[c_ptr0], #1, MUL VL]\n"
                        "ld1w z18.s, p2/z, [%[c_ptr0], #2, MUL VL]\n"
                        "ld1w z19.s, p3/z, [%[c_ptr0], #3, MUL VL]\n"
                        "ld1w z20.s, p0/z, [c_ptr1]\n"
                        "ld1w z21.s, p1/z, [c_ptr1, #1, MUL VL]\n"
                        "ld1w z22.s, p2/z, [c_ptr1, #2, MUL VL]\n"
                        "ld1w z23.s, p3/z, [c_ptr1, #3, MUL VL]\n"
                        "ld1w z24.s, p0/z, [c_ptr2]\n"
                        "ld1w z25.s, p1/z, [c_ptr2, #1, MUL VL]\n"
                        "ld1w z26.s, p2/z, [c_ptr2, #2, MUL VL]\n"
                        "ld1w z27.s, p3/z, [c_ptr2, #3, MUL VL]\n"
                        "ld1w z28.s, p0/z, [c_ptr3]\n"
                        "ld1w z29.s, p1/z, [c_ptr3, #1, MUL VL]\n"
                        "ld1w z30.s, p2/z, [c_ptr3, #2, MUL VL]\n"
                        "ld1w z31.s, p3/z, [c_ptr3, #3, MUL VL]\n"
                        "ld1rqb z0.b, p7/z, [%[a_ptr0]]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x10\n"
                        "ld1rqb z1.b, p7/z, [a_ptr1]\n"
                        "add a_ptr1, a_ptr1, #0x10\n"
                        "ld1rqb z2.b, p7/z, [a_ptr2]\n"
                        "add a_ptr2, a_ptr2, #0x10\n"
                        "ld1rqb z3.b, p7/z, [a_ptr3]\n"
                        "add a_ptr3, a_ptr3, #0x10\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "cbz %[loops], 2f\n"
                        "3:\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z20.s, z8.b, z1.b[0]\n"
                        "ld1rqb z4.b, p7/z, [%[a_ptr0]]\n"
                        "sdot z24.s, z8.b, z2.b[0]\n"
                        "ld1rqb z5.b, p7/z, [a_ptr1]\n"
                        "sdot z28.s, z8.b, z3.b[0]\n"
                        "ld1rqb z6.b, p7/z, [a_ptr2]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "ld1rqb z7.b, p7/z, [a_ptr3]\n"
                        "sdot z21.s, z9.b, z1.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z25.s, z9.b, z2.b[0]\n"
                        "subs %[loops], %[loops], #0x1\n"
                        "sdot z29.s, z9.b, z3.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "add %[a_ptr0], %[a_ptr0], #0x20\n"
                        "sdot z22.s, z10.b, z1.b[0]\n"
                        "add a_ptr1, a_ptr1, #0x20\n"
                        "sdot z26.s, z10.b, z2.b[0]\n"
                        "add a_ptr2, a_ptr2, #0x20\n"
                        "sdot z30.s, z10.b, z3.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "add a_ptr3, a_ptr3, #0x20\n"
                        "sdot z23.s, z11.b, z1.b[0]\n"
                        "sdot z27.s, z11.b, z2.b[0]\n"
                        "sdot z31.s, z11.b, z3.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "sdot z20.s, z12.b, z1.b[1]\n"
                        "sdot z24.s, z12.b, z2.b[1]\n"
                        "sdot z28.s, z12.b, z3.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "sdot z21.s, z13.b, z1.b[1]\n"
                        "sdot z25.s, z13.b, z2.b[1]\n"
                        "sdot z29.s, z13.b, z3.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "sdot z22.s, z14.b, z1.b[1]\n"
                        "sdot z26.s, z14.b, z2.b[1]\n"
                        "sdot z30.s, z14.b, z3.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "sdot z23.s, z15.b, z1.b[1]\n"
                        "sdot z27.s, z15.b, z2.b[1]\n"
                        "sdot z31.s, z15.b, z3.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "sdot z20.s, z8.b, z1.b[2]\n"
                        "sdot z24.s, z8.b, z2.b[2]\n"
                        "sdot z28.s, z8.b, z3.b[2]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "sdot z21.s, z9.b, z1.b[2]\n"
                        "sdot z25.s, z9.b, z2.b[2]\n"
                        "sdot z29.s, z9.b, z3.b[2]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "sdot z22.s, z10.b, z1.b[2]\n"
                        "sdot z26.s, z10.b, z2.b[2]\n"
                        "sdot z30.s, z10.b, z3.b[2]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "sdot z23.s, z11.b, z1.b[2]\n"
                        "sdot z27.s, z11.b, z2.b[2]\n"
                        "sdot z31.s, z11.b, z3.b[2]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "sdot z20.s, z12.b, z1.b[3]\n"
                        "sdot z24.s, z12.b, z2.b[3]\n"
                        "sdot z28.s, z12.b, z3.b[3]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "sdot z21.s, z13.b, z1.b[3]\n"
                        "sdot z25.s, z13.b, z2.b[3]\n"
                        "sdot z29.s, z13.b, z3.b[3]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "sdot z22.s, z14.b, z1.b[3]\n"
                        "sdot z26.s, z14.b, z2.b[3]\n"
                        "sdot z30.s, z14.b, z3.b[3]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "ld1rqb z0.b, p7/z, [%[a_ptr0], #-0x10]\n"
                        "sdot z23.s, z15.b, z1.b[3]\n"
                        "ld1rqb z1.b, p7/z, [a_ptr1, #-0x10]\n"
                        "sdot z27.s, z15.b, z2.b[3]\n"
                        "ld1rqb z2.b, p7/z, [a_ptr2, #-0x10]\n"
                        "sdot z31.s, z15.b, z3.b[3]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[0]\n"
                        "ld1rqb z3.b, p7/z, [a_ptr3, #-0x10]\n"
                        "sdot z20.s, z8.b, z5.b[0]\n"
                        "sdot z24.s, z8.b, z6.b[0]\n"
                        "sdot z28.s, z8.b, z7.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z17.s, z9.b, z4.b[0]\n"
                        "sdot z21.s, z9.b, z5.b[0]\n"
                        "sdot z25.s, z9.b, z6.b[0]\n"
                        "sdot z29.s, z9.b, z7.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z18.s, z10.b, z4.b[0]\n"
                        "sdot z22.s, z10.b, z5.b[0]\n"
                        "sdot z26.s, z10.b, z6.b[0]\n"
                        "sdot z30.s, z10.b, z7.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z11.b, z4.b[0]\n"
                        "sdot z23.s, z11.b, z5.b[0]\n"
                        "sdot z27.s, z11.b, z6.b[0]\n"
                        "sdot z31.s, z11.b, z7.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[1]\n"
                        "sdot z20.s, z12.b, z5.b[1]\n"
                        "sdot z24.s, z12.b, z6.b[1]\n"
                        "sdot z28.s, z12.b, z7.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z13.b, z4.b[1]\n"
                        "sdot z21.s, z13.b, z5.b[1]\n"
                        "sdot z25.s, z13.b, z6.b[1]\n"
                        "sdot z29.s, z13.b, z7.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z14.b, z4.b[1]\n"
                        "sdot z22.s, z14.b, z5.b[1]\n"
                        "sdot z26.s, z14.b, z6.b[1]\n"
                        "sdot z30.s, z14.b, z7.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z15.b, z4.b[1]\n"
                        "sdot z23.s, z15.b, z5.b[1]\n"
                        "sdot z27.s, z15.b, z6.b[1]\n"
                        "sdot z31.s, z15.b, z7.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "sdot z20.s, z8.b, z5.b[2]\n"
                        "sdot z24.s, z8.b, z6.b[2]\n"
                        "sdot z28.s, z8.b, z7.b[2]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "sdot z17.s, z9.b, z4.b[2]\n"
                        "sdot z21.s, z9.b, z5.b[2]\n"
                        "sdot z25.s, z9.b, z6.b[2]\n"
                        "sdot z29.s, z9.b, z7.b[2]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "sdot z18.s, z10.b, z4.b[2]\n"
                        "sdot z22.s, z10.b, z5.b[2]\n"
                        "sdot z26.s, z10.b, z6.b[2]\n"
                        "sdot z30.s, z10.b, z7.b[2]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "sdot z19.s, z11.b, z4.b[2]\n"
                        "sdot z23.s, z11.b, z5.b[2]\n"
                        "sdot z27.s, z11.b, z6.b[2]\n"
                        "sdot z31.s, z11.b, z7.b[2]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[3]\n"
                        "sdot z20.s, z12.b, z5.b[3]\n"
                        "sdot z24.s, z12.b, z6.b[3]\n"
                        "sdot z28.s, z12.b, z7.b[3]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "sdot z17.s, z13.b, z4.b[3]\n"
                        "sdot z21.s, z13.b, z5.b[3]\n"
                        "sdot z25.s, z13.b, z6.b[3]\n"
                        "sdot z29.s, z13.b, z7.b[3]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "sdot z18.s, z14.b, z4.b[3]\n"
                        "sdot z22.s, z14.b, z5.b[3]\n"
                        "sdot z26.s, z14.b, z6.b[3]\n"
                        "sdot z30.s, z14.b, z7.b[3]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "sdot z19.s, z15.b, z4.b[3]\n"
                        "sdot z23.s, z15.b, z5.b[3]\n"
                        "sdot z27.s, z15.b, z6.b[3]\n"
                        "sdot z31.s, z15.b, z7.b[3]\n"
                        "b.ne 3b\n"
                        "2:\n"
                        "cbz %[regs], 4f\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z20.s, z8.b, z1.b[0]\n"
                        "ld1rqb z4.b, p7/z, [%[a_ptr0]]\n"
                        "sdot z24.s, z8.b, z2.b[0]\n"
                        "ld1rqb z5.b, p7/z, [a_ptr1]\n"
                        "sdot z28.s, z8.b, z3.b[0]\n"
                        "ld1rqb z6.b, p7/z, [a_ptr2]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "ld1rqb z7.b, p7/z, [a_ptr3]\n"
                        "sdot z21.s, z9.b, z1.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z25.s, z9.b, z2.b[0]\n"
                        "sdot z29.s, z9.b, z3.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "sdot z22.s, z10.b, z1.b[0]\n"
                        "sdot z26.s, z10.b, z2.b[0]\n"
                        "sdot z30.s, z10.b, z3.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "sdot z23.s, z11.b, z1.b[0]\n"
                        "sdot z27.s, z11.b, z2.b[0]\n"
                        "sdot z31.s, z11.b, z3.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "sdot z20.s, z12.b, z1.b[1]\n"
                        "sdot z24.s, z12.b, z2.b[1]\n"
                        "sdot z28.s, z12.b, z3.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "sdot z21.s, z13.b, z1.b[1]\n"
                        "sdot z25.s, z13.b, z2.b[1]\n"
                        "sdot z29.s, z13.b, z3.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "sdot z22.s, z14.b, z1.b[1]\n"
                        "sdot z26.s, z14.b, z2.b[1]\n"
                        "sdot z30.s, z14.b, z3.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "sdot z23.s, z15.b, z1.b[1]\n"
                        "sdot z27.s, z15.b, z2.b[1]\n"
                        "sdot z31.s, z15.b, z3.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "sdot z20.s, z8.b, z1.b[2]\n"
                        "sdot z24.s, z8.b, z2.b[2]\n"
                        "sdot z28.s, z8.b, z3.b[2]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "sdot z21.s, z9.b, z1.b[2]\n"
                        "sdot z25.s, z9.b, z2.b[2]\n"
                        "sdot z29.s, z9.b, z3.b[2]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "sdot z22.s, z10.b, z1.b[2]\n"
                        "sdot z26.s, z10.b, z2.b[2]\n"
                        "sdot z30.s, z10.b, z3.b[2]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "sdot z23.s, z11.b, z1.b[2]\n"
                        "sdot z27.s, z11.b, z2.b[2]\n"
                        "sdot z31.s, z11.b, z3.b[2]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "sdot z20.s, z12.b, z1.b[3]\n"
                        "sdot z24.s, z12.b, z2.b[3]\n"
                        "sdot z28.s, z12.b, z3.b[3]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "sdot z21.s, z13.b, z1.b[3]\n"
                        "sdot z25.s, z13.b, z2.b[3]\n"
                        "sdot z29.s, z13.b, z3.b[3]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "sdot z22.s, z14.b, z1.b[3]\n"
                        "sdot z26.s, z14.b, z2.b[3]\n"
                        "sdot z30.s, z14.b, z3.b[3]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "ld1rqb z0.b, p6/z, [%[a_ptr0], #0x10]\n"
                        "sdot z23.s, z15.b, z1.b[3]\n"
                        "ld1rqb z1.b, p6/z, [a_ptr1, #0x10]\n"
                        "sdot z27.s, z15.b, z2.b[3]\n"
                        "ld1rqb z2.b, p6/z, [a_ptr2, #0x10]\n"
                        "sdot z31.s, z15.b, z3.b[3]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[0]\n"
                        "ld1rqb z3.b, p6/z, [a_ptr3, #0x10]\n"
                        "sdot z20.s, z8.b, z5.b[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #2\n"
                        "sdot z24.s, z8.b, z6.b[0]\n"
                        "addvl a_ptr1, a_ptr1, #2\n"
                        "sdot z28.s, z8.b, z7.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z17.s, z9.b, z4.b[0]\n"
                        "addvl a_ptr2, a_ptr2, #2\n"
                        "sdot z21.s, z9.b, z5.b[0]\n"
                        "addvl a_ptr3, a_ptr3, #2\n"
                        "sdot z25.s, z9.b, z6.b[0]\n"
                        "sdot z29.s, z9.b, z7.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z18.s, z10.b, z4.b[0]\n"
                        "sdot z22.s, z10.b, z5.b[0]\n"
                        "sdot z26.s, z10.b, z6.b[0]\n"
                        "sdot z30.s, z10.b, z7.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z11.b, z4.b[0]\n"
                        "sdot z23.s, z11.b, z5.b[0]\n"
                        "sdot z27.s, z11.b, z6.b[0]\n"
                        "sdot z31.s, z11.b, z7.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[1]\n"
                        "sdot z20.s, z12.b, z5.b[1]\n"
                        "sdot z24.s, z12.b, z6.b[1]\n"
                        "sdot z28.s, z12.b, z7.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z13.b, z4.b[1]\n"
                        "sdot z21.s, z13.b, z5.b[1]\n"
                        "sdot z25.s, z13.b, z6.b[1]\n"
                        "sdot z29.s, z13.b, z7.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z14.b, z4.b[1]\n"
                        "sdot z22.s, z14.b, z5.b[1]\n"
                        "sdot z26.s, z14.b, z6.b[1]\n"
                        "sdot z30.s, z14.b, z7.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z15.b, z4.b[1]\n"
                        "sdot z23.s, z15.b, z5.b[1]\n"
                        "sdot z27.s, z15.b, z6.b[1]\n"
                        "sdot z31.s, z15.b, z7.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "sdot z20.s, z8.b, z5.b[2]\n"
                        "sdot z24.s, z8.b, z6.b[2]\n"
                        "sdot z28.s, z8.b, z7.b[2]\n"
                        "sdot z17.s, z9.b, z4.b[2]\n"
                        "sdot z21.s, z9.b, z5.b[2]\n"
                        "sdot z25.s, z9.b, z6.b[2]\n"
                        "sdot z29.s, z9.b, z7.b[2]\n"
                        "sdot z18.s, z10.b, z4.b[2]\n"
                        "sdot z22.s, z10.b, z5.b[2]\n"
                        "sdot z26.s, z10.b, z6.b[2]\n"
                        "sdot z30.s, z10.b, z7.b[2]\n"
                        "sdot z19.s, z11.b, z4.b[2]\n"
                        "sdot z23.s, z11.b, z5.b[2]\n"
                        "sdot z27.s, z11.b, z6.b[2]\n"
                        "sdot z31.s, z11.b, z7.b[2]\n"
                        "sdot z16.s, z12.b, z4.b[3]\n"
                        "sdot z20.s, z12.b, z5.b[3]\n"
                        "sdot z24.s, z12.b, z6.b[3]\n"
                        "sdot z28.s, z12.b, z7.b[3]\n"
                        "sdot z17.s, z13.b, z4.b[3]\n"
                        "sdot z21.s, z13.b, z5.b[3]\n"
                        "sdot z25.s, z13.b, z6.b[3]\n"
                        "sdot z29.s, z13.b, z7.b[3]\n"
                        "sdot z18.s, z14.b, z4.b[3]\n"
                        "sdot z22.s, z14.b, z5.b[3]\n"
                        "sdot z26.s, z14.b, z6.b[3]\n"
                        "sdot z30.s, z14.b, z7.b[3]\n"
                        "sdot z19.s, z15.b, z4.b[3]\n"
                        "sdot z23.s, z15.b, z5.b[3]\n"
                        "sdot z27.s, z15.b, z6.b[3]\n"
                        "sdot z31.s, z15.b, z7.b[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z20.s, z8.b, z1.b[0]\n"
                        "sdot z24.s, z8.b, z2.b[0]\n"
                        "sdot z28.s, z8.b, z3.b[0]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "sdot z21.s, z9.b, z1.b[0]\n"
                        "sdot z25.s, z9.b, z2.b[0]\n"
                        "sdot z29.s, z9.b, z3.b[0]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "sdot z22.s, z10.b, z1.b[0]\n"
                        "sdot z26.s, z10.b, z2.b[0]\n"
                        "sdot z30.s, z10.b, z3.b[0]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "sdot z23.s, z11.b, z1.b[0]\n"
                        "sdot z27.s, z11.b, z2.b[0]\n"
                        "sdot z31.s, z11.b, z3.b[0]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z20.s, z12.b, z1.b[1]\n"
                        "sdot z24.s, z12.b, z2.b[1]\n"
                        "sdot z28.s, z12.b, z3.b[1]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "sdot z21.s, z13.b, z1.b[1]\n"
                        "sdot z25.s, z13.b, z2.b[1]\n"
                        "sdot z29.s, z13.b, z3.b[1]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "sdot z22.s, z14.b, z1.b[1]\n"
                        "sdot z26.s, z14.b, z2.b[1]\n"
                        "sdot z30.s, z14.b, z3.b[1]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "sdot z23.s, z15.b, z1.b[1]\n"
                        "sdot z27.s, z15.b, z2.b[1]\n"
                        "sdot z31.s, z15.b, z3.b[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "sdot z20.s, z8.b, z1.b[2]\n"
                        "sdot z24.s, z8.b, z2.b[2]\n"
                        "sdot z28.s, z8.b, z3.b[2]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "sdot z21.s, z9.b, z1.b[2]\n"
                        "sdot z25.s, z9.b, z2.b[2]\n"
                        "sdot z29.s, z9.b, z3.b[2]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "sdot z22.s, z10.b, z1.b[2]\n"
                        "sdot z26.s, z10.b, z2.b[2]\n"
                        "sdot z30.s, z10.b, z3.b[2]\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "sdot z23.s, z11.b, z1.b[2]\n"
                        "sdot z27.s, z11.b, z2.b[2]\n"
                        "sdot z31.s, z11.b, z3.b[2]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "sdot z20.s, z12.b, z1.b[3]\n"
                        "sdot z24.s, z12.b, z2.b[3]\n"
                        "sdot z28.s, z12.b, z3.b[3]\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "sdot z21.s, z13.b, z1.b[3]\n"
                        "sdot z25.s, z13.b, z2.b[3]\n"
                        "sdot z29.s, z13.b, z3.b[3]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "sdot z22.s, z14.b, z1.b[3]\n"
                        "sdot z26.s, z14.b, z2.b[3]\n"
                        "sdot z30.s, z14.b, z3.b[3]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "sdot z23.s, z15.b, z1.b[3]\n"
                        "sdot z27.s, z15.b, z2.b[3]\n"
                        "sdot z31.s, z15.b, z3.b[3]\n"
                        "b 5f\n"
                        "4:\n"
                        "sdot z16.s, z8.b, z0.b[0]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z20.s, z8.b, z1.b[0]\n"
                        "ld1rqb z4.b, p6/z, [%[a_ptr0]]\n"
                        "sdot z24.s, z8.b, z2.b[0]\n"
                        "ld1rqb z5.b, p6/z, [a_ptr1]\n"
                        "sdot z28.s, z8.b, z3.b[0]\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "sdot z17.s, z9.b, z0.b[0]\n"
                        "ld1rqb z6.b, p6/z, [a_ptr2]\n"
                        "sdot z21.s, z9.b, z1.b[0]\n"
                        "ld1rqb z7.b, p6/z, [a_ptr3]\n"
                        "sdot z25.s, z9.b, z2.b[0]\n"
                        "addvl %[a_ptr0], %[a_ptr0], #1\n"
                        "sdot z29.s, z9.b, z3.b[0]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "sdot z18.s, z10.b, z0.b[0]\n"
                        "addvl a_ptr1, a_ptr1, #1\n"
                        "sdot z22.s, z10.b, z1.b[0]\n"
                        "addvl a_ptr2, a_ptr2, #1\n"
                        "sdot z26.s, z10.b, z2.b[0]\n"
                        "addvl a_ptr3, a_ptr3, #1\n"
                        "sdot z30.s, z10.b, z3.b[0]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z19.s, z11.b, z0.b[0]\n"
                        "sdot z23.s, z11.b, z1.b[0]\n"
                        "sdot z27.s, z11.b, z2.b[0]\n"
                        "sdot z31.s, z11.b, z3.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z16.s, z12.b, z0.b[1]\n"
                        "sdot z20.s, z12.b, z1.b[1]\n"
                        "sdot z24.s, z12.b, z2.b[1]\n"
                        "sdot z28.s, z12.b, z3.b[1]\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "sdot z17.s, z13.b, z0.b[1]\n"
                        "sdot z21.s, z13.b, z1.b[1]\n"
                        "sdot z25.s, z13.b, z2.b[1]\n"
                        "sdot z29.s, z13.b, z3.b[1]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "sdot z18.s, z14.b, z0.b[1]\n"
                        "sdot z22.s, z14.b, z1.b[1]\n"
                        "sdot z26.s, z14.b, z2.b[1]\n"
                        "sdot z30.s, z14.b, z3.b[1]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z19.s, z15.b, z0.b[1]\n"
                        "sdot z23.s, z15.b, z1.b[1]\n"
                        "sdot z27.s, z15.b, z2.b[1]\n"
                        "sdot z31.s, z15.b, z3.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z16.s, z8.b, z0.b[2]\n"
                        "addvl %[b_ptr0], %[b_ptr0], #8\n"
                        "sdot z20.s, z8.b, z1.b[2]\n"
                        "sdot z24.s, z8.b, z2.b[2]\n"
                        "sdot z28.s, z8.b, z3.b[2]\n"
                        "sdot z17.s, z9.b, z0.b[2]\n"
                        "sdot z21.s, z9.b, z1.b[2]\n"
                        "sdot z25.s, z9.b, z2.b[2]\n"
                        "sdot z29.s, z9.b, z3.b[2]\n"
                        "sdot z18.s, z10.b, z0.b[2]\n"
                        "sdot z22.s, z10.b, z1.b[2]\n"
                        "sdot z26.s, z10.b, z2.b[2]\n"
                        "sdot z30.s, z10.b, z3.b[2]\n"
                        "sdot z19.s, z11.b, z0.b[2]\n"
                        "sdot z23.s, z11.b, z1.b[2]\n"
                        "sdot z27.s, z11.b, z2.b[2]\n"
                        "sdot z31.s, z11.b, z3.b[2]\n"
                        "sdot z16.s, z12.b, z0.b[3]\n"
                        "sdot z20.s, z12.b, z1.b[3]\n"
                        "sdot z24.s, z12.b, z2.b[3]\n"
                        "sdot z28.s, z12.b, z3.b[3]\n"
                        "sdot z17.s, z13.b, z0.b[3]\n"
                        "sdot z21.s, z13.b, z1.b[3]\n"
                        "sdot z25.s, z13.b, z2.b[3]\n"
                        "sdot z29.s, z13.b, z3.b[3]\n"
                        "sdot z18.s, z14.b, z0.b[3]\n"
                        "sdot z22.s, z14.b, z1.b[3]\n"
                        "sdot z26.s, z14.b, z2.b[3]\n"
                        "sdot z30.s, z14.b, z3.b[3]\n"
                        "sdot z19.s, z15.b, z0.b[3]\n"
                        "sdot z23.s, z15.b, z1.b[3]\n"
                        "sdot z27.s, z15.b, z2.b[3]\n"
                        "sdot z31.s, z15.b, z3.b[3]\n"
                        "cbz %[blocks], 5f\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0]]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #1, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #2, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[0]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #3, MUL VL]\n"
                        "sdot z20.s, z8.b, z5.b[0]\n"
                        "sdot z24.s, z8.b, z6.b[0]\n"
                        "sdot z28.s, z8.b, z7.b[0]\n"
                        "sdot z17.s, z9.b, z4.b[0]\n"
                        "sdot z21.s, z9.b, z5.b[0]\n"
                        "sdot z25.s, z9.b, z6.b[0]\n"
                        "sdot z29.s, z9.b, z7.b[0]\n"
                        "sdot z18.s, z10.b, z4.b[0]\n"
                        "sdot z22.s, z10.b, z5.b[0]\n"
                        "sdot z26.s, z10.b, z6.b[0]\n"
                        "sdot z30.s, z10.b, z7.b[0]\n"
                        "sdot z19.s, z11.b, z4.b[0]\n"
                        "sdot z23.s, z11.b, z5.b[0]\n"
                        "sdot z27.s, z11.b, z6.b[0]\n"
                        "sdot z31.s, z11.b, z7.b[0]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #4, MUL VL]\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #5, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #6, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[1]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #7, MUL VL]\n"
                        "sdot z20.s, z12.b, z5.b[1]\n"
                        "sdot z24.s, z12.b, z6.b[1]\n"
                        "sdot z28.s, z12.b, z7.b[1]\n"
                        "sdot z17.s, z13.b, z4.b[1]\n"
                        "sdot z21.s, z13.b, z5.b[1]\n"
                        "sdot z25.s, z13.b, z6.b[1]\n"
                        "sdot z29.s, z13.b, z7.b[1]\n"
                        "sdot z18.s, z14.b, z4.b[1]\n"
                        "sdot z22.s, z14.b, z5.b[1]\n"
                        "sdot z26.s, z14.b, z6.b[1]\n"
                        "sdot z30.s, z14.b, z7.b[1]\n"
                        "sdot z19.s, z15.b, z4.b[1]\n"
                        "sdot z23.s, z15.b, z5.b[1]\n"
                        "sdot z27.s, z15.b, z6.b[1]\n"
                        "sdot z31.s, z15.b, z7.b[1]\n"
                        "b.eq 5f\n"
                        "addvl %[b_ptr0], %[b_ptr0], #16\n"
                        "subs %[blocks], %[blocks], #0x1\n"
                        "ld1b z8.b, p7/z, [%[b_ptr0], #-8, MUL VL]\n"
                        "ld1b z9.b, p7/z, [%[b_ptr0], #-7, MUL VL]\n"
                        "ld1b z10.b, p7/z, [%[b_ptr0], #-6, MUL VL]\n"
                        "ld1b z11.b, p7/z, [%[b_ptr0], #-5, MUL VL]\n"
                        "sdot z16.s, z8.b, z4.b[2]\n"
                        "sdot z20.s, z8.b, z5.b[2]\n"
                        "sdot z24.s, z8.b, z6.b[2]\n"
                        "sdot z28.s, z8.b, z7.b[2]\n"
                        "sdot z17.s, z9.b, z4.b[2]\n"
                        "sdot z21.s, z9.b, z5.b[2]\n"
                        "sdot z25.s, z9.b, z6.b[2]\n"
                        "sdot z29.s, z9.b, z7.b[2]\n"
                        "sdot z18.s, z10.b, z4.b[2]\n"
                        "sdot z22.s, z10.b, z5.b[2]\n"
                        "sdot z26.s, z10.b, z6.b[2]\n"
                        "sdot z30.s, z10.b, z7.b[2]\n"
                        "sdot z19.s, z11.b, z4.b[2]\n"
                        "sdot z23.s, z11.b, z5.b[2]\n"
                        "sdot z27.s, z11.b, z6.b[2]\n"
                        "sdot z31.s, z11.b, z7.b[2]\n"
                        "b.eq 5f\n"
                        "ld1b z12.b, p7/z, [%[b_ptr0], #-4, MUL VL]\n"
                        "ld1b z13.b, p7/z, [%[b_ptr0], #-3, MUL VL]\n"
                        "ld1b z14.b, p7/z, [%[b_ptr0], #-2, MUL VL]\n"
                        "ld1b z15.b, p7/z, [%[b_ptr0], #-1, MUL VL]\n"
                        "sdot z16.s, z12.b, z4.b[3]\n"
                        "sdot z20.s, z12.b, z5.b[3]\n"
                        "sdot z24.s, z12.b, z6.b[3]\n"
                        "sdot z28.s, z12.b, z7.b[3]\n"
                        "sdot z17.s, z13.b, z4.b[3]\n"
                        "sdot z21.s, z13.b, z5.b[3]\n"
                        "sdot z25.s, z13.b, z6.b[3]\n"
                        "sdot z29.s, z13.b, z7.b[3]\n"
                        "sdot z18.s, z14.b, z4.b[3]\n"
                        "sdot z22.s, z14.b, z5.b[3]\n"
                        "sdot z26.s, z14.b, z6.b[3]\n"
                        "sdot z30.s, z14.b, z7.b[3]\n"
                        "sdot z19.s, z15.b, z4.b[3]\n"
                        "sdot z23.s, z15.b, z5.b[3]\n"
                        "sdot z27.s, z15.b, z6.b[3]\n"
                        "sdot z31.s, z15.b, z7.b[3]\n"
                        "5:\n"
                        "st1w z16.s, p0, [%[c_ptr0]]\n"
                        "st1w z17.s, p1, [%[c_ptr0], #1, MUL VL]\n"
                        "st1w z18.s, p2, [%[c_ptr0], #2, MUL VL]\n"
                        "st1w z19.s, p3, [%[c_ptr0], #3, MUL VL]\n"
                        "addvl %[c_ptr0], %[c_ptr0], #4\n"
                        "st1w z20.s, p0, [c_ptr1]\n"
                        "st1w z21.s, p1, [c_ptr1, #1, MUL VL]\n"
                        "st1w z22.s, p2, [c_ptr1, #2, MUL VL]\n"
                        "st1w z23.s, p3, [c_ptr1, #3, MUL VL]\n"
                        "st1w z24.s, p0, [c_ptr2]\n"
                        "st1w z25.s, p1, [c_ptr2, #1, MUL VL]\n"
                        "st1w z26.s, p2, [c_ptr2, #2, MUL VL]\n"
                        "st1w z27.s, p3, [c_ptr2, #3, MUL VL]\n"
                        "st1w z28.s, p0, [c_ptr3]\n"
                        "st1w z29.s, p1, [c_ptr3, #1, MUL VL]\n"
                        "st1w z30.s, p2, [c_ptr3, #2, MUL VL]\n"
                        "st1w z31.s, p3, [c_ptr3, #3, MUL VL]\n"
                        ".unreq a_ptr1\n"
                        ".unreq a_ptr2\n"
                        ".unreq a_ptr3\n"
                        ".unreq c_ptr1\n"
                        ".unreq c_ptr2\n"
                        ".unreq c_ptr3\n"
                        : [a_ptr0] "+r" (a_ptr0), [b_ptr0] "+r" (b_ptr0), [c_ptr0] "+r" (c_ptr0), [loops] "+r" (loops), [regs] "+r" (regs), [temp] "+r" (temp), [blocks] "+r" (blocks)
                        : [width] "r" (width), [append] "r" (static_cast<uint64_t>(append)), [lda] "r" (ldab), [ldc] "r" (ldcb), [leftovers] "r" (leftovers)
                        : "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31", "x0", "x1", "x2", "x3", "x4", "x5", "cc", "memory"
                    );
                    break;
            }

        }
    }
}

} // namespace arm_gemm

#endif // __ARM_FEATURE_SVE
