/*
 * Copyright (c) 2019-2020, NVIDIA CORPORATION. All rights reserved.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

#include <assert.h>
#include <common/bl_common.h>
#include <mce.h>
#include <memctrl_v2.h>
#include <tegra_mc_def.h>
#include <tegra_platform.h>

/*******************************************************************************
 * Array to hold the security configs for stream IDs
 ******************************************************************************/
const static mc_streamid_security_cfg_t tegra194_streamid_sec_cfgs[] = {
	mc_make_sec_cfg(PTCR, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(HDAR, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(HOST1XDMAR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(NVENCSRD, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(SATAR, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MPCORER, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(NVENCSWR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(HDAW, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MPCOREW, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(SATAW, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(ISPRA, NON_SECURE, NO_OVERRIDE, ENABLE),
	mc_make_sec_cfg(ISPFALR, NON_SECURE, NO_OVERRIDE, ENABLE),
	mc_make_sec_cfg(ISPWA, NON_SECURE, NO_OVERRIDE, ENABLE),
	mc_make_sec_cfg(ISPWB, NON_SECURE, NO_OVERRIDE, ENABLE),
	mc_make_sec_cfg(XUSB_HOSTR, NON_SECURE, NO_OVERRIDE, ENABLE),
	mc_make_sec_cfg(XUSB_HOSTW, NON_SECURE, NO_OVERRIDE, ENABLE),
	mc_make_sec_cfg(XUSB_DEVR, NON_SECURE, NO_OVERRIDE, ENABLE),
	mc_make_sec_cfg(XUSB_DEVW, NON_SECURE, NO_OVERRIDE, ENABLE),
	mc_make_sec_cfg(TSECSRD, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(TSECSWR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(SDMMCRA, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(SDMMCR, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(SDMMCRAB, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(SDMMCWA, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(SDMMCW, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(SDMMCWAB, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(VICSRD, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(VICSWR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(VIW, NON_SECURE, NO_OVERRIDE, ENABLE),
	mc_make_sec_cfg(NVDECSRD, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(NVDECSWR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(APER, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(APEW, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(NVJPGSRD, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(NVJPGSWR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(SESRD, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(SESWR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(AXIAPR, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(AXIAPW, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(ETRR, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(ETRW, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(TSECSRDB, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(TSECSWRB, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(AXISR, SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(AXISW, SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(EQOSR, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(EQOSW, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(UFSHCR, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(UFSHCW, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(NVDISPLAYR, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(BPMPR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(BPMPW, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(BPMPDMAR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(BPMPDMAW, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(AONR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(AONW, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(AONDMAR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(AONDMAW, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(SCER, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(SCEW, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(SCEDMAR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(SCEDMAW, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(APEDMAR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(APEDMAW, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(NVDISPLAYR1, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(VICSRD1, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(NVDECSRD1, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(VIFALR, NON_SECURE, NO_OVERRIDE, ENABLE),
	mc_make_sec_cfg(VIFALW, NON_SECURE, NO_OVERRIDE, ENABLE),
	mc_make_sec_cfg(DLA0RDA, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(DLA0FALRDB, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(DLA0WRA, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(DLA0FALWRB, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(DLA1RDA, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(DLA1FALRDB, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(DLA1WRA, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(DLA1FALWRB, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA0RDA, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA0RDB, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA0RDC, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA0WRA, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA0WRB, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA0WRC, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA1RDA, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA1RDB, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA1RDC, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA1WRA, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA1WRB, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA1WRC, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(RCER, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(RCEW, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(RCEDMAR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(RCEDMAW, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(NVENC1SRD, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(NVENC1SWR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PCIE0R, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(PCIE0W, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(PCIE1R, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(PCIE1W, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(PCIE2AR, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(PCIE2AW, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(PCIE3R, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(PCIE3W, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(PCIE4R, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(PCIE4W, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(PCIE5R, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(PCIE5W, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(ISPFALW, NON_SECURE, NO_OVERRIDE, ENABLE),
	mc_make_sec_cfg(DLA0RDA1, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(DLA1RDA1, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA0RDA1, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA0RDB1, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA1RDA1, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PVA1RDB1, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(PCIE5R1, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(NVENCSRD1, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(NVENC1SRD1, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(ISPRA1, NON_SECURE, NO_OVERRIDE, ENABLE),
	mc_make_sec_cfg(PCIE0R1, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(NVDEC1SRD, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(NVDEC1SRD1, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(NVDEC1SWR, NON_SECURE, NO_OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU0R, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU0W, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU1R, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU1W, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU2R, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU2W, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU3R, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU3W, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU4R, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU4W, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU5R, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU5W, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU6R, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU6W, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU7R, NON_SECURE, OVERRIDE, DISABLE),
	mc_make_sec_cfg(MIU7W, NON_SECURE, OVERRIDE, DISABLE)
};

/*******************************************************************************
 * Array to hold MC context for Tegra194
 ******************************************************************************/
static __attribute__((aligned(16))) mc_regs_t tegra194_mc_context[] = {
	_START_OF_TABLE_,
	mc_make_sid_security_cfg(HDAR),
	mc_make_sid_security_cfg(HOST1XDMAR),
	mc_make_sid_security_cfg(NVENCSRD),
	mc_make_sid_security_cfg(SATAR),
	mc_make_sid_security_cfg(NVENCSWR),
	mc_make_sid_security_cfg(HDAW),
	mc_make_sid_security_cfg(SATAW),
	mc_make_sid_security_cfg(ISPRA),
	mc_make_sid_security_cfg(ISPFALR),
	mc_make_sid_security_cfg(ISPWA),
	mc_make_sid_security_cfg(ISPWB),
	mc_make_sid_security_cfg(XUSB_HOSTR),
	mc_make_sid_security_cfg(XUSB_HOSTW),
	mc_make_sid_security_cfg(XUSB_DEVR),
	mc_make_sid_security_cfg(XUSB_DEVW),
	mc_make_sid_security_cfg(TSECSRD),
	mc_make_sid_security_cfg(TSECSWR),
	mc_make_sid_security_cfg(SDMMCRA),
	mc_make_sid_security_cfg(SDMMCR),
	mc_make_sid_security_cfg(SDMMCRAB),
	mc_make_sid_security_cfg(SDMMCWA),
	mc_make_sid_security_cfg(SDMMCW),
	mc_make_sid_security_cfg(SDMMCWAB),
	mc_make_sid_security_cfg(VICSRD),
	mc_make_sid_security_cfg(VICSWR),
	mc_make_sid_security_cfg(VIW),
	mc_make_sid_security_cfg(NVDECSRD),
	mc_make_sid_security_cfg(NVDECSWR),
	mc_make_sid_security_cfg(APER),
	mc_make_sid_security_cfg(APEW),
	mc_make_sid_security_cfg(NVJPGSRD),
	mc_make_sid_security_cfg(NVJPGSWR),
	mc_make_sid_security_cfg(SESRD),
	mc_make_sid_security_cfg(SESWR),
	mc_make_sid_security_cfg(AXIAPR),
	mc_make_sid_security_cfg(AXIAPW),
	mc_make_sid_security_cfg(ETRR),
	mc_make_sid_security_cfg(ETRW),
	mc_make_sid_security_cfg(TSECSRDB),
	mc_make_sid_security_cfg(TSECSWRB),
	mc_make_sid_security_cfg(AXISR),
	mc_make_sid_security_cfg(AXISW),
	mc_make_sid_security_cfg(EQOSR),
	mc_make_sid_security_cfg(EQOSW),
	mc_make_sid_security_cfg(UFSHCR),
	mc_make_sid_security_cfg(UFSHCW),
	mc_make_sid_security_cfg(NVDISPLAYR),
	mc_make_sid_security_cfg(BPMPR),
	mc_make_sid_security_cfg(BPMPW),
	mc_make_sid_security_cfg(BPMPDMAR),
	mc_make_sid_security_cfg(BPMPDMAW),
	mc_make_sid_security_cfg(AONR),
	mc_make_sid_security_cfg(AONW),
	mc_make_sid_security_cfg(AONDMAR),
	mc_make_sid_security_cfg(AONDMAW),
	mc_make_sid_security_cfg(SCER),
	mc_make_sid_security_cfg(SCEW),
	mc_make_sid_security_cfg(SCEDMAR),
	mc_make_sid_security_cfg(SCEDMAW),
	mc_make_sid_security_cfg(APEDMAR),
	mc_make_sid_security_cfg(APEDMAW),
	mc_make_sid_security_cfg(NVDISPLAYR1),
	mc_make_sid_security_cfg(VICSRD1),
	mc_make_sid_security_cfg(NVDECSRD1),
	mc_make_sid_security_cfg(VIFALR),
	mc_make_sid_security_cfg(VIFALW),
	mc_make_sid_security_cfg(DLA0RDA),
	mc_make_sid_security_cfg(DLA0FALRDB),
	mc_make_sid_security_cfg(DLA0WRA),
	mc_make_sid_security_cfg(DLA0FALWRB),
	mc_make_sid_security_cfg(DLA1RDA),
	mc_make_sid_security_cfg(DLA1FALRDB),
	mc_make_sid_security_cfg(DLA1WRA),
	mc_make_sid_security_cfg(DLA1FALWRB),
	mc_make_sid_security_cfg(PVA0RDA),
	mc_make_sid_security_cfg(PVA0RDB),
	mc_make_sid_security_cfg(PVA0RDC),
	mc_make_sid_security_cfg(PVA0WRA),
	mc_make_sid_security_cfg(PVA0WRB),
	mc_make_sid_security_cfg(PVA0WRC),
	mc_make_sid_security_cfg(PVA1RDA),
	mc_make_sid_security_cfg(PVA1RDB),
	mc_make_sid_security_cfg(PVA1RDC),
	mc_make_sid_security_cfg(PVA1WRA),
	mc_make_sid_security_cfg(PVA1WRB),
	mc_make_sid_security_cfg(PVA1WRC),
	mc_make_sid_security_cfg(RCER),
	mc_make_sid_security_cfg(RCEW),
	mc_make_sid_security_cfg(RCEDMAR),
	mc_make_sid_security_cfg(RCEDMAW),
	mc_make_sid_security_cfg(NVENC1SRD),
	mc_make_sid_security_cfg(NVENC1SWR),
	mc_make_sid_security_cfg(PCIE0R),
	mc_make_sid_security_cfg(PCIE0W),
	mc_make_sid_security_cfg(PCIE1R),
	mc_make_sid_security_cfg(PCIE1W),
	mc_make_sid_security_cfg(PCIE2AR),
	mc_make_sid_security_cfg(PCIE2AW),
	mc_make_sid_security_cfg(PCIE3R),
	mc_make_sid_security_cfg(PCIE3W),
	mc_make_sid_security_cfg(PCIE4R),
	mc_make_sid_security_cfg(PCIE4W),
	mc_make_sid_security_cfg(PCIE5R),
	mc_make_sid_security_cfg(PCIE5W),
	mc_make_sid_security_cfg(ISPFALW),
	mc_make_sid_security_cfg(DLA0RDA1),
	mc_make_sid_security_cfg(DLA1RDA1),
	mc_make_sid_security_cfg(PVA0RDA1),
	mc_make_sid_security_cfg(PVA0RDB1),
	mc_make_sid_security_cfg(PVA1RDA1),
	mc_make_sid_security_cfg(PVA1RDB1),
	mc_make_sid_security_cfg(PCIE5R1),
	mc_make_sid_security_cfg(NVENCSRD1),
	mc_make_sid_security_cfg(NVENC1SRD1),
	mc_make_sid_security_cfg(ISPRA1),
	mc_make_sid_security_cfg(PCIE0R1),
	mc_make_sid_security_cfg(MIU0R),
	mc_make_sid_security_cfg(MIU0W),
	mc_make_sid_security_cfg(MIU1R),
	mc_make_sid_security_cfg(MIU1W),
	mc_make_sid_security_cfg(MIU2R),
	mc_make_sid_security_cfg(MIU2W),
	mc_make_sid_security_cfg(MIU3R),
	mc_make_sid_security_cfg(MIU3W),
	mc_make_sid_override_cfg(HDAR),
	mc_make_sid_override_cfg(HOST1XDMAR),
	mc_make_sid_override_cfg(NVENCSRD),
	mc_make_sid_override_cfg(SATAR),
	mc_make_sid_override_cfg(NVENCSWR),
	mc_make_sid_override_cfg(HDAW),
	mc_make_sid_override_cfg(SATAW),
	mc_make_sid_override_cfg(ISPRA),
	mc_make_sid_override_cfg(ISPFALR),
	mc_make_sid_override_cfg(ISPWA),
	mc_make_sid_override_cfg(ISPWB),
	mc_make_sid_override_cfg(XUSB_HOSTR),
	mc_make_sid_override_cfg(XUSB_HOSTW),
	mc_make_sid_override_cfg(XUSB_DEVR),
	mc_make_sid_override_cfg(XUSB_DEVW),
	mc_make_sid_override_cfg(TSECSRD),
	mc_make_sid_override_cfg(TSECSWR),
	mc_make_sid_override_cfg(SDMMCRA),
	mc_make_sid_override_cfg(SDMMCR),
	mc_make_sid_override_cfg(SDMMCRAB),
	mc_make_sid_override_cfg(SDMMCWA),
	mc_make_sid_override_cfg(SDMMCW),
	mc_make_sid_override_cfg(SDMMCWAB),
	mc_make_sid_override_cfg(VICSRD),
	mc_make_sid_override_cfg(VICSWR),
	mc_make_sid_override_cfg(VIW),
	mc_make_sid_override_cfg(NVDECSRD),
	mc_make_sid_override_cfg(NVDECSWR),
	mc_make_sid_override_cfg(APER),
	mc_make_sid_override_cfg(APEW),
	mc_make_sid_override_cfg(NVJPGSRD),
	mc_make_sid_override_cfg(NVJPGSWR),
	mc_make_sid_override_cfg(SESRD),
	mc_make_sid_override_cfg(SESWR),
	mc_make_sid_override_cfg(AXIAPR),
	mc_make_sid_override_cfg(AXIAPW),
	mc_make_sid_override_cfg(ETRR),
	mc_make_sid_override_cfg(ETRW),
	mc_make_sid_override_cfg(TSECSRDB),
	mc_make_sid_override_cfg(TSECSWRB),
	mc_make_sid_override_cfg(AXISR),
	mc_make_sid_override_cfg(AXISW),
	mc_make_sid_override_cfg(EQOSR),
	mc_make_sid_override_cfg(EQOSW),
	mc_make_sid_override_cfg(UFSHCR),
	mc_make_sid_override_cfg(UFSHCW),
	mc_make_sid_override_cfg(NVDISPLAYR),
	mc_make_sid_override_cfg(BPMPR),
	mc_make_sid_override_cfg(BPMPW),
	mc_make_sid_override_cfg(BPMPDMAR),
	mc_make_sid_override_cfg(BPMPDMAW),
	mc_make_sid_override_cfg(AONR),
	mc_make_sid_override_cfg(AONW),
	mc_make_sid_override_cfg(AONDMAR),
	mc_make_sid_override_cfg(AONDMAW),
	mc_make_sid_override_cfg(SCER),
	mc_make_sid_override_cfg(SCEW),
	mc_make_sid_override_cfg(SCEDMAR),
	mc_make_sid_override_cfg(SCEDMAW),
	mc_make_sid_override_cfg(APEDMAR),
	mc_make_sid_override_cfg(APEDMAW),
	mc_make_sid_override_cfg(NVDISPLAYR1),
	mc_make_sid_override_cfg(VICSRD1),
	mc_make_sid_override_cfg(NVDECSRD1),
	mc_make_sid_override_cfg(VIFALR),
	mc_make_sid_override_cfg(VIFALW),
	mc_make_sid_override_cfg(DLA0RDA),
	mc_make_sid_override_cfg(DLA0FALRDB),
	mc_make_sid_override_cfg(DLA0WRA),
	mc_make_sid_override_cfg(DLA0FALWRB),
	mc_make_sid_override_cfg(DLA1RDA),
	mc_make_sid_override_cfg(DLA1FALRDB),
	mc_make_sid_override_cfg(DLA1WRA),
	mc_make_sid_override_cfg(DLA1FALWRB),
	mc_make_sid_override_cfg(PVA0RDA),
	mc_make_sid_override_cfg(PVA0RDB),
	mc_make_sid_override_cfg(PVA0RDC),
	mc_make_sid_override_cfg(PVA0WRA),
	mc_make_sid_override_cfg(PVA0WRB),
	mc_make_sid_override_cfg(PVA0WRC),
	mc_make_sid_override_cfg(PVA1RDA),
	mc_make_sid_override_cfg(PVA1RDB),
	mc_make_sid_override_cfg(PVA1RDC),
	mc_make_sid_override_cfg(PVA1WRA),
	mc_make_sid_override_cfg(PVA1WRB),
	mc_make_sid_override_cfg(PVA1WRC),
	mc_make_sid_override_cfg(RCER),
	mc_make_sid_override_cfg(RCEW),
	mc_make_sid_override_cfg(RCEDMAR),
	mc_make_sid_override_cfg(RCEDMAW),
	mc_make_sid_override_cfg(NVENC1SRD),
	mc_make_sid_override_cfg(NVENC1SWR),
	mc_make_sid_override_cfg(PCIE0R),
	mc_make_sid_override_cfg(PCIE0W),
	mc_make_sid_override_cfg(PCIE1R),
	mc_make_sid_override_cfg(PCIE1W),
	mc_make_sid_override_cfg(PCIE2AR),
	mc_make_sid_override_cfg(PCIE2AW),
	mc_make_sid_override_cfg(PCIE3R),
	mc_make_sid_override_cfg(PCIE3W),
	mc_make_sid_override_cfg(PCIE4R),
	mc_make_sid_override_cfg(PCIE4W),
	mc_make_sid_override_cfg(PCIE5R),
	mc_make_sid_override_cfg(PCIE5W),
	mc_make_sid_override_cfg(ISPFALW),
	mc_make_sid_override_cfg(DLA0RDA1),
	mc_make_sid_override_cfg(DLA1RDA1),
	mc_make_sid_override_cfg(PVA0RDA1),
	mc_make_sid_override_cfg(PVA0RDB1),
	mc_make_sid_override_cfg(PVA1RDA1),
	mc_make_sid_override_cfg(PVA1RDB1),
	mc_make_sid_override_cfg(PCIE5R1),
	mc_make_sid_override_cfg(NVENCSRD1),
	mc_make_sid_override_cfg(NVENC1SRD1),
	mc_make_sid_override_cfg(ISPRA1),
	mc_make_sid_override_cfg(PCIE0R1),
	mc_make_sid_override_cfg(MIU0R),
	mc_make_sid_override_cfg(MIU0W),
	mc_make_sid_override_cfg(MIU1R),
	mc_make_sid_override_cfg(MIU1W),
	mc_make_sid_override_cfg(MIU2R),
	mc_make_sid_override_cfg(MIU2W),
	mc_make_sid_override_cfg(MIU3R),
	mc_make_sid_override_cfg(MIU3W),
	mc_smmu_bypass_cfg,	/* TBU settings */
	_END_OF_TABLE_,
};

/*******************************************************************************
 * Handler to return the pointer to the MC's context struct
 ******************************************************************************/
static mc_regs_t *tegra194_get_mc_system_suspend_ctx(void)
{
	/* index of _END_OF_TABLE_ */
	tegra194_mc_context[0].val = (uint32_t)ARRAY_SIZE(tegra194_mc_context) - 1U;

	return tegra194_mc_context;
}

/*******************************************************************************
 * Struct to hold the memory controller settings
 ******************************************************************************/
static tegra_mc_settings_t tegra194_mc_settings = {
	.streamid_security_cfg = tegra194_streamid_sec_cfgs,
	.num_streamid_security_cfgs = (uint32_t)ARRAY_SIZE(tegra194_streamid_sec_cfgs),
	.get_mc_system_suspend_ctx = tegra194_get_mc_system_suspend_ctx
};

/*******************************************************************************
 * Handler to return the pointer to the memory controller's settings struct
 ******************************************************************************/
tegra_mc_settings_t *tegra_get_mc_settings(void)
{
	return &tegra194_mc_settings;
}

/*******************************************************************************
 * Handler to program the scratch registers with TZDRAM settings for the
 * resume firmware
 ******************************************************************************/
void plat_memctrl_tzdram_setup(uint64_t phys_base, uint64_t size_in_bytes)
{
	uint32_t sec_reg_ctrl = tegra_mc_read_32(MC_SECURITY_CFG_REG_CTRL_0);

	/*
	 * Check TZDRAM carveout register access status. Setup TZDRAM fence
	 * only if access is enabled.
	 */
	if ((sec_reg_ctrl & SECURITY_CFG_WRITE_ACCESS_BIT) ==
	     SECURITY_CFG_WRITE_ACCESS_ENABLE) {

		/*
		 * Setup the Memory controller to allow only secure accesses to
		 * the TZDRAM carveout
		 */
		INFO("Configuring TrustZone DRAM Memory Carveout\n");

		tegra_mc_write_32(MC_SECURITY_CFG0_0, (uint32_t)phys_base);
		tegra_mc_write_32(MC_SECURITY_CFG3_0, (uint32_t)(phys_base >> 32));
		tegra_mc_write_32(MC_SECURITY_CFG1_0, (uint32_t)(size_in_bytes >> 20));

		/*
		 * MCE propagates the security configuration values across the
		 * CCPLEX.
		 */
		(void)mce_update_gsc_tzdram();
	}
}
