/*
 *  Copyright 1994-2018 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include "SDL.h"
#include "ttf.h"
#include "src/defaults.h"

#define NO_MOUSE_CURSOR

u_long id = 1095629101;
u_long options = BE_NONE;

#ifndef FIXED
u_short out_width;
u_short out_height;
#define out_buffsize (out_width * out_height)
static struct SwsContext *sws_context = NULL;
static uint8_t *bufp = NULL, *dstbufp = NULL;
static const uint8_t *bufp_src[3] = {NULL, NULL, NULL}; /* FIXME not 4 ? */
static uint8_t *dstbufp_src[3] = {NULL, NULL, NULL}; /* FIXME only first is used ? */
#endif
static Buffer8_t *out;


static void
biniou_sdl_set_videomode(const u_short width, const u_short height, const Uint32 flags)
{
  drv.sc = SDL_SetVideoMode(width, height, 0, flags);
  if (NULL == drv.sc)
    xerror("Couldn't set %dx%d video mode: %s\n", width, height, SDL_GetError());
#ifdef DEBUG
  else 
    printf("[+] Set screen %dx%d at %dbpp\n",
	   width, height, drv.sc->format->BitsPerPixel);
#endif
  drv.must_lock_screen = SDL_MUSTLOCK(drv.sc);
}


static inline void
SDL_refresh_32bpp(Context_t *ctx, SDL_Surface *sc)
{
  const RGBA_t *src;

  Buffer8_flip_v(active_buffer(ctx));
  src = export_RGBA_active_buffer(ctx);
  Buffer8_flip_v(active_buffer(ctx));
#ifndef FIXED
  int srcstride[3] = {WIDTH*4, 0, 0};
  int dststride[3] = {out_width*4, 0, 0};
  int ret;

  if (bufp == NULL) {
    bufp = xcalloc(4*BUFFSIZE, sizeof(uint8_t)); /* FIXME 1*RGBA_t ? */
  }
  if (dstbufp == NULL) {
    dstbufp = xcalloc(4*out_buffsize, sizeof(uint8_t)); /* FIXME 1*RGBA_t ? */
    dstbufp_src[0] = sc->pixels;
  }
  bufp_src[0] = (uint8_t *)src;
#else
  memcpy(sc->pixels, src, BUFFSIZE*sizeof(RGBA_t));
#endif

#ifndef FIXED
  if (NULL == sws_context)
    sws_context = sws_getContext(WIDTH, HEIGHT, AV_PIX_FMT_RGB32,
				 out_width, out_height, AV_PIX_FMT_BGR32,
				 SWS_FAST_BILINEAR, NULL, NULL, NULL);
  if (NULL == sws_context)
    xerror("sws_getContext\n");

  ret = sws_scale(sws_context, bufp_src, srcstride, 0, HEIGHT, dstbufp_src, dststride);
  if (ret < 0)
    xerror("sws_scale\n");
#endif
}


static void
SDL_get_event(Context_t *ctx)
{
  // TODO middle = change color, right = erase (3x3)

  SDL_Event evt;
  memset(&evt, 0, sizeof(SDL_Event));

  while (SDL_PollEvent(&evt) != 0) {
    BKey_t key;

    switch (evt.type) {
    case SDL_KEYDOWN:
      if (NULL != ctx->events_cb) {
	key.val = evt.key.keysym.sym;
	key.mod = evt.key.keysym.mod;
	
	ctx->events_cb(ctx, &key);
      }
      break;
	
    case SDL_QUIT:
      Context_send_event(ctx, BT_CONTEXT, BC_QUIT, BA_NONE);
      break;
			
    case SDL_MOUSEMOTION:
      switch (evt.motion.state) {
      case SDL_BUTTON_LEFT:
	ctx->params3d.xe = evt.motion.x;
	ctx->params3d.ye = evt.motion.y;
	Params3d_rotate(&ctx->params3d);
	break;

      case SDL_BUTTON_RIGHT+SDL_BUTTON_LEFT: /* <- WTF ? */
	// printf("right button motion @ %d %d\n",  evt.motion.x, evt.motion.y);
	set_pixel_nc(active_buffer(ctx), evt.motion.x, MAXY-evt.motion.y, 255);
	break;

      default:
	break;
      }
      break;

    case SDL_MOUSEBUTTONDOWN:
      /* printf("type= %d, button= %d\n", evt.button.type, evt.button.button); */
      switch (evt.button.button) {
      case SDL_BUTTON_LEFT:
	ctx->params3d.xs = evt.motion.x;
	ctx->params3d.ys = evt.motion.y;
	break;

      case SDL_BUTTON_RIGHT:
	// printf("button down @ %d %d\n",  evt.motion.x, evt.motion.y);
	set_pixel_nc(active_buffer(ctx), evt.motion.x, MAXY-evt.motion.y, 255);
	break;
	
      case SDL_BUTTON_WHEELUP:
	ctx->params3d.scale_factor /= 0.9;
	/* printf("scale: %d\n", ctx->params3d->scale_factor); */
	break;

      case SDL_BUTTON_WHEELDOWN:
	if (ctx->params3d.scale_factor > 11)
	  ctx->params3d.scale_factor *= 0.9;
	break;

      default:
	break;
      }
      break;

#ifndef FIXED
    case SDL_VIDEORESIZE:
      out_width = (evt.resize.w >= 8) ? evt.resize.w : out_width;
      out_height = (evt.resize.h >= 8) ? evt.resize.h : out_height;

      sws_freeContext(sws_context);
      sws_context = NULL;
      xfree(bufp);
      xfree(dstbufp);

      biniou_sdl_set_videomode(out_width, out_height, drv.sc->flags);
      break;
#endif

    default:
      break;
    }
  }
}


static void
osd_random_mode_elapsed(const Context_t *ctx)
{
  float pct = Alarm_elapsed_pct(ctx->a_random);
  u_char color = 255; /* TODO colormap->max */
  u_short height;
  SDL_Rect r;

#ifdef FIXED
#define out_width  WIDTH
#define out_height HEIGHT
#endif
  height = (u_short)((1.0 - pct) * out_height);
  r.x = out_width-PB_WIDTH;
  r.y = out_height-height;
  r.w = PB_WIDTH;
  r.h = height;

  SDL_FillRect(drv.sc, &r, color);
}


void
run(Context_t *ctx)
{
  lock_screen(drv);
  SDL_refresh_32bpp(ctx, drv.sc);

  if (ctx->osd_mode != OSD_NONE) {
    osd(ctx);
    
    if (ctx->random_mode != BR_NONE)
      osd_random_mode_elapsed(ctx);
  }

  unlock_screen(drv);
  SDL_get_event(ctx);
}


void
create(Context_t *ctx)
{
  char *window_title;
  char *icon_file;
  SDL_Surface *icon = NULL;
  Uint32 flags = 0;
  Uint32 colorkey;
  Uint32 subsystems;
  int ret;

  /* Initialize SDL */
  subsystems = SDL_WasInit(SDL_INIT_VIDEO);
  if (subsystems == 0) {
    ret = SDL_Init(SDL_INIT_VIDEO);
    if (ret == -1)
      xerror("Couldn't initialize SDL: %s\n", SDL_GetError());
  }

  ttf_init();

  /* XXX hardcoded icon name */
  icon_file = g_strdup_printf("%s/lebiniou.bmp", DEFAULT_DATADIR);
  icon = SDL_LoadBMP(icon_file);
  g_free(icon_file);
  colorkey = SDL_MapRGB(icon->format, 0, 0, 0);
  SDL_SetColorKey(icon, SDL_SRCCOLORKEY, colorkey);
  SDL_WM_SetIcon(icon, NULL);
  SDL_FreeSurface(icon);

  flags = SDL_HWSURFACE | SDL_ANYFORMAT | SDL_HWPALETTE |
          SDL_DOUBLEBUF | SDL_HWACCEL | SDL_RLEACCEL;

#ifndef FIXED
  flags |= SDL_RESIZABLE;
  out_width = WIDTH;
  out_height = HEIGHT;
  biniou_sdl_set_videomode(out_width, out_height, flags);
#else
  biniou_sdl_set_videomode(WIDTH, HEIGHT, flags);
#endif

  window_title = g_strdup_printf("Le Biniou (%dx%d)", WIDTH, HEIGHT);
  SDL_WM_SetCaption(window_title, NULL);

  g_free(window_title);

#ifdef NO_MOUSE_CURSOR
  SDL_ShowCursor(SDL_DISABLE);
#endif
  SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);

  out = Buffer8_new();
}


void
destroy(Context_t *ctx)
{
#ifndef FIXED
  sws_freeContext(sws_context);
  xfree(bufp);
  xfree(dstbufp);
#endif
  SDL_FreeSurface(drv.sc);
  ttf_quit();
  SDL_Quit();
  Buffer8_delete(out);
}


void
fullscreen(const int fs)
{
  int do_it = drv.sc->flags & SDL_FULLSCREEN;

  /* houlala c'est loin l'algebre de bool en C... */
  do_it = (fs && !do_it) || (!fs && do_it);

  if (do_it) {
    printf("[S] Toggle full-screen\n");
    SDL_WM_ToggleFullScreen(drv.sc);
  }
}


void
switch_cursor()
{
  int on;

  on = SDL_ShowCursor(SDL_QUERY);
  SDL_ShowCursor(on ? SDL_DISABLE : SDL_ENABLE);
}
