/***********************************************************************************

    Copyright (C) 2007-2019 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/

#include <src/dialog_sync.hpp>
#include <cassert>

#include "lifeograph.hpp"

using namespace LIFEO;

constexpr char COLOR_IGNORE[]      = "#999999";
constexpr char COLOR_ADD[]         = "#ccffcc";
constexpr char COLOR_OVERWRITE[]   = "#d0e0bb";

DialogSync::DialogSync( BaseObjectType* cobject,
                        const Glib::RefPtr< Gtk::Builder >& refbuilder )
:   DialogEvent( cobject, refbuilder ), m_dialog_password( nullptr ), m_diary( nullptr )
{
    Gtk::CellRendererPixbuf* CRP_icon{ nullptr };
    Gtk::CellRendererText* CRT_name{ nullptr };
    Gtk::CellRendererCombo* CRC_action{ nullptr };
    Gtk::TreeViewColumn* TVC_name{ nullptr };
    Gtk::Button* B_ignore{ nullptr };

    Gtk::CellRendererText* CRT_comp_l{ nullptr };
    Gtk::CellRendererText* CRT_comp_r{ nullptr };
    Gtk::TreeViewColumn* TVC_comp_l{ nullptr };
    Gtk::TreeViewColumn* TVC_comp_r{ nullptr };

    try
    {
        m_TS_content = TreeStoreImport::create();

        refbuilder->get_widget( "B_sync_go", m_B_go );
        refbuilder->get_widget( "B_sync_cancel", m_B_cancel );
        refbuilder->get_widget( "IB_sync", m_IB_sync );
        refbuilder->get_widget( "L_sync_info", m_L_info );
        refbuilder->get_widget( "FB_sync_diary", m_FCB_diary );
        refbuilder->get_widget_derived( "E_sync_add_tag", m_tag_widget );
        refbuilder->get_widget( "RB_sync_changed_ignore", m_RB_changed_ignore );
        refbuilder->get_widget( "RB_sync_changed_overwrite", m_RB_changed_overwrite );
        refbuilder->get_widget( "RB_sync_changed_add", m_RB_changed_add );
        refbuilder->get_widget( "RB_sync_new_ignore", m_RB_new_ignore );
        refbuilder->get_widget( "RB_sync_new_add", m_RB_new_add );
        refbuilder->get_widget( "B_sync_apply_scheme", m_B_apply_scheme );

        refbuilder->get_widget( "TV_sync_content", m_TV_contents );
        refbuilder->get_widget( "P_sync", m_P_contents );
        refbuilder->get_widget( "Bx_sync_operations", m_Bx_operations );
        refbuilder->get_widget( "TV_sync_compare", m_TV_compare );
        refbuilder->get_widget( "Sw_sync_similar", m_Sw_similar );
        refbuilder->get_widget( "Po_sync_status", m_Po_status );
        refbuilder->get_widget( "Po_sync_action", m_Po_action );
        refbuilder->get_widget( "B_sync_ignore", B_ignore );
        refbuilder->get_widget( "B_sync_overwrite", m_B_ow );
        refbuilder->get_widget( "B_sync_add", m_B_add );
        refbuilder->get_widget( "CB_sync_show_accepted", m_CB_show_accepted );
        refbuilder->get_widget( "CB_sync_show_ignored", m_CB_show_ignored );
        refbuilder->get_widget( "CB_sync_show_new", m_CB_show_new );
        refbuilder->get_widget( "CB_sync_show_changed", m_CB_show_changed );

        CRP_icon = Gtk::manage( new Gtk::CellRendererPixbuf );
        CRT_name = Gtk::manage( new Gtk::CellRendererText );
        CRC_action = Gtk::manage( new Gtk::CellRendererCombo );
        TVC_name = Gtk::manage( new Gtk::TreeViewColumn( _( "Name" ) ) );
        m_TVC_status = Gtk::manage( new Gtk::TreeViewColumn( _( "Status" ),
                                                             m_TS_content->colrec.status ) );
        m_TVC_action = Gtk::manage( new Gtk::TreeViewColumn( _( "Action" ) ) );

        TVC_comp_l = Gtk::manage( new Gtk::TreeViewColumn( _( "Local" ) ) );
        TVC_comp_r = Gtk::manage( new Gtk::TreeViewColumn( _( "Remote" ) ) );
        CRT_comp_l = Gtk::manage( new Gtk::CellRendererText );
        CRT_comp_r = Gtk::manage( new Gtk::CellRendererText );
    }
    catch( ... )
    {
        throw LIFEO::Error( "Failed to create sync dialog" );
    }

    // ACTION COMBOBOX
    m_LS_action_ow = Gtk::ListStore::create( m_colrec_combo );
    auto row = *( m_LS_action_ow->append() );
    row[ m_colrec_combo.name ] = STR0/SI::SI_IGNORE;
    row = *( m_LS_action_ow->append() );
    row[ m_colrec_combo.name ] = STR0/SI::OVERWRITE;

    m_LS_action_add = Gtk::ListStore::create( m_colrec_combo );
    row = *( m_LS_action_add->append() );
    row[ m_colrec_combo.name ] = STR0/SI::SI_IGNORE;
    row = *( m_LS_action_add->append() );
    row[ m_colrec_combo.name ] = STR0/SI::ADD;

    m_LS_action_add_ow = Gtk::ListStore::create( m_colrec_combo );
    row = *( m_LS_action_add_ow->append() );
    row[ m_colrec_combo.name ] = STR0/SI::SI_IGNORE;
    row = *( m_LS_action_add_ow->append() );
    row[ m_colrec_combo.name ] = STR0/SI::ADD;
    row = *( m_LS_action_add_ow->append() );
    row[ m_colrec_combo.name ] = STR0/SI::OVERWRITE;

    // TREEVIEW CONTENT
    m_TV_contents->set_model( m_TS_content );

    // not at all sure about the below approach to sizing
    TVC_name->set_sizing( Gtk::TREE_VIEW_COLUMN_FIXED );
    m_TVC_status->set_sizing( Gtk::TREE_VIEW_COLUMN_FIXED );
    m_TVC_action->set_sizing( Gtk::TREE_VIEW_COLUMN_FIXED );
    m_TVC_status->set_fixed_width( 110 );
    m_TVC_action->set_fixed_width( 110 );

    m_TVC_status->set_sort_indicator( true );
    m_TVC_action->set_sort_indicator( true );
    m_TV_contents->append_column( *TVC_name );
    m_TV_contents->append_column( *m_TVC_status );
    m_TV_contents->append_column( *m_TVC_action );

    // NAME COLUMN
    CRT_name->property_ellipsize() = Pango::ELLIPSIZE_END;
    TVC_name->pack_start( *CRP_icon, false );
    TVC_name->pack_start( *CRT_name );
    TVC_name->add_attribute( CRP_icon->property_pixbuf(), m_TS_content->colrec.icon );
    TVC_name->add_attribute( CRT_name->property_markup(), m_TS_content->colrec.name );
    TVC_name->set_expand( true );

    // STATUS COLUMN
    m_TVC_status->set_clickable( true );
    m_TVC_status->signal_clicked().connect(
            sigc::mem_fun( this, &DialogSync::show_status_filter ) );
    m_TVC_status->set_cell_data_func( * m_TV_contents->get_column_cell_renderer( 1 ),
                                      sigc::mem_fun( this, &DialogSync::set_cell_color_s ) );

    // ACTION COLUMN
    m_TVC_action->pack_start( *CRC_action, false );
    m_TVC_action->add_attribute( CRC_action->property_text(), m_TS_content->colrec.action );
    m_TVC_action->add_attribute( CRC_action->property_model(), m_TS_content->colrec.action_combo );
    m_TVC_action->set_clickable( true );
    CRC_action->property_text_column() = 0;
    CRC_action->property_editable() = true;
    CRC_action->property_has_entry() = false;
    CRC_action->signal_edited().connect(
            sigc::mem_fun( this, &DialogSync::handle_TV_action_edited ) );
    m_TVC_action->signal_clicked().connect(
            sigc::mem_fun( this, &DialogSync::show_action_filter ) );
    m_TVC_status->set_cell_data_func( * m_TV_contents->get_column_cell_renderer( 2 ),
                                      sigc::mem_fun( this, &DialogSync::set_cell_color_a ) );

    // FILTERING
    DialogOpenDiary::init_filters();
    m_FCB_diary->add_filter( DialogOpenDiary::filter_any );
    m_FCB_diary->add_filter( DialogOpenDiary::filter_diary );
    m_FCB_diary->set_filter( DialogOpenDiary::filter_diary );

    // TREEVIEW COMPARE
    CRT_comp_l->property_wrap_mode() = Pango::WRAP_WORD;
    CRT_comp_r->property_wrap_mode() = Pango::WRAP_WORD;
    CRT_comp_l->property_wrap_width() = 300;
    CRT_comp_r->property_wrap_width() = 300;
    CRT_comp_l->property_ellipsize() = Pango::ELLIPSIZE_END;
    CRT_comp_r->property_ellipsize() = Pango::ELLIPSIZE_END;
    CRT_comp_l->property_cell_background_rgba() = Gdk::RGBA{ "#b0b0b0" };
    CRT_comp_r->property_cell_background_rgba() = Gdk::RGBA{ "#b0b0b0" };
    m_LS_compare = Gtk::ListStore::create( m_CR_compare );
    m_TV_compare->set_model( m_LS_compare );
    m_TV_compare->append_column( _( "Name" ), m_CR_compare.name );
    TVC_comp_l->pack_start( *CRT_comp_l );
    TVC_comp_r->pack_start( *CRT_comp_r );
    TVC_comp_l->add_attribute( CRT_comp_l->property_markup(), m_CR_compare.str_l );
    TVC_comp_r->add_attribute( CRT_comp_r->property_markup(), m_CR_compare.str_r );
    TVC_comp_l->set_expand( true );
    TVC_comp_r->set_expand( true );
    m_TV_compare->append_column( *TVC_comp_l );
    m_TV_compare->append_column( *TVC_comp_r );

    // BUTTONS
    set_widget_css( B_ignore, Ustring::compose( "button { background: %1 }", COLOR_IGNORE ) );
    set_widget_css( m_B_ow, Ustring::compose( "button { background: %1 }", COLOR_OVERWRITE ) );
    set_widget_css( m_B_add, Ustring::compose( "button { background: %1 }", COLOR_ADD ) );

    // SIGNALS
    m_B_apply_scheme->signal_clicked().connect(
            sigc::mem_fun( this, &DialogSync::update_maps ) );

    m_TV_contents->get_selection()->signal_changed().connect(
            sigc::mem_fun( this, &DialogSync::handle_TV_selection_changed ) );
    m_CB_show_accepted->signal_toggled().connect(
            sigc::mem_fun( *this, &DialogSync::update_filter ) );
    m_CB_show_ignored->signal_toggled().connect(
            sigc::mem_fun( *this, &DialogSync::update_filter ) );
    m_CB_show_new->signal_toggled().connect(
            sigc::mem_fun( *this, &DialogSync::update_filter ) );
    m_CB_show_changed->signal_toggled().connect(
            sigc::mem_fun( *this, &DialogSync::update_filter ) );

    B_ignore->signal_clicked().connect(
            sigc::bind( sigc::mem_fun( this, &DialogSync::change_action ), SI::SI_IGNORE ) );
    m_B_add->signal_clicked().connect(
            sigc::bind( sigc::mem_fun( this, &DialogSync::change_action ), SI::ADD ) );
    m_B_ow->signal_clicked().connect(
            sigc::bind( sigc::mem_fun( this, &DialogSync::change_action ), SI::OVERWRITE ) );

    m_FCB_diary->signal_file_set().connect(
            sigc::mem_fun( this, &DialogSync::open_remote_diary ) );

    m_B_go->signal_clicked().connect(
            sigc::bind( sigc::mem_fun( this, &DialogEvent::response ), RESPONSE_GO ) );
    m_B_cancel->signal_clicked().connect(
            sigc::bind( sigc::mem_fun( this, &DialogEvent::response ), RESPONSE_CANCEL ) );

    m_IB_sync->signal_response().connect( [ this ]( int ){ m_IB_sync->set_visible( false ); } );
}

void
DialogSync::on_show()
{
    Gtk::Dialog::on_show();
    set_not_ready( _( "Select a diary file for synchronizing" ) );

    m_Bx_operations->set_visible( false );
    m_CB_show_changed->set_active( true );
    m_CB_show_new->set_active( true );
    m_CB_show_accepted->set_active( true );
    m_CB_show_ignored->set_active( true );

    m_RB_changed_add->set_active( true );
    m_RB_new_add->set_active( true );

    m_FCB_diary->unselect_all();
    m_tag_widget->clear();

    m_tag_widget->populate();

    m_FCB_diary->grab_focus();

    m_content_filter = FILTER_DEFAULT;
}

void
DialogSync::open_remote_diary()
{
    if( m_diary != nullptr )
        delete m_diary;

    m_diary = new Diary;

    if( m_diary->set_path( m_FCB_diary->get_filename(), Diary::SPT_READ_ONLY ) == SUCCESS )
    {
        if( m_diary->read_header() == SUCCESS )
        {
            if( m_diary->is_encrypted() )
                ask_for_password();
            else
                open_remote_diary2();
        }
        else
        {
            set_not_ready( _( "Selected file does not seem to be a valid diary" ) );
        }
    }
    else
        set_not_ready( _( "Selected file cannot be read" ) );
}

void
DialogSync::open_remote_diary2()
{
    switch( m_diary->read_body())
    {
        case SUCCESS:
        {
            DialogPassword::finish( m_diary->get_path() );
            m_P_contents->set_position( get_height() * 0.55 );
            update_maps();
            populate_content();
            if( m_flag_identical )
                set_not_ready( "Diaries are identical" );
            else
            {
                set_ready();
                m_L_info->set_text(_( "Please select what to import/sync and press Go" ) );
            }
        }
            break;
        case WRONG_PASSWORD:
            open_remote_diary();
            break;
        default:
            set_not_ready( _( "Selected file cannot be read" ) );
            break;
    }
}

void
DialogSync::ask_for_password()
{
    DialogPassword::launch( DialogPassword::OT_OPEN,
                            m_diary, nullptr, m_FCB_diary,
                            sigc::mem_fun( this, &DialogSync::open_remote_diary2 ),
                            [ this ] { m_FCB_diary->unselect_all(); } );
}

void
DialogSync::update_maps()
{
    PRINT_DEBUG( "DialogImport::handle_scheme_changed()" );

    m_comparison_map.clear();
    m_action_map.clear();
    m_flag_identical = true;

    for( auto kv_elem : m_diary->m_ids )
    {
        const DiaryElement* elem_l;
        SI cmp_res{ Diary::d->compare_foreign_elem( kv_elem.second, elem_l ) };
        m_comparison_map[ kv_elem.first ] = cmp_res;

        if( cmp_res == SI::CHANGED )
        {
            if( m_RB_changed_ignore->get_active() )
                m_action_map[ kv_elem.first ] = SI::SI_IGNORE;
            else if( m_RB_changed_overwrite->get_active() )
                m_action_map[ kv_elem.first ] = SI::OVERWRITE;
            else //if( m_RB_changed_add->get_active() )
                m_action_map[ kv_elem.first ] =
                        ( elem_l->get_type() == DiaryElement::ET_ENTRY ||
                          elem_l->get_type() == DiaryElement::ET_TAG ) ?
                                SI::ADD : SI::OVERWRITE;

            if( m_flag_identical )
                m_flag_identical = false;
        }
        else if( cmp_res == SI::NEW )
        {
            if( m_RB_new_ignore->get_active() )
                m_action_map[ kv_elem.first ] = SI::SI_IGNORE;
            else //if( m_RB_new_add->get_active() )
                m_action_map[ kv_elem.first ] = SI::ADD;

            if( m_flag_identical )
                m_flag_identical = false;
        }
        else
            m_action_map[ kv_elem.first ] = SI::SI_IGNORE;
    }

    populate_content();
}

void
DialogSync::handle_TV_selection_changed()
{
    if( !is_ready() )
        return;

    m_LS_compare->clear();
    m_Bx_operations->set_visible( false );
    m_B_ow->set_visible( false );
    m_B_add->set_visible( false );

    Gtk::TreeIter iter{ m_TV_contents->get_selection()->get_selected() };

    if( !iter )
        return;

    Gtk::TreeRow row{ *( iter ) };
    const DiaryElement* elem_r{ row[ m_TS_content->colrec.ptr ] };
    const SI si{ row[ m_TS_content->colrec.comp_res ] };

    if( elem_r == nullptr || si == SI::INTACT )
        return;

    m_Bx_operations->set_visible( true );

    switch( elem_r->get_type() )
    {
        case DiaryElement::ET_ENTRY:
        case DiaryElement::ET_TAG:
            m_B_add->set_visible( true );
            m_B_ow->set_visible( si == SI::CHANGED );
            break;
        default:
            m_B_add->set_visible( si == SI::NEW );
            m_B_ow->set_visible( si == SI::CHANGED );
            break;
    }

    const DiaryElement* elem_l{ Diary::d->get_corresponding_elem( elem_r ) };

    static const Ustring tpl[] = { "<span color=\"red\">∅</span>", // 0: DELETED
                                   "<span color=\"green\">%1</span>", // 1: NEW
                                   "<span color=\"blue\">%1</span>", // 2: CHANGED
                                   "<span color=\"%1\">%2</span>" };

    auto add_item = [ & ]( const SI id, const Ustring& str_l,
                           const Ustring& str_r, int l, int r )
    {
        row = * ( m_LS_compare->append() );

        row[ m_CR_compare.name ] = STR0/id;
        row[ m_CR_compare.str_l ] =
                Ustring::compose( tpl[ l ], Glib::Markup::escape_text( str_l ) );
        row[ m_CR_compare.str_r ] =
                Ustring::compose( tpl[ r ], Glib::Markup::escape_text( str_r ) );
    };

    if( elem_l )
    {
        const SKVVec props_l = elem_l->get_as_string();
        const SKVVec props_r = elem_r->get_as_string();
        const int loop_size{ max( props_l.size(), props_r.size() ) };
        unsigned i_l{ 0 }, i_r{ 0 };

        for( int i = 0; i < loop_size; i++ )
        {
            if( i_l >= props_l.size() )
            {
                add_item( props_r[ i_r ].first, "", props_r[ i_r ].second, 0, 1 );
                i_r++;
            }
            else
            if( i_r >= props_r.size() )
            {
                add_item( props_l[ i_l ].first, props_l[ i_l ].second, "", 1, 0 );
                i_l++;
            }
            else
            if( props_l[ i_l ].first != props_r[ i_r ].first )
            {
                if( props_l[ i_l ].first < props_r[ i_r ].first )
                {
                    add_item( props_l[ i_l ].first, props_l[ i_l ].second, "", 1, 0 );
                    i_l++;
                }
                else
                {
                    add_item( props_r[ i_r ].first, "", props_r[ i_r ].second, 0, 1 );
                    i_r++;
                }
            }
            else
            if( props_l[ i_l ].second != props_r[ i_r ].second )
            {
                if( i_l + 1 < props_l.size() &&
                    props_l[ i_l + 1 ].first == props_r[ i_r ].first &&
                    props_l[ i_l + 1 ].second == props_r[ i_r ].second )
                {
                    add_item( props_l[ i_l ].first, props_l[ i_l ].second, "", 1, 0 );
                    i_l++;
                }
                else
                if( i_r + 1 < props_r.size() &&
                    props_l[ i_l ].first == props_r[ i_r + 1 ].first &&
                    props_l[ i_l ].second == props_r[ i_r + 1 ].second )
                {
                    add_item( props_r[ i_r ].first, "", props_r[ i_r ].second, 0, 1 );
                    i_r++;
                }
                else
                {
                    add_item( props_l[ i_l ].first, props_l[ i_l ].second,
                              props_r[ i_r ].second, 2, 2 );
                    i_l++;
                    i_r++;
                }
            }
            else
            {
                i_l++;
                i_r++;
            }
        }
    }
}

void
DialogSync::handle_TV_action_edited( const Ustring& path_str, const Ustring& new_text )
{
    Gtk::TreePath path( path_str );
    auto iter = m_TS_content->get_iter( path );
    if( iter )
    {
        auto row = *iter;
        DiaryElement* elem{ row[ m_TS_content->colrec.ptr ] };
        SI action{ new_text == STR0/SI::ADD ? SI::ADD :
                ( new_text == STR0/SI::SI_IGNORE ? SI::SI_IGNORE : SI::OVERWRITE ) };
        if( elem )
        {
            m_action_map[ elem->get_id() ] = action;

            if( m_Sw_similar->get_active() )
            {
                for( auto& kv : m_action_map )
                {
                    DiaryElement* elem_i{ m_diary->get_element( kv.first ) };
                    if( elem_i->get_type() == elem->get_type() &&
                        m_comparison_map[ kv.first ] == m_comparison_map[ elem->get_id() ] )
                        m_action_map[ kv.first ] = action;
                }
            }
        }

        populate_content();
    }
}

void
DialogSync::update_filter()
{
    if( ! m_diary )
        return;

    m_content_filter = 0;

    if( m_CB_show_changed->get_active() )
        m_content_filter |= FILTER_SHOW_CHANGED;
    if( m_CB_show_new->get_active() )
        m_content_filter |= FILTER_SHOW_NEW;

    if( m_CB_show_ignored->get_active() )
        m_content_filter |= FILTER_SHOW_IGNORED;
    if( m_CB_show_accepted->get_active() )
        m_content_filter |= FILTER_SHOW_ACCEPTED;

    populate_content();
}

void
DialogSync::show_status_filter()
{
    Gdk::Rectangle rect{ m_TVC_status->get_x_offset(), 0,
                         m_TVC_status->get_width(),
                         m_TVC_status->get_button()->get_allocated_height() };
    m_Po_status->set_relative_to( *m_TV_contents );
    m_Po_status->set_pointing_to( rect );
    m_Po_status->show();
}

void
DialogSync::show_action_filter()
{
    Gdk::Rectangle rect{ m_TVC_action->get_x_offset(), 0,
                         m_TVC_action->get_width(),
                         m_TVC_action->get_button()->get_allocated_height() };
    m_Po_action->set_relative_to( *m_TV_contents );
    m_Po_action->set_pointing_to( rect );
    m_Po_action->show();
}

void
DialogSync::change_action( SI si )
{
    auto treesel{ m_TV_contents->get_selection() };
    if( treesel->count_selected_rows() < 1 )
        return;

    handle_TV_action_edited( m_TS_content->get_path( treesel->get_selected() ).to_string(),
                             STR0/si );
}

void
DialogSync::populate_content()
{
    auto add_header = [ & ]( const Icon& icon, int header_type )
    {
        Gtk::TreeRow row{ * m_TS_content->append() };
        row[ m_TS_content->colrec.ptr ] = nullptr;
        row[ m_TS_content->colrec.icon ] = icon;
        row[ m_TS_content->colrec.header_type ] = header_type;
        row[ m_TS_content->colrec.comp_res ] = SI::_NONE_;
        return row;
    };

    auto update_header = [ & ]( Gtk::TreeRow& row, const Ustring& name, const int size )
    {
        if( row.children().empty() )
            m_TS_content->erase( row );
        else
            row[ m_TS_content->colrec.name ] = Glib::ustring::compose( "<b>%1</b> (%2/%3)",
                                                                       name,
                                                                       row.children().size(),
                                                                       size );
    };

    auto get_action_combo = [ & ]( DiaryElement* elem, SI si )
    {
        switch( elem->get_type() )
        {
            case DiaryElement::ET_ENTRY:
            case DiaryElement::ET_TAG:
                return( si == SI::NEW ? m_LS_action_add : m_LS_action_add_ow );
            default:
                return( si == SI::NEW ? m_LS_action_add : m_LS_action_ow );
        }
    };

    auto add_elem = [ & ]( DiaryElement* elem_r, Gtk::TreeRow& parent )
    {
        SI cmp_res{ m_comparison_map[ elem_r->get_id() ] };
        SI action{ m_action_map[ elem_r->get_id() ] };

        if( cmp_res == SI::INTACT ||
            ( cmp_res == SI::NEW && !( m_content_filter & FILTER_SHOW_NEW ) ) ||
            ( cmp_res == SI::CHANGED && !( m_content_filter & FILTER_SHOW_CHANGED ) ) ||
            ( action == SI::SI_IGNORE && !( m_content_filter & FILTER_SHOW_IGNORED ) ) ||
            ( action != SI::SI_IGNORE && !( m_content_filter & FILTER_SHOW_ACCEPTED ) ) )
            return;

        Gtk::TreeRow row = *( m_TS_content->append( parent.children() ) );

        row[ m_TS_content->colrec.name ] = elem_r->get_list_str();
        row[ m_TS_content->colrec.ptr ] = elem_r;
        row[ m_TS_content->colrec.icon ] = elem_r->get_icon();
        row[ m_TS_content->colrec.status ] = STR0/cmp_res;
        row[ m_TS_content->colrec.action ] = STR0/action;
        row[ m_TS_content->colrec.action_combo ] = get_action_combo( elem_r, cmp_res );
        row[ m_TS_content->colrec.header_type ] = 0; //TreeStoreImport::HT_NONE;
        row[ m_TS_content->colrec.comp_res ] = cmp_res;
    };

    Gtk::TreePath path_top, path_btm;
    m_TV_contents->get_visible_range( path_top, path_btm );

    m_TS_content->clear();

    // ADD ELEMENTS
    Gtk::TreeRow row_tag_ctgs =
            add_header( Lifeograph::icons->tag_category_16, TreeStoreImport::HT_TAG_CTGS );
    Gtk::TreeRow row_tags =
            add_header( Lifeograph::icons->tag_16, TreeStoreImport::HT_TAGS );
    Gtk::TreeRow row_topics =
            add_header( Lifeograph::icons->chapter_ordr_16, TreeStoreImport::HT_NMBR_CHAPTERS );
    Gtk::TreeRow row_groups =
            add_header( Lifeograph::icons->chapter_free_16, TreeStoreImport::HT_FREE_CHAPTERS );
    Gtk::TreeRow row_entries =
            add_header( Lifeograph::icons->entry_16, TreeStoreImport::HT_ENTRIES );
    for( auto& kv_elem : m_action_map )
    {
        DiaryElement* elem{ m_diary->get_element( kv_elem.first ) };
        switch( elem->get_type() )
        {
            case DiaryElement::ET_TAG_CTG:
                add_elem( elem, row_tag_ctgs );
                break;
            case DiaryElement::ET_TAG:
                add_elem( elem, row_tags );
                break;
            case DiaryElement::ET_TOPIC:
                add_elem( elem, row_topics );
                break;
            case DiaryElement::ET_GROUP:
                add_elem( elem, row_groups );
                break;
            case DiaryElement::ET_ENTRY:
                add_elem( elem, row_entries );
                break;
            default:
                break;
        }
    }
    update_header( row_tag_ctgs, _( "TAG CATEGORIES" ), m_diary->m_tag_categories.size() );
    update_header( row_tags, _( "TAGS" ), m_diary->get_tags()->size() );
    update_header( row_topics, _( "NUMBERED CHAPTERS" ), m_diary->m_topics->size() );
    update_header( row_groups, _( "FREE CHAPTERS" ), m_diary->m_groups->size() );
    update_header( row_entries, _( "ENTRIES" ), m_diary->get_entries().size() );

    // CHAPTER CATEGORIES
    Gtk::TreeRow row, row2;
    for( auto& kv_ctg : m_diary->m_chapter_categories )
    {
        SI cmp_res{ m_comparison_map[ kv_ctg.second->get_id() ] };
        row = add_header( Lifeograph::icons->chapter_temp_16, TreeStoreImport::HT_CHAPTER_CTG );
        // Chapter Categories are not plain headers but also elements per se
        // so we set element related fields manually:
        row[ m_TS_content->colrec.ptr ] = kv_ctg.second;
        row[ m_TS_content->colrec.status ] = STR0/cmp_res;
        row[ m_TS_content->colrec.action ] = STR0/m_action_map[ kv_ctg.second->get_id() ];
        row[ m_TS_content->colrec.comp_res ] = cmp_res;
        // DATED CHAPTERS
        for( auto& kv_chapter : * kv_ctg.second )
            add_elem( kv_chapter.second, row );

        if( cmp_res == SI::INTACT && row.children().empty() )
            m_TS_content->erase( row );
        else
            update_header( row, kv_ctg.second->get_name(), kv_ctg.second->size() );
    }

    m_TV_contents->expand_all();
    if( path_top.size() > 0 )
    {
        m_TV_contents->scroll_to_row( path_top );
        m_TV_contents->scroll_to_row( path_btm );
    }
}

void
DialogSync::set_cell_color_s( Gtk::CellRenderer* cell, const Gtk::TreeModel::iterator& iter )
{
    Gdk::RGBA color_b{ "#b0b0b0" };
    Gdk::RGBA color_f{ "#000000" };

    switch( ( * iter )[ m_TS_content->colrec.comp_res ]  )
    {
        case SI::CHANGED:
            color_b.set( "#ffcc66" );
            color_f.set( "#664422" );
            break;
        case SI::NEW:
            color_b.set( "#66cc66" );
            color_f.set( "#446644" );
            break;
        case SI::DELETED:
            color_b.set( "#cc6666" );
            color_f.set( "#664444" );
            break;
        default:
            break;
    }

    cell->property_cell_background_rgba() = color_b;
    dynamic_cast< Gtk::CellRendererText* >( cell )->property_foreground_rgba() = color_f;
}

void
DialogSync::set_cell_color_a( Gtk::CellRenderer* cell, const Gtk::TreeModel::iterator& iter )
{
    DiaryElement* elem{ ( * iter )[ m_TS_content->colrec.ptr ] };
    Gdk::RGBA color_b{ "#b0b0b0" };
    Gdk::RGBA color_f{ "#000000" };

    if( elem != nullptr )
    {
        switch( m_action_map[ elem->get_id() ] )
        {
            case SI::ADD:
                color_b.set( COLOR_ADD );
                break;
            case SI::OVERWRITE:
                color_b.set( COLOR_OVERWRITE );
                break;
            default:
                color_b.set( COLOR_IGNORE );
                break;
        }
    }

    cell->property_cell_background_rgba() = color_b;
    dynamic_cast< Gtk::CellRendererText* >( cell )->property_foreground_rgba() = color_f;
}

inline bool
DialogSync::is_ready()
{
    return( not( m_flag_identical ) && m_flag_diary_is_ready );
}

void
DialogSync::set_ready()
{
    m_B_apply_scheme->set_sensitive( true );
    m_B_go->set_sensitive( true );
    m_flag_diary_is_ready = true;
}

void
DialogSync::set_not_ready( const Ustring& message )
{
    m_flag_diary_is_ready = false;

    m_B_go->set_sensitive( false );
    m_B_apply_scheme->set_sensitive( false );

    if( m_diary )
    {
        m_TS_content->clear();
        m_action_map.clear();
        delete m_diary;
        m_diary = nullptr;
    }

    if( ! m_IB_sync->is_visible() )
        m_IB_sync->set_visible( true );

    m_L_info->set_text( message );
}

void
DialogSync::on_response( int response )
{
    if( response == RESPONSE_GO )
    {
        // prepare the new tag
        Tag* tag_all{ nullptr };
        NameAndValue nav{ m_tag_widget->get_nav() };
        SI filter;

        if( ! nav.name.empty() )
            tag_all = ( nav.status & NameAndValue::HAS_VALUE ?
                    Diary::d->create_tag( nav.name, nullptr, ChartPoints::DEFAULT ) :
                    Diary::d->create_tag( nav.name, nullptr ) );

        for( auto& kv_tag_ctg : m_diary->m_tag_categories )
        {
            filter = m_action_map[ kv_tag_ctg.second->get_id() ];
            if( filter != SI::SI_IGNORE )
                Diary::d->import_tag_ctg( kv_tag_ctg.second, filter == SI::ADD );
        }

        for( auto& kv_tag : m_diary->m_tags )
        {
            filter = m_action_map[ kv_tag.second->get_id() ];
            if( filter != SI::SI_IGNORE )
                Diary::d->import_tag( kv_tag.second, filter == SI::ADD );
        }

        for( auto& kv_cpt_ctg : m_diary->m_chapter_categories )
        {
            filter = m_action_map[ kv_cpt_ctg.second->get_id() ];
            if( filter != SI::SI_IGNORE )
                Diary::d->import_chapter_ctg( kv_cpt_ctg.second, filter == SI::ADD );

            for( auto& kv_cpt : * kv_cpt_ctg.second )
            {
                filter = m_action_map[ kv_cpt.second->get_id() ];
                if( filter != SI::SI_IGNORE )
                    Diary::d->import_chapter( kv_cpt.second, filter == SI::ADD );
            }
        }

        for( auto& kv_topic : *( m_diary->m_topics ) )
        {
            filter = m_action_map[ kv_topic.second->get_id() ];
            if( filter != SI::SI_IGNORE )
                Diary::d->import_chapter( kv_topic.second, filter == SI::ADD );
        }

        for( auto& kv_group : *( m_diary->m_groups ) )
        {
            filter = m_action_map[ kv_group.second->get_id() ];
            if( filter != SI::SI_IGNORE )
                Diary::d->import_chapter( kv_group.second, filter == SI::ADD );
        }

        for( auto& kv_entry : m_diary->m_entries )
        {
            filter = m_action_map[ kv_entry.second->get_id() ];
            if( filter != SI::SI_IGNORE )
                Diary::d->import_entry( kv_entry.second, tag_all, nav.value, filter == SI::ADD );
        }
    }

    set_not_ready( "" );

    Gtk::Dialog::on_response( response );
}
