/*
    Copyright (c) 1996-2003 Maximum Entropy Data Consultants Ltd,
                            114c Milton Road, Cambridge CB4 1XE, England

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>

#define BITS 5			/* number of bits used to store the axis values, size of Hilbert space */
#define DIMS 3			/* number of dimensions in the Hilbert space */

/** Hilbert transpose for @b bits and @n dimensions */
static void AxestoTranspose(uint32_t X[DIMS])
{
	uint32_t P, Q, t;
	int b = BITS, n = DIMS, i;

	/* Inverse undo */
	for (Q = 1 << (b - 1); Q > 1; Q >>= 1) {
		P = Q - 1;
		if (X[0] & Q)
			X[0] ^= P;	/* invert */
		for (i = 1; i < n; i++)
			if (X[i] & Q)
				X[0] ^= P;	/* invert */
			else {
				t = (X[0] ^ X[i]) & P;
				X[0] ^= t;
				X[i] ^= t;
			}	/* exchange */
	}

	/* Gray encode (inverse of decode) */
	for (i = 1; i < n; i++)
		X[i] ^= X[i - 1];
	t = X[n - 1];
	for (i = 1; i < b; i <<= 1)
		X[n - 1] ^= X[n - 1] >> i;
	t ^= X[n - 1];
	for (i = n - 2; i >= 0; i--)
		X[i] ^= t;
}

static void TransposetoAxes(uint32_t X[DIMS])
{
	uint32_t M, P, Q, t;
	int b = BITS, n = DIMS, i;

	/* Gray decode by  H ^ (H/2) */
	t = X[n - 1] >> 1;
	for (i = n - 1; i; i--)
		X[i] ^= X[i - 1];
	X[0] ^= t;

	/* Undo excess work */
	M = 2 << (b - 1);
	for (Q = 2; Q != M; Q <<= 1) {
		P = Q - 1;
		for (i = n - 1; i; i--)
			if (X[i] & Q) {
				X[0] ^= P;	/* invert */
			} else {
				t = (X[0] ^ X[i]) & P;
				X[0] ^= t;
				X[i] ^= t;
			}			/* exchange */
		if (X[0] & Q)
			X[0] ^= P;		/* invert */
	}
}

void hilbert_inverse(uint32_t H, uint8_t *x, uint8_t *y, uint8_t *z)
{
	uint32_t X[DIMS];

	X[0] = (H >> 2 & 1) | (H >> 4 & 2) | (H >> 6 & 4) | (H >> 8 & 8) | (H >> 10 & 16);
	X[1] = (H >> 1 & 1) | (H >> 3 & 2) | (H >> 5 & 4) | (H >> 7 & 8) | (H >> 9 & 16);
	X[2] = (H >> 0 & 1) | (H >> 2 & 2) | (H >> 4 & 4) | (H >> 6 & 8) | (H >> 8 & 16);
	TransposetoAxes(X);
	*x = X[0], *y = X[1], *z = X[2];
}

int hilbert_integer(uint8_t x, uint8_t y, uint8_t z)
{
	uint32_t X[DIMS] = {x, y , z};	/* any position in 32x32x32 cube for BITS=5 */

#ifdef _WANT_MAIN
	printf("Axis coordinates = %d %d %d\n", X[0], X[1], X[2]);
#endif
	AxestoTranspose(X);
#ifdef _WANT_MAIN
	printf("Hilbert integer  = (%d%d%d%d%d%d%d%d%d%d%d%d%d%d%d)\n",
	       X[0] >> 4 & 1, X[1] >> 4 & 1, X[2] >> 4 & 1, X[0] >> 3 & 1,
	       X[1] >> 3 & 1, X[2] >> 3 & 1, X[0] >> 2 & 1, X[1] >> 2 & 1,
	       X[2] >> 2 & 1, X[0] >> 1 & 1, X[1] >> 1 & 1, X[2] >> 1 & 1,
	       X[0] >> 0 & 1, X[1] >> 0 & 1, X[2] >> 0 & 1);
#endif
	return	((X[2] >> 0 & 1) <<  0) + ((X[1] >> 0 & 1) <<  1) +
		((X[0] >> 0 & 1) <<  2) + ((X[2] >> 1 & 1) <<  3) +
		((X[1] >> 1 & 1) <<  4) + ((X[0] >> 1 & 1) <<  5) +
		((X[2] >> 2 & 1) <<  6) + ((X[1] >> 2 & 1) <<  7) +
		((X[0] >> 2 & 1) <<  8) + ((X[2] >> 3 & 1) <<  9) +
		((X[1] >> 3 & 1) << 10) + ((X[0] >> 3 & 1) << 11) +
		((X[2] >> 4 & 1) << 12) + ((X[1] >> 4 & 1) << 13) +
		((X[0] >> 4 & 1) << 14);
}

#ifdef _WANT_MAIN
int main(int argc, char **argv)
{
	int i, H;
	uint32_t X[DIMS];

	if (argc != DIMS + 1) {
		printf("usage: %s X Y Z\n", argv[0]);
		exit(1);
	}
	for (i = 0; i < DIMS; i++)
		X[i] = atoi(argv[i + 1]);

	printf("Hilbert integer  = %d \n", hilbert_integer(X[0], X[1], X[2]));
	return 0;
}
#endif
