/* s7, a Scheme interpreter
 *
 *    derived from:
 *
 * --------------------------------------------------------------------------------
 * T I N Y S C H E M E    1 . 3 9
 *   Dimitrios Souflis (dsouflis@acm.org)
 *   Based on MiniScheme (original credits follow)
 * (MINISCM)               coded by Atsushi Moriwaki (11/5/1989)
 * (MINISCM)           E-MAIL :  moriwaki@kurims.kurims.kyoto-u.ac.jp
 * (MINISCM) This version has been modified by R.C. Secrist.
 * (MINISCM)
 * (MINISCM) Mini-Scheme is now maintained by Akira KIDA.
 * (MINISCM)
 * (MINISCM) This is a revised and modified version by Akira KIDA.
 * (MINISCM)	current version is 0.85k4 (15 May 1994)
 * --------------------------------------------------------------------------------
 *
 * apparently tinyScheme is under the BSD license, so I guess s7 is too.
 * Here is Snd's verbiage which can apply here:
 *
 *     The authors hereby grant permission to use, copy, modify, distribute,
 *     and license this software and its documentation for any purpose.  No
 *     written agreement, license, or royalty fee is required.  Modifications
 *     to this software may be copyrighted by their authors and need not
 *     follow the licensing terms described here.
 *
 * followed by the usual all-caps shouting about liability.
 *
 * --------------------------------------------------------------------------------
 *
 * s7, Bill Schottstaedt, Aug-08, bil@ccrma.stanford.edu
 *
 * Mike Scholz provided the FreeBSD support (complex trig funcs, etc)
 * Rick Taube, Andrew Burnson, Donny Ward, and Greg Santucci provided the MS Visual C++ support
 *
 * Documentation is in s7.h and s7.html.
 * s7test.scm is a regression test.
 * glistener.c is a gtk-based listener.
 * repl.scm is a vt100-based listener.
 * cload.scm and lib*.scm tie in various C libraries.
 * lint.scm checks Scheme code for infelicities.
 * r7rs.scm implements some of r7rs (small).
 * write.scm currrently has pretty-print.
 * mockery.scm has the mock-data definitions.
 * reactive.scm has reactive-set and friends.
 * stuff.scm has some stuff.
 * timing tests are in the snd tools directory
 *
 * s7.c is organized as follows:
 *
 *    structs and type flags
 *    constants
 *    GC
 *    stacks
 *    symbols and keywords
 *    environments
 *    continuations
 *    numbers
 *    characters
 *    strings and byte-vectors
 *    ports
 *    format
 *    lists
 *    vectors
 *    hash-tables
 *    c-objects
 *    functions
 *    equal?
 *    generic length, copy, reverse, fill!, append
 *    error handlers
 *    sundry leftovers
 *    the optimizers
 *    multiple-values, quasiquote
 *    eval
 *    multiprecision arithmetic
 *    *s7* environment
 *    initialization
 *    repl
 *
 * naming conventions: s7_* usually are C accessible (s7.h), g_* are scheme accessible (FFI),
 *   H_* are documentation strings, Q_* are procedure signatures,
 *   *_1 are ancillary functions, big_* refer to gmp,
 *   scheme "?" corresponds to C "is_", scheme "->" to C "_to_".
 *
 * ---------------- compile time switches ----------------
 */

#include "mus-config.h"

/*
 * Your config file goes here, or just replace that #include line with the defines you need.
 * The compile-time switches involve booleans, complex numbers, and multiprecision arithmetic.
 * Currently we assume we have setjmp.h (used by the error handlers).
 *
 * Complex number support which is problematic in C++, Solaris, and netBSD
 *   is on the HAVE_COMPLEX_NUMBERS switch. In OSX or Linux, if you're not using C++,
 *
 *   #define HAVE_COMPLEX_NUMBERS 1
 *   #define HAVE_COMPLEX_TRIG 1
 *
 *   In C++ I use:
 *
 *   #define HAVE_COMPLEX_NUMBERS 1
 *   #define HAVE_COMPLEX_TRIG 0
 *
 *   In windows, both are 0.
 *
 *   Some systems (FreeBSD) have complex.h, but some random subset of the trig funcs, so
 *   HAVE_COMPLEX_NUMBERS means we can find
 *      cimag creal cabs csqrt carg conj
 *   and HAVE_COMPLEX_TRIG means we have
 *      cacos cacosh casin casinh catan catanh ccos ccosh cexp clog cpow csin csinh ctan ctanh
 *
 * When HAVE_COMPLEX_NUMBERS is 0, the complex functions are stubs that simply return their
 *   argument -- this will be very confusing for the s7 user because, for example, (sqrt -2)
 *   will return something bogus (it will not signal an error).
 *
 * so the incoming (non-s7-specific) compile-time switches are
 *     HAVE_COMPLEX_NUMBERS, HAVE_COMPLEX_TRIG, SIZEOF_VOID_P
 * if SIZEOF_VOID_P is not defined, we look for __SIZEOF_POINTER__ instead
 *   the default is to assume that we're running on a 64-bit machine.
 *
 * To get multiprecision arithmetic, set WITH_GMP to 1.
 *   You'll also need libgmp, libmpfr, and libmpc (version 0.8.0 or later)
 *   In highly numerical contexts, the gmp version of s7 is about 50(!) times slower than the non-gmp version.
 *
 * and we use these predefined macros: __cplusplus, _MSC_VER, __GNUC__, __clang__, __ANDROID__
 *
 * if WITH_SYSTEM_EXTRAS is 1 (default is 1 unless _MSC_VER), various OS and file related functions are included.
 * in openBSD I think you need to include -ftrampolines in CFLAGS.
 * if you want this file to compile into a stand-alone interpreter, define WITH_MAIN
 *
 * -O3 produces segfaults, and is often slower than -O2 (at least according to callgrind)
 * -march=native seems to improve tree-vectorization which is important in Snd
 * -ffast-math makes a mess of NaNs, and does not appear to be faster
 * for timing tests, I use: -O2 -march=native -fomit-frame-pointer -funroll-loops
 * according to callgrind, clang is normally about 10% slower than gcc, and vectorization either doesn't work or is much worse than gcc's
 *   also g++ appears to be slightly slower than gcc
 */

#if (defined(__GNUC__) || defined(__clang__)) /* s7 uses PRId64 so (for example) g++ 4.4 is too old */
  #define WITH_GCC 1
#else
  #define WITH_GCC 0
#endif


/* ---------------- initial sizes ---------------- */

#ifndef INITIAL_HEAP_SIZE
  #define INITIAL_HEAP_SIZE 128000
#endif
/* the heap grows as needed, this is its initial size.
 * If the initial heap is small, s7 can run in about 2.5 Mbytes of memory. There are (many) cases where a bigger heap is faster.
 * The heap size must be a multiple of 32.  Each object takes 48 bytes.
 *
 * repl runs in    4Mb (18v) (64bit) if heap is 8192
 *                11Mb (25v)         if 128k heap
 *  snd (no gui)  15Mb (151v)
 *  snd (no gui)  17Mb (33v)         (no gsl, fftw, audio)
 *                21Mb (155v)        same but with ALSA!?
 *  snd (motif)   12Mb (285v)
 *  snd (gtk)     32Mb (515v!)
 */

#ifndef SYMBOL_TABLE_SIZE
  #define SYMBOL_TABLE_SIZE 32749
#endif
/* names are hashed into the symbol table (a vector) and collisions are chained as lists. */

#ifndef INITIAL_STACK_SIZE
  #define INITIAL_STACK_SIZE 512
#endif
/* the stack grows as needed, each frame takes 4 entries, this is its initial size.
 *   this needs to be big enough to handle the eval_c_string's at startup (ca 100)
 *   In s7test.scm, the maximum stack size is ca 440.  In snd-test.scm, it's ca 200.
 *   This number matters only because call/cc copies the stack, which requires filling
 *   the unused portion of the new stack, which requires memcpy of #<unspecified>'s.
 */

#ifndef INITIAL_PROTECTED_OBJECTS_SIZE
  #define INITIAL_PROTECTED_OBJECTS_SIZE 16
#endif
/* a vector of objects that are (semi-permanently) protected from the GC, grows as needed */

#ifndef GC_TEMPS_SIZE
  #define GC_TEMPS_SIZE 256
#endif
/* the number of recent objects that are temporarily gc-protected; 8 works for s7test and snd-test.
 *    For the FFI, this sets the lag between a call on s7_cons and the first moment when its result
 *    might be vulnerable to the GC.
 */


/* ---------------- scheme choices ---------------- */

#ifndef WITH_GMP
  #define WITH_GMP 0
  /* this includes multiprecision arithmetic for all numeric types and functions, using gmp, mpfr, and mpc
   * WITH_GMP adds the following functions: bignum and bignum?, and (*s7* 'bignum-precision)
   * using gmp with precision=128 is about 50 times slower than using C doubles and int64_ts.
   */
#endif

#define DEFAULT_BIGNUM_PRECISION 128

#ifndef WITH_PURE_S7
  #define WITH_PURE_S7 0
#endif
#if WITH_PURE_S7
  #define WITH_EXTRA_EXPONENT_MARKERS 0
  #define WITH_IMMUTABLE_UNQUOTE 1
  /* also omitted: *-ci* functions, char-ready?, cond-expand, multiple-values-bind|set!, call-with-values, defmacro(*)
   *   and a lot more (inexact/exact, integer-length,  etc) -- see s7.html.
   */
#endif

#ifndef WITH_EXTRA_EXPONENT_MARKERS
  #define WITH_EXTRA_EXPONENT_MARKERS 0
#endif
/* if 1, s7 recognizes "d", "f", "l", and "s" as exponent markers, in addition to "e" (also "D", "F", "L", "S") */

#ifndef WITH_SYSTEM_EXTRAS
  #define WITH_SYSTEM_EXTRAS (!_MSC_VER)
  /* this adds several functions that access file info, directories, times, etc
   *    this may be replaced by the cload business below
   */
#endif

#ifndef WITH_IMMUTABLE_UNQUOTE
  #define WITH_IMMUTABLE_UNQUOTE 0
  /* this removes the name "unquote" */
#endif

#ifndef WITH_C_LOADER
  #if WITH_GCC
    #define WITH_C_LOADER 1
  /* (load file.so [e]) looks for (e 'init_func) and if found, calls it
   *   as the shared object init function.  If WITH_SYSTEM_EXTRAS is 0, the caller
   *   needs to supply system and delete-file so that cload.scm works.
   */
  #else
    #define WITH_C_LOADER 0
  #endif
#endif

#ifndef WITH_HISTORY
  #define WITH_HISTORY 0
  /* this includes a circular buffer of previous evaluations for debugging, ((owlet) 'error-history) and (*s7* 'history-size) */
#endif

#ifndef DEFAULT_HISTORY_SIZE
  #define DEFAULT_HISTORY_SIZE 8
  /* this is the default length of the eval history buffer */
#endif
#if WITH_HISTORY
  #define MAX_HISTORY_SIZE 1048576
#endif

#define DEFAULT_PRINT_LENGTH 32 /* (*s7* 'print-length) */

#ifndef WITH_PROFILE
  #define WITH_PROFILE 0
  /* this includes profiling data collection accessible from scheme via the hash-table (*s7* 'profile-info) */
#endif

/* in case mus-config.h forgets these */
#ifdef _MSC_VER
  #ifndef HAVE_COMPLEX_NUMBERS
    #define HAVE_COMPLEX_NUMBERS 0
  #endif
  #ifndef HAVE_COMPLEX_TRIG
    #define HAVE_COMPLEX_TRIG 0
  #endif
#else
  #ifndef HAVE_COMPLEX_NUMBERS
    #define HAVE_COMPLEX_NUMBERS 1
  #endif
  #if __cplusplus
    #ifndef HAVE_COMPLEX_TRIG
      #define HAVE_COMPLEX_TRIG 0
    #endif
  #else
    #ifndef HAVE_COMPLEX_TRIG
      #define HAVE_COMPLEX_TRIG 1
    #endif
  #endif
#endif

/* -------------------------------------------------------------------------------- */

#ifndef WITH_MULTITHREAD_CHECKS
  #define WITH_MULTITHREAD_CHECKS 0
  /* debugging aid if using s7 in a multithreaded program -- this code courtesy of Kjetil Matheussen */
#endif

#ifndef S7_DEBUGGING
  #define S7_DEBUGGING 0
#endif

#define CYCLE_DEBUGGING S7_DEBUGGING

#undef DEBUGGING
#define DEBUGGING typo!

#ifndef OP_NAMES
  #define OP_NAMES 0
#endif

#ifndef _GNU_SOURCE
#define _GNU_SOURCE
/* for qsort_r, grumble... */
#endif

#ifndef _MSC_VER
  #include <unistd.h>
  #include <sys/param.h>
  #include <strings.h>
  #include <errno.h>
  #include <locale.h>
#else
  /* in Snd these are in mus-config.h */
  #ifndef MUS_CONFIG_H_LOADED
    #define snprintf _snprintf
    #if _MSC_VER > 1200
      #define _CRT_SECURE_NO_DEPRECATE 1
      #define _CRT_NONSTDC_NO_DEPRECATE 1
      #define _CRT_SECURE_CPP_OVERLOAD_STANDARD_NAMES 1
    #endif
  #endif
  #include <io.h>
  #pragma warning(disable: 4244) /* conversion might cause loss of data warning */
#endif

#ifndef WITH_VECTORIZE
#if (defined(__GNUC__) && __GNUC__ >= 5)
  #define WITH_VECTORIZE 1
#else
  #define WITH_VECTORIZE 0
#endif
#endif

#include <stdio.h>
#include <limits.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <time.h>
#include <stdarg.h>
#include <stddef.h>
#include <stdint.h>
#include <inttypes.h>
#include <setjmp.h>

#ifdef _MSC_VER
  #define MS_WINDOWS 1
#else
  #define MS_WINDOWS 0
#endif

#if (!MS_WINDOWS)
#include <pthread.h>
#endif

#if __cplusplus
  #include <cmath>
  #define s7_inline /* g++ 8.1.1 stupidity */
#else
  #include <math.h>
  #define s7_inline inline
#endif

#if HAVE_COMPLEX_NUMBERS
  #if __cplusplus
    #include <complex>
  #else
    #include <complex.h>
    #ifndef __SUNPRO_C
      #if defined(__sun) && defined(__SVR4)
        #undef _Complex_I
        #define _Complex_I 1.0fi
      #endif
    #endif
  #endif

#ifndef CMPLX
  #if (!(defined(__cplusplus))) && (__GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ >= 7)) && !defined(__INTEL_COMPILER)
    #define CMPLX(x, y) __builtin_complex ((double) (x), (double) (y))
  #else
    #define CMPLX(r, i) ((r) + ((i) * _Complex_I))
  #endif
#endif
#endif

#include "s7.h"

#ifndef M_PI
  #define M_PI 3.1415926535897932384626433832795029L
#endif

#ifndef INFINITY
  #define INFINITY (-log(0.0))
  /* 1.0 / 0.0 is also used, there is sometimes a function, infinity(), MSC apparently uses HUGE_VALF, gcc has __builtin_huge_val() */
#endif

#ifndef NAN
  #define NAN (INFINITY / INFINITY) /* gcc has __builtin_nan(str) */
#endif

#define BOLD_TEXT "\033[1m"
#define UNBOLD_TEXT "\033[22m"

#if ((!__NetBSD__) && ((_MSC_VER) || (!defined(__STC__)) || (defined(__STDC_VERSION__) && (__STDC_VERSION__ < 199901L))))
  #define __func__ __FUNCTION__
#endif

#define DISPLAY(Obj) string_value(s7_object_to_string(sc, Obj, false))
#define DISPLAY_80(Obj) string_value(object_to_truncated_string(sc, Obj, 80))

typedef intptr_t opcode_t;

#define PRINT_NAME_PADDING 16
#if (((defined(SIZEOF_VOID_P)) && (SIZEOF_VOID_P == 4)) || ((defined(__SIZEOF_POINTER__)) && (__SIZEOF_POINTER__ == 4)))
  #define PRINT_NAME_SIZE (20 - PRINT_NAME_PADDING - 2) /* pointless */
  #define POINTER_32 true
#else
  #define PRINT_NAME_SIZE (40 - PRINT_NAME_PADDING - 2)
  #define POINTER_32 false
#endif

#define WRITE_REAL_PRECISION 16
typedef long double long_double;

#define print_s7_int PRId64
#define print_int32  PRId32
#define print_pointer PRIdPTR

#define MAX_FLOAT_FORMAT_PRECISION 128

/* types */
#define T_FREE                 0
#define T_PAIR                 1
#define T_NIL                  2
#define T_UNDEFINED            3
#define T_UNSPECIFIED          4
#define T_EOF_OBJECT           5
#define T_BOOLEAN              6
#define T_CHARACTER            7
#define T_SYNTAX               8
#define T_SYMBOL               9

#define T_INTEGER             10
#define T_RATIO               11
#define T_REAL                12
#define T_COMPLEX             13

#define T_BIG_INTEGER         14 /* these four used only if WITH_GMP -- order matters */
#define T_BIG_RATIO           15
#define T_BIG_REAL            16
#define T_BIG_COMPLEX         17

#define T_STRING              18
#define T_C_OBJECT            19
#define T_VECTOR              20
#define T_INT_VECTOR          21
#define T_FLOAT_VECTOR        22
#define T_BYTE_VECTOR         23

#define T_CATCH               24
#define T_DYNAMIC_WIND        25
#define T_HASH_TABLE          26
#define T_LET                 27
#define T_ITERATOR            28
#define T_STACK               29
#define T_COUNTER             30
#define T_SLOT                31
#define T_C_POINTER           32
#define T_OUTPUT_PORT         33
#define T_INPUT_PORT          34
#define T_BAFFLE              35
#define T_RANDOM_STATE        36

#define T_CONTINUATION        37
#define T_GOTO                38
#define T_CLOSURE             39
#define T_CLOSURE_STAR        40
#define T_C_MACRO             41
#define T_MACRO               42
#define T_MACRO_STAR          43
#define T_BACRO               44
#define T_BACRO_STAR          45
#define T_C_FUNCTION_STAR     46
#define T_C_FUNCTION          47
#define T_C_ANY_ARGS_FUNCTION 48
#define T_C_OPT_ARGS_FUNCTION 49
#define T_C_RST_ARGS_FUNCTION 50

#define NUM_TYPES        51

/* T_STACK, T_SLOT, T_BAFFLE, T_DYNAMIC_WIND, and T_COUNTER are internal */

typedef struct block_t {
  union {
    void *data;
    s7_pointer d_ptr;
    s7_int *i_ptr;
  } dx;
  int32_t index;
  union {
    bool filler;
    int32_t len;
    uint32_t tag;
  } ln;
  s7_int size;
  union {
    struct block_t *next;
    char *documentation;
    s7_pointer ksym;
    s7_int nx_int;
    s7_int *ix_ptr;
    struct {
      uint32_t i1, i2;
    } ix;
  } nx;
  union {
    s7_pointer ex_ptr;
    void *ex_info;
    struct {
      uint32_t i3;
      int32_t i4;
    } jx;
  } ex;
} block_t;

#define NUM_BLOCK_LISTS 18
#define TOP_BLOCK_LIST 17
#define BLOCK_LIST 0

#define block_data(p)                    p->dx.data
#define block_index(p)                   p->index
#define block_set_index(p, Index)        p->index = Index
#define block_size(p)                    p->size
#define block_set_size(p, Size)          p->size = Size
#define block_next(p)                    p->nx.next
#define block_info(p)                    p->ex.ex_info

typedef block_t hash_entry_t;
#define hash_entry_key(p)                p->dx.d_ptr
#define hash_entry_value(p)              p->ex.ex_ptr
#define hash_entry_set_value(p, Val)     p->ex.ex_ptr = Val
#define hash_entry_next(p)               block_next(p)
#define hash_entry_raw_hash(p)           block_size(p)
#define hash_entry_set_raw_hash(p, Hash) block_set_size(p, Hash)

typedef block_t optfix_t;
#define optfix_expr(p)                   p->dx.d_ptr
#define optfix_op(p)                     block_size(p)
#define optfix_next(p)                   block_next(p)

typedef block_t vdims_t;
#define vdims_rank(p)                    p->size
#define vector_elements_should_be_freed(p) p->ln.filler
#define vdims_dims(p)                    p->dx.i_ptr
#define vdims_offsets(p)                 p->nx.ix_ptr
#define vdims_original(p)                p->ex.ex_ptr


typedef enum {TOKEN_EOF, TOKEN_LEFT_PAREN, TOKEN_RIGHT_PAREN, TOKEN_DOT, TOKEN_ATOM, TOKEN_QUOTE, TOKEN_DOUBLE_QUOTE, TOKEN_BACK_QUOTE,
	      TOKEN_COMMA, TOKEN_AT_MARK, TOKEN_SHARP_CONST, TOKEN_VECTOR, TOKEN_BYTE_VECTOR, TOKEN_INT_VECTOR, TOKEN_FLOAT_VECTOR} token_t;

typedef enum {FILE_PORT, STRING_PORT, FUNCTION_PORT} port_type_t;

typedef struct {
  bool needs_free, needs_unprotect, is_closed;
  port_type_t ptype;
  FILE *file;
  char *filename;
  block_t *filename_block;
  uint32_t line_number, file_number;
  s7_int gc_loc, filename_length;
  block_t *block;
  s7_pointer (*input_function)(s7_scheme *sc, s7_read_t read_choice, s7_pointer port);
  void (*output_function)(s7_scheme *sc, uint8_t c, s7_pointer port);
  /* a version of string ports using a pointer to the current location and a pointer to the end
   *   (rather than an integer for both, indexing from the base string) was not faster.
   */
  s7_pointer orig_str;                                                   /* GC protection for string port string */
  int32_t (*read_character)(s7_scheme *sc, s7_pointer port);             /* function to read a character, int32_t for EOF */
  void (*write_character)(s7_scheme *sc, uint8_t c, s7_pointer port);    /* function to write a character */
  void (*write_string)(s7_scheme *sc, const char *str, s7_int len, s7_pointer port); /* function to write a string of known length */
  token_t (*read_semicolon)(s7_scheme *sc, s7_pointer port);             /* internal skip-to-semicolon reader */
  int32_t (*read_white_space)(s7_scheme *sc, s7_pointer port);           /* internal skip white space reader */
  s7_pointer (*read_name)(s7_scheme *sc, s7_pointer pt);                 /* internal get-next-name reader */
  s7_pointer (*read_sharp)(s7_scheme *sc, s7_pointer pt);                /* internal get-next-sharp-constant reader */
  s7_pointer (*read_line)(s7_scheme *sc, s7_pointer pt, bool eol_case, bool copied);  /* function to read a string up to \n */
  void (*display)(s7_scheme *sc, const char *s, s7_pointer pt);
} port_t;


typedef struct opt_funcs {
  int32_t typ;
  void *func;
  struct opt_funcs *next;
} opt_funcs;

typedef struct {
  const char *name;
  int32_t name_length;
  uint32_t id;
  char *doc;
  block_t *block;
  opt_funcs *opt_data; /* vunion-functions (see below) not optlists */
  s7_pointer generic_ff, setter, signature, pars;
  s7_pointer (*chooser)(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops);
  /* arg_defaults|names call_args only T_C_FUNCTION_STAR -- call args for GC protection */
  union {
    s7_pointer *arg_defaults;
    s7_pointer bool_setter;
  } dam;
  union {
    s7_pointer *arg_names;
    s7_pointer c_sym;
  } sam;
  union {
    s7_pointer call_args;
    void (*marker)(s7_pointer p, s7_int len);
  } cam;
} c_proc_t;


typedef struct {
  s7_int type, outer_type;
  s7_pointer scheme_name;
  void (*free)(void *value);
  void (*mark)(void *val);
  bool (*equal)(void *val1, void *val2);       /* can this be wrapped? */
#if (!DISABLE_DEPRECATED)
  char *(*print)(s7_scheme *sc, void *value);
#endif
  s7_pointer (*ref)      (s7_scheme *sc, s7_pointer args);
  s7_pointer (*set)      (s7_scheme *sc, s7_pointer args);
  s7_pointer (*length)   (s7_scheme *sc, s7_pointer args);
  s7_pointer (*reverse)  (s7_scheme *sc, s7_pointer args);
  s7_pointer (*copy)     (s7_scheme *sc, s7_pointer args);
  s7_pointer (*fill)     (s7_scheme *sc, s7_pointer args);
  s7_pointer (*to_list)  (s7_scheme *sc, s7_pointer args);
  s7_pointer (*to_string)(s7_scheme *sc, s7_pointer args);
} c_object_t;


typedef s7_int (*hash_map_t)(s7_scheme *sc, s7_pointer table, s7_pointer key);          /* hash-table object->location mapper */
typedef hash_entry_t *(*hash_check_t)(s7_scheme *sc, s7_pointer table, s7_pointer key); /* hash-table object equality function */
static hash_map_t default_hash_map[NUM_TYPES];


/* -------------------------------- */
typedef s7_int (*s7_i_7pi_t)(s7_scheme *sc, s7_pointer p, s7_int i1);
typedef s7_int (*s7_i_7pii_t)(s7_scheme *sc, s7_pointer p, s7_int i1, s7_int i2);
typedef s7_int (*s7_i_iii_t)(s7_int i1, s7_int i2, s7_int i3);
typedef s7_int (*s7_i_7i_t)(s7_scheme *sc, s7_int i1);
typedef s7_int (*s7_i_7ii_t)(s7_scheme *sc, s7_int i1, s7_int i2);
typedef bool (*s7_b_pp_t)(s7_pointer p1, s7_pointer p2);
typedef bool (*s7_b_7pp_t)(s7_scheme *sc, s7_pointer p1, s7_pointer p2);
typedef bool (*s7_b_7p_t)(s7_scheme *sc, s7_pointer p1);
typedef bool (*s7_b_pi_t)(s7_scheme *sc, s7_pointer p1, s7_int i2);
typedef bool (*s7_b_d_t)(s7_double p1);
typedef bool (*s7_b_i_t)(s7_int p1);
typedef bool (*s7_b_ii_t)(s7_int p1, s7_int p2);
typedef bool (*s7_b_dd_t)(s7_double p1, s7_double p2);
typedef s7_pointer (*s7_p_p_t)(s7_scheme *sc, s7_pointer p);
typedef s7_pointer (*s7_p_t)(s7_scheme *sc);
typedef s7_pointer (*s7_p_pp_t)(s7_scheme *sc, s7_pointer p1, s7_pointer p2);
typedef s7_pointer (*s7_p_ppi_t)(s7_scheme *sc, s7_pointer p1, s7_pointer p2, s7_int i1);
typedef s7_pointer (*s7_p_ppp_t)(s7_scheme *sc, s7_pointer p1, s7_pointer p2, s7_pointer p3);
typedef s7_pointer (*s7_p_pi_t)(s7_scheme *sc, s7_pointer p1, s7_int i1);
typedef s7_pointer (*s7_p_pip_t)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_pointer p2);
typedef s7_pointer (*s7_p_ii_t)(s7_scheme *sc, s7_int i1, s7_int i2);
typedef s7_pointer (*s7_p_dd_t)(s7_scheme *sc, s7_double x1, s7_double x2);
typedef s7_double (*s7_d_7d_t)(s7_scheme *sc, s7_double p1);
typedef s7_double (*s7_d_7dd_t)(s7_scheme *sc, s7_double p1, s7_double p2);
typedef s7_double (*s7_d_7p_t)(s7_scheme *sc, s7_pointer p1);

typedef union {
  s7_int i;
  s7_double x;
  s7_pointer p;
  void *obj;
  s7_function cf;
  s7_double (*d_f)(void);
  s7_double (*d_d_f)(s7_double x);
  s7_double (*d_7d_f)(s7_scheme *sc, s7_double x);
  s7_double (*d_dd_f)(s7_double x1, s7_double x2);
  s7_double (*d_7dd_f)(s7_scheme *sc, s7_double x1, s7_double x2);
  s7_double (*d_ddd_f)(s7_double x1, s7_double x2, s7_double x3);
  s7_double (*d_dddd_f)(s7_double x1, s7_double x2, s7_double x3, s7_double x4);
  s7_double (*d_v_f)(void *obj);
  s7_double (*d_vd_f)(void *obj, s7_double fm);
  s7_double (*d_vdd_f)(void *obj, s7_double x1, s7_double x2);
  s7_double (*d_vid_f)(void *obj, s7_int i, s7_double fm);
  s7_double (*d_id_f)(s7_int i, s7_double fm);
  s7_double (*d_7pi_f)(s7_scheme *sc, s7_pointer obj, s7_int i1);
  s7_double (*d_ip_f)(s7_int i1, s7_pointer p);
  s7_double (*d_pd_f)(s7_pointer obj, s7_double x);
  s7_double (*d_7pid_f)(s7_scheme *sc, s7_pointer obj, s7_int i1, s7_double x);
  s7_double (*d_p_f)(s7_pointer p);
  s7_double (*d_7p_f)(s7_scheme *sc, s7_pointer p);
  s7_int (*i_7d_f)(s7_scheme *sc, s7_double i1);
  s7_int (*i_7p_f)(s7_scheme *sc, s7_pointer i1);
  s7_int (*i_i_f)(s7_int i1);
  s7_int (*i_7i_f)(s7_scheme *sc, s7_int i1);
  s7_int (*i_ii_f)(s7_int i1, s7_int i2);
  s7_int (*i_7ii_f)(s7_scheme *sc, s7_int i1, s7_int i2);
  s7_int (*i_iii_f)(s7_int i1, s7_int i2, s7_int i3);
  s7_int (*i_7pi_f)(s7_scheme *sc, s7_pointer p, s7_int i1);
  s7_int (*i_7pii_f)(s7_scheme *sc, s7_pointer p, s7_int i1, s7_int i2);
  bool (*b_i_f)(s7_int p);
  bool (*b_d_f)(s7_double p);
  bool (*b_p_f)(s7_pointer p);
  bool (*b_pp_f)(s7_pointer p1, s7_pointer p2);
  bool (*b_7pp_f)(s7_scheme *sc, s7_pointer p1, s7_pointer p2);
  bool (*b_7p_f)(s7_scheme *sc, s7_pointer p1);
  bool (*b_pi_f)(s7_scheme *sc, s7_pointer p1, s7_int i2);
  bool (*b_ii_f)(s7_int i1, s7_int i2);
  bool (*b_dd_f)(s7_double x1, s7_double x2);
  s7_pointer (*p_f)(s7_scheme *sc);
  s7_pointer (*p_p_f)(s7_scheme *sc, s7_pointer p);
  s7_pointer (*p_pp_f)(s7_scheme *sc, s7_pointer p1, s7_pointer p2);
  s7_pointer (*p_ppp_f)(s7_scheme *sc, s7_pointer p, s7_pointer p2, s7_pointer p3);
  s7_pointer (*p_pi_f)(s7_scheme *sc, s7_pointer p1, s7_int i1);
  s7_pointer (*p_ppi_f)(s7_scheme *sc, s7_pointer p1, s7_pointer p2, s7_int i1);
  s7_pointer (*p_pip_f)(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_pointer p2);
  s7_pointer (*p_ii_f)(s7_scheme *sc, s7_int x1, s7_int x2);
  s7_pointer (*p_dd_f)(s7_scheme *sc, s7_double x1, s7_double x2);
  s7_double (*fd)(void *o);
  s7_int (*fi)(void *o);
  bool (*fb)(void *o);
  s7_pointer (*fp)(void *o);
} vunion;

#define NUM_VUNIONS 16
typedef struct {
  union {
    int64_t vtype;
    uint8_t vt[8];
  } typ;
  s7_scheme *sc;
  vunion v[NUM_VUNIONS];
#if S7_DEBUGGING
  s7_pointer vexpr;
  const char *func;
  int32_t line;
#endif
} opt_info;


/* -------------------------------- cell structure -------------------------------- */

typedef struct s7_cell {
  union {
    uint64_t flag;                /* type info */
    uint8_t type_field;
    uint16_t sflag;
    struct {
      uint32_t low_flag;
      uint16_t opt_choice;
      uint16_t high_flag;
    } opts;
  } tf;
  union {

    union {                       /* integers, floats */
      s7_int integer_value;
      s7_double real_value;

      struct {
	uint8_t padding[PRINT_NAME_PADDING];
	uint8_t name[PRINT_NAME_SIZE + 2];
      } pval;

      struct {                    /* ratios */
	s7_int numerator;
	s7_int denominator;
      } fraction_value;

      struct {                    /* complex numbers */
	s7_double rl;
	s7_double im;
      } complex_value;

#if WITH_GMP
      mpz_t big_integer;          /* bignums */
      mpq_t big_ratio;
      mpfr_t big_real;
      mpc_t big_complex;
      /* using free_lists here was not faster, and avoiding the extra init/clear too tricky.  These make up
       *   no more than ca. 5% of the gmp computation -- it is totally dominated by stuff like __gmpz_mul,
       *   so I can't see much point in optimizing the background noise.  In a very numerical context,
       *   gmp slows us down by a factor of 50.
       */
#endif
    } number;

    struct {                       /* ports */
      port_t *port;
      uint8_t *data;
      s7_int size, point;
      block_t *block;
    } prt;

    struct{                        /* characters */
      uint8_t c, up_c;
      int32_t length;
      bool alpha_c, digit_c, space_c, upper_c, lower_c;
      char c_name[12];
    } chr;

    struct {                       /* c-pointers */
      void *c_pointer;
      s7_pointer c_type, info, weak1, weak2;
    } cptr;

    s7_int baffle_key;             /* baffles */

    struct {                       /* vectors */
      s7_int length;
      union {
	s7_pointer *objects;
	s7_int *ints;
	s7_double *floats;
	uint8_t *bytes;
      } elements;
      block_t *block;
      s7_pointer (*vget)(s7_scheme *sc, s7_pointer vec, s7_int loc);
      union {
	s7_pointer (*vset)(s7_scheme *sc, s7_pointer vec, s7_int loc, s7_pointer val);
	s7_pointer fset;
      }setv;
    } vector;

    struct {                        /* stacks (internal) struct must match vector above for length/objects */
      s7_int length;
      s7_pointer *objects;
      block_t *block;
      int64_t top;
    } stk;

    struct {                        /* hash-tables */
      s7_int mask;
      hash_entry_t **elements;
      hash_check_t hash_func;
      hash_map_t *loc;
      block_t *block;
    } hasher;

    struct {                        /* iterators */
      s7_pointer obj, cur;
      union {
	s7_int loc;
	s7_pointer lcur;
      } lc;
      union {
	s7_int len;
	s7_pointer slow;
	hash_entry_t *hcur;
      } lw;
      s7_pointer (*next)(s7_scheme *sc, s7_pointer iterator);
    } iter;

    struct {
      c_proc_t *c_proc;              /* C functions, macros */
      s7_function ff;
      s7_int required_args, optional_args, all_args;
    } fnc;

    struct {                         /* pairs */
      s7_pointer car, cdr, opt1, opt2, opt3;
    } cons;

    struct {                         /* pairs */
      s7_pointer car, cdr, opt1, opt2;
      union {
	uint8_t opt_type;
	int32_t ctr;
      } ce;
    } cons_ext;

    struct {                         /* special purpose pairs (symbol-table etc) */
      s7_pointer unused_car, unused_cdr;
      uint64_t hash;
      const char *fstr;
      uint32_t file, line;          /* line|file=pair_line|file, line also used in symbol_table (raw_len) */
    } sym_cons;

    struct {                        /* scheme functions */
      s7_pointer args, body, env, setter; /* args can be a symbol, as well as a list */
      int32_t arity;
    } func;

    struct {                        /* strings */
      s7_int length;
      char *svalue;
      uint64_t hash;                /* string hash-index */
      block_t *block;
      block_t *gensym_block;
    } string;

    struct {                       /* symbols */
      s7_pointer name, global_slot, local_slot;
      int64_t id;                  /* which let last bound the symbol -- for faster symbol lookup */
      uint32_t ctr;                /* how many times has symbol been set */
      uint32_t tag;                /* symbol as member of a set (tree-set-memq etc), high 32 bits are in symbol_info (the string block) */
    } sym;

    struct {                       /* syntax */
      s7_pointer symbol;
      opcode_t op;
      int32_t min_args, max_args;
      const char *documentation;
    } syn;

    struct {                       /* slots (bindings) */
      s7_pointer sym, val, nxt, pending_value, expr;
    } slt;

    struct {                       /* environments (frames, lets) */
      s7_pointer slots, nxt;
      int64_t id;                  /* id of rootlet is -1 */
      union {
	struct {
	  s7_pointer function;     /* __func__ (code) if this is a funclet */
	  uint32_t line, file;     /* __func__ location if it is known */
	} efnc;
	struct {
	  s7_pointer dox1, dox2;   /* do loop variables */
	} dox;
	struct {                   /* (catch #t ...) opts */
	  uint64_t op_stack_loc, goto_loc;
	} ctall;
      } edat;
    } envr;

    struct {                        /* special stuff like #<unspecified> */
      s7_pointer car, cdr;          /* unique_car|cdr, for sc->nil these are sc->unspecified for faster assoc etc */
      int64_t unused_let_id;        /* let_id(sc->nil) is -1, so this needs to align with envr.id above */
      /* these two fields are for some special case objects like #<unspecified> */
      union {
	const char *name;
	char *unknown_name;
      } nm;
      s7_int len;
    } unq;

    struct {                        /* counter (internal) */
      s7_pointer result, list, env, slots; /* env = counter_let (curlet after map/for-each frame created) */
      uint64_t cap;                 /* sc->capture_let_counter for frame reuse */
    } ctr;

    struct {                        /* random-state */
#if WITH_GMP
      gmp_randstate_t state;
#else
      uint64_t seed, carry;
#endif
    } rng;

    struct {                        /* additional object types (C) */
      s7_int type;
      void *value;                  /*  the value the caller associates with the c_object */
      s7_pointer e;                 /*   the method list, if any (openlet) */
      void (*mark)(void *val);
    } c_obj;

    struct {                        /* continuations */
      block_t *block;
      s7_pointer stack;
      s7_pointer *stack_start, *stack_end, *op_stack;
    } cwcc;

    struct {                        /* call-with-exit */
      uint64_t goto_loc, op_stack_loc;
      bool active;
    } rexit;

    struct {                        /* catch */
      uint64_t goto_loc, op_stack_loc;
      s7_pointer tag;
      s7_pointer handler;
    } rcatch; /* C++ reserves "catch" I guess */

    struct {                        /* dynamic-wind */
      s7_pointer in, out, body;
      uint32_t state;
    } winder;
  } object;

#if WITH_PROFILE
  int32_t file_and_line;
#endif
#if S7_DEBUGGING
  int32_t current_alloc_line, previous_alloc_line, uses, explicit_free_line;
  int64_t current_alloc_type, previous_alloc_type, debugger_bits;
  const char *current_alloc_func, *previous_alloc_func;
#endif

} s7_cell;


typedef struct s7_big_cell {
  s7_cell cell;
  int64_t big_hloc;
} s7_big_cell;
typedef struct s7_big_cell *s7_big_pointer;

typedef struct heap_block_t {
  intptr_t start, end;
  int64_t offset;
  struct heap_block_t *next;
} heap_block_t;


typedef struct {
  s7_pointer *objs;
  int32_t size, top, ref, size2;
  bool has_hits;
  int32_t *refs;
  s7_pointer cycle_port, init_port;
  s7_int cycle_loc, init_loc;
  bool *defined;
} shared_info;

typedef struct {
  s7_int loc, curly_len, ctr;
  char *curly_str;
  s7_pointer args, orig_str, curly_arg;
  s7_pointer port, strport;
} format_data;

typedef struct gc_obj {
  s7_pointer p;
  struct gc_obj *nxt;
} gc_obj;

typedef struct {
  s7_pointer *list;
  s7_int size, loc;
} gc_list;


struct s7_scheme {
  s7_pointer code;
  s7_pointer envir;                   /* curlet, layout of first 4 entries should match stack frame layout */
  s7_pointer args;                    /* arguments of current function */
  opcode_t cur_op;
  s7_pointer value;
  s7_pointer cur_code;
  token_t tok;

  s7_pointer stack;                   /* stack is a vector */
  uint32_t stack_size;
  s7_pointer *stack_start, *stack_end, *stack_resize_trigger;

  s7_pointer *op_stack, *op_stack_now, *op_stack_end;
  uint32_t op_stack_size, max_stack_size;

  s7_cell **heap, **free_heap, **free_heap_top, **free_heap_trigger, **previous_free_heap_top;
  int64_t heap_size, gc_freed, max_heap_size;

#if WITH_HISTORY
  s7_pointer eval_history1, eval_history2, error_history;
  bool using_history1;
#endif
  bool history_enabled;

#if WITH_MULTITHREAD_CHECKS
  int32_t lock_count;
  pthread_mutex_t lock;
#endif

  gc_obj *permanent_objects;
  s7_pointer protected_objects, protected_setters, protected_setter_symbols;  /* vectors of gc-protected objects */
  s7_int *gpofl;
  s7_int protected_objects_size, protected_setters_size, protected_setters_loc;
  s7_int gpofl_loc;

  s7_pointer nil;                     /* empty list */
  s7_pointer T;                       /* #t */
  s7_pointer F;                       /* #f */
  s7_pointer undefined;               /* #<undefined> */
  s7_pointer unspecified;             /* #<unspecified> */
  s7_pointer no_value;                /* the (values) value */
  s7_pointer gc_nil;                  /* a marker for an unoccupied slot in sc->protected_objects (and other similar stuff) */

  s7_pointer symbol_table;            /* symbol table */
  s7_pointer rootlet, shadow_rootlet; /* rootlet */
  s7_int rootlet_entries;
  s7_pointer unlet;                   /* original bindings of predefined functions */

  s7_pointer input_port;              /* current-input-port */
  s7_pointer input_port_stack;        /*   input port stack (load and read internally) */
  s7_pointer output_port;             /* current-output-port */
  s7_pointer error_port;              /* current-error-port */
  s7_pointer owlet;                   /* owlet */
  s7_pointer error_type, error_data, error_code, error_line, error_file; /* owlet slots */
  s7_pointer standard_input, standard_output, standard_error;

  s7_pointer sharp_readers;           /* the binding pair for the global *#readers* list */
  s7_pointer load_hook;               /* *load-hook* hook object */
  s7_pointer unbound_variable_hook;   /* *unbound-variable-hook* hook object */
  s7_pointer missing_close_paren_hook, rootlet_redefinition_hook;
  s7_pointer error_hook, read_error_hook; /* *error-hook* hook object, and *read-error-hook* */
  bool gc_off;                        /* gc_off: if true, the GC won't run */
  uint32_t gc_stats, gensym_counter, f_class, add_class, multiply_class, subtract_class, equal_class;
  int32_t format_column;
  uint64_t capture_let_counter;
  bool short_print, is_autoloading, in_with_let, object_out_locked;
  int64_t let_number;
  s7_double default_rationalize_error, morally_equal_float_epsilon, hash_table_float_epsilon;
  s7_int default_hash_table_length, initial_string_port_length, print_length, objstr_max_len, history_size, true_history_size;
  s7_int max_vector_length, max_string_length, max_list_length, max_vector_dimensions, max_format_length;
  s7_pointer stacktrace_defaults;
  int32_t float_format_precision;
  vdims_t *wrap_only;

  char *typnam;
  int32_t typnam_len, print_width;
  s7_pointer *singletons;
  block_t *unentry;                   /* hash-table lookup failure indicator */

  #define INITIAL_FILE_NAMES_SIZE 8
  s7_pointer *file_names;
  int32_t file_names_size, file_names_top;

  #define INITIAL_STRBUF_SIZE 1024
  s7_int strbuf_size;
  char *strbuf;

  char *read_line_buf;
  s7_int read_line_buf_size;

  s7_pointer u, v, w, x, y, z;         /* evaluator local vars */
  s7_pointer temp1, temp2, temp3, temp4, temp6, temp7, temp8, temp9, temp10, temp11;
  s7_pointer temp_cell, temp_cell_1, temp_cell_2, u1_1;
  s7_pointer t1_1, t2_1, t2_2, t3_1, t3_2, t3_3, z2_1, z2_2;
  s7_pointer a1_1, a2_1, a2_2, a3_1, a3_2, a3_3, a4_1, a4_2, a4_3, a4_4;
  #define T_TEMPS_SIZE 32
  s7_pointer t_temps[T_TEMPS_SIZE];    /* more eval temps */
  int32_t t_temp_ctr;

  jmp_buf goto_start;
  bool longjmp_ok;
  int32_t setjmp_loc;

  void (*begin_hook)(s7_scheme *sc, bool *val);
  opcode_t begin_op;

  s7_int current_line, s7_call_line, safety;
  const char *current_file, *s7_call_file, *s7_call_name;

  shared_info *circle_info;
  format_data **fdats;
  int32_t num_fdats, last_error_line;
  s7_pointer elist_1, elist_2, elist_3, elist_4, elist_5, plist_1, plist_2, plist_2_2, plist_3, qlist_2, clist_1;
  gc_list *strings, *vectors, *input_ports, *output_ports, *continuations, *c_objects, *hash_tables, *gensyms, *unknowns, *lambdas, *multivectors, *optlists, *weak_refs;
  s7_pointer *setters;
  s7_int setters_size, setters_loc;
  s7_pointer *tree_pointers;
  int32_t tree_pointers_size, tree_pointers_top, permanent_cells, string_wrapper_pos, num_to_str_size;
  s7_pointer format_ports;
  uint32_t alloc_pointer_k, alloc_function_k, alloc_symbol_k;
  s7_cell *alloc_pointer_cells;
  c_proc_t *alloc_function_cells;
  uint32_t alloc_big_pointer_k;
  s7_big_cell *alloc_big_pointer_cells;
  s7_pointer *string_wrappers;
  uint8_t *alloc_symbol_cells;
  char *num_to_str;

  block_t *block_lists[NUM_BLOCK_LISTS];
  size_t alloc_string_k;
  char *alloc_string_cells;

  c_object_t **c_object_types;
  int32_t c_object_types_size, num_c_object_types;
  s7_pointer type_to_typers[NUM_TYPES];

  uint32_t syms_tag, syms_tag2;
  int32_t bignum_precision;
  s7_int baffle_ctr;
  s7_pointer default_rng;

  s7_pointer sort_body, sort_begin, sort_v1, sort_v2;
  opcode_t sort_op;
  s7_int sort_body_len;
  s7_b_7pp_t sort_f;

  #define INT_TO_STR_SIZE 32
  char int_to_str1[INT_TO_STR_SIZE], int_to_str2[INT_TO_STR_SIZE], int_to_str3[INT_TO_STR_SIZE], int_to_str4[INT_TO_STR_SIZE], int_to_str5[INT_TO_STR_SIZE];

  s7_pointer abs_symbol, acos_symbol, acosh_symbol, add_symbol, angle_symbol, append_symbol, apply_symbol, apply_values_symbol, arity_symbol,
             ash_symbol, asin_symbol, asinh_symbol, assoc_symbol, assq_symbol, assv_symbol, atan_symbol, atanh_symbol,
             autoload_symbol, autoloader_symbol,
             byte_vector_symbol, byte_vector_ref_symbol, byte_vector_set_symbol, byte_vector_to_string_symbol,
             c_pointer_symbol, c_pointer_info_symbol, c_pointer_to_list_symbol, c_pointer_type_symbol, c_pointer_weak1_symbol, c_pointer_weak2_symbol,
             caaaar_symbol, caaadr_symbol, caaar_symbol, caadar_symbol, caaddr_symbol, caadr_symbol,
             caar_symbol, cadaar_symbol, cadadr_symbol, cadar_symbol, caddar_symbol, cadddr_symbol, caddr_symbol, cadr_symbol,
             call_cc_symbol, call_with_current_continuation_symbol, call_with_exit_symbol, call_with_input_file_symbol,
             call_with_input_string_symbol, call_with_output_file_symbol, call_with_output_string_symbol, car_symbol,
             catch_symbol, cdaaar_symbol, cdaadr_symbol, cdaar_symbol, cdadar_symbol, cdaddr_symbol, cdadr_symbol, cdar_symbol,
             cddaar_symbol, cddadr_symbol, cddar_symbol, cdddar_symbol, cddddr_symbol, cdddr_symbol, cddr_symbol, cdr_symbol,
             ceiling_symbol, char_downcase_symbol, char_eq_symbol, char_geq_symbol, char_gt_symbol, char_leq_symbol, char_lt_symbol,
             char_position_symbol, char_to_integer_symbol, char_upcase_symbol, cload_directory_symbol, close_input_port_symbol,
             close_output_port_symbol, complex_symbol, cons_symbol, copy_symbol, cos_symbol, cosh_symbol, coverlet_symbol,
             curlet_symbol, current_error_port_symbol, current_input_port_symbol, current_output_port_symbol, cutlet_symbol, cyclic_sequences_symbol,
             denominator_symbol, dilambda_symbol, display_symbol, divide_symbol, documentation_symbol, dynamic_wind_symbol,
             eq_symbol, error_symbol, eval_string_symbol, eval_symbol, exact_to_inexact_symbol, exit_symbol, exp_symbol, expt_symbol,
             features_symbol, fill_symbol, float_vector_ref_symbol, float_vector_set_symbol, float_vector_symbol, floor_symbol,
             flush_output_port_symbol, for_each_symbol, format_symbol, funclet_symbol,
             gc_symbol, gcd_symbol, gensym_symbol, geq_symbol, get_output_string_symbol, gt_symbol,
             hash_table_entries_symbol, hash_table_ref_symbol, hash_table_set_symbol, hash_table_star_symbol, hash_table_symbol, help_symbol,
             imag_part_symbol, immutable_symbol, inexact_to_exact_symbol, inlet_symbol, int_vector_ref_symbol, int_vector_set_symbol, int_vector_symbol,
             integer_decode_float_symbol, integer_to_char_symbol, is_aritable_symbol, is_boolean_symbol, is_byte_symbol, is_byte_vector_symbol,
             is_c_object_symbol, c_object_type_symbol, is_c_pointer_symbol, is_char_alphabetic_symbol, is_char_lower_case_symbol, is_char_numeric_symbol,
             is_char_symbol, is_char_upper_case_symbol, is_char_whitespace_symbol, is_complex_symbol, is_constant_symbol,
             is_continuation_symbol, is_defined_symbol, is_dilambda_symbol, is_eof_object_symbol, is_eq_symbol, is_equal_symbol,
             is_eqv_symbol, is_even_symbol, is_exact_symbol, is_float_vector_symbol, is_gensym_symbol, is_hash_table_symbol, is_immutable_symbol,
             is_inexact_symbol, is_infinite_symbol, is_input_port_symbol, is_int_vector_symbol, is_integer_symbol, is_iterator_symbol,
             is_keyword_symbol, is_let_symbol, is_list_symbol, is_macro_symbol, is_morally_equal_symbol, is_nan_symbol, is_negative_symbol,
             is_null_symbol, is_number_symbol, is_odd_symbol, is_openlet_symbol, is_output_port_symbol, is_pair_symbol,
             is_port_closed_symbol, is_positive_symbol, is_procedure_symbol, is_proper_list_symbol, is_provided_symbol,
             is_random_state_symbol, is_rational_symbol, is_real_symbol, is_sequence_symbol, is_string_symbol, is_subvector_symbol,
             is_symbol_symbol, is_syntax_symbol, is_vector_symbol, is_weak_hash_table_symbol, is_zero_symbol,
             iterate_symbol, iterator_is_at_end_symbol, iterator_sequence_symbol,
             is_float_symbol, is_integer_or_real_at_end_symbol, is_integer_or_any_at_end_symbol, is_unspecified_symbol, is_undefined_symbol,
             keyword_to_symbol_symbol,
             lcm_symbol, length_symbol, leq_symbol, let_ref_fallback_symbol, let_ref_symbol, let_set_fallback_symbol,
             let_set_symbol, let_temporarily_symbol, list_ref_symbol, list_set_symbol, list_symbol, list_tail_symbol, list_values_symbol,
             load_path_symbol, load_symbol, log_symbol, logand_symbol, logbit_symbol, logior_symbol, lognot_symbol, logxor_symbol, lt_symbol,
             magnitude_symbol, make_byte_vector_symbol, make_float_vector_symbol, make_hash_table_symbol, make_weak_hash_table_symbol,
             make_int_vector_symbol, make_iterator_symbol, string_to_keyword_symbol, make_list_symbol, make_string_symbol,
             make_vector_symbol, map_symbol, max_symbol, member_symbol, memq_symbol, memv_symbol, min_symbol, modulo_symbol, multiply_symbol,
             newline_symbol, not_symbol, number_to_string_symbol, numerator_symbol,
             object_to_string_symbol, object_to_let_symbol, open_input_file_symbol, open_input_string_symbol, open_output_file_symbol,
             open_output_string_symbol, openlet_symbol, outlet_symbol, owlet_symbol,
             pair_filename_symbol, pair_line_number_symbol, peek_char_symbol, pi_symbol, port_filename_symbol, port_line_number_symbol,
             procedure_source_symbol, provide_symbol,
             quotient_symbol,
             random_state_symbol, random_state_to_list_symbol, random_symbol, rationalize_symbol, read_byte_symbol,
             read_char_symbol, read_line_symbol, read_string_symbol, read_symbol, real_part_symbol, remainder_symbol,
             require_symbol, reverse_symbol, reverseb_symbol, rootlet_symbol, round_symbol,
             setter_symbol, set_car_symbol, set_cdr_symbol,
             set_current_error_port_symbol, set_current_input_port_symbol, set_current_output_port_symbol,
             signature_symbol, sin_symbol, sinh_symbol, sort_symbol, sqrt_symbol,
             stacktrace_symbol, string_append_symbol, string_downcase_symbol, string_eq_symbol, string_fill_symbol,
             string_geq_symbol, string_gt_symbol, string_leq_symbol, string_lt_symbol, string_position_symbol, string_ref_symbol,
             string_set_symbol, string_symbol, string_to_number_symbol, string_to_symbol_symbol, string_upcase_symbol,
             sublet_symbol, substring_symbol, subtract_symbol, subvector_symbol, subvector_position_symbol, subvector_vector_symbol,
             symbol_symbol, symbol_to_dynamic_value_symbol,
             symbol_to_keyword_symbol, symbol_to_string_symbol, symbol_to_value_symbol, s7_version_symbol,
             tan_symbol, tanh_symbol, throw_symbol, string_to_byte_vector_symbol,
             tree_count_symbol, tree_leaves_symbol, tree_memq_symbol, tree_set_memq_symbol, tree_is_cyclic_symbol, truncate_symbol, type_of_symbol,
             unlet_symbol,
             values_symbol, varlet_symbol, vector_append_symbol, vector_dimensions_symbol, vector_fill_symbol, vector_ref_symbol,
             vector_set_symbol, vector_symbol,
             with_input_from_file_symbol, with_input_from_string_symbol, with_output_to_file_symbol, with_output_to_string_symbol,
             write_byte_symbol, write_char_symbol, write_string_symbol, write_symbol,
             local_documentation_symbol, local_signature_symbol, local_setter_symbol, local_iterator_symbol;
#if (!WITH_PURE_S7)
  s7_pointer is_char_ready_symbol, char_ci_leq_symbol, char_ci_lt_symbol, char_ci_eq_symbol, char_ci_geq_symbol, char_ci_gt_symbol,
             let_to_list_symbol, integer_length_symbol, string_ci_leq_symbol, string_ci_lt_symbol, string_ci_eq_symbol,
             string_ci_geq_symbol, string_ci_gt_symbol, string_to_list_symbol, vector_to_list_symbol, string_length_symbol,
             string_copy_symbol, list_to_string_symbol, list_to_vector_symbol, vector_length_symbol, make_polar_symbol,
             make_rectangular_symbol;
#endif

  /* *s7* fields */
  s7_pointer stack_top_symbol, heap_size_symbol, gc_freed_symbol, gc_protected_objects_symbol,
             free_heap_size_symbol, file_names_symbol, symbol_table_symbol, cpu_time_symbol, float_format_precision_symbol, max_heap_size_symbol,
             stack_size_symbol, rootlet_size_symbol, c_types_symbol, safety_symbol, max_stack_size_symbol, gc_stats_symbol, autoloading_symbol,
             catches_symbol, stack_symbol, default_rationalize_error_symbol, max_string_length_symbol, default_random_state_symbol, history_enabled_symbol,
             max_list_length_symbol, max_vector_length_symbol, max_vector_dimensions_symbol, default_hash_table_length_symbol, profile_info_symbol,
             hash_table_float_epsilon_symbol, morally_equal_float_epsilon_symbol, initial_string_port_length_symbol, memory_usage_symbol, max_format_length_symbol,
             undefined_identifier_warnings_symbol, print_length_symbol, bignum_precision_symbol, stacktrace_defaults_symbol, history_symbol, history_size_symbol;

  /* syntax symbols et al */
  s7_pointer else_symbol, lambda_symbol, lambda_star_symbol, let_symbol, quote_symbol, unquote_symbol, macroexpand_symbol,
             define_expansion_symbol, baffle_symbol, with_let_symbol, if_symbol, autoload_error_symbol,
             when_symbol, unless_symbol, begin_symbol, cond_symbol, case_symbol, and_symbol, or_symbol, do_symbol,
             define_symbol, define_star_symbol, define_constant_symbol, with_baffle_symbol, define_macro_symbol,
             define_macro_star_symbol, define_bacro_symbol, define_bacro_star_symbol, letrec_symbol, letrec_star_symbol,
             let_star_symbol, key_rest_symbol, key_allow_other_keys_symbol, key_readable_symbol, value_symbol, type_symbol,
             baffled_symbol, __func___symbol, set_symbol, body_symbol, class_name_symbol, feed_to_symbol, format_error_symbol,
             wrong_number_of_args_symbol, read_error_symbol, string_read_error_symbol, syntax_error_symbol, division_by_zero_symbol,
             no_catch_symbol, io_error_symbol, invalid_escape_function_symbol, wrong_type_arg_symbol, out_of_range_symbol,
             missing_method_symbol, unbound_variable_symbol, key_if_symbol;
  s7_pointer dox_slot_symbol;

  s7_pointer string_signature, vector_signature, float_vector_signature, int_vector_signature, byte_vector_signature,
             c_object_signature, let_signature, hash_table_signature, pair_signature;
  s7_pointer pcl_bc, pcl_bs, pcl_bt, pcl_c, pcl_e, pcl_f, pcl_i, pcl_n, pcl_r, pcl_s, pcl_v, pl_bc, pl_bn, pl_bt, pl_p, pl_sf, pl_tl;

  /* optimizer s7_functions */
  s7_pointer add_2, add_1s, add_s1, add_cs1, add_si, add_sf, add_fs, add_f_sf, subtract_1, subtract_2, subtract_s1,
           subtract_cs1, subtract_csn, subtract_sf, subtract_2f, subtract_fs,  simple_char_eq, char_equal_s_ic,
           char_equal_2, char_greater_2, char_less_2, char_position_csi, string_equal_2, substring_to_temp, 
           string_greater_2, string_less_2, symbol_to_string_uncopied, vector_ref_ic, vector_ref_ic_0, vector_ref_ic_1,
           vector_ref_ic_2, vector_ref_ic_3, vector_ref_2, vector_ref_2_direct, vector_set_ic, vector_set_3, fv_ref,
           fv_ref_3, fv_set, fv_set_unchecked, iv_ref, iv_ref_0, iv_set, bv_ref, bv_set, list_set_ic, hash_table_ref_2, hash_table_ref_ss, hash_table_star_2,
           hash_table_ref_car, format_allg, format_allg_no_column, format_just_control_string, format_as_objstr,
           not_is_pair_s, not_is_null_s, not_is_symbol_s, not_is_number_s, not_is_eq_ss, not_is_eq_sq, not_is_pair_car_s,
           not_c_c, is_pair_car_s, is_pair_cdr_s, is_pair_cddr_s, is_pair_cadr_s, is_null_cdr, is_null_cddr_s,
           is_null_cadr_s, is_symbol_cadr_s, is_eq_car, is_eq_car_q, is_eq_caar_q, member_ss, member_sq, memq_2,
           memq_3, memq_4, memq_any, memq_car, memq_car_2, tree_set_memq_syms, read_line_uncopied, simple_inlet,
           lint_let_ref, lint_let_set, or_n, or_2, or_3, and_n, and_2, and_3, and_sc, if_x1, if_x2, if_not_x1,
           if_not_x2, if_x_qq, if_x_qa, or_s_direct, and_s_direct, geq_2, or_s_direct_2, and_s_direct_2, or_s_type_2;

#if (!WITH_GMP)
  s7_pointer multiply_2, multiply_is, multiply_si, multiply_fs, multiply_sf, sqr_ss, invert_1, divide_1r, mod_si, equal_s_ic,
           equal_length_ic, equal_2, equal_2i, less_s_ic, less_s0, less_2, less_length_ic, greater_s_ic, greater_s_fc, greater_2,
           leq_s_ic, leq_2, geq_s_ic, geq_s_fc, random_ic, random_rc, random_1, 
           mul_2_ff, mul_2_ii, mul_2_if, mul_2_fi, mul_2_xi, mul_2_ix, mul_2_fx, mul_2_xf,
           add_2_ff, add_2_ii, add_2_if, add_2_fi, add_2_xi, add_2_ix, add_2_fx, add_2_xf;
#endif
#if WITH_GMP
  s7_pointer bignum_symbol, is_bignum_symbol;
  gc_list *bigints, *bigratios, *bigreals, *bignumbers;
#endif
  /* object->let symbols */
#if (!WITH_GMP)
  s7_pointer seed_symbol, carry_symbol;
#endif
  s7_pointer active_symbol, goto_symbol, data_symbol, weak_symbol, dimensions_symbol, info_symbol, c_type_symbol, source_symbol, c_object_ref_symbol,
             at_end_symbol, sequence_symbol, position_symbol, entries_symbol, locked_symbol, function_symbol, open_symbol, alias_symbol, port_type_symbol,
             file_symbol, line_symbol, c_object_let_symbol, class_symbol, c_object_length_symbol, c_object_set_symbol, current_value_symbol,
             c_object_copy_symbol, c_object_fill_symbol, c_object_reverse_symbol, c_object_to_list_symbol, c_object_to_string_symbol, closed_symbol;

#if WITH_SYSTEM_EXTRAS
  s7_pointer is_directory_symbol, file_exists_symbol, delete_file_symbol, getenv_symbol, system_symbol, directory_to_list_symbol, file_mtime_symbol;
#endif

  s7_pointer vector_set_function, string_set_function, list_set_function, hash_table_set_function, let_set_function, c_object_set_function, last_function;

  s7_pointer wrong_type_arg_info, out_of_range_info, simple_wrong_type_arg_info, simple_out_of_range_info;
  s7_pointer integer_wrapper1, integer_wrapper2, integer_wrapper3;
  s7_pointer real_wrapper1, real_wrapper2, real_wrapper3, real_wrapper4;

  #define NUM_SAFE_PRELISTS 8
  #define NUM_SAFE_LISTS 64               /* 36 is the biggest normally (lint.scm), 49 in s7test, 57 in snd-test */
  s7_pointer safe_lists[NUM_SAFE_LISTS];
  int32_t current_safe_list;

  s7_pointer autoload_table, libraries, profile_info, s7_let;
  const char ***autoload_names;
  s7_int *autoload_names_sizes;
  bool **autoloaded_already;
  s7_int autoload_names_loc, autoload_names_top;
  int32_t format_depth;

  bool undefined_identifier_warnings;
  optfix_t *optimizer_fixups;

  jmp_buf opt_exit;
  int32_t pc;
  bool opt_has_local_let;
  #define OPTS_SIZE 256          /* 128 overflows twice in s7test, 64 overflows 4 times in s7test, once in tall, pqw-vox needs 173 */
  opt_info *opts[OPTS_SIZE + 1]; /* this form is a lot faster than opt_info**! */
  opt_info *base_opts;

  heap_block_t *heap_blocks;
};

static s7_scheme *cur_sc = NULL; /* intended for gdb (see gdbinit) */


/* -------------------------------- mallocate -------------------------------- */

static const int32_t bits[256] =
  {0, 1, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5,
   6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6,
   7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
   7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7,
   8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
   8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
   8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
   8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8};

static void memclr(void *s, size_t n)
{
  uint8_t *s2;
#if S7_ALIGNED
  s2 = (uint8_t *)s;
#else
#if (defined(__x86_64__) || defined(__i386__))
  if (n >= 8)
    {
      int64_t *s1 = (int64_t *)s;
      size_t n8 = n >> 3;
      do {*s1++ = 0;} while (--n8 > 0);
      n &= 7;
      s2 = (uint8_t *)s1;
    }
  else s2 = (uint8_t *)s;
#else
  s2 = (uint8_t *)s;
#endif
#endif
  while (n > 0)
    {
      *s2++ = 0;
      n--;
    }
}

#define LOOP_4(Code) do {Code; Code; Code; Code;} while (0)
#define LOOP_8(Code) do {Code; Code; Code; Code; Code; Code; Code; Code;} while (0)

#if POINTER_32
#define memclr64 memclr
#else
#if WITH_VECTORIZE
static void memclr64(void *p, size_t bytes) __attribute__((optimize("tree-vectorize")));
#endif

static void memclr64(void *p, size_t bytes)
{
  size_t i, n;
  int64_t *vals;
  vals = (int64_t *)p;
  n = bytes >> 3;
  for (i = 0; i < n; )
    LOOP_8(vals[i++] = 0);
}
#endif

static void init_block_lists(s7_scheme *sc)
{
  int32_t i;
  for (i = 0; i < NUM_BLOCK_LISTS; i++)
    sc->block_lists[i] = NULL;
}

/* clear_block_lists (in (gc)?) can free the data blocks and put all blocks on block_list[BLOCK_LIST],
 *  but the blocks themselves can't be freed (they're allocated by malloc in arbitrary batches, and
 *  the malloc pointer block is not currently recognizable).
 */

static inline void liberate(s7_scheme *sc, block_t *p)
{
  if (block_index(p) != TOP_BLOCK_LIST)
    {
      block_next(p) = (struct block_t *)sc->block_lists[block_index(p)];
      sc->block_lists[block_index(p)] = p;
    }
  else
    {
      if (block_data(p)) {free(block_data(p)); block_data(p) = NULL;}
      block_set_index(p, BLOCK_LIST);
      block_next(p) = (struct block_t *)sc->block_lists[block_index(p)];
      sc->block_lists[block_index(p)] = p;
    }
}

static inline void liberate_block(s7_scheme *sc, block_t *p)
{
  block_next(p) = (struct block_t *)sc->block_lists[BLOCK_LIST];
  sc->block_lists[BLOCK_LIST] = p;
}

static void fill_block_list(s7_scheme *sc)
{
  int32_t i;
  block_t *b;
  #define BLOCK_MALLOC_SIZE 256
  b = (block_t *)malloc(BLOCK_MALLOC_SIZE * sizeof(block_t)); /* batch alloc means blocks in this batch can't be freed, only returned to the list */
  sc->block_lists[BLOCK_LIST] = b;
  for (i = 0; i < BLOCK_MALLOC_SIZE - 1; i++)
    {
      block_next(b) = (block_t *)(b + 1);
      block_set_index(b, BLOCK_LIST);
      b++;
    }
  block_next(b) = NULL;
  block_set_index(b, BLOCK_LIST);
}

static inline block_t *mallocate_block(s7_scheme *sc)
{
  block_t *p;
  if (!sc->block_lists[BLOCK_LIST])
    fill_block_list(sc);                /* this is much faster than allocating blocks as needed */
  p = sc->block_lists[BLOCK_LIST];
  sc->block_lists[BLOCK_LIST] = (block_t *)(block_next(p));
  block_next(p) = NULL;
  block_index(p) = BLOCK_LIST;
  return(p);
}

static inline char *alloc_permanent_string(s7_scheme *sc, size_t len)
{
  #define ALLOC_STRING_SIZE 32768
  #define ALLOC_MAX_STRING 256
  char *result;
  size_t next_k;

  len = (len + 7) & (~7);            /* 8-byte aligned -- more than half the time, len is already 8-byte aligned */
  next_k = sc->alloc_string_k + len;
  if (next_k >= ALLOC_STRING_SIZE)
    {
      if (len >= ALLOC_MAX_STRING)
	return((char *)malloc(len));
      sc->alloc_string_cells = (char *)malloc(ALLOC_STRING_SIZE);
      sc->alloc_string_k = 0;
      next_k = len;
    }
  result = &(sc->alloc_string_cells[sc->alloc_string_k]);
  sc->alloc_string_k = next_k;
  return(result);
}

static inline block_t *mallocate(s7_scheme *sc, size_t bytes)
{
  block_t *p;
  if (bytes > 0)
    {
      int32_t index;
      if (bytes <= 8)
	index = 3;
      else
	{
	  if (bytes <= 256)
	    index = bits[bytes - 1];
	  else
	    {
	      if (bytes <= 65536)
		index = 8 + bits[(bytes - 1) >> 8];
	      else index = TOP_BLOCK_LIST; /* expansion to (1 << 17) made no difference */
	    }
	}
      p = sc->block_lists[index];
      if (p)
	{
	  sc->block_lists[index] = (block_t *)block_next(p);
	  block_next(p) = NULL;
	}
      else
	{
	  p = mallocate_block(sc);
	  block_data(p) = (void *)alloc_permanent_string(sc, (index < TOP_BLOCK_LIST) ? (size_t)(1 << index) : bytes);
	  block_set_index(p, index);
	}
    }
  else p = mallocate_block(sc);
  block_set_size(p, bytes);
  return(p);
}

static block_t *callocate(s7_scheme *sc, size_t bytes)
{
  block_t *p;
  p = mallocate(sc, bytes);
  if ((block_data(p)) && (block_index(p) != BLOCK_LIST))
    {
      if (block_index(p) >= 6)
	memclr64((void *)block_data(p), bytes);
      else memclr((void *)(block_data(p)), bytes);
    }
  return(p);
}

static block_t *reallocate(s7_scheme *sc, block_t *op, size_t bytes)
{
  block_t *np;
  np = mallocate(sc, bytes);
  if (block_data(op))  /* presumably block_data(np) is not null */
    memcpy((uint8_t *)(block_data(np)), (uint8_t *)(block_data(op)), block_size(op));
  liberate(sc, op);
  return(np);
}
/* -------------------------------------------------------------------------------- */

/* (*s7* 'safety) settings */
#define NO_SAFETY 0
#define IMMUTABLE_VECTOR_SAFETY 1

typedef enum {P_DISPLAY, P_WRITE, P_READABLE, P_KEY} use_write_t;

#define INITIAL_AUTOLOAD_NAMES_SIZE 4

static s7_pointer prepackaged_type_names[NUM_TYPES];

static s7_pointer too_many_arguments_string, not_enough_arguments_string, missing_method_string,
  a_boolean_string, a_byte_vector_string, a_format_port_string, a_let_string, a_list_string, a_non_constant_symbol_string,
  a_non_negative_integer_string, a_normal_procedure_string, a_normal_real_string, a_number_string, a_procedure_string,
  a_proper_list_string, a_random_state_object_string, a_rational_string, a_sequence_string, a_symbol_string, a_thunk_string,
  a_valid_radix_string, an_association_list_string, an_eq_func_string, an_input_file_port_string, an_input_port_string,
  an_input_string_port_string, an_open_port_string, an_output_file_port_string, an_output_port_string, an_output_string_port_string,
  an_unsigned_byte_string, caaar_a_list_string, caadr_a_list_string, caar_a_list_string, cadar_a_list_string, caddr_a_list_string,
  cadr_a_list_string, car_a_list_string, cdaar_a_list_string, cdadr_a_list_string, cdar_a_list_string, cddar_a_list_string,
  cdddr_a_list_string, cddr_a_list_string, cdr_a_list_string, immutable_error_string, its_infinite_string, its_nan_string,
  its_negative_string, its_too_large_string, its_too_small_string, parameter_set_twice_string, result_is_too_large_string,
  something_applicable_string, too_many_indices_string, value_is_missing_string,
  format_string_1, format_string_2, format_string_3, format_string_4;

static bool t_number_p[NUM_TYPES], t_real_p[NUM_TYPES], t_rational_p[NUM_TYPES], t_big_number_p[NUM_TYPES];
static bool t_simple_p[NUM_TYPES], t_structure_p[NUM_TYPES];
static bool t_any_macro_p[NUM_TYPES], t_any_closure_p[NUM_TYPES], t_has_closure_let[NUM_TYPES];
static bool t_mappable_p[NUM_TYPES], t_sequence_p[NUM_TYPES], t_vector_p[NUM_TYPES];
static bool t_procedure_p[NUM_TYPES], t_applicable_p[NUM_TYPES];
#if S7_DEBUGGING
static bool t_freeze_p[NUM_TYPES];
#endif

static void init_types(void)
{
  int32_t i;
  for (i = 0; i < NUM_TYPES; i++)
    {
      t_number_p[i] = false;
      t_real_p[i] = false;
      t_rational_p[i] = false;
      t_simple_p[i] = false;
      t_structure_p[i] = false;
      t_any_macro_p[i] = false;
      t_any_closure_p[i] = false;
      t_has_closure_let[i] = false;
      t_sequence_p[i] = false;
      t_mappable_p[i] = false;
      t_vector_p[i] = false;
      t_applicable_p[i] = false;
      t_procedure_p[i] = false;
#if S7_DEBUGGING
      t_freeze_p[i] = false;
#endif
    }
  t_number_p[T_INTEGER] = true;
  t_number_p[T_RATIO] = true;
  t_number_p[T_REAL] = true;
  t_number_p[T_COMPLEX] = true;

  t_rational_p[T_INTEGER] = true;
  t_rational_p[T_RATIO] = true;

  t_real_p[T_INTEGER] = true;
  t_real_p[T_RATIO] = true;
  t_real_p[T_REAL] = true;

  t_big_number_p[T_BIG_INTEGER] = true;
  t_big_number_p[T_BIG_RATIO] = true;
  t_big_number_p[T_BIG_REAL] = true;
  t_big_number_p[T_BIG_COMPLEX] = true;

  t_structure_p[T_PAIR] = true;
  t_structure_p[T_VECTOR] = true;
  t_structure_p[T_HASH_TABLE] = true;
  t_structure_p[T_SLOT] = true;
  t_structure_p[T_LET] = true;
  t_structure_p[T_ITERATOR] = true;
  t_structure_p[T_C_POINTER] = true;
  t_structure_p[T_C_OBJECT] = true;

  t_sequence_p[T_NIL] = true;
  t_sequence_p[T_PAIR] = true;
  t_sequence_p[T_STRING] = true;
  t_sequence_p[T_VECTOR] = true;
  t_sequence_p[T_INT_VECTOR] = true;
  t_sequence_p[T_FLOAT_VECTOR] = true;
  t_sequence_p[T_BYTE_VECTOR] = true;
  t_sequence_p[T_HASH_TABLE] = true;
  t_sequence_p[T_LET] = true;
  t_sequence_p[T_C_OBJECT] = true;

  t_mappable_p[T_PAIR] = true;
  t_mappable_p[T_STRING] = true;
  t_mappable_p[T_VECTOR] = true;
  t_mappable_p[T_INT_VECTOR] = true;
  t_mappable_p[T_FLOAT_VECTOR] = true;
  t_mappable_p[T_BYTE_VECTOR] = true;
  t_mappable_p[T_HASH_TABLE] = true;
  t_mappable_p[T_LET] = true;
  t_mappable_p[T_C_OBJECT] = true;
  t_mappable_p[T_ITERATOR] = true;
  t_mappable_p[T_C_MACRO] = true;
  t_mappable_p[T_MACRO] = true;
  t_mappable_p[T_BACRO] = true;
  t_mappable_p[T_MACRO_STAR] = true;
  t_mappable_p[T_BACRO_STAR] = true;
  t_mappable_p[T_CLOSURE] = true;
  t_mappable_p[T_CLOSURE_STAR] = true;

  t_vector_p[T_VECTOR] = true;
  t_vector_p[T_INT_VECTOR] = true;
  t_vector_p[T_FLOAT_VECTOR] = true;
  t_vector_p[T_BYTE_VECTOR] = true;

  t_applicable_p[T_PAIR] = true;
  t_applicable_p[T_STRING] = true;
  t_applicable_p[T_VECTOR] = true;
  t_applicable_p[T_INT_VECTOR] = true;
  t_applicable_p[T_FLOAT_VECTOR] = true;
  t_applicable_p[T_BYTE_VECTOR] = true;
  t_applicable_p[T_HASH_TABLE] = true;
  t_applicable_p[T_ITERATOR] = true;
  t_applicable_p[T_LET] = true;
  t_applicable_p[T_C_OBJECT] = true;
  t_applicable_p[T_C_MACRO] = true;
  t_applicable_p[T_MACRO] = true;
  t_applicable_p[T_BACRO] = true;
  t_applicable_p[T_MACRO_STAR] = true;
  t_applicable_p[T_BACRO_STAR] = true;
  t_applicable_p[T_SYNTAX] = true;
  t_applicable_p[T_C_FUNCTION] = true;
  t_applicable_p[T_C_FUNCTION_STAR] = true;
  t_applicable_p[T_C_ANY_ARGS_FUNCTION] = true;
  t_applicable_p[T_C_OPT_ARGS_FUNCTION] = true;
  t_applicable_p[T_C_RST_ARGS_FUNCTION] = true;
  t_applicable_p[T_CLOSURE] = true;
  t_applicable_p[T_CLOSURE_STAR] = true;
  t_applicable_p[T_GOTO] = true;
  t_applicable_p[T_CONTINUATION] = true;

  /* t_procedure_p[T_C_OBJECT] = true; */
  t_procedure_p[T_C_FUNCTION] = true;
  t_procedure_p[T_C_FUNCTION_STAR] = true;
  t_procedure_p[T_C_ANY_ARGS_FUNCTION] = true;
  t_procedure_p[T_C_OPT_ARGS_FUNCTION] = true;
  t_procedure_p[T_C_RST_ARGS_FUNCTION] = true;
  t_procedure_p[T_CLOSURE] = true;
  t_procedure_p[T_CLOSURE_STAR] = true;
  t_procedure_p[T_GOTO] = true;
  t_procedure_p[T_CONTINUATION] = true;

  t_any_macro_p[T_C_MACRO] = true;
  t_any_macro_p[T_MACRO] = true;
  t_any_macro_p[T_BACRO] = true;
  t_any_macro_p[T_MACRO_STAR] = true;
  t_any_macro_p[T_BACRO_STAR] = true;

  t_any_closure_p[T_CLOSURE] = true;
  t_any_closure_p[T_CLOSURE_STAR] = true;

  t_has_closure_let[T_MACRO] = true;
  t_has_closure_let[T_BACRO] = true;
  t_has_closure_let[T_MACRO_STAR] = true;
  t_has_closure_let[T_BACRO_STAR] = true;
  t_has_closure_let[T_CLOSURE] = true;
  t_has_closure_let[T_CLOSURE_STAR] = true;

  t_simple_p[T_NIL] = true;
  t_simple_p[T_UNDEFINED] = true;
  t_simple_p[T_EOF_OBJECT] = true;
  t_simple_p[T_BOOLEAN] = true;
  t_simple_p[T_CHARACTER] = true;
  t_simple_p[T_SYMBOL] = true;
  t_simple_p[T_SYNTAX] = true;
  t_simple_p[T_C_MACRO] = true;
  t_simple_p[T_C_FUNCTION] = true;
  t_simple_p[T_C_FUNCTION_STAR] = true;
  t_simple_p[T_C_ANY_ARGS_FUNCTION] = true;
  t_simple_p[T_C_OPT_ARGS_FUNCTION] = true;
  t_simple_p[T_C_RST_ARGS_FUNCTION] = true;
  /* not completely sure about the next ones */
  t_simple_p[T_LET] = true;
  t_simple_p[T_INPUT_PORT] = true;
  t_simple_p[T_OUTPUT_PORT] = true;

#if S7_DEBUGGING
  t_freeze_p[T_STRING] = true;
  t_freeze_p[T_BYTE_VECTOR] = true;
  t_freeze_p[T_VECTOR] = true;
  t_freeze_p[T_FLOAT_VECTOR] = true;
  t_freeze_p[T_INT_VECTOR] = true;
  t_freeze_p[T_UNDEFINED] = true;
  t_freeze_p[T_C_OBJECT] = true;
  t_freeze_p[T_LET] = true;
  t_freeze_p[T_HASH_TABLE] = true;
  t_freeze_p[T_C_FUNCTION] = true;
  t_freeze_p[T_CONTINUATION] = true;
  t_freeze_p[T_INPUT_PORT] = true;
  t_freeze_p[T_OUTPUT_PORT] = true;
#endif
}

#if WITH_HISTORY
#define current_code(Sc)           car(Sc->cur_code)
#define set_current_code(Sc, Code) do {if (Sc->history_enabled) {Sc->cur_code = cdr(Sc->cur_code); set_car(Sc->cur_code, Code);} } while (0)
#define mark_current_code(Sc)      do {int32_t i; s7_pointer p; for (p = Sc->cur_code, i = 0; i < sc->history_size; i++, p = cdr(p)) gc_mark(car(p));} while (0)
#else
#define current_code(Sc)           Sc->cur_code
#define set_current_code(Sc, Code) Sc->cur_code = Code
#define mark_current_code(Sc)      gc_mark(Sc->cur_code)
#endif

#define typeflag(p)  ((p)->tf.flag)
#define typesflag(p) ((p)->tf.sflag)

#if S7_DEBUGGING
  static const char *check_name(int32_t typ);
  static s7_pointer check_ref(s7_pointer p, uint8_t expected_type, const char *func, int32_t line, const char *func1, const char *func2);
  static s7_pointer check_ref2(s7_pointer p, uint8_t expected_type, int32_t other_type, const char *func, int32_t line, const char *func1, const char *func2);
  static s7_pointer check_ref3(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref4(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref5(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref6(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref7(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref8(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref9(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref10(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref11(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_ref12(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_nref(s7_pointer p, const char *func, int32_t line);
  static s7_pointer check_let_ref(s7_pointer p, uint64_t role, const char *func, int32_t line);
  static s7_pointer check_let_set(s7_pointer p, uint64_t role, const char *func, int32_t line);
  static s7_pointer check_cell(s7_pointer p, const char *func, int32_t line);
  static void print_gc_info(s7_pointer obj, int32_t line);

  static s7_pointer opt1_1(s7_pointer p, uint32_t role, const char *func, int32_t line);
  static s7_pointer set_opt1_1(s7_pointer p, s7_pointer x, uint32_t role, const char *func, int32_t line);
  static s7_pointer opt2_1(s7_scheme *sc, s7_pointer p, uint32_t role, const char *func, int32_t line);
  static void set_opt2_1(s7_scheme *sc, s7_pointer p, s7_pointer x, uint32_t role, const char *func, int32_t line);
  static s7_pointer opt3_1(s7_pointer p, uint32_t role, const char *func, int32_t line);
  static void set_opt3_1(s7_pointer p, s7_pointer x, uint32_t role, const char *func, int32_t line);

  static uint64_t s_hash_1(s7_pointer p, const char *func, int32_t line);
  static void set_s_hash_1(s7_pointer p, uint64_t x, const char *func, int32_t line);
  static const char *s_name_1(s7_pointer p, const char *func, int32_t line);
  static void set_s_name_1(s7_pointer p, const char *str, const char *func, int32_t line);
  static uint32_t s_line_1(s7_pointer p, const char *func, int32_t line);
  static void set_s_line_1(s7_pointer p, uint32_t x, const char *func, int32_t line);
  static void set_s_file_1(s7_scheme *sc, s7_pointer p, uint32_t x, const char *func, int32_t line);
  static uint32_t s_len_1(s7_pointer p, const char *func, int32_t line);
  static void set_s_len_1(s7_pointer p, uint32_t x, const char *func, int32_t line);
  #define unchecked_type(p)           ((p)->tf.type_field)
#if WITH_GCC
  #define type(p) ({uint8_t _t_; _t_ = (p)->tf.type_field; if (((_t_ == T_FREE)) || (_t_ >= NUM_TYPES)) print_gc_info(p, __LINE__); _t_;})
#else
  #define type(p) (p)->tf.type_field
#endif

  #define set_type(p, f)						\
    do {								\
      p->previous_alloc_line = p->current_alloc_line;			\
      p->previous_alloc_func = p->current_alloc_func;			\
      p->previous_alloc_type = p->current_alloc_type;			\
      p->current_alloc_line = __LINE__;					\
      p->current_alloc_func = __func__;					\
      p->current_alloc_type = f;					\
      p->explicit_free_line = 0;					\
      p->uses++; 					                \
      if ((((f) & 0xff) == T_FREE) || (((f) & 0xff) >= NUM_TYPES))	\
        fprintf(stderr, "%d: set free %p type to %" PRIx64 "\n", __LINE__, p, (int64_t)(f)); \
      else								\
	{								\
	  if (((typeflag(p) & T_IMMUTABLE) != 0) && ((typeflag(p) != (uint64_t)(f))))						\
	    {fprintf(stderr, "%s[%d]: set immutable %p type %d to %" print_s7_int "\n", __func__, __LINE__, p, unchecked_type(p), (int64_t)(f)); abort();} \
          if (((typeflag(p) & T_UNHEAP) != 0) && (((f) & T_UNHEAP) == 0)) \
	    fprintf(stderr, "%s[%d]: clearing unheap in set type!\n", __func__, __LINE__);		\
	}								\
      typeflag(p) = f;							\
    } while (0)

  /* these check most s7cell field references (and many type bits) for consistency */
  #define T_Int(P) check_ref(P, T_INTEGER,           __func__, __LINE__, NULL, NULL)
  #define T_Rel(P) check_ref(P, T_REAL,              __func__, __LINE__, NULL, NULL)
  #define T_Frc(P) check_ref2(P, T_RATIO, T_INTEGER, __func__, __LINE__, NULL, NULL)
  #define T_Cmp(P) check_ref(P, T_COMPLEX,           __func__, __LINE__, NULL, NULL)
  #define T_Bgi(P) check_ref(P, T_BIG_INTEGER,       __func__, __LINE__, "sweep", NULL)
  #define T_Bgr(P) check_ref(P, T_BIG_REAL,          __func__, __LINE__, "sweep", NULL)
  #define T_Bgf(P) check_ref(P, T_BIG_RATIO,         __func__, __LINE__, "sweep", NULL)
  #define T_Bgz(P) check_ref(P, T_BIG_COMPLEX,       __func__, __LINE__, "sweep", NULL)

  #define T_Chr(P) check_ref(P, T_CHARACTER,         __func__, __LINE__, NULL, NULL)
  #define T_Ctr(P) check_ref(P, T_COUNTER,           __func__, __LINE__, NULL, NULL)
  #define T_Ptr(P) check_ref(P, T_C_POINTER,         __func__, __LINE__, NULL, NULL)
  #define T_Bfl(P) check_ref(P, T_BAFFLE,            __func__, __LINE__, NULL, NULL)
  #define T_Got(P) check_ref(P, T_GOTO,              __func__, __LINE__, NULL, NULL)
  #define T_Stk(P) check_ref(P, T_STACK,             __func__, __LINE__, NULL, NULL)
  #define T_Pair(P) check_ref(P, T_PAIR,             __func__, __LINE__, NULL, NULL)
  #define T_Cat(P) check_ref(P, T_CATCH,             __func__, __LINE__, NULL, NULL)
  #define T_Dyn(P) check_ref(P, T_DYNAMIC_WIND,      __func__, __LINE__, NULL, NULL)
  #define T_Slt(P) check_ref(P, T_SLOT,              __func__, __LINE__, NULL, NULL)
  #define T_Slp(P) check_ref2(P, T_SLOT, T_PAIR,     __func__, __LINE__, NULL, NULL)
  #define T_Sln(P) check_ref2(P, T_SLOT, T_NIL,      __func__, __LINE__, NULL, NULL)
  #define T_Sld(P) check_ref2(P, T_SLOT, T_UNDEFINED,__func__, __LINE__, NULL, NULL)
  #define T_Syn(P) check_ref(P, T_SYNTAX,            __func__, __LINE__, NULL, NULL)
  #define T_Mac(P) check_ref(P, T_C_MACRO,           __func__, __LINE__, NULL, NULL)
  #define T_Let(P) check_ref(P, T_LET,               __func__, __LINE__, NULL, NULL)
  #define T_Lid(P) check_ref2(P, T_LET, T_NIL,       __func__, __LINE__, NULL, NULL)
  #define T_Ran(P) check_ref(P, T_RANDOM_STATE,      __func__, __LINE__, NULL, NULL)
  #define T_Lst(P) check_ref2(P, T_PAIR, T_NIL,      __func__, __LINE__, "gc", NULL)
  #define T_Str(P) check_ref(P, T_STRING,            __func__, __LINE__, "sweep", NULL)
  #define T_BVc(P) check_ref(P, T_BYTE_VECTOR,       __func__, __LINE__, "sweep", NULL)
  #define T_BSt(P) check_ref2(P, T_BYTE_VECTOR, T_STRING, __func__, __LINE__, "sweep", NULL)
  #define T_Obj(P) check_ref(P, T_C_OBJECT,          __func__, __LINE__, "sweep", NULL)
  #define T_Hsh(P) check_ref(P, T_HASH_TABLE,        __func__, __LINE__, "sweep", "free_hash_table")
  #define T_Itr(P) check_ref(P, T_ITERATOR,          __func__, __LINE__, "sweep", NULL)
  #define T_Con(P) check_ref(P, T_CONTINUATION,      __func__, __LINE__, "sweep", NULL)
  #define T_Fvc(P) check_ref(P, T_FLOAT_VECTOR,      __func__, __LINE__, "sweep", NULL)
  #define T_Ivc(P) check_ref(P, T_INT_VECTOR,        __func__, __LINE__, "sweep", NULL)
  #define T_Nvc(P) check_ref(P, T_VECTOR,            __func__, __LINE__, "sweep", NULL)
  #define T_Sym(P) check_ref(P, T_SYMBOL,            __func__, __LINE__, "sweep", "remove_gensym_from_symbol_table")
  #define T_Fst(P) check_ref(P, T_C_FUNCTION_STAR,   __func__, __LINE__, "sweep", NULL)
  #define T_Exp(P) check_ref2(P, T_MACRO, T_SYMBOL,  __func__, __LINE__, NULL, NULL)
  #define T_Prt(P) check_ref3(P,                     __func__, __LINE__) /* input|output_port, or free */
  #define T_Vec(P) check_ref4(P,                     __func__, __LINE__) /* any vector or free */
  #define T_Clo(P) check_ref5(P,                     __func__, __LINE__) /* has closure let */
  #define T_Clp(P) check_ref12(P,                    __func__, __LINE__) /* any closure or pair */
  #define T_Fnc(P) check_ref6(P,                     __func__, __LINE__) /* any c_function|c_macro */
  #define T_Num(P) check_ref7(P,                     __func__, __LINE__) /* any number (not bignums I think) */
  #define T_Seq(P) check_ref8(P,                     __func__, __LINE__) /* any sequence or structure */
  #define T_Met(P) check_ref9(P,                     __func__, __LINE__) /* anything that might contain a method */
  #define T_Arg(P) check_ref10(P,                    __func__, __LINE__) /* closure arg (list, symbol) */
  #define T_App(P) check_ref11(P,                    __func__, __LINE__) /* applicable or #f */
  #define T_Pos(P) check_nref(P,                     __func__, __LINE__) /* not free */
  #define T_Any(P) check_cell(P,                     __func__, __LINE__) /* any cell */

#else
  #define unchecked_type(p)            ((p)->tf.type_field)
  #define type(p)                      ((p)->tf.type_field)
  #define set_type(p, f)               typeflag(p) = f
  #define T_Int(P)                     P
  #define T_Rel(P)                     P
  #define T_Frc(P)                     P
  #define T_Cmp(P)                     P
  #define T_Bgi(P)                     P
  #define T_Bgr(P)                     P
  #define T_Bgf(P)                     P
  #define T_Bgz(P)                     P
  #define T_Str(P)                     P
  #define T_BVc(P)                     P
  #define T_BSt(P)                     P
  #define T_Syn(P)                     P
  #define T_Chr(P)                     P
  #define T_Obj(P)                     P
  #define T_Ctr(P)                     P
  #define T_Hsh(P)                     P
  #define T_Itr(P)                     P
  #define T_Ptr(P)                     P
  #define T_Bfl(P)                     P
  #define T_Got(P)                     P
  #define T_Con(P)                     P
  #define T_Stk(P)                     P
  #define T_Prt(P)                     P
  #define T_Ivc(P)                     P
  #define T_Fvc(P)                     P
  #define T_Nvc(P)                     P
  #define T_Vec(P)                     P
  #define T_Pair(P)                    P
  #define T_Ran(P)                     P
  #define T_Dyn(P)                     P
  #define T_Cat(P)                     P
  #define T_Clo(P)                     P
  #define T_Clp(P)                     P
  #define T_Fnc(P)                     P
  #define T_Fst(P)                     P
  #define T_Slt(P)                     P
  #define T_Sln(P)                     P
  #define T_Sld(P)                     P
  #define T_Slp(P)                     P
  #define T_Sym(P)                     P
  #define T_Let(P)                     P
  #define T_Lid(P)                     P
  #define T_Lst(P)                     P
  #define T_Num(P)                     P
  #define T_Seq(P)                     P
  #define T_Met(P)                     P
  #define T_Mac(P)                     P
  #define T_Arg(P)                     P
  #define T_App(P)                     P
  #define T_Exp(P)                     P
  #define T_Pos(P)                     P
  #define T_Any(P)                     P
#endif

#define is_number(P)                   t_number_p[type(P)]
#define is_integer(P)                  (type(P) == T_INTEGER)
#define is_real(P)                     t_real_p[type(P)]
#define is_rational(P)                 t_rational_p[type(P)]
#define is_big_number(p)               t_big_number_p[type(p)]
#define is_t_integer(p)                (type(p) == T_INTEGER)
#define is_t_ratio(p)                  (type(p) == T_RATIO)
#define is_t_real(p)                   (type(p) == T_REAL)
#define is_t_complex(p)                (type(p) == T_COMPLEX)
#define is_t_big_integer(p)            (type(p) == T_BIG_INTEGER)
#define is_t_big_ratio(p)              (type(p) == T_BIG_RATIO)
#define is_t_big_real(p)               (type(p) == T_BIG_REAL)
#define is_t_big_complex(p)            (type(p) == T_BIG_COMPLEX)
#define is_float(p)                    ((is_real(p)) && (!is_rational(p)))

#define is_free(p)                     (type(p) == T_FREE)
#define is_free_and_clear(p)           (typeflag(p) == T_FREE)
#define is_simple(P)                   t_simple_p[type(P)]
#define has_structure(P)               ((t_structure_p[type(P)]) && ((!is_normal_vector(P)) || (!has_simple_elements(P))))

#define is_any_macro(P)                t_any_macro_p[type(P)]
#define is_any_closure(P)              t_any_closure_p[type(P)]
#define is_procedure_or_macro(P)       ((t_procedure_p[type(P)]) || (t_any_macro_p[type(P)]))
#define is_any_procedure(P)            (type(P) >= T_CLOSURE)
#define has_closure_let(P)             t_has_closure_let[type(P)]

#define is_simple_sequence(P)          (t_sequence_p[type(P)])
#define is_sequence(P)                 ((t_sequence_p[type(P)]) || (has_methods(P)))
#define is_mutable_sequence(P)         (((t_sequence_p[type(P)]) || (has_methods(P))) && (!is_immutable(P)))
#define is_mappable(P)                 (t_mappable_p[type(P)])
#define is_applicable(P)               (t_applicable_p[type(P)])
/* this misses #() which actually is not applicable to anything, probably "" also, and inapplicable c-objects like random-state */
#define is_procedure(p)                ((t_procedure_p[type(p)]) || ((is_c_object(p)) && (is_safe_procedure(p))))
#define is_t_procedure(p)              (t_procedure_p[type(p)])

/* the layout of these bits does matter in several cases -- don't shadow SYNTACTIC_PAIR and OPTIMIZED_PAIR */
#define TYPE_BITS                      8

#define set_type_bit(p, b)             typeflag(p) |= (b)
#define clear_type_bit(p, b)           typeflag(p) &= (~(b))
#define has_type_bit(p, b)             ((typeflag(p) & (b)) != 0)

#define set_type0_bit(p, b)            typesflag(p) |= (b)
#define clear_type0_bit(p, b)          typesflag(p) &= (~(b))
#define has_type0_bit(p, b)            ((typesflag(p) & (b)) != 0)

#define set_type1_bit(p, b)            (p)->tf.opts.high_flag |= (b)
#define clear_type1_bit(p, b)          (p)->tf.opts.high_flag &= (~(b))
#define has_type1_bit(p, b)            (((p)->tf.opts.high_flag & (b)) != 0)

#define T_SYNTACTIC                    (1 << (TYPE_BITS + 1))
#define is_syntactic(p)                has_type0_bit(T_Pos(p), T_SYNTACTIC)
#define is_syntactic_symbol(p)         (typesflag(T_Pos(p)) == (uint16_t)(T_SYMBOL | T_SYNTACTIC))
#define is_syntactic_pair(p)           (typesflag(T_Pos(p)) == (uint16_t)(T_PAIR | T_SYNTACTIC))
/* this marks symbols that represent syntax objects, it should be in the second byte */


#define T_SIMPLE_ARG_DEFAULTS          (1 << (TYPE_BITS + 2))
#define lambda_has_simple_defaults(p)  has_type_bit(T_Pair(p), T_SIMPLE_ARG_DEFAULTS)
#define lambda_set_simple_defaults(p)  set_type_bit(T_Pair(p), T_SIMPLE_ARG_DEFAULTS)
/* are all lambda* default values simple? This is set on closure_body, so it doesn't mess up closure_is_ok_1 */

#define T_LIST_IN_USE                  T_SIMPLE_ARG_DEFAULTS
#define list_is_in_use(p)              has_type0_bit(T_Pair(p), T_LIST_IN_USE)
#define set_list_in_use(p)             set_type_bit(T_Pair(p), T_LIST_IN_USE)
#define clear_list_in_use(p)           clear_type_bit(T_Pair(p), T_LIST_IN_USE)
/* if (!is_immutable(p)) free_vlist(cur_sc, p) seems plausible here, but it got no hits in s7test and other cases */

#define T_ONE_FORM                     T_SIMPLE_ARG_DEFAULTS
#define set_closure_has_one_form(p)    set_type_bit(T_Clo(p), T_ONE_FORM)
#define T_MULTIFORM                    (1 << (TYPE_BITS + 0))
#define set_closure_has_multiform(p)   set_type_bit(T_Clo(p), T_MULTIFORM)
#define set_closure_has_fx(p)          set_type_bit(T_Clo(p), T_ONE_FORM | T_MULTIFORM)
#define closure_clear_multiform(p)     clear_type_bit(T_Clo(p), T_MULTIFORM)
/* temporary extra bit (simplify development...) */

#define T_OPTIMIZED                    (1 << (TYPE_BITS + 3))
#define set_optimized(p)               set_type0_bit(T_Pair(p), T_OPTIMIZED)
#define clear_optimized(p)             clear_type0_bit(T_Pair(p), T_OPTIMIZED | T_SYNTACTIC | T_HAS_FX)
#define OPTIMIZED_PAIR                 (uint16_t)(T_PAIR | T_OPTIMIZED)
#define is_optimized(p)                (typesflag(T_Pos(p)) == OPTIMIZED_PAIR)
/* optimizer flag for an expression that has optimization info, it should be in the second byte */

#define T_SCOPE_SAFE                   T_OPTIMIZED
#define is_scope_safe(p)               has_type_bit(T_Fnc(p), T_SCOPE_SAFE)
#define set_scope_safe(p)              set_type_bit(T_Fnc(p), T_SCOPE_SAFE)


#define T_SAFE_CLOSURE                 (1 << (TYPE_BITS + 4))
#define is_safe_closure(p)             has_type0_bit(T_Clp(p), T_SAFE_CLOSURE)  /* T_Pos 1-Nov-18 */
#define set_safe_closure(p)            set_type0_bit(T_Clp(p), T_SAFE_CLOSURE)  /* T_Clp: has_closure_let, pair #f */
#define clear_safe_closure(p)          clear_type0_bit(T_Clp(p), T_SAFE_CLOSURE)

/* optimizer flag for a closure body that is completely simple (every expression is safe)
 *   set_safe_closure happens only in optimize_lambda (and define_funchcecked?), clear only in procedure_source, bits only here
 *   this has to be separate from T_SAFE_PROCEDURE, and should be in the second byte (closure_is_ok_1 checks typesflag).
 *   It can be set on either the body (a pair) or the closure itself.
 * define -> optimize_lambda sets safe -> define_funchecked -> make_funclet for the frame
 *   similarly, named let -> optimize_lambda, then let creates the frame if safe
 *   thereafter, optimizer uses OP_SAFE_CLOSURE* which calls old_frame*
 */

#define T_DONT_EVAL_ARGS               (1 << (TYPE_BITS + 5))
#define dont_eval_args(p)              has_type0_bit(T_Pos(p), T_DONT_EVAL_ARGS)
/* this marks things that don't evaluate their arguments */

#define T_EXPANSION                    (1 << (TYPE_BITS + 6))
#define is_expansion(p)                has_type0_bit(T_Exp(p), T_EXPANSION)
#define clear_expansion(p)             clear_type0_bit(T_Sym(p), T_EXPANSION)
/* this marks the symbol and its run-time macro value, distinguishing it from an ordinary macro */

#define T_MULTIPLE_VALUE               (1 << (TYPE_BITS + 7))
#define is_multiple_value(p)           has_type0_bit(T_Pos(p), T_MULTIPLE_VALUE)
#define set_multiple_value(p)          set_type0_bit(T_Pair(p), T_MULTIPLE_VALUE)
#define clear_multiple_value(p)        clear_type0_bit(T_Pair(p), T_MULTIPLE_VALUE)
#define multiple_value(p)              p
/* this bit marks a list (from "values") that is waiting for a
 *    chance to be spliced into its caller's argument list.  It is normally
 *    on only for a very short time.
 */

#define T_MATCHED                      T_MULTIPLE_VALUE
#define is_matched_pair(p)             has_type0_bit(T_Pair(p), T_MATCHED)
#define set_match_pair(p)              set_type0_bit(T_Pair(p), T_MATCHED)
#define clear_match_pair(p)            clear_type0_bit(T_Pair(p), T_MATCHED)
#define is_matched_symbol(p)           has_type0_bit(T_Sym(p), T_MATCHED)
#define set_match_symbol(p)            set_type0_bit(T_Sym(p), T_MATCHED)
#define clear_match_symbol(p)          clear_type0_bit(T_Sym(p), T_MATCHED)

#define T_GLOBAL                       (1 << (TYPE_BITS + 8))
#define T_LOCAL                        (1 << (TYPE_BITS + 12))
#define is_global(p)                   has_type_bit(T_Sym(p), T_GLOBAL)
#define set_global(p)                  do {if ((typeflag(T_Sym(p)) & T_LOCAL) == 0) typeflag(p) |= T_GLOBAL;} while (0)
/* T_LOCAL marks a symbol that has been used locally */
/* T_GLOBAL marks something defined (bound) at the top-level, and never defined locally */

#if 0
  /* to find who is stomping on our symbols: */
  static void set_local_1(s7_scheme *sc, s7_pointer symbol, const char *func, int32_t line)
  {
    if (is_global(symbol))
      fprintf(stderr, "%s[%d]: %s%s%s in %s\n",
	      func, line,
	      BOLD_TEXT, s7_object_to_c_string(sc, symbol), UNBOLD_TEXT,
	      s7_object_to_c_string(sc, sc->cur_code));
    typeflag(symbol) = (typeflag(symbol) & ~(T_DONT_EVAL_ARGS | T_GLOBAL | T_SYNTACTIC));
  }
  #define set_local(Symbol) set_local_1(sc, Symbol, __func__, __LINE__)
#else
#define set_local(p)                   typeflag(T_Sym(p)) = ((typeflag(p) | T_LOCAL) & ~(T_DONT_EVAL_ARGS | T_GLOBAL | T_SYNTACTIC))
#endif

#define T_UNSAFE_DO                    T_GLOBAL
#define is_unsafe_do(p)                has_type_bit(T_Pair(p), T_UNSAFE_DO)
#define set_unsafe_do(p)               set_type_bit(T_Pair(p), T_UNSAFE_DO)
/* marks do-loops that resist optimization */

#define T_COLLECTED                    (1 << (TYPE_BITS + 9))
#define is_collected(p)                has_type_bit(T_Seq(p), T_COLLECTED)
#define set_collected(p)               set_type_bit(T_Seq(p), T_COLLECTED)
/* #define clear_collected(p)          clear_type_bit(T_Seq(p), T_COLLECTED) */
/* this is a transient flag used by the printer to catch cycles.  It affects only objects that have structure.
 *   We can't use a low bit (bit 7 for example), because collect_shared_info inspects the object's type.
 */

#define T_LINE_NUMBER                  (1 << (TYPE_BITS + 10))
#define has_line_number(p)             has_type_bit(T_Pair(p), T_LINE_NUMBER)
#define set_has_line_number(p)         set_type_bit(T_Pair(p), T_LINE_NUMBER)
/* pair in question has line/file info added during read, or the environment has function placement info
 *   this bit should not be in the first byte -- SYNTACTIC_PAIR ignores it.
 */

#define T_LOADER_PORT                  T_LINE_NUMBER
#define is_loader_port(p)              has_type_bit(T_Prt(p), T_LOADER_PORT)
#define set_loader_port(p)             set_type_bit(T_Prt(p), T_LOADER_PORT)
#define clear_loader_port(p)           clear_type_bit(T_Prt(p), T_LOADER_PORT)
/* to block random load-time reads from screwing up the load process, this bit marks a port used by the loader */

#define T_HAS_SETTER                   T_LINE_NUMBER
#define symbol_has_setter(p)           has_type_bit(T_Sym(p), T_HAS_SETTER)
#define symbol_set_has_setter(p)       set_type_bit(T_Sym(p), T_HAS_SETTER)
#define slot_has_setter(p)             has_type_bit(T_Slt(p), T_HAS_SETTER)
#define slot_set_has_setter(p)         set_type_bit(T_Slt(p), T_HAS_SETTER)
/* marks a slot that has a setter or symbol that might have a setter */

#define T_WITH_LET_LET                 T_LINE_NUMBER
#define is_with_let_let(p)             has_type_bit(T_Let(p), T_WITH_LET_LET)
#define set_with_let_let(p)            set_type_bit(T_Let(p), T_WITH_LET_LET)
/* marks a let that is the argument to with-let */

#define T_SIMPLE_DEFAULTS              T_LINE_NUMBER
#define c_func_has_simple_defaults(p)  has_type_bit(T_Fnc(p), T_SIMPLE_DEFAULTS)
#define c_func_set_simple_defaults(p)  set_type_bit(T_Fnc(p), T_SIMPLE_DEFAULTS)
#define c_func_clear_simple_defaults(p) clear_type_bit(T_Fnc(p), T_SIMPLE_DEFAULTS)
/* flag c_func_star arg defaults that need GC protection */

#define T_NO_SETTER                    T_LINE_NUMBER
#define closure_no_setter(p)           has_type_bit(T_Clo(p), T_NO_SETTER)
#define closure_set_no_setter(p)       set_type_bit(T_Clo(p), T_NO_SETTER)

#define T_SHARED                       (1 << (TYPE_BITS + 11))
#define is_shared(p)                   has_type_bit(T_Seq(p), T_SHARED)
#define set_shared(p)                  set_type_bit(T_Seq(p), T_SHARED)
#define is_collected_or_shared(p)      has_type_bit(p, T_COLLECTED | T_SHARED)
#define clear_collected_and_shared(p)  clear_type_bit(p, T_COLLECTED | T_SHARED) /* this can clear free cells = calloc */

#define T_SAFE_PROCEDURE               (1 << (TYPE_BITS + 13))
#define is_safe_procedure(p)           has_type_bit(T_Pos(p), T_SAFE_PROCEDURE)
#define is_safe_or_scope_safe_procedure(p) ((typeflag(T_Fnc(p)) & (T_SCOPE_SAFE | T_SAFE_PROCEDURE)) != 0)
/* applicable objects that do not return or modify their arg list directly (no :rest arg in particular),
 *    and that can't call apply themselves either directly or via s7_call, and that don't mess with the stack.
 */

#define T_CHECKED                      (1 << (TYPE_BITS + 14))
#define set_checked(p)                 set_type_bit(T_Pair(p), T_CHECKED)
#define is_checked(p)                  has_type_bit(T_Pair(p), T_CHECKED)
#define clear_checked(p)               clear_type_bit(T_Pair(p), T_CHECKED)
#define set_checked_slot(p)            set_type_bit(T_Slt(p), T_CHECKED)
#define is_checked_slot(p)             has_type_bit(T_Slt(p), T_CHECKED)
#define clear_checked_slot(p)          clear_type_bit(T_Slt(p), T_CHECKED)

#define T_UNSAFE                       (1 << (TYPE_BITS + 15))
#define set_unsafe(p)                  set_type_bit(T_Pair(p), T_UNSAFE)
#define set_unsafely_optimized(p)      typeflag(T_Pair(p)) = (typeflag(p) | T_UNSAFE | T_OPTIMIZED)
#define is_unsafe(p)                   has_type_bit(T_Pair(p), T_UNSAFE)
#define clear_unsafe(p)                clear_type_bit(T_Pair(p), T_UNSAFE)
#define is_safely_optimized(p)         ((typeflag(T_Pos(p)) & (T_OPTIMIZED | T_UNSAFE)) == T_OPTIMIZED)
/* optimizer flag saying "this expression is not completely self-contained.  It might involve the stack, etc" */

#define T_CLEAN_SYMBOL                 T_UNSAFE
#define is_clean_symbol(p)             has_type_bit(T_Sym(p), T_CLEAN_SYMBOL)
#define set_clean_symbol(p)            set_type_bit(T_Sym(p), T_CLEAN_SYMBOL)
/* set if we know the symbol name can be printed without quotes (slashification) */

#define T_HAS_STEPPER                  T_UNSAFE
#define has_stepper(p)                 has_type_bit(T_Slt(p), T_HAS_STEPPER)
#define set_has_stepper(p)             set_type_bit(T_Slt(p), T_HAS_STEPPER)

#define T_IMMUTABLE                    (1 << (TYPE_BITS + 16))
#define is_immutable(p)                has_type_bit(T_Pos(p), T_IMMUTABLE)
#define set_immutable(p)               set_type_bit(T_Pos(p), T_IMMUTABLE)
#define is_immutable_port(p)           has_type_bit(T_Prt(p), T_IMMUTABLE)
#define is_immutable_symbol(p)         has_type_bit(T_Sym(p), T_IMMUTABLE)
#define is_immutable_slot(p)           has_type_bit(T_Slt(p), T_IMMUTABLE)
#define is_immutable_pair(p)           has_type_bit(T_Pair(p), T_IMMUTABLE)
#define is_immutable_vector(p)         has_type_bit(T_Vec(p), T_IMMUTABLE)
#define is_immutable_string(p)         has_type_bit(T_Str(p), T_IMMUTABLE)

#define T_SETTER                       (1 << (TYPE_BITS + 17))
#define set_setter(p)                  set_type_bit(T_Sym(p), T_SETTER)
#define is_setter(p)                   has_type_bit(T_Sym(p), T_SETTER)
/* optimizer flag for a procedure that sets some variable (set-car! for example). */

#define T_ALLOW_OTHER_KEYS             T_SETTER
#define set_allow_other_keys(p)        set_type_bit(T_Pair(p), T_ALLOW_OTHER_KEYS)
#define allows_other_keys(p)           has_type_bit(T_Pair(p), T_ALLOW_OTHER_KEYS)
/* marks arglist that allows keyword args other than those in the parameter list; can't allow
 *   (define* (f :allow-other-keys)...) because there's only one nil, and besides, it does say "other".
 */

#define T_HASH_REMOVED                 T_SETTER
#define hash_table_set_removed(p)      set_type_bit(T_Hsh(p), T_HASH_REMOVED)
#define hash_table_removed(p)          has_type_bit(T_Hsh(p), T_HASH_REMOVED)

#define T_LET_REMOVED                  T_SETTER
#define let_set_removed(p)             set_type_bit(T_Let(p), T_LET_REMOVED)
#define let_removed(p)                 has_type_bit(T_Let(p), T_LET_REMOVED)
/* these mark objects that have been removed from the heap or checked for that possibility */

#define T_HAS_EXPRESSION               T_SETTER
#define slot_set_has_expression(p)     set_type_bit(T_Slt(p), T_HAS_EXPRESSION)
#define slot_has_expression(p)         has_type_bit(T_Slt(p), T_HAS_EXPRESSION)

#define T_MUTABLE                      (1 << (TYPE_BITS + 18))
#define is_mutable(p)                  has_type_bit(T_Num(p), T_MUTABLE)
#define clear_mutable(p)               clear_type_bit(T_Num(p), T_MUTABLE)
/* used for mutable numbers */

#define T_HAS_KEYWORD                  T_MUTABLE
#define has_keyword(p)                 has_type_bit(T_Sym(p), T_HAS_KEYWORD)
#define set_has_keyword(p)             set_type_bit(T_Sym(p), T_HAS_KEYWORD)

#define T_MARK_SEQ                     T_MUTABLE
#define is_mark_seq(p)                 has_type_bit(T_Itr(p), T_MARK_SEQ)
#define set_mark_seq(p)                set_type_bit(T_Itr(p), T_MARK_SEQ)
/* used in iterators for GC mark of sequence */

#define T_STEP_END                     T_MUTABLE
#define is_step_end(p)                 has_type_bit(T_Slt(p), T_STEP_END)
#define set_step_end(p)                set_type_bit(T_Slt(p), T_STEP_END)
/* marks a slot that holds a do-loop's step-or-end variable, numerator=current, denominator=end */

#define T_PAIR_NO_OPT                  T_MUTABLE
#define set_pair_no_opt(p)             set_type_bit(T_Pair(p), T_PAIR_NO_OPT)
#define pair_no_opt(p)                 has_type_bit(T_Pair(p), T_PAIR_NO_OPT)

#define T_NO_INT_OPT                   T_SETTER
#define set_no_int_opt(p)              set_type_bit(T_Pair(p), T_NO_INT_OPT)
#define no_int_opt(p)                  has_type_bit(T_Pair(p), T_NO_INT_OPT)

#define T_NO_FLOAT_OPT                 T_UNSAFE
#define set_no_float_opt(p)            set_type_bit(T_Pair(p), T_NO_FLOAT_OPT)
#define no_float_opt(p)                has_type_bit(T_Pair(p), T_NO_FLOAT_OPT)

#define T_NO_BOOL_OPT                  T_SAFE_STEPPER
#define set_no_bool_opt(p)             set_type_bit(T_Pair(p), T_NO_BOOL_OPT)
#define no_bool_opt(p)                 has_type_bit(T_Pair(p), T_NO_BOOL_OPT)

#define T_DIRECT_X_OPT                 T_SAFE_STEPPER
#define set_direct_x_opt(p)            set_type_bit(T_Pair(p), T_DIRECT_X_OPT)
#define has_direct_x_opt(p)            has_type_bit(T_Pair(p), T_DIRECT_X_OPT)

#define T_SAFE_STEPPER                 (1 << (TYPE_BITS + 19))
#define is_safe_stepper(p)             has_type_bit(T_Slp(p), T_SAFE_STEPPER)
#define set_safe_stepper(p)            set_type_bit(T_Slp(p), T_SAFE_STEPPER)
#define clear_safe_stepper(p)          clear_type_bit(T_Slp(p), T_SAFE_STEPPER) /* T_Pos 1-Nov-18 */

#define T_PRINT_NAME                   T_SAFE_STEPPER
#define has_print_name(p)              has_type_bit(T_Num(p), T_PRINT_NAME)
#define set_has_print_name(p)          set_type_bit(T_Num(p), T_PRINT_NAME)
/* marks numbers that have a saved version of their string representation */

#define T_MAYBE_SAFE                   T_SAFE_STEPPER
#define is_maybe_safe(p)               has_type_bit(T_Fnc(p), T_MAYBE_SAFE)
#define set_maybe_safe(p)              set_type_bit(T_Fnc(p), T_MAYBE_SAFE)

#define T_HAS_LET_SET_FALLBACK         T_SAFE_STEPPER
#define T_HAS_LET_REF_FALLBACK         T_MUTABLE
#define has_let_ref_fallback(p)        ((typeflag(T_Lid(p)) & (T_HAS_LET_REF_FALLBACK | T_HAS_METHODS)) == (T_HAS_LET_REF_FALLBACK | T_HAS_METHODS))
#define has_let_set_fallback(p)        ((typeflag(T_Lid(p)) & (T_HAS_LET_SET_FALLBACK | T_HAS_METHODS)) == (T_HAS_LET_SET_FALLBACK | T_HAS_METHODS))
#define set_has_let_ref_fallback(p)    set_type_bit(T_Let(p), T_HAS_LET_REF_FALLBACK)
#define set_has_let_set_fallback(p)    set_type_bit(T_Let(p), T_HAS_LET_SET_FALLBACK)
#define set_all_methods(p, e)          typeflag(T_Let(p)) |= (typeflag(e) & (T_HAS_METHODS | T_HAS_LET_REF_FALLBACK | T_HAS_LET_SET_FALLBACK))

#define T_WEAK_HASH                    T_SAFE_STEPPER
#define set_weak_hash_table(p)         set_type_bit(T_Hsh(p), T_WEAK_HASH)
#define is_weak_hash_table(p)          has_type_bit(T_Hsh(p), T_WEAK_HASH)

#define T_COPY_ARGS                    (1 << (TYPE_BITS + 20))
#define needs_copied_args(p)           has_type_bit(T_Pos(p), T_COPY_ARGS)
/* this marks something that might mess with its argument list, it should not be in the second byte */

#define T_GENSYM                       (1 << (TYPE_BITS + 21))
#define is_gensym(p)                   has_type_bit(T_Sym(p), T_GENSYM)
/* symbol is from gensym (GC-able etc) */

#define T_FUNCLET                      T_GENSYM
#define is_funclet(p)                  has_type_bit(T_Let(p), T_FUNCLET)
#define set_funclet(p)                 set_type_bit(T_Let(p), T_FUNCLET)
/* this marks a funclet */

#define T_HASH_CHOSEN                  T_GENSYM
#define hash_chosen(p)                 has_type_bit(T_Hsh(p), T_HASH_CHOSEN)
#define hash_set_chosen(p)             set_type_bit(T_Hsh(p), T_HASH_CHOSEN)
#define hash_clear_chosen(p)           clear_type_bit(T_Hsh(p), T_HASH_CHOSEN)

#define T_DOCUMENTED                   T_GENSYM
#define is_documented(p)               has_type_bit(T_Str(p), T_DOCUMENTED)
#define set_documented(p)              set_type_bit(T_Str(p), T_DOCUMENTED)
/* this marks a symbol that has documentation (bit is set on name cell) */

#define T_DOTTED_PAIR                  T_GENSYM
#define is_dotted_pair(p)              has_type_bit(T_Lst(p), T_DOTTED_PAIR)
#define pair_set_dotted(p)             set_type_bit(T_Pair(p), T_DOTTED_PAIR)
/* reader indication that a list it just read was dotted */

#define T_SUBVECTOR                    T_GENSYM
#define is_subvector(p)                has_type_bit(T_Vec(p), T_SUBVECTOR)

#define T_HAS_PENDING_VALUE            T_GENSYM
#define slot_set_has_pending_value(p)  set_type_bit(T_Slt(p), T_HAS_PENDING_VALUE)
#define slot_has_pending_value(p)      has_type_bit(T_Slt(p), T_HAS_PENDING_VALUE)

#define T_HAS_METHODS                  (1 << (TYPE_BITS + 22))
#define has_methods(p)                 has_type_bit(T_Pos(p), T_HAS_METHODS)
#define set_has_methods(p)             set_type_bit(T_Met(p), T_HAS_METHODS)
#define clear_has_methods(p)           clear_type_bit(T_Met(p), T_HAS_METHODS)
/* this marks an environment or closure that is "open" for generic functions etc, don't reuse this bit */

#define T_ITER_OK                      (1LL << (TYPE_BITS + 23))
#define iter_ok(p)                     has_type_bit(T_Pos(p), T_ITER_OK) /* not TItr(p) here because this bit is globally unique */
#define clear_iter_ok(p)               clear_type_bit(T_Itr(p), T_ITER_OK)

/* its faster here to use the high_flag bits rather than typeflag bits */
#define BIT_ROOM                       16
#define T_FULL_SYMCONS                 (1LL << (TYPE_BITS + BIT_ROOM + 24))
#define T_SYMCONS                      (1 << 0)
#define is_possibly_constant(p)        has_type1_bit(T_Sym(p), T_SYMCONS)
#define set_possibly_constant(p)       set_type1_bit(T_Sym(p), T_SYMCONS)

#define T_HAS_LET_ARG                  T_SYMCONS
#define has_let_arg(p)                 has_type1_bit(T_App(p), T_HAS_LET_ARG) /* was T_Pos 1-Nov-18 */
#define set_has_let_arg(p)             set_type1_bit(T_App(p), T_HAS_LET_ARG)
/* p is a setter procedure, "let arg" refers to the setter's optional third (let) argument */

#define T_FULL_S7_LET_FIELD            (1LL << (TYPE_BITS + BIT_ROOM + 25))
#define T_S7_LET_FIELD                 (1 << 1)
#define is_s7_let_field(p)             has_type1_bit(T_Sym(p), T_S7_LET_FIELD)
#define set_s7_let_field(p)            set_type1_bit(T_Sym(p), T_S7_LET_FIELD)

#define T_HAS_LET_FILE                 T_S7_LET_FIELD
#define has_let_file(p)                has_type1_bit(T_Let(p), T_HAS_LET_FILE)
#define set_has_let_file(p)            set_type1_bit(T_Let(p), T_HAS_LET_FILE)
#define clear_has_let_file(p)          clear_type1_bit(T_Let(p), T_HAS_LET_FILE)

#define T_HAS_OPTLIST                  T_S7_LET_FIELD
#define has_optlist(p)                 has_type1_bit(T_Pair(p), T_HAS_OPTLIST)
#define set_has_optlist(p)             set_type1_bit(T_Pair(p), T_HAS_OPTLIST)

#define T_TYPED_VECTOR                 T_S7_LET_FIELD
#define is_typed_vector(p)             has_type1_bit(T_Vec(p), T_TYPED_VECTOR)
#define set_typed_vector(p)            set_type1_bit(T_Vec(p), T_TYPED_VECTOR)

#define T_TYPED_HASH_TABLE             T_S7_LET_FIELD
#define is_typed_hash_table(p)         has_type1_bit(T_Hsh(p), T_TYPED_HASH_TABLE)
#define set_typed_hash_table(p)        set_type1_bit(T_Hsh(p), T_TYPED_HASH_TABLE)

#define T_BOOL_SETTER                  T_S7_LET_FIELD
#define c_function_has_bool_setter(p)  has_type1_bit(T_Fnc(p), T_BOOL_SETTER)
#define c_function_set_has_bool_setter(p) set_type1_bit(T_Fnc(p), T_BOOL_SETTER)

#define T_FULL_DEFINER                 (1LL << (TYPE_BITS + BIT_ROOM + 26))
#define T_DEFINER                      (1 << 2)
#define is_definer(p)                  has_type1_bit(T_Sym(p), T_DEFINER)
#define set_is_definer(p)              set_type1_bit(T_Sym(p), T_DEFINER)
/* this marks "definers" like define and define-macro */

#define T_HAS_FX                       T_DEFINER
#define set_has_fx(p)                  set_type1_bit(T_Pair(p), T_HAS_FX)
#define has_fx(p)                      has_type1_bit(T_Pair(p), T_HAS_FX)
#define clear_has_fx(p)                clear_type1_bit(T_Pair(p), T_HAS_FX)

#define T_SLOT_DEFAULTS                T_DEFINER
#define slot_defaults(p)               has_type1_bit(T_Slt(p), T_SLOT_DEFAULTS)
#define set_slot_defaults(p)           set_type1_bit(T_Slt(p), T_SLOT_DEFAULTS)

#define T_TREE_COLLECTED               (1LL << (TYPE_BITS + BIT_ROOM + 27))
#define T_SHORT_TREE_COLLECTED         (1 << 3)
#define is_tree_collected_or_shared(p) has_type_bit(T_Pair(p), (T_TREE_COLLECTED | T_SHARED))
#define set_tree_collected(p)          set_type1_bit(T_Pair(p), T_SHORT_TREE_COLLECTED)
#define clear_tree_bits(p)             clear_type_bit(T_Pair(p), T_TREE_COLLECTED | T_SHARED)

#define T_FULL_BINDER                  T_TREE_COLLECTED
#define T_BINDER                       T_SHORT_TREE_COLLECTED
#define is_binder(p)                   has_type1_bit(T_Sym(p), T_BINDER)
#define is_definer_or_binder(p)        has_type1_bit(T_Sym(p), T_DEFINER | T_BINDER)
#define set_is_binder(p)               set_type1_bit(T_Sym(p), T_BINDER)
/* this marks "binders" like let */

#define T_TYPE_INFO                    T_SHORT_TREE_COLLECTED
#define has_type_info(p)               has_type1_bit(T_Fnc(p), T_TYPE_INFO)
#define set_has_type_info(p)           set_type1_bit(T_Fnc(p), T_TYPE_INFO)

#define T_SIMPLE_VALUES                T_SHORT_TREE_COLLECTED
#define has_simple_values(p)           has_type1_bit(T_Hsh(p), T_SIMPLE_VALUES)
#define set_has_simple_values(p)       set_type1_bit(T_Hsh(p), T_SIMPLE_VALUES)

#define T_VERY_SAFE_CLOSURE            (1LL << (TYPE_BITS + BIT_ROOM + 28))
#define T_SHORT_VERY_SAFE_CLOSURE      (1 << 4)
#define is_very_safe_closure(p)        has_type1_bit(T_Clp(p), T_SHORT_VERY_SAFE_CLOSURE)
#define set_very_safe_closure(p)       set_type1_bit(T_Clp(p), T_SHORT_VERY_SAFE_CLOSURE) /* T_Pos 1-Nov-18 (can be a pair) */
#define closure_bits(p)                (typeflag(T_Clp(p)) & (T_SAFE_CLOSURE | T_VERY_SAFE_CLOSURE))

#define T_CYCLIC                       (1LL << (TYPE_BITS + BIT_ROOM + 29))
#define T_SHORT_CYCLIC                 (1 << 5)
#define is_cyclic(p)                   has_type1_bit(T_Seq(p), T_SHORT_CYCLIC) /* T_Pos 1-Nov-18 */
#define set_cyclic(p)                  set_type1_bit(T_Seq(p), T_SHORT_CYCLIC)

#define T_CYCLIC_SET                   (1LL << (TYPE_BITS + BIT_ROOM + 30))
#define T_SHORT_CYCLIC_SET             (1 << 6)
#define is_cyclic_set(p)               has_type1_bit(T_Pos(p), T_SHORT_CYCLIC_SET)
#define set_cyclic_set(p)              set_type1_bit(T_Seq(p), T_SHORT_CYCLIC_SET)  /* T_Pos 1-Nov-18 */
#define clear_cyclic_bits(p)           clear_type_bit(p, T_COLLECTED | T_SHARED | T_CYCLIC | T_CYCLIC_SET)

#define T_KEYWORD                      (1LL << (TYPE_BITS + BIT_ROOM + 31))
#define T_SHORT_KEYWORD                (1 << 7)
#define is_keyword(p)                  has_type1_bit(T_Pos(p), T_SHORT_KEYWORD)
/* this bit distinguishes a symbol from a symbol that is also a keyword */

#define T_FULL_SIMPLE_ELEMENTS         (1LL << (TYPE_BITS + BIT_ROOM + 32))
#define T_SIMPLE_ELEMENTS              (1 << 8)
#define has_simple_elements(p)         has_type1_bit(T_Nvc(p), T_SIMPLE_ELEMENTS)
#define set_has_simple_elements(p)     set_type1_bit(T_Nvc(p), T_SIMPLE_ELEMENTS)
#define c_function_has_simple_elements(p)     has_type1_bit(T_Fnc(p), T_SIMPLE_ELEMENTS)
#define c_function_set_has_simple_elements(p) set_type1_bit(T_Fnc(p), T_SIMPLE_ELEMENTS)

#define T_SIMPLE_KEYS                  T_SIMPLE_ELEMENTS
#define has_simple_keys(p)             has_type1_bit(T_Hsh(p), T_SIMPLE_KEYS)
#define set_has_simple_keys(p)         set_type1_bit(T_Hsh(p), T_SIMPLE_KEYS)

#define T_CTR3_SET                     T_SIMPLE_ELEMENTS
#define ctr3_is_set(p)                 has_type1_bit(T_Pair(p), T_CTR3_SET)
#define set_ctr3_is_set(p)             do {set_type1_bit(T_Pair(p), T_CTR3_SET); clear_type_bit(p, T_LINE_NUMBER);} while (0)

#define UNUSED_BITS                    0x3e00000000000000

#define T_GC_MARK                      0x8000000000000000
#define is_marked(p)                   has_type_bit(p, T_GC_MARK)
#define set_mark(p)                    set_type_bit(T_Pos(p), T_GC_MARK)
#define clear_mark(p)                  clear_type_bit(p, T_GC_MARK)
/* using the sign bit, bit 23 (or 55) == 31 (or 63) for this makes a big difference in the GC */

#define T_UNHEAP                       0x4000000000000000
#define T_SHORT_UNHEAP                 (1 << 14)
#define not_in_heap(p)                 has_type1_bit(T_Pos(p), T_SHORT_UNHEAP)
#define in_heap(p)                     (((T_Pos(p))->tf.opts.high_flag & T_SHORT_UNHEAP) == 0)
#define unheap(sc, p)                  set_type1_bit(T_Pos(p), T_SHORT_UNHEAP)

#define is_eof(p)                      ((T_Pos(p)) == eof_object)
#define is_undefined(p)                (type(p) == T_UNDEFINED)
#define is_true(Sc, p)                 ((T_Pos(p)) != Sc->F)
#define is_false(Sc, p)                ((T_Pos(p)) == Sc->F)

#ifdef _MSC_VER
  static s7_pointer make_boolean(s7_scheme *sc, bool val) {if (val) return(sc->T); return(sc->F);}
#else
  #define make_boolean(sc, Val)        ((Val) ? sc->T : sc->F)
#endif

#define is_pair(p)                     (type(p) == T_PAIR)
#define is_mutable_pair(p)             ((typeflag(T_Pos(p)) & (0xff | T_IMMUTABLE)) == T_PAIR)
#define is_null(p)                     ((T_Pos(p)) == sc->nil)
#define is_not_null(p)                 ((T_Pos(p)) != sc->nil)
#define is_list(p)                     ((is_pair(p)) || (type(p) == T_NIL))

#define raw_opt1(p)                    ((p)->object.cons.opt1)

#if (!S7_DEBUGGING)

#define opt1(p, r)                     ((p)->object.cons.opt1)
#define set_opt1(p, x, r)              (p)->object.cons.opt1 = x
#define opt2(p, r)                     ((p)->object.cons.opt2)
#define set_opt2(p, x, r)              (p)->object.cons.opt2 = (s7_pointer)(x)
#define opt3(p, r)                     ((p)->object.cons.opt3)
#define set_opt3(p, x, r)              do {(p)->object.cons.opt3 = x; clear_type_bit(p, T_LINE_NUMBER);} while (0)
/* 29-Oct-18 this used to clear T_OPTIMIZED -- optimize_op was sharing opt3 with line info */

#define pair_line(p)                   (p)->object.sym_cons.line
#define pair_set_line(p, X)            (p)->object.sym_cons.line = X
#define pair_file(p)                   (p)->object.sym_cons.file
#define pair_set_file(p, X)            (p)->object.sym_cons.file = X
#define pair_raw_hash(p)               (p)->object.sym_cons.hash
#define pair_set_raw_hash(p, X)        (p)->object.sym_cons.hash = X
#define pair_raw_len(p)                (p)->object.sym_cons.line
#define pair_set_raw_len(p, X)         (p)->object.sym_cons.line = X
#define pair_raw_name(p)               (p)->object.sym_cons.fstr
#define pair_set_raw_name(p, X)        (p)->object.sym_cons.fstr = X
/* opt1 == raw_hash, opt2 == raw_name, opt3 == line|ctr + len, but hash/name/len only apply to the symbol table so there's no collision */

#else

#define S_NAME                         (1 << 25)
#define S_HASH                         (1 << 26)
#define S_LINE                         (1 << 27)
#define S_LEN                          (1 << 28)

/* these 3 fields (or 8 counting sym_cons) hold most of the varigated optimizer info, so they are used in many conflicting ways.
 * the bits and funcs here try to track each such use, and report any cross-talk or collisions.
 * all of this machinery vanishes if debugging is turned off.
 */
#define E_SET                          (1 << 0)
#define E_FAST                         (1 << 7)   /* fast list in member/assoc circular list check */
#define E_CFUNC                        (1 << 8)   /* c-function */
#define E_CLAUSE                       (1 << 9)   /* case clause */
#define E_LAMBDA                       (1 << 11)  /* lambda(*) */
#define E_SYM                          (1 << 12)  /* symbol */
#define E_PAIR                         (1 << 13)  /* pair */
#define E_CON                          (1 << 14)  /* constant from eval's point of view */
#define E_GOTO                         (1 << 15)  /* call-with-exit exit func */
#define E_ANY                          (1 << 16)  /* anything -- deliberate unchecked case */
#define E_SLOT                         (1 << 17)  /* slot */
#define E_MASK                         (E_FAST | E_CFUNC | E_CLAUSE | E_LAMBDA | E_SYM | E_PAIR | E_CON | E_GOTO | E_ANY | E_SLOT | S_HASH)

#define opt1_is_set(p)                 (((p)->debugger_bits & E_SET) != 0)
#define set_opt1_is_set(p)             (p)->debugger_bits |= E_SET
#define opt1_role_matches(p, Role)     (((p)->debugger_bits & E_MASK) == Role)
#define set_opt1_role(p, Role)         (p)->debugger_bits = (Role | ((p)->debugger_bits & ~E_MASK))
#define opt1(p, Role)                  opt1_1(T_Pair(p), Role, __func__, __LINE__)
#define set_opt1(p, x, Role)           set_opt1_1(T_Pair(p), x, Role, __func__, __LINE__)

#define F_SET                          (1 << 1)
#define F_KEY                          (1 << 18)  /* case key */
#define F_SLOW                         (1 << 19)  /* slow list in member/assoc circular list check */
#define F_SYM                          (1 << 20)  /* symbol */
#define F_PAIR                         (1 << 21)  /* pair */
#define F_CON                          (1 << 22)  /* constant as above */
#define F_CALL                         (1 << 23)  /* c-func */
#define F_LAMBDA                       (1 << 24)  /* lambda form */
#define F_MASK                         (F_KEY | F_SLOW | F_SYM | F_PAIR | F_CON | F_CALL | F_LAMBDA | S_NAME)

#define opt2_is_set(p)                 (((p)->debugger_bits & F_SET) != 0)
#define set_opt2_is_set(p)             (p)->debugger_bits |= F_SET
#define opt2_role_matches(p, Role)     (((p)->debugger_bits & F_MASK) == Role)
#define set_opt2_role(p, Role)         (p)->debugger_bits = (Role | ((p)->debugger_bits & ~F_MASK))
#define opt2(p, Role)                  opt2_1(sc, T_Pair(p), Role, __func__, __LINE__)
#define set_opt2(p, x, Role)           set_opt2_1(sc, T_Pair(p), (s7_pointer)(x), Role, __func__, __LINE__)

#define G_SET                          (1 << 2)
#define G_ARGLEN                       (1 << 3)  /* arglist length */
#define G_SYM                          (1 << 4)  /* expression symbol access */
#define G_AND                          (1 << 5)  /* and second clause */
#define G_DIRECT                       (1 << 6)  /* direct call info */
#define G_ANY                          (1 << 29) 
#define G_CTR                          (1 << 30) 
#define G_CON                          0x80000000 /* not (1LL < 31) ! */
#define G_MASK                         (G_ARGLEN | G_SYM | G_AND | G_ANY | G_CTR | G_CON | S_LINE | S_LEN | G_DIRECT)

#define opt3_is_set(p)                 (((p)->debugger_bits & G_SET) != 0)
#define set_opt3_is_set(p)             (p)->debugger_bits |= G_SET
#define opt3_role_matches(p, Role)     (((p)->debugger_bits & G_MASK) == Role)
#define set_opt3_role(p, Role)         (p)->debugger_bits = (Role | ((p)->debugger_bits & ~G_MASK))
#define opt3(p, Role)                  opt3_1(T_Pair(p), Role, __func__, __LINE__)
#define set_opt3(p, x, Role)           set_opt3_1(T_Pair(p), x, Role, __func__, __LINE__)

#define pair_line(p)                   s_line_1(T_Pair(p), __func__, __LINE__)
#define pair_set_line(p, X)            set_s_line_1(T_Pair(p), X, __func__, __LINE__)
#define pair_file(p)                   (p)->object.sym_cons.file
#define pair_set_file(p, X)            set_s_file_1(sc, T_Pair(p), X, __func__, __LINE__)
#define pair_raw_hash(p)               s_hash_1(T_Pair(p), __func__, __LINE__)
#define pair_set_raw_hash(p, X)        set_s_hash_1(T_Pair(p), X, __func__, __LINE__)
#define pair_raw_len(p)                s_len_1(T_Pair(p), __func__, __LINE__)
#define pair_set_raw_len(p, X)         set_s_len_1(T_Pair(p), X, __func__, __LINE__)
#define pair_raw_name(p)               s_name_1(T_Pair(p), __func__, __LINE__)
#define pair_set_raw_name(p, X)        set_s_name_1(T_Pair(p), X, __func__, __LINE__)

#define L_HIT                          (1LL < 40) /* "L_SET" is taken */
#define L_FUNC                         (1LL < 41)
#define L_DOX                          (1LL < 42)
#define L_CATCH                        (1LL < 43)
#define L_MASK                         (L_FUNC | L_DOX | L_CATCH)
#endif

#define opt1_fast(P)                   T_Lst(opt1(P,                E_FAST))
#define set_opt1_fast(P, X)            set_opt1(P, T_Pair(X),       E_FAST)
#define opt1_cfunc(P)                  T_Pos(opt1(P,                E_CFUNC))
#define set_opt1_cfunc(P, X)           set_opt1(P, T_Pos(X),        E_CFUNC)
#define opt1_lambda_unchecked(P)       opt1(P,                      E_LAMBDA) /* can be free/null? from s7_call? */
#define opt1_lambda(P)                 T_Clo(opt1(P,                E_LAMBDA))
#define set_opt1_lambda(P, X)          set_opt1(P, T_Pos(X),        E_LAMBDA)
#define opt1_goto(P)                   T_Pos(opt1(P,                E_GOTO))  /* used when checking for non-goto unknown in eval, so can't be T_Got */
#define set_opt1_goto(P, X)            set_opt1(P, T_Pos(X),        E_GOTO)
#define opt1_clause(P)                 T_Pos(opt1(P,                E_CLAUSE))
#define set_opt1_clause(P, X)          set_opt1(P, T_Pos(X),        E_CLAUSE)
#define opt1_sym(P)                    T_Sym(opt1(P,                E_SYM))
#define set_opt1_sym(P, X)             set_opt1(P, T_Sym(X),        E_SYM)
#define opt1_pair(P)                   T_Lst(opt1(P,                E_PAIR))
#define set_opt1_pair(P, X)            set_opt1(P, T_Lst(X),        E_PAIR)
#define opt1_con(P)                    T_Pos(opt1(P,                E_CON))
#define set_opt1_con(P, X)             set_opt1(P, T_Pos(X),        E_CON)
#define opt1_any(P)                    opt1(P,                      E_ANY)    /* can be free in closure_is_ok */
#define set_opt1_any(P, X)             set_opt1(P, X,               E_ANY)
#define opt1_slot(P)                   T_Slt(opt1(P,                E_SLOT))
#define set_opt1_slot(P, X)            set_opt1(P, T_Slt(X),        E_SLOT)

#define opt2_any_unchecked(P)          P->object.cons.opt2
#define set_opt2_any_unchecked(P, X)   P->object.cons.opt2 = X
#define opt2_any(P)                    opt2(P,                      F_KEY)
#define set_opt2_any(P, X)             set_opt2(P, X,               F_KEY)
#define opt2_slow(P)                   T_Lst(opt2(P,                F_SLOW))
#define set_opt2_slow(P, X)            set_opt2(P, T_Pair(X),       F_SLOW)
#define opt2_sym(P)                    T_Sym(opt2(P,                F_SYM))
#define set_opt2_sym(P, X)             set_opt2(P, T_Sym(X),        F_SYM)
#define opt2_pair(P)                   T_Lst(opt2(P,                F_PAIR))
#define set_opt2_pair(P, X)            set_opt2(P, T_Lst(X),        F_PAIR)
#define opt2_con(P)                    T_Pos(opt2(P,                F_CON))
#define set_opt2_con(P, X)             set_opt2(P, T_Pos(X),        F_CON)
#define opt2_lambda(P)                 T_Pair(opt2(P,               F_LAMBDA))
#define set_opt2_lambda(P, X)          set_opt2(P, T_Pair(X),       F_LAMBDA)
#define opt2_direct_x_call(P)          opt2(P,                      F_LAMBDA)
#define set_opt2_direct_x_call(P, X)   set_opt2(P, (s7_pointer)(X), F_LAMBDA)

#define opt3_arglen(P)                 T_Int(opt3(cdr(P),           G_ARGLEN))
#define set_opt3_arglen(P, X)          set_opt3(cdr(P), T_Int(X),   G_ARGLEN)
#define opt3_sym(P)                    T_Sym(opt3(P,                G_SYM))
#define set_opt3_sym(P, X)             set_opt3(P, T_Sym(X),        G_SYM)
#define opt3_pair(P)                   T_Pair(opt3(P,               G_AND))
#define set_opt3_pair(P, X)            set_opt3(P, T_Pair(X),       G_AND)
#define opt3_any(P)                    opt3(P,                      G_ANY)
#define set_opt3_any(P, X)             set_opt3(P, X,               G_ANY)
#define opt3_direct_x(P)               opt3(P,                      G_DIRECT)
#define set_opt3_direct_x(P, X)        set_opt3(P, (s7_pointer)(X), G_DIRECT)

#if S7_DEBUGGING
#define opt3_con(p)                    opt3_con_1(T_Pair(p),           G_CON, __func__, __LINE__)
#define set_opt3_con(p, x)             set_opt3_con_1(T_Pair(p), x,    G_CON, __func__, __LINE__)
#define opt3_ctr(p)                    opt3_ctr_1(T_Pair(p),           G_CTR, __func__, __LINE__)
#define set_opt3_ctr(p, x)             set_opt3_ctr_1(T_Pair(p), x,    G_CTR, __func__, __LINE__)
#define increment_opt3_ctr(p)          increment_opt3_ctr_1(T_Pair(p), G_CTR, __func__, __LINE__)
#else
#define opt3_con(P)                    T_Pair(P)->object.cons_ext.ce.opt_type /* op_if_is_type */
#define set_opt3_con(P, X)             do {T_Pair(P)->object.cons_ext.ce.opt_type = X; clear_type_bit(P, T_LINE_NUMBER);} while (0)
#define opt3_ctr(P)                    T_Pair(P)->object.cons_ext.ce.ctr
#define set_opt3_ctr(P, X)             do {T_Pair(P)->object.cons_ext.ce.ctr = X; set_ctr3_is_set(P);} while(0)
#define increment_opt3_ctr(P)          do {if (ctr3_is_set(P)) P->object.cons_ext.ce.ctr++; else set_opt3_ctr(P, 0);} while (0)
#endif

#define c_callee(f)                    ((s7_function)opt2(f,      F_CALL))
#define c_call(f)                      ((s7_function)opt2(f,      F_CALL))
#define set_c_call_checked(f, _X_)     do {s7_pointer X; X = (s7_pointer)(_X_); set_opt2(f, X, F_CALL); if (X) set_has_fx(f); else clear_has_fx(f);} while (0)
#if S7_DEBUGGING
  #define set_c_call(f, _X_)           do {s7_pointer X; X = (s7_pointer)(_X_); if (!(X)) fprintf(stderr, "%s[%d] x_call null: %s\n", __func__, __LINE__, DISPLAY(f)); set_opt2(f, X, F_CALL); if (X) set_has_fx(f); else clear_has_fx(f);} while (0)
#else
  #define set_c_call(f, X)             do {set_opt2(f, (s7_pointer)(X), F_CALL); set_has_fx(f);} while (0)
#endif
#define set_c_call_direct(f, X)        set_opt2(f, (s7_pointer)(X), F_CALL)
#define set_c_call_unchecked(f, _X_)   do {s7_pointer X; X = (s7_pointer)(_X_); set_opt2(f, X, F_CALL); if (X) set_has_fx(f); else clear_has_fx(f);} while (0)


#define car(p)                         (T_Pair(p))->object.cons.car
#define set_car(p, Val)                (T_Pair(p))->object.cons.car = T_Pos(Val)
#define cdr(p)                         (T_Pair(p))->object.cons.cdr
#define set_cdr(p, Val)                (T_Pair(p))->object.cons.cdr = T_Pos(Val)
#define unchecked_car(p)               (T_Pos(p))->object.cons.car
#define unchecked_cdr(p)               (T_Pos(p))->object.cons.cdr

#define caar(p)                        car(car(p))
#define cadr(p)                        car(cdr(p))
#define set_cadr(p, Val)               (T_Pair(p))->object.cons.cdr->object.cons.car = T_Pos(Val)
#define cdar(p)                        cdr(car(p))
#define set_cdar(p, Val)               (T_Pair(p))->object.cons.car->object.cons.cdr = T_Pos(Val)
#define cddr(p)                        cdr(cdr(p))

#define caaar(p)                       car(car(car(p)))
#define cadar(p)                       car(cdr(car(p)))
#define cdadr(p)                       cdr(car(cdr(p)))
#define caddr(p)                       car(cdr(cdr(p)))
#define set_caddr(p, Val)              (T_Pair(p))->object.cons.cdr->object.cons.cdr->object.cons.car = T_Pos(Val)
#define caadr(p)                       car(car(cdr(p)))
#define cdaar(p)                       cdr(car(car(p)))
#define cdddr(p)                       cdr(cdr(cdr(p)))
#define cddar(p)                       cdr(cdr(car(p)))

#define caaadr(p)                      car(car(car(cdr(p))))
#define caadar(p)                      car(car(cdr(car(p))))
#define cadaar(p)                      car(cdr(car(car(p))))
#define cadddr(p)                      car(cdr(cdr(cdr(p))))
#define caaddr(p)                      car(car(cdr(cdr(p))))
#define cddddr(p)                      cdr(cdr(cdr(cdr(p))))
#define caddar(p)                      car(cdr(cdr(car(p))))
#define cdadar(p)                      cdr(car(cdr(car(p))))
#define cdaddr(p)                      cdr(car(cdr(cdr(p))))
#define caaaar(p)                      car(car(car(car(p))))
#define cadadr(p)                      car(cdr(car(cdr(p))))
#define cdaadr(p)                      cdr(car(car(cdr(p))))
#define cdaaar(p)                      cdr(car(car(car(p))))
#define cdddar(p)                      cdr(cdr(cdr(car(p))))
#define cddadr(p)                      cdr(cdr(car(cdr(p))))
#define cddaar(p)                      cdr(cdr(car(car(p))))

#if WITH_GCC
  /* slightly tricky because cons can be called recursively */
  #define cons(Sc, A, B)   ({s7_pointer _X_, _A_, _B_; _A_ = A; _B_ = B; new_cell(Sc, _X_, T_PAIR | T_SAFE_PROCEDURE); set_car(_X_, _A_); set_cdr(_X_, _B_); _X_;})
#else
  #define cons(Sc, A, B)               s7_cons(Sc, A, B)
#endif

#define list_1(Sc, A)                  cons(Sc, A, Sc->nil)
#define list_2(Sc, A, B)               cons_unchecked(Sc, A, cons(Sc, B, Sc->nil))
#define list_3(Sc, A, B, C)            cons_unchecked(Sc, A, cons_unchecked(Sc, B, cons(Sc, C, Sc->nil)))
#define list_4(Sc, A, B, C, D)         cons_unchecked(Sc, A, cons_unchecked(Sc, B, cons_unchecked(Sc, C, cons(Sc, D, Sc->nil))))

#define is_string(p)                   (type(p) == T_STRING)
#define is_mutable_string(p)           ((typeflag(T_Pos(p)) & (0xff | T_IMMUTABLE)) == T_STRING)
#define string_value(p)                (T_Str(p))->object.string.svalue
#define string_length(p)               (T_Str(p))->object.string.length
#define string_hash(p)                 (T_Str(p))->object.string.hash
#define string_block(p)                (T_Str(p))->object.string.block

#define character(p)                   (T_Chr(p))->object.chr.c
#define upper_character(p)             (T_Chr(p))->object.chr.up_c
#define is_char_alphabetic(p)          (T_Chr(p))->object.chr.alpha_c
#define is_char_numeric(p)             (T_Chr(p))->object.chr.digit_c
#define is_char_whitespace(p)          (T_Chr(p))->object.chr.space_c
#define is_char_uppercase(p)           (T_Chr(p))->object.chr.upper_c
#define is_char_lowercase(p)           (T_Chr(p))->object.chr.lower_c
#define character_name(p)              (T_Chr(p))->object.chr.c_name
#define character_name_length(p)       (T_Chr(p))->object.chr.length

#define optimize_op(P)                 (P)->tf.opts.opt_choice
#define set_optimize_op(P, Op)         (P)->tf.opts.opt_choice = Op
#define optimize_op_match(P, Q)        ((is_optimized(P)) && ((optimize_op(P) & 0xfffe) == Q))
#define op_no_hop(P)                   (optimize_op(P) & 0xfffe)
#define clear_hop(P)                   set_optimize_op(P, op_no_hop(P))
#define clear_optimize_op(P)           set_optimize_op(P, 0)
#define set_safe_optimize_op(P, Q)     do {set_optimized(P); set_optimize_op(P, Q);} while (0)
#define set_unsafe_optimize_op(P, Q)   do {set_unsafely_optimized(P); set_optimize_op(P, Q);} while (0)

#define is_symbol(p)                   (type(p) == T_SYMBOL)
#define is_normal_symbol(p)            ((is_symbol(p)) && (!is_keyword(p)))
#define is_safe_symbol(p)              ((is_symbol(p)) && (is_slot(symbol_to_slot(sc, p))))
#define symbol_name_cell(p)            T_Str((T_Sym(p))->object.sym.name)
#define symbol_set_name_cell(p, S)     (T_Sym(p))->object.sym.name = T_Str(S)
#define symbol_name(p)                 string_value(symbol_name_cell(p))
#define symbol_name_length(p)          string_length(symbol_name_cell(p))
#define gensym_block(p)                symbol_name_cell(p)->object.string.gensym_block
#define symbol_hmap(p)                 (s7_int)((intptr_t)(p) >> 8)
#define symbol_id(p)                   (T_Sym(p))->object.sym.id
#define symbol_set_id_unchecked(p, X)  (T_Sym(p))->object.sym.id = X
#if S7_DEBUGGING
static void symbol_set_id(s7_pointer p, s7_int id)
{
  if (id < symbol_id(p))
    {
      fprintf(stderr, "id mismatch: sym: %s %" print_s7_int ", let: %" print_s7_int "\n", symbol_name(p), symbol_id(p), id);
      abort();
    }
  (T_Sym(p))->object.sym.id = id;
}
#else
#define symbol_set_id(p, X)            (T_Sym(p))->object.sym.id = X
#endif
/* we need 64-bits here, since we don't want this thing to wrap around, and frames are created at a great rate
 *    callgrind says this is faster than an uint32_t!
 */
#define symbol_info(p)                 (symbol_name_cell(p))->object.string.block
#define symbol_type(p)                 block_size(symbol_info(p))
#define symbol_set_type(p, Type)       block_set_size(symbol_info(p), Type)
#define initial_slot(p)                symbol_info(p)->ex.ex_ptr
#define set_initial_slot(p, Val)       symbol_info(p)->ex.ex_ptr = T_Sld(Val)

#define global_slot(p)                 (T_Sym(p))->object.sym.global_slot
#define set_global_slot(p, Val)        (T_Sym(p))->object.sym.global_slot = T_Sld(Val)
#define local_slot(p)                  (T_Sym(p))->object.sym.local_slot
#define set_local_slot(p, Val)         (T_Sym(p))->object.sym.local_slot = T_Sln(Val)
#define keyword_symbol(p)              symbol_info(p)->nx.ksym               /* keyword only, so does not collide with documentation */
#define keyword_set_symbol(p, Val)     symbol_info(p)->nx.ksym = T_Sym(Val)
#define symbol_help(p)                 symbol_info(p)->nx.documentation
#define symbol_set_help(p, Doc)        symbol_info(p)->nx.documentation = Doc
#define symbol_tag(p)                  (T_Sym(p))->object.sym.tag
#define symbol_set_tag(p, Val)         (T_Sym(p))->object.sym.tag = Val
#define symbol_ctr(p)                  (T_Sym(p))->object.sym.ctr
#define symbol_set_ctr(p, Val)         (T_Sym(p))->object.sym.ctr = Val
#define symbol_increment_ctr(p)        (T_Sym(p))->object.sym.ctr++
#define symbol_tag2(p)                 symbol_info(p)->ln.tag
#define symbol_set_tag2(p, Val)        symbol_info(p)->ln.tag = Val
#define symbol_has_help(p)             (is_documented(symbol_name_cell(p)))
#define symbol_set_has_help(p)         set_documented(symbol_name_cell(p))

#define symbol_set_local_unchecked(Symbol, Id, Slot) do {set_local_slot(Symbol, Slot); symbol_set_id_unchecked(Symbol, Id); symbol_increment_ctr(Symbol);} while (0)
#define symbol_set_local(Symbol, Id, Slot) do {set_local_slot(Symbol, Slot); symbol_set_id(Symbol, Id); symbol_increment_ctr(Symbol);} while (0)
/* set slot before id in case Slot is an expression that tries to find the current Symbol slot (using its old Id obviously) */

#define is_slot(p)                     (type(p) == T_SLOT)
#define slot_symbol(p)                 T_Sym((T_Slt(p))->object.slt.sym)
#define slot_set_symbol(p, Sym)        (T_Slt(p))->object.slt.sym = T_Sym(Sym)
#define slot_value(p)                  T_Pos((T_Slt(p))->object.slt.val)
#define unchecked_slot_value(p)        p->object.slt.val
#define slot_set_value(p, Val)         (T_Slt(p))->object.slt.val = T_Pos(Val)
#define slot_set_value_with_hook(Slot, Value) \
  do {if (hook_has_functions(sc->rootlet_redefinition_hook)) slot_set_value_with_hook_1(sc, Slot, Value); else slot_set_value(Slot, Value);} while (0)
#define next_slot(p)                   (T_Slt(p))->object.slt.nxt
#define set_next_slot(p, Val)          (T_Slt(p))->object.slt.nxt = T_Sln(Val)
#define slot_set_pending_value(p, Val) do {(T_Slt(p))->object.slt.pending_value = T_Pos(Val); slot_set_has_pending_value(p);} while (0)
#if S7_DEBUGGING
static s7_pointer slot_pending_value(s7_pointer p) {if (slot_has_pending_value(p)) return(p->object.slt.pending_value); fprintf(stderr, "slot: no pending value\n"); abort();}
static s7_pointer slot_expression(s7_pointer p)    {if (slot_has_expression(p)) return(p->object.slt.expr); fprintf(stderr, "slot: no expression\n"); abort();}
#else
#define slot_pending_value(p)          (T_Slt(p))->object.slt.pending_value
#define slot_expression(p)             (T_Slt(p))->object.slt.expr
#endif
#define slot_set_expression(p, Val)    do {(T_Slt(p))->object.slt.expr = T_Pos(Val); slot_set_has_expression(p);} while (0)
#define slot_just_set_expression(p, Val) (T_Slt(p))->object.slt.expr = T_Pos(Val)
#define slot_setter(p)                 (T_Slt(p))->object.slt.expr
#define slot_set_setter_1(p, Val)      (T_Slt(p))->object.slt.expr = T_App(Val)

#define is_syntax(p)                   (type(p) == T_SYNTAX)
#define syntax_symbol(p)               T_Sym((T_Syn(p))->object.syn.symbol)
#define syntax_set_symbol(p, Sym)      (T_Syn(p))->object.syn.symbol = T_Sym(Sym)
#define syntax_opcode(p)               (T_Syn(p))->object.syn.op
#define syntax_min_args(p)             (T_Syn(p))->object.syn.min_args
#define syntax_max_args(p)             (T_Syn(p))->object.syn.max_args
#define syntax_documentation(p)        (T_Syn(p))->object.syn.documentation

#define set_syntactic_pair(p)          typeflag(T_Pair(p)) = (T_PAIR | T_SYNTACTIC | (typeflag(p) & (0xffffffffffff0000 & ~T_OPTIMIZED)))
#define pair_set_syntax_op(p, X)       do {set_optimize_op(p, X); set_syntactic_pair(p);} while (0)
#define symbol_syntax_op_checked(p)    ((is_syntactic_pair(p)) ? optimize_op(p) : symbol_syntax_op(car(p)))
#define symbol_syntax_op(p)            syntax_opcode(slot_value(global_slot(p)))

#define ROOTLET_SIZE                   512
#define let_id(p)                      (T_Lid(p))->object.envr.id
#define is_let(p)                      (type(p) == T_LET)
#define let_slots(p)                   (T_Let(p))->object.envr.slots
#define let_set_slots(p, Slot)         (T_Let(p))->object.envr.slots = T_Sln(Slot)
#define outlet(p)                      (T_Let(p))->object.envr.nxt
#define set_outlet(p, ol)              (T_Let(p))->object.envr.nxt = T_Lid(ol)
#if S7_DEBUGGING
#define C_Let(p, role)                 check_let_ref(p, role, __func__, __LINE__)
#define S_Let(p, role)                 check_let_set(p, role, __func__, __LINE__)
#else
#define C_Let(p, role)                 p
#define S_Let(p, role)                 p
#endif
#define funclet_function(p)            T_Sym((C_Let(p, L_FUNC))->object.envr.edat.efnc.function)
#define funclet_set_function(p, F)     (S_Let(p, L_FUNC))->object.envr.edat.efnc.function = T_Sym(F)
#define let_line(p)                    (C_Let(p, L_FUNC))->object.envr.edat.efnc.line
#define let_set_line(p, L)             (S_Let(p, L_FUNC))->object.envr.edat.efnc.line = L
#define let_file(p)                    (C_Let(p, L_FUNC))->object.envr.edat.efnc.file
#define let_set_file(p, F)             (S_Let(p, L_FUNC))->object.envr.edat.efnc.file = F
#define dox_slot1(p)                   T_Slt((C_Let(p, L_DOX))->object.envr.edat.dox.dox1)
#define dox_set_slot1(p, S)            (S_Let(p, L_DOX))->object.envr.edat.dox.dox1 = T_Slt(S)
#define dox_slot2(p)                   T_Slt((C_Let(p, L_DOX))->object.envr.edat.dox.dox2)
#define dox_set_slot2(p, S)            (S_Let(p, L_DOX))->object.envr.edat.dox.dox2 = T_Slt(S)
#define dox_slot2_unchecked(p)         C_Let(p, L_DOX)->object.envr.edat.dox.dox2
#define dox_set_slot2_unchecked(p, S)  S_Let(p, L_DOX)->object.envr.edat.dox.dox2 = (S)

#define unique_name(p)                 (p)->object.unq.nm.name
#define unique_name_length(p)          (p)->object.unq.len
#define unknown_name(p)                (p)->object.unq.nm.unknown_name
#define is_unspecified(p)              (type(p) == T_UNSPECIFIED)
#define unique_car(p)                  (p)->object.unq.car
#define unique_cdr(p)                  (p)->object.unq.cdr

#define is_any_vector(p)               t_vector_p[type(p)]
#define is_normal_vector(p)            (type(p) == T_VECTOR)
#define vector_length(p)               (p)->object.vector.length
#define unchecked_vector_elements(p)   (p)->object.vector.elements.objects
#define unchecked_vector_element(p, i) ((p)->object.vector.elements.objects[i])
#define vector_element(p, i)           ((T_Vec(p))->object.vector.elements.objects[i])
#define vector_elements(p)             (T_Vec(p))->object.vector.elements.objects
#define vector_getter(p)               (T_Vec(p))->object.vector.vget
#define vector_setter(p)               (T_Vec(p))->object.vector.setv.vset
#define typed_vector_typer(p)          (T_Vec(p))->object.vector.setv.fset
#define vector_block(p)                (T_Vec(p))->object.vector.block
#define unchecked_vector_block(p)      p->object.vector.block

#define is_int_vector(p)               (type(p) == T_INT_VECTOR)
#define int_vector(p, i)               ((T_Ivc(p))->object.vector.elements.ints[i])
#define int_vector_ints(p)             (T_Ivc(p))->object.vector.elements.ints

#define is_float_vector(p)             (type(p) == T_FLOAT_VECTOR)
#define float_vector(p, i)             ((T_Fvc(p))->object.vector.elements.floats[i])
#define float_vector_floats(p)         (T_Fvc(p))->object.vector.elements.floats

#define is_byte_vector(p)              (type(p) == T_BYTE_VECTOR)
#define is_mutable_byte_vector(p)      ((typeflag(T_Pos(p)) & (0xff | T_IMMUTABLE)) == T_BYTE_VECTOR)
#define byte_vector_length(p)          (T_BVc(p))->object.vector.length
#define byte_vector_bytes(p)           (T_BVc(p))->object.vector.elements.bytes
#define byte_vector(p, i)              ((T_BVc(p))->object.vector.elements.bytes[i])
#define is_string_or_byte_vector(p)    ((type(p) == T_STRING) || (type(p) == T_BYTE_VECTOR))

#define vector_dimension_info(p)       ((vdims_t *)(T_Vec(p))->object.vector.block->ex.ex_info)
#define vector_set_dimension_info(p, d) (T_Vec(p))->object.vector.block->ex.ex_info = (void  *)d
#define vector_ndims(p)                vdims_rank(vector_dimension_info(p))
#define vector_dimension(p, i)         vdims_dims(vector_dimension_info(p))[i]
#define vector_dimensions(p)           vdims_dims(vector_dimension_info(p))
#define vector_offset(p, i)            vdims_offsets(vector_dimension_info(p))[i]
#define vector_offsets(p)              vdims_offsets(vector_dimension_info(p))
#define vector_rank(p)                 ((vector_dimension_info(p)) ? vector_ndims(p) : 1)
#define vector_has_dimensional_info(p) (vector_dimension_info(p))

#define subvector_vector(p)            ((vector_dimension_info(p)) ? vdims_original(vector_dimension_info(p)) : (T_Vec(p))->object.vector.block->nx.ksym)
#define subvector_set_vector(p, vect)  (T_Vec(p))->object.vector.block->nx.ksym = vect

#define rootlet_element(p, i)          unchecked_vector_element(p, i)
#define rootlet_elements(p)            unchecked_vector_elements(p)
#define rootlet_block(p)               unchecked_vector_block(p)
#define stack_element(p, i)            unchecked_vector_element(T_Stk(p), i)
#define stack_elements(p)              unchecked_vector_elements(T_Stk(p))
#define stack_block(p)                 unchecked_vector_block(T_Stk(p))

#define is_hash_table(p)               (type(p) == T_HASH_TABLE)
#define is_mutable_hash_table(p)       ((typeflag(T_Pos(p)) & (0xff | T_IMMUTABLE)) == T_HASH_TABLE)
#define hash_table_mask(p)             (T_Hsh(p))->object.hasher.mask
#define hash_table_block(p)            (T_Hsh(p))->object.hasher.block
#define hash_table_set_block(p, b)     (T_Hsh(p))->object.hasher.block = b
#define hash_table_element(p, i)       (T_Hsh(p))->object.hasher.elements[i]
#define hash_table_elements(p)         (T_Hsh(p))->object.hasher.elements
#define hash_table_entries(p)          hash_table_block(p)->nx.nx_int
#define hash_table_checker(p)          (T_Hsh(p))->object.hasher.hash_func
#define hash_table_mapper(p)           (T_Hsh(p))->object.hasher.loc
#define hash_table_checker_locked(p)   (hash_table_mapper(p) != default_hash_map)
#define hash_table_procedures(p)       T_Lst(hash_table_block(p)->ex.ex_ptr)
#define hash_table_set_procedures(p, Lst) hash_table_block(p)->ex.ex_ptr = T_Lst(Lst)
#define hash_table_procedures_checker(p)  car(hash_table_procedures(p))
#define hash_table_procedures_mapper(p)   cdr(hash_table_procedures(p))
#define hash_table_key_typer(p)           opt1_any(hash_table_procedures(p))
#define hash_table_set_key_typer(p, Fnc)  set_opt1_any(p, Fnc)
#define hash_table_value_typer(p)         opt2_any(hash_table_procedures(p))
#define hash_table_set_value_typer(p, Fnc) set_opt2_any(p, Fnc)

#if S7_DEBUGGING
#define T_Itr_Pos(p)                   titr_pos(sc, T_Itr(p), __func__, __LINE__)
#define T_Itr_Len(p)                   titr_len(T_Itr(p), __func__, __LINE__)
#define T_Itr_Hash(p)                  titr_hash(T_Itr(p), __func__, __LINE__)
#define T_Itr_Let(p)                   titr_let(T_Itr(p), __func__, __LINE__)
#define T_Itr_Pair(p)                  titr_pair(T_Itr(p), __func__, __LINE__)
#else
#define T_Itr_Pos(p)                   p
#define T_Itr_Len(p)                   p
#define T_Itr_Hash(p)                  p
#define T_Itr_Let(p)                   p
#define T_Itr_Pair(p)                  p
#endif

#define is_iterator(p)                 (type(p) == T_ITERATOR)
#define iterator_sequence(p)           (T_Itr(p))->object.iter.obj
#define iterator_position(p)           (T_Itr_Pos(p))->object.iter.lc.loc
#define iterator_length(p)             (T_Itr_Len(p))->object.iter.lw.len
#define iterator_next(p)               (T_Itr(p))->object.iter.next
#define iterator_is_at_end(p)          ((typeflag(T_Itr(p)) & T_ITER_OK) == 0)
#define iterator_slow(p)               T_Lst((T_Itr_Pair(p))->object.iter.lw.slow)
#define iterator_set_slow(p, Val)      (T_Itr_Pair(p))->object.iter.lw.slow = T_Lst(Val)
#define iterator_hash_current(p)       (T_Itr_Hash(p))->object.iter.lw.hcur
#define iterator_current(p)            (T_Itr(p))->object.iter.cur
#define iterator_current_slot(p)       T_Sln((T_Itr_Let(p))->object.iter.lc.lcur)
#define iterator_set_current_slot(p, Val) (T_Itr_Let(p))->object.iter.lc.lcur = T_Sln(Val)
#define iterator_let_cons(p)           (T_Itr_Let(p))->object.iter.cur

#define ITERATOR_END                   eof_object
#define ITERATOR_END_NAME              "#<eof>"

#define is_input_port(p)               (type(p) == T_INPUT_PORT)
#define is_output_port(p)              (type(p) == T_OUTPUT_PORT)
#define port_port(p)                   (T_Prt(p))->object.prt.port
#define is_string_port(p)              (port_type(p) == STRING_PORT)
#define is_file_port(p)                (port_type(p) == FILE_PORT)
#define is_function_port(p)            (port_type(p) == FUNCTION_PORT)
#define port_filename_block(p)         port_port(p)->filename_block
#define port_filename(p)               port_port(p)->filename
#define port_filename_length(p)        port_port(p)->filename_length
#define port_file(p)                   port_port(p)->file
#define port_data_block(p)             port_port(p)->block
#define port_line_number(p)            port_port(p)->line_number
#define port_file_number(p)            port_port(p)->file_number
#define port_data(p)                   (T_Prt(p))->object.prt.data
#define port_data_size(p)              (T_Prt(p))->object.prt.size
#define port_position(p)               (T_Prt(p))->object.prt.point
#define port_block(p)                  (T_Prt(p))->object.prt.block
#define port_type(p)                   port_port(p)->ptype
#define port_is_closed(p)              port_port(p)->is_closed
#define port_set_closed(p, Val)        port_port(p)->is_closed = Val /* this can't be a type bit because sweep checks it after the type has been cleared */
#define port_needs_free(p)             port_port(p)->needs_free
#define port_next(p)                   port_block(p)->nx.next
#define port_output_function(p)        port_port(p)->output_function /* these two are for function ports */
#define port_input_function(p)         port_port(p)->input_function
#define port_original_input_string(p)  port_port(p)->orig_str
#define port_read_character(p)         port_port(p)->read_character
#define port_read_line(p)              port_port(p)->read_line
#define port_display(p)                port_port(p)->display
#define port_write_character(p)        port_port(p)->write_character
#define port_write_string(p)           port_port(p)->write_string
#define port_read_semicolon(p)         port_port(p)->read_semicolon
#define port_read_white_space(p)       port_port(p)->read_white_space
#define port_read_name(p)              port_port(p)->read_name
#define port_read_sharp(p)             port_port(p)->read_sharp
#define port_gc_loc(p)                 port_port(p)->gc_loc
#define port_needs_unprotect(p)        port_port(p)->needs_unprotect

#define is_c_function(f)               (type(f) >= T_C_FUNCTION)
#define is_c_function_star(f)          (type(f) == T_C_FUNCTION_STAR)
#define is_any_c_function(f)           (type(f) >= T_C_FUNCTION_STAR)
#define c_function_data(f)             (T_Fnc(f))->object.fnc.c_proc
#define c_function_call(f)             (T_Fnc(f))->object.fnc.ff
#define c_function_required_args(f)    (T_Fnc(f))->object.fnc.required_args
#define c_function_optional_args(f)    (T_Fnc(f))->object.fnc.optional_args
#define c_function_all_args(f)         (T_Fnc(f))->object.fnc.all_args
#define c_function_name(f)             c_function_data(f)->name
#define c_function_name_length(f)      c_function_data(f)->name_length
#define c_function_documentation(f)    c_function_data(f)->doc
#define c_function_signature(f)        c_function_data(f)->signature
#define c_function_setter(f)           T_App(c_function_data(f)->setter)
#define c_function_set_setter(f, Val)  c_function_data(f)->setter = T_App(Val)
#define c_function_block(f)            (f)->object.fnc.c_proc->block /* no type checking here */
#define c_function_class(f)            c_function_data(f)->id
#define c_function_chooser(f)          c_function_data(f)->chooser
#define c_function_base(f)             T_App(c_function_data(f)->generic_ff)
#define c_function_set_base(f, Val)    c_function_data(f)->generic_ff = T_App(Val)
#define c_function_marker(f)           c_function_data(f)->cam.marker
#define c_function_set_marker(f, Val)  c_function_data(f)->cam.marker = Val
#define c_function_symbol(f)           c_function_data(f)->sam.c_sym

#define c_function_bool_setter(f)      c_function_data(f)->dam.bool_setter
#define c_function_set_bool_setter(f, Val) c_function_data(f)->dam.bool_setter = Val
#define c_function_arg_defaults(f)     c_function_data(T_Fst(f))->dam.arg_defaults
#define c_function_call_args(f)        c_function_data(T_Fst(f))->cam.call_args
#define c_function_arg_names(f)        c_function_data(T_Fst(f))->sam.arg_names

#define set_c_function(X, f)           do {set_opt1_cfunc(X, f); set_c_call_direct(X, c_function_call(f));} while (0)
#define c_function_opt_data(f)         c_function_data(f)->opt_data

#define is_c_macro(p)                  (type(p) == T_C_MACRO)
#define c_macro_data(f)                (T_Mac(f))->object.fnc.c_proc
#define c_macro_call(f)                (T_Mac(f))->object.fnc.ff
#define c_macro_name(f)                c_macro_data(f)->name
#define c_macro_name_length(f)         c_macro_data(f)->name_length
#define c_macro_required_args(f)       (T_Mac(f))->object.fnc.required_args
#define c_macro_all_args(f)            (T_Mac(f))->object.fnc.all_args
#define c_macro_setter(f)              T_App(c_macro_data(f)->setter)
#define c_macro_set_setter(f, Val)     c_macro_data(f)->setter = T_App(Val)

#define is_random_state(p)             (type(p) == T_RANDOM_STATE)
#if WITH_GMP
#define random_gmp_state(p)            (T_Ran(p))->object.rng.state
#else
#define random_seed(p)                 (T_Ran(p))->object.rng.seed
#define random_carry(p)                (T_Ran(p))->object.rng.carry
#endif

#define continuation_block(p)          (T_Con(p))->object.cwcc.block
#define continuation_stack(p)          (T_Con(p))->object.cwcc.stack
#define continuation_set_stack(p, Val) (T_Con(p))->object.cwcc.stack = T_Stk(Val)
#define continuation_stack_end(p)      (T_Con(p))->object.cwcc.stack_end
#define continuation_stack_start(p)    (T_Con(p))->object.cwcc.stack_start
#define continuation_stack_top(p)      (continuation_stack_end(p) - continuation_stack_start(p))
#define continuation_op_stack(p)       (T_Con(p))->object.cwcc.op_stack
#define continuation_stack_size(p)     continuation_block(p)->nx.ix.i1
#define continuation_op_loc(p)         continuation_block(p)->nx.ix.i2
#define continuation_op_size(p)        continuation_block(p)->ex.jx.i3
#define continuation_key(p)            continuation_block(p)->ex.jx.i4

#define call_exit_goto_loc(p)          (T_Got(p))->object.rexit.goto_loc
#define call_exit_op_loc(p)            (T_Got(p))->object.rexit.op_stack_loc
#define call_exit_active(p)            (T_Got(p))->object.rexit.active

#define temp_stack_top(p)              (T_Stk(p))->object.stk.top
#define s7_stack_top(Sc)               ((Sc)->stack_end - (Sc)->stack_start)

#define is_continuation(p)             (type(p) == T_CONTINUATION)
#define is_goto(p)                     (type(p) == T_GOTO)
#define is_macro(p)                    (type(p) == T_MACRO)
/* #define is_bacro(p)                 (type(p) == T_BACRO) */
#define is_macro_star(p)               (type(p) == T_MACRO_STAR)
#define is_bacro_star(p)               (type(p) == T_BACRO_STAR)

#define is_closure(p)                  (type(p) == T_CLOSURE)
#define is_closure_star(p)             (type(p) == T_CLOSURE_STAR)
#define closure_args(p)                (T_Clo(p))->object.func.args
#define closure_set_args(p, Val)       (T_Clo(p))->object.func.args = T_Arg(Val)
#define closure_body(p)                (T_Pair((T_Clo(p))->object.func.body))
#define closure_set_body(p, Val)       (T_Clo(p))->object.func.body = T_Pair(Val)
#define closure_let(p)                 T_Lid((T_Clo(p))->object.func.env)
#define closure_set_let(p, L)          (T_Clo(p))->object.func.env = T_Lid(L)
#define closure_arity(p)               (T_Clo(p))->object.func.arity
#define closure_setter(p)              (T_Clo(p))->object.func.setter
#define closure_set_setter(p, Val)     (T_Clo(p))->object.func.setter = T_Pos(Val)

#define CLOSURE_ARITY_NOT_SET          0x40000000
#define MAX_ARITY                      0x20000000
#define closure_arity_unknown(p)       (closure_arity(p) == CLOSURE_ARITY_NOT_SET)
#define is_thunk(Sc, Fnc)              ((type(Fnc) >= T_GOTO) && (s7_is_aritable(Sc, Fnc, 0)))

#define hook_has_functions(p)          (is_pair(s7_hook_functions(sc, T_Clo(p))))

#define catch_tag(p)                   (T_Cat(p))->object.rcatch.tag
#define catch_goto_loc(p)              (T_Cat(p))->object.rcatch.goto_loc
#define catch_op_loc(p)                (T_Cat(p))->object.rcatch.op_stack_loc
#define catch_handler(p)               T_Pos((T_Cat(p))->object.rcatch.handler)
#define catch_set_handler(p, val)      (T_Cat(p))->object.rcatch.handler = T_Pos(val)

#define catch_all_goto_loc(p)          (C_Let(p, L_CATCH))->object.envr.edat.ctall.goto_loc
#define catch_all_set_goto_loc(p, L)   (S_Let(p, L_CATCH))->object.envr.edat.ctall.goto_loc = L
#define catch_all_op_loc(p)            (C_Let(p, L_CATCH))->object.envr.edat.ctall.op_stack_loc
#define catch_all_set_op_loc(p, L)     (S_Let(p, L_CATCH))->object.envr.edat.ctall.op_stack_loc = L

enum {DWIND_INIT, DWIND_BODY, DWIND_FINISH};
#define dynamic_wind_state(p)          (T_Dyn(p))->object.winder.state
#define dynamic_wind_in(p)             (T_Dyn(p))->object.winder.in
#define dynamic_wind_out(p)            (T_Dyn(p))->object.winder.out
#define dynamic_wind_body(p)           (T_Dyn(p))->object.winder.body

#define is_c_object(p)                 (type(p) == T_C_OBJECT)
#define c_object_value(p)              (T_Obj(p))->object.c_obj.value
#define c_object_type(p)               (T_Obj(p))->object.c_obj.type
#define c_object_let(p)                T_Lid((T_Obj(p))->object.c_obj.e)
#define c_object_set_let(p, L)         (T_Obj(p))->object.c_obj.e = T_Lid(L)
#define c_object_mark(p)               (T_Obj(p))->object.c_obj.mark

#define c_object_info(Sc, p)           Sc->c_object_types[c_object_type(T_Obj(p))]
#define c_object_free(Sc, p)           c_object_info(Sc, p)->free
#define c_object_ref(Sc, p)            c_object_info(Sc, p)->ref
#define c_object_set(Sc, p)            c_object_info(Sc, p)->set
#if (!DISABLE_DEPRECATED)
  #define c_object_print(Sc, p)        c_object_info(Sc, p)->print
#endif
#define c_object_len(Sc, p)            c_object_info(Sc, p)->length
#define c_object_eql(Sc, p)            c_object_info(Sc, p)->equal
#define c_object_fill(Sc, p)           c_object_info(Sc, p)->fill
#define c_object_copy(Sc, p)           c_object_info(Sc, p)->copy
#define c_object_reverse(Sc, p)        c_object_info(Sc, p)->reverse
#define c_object_to_list(Sc, p)        c_object_info(Sc, p)->to_list
#define c_object_to_string(Sc, p)      c_object_info(Sc, p)->to_string
#define c_object_scheme_name(Sc, p)    T_Str(c_object_info(Sc, p)->scheme_name)

#define c_pointer(p)                   (T_Ptr(p))->object.cptr.c_pointer
#define c_pointer_type(p)              (T_Ptr(p))->object.cptr.c_type
#define c_pointer_info(p)              (T_Ptr(p))->object.cptr.info
#define c_pointer_weak1(p)             (T_Ptr(p))->object.cptr.weak1
#define c_pointer_weak2(p)             (T_Ptr(p))->object.cptr.weak2
#define c_pointer_set_weak1(p, q)      (T_Ptr(p))->object.cptr.weak1 = q
#define c_pointer_set_weak2(p, q)      (T_Ptr(p))->object.cptr.weak2 = q
#define is_c_pointer(p)                (type(p) == T_C_POINTER)

#define is_counter(p)                  (type(p) == T_COUNTER)
#define counter_result(p)              (T_Ctr(p))->object.ctr.result
#define counter_set_result(p, Val)     (T_Ctr(p))->object.ctr.result = T_Pos(Val)
#define counter_list(p)                (T_Ctr(p))->object.ctr.list
#define counter_set_list(p, Val)       (T_Ctr(p))->object.ctr.list = T_Pos(Val)
#define counter_capture(p)             (T_Ctr(p))->object.ctr.cap
#define counter_set_capture(p, Val)    (T_Ctr(p))->object.ctr.cap = Val
#define counter_let(p)                 T_Lid((T_Ctr(p))->object.ctr.env)
#define counter_set_let(p, L)          (T_Ctr(p))->object.ctr.env = T_Lid(L)
#define counter_slots(p)               (T_Ctr(p))->object.ctr.slots
#define counter_set_slots(p, Val)      (T_Ctr(p))->object.ctr.slots = T_Sln(Val)

#define is_baffle(p)                   (type(p) == T_BAFFLE)
#define baffle_key(p)                  (T_Bfl(p))->object.baffle_key

#if __cplusplus && HAVE_COMPLEX_NUMBERS
  using namespace std;                /* the code has to work in C as well as C++, so we can't scatter std:: all over the place */
  typedef complex<s7_double> s7_complex;
  static s7_double Real(complex<s7_double> x) {return(real(x));} /* protect the C++ name */
  static s7_double Imag(complex<s7_double> x) {return(imag(x));}
#endif

#define integer(p)                     (T_Int(p))->object.number.integer_value
#define set_integer(p, x)              integer(p) = x
#define real(p)                        (T_Rel(p))->object.number.real_value
#define set_real(p, x)                 real(p) = x
#define numerator(p)                   (T_Frc(p))->object.number.fraction_value.numerator
#define denominator(p)                 (T_Frc(p))->object.number.fraction_value.denominator
#define fraction(p)                    (((long_double)numerator(p)) / ((long_double)denominator(p)))
#define inverted_fraction(p)           (((long_double)denominator(p)) / ((long_double)numerator(p)))
#define real_part(p)                   (T_Cmp(p))->object.number.complex_value.rl
#define set_real_part(p, x)            real_part(p) = x
#define imag_part(p)                   (T_Cmp(p))->object.number.complex_value.im
#define set_imag_part(p, x)            imag_part(p) = x
#if HAVE_COMPLEX_NUMBERS
  #define as_c_complex(p)              CMPLX(real_part(p), imag_part(p))
#endif

#if WITH_GMP
#define big_integer(p)                 ((T_Bgi(p))->object.number.big_integer)
#define big_ratio(p)                   ((T_Bgf(p))->object.number.big_ratio)
#define big_real(p)                    ((T_Bgr(p))->object.number.big_real)
#define big_complex(p)                 ((T_Bgz(p))->object.number.big_complex)
#endif

#define print_name(p)                  (char *)((T_Num(p))->object.number.pval.name + 1)
#define print_name_length(p)           (T_Num(p))->object.number.pval.name[0]

static void set_print_name(s7_pointer p, const char *name, int32_t len)
{
  if ((len < (PRINT_NAME_SIZE - 1)) &&
      (!is_mutable(p)))
    {
      set_has_print_name(p);
      print_name_length(p) = (uint8_t)len;
      memcpy((void *)print_name(p), (void *)name, len);
      (print_name(p))[len] = 0;
    }
}

static s7_int s7_int_max = 0, s7_int_min = 0;
static int32_t s7_int32_max = 0, s7_int32_min = 0, s7_int_bits = 0, s7_int_digits = 0;
static int32_t s7_int_digits_by_radix[17];

#define S7_LLONG_MAX 9223372036854775807LL
#define S7_LLONG_MIN (-S7_LLONG_MAX - 1LL)

#define S7_LONG_MAX 2147483647LL
#define S7_LONG_MIN (-S7_LONG_MAX - 1LL)

#define S7_SHORT_MAX 32767
#define S7_SHORT_MIN -32768

static void init_int_limits(void)
{
  int32_t i, top;
#if WITH_GMP
#define S7_LOG_LLONG_MAX 36.736800
#define S7_LOG_LONG_MAX  16.6355322
#else
  /* actually not safe = (log (- (expt 2 63) 1)) and (log (- (expt 2 31) 1)) (using 63 and 31 bits) */
#define S7_LOG_LLONG_MAX 43.668274
#define S7_LOG_LONG_MAX  21.487562
#endif

  top = sizeof(s7_int);
  s7_int32_max = (top == 8) ? S7_LONG_MAX : S7_SHORT_MAX;
  s7_int32_min = (top == 8) ? S7_LONG_MIN : S7_SHORT_MIN;
  s7_int_bits = (top == 8) ? 63 : 31;
  s7_int_digits = (top == 8) ? 18 : 8;

  s7_int_max = (top == 8) ? S7_LLONG_MAX : S7_LONG_MAX;
  s7_int_min = (top == 8) ? S7_LLONG_MIN : S7_LONG_MIN;

  s7_int_digits_by_radix[0] = 0;
  s7_int_digits_by_radix[1] = 0;

  for (i = 2; i < 17; i++)
    s7_int_digits_by_radix[i] = (int32_t)(floor(((top == 8) ? S7_LOG_LLONG_MAX : S7_LOG_LONG_MAX) / log((double)i)));
}

static s7_pointer make_permanent_integer_unchecked(s7_int i)
{
  s7_pointer p;
  p = (s7_pointer)calloc(1, sizeof(s7_cell));
  set_type_bit(p, T_IMMUTABLE | T_INTEGER | T_UNHEAP);
  integer(p) = i;
  return(p);
}

#define NUM_SMALL_INTS 2048
static s7_pointer small_ints[NUM_SMALL_INTS + 1];
#define small_int(Val) small_ints[Val]
#define is_small(n)    ((n & ~(NUM_SMALL_INTS - 1)) == 0)

static s7_pointer real_zero, real_NaN, real_pi, real_one, arity_not_set, max_arity, real_infinity, real_minus_infinity, minus_one, minus_two, mostfix, leastfix;

static void init_small_ints(void)
{
  const char *ones[10] = {"0", "1", "2", "3", "4", "5", "6", "7", "8", "9"};
  s7_cell *cells;
  int32_t i;
  cells = (s7_cell *)calloc((NUM_SMALL_INTS + 1), sizeof(s7_cell));
  for (i = 0; i <= NUM_SMALL_INTS; i++)
    {
      s7_pointer p;
      small_ints[i] = &cells[i];
      p = small_ints[i];
      set_type_bit(p, T_IMMUTABLE | T_INTEGER | T_UNHEAP);
      integer(p) = i;
    }
  for (i = 0; i < 10; i++)
    set_print_name(small_ints[i], ones[i], 1);

  /* setup a few other numbers while we're here */
  #define EXTRA_NUMBERS 10
  cells = (s7_cell *)calloc(EXTRA_NUMBERS, sizeof(s7_cell));

  #define init_real(Ptr, Num, Name, Name_Len) \
    do {set_type(Ptr, T_REAL | T_IMMUTABLE | T_UNHEAP); set_real(Ptr, Num); if (Name) set_print_name(Ptr, Name, Name_Len);} while (0)

  real_zero = &cells[0]; init_real(real_zero, 0.0, "0.0", 3);
  real_one = &cells[1]; init_real(real_one, 1.0, "1.0", 3);
  real_NaN = &cells[2]; init_real(real_NaN, NAN, "+nan.0", 6);
  real_infinity = &cells[3]; init_real(real_infinity, INFINITY, "+inf.0", 6);
  real_minus_infinity = &cells[4]; init_real(real_minus_infinity, -INFINITY, "-inf.0", 6);
  real_pi = &cells[5]; init_real(real_pi, 3.1415926535897932384626433832795029L, NULL, 0); /* M_PI is not good enough for s7_double = long double */

  #define init_integer(Ptr, Num, Name, Name_Len) \
    do {set_type(Ptr, T_INTEGER | T_IMMUTABLE | T_UNHEAP); set_integer(Ptr, Num); if (Name) set_print_name(Ptr, Name, Name_Len);} while (0)

  arity_not_set = &cells[6]; init_integer(arity_not_set, CLOSURE_ARITY_NOT_SET, NULL, 0);
  max_arity = &cells[7]; init_integer(max_arity, MAX_ARITY, NULL, 0);
  minus_one = &cells[8]; init_integer(minus_one, -1, "-1", 2);
  minus_two = &cells[9]; init_integer(minus_two, -2, "-2", 2);

  mostfix = make_permanent_integer_unchecked(s7_int_max);
  leastfix = make_permanent_integer_unchecked(s7_int_min);
  if (s7_int_bits == 63)
    {
      set_print_name(mostfix, "9223372036854775807", 19);
      set_print_name(leastfix, "-9223372036854775808", 20);
    }
  else
    {
      set_print_name(mostfix, "2147483647", 10);
      set_print_name(leastfix, "-2147483648", 11);
    }
}

static void slot_set_setter(s7_pointer p, s7_pointer val)
{
  if ((type(val) == T_C_FUNCTION) &&
      (c_function_has_bool_setter(val)))
    slot_set_setter_1(p, c_function_bool_setter(val));
  else slot_set_setter_1(p, val);
}


/* -------------------------------------------------------------------------------- */
#define GC_TRIGGER_SIZE 64

/* new_cell has to include the new cell's type.  In the free list, it is 0 (T_FREE).  If we remove it here,
 *   but then hit some error before setting the type, the GC sweep thinks it is a free cell already and
 *   does not return it to the free list: a memory leak.
 */

#define GC_STATS 1
#define HEAP_STATS 2
#define STACK_STATS 4

#define show_gc_stats(Sc) ((Sc->gc_stats & GC_STATS) != 0)
#define show_stack_stats(Sc) ((Sc->gc_stats & STACK_STATS) != 0)
#define show_heap_stats(Sc) ((Sc->gc_stats & HEAP_STATS) != 0)


#if (!S7_DEBUGGING)
#define new_cell(Sc, Obj, Type)			\
  do {						\
    if (Sc->free_heap_top <= Sc->free_heap_trigger) try_to_call_gc(Sc); \
    Obj = (*(--(Sc->free_heap_top))); \
    set_type(Obj, Type);	      \
    } while (0)

#define new_cell_no_check(Sc, Obj, Type) do {Obj = (*(--(Sc->free_heap_top))); set_type(Obj, Type);} while (0)
  /* since sc->free_heap_trigger is GC_TRIGGER_SIZE above the free heap base, we don't need
   *   to check it repeatedly after the first such check.
   */
#else

#define new_cell(Sc, Obj, Type)						\
  do {									\
    if (Sc->free_heap_top <= Sc->free_heap_trigger) {if (show_gc_stats(Sc)) fprintf(stderr, "%s[%d]: gc\n", __func__, __LINE__);  try_to_call_gc(Sc);} \
    Obj = (*(--(Sc->free_heap_top)));					\
    Obj->debugger_bits = 0;						\
    set_type(Obj, Type);						\
  } while (0)

#define new_cell_no_check(Sc, Obj, Type)		    \
  do {							    \
    Obj = (*(--(Sc->free_heap_top)));			    \
    Obj->debugger_bits = 0;				    \
    set_type(Obj, Type);				    \
    } while (0)
#endif

#if WITH_GCC
#define make_integer(Sc, N) ({ s7_int _N_; _N_ = (N); (is_small(_N_) ? small_int(_N_) : ({ s7_pointer _I_; new_cell(Sc, _I_, T_INTEGER); integer(_I_) = _N_; _I_;}) ); })

#define make_real(Sc, X) ({ s7_pointer _R_; s7_double _N_ = (X); new_cell(Sc, _R_, T_REAL); set_real(_R_, _N_); _R_;})

#define make_complex(Sc, R, I)						\
  ({ s7_double im; im = (I); ((im == 0.0) ? make_real(Sc, R) : ({ s7_pointer _C_; new_cell(Sc, _C_, T_COMPLEX); set_real_part(_C_, R); set_imag_part(_C_, im); _C_;}) ); })

#define real_to_double(Sc, X, Caller)   ({ s7_pointer _x_; _x_ = (X); ((type(_x_) == T_REAL) ? real(_x_) : s7_number_to_real_with_caller(Sc, _x_, Caller)); })
#define rational_to_double(Sc, X)       ({ s7_pointer _x_; _x_ = (X); ((type(_x_) == T_INTEGER) ? (s7_double)integer(_x_) : fraction(_x_)); })

#else

#define make_integer(Sc, N)           s7_make_integer(Sc, N)
#define make_real(Sc, X)              s7_make_real(Sc, X)
#define make_complex(Sc, R, I)        s7_make_complex(Sc, R, I)
#define real_to_double(Sc, X, Caller) s7_number_to_real_with_caller(Sc, X, Caller)
#define rational_to_double(Sc, X)     s7_number_to_real(Sc, X)
#endif

static inline s7_pointer wrap_real(s7_scheme *sc, s7_double x) {real(sc->real_wrapper1) = x; return(sc->real_wrapper1);}
static inline s7_pointer wrap_integer1(s7_scheme *sc, s7_int x) {integer(sc->integer_wrapper1) = x; return(sc->integer_wrapper1);}
static inline s7_pointer wrap_integer2(s7_scheme *sc, s7_int x) {integer(sc->integer_wrapper2) = x; return(sc->integer_wrapper2);}
static inline s7_pointer wrap_integer3(s7_scheme *sc, s7_int x) {integer(sc->integer_wrapper3) = x; return(sc->integer_wrapper3);}
#if (!WITH_GMP)
static inline s7_pointer wrap_real2(s7_scheme *sc, s7_double x) {real(sc->real_wrapper2) = x; return(sc->real_wrapper2);}
#endif

/* 9007199254740991LL is where a truncated double starts to skip integers (expt 2 53) = ca 1e16
 *   :(ceiling (+ 1e16 1))
 *   10000000000000000
 *   :(> 9007199254740993.0 9007199254740992.0)
 *   #f ; in non-gmp 64-bit doubles
 *
 * but we can't fix this except in the gmp case because:
 *   :(integer-decode-float (+ (expt 2.0 62) 100))
 *   (4503599627370496 10 1)
 *   :(integer-decode-float (+ (expt 2.0 62) 500))
 *   (4503599627370496 10 1)
 *   :(> (+ (expt 2.0 62) 500) (+ (expt 2.0 62) 100))
 *   #f ; non-gmp again
 *
 * i.e. the bits are identical.  We can't even detect when it has happened, so should
 *   we just give an error for any floor (or whatever) of an arg>1e16?  (sin has a similar problem)?
 *   I think in the non-gmp case I'll throw an error in these cases because the results are
 *   bogus:
 *   :(floor (+ (expt 2.0 62) 512))
 *   4611686018427387904
 *   :(floor (+ (expt 2.0 62) 513))
 *   4611686018427388928
 *
 * another case at the edge: (round 9007199254740992.51) -> 9007199254740992
 *
 * This spells trouble for normal arithmetic in this range.  If no gmp,
 *    (- (+ (expt 2.0 62) 512) (+ (expt 2.0 62) 513)) = -1024.0 (should be -1.0)
 *    but we don't currently give an error in this case -- not sure what the right thing is.
 */


/* --------------------------------------------------------------------------------
 * local versions of some standard C library functions
 * timing tests involving these are very hard to interpret
 * local_memset and memclr are faster using int64_t than int32_t
 */

static void local_memset(void *s, uint8_t val, size_t n)
{
  uint8_t *s2;
#if S7_ALIGNED
  s2 = (uint8_t *)s;
#else
#if (defined(__x86_64__) || defined(__i386__))
  if (n >= 8)
    {
      int64_t ival;
      int64_t *s1 = (int64_t *)s;
      size_t n8 = n >> 3;
      ival = val | (val << 8) | (val << 16) | (val << 24); /* make gcc happy */
      ival = (ival << 32) | ival;
      do {*s1++ = ival;} while (--n8 > 0);
      n &= 7;
      s2 = (uint8_t *)s1;
    }
  else s2 = (uint8_t *)s;
#else
  s2 = (uint8_t *)s;
#endif
#endif
  while (n > 0)
    {
      *s2++ = val;
      n--;
    }
}

static inline s7_int safe_strlen(const char *str)
{
  /* this is safer than strlen, and slightly faster */
  char *tmp = (char *)str;
  if ((!tmp) || (!(*tmp))) return(0);
  while (*tmp++) {};
  return(tmp - str - 1);
}


static char *copy_string_with_length(const char *str, s7_int len)
{
  char *newstr;
#if S7_DEBUGGING
  if ((len <= 0) || (!str))
    fprintf(stderr, "%s[%d]: len: %" print_s7_int ", str: %s\n", __func__, __LINE__, len, str);
#endif
  newstr = (char *)malloc((len + 1) * sizeof(char));
  if (len != 0)
    memcpy((void *)newstr, (void *)str, len);
  newstr[len] = '\0';
  return(newstr);
}

static char *copy_string(const char *str)
{
  return(copy_string_with_length(str, safe_strlen(str)));
}


static bool local_strcmp(const char *s1, const char *s2)
{
  while (true)
    {
      if (*s1 != *s2++) return(false);
      if (*s1++ == 0) return(true);
    }
  return(true);
}

#define strings_are_equal(Str1, Str2) (local_strcmp(Str1, Str2))
/* this should only be used for internal strings -- scheme strings can have embedded nulls. */

static bool safe_strcmp(const char *s1, const char *s2)
{
  if ((!s1) || (!s2)) return(s1 == s2);
  return(local_strcmp(s1, s2));
}

static bool local_strncmp(const char *s1, const char *s2, size_t n)
{
#if S7_ALIGNED
  return(strncmp(s1, s2, n) == 0);
#else
#if (defined(__x86_64__) || defined(__i386__)) /* unaligned accesses are safe on i386 hardware, sez everyone */
  if (n >= 8)
    {
      int64_t *is1, *is2;
      size_t n8 = n >> 3;
      is1 = (int64_t *)s1;
      is2 = (int64_t *)s2;
      do {if (*is1++ != *is2++) return(false);} while (--n8 > 0);
      s1 = (const char *)is1;
      s2 = (const char *)is2;
      n &= 7;
    }
#endif
  while (n > 0)
    {
      if (*s1++ != *s2++) return(false);
      n--;
    }
  return(true);
#endif
}

#define strings_are_equal_with_length(Str1, Str2, Len) (local_strncmp(Str1, Str2, Len))

static size_t catstrs(char *dst, size_t len, ...) /* NULL-terminated arg list */
{
  const char *s, *dend;
  char *d;
  va_list ap;
  d = dst;
  dend = (const char *)(dst + len);
  while ((*d) && (d < dend)) d++;
  va_start(ap, len);
  for (s = va_arg(ap, const char *); s != NULL; s = va_arg(ap, const char *))
    while ((*s) && (d < dend)) {*d++ = *s++;}
  *d = '\0';
  va_end (ap);
  return(d - dst);
}

static size_t catstrs_direct(char *dst, const char *s1, ...) /* NULL-terminated arg list, dst is destination only (assumed empty), all args known to fit in dst */
{
  const char *s;
  char *d;
  va_list ap;
  d = dst;
  va_start(ap, s1);
  for (s = s1; s != NULL; s = va_arg(ap, const char *))
    while (*s) {*d++ = *s++;}
  *d = '\0';
  va_end (ap);
  return(d - dst);
}

static char *pos_int_to_str(s7_scheme *sc, s7_int num, s7_int *len, char endc)
{
  char *p, *op;

  p = (char *)(sc->int_to_str3 + INT_TO_STR_SIZE - 1);
  op = p;
  *p-- = '\0';
  if (endc != '\0') *p-- = endc;
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  (*len) = op - p;           /* this includes the trailing #\null */
  return((char *)(p + 1));
}

static char *pos_int_to_str_direct(s7_scheme *sc, s7_int num)
{
  char *p;
  p = (char *)(sc->int_to_str4 + INT_TO_STR_SIZE - 1);
  *p-- = '\0';
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  return((char *)(p + 1));
}

static char *pos_int_to_str_direct_1(s7_scheme *sc, s7_int num)
{
  char *p;
  p = (char *)(sc->int_to_str5 + INT_TO_STR_SIZE - 1);
  *p-- = '\0';
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  return((char *)(p + 1));
}


/* ---------------- forward decls ---------------- */

static void try_to_call_gc(s7_scheme *sc);
static s7_pointer eval(s7_scheme *sc, opcode_t first_op);
static s7_pointer prepackaged_type_name(s7_scheme *sc, s7_pointer x);
static const char *type_name(s7_scheme *sc, s7_pointer arg, int32_t article);
static void s7_warn(s7_scheme *sc, s7_int len, const char *ctrl, ...);
static s7_pointer safe_reverse_in_place(s7_scheme *sc, s7_pointer list);
static s7_pointer cons_unchecked(s7_scheme *sc, s7_pointer a, s7_pointer b);
static s7_pointer cons_unchecked_with_type(s7_scheme *sc, s7_pointer p, s7_pointer a, s7_pointer b);
static s7_pointer permanent_cons(s7_scheme *sc, s7_pointer a, s7_pointer b, uint64_t type);
static s7_pointer make_atom(s7_scheme *sc, char *q, s7_int radix, bool want_symbol, bool with_error);
static s7_pointer apply_error(s7_scheme *sc, s7_pointer obj, s7_pointer args);
static s7_pointer division_by_zero_error(s7_scheme *sc, s7_pointer caller, s7_pointer arg);
static s7_pointer file_error(s7_scheme *sc, const char *caller, const char *descr, const char *name);
static s7_pointer unbound_variable(s7_scheme *sc, s7_pointer sym);
static void check_for_substring_temp(s7_scheme *sc, s7_pointer expr);
static s7_pointer splice_in_values(s7_scheme *sc, s7_pointer args);
static void pop_input_port(s7_scheme *sc);
static s7_pointer object_to_truncated_string(s7_scheme *sc, s7_pointer p, s7_int len);
static token_t token(s7_scheme *sc);
static s7_pointer implicit_index(s7_scheme *sc, s7_pointer obj, s7_pointer indices);
static void free_hash_table(s7_scheme *sc, s7_pointer table);
static s7_pointer g_cdr(s7_scheme *sc, s7_pointer args);
static s7_pointer s7_length(s7_scheme *sc, s7_pointer lst);
static bool tree_is_cyclic(s7_scheme *sc, s7_pointer tree);
static inline s7_pointer symbol_to_slot(s7_scheme *sc, s7_pointer symbol);
static inline s7_pointer make_simple_vector(s7_scheme *sc, s7_int len);

#if S7_DEBUGGING
#define wrap_string(Sc, Str, Len) wrap_string_1(Sc, Str, Len, __func__, __LINE__)
static s7_pointer wrap_string_1(s7_scheme *sc, const char *str, s7_int len, const char *func, int line);
#else
static s7_pointer wrap_string(s7_scheme *sc, const char *str, s7_int len);
#endif

#if WITH_GMP
static s7_int big_integer_to_s7_int(mpz_t n);
#else
static double next_random(s7_pointer r);
#endif

#if S7_DEBUGGING && WITH_GCC
  static s7_pointer symbol_to_value_unchecked_1(s7_scheme *sc, s7_pointer symbol);
  #define symbol_to_value_unchecked(Sc, Sym) check_null_sym(Sc, symbol_to_value_unchecked_1(Sc, Sym), Sym, __LINE__, __func__)
  static s7_pointer check_null_sym(s7_scheme *sc, s7_pointer p, s7_pointer sym, int32_t line, const char *func);
  #define symbol_to_value_unexamined(Sc, Sym) symbol_to_value_unchecked_1(Sc, Sym)
#else
  static inline s7_pointer symbol_to_value_unchecked(s7_scheme *sc, s7_pointer symbol);
  #define symbol_to_value_unexamined(Sc, Sym) symbol_to_value_unchecked(Sc, Sym)
#endif

#if WITH_GCC
  #if S7_DEBUGGING
    #define symbol_to_value_checked(Sc, Sym) ({s7_pointer _x_; _x_ = symbol_to_value_unchecked_1(Sc, Sym); ((_x_) ? _x_ : unbound_variable(Sc, Sym));})
  #else
    #define symbol_to_value_checked(Sc, Sym) ({s7_pointer _x_; _x_ = symbol_to_value_unchecked(Sc, Sym); ((_x_) ? _x_ : unbound_variable(Sc, Sym));})
  #endif
#else
  #define symbol_to_value_checked(Sc, Sym) symbol_to_value_unchecked(Sc, Sym)
#endif

static s7_pointer find_method(s7_scheme *sc, s7_pointer env, s7_pointer symbol);
static s7_pointer find_let(s7_scheme *sc, s7_pointer obj);
static bool call_begin_hook(s7_scheme *sc);
static s7_pointer default_vector_setter(s7_scheme *sc, s7_pointer vec, s7_int loc, s7_pointer val);
static s7_pointer default_vector_getter(s7_scheme *sc, s7_pointer vec, s7_int loc);

static s7_pointer simple_wrong_type_arg_error_prepackaged(s7_scheme *sc, s7_pointer caller, s7_pointer arg, s7_pointer typnam, s7_pointer descr);
static s7_pointer wrong_type_arg_error_prepackaged(s7_scheme *sc, s7_pointer caller, s7_pointer arg_n, s7_pointer arg, s7_pointer typnam, s7_pointer descr);
static s7_pointer out_of_range_error_prepackaged(s7_scheme *sc, s7_pointer caller, s7_pointer arg_n, s7_pointer arg, s7_pointer descr);
static s7_pointer simple_out_of_range_error_prepackaged(s7_scheme *sc, s7_pointer caller, s7_pointer arg, s7_pointer descr);

/* putting off the type description until s7_error via the sc->gc_nil marker below makes it possible
 *    for gcc to speed up the functions that call these as tail-calls.  1-2% overall speedup!
 */
#define simple_wrong_type_argument(Sc, Caller, Arg, Desired_Type) \
  simple_wrong_type_arg_error_prepackaged(Sc, symbol_name_cell(Caller), Arg, Sc->gc_nil, prepackaged_type_names[Desired_Type])

#define wrong_type_argument(Sc, Caller, Num, Arg, Desired_Type)	\
  wrong_type_arg_error_prepackaged(Sc, symbol_name_cell(Caller), make_integer(Sc, Num), Arg, Sc->gc_nil, prepackaged_type_names[Desired_Type])

#define simple_wrong_type_argument_with_type(Sc, Caller, Arg, Type) \
  simple_wrong_type_arg_error_prepackaged(Sc, symbol_name_cell(Caller), Arg, Sc->gc_nil, Type)

#define wrong_type_argument_with_type(Sc, Caller, Num, Arg, Type)	\
  wrong_type_arg_error_prepackaged(Sc, symbol_name_cell(Caller), make_integer(Sc, Num), Arg, Sc->gc_nil, Type)

#define simple_out_of_range(Sc, Caller, Arg, Description)   simple_out_of_range_error_prepackaged(Sc, symbol_name_cell(Caller), Arg, Description)
#define out_of_range(Sc, Caller, Arg_Num, Arg, Description) out_of_range_error_prepackaged(Sc, symbol_name_cell(Caller), Arg_Num, Arg, Description)


#if (!HAVE_COMPLEX_NUMBERS)
static s7_pointer no_complex_numbers_string;
#endif


/* ---------------- evaluator ops ---------------- */

/* C=constant, S=symbol, A=fx-callable, Q=quote, D=list of constants */
enum {OP_UNOPT, HOP_UNOPT, OP_SYM, HOP_SYM, OP_CON, HOP_CON,
      OP_PAIR_SYM, HOP_PAIR_SYM, OP_PAIR_PAIR, HOP_PAIR_PAIR, OP_PAIR_ANY, HOP_PAIR_ANY,
      OP_SAFE_C_D, HOP_SAFE_C_D, OP_SAFE_C_AND2, HOP_SAFE_C_AND2, OP_SAFE_C_OR2, HOP_SAFE_C_OR2,
      OP_SAFE_C_S, HOP_SAFE_C_S, OP_SAFE_CAR_S, HOP_SAFE_CAR_S, OP_SAFE_CDR_S, HOP_SAFE_CDR_S, OP_SAFE_CADR_S, HOP_SAFE_CADR_S,
      OP_SAFE_IS_PAIR_S, HOP_SAFE_IS_PAIR_S, OP_SAFE_IS_NULL_S, HOP_SAFE_IS_NULL_S, OP_SAFE_IS_SYMBOL_S, HOP_SAFE_IS_SYMBOL_S, /* order matters here */
      OP_SAFE_C_SS, HOP_SAFE_C_SS, OP_SAFE_C_SC, HOP_SAFE_C_SC, OP_SAFE_C_CS, HOP_SAFE_C_CS,
      OP_SAFE_C_CQ, HOP_SAFE_C_CQ, 
      OP_SAFE_C_SSS, HOP_SAFE_C_SSS, OP_SAFE_C_SCS, HOP_SAFE_C_SCS, OP_SAFE_C_SSC, HOP_SAFE_C_SSC, OP_SAFE_C_CSS, HOP_SAFE_C_CSS,
      OP_SAFE_C_SCC, HOP_SAFE_C_SCC, OP_SAFE_C_CSC, HOP_SAFE_C_CSC, OP_SAFE_C_CCS, HOP_SAFE_C_CCS,
      OP_SAFE_C_ALL_S, HOP_SAFE_C_ALL_S,
      OP_SAFE_C_opDq, HOP_SAFE_C_opDq, OP_SAFE_C_opSq, HOP_SAFE_C_opSq,
      OP_SAFE_C_opDq_opSq, HOP_SAFE_C_opDq_opSq, OP_SAFE_C_opSq_opDq, HOP_SAFE_C_opSq_opDq,
      OP_SAFE_C_opSSq, HOP_SAFE_C_opSSq, OP_SAFE_C_opSCq, HOP_SAFE_C_opSCq, 
      OP_SAFE_C_opCSq, HOP_SAFE_C_opCSq, OP_SAFE_C_S_opSq, HOP_SAFE_C_S_opSq,
      OP_SAFE_C_C_opSCq, HOP_SAFE_C_C_opSCq,
      OP_SAFE_C_S_opSCq, HOP_SAFE_C_S_opSCq, OP_SAFE_C_S_opCSq, HOP_SAFE_C_S_opCSq,
      OP_SAFE_C_opSq_S, HOP_SAFE_C_opSq_S, OP_SAFE_C_CAR_S_S, HOP_SAFE_C_CAR_S_S, OP_SAFE_C_opSq_C, HOP_SAFE_C_opSq_C,
      OP_SAFE_C_opSq_opSq, HOP_SAFE_C_opSq_opSq, OP_SAFE_C_S_opSSq, HOP_SAFE_C_S_opSSq, OP_SAFE_C_C_opSq, HOP_SAFE_C_C_opSq,
      OP_SAFE_C_C_opCSq, HOP_SAFE_C_C_opCSq, OP_SAFE_C_opCSq_C, HOP_SAFE_C_opCSq_C,
      OP_SAFE_C_S_opDq, HOP_SAFE_C_S_opDq, OP_SAFE_C_opSSq_C, HOP_SAFE_C_opSSq_C, OP_SAFE_C_C_opSSq, HOP_SAFE_C_C_opSSq,
      OP_SAFE_C_C_opDq, HOP_SAFE_C_C_opDq, OP_SAFE_C_opDq_S, HOP_SAFE_C_opDq_S,
      OP_SAFE_C_opDq_opDq, HOP_SAFE_C_opDq_opDq, OP_SAFE_C_opDq_C, HOP_SAFE_C_opDq_C,
      OP_SAFE_C_opSCq_opSCq, HOP_SAFE_C_opSCq_opSCq, OP_SAFE_C_opSSq_opSSq, HOP_SAFE_C_opSSq_opSSq,
      OP_SAFE_C_opSSq_opDq, HOP_SAFE_C_opSSq_opDq, OP_SAFE_C_opSSq_opSq, HOP_SAFE_C_opSSq_opSq, OP_SAFE_C_opSq_opSSq, HOP_SAFE_C_opSq_opSSq,
      OP_SAFE_C_opSSq_S, HOP_SAFE_C_opSSq_S, OP_SAFE_C_opSCq_S, HOP_SAFE_C_opSCq_S, OP_SAFE_C_opCSq_S, HOP_SAFE_C_opCSq_S,
      OP_SAFE_C_opSCq_C, HOP_SAFE_C_opSCq_C, OP_SAFE_C_opDq_opSSq, HOP_SAFE_C_opDq_opSSq,
      OP_SAFE_C_S_op_opSq_Cq, HOP_SAFE_C_S_op_opSq_Cq,
      OP_SAFE_C_S_op_S_opSSqq, HOP_SAFE_C_S_op_S_opSSqq, OP_SAFE_C_S_op_S_opSqq, HOP_SAFE_C_S_op_S_opSqq,
      OP_SAFE_C_op_opSSq_q_C, HOP_SAFE_C_op_opSSq_q_C, OP_SAFE_C_op_opSq_q_C, HOP_SAFE_C_op_opSq_q_C,
      OP_SAFE_C_op_opSSq_q_S, HOP_SAFE_C_op_opSSq_q_S, OP_SAFE_C_op_opSq_q_S, HOP_SAFE_C_op_opSq_q_S,
      OP_SAFE_C_S_op_opSSq_opSSqq, HOP_SAFE_C_S_op_opSSq_opSSqq, OP_SAFE_C_op_opSSq_Sq_S, HOP_SAFE_C_op_opSSq_Sq_S,
      OP_SAFE_C_op_opSq_q, HOP_SAFE_C_op_opSq_q,
      OP_SAFE_C_op_S_opSq_q, HOP_SAFE_C_op_S_opSq_q, OP_SAFE_C_op_opSq_S_q, HOP_SAFE_C_op_opSq_S_q,
      OP_SAFE_C_opSq_CS, HOP_SAFE_C_opSq_CS,

      OP_SAFE_C_A, HOP_SAFE_C_A, OP_SAFE_C_AA, HOP_SAFE_C_AA, OP_SAFE_C_AAA, HOP_SAFE_C_AAA, OP_SAFE_C_AAAA, HOP_SAFE_C_AAAA,
      OP_SAFE_C_FX, HOP_SAFE_C_FX, OP_SAFE_C_ALL_CA, HOP_SAFE_C_ALL_CA,
      OP_SAFE_C_SSA, HOP_SAFE_C_SSA, OP_SAFE_C_SAS, HOP_SAFE_C_SAS,
      OP_SAFE_C_CSA, HOP_SAFE_C_CSA, OP_SAFE_C_SCA, HOP_SAFE_C_SCA,
      OP_SAFE_C_CAC, HOP_SAFE_C_CAC,
      OP_SAFE_C_opAq, HOP_SAFE_C_opAq, OP_SAFE_C_opAAq, HOP_SAFE_C_opAAq, OP_SAFE_C_opAAAq, HOP_SAFE_C_opAAAq,
      OP_SAFE_C_S_opAq, HOP_SAFE_C_S_opAq, OP_SAFE_C_opAq_S, HOP_SAFE_C_opAq_S,
      OP_SAFE_C_S_opAAq, HOP_SAFE_C_S_opAAq, OP_SAFE_C_S_opAAAq, HOP_SAFE_C_S_opAAAq,
      OP_SAFE_IFA_SS_A, HOP_SAFE_IFA_SS_A,
      OP_SAFE_C_STAR, HOP_SAFE_C_STAR, OP_SAFE_C_STAR_A, HOP_SAFE_C_STAR_A, OP_SAFE_C_STAR_AA, HOP_SAFE_C_STAR_AA, OP_SAFE_C_STAR_FX, HOP_SAFE_C_STAR_FX,
      OP_SAFE_C_P, HOP_SAFE_C_P, 

      OP_THUNK, HOP_THUNK, OP_THUNK_P, HOP_THUNK_P,
      OP_SAFE_THUNK, HOP_SAFE_THUNK, OP_SAFE_THUNK_P, HOP_SAFE_THUNK_P, OP_SAFE_THUNK_A, HOP_SAFE_THUNK_A,

      OP_CLOSURE_S, HOP_CLOSURE_S, OP_CLOSURE_S_P, HOP_CLOSURE_S_P,
      OP_SAFE_CLOSURE_S, HOP_SAFE_CLOSURE_S, OP_SAFE_CLOSURE_S_P, HOP_SAFE_CLOSURE_S_P, OP_SAFE_CLOSURE_S_A, HOP_SAFE_CLOSURE_S_A,
      OP_CLOSURE_C, HOP_CLOSURE_C, OP_CLOSURE_C_P, HOP_CLOSURE_C_P,
      OP_SAFE_CLOSURE_C, HOP_SAFE_CLOSURE_C, OP_SAFE_CLOSURE_C_P, HOP_SAFE_CLOSURE_C_P, OP_SAFE_CLOSURE_C_A, HOP_SAFE_CLOSURE_C_A,

      OP_CLOSURE_A, HOP_CLOSURE_A, OP_CLOSURE_A_P, HOP_CLOSURE_A_P, OP_CLOSURE_SUB_P, HOP_CLOSURE_SUB_P,
      OP_SAFE_CLOSURE_A, HOP_SAFE_CLOSURE_A, OP_SAFE_CLOSURE_A_P, HOP_SAFE_CLOSURE_A_P, OP_SAFE_CLOSURE_A_A, HOP_SAFE_CLOSURE_A_A,

      OP_CLOSURE_P, HOP_CLOSURE_P, OP_SAFE_CLOSURE_P, HOP_SAFE_CLOSURE_P,

      OP_CLOSURE_AP, HOP_CLOSURE_AP, OP_CLOSURE_PA, HOP_CLOSURE_PA,
      OP_SAFE_CLOSURE_AP, HOP_SAFE_CLOSURE_AP, OP_SAFE_CLOSURE_PA, HOP_SAFE_CLOSURE_PA,
      OP_CLOSURE_FA, HOP_CLOSURE_FA,

      OP_CLOSURE_SS, HOP_CLOSURE_SS, OP_CLOSURE_SS_P, HOP_CLOSURE_SS_P,
      OP_SAFE_CLOSURE_SS, HOP_SAFE_CLOSURE_SS, OP_SAFE_CLOSURE_SS_P, HOP_SAFE_CLOSURE_SS_P, OP_SAFE_CLOSURE_SS_A, HOP_SAFE_CLOSURE_SS_A,
      OP_CLOSURE_SC, HOP_CLOSURE_SC, OP_CLOSURE_SC_P, HOP_CLOSURE_SC_P,
      OP_SAFE_CLOSURE_SC, HOP_SAFE_CLOSURE_SC, OP_SAFE_CLOSURE_SC_P, HOP_SAFE_CLOSURE_SC_P,
      OP_CLOSURE_CS, HOP_CLOSURE_CS,
      OP_SAFE_CLOSURE_CS, HOP_SAFE_CLOSURE_CS,

      OP_CLOSURE_AA, HOP_CLOSURE_AA, OP_CLOSURE_AA_P, HOP_CLOSURE_AA_P,
      OP_SAFE_CLOSURE_AA, HOP_SAFE_CLOSURE_AA, OP_SAFE_CLOSURE_AA_P, HOP_SAFE_CLOSURE_AA_P, OP_SAFE_CLOSURE_AA_A, HOP_SAFE_CLOSURE_AA_A,

      OP_CLOSURE_FX, HOP_CLOSURE_FX, OP_CLOSURE_ALL_S, HOP_CLOSURE_ALL_S, OP_CLOSURE_ANY_FX, HOP_CLOSURE_ANY_FX,
      OP_SAFE_CLOSURE_SA, HOP_SAFE_CLOSURE_SA, OP_SAFE_CLOSURE_SAA, HOP_SAFE_CLOSURE_SAA, OP_SAFE_CLOSURE_FX, HOP_SAFE_CLOSURE_FX,

      OP_CLOSURE_STAR_A, HOP_CLOSURE_STAR_A, OP_CLOSURE_STAR_FX, HOP_CLOSURE_STAR_FX,
      OP_SAFE_CLOSURE_STAR_A, HOP_SAFE_CLOSURE_STAR_A, OP_SAFE_CLOSURE_STAR_AA, HOP_SAFE_CLOSURE_STAR_AA,
      OP_SAFE_CLOSURE_STAR_FX, HOP_SAFE_CLOSURE_STAR_FX, OP_SAFE_CLOSURE_STAR_FX_0, HOP_SAFE_CLOSURE_STAR_FX_0, 
      OP_SAFE_CLOSURE_STAR_FX_1, HOP_SAFE_CLOSURE_STAR_FX_1, OP_SAFE_CLOSURE_STAR_FX_2, HOP_SAFE_CLOSURE_STAR_FX_2,

      /* these can't be embedded, and have to be the last thing called */
      OP_APPLY_SS, HOP_APPLY_SS, OP_APPLY_SA, HOP_APPLY_SA, OP_APPLY_SL, HOP_APPLY_SL,
      OP_C_FX, HOP_C_FX, OP_CALL_WITH_EXIT, HOP_CALL_WITH_EXIT, OP_CALL_WITH_EXIT_P, HOP_CALL_WITH_EXIT_P,
      OP_C_CATCH, HOP_C_CATCH, OP_C_CATCH_ALL, HOP_C_CATCH_ALL, OP_C_CATCH_ALL_P, HOP_C_CATCH_ALL_P,
      OP_C_S_opSq, HOP_C_S_opSq, OP_C_S_opDq, HOP_C_S_opDq, OP_C_SS, HOP_C_SS,
      OP_C_S, HOP_C_S, OP_READ_S, HOP_READ_S, OP_C_P, HOP_C_P, OP_C_AP, HOP_C_AP, OP_NOT_P, HOP_NOT_P,
      OP_C_A, HOP_C_A, OP_C_SCS, HOP_C_SCS,
      OP_C_FA, HOP_C_FA, OP_C_AA, HOP_C_AA,

      OP_SAFE_C_PP, HOP_SAFE_C_PP,
      OP_SAFE_C_opSq_P, HOP_SAFE_C_opSq_P,
      OP_SAFE_C_SP, HOP_SAFE_C_SP, OP_SAFE_C_CP, HOP_SAFE_C_CP, 
      OP_SAFE_C_AP, HOP_SAFE_C_AP, OP_SAFE_C_PA, HOP_SAFE_C_PA,
      OP_SAFE_C_PS, HOP_SAFE_C_PS, OP_SAFE_C_PC, HOP_SAFE_C_PC, 
      OP_SAFE_C_SSP, HOP_SAFE_C_SSP, OP_SAFE_C_FP, HOP_SAFE_C_FP,

      OP_S, OP_S_S, OP_S_C, OP_S_A, OP_C_FA_1, OP_S_AA,
      OP_GOTO, OP_GOTO_A,
      OP_ITERATE, OP_CONTINUATION_A, OP_VECTOR_A, OP_STRING_A, OP_C_OBJECT_A, OP_PAIR_A, OP_HASH_TABLE_A, OP_ENVIRONMENT_C, OP_ENVIRONMENT_A,
      OP_UNKNOWN, OP_UNKNOWN_ALL_S, OP_UNKNOWN_FX, OP_UNKNOWN_G, OP_UNKNOWN_GG, OP_UNKNOWN_A, OP_UNKNOWN_AA,

      OP_GC_PROTECT,
      OP_READ_INTERNAL, OP_EVAL,
      OP_EVAL_ARGS, OP_EVAL_ARGS1, OP_EVAL_ARGS2, OP_EVAL_ARGS3, OP_EVAL_ARGS4, OP_EVAL_ARGS5,
      OP_APPLY, OP_EVAL_MACRO, OP_LAMBDA, OP_QUOTE, OP_MACROEXPAND,
      OP_DEFINE, OP_DEFINE1, OP_BEGIN, OP_BEGIN0, OP_BEGIN1, OP_BEGIN_UNCHECKED,
      OP_IF, OP_IF1, OP_WHEN, OP_UNLESS, OP_SET, OP_SET1, OP_SET2,
      OP_LET, OP_LET1, OP_LET_STAR, OP_LET_STAR1, OP_LET_STAR2,
      OP_LETREC, OP_LETREC1, OP_LETREC_STAR, OP_LETREC_STAR1,
      OP_LET_TEMPORARILY, OP_LET_TEMP_UNCHECKED, OP_LET_TEMP_INIT1, OP_LET_TEMP_INIT2, OP_LET_TEMP_DONE, OP_LET_TEMP_DONE1,
      OP_COND, OP_COND1, OP_FEED_TO_1, OP_COND_SIMPLE, OP_COND1_SIMPLE, OP_COND_SIMPLE_P, OP_COND1_SIMPLE_P,
      OP_AND, OP_AND1, OP_OR, OP_OR1,
      OP_DEFINE_MACRO, OP_DEFINE_MACRO_STAR, OP_DEFINE_EXPANSION,
      OP_CASE,
      OP_READ_LIST, OP_READ_NEXT, OP_READ_DOT, OP_READ_QUOTE,
      OP_READ_QUASIQUOTE, OP_READ_UNQUOTE, OP_READ_APPLY_VALUES,
      OP_READ_VECTOR, OP_READ_BYTE_VECTOR, OP_READ_INT_VECTOR, OP_READ_FLOAT_VECTOR, OP_READ_DONE,
      OP_LOAD_RETURN_IF_EOF, OP_LOAD_CLOSE_AND_POP_IF_EOF, OP_EVAL_DONE,
      OP_CATCH, OP_DYNAMIC_WIND, OP_DEFINE_CONSTANT, OP_DEFINE_CONSTANT1,
      OP_DO, OP_DO_END, OP_DO_END1, OP_DO_STEP, OP_DO_STEP2, OP_DO_INIT,
      OP_DEFINE_STAR, OP_LAMBDA_STAR, OP_LAMBDA_STAR_DEFAULT, OP_ERROR_QUIT, OP_UNWIND_INPUT, OP_UNWIND_OUTPUT,
      OP_ERROR_HOOK_QUIT,
      OP_WITH_LET, OP_WITH_LET1, OP_WITH_LET_UNCHECKED, OP_WITH_LET_S,
      OP_WITH_BAFFLE, OP_WITH_BAFFLE_UNCHECKED, OP_EXPANSION,
      OP_FOR_EACH, OP_FOR_EACH_1, OP_FOR_EACH_2, OP_FOR_EACH_3,
      OP_MAP, OP_MAP_1, OP_MAP_2, OP_MAP_GATHER, OP_MAP_GATHER_1, OP_MAP_GATHER_2, OP_MAP_GATHER_3,
      OP_BARRIER, OP_DEACTIVATE_GOTO,

      OP_DEFINE_BACRO, OP_DEFINE_BACRO_STAR,
      OP_GET_OUTPUT_STRING,
      OP_SORT, OP_SORT1, OP_SORT2, OP_SORT3, OP_SORT_PAIR_END, OP_SORT_VECTOR_END, OP_SORT_STRING_END,
      OP_EVAL_STRING,
      OP_MEMBER_IF, OP_ASSOC_IF, OP_MEMBER_IF1, OP_ASSOC_IF1,

      OP_LAMBDA_UNCHECKED, OP_LET_UNCHECKED,
      OP_CATCH_1, OP_CATCH_2, OP_CATCH_ALL,

      OP_SET_UNCHECKED, OP_SET_SYMBOL_C, OP_SET_SYMBOL_S, OP_SET_SYMBOL_P, OP_SET_SYMBOL_A,
      OP_SET_SYMBOL_opSq, OP_SET_SYMBOL_opDq, OP_SET_SYMBOL_opSSq,
      OP_SET_NORMAL, OP_SET_PAIR, OP_SET_DILAMBDA, OP_SET_DILAMBDA_P, OP_SET_DILAMBDA_P_1, OP_SET_PAIR_A, OP_SET_PAIR_P, OP_SET_PAIR_ZA,
      OP_SET_PAIR_P_1, OP_SET_WITH_SETTER, OP_SET_PWS, OP_SET_LET_S, OP_SET_LET_FX,
      OP_SET_SAFE,
      OP_INCREMENT_1, OP_DECREMENT_1, OP_SET_CONS,
      OP_INCREMENT_SS, OP_INCREMENT_SSS, OP_INCREMENT_SP, OP_INCREMENT_SA, OP_INCREMENT_SAA,

      OP_LETREC_UNCHECKED, OP_LETREC_STAR_UNCHECKED, OP_COND_UNCHECKED,
      OP_LAMBDA_STAR_UNCHECKED, OP_DO_UNCHECKED, OP_DEFINE_UNCHECKED, OP_DEFINE_STAR_UNCHECKED, OP_DEFINE_FUNCHECKED, OP_DEFINE_CONSTANT_UNCHECKED,
      OP_DEFINE_WITH_SETTER, OP_DEFINE_MACRO_WITH_SETTER,

      OP_LET_NO_VARS, OP_NAMED_LET, OP_NAMED_LET_NO_VARS, OP_NAMED_LET_STAR,
      OP_LET_C, OP_LET_S, OP_LET_S_P, OP_LET_ALL_C, OP_LET_ALL_S, OP_LET_FX,
      OP_LET_STAR_FX, OP_LET_STAR_A2, OP_LET_STAR_A, OP_LET_opDq, OP_LET_opSSq, OP_LET_opSSq_E, OP_LET_opaSSq, OP_LET_opaSSq_E,
      OP_LET_opSq, OP_LET_ALL_opSq, OP_LET_opSq_P, OP_LET_CAR, OP_LET_ONE, OP_LET_ONE_1, OP_LET_ONE_P, OP_LET_ONE_P_1,
      OP_LET_A, OP_LET_A_P,

      OP_CASE_A_E_S, OP_CASE_A_I_S, OP_CASE_A_G_S, OP_CASE_A_E_G, OP_CASE_A_G_G,
      OP_CASE_S_E_S, OP_CASE_S_I_S, OP_CASE_S_G_S, OP_CASE_S_E_G, OP_CASE_S_G_G,
      OP_CASE_P_E_S, OP_CASE_P_I_S, OP_CASE_P_G_S, OP_CASE_P_E_G, OP_CASE_P_G_G,
      OP_CASE_E_S, OP_CASE_I_S, OP_CASE_G_S, OP_CASE_E_G, OP_CASE_G_G,

      OP_IF_UNCHECKED, OP_AND_P, OP_AND_P1, OP_AND_AP, OP_AND_SAFE_P, OP_AND_SAFE_AA, OP_AND_PAIR_P,
      OP_OR_P, OP_OR_P1, OP_OR_AP, OP_OR_SAFE_P, OP_OR_SAFE_AA,
      OP_COND_FEED, OP_COND_FEED_1, OP_WHEN_S, OP_WHEN_A, OP_WHEN_P, OP_UNLESS_S, OP_UNLESS_A, OP_UNLESS_P,

      OP_IF_S_P, OP_IF_S_P_P, OP_IF_S_R, OP_IF_S_N, OP_IF_S_N_N,
      OP_IF_C_P, OP_IF_C_P_P, OP_IF_C_R, OP_IF_C_N, OP_IF_C_N_N,
      OP_IF_CS_P, OP_IF_CS_P_P, OP_IF_CS_R, OP_IF_CS_N, OP_IF_CS_N_N,
      OP_IF_CSS_P, OP_IF_CSS_P_P, OP_IF_CSS_R, OP_IF_CSS_N, OP_IF_CSS_N_N,
      OP_IF_CSC_P, OP_IF_CSC_P_P, OP_IF_CSC_R, OP_IF_CSC_N, OP_IF_CSC_N_N,
      OP_IF_opSq_P, OP_IF_opSq_P_P, OP_IF_opSq_R, OP_IF_opSq_N, OP_IF_opSq_N_N,
      OP_IF_S_opDq_P, OP_IF_S_opDq_P_P, OP_IF_S_opDq_R, OP_IF_S_opDq_N, OP_IF_S_opDq_N_N,
      OP_IF_IS_TYPE_S_P, OP_IF_IS_TYPE_S_P_P, OP_IF_IS_TYPE_S_R, OP_IF_IS_TYPE_S_N, OP_IF_IS_TYPE_S_N_N,
      OP_IF_IS_TYPE_opSq_P, OP_IF_IS_TYPE_opSq_P_P, OP_IF_IS_TYPE_opSq_R, OP_IF_IS_TYPE_opSq_N, OP_IF_IS_TYPE_opSq_N_N,
      OP_IF_A_P, OP_IF_A_P_P, OP_IF_A_R, OP_IF_A_N, OP_IF_A_N_N,
      OP_IF_AND2_P, OP_IF_AND2_P_P, OP_IF_AND2_R, OP_IF_AND2_N, OP_IF_AND2_N_N,
      OP_IF_AND3_P, OP_IF_AND3_P_P, OP_IF_AND3_R, OP_IF_AND3_N, OP_IF_AND3_N_N,
      OP_IF_P_P, OP_IF_P_P_P, OP_IF_P_R, OP_IF_P_N, OP_IF_P_N_N,
      OP_IF_ANDP_P, OP_IF_ANDP_P_P, OP_IF_ANDP_R, OP_IF_ANDP_N, OP_IF_ANDP_N_N,
      OP_IF_ORP_P, OP_IF_ORP_P_P, OP_IF_ORP_R, OP_IF_ORP_N, OP_IF_ORP_N_N,
      OP_IF_OR2_P, OP_IF_OR2_P_P, OP_IF_OR2_R, OP_IF_OR2_N, OP_IF_OR2_N_N,

      OP_IF_PPP, OP_IF_PP, OP_IF_PR, OP_IF_PRR,
      OP_WHEN_PP, OP_UNLESS_PP,

      OP_COND_FX, OP_COND_FX_2, OP_COND_FX_P, OP_COND_FX_1P_ELSE, OP_COND_FX_2P_ELSE,
      OP_SIMPLE_DO, OP_SIMPLE_DO_STEP, OP_SAFE_DOTIMES, OP_SAFE_DOTIMES_STEP, OP_SAFE_DOTIMES_STEP_P, OP_SAFE_DOTIMES_STEP_O,
      OP_SAFE_DO, OP_SAFE_DO_STEP, OP_DOX, OP_DOX_STEP, OP_DOX_STEP_P, OP_SIMPLE_DOX, OP_DOX_INIT,
      OP_DOTIMES_P, OP_DOTIMES_STEP_P,
      OP_DO_NO_VARS, OP_DO_NO_VARS_NO_OPT, OP_DO_NO_VARS_NO_OPT_1,

      OP_SAFE_C_P_1, OP_SAFE_C_PP_1, OP_SAFE_C_PP_3_MV, OP_SAFE_C_PP_5, OP_SAFE_C_PP_6_MV,
      OP_SAFE_C_SP_1, OP_SAFE_C_SP_MV, OP_SAFE_CONS_SP_1, OP_SAFE_MEMQ_SP_1, OP_SAFE_ADD_SP_1, OP_SAFE_MULTIPLY_SP_1, OP_SAFE_SUBTRACT_SP_1,
      OP_SAFE_C_PS_1, OP_SAFE_C_PC_1, OP_SAFE_C_PS_MV, OP_SAFE_C_PC_MV,
      OP_EVAL_MACRO_MV, OP_MACROEXPAND_1, OP_APPLY_LAMBDA,
      OP_SAFE_CLOSURE_P_1, OP_CLOSURE_P_1, OP_SAFE_CLOSURE_AP_1, OP_SAFE_CLOSURE_PA_1,
      OP_INCREMENT_SP_1, OP_INCREMENT_SP_MV,
      OP_SAFE_C_FP_1, OP_SAFE_C_FP_MV_1, OP_SAFE_C_SSP_1, OP_SAFE_C_SSP_MV_1,
      OP_C_P_1, OP_C_P_MV, OP_C_AP_1, OP_NOT_P_1,
      OP_CLOSURE_AP_1, OP_CLOSURE_PA_1,
      OP_CLOSURE_P_MV, OP_CLOSURE_AP_MV, OP_CLOSURE_PA_MV,
      OP_SAFE_C_PA_1, OP_SAFE_C_PA_MV,

      OP_SET_WITH_LET_1, OP_SET_WITH_LET_2, OP_S7_LET,
      OP_MAX_DEFINED_1};

#define OP_MAX_DEFINED (OP_MAX_DEFINED_1 + 1)

typedef enum{E_C_P, E_C_PP, E_C_CP, E_C_SP, E_C_PC, E_C_PS} combine_op_t;

#if S7_DEBUGGING || OP_NAMES

static const char* op_names[OP_MAX_DEFINED_1] =
     {"unopt", "h_unopt", "sym", "h_sym", "con", "h_con",
      "pair_sym", "h_pair_sym", "pair_pair", "h_pair_pair", "pair_any", "h_pair_any",
      "safe_c_d", "h_safe_c_d", "safe_c_and2", "h_safe_c_and2", "safe_c_or2", "h_safe_c_or2",
      "safe_c_s", "h_safe_c_s", "safe_car_s", "h_safe_car_s", "safe_cdr_s", "h_safe_cdr_s", "safe_cadr_s", "h_safe_cadr_s",
      "safe_is_pair_s", "h_safe_is_pair_s", "safe_is_null_s", "h_safe_is_null_s", "safe_is_symbol_s", "h_safe_is_symbol_s",
      "safe_c_ss", "h_safe_c_ss", "safe_c_sc", "h_safe_c_sc", "safe_c_cs", "h_safe_c_cs",
      "safe_c_cq", "h_safe_c_cq", 
      "safe_c_sss", "h_safe_c_sss", "safe_c_scs", "h_safe_c_scs", "safe_c_ssc", "h_safe_c_ssc", "safe_c_css", "h_safe_c_css",
      "safe_c_scc", "h_safe_c_scc", "safe_c_csc", "h_safe_c_csc", "safe_c_ccs", "h_safe_c_ccs",
      "safe_c_all_s", "h_safe_c_all_s",
      "safe_c_opdq", "h_safe_c_opdq", "safe_c_opsq", "h_safe_c_opsq",
      "safe_c_opdq_opsq", "h_safe_c_opdq_opsq", "safe_c_opsq_opdq", "h_safe_c_opsq_opdq",
      "safe_c_opssq", "h_safe_c_opssq", "safe_c_opscq", "h_safe_c_opscq", 
      "safe_c_opcsq", "h_safe_c_opcsq", "safe_c_s_opsq", "h_safe_c_s_opsq",
      "safe_c_c_opscq", "h_safe_c_c_opscq",
      "safe_c_s_opscq", "h_safe_c_s_opscq", "safe_c_s_opcsq", "h_safe_c_s_opcsq",
      "safe_c_opsq_s", "h_safe_c_opsq_s", "safe_c_car_s_s", "h_safe_c_car_s_s", "safe_c_opsq_c", "h_safe_c_opsq_c",
      "safe_c_opsq_opsq", "h_safe_c_opsq_opsq", "safe_c_s_opssq", "h_safe_c_s_opssq", "safe_c_c_opsq", "h_safe_c_c_opsq",
      "safe_c_c_opcsq", "h_safe_c_c_opcsq", "safe_c_opcsq_c", "h_safe_c_opcsq_c",
      "safe_c_s_opdq", "h_safe_c_s_opdq", "safe_c_opssq_c", "h_safe_c_opssq_c", "safe_c_c_opssq", "h_safe_c_c_opssq",
      "safe_c_c_opdq", "h_safe_c_c_opdq", "safe_c_opdq_s", "h_safe_c_opdq_s",
      "safe_c_opdq_opdq", "h_safe_c_opdq_opdq", "safe_c_opdq_c", "h_safe_c_opdq_c",
      "safe_c_opscq_opscq", "h_safe_c_opscq_opscq", "safe_c_opssq_opssq", "h_safe_c_opssq_opssq",
      "safe_c_opssq_opdq", "h_safe_c_opssq_opdq", "safe_c_opssq_opsq", "h_safe_c_opssq_opsq", "safe_c_opsq_opssq", "h_safe_c_opsq_opssq",
      "safe_c_opssq_s", "h_safe_c_opssq_s", "safe_c_opscq_s", "h_safe_c_opscq_s", "safe_c_opcsq_s", "h_safe_c_opcsq_s",
      "safe_c_opscq_c", "h_safe_c_opscq_c", "safe_c_opdq_opssq", "h_safe_c_opdq_opssq",
      "safe_c_s_op_opsq_cq", "h_safe_c_s_op_opsq_cq",
      "safe_c_s_op_s_opssqq", "h_safe_c_s_op_s_opssqq", "safe_c_s_op_s_opsqq", "h_safe_c_s_op_s_opsqq",
      "safe_c_op_opssq_q_c", "h_safe_c_op_opssq_q_c", "safe_c_op_opsq_q_c", "h_safe_c_op_opsq_q_c",
      "safe_c_op_opssq_q_s", "h_safe_c_op_opssq_q_s", "safe_c_op_opsq_q_s", "h_safe_c_op_opsq_q_s",
      "safe_c_s_op_opssq_opssqq", "h_safe_c_s_op_opssq_opssqq", "safe_c_opssq_sq_s", "h_safe_c_opssq_sq_s",
      "safe_c_op_opsq_q", "h_safe_c_op_opsq_q",
      "safe_c_op_s_opsq_q", "h_safe_c_op_s_opsq_q", "safe_c_op_opsq_s_q", "h_safe_c_op_opsq_s_q",
      "safe_c_opsq_cs", "h_safe_c_opsq_cs",

      "safe_c_a", "h_safe_c_a", "safe_c_aa", "h_safe_c_aa", "safe_c_aaa", "h_safe_c_aaa", "safe_c_aaaa", "h_safe_c_aaaa",
      "safe_c_fx", "h_safe_c_fx", "safe_c_all_ca", "h_safe_c_all_ca",
      "safe_c_ssa", "h_safe_c_ssa", "safe_c_sas", "h_safe_c_sas",
      "safe_c_csa", "h_safe_c_csa", "safe_c_sca", "h_safe_c_sca",
      "safe_c_cac", "h_safe_c_cac",
      "safe_c_opaq", "h_safe_c_opaq", "safe_c_opaaq", "h_safe_c_opaaq", "safe_c_opaaaq", "h_safe_c_opaaaq",
      "safe_c_s_opaq", "h_safe_c_s_opaq", "safe_c_opaq_s", "h_safe_c_opaq_s",
      "safe_c_s_opaaq", "h_safe_c_s_opaaq", "safe_c_s_opaaaq", "h_safe_c_s_opaaaq",
      "safe_ifa_ss_a", "h_safe_ifa_ss_a",
      "safe_c*", "h_safe_c*", "safe_c*_a", "h_safe_c*_a", "safe_c*_aa", "h_safe_c*_aa", "safe_c*_fx", "h_safe_c*_fx",
      "safe_c_p", "h_safe_c_p", 

      "thunk", "h_thunk", "thunk_p", "h_thunk_p",
      "safe_thunk", "h_safe_thunk", "safe_thunk_p", "h_safe_thunk_p", "safe_thunk_a", "h_safe_thunk_a",

      "closure_s", "h_closure_s", "closure_s_p", "h_closure_s_p",
      "safe_closure_s", "h_safe_closure_s", "safe_closure_s_p", "h_safe_closure_s_p", "safe_closure_s_a", "h_safe_closure_s_a",
      "closure_c", "h_closure_c", "closure_c_p", "h_closure_c_p",
      "safe_closure_c", "h_safe_closure_c", "safe_closure_c_p", "h_safe_closure_c_p", "safe_closure_c_a", "h_safe_closure_c_a",

      "closure_a", "h_closure_a", "closure_a_p", "h_closure_a_p", "closure_sub_p", "h_closure_sub_p",
      "safe_closure_a", "h_safe_closure_a", "safe_closure_a_p", "h_safe_closure_a_p", "safe_closure_a_a", "h_safe_closure_a_a",

      "closure_p", "h_closure_p", "safe_closure_p", "h_safe_closure_p",
      "closure_ap", "h_closure_ap", "closure_pa", "h_closure_pa",
      "safe_closure_ap", "h_safe_closure_ap", "safe_closure_pa", "h_safe_closure_pa",
      "closure_fa", "h_closure_fa",

      "closure_ss", "h_closure_ss", "closure_ss_p", "h_closure_ss_p",
      "safe_closure_ss", "h_safe_closure_ss", "safe_closure_ss_p", "h_safe_closure_ss_p", "safe_closure_ss_a", "h_safe_closure_ss_a",
      "closure_sc", "h_closure_sc", "closure_sc_p", "h_closure_sc_p",
      "safe_closure_sc", "h_safe_closure_sc", "safe_closure_sc_p", "h_safe_closure_sc_p",
      "closure_cs", "h_closure_cs",
      "safe_closure_cs", "h_safe_closure_cs",
      "closure_aa", "h_closure_aa", "closure_aa_p", "h_closure_aa_p",
      "safe_closure_aa", "h_safe_closure_aa", "safe_closure_aa_p", "h_safe_closure_aa_p", "safe_closure_aa_a", "h_safe_closure_aa_a",

      "closure_fx", "h_closure_fx", "closure_all_s", "h_closure_all_s", "closure_any_fx", "h_closure_any_fx",

      "safe_closure_sa", "h_safe_closure_sa", "safe_closure_saa", "h_safe_closure_saa", "safe_closure_fx", "h_safe_closure_fx",

      "closure*_a", "h_closure*_a", "closure*_fx", "h_closure*_fx",
      "safe_closure*_a", "h_safe_closure*_a", "safe_closure*_aa", "h_safe_closure*_aa",
      "safe_closure*_fx", "h_safe_closure*_fx", "safe_closure*_fx_0", "h_safe_closure*_fx_0",
      "safe_closure*_fx_1", "h_safe_closure*_fx_1", "safe_closure*_fx_2", "h_safe_closure*_fx_2",

      "apply_ss", "h_apply_ss", "apply_sa", "h_apply_sa", "apply_sl", "h_apply_sl",
      "c_fx", "h_c_fx", "call_with_exit", "h_call_with_exit", "call_with_exit_p", "h_call_with_exit_p",
      "c_catch", "h_c_catch", "c_catch_all", "h_c_catch_all", "c_catch_all_p", "h_c_catch_all_p",
      "c_s_opsq", "h_c_s_opsq", "c_s_opdq", "h_c_s_opdq", "c_ss", "h_c_ss",
      "c_s", "h_c_s", "read_s", "h_read_s", "c_p", "h_c_p", "c_ap", "h_c_ap", "c_not", "h_c_not",
      "c_a", "h_c_a", "c_scs", "h_c_scs",
      "c_fa", "h_c_fa", "c_aa", "h_c_aa",

      "safe_c_pp", "h_safe_c_pp",
      "safe_c_opsq_p", "h_safe_c_opsq_p",
      "safe_c_sp", "h_safe_c_sp", "safe_c_cp", "h_safe_c_cp", 
      "safe_c_ap", "h_safe_c_ap", "safe_c_pa", "h_safe_c_pa",
      "safe_c_ps", "h_safe_c_ps", "safe_c_pc", "h_safe_c_pc", 
      "safe_c_ssp", "h_safe_c_ssp", "safe_c_fp", "h_safe_c_fp",
      "s", "s_s", "s_c", "s_a", "c_fa_1", "s_aa",
      "goto", "goto_a",
      "iterate", "continuation_a", "vector_a", "string_a", "c_object_a", "pair_a", "hash_table_a", "environment_c", "environment_a",
      "unknown", "unknown_all_s", "unknown_fx", "unknown_g", "unknown_gg", "unknown_a", "unknown_aa",

      "gc_protect",
      "read_internal", "eval",
      "eval_args", "eval_args1", "eval_args2", "eval_args3", "eval_args4", "eval_args5",
      "apply", "eval_macro", "lambda", "quote", "macroexpand",
      "define", "define1", "begin", "begin0", "begin1", "begin_unchecked",
      "if", "if1", "when", "unless", "set", "set1", "set2",
      "let", "let1", "let*", "let*1", "let*2",
      "letrec", "letrec1", "letrec*", "letrec*1",
      "let_temporarily", "let_temp_unchecked", "let_temp_init1", "let_temp_init2", "let_temp_done", "let_temp_done1",
      "cond", "cond1", "feed_to_1", "cond_simple", "cond1_simple", "cond_simple_p", "cond1_simple_p",
      "and", "and1", "or", "or1",
      "define_macro", "define_macro*", "define_expansion",
      "case", "read_list", "read_next", "read_dot", "read_quote",
      "read_quasiquote", "read_unquote", "read_apply_values",
      "read_vector", "read_byte_vector", "read_int_vector", "read_float_vector", "read_done",
      "load_return_if_eof", "load_close_and_pop_if_eof", "eval_done",
      "catch", "dynamic_wind", "define_constant", "define_constant1",
      "do", "do_end", "do_end1", "do_step", "do_step2", "do_init",
      "define*", "lambda*", "lambda*_default", "error_quit", "unwind_input", "unwind_output",
      "error_hook_quit",
      "with_let", "with_let1", "with_let_unchecked", "with_let_s",
      "with_baffle", "with_baffle_unchecked", "expansion",
      "for_each", "for_each_1", "for_each_2", "for_each_3",
      "map", "map_1", "map_2", "map_gather", "map_gather_1", "map_gather_2", "map_gather_3",
      "barrier", "deactivate_goto",

      "define_bacro", "define_bacro*",
      "get_output_string",
      "sort", "sort1", "sort2", "sort3", "sort_pair_end", "sort_vector_end", "sort_string_end",
      "eval_string",
      "member_if", "assoc_if", "member_if1", "assoc_if1",

      "lambda_unchecked", "let_unchecked",
      "catch_1", "catch_2", "catch_all",

      "set_unchecked", "set_symbol_c", "set_symbol_s", "set_symbol_p", "set_symbol_a",
      "set_symbol_opsq", "set_symbol_opdq", "set_symbol_opssq",
      "set_normal", "set_pair", "set_dilambda", "set_dilambda_p", "set_dilambda_p_1",
      "set_pair_a", "set_pair_p", "set_pair_za",
      "set_pair_p_1", "set_with_setter", "set_pws", "set_let_s", "set_let_fx",
      "set_safe",
      "increment_1", "decrement_1", "set_cons",
      "increment_ss", "increment_sss", "increment_sp", "increment_sa", "increment_saa",

      "letrec_unchecked", "letrec*_unchecked", "cond_unchecked",
      "lambda*_unchecked", "do_unchecked", "define_unchecked", "define*_unchecked", "define_funchecked", "define_constant_unchecked",
      "define_with_setter", "define_macro_with_setter",

      "let_no_vars", "named_let", "named_let_no_vars", "named_let*",
      "let_c", "let_s", "let_s_p", "let_all_c", "let_all_s", "let_fx",
      "let*_fx", "let*_a2", "let*_a", "let_opdq", "let_opssq", "let_opssq_e", "let_opassq", "let_opassq_e",
      "let_opsq", "let_all_opsq", "let_opsq_p", "let_car", "let_one", "let_one_1", "let_one_p", "let_one_p_1",
      "let_a", "let_a_p",

      "case_a_e_s", "case_a_i_s", "case_a_g_s", "case_a_e_g", "case_a_g_g",
      "case_s_e_s", "case_s_i_s", "case_s_g_s", "case_s_e_g", "case_s_g_g",
      "case_p_e_s", "case_p_i_s", "case_p_g_s", "case_p_e_g", "case_p_g_g",
      "case_e_s", "case_i_s", "case_g_s", "case_e_g", "case_g_g",

      "if_unchecked", "and_p", "and_p1", "and_ap", "and_safe_p", "and_safe_aa", "and_pair_p",
      "or_p", "or_p1", "or_ap", "or_safe_p", "or_safe_aa",
      "cond_feed", "cond_feed_1", "when_s", "when_a", "when_p", "unless_s", "unless_a", "unless_p",

      "if_s_p", "if_s_p_p", "if_s_r", "if_s_n", "if_s_n_n",
      "if_c_p", "if_c_p_p", "if_c_r", "if_c_n", "if_c_n_n",
      "if_cs_p", "if_cs_p_p", "if_cs_r", "if_cs_n", "if_cs_n_n",
      "if_css_p", "if_css_p_p", "if_css_r","if_css_n", "if_css_n_n",
      "if_csc_p", "if_csc_p_p", "if_csc_r", "if_csc_n", "if_csc_n_n",
      "if_opsq_p", "if_opsq_p_p", "if_opsq_r", "if_opsq_n", "if_opsq_n_n",
      "if_s_opdq_p", "if_s_opdq_p_p", "if_s_opdq_r","if_s_opdq_n", "if_s_opdq_n_n",
      "if_is_type_s_p", "if_is_type_s_p_p", "if_is_type_s_r", "if_is_type_s_n", "if_is_type_s_n_n",
      "if_is_type_opsq_p", "if_is_type_opsq_p_p", "if_is_type_opsq_r", "if_is_type_opsq_n", "if_is_type_opsq_n_n",
      "if_a_p", "if_a_p_p", "if_a_r", "if_a_n", "if_a_n_n",
      "if_and2_p", "if_and2_p_p", "if_and2_r","if_and2_n", "if_and2_n_n",
      "if_and3_p", "if_and3_p_p", "if_and3_r","if_and3_n", "if_and3_n_n",
      "if_p_p", "if_p_p_p", "if_p_r", "if_p_n", "if_p_n_n",
      "if_andp_p", "if_andp_p_p", "if_andp_r", "if_andp_n", "if_andp_n_n",
      "if_orp_p", "if_orp_p_p", "if_orp_r","if_orp_n", "if_orp_n_n",
      "if_or2_p", "if_or2_p_p", "if_or2_r","if_or2_n", "if_or2_n_n",

      "if_ppp", "if_pp", "if_pr", "if_prr",
      "when_pp", "unless_pp",

      "cond_fx", "cond_fx_2", "cond_fx_p", "cond_fx_1p_else", "cond_fx_2p_else",
      "simple_do", "simple_do_step", "safe_dotimes", "safe_dotimes_step", "safe_dotimes_step_p", "safe_dotimes_step_o",
      "safe_do", "safe_do_step", "dox", "dox_step", "dox_step_p", "simple_dox", "dox_init",
      "dotimes_p", "dotimes_step_p",
      "do_no_vars", "do_no_vars_no_opt", "do_no_vars_no_opt_1",

      "safe_c_p_1", "safe_c_pp_1", "safe_c_pp_3_mv", "safe_c_pp_5", "safe_c_pp_6_mv",
      "safe_c_sp_1", "safe_c_sp_mv", "safe_cons_sp_1", "safe_memq_sp_1", "safe_add_sp_1", "safe_multiply_sp_1", "safe_subtract_sp_1",
      "safe_c_ps_1", "safe_c_pc_1", "safe_c_ps_mv", "safe_c_pc_mv",
      "eval_macro_mv", "macroexpand_1", "apply_lambda",
      "safe_closure_p_1", "closure_p_1", "safe_closure_ap_1", "safe_closure_pa_1",
      "increment_sp_1", "increment_sp_mv",
      "safe_c_fp_1", "safe_c_fp_mv_1", "safe_c_ssp_1", "safe_c_ssp_mv_1",
      "c_p_1", "c_p_mv", "c_ap_1", "not_1",
      "closure_ap_1", "closure_pa_1",
      "closure_p_mv", "closure_ap_mv", "closure_pa_mv",
      "safe_c_pa_1", "safe_c_pa_mv",

      "set_with_let_1", "set_with_let_2", "*s7*",
};
#endif

#define op_names op_names
#define OPT_MAX_DEFINED OP_GC_PROTECT

#define in_reader(Sc)        ((sc->cur_op >= OP_READ_LIST) && (sc->cur_op <= OP_READ_DONE) && (is_input_port(Sc->input_port)))
#define is_safe_c_op(op)     ((op >= OP_SAFE_C_D) && (op < OP_THUNK))
#define is_unknown_op(op)    ((op >= OP_UNKNOWN) && (op <= OP_UNKNOWN_AA))
#define is_callable_c_op(op) ((is_safe_c_op(op)) || (op >= OP_SAFE_C_PP)) /* used only in check_set */
#define is_fxa_op(op)        ((op < OP_SAFE_C_D) || (op >= OP_SAFE_C_A))
#define is_h_safe_c_d(P)     ((is_optimized(P)) && (optimize_op(P) >= HOP_SAFE_C_D) && (optimize_op(P) < OP_SAFE_C_S) && ((optimize_op(P) & 1) != 0))
#define is_h_safe_c_s(P)     ((is_optimized(P)) && (optimize_op(P) >= HOP_SAFE_C_S) && (optimize_op(P) <= HOP_SAFE_IS_SYMBOL_S) && ((optimize_op(P) & 1) != 0))
#define is_safe_c_s(P)       ((is_optimized(P)) && (optimize_op(P) >= HOP_SAFE_C_S) && (optimize_op(P) <= HOP_SAFE_IS_SYMBOL_S))

static bool is_h_optimized(s7_pointer p)
{
  return((is_optimized(p)) &&
	 ((optimize_op(p) & 1) != 0) &&
	 (!is_unknown_op(optimize_op(p))));
}

/* -------- */
static s7_pointer set_elist_1(s7_scheme *sc, s7_pointer x1)
{
  set_car(sc->elist_1, x1);
  return(sc->elist_1);
}

static s7_pointer set_elist_2(s7_scheme *sc, s7_pointer x1, s7_pointer x2)
{
  set_car(sc->elist_2, x1);
  set_cadr(sc->elist_2, x2);
  return(sc->elist_2);
}

static s7_pointer set_elist_3(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3)
{
  s7_pointer p;
  p = sc->elist_3;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3);
  return(sc->elist_3);
}

static s7_pointer set_elist_4(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4)
{
  s7_pointer p;
  p = sc->elist_4;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3); p = cdr(p);
  set_car(p, x4);
  return(sc->elist_4);
}

static s7_pointer set_elist_5(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4, s7_pointer x5)
{
  s7_pointer p;
  p = sc->elist_5;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3); p = cdr(p);
  set_car(p, x4); p = cdr(p);
  set_car(p, x5);
  return(sc->elist_5);
}

static s7_pointer set_wlist_3(s7_pointer lst, s7_pointer x1, s7_pointer x2, s7_pointer x3)
{
  s7_pointer p;
  p = lst;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3);
  return(lst);
}

static s7_pointer set_wlist_4(s7_pointer lst, s7_pointer x1, s7_pointer x2, s7_pointer x3, s7_pointer x4)
{
  s7_pointer p;
  p = lst;
  set_car(p, x1); p = cdr(p);
  set_car(p, x2); p = cdr(p);
  set_car(p, x3); p = cdr(p);
  set_car(p, x4);
  return(lst);
}

static s7_pointer set_plist_1(s7_scheme *sc, s7_pointer x1)
{
  set_car(sc->plist_1, x1);
  return(sc->plist_1);
}

static s7_pointer set_plist_2(s7_scheme *sc, s7_pointer x1, s7_pointer x2)
{
  set_car(sc->plist_2, x1);
  set_car(sc->plist_2_2, x2);
  return(sc->plist_2);
}

static s7_pointer set_qlist_2(s7_scheme *sc, s7_pointer x1, s7_pointer x2)
{
  set_car(sc->qlist_2, x1);
  set_cadr(sc->qlist_2, x2);
  return(sc->qlist_2);
}

static s7_pointer set_clist_1(s7_scheme *sc, s7_pointer x1)
{
  set_car(sc->clist_1, x1);
  return(sc->clist_1);
}

static s7_pointer set_plist_3(s7_scheme *sc, s7_pointer x1, s7_pointer x2, s7_pointer x3)
{
  return(set_wlist_3(sc->plist_3, x1, x2, x3));
}


static int32_t position_of(s7_pointer p, s7_pointer args)
{
  int32_t i;
  for (i = 1; p != args; i++, args = cdr(args));
  return(i);
}

s7_pointer s7_method(s7_scheme *sc, s7_pointer obj, s7_pointer method)
{
  if (has_methods(obj))
    return(find_method(sc, find_let(sc, obj), method));
  return(sc->undefined);
}


/* if a method is shadowing a built-in like abs, it should expect the same args as abs and
 *   behave the same -- no multiple values etc.
 */
static s7_pointer copy_list(s7_scheme *sc, s7_pointer lst);

#define check_method(Sc, Obj, Method, Args)		\
  {							\
    s7_pointer func;					\
    if ((has_methods(Obj)) &&				\
	((func = find_method(Sc, find_let(Sc, Obj), Method)) != Sc->undefined)) \
      return(s7_apply_function(Sc, func, copy_list(Sc, Args))); \
  }

#define check_method_uncopied(Sc, Obj, Method, Args)		\
  {							\
    s7_pointer func;					\
    if ((has_methods(Obj)) &&				\
	((func = find_method(Sc, find_let(Sc, Obj), Method)) != Sc->undefined)) \
      return(s7_apply_function(Sc, func, Args)); \
  }

#define apply_known_method(Sc, Let, Method, Args) return(s7_apply_function(Sc, find_method(Sc, Let, Method), Args))

static s7_pointer check_value_slot(s7_scheme *sc, s7_pointer obj)
{
  if (has_methods(obj))
    return(s7_let_ref(sc, find_let(sc, obj), sc->value_symbol));
  return(sc->gc_nil);
}

/* unfortunately, in the simplest cases, where a function (like number?) accepts any argument,
 *   this costs about a factor of 1.5 in speed (we're doing the normal check like s7_is_number,
 *   but then have to check has_methods before returning #f).  We can't use the old form until
 *   openlet is seen because the prior code might use #_number? which gets the value
 *   before the switch.  These simple functions normally do not dominate timing info, so I'll
 *   go ahead. It's mostly boilerplate:
 */

static s7_pointer apply_boolean_method(s7_scheme *sc, s7_pointer obj, s7_pointer method)
{
  s7_pointer func;
  func = find_method(sc, find_let(sc, obj), method);
  if (func == sc->undefined) return(sc->F);
  return(s7_apply_function(sc, func, list_1(sc, obj)));
}

static s7_pointer missing_method_error(s7_scheme *sc, s7_pointer method, s7_pointer obj)
{
  return(s7_error(sc, sc->missing_method_symbol, set_elist_3(sc, missing_method_string, method, obj)));
}

#define check_boolean_method(Sc, Checker, Method, Args)	       \
  {							       \
    s7_pointer p;					       \
    p = car(Args);					       \
    if (Checker(p)) return(Sc->T);			       \
    if (!has_methods(p)) return(Sc->F);			       \
    return(apply_boolean_method(Sc, p, Method));	       \
  }

#define check_boolean_not_method(Sc, Checker, Method, Args)		\
  {									\
    s7_pointer p;							\
    p = symbol_to_value_unchecked(sc, cadar(Args));			\
    if (Checker(p)) return(Sc->F);					\
    if (!has_methods(p)) return(Sc->T);					\
    return((apply_boolean_method(Sc, p, Method) == sc->F) ? sc->T : sc->F); \
  }

static s7_pointer find_and_apply_method(s7_scheme *sc, s7_pointer lt, s7_pointer sym, s7_pointer args)
{
  s7_pointer func;
  func = find_method(sc, lt, sym);
  if (func != sc->undefined)
    return(s7_apply_function(sc, func, args));
  return(missing_method_error(sc, sym, lt));
}

static s7_pointer method_or_bust(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer args, uint8_t typ, int32_t num)
{
  if (has_methods(obj))
    return(find_and_apply_method(sc, find_let(sc, obj), method, args));
  return(wrong_type_argument(sc, method, num, obj, typ));
}

static s7_pointer immutable_object_error(s7_scheme *sc, s7_pointer info) {return(s7_error(sc, sc->error_symbol, info));}

static s7_pointer mutable_method_or_bust(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer args, uint8_t typ, int32_t num)
{
  if (has_methods(obj))
    return(find_and_apply_method(sc, find_let(sc, obj), method, args));
  if (type(obj) != typ)
    return(wrong_type_argument(sc, method, num, obj, typ));
  if (is_immutable(obj))
    return(immutable_object_error(sc, set_elist_3(sc, immutable_error_string, method, obj)));
  return(wrong_type_argument(sc, method, num, obj, typ));
}

static s7_pointer method_or_bust_one_arg(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer args, uint8_t typ)
{
  if (has_methods(obj))
    return(find_and_apply_method(sc, find_let(sc, obj), method, args));
  return(simple_wrong_type_argument(sc, method, obj, typ));
}

static s7_pointer method_or_bust_with_type(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer args, s7_pointer typ, int32_t num)
{
  if (has_methods(obj))
    return(find_and_apply_method(sc, find_let(sc, obj), method, args));
  return(wrong_type_argument_with_type(sc, method, num, obj, typ));
}

static s7_pointer method_or_bust_with_type_one_arg(s7_scheme *sc, s7_pointer obj, s7_pointer method, s7_pointer args, s7_pointer typ)
{
  if (has_methods(obj))
    return(find_and_apply_method(sc, find_let(sc, obj), method, args));
  return(simple_wrong_type_argument_with_type(sc, method, obj, typ));
}


#define eval_error_any(Sc, ErrType, ErrMsg, Len, Obj) \
  return(s7_error(Sc, ErrType, set_elist_2(Sc, wrap_string(Sc, ErrMsg, Len), Obj)))

#define eval_error(Sc, ErrMsg, Len, Obj) \
  eval_error_any(Sc, Sc->syntax_error_symbol, ErrMsg, Len, Obj)

#define eval_type_error(Sc, ErrMsg, Len, Obj) \
  eval_error_any(Sc, Sc->wrong_type_arg_symbol, ErrMsg, Len, Obj)

#define eval_range_error(Sc, ErrMsg, Len, Obj) \
  eval_error_any(Sc, Sc->out_of_range_symbol, ErrMsg, Len, Obj)

#define eval_error_no_return(Sc, ErrType, ErrMsg, Len, Obj) \
  s7_error(Sc, ErrType, set_elist_2(Sc, wrap_string(Sc, ErrMsg, Len), Obj))

#define eval_error_with_caller(Sc, ErrMsg, Len, Caller, Obj)	\
  return(s7_error(Sc, Sc->syntax_error_symbol, set_elist_3(Sc, wrap_string(Sc, ErrMsg, Len), Caller, Obj)))

#define eval_error_with_caller2(Sc, ErrMsg, Len, Caller, Name, Obj)	\
  return(s7_error(Sc, Sc->syntax_error_symbol, set_elist_4(Sc, wrap_string(Sc, ErrMsg, Len), Caller, Name, Obj)))



/* -------------------------------- constants -------------------------------- */

s7_pointer s7_f(s7_scheme *sc) {return(sc->F);}
s7_pointer s7_t(s7_scheme *sc) {return(sc->T);}

s7_pointer s7_nil(s7_scheme *sc)             {return(sc->nil);}
bool s7_is_null(s7_scheme *sc, s7_pointer p) {return(is_null(p));}
static bool is_null_b(s7_pointer p)          {return(type(p) == T_NIL);}

s7_pointer s7_undefined(s7_scheme *sc)                {return(sc->undefined);}
s7_pointer s7_unspecified(s7_scheme *sc)              {return(sc->unspecified);}
bool s7_is_unspecified(s7_scheme *sc, s7_pointer val) {return(is_unspecified(val));}

static s7_pointer g_is_undefined(s7_scheme *sc, s7_pointer args)
{
  #define H_is_undefined "(undefined? val) returns #t if val is #<undefined> or its reader equivalent"
  #define Q_is_undefined sc->pl_bt
  check_boolean_method(sc, is_undefined, sc->is_undefined_symbol, args);
}

static s7_pointer g_is_unspecified(s7_scheme *sc, s7_pointer args)
{
  #define H_is_unspecified "(unspecified? val) returns #t if val is #<unspecified>"
  #define Q_is_unspecified sc->pl_bt
  check_boolean_method(sc, is_unspecified, sc->is_unspecified_symbol, args);
}


/* -------------------------------- eof-object? -------------------------------- */
s7_pointer eof_object = NULL;

s7_pointer s7_eof_object(s7_scheme *sc) {return(eof_object);}

static s7_pointer g_is_eof_object(s7_scheme *sc, s7_pointer args)
{
  #define H_is_eof_object "(eof-object? val) returns #t if val is the end-of-file object"
  #define Q_is_eof_object sc->pl_bt
  check_boolean_method(sc, is_eof, sc->is_eof_object_symbol, args);
}

static bool s7_is_eof_object(s7_pointer p) {return(p == eof_object);}

/* -------------------------------- not -------------------------------- */
static s7_pointer g_not(s7_scheme *sc, s7_pointer args)
{
  #define H_not "(not obj) returns #t if obj is #f, otherwise #t: (not ()) -> #f"
  #define Q_not sc->pl_bt
  return((car(args) == sc->F) ? sc->T : sc->F);
}

static bool not_b_7p(s7_scheme *sc, s7_pointer p) {return(p == sc->F);}


bool s7_boolean(s7_scheme *sc, s7_pointer x) {return(x != sc->F);}
s7_pointer s7_make_boolean(s7_scheme *sc, bool x) {return(make_boolean(sc, x));}


/* -------------------------------- boolean? -------------------------------- */
bool s7_is_boolean(s7_pointer x)
{
  return(type(x) == T_BOOLEAN);
}

static s7_pointer g_is_boolean(s7_scheme *sc, s7_pointer args)
{
  #define H_is_boolean "(boolean? obj) returns #t if obj is #f or #t: (boolean? ()) -> #f"
  #define Q_is_boolean sc->pl_bt
  check_boolean_method(sc, s7_is_boolean, sc->is_boolean_symbol, args);
}

/* -------------------------------- constant? -------------------------------- */
static inline bool is_constant_symbol(s7_scheme *sc, s7_pointer sym)
{
  if (is_immutable_symbol(sym))    /* for keywords */
    return(true);
  if (is_possibly_constant(sym))
    {
      s7_pointer slot;
      slot = symbol_to_slot(sc, sym);
      return((is_slot(slot)) && (is_immutable_slot(slot)));
    }
  return(false);
}

#define is_constant(sc, p) ((type(p) != T_SYMBOL) || (is_constant_symbol(sc, p)))

static s7_pointer g_is_constant(s7_scheme *sc, s7_pointer args)
{
  #define H_is_constant "(constant? obj) returns #t if obj either evaluates to itself, or is a symbol whose binding is constant"
  #define Q_is_constant sc->pl_bt
  return(make_boolean(sc, is_constant(sc, car(args))));
}

static s7_pointer is_constant_p_p(s7_scheme *sc, s7_pointer p) {return(make_boolean(sc, is_constant(sc, p)));}

/* -------------------------------- immutable? -------------------------------- */
bool s7_is_immutable(s7_pointer p)
{
  return(is_immutable(p));
}

static s7_pointer g_is_immutable(s7_scheme *sc, s7_pointer args)
{
  #define H_is_immutable "(immutable? sequence) returns #t if the sequence is immutable. (This function is work-in-progress)"
  #define Q_is_immutable sc->pl_bt
  return((is_immutable(car(args))) ? sc->T : sc->F);
}

/* -------------------------------- immutable! -------------------------------- */
s7_pointer s7_immutable(s7_pointer p)
{
  set_immutable(p);
  return(p);
}

static s7_pointer g_immutable(s7_scheme *sc, s7_pointer args)
{
  #define H_immutable "(immutable! sequence) declares that the sequence's entries can't be changed. The sequence is returned."
  #define Q_immutable s7_make_signature(sc, 2, sc->T, sc->T)
  s7_pointer p;
  p = car(args);
  if (is_symbol(p))
    {
      s7_pointer slot;
      slot = symbol_to_slot(sc, p);
      if (is_slot(slot))
	{
	  set_immutable(slot);
	  return(p);
	}
    }
  return(s7_immutable(p));
}


/* -------------------------------- GC -------------------------------- */

/* in most code, pairs, lets, and slots dominate the heap -- each about 25% to 40% of the
 *   total cell allocations.  In snd-test, reals are 50%. slots need not be in the heap,
 *   but moving them out to their own free list was actually slower because we need (in that
 *   case) to manage them in the sweep process by tracking lets.
 */

#if S7_DEBUGGING
static s7_int s7_gc_protect_2(s7_scheme *sc, s7_pointer x, int32_t line)
{
  s7_int loc;
  loc = s7_gc_protect(sc, x);
  if (loc > 8192)
    {
      fprintf(stderr, "infinite loop at line %d %s?\n", line, string_value(s7_object_to_string(sc, current_code(sc), false)));
      abort();
    }
  return(loc);
}
#define s7_gc_protect_1(Sc, X) s7_gc_protect_2(Sc, X, __LINE__)
#else
#define s7_gc_protect_1(Sc, X) s7_gc_protect(Sc, X)
#endif

static void resize_gc_protect(s7_scheme *sc)
{
  s7_int i, size, new_size;
  block_t *ob, *nb;
  size = sc->protected_objects_size;
  new_size = 2 * size;
  ob = vector_block(sc->protected_objects);
  nb = reallocate(sc, ob, new_size * sizeof(s7_pointer));
  block_info(nb) = NULL;
  vector_block(sc->protected_objects) = nb;
  vector_elements(sc->protected_objects) = (s7_pointer *)block_data(nb);
  vector_length(sc->protected_objects) = new_size;
  sc->protected_objects_size = new_size;
  sc->gpofl = (s7_int *)realloc(sc->gpofl, new_size * sizeof(s7_int));
  for (i = size; i < new_size; i++)
    {
      vector_element(sc->protected_objects, i) = sc->gc_nil;
      sc->gpofl[++sc->gpofl_loc] = i;
    }
}

s7_inline s7_int s7_gc_protect(s7_scheme *sc, s7_pointer x)
{
  s7_int loc;
  if (sc->gpofl_loc < 0)
    resize_gc_protect(sc);
  loc = sc->gpofl[sc->gpofl_loc--];
  vector_element(sc->protected_objects, loc) = x;
  return(loc);
}

void s7_gc_unprotect(s7_scheme *sc, s7_pointer x)
{
  s7_int i;
  for (i = 0; i < sc->protected_objects_size; i++)
    if (vector_element(sc->protected_objects, i) == x)
      {
	vector_element(sc->protected_objects, i) = sc->gc_nil;
	sc->gpofl[++sc->gpofl_loc] = i;
	return;
      }
}

void s7_gc_unprotect_at(s7_scheme *sc, s7_int loc)
{
  if (loc < sc->protected_objects_size)
    {
      if (vector_element(sc->protected_objects, loc) != sc->gc_nil)
	sc->gpofl[++sc->gpofl_loc] = loc;
      vector_element(sc->protected_objects, loc) = sc->gc_nil;
    }
}

s7_pointer s7_gc_protected_at(s7_scheme *sc, s7_int loc)
{
  s7_pointer obj;

  obj = sc->unspecified;
  if (loc < sc->protected_objects_size)
    obj = vector_element(sc->protected_objects, loc);
  if (obj == sc->gc_nil)
    return(sc->unspecified);

  return(obj);
}

#define gc_protected_at(Sc, Loc) vector_element(Sc->protected_objects, Loc)


static void (*mark_function[NUM_TYPES])(s7_pointer p);

void s7_mark(s7_pointer p)
{
  if (!is_marked(p))
    (*mark_function[unchecked_type(p)])(p);
}

static inline void gc_mark(s7_pointer p)
{
  if (!is_marked(p))
    (*mark_function[unchecked_type(p)])(p);
}

static inline void mark_slot(s7_pointer p)
{
  set_mark(p);
  gc_mark(slot_value(p));
  if (slot_has_setter(p))
    gc_mark(slot_setter(p));
  set_mark(slot_symbol(p));
}

static void mark_noop(s7_pointer p) {}

static void close_output_port(s7_scheme *sc, s7_pointer p);
static void clear_weak_hash_table(s7_scheme *sc, s7_pointer table);
static void remove_gensym_from_symbol_table(s7_scheme *sc, s7_pointer sym);

static void sweep(s7_scheme *sc)
{
  s7_int i, j;
  s7_pointer s1;
  gc_list *gp;

  gp = sc->strings;
  if (gp->loc > 0)
    {
      /* unrolling this loop (even via LOOP_8) is not an improvement */
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s1 = gp->list[i];
	  if (is_free_and_clear(s1))
	    liberate(sc, string_block(s1));
	  else gp->list[j++] = s1;
	}
      gp->loc = j;
    }

  gp = sc->gensyms;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s1 = gp->list[i];
	  if (is_free_and_clear(s1))
	    {
	      remove_gensym_from_symbol_table(sc, s1); /* this uses symbol_name_cell data */
	      liberate(sc, gensym_block(s1));
	    }
	  else gp->list[j++] = s1;
	}
      gp->loc = j;
      if (j == 0) mark_function[T_SYMBOL] = mark_noop;
    }

  gp = sc->unknowns;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s1 = gp->list[i];
	  if (is_free_and_clear(s1))
	    free(unknown_name(s1));
	  else gp->list[j++] = s1;
	}
      gp->loc = j;
    }

  gp = sc->c_objects;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s1 = gp->list[i];
	  if (is_free_and_clear(s1))
	    (*(c_object_free(sc, s1)))(c_object_value(s1));
	  else gp->list[j++] = s1;
	}
      gp->loc = j;
    }

  gp = sc->lambdas;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s1 = gp->list[i];
	  if (is_free_and_clear(s1))
	    liberate(sc, c_function_block(s1));
	  else gp->list[j++] = s1;
	}
      gp->loc = j;
    }

  gp = sc->vectors;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s1 = gp->list[i];

	  if (is_free_and_clear(s1))
	    liberate(sc, vector_block(s1));
	  else gp->list[j++] = s1;
	}
      gp->loc = j;
    }

  gp = sc->multivectors;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s1 = gp->list[i];
	  if (is_free_and_clear(s1))
	    {
	      vdims_t *info;
	      info = vector_dimension_info(s1);  /* a multidimensional empty vector can have dimension info, wrapped vectors always have dimension info */
	      if ((info) &&
		  (info != sc->wrap_only))
		{
		  if (vector_elements_should_be_freed(info)) /* a kludge for foreign code convenience */
		    {
		      free(vector_elements(s1));
		      vector_elements_should_be_freed(info) = false;
		    }
		  liberate(sc, info);
		  vector_set_dimension_info(s1, NULL);
		}
	      liberate(sc, vector_block(s1));
	    }
	  else gp->list[j++] = s1;
	}
      gp->loc = j;
    }

  gp = sc->hash_tables;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s1 = gp->list[i];
	  if (is_free_and_clear(s1))
	    free_hash_table(sc, s1);
	  else
	    {
	      if (is_weak_hash_table(s1))
		clear_weak_hash_table(sc, s1);
	      gp->list[j++] = s1;
	    }
	}
      gp->loc = j;
    }

  gp = sc->input_ports;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s1 = gp->list[i];
	  if (is_free_and_clear(s1))
	    {
	      if (!port_is_closed(s1))
		{
		  if (is_file_port(s1))
		    {
		      if (port_file(s1))
			{
			  fclose(port_file(s1));
			  port_file(s1) = NULL;
			}
		    }
		  else
		    {
		      if ((is_string_port(s1)) &&
			  (port_needs_unprotect(s1)))
			{
			  s7_gc_unprotect_at(sc, port_gc_loc(s1));
			  port_needs_unprotect(s1) = false;
			}
		    }
		}
	      if (port_needs_free(s1))
		{
		  if (port_data(s1))
		    {
		      liberate(sc, port_data_block(s1));
		      port_data_block(s1) = NULL;
		      port_data(s1) = NULL;
		      port_data_size(s1) = 0;
		    }
		  port_needs_free(s1) = false;
		}
	      if (port_filename(s1))
		{
		  liberate(sc, port_filename_block(s1));
		  port_filename(s1) = NULL;
		}
	      liberate(sc, port_block(s1));
	    }
	  else gp->list[j++] = s1;
	}
      gp->loc = j;
    }

  gp = sc->output_ports;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s1 = gp->list[i];
	  if (is_free_and_clear(s1))
	    {
	      close_output_port(sc, s1); /* needed for free filename, etc */
	      liberate(sc, port_block(s1));
	      if (port_needs_free(s1))
		{
		  if (port_data_block(s1))
		    {
		      liberate(sc, port_data_block(s1));
		      port_data_block(s1) = NULL;
		    }
		  port_needs_free(s1) = false;
		}
	    }
	  else gp->list[j++] = s1;
	}
      gp->loc = j;
    }

  gp = sc->continuations;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s1 = gp->list[i];
	  if (is_free_and_clear(s1))
	    {
	      if (continuation_op_stack(s1))
		{
		  free(continuation_op_stack(s1));
		  continuation_op_stack(s1) = NULL;
		}
	      liberate_block(sc, continuation_block(s1));
	    }
	  else gp->list[j++] = s1;
	}
      gp->loc = j;
    }

  gp = sc->optlists;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s1 = gp->list[i];
	  if ((is_free_and_clear(s1)) &&
	      (opt2_any_unchecked(s1)))
	    {
	      liberate(sc, (block_t *)opt2_any_unchecked(s1));
	      set_opt2_any_unchecked(s1, NULL);
	    }
	  else gp->list[j++] = s1;
	}
      gp->loc = j;
    }

  gp = sc->weak_refs;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s1 = gp->list[i];
	  if (!is_free_and_clear(s1))
	    {
	      if (is_free_and_clear(c_pointer_weak1(s1)))
		c_pointer_weak1(s1) = sc->F;
	      if (is_free_and_clear(c_pointer_weak2(s1)))
		c_pointer_weak2(s1) = sc->F;
	      if ((c_pointer_weak1(s1) != sc->F) ||
		  (c_pointer_weak2(s1) != sc->F))
		gp->list[j++] = s1;
	    }
	}
      gp->loc = j;
    }

#if WITH_GMP
  gp = sc->bigints;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s7_pointer s1;
	  s1 = gp->list[i];
	  if (is_free_and_clear(s1))
	    mpz_clear(big_integer(s1));
	  else gp->list[j++] = s1;
	}
      gp->loc = j;
    }

  gp = sc->bigratios;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s7_pointer s1;
	  s1 = gp->list[i];
	  if (is_free_and_clear(s1))
	    mpq_clear(big_ratio(s1));
	  else gp->list[j++] = s1;
	}
      gp->loc = j;
    }

  gp = sc->bigreals;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s7_pointer s1;
	  s1 = gp->list[i];
	  if (is_free_and_clear(s1))
	    mpfr_clear(big_real(s1));
	  else gp->list[j++] = s1;
	}
      gp->loc = j;
    }

  gp = sc->bignumbers;
  if (gp->loc > 0)
    {
      for (i = 0, j = 0; i < gp->loc; i++)
	{
	  s7_pointer s1;
	  s1 = gp->list[i];
	  if (is_free_and_clear(s1))
	    mpc_clear(big_complex(s1));
	  else gp->list[j++] = s1;
	}
      gp->loc = j;
    }
#endif
}

static inline void add_to_gc_list(gc_list *gp, s7_pointer p)
{
  if (gp->loc == gp->size)
    {
      gp->size *= 2;
      gp->list = (s7_pointer *)realloc(gp->list, gp->size * sizeof(s7_pointer));
    }
  gp->list[gp->loc++] = p;
}

static gc_list *make_gc_list(void)
{
  gc_list *gp;
  #define INIT_GC_CACHE_SIZE 4
  gp = (gc_list *)malloc(sizeof(gc_list));
  gp->size = INIT_GC_CACHE_SIZE;
  gp->loc = 0;
  gp->list = (s7_pointer *)malloc(gp->size * sizeof(s7_pointer));
  return(gp);
}

static void just_mark(s7_pointer p)
{
  set_mark(p);
}

static void add_gensym(s7_scheme *sc, s7_pointer p)
{
  add_to_gc_list(sc->gensyms, p);
  mark_function[T_SYMBOL] = just_mark;
}


#define add_c_object(sc, p)        add_to_gc_list(sc->c_objects, p)
#define add_hash_table(sc, p)      add_to_gc_list(sc->hash_tables, p)
#define add_string(sc, p)          add_to_gc_list(sc->strings, p)
#define add_input_port(sc, p)      add_to_gc_list(sc->input_ports, p)
#define add_output_port(sc, p)     add_to_gc_list(sc->output_ports, p)
#define add_continuation(sc, p)    add_to_gc_list(sc->continuations, p)
#define add_unknown(sc, p)         add_to_gc_list(sc->unknowns, p)
#define add_vector(sc, p)          add_to_gc_list(sc->vectors, p)
#define add_multivector(sc, p)     add_to_gc_list(sc->multivectors, p)
#define add_lambda(sc, p)          add_to_gc_list(sc->lambdas, p)
#define add_optlist(sc, p)         add_to_gc_list(sc->optlists, p)
#define add_weak_ref(sc, p)        add_to_gc_list(sc->weak_refs, p)

#if WITH_GMP
#define add_bigint(sc, p)          add_to_gc_list(sc->bigints, p)
#define add_bigratio(sc, p)        add_to_gc_list(sc->bigratios, p)
#define add_bigreal(sc, p)         add_to_gc_list(sc->bigreals, p)
#define add_bignumber(sc, p)       add_to_gc_list(sc->bignumbers, p)
#endif

static void init_gc_caches(s7_scheme *sc)
{
  sc->strings = make_gc_list();
  sc->gensyms = make_gc_list();
  sc->unknowns = make_gc_list();
  sc->vectors = make_gc_list();
  sc->multivectors = make_gc_list();
  sc->hash_tables = make_gc_list();
  sc->input_ports = make_gc_list();
  sc->output_ports = make_gc_list();
  sc->continuations = make_gc_list();
  sc->c_objects = make_gc_list();
  sc->lambdas = make_gc_list();
  sc->optlists = make_gc_list();
  sc->weak_refs = make_gc_list();
#if WITH_GMP
  sc->bigints = make_gc_list();
  sc->bigratios = make_gc_list();
  sc->bigreals = make_gc_list();
  sc->bignumbers = make_gc_list();
#endif

  /* slightly unrelated... */
  sc->setters_size = 4;
  sc->setters_loc = 0;
  sc->setters = (s7_pointer *)malloc(sc->setters_size * sizeof(s7_pointer));
}


static void add_setter(s7_scheme *sc, s7_pointer p, s7_pointer setter)
{
  /* setters GC-protected. The c_function_setter field can't be used because the built-in functions
   *   are often removed from the heap and never thereafter marked.  Only closures and macros are protected here.
   */
  s7_int i;
#if S7_DEBUGGING
  if ((!is_any_closure(setter)) && (!is_any_macro(setter)))
    fprintf(stderr, "add_setter: %s %d?\n", DISPLAY(setter), type(setter));
#endif
  for (i = 0; i < sc->setters_loc; i++)
    {
      s7_pointer x;
      x = sc->setters[i];
      if (car(x) == p)
	{
	  set_cdr(x, setter);
	  return;
 	}
    }
  if (sc->setters_loc == sc->setters_size)
    {
      sc->setters_size *= 2;
      sc->setters = (s7_pointer *)realloc(sc->setters, sc->setters_size * sizeof(s7_pointer));
    }
  sc->setters[sc->setters_loc++] = permanent_cons(sc, p, setter, T_PAIR | T_IMMUTABLE);
}

static void mark_symbol_vector(s7_pointer p, s7_int len)
{
  set_mark(p);
  if (mark_function[T_SYMBOL] != mark_noop) /* else no gensyms */
    {
      s7_int i;
      s7_pointer *e;
      e = vector_elements(p);
      for (i = 0; i < len; i++)
	if (is_gensym(e[i]))
	  set_mark(e[i]);
    }
}

static void mark_simple_vector(s7_pointer p, s7_int len)
{
  s7_int i;
  s7_pointer *e;
  set_mark(p);
  e = vector_elements(p);
  for (i = 0; i < len; i++)
    set_mark(e[i]);
}

static void just_mark_vector(s7_pointer p, s7_int len)
{
  set_mark(p);
}

static void mark_vector_1(s7_pointer p, s7_int top)
{
  s7_pointer *tp, *tend, *tend4;

  set_mark(p);

  tp = (s7_pointer *)(vector_elements(p));
  if (!tp) return;
  tend = (s7_pointer *)(tp + top);

  tend4 = (s7_pointer *)(tend - 8);
  while (tp <= tend4)
    LOOP_8(gc_mark(*tp++));
  while (tp < tend)
    gc_mark(*tp++);
}

static void mark_let(s7_pointer env)
{
  s7_pointer x;
  for (x = env; is_let(x) && (!is_marked(x)); x = outlet(x))
    {
      s7_pointer y;
      set_mark(x);
      for (y = let_slots(x); is_slot(y); y = next_slot(y))
	if (!is_marked(y)) /* slot value might be the enclosing let */
	  mark_slot(y);
    }
}

static void mark_owlet(s7_scheme *sc)
{
  /* sc->error_type and friends are slots in owlet */
  mark_slot(sc->error_type);
  slot_set_value(sc->error_data, sc->F);
  mark_slot(sc->error_data);
  /* error_data is normally a permanent list with impermanent contents, so we have to traverse it explicitly
   *   mark_owlet is not called very often
   */
  mark_slot(sc->error_code);
  /* can sc->code be garbage at EVAL when sc->cur_code is set? or freed later by hand? */
  mark_slot(sc->error_line);
  mark_slot(sc->error_file);
#if WITH_HISTORY
  mark_slot(sc->error_history);
#endif
  set_mark(sc->owlet);
  mark_let(outlet(sc->owlet));
}

static void mark_c_pointer(s7_pointer p)
{
  set_mark(p);
  gc_mark(c_pointer_type(p));
  gc_mark(c_pointer_info(p));
}

static void mark_c_proc_star(s7_pointer p)
{
  set_mark(p);
  if (!c_func_has_simple_defaults(p))
    {
      s7_pointer arg;
      for (arg = c_function_call_args(p); is_pair(arg); arg = cdr(arg))
	gc_mark(car(arg));
    }
}

static void mark_pair(s7_pointer p)
{
  do {
    set_mark(p);
    gc_mark(car(p)); /* expanding this to avoid recursion is slower */
    p = cdr(p);
  } while ((is_pair(p)) && (!is_marked(p))); /* ((typeflag(p) & (0xff | T_GC_MARK)) == T_PAIR) is much slower */
  gc_mark(p);
}
/* in snd-14 or so through 15.3, sc->temp_cell_2|3 were used for trailing args in eval, but that meant
 *   the !is_marked check below (which is intended to catch cyclic lists) caused cells to be missed;
 *   since sc->args could contain permanently marked cells, if these were passed to g_vector, for example, and
 *   make_vector_1 triggered a GC call, we needed to mark both the permanent (always marked) cell and its contents,
 *   and continue through the rest of the list.  But adding temp_cell_2|3 to sc->permanent_objects was not enough.
 *   Now I've already forgotten the rest of the story, and it was just an hour ago! -- the upshot is that temp_cell_2|3
 *   are not now used as arg list members.
 */

static void mark_counter(s7_pointer p)
{
  set_mark(p);
  gc_mark(counter_result(p));
  gc_mark(counter_list(p));
  gc_mark(counter_let(p));
}

static void mark_closure(s7_pointer p)
{
  set_mark(p);
  gc_mark(closure_args(p));
  gc_mark(closure_body(p));
  mark_let(closure_let(p));
  gc_mark(closure_setter(p));
}

static void mark_stack_1(s7_pointer p, s7_int top)
{
  s7_pointer *tp, *tend;
  set_mark(p);

  tp = (s7_pointer *)(stack_elements(p));
  if (!tp) return;
  tend = (s7_pointer *)(tp + top);

  while (tp < tend)
    {
      gc_mark(*tp++);
      gc_mark(*tp++);
      gc_mark(*tp++);
      tp++;
    }
}

static void mark_stack(s7_pointer p)
{
  /* we can have a bare stack awaiting a continuation to hold it if the new_cell for the continuation triggers the GC!  But we need a top-of-stack?? */
  mark_stack_1(p, temp_stack_top(p));
}

static void mark_continuation(s7_pointer p)
{
  uint32_t i;
  set_mark(p);
  mark_stack_1(continuation_stack(p), continuation_stack_top(p));
  for (i = 0; i < continuation_op_loc(p); i++)
    gc_mark(continuation_op_stack(p)[i]);
}

static void mark_vector(s7_pointer p)
{
  if (is_typed_vector(p))
    c_function_marker(typed_vector_typer(p))(p, vector_length(p));
  else mark_vector_1(p, vector_length(p));
}

static void mark_vector_possibly_shared(s7_pointer p)
{
  /* If a subvector (an inner dimension) of a vector is the only remaining reference
   *    to the main vector, we want to make sure the main vector is not GC'd until
   *    the subvector is also GC-able.  The subvector field either points to the
   *    parent vector, or it is sc->F, so we need to check for a vector parent if
   *    the current is multidimensional (this will include 1-dim slices).  We need
   *    to keep the parent case separate (i.e. sc->F means the current is the original)
   *    so that we only free once (or remove_from_heap once).
   *
   * If we have a subvector of a subvector, and the middle and original are not otherwise
   *   in use, we mark the middle one, but (since it itself is not in use anywhere else)
   *   we don't mark the original!  So we need to follow the share-vector chain marking every one.
   *
   * To remove a cell from the heap, we need its current heap location so that we can replace it.
   *   The heap is allocated as needed in monolithic blocks of (say) 1/2M s7_cells. When a cell
   *   is replaced, the new cell (at heap[x] say) is no longer from the original block. Since the
   *   GC clears all type bits when it frees a cell, we can't use a type bit to distinguish the
   *   replacements from the originals, but we need that info because in the base case, we use
   *   the distance of the cell from the base cell to get "x", its location.  In the replacement
   *   case, we add the location at the end of the s7_cell (s7_big_cell).  We track the current
   *   heap blocks via the sc->heap_blocks list.  To get the location of "p" above, we run through
   *   that list looking for a block it fits in.  If none is found, we assume it is an s7_big_cell
   *   and use the saved location.
   */
  if ((is_subvector(p)) &&
      (is_any_vector(subvector_vector(p))))
    mark_vector_possibly_shared(subvector_vector(p));

  /* mark_vector_1 does not check the marked bit, so if subvector below is in a cycle involving
   *   the calling vector, we get infinite recursion unless we check the mark bit here.
   */
  if (!is_marked(p))
    mark_vector_1(p, vector_length(p));
}

static void mark_int_or_float_vector(s7_pointer p)
{
  set_mark(p);
}

static void mark_int_or_float_vector_possibly_shared(s7_pointer p)
{
  if ((vector_has_dimensional_info(p)) &&
      (is_any_vector(subvector_vector(p))))
    mark_int_or_float_vector_possibly_shared(subvector_vector(p));

  set_mark(p);
}

static void mark_c_object(s7_pointer p)
{
  set_mark(p);
  (*(c_object_mark(p)))(c_object_value(p));
}

static void mark_catch(s7_pointer p)
{
  set_mark(p);
  gc_mark(catch_tag(p));
  gc_mark(catch_handler(p));
}

static void mark_dynamic_wind(s7_pointer p)
{
  set_mark(p);
  gc_mark(dynamic_wind_in(p));
  gc_mark(dynamic_wind_out(p));
  gc_mark(dynamic_wind_body(p));
}

/* if is_typed_hash_table then if c_function_marker(key|value_typer) is just_mark_vector, we can ignore that field,
 *    if it's mark_simple_vector, we just set_mark (key|value), else we gc_mark
 */
static void mark_hash_table(s7_pointer p)
{
  set_mark(p);
  gc_mark(hash_table_procedures(p));
  if (hash_table_entries(p) > 0)
    {
      s7_int len;
      hash_entry_t **entries, **last;

      entries = hash_table_elements(p);
      len = hash_table_mask(p) + 1;
      last = (hash_entry_t **)(entries + len);

      if (is_weak_hash_table(p))
	{
	  while (entries < last)
	    {
	      hash_entry_t *xp;
	      for (xp = *entries++; xp; xp = hash_entry_next(xp))
		gc_mark(hash_entry_value(xp));
	      for (xp = *entries++; xp; xp = hash_entry_next(xp))
		gc_mark(hash_entry_value(xp));
	    }
	}
      else
	{
	  while (entries < last) /* counting entries here was slightly faster */
	    {
	      hash_entry_t *xp;
	      for (xp = *entries++; xp; xp = hash_entry_next(xp))
		{
		  gc_mark(hash_entry_key(xp));
		  gc_mark(hash_entry_value(xp));
		}
	      for (xp = *entries++; xp; xp = hash_entry_next(xp))
		{
		  gc_mark(hash_entry_key(xp));
		  gc_mark(hash_entry_value(xp));
		}
	    }
	}
    }
}

static void mark_iterator(s7_pointer p)
{
  set_mark(p);
  gc_mark(iterator_sequence(p));
  if (is_mark_seq(p))
    gc_mark(iterator_current(p));
}

static void mark_input_port(s7_pointer p)
{
  set_mark(p);
  set_mark(port_original_input_string(p));
}

#define clear_type(p) typeflag(p) = T_FREE

static void init_mark_functions(void)
{
  mark_function[T_FREE]                = mark_noop;
  mark_function[T_UNDEFINED]           = just_mark;
  mark_function[T_EOF_OBJECT]          = mark_noop;
  mark_function[T_UNSPECIFIED]         = mark_noop;
  mark_function[T_NIL]                 = mark_noop;
  mark_function[T_BOOLEAN]             = mark_noop;
  mark_function[T_SYNTAX]              = mark_noop;
  mark_function[T_CHARACTER]           = mark_noop;
  mark_function[T_SYMBOL]              = mark_noop; /* this changes to just_mark when gensyms are in the heap */
  mark_function[T_STRING]              = just_mark;
  mark_function[T_BYTE_VECTOR]         = just_mark;
  mark_function[T_INTEGER]             = just_mark;
  mark_function[T_RATIO]               = just_mark;
  mark_function[T_REAL]                = just_mark;
  mark_function[T_COMPLEX]             = just_mark;
  mark_function[T_BIG_INTEGER]         = just_mark;
  mark_function[T_BIG_RATIO]           = just_mark;
  mark_function[T_BIG_REAL]            = just_mark;
  mark_function[T_BIG_COMPLEX]         = just_mark;
  mark_function[T_RANDOM_STATE]        = just_mark;
  mark_function[T_GOTO]                = just_mark;
  mark_function[T_OUTPUT_PORT]         = just_mark;
  mark_function[T_BAFFLE]              = just_mark;
  mark_function[T_C_MACRO]             = just_mark;
  mark_function[T_C_POINTER]           = mark_c_pointer;
  mark_function[T_C_FUNCTION]          = just_mark;
  mark_function[T_C_FUNCTION_STAR]     = just_mark;  /* changes to mark_c_proc_star if defaults involve an expression */
  mark_function[T_C_ANY_ARGS_FUNCTION] = just_mark;
  mark_function[T_C_OPT_ARGS_FUNCTION] = just_mark;
  mark_function[T_C_RST_ARGS_FUNCTION] = just_mark;
  mark_function[T_PAIR]                = mark_pair;
  mark_function[T_CLOSURE]             = mark_closure;
  mark_function[T_CLOSURE_STAR]        = mark_closure;
  mark_function[T_CONTINUATION]        = mark_continuation;
  mark_function[T_INPUT_PORT]          = mark_input_port;
  mark_function[T_VECTOR]              = mark_vector; /* this changes if subvector created (similarly below) */
  mark_function[T_INT_VECTOR]          = mark_int_or_float_vector;
  mark_function[T_FLOAT_VECTOR]        = mark_int_or_float_vector;
  mark_function[T_MACRO]               = mark_closure;
  mark_function[T_BACRO]               = mark_closure;
  mark_function[T_MACRO_STAR]          = mark_closure;
  mark_function[T_BACRO_STAR]          = mark_closure;
  mark_function[T_C_OBJECT]            = mark_c_object;
  mark_function[T_CATCH]               = mark_catch;
  mark_function[T_DYNAMIC_WIND]        = mark_dynamic_wind;
  mark_function[T_HASH_TABLE]          = mark_hash_table;
  mark_function[T_ITERATOR]            = mark_iterator;
  mark_function[T_LET]                 = mark_let;
  mark_function[T_STACK]               = mark_stack;
  mark_function[T_COUNTER]             = mark_counter;
  mark_function[T_SLOT]                = mark_slot;
}

static void mark_op_stack(s7_scheme *sc)
{
  s7_pointer *p, *tp;
  tp = sc->op_stack_now;
  p = sc->op_stack;
  while (p < tp)
    gc_mark(*p++);
}

static void mark_rootlet(s7_scheme *sc)
{
  s7_pointer ge;
  s7_pointer *tmp, *top;

  ge = sc->rootlet;
  tmp = rootlet_elements(ge);
  top = (s7_pointer *)(tmp + sc->rootlet_entries);

  set_mark(ge);
  while (tmp < top)
    gc_mark(slot_value(*tmp++));
  /* slot_setter is handled below with an explicit list -- more code than its worth probably */
  /* we're not marking slot_symbol above which makes me worry that a top-level gensym won't protected
   *   (apply define (gensym) '(32)), then try to get the GC to clobber {gensym}-0,
   *   but I can't get it to break, so they must be protected somehow; apparently they are
   *   removed from the heap!  At least:
   *   (define-macro (defit) (let ((n (gensym))) `(define (,n) (format #t "fun")))) (defit)
   *   removes the function from the heap (protecting the gensym).
   */
}

static void mark_permanent_objects(s7_scheme *sc)
{
  gc_obj *g;
  for (g = sc->permanent_objects; g; g = (gc_obj *)(g->nxt))
    gc_mark(g->p);
}

static void unmark_permanent_objects(s7_scheme *sc)
{
  gc_obj *g;
  for (g = sc->permanent_objects; g; g = (gc_obj *)(g->nxt))
    clear_mark(g->p);
}


#if (!MS_WINDOWS)
  #include <time.h>
  #include <sys/time.h>
#endif

#if S7_DEBUGGING
static char *describe_type_bits(s7_scheme *sc, s7_pointer obj);
static bool has_odd_bits(s7_pointer obj);
#endif
void s7_show_let(s7_scheme *sc);

static int64_t gc(s7_scheme *sc)
{
  s7_cell **old_free_heap_top;
#if (!MS_WINDOWS)
  struct timeval start_time;
  struct timezone z0;
#endif

  /* mark all live objects (the symbol table is in permanent memory, not the heap) */

#if (!MS_WINDOWS)
  if (show_gc_stats(sc))
    gettimeofday(&start_time, &z0);
  /* this is apparently deprecated in favor of clock_gettime -- what compile-time switch to use here?
   *   _POSIX_TIMERS, or perhaps use CLOCK_REALTIME, but clock_gettime requires -lrt -- no thanks.
   */
#endif
  mark_rootlet(sc);
  gc_mark(sc->args);
  mark_let(sc->envir);
  /* slot_set_value(sc->error_data, sc->F); */
  /* the other choice here is to explicitly mark slot_value(sc->error_data) as we do eval_history1/2 below.
   *    in both cases, the values are permanent lists that do not mark impermanent contents.
   *    this will need circular list checks, and can't depend on marked to exit early
   */
  /* mark_let(sc->owlet); */

#if WITH_HISTORY
  {
    s7_pointer p1, p2;
    for (p1 = sc->eval_history1, p2 = sc->eval_history2; ; p2 = cdr(p2))
      {
	gc_mark(car(p1));
	gc_mark(car(p2));
	p1 = cdr(p1);
	if (p1 == sc->eval_history1) break; /* these are circular lists */
      }
  }
#endif
  mark_owlet(sc);

  gc_mark(sc->code);
  mark_current_code(sc); /* probably redundant if with_history */
  mark_stack_1(sc->stack, s7_stack_top(sc));
  gc_mark(sc->u);
  gc_mark(sc->v);
  gc_mark(sc->w);
  gc_mark(sc->x);
  gc_mark(sc->y);
  gc_mark(sc->z);
  gc_mark(sc->value);

  gc_mark(sc->temp1);
  gc_mark(sc->temp2);
  gc_mark(sc->temp3);
  gc_mark(sc->temp4);
  gc_mark(sc->temp6);
  gc_mark(sc->temp7);
  gc_mark(sc->temp8);
  gc_mark(sc->temp9);
  gc_mark(sc->temp10);
  gc_mark(sc->temp11);
  {
    int32_t i;
    for (i = 0; i < T_TEMPS_SIZE; i++) {gc_mark(sc->t_temps[i]);}
  }
  set_mark(sc->input_port);
  gc_mark(sc->input_port_stack);
  set_mark(sc->output_port);
  set_mark(sc->error_port);
  gc_mark(sc->stacktrace_defaults);
  gc_mark(sc->autoload_table);
  gc_mark(sc->default_rng);

  /* permanent lists that might escape and therefore need GC protection */
  mark_pair(sc->temp_cell_1);
  mark_pair(sc->temp_cell_2);
  gc_mark(car(sc->t1_1));
  gc_mark(car(sc->t2_1)); gc_mark(car(sc->t2_2));
  gc_mark(car(sc->t3_1)); gc_mark(car(sc->t3_2)); gc_mark(car(sc->t3_3));
  gc_mark(car(sc->a4_1)); gc_mark(car(sc->a4_2)); gc_mark(car(sc->a4_3)); gc_mark(car(sc->a4_4));
  gc_mark(car(sc->plist_1));
  gc_mark(car(sc->clist_1));
  gc_mark(car(sc->plist_2)); gc_mark(cadr(sc->plist_2));
  gc_mark(car(sc->qlist_2)); gc_mark(cadr(sc->qlist_2));
  gc_mark(sc->u1_1);

  {
    s7_pointer p;
    for (p = sc->wrong_type_arg_info; is_pair(p); p = cdr(p)) gc_mark(car(p));
    for (p = sc->simple_wrong_type_arg_info; is_pair(p); p = cdr(p)) gc_mark(car(p));
    for (p = sc->out_of_range_info; is_pair(p); p = cdr(p)) gc_mark(car(p));
    for (p = sc->simple_out_of_range_info; is_pair(p); p = cdr(p)) gc_mark(car(p));
    gc_mark(car(sc->elist_1));
    gc_mark(car(sc->elist_2));
    gc_mark(cadr(sc->elist_2));
    for (p = sc->plist_3; is_pair(p); p = cdr(p)) gc_mark(car(p));
    for (p = sc->elist_3; is_pair(p); p = cdr(p)) gc_mark(car(p));
    for (p = sc->elist_4; is_pair(p); p = cdr(p)) gc_mark(car(p));
    for (p = sc->elist_5; is_pair(p); p = cdr(p)) gc_mark(car(p));
  }

  {
    s7_int i;
    s7_pointer p;
    /* perhaps: if (sc->current_safe_list > 0) ... but this loop is down in the noise */
    for (i = 1; i < NUM_SAFE_LISTS; i++)
      if ((is_pair(sc->safe_lists[i])) &&
	  (list_is_in_use(sc->safe_lists[i])))
	for (p = sc->safe_lists[i]; is_pair(p); p = cdr(p))
	  gc_mark(car(p));

    for (i = 0; i < sc->setters_loc; i++)
      gc_mark(cdr(sc->setters[i]));

    for (i = 0; i < sc->num_fdats; i++)
      if (sc->fdats[i])
	gc_mark(sc->fdats[i]->curly_arg);
  }
  mark_vector(sc->protected_objects);
  mark_vector(sc->protected_setters);
  set_mark(sc->protected_setter_symbols);

  /* now protect recent allocations using the free_heap cells above the current free_heap_top (if any).
   *
   * cells above sc->free_heap_top might be malloc'd garbage (after heap reallocation), so we keep track of
   *   where the last actually freed cells were after the previous GC call.  We're trying to
   *   GC protect the previous GC_TEMPS_SIZE allocated pointers so that the caller doesn't have
   *   to gc-protect every temporary cell.
   *
   * There's one remaining possible problem.  s7_remove_from_heap frees cells outside
   *   the GC and might push free_heap_top beyond its previous_free_heap_top, then
   *   an immediate explicit gc call might not see those temp cells.
   */
  {
    s7_pointer *tmps, *tmps_top;

    tmps = sc->free_heap_top;
    tmps_top = tmps + GC_TEMPS_SIZE;
    if (tmps_top > sc->previous_free_heap_top)
      tmps_top = sc->previous_free_heap_top;

    while (tmps < tmps_top)
      gc_mark(*tmps++);
  }
  mark_op_stack(sc);
  mark_permanent_objects(sc);

  /* free up all unmarked objects */
  old_free_heap_top = sc->free_heap_top;

  {
    s7_pointer *fp, *tp, *heap_top;
    fp = sc->free_heap_top;

    tp = sc->heap;
    heap_top = (s7_pointer *)(sc->heap + sc->heap_size);

#if S7_DEBUGGING
#define gc_call(P, Tp)							\
    p = (*tp++);							\
    if (is_marked(T_Any(p)))						\
      clear_mark(p);							\
    else								\
      {									\
        if (!is_free_and_clear(p))					\
          {								\
	    p->debugger_bits = 0;					\
	    if (has_odd_bits(p))					\
	      {char *s; fprintf(stderr, "odd bits: %s\n", s = describe_type_bits(sc, p)); free(s);} \
            clear_type(p);						\
            (*fp++) = p;						\
          }}
#else
  #define gc_call(P, Tp) p = (*tp++); if (is_marked(p)) clear_mark(p); else {if (!is_free_and_clear(p)) {clear_type(p); (*fp++) = p;}}
#endif

    while (tp < heap_top)          /* != here or ^ makes no difference */
      {
	s7_pointer p;
	LOOP_8(gc_call(p, tp));
	LOOP_8(gc_call(p, tp));
	LOOP_8(gc_call(p, tp));
	LOOP_8(gc_call(p, tp));
      }

    sc->free_heap_top = fp;
    sweep(sc);
  }

  unmark_permanent_objects(sc);
  sc->gc_freed = (int64_t)(sc->free_heap_top - old_free_heap_top);

  if (show_gc_stats(sc))
    {
#if (!MS_WINDOWS)
      struct timeval t0;
      double secs;
      gettimeofday(&t0, &z0);
      secs = (t0.tv_sec - start_time.tv_sec) +  0.000001 * (t0.tv_usec - start_time.tv_usec);
      s7_warn(sc, 256, "gc freed %" print_s7_int "/%" print_s7_int " (free: %" print_pointer "), time: %f\n", sc->gc_freed, sc->heap_size, (intptr_t)(sc->free_heap_top - sc->free_heap), secs);
#else
      s7_warn(sc, 128, "gc freed %" print_s7_int "/%" print_s7_int "\n", sc->gc_freed, sc->heap_size);
#endif
    }
  sc->previous_free_heap_top = sc->free_heap_top;

  return(sc->gc_freed); /* needed by cell allocator to decide when to increase heap size */
}

void s7_gc_stats(s7_scheme *sc, bool on) {sc->gc_stats = (on) ? GC_STATS : 0;}


static void resize_heap_to(s7_scheme *sc, int64_t size)
{
  /* alloc more heap */
  int64_t old_size, old_free, k;
  s7_cell *cells;
  s7_pointer p;
  heap_block_t *hp;

  old_size = sc->heap_size;
  old_free = sc->free_heap_top - sc->free_heap;

  if (size == 0)
    {
      if (sc->heap_size < 512000)
	sc->heap_size *= 2;
      else sc->heap_size += 512000;
    }
  else
    {
      if (size > sc->heap_size)
	while (sc->heap_size < size) sc->heap_size *= 2;
    }

  sc->heap = (s7_cell **)realloc(sc->heap, sc->heap_size * sizeof(s7_cell *));
  if (!(sc->heap))
    s7_warn(sc, 256, "heap reallocation failed! tried to get %" print_s7_int " bytes\n", (int64_t)(sc->heap_size * sizeof(s7_cell *)));

  sc->free_heap = (s7_cell **)realloc(sc->free_heap, sc->heap_size * sizeof(s7_cell *));
  if (!(sc->free_heap))
    s7_warn(sc, 256, "free heap reallocation failed! tried to get %" print_s7_int " bytes\n", (int64_t)(sc->heap_size * sizeof(s7_cell *)));

  sc->free_heap_trigger = (s7_cell **)(sc->free_heap + GC_TRIGGER_SIZE);
  sc->free_heap_top = sc->free_heap + old_free; /* incremented below, added old_free 21-Aug-12?!? */

  cells = (s7_cell *)calloc(sc->heap_size - old_size, sizeof(s7_cell)); /* optimization suggested by K Matheussen, malloc makes no difference */
  for (p = cells, k = old_size; k < sc->heap_size;)
    LOOP_4(sc->heap[k++] = p; (*sc->free_heap_top++) = p++);

  hp = (heap_block_t *)malloc(sizeof(heap_block_t));
  hp->start = (intptr_t)cells;
  hp->end = (intptr_t)cells + ((sc->heap_size - old_size) * sizeof(s7_cell));
  hp->offset = old_size;
  hp->next = sc->heap_blocks;
  sc->heap_blocks = hp;

  sc->previous_free_heap_top = sc->free_heap_top;

  if (show_heap_stats(sc))
    s7_warn(sc, 256, "heap grows to %" print_s7_int " (old free/size: %" print_s7_int "/%" print_s7_int ")\n", sc->heap_size, old_free, old_size);

  if (sc->heap_size >= sc->max_heap_size)
    {
#if S7_DEBUGGING
      fprintf(stderr, "heap %" print_s7_int ", %s\n", sc->heap_size, DISPLAY(current_code(sc)));
      s7_show_let(sc);
      abort();
#endif
      s7_error(sc, s7_make_symbol(sc, "heap-too-big"), set_elist_1(sc, wrap_string(sc, "heap has grown past (*s7* 'max-heap-size)", 41)));
    }
}

#define resize_heap(Sc) resize_heap_to(Sc, 0)

static void try_to_call_gc(s7_scheme *sc)
{
  /* called only from new_cell */
  if (sc->gc_off)
    {
      /* we can't just return here!  Someone needs a new cell, and once the heap free list is exhausted, segfault */
      resize_heap(sc);
    }
  else
    {
#if (!S7_DEBUGGING)
      int64_t freed_heap;
      freed_heap = gc(sc);
      if ((freed_heap < sc->heap_size / 2) &&
	  (freed_heap < 1000000)) /* if huge heap */
	resize_heap(sc);
#else
      gc(sc);
      if ((int64_t)(sc->free_heap_top - sc->free_heap) < sc->heap_size / 2)
	resize_heap(sc);
#endif
    }
}
  /* originally I tried to mark each temporary value until I was done with it, but
   *   that way madness lies... By delaying GC of _every_ %$^#%@ pointer, I can dispense
   *   with hundreds of individual protections.  So the free_heap's last GC_TEMPS_SIZE
   *   allocated pointers are protected during the mark sweep.
   */

static s7_pointer g_gc(s7_scheme *sc, s7_pointer args)
{
  #define H_gc "(gc (on #t)) runs the garbage collector.  If 'on' is supplied, it turns the GC on or off. \
Evaluation produces a surprising amount of garbage, so don't leave the GC off for very long!"
  #define Q_gc s7_make_signature(sc, 2, sc->T, sc->is_boolean_symbol)

  /* g_gc can't be called in a situation where these lists matter (I think...) */
  set_elist_1(sc, sc->nil);
  set_plist_1(sc, sc->nil);
  set_elist_2(sc, sc->nil, sc->nil);
  set_plist_2(sc, sc->nil, sc->nil);
  set_clist_1(sc, sc->nil);
  set_qlist_2(sc, sc->nil, sc->nil);
  set_elist_3(sc, sc->nil, sc->nil, sc->nil);
  set_plist_3(sc, sc->nil, sc->nil, sc->nil);
  set_elist_4(sc, sc->nil, sc->nil, sc->nil, sc->nil);
  set_elist_5(sc, sc->nil, sc->nil, sc->nil, sc->nil, sc->nil);

  if (is_not_null(args))
    {
      if (!s7_is_boolean(car(args)))
	return(method_or_bust_one_arg(sc, car(args), sc->gc_symbol, args, T_BOOLEAN));
      sc->gc_off = (car(args) == sc->F);
      if (sc->gc_off)
	return(sc->F);
    }
  gc(sc);
  return(sc->unspecified);
}

s7_pointer s7_gc_on(s7_scheme *sc, bool on)
{
  sc->gc_off = !on;
  return(s7_make_boolean(sc, on));
}


#define ALLOC_POINTER_SIZE 256
static s7_cell *alloc_pointer(s7_scheme *sc)
{
  if (sc->alloc_pointer_k == ALLOC_POINTER_SIZE)     /* if either no current block or the block is used up, make a new block */
    {
      sc->permanent_cells += ALLOC_POINTER_SIZE;
      sc->alloc_pointer_cells = (s7_cell *)calloc(ALLOC_POINTER_SIZE, sizeof(s7_cell));
      sc->alloc_pointer_k = 0;
    }
  return(&(sc->alloc_pointer_cells[sc->alloc_pointer_k++]));
}

#define ALLOC_BIG_POINTER_SIZE 256
static s7_big_cell *alloc_big_pointer(s7_scheme *sc, int64_t loc)
{
  s7_big_pointer p;
  if (sc->alloc_big_pointer_k == ALLOC_BIG_POINTER_SIZE)
    {
      sc->permanent_cells += ALLOC_BIG_POINTER_SIZE;
      sc->alloc_big_pointer_cells = (s7_big_cell *)calloc(ALLOC_BIG_POINTER_SIZE, sizeof(s7_big_cell));
      sc->alloc_big_pointer_k = 0;
    }
  p = (&(sc->alloc_big_pointer_cells[sc->alloc_big_pointer_k++]));
  p->big_hloc = loc;
  return(p);
}


static void add_permanent_object(s7_scheme *sc, s7_pointer obj)
{
  gc_obj *g;
  g = (gc_obj *)malloc(sizeof(gc_obj));
  g->p = obj;
  g->nxt = sc->permanent_objects;
  sc->permanent_objects = g;
}

#if S7_DEBUGGING
static const char *type_name_from_type(int32_t typ, int32_t article);

#define free_cell(Sc, P) free_cell_1(Sc, P, __LINE__)
static void free_cell_1(s7_scheme *sc, s7_pointer p, int32_t line)
#else
static void free_cell(s7_scheme *sc, s7_pointer p)
#endif
{
#if S7_DEBUGGING
  /* anything that needs gc_list attention should not be freed here */
  uint8_t typ;
  typ = unchecked_type(p);
  if ((t_freeze_p[typ]) || ((typ == T_SYMBOL) && (is_gensym(p))))
    {
      fprintf(stderr, "free_cell of %s?\n", type_name_from_type(typ, 0));
      /* abort(); */
    }
  p->debugger_bits = 0;
  p->explicit_free_line = line;
#endif
  clear_type(p);
  (*(sc->free_heap_top++)) = p;
}

static void free_vlist(s7_scheme *sc, s7_pointer lst)
{
  if (is_pair(lst))
    {
      s7_pointer p, np;
      for (p = lst, np = cdr(lst); is_pair(p); p = np, np = unchecked_cdr(np))
	free_cell(sc, p);
    }
}

static int64_t heap_location(s7_scheme *sc, s7_pointer p)
{
  heap_block_t *hp;
  for (hp = sc->heap_blocks; hp; hp = hp->next)
    {
      if (((intptr_t)p >= hp->start) && ((intptr_t)p < hp->end))
	return(hp->offset + (((intptr_t)p - hp->start) / sizeof(s7_cell)));
    }
  return(((s7_big_pointer)p)->big_hloc);
}

#if S7_DEBUGGING
static void check_heap_location(s7_scheme *sc, s7_pointer x, int64_t loc, const char *func, int line)
{
  if ((in_heap(x)) && ((loc < 0) || (loc > sc->heap_size) || (sc->heap[loc] != x)))
    {
      s7_int i;
      char *s;
      heap_block_t *hp;
      fprintf(stderr, "%s[%d]: sc->heap[%ld] (%p) is not %p\n", func, line, loc, ((loc >= 0) && (loc < sc->heap_size)) ? sc->heap[loc] : NULL, x);
      for (i = 0; i < sc->heap_size; i++)
	if (sc->heap[i] == x)
	  break;
      if (i < sc->heap_size)
	fprintf(stderr, "  correct location: %ld\n", i);
      else fprintf(stderr, "  %p is not in the heap\n", x);
      fprintf(stderr, "  bits: %s\n", s = describe_type_bits(sc, x));
      free(s);
      fprintf(stderr, "blocks (x is %ld, big_hloc: %ld):\n", (intptr_t)x, ((s7_big_pointer)x)->big_hloc);
      for (hp = sc->heap_blocks; hp; hp = hp->next)
	{
	  fprintf(stderr, "  %ld: %ld to %ld\n", hp->offset, hp->start, hp->end);
	  if (((intptr_t)x >= hp->start) && ((intptr_t)x < hp->end))
	    fprintf(stderr, "   (found it here: %ld\n", hp->offset + (((intptr_t)x - hp->start) / sizeof(s7_cell)));
	}
      abort();
    }
}
#endif

static inline s7_pointer petrify(s7_scheme *sc, s7_pointer x)
{
  s7_pointer p;
  int64_t loc;
  loc = heap_location(sc, x);
#if S7_DEBUGGING
  check_heap_location(sc, x, loc, __func__, __LINE__);
#endif
  p = (s7_pointer)alloc_big_pointer(sc, loc);
  sc->heap[loc] = p;
  free_cell(sc, p);
  unheap(sc, x);
  return(x);
}

static inline void s7_remove_from_heap(s7_scheme *sc, s7_pointer x)
{
  /* global functions are very rarely redefined, so we can remove the function body from
   *   the heap when it is defined.  If redefined, we currently lose the memory held by the
   *   old definition.  (It is not trivial to recover this memory because it is allocated
   *   in blocks, not by the pointer, I think, but s7_define is the point to try).
   *
   * There is at least one problem with this: if, for example, a function has
   *    a quoted (constant) list, then uses list-set! to change an element of it,
   *    then a GC happens, and the new element is GC'd because no one in the heap
   *    points to it, then we call the function again, and it tries to access
   *    that element.
   *
   *    (define (bad-idea)
   *      (let ((lst '(1 2 3)))
   *        (let ((result (list-ref lst 1)))
   *          (list-set! lst 1 (* 2.0 16.6))
   *          (gc)
   *          result)))
   *
   *     put that in a file, load it (to force removal), then call bad-idea a few times.
   * so... if (*s7* 'safety) is not 0, remove-from-heap is disabled.
   */
  if (not_in_heap(x)) return;
  if (is_pair(x))
    {
      s7_pointer p;
      p = x;
      do {
	petrify(sc, p);
	s7_remove_from_heap(sc, car(p));
	p = cdr(p);
      } while (is_pair(p) && (in_heap(p)));
      if (in_heap(p)) petrify(sc, p);
      return;
    }

  switch (type(x))
    {
    case T_HASH_TABLE:
    case T_LET:
    case T_VECTOR:
      /* not int|float_vector or string because none of their elements are GC-able (so unheap below is ok)
       *   but hash-table and let seem like they need protection? And let does happen via define-class.
       */
      add_permanent_object(sc, x);
      return;

    case T_SYMBOL:
      if (is_gensym(x))
	{
	  s7_int i;
	  gc_list *gp;
	  int64_t loc;
	  loc = heap_location(sc, x);
#if S7_DEBUGGING
	  check_heap_location(sc, x, loc, __func__, __LINE__);
#endif
	  sc->heap[loc] = (s7_pointer)alloc_big_pointer(sc, loc);
	  free_cell(sc, sc->heap[loc]);
	  unheap(sc, x);

	  gp = sc->gensyms;
	  for (i = 0; i < gp->loc; i++) /* sc->gensyms reaches size 512 during s7test, but this search is called 3 times and costs nothing */
	    if (gp->list[i] == x)
	      {
		s7_int j;
		for (j = i + 1; i < gp->loc - 1; i++, j++)
		  gp->list[i] = gp->list[j];
		gp->list[i] = NULL;
		gp->loc--;
		if (gp->loc == 0) mark_function[T_SYMBOL] = mark_noop;
		break;
	      }
	}
      return;

    case T_CLOSURE: case T_CLOSURE_STAR:
    case T_MACRO:   case T_MACRO_STAR:
    case T_BACRO:   case T_BACRO_STAR:
      return;

    default:
      break;
    }

  petrify(sc, x);
}


/* -------------------------------- stacks -------------------------------- */

#define OP_STACK_INITIAL_SIZE 8

#define stop_at_error true
#if S7_DEBUGGING

static void push_op_stack(s7_scheme *sc, s7_pointer op)
{
  (*sc->op_stack_now++) = T_Pos(op);
  if (sc->op_stack_now > (sc->op_stack + sc->op_stack_size))
    {
      fprintf(stderr, "%sop_stack overflow%s\n", BOLD_TEXT, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
}

static s7_pointer pop_op_stack(s7_scheme *sc)
{
  s7_pointer op;
  op = (*(--(sc->op_stack_now)));
  if (sc->op_stack_now < sc->op_stack)
    {
      fprintf(stderr, "%sop_stack underflow%s\n", BOLD_TEXT, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(T_Pos(op));
}
#else
#define push_op_stack(Sc, Op) (*Sc->op_stack_now++) = Op
#define pop_op_stack(Sc)      (*(--(Sc->op_stack_now)))
#endif

static void initialize_op_stack(s7_scheme *sc)
{
  int32_t i;
  sc->op_stack = (s7_pointer *)malloc(OP_STACK_INITIAL_SIZE * sizeof(s7_pointer));
  sc->op_stack_size = OP_STACK_INITIAL_SIZE;
  sc->op_stack_now = sc->op_stack;
  sc->op_stack_end = (s7_pointer *)(sc->op_stack + sc->op_stack_size);
  for (i = 0; i < OP_STACK_INITIAL_SIZE; i++)
    sc->op_stack[i] = sc->nil;
}

static void resize_op_stack(s7_scheme *sc)
{
  int32_t i, loc, new_size;
  loc = (int32_t)(sc->op_stack_now - sc->op_stack);
  new_size = sc->op_stack_size * 2;
  sc->op_stack = (s7_pointer *)realloc((void *)(sc->op_stack), new_size * sizeof(s7_pointer));
  for (i = sc->op_stack_size; i < new_size; i++)
    sc->op_stack[i] = sc->nil;
  sc->op_stack_size = (uint32_t)new_size;
  sc->op_stack_now = (s7_pointer *)(sc->op_stack + loc);
  sc->op_stack_end = (s7_pointer *)(sc->op_stack + sc->op_stack_size);
}

#define stack_code(Stack, Loc)  stack_element(Stack, Loc - 3)
#define stack_let(Stack, Loc)   stack_element(Stack, Loc - 2)
#define stack_args(Stack, Loc)  stack_element(Stack, Loc - 1)
#define stack_op(Stack, Loc)    ((opcode_t)(stack_element(Stack, Loc)))

#if S7_DEBUGGING
static void pop_stack(s7_scheme *sc)
{
  sc->stack_end -= 4;
  if (sc->stack_end < sc->stack_start)
    {
      fprintf(stderr, "%sstack underflow%s\n", BOLD_TEXT, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  /* here and in push_stack, both code and args might be non-free only because they've been retyped
   *   inline (as in named let) -- they actually don't make sense in these cases, but are ignored,
   *   and are carried around as GC protection in other cases.
   */
  sc->code =  T_Pos(sc->stack_end[0]);
  sc->envir = T_Lid(sc->stack_end[1]);
  sc->args =  T_Pos(sc->stack_end[2]);
  sc->cur_op = (opcode_t)(sc->stack_end[3]);

  if (sc->cur_op > OP_MAX_DEFINED)
    {
      fprintf(stderr, "%spop_stack[%d] invalid opcode: %" print_pointer " %s\n", BOLD_TEXT, __LINE__, sc->cur_op, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
}

static void pop_stack_no_op(s7_scheme *sc)
{
  sc->stack_end -= 4;
  if (sc->stack_end < sc->stack_start)
    {
      fprintf(stderr, "%sstack underflow%s\n", BOLD_TEXT, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  sc->code =  T_Pos(sc->stack_end[0]);
  sc->envir = T_Lid(sc->stack_end[1]);
  sc->args =  T_Pos(sc->stack_end[2]);
}

static void push_stack(s7_scheme *sc, opcode_t op, s7_pointer args, s7_pointer code)
{
  if (sc->stack_end >= sc->stack_start + sc->stack_size)
    {
      fprintf(stderr, "%sstack overflow%s\n", BOLD_TEXT, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  if (op > OP_MAX_DEFINED)
    {
      fprintf(stderr, "%spush_stack[%d] invalid opcode: %" print_pointer " %s\n", BOLD_TEXT, __LINE__, sc->cur_op, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  if (code) sc->stack_end[0] = T_Pos(code);
  sc->stack_end[1] = T_Lid(sc->envir);
  if (args) sc->stack_end[2] = T_Pos(args);
  sc->stack_end[3] = (s7_pointer)op;
  sc->stack_end += 4;
}

#define push_stack_no_code(Sc, Op, Args)        push_stack(Sc, Op, Args, Sc->gc_nil)
#define push_stack_no_let_no_code(Sc, Op, Args) push_stack(Sc, Op, Args, Sc->gc_nil)
#define push_stack_no_args(Sc, Op, Code)        push_stack(Sc, Op, Sc->gc_nil, Code)
#define push_stack_no_let(Sc, Op, Args, Code)   push_stack(Sc, Op, Args, Code)
#define push_stack_op(Sc, Op)                   push_stack(Sc, Op, Sc->gc_nil, Sc->gc_nil)
#define push_stack_op_let(Sc, Op)               push_stack(Sc, Op, Sc->gc_nil, Sc->gc_nil)
/* in the non-debugging case, the sc->gc_nil's here are not set, so we can (later) pop free cells */

#else

#define pop_stack(Sc) do {Sc->stack_end -= 4; memcpy((void *)Sc, (void *)(Sc->stack_end), 4 * sizeof(s7_pointer));} while (0)
#define pop_stack_no_op(Sc) {Sc->stack_end -= 4; memcpy((void *)Sc, (void *)(Sc->stack_end), 3 * sizeof(s7_pointer));} while (0)

#define push_stack(Sc, Op, Args, Code) \
  do { \
      Sc->stack_end[0] = Code; \
      Sc->stack_end[1] = Sc->envir; \
      Sc->stack_end[2] = Args; \
      Sc->stack_end[3] = (s7_pointer)Op; \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_no_code(Sc, Op, Args) \
  do { \
      Sc->stack_end[1] = Sc->envir; \
      Sc->stack_end[2] = Args; \
      Sc->stack_end[3] = (s7_pointer)Op; \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_no_let_no_code(Sc, Op, Args) \
  do { \
      Sc->stack_end[2] = Args; \
      Sc->stack_end[3] = (s7_pointer)Op; \
      Sc->stack_end += 4; \
  } while (0)

#define push_stack_no_args(Sc, Op, Code) \
  do { \
      Sc->stack_end[0] = Code; \
      Sc->stack_end[1] = Sc->envir; \
      Sc->stack_end[3] = (s7_pointer)Op; \
      Sc->stack_end += 4; \
  } while (0)
#define push_stack_no_let(Sc, Op, Args, Code) \
  do { \
      Sc->stack_end[0] = Code; \
      Sc->stack_end[2] = Args; \
      Sc->stack_end[3] = (s7_pointer)Op; \
      Sc->stack_end += 4; \
  } while (0)
#define push_stack_op(Sc, Op) \
  do { \
      Sc->stack_end[3] = (s7_pointer)Op; \
      Sc->stack_end += 4; \
  } while (0)
#define push_stack_op_let(Sc, Op) \
  do { \
      Sc->stack_end[1] = Sc->envir; \
      Sc->stack_end[3] = (s7_pointer)Op; \
      Sc->stack_end += 4; \
  } while (0)
#endif
/* since we don't GC mark the stack past the stack_top, push_stack_no_args and friends can cause pop_stack to set
 *   sc->code and sc->args to currently free objects.
 */

#define main_stack_op(Sc)   ((opcode_t)(Sc->stack_end[-1]))
/* #define main_stack_args(Sc) (Sc->stack_end[-2]) */
/* #define main_stack_let(Sc)  (Sc->stack_end[-3]) */
/* #define main_stack_code(Sc) (Sc->stack_end[-4]) */
/* #define pop_main_stack(Sc)  Sc->stack_end -= 4 */

/* beware of main_stack_code!  If a function has a tail-call, the main_stack_code that form sees
 *   if main_stack_op==op-begin1 can change from call to call -- the begin actually refers
 *   to the caller, which is dependent on where the current function was called, so we can't hard-wire
 *   any optimizations based on that sequence.
 */

static void stack_reset(s7_scheme *sc)
{
  sc->stack_end = sc->stack_start;
  push_stack_op(sc, OP_EVAL_DONE);
  push_stack_op(sc, OP_BARRIER);
}

static void resize_stack(s7_scheme *sc)
{
  uint64_t loc;
  uint32_t new_size;
  block_t *ob, *nb;

  loc = s7_stack_top(sc);
  new_size = sc->stack_size * 2;

  /* how can we trap infinite recursion?  Is a warning in order here? I think I'll add 'max-stack-size */
  if (new_size > sc->max_stack_size)
    s7_error(sc, s7_make_symbol(sc, "stack-too-big"), set_elist_1(sc, wrap_string(sc, "stack has grown past (*s7* 'max-stack-size)", 43)));

  ob = stack_block(sc->stack);
  nb = reallocate(sc, ob, new_size * sizeof(s7_pointer));
  block_info(nb) = NULL;
  stack_block(sc->stack) = nb;
  stack_elements(sc->stack) = (s7_pointer *)block_data(nb);
  if (!stack_elements(sc->stack))
    {
      s7_warn(sc, 32, "can't allocate additional stack\n");
#if S7_DEBUGGING
      abort();
#endif
      s7_error(sc, s7_make_symbol(sc, "stack-too-big"), set_elist_1(sc, wrap_string(sc, "no room to expand stack?", 24)));
    }
#if 0
  for (i = sc->stack_size; i < new_size; i++)
    stack_element(sc->stack, i) = sc->nil;
#else
  {
    s7_pointer *orig;
    s7_int i, left;
    i = sc->stack_size;
    left = new_size - i - 8;
    orig = stack_elements(sc->stack);
    while (i <= left)
      LOOP_8(orig[i++] = sc->nil);
    for (; i < new_size; i++)
      orig[i] = sc->nil;
  }
#endif
  vector_length(sc->stack) = new_size;
  sc->stack_size = new_size;

  sc->stack_start = stack_elements(sc->stack);
  sc->stack_end = (s7_pointer *)(sc->stack_start + loc);
  sc->stack_resize_trigger = (s7_pointer *)(sc->stack_start + sc->stack_size / 2);

  if (show_stack_stats(sc))
    {
      s7_warn(sc, 128, "stack grows to %u, %s\n", new_size, DISPLAY_80(sc->code));
      s7_show_let(sc);
    }
}

#define check_stack_size(Sc) do {if (Sc->stack_end >= Sc->stack_resize_trigger) resize_stack(Sc);} while (0)

s7_pointer s7_gc_protect_via_stack(s7_scheme *sc, s7_pointer x)
{
  push_stack(sc, OP_GC_PROTECT, sc->args, sc->code);
  return(x);
}


/* -------------------------------- symbols -------------------------------- */

static inline uint64_t raw_string_hash(const uint8_t *key, s7_int len)
{
  uint64_t x;
  uint8_t *cx = (uint8_t *)&x;
  x = 0;
  if (len <= 8)
    memcpy((void *)cx, (void *)key, len);
  else
    {
      uint64_t y;
      uint8_t *cy = (uint8_t *)&y;
      memcpy((void *)cx, (void *)key, 8);
      y = 0;
      len -= 8;
      memcpy((void *)cy, (void *)(key + 8), (len > 8) ? 8 : len);
      x += y;  /* better than |= but still not great if (for example) > 1B gensyms -- maybe add z? */
    }
  return(x);
}

static uint8_t *alloc_symbol(s7_scheme *sc)
{
  #define SYMBOL_SIZE (3 * sizeof(s7_cell) + sizeof(block_t))
  #define ALLOC_SYMBOL_SIZE (64 * SYMBOL_SIZE)
  uint8_t *result;

  if (sc->alloc_symbol_k == ALLOC_SYMBOL_SIZE)
    {
      sc->alloc_symbol_cells = (uint8_t *)malloc(ALLOC_SYMBOL_SIZE);
      sc->alloc_symbol_k = 0;
    }
  result = &(sc->alloc_symbol_cells[sc->alloc_symbol_k]);
  sc->alloc_symbol_k += SYMBOL_SIZE;
  return(result);
}

static inline s7_pointer make_symbol_with_length(s7_scheme *sc, const char *name, s7_int len);
static s7_pointer permanent_slot(s7_scheme *sc, s7_pointer symbol, s7_pointer value);

static inline s7_pointer new_symbol(s7_scheme *sc, const char *name, s7_int len, uint64_t hash, uint32_t location)
{
  /* name might not be null-terminated, these are permanent symbols even in s7_gensym; g_gensym handles everything separately */
  s7_pointer x, str, p;
  uint8_t *base, *val;

  base = alloc_symbol(sc);
  x = (s7_pointer)base;
  str = (s7_pointer)(base + sizeof(s7_cell));
  p = (s7_pointer)(base + 2 * sizeof(s7_cell));
  val = (uint8_t *)alloc_permanent_string(sc, len + 1);
  memcpy((void *)val, (void *)name, len);
  val[len] = '\0';

  typeflag(str) = T_STRING | T_IMMUTABLE | T_UNHEAP;       /* avoid debugging confusion involving set_type (also below) */
  string_length(str) = len;
  string_value(str) = (char *)val;
  string_hash(str) = hash;

  typeflag(x) = T_SYMBOL | T_UNHEAP;
  symbol_set_name_cell(x, str);
  set_global_slot(x, sc->undefined);                       /* was sc->nil */
  symbol_info(x) = (block_t *)(base + 3 * sizeof(s7_cell));
  set_initial_slot(x, sc->undefined);
  symbol_set_local_unchecked(x, 0LL, sc->nil);
  symbol_set_tag(x, 0);
  symbol_set_tag2(x, 0);
  symbol_set_ctr(x, 0);
  symbol_set_type(x, 0);

  if (len > 1)                                             /* not 0, otherwise : is a keyword */
    {
      if ((name[0] == ':') || (name[len - 1] == ':'))
	{
	  s7_pointer slot;
	  set_type_bit(x, T_IMMUTABLE | T_KEYWORD | T_GLOBAL);
	  keyword_set_symbol(x, make_symbol_with_length(sc, (name[0] == ':') ? (char *)(name + 1) : name, len - 1));
	  set_has_keyword(keyword_symbol(x));
	  slot = permanent_slot(sc, x, x);
	  set_global_slot(x, slot);
	  set_local_slot(x, slot);
	}
    }
  typeflag(p) = T_PAIR | T_IMMUTABLE | T_UNHEAP;
  set_car(p, x);
  set_cdr(p, vector_element(sc->symbol_table, location));
  vector_element(sc->symbol_table, location) = p;
  pair_set_raw_hash(p, hash);
  pair_set_raw_len(p, (uint32_t)len); /* symbol name length, so it ought to fit! */
  pair_set_raw_name(p, string_value(str));

  return(x);
}

static inline s7_pointer make_symbol_with_length(s7_scheme *sc, const char *name, s7_int len)
{
  s7_pointer x;
  uint64_t hash;
  uint32_t location;

  hash = raw_string_hash((const uint8_t *)name, len);
  location = hash % SYMBOL_TABLE_SIZE;

  if (len <= 8)
    {
      for (x = vector_element(sc->symbol_table, location); is_pair(x); x = cdr(x))
	if ((hash == pair_raw_hash(x)) &&
	    (len == pair_raw_len(x)))
	  return(car(x));
    }
  else
    {
      for (x = vector_element(sc->symbol_table, location); is_pair(x); x = cdr(x))
	if ((hash == pair_raw_hash(x)) &&
	    (len == pair_raw_len(x)) &&
	    (strings_are_equal_with_length(name, pair_raw_name(x), len))) /* length here because name might not be null-terminated */
	  return(car(x));
    }
  return(new_symbol(sc, name, len, hash, location));
}


static s7_pointer make_symbol(s7_scheme *sc, const char *name)
{
  return(make_symbol_with_length(sc, name, safe_strlen(name)));
}


s7_pointer s7_make_symbol(s7_scheme *sc, const char *name)
{
  if (!name) return(sc->F);
  return(make_symbol_with_length(sc, name, safe_strlen(name)));
}

static s7_pointer symbol_table_find_by_name(s7_scheme *sc, const char *name, uint64_t hash, uint32_t location)
{
  s7_pointer x;
  for (x = vector_element(sc->symbol_table, location); is_not_null(x); x = cdr(x))
    if ((hash == pair_raw_hash(x)) &&
	(strings_are_equal(name, pair_raw_name(x))))
      return(car(x));
  return(sc->nil);
}

s7_pointer s7_symbol_table_find_name(s7_scheme *sc, const char *name)
{
  uint64_t hash;
  uint32_t location;
  s7_pointer result;

  hash = raw_string_hash((const uint8_t *)name, safe_strlen(name));
  location = hash % SYMBOL_TABLE_SIZE;
  result = symbol_table_find_by_name(sc, name, hash, location);
  if (is_null(result))
    return(NULL);

  return(result);
}


#define FILLED true
#define NOT_FILLED false

static s7_pointer g_symbol_table(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_table "(symbol-table) returns a vector containing the current symbol-table symbols"
  #define Q_symbol_table s7_make_signature(sc, 1, sc->is_vector_symbol)

  s7_pointer lst, x;
  s7_pointer *els;
  int32_t i, j, syms = 0;

  /* this can't be optimized by returning the actual symbol-table (a vector of lists), because
   *    gensyms can cause the table's lists and symbols to change at any time.  This wreaks havoc
   *    on traversals like for-each.  So, symbol-table returns a snap-shot of the table contents
   *    at the time it is called, and we call gc before making the list.  I suppose the next step
   *    is to check that we have room, and increase the heap here if necessary!
   *
   *    (define (for-each-symbol func num) (for-each (lambda (sym) (if (> num 0) (for-each-symbol func (- num 1)) (func sym))) (symbol-table)))
   *    (for-each-symbol (lambda (sym) (gensym) 1))
   */

  for (i = 0; i < SYMBOL_TABLE_SIZE; i++)
    for (x = vector_element(sc->symbol_table, i); is_not_null(x); x = cdr(x))
      syms++;
  sc->w = make_simple_vector(sc, syms);
  els = vector_elements(sc->w);

  for (i = 0, j = 0; i < SYMBOL_TABLE_SIZE; i++)
    for (x = vector_element(sc->symbol_table, i); is_not_null(x); x = cdr(x))
      els[j++] = car(x);

  lst = sc->w;
  sc->w = sc->nil;
  return(lst);
}

bool s7_for_each_symbol_name(s7_scheme *sc, bool (*symbol_func)(const char *symbol_name, void *data), void *data)
{
  /* this includes the special constants #<unspecified> and so on for simplicity -- are there any others? */
  int32_t i;
  s7_pointer x;

  for (i = 0; i < SYMBOL_TABLE_SIZE; i++)
    for (x = vector_element(sc->symbol_table, i); is_not_null(x); x = cdr(x))
      if (symbol_func(symbol_name(car(x)), data))
	return(true);

  return((symbol_func("#t", data))             ||
	 (symbol_func("#f", data))             ||
	 (symbol_func("#<unspecified>", data)) ||
	 (symbol_func("#<undefined>", data))   ||
	 (symbol_func("#<eof>", data))         ||
	 (symbol_func("#true", data))          ||
	 (symbol_func("#false", data)));
}

bool s7_for_each_symbol(s7_scheme *sc, bool (*symbol_func)(const char *symbol_name, void *data), void *data)
{
  int32_t i;
  s7_pointer x;

  for (i = 0; i < SYMBOL_TABLE_SIZE; i++)
    for (x = vector_element(sc->symbol_table, i); is_not_null(x); x = cdr(x))
      if (symbol_func(symbol_name(car(x)), data))
	return(true);

  return(false);
}

/* -------------------------------- gensym -------------------------------- */
static void remove_gensym_from_symbol_table(s7_scheme *sc, s7_pointer sym)
{
  /* sym is a free cell at this point (we're called after the GC), but the name_cell is still intact */
  s7_pointer x, name;
  uint32_t location;

  name = symbol_name_cell(sym);
  location = string_hash(name) % SYMBOL_TABLE_SIZE;
  x = vector_element(sc->symbol_table, location);

  if (car(x) == sym)
    vector_element(sc->symbol_table, location) = cdr(x);
  else
    {
      s7_pointer y;
	for (y = x, x = cdr(x); is_pair(x); y = x, x = cdr(x))
	if (car(x) == sym)
	  {
	    set_cdr(y, cdr(x));
	    return;
	  }
#if S7_DEBUGGING
      fprintf(stderr, "could not remove %s?\n", string_value(name));
#endif
    }
}

s7_pointer s7_gensym(s7_scheme *sc, const char *prefix)
{
  block_t *b;
  char *name;
  uint32_t location;
  s7_int len;
  uint64_t hash;
  s7_pointer x;

  len = safe_strlen(prefix) + 32;
  b = mallocate(sc, len);
  name = (char *)block_data(b);
  /* there's no point in heroic efforts here to avoid name collisions -- the user can screw up no matter what we do */
  name[0] = '\0';
  len = catstrs(name, len, "{", (prefix) ? prefix : "", "}-", pos_int_to_str_direct(sc, sc->gensym_counter++), NULL);
  hash = raw_string_hash((const uint8_t *)name, len);
  location = hash % SYMBOL_TABLE_SIZE;
  x = new_symbol(sc, name, len, hash, location);  /* not T_GENSYM -- might be called from outside */
  liberate(sc, b);
  return(x);
}

static bool s7_is_gensym(s7_pointer g) {return((is_symbol(g)) && (is_gensym(g)));}

static s7_pointer g_is_gensym(s7_scheme *sc, s7_pointer args)
{
  #define H_is_gensym "(gensym? sym) returns #t if sym is a gensym"
  #define Q_is_gensym sc->pl_bt

  check_boolean_method(sc, s7_is_gensym, sc->is_gensym_symbol, args);
}

static s7_pointer g_gensym(s7_scheme *sc, s7_pointer args)
{
  #define H_gensym "(gensym (prefix \"gensym\")) returns a new, unused symbol"
  #define Q_gensym s7_make_signature(sc, 2, sc->is_gensym_symbol, sc->is_string_symbol)

  const char *prefix;
  char *name, *p, *base;
  s7_int len, plen, nlen;
  uint32_t location;
  uint64_t hash;
  s7_pointer x, str, stc;
  block_t *b, *ib;

  /* get symbol name */
  if (is_not_null(args))
    {
      s7_pointer gname;
      gname = car(args);
      if (!is_string(gname))
	return(method_or_bust_one_arg(sc, gname, sc->gensym_symbol, args, T_STRING));
      prefix = string_value(gname);
      plen = safe_strlen(prefix);
    }
  else
    {
      prefix = "gensym";
      plen = 6;
    }
  len = plen + 32; /* why 32 -- we need room for the gensym_counter integer, but (length "9223372036854775807") = 19 */

  b = mallocate(sc, len + sizeof(block_t) + 2 * sizeof(s7_cell));
  /* only 16 of block_t size is actually needed here because only the ln.tag (symbol_tag2) field is used in the embedded block_t */
  base = (char *)block_data(b);
  str = (s7_cell *)base;
  stc = (s7_cell *)(base + sizeof(s7_cell));
  ib = (block_t *)(base + 2 * sizeof(s7_cell));
  name = (char *)(base + sizeof(block_t) + 2 * sizeof(s7_cell));

  name[0] = '{';
  if (plen > 0) memcpy((void *)(name + 1), prefix, plen);
  name[plen + 1] = '}';
  name[plen + 2] = '-'; /* {gensym}-nnn */

  p = pos_int_to_str(sc, sc->gensym_counter++, &len, '\0');
  memcpy((void *)(name + plen + 3), (void *)p, len);
  nlen = len + plen + 2;

  hash = raw_string_hash((const uint8_t *)name, nlen);
  location = hash % SYMBOL_TABLE_SIZE;

  /* make-string for symbol name */
#if S7_DEBUGGING
  typeflag(str) = 0; /* here and below, this is needed to avoid set_type check errors (mallocate above) */
#endif
  set_type(str, T_STRING | T_IMMUTABLE | T_UNHEAP);
  string_length(str) = nlen;
  string_value(str) = name;
  string_hash(str) = hash;

  /* allocate the symbol in the heap so GC'd when inaccessible */
  new_cell(sc, x, T_SYMBOL | T_GENSYM);
  symbol_set_name_cell(x, str);
  symbol_info(x) = ib;
  set_global_slot(x, sc->undefined);
  /* set_initial_slot(x, sc->undefined); */
  symbol_set_local_unchecked(x, 0LL, sc->nil);
  symbol_set_ctr(x, 0);
  symbol_set_tag(x, 0);
  symbol_set_tag2(x, 0);
  gensym_block(x) = b;

  /* place new symbol in symbol-table, but using calloc so we can easily free it (remove it from the table) in GC sweep */
#if S7_DEBUGGING
  typeflag(stc) = 0;
#endif
  set_type(stc, T_PAIR | T_IMMUTABLE | T_UNHEAP);
  set_car(stc, x);
  set_cdr(stc, vector_element(sc->symbol_table, location));
  vector_element(sc->symbol_table, location) = stc;
  pair_set_raw_hash(stc, hash);
  pair_set_raw_len(stc, (uint32_t)string_length(str));
  pair_set_raw_name(stc, string_value(str));

  add_gensym(sc, x);
  return(x);
}


/* -------------------------------- syntax? -------------------------------- */
bool s7_is_syntax(s7_pointer p)
{
  return(is_syntax(p));
}

static s7_pointer g_is_syntax(s7_scheme *sc, s7_pointer args)
{
  #define H_is_syntax "(syntax? obj) returns #t if obj is a syntactic value (e.g. lambda)"
  #define Q_is_syntax sc->pl_bt

  check_boolean_method(sc, is_syntax, sc->is_syntax_symbol, args);
}

/* -------------------------------- symbol? -------------------------------- */
bool s7_is_symbol(s7_pointer p)
{
  return(is_symbol(p));
}

static s7_pointer g_is_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_is_symbol "(symbol? obj) returns #t if obj is a symbol"
  #define Q_is_symbol sc->pl_bt

  check_boolean_method(sc, is_symbol, sc->is_symbol_symbol, args);
}


const char *s7_symbol_name(s7_pointer p)
{
  return(symbol_name(p));
}

s7_pointer s7_name_to_value(s7_scheme *sc, const char *name)
{
  return(s7_symbol_value(sc, make_symbol(sc, name)));
}

/* -------------------------------- symbol->string -------------------------------- */
static inline s7_pointer make_string_with_length(s7_scheme *sc, const char *str, s7_int len)
{
  s7_pointer x;
  new_cell(sc, x, T_STRING | T_SAFE_PROCEDURE);
  string_block(x) = mallocate(sc, len + 1);
  string_value(x) = (char *)block_data(string_block(x));
  if (len > 0)
    memcpy((void *)string_value(x), (void *)str, len);
  string_value(x)[len] = 0;
  string_length(x) = len;
  string_hash(x) = 0;
  add_string(sc, x);
  return(x);
}

static s7_pointer g_symbol_to_string(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_string "(symbol->string sym) returns the symbol sym converted to a string"
  #define Q_symbol_to_string s7_make_signature(sc, 2, sc->is_string_symbol, sc->is_symbol_symbol)
  s7_pointer sym;

  sym = car(args);
  if (!is_symbol(sym))
    return(method_or_bust_one_arg(sc, sym, sc->symbol_to_string_symbol, args, T_SYMBOL));
  /* s7_make_string uses strlen which stops at an embedded null */
  return(make_string_with_length(sc, symbol_name(sym), symbol_name_length(sym)));    /* return a copy */
}

static s7_pointer g_symbol_to_string_uncopied(s7_scheme *sc, s7_pointer args)
{
  s7_pointer sym;

  sym = car(args);
  if (!is_symbol(sym))
    return(method_or_bust_one_arg(sc, sym, sc->symbol_to_string_symbol, args, T_SYMBOL));
  if (is_gensym(sym))
    return(make_string_with_length(sc, symbol_name(sym), symbol_name_length(sym)));    /* return a copy of gensym name (which will be freed) */
  return(symbol_name_cell(sym));
}

static s7_pointer symbol_to_string_p(s7_scheme *sc, s7_pointer sym)
{
  if (!is_symbol(sym))
    simple_wrong_type_argument(sc, sc->symbol_to_string_symbol, sym, T_SYMBOL);
  return(make_string_with_length(sc, symbol_name(sym), symbol_name_length(sym)));
}

static s7_pointer symbol_to_string_uncopied_p(s7_scheme *sc, s7_pointer sym)
{
  if (!is_symbol(sym))
    simple_wrong_type_argument(sc, sc->symbol_to_string_symbol, sym, T_SYMBOL);
  if (is_gensym(sym))
    return(make_string_with_length(sc, symbol_name(sym), symbol_name_length(sym)));
  return(symbol_name_cell(sym));
}

/* -------------------------------- string->symbol -------------------------------- */
static inline s7_pointer g_string_to_symbol_1(s7_scheme *sc, s7_pointer str, s7_pointer caller)
{
  if (is_string(str))
    {
      if (string_length(str) > 0)
	return(make_symbol_with_length(sc, string_value(str), string_length(str)));
      return(simple_wrong_type_argument_with_type(sc, caller, str, wrap_string(sc, "a non-null string", 17)));
      /* currently if the string has an embedded null, it marks the end of the new symbol name. */
    }
  return(method_or_bust_one_arg(sc, str, caller, list_1(sc, str), T_STRING));
}

static s7_pointer g_string_to_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_string_to_symbol "(string->symbol str) returns the string str converted to a symbol"
  #define Q_string_to_symbol s7_make_signature(sc, 2, sc->is_symbol_symbol, sc->is_string_symbol)
  return(g_string_to_symbol_1(sc, car(args), sc->string_to_symbol_symbol));
}

static s7_pointer string_to_symbol_p_p(s7_scheme *sc, s7_pointer p)
{
  if (is_string(p))
    {
      if (string_length(p) > 0)
	return(make_symbol_with_length(sc, string_value(p), string_length(p)));
      simple_wrong_type_argument_with_type(sc, sc->string_to_symbol_symbol, p, wrap_string(sc, "a non-null string", 17));
    }
  else simple_wrong_type_argument(sc, sc->string_to_symbol_symbol, p, T_STRING);
  return(p);
}

/* -------------------------------- symbol -------------------------------- */
static s7_pointer g_string_append_1(s7_scheme *sc, s7_pointer args, s7_pointer caller);

static s7_pointer g_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol "(symbol str ...) returns its string arguments concatenated and converted to a symbol"
  #define Q_symbol s7_make_circular_signature(sc, 1, 2, sc->is_symbol_symbol, sc->is_string_symbol)

  s7_int len = 0, cur_len;
  s7_pointer p, sym;
  block_t *b;
  char *name;

  for (p = args; is_pair(p); p = cdr(p))
    if (is_string(car(p)))
      len += string_length(car(p));
    else break;

  if (is_pair(p))
    {
      if (is_null(cdr(args)))
	return(g_string_to_symbol_1(sc, car(args), sc->symbol_symbol));
      return(g_string_to_symbol_1(sc, g_string_append_1(sc, args, sc->symbol_symbol), sc->symbol_symbol));
    }
  if (len == 0)
    return(simple_wrong_type_argument_with_type(sc, sc->symbol_symbol, car(args), wrap_string(sc, "a non-null string", 17)));

  b = mallocate(sc, len + 1);
  name = (char *)block_data(b);
  /* can't use catstrs_direct here because it stops at embedded null */
  cur_len = 0;
  for (p = args; is_pair(p); p = cdr(p))
    {
      s7_pointer str;
      str = car(p);
      if (string_length(str) > 0)
	{
	  memcpy((void *)(name + cur_len), (void *)string_value(str), string_length(str));
	  cur_len += string_length(str);
	}
    }
  name[len] = '\0';
  sym = make_symbol_with_length(sc, name, len);
  liberate(sc, b);
  return(sym);
}


/* -------- symbol sets -------- */
static inline s7_pointer add_symbol_to_list(s7_scheme *sc, s7_pointer sym)
{
  symbol_set_tag(sym, sc->syms_tag);
  symbol_set_tag2(sym, sc->syms_tag2);
  return(sym);
}

static inline void clear_symbol_list(s7_scheme *sc)
{
  sc->syms_tag++;
  if (sc->syms_tag == 0)
    {
      sc->syms_tag = 1; /* we're assuming (in let_equal) that this tag is not 0 */
      sc->syms_tag2++;
    }
}

#define symbol_is_in_list(Sc, Sym) ((symbol_tag(Sym) == Sc->syms_tag) && (symbol_tag2(Sym) == Sc->syms_tag2))


/* -------------------------------- environments -------------------------------- */

#define new_frame(Sc, Old_Env, New_Env)		      \
  do {						      \
    s7_pointer _x_;				      \
      new_cell(Sc, _x_, T_LET | T_SAFE_PROCEDURE);    \
      let_id(_x_) = ++sc->let_number;		      \
      let_set_slots(_x_, Sc->nil);	              \
      set_outlet(_x_, Old_Env);			      \
      New_Env = _x_;				      \
  } while (0)
/* this macro is noticeably faster than using the equivalent inlined new_frame_in_env function below */

static inline s7_pointer new_frame_in_env(s7_scheme *sc, s7_pointer old_env)
{
  s7_pointer x;
  new_cell(sc, x, T_LET | T_SAFE_PROCEDURE);
  let_id(x) = ++sc->let_number;
  let_set_slots(x, sc->nil);
  set_outlet(x, old_env);
  return(x);
}

static inline s7_pointer make_simple_let(s7_scheme *sc)
{
  s7_pointer frame;
  new_cell(sc, frame, T_LET | T_SAFE_PROCEDURE);
  let_id(frame) = sc->let_number + 1;
  let_set_slots(frame, sc->nil);
  set_outlet(frame, sc->envir);
  return(frame);
}

/* in all these macros, symbol_set_local should follow slot_set_value so that we can evaluate the slot's value in its old state. */
#define add_slot(Frame, Symbol, Value)			\
  do {							\
    s7_pointer _slot_, _sym_, _val_;			\
    _sym_ = Symbol; _val_ = Value;			\
    new_cell_no_check(sc, _slot_, T_SLOT);		\
    slot_set_symbol(_slot_, _sym_);			\
    slot_set_value(_slot_, _val_);			\
    symbol_set_local(_sym_, let_id(Frame), _slot_);	\
    set_next_slot(_slot_, let_slots(Frame));		\
    let_set_slots(Frame, _slot_);	                \
  } while (0)

#define new_frame_with_slot(Sc, Old_Env, New_Env, Symbol, Value) \
  do {								 \
    s7_pointer _x_, _slot_, _sym_, _val_;			 \
    _sym_ = Symbol; _val_ = Value;				\
    new_cell(Sc, _x_, T_LET | T_SAFE_PROCEDURE);		\
    let_id(_x_) = ++sc->let_number;				\
    set_outlet(_x_, Old_Env);			                \
    New_Env = _x_;						\
    new_cell_no_check(Sc, _slot_, T_SLOT);	                \
    slot_set_symbol(_slot_, _sym_);				\
    slot_set_value(_slot_, _val_);	                        \
    symbol_set_local(_sym_, sc->let_number, _slot_);            \
    set_next_slot(_slot_, sc->nil);			        \
    let_set_slots(_x_, _slot_);					\
  } while (0)

#define new_frame_with_two_slots(Sc, Old_Env, New_Env, Symbol1, Value1, Symbol2, Value2) \
  do {                                   \
    s7_pointer _x_, _slot_, _sym1_, _val1_, _sym2_, _val2_;		\
    _sym1_ = Symbol1; _val1_ = Value1;					\
    _sym2_ = Symbol2; _val2_ = Value2;					\
    new_cell(Sc, _x_, T_LET | T_SAFE_PROCEDURE);			\
    let_id(_x_) = ++sc->let_number;					\
    set_outlet(_x_, Old_Env);				                \
    New_Env = _x_;							\
    new_cell_no_check(Sc, _slot_, T_SLOT);		                \
    slot_set_symbol(_slot_, _sym1_);					\
    slot_set_value(_slot_, _val1_);					\
    symbol_set_local(_sym1_, sc->let_number, _slot_);			\
    let_set_slots(_x_, _slot_);			                        \
    new_cell_no_check(Sc, _x_, T_SLOT);			                \
    slot_set_symbol(_x_, _sym2_);					\
    slot_set_value(_x_, _val2_);					\
    symbol_set_local(_sym2_, sc->let_number, _x_);			\
    set_next_slot(_x_, sc->nil);				        \
    set_next_slot(_slot_, _x_);			                        \
  } while (0)

static s7_pointer reuse_as_let(s7_scheme *sc, s7_pointer frame, s7_pointer next_frame)
{
  /* we're reusing frame here as a let -- it was probably a pair */
#if S7_DEBUGGING
  frame->debugger_bits = 0;
#endif
  set_type(frame, T_LET | T_SAFE_PROCEDURE);
  let_set_slots(frame, sc->nil);
  set_outlet(frame, next_frame);
  let_id(frame) = ++sc->let_number;
  return(frame);
}

static s7_pointer reuse_as_slot(s7_pointer slot, s7_pointer symbol, s7_pointer value)
{
#if S7_DEBUGGING
  slot->debugger_bits = 0;
#endif
  set_type(slot, T_SLOT);
  slot_set_symbol(slot, symbol);
  slot_set_value(slot, T_Pos(value));
  return(slot);
}

static s7_pointer old_frame_with_slot(s7_scheme *sc, s7_pointer env, s7_pointer val)
{
  s7_pointer x, sym;
  uint64_t id;

  id = ++sc->let_number;
  let_id(env) = id;
  x = let_slots(env);
  slot_set_value(x, val);
  sym = slot_symbol(x);
  symbol_set_local(sym, id, x);

  return(env);
}

static s7_pointer old_frame_with_two_slots(s7_scheme *sc, s7_pointer env, s7_pointer val1, s7_pointer val2)
{
  s7_pointer x, sym;
  uint64_t id;

  id = ++sc->let_number;
  let_id(env) = id;
  x = let_slots(env);
  slot_set_value(x, val1);
  sym = slot_symbol(x);
  symbol_set_local(sym, id, x);
  x = next_slot(x);
  slot_set_value(x, val2);
  sym = slot_symbol(x);
  symbol_set_local(sym, id, x);

  return(env);
}

static s7_pointer old_frame_with_three_slots(s7_scheme *sc, s7_pointer env, s7_pointer val1, s7_pointer val2, s7_pointer val3)
{
  s7_pointer x, sym;
  uint64_t id;

  id = ++sc->let_number;
  let_id(env) = id;
  x = let_slots(env);

  slot_set_value(x, val1);
  sym = slot_symbol(x);
  symbol_set_local(sym, id, x);
  x = next_slot(x);

  slot_set_value(x, val2);
  sym = slot_symbol(x);
  symbol_set_local(sym, id, x);
  x = next_slot(x);

  slot_set_value(x, val3);
  sym = slot_symbol(x);
  symbol_set_local(sym, id, x);

  return(env);
}

static s7_pointer permanent_slot(s7_scheme *sc, s7_pointer symbol, s7_pointer value)
{
  s7_pointer x;
  x = alloc_pointer(sc);
  set_type(x, T_SLOT | T_UNHEAP);
  slot_set_symbol(x, symbol);
  slot_set_value(x, value);
  return(x);
}

static s7_pointer find_let(s7_scheme *sc, s7_pointer obj)
{
  if (is_let(obj)) return(obj);
  switch (type(obj))
    {
    case T_MACRO:   case T_MACRO_STAR:
    case T_BACRO:   case T_BACRO_STAR:
    case T_CLOSURE: case T_CLOSURE_STAR:
      return(closure_let(obj));

    case T_C_OBJECT:
      return(c_object_let(obj));

    case T_C_POINTER:
      if ((is_let(c_pointer_info(obj))) &&
	  (c_pointer_info(obj) != sc->rootlet))
	return(c_pointer_info(obj));
    }
  return(sc->nil);
}


static s7_pointer let_fill(s7_scheme *sc, s7_pointer args)
{
  s7_pointer e, val;
  e = car(args);

  if (e == sc->rootlet)
    return(out_of_range(sc, sc->fill_symbol, small_int(1), e, wrap_string(sc, "can't fill! rootlet", 19)));
  if (e == sc->owlet) /* (owlet) copies sc->owlet, so this probably can't happen */
    return(out_of_range(sc, sc->fill_symbol, small_int(1), e, wrap_string(sc, "can't fill! owlet", 17)));
  if (is_funclet(e))
    return(out_of_range(sc, sc->fill_symbol, small_int(1), e, wrap_string(sc, "can't fill! a funclet", 21)));
  if (is_immutable(e))
    return(immutable_object_error(sc, set_elist_3(sc, immutable_error_string, sc->fill_symbol, e)));

  val = cadr(args);
  if (val == sc->undefined)
    {
      let_set_slots(e, sc->nil);
      let_id(e) = ++sc->let_number; /* else previous symbol_id matches! */
    }
  else
    {
      s7_pointer p;
      for (p = let_slots(e); is_slot(p); p = next_slot(p))
	slot_set_value(p, val);
    }
  return(val);
}


static s7_pointer find_method(s7_scheme *sc, s7_pointer env, s7_pointer symbol)
{
  s7_pointer x;
  if (symbol_id(symbol) == 0) /* this means the symbol has never been used locally, so how can it be a method? */
    return(sc->undefined);

  /* I think the symbol_id is in sync with let_id, so the standard search should work */
  if (let_id(env) == symbol_id(symbol))
    return(slot_value(local_slot(symbol)));

  for (x = env; symbol_id(symbol) < let_id(x); x = outlet(x));

  if (let_id(x) == symbol_id(symbol))
    return(slot_value(local_slot(symbol)));

  for (; is_let(x); x = outlet(x))
    {
      s7_pointer y;
      for (y = let_slots(x); is_slot(y); y = next_slot(y))
	if (slot_symbol(y) == symbol)
	  return(slot_value(y));
    }
  return(sc->undefined);
}

static s7_int let_length(s7_scheme *sc, s7_pointer e)
{
  /* used by length, applicable_length, copy, and some length optimizations */
  s7_int i;
  s7_pointer p;

  if (e == sc->rootlet)
    return(sc->rootlet_entries);

  if (has_methods(e))
    {
      s7_pointer length_func;
      length_func = find_method(sc, e, sc->length_symbol);
      if (length_func != sc->undefined)
	{
	  p = s7_apply_function(sc, length_func, list_1(sc, e));
	  if (s7_is_integer(p))
	    return(s7_integer(p));
	  return(-1); /* ?? */
	}
    }

  for (i = 0, p = let_slots(e); is_slot(p); i++, p = next_slot(p));
  return(i);
}


static void slot_set_value_with_hook_1(s7_scheme *sc, s7_pointer slot, s7_pointer value)
  {
    /* (set! (hook-functions *rootlet-redefinition-hook*) (list (lambda (hook) (format *stderr* "~A ~A~%" (hook 'symbol) (hook 'value))))) */
    s7_pointer symbol;
    symbol = slot_symbol(slot);
    if ((global_slot(symbol) == slot) &&
	(value != slot_value(slot)))
      s7_call(sc, sc->rootlet_redefinition_hook, set_elist_2(sc, symbol, value));
    slot_set_value(slot, value);
  }


static s7_pointer make_slot_1(s7_scheme *sc, s7_pointer env, s7_pointer symbol, s7_pointer value)
{
  /* env is not rootlet and is a let */
  s7_pointer slot;
  new_cell(sc, slot, T_SLOT);
  slot_set_symbol(slot, symbol);
  slot_set_value(slot, value);
  set_next_slot(slot, let_slots(env));
  let_set_slots(env, slot);
  set_local(symbol);
  /* this is called by varlet so we have to be careful about the resultant let_id
   *   check for greater to ensure shadowing stays in effect, and equal to do updates (set! in effect)
   */
  if (let_id(env) >= symbol_id(symbol))
    symbol_set_local(symbol, let_id(env), slot);
  return(slot);
}

static hash_entry_t *hash_eq(s7_scheme *sc, s7_pointer table, s7_pointer key);
static s7_pointer hash_table_iterate(s7_scheme *sc, s7_pointer iterator);
static void remove_function_from_heap(s7_scheme *sc, s7_pointer value);

static void remove_let_from_heap(s7_scheme *sc, s7_pointer lt)
{
  s7_pointer p;
  for (p = let_slots(lt); is_slot(p); p = next_slot(p))
    {
      s7_pointer val;
      val = slot_value(p);
      if ((has_closure_let(val)) &&
	  (in_heap(closure_args(val))))
	remove_function_from_heap(sc, val);
      else
	{
	  /* an experiment... */
	  if ((is_hash_table(val)) &&
	      (!hash_table_removed(val)))
	    {
	      s7_pointer iterator, ip;
	      s7_int gc_iter;
	      s7_int i, len;

	      len = hash_table_entries(val);
	      iterator = s7_make_iterator(sc, val);
	      gc_iter = s7_gc_protect_1(sc, iterator);
	      ip = cons(sc, sc->F, sc->F);
	      iterator_current(iterator) = ip;
	      set_mark_seq(iterator);
	      for (i = 0; i < len; i++)
		{
		  s7_pointer key_val;
		  key_val = hash_table_iterate(sc, iterator);
		  if ((has_closure_let(cdr(key_val))) &&
		      (in_heap(closure_args(cdr(key_val)))))
		    remove_function_from_heap(sc, cdr(key_val));
		}
	      hash_table_set_removed(val);
	      s7_gc_unprotect_at(sc, gc_iter);
	      iterator_current(iterator) = sc->nil;
	      free_cell(sc, ip);
	      free_cell(sc, iterator);
	    }
	}
    }
  let_set_removed(lt);
}

static void remove_function_from_heap(s7_scheme *sc, s7_pointer value)
{
  s7_pointer lt;

  s7_remove_from_heap(sc, closure_args(value));
  s7_remove_from_heap(sc, closure_body(value));

  /* remove closure if it's local to current func (meaning (define f (let ...) (lambda ...)) removes the enclosing let) */
  lt = closure_let(value);
  if ((is_let(lt)) && (!let_removed(lt)) && (lt != sc->rootlet) && (lt != sc->shadow_rootlet))
    {
      lt = outlet(lt);
      if ((is_let(lt)) && (!let_removed(lt)) && (lt != sc->rootlet) && (lt != sc->shadow_rootlet))
	{
	  remove_let_from_heap(sc, lt);
	  lt = outlet(lt);
	  if ((is_let(lt)) && (!let_removed(lt)) && (lt != sc->rootlet) && (lt != sc->shadow_rootlet))
	    remove_let_from_heap(sc, lt);
	}
    }
}

s7_pointer s7_make_slot(s7_scheme *sc, s7_pointer env, s7_pointer symbol, s7_pointer value)
{
  if ((!is_let(env)) ||
      (env == sc->rootlet))
    {
      s7_pointer ge, slot;
      if (is_immutable(sc->rootlet)) /* PERHAPS: since this is undoable and disables much of s7, it should be an error? */
	return(immutable_object_error(sc, set_elist_3(sc, immutable_error_string, sc->define_symbol, sc->rootlet)));
      if ((sc->safety == NO_SAFETY) &&
	  (has_closure_let(value)))
	remove_function_from_heap(sc, value);

      /* first look for existing slot -- this is not always checked before calling s7_make_slot */
      if (is_slot(global_slot(symbol)))
	{
	  slot = global_slot(symbol);
	  slot_set_value_with_hook(slot, value);
	  return(slot);
	}

      ge = sc->rootlet;
      slot = permanent_slot(sc, symbol, value);
      rootlet_element(ge, sc->rootlet_entries++) = slot;
      if (sc->rootlet_entries >= vector_length(ge))
	{
	  s7_int i, len;
	  block_t *ob, *nb;
	  vector_length(ge) *= 2;
	  len = vector_length(ge);
	  ob = rootlet_block(ge);
	  nb = reallocate(sc, ob, len * sizeof(s7_pointer));
	  block_info(nb) = NULL;
	  rootlet_block(ge) = nb;
	  rootlet_elements(ge) = (s7_pointer *)block_data(nb);
	  for (i = sc->rootlet_entries; i < len; i++)
	    rootlet_element(ge, i) = sc->nil;
	}
      set_global_slot(symbol, slot);

      if (symbol_id(symbol) == 0)    /* never defined locally? */
	{
	  if (!is_gensym(symbol))
	    {
#if S7_DEBUGGING
	      if (!symbol_info(symbol)) fprintf(stderr, "%s info is null?\n", symbol_name(symbol));
#endif
	      if (initial_slot(symbol) == sc->undefined)
		set_initial_slot(symbol, permanent_slot(sc, symbol, value));
	    }
	  set_local_slot(symbol, slot);
	  symbol_increment_ctr(symbol);
	  set_global(symbol);
	}
      if (is_gensym(symbol))
	s7_remove_from_heap(sc, symbol);
      return(slot);
    }

  return(make_slot_1(sc, env, symbol, value));
  /* there are about the same number of frames as local variables -- this
   *   strikes me as surprising, but it holds up across a lot of code.
   */
}


static s7_pointer make_slot(s7_scheme *sc, s7_pointer variable, s7_pointer value)
{
  /* this is for a do-loop optimization -- an unattached slot */
  s7_pointer y;
  new_cell(sc, y, T_SLOT);
  slot_set_symbol(y, variable);
  slot_set_value(y, value);
  return(y);
}


/* -------------------------------- let? -------------------------------- */
bool s7_is_let(s7_pointer e)
{
  return(is_let(e));
}

static s7_pointer g_is_let(s7_scheme *sc, s7_pointer args)
{
  #define H_is_let "(let? obj) returns #t if obj is a let (an environment)."
  #define Q_is_let sc->pl_bt

  check_boolean_method(sc, is_let, sc->is_let_symbol, args);
}


/* -------------------------------- unlet -------------------------------- */
#define UNLET_ENTRIES 512 /* 397 if not --disable-deprecated etc */

static void save_unlet(s7_scheme *sc)
{
  int32_t i, k = 0;
  s7_pointer x;
  s7_pointer *inits;
  block_t *block;

  sc->unlet = (s7_pointer)calloc(1, sizeof(s7_cell));
  set_type(sc->unlet, T_VECTOR | T_UNHEAP);
  vector_length(sc->unlet) = UNLET_ENTRIES;
  block = mallocate(sc, UNLET_ENTRIES * sizeof(s7_pointer));
  vector_block(sc->unlet) = block;
  vector_elements(sc->unlet) = (s7_pointer *)block_data(block);
  vector_set_dimension_info(sc->unlet, NULL);
  vector_getter(sc->unlet) = default_vector_getter;
  vector_setter(sc->unlet) = default_vector_setter;
  inits = vector_elements(sc->unlet);
  s7_vector_fill(sc, sc->unlet, sc->nil);

  for (i = 0; i < SYMBOL_TABLE_SIZE; i++)
    for (x = vector_element(sc->symbol_table, i); is_not_null(x); x = cdr(x))
      {
	s7_pointer sym;
	sym = car(x);
	if ((!is_gensym(sym)) && (is_slot(initial_slot(sym))))
	  {
	    s7_pointer val;
	    val = slot_value(initial_slot(sym));
	    if ((is_c_function(val)) || (is_syntax(val))) /* we're assuming the initial_slots values of these guys need no GC protection */
	      inits[k++] = initial_slot(sym);

	    /* non-c_functions that are not 'set! (and therefore initial_slot GC) protected by default:
	     *    make-hook hook-functions
	     * if these initial_slot values are added to unlet, they need explicit GC protection.
	     */
	    /* (let ((begin +)) (with-let (unlet) (begin 1 2))) */
#if S7_DEBUGGING
	    if (k >= UNLET_ENTRIES)
	      fprintf(stderr, "unlet overflow\n");
#endif
	  }
      }
}

static s7_pointer g_unlet(s7_scheme *sc, s7_pointer args)
{
  /* add sc->unlet bindings to the current environment */
  #define H_unlet "(unlet) returns a let that establishes the original bindings of all the predefined functions"
  #define Q_unlet s7_make_signature(sc, 1, sc->is_let_symbol)

  /* slightly confusing:
   *    :((unlet) 'abs)
   *    #<undefined>
   *    :(defined? 'abs (unlet))
   *    #t
   * this is because unlet sets up a local environment of unshadowed symbols,
   *   and s7_let_ref below only looks at the local env chain (that is, if env is not
   *   the global env, then the global env is not searched).
   *
   * Also (define hi 3) #_hi => 3, (set! hi 4), #_hi -> 3 but (with-let (unlet) hi) -> 4!
   */
  int32_t i;
  s7_pointer *inits;
  s7_pointer x;

  sc->w = new_frame_in_env(sc, sc->envir);
  inits = vector_elements(sc->unlet);

  for (i = 0; (i < UNLET_ENTRIES) && (is_slot(inits[i])); i++)
    {
      s7_pointer sym;
      x = slot_value(inits[i]);
      sym = slot_symbol(inits[i]);
      if (!is_immutable_symbol(sym))
	{
	  if (is_t_procedure(x))
	    {
	      if (((!is_global(sym)) &&                  /* it might be shadowed locally */
		   (s7_symbol_local_value(sc, sym, sc->envir) != slot_value(global_slot(sym)))) ||
		  (x != slot_value(global_slot(sym))))   /* it's not shadowed, but has been changed globally */
		make_slot_1(sc, sc->w, sym, x);
	    }
	  else
	    {
	      if ((is_syntax(x)) &&
		  (local_slot(sym) != sc->nil))          /* this can be a freed cell, will be nil if unchanged */
		make_slot_1(sc, sc->w, sym, x);
	    }
	}
    }
  /* if (set! + -) then + needs to be overridden, but the local bit isn't set,
   *   so we have to check the actual values in the non-local case.
   *   (define (f x) (with-let (unlet) (+ x 1)))
   */

  x = sc->w;
  sc->w = sc->nil;
  return(x);
}


/* -------------------------------- openlet? -------------------------------- */
bool s7_is_openlet(s7_pointer e)
{
  return(has_methods(e));
}

static s7_pointer g_is_openlet(s7_scheme *sc, s7_pointer args)
{
  #define H_is_openlet "(openlet? obj) returns #t is 'obj' has methods."
  #define Q_is_openlet sc->pl_bt
  s7_pointer e;

  e = car(args);
  /* if e is not a let, should this raise an error? -- no, easier to use this way in cond */
  check_method(sc, e, sc->is_openlet_symbol, args);
  return(make_boolean(sc, has_methods(e)));
}


/* -------------------------------- openlet -------------------------------- */
s7_pointer s7_openlet(s7_scheme *sc, s7_pointer e)
{
  set_has_methods(e);
  return(e);
}

static s7_pointer g_openlet(s7_scheme *sc, s7_pointer args)
{
  #define H_openlet "(openlet e) tells the built-in generic functions that the let 'e might have an over-riding method."
  #define Q_openlet sc->pcl_e
  s7_pointer e, elet, func;

  e = car(args);
  if ((e == sc->rootlet) || (e == sc->nil))
    s7_error(sc, sc->error_symbol, set_elist_1(sc, wrap_string(sc, "can't openlet rootlet", 21)));
  elet = find_let(sc, e); /* returns nil if no let found, so has to follow error check above */
  if (!is_let(elet))
    return(simple_wrong_type_argument_with_type(sc, sc->openlet_symbol, e, a_let_string));

  if ((has_methods(e)) &&
      ((func = find_method(sc, elet, sc->openlet_symbol)) != sc->undefined))
    return(s7_apply_function(sc, func, args));

  set_has_methods(e);
  return(e);
}


/* -------------------------------- coverlet -------------------------------- */

static s7_pointer g_coverlet(s7_scheme *sc, s7_pointer args)
{
  s7_pointer e;
  #define H_coverlet "(coverlet e) undoes an earlier openlet."
  #define Q_coverlet sc->pcl_e

  e = car(args);
  sc->temp3 = e;
  check_method_uncopied(sc, e, sc->coverlet_symbol, list_1(sc, e));
  sc->temp3 = sc->nil;
  if (e == sc->rootlet)
    s7_error(sc, sc->error_symbol, set_elist_1(sc, wrap_string(sc, "can't coverlet rootlet", 22)));

  if ((is_let(e)) ||
      (has_closure_let(e)) ||
      ((is_c_object(e)) && (c_object_let(e) != sc->nil)) ||
      ((is_c_pointer(e)) && (is_let(c_pointer_info(e)))))
    {
      clear_has_methods(e);
      return(e);
    }
  return(simple_wrong_type_argument_with_type(sc, sc->coverlet_symbol, e, a_let_string));
}


/* -------------------------------- varlet -------------------------------- */
static void append_let(s7_scheme *sc, s7_pointer new_e, s7_pointer old_e)
{
  s7_pointer x;

  if (old_e == sc->rootlet)
    return;

  if (new_e != sc->rootlet)
    {
      for (x = let_slots(old_e); is_slot(x); x = next_slot(x))
	make_slot_1(sc, new_e, slot_symbol(x), slot_value(x)); /* not add_slot here because we might run off the free heap end */
    }
  else
    {
      for (x = let_slots(old_e); is_slot(x); x = next_slot(x))
	{
	  s7_pointer sym, val;
	  sym = slot_symbol(x);
	  val = slot_value(x);
	  if (is_slot(global_slot(sym)))
	    slot_set_value(global_slot(sym), val);
	  else s7_make_slot(sc, new_e, sym, val);
	}
    }
}

static s7_pointer check_c_obj_env(s7_scheme *sc, s7_pointer old_e, s7_pointer caller)
{
  if (is_c_object(old_e))
    old_e = c_object_let(old_e);
  if (!is_let(old_e))
    return(simple_wrong_type_argument_with_type(sc, caller, old_e, a_let_string));
  return(old_e);
}


s7_pointer s7_varlet(s7_scheme *sc, s7_pointer env, s7_pointer symbol, s7_pointer value)
{
  if (!is_let(env))
    return(wrong_type_argument_with_type(sc, sc->varlet_symbol, 1, env, a_let_string));

  if (!is_symbol(symbol))
    return(wrong_type_argument_with_type(sc, sc->varlet_symbol, 2, symbol, a_symbol_string));

  if ((is_slot(global_slot(symbol))) &&
      (is_syntax(slot_value(global_slot(symbol)))))
    return(wrong_type_argument_with_type(sc, sc->varlet_symbol, 2, symbol, wrap_string(sc, "a non-syntactic name", 20)));

  if (env == sc->rootlet)
    {
      if (is_slot(global_slot(symbol)))
	slot_set_value(global_slot(symbol), value);
      else s7_make_slot(sc, env, symbol, value);
    }
  else make_slot_1(sc, env, symbol, value);
  return(value);
}


static s7_pointer g_varlet(s7_scheme *sc, s7_pointer args)
{
  #define H_varlet "(varlet env ...) adds its arguments (a let, a cons: symbol . value, or a pair of arguments, the symbol and its value) \
to the let env, and returns env."
  #define Q_varlet s7_make_circular_signature(sc, 2, 4, sc->is_let_symbol, \
                     s7_make_signature(sc, 2, sc->is_let_symbol, sc->is_null_symbol), \
                       s7_make_signature(sc, 3, sc->is_pair_symbol, sc->is_symbol_symbol, sc->is_let_symbol), \
                         sc->T)
  /* varlet = with-let + define */
  s7_pointer x, e, sym, val, p;

  e = car(args);
  if (is_null(e))
    e = sc->rootlet;
  else
    {
      check_method(sc, e, sc->varlet_symbol, args);
      if (!is_let(e))
	return(wrong_type_argument_with_type(sc, sc->varlet_symbol, 1, e, a_let_string));
      if (is_immutable(e))
	return(s7_wrong_type_arg_error(sc, "varlet", 1, e, "a mutable let"));
    }
  for (x = cdr(args); is_pair(x); x = cdr(x))
    {
      p = car(x);
      switch (type(p))
	{
	case T_SYMBOL:
	  if (is_keyword(p))
	    sym = keyword_symbol(p);
	  else sym = p;
	  if (!is_pair(cdr(x)))
	    s7_error(sc, sc->error_symbol, set_elist_3(sc, value_is_missing_string, sc->varlet_symbol, car(x)));
	  x = cdr(x);
	  val = car(x);
	  break;

	case T_PAIR:
	  sym = car(p);
	  if (!is_symbol(sym))
	    return(wrong_type_argument_with_type(sc, sc->varlet_symbol, position_of(x, args), p, a_symbol_string));
	  val = cdr(p);
	  break;

	case T_LET:
	  append_let(sc, e, check_c_obj_env(sc, p, sc->varlet_symbol));
	  continue;

	default:
	  return(wrong_type_argument_with_type(sc, sc->varlet_symbol, position_of(x, args), p, a_symbol_string));
	}

      if (is_constant_symbol(sc, sym))
	return(wrong_type_argument_with_type(sc, sc->varlet_symbol, position_of(x, args), sym, a_non_constant_symbol_string));

      if (e == sc->rootlet)
	{
	  if (is_slot(global_slot(sym)))
	    {
	      if (is_syntax(slot_value(global_slot(sym))))
		return(wrong_type_argument_with_type(sc, sc->varlet_symbol, position_of(x, args), p, wrap_string(sc, "a non-syntactic keyword", 23)));
	      /*  without this check we can end up turning our code into gibberish:
	       *   :(set! quote 1)
	       *   ;can't set! quote
	       *   :(varlet (rootlet) '(quote . 1))
	       *   :quote
	       *   1
	       * or worse set quote to a function of one arg that tries to quote something -- infinite loop
	       */
	      slot_set_value_with_hook(global_slot(sym), val);
	    }
	  else s7_make_slot(sc, e, sym, val);
	}
      else make_slot_1(sc, e, sym, val);
      /* this used to check for sym already defined, and set its value, but that greatly slows down
       *   the most common use (adding a slot), and makes it hard to shadow explicitly.  Don't use
       *   varlet as a substitute for set!/let-set!.
       */
    }
  return(e);
}


/* -------------------------------- cutlet -------------------------------- */
static s7_pointer g_cutlet(s7_scheme *sc, s7_pointer args)
{
  #define H_cutlet "(cutlet e symbol ...) removes symbols from the let e."
  #define Q_cutlet s7_make_circular_signature(sc, 2, 3, sc->is_let_symbol, sc->is_let_symbol, sc->is_symbol_symbol)

  s7_pointer e, syms;
  #define THE_UN_ID ++sc->let_number

  e = car(args);
  if (is_null(e))
    e = sc->rootlet;
  else
    {
      check_method(sc, e, sc->cutlet_symbol, args);
      if (!is_let(e))
	return(wrong_type_argument_with_type(sc, sc->cutlet_symbol, 1, e, a_let_string));
      if (is_immutable(e))
	return(immutable_object_error(sc, set_elist_3(sc, immutable_error_string, sc->cutlet_symbol, e)));
    }
  /* besides removing the slot we have to make sure the symbol_id does not match else
   *   let-ref and others will use the old slot!  What's the un-id?  Perhaps the next one?
   *   (let ((b 1)) (let ((b 2)) (cutlet (curlet) 'b)) b)
   */

  for (syms = cdr(args); is_pair(syms); syms = cdr(syms))
    {
      s7_pointer sym, slot;
      sym = car(syms);

      if (!is_symbol(sym))
	return(wrong_type_argument_with_type(sc, sc->cutlet_symbol, position_of(syms, args), sym, a_symbol_string));

      if (is_keyword(sym))
	sym = keyword_symbol(sym);

      if (e == sc->rootlet)
	{
	  if (is_slot(global_slot(sym)))
	    {
	      symbol_set_id(sym, THE_UN_ID);
	      slot_set_value(global_slot(sym), sc->undefined);
	    }
	}
      else
	{
	  slot = let_slots(e);
	  if (is_slot(slot))
	    {
	      if (slot_symbol(slot) == sym)
		{
		  let_set_slots(e, next_slot(let_slots(e)));
		  symbol_set_id(sym, THE_UN_ID);
		}
	      else
		{
		  s7_pointer last_slot;
		  last_slot = slot;
		  for (slot = next_slot(let_slots(e)); is_slot(slot); last_slot = slot, slot = next_slot(slot))
		    {
		      if (slot_symbol(slot) == sym)
			{
			  symbol_set_id(sym, THE_UN_ID);
			  set_next_slot(last_slot, next_slot(slot));
			  break;
			}}}}}
    }
  return(e);
}


/* -------------------------------- sublet -------------------------------- */
static s7_pointer sublet_1(s7_scheme *sc, s7_pointer e, s7_pointer bindings, s7_pointer caller)
{
  s7_pointer new_e;

  if (e == sc->rootlet)
    new_e = new_frame_in_env(sc, sc->nil);
  else new_e = new_frame_in_env(sc, e);
  set_all_methods(new_e, e);

  if (!is_null(bindings))
    {
      s7_pointer x;
      sc->temp3 = new_e;

      for (x = bindings; is_pair(x); x = cdr(x))
	{
	  s7_pointer p, sym, val;

	  p = car(x);
	  switch (type(p))
	    {
	      /* should this insist on one style of field arg?  i.e. (cons sym val) throughout, or :sym val etc? */
	    case T_SYMBOL:
	      if (is_keyword(p))
		sym = keyword_symbol(p);
	      else sym = p;
	      if (!is_pair(cdr(x)))
		s7_error(sc, sc->error_symbol, set_elist_3(sc, value_is_missing_string, caller, car(x)));
	      x = cdr(x);
	      val = car(x);
	      break;

	    case T_PAIR:
	      sym = car(p);
	      if (!is_symbol(sym))
		return(wrong_type_argument_with_type(sc, caller, position_of(x, bindings), p, a_symbol_string));
	      if (is_keyword(sym))
		sym = keyword_symbol(sym);
	      val = cdr(p);
	      break;

	    case T_LET:
	      append_let(sc, new_e, check_c_obj_env(sc, p, caller));
	      continue;

	    default:
	      return(wrong_type_argument_with_type(sc, caller, position_of(x, bindings), p, a_symbol_string));
	    }

	  if (is_constant_symbol(sc, sym))
	    return(wrong_type_argument_with_type(sc, caller, position_of(x, bindings), sym, a_non_constant_symbol_string));
	  if ((is_slot(global_slot(sym))) &&
	      (is_syntax(slot_value(global_slot(sym)))))
	    return(wrong_type_argument_with_type(sc, caller, 2, sym, wrap_string(sc, "a non-syntactic name", 20)));

	  /* here we know new_e is a let and is not rootlet */
	  make_slot_1(sc, new_e, sym, val);
	  if (sym == sc->let_ref_fallback_symbol)
	    set_has_let_ref_fallback(new_e);
	  else
	    {
	      if (sym == sc->let_set_fallback_symbol)
		set_has_let_set_fallback(new_e);
	    }
	}
      sc->temp3 = sc->nil;
    }
  return(new_e);
}

s7_pointer s7_sublet(s7_scheme *sc, s7_pointer e, s7_pointer bindings)
{
  return(sublet_1(sc, e, bindings, sc->sublet_symbol));
}

static s7_pointer g_sublet(s7_scheme *sc, s7_pointer args)
{
  #define H_sublet "(sublet env ...) adds its arguments (each a let or a cons: '(symbol . value)) to env, and returns the new environment."
  #define Q_sublet Q_varlet
  s7_pointer e;

  e = car(args);
  if (is_null(e))
    e = sc->rootlet;
  else
    {
      check_method(sc, e, sc->sublet_symbol, args);
      if (!is_let(e))
	return(wrong_type_argument_with_type(sc, sc->sublet_symbol, 1, e, a_let_string));
    }
  return(sublet_1(sc, e, cdr(args), sc->sublet_symbol));
}


/* -------------------------------- inlet -------------------------------- */
s7_pointer s7_inlet(s7_scheme *sc, s7_pointer args)
{
  #define H_inlet "(inlet ...) adds its \
arguments, each a let, a cons: '(symbol . value), or a keyword/value pair, to a new environment, and returns the \
new environment. (inlet :a 1 :b 2) or (inlet '(a . 1) '(b . 2))"
  #define Q_inlet s7_make_circular_signature(sc, 1, 2, sc->is_let_symbol, sc->T)

  return(sublet_1(sc, sc->rootlet, args, sc->inlet_symbol));
}

#define g_inlet s7_inlet

static s7_pointer g_simple_inlet(s7_scheme *sc, s7_pointer args)
{
  /* here all args are paired with normal symbol/value, no fallbacks, no immutable symbols etc */
  s7_pointer new_e, x;
  int64_t id;
  new_e = new_frame_in_env(sc, sc->nil);
  sc->temp3 = new_e;
  id = let_id(new_e);
  for (x = args; is_pair(x); x = cddr(x))
    {
      s7_pointer symbol, slot;
      symbol = car(x);
      if (is_keyword(symbol))                 /* (inlet ':allow-other-keys 3) */
	symbol = keyword_symbol(symbol);
      new_cell(sc, slot, T_SLOT);
      slot_set_symbol(slot, symbol);
      slot_set_value(slot, cadr(x));
      set_next_slot(slot, let_slots(new_e));
      let_set_slots(new_e, slot);
      set_local(symbol);
      symbol_set_local(symbol, id, slot);
    }
  sc->temp3 = sc->nil;
  return(new_e);
}

static s7_pointer inlet_p_pp(s7_scheme *sc, s7_pointer symbol, s7_pointer value)
{
  s7_pointer x, slot;
  if (!is_symbol(symbol))
    return(sublet_1(sc, sc->nil, list_2(sc, symbol, value), sc->inlet_symbol));
  new_cell(sc, x, T_LET | T_SAFE_PROCEDURE);
  sc->temp3 = x;
  let_id(x) = ++sc->let_number;
  set_outlet(x, sc->nil);
  if (is_keyword(symbol))
    symbol = keyword_symbol(symbol);
  new_cell(sc, slot, T_SLOT);
  slot_set_symbol(slot, symbol);
  slot_set_value(slot, value);
  set_next_slot(slot, sc->nil);
  let_set_slots(x, slot);
  set_local(symbol);
  symbol_set_local(symbol, let_id(x), slot);
  sc->temp3 = sc->nil;
  return(x);
}

static s7_pointer g_local_inlet(s7_scheme *sc, s7_int num_args, ...)
{
  va_list ap;
  s7_int i;
  s7_pointer new_e;
  int64_t id;

  new_e = new_frame_in_env(sc, sc->nil);
  sc->temp3 = new_e;
  id = let_id(new_e);

  va_start(ap, num_args);
  for (i = 0; i < num_args; i += 2)
    {
      s7_pointer symbol, slot, value;
      symbol = va_arg(ap, s7_pointer);
      value = va_arg(ap, s7_pointer);
      if (is_keyword(symbol))                 /* (inlet ':allow-other-keys 3) */
	symbol = keyword_symbol(symbol);
      new_cell(sc, slot, T_SLOT);
      slot_set_symbol(slot, symbol);
      slot_set_value(slot, value);
      set_next_slot(slot, let_slots(new_e));
      let_set_slots(new_e, slot);
      set_local(symbol);
      symbol_set_local(symbol, id, slot);
    }
  va_end(ap);

  sc->temp3 = sc->nil;
  return(new_e);
}

static bool is_proper_quote(s7_scheme *sc, s7_pointer p)
{
  return((is_pair(p)) &&
	 (car(p) == sc->quote_symbol) &&
	 (is_pair(cdr(p))) &&
	 (is_null(cddr(p))));
}

static s7_pointer inlet_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if (!ops) return(f);
  if ((args > 0) &&
      ((args % 2) == 0))
    {
      s7_pointer p;
      for (p = cdr(expr); is_pair(p); p = cddr(p))
	{
	  s7_pointer sym;
	  if (!is_proper_quote(sc, car(p)))             /* 'abs etc, but tricky: ':abs */
	    return(f);
	  sym = cadar(p);
	  if ((!is_symbol(sym)) ||
	      (is_possibly_constant(sym)) ||            /* define-constant etc */
	      (is_syntactic_symbol(sym))  ||            /* (inlet 'if 3) */
	      ((is_slot(global_slot(sym))) &&
	       (is_syntactic(slot_value(global_slot(sym))))) ||
	      (sym == sc->let_ref_fallback_symbol) ||
	      (sym == sc->let_set_fallback_symbol))
	    return(f);
	}
      return(sc->simple_inlet);
    }
  return(f);
}


/* -------------------------------- let->list -------------------------------- */
s7_pointer s7_let_to_list(s7_scheme *sc, s7_pointer env)
{
  s7_pointer x;

  sc->temp3 = sc->w;
  sc->w = sc->nil;

  if (env == sc->rootlet)
    {
      s7_int i, lim2;
      s7_pointer *entries;

      entries = rootlet_elements(env);
      lim2 = sc->rootlet_entries;
      if (lim2 & 1) lim2--;

      for (i = 0; i < lim2; )
	{
	  sc->w = cons_unchecked(sc, cons(sc, slot_symbol(entries[i]), slot_value(entries[i])), sc->w); i++;
	  sc->w = cons_unchecked(sc, cons_unchecked(sc, slot_symbol(entries[i]), slot_value(entries[i])), sc->w); i++;
	}
      if (lim2 < sc->rootlet_entries)
	sc->w = cons_unchecked(sc, cons(sc, slot_symbol(entries[i]), slot_value(entries[i])), sc->w);
    }
  else
    {
      s7_pointer iter, func;
      /* need to check make-iterator method before dropping into let->list */

      if ((has_methods(env)) && ((func = find_method(sc, env, sc->make_iterator_symbol)) != sc->undefined))
	iter = s7_apply_function(sc, func, list_1(sc, env));
      else iter = sc->nil;

      if (is_null(iter))
	{
	  for (x = let_slots(env); is_slot(x); x = next_slot(x))
	    sc->w = cons_unchecked(sc, cons(sc, slot_symbol(x), slot_value(x)), sc->w);
	}
      else
	{
	  /* (begin (load "mockery.scm") (let ((lt ((*mock-pair* 'mock-pair) 1 2 3))) (format *stderr* "~{~A ~}" lt))) */
	  while (true)
	    {
	      x = s7_iterate(sc, iter);
	      if (iterator_is_at_end(iter)) break;
	      sc->w = cons(sc, x, sc->w);
	    }
	  sc->w = safe_reverse_in_place(sc, sc->w);
	}
    }
  x = sc->w;
  sc->w = sc->temp3;
  sc->temp3 = sc->nil;
  return(x);
}

#if (!WITH_PURE_S7)
static s7_pointer g_let_to_list(s7_scheme *sc, s7_pointer args)
{
  #define H_let_to_list "(let->list env) returns env's bindings as a list of cons's: '(symbol . value)."
  #define Q_let_to_list s7_make_signature(sc, 2, sc->is_pair_symbol, sc->is_let_symbol)

  s7_pointer env;
  env = car(args);
  check_method(sc, env, sc->let_to_list_symbol, args);
  if (!is_let(env))
    {
      if (is_c_object(env))
	env = c_object_let(env);
      else
	{
	  if (is_c_pointer(env))
	    env = c_pointer_info(env);
	}
      if (!is_let(env))
        return(simple_wrong_type_argument_with_type(sc, sc->let_to_list_symbol, env, a_let_string));
    }
  return(s7_let_to_list(sc, env));
}
#endif


/* -------------------------------- let-ref -------------------------------- */

inline s7_pointer s7_let_ref(s7_scheme *sc, s7_pointer env, s7_pointer symbol)
{
  s7_pointer x, y;
  /* (let ((a 1)) ((curlet) 'a))
   * ((rootlet) 'abs)
   */
  if (!is_let(env))
    return(wrong_type_argument_with_type(sc, sc->let_ref_symbol, 1, env, a_let_string));

  if (!is_symbol(symbol))
    {
      if (has_let_ref_fallback(env))
	check_method_uncopied(sc, env, sc->let_ref_fallback_symbol, list_2(sc, env, symbol));
      return(wrong_type_argument_with_type(sc, sc->let_ref_symbol, 2, symbol, a_symbol_string));
    }

  check_method_uncopied(sc, env, sc->let_ref_symbol, list_2(sc, env, symbol));
  /* a let-ref method is almost impossible to write without creating an infinite loop:
   *   any reference to the let will probably call let-ref somewhere, calling us again, and looping.
   *   This is not a problem in c-objects and funclets because c-object-ref and funclet-ref don't
   *   exist -- perhaps let-ref should not also.
   */

  if (is_keyword(symbol))
    symbol = keyword_symbol(symbol);

  if (env == sc->rootlet)
    {
      y = global_slot(symbol);
      if (is_slot(y))
	return(slot_value(y));
      return(sc->undefined);
    }

  if (let_id(env) == symbol_id(symbol))
    return(slot_value(local_slot(symbol))); /* this obviously has to follow the global-env check */

  for (x = env; is_let(x); x = outlet(x))
    for (y = let_slots(x); is_slot(y); y = next_slot(y))
      if (slot_symbol(y) == symbol)
	return(slot_value(y));

  if (has_methods(env))
    {
      /* this is not a redundant check -- if has_methods, don't check global slot */

      /* If a let is a mock-hash-table (for example), implicit
       *   indexing of the hash-table collides with the same thing for the let (field names
       *   versus keys), and we can't just try again here because that makes it too easy to
       *   get into infinite recursion.  So, 'let-ref-fallback...
       */
      if (has_let_ref_fallback(env))
	apply_known_method(sc, env, sc->let_ref_fallback_symbol, set_qlist_2(sc, env, symbol));
    }
  else
    {
      y = global_slot(symbol);  /* (let () ((curlet) 'pi)) */
      if (is_slot(y))
	return(slot_value(y));
    }
  return(sc->undefined);
}

static s7_pointer g_let_ref(s7_scheme *sc, s7_pointer args)
{
  #define H_let_ref "(let-ref env sym) returns the value of the symbol sym in the environment env"
  #define Q_let_ref s7_make_signature(sc, 3, sc->T, sc->is_let_symbol, sc->is_symbol_symbol)
  return(s7_let_ref(sc, car(args), cadr(args)));
}

static s7_pointer slot_in_let(s7_scheme *sc, s7_pointer e, s7_pointer sym)
{
  s7_pointer y;
  for (y = let_slots(e); is_slot(y); y = next_slot(y))
    if (slot_symbol(y) == sym)
      return(y);
  return(sc->undefined);
}

static s7_pointer lint_let_ref_1(s7_scheme *sc, s7_pointer lt, s7_pointer sym)
{
  s7_pointer x, y;
  for (x = lt; is_let(x); x = outlet(x))
    for (y = let_slots(x); is_slot(y); y = next_slot(y))
      if (slot_symbol(y) == sym)
	return(slot_value(y));

  if (has_methods(lt))
    {
      if (has_let_ref_fallback(lt))
	apply_known_method(sc, lt, sc->let_ref_fallback_symbol, set_qlist_2(sc, lt, sym));
    }
  else
    {
      y = global_slot(sym);
      if (is_slot(y))
	return(slot_value(y));
    }
  return(sc->undefined);
}

static s7_pointer let_ref_p_pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2) {return(s7_let_ref(sc, p1, p2));}

static inline s7_pointer g_lint_let_ref(s7_scheme *sc, s7_pointer args)
{
  s7_pointer lt;
  lt = symbol_to_value_unchecked(sc, opt2_sym(args)); /* cadar */
  if (is_pair(lt))
    {
      lt = cdr(lt);
      if (is_let(lt))
	{
	  s7_pointer y, sym;
	  sym = opt3_sym(args); /* cadadr */
	  for (y = let_slots(lt); is_slot(y); y = next_slot(y))
	    if (slot_symbol(y) == sym)
	      return(slot_value(y));
	  return(lint_let_ref_1(sc, outlet(lt), sym));
	}
      return(wrong_type_argument_with_type(sc, sc->let_ref_symbol, 1, lt, a_let_string));
    }
  return(simple_wrong_type_argument(sc, sc->cdr_symbol, lt, T_PAIR));
}

static s7_pointer let_ref_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if ((!ops) || (!is_global(sc->let_ref_symbol))) return(f);
  if ((is_h_safe_c_d(expr)) && (raw_opt1(expr) == sc->lint_let_ref))
    return(raw_opt1(expr));

  if (optimize_op(expr) == HOP_SAFE_C_opSq_C)
    {
      s7_pointer arg1, arg2;
      arg1 = cadr(expr);
      arg2 = caddr(expr);
      if ((car(arg1) == sc->cdr_symbol) &&
	  ((is_keyword(arg2)) ||
	   ((is_pair(arg2)) &&
	    (car(arg2) == sc->quote_symbol) &&
	    (is_symbol(cadr(arg2))) &&
	    (!is_possibly_constant(cadr(arg2))))))
	{
	  set_optimize_op(expr, HOP_SAFE_C_D);
	  set_opt2_sym(cdr(expr), cadr(arg1));
	  set_opt3_sym(cdr(expr), (is_keyword(arg2)) ? arg2 : cadr(arg2));
	  return(sc->lint_let_ref);
	}
    }
  return(f);
}

static bool op_environment_c(s7_scheme *sc)
{
  s7_pointer s;
  s = symbol_to_value_checked(sc, car(sc->code));
  if (!is_let(s)) {sc->last_function = s; return(false);}
  sc->value = s7_let_ref(sc, T_Pos(s), opt3_any(sc->code));
  return(true);
}

static bool op_environment_a(s7_scheme *sc)
{
  s7_pointer s;
  s = symbol_to_value_checked(sc, car(sc->code));
  if (!is_let(s)) {sc->last_function = s; return(false);}
  sc->value = s7_let_ref(sc, s, c_call(cdr(sc->code))(sc, cadr(sc->code)));
  return(true);
}


/* -------------------------------- let-set! -------------------------------- */
static s7_pointer call_setter(s7_scheme *sc, s7_pointer slot, s7_pointer old_value);

static s7_pointer let_set_1(s7_scheme *sc, s7_pointer env, s7_pointer symbol, s7_pointer value)
{
  s7_pointer x, y;

  if (is_keyword(symbol))
    symbol = keyword_symbol(symbol);

  if (env == sc->rootlet)
    {
      if (is_constant_symbol(sc, symbol))  /* (let-set! (rootlet) 'pi #f) */
	return(wrong_type_argument_with_type(sc, sc->let_set_symbol, 2, symbol, a_non_constant_symbol_string));

      y = global_slot(symbol);
      if (is_slot(y))
	{
	  if (is_syntax(slot_value(y)))
	    return(wrong_type_argument_with_type(sc, sc->let_set_symbol, 2, symbol, wrap_string(sc, "a non-syntactic keyword", 23)));

	  if (slot_has_setter(y))
	    slot_set_value(y, call_setter(sc, y, value));
	  else slot_set_value(y, value);
	  return(slot_value(y));
	}
      return(s7_error(sc, sc->wrong_type_arg_symbol, set_elist_3(sc, wrap_string(sc, "let-set!: ~A is not defined in ~A", 33), symbol, env)));
    }

  if (let_id(env) == symbol_id(symbol))
   {
     y = local_slot(symbol);
     if (is_slot(y))
       {
	 if (slot_has_setter(y))
	   slot_set_value(y, call_setter(sc, y, value));
	 else slot_set_value(y, value);
	 return(slot_value(y));
       }
   }

  for (x = env; is_let(x); x = outlet(x))
    for (y = let_slots(x); is_slot(y); y = next_slot(y))
      if (slot_symbol(y) == symbol)
	{
	  if (slot_has_setter(y))
	    slot_set_value(y, call_setter(sc, y, value));
	  else slot_set_value(y, value);
	  return(slot_value(y));
	}

  if (has_methods(env))
    {
      if (has_let_set_fallback(env))
	apply_known_method(sc, env, sc->let_set_fallback_symbol, sc->w = list_3(sc, env, symbol, value));
    }

  return(s7_error(sc, sc->wrong_type_arg_symbol, set_elist_3(sc, wrap_string(sc, "let-set!: ~A is not defined in ~A", 33), symbol, env)));
  /* not sure about this -- what's the most useful choice? */
}

s7_pointer s7_let_set(s7_scheme *sc, s7_pointer env, s7_pointer symbol, s7_pointer value)
{
  if (!is_let(env))
    return(wrong_type_argument_with_type(sc, sc->let_set_symbol, 1, env, a_let_string));

  if (is_immutable(env)) /* opt_p_ppp_fff can't check at opt time for immutable let, and it calls us */
    return(immutable_object_error(sc, set_elist_3(sc, immutable_error_string, sc->let_set_symbol, env)));

  if (!is_symbol(symbol))
    {
      if (has_let_set_fallback(env))
	apply_known_method(sc, env, sc->let_set_fallback_symbol, sc->w = list_3(sc, env, symbol, value));
      return(wrong_type_argument_with_type(sc, sc->let_set_symbol, 2, symbol, a_symbol_string));
    }

  check_method_uncopied(sc, env, sc->let_set_symbol, sc->w = list_3(sc, env, symbol, value));
  return(let_set_1(sc, env, symbol, value));
}

static s7_pointer g_let_set(s7_scheme *sc, s7_pointer args)
{
  /* (let ((a 1)) (set! ((curlet) 'a) 32) a) */
  #define H_let_set "(let-set! env sym val) sets the symbol sym's value in the environment env to val"
  #define Q_let_set s7_make_signature(sc, 4, sc->T, sc->is_let_symbol, sc->is_symbol_symbol, sc->T)

  return(s7_let_set(sc, car(args), cadr(args), caddr(args)));
}

static s7_pointer let_set_p_ppp(s7_scheme *sc, s7_pointer p1, s7_pointer p2, s7_pointer p3)
{
  return(s7_let_set(sc, p1, p2, p3));
}

static s7_pointer let_set_p_ppp_1(s7_scheme *sc, s7_pointer p1, s7_pointer p2, s7_pointer p3)
{
  return(let_set_1(sc, p1, p2, p3));
}

static s7_pointer let_set_p_ppp_2(s7_scheme *sc, s7_pointer p1, s7_pointer p2, s7_pointer p3)
{
  if (!is_symbol(p2))
    return(wrong_type_argument_with_type(sc, sc->let_set_symbol, 2, p2, a_symbol_string));
  return(let_set_1(sc, p1, p2, p3));
}

static s7_pointer g_lint_let_set_1(s7_scheme *sc, s7_pointer lt1, s7_pointer sym, s7_pointer val)
{
  s7_pointer lt, x, y;

  lt = (is_pair(lt1)) ? cdr(lt1) : g_cdr(sc, set_plist_1(sc, lt1));
  if (!is_let(lt))
    return(wrong_type_argument_with_type(sc, sc->let_set_symbol, 1, lt, a_let_string));
  if (is_immutable(lt))
    return(immutable_object_error(sc, set_elist_3(sc, immutable_error_string, sc->let_set_symbol, lt)));

  if (lt == sc->rootlet)
    {
      y = global_slot(sym);
      if (is_slot(y))
	{
	  if (slot_has_setter(y))
	    slot_set_value(y, call_setter(sc, y, val));
	  else slot_set_value(y, val);
	  return(slot_value(y));
	}
      return(s7_error(sc, sc->wrong_type_arg_symbol, set_elist_3(sc, wrap_string(sc, "let-set!: ~A is not defined in ~A", 33), sym, lt)));
    }

  for (x = lt; is_let(x); x = outlet(x))
    for (y = let_slots(x); is_slot(y); y = next_slot(y))
      if (slot_symbol(y) == sym)
	{
	  if (slot_has_setter(y))
	    slot_set_value(y, call_setter(sc, y, val));
	  else slot_set_value(y, val);
	  return(slot_value(y));
	}

  if (has_methods(lt))
    {
      if (has_let_set_fallback(lt))
	apply_known_method(sc, lt, sc->let_set_fallback_symbol, sc->w = list_3(sc, lt, sym, val));
    }
  else
    {
      y = global_slot(sym);
      if (is_slot(y))
	{
	  if (slot_has_setter(y))
	    slot_set_value(y, call_setter(sc, y, val));
	  else slot_set_value(y, val);
	  return(slot_value(y));
	}
    }
  return(s7_error(sc, sc->wrong_type_arg_symbol, set_elist_3(sc, wrap_string(sc, "let-set!: ~A is not defined in ~A", 33), sym, lt)));
}

static s7_pointer g_lint_let_set(s7_scheme *sc, s7_pointer args)
{
  return(g_lint_let_set_1(sc, symbol_to_value_checked(sc, opt3_sym(args)), opt1_con(args), symbol_to_value_unchecked(sc, opt2_sym(args))));
}

static s7_pointer let_set_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if ((!ops) || (!is_global(sc->let_set_symbol))) return(f);
  if ((is_h_safe_c_d(expr)) && (raw_opt1(expr) == sc->lint_let_set))
    return(raw_opt1(expr));

  if (optimize_op(expr) == HOP_SAFE_C_opSq_CS)
    {
      s7_pointer arg1, arg2, arg3;
      arg1 = cadr(expr);
      arg2 = caddr(expr);
      arg3 = cadddr(expr);
      if ((car(arg1) == sc->cdr_symbol) &&
	  (car(arg2) == sc->quote_symbol) &&
	  (is_symbol(cadr(arg2))) &&
	  (!is_possibly_constant(cadr(arg2))) &&
	  (is_symbol(arg3)) &&
	  (!is_possibly_constant(arg3)))
	{
	  set_optimize_op(expr, HOP_SAFE_C_D);
	  return(sc->lint_let_set);
	}
    }
  return(f);
}


static s7_pointer reverse_slots(s7_scheme *sc, s7_pointer list)
{
  s7_pointer p = list, result, q;
  result = sc->nil;

  while (is_slot(p))
    {
      q = next_slot(p);
      set_next_slot(p, result);
      result = p;
      p = q;
    }
  return(result);
}


static s7_pointer let_copy(s7_scheme *sc, s7_pointer env)
{
  if (is_let(env))
    {
      s7_pointer new_e;

      if (env == sc->rootlet)   /* (copy (rootlet)) or (copy (funclet abs)) etc */
	return(sc->rootlet);

      /* we can't make copy handle environments-as-objects specially because the
       *   make-object function in define-class uses copy to make a new object!
       *   So if it is present, we get it here, and then there's almost surely trouble.
       */
      new_e = new_frame_in_env(sc, outlet(env));
      set_all_methods(new_e, env);
      sc->temp3 = new_e;
      if (is_slot(let_slots(env)))
	{
	  s7_int id;
	  s7_pointer x, y = NULL;

	  id = let_id(new_e);
	  for (x = let_slots(env); is_slot(x); x = next_slot(x))
	    {
	      s7_pointer z;
	      new_cell(sc, z, T_SLOT);
	      slot_set_symbol(z, slot_symbol(x));
	      slot_set_value(z, slot_value(x));
	      if (symbol_id(slot_symbol(z)) != id) /* keep shadowing intact */
		symbol_set_local(slot_symbol(x), id, z);
	      if (is_slot(let_slots(new_e)))
		set_next_slot(y, z);
	      else let_set_slots(new_e, z);
	      set_next_slot(z, sc->nil);              /* in case GC runs during this loop */
	      y = z;
	    }
	}
      /* We can't do a (normal) loop here then reverse the slots later because the symbol's local_slot has to
       *    match the unshadowed slot, not the last in the list:
       *    (let ((e1 (inlet 'a 1 'a 2))) (let ((e2 (copy e1))) (list (equal? e1 e2) (equal? (e1 'a) (e2 'a)))))
       */
      sc->temp3 = sc->nil;
      return(new_e);
    }
  return(sc->nil);
}


/* -------------------------------- rootlet -------------------------------- */
static s7_pointer g_rootlet(s7_scheme *sc, s7_pointer ignore)
{
  #define H_rootlet "(rootlet) returns the current top-level definitions (symbol bindings)."
  #define Q_rootlet s7_make_signature(sc, 1, sc->is_let_symbol)
  return(sc->rootlet);
}
/* as with the symbol-table, this function can lead to disaster -- user could
 *   clobber the environment etc.  But we want it to be editable and augmentable,
 *   so I guess I'll leave it alone.  (See curlet|funclet as well).
 */

s7_pointer s7_rootlet(s7_scheme *sc)
{
  return(sc->rootlet);
}

s7_pointer s7_shadow_rootlet(s7_scheme *sc)
{
  return(sc->shadow_rootlet);
}

s7_pointer s7_set_shadow_rootlet(s7_scheme *sc, s7_pointer let)
{
  sc->shadow_rootlet = let;
  return(let);
}


/* -------------------------------- curlet -------------------------------- */
static s7_pointer g_curlet(s7_scheme *sc, s7_pointer args)
{
  #define H_curlet "(curlet) returns the current definitions (symbol bindings)"
  #define Q_curlet s7_make_signature(sc, 1, sc->is_let_symbol)

  sc->capture_let_counter++;
  if (is_let(sc->envir))
    return(sc->envir);
  return(sc->rootlet);
}

s7_pointer s7_curlet(s7_scheme *sc)
{
  sc->capture_let_counter++;
  return(sc->envir);
}

s7_pointer s7_set_curlet(s7_scheme *sc, s7_pointer e)
{
  s7_pointer p, old_e;
  old_e = sc->envir;
  sc->envir = e;

  if ((is_let(e)) && (let_id(e) > 0)) /* might be () [id=-1] or rootlet [id=0] etc */
    {
      let_id(e) = ++sc->let_number;
      for (p = let_slots(e); is_slot(p); p = next_slot(p))
	{
	  s7_pointer sym;
	  sym = slot_symbol(p);
	  if (symbol_id(sym) != sc->let_number)
	    symbol_set_local(sym, sc->let_number, p);
	}
    }

  return(old_e);
}


/* -------------------------------- outlet -------------------------------- */
s7_pointer s7_outlet(s7_scheme *sc, s7_pointer e)
{
  return(outlet(e));
}

static s7_pointer g_outlet(s7_scheme *sc, s7_pointer args)
{
  #define H_outlet "(outlet env) is the environment that contains env."
  #define Q_outlet s7_make_signature(sc, 2, sc->is_let_symbol, sc->is_let_symbol)

  s7_pointer env;
  env = car(args);
  if (!is_let(env))
    return(method_or_bust_with_type_one_arg(sc, env, sc->outlet_symbol, args, a_let_string));

  if ((env == sc->rootlet) ||
      (is_null(outlet(env))))
    return(sc->rootlet);
  return(outlet(env));
}


static s7_pointer g_set_outlet(s7_scheme *sc, s7_pointer args)
{
  /* (let ((a 1)) (let ((b 2)) (set! (outlet (curlet)) (rootlet)) ((curlet) 'a))) */
  s7_pointer env, new_outer;

  env = car(args);
  if (!is_let(env))
    return(s7_wrong_type_arg_error(sc, "set! outlet", 1, env, "a let"));
  if (is_immutable(env))
    return(s7_wrong_type_arg_error(sc, "set! outlet", 1, env, "a mutable let"));

  new_outer = cadr(args);
  if (!is_let(new_outer))
    return(s7_wrong_type_arg_error(sc, "set! outlet", 2, new_outer, "a let"));

  if (env != sc->rootlet)
    set_outlet(env, (new_outer == sc->rootlet) ? sc->nil : new_outer);
  return(new_outer);
}

static inline s7_pointer symbol_to_slot(s7_scheme *sc, s7_pointer symbol)
{
  s7_pointer x;
  if (let_id(sc->envir) == symbol_id(symbol))
    return(local_slot(symbol));
  for (x = sc->envir; symbol_id(symbol) < let_id(x); x = outlet(x));
  if (let_id(x) == symbol_id(symbol))
    return(local_slot(symbol));
  for (; is_let(x); x = outlet(x))
    {
      s7_pointer y;
      for (y = let_slots(x); is_slot(y); y = next_slot(y))
	if (slot_symbol(y) == symbol)
	  return(y);
    }
  return(global_slot(symbol));
}

#if WITH_GCC && S7_DEBUGGING
static s7_pointer symbol_to_value_unchecked_1(s7_scheme *sc, s7_pointer symbol)
#else
static inline s7_pointer symbol_to_value_unchecked(s7_scheme *sc, s7_pointer symbol) /* symbol_to_value_checked includes the unbound_variable call */
#endif
{
  s7_pointer x;
  if (let_id(sc->envir) == symbol_id(symbol))
    return(slot_value(local_slot(symbol)));
  for (x = sc->envir; symbol_id(symbol) < let_id(x); x = outlet(x));
  if (let_id(x) == symbol_id(symbol))
    return(slot_value(local_slot(symbol)));
  for (; is_let(x); x = outlet(x))
    {
      s7_pointer y;
      for (y = let_slots(x); is_slot(y); y = next_slot(y))
	if (slot_symbol(y) == symbol)
	  return(slot_value(y));
    }
  x = global_slot(symbol);
  if (is_slot(x)) return(slot_value(x));
#if WITH_GCC
  return(NULL); /* much faster than various alternatives */
#else
  return(unbound_variable(sc, symbol));
#endif
}


s7_pointer s7_slot(s7_scheme *sc, s7_pointer symbol)
{
  return(symbol_to_slot(sc, symbol));
}

s7_pointer s7_slot_value(s7_pointer slot)
{
  return(slot_value(slot));
}

s7_pointer s7_slot_set_value(s7_scheme *sc, s7_pointer slot, s7_pointer value)
{
  slot_set_value(slot, value);
  return(value);
}

void s7_slot_set_real_value(s7_scheme *sc, s7_pointer slot, s7_double value)
{
  set_real(slot_value(slot), value);
}

static s7_pointer symbol_to_local_slot(s7_scheme *sc, s7_pointer symbol, s7_pointer e)
{
  if (!is_let(e))
    return(global_slot(symbol));

  if (symbol_id(symbol) != 0)
    {
      s7_pointer y;
      for (y = let_slots(e); is_slot(y); y = next_slot(y))
	if (slot_symbol(y) == symbol)
	  return(y);
    }
  return(sc->undefined);
}

s7_pointer s7_symbol_value(s7_scheme *sc, s7_pointer sym)
{
  s7_pointer x;
  x = symbol_to_slot(sc, sym);
  if (is_slot(x))
    return(slot_value(x));
  return(sc->undefined);
}

s7_pointer s7_symbol_local_value(s7_scheme *sc, s7_pointer sym, s7_pointer local_env)
{
  /* restrict the search to local_env outward */
  if ((local_env == sc->rootlet) || (is_global(sym)))
    {
      if (is_slot(global_slot(sym)))
	return(slot_value(global_slot(sym)));
      return(sc->undefined);
    }

  if (is_let(local_env))
    {
      s7_pointer x;

      if (let_id(local_env) == symbol_id(sym))
	return(slot_value(local_slot(sym)));
      for (x = local_env; symbol_id(sym) < let_id(x); x = outlet(x));
      if (let_id(x) == symbol_id(sym))
	return(slot_value(local_slot(sym)));

      for (; is_let(x); x = outlet(x))
	{
	  s7_pointer y;
	  for (y = let_slots(x); is_slot(y); y = next_slot(y))
	    if (slot_symbol(y) == sym)
	      return(slot_value(y));
	}
      /* need to check rootlet before giving up */
      if (is_slot(global_slot(sym)))
	return(slot_value(global_slot(sym)));

      /* (let ((e (curlet))) (let ((a 1)) (symbol->value 'a e))) -> #<undefined> not 1 */
      return(sc->undefined); /* 29-Nov-17 */
    }
  return(s7_symbol_value(sc, sym));
}


/* -------------------------------- symbol->value -------------------------------- */

#define find_global_symbol_checked(Sc, Sym) ((is_global(Sym)) ? slot_value(global_slot(Sym)) : symbol_to_value_checked(Sc, Sym))

static s7_pointer g_s7_let_ref_fallback(s7_scheme *sc, s7_pointer args);

static s7_pointer g_symbol_to_value(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_value "(symbol->value sym (env (curlet))) returns the binding of (the value associated with) the \
symbol sym in the given environment: (let ((x 32)) (symbol->value 'x)) -> 32"
  #define Q_symbol_to_value s7_make_signature(sc, 3, sc->T, sc->is_symbol_symbol, sc->is_let_symbol)
  /* (symbol->value 'x e) => (e 'x)? */

  s7_pointer sym;
  sym = car(args);

  if (!is_symbol(sym))
    return(method_or_bust(sc, sym, sc->symbol_to_value_symbol, args, T_SYMBOL, 1));

  if (is_not_null(cdr(args)))
    {
      s7_pointer local_env;

      local_env = cadr(args);
      if (local_env == sc->unlet_symbol)
	return((is_slot(initial_slot(sym))) ? slot_value(initial_slot(sym)) : sc->undefined);

      if (!is_let(local_env))
	return(method_or_bust_with_type(sc, local_env, sc->symbol_to_value_symbol, args, a_let_string, 2));

      if (local_env == sc->s7_let)
	return(g_s7_let_ref_fallback(sc, set_qlist_2(sc, local_env, sym)));

      return(s7_symbol_local_value(sc, sym, local_env));
    }

  if (is_global(sym))
    return(slot_value(global_slot(sym)));

  return(s7_symbol_value(sc, sym));
}


s7_pointer s7_symbol_set_value(s7_scheme *sc, s7_pointer sym, s7_pointer val)
{
  s7_pointer x;
  /* if immutable should this return an error? */
  x = symbol_to_slot(sc, sym);
  if (is_slot(x))
    slot_set_value(x, val); /* with_hook? */
  return(val);
}


/* -------------------------------- symbol->dynamic-value -------------------------------- */

static s7_pointer find_dynamic_value(s7_scheme *sc, s7_pointer x, s7_pointer sym, int64_t *id)
{
  for (; symbol_id(sym) < let_id(x); x = outlet(x));

  if (let_id(x) == symbol_id(sym))
    {
      (*id) = let_id(x);
      return(slot_value(local_slot(sym)));
    }
  for (; (is_let(x)) && (let_id(x) > (*id)); x = outlet(x))
    {
      s7_pointer y;
      for (y = let_slots(x); is_slot(y); y = next_slot(y))
	if (slot_symbol(y) == sym)
	  {
	    (*id) = let_id(x);
	    return(slot_value(y));
	  }
    }
  return(sc->gc_nil);
}

static s7_pointer g_symbol_to_dynamic_value(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_dynamic_value "(symbol->dynamic-value sym) returns the dynamic binding of the symbol sym"
  #define Q_symbol_to_dynamic_value s7_make_signature(sc, 2, sc->T, sc->is_symbol_symbol)

  s7_pointer sym, val;
  int64_t i, top_id;

  sym = car(args);
  if (!is_symbol(sym))
    return(method_or_bust(sc, sym, sc->symbol_to_dynamic_value_symbol, args, T_SYMBOL, 1));

  if (is_global(sym))
    return(slot_value(global_slot(sym)));

  if (let_id(sc->envir) == symbol_id(sym))
    return(slot_value(local_slot(sym)));

  top_id = -1;
  val = find_dynamic_value(sc, sc->envir, sym, &top_id);
  if (top_id == symbol_id(sym))
    return(val);

  for (i = s7_stack_top(sc) - 1; i > 0; i -= 4)
    {
      s7_pointer cur_val;
      cur_val = find_dynamic_value(sc, stack_let(sc->stack, i), sym, &top_id);
      if (cur_val != sc->gc_nil)
	val = cur_val;
      if (top_id == symbol_id(sym))
	return(val);
    }

  if (val == sc->gc_nil)
    return(s7_symbol_value(sc, sym));
  return(val);
}


typedef bool (safe_sym_t)(s7_scheme *sc, s7_pointer sym, s7_pointer e);

static bool direct_memq(s7_pointer symbol, s7_pointer symbols)
{
  s7_pointer x;
  for (x = symbols; is_pair(x); x = cdr(x))
    if (car(x) == symbol)
	return(true);
  return(false);
}

static bool direct_assq(s7_pointer symbol, s7_pointer symbols)
{ /* used only below in do_symbol_is_safe */
  s7_pointer x;
  for (x = symbols; is_pair(x); x = cdr(x))
    if (caar(x) == symbol)
      return(true);
  return(false);
}

static bool do_symbol_is_safe(s7_scheme *sc, s7_pointer sym, s7_pointer e)
{
  return((is_slot(global_slot(sym))) ||
	 (direct_assq(sym, e)) ||
	 (is_slot(symbol_to_slot(sc, sym))));
}

static bool let_symbol_is_safe(s7_scheme *sc, s7_pointer sym, s7_pointer e)
{
  if (is_slot(global_slot(sym)))
    return(true);
  if (is_null(e))
    e = sc->rootlet;
  return((!is_with_let_let(e)) &&
	 (is_slot(symbol_to_slot(sc, sym))));
}

static bool let_star_symbol_is_safe(s7_scheme *sc, s7_pointer sym, s7_pointer e)
{
  return((symbol_is_in_list(sc, sym)) ||
	 (is_slot(global_slot(sym))) ||
	 ((is_let(e)) && (!is_with_let_let(e)) && (is_slot(symbol_to_slot(sc, sym)))));
}

static bool pair_symbol_is_safe(s7_scheme *sc, s7_pointer sym, s7_pointer e)
{
  return((is_slot(global_slot(sym))) ||
	 (direct_memq(sym, e)));
}


static inline s7_pointer collect_variables(s7_scheme *sc, s7_pointer lst, s7_pointer e)
{
  /* collect local variable names from let/do (pre-error-check) */
  s7_pointer p;
  sc->w = e;
  for (p = lst; is_pair(p); p = cdr(p))
    sc->w = cons(sc, add_symbol_to_list(sc, caar(p)), sc->w);
  return(sc->w);
}

static s7_pointer collect_parameters(s7_scheme *sc, s7_pointer lst, s7_pointer e)
{
  /* collect local variable names from lambda arglists (pre-error-check) */
  s7_pointer p;
  if (is_symbol(lst))
    return(cons(sc, add_symbol_to_list(sc, lst), e));
  sc->w = e;
  for (p = lst; is_pair(p); p = cdr(p))
    {
      s7_pointer car_p;
      car_p = car(p);
      if (is_pair(car_p))
	car_p = car(car_p);
      if ((is_symbol(car_p)) &&
	  (!is_keyword(car_p)))
	sc->w = cons(sc, add_symbol_to_list(sc, car_p), sc->w);
    }
  if (is_symbol(p)) /* rest arg */
    sc->w = cons(sc, add_symbol_to_list(sc, p), sc->w);
  return(sc->w);
}


typedef enum {OPT_F, OPT_T, OPT_OOPS} opt_t;
static opt_t optimize(s7_scheme *sc, s7_pointer code, int32_t hop, s7_pointer e);

static void clear_all_optimizations(s7_scheme *sc, s7_pointer p)
{
  /* I believe that we would not have been optimized to begin with if the tree were circular,
   *   and this tree is supposed to be a function call + args -- a circular list here is a bug.
   */
  if (is_pair(p))
    {
      if ((is_optimized(p)) &&
	  ((optimize_op(p) & 1) == 0)) /* protect possibly shared code?  Elsewhere we assume these aren't changed */
	{
	  clear_optimized(p);     /* includes T_SYNTACTIC */
	  clear_optimize_op(p);
	}
      clear_all_optimizations(sc, cdr(p));
      clear_all_optimizations(sc, car(p));
    }
}


static s7_pointer make_macro(s7_scheme *sc, opcode_t op)
{
  s7_pointer cx, mac;
  uint64_t typ;

  if (op == OP_DEFINE_MACRO)
    typ = T_MACRO | T_DONT_EVAL_ARGS | T_COPY_ARGS;
  else
    {
      if (op == OP_DEFINE_MACRO_STAR)
	typ = T_MACRO_STAR | T_DONT_EVAL_ARGS | T_COPY_ARGS;
      else
	{
	  if (op == OP_DEFINE_BACRO)
	    typ = T_BACRO | T_DONT_EVAL_ARGS | T_COPY_ARGS;
	  else
	    {
	      if (op == OP_DEFINE_BACRO_STAR)
		typ = T_BACRO_STAR | T_DONT_EVAL_ARGS | T_COPY_ARGS;
	      else
		{
		  if ((op == OP_DEFINE_EXPANSION) &&
		      (!is_let(sc->envir)))        /* local expansions are just normal macros */
		    typ = T_MACRO | T_EXPANSION | T_DONT_EVAL_ARGS | T_COPY_ARGS;
		  else typ = T_MACRO | T_DONT_EVAL_ARGS | T_COPY_ARGS;
		}
	    }
	}
    }

  new_cell_no_check(sc, mac, typ);
  sc->temp6 = mac;
  closure_set_args(mac, cdar(sc->code));
  closure_set_body(mac, cdr(sc->code));
  closure_set_setter(mac, sc->F);
  closure_set_let(mac, sc->envir);
  closure_arity(mac) = CLOSURE_ARITY_NOT_SET;

  sc->capture_let_counter++;
  sc->code = caar(sc->code);
  if ((op == OP_DEFINE_EXPANSION) &&
      (!is_let(sc->envir)))
    set_type(sc->code, T_EXPANSION | T_SYMBOL | (typeflag(sc->code) & T_UNHEAP)); /* see comment under READ_TOK */
  /* symbol? macro name has already been checked, find name in environment, and define it */
  cx = symbol_to_local_slot(sc, sc->code, sc->envir);
  if (is_slot(cx))
    slot_set_value_with_hook(cx, mac);
  else s7_make_slot(sc, sc->envir, sc->code, mac); /* was current but we've checked immutable already */

  clear_symbol_list(sc); /* tracks names local to this macro */
  if (optimize(sc, closure_body(mac), 1, collect_parameters(sc, closure_args(mac), sc->nil)) == OPT_OOPS)
    clear_all_optimizations(sc, closure_body(mac));

  sc->temp6 = sc->nil;
  return(mac);
}


static s7_pointer make_closure(s7_scheme *sc, s7_pointer args, s7_pointer code, uint64_t type, int32_t arity)
{
  /* this is called every time a lambda form is evaluated, or during letrec, etc */
  s7_pointer x;
  new_cell(sc, x, (type | closure_bits(code)));
  closure_set_args(x, args);
  closure_set_body(x, code);
  if (is_pair(cdr(code))) set_closure_has_multiform(x); else set_closure_has_one_form(x);
  closure_set_let(x, sc->envir);
  closure_set_setter(x, sc->F);
  closure_arity(x) = arity;
  sc->capture_let_counter++;
  return(x);
}

#define make_closure_with_let(Sc, X, Args, Code, Env, Arity)	\
  do {							\
    new_cell(Sc, X, T_CLOSURE | T_COPY_ARGS | closure_bits(Code));	\
    closure_set_args(X, Args);						\
    closure_set_body(X, Code);				                \
    if (is_pair(cdr(Code))) set_closure_has_multiform(X); else set_closure_has_one_form(X);\
    closure_set_let(X, Env);						\
    closure_set_setter(X, sc->F);					\
    closure_arity(X) = Arity;						\
    sc->capture_let_counter++;						\
  } while (0)

static int32_t closure_length(s7_scheme *sc, s7_pointer e)
{
  /* we can't use let_length(sc, closure_let(e)) because the closure_let(closure)
   *   changes.  So the open bit is not always on.  Besides, the fallbacks need to be for closures, not environments.
   */
  s7_pointer length_func;
  length_func = find_method(sc, closure_let(e), sc->length_symbol);
  if (length_func != sc->undefined)
    return((int32_t)s7_integer(s7_apply_function(sc, length_func, list_1(sc, e))));

  /* there are cases where this should raise a wrong-type-arg error, but for now... */
  return(-1);
}

#define check_closure_for(Sc, Fnc, Sym)				    \
  if ((has_closure_let(Fnc)) && (is_let(closure_let(Fnc))))	    \
    {								    \
      s7_pointer val;						    \
      val = symbol_to_local_slot(Sc, Sym, closure_let(Fnc));	    \
      if ((!is_slot(val)) && (is_let(outlet(closure_let(Fnc)))))    \
	val = symbol_to_local_slot(Sc, Sym, outlet(closure_let(Fnc))); \
      if (is_slot(val))						    \
	return(slot_value(val));				    \
    }


static s7_pointer copy_tree_with_type(s7_scheme *sc, s7_pointer tree)
{
  /* if sc->safety > NO_SAFETY, '(1 2) is set immutable by the reader, but eval (in that safety case) calls
   *   copy_body on the incoming tree, so we have to preserve T_IMMUTABLE in that case.
   */
#if WITH_GCC
  #define COPY_TREE_WITH_TYPE(P) ({s7_pointer _p; _p = P; \
                                   cons_unchecked_with_type(sc, _p, (is_pair(car(_p))) ? copy_tree_with_type(sc, car(_p)) : car(_p), \
                                                                    (is_pair(cdr(_p))) ? copy_tree_with_type(sc, cdr(_p)) : cdr(_p));})
#else
  #define COPY_TREE_WITH_TYPE(P) copy_tree_with_type(sc, P)
#endif

  return(cons_unchecked_with_type(sc, tree,
				  (is_pair(car(tree))) ? COPY_TREE_WITH_TYPE(car(tree)) : car(tree),
				  (is_pair(cdr(tree))) ? COPY_TREE_WITH_TYPE(cdr(tree)) : cdr(tree)));
}

static s7_pointer copy_tree(s7_scheme *sc, s7_pointer tree)
{
#if WITH_GCC
  #define COPY_TREE(P) ({s7_pointer _p; _p = P; \
                         cons_unchecked(sc, (is_pair(car(_p))) ? copy_tree(sc, car(_p)) : car(_p), \
                                            (is_pair(cdr(_p))) ? copy_tree(sc, cdr(_p)) : cdr(_p));})
#else
  #define COPY_TREE(P) copy_tree(sc, P)
#endif

  return(cons_unchecked(sc,
			(is_pair(car(tree))) ? COPY_TREE(car(tree)) : car(tree),
			(is_pair(cdr(tree))) ? COPY_TREE(cdr(tree)) : cdr(tree)));
}


static inline bool tree_is_cyclic_1(s7_scheme *sc, s7_pointer tree)
{
  s7_pointer p;
  if (car(tree) == sc->quote_symbol) return(false); /* not quite correct (given apply) */
  for (p = tree; is_pair(p); p = cdr(p))
    {
      if (is_tree_collected_or_shared(p))
	return(!is_shared(p));
      set_tree_collected(p);

      if (sc->tree_pointers_top == sc->tree_pointers_size)
	{
	  if (sc->tree_pointers_size == 0)
	    {
	      sc->tree_pointers_size = 8;
	      sc->tree_pointers = (s7_pointer *)malloc(sc->tree_pointers_size * sizeof(s7_pointer));
	    }
	  else
	    {
	      sc->tree_pointers_size *= 2;
	      sc->tree_pointers = (s7_pointer *)realloc(sc->tree_pointers, sc->tree_pointers_size * sizeof(s7_pointer));
	    }
	}
      sc->tree_pointers[sc->tree_pointers_top++] = p;

      if ((is_pair(car(p))) &&
	  (tree_is_cyclic_1(sc, car(p))))
	return(true);
    }
  set_shared(tree);
  return(false);
}

static bool tree_is_cyclic(s7_scheme *sc, s7_pointer tree)
{
  if (is_pair(tree))
    {
      bool result;
      int32_t i;
      result = tree_is_cyclic_1(sc, tree);
      for (i = 0; i < sc->tree_pointers_top; i++)
	clear_tree_bits(sc->tree_pointers[i]);
      sc->tree_pointers_top = 0;
      return(result);
    }
  return(false);
}

static s7_pointer g_tree_is_cyclic(s7_scheme *sc, s7_pointer args)
{
  #define H_tree_is_cyclic "(tree-cyclic? tree) returns #t if the tree has a cycle."
  #define Q_tree_is_cyclic sc->pl_bt
  return(make_boolean(sc, tree_is_cyclic(sc, car(args))));
}

static s7_pointer copy_body(s7_scheme *sc, s7_pointer p)
{
  if ((sc->safety > NO_SAFETY) &&
      (tree_is_cyclic(sc, p)))
    s7_error(sc, sc->wrong_type_arg_symbol, wrap_string(sc, "copy: tree is cyclic", 20));
  if (8192 >= (sc->free_heap_top - sc->free_heap))
    {
      sc->w = p;
      gc(sc);
      while (8192 >= (sc->free_heap_top - sc->free_heap))
	resize_heap(sc);
    }
  if (sc->safety > NO_SAFETY)
    sc->w = copy_tree_with_type(sc, p);
  else sc->w = copy_tree(sc, p);
  p = sc->w;
  sc->w = sc->nil;
  return(p);
}

static s7_pointer copy_closure(s7_scheme *sc, s7_pointer fnc)
{
  /* copy the source tree annotating (for eventual optimization), return a thing of the same type as fnc */
  s7_pointer x, body;

  body = copy_body(sc, closure_body(fnc));
  new_cell(sc, x, typeflag(fnc));
  closure_set_args(x, closure_args(fnc));
  closure_set_body(x, body);
  closure_set_setter(x, closure_setter(fnc));
  closure_arity(x) = closure_arity(fnc);
  closure_set_let(x, closure_let(fnc));
  return(x);
}


/* -------------------------------- defined? -------------------------------- */
static s7_pointer g_is_defined(s7_scheme *sc, s7_pointer args)
{
  #define H_is_defined "(defined? obj (env (curlet)) ignore-globals) returns #t if obj has a binding (a value) in the environment env"
  #define Q_is_defined s7_make_signature(sc, 4, sc->is_boolean_symbol, sc->is_symbol_symbol, sc->is_let_symbol, sc->is_boolean_symbol)

  s7_pointer sym;

  /* is this correct?
   *    (defined? '_x) #f (symbol->value '_x) #<undefined>
   *    (define x #<undefined>) (defined? 'x) #t
   * can't return the value here because it might be #f
   */

  sym = car(args);
  if (!is_symbol(sym))
    return(method_or_bust(sc, sym, sc->is_defined_symbol, args, T_SYMBOL, 1));

  if (is_pair(cdr(args)))
    {
      s7_pointer e, b, x;

      e = cadr(args);
      if (!is_let(e))
	return(wrong_type_argument_with_type(sc, sc->is_defined_symbol, 2, e, a_let_string));
      if (e == sc->s7_let)
	return(make_boolean(sc, is_s7_let_field(sym)));

      if (is_pair(cddr(args)))
	{
	  b = caddr(args);
	  if (!s7_is_boolean(b))
	    return(method_or_bust_with_type(sc, b, sc->is_defined_symbol, args, a_boolean_string, 3));
	}
      else b = sc->F;

      if (e == sc->rootlet) /* we checked (let? e) above */
	{
	  if (b == sc->F)
	    return(make_boolean(sc, is_slot(global_slot(sym)))); /* new_symbol and gensym initialize global_slot to #<undefined> */
	  return(sc->F);
	}

      x = symbol_to_local_slot(sc, sym, e);
      if (is_slot(x))
	return(sc->T);

      if (b == sc->T)
	return(sc->F);

      return(make_boolean(sc, is_slot(global_slot(sym))));
    }
  else
    {
      if (is_global(sym))
	return(sc->T);
    }
  return(make_boolean(sc, is_slot(symbol_to_slot(sc, sym))));
}


bool s7_is_defined(s7_scheme *sc, const char *name)
{
  s7_pointer x;
  x = s7_symbol_table_find_name(sc, name);
  if (x)
    {
      x = symbol_to_slot(sc, x);
      return(is_slot(x));
    }
  return(false);
}

static bool is_defined_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_symbol(p))
    simple_wrong_type_argument(sc, sc->is_defined_symbol, p, T_SYMBOL);
  return(is_slot(symbol_to_slot(sc, p)));
}

static bool is_defined_b_7pp(s7_scheme *sc, s7_pointer p, s7_pointer e) {return(g_is_defined(sc, set_plist_2(sc, p, e)) != sc->F);}


void s7_define(s7_scheme *sc, s7_pointer envir, s7_pointer symbol, s7_pointer value)
{
  s7_pointer x;
  if ((envir == sc->nil) ||
      (envir == sc->rootlet))
    envir = sc->shadow_rootlet;
  x = symbol_to_local_slot(sc, symbol, envir);
  if (is_slot(x))
    slot_set_value_with_hook(x, value);
  else
    {
      s7_make_slot(sc, envir, symbol, value); /* I think this means C code can override "constant" defs */
      if ((envir == sc->shadow_rootlet) &&
	  (!is_slot(global_slot(symbol))))
	{
	  set_global(symbol); /* is_global => global_slot is usable */
	  set_global_slot(symbol, local_slot(symbol));
	}
    }
}

s7_pointer s7_define_variable(s7_scheme *sc, const char *name, s7_pointer value)
{
  s7_pointer sym;
  sym = make_symbol(sc, name);
  s7_define(sc, sc->nil, sym, value);
  return(sym);
}

s7_pointer s7_define_variable_with_documentation(s7_scheme *sc, const char *name, s7_pointer value, const char *help)
{
  s7_pointer sym;
  sym = s7_define_variable(sc, name, value);
  symbol_set_has_help(sym);
  symbol_set_help(sym, copy_string(help));
  return(sym);
}

s7_pointer s7_define_constant(s7_scheme *sc, const char *name, s7_pointer value)
{
  s7_pointer sym;
  sym = make_symbol(sc, name);
  s7_define(sc, sc->nil, sym, value);
  set_immutable(sym);
  set_possibly_constant(sym);
  set_immutable(global_slot(sym));
  set_immutable(local_slot(sym));
  return(sym);
}

/* (define (func a) (let ((cvar (+ a 1))) cvar)) (define-constant cvar 23) (func 1) -> ;can't bind an immutable object: cvar
 * (let ((aaa 1)) (define-constant aaa 32) (set! aaa 3)) -> set!: can't alter immutable object: aaa
 */

s7_pointer s7_define_constant_with_documentation(s7_scheme *sc, const char *name, s7_pointer value, const char *help)
{
  s7_pointer sym;
  sym = s7_define_constant(sc, name, value);
  symbol_set_has_help(sym);
  symbol_set_help(sym, copy_string(help));
  return(value); /* inconsistent with variable above, but consistent with define_function? */
}


/* -------------------------------- keyword? -------------------------------- */

bool s7_is_keyword(s7_pointer obj)
{
  return(is_keyword(obj));
}

static s7_pointer g_is_keyword(s7_scheme *sc, s7_pointer args)
{
  #define H_is_keyword "(keyword? obj) returns #t if obj is a keyword, (keyword? :rest) -> #t"
  #define Q_is_keyword sc->pl_bt
  check_boolean_method(sc, is_keyword, sc->is_keyword_symbol, args);
}


/* -------------------------------- string->keyword -------------------------------- */
s7_pointer s7_make_keyword(s7_scheme *sc, const char *key)
{
  s7_pointer sym;
  block_t *b;
  char *name;
  size_t slen;
  slen = (size_t)safe_strlen(key);
  b = mallocate(sc, slen + 2);
  name = (char *)block_data(b);
  catstrs_direct(name, ":", key, NULL);              /* use catstrs_direct to get around a bug in gcc 8.1 */
  sym = make_symbol_with_length(sc, name, slen + 1); /* keyword slot etc taken care of here (in new_symbol actually) */
  liberate(sc, b);
  return(sym);
}

static s7_pointer g_string_to_keyword(s7_scheme *sc, s7_pointer args)
{
  #define H_string_to_keyword "(string->keyword str) prepends ':' to str and defines that as a keyword"
  #define Q_string_to_keyword s7_make_signature(sc, 2, sc->is_keyword_symbol, sc->is_string_symbol)

  s7_pointer str;
  str = car(args);
  if (!is_string(str))
    return(method_or_bust_one_arg(sc, str, sc->string_to_keyword_symbol, args, T_STRING));
  if ((string_length(str) == 0) ||
      (string_value(str)[0] == '\0'))
    return(s7_error(sc, sc->out_of_range_symbol, set_elist_2(sc, wrap_string(sc, "string->keyword wants a non-null string: ~S", 43), str)));
  return(s7_make_keyword(sc, string_value(str)));
}

/* -------------------------------- keyword->symbol -------------------------------- */
static s7_pointer g_keyword_to_symbol(s7_scheme *sc, s7_pointer args)
{
  #define H_keyword_to_symbol "(keyword->symbol key) returns a symbol with the same name as key but no prepended colon"
  #define Q_keyword_to_symbol s7_make_signature(sc, 2, sc->is_symbol_symbol, sc->is_keyword_symbol)

  s7_pointer sym;
  sym = car(args);
  if (!is_keyword(sym))
    return(method_or_bust_with_type_one_arg(sc, sym, sc->keyword_to_symbol_symbol, args, wrap_string(sc, "a keyword", 9)));
  return(keyword_symbol(sym));
}

s7_pointer s7_keyword_to_symbol(s7_scheme *sc, s7_pointer key)
{
  return(keyword_symbol(key));
}


/* -------------------------------- symbol->keyword -------------------------------- */
static s7_pointer symbol_to_keyword(s7_scheme *sc, s7_pointer sym)
{
  return(s7_make_keyword(sc, symbol_name(sym)));
}

static s7_pointer g_symbol_to_keyword(s7_scheme *sc, s7_pointer args)
{
  #define H_symbol_to_keyword "(symbol->keyword sym) returns a keyword with the same name as sym, but with a colon prepended"
  #define Q_symbol_to_keyword s7_make_signature(sc, 2, sc->is_keyword_symbol, sc->is_symbol_symbol)

  if (!is_symbol(car(args)))
    return(method_or_bust_one_arg(sc, car(args), sc->symbol_to_keyword_symbol, args, T_SYMBOL));
  return(symbol_to_keyword(sc, car(args)));
}


/* -------------------------------- c-pointer? -------------------------------- */
bool s7_is_c_pointer(s7_pointer arg)
{
  return(is_c_pointer(arg));
}

bool s7_is_c_pointer_of_type(s7_pointer arg, s7_pointer type)
{
  return((is_c_pointer(arg)) && (c_pointer_type(arg) == type));
}

static s7_pointer g_is_c_pointer(s7_scheme *sc, s7_pointer args)
{
  #define H_is_c_pointer "(c-pointer? obj type) returns #t if obj is a C pointer being held in s7.  If type is given, the c_pointer's type is also checked."
  #define Q_is_c_pointer s7_make_signature(sc, 3, sc->is_boolean_symbol, sc->T, sc->T)

  s7_pointer p;
  p = car(args);
  if (!is_c_pointer(p))
    {
      if (!has_methods(p)) return(sc->F);
      return(apply_boolean_method(sc, p, sc->is_c_pointer_symbol));
    }
  if (is_pair(cdr(args)))
    return(make_boolean(sc, c_pointer_type(p) == cadr(args)));
  return(sc->T);
}

/* -------------------------------- c-pointer -------------------------------- */
void *s7_c_pointer(s7_pointer p)
{
  if ((is_number(p)) &&
      (s7_integer(p) == 0))
    return(NULL); /* special case where the null pointer has been cons'd up by hand */

  if (!is_c_pointer(p))
    return(NULL);

  return(c_pointer(p));
}

s7_pointer s7_make_c_pointer_with_type(s7_scheme *sc, void *ptr, s7_pointer type, s7_pointer info)
{
  s7_pointer x;
  new_cell(sc, x, T_C_POINTER);
  c_pointer(x) = ptr;
  c_pointer_type(x) = type;
  c_pointer_info(x) = info;
  c_pointer_weak1(x) = sc->F;
  c_pointer_weak2(x) = sc->F;
  return(x);
}

s7_pointer s7_make_c_pointer(s7_scheme *sc, void *ptr) {return(s7_make_c_pointer_with_type(sc, ptr, sc->F, sc->F));}

static s7_pointer g_c_pointer(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer "(c-pointer int type info unmarked) returns a c-pointer object. The type and info args are optional, defaulting to #f."
  #define Q_c_pointer s7_make_circular_signature(sc, 2, 3, sc->is_c_pointer_symbol, sc->is_integer_symbol, sc->T)

  s7_pointer arg, type, info, weak1, weak2, cp;
  intptr_t p;

  type = sc->F;
  info = sc->F;
  weak1 = sc->F;
  weak2 = sc->F;
  arg = car(args);
  if (!s7_is_integer(arg))
    return(method_or_bust(sc, arg, sc->c_pointer_symbol, args, T_INTEGER, 1));
  p = (intptr_t)s7_integer(arg);             /* (c-pointer (bignum "1234")) */
  args = cdr(args);
  if (is_pair(args))
    {
      type = car(args);
      args = cdr(args);
      if (is_pair(args))
	{
	  info = car(args);
	  args = cdr(args);
	  if (is_pair(args))
	    {
	      weak1 = car(args);
	      args = cdr(args);
	      if (is_pair(args))
		weak2 = car(args);
	    }
	}
    }
  cp = s7_make_c_pointer_with_type(sc, (void *)p, type, info);
  c_pointer_set_weak1(cp, weak1);
  c_pointer_set_weak2(cp, weak2);
  if ((weak1 != sc->F) || (weak2 != sc->F))
    add_weak_ref(sc, cp);
  return(cp);
}

/* -------------------------------- c-pointer-info -------------------------------- */
static s7_pointer c_pointer_info_p_p(s7_scheme *sc, s7_pointer p)
{
  if (!is_c_pointer(p))
    return(method_or_bust(sc, p, sc->c_pointer_info_symbol, list_1(sc, p), T_C_POINTER, 1));
  return(c_pointer_info(p));
}

static s7_pointer g_c_pointer_info(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer_info "(c-pointer-info obj) returns the c-pointer info field"
  #define Q_c_pointer_info s7_make_signature(sc, 2, sc->T, sc->is_c_pointer_symbol)
  return(c_pointer_info_p_p(sc, car(args)));
}

/* -------------------------------- c-pointer-type -------------------------------- */
s7_pointer s7_c_pointer_type(s7_pointer p)
{
  if (!is_c_pointer(p))
    return(NULL); /* as above */
  return(c_pointer_type(p));
}

static s7_pointer c_pointer_type_p_p(s7_scheme *sc, s7_pointer p)
{
  if (!is_c_pointer(p))
    return(method_or_bust(sc, p, sc->c_pointer_type_symbol, list_1(sc, p), T_C_POINTER, 1));
  return(c_pointer_type(p));
}

static s7_pointer g_c_pointer_type(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer_type "(c-pointer-type obj) returns the c-pointer type field"
  #define Q_c_pointer_type s7_make_signature(sc, 2, sc->T, sc->is_c_pointer_symbol)
  return(c_pointer_type_p_p(sc, car(args)));
}

/* -------------------------------- c-pointer-weak1/2 -------------------------------- */
static s7_pointer c_pointer_weak1_p_p(s7_scheme *sc, s7_pointer p)
{
  if (!is_c_pointer(p))
    return(method_or_bust(sc, p, sc->c_pointer_weak1_symbol, list_1(sc, p), T_C_POINTER, 1));
  return(c_pointer_weak1(p));
}

static s7_pointer g_c_pointer_weak1(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer_weak1 "(c-pointer-weak1 obj) returns the c-pointer weak1 field"
  #define Q_c_pointer_weak1 s7_make_signature(sc, 2, sc->T, sc->is_c_pointer_symbol)
  return(c_pointer_weak1_p_p(sc, car(args)));
}

static s7_pointer c_pointer_weak2_p_p(s7_scheme *sc, s7_pointer p)
{
  if (!is_c_pointer(p))
    return(method_or_bust(sc, p, sc->c_pointer_weak2_symbol, list_1(sc, p), T_C_POINTER, 1));
  return(c_pointer_weak2(p));
}

static s7_pointer g_c_pointer_weak2(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer_weak2 "(c-pointer-weak2 obj) returns the c-pointer weak2 field"
  #define Q_c_pointer_weak2 s7_make_signature(sc, 2, sc->T, sc->is_c_pointer_symbol)
  return(c_pointer_weak2_p_p(sc, car(args)));
}

/* -------------------------------- c-pointer->list -------------------------------- */
static s7_pointer g_c_pointer_to_list(s7_scheme *sc, s7_pointer args)
{
  #define H_c_pointer_to_list "(c-pointer->list obj) returns the c-pointer data as (list pointer-as-int type info)"
  #define Q_c_pointer_to_list s7_make_signature(sc, 2, sc->is_pair_symbol, sc->is_c_pointer_symbol)

  s7_pointer p;
  p = car(args);
  if (!is_c_pointer(p))
    return(method_or_bust(sc, p, sc->c_pointer_to_list_symbol, args, T_C_POINTER, 1));
  return(s7_list(sc, 3, s7_make_integer(sc, (s7_int)((intptr_t)c_pointer(p))), c_pointer_type(p), c_pointer_info(p)));
}


/* -------------------------------- continuations and gotos -------------------------------- */

enum {NO_JUMP, CALL_WITH_EXIT_JUMP, THROW_JUMP, CATCH_JUMP, ERROR_JUMP, ERROR_QUIT_JUMP};
enum {NO_SET_JUMP, READ_SET_JUMP, LOAD_SET_JUMP, DYNAMIC_WIND_SET_JUMP, S7_CALL_SET_JUMP, EVAL_SET_JUMP};

static s7_pointer g_is_continuation(s7_scheme *sc, s7_pointer args)
{
  #define H_is_continuation "(continuation? obj) returns #t if obj is a continuation"
  #define Q_is_continuation sc->pl_bt

  check_boolean_method(sc, is_continuation, sc->is_continuation_symbol, args);
  /* is this the right thing?  It returns #f for call-with-exit ("goto") because
   *   that form of continuation can't continue (via a jump back to its context).
   * how to recognize the call-with-exit function?  "goto" is an internal name.
   */
}

static bool s7_is_continuation(s7_pointer p) {return(is_continuation(p));}


static s7_pointer protected_list_copy(s7_scheme *sc, s7_pointer a)
{
  s7_pointer slow, fast, p;

  sc->w = cons(sc, car(a), sc->nil);
  p = sc->w;

  slow = fast = cdr(a);
  while (true)
    {
      if (!is_pair(fast))
	{
	  if (is_null(fast))
	    return(sc->w);
	  set_cdr(p, fast);
	  return(sc->w);
	}

      set_cdr(p, cons(sc, car(fast), sc->nil));
      p = cdr(p);

      fast = cdr(fast);
      if (!is_pair(fast))
	{
	  if (is_null(fast))
	    return(sc->w);
	  set_cdr(p, fast);
	  return(sc->w);
	}
      /* if unrolled further, it's a lot slower? */
      set_cdr(p, cons(sc, car(fast), sc->nil));
      p = cdr(p);

      fast = cdr(fast);
      slow = cdr(slow);
      if (fast == slow)
	{
	  /* try to preserve the original cyclic structure */
	  s7_pointer p1, f1, p2, f2;
	  set_match_pair(a);
	  for (p1 = sc->w, f1 = a; !(is_matched_pair(cdr(f1))); f1 = cdr(f1), p1 = cdr(p1))
	    set_match_pair(f1);
	  for (p2 = sc->w, f2 = a; cdr(f1) != f2; f2 = cdr(f2), p2 = cdr(p2))
	    clear_match_pair(f2);
	  for (f1 = f2; is_pair(f1); f1 = cdr(f1), f2 = cdr(f2))
	    {
	      clear_match_pair(f1);
	      f1 = cdr(f1);
	      clear_match_pair(f1);
	      if (f1 == f2) break;
	    }
	  if (is_null(p1))
	    set_cdr(p2, p2);
	  else set_cdr(p1, p2);
	  return(sc->w);
	}
    }
  return(sc->w);
}


static s7_pointer copy_counter(s7_scheme *sc, s7_pointer obj)
{
  s7_pointer nobj;
  new_cell(sc, nobj, T_COUNTER);
  counter_set_result(nobj, counter_result(obj));
  counter_set_list(nobj, counter_list(obj));
  counter_set_capture(nobj, counter_capture(obj));
  counter_set_let(nobj, counter_let(obj));
  counter_set_slots(nobj, counter_slots(obj));
  return(nobj);
}

static s7_pointer copy_stack(s7_scheme *sc, s7_pointer old_v, int64_t top)
{
  #define CC_INITIAL_STACK_SIZE 256 /* 128 is too small here */
  int64_t i, len;
  s7_pointer new_v;
  s7_pointer *nv, *ov;

  /* stacks can grow temporarily, so sc->stack_size grows, but we don't normally need all that
   *   leftover space here, so choose the original stack size if it's smaller.
   */
  len = vector_length(old_v);
  if (len > CC_INITIAL_STACK_SIZE)
    {
      if (top < CC_INITIAL_STACK_SIZE / 4)
	len = CC_INITIAL_STACK_SIZE;
    }
  else
    {
      if (len < CC_INITIAL_STACK_SIZE)
	len = CC_INITIAL_STACK_SIZE;
    }
  if ((int64_t)(sc->free_heap_top - sc->free_heap) < (int64_t)(sc->heap_size / 4)) gc(sc);
  /* this gc call is needed if there are lots of call/cc's -- by pure bad luck
   *   we can end up hitting the end of the gc free list time after time while
   *   in successive copy_stack's below, causing s7 to core up until it runs out of memory.
   *   It seems like it would make more sense to use len*32 or something similar as the
   *   trigger, but that was slower in my timing tests!?
   */

  new_v = make_simple_vector(sc, len);
  set_type(new_v, T_STACK);
  temp_stack_top(new_v) = top;
  nv = stack_elements(new_v);
  ov = stack_elements(old_v);
  if (len > 0)
    memcpy((void *)nv, (void *)ov, ((len > vector_length(old_v)) ? vector_length(old_v) : len) * sizeof(s7_pointer));

  s7_gc_on(sc, false);

  for (i = 2; i < top; i += 4)
    {
      s7_pointer p;
      p = ov[i];                            /* args */
      if (is_pair(p))                       /* args need not be a list (it can be a port or #f, etc) */
	{
	  nv[i] = protected_list_copy(sc, p);                /* args (copy is needed -- see s7test.scm) */
	  set_type(nv[i], (typeflag(p) & (~T_HAS_OPTLIST))); /* carry over T_IMMUTABLE, but not T_HAS_OPTLIST since opt2 is not copied in protected_list_copy */
	}
      /* lst can be dotted or circular here.  The circular list only happens in a case like:
       *    (dynamic-wind (lambda () (eq? (let ((lst (cons 1 2))) (set-cdr! lst lst) lst) (call/cc (lambda (k) k)))) (lambda () #f) (lambda () #f))
       */
      else
	{
	  if (is_counter(p))              /* these can only occur in this context */
	    nv[i] = copy_counter(sc, p);
	}
    }
  s7_gc_on(sc, true);
  return(new_v);
}


static inline s7_pointer make_goto(s7_scheme *sc)
{
  s7_pointer x;
  new_cell(sc, x, T_GOTO);
  call_exit_goto_loc(x) = s7_stack_top(sc);
  call_exit_op_loc(x) = (int32_t)(sc->op_stack_now - sc->op_stack);
  call_exit_active(x) = true;
  return(x);
}


static s7_pointer *copy_op_stack(s7_scheme *sc)
{
  int32_t len;
  s7_pointer *ops;
  ops = (s7_pointer *)malloc(sc->op_stack_size * sizeof(s7_pointer));
  len = (int32_t)(sc->op_stack_now - sc->op_stack);
  if (len > 0)
    memcpy((void *)ops, (void *)(sc->op_stack), len * sizeof(s7_pointer));
  return(ops);
}


/* (with-baffle . body) calls body guaranteeing that there can be no jumps into the
 *    middle of it from outside -- no outer evaluation of a continuation can jump across this
 *    barrier:  The flip-side of call-with-exit.
 *    It sets a T_BAFFLE var in a new env, that has a unique key.  Call/cc then always
 *    checks the env chain for any such variable, saving the localmost.  Apply of a continuation
 *    looks for such a saved variable, if none, go ahead, else check the current env (before the
 *    jump) for that variable.  If none, error, else go ahead.  This is different from a delimited
 *    continuation which simply delimits the extent of the continuation (why not use lambda?) -- we want to block it
 *    from coming at us from some unknown place.
 */

static s7_pointer make_baffle(s7_scheme *sc)
{
  s7_pointer x;
  new_cell(sc, x, T_BAFFLE);
  baffle_key(x) = sc->baffle_ctr++;
  return(x);
}


static bool find_baffle(s7_scheme *sc, s7_int key)
{
  /* search backwards through sc->envir for sc->baffle_symbol with key as value */
  s7_pointer x, y;
  for (x = sc->envir; is_let(x); x = outlet(x))
    for (y = let_slots(x); is_slot(y); y = next_slot(y))
      if ((slot_symbol(y) == sc->baffle_symbol) &&
	  (baffle_key(slot_value(y)) == key))
	return(true);

  if ((is_slot(global_slot(sc->baffle_symbol))) &&
      (is_baffle(slot_value(global_slot(sc->baffle_symbol)))))
    return(baffle_key(slot_value(global_slot(sc->baffle_symbol))) == key);

  return(false);
}

static s7_int find_any_baffle(s7_scheme *sc)
{
  /* search backwards through sc->envir for any sc->baffle_symbol */
  if (sc->baffle_ctr > 0)
    {
      s7_pointer x, y;
      for (x = sc->envir; is_let(x); x = outlet(x))
	for (y = let_slots(x); is_slot(y); y = next_slot(y))
	  if (slot_symbol(y) == sc->baffle_symbol)
	    return(baffle_key(slot_value(y)));

      if ((is_slot(global_slot(sc->baffle_symbol))) &&
	  (is_baffle(slot_value(global_slot(sc->baffle_symbol)))))
	return(baffle_key(slot_value(global_slot(sc->baffle_symbol))));
    }
  return(-1);
}


s7_pointer s7_make_continuation(s7_scheme *sc)
{
  s7_pointer x, stack;
  int64_t loc;
  block_t *block;

  loc = s7_stack_top(sc);
  stack = copy_stack(sc, sc->stack, loc);
  sc->temp8 = stack;

  new_cell(sc, x, T_CONTINUATION);
  block = mallocate_block(sc);
  continuation_block(x) = block;
  continuation_set_stack(x, stack);
  continuation_stack_size(x) = vector_length(continuation_stack(x));   /* copy_stack can return a smaller stack than the current one */
  continuation_stack_start(x) = stack_elements(continuation_stack(x));
  continuation_stack_end(x) = (s7_pointer *)(continuation_stack_start(x) + loc);
  continuation_op_stack(x) = copy_op_stack(sc);                        /* no heap allocation here */
  continuation_op_loc(x) = (int32_t)(sc->op_stack_now - sc->op_stack);
  continuation_op_size(x) = sc->op_stack_size;
  continuation_key(x) = find_any_baffle(sc);
  sc->temp8 = sc->nil;

  add_continuation(sc, x);
  return(x);
}

static void let_temp_done(s7_scheme *sc, s7_pointer args, s7_pointer code, s7_pointer let)
{
  push_stack(sc, OP_EVAL_DONE, sc->args, sc->code);
  sc->args = args;
  sc->code = code;
  sc->envir = let;
  eval(sc, OP_LET_TEMP_DONE);
}

static bool check_for_dynamic_winds(s7_scheme *sc, s7_pointer c)
{
  /* called only from call_with_current_continuation.
   *   if call/cc jumps into a dynamic-wind, the init/finish funcs are wrapped in with-baffle
   *   so they'll complain.  Otherwise we're supposed to re-run the init func before diving
   *   into the body.  Similarly for let-temporarily.  If a call/cc jumps out of a dynamic-wind
   *   body-func, we're supposed to call the finish-func.  The continuation is called at
   *   s7_stack_top(sc); the continuation form is at continuation_stack_top(c).
   */
  int64_t i;
  opcode_t op;

  /* check sc->stack for dynamic-winds we're jumping out of */
  for (i = s7_stack_top(sc) - 1; i > 0; i -= 4)
    {
      op = stack_op(sc->stack, i);
      switch (op)
	{
	case OP_DYNAMIC_WIND:
	case OP_LET_TEMP_DONE:
	  {
	    s7_pointer x;
	    int64_t j, s_base = 0;
	    x = stack_code(sc->stack, i);
	    for (j = 3; j < continuation_stack_top(c); j += 4)
	      if (((stack_op(continuation_stack(c), j) == OP_DYNAMIC_WIND) ||
		   (stack_op(continuation_stack(c), j) == OP_LET_TEMP_DONE)) &&
		  (x == stack_code(continuation_stack(c), j)))
		{
		  s_base = i;
		  break;
		}
	    if (s_base == 0)
	      {
		if (op == OP_DYNAMIC_WIND)
		  {
		    if (dynamic_wind_state(x) == DWIND_BODY)
		      {
			dynamic_wind_state(x) = DWIND_FINISH;
			if (dynamic_wind_out(x) != sc->F)
			  {
			    push_stack(sc, OP_EVAL_DONE, sc->args, sc->code);
			    sc->args = sc->nil;
			    sc->code = dynamic_wind_out(x);
			    eval(sc, OP_APPLY);
			  }
		      }
		  }
		else let_temp_done(sc, stack_args(sc->stack, i), stack_code(sc->stack, i), stack_let(sc->stack, i));
	      }
	  }
	  break;

	case OP_BARRIER:
	  if (i > continuation_stack_top(c))  /* otherwise it's some unproblematic outer eval-string? */
	    return(false);                    /*    but what if we've already evaluated a dynamic-wind closer? */
	  break;

	case OP_DEACTIVATE_GOTO:              /* here we're jumping out of an unrelated call-with-exit block */
	  if (i > continuation_stack_top(c))
	    call_exit_active(stack_args(sc->stack, i)) = false;
	  break;

	default:
	  break;
	}
    }

  /* check continuation-stack for dynamic-winds we're jumping into */
  for (i = s7_stack_top(sc) - 1; i < continuation_stack_top(c); i += 4)
    {
      op = stack_op(continuation_stack(c), i);
      if (op == OP_DYNAMIC_WIND)
	{
	  s7_pointer x;
	  x = stack_code(continuation_stack(c), i);
	  if (dynamic_wind_in(x) != sc->F)
	    {
	      push_stack(sc, OP_EVAL_DONE, sc->args, sc->code);
	      sc->args = sc->nil;
	      sc->code = dynamic_wind_in(x);
	      eval(sc, OP_APPLY);
	    }
	  dynamic_wind_state(x) = DWIND_BODY;
	}
      else
	{
	  if (op == OP_DEACTIVATE_GOTO)
	    call_exit_active(stack_args(continuation_stack(c), i)) = true;
	  /* not let_temp_done here! */
	}
    }
  return(true);
}

static bool call_with_current_continuation(s7_scheme *sc)
{
  s7_pointer c;
  c = sc->code;

  /* check for (baffle ...) blocking the current attempt to continue */
  if ((continuation_key(c) >= 0) &&
      (!(find_baffle(sc, continuation_key(c))))) /* should this raise an error? */
    return(false);

  if (!check_for_dynamic_winds(sc, c))
    return(true);

  /* we push_stack sc->code before calling an embedded eval above, so sc->code should still be c here, etc */
  sc->stack = copy_stack(sc, continuation_stack(c), continuation_stack_top(c));
  sc->stack_size = continuation_stack_size(c);
  sc->stack_start = stack_elements(sc->stack);
  sc->stack_end = (s7_pointer *)(sc->stack_start + continuation_stack_top(c));
  sc->stack_resize_trigger = (s7_pointer *)(sc->stack_start + sc->stack_size / 2);

  {
    int32_t i, top;
    top = continuation_op_loc(c);
    sc->op_stack_now = (s7_pointer *)(sc->op_stack + top);
    sc->op_stack_size = continuation_op_size(c);
    sc->op_stack_end = (s7_pointer *)(sc->op_stack + sc->op_stack_size);
    for (i = 0; i < top; i++)
      sc->op_stack[i] = continuation_op_stack(c)[i];
  }

  if (is_null(sc->args))
    sc->value = sc->nil;
  else
    {
      if (is_null(cdr(sc->args)))
	sc->value = car(sc->args);
      else sc->value = splice_in_values(sc, sc->args);
    }
  return(true);
}

static s7_pointer g_call_cc(s7_scheme *sc, s7_pointer args)
{
  #define H_call_cc "(call-with-current-continuation (lambda (continuer)...)) is always a mistake!"
  #define Q_call_cc s7_make_signature(sc, 2, sc->T, sc->is_procedure_symbol)

  s7_pointer p;
  p = car(args);                             /* this is the procedure passed to call/cc */
  if (!is_t_procedure(p))                    /* this includes continuations */
    {
      check_method(sc, p, sc->call_cc_symbol, args);
      check_method(sc, p, sc->call_with_current_continuation_symbol, args);
      return(simple_wrong_type_argument_with_type(sc, sc->call_cc_symbol, p, a_procedure_string));
    }
  if (!s7_is_aritable(sc, p, 1))
    return(s7_error(sc, sc->wrong_type_arg_symbol, set_elist_2(sc, wrap_string(sc, "call/cc procedure, ~A, should take one argument", 47), p)));

  sc->w = s7_make_continuation(sc);
  push_stack(sc, OP_APPLY, list_1(sc, sc->w), p);
  sc->w = sc->nil;

  return(sc->nil);
}

/* we can't naively optimize call/cc to call-with-exit if the continuation is only
 *   used as a function in the call/cc body because it might (for example) be wrapped
 *   in a lambda form that is being exported.  See b-func in s7test for an example.
 */

static void apply_continuation(s7_scheme *sc)               /* -------- continuation ("call/cc") -------- */
{
  if (!call_with_current_continuation(sc))
    s7_error(sc, sc->baffled_symbol,
	     set_elist_1(sc, wrap_string(sc, "continuation can't jump into with-baffle", 40)));
}

static bool op_continuation_a(s7_scheme *sc)
{
  s7_pointer s, code;
  code = sc->code;
  s = symbol_to_value_checked(sc, car(code));
  if (!is_continuation(s)) {sc->last_function = s; return(false);}
  sc->code = s;
  sc->args = set_plist_1(sc, c_call(cdr(code))(sc, cadr(code)));
  apply_continuation(sc);
  return(true);
}

/* -------------------------------- with-baffle -------------------------------- */
static s7_pointer check_with_baffle(s7_scheme *sc)
{
  if (!s7_is_proper_list(sc, sc->code))
    eval_error(sc, "with-baffle: unexpected dot? ~A", 31, sc->code);
  if (!is_null(cdr(sc->code)))
    pair_set_syntax_op(sc->code, OP_WITH_BAFFLE_UNCHECKED);
  return(sc->code);
}

static bool op_with_baffle_unchecked(s7_scheme *sc)
{
  set_current_code(sc, sc->code);
  sc->code = cdr(sc->code);
  if (is_null(sc->code))
    {
      sc->value = sc->nil;
      return(true);
    }
  new_frame(sc, sc->envir, sc->envir);
  make_slot_1(sc, sc->envir, sc->baffle_symbol, make_baffle(sc));
  return(false);
}


/* -------------------------------- call-with-exit -------------------------------- */
static void call_with_exit(s7_scheme *sc)
{
  int64_t i, new_stack_top, quit = 0;

  if (!call_exit_active(sc->code))
    s7_error(sc, sc->invalid_escape_function_symbol, set_elist_1(sc, wrap_string(sc, "call-with-exit escape procedure called outside its block", 56)));

  call_exit_active(sc->code) = false;
  new_stack_top = call_exit_goto_loc(sc->code);
  sc->op_stack_now = (s7_pointer *)(sc->op_stack + call_exit_op_loc(sc->code));

  /* look for dynamic-wind in the stack section that we are jumping out of */
  for (i = s7_stack_top(sc) - 1; i > new_stack_top; i -= 4)
    {
      opcode_t op;

      op = stack_op(sc->stack, i);
      switch (op)
	{
	case OP_DYNAMIC_WIND:
	  {
	    s7_pointer lx;
	    lx = stack_code(sc->stack, i);
	    if (dynamic_wind_state(lx) == DWIND_BODY)
	      {
		dynamic_wind_state(lx) = DWIND_FINISH;
		if (dynamic_wind_out(lx) != sc->F)
		  {
		    push_stack(sc, OP_EVAL_DONE, sc->args, sc->code);
		    sc->args = sc->nil;
		    sc->code = dynamic_wind_out(lx);
		    eval(sc, OP_APPLY);
		  }
	      }
	  }
	  break;

	case OP_EVAL_STRING:
	  s7_close_input_port(sc, sc->input_port);
	  pop_input_port(sc);
	  break;

	case OP_BARRIER:                /* oops -- we almost certainly went too far */
	  goto SET_VALUE;

	case OP_DEACTIVATE_GOTO:        /* here we're jumping into an unrelated call-with-exit block */
	  call_exit_active(stack_args(sc->stack, i)) = false;
	  break;

	case OP_LET_TEMP_DONE:
	  let_temp_done(sc, stack_args(sc->stack, i), stack_code(sc->stack, i), stack_let(sc->stack, i));
	  break;

	  /* call/cc does not close files, but I think call-with-exit should */
	case OP_GET_OUTPUT_STRING:
	case OP_UNWIND_OUTPUT:
	  {
	    s7_pointer x;
	    x = stack_code(sc->stack, i);                /* "code" = port that we opened */
	    s7_close_output_port(sc, x);
	    x = stack_args(sc->stack, i);                /* "args" = port that we shadowed, if not #<gc-nil> */
	    if (x != sc->gc_nil)
	      sc->output_port = x;
	  }
	  break;

	case OP_UNWIND_INPUT:
	  s7_close_input_port(sc, stack_code(sc->stack, i)); /* "code" = port that we opened */
	  sc->input_port = stack_args(sc->stack, i);         /* "args" = port that we shadowed */
	  break;

	case OP_EVAL_DONE: /* goto called in a method -- put off the inner eval return(s) until we clean up the stack */
	  quit++;
	  break;

	default:
	  break;
	}
    }

  /* is this right? maybe the SET_VALUE should skip setting stack_end? */
 SET_VALUE:
  sc->stack_end = (s7_pointer *)(sc->stack_start + new_stack_top);

  /* the return value should have an implicit values call, just as in call/cc */
  if (is_null(sc->args))
    sc->value = sc->nil;
  else
    {
      if (is_null(cdr(sc->args)))
	sc->value = car(sc->args);
      else sc->value = splice_in_values(sc, sc->args);
    }

  if (quit > 0)
    {
      if (sc->longjmp_ok)
	{
	  pop_stack(sc);
	  longjmp(sc->goto_start, CALL_WITH_EXIT_JUMP);
	}
      for (i = 0; i < quit; i++)
	push_stack_op_let(sc, OP_EVAL_DONE);
    }
}

static s7_pointer g_call_with_exit(s7_scheme *sc, s7_pointer args)
{
  #define H_call_with_exit "(call-with-exit (lambda (exiter) ...)) is call/cc without the ability to jump back into a previous computation."
  #define Q_call_with_exit s7_make_signature(sc, 2, sc->values_symbol, sc->is_procedure_symbol)

  s7_pointer p, x;
  /* (call-with-exit (lambda (return) ...)) */

  p = car(args);
  if (!is_t_procedure(p))                  /* this includes continuations */
    return(method_or_bust_with_type_one_arg(sc, p, sc->call_with_exit_symbol, args, a_procedure_string));

  x = make_goto(sc);
  push_stack(sc, OP_DEACTIVATE_GOTO, x, p); /* this means call-with-exit is not tail-recursive */
  push_stack(sc, OP_APPLY, cons_unchecked(sc, x, sc->nil), p);

  /* if the lambda body calls the argument as a function,
   *   it is applied to its arguments, apply notices that it is a goto, and...
   *
   *      (conceptually...) sc->stack_top = call_exit_goto_loc(sc->code);
   *      s_pop(sc, (is_not_null(sc->args)) ? car(sc->args) : sc->nil);
   *
   *   which jumps to the point of the goto returning car(args).
   *
   * There is one gotcha: we can't jump back in from outside, so if the caller saves the goto
   *   and tries to invoke it outside the call-with-exit block, we have to
   *   make sure it triggers an error.  So, if the escape is called, it then
   *   deactivates itself.  Otherwise the block returns, we pop to OP_DEACTIVATE_GOTO,
   *   and it finds the goto in sc->args.
   * Even worse:
       (let ((cc #f))
         (call-with-exit
           (lambda (c3)
             (call/cc (lambda (ret) (set! cc ret)))
             (c3)))
         (cc))
   * where we jump back into a call-with-exit body via call/cc, the goto has to be
   * re-established.
   *
   * I think call-with-exit could be based on catch, but it's a simpler notion,
   *   and certainly at the source level it is easier to read.
   */
  return(sc->nil);
}

static s7_pointer op_call_with_exit(s7_scheme *sc)
{
  s7_pointer go, args;
  set_current_code(sc, sc->code);
  args = opt2_pair(sc->code);
  go = make_goto(sc);
  push_stack_no_let_no_code(sc, OP_DEACTIVATE_GOTO, go); /* was also pushing code */
  new_frame_with_slot(sc, sc->envir, sc->envir, caar(args), go);
  sc->code = T_Pair(cdr(args));
  return(NULL);
}

static s7_pointer op_call_with_exit_p(s7_scheme *sc)
{
  s7_pointer go, args;
  set_current_code(sc, sc->code);
  args = opt2_pair(sc->code);
  go = make_goto(sc);
  push_stack_no_let_no_code(sc, OP_DEACTIVATE_GOTO, go);
  new_frame_with_slot(sc, sc->envir, sc->envir, caar(args), go);
  sc->code = cadr(args);
  return(NULL);
}

static bool op_goto(s7_scheme *sc)
{
  set_opt1_goto(sc->code, symbol_to_value_checked(sc, car(sc->code)));
  if (!is_goto(opt1_goto(sc->code))) return(false);
  sc->args = sc->nil;
  sc->code = T_Got(opt1_goto(sc->code));
  call_with_exit(sc);
  return(true);
}

static bool op_goto_a(s7_scheme *sc)
{
  set_opt1_goto(sc->code, symbol_to_value_checked(sc, car(sc->code)));
  if (!is_goto(opt1_goto(sc->code))) return(false);
  sc->args = list_1(sc, c_call(cdr(sc->code))(sc, cadr(sc->code)));
  sc->code = T_Got(opt1_goto(sc->code));
  call_with_exit(sc);
  return(true);
}


/* -------------------------------- numbers -------------------------------- */

#if WITH_GMP
  static char *big_number_to_string_with_radix(s7_pointer p, s7_int radix, s7_int width, s7_int *nlen, use_write_t use_write);
  static bool big_numbers_are_eqv(s7_pointer a, s7_pointer b);
  static s7_pointer string_to_either_integer(s7_scheme *sc, const char *str, s7_int radix);
  static s7_pointer string_to_either_ratio(s7_scheme *sc, const char *nstr, const char *dstr, s7_int radix);
  static s7_pointer string_to_either_real(s7_scheme *sc, const char *str, s7_int radix);
  static s7_pointer string_to_either_complex(s7_scheme *sc, char *q, char *slash1, char *ex1, bool has_dec_point1,
					     char *plus, char *slash2, char *ex2, bool has_dec_point2, s7_int radix, int32_t has_plus_or_minus);
  static s7_pointer big_add(s7_scheme *sc, s7_pointer args);
  static s7_pointer big_subtract(s7_scheme *sc, s7_pointer args);
  static s7_pointer big_multiply(s7_scheme *sc, s7_pointer args);
  static s7_pointer big_divide(s7_scheme *sc, s7_pointer args);
  static s7_pointer big_random(s7_scheme *sc, s7_pointer args);
  static s7_pointer s7_int_to_big_integer(s7_scheme *sc, s7_int val);
  static s7_pointer s7_ratio_to_big_ratio(s7_scheme *sc, s7_int num, s7_int den);
  static s7_pointer s7_number_to_big_real(s7_scheme *sc, s7_pointer p);
  static s7_pointer promote_number(s7_scheme *sc, int32_t type, s7_pointer x);
  static s7_pointer big_equal(s7_scheme *sc, s7_pointer args);
  static s7_pointer big_negate(s7_scheme *sc, s7_pointer args);
  static s7_pointer big_invert(s7_scheme *sc, s7_pointer args);
#if (!WITH_PURE_S7)
  static s7_pointer big_inexact_to_exact(s7_scheme *sc, s7_pointer args);
  static s7_pointer big_exact_to_inexact(s7_scheme *sc, s7_pointer args);
#endif
  static s7_pointer mpz_to_big_integer(s7_scheme *sc, mpz_t val);
  static s7_pointer mpq_to_big_ratio(s7_scheme *sc, mpq_t val);
  static s7_pointer mpfr_to_big_real(s7_scheme *sc, mpfr_t val);
  static s7_pointer mpc_to_big_complex(s7_scheme *sc, mpc_t val);
#endif

#ifndef HAVE_OVERFLOW_CHECKS
#if ((defined(__clang__) && (!POINTER_32) && ((__clang_major__ > 3) || (__clang_major__ == 3 && __clang_minor__ >= 4))) || (defined(__GNUC__) && __GNUC__ >= 5))
  #define HAVE_OVERFLOW_CHECKS 1
#else
  #define HAVE_OVERFLOW_CHECKS 0
  #if (!WITH_GMP)
    #warning "no arithmetic overflow checks in this version of s7"
  #endif
#endif
#endif

#if (defined(__clang__) && (!POINTER_32) && ((__clang_major__ > 3) || (__clang_major__ == 3 && __clang_minor__ >= 4)))
  #define subtract_overflow(A, B, C)       __builtin_ssubll_overflow((long long)A, (long long)B, (long long *)C)
  #define add_overflow(A, B, C)            __builtin_saddll_overflow((long long)A, (long long)B, (long long *)C)
  #define multiply_overflow(A, B, C)       __builtin_smulll_overflow((long long)A, (long long)B, (long long *)C)
  /* #define int32_subtract_overflow(A, B, C) __builtin_ssub_overflow(A, B, C) */
  #define int32_add_overflow(A, B, C)      __builtin_sadd_overflow(A, B, C)
  #define int32_multiply_overflow(A, B, C) __builtin_smul_overflow(A, B, C)
#else
#if (defined(__GNUC__) && __GNUC__ >= 5)
  #define subtract_overflow(A, B, C)       __builtin_sub_overflow(A, B, C)
  #define add_overflow(A, B, C)            __builtin_add_overflow(A, B, C)
  #define multiply_overflow(A, B, C)       __builtin_mul_overflow(A, B, C)
  /* #define int32_subtract_overflow(A, B, C) __builtin_sub_overflow(A, B, C) */
  #define int32_add_overflow(A, B, C)      __builtin_add_overflow(A, B, C)
  #define int32_multiply_overflow(A, B, C) __builtin_mul_overflow(A, B, C)
#endif
#endif

#if WITH_GCC
#define s7_int_abs(x) ({s7_int _X_; _X_ = x; _X_ >= 0 ? _X_ : -_X_;})
#else
#define s7_int_abs(x) ((x) >= 0 ? (x) : -(x))
#endif
/* can't use abs even in gcc -- it doesn't work with int64_ts! */

#if (!__NetBSD__)
  #define s7_fabsl(X) fabsl(X)
#else
  static double s7_fabsl(long double x) {if (x < 0.0) return(-x);  return(x);}
#endif


static bool is_NaN(s7_double x) {return(x != x);}
/* callgrind says this is faster than isnan, I think (very confusing data...) */


#if defined(__sun) && defined(__SVR4)
  static bool is_inf(s7_double x) {return((x == x) && (is_NaN(x - x)));} /* there's no isinf in Solaris */
#else
#if (!MS_WINDOWS)

  #if __cplusplus
    #define is_inf(x) std::isinf(x)
  #else
    #define is_inf(x) isinf(x)
  #endif

#else
  static bool is_inf(s7_double x) {return((x == x) && (is_NaN(x - x)));}  /* Another possibility: (x * 0) != 0 */

  /* in MS C, we need to provide inverse hyperbolic trig funcs and cbrt */
  static double asinh(double x) {return(log(x + sqrt(1.0 + x * x)));}
  static double acosh(double x) {return(log(x + sqrt(x * x - 1.0)));}
  /* perhaps less prone to numerical troubles (untested): 2.0 * log(sqrt(0.5 * (x + 1.0)) + sqrt(0.5 * (x - 1.0))) */
  static double atanh(double x) {return(log((1.0 + x) / (1.0 - x)) / 2.0);}
  static double cbrt(double x) {if (x >= 0.0) return(pow(x, 1.0 / 3.0)); return(-pow(-x, 1.0 / 3.0));}
#endif /* windows */
#endif /* sun */


/* for g_log, we also need round. this version is from stackoverflow, see also round_per_R5RS below */
double s7_round(double number) {return((number < 0.0) ? ceil(number - 0.5) : floor(number + 0.5));}

#if HAVE_COMPLEX_NUMBERS
#if __cplusplus
  #define _Complex_I (complex<s7_double>(0.0, 1.0))
  #define creal(x) Real(x)
  #define cimag(x) Imag(x)
  #define carg(x) arg(x)
  #define cabs(x) abs(x)
  #define csqrt(x) sqrt(x)
  #define cpow(x, y) pow(x, y)
  #define clog(x) log(x)
  #define cexp(x) exp(x)
  #define csin(x) sin(x)
  #define ccos(x) cos(x)
  #define csinh(x) sinh(x)
  #define ccosh(x) cosh(x)
#else
  typedef double complex s7_complex;
#endif


#if (!HAVE_COMPLEX_TRIG)
#if (__cplusplus)

  static s7_complex ctan(s7_complex z)   {return(csin(z) / ccos(z));}
  static s7_complex ctanh(s7_complex z)  {return(csinh(z) / ccosh(z));}
  static s7_complex casin(s7_complex z)  {return(-_Complex_I * clog(_Complex_I * z + csqrt(1.0 - z * z)));}
  static s7_complex cacos(s7_complex z)  {return(-_Complex_I * clog(z + _Complex_I * csqrt(1.0 - z * z)));}
  static s7_complex catan(s7_complex z)  {return(_Complex_I * clog((_Complex_I + z) / (_Complex_I - z)) / 2.0);}
  static s7_complex casinh(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
  static s7_complex cacosh(s7_complex z) {return(clog(z + csqrt(z * z - 1.0)));}
  static s7_complex catanh(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
#else

/* still not in FreeBSD! */
static s7_complex clog(s7_complex z) {return(log(fabs(cabs(z))) + carg(z) * _Complex_I);}
static s7_complex cpow(s7_complex x, s7_complex y)
{
  s7_double r = cabs(x);
  s7_double theta = carg(x);
  s7_double yre = creal(y);
  s7_double yim = cimag(y);
  s7_double nr = exp(yre * log(r) - yim * theta);
  s7_double ntheta = yre * theta + yim * log(r);
  return(nr * cos(ntheta) + (nr * sin(ntheta)) * _Complex_I); /* make-polar */
}

#if (!defined(__FreeBSD__)) || (__FreeBSD__ < 9) /* untested -- this orignally looked at __FreeBSD_version which apparently no longer exists */
  static s7_complex cexp(s7_complex z) {return(exp(creal(z)) * cos(cimag(z)) + (exp(creal(z)) * sin(cimag(z))) * _Complex_I);}
#endif

#if (!defined(__FreeBSD__)) || (__FreeBSD__ < 10)
  static s7_complex csin(s7_complex z)   {return(sin(creal(z)) * cosh(cimag(z)) + (cos(creal(z)) * sinh(cimag(z))) * _Complex_I);}
  static s7_complex ccos(s7_complex z)   {return(cos(creal(z)) * cosh(cimag(z)) + (-sin(creal(z)) * sinh(cimag(z))) * _Complex_I);}
  static s7_complex csinh(s7_complex z)  {return(sinh(creal(z)) * cos(cimag(z)) + (cosh(creal(z)) * sin(cimag(z))) * _Complex_I);}
  static s7_complex ccosh(s7_complex z)  {return(cosh(creal(z)) * cos(cimag(z)) + (sinh(creal(z)) * sin(cimag(z))) * _Complex_I);}
  static s7_complex ctan(s7_complex z)   {return(csin(z) / ccos(z));}
  static s7_complex ctanh(s7_complex z)  {return(csinh(z) / ccosh(z));}
  static s7_complex casin(s7_complex z)  {return(-_Complex_I * clog(_Complex_I * z + csqrt(1.0 - z * z)));}
  static s7_complex cacos(s7_complex z)  {return(-_Complex_I * clog(z + _Complex_I * csqrt(1.0 - z * z)));}
  static s7_complex catan(s7_complex z)  {return(_Complex_I * clog((_Complex_I + z) / (_Complex_I - z)) / 2.0);}
  static s7_complex catanh(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
  static s7_complex casinh(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
  static s7_complex cacosh(s7_complex z) {return(clog(z + csqrt(z * z - 1.0)));}
  /* perhaps less prone to numerical troubles (untested): 2.0 * clog(csqrt(0.5 * (z + 1.0)) + csqrt(0.5 * (z - 1.0))) */
#endif /* not FreeBSD 10 */
#endif /* not c++ */
#endif /* not HAVE_COMPLEX_TRIG */

#else  /* not HAVE_COMPLEX_NUMBERS */
  typedef double s7_complex;
  #define _Complex_I 1
  #define creal(x) x
  #define cimag(x) x
  #define csin(x) sin(x)
  #define casin(x) x
  #define ccos(x) cos(x)
  #define cacos(x) x
  #define ctan(x) x
  #define catan(x) x
  #define csinh(x) x
  #define casinh(x) x
  #define ccosh(x) x
  #define cacosh(x) x
  #define ctanh(x) x
  #define catanh(x) x
  #define cexp(x) exp(x)
  #define cpow(x, y) pow(x, y)
  #define clog(x) log(x)
  #define csqrt(x) sqrt(x)
  #define conj(x) x
#endif

#ifdef __OpenBSD__
  /* openbsd's builtin versions of these functions are not usable */
  static s7_complex catanh_1(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
  static s7_complex casinh_1(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
  static s7_complex cacosh_1(s7_complex z) {return(clog(z + csqrt(z * z - 1.0)));}
#endif
#ifdef __NetBSD__
  static s7_complex catanh_1(s7_complex z) {return(clog((1.0 + z) / (1.0 - z)) / 2.0);}
  static s7_complex casinh_1(s7_complex z) {return(clog(z + csqrt(1.0 + z * z)));}
#endif


bool s7_is_number(s7_pointer p)
{
#if WITH_GMP
  return((is_number(p)) || (is_big_number(p)));
#else
  return(is_number(p));
#endif
}


bool s7_is_integer(s7_pointer p)
{
#if WITH_GMP
  return((is_t_integer(p)) ||
	 (is_t_big_integer(p)));
#else
  return(is_integer(p));
#endif
}

bool s7_is_real(s7_pointer p)
{
#if WITH_GMP
  return((is_real(p)) ||
	 (is_t_big_integer(p)) ||
	 (is_t_big_ratio(p)) ||
	 (is_t_big_real(p)));
#else
  return(is_real(p)); /* in GSL, a NaN or inf is not a real, or perhaps better, finite = not (nan or inf) */
#endif
}


bool s7_is_rational(s7_pointer p)
{
#if WITH_GMP
  return((is_rational(p)) ||
	 (is_t_big_integer(p)) ||
	 (is_t_big_ratio(p)));
#else
  return(is_rational(p));
#endif
}


bool s7_is_ratio(s7_pointer p)
{
#if WITH_GMP
  return((is_t_ratio(p)) || (is_t_big_ratio(p)));
#else
  return(is_t_ratio(p));
#endif
}


bool s7_is_complex(s7_pointer p)
{
#if WITH_GMP
  return((is_number(p)) || (is_big_number(p)));
#else
  return(is_number(p));
#endif
}


static s7_int c_gcd(s7_int u, s7_int v)
{
  s7_int a, b;

  if ((u == s7_int_min) || (v == s7_int_min))
    {
      /* can't take abs of these (below) so do it by hand */
      s7_int divisor = 1;
      if (u == v) return(u);
      while (((u & 1) == 0) && ((v & 1) == 0))
	{
	  u /= 2;
	  v /= 2;
	  divisor *= 2;
	}
      return(divisor);
    }

  a = s7_int_abs(u);
  b = s7_int_abs(v);
  /* there are faster gcd algorithms but does it ever matter? */
  while (b != 0)
    {
      s7_int temp;
      temp = a % b;
      a = b;
      b = temp;
    }
  /* if (a < 0) return(-a); */ /* why this? */
  return(a);
}


static bool c_rationalize(s7_double ux, s7_double error, s7_int *numer, s7_int *denom)
{
  /*
    (define* (rat ux (err 0.0000001))
      ;; translated from CL code in Canny, Donald, Ressler, "A Rational Rotation Method for Robust Geometric Algorithms"
      (let ((x0 (- ux error))
	    (x1 (+ ux error)))
        (let ((i (ceiling x0))
	      (i0 (floor x0))
	      (i1 (ceiling x1))
	      (r 0))
          (if (>= x1 i)
	      i
	      (do ((p0 i0 (+ p1 (* r p0)))
	           (q0 1 (+ q1 (* r q0)))
	           (p1 i1 p0)
	           (q1 1 q0)
	           (e0 (- i1 x0) e1p)
	           (e1 (- x0 i0) (- e0p (* r e1p)))
	           (e0p (- i1 x1) e1)
	           (e1p (- x1 i0) (- e0 (* r e1))))
	          ((<= x0 (/ p0 q0) x1)
	           (/ p0 q0))
	        (set! r (min (floor (/ e0 e1))
			     (ceiling (/ e0p e1p)))))))))
  */

  double x0, x1;
  s7_int i, i0, i1, p0, q0, p1, q1;
  double e0, e1, e0p, e1p;
  int32_t tries = 0;
  /* don't use long double: the loop below will hang */

  /* #e1e19 is a killer -- it's bigger than most-positive-fixnum, but if we ceil(ux) below
   *   it turns into most-negative-fixnum.  1e19 is trouble in many places.
   */
  if ((ux > s7_int_max) || (ux < s7_int_min))
    {
      /* can't return false here because that confuses some of the callers! */
      if (ux > s7_int_min) (*numer) = s7_int_max; else (*numer) = s7_int_min;
      (*denom) = 1;
      return(true);
    }

  if (error < 0.0) error = -error;
  x0 = ux - error;
  x1 = ux + error;
  i = (s7_int)ceil(x0);

  if (error >= 1.0) /* aw good grief! */
    {
      if (x0 < 0)
	{
	  if (x1 < 0)
	    (*numer) = (s7_int)floor(x1);
	  else (*numer) = 0;
	}
      else (*numer) = i;
      (*denom) = 1;
      return(true);
    }

  if (x1 >= i)
    {
      if (i >= 0)
	(*numer) = i;
      else (*numer) = (s7_int)floor(x1);
      (*denom) = 1;
      return(true);
    }

  i0 = (s7_int)floor(x0);
  i1 = (s7_int)ceil(x1);

  p0 = i0;
  q0 = 1;
  p1 = i1;
  q1 = 1;
  e0 = i1 - x0;
  e1 = x0 - i0;
  e0p = i1 - x1;
  e1p = x1 - i0;

  while (true)
    {
      s7_int old_p1, old_q1;
      double old_e0, old_e1, old_e0p, val, r, r1;
      val = (double)p0 / (double)q0;

      if (((x0 <= val) && (val <= x1)) ||
	  (e1 == 0)                    ||
	  (e1p == 0)                   ||
	  (tries > 100))
	{
	  if ((q0 == s7_int_min) && (p0 == 1)) /* (rationalize 1.000000004297917e-12) when error is 1e-12 */
	    {
	      (*numer) = 0;
	      (*denom) = 1;
	    }
	  else
	    {
	      (*numer) = p0;
	      (*denom) = q0;
	    }
	  return(true);
	}
      tries++;

      r = (s7_int)floor(e0 / e1);
      r1 = (s7_int)ceil(e0p / e1p);
      if (r1 < r) r = r1;

      /* do handles all step vars in parallel */
      old_p1 = p1;
      p1 = p0;
      old_q1 = q1;
      q1 = q0;
      old_e0 = e0;
      e0 = e1p;
      old_e0p = e0p;
      e0p = e1;
      old_e1 = e1;

      p0 = old_p1 + r * p0;
      q0 = old_q1 + r * q0;
      e1 = old_e0p - r * e1p;
      /* if the error is set too low, we can get e1 = 0 here: (rationalize (/ pi) 1e-17) */
      e1p = old_e0 - r * old_e1;
    }
  return(false);
}

s7_pointer s7_rationalize(s7_scheme *sc, s7_double x, s7_double error)
{
  s7_int numer = 0, denom = 1;
  if (c_rationalize(x, error, &numer, &denom))
    return(s7_make_ratio(sc, numer, denom));
  return(make_real(sc, x));
}


static s7_int number_to_numerator(s7_pointer n)
{
  if (is_t_ratio(n))
    return(numerator(n));
  return(integer(n));
}

static s7_int number_to_denominator(s7_pointer n)
{
  if (is_t_ratio(n))
    return(denominator(n));
  return(1);
}


s7_pointer s7_make_integer(s7_scheme *sc, s7_int n)
{
  s7_pointer x;
  if (is_small(n))              /* ((n >= 0) && (n < NUM_SMALL_INTS)) is slower */
    return(small_int(n));

  new_cell(sc, x, T_INTEGER);
  integer(x) = n;
  return(x);
}

static s7_pointer make_mutable_integer(s7_scheme *sc, s7_int n)
{
  s7_pointer x;
  new_cell(sc, x, T_INTEGER | T_MUTABLE);
  integer(x) = n;
  return(x);
}

static s7_pointer make_permanent_integer(s7_int i)
{
  if (is_small(i)) return(small_int(i));

  if (i == MAX_ARITY) return(max_arity);
  if (i == CLOSURE_ARITY_NOT_SET) return(arity_not_set);
  if (i == -1) return(minus_one);
  if (i == -2) return(minus_two);
  /* a few -3 */

  return(make_permanent_integer_unchecked(i));
}

s7_pointer s7_make_real(s7_scheme *sc, s7_double n)
{
  s7_pointer x;
  new_cell(sc, x, T_REAL);
  set_real(x, n);
  return(x);
}

s7_pointer s7_make_mutable_real(s7_scheme *sc, s7_double n)
{
  s7_pointer x;
  new_cell(sc, x, T_REAL | T_MUTABLE);
  set_real(x, n);
  return(x);
}

s7_pointer s7_make_complex(s7_scheme *sc, s7_double a, s7_double b)
{
  s7_pointer x;
  if (b == 0.0)
    {
      new_cell(sc, x, T_REAL);
      set_real(x, a);
    }
  else
    {
      new_cell(sc, x, T_COMPLEX);
      set_real_part(x, a);
      set_imag_part(x, b);
    }
  return(x);
}


s7_pointer s7_make_ratio(s7_scheme *sc, s7_int a, s7_int b)
{
  s7_pointer x;
  s7_int divisor;

  if (b == 0)
    return(division_by_zero_error(sc, wrap_string(sc, "make-ratio", 10), set_elist_2(sc, wrap_integer1(sc, a), small_int(0))));
  if (a == 0)
    return(small_int(0));
  if (b == 1)
    return(make_integer(sc, a));

#if (!WITH_GMP)
  if (b == s7_int_min)
    {
      if (a == b)
	return(small_int(1));
      /* we've got a problem... This should not trigger an error during reading -- we might have the
       *   ratio on a switch with-bignums or whatever, so its mere occurrence is just an annoyance.
       */
      if (a & 1)
	return(make_real(sc, (long double)a / (long double)b));
      a /= 2;
      b /= 2;
    }
#endif

  if (b < 0)
    {
      a = -a;
      b = -b;
    }
  divisor = c_gcd(a, b);
  if (divisor != 1)
    {
      a /= divisor;
      b /= divisor;
    }
  if (b == 1)
    return(make_integer(sc, a));

  new_cell(sc, x, T_RATIO);
  numerator(x) = a;
  denominator(x) = b;

  return(x);
}

static inline s7_pointer make_simple_ratio(s7_scheme *sc, s7_int num, s7_int den)
{
  s7_pointer x;
  if (den == 1)
    return(make_integer(sc, num));
  if (den == -1)
    return(make_integer(sc, -num));
  if ((den == s7_int_min) && ((num & 1) != 0))
    return(make_real(sc, (long double)num / (long double)den));
  new_cell(sc, x, T_RATIO);
  if (den < 0)
    {
      numerator(x) = -num;
      denominator(x) = -den;
    }
  else
    {
      numerator(x) = num;
      denominator(x) = den;
    }
  return(x);
}


#define WITH_OVERFLOW_ERROR true
#define WITHOUT_OVERFLOW_ERROR false

#if (!WITH_PURE_S7) && (!WITH_GMP)
static s7_pointer exact_to_inexact(s7_scheme *sc, s7_pointer x)
{
  /* this is tricky because a big int32_t can mess up when turned into a double: (truncate (exact->inexact most-positive-fixnum)) -> -9223372036854775808 */
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, (s7_double)(integer(x))));
    case T_RATIO:   return(make_real(sc, (s7_double)(fraction(x))));
    case T_REAL:
    case T_COMPLEX: return(x); /* apparently (exact->inexact 1+i) is not an error */
    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->exact_to_inexact_symbol, list_1(sc, x), a_number_string));
    }
}

static s7_pointer inexact_to_exact(s7_scheme *sc, s7_pointer x, bool with_error)
{
  switch (type(x))
    {
    case T_INTEGER:
    case T_RATIO:
      return(x);

    case T_REAL:
      {
	s7_int numer = 0, denom = 1;
	s7_double val;

	val = real(x); /* no fall through */
	if ((is_inf(val)) || (is_NaN(val)))
	  {
	    if (with_error)
	      return(simple_wrong_type_argument_with_type(sc, sc->inexact_to_exact_symbol, x, a_normal_real_string));
	    return(sc->nil);
	  }

	if ((val > s7_int_max) ||
	    (val < s7_int_min))
	  {
	    if (with_error)
	      return(simple_out_of_range(sc, sc->inexact_to_exact_symbol, x, its_too_large_string));
	    return(sc->nil);
	  }

	if (c_rationalize(val, sc->default_rationalize_error, &numer, &denom))
	  return(s7_make_ratio(sc, numer, denom));
      }

    default:
      if (with_error)
	return(method_or_bust_one_arg(sc, x, sc->inexact_to_exact_symbol, list_1(sc, x), T_REAL));
      return(sc->nil);
    }
  return(x);
}
#endif

/* this is a mess -- it's too late to clean up s7.h (sigh) */
s7_double s7_number_to_real_with_caller(s7_scheme *sc, s7_pointer x, const char *caller)
{
  if (is_t_real(x))
    return(real(x));

  switch (type(x))
    {
    case T_INTEGER:     return((s7_double)integer(x));
    case T_RATIO:       return((s7_double)numerator(x) / (s7_double)denominator(x));
    case T_REAL:        return(real(x));
#if WITH_GMP
    case T_BIG_INTEGER: return((s7_double)big_integer_to_s7_int(big_integer(x)));
    case T_BIG_RATIO:   return((s7_double)((long_double)big_integer_to_s7_int(mpq_numref(big_ratio(x))) /
					   (long_double)big_integer_to_s7_int(mpq_denref(big_ratio(x)))));
    case T_BIG_REAL:    return((s7_double)mpfr_get_d(big_real(x), GMP_RNDN));
#endif
    }
  s7_wrong_type_arg_error(sc, caller, 0, x, "a real number");
  return(0.0);
}

s7_double s7_number_to_real(s7_scheme *sc, s7_pointer x)
{
  return(s7_number_to_real_with_caller(sc, x, "s7_number_to_real"));
}

s7_int s7_number_to_integer_with_caller(s7_scheme *sc, s7_pointer x, const char *caller)
{
  if (is_t_integer(x))
    return(integer(x));

#if WITH_GMP
  if (is_t_big_integer(x))
    return(big_integer_to_s7_int(big_integer(x)));
#endif
  s7_wrong_type_arg_error(sc, caller, 0, x, "an integer");
  return(0);
}

s7_int s7_number_to_integer(s7_scheme *sc, s7_pointer x)
{
  return(s7_number_to_integer_with_caller(sc, x, "s7_number_to_integer"));
}


s7_int s7_numerator(s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:     return(integer(x));
    case T_RATIO:       return(numerator(x));
#if WITH_GMP
    case T_BIG_INTEGER: return(big_integer_to_s7_int(big_integer(x)));
    case T_BIG_RATIO:   return(big_integer_to_s7_int(mpq_numref(big_ratio(x))));
#endif
    }
  return(0);
}


s7_int s7_denominator(s7_pointer x)
{
  switch (type(x))
    {
    case T_RATIO:     return(denominator(x));
#if WITH_GMP
    case T_BIG_RATIO: return(big_integer_to_s7_int(mpq_denref(big_ratio(x))));
#endif
    }
  return(1);
}


s7_int s7_integer(s7_pointer p)
{
  if (is_t_integer(p))
    return(integer(p));

#if WITH_GMP
  if (is_t_big_integer(p))
    return(big_integer_to_s7_int(big_integer(p)));
#endif

  return(0);
}

s7_double s7_real(s7_pointer x)
{
  if (is_t_real(x))
    return(real(x));
  return(s7_number_to_real_with_caller(cur_sc, x, "s7_real"));
}

#if (!WITH_GMP)
static s7_complex s7_to_c_complex(s7_pointer p)
{
#if HAVE_COMPLEX_NUMBERS
  return(CMPLX(s7_real_part(p), s7_imag_part(p)));
#else
  return(0.0);
#endif
}


static s7_pointer s7_from_c_complex(s7_scheme *sc, s7_complex z)
{
  return(s7_make_complex(sc, creal(z), cimag(z)));
}
#endif


#if (!WITH_GMP)
static s7_pointer s7_negate(s7_scheme *sc, s7_pointer p)     /* can't use "negate" because it confuses C++! */
{
  switch (type(p))
    {
    case T_INTEGER:
      if (integer(p) == s7_int_min)
	return(simple_out_of_range(sc, sc->subtract_symbol, p, wrap_string(sc, "most-negative-fixnum can't be negated", 37)));
      return(make_integer(sc, -integer(p)));
    case T_RATIO:   return(make_simple_ratio(sc, -numerator(p), denominator(p)));
    case T_REAL:    return(make_real(sc, -real(p)));
    default:        return(s7_make_complex(sc, -real_part(p), -imag_part(p)));
    }
}
#endif


static s7_pointer s7_invert(s7_scheme *sc, s7_pointer p)      /* s7_ to be consistent... */
{
  switch (type(p))
    {
    case T_INTEGER:
      return(make_simple_ratio(sc, 1, integer(p)));           /* a already checked, not 0 */

    case T_RATIO:
      return(make_simple_ratio(sc, denominator(p), numerator(p)));

    case T_REAL:
      return(make_real(sc, 1.0 / real(p)));

    case T_COMPLEX:
      {
	s7_double r2, i2, den;
	r2 = real_part(p);
	i2 = imag_part(p);
	den = (r2 * r2 + i2 * i2);
	return(s7_make_complex(sc, r2 / den, -i2 / den));
      }

    default:
      return(wrong_type_argument_with_type(sc, sc->divide_symbol, 1, p, a_number_string));
    }
}


static s7_pointer subtract_ratios(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  s7_int d1, d2, n1, n2;
  d1 = number_to_denominator(x);
  n1 = number_to_numerator(x);
  d2 = number_to_denominator(y);
  n2 = number_to_numerator(y);

  if (d1 == d2)                                     /* the easy case -- if overflow here, it matches the int32_t case */
    return(s7_make_ratio(sc, n1 - n2, d1));

#if (!WITH_GMP) && HAVE_OVERFLOW_CHECKS
  {
    s7_int n1d2, n2d1, d1d2, dn;
    if ((multiply_overflow(d1, d2, &d1d2)) ||
	(multiply_overflow(n1, d2, &n1d2)) ||
	(multiply_overflow(n2, d1, &n2d1)) ||
	(subtract_overflow(n1d2, n2d1, &dn)))
      return(make_real(sc, ((long_double)n1 / (long_double)d1) - ((long_double)n2 / (long_double)d2)));
    return(s7_make_ratio(sc, dn, d1d2));
  }
#else
  return(s7_make_ratio(sc, n1 * d2 - n2 * d1, d1 * d2));
#endif
}

static bool s7_is_negative(s7_pointer obj)
{
  switch (type(obj))
    {
    case T_INTEGER:     return(integer(obj) < 0);
    case T_RATIO:       return(numerator(obj) < 0);
#if WITH_GMP
    case T_BIG_INTEGER: return(mpz_cmp_ui(big_integer(obj), 0) < 0);
    case T_BIG_RATIO:   return(mpq_cmp_ui(big_ratio(obj), 0, 1) < 0);
    case T_BIG_REAL:    return(mpfr_cmp_ui(big_real(obj), 0) < 0);
#endif
    default:            return(real(obj) < 0);
    }
}

static bool s7_is_positive(s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:     return(integer(x) > 0);
    case T_RATIO:       return(numerator(x) > 0);
#if WITH_GMP
    case T_BIG_INTEGER: return(mpz_cmp_ui(big_integer(x), 0) > 0);
    case T_BIG_RATIO:   return(mpq_cmp_ui(big_ratio(x), 0, 1) > 0);
    case T_BIG_REAL:    return(mpfr_cmp_ui(big_real(x), 0) > 0);
#endif
    default:            return(real(x) > 0.0);
    }
}

static bool s7_is_zero(s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:     return(integer(x) == 0);
    case T_REAL:        return(real(x) == 0.0);
#if WITH_GMP
    case T_BIG_INTEGER: return(mpz_cmp_ui(big_integer(x), 0) == 0);
    case T_BIG_RATIO:   return(mpz_cmp_ui(mpq_numref(big_ratio(x)), 0) == 0);   /* a big_ratio can be zero! */
    case T_BIG_REAL:    return(mpfr_zero_p(big_real(x)));
#endif
    default:            return(false); /* ratios and complex numbers here are already collapsed into integers and reals */
    }
}

static bool s7_is_one(s7_pointer x)
{
    return(((is_integer(x)) && (integer(x) == 1)) ||
	   ((is_t_real(x)) && (real(x) == 1.0)));
}


/* -------- optimize exponents -------- */

#define MAX_POW 64
static double pepow[17][MAX_POW * 2];

static void init_pows(void)
{
  int32_t i, j;
  for (i = 2; i < 17; i++)        /* radix between 2 and 16 */
    for (j = -MAX_POW; j < MAX_POW; j++) /* saved exponent between 0 and +/- MAX_POW */
      pepow[i][j + MAX_POW] = pow((double)i, (double)j);
}

static double ipow(int32_t x, int32_t y)
{
  if ((y >= MAX_POW) || (y < -MAX_POW))
    return(pow((double)x, (double)y));
  return(pepow[x][y + MAX_POW]);
}


/* -------------------------------- number->string -------------------------------- */
static const char dignum[] = "0123456789abcdef";

static size_t integer_to_string_any_base(char *p, s7_int n, s7_int radix)  /* called by number_to_string_with_radix */
{
  s7_int i, len, end;
  bool sign;
  s7_int pown;

  if ((radix < 2) || (radix > 16))
    return(0);

  if (n == s7_int_min) /* can't negate this, so do it by hand */
    {
      static const char *mnfs[17] = {"","",
	"-1000000000000000000000000000000000000000000000000000000000000000", "-2021110011022210012102010021220101220222",
	"-20000000000000000000000000000000", "-1104332401304422434310311213", "-1540241003031030222122212",
	"-22341010611245052052301", "-1000000000000000000000", "-67404283172107811828",	"-9223372036854775808",
	"-1728002635214590698",	"-41a792678515120368", "-10b269549075433c38", "-4340724c6c71dc7a8", "-160e2ad3246366808", "-8000000000000000"};

      len = safe_strlen(mnfs[radix]);
      memcpy((void *)p, (void *)mnfs[radix], len);
      p[len] = '\0';
      return(len);
    }

  sign = (n < 0);
  if (sign) n = -n;

  /* the previous version that counted up to n, rather than dividing down below n, as here,
   *   could be confused by large ints on 64 bit machines
   */
  pown = n;
  for (i = 1; i < 100; i++)
    {
      if (pown < radix)
	break;
      pown /= (s7_int)radix;
    }
  len = i - 1;
  if (sign) len++;
  end = 0;
  if (sign)
    {
      p[0] = '-';
      end++;
    }
  for (i = len; i >= end; i--)
    {
      p[i] = dignum[n % radix];
      n /= radix;
    }
  p[len + 1] = '\0';
  return(len + 1);
}

static char *integer_to_string(s7_scheme *sc, s7_int num, s7_int *nlen) /* do not free the returned string */
{
  char *p, *op;
  bool sign;

  if (num == s7_int_min)
    {
      (*nlen) = 20;
      return((char *)"-9223372036854775808");
    }
  p = (char *)(sc->int_to_str1 + INT_TO_STR_SIZE - 1);
  op = p;
  *p-- = '\0';

  sign = (num < 0);
  if (sign) num = -num;  /* we need a positive index below */
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  if (sign)
    {
      *p = '-';
      (*nlen) = op - p;
      return(p);
    }

  (*nlen) = op - p - 1;
  return(++p);
}

static char *integer_to_string_no_length(s7_scheme *sc, s7_int num) /* do not free the returned string */
{
  char *p;
  bool sign;

  if (num == s7_int_min)
    return((char *)"-9223372036854775808");
  p = (char *)(sc->int_to_str2 + INT_TO_STR_SIZE - 1);
  *p-- = '\0';
  sign = (num < 0);
  if (sign) num = -num;
  do {*p-- = "0123456789"[num % 10]; num /= 10;} while (num);
  if (sign)
    {
      *p = '-';
      return(p);
    }
  return(++p);
}

#define BASE_10 10

static char *floatify(char *str, s7_int *nlen)
{
  if ((!strchr(str, '.')) &&
      (!strchr(str, 'e')))
    {
      /* this assumes there is room in str for 2 more chars */
      s7_int len;
      len = *nlen;
      str[len]='.';
      str[len + 1]='0';
      str[len + 2]='\0';
      (*nlen) = len + 2;
    }
  return(str);
}

static void insert_spaces(s7_scheme *sc, char *src, s7_int width, s7_int len)
{
  s7_int spaces;
  if (width >= sc->num_to_str_size)
    {
      sc->num_to_str_size = width + 1;
      sc->num_to_str = (char *)realloc(sc->num_to_str, sc->num_to_str_size * sizeof(char));
    }
  spaces = width - len;
  sc->num_to_str[width] = '\0';
  memmove((void *)(sc->num_to_str + spaces), (void *)src, len);
  memset((void *)(sc->num_to_str), (int)' ', spaces);
}

static char *number_to_string_base_10(s7_scheme *sc, s7_pointer obj, s7_int width, s7_int precision, char float_choice, s7_int *nlen, use_write_t choice) /* don't free result */
{
  /* called by number_to_string_with_radix g_number_to_string, number_to_string_p_p number_to_port format_number */
  /* the rest of s7 assumes nlen is set to the correct length
   *   a tricky case: (format #f "~f" 1e308) -- tries to print 308 digits! so 256 as default len is too small.
   *   but then even worse: (format #f "~F" 1e308+1e308i)!
   */
  s7_int len;

  if ((width + precision) > 512)
    len = 512 + 2 * (width + precision);
  else len = 1024;

  if (len > sc->num_to_str_size)
    {
      if (!sc->num_to_str)
	sc->num_to_str = (char *)malloc(len * sizeof(char));
      else sc->num_to_str = (char *)realloc(sc->num_to_str, len * sizeof(char));
      sc->num_to_str_size = len;
    }

  /* bignums can't happen here */
  switch (type(obj))
    {
    case T_INTEGER:
      if (width == 0)
	{
	  if (has_print_name(obj))
	    {
	      (*nlen) = print_name_length(obj);
	      return((char *)print_name(obj));
	    }
	  return(integer_to_string(sc, integer(obj), nlen));
	}
      {
	char *p;
	p = integer_to_string(sc, integer(obj), &len);
	if (width > len)
	  {
	    insert_spaces(sc, p, width, len);
	    (*nlen) = width;
	    return(sc->num_to_str);
	  }
	(*nlen) = len;
	return(p);
      }

    case T_RATIO:
      len = catstrs_direct(sc->num_to_str, integer_to_string_no_length(sc, numerator(obj)), "/", pos_int_to_str_direct(sc, denominator(obj)), NULL);
      if (width > len)
	{
	  insert_spaces(sc, sc->num_to_str, width, len);
	  (*nlen) = width;
	}
      else (*nlen) = len;
      return(sc->num_to_str);

    case T_REAL:
      if (width == 0)
	len = snprintf(sc->num_to_str, sc->num_to_str_size - 4,
		       (float_choice == 'g') ? "%.*g" : ((float_choice == 'f') ? "%.*f" : "%.*e"),
		       (int32_t)precision, real(obj)); /* -4 for floatify */
      else len = snprintf(sc->num_to_str, sc->num_to_str_size - 4,
			  (float_choice == 'g') ? "%*.*g" : ((float_choice == 'f') ? "%*.*f" : "%*.*e"),
			  (int32_t)width, (int32_t)precision, real(obj)); /* -4 for floatify */
      (*nlen) = len;
      floatify(sc->num_to_str, nlen);
      return(sc->num_to_str);

    default:
      {
	if ((choice == P_READABLE) &&
	    ((is_NaN(real_part(obj))) || (is_NaN(imag_part(obj))) || ((is_inf(real_part(obj))) || (is_inf(imag_part(obj))))))
	  {
	    char rbuf[128], ibuf[128];
	    char *rp, *ip;
	    if (is_NaN(real_part(obj)))
	      rp = (char *)"+nan.0";
	    else
	      {
		if (is_inf(real_part(obj)))
		  {
		    if (real_part(obj) < 0.0)
		      rp = (char *)"-inf.0";
		    else rp = (char *)"+inf.0";
		  }
		else
		  {
		    snprintf(rbuf, 128, "%.*g", (int32_t)precision, (double)real_part(obj));
		    rp = rbuf;
		  }
	      }
	    if (is_NaN(imag_part(obj)))
	      ip = (char *)"+nan.0";
	    else
	      {
		if (is_inf(imag_part(obj)))
		  {
		    if (imag_part(obj) < 0.0)
		      ip = (char *)"-inf.0";
		    else ip = (char *)"+inf.0";
		  }
		else
		  {
		    snprintf(ibuf, 128, "%.*g", (int32_t)precision, (double)imag_part(obj));
		    ip = ibuf;
		  }
	      }
	    sc->num_to_str[0] = '\0';
	    len = catstrs(sc->num_to_str, sc->num_to_str_size, "(complex ", rp, " ", ip, ")", NULL);
	  }
	else
	  {
	    if (imag_part(obj) >= 0.0)
	      len = snprintf(sc->num_to_str, sc->num_to_str_size,
			     (float_choice == 'g') ? "%.*g+%.*gi" : ((float_choice == 'f') ? "%.*f+%.*fi" : "%.*e+%.*ei"),
			     (int32_t)precision, (double)real_part(obj), (int32_t)precision, (double)imag_part(obj));
	    else len = snprintf(sc->num_to_str, sc->num_to_str_size,
				(float_choice == 'g') ? "%.*g%.*gi" : ((float_choice == 'f') ? "%.*f%.*fi" :"%.*e%.*ei"), /* minus sign comes with the imag_part */
				(int32_t)precision, (double)real_part(obj), (int32_t)precision, (double)imag_part(obj));
	  }

	if (width > len)  /* (format #f "~20g" 1+i) */
	  {
	    insert_spaces(sc, sc->num_to_str, width, len);
	    (*nlen) = width;
	  }
	else (*nlen) = len;
      }
      break;
    }
  return(sc->num_to_str);
}

static char *number_to_string_with_radix(s7_scheme *sc, s7_pointer obj, s7_int radix, s7_int width, s7_int precision, char float_choice, s7_int *nlen)
{ /* called by s7_number_to_string (char*), g_number_to_string (strp), number_to_string_p_pp (strp), format_number (strp basically) */
  /* the rest of s7 assumes nlen is set to the correct length */
  char *p;
  s7_int len, str_len;

#if WITH_GMP
  if (s7_is_bignum(obj))
    return(big_number_to_string_with_radix(obj, radix, width, nlen, P_WRITE));
  /* this ignores precision because it's way too hard to get the mpfr string to look like
   *   C's output -- we either have to call mpfr_get_str twice (the first time just to
   *   find out what the exponent is and how long the string actually is), or we have
   *   to do messy string manipulations.  So (format #f "",3F" pi) ignores the "3" and
   *   prints the full string.
   */
#endif

  if (radix == 10)
    {
      p = number_to_string_base_10(sc, obj, width, precision, float_choice, nlen, P_WRITE);
      return(copy_string_with_length(p, *nlen));
    }

  switch (type(obj))
    {
    case T_INTEGER:
      {
	size_t len;
	p = (char *)malloc((128 + width) * sizeof(char));
	len = integer_to_string_any_base(p, integer(obj), radix);
	if ((size_t)width > len)
	  {
	    size_t start;
	    start = width - len;
	    memmove((void *)(p + start), (void *)p, len);
	    memset((void *)p, (int)' ', start);
	    p[width] = '\0';
	    *nlen = width;
	  }
	else *nlen = len;
	return(p);
      }
    case T_RATIO:
      {
	size_t len1, len2;
	str_len = 256 + width;
	p = (char *)malloc(str_len * sizeof(char));
	len1 = integer_to_string_any_base(p, numerator(obj), radix);
	p[len1] = '/';
	len2 = integer_to_string_any_base((char *)(p + len1 + 1), denominator(obj), radix);
        len = len1 + 1 + len2;
        p[len] = '\0';
      }
      break;

    case T_REAL:
      {
	int32_t i;
	s7_int int_part;
	s7_double x, frac_part, min_frac, base;
	bool sign = false;
	char n[128], d[256];

	x = real(obj);

	if (is_NaN(x))
	  return(copy_string_with_length("+nan.0", *nlen = 6));
	if (is_inf(x))
	  {
	    if (x < 0.0)
	      return(copy_string_with_length("-inf.0", *nlen = 6));
	    return(copy_string_with_length("+inf.0", *nlen = 6));
	  }

	if (x < 0.0)
	  {
	    sign = true;
	    x = -x;
	  }

	if (x > 1.0e18) /* i.e. close to or greater than most-positive-fixnum (9.22e18), so the code below is unlikely to work, (format #f "~X" 1e19) */
	  {
	    int32_t ep;
	    char *p1;
	    len = 0;
	    ep = (int32_t)floor(log(x) / log((double)radix));
 	    real(sc->real_wrapper3) = x / pow((double)radix, (double)ep); /* divide it down to one digit, then the fractional part */
	    p1 = number_to_string_with_radix(sc, sc->real_wrapper3, radix, width, precision, float_choice, &len);
	    p = (char *)malloc((len + 8) * sizeof(char));
	    p[0] = '\0';
	    (*nlen) = catstrs(p, len + 8, (sign) ? "-" : "", p1, "e", integer_to_string_no_length(sc, ep), NULL);
	    free(p1);
	    return(p);
	  }

	int_part = (s7_int)floor(x);
	frac_part = x - int_part;
	integer_to_string_any_base(n, int_part, radix);
	min_frac = (s7_double)ipow(radix, -precision);

	/* doesn't this assume precision < 128/256 and that we can fit in 256 digits (1e308)? */

	for (i = 0, base = radix; (i < precision) && (frac_part > min_frac); i++, base *= radix)
	  {
	    s7_int ipart;
	    ipart = (s7_int)(frac_part * base);
	    if (ipart >= radix)         /* rounding confusion */
	      ipart = radix - 1;
	    frac_part -= (ipart / base);
	    if (ipart < 10)
	      d[i] = (char)('0' + ipart);
	    else d[i] = (char)('a' + ipart -  10);
	  }
	if (i == 0)
	  d[i++] = '0';
	d[i] = '\0';
	p = (char *)malloc(256 * sizeof(char));
	p[0] = '\0';
	len = catstrs(p, 256, (sign) ? "-" : "", n, ".", d, NULL);
	str_len = 256;
      }
      break;

    default:
      {
	char *n, *d;
	p = (char *)malloc(512 * sizeof(char));
 	real(sc->real_wrapper3) = real_part(obj);
	n = number_to_string_with_radix(sc, sc->real_wrapper3, radix, 0, precision, float_choice, &len);
 	real(sc->real_wrapper4) = imag_part(obj);
	d = number_to_string_with_radix(sc, sc->real_wrapper4, radix, 0, precision, float_choice, &len);
	p[0] = '\0';
	len = catstrs(p, 512, n, (imag_part(obj) < 0.0) ? "" : "+", d, "i", NULL);
	str_len = 512;
	free(n);
	free(d);
      }
      break;
    }

  if (width > len)
    {
      s7_int spaces;
      if (width >= str_len)
	{
	  str_len = width + 1;
	  p = (char *)realloc(p, str_len * sizeof(char));
	}
      spaces = width - len;
      p[width] = '\0';
      memmove((void *)(p + spaces), (void *)p, len);
      memset((void *)p, (int)' ', spaces);
      (*nlen) = width;
    }
  else (*nlen) = len;
  return(p);
}

char *s7_number_to_string(s7_scheme *sc, s7_pointer obj, s7_int radix)
{
  s7_int nlen = 0;
  return(number_to_string_with_radix(sc, obj, radix, 0, 20, 'g', &nlen));  /* (log top 10) so we get all the digits in base 10 (??) */
}

static s7_pointer block_to_string(s7_scheme *sc, block_t *block, s7_int len);

static s7_pointer g_number_to_string(s7_scheme *sc, s7_pointer args)
{
  #define H_number_to_string "(number->string num (radix 10)) converts the number num into a string."
  #define Q_number_to_string s7_make_signature(sc, 3, sc->is_string_symbol, sc->is_number_symbol, sc->is_integer_symbol)

  s7_int nlen = 0, radix = 10;
  char *res;
  s7_pointer x;

  x = car(args);
  if (!s7_is_number(x))
    return(method_or_bust_with_type(sc, x, sc->number_to_string_symbol, args, a_number_string, 1));

  if (is_pair(cdr(args)))
    {
      s7_pointer y;
      y = cadr(args);
      if (s7_is_integer(y))
	radix = s7_integer(y);
      else return(method_or_bust(sc, y, sc->number_to_string_symbol, args, T_INTEGER, 2));
      if ((radix < 2) || (radix > 16))
	return(out_of_range(sc, sc->number_to_string_symbol, small_int(2), y, a_valid_radix_string));
    }

#if WITH_GMP
  if (s7_is_bignum(x))
    {
      s7_pointer p;
      res = big_number_to_string_with_radix(x, radix, 0, &nlen, P_WRITE);
      p = make_string_with_length(sc, res, nlen);
      free(res);
      return(p);
    }
#endif

  if (radix != 10)
    {
#if 0
      /* weird -- this is much slower due to malloc? */
      block_t *b;
      res = number_to_string_with_radix(sc, x, radix, 0, sc->float_format_precision, 'g', &nlen);
      b = mallocate_block(sc);
      block_data(b) = (void *)res;
      block_set_index(b, TOP_BLOCK_LIST);
      return(block_to_string(sc, b, nlen));
#else
      s7_pointer p;
      res = number_to_string_with_radix(sc, x, radix, 0, sc->float_format_precision, 'g', &nlen);
      p = make_string_with_length(sc, res, nlen);
      free(res);
      return(p);
#endif
    }
  res = number_to_string_base_10(sc, x, 0, sc->float_format_precision, 'g', &nlen, P_WRITE);
  return(make_string_with_length(sc, res, nlen));
}

static s7_pointer number_to_string_p_p(s7_scheme *sc, s7_pointer p)
{
  s7_int nlen = 0;
  char *res;
  if (!is_number(p))
    return(wrong_type_argument_with_type(sc, sc->number_to_string_symbol, 1, p, a_number_string));
  res = number_to_string_base_10(sc, p, 0, sc->float_format_precision, 'g', &nlen, P_WRITE);
  return(make_string_with_length(sc, res, nlen));
}

static s7_pointer number_to_string_p_pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  s7_int nlen = 0, radix;
  char *res;
  s7_pointer p;
  if (!is_number(p1))
    return(wrong_type_argument_with_type(sc, sc->number_to_string_symbol, 1, p1, a_number_string));
  if (!is_integer(p2))
    return(wrong_type_argument(sc, sc->number_to_string_symbol, 2, p2, T_INTEGER));
  radix = s7_integer(p2);
  if ((radix < 2) || (radix > 16))
    return(out_of_range(sc, sc->number_to_string_symbol, small_int(2), p2, a_valid_radix_string));
  res = number_to_string_with_radix(sc, p1, radix, 0, sc->float_format_precision, 'g', &nlen);
  p = make_string_with_length(sc, res, nlen);
  free(res);
  return(p);
}


/* -------------------------------------------------------------------------------- */
#define CTABLE_SIZE 256
static bool *exponent_table, *slashify_table, *char_ok_in_a_name, *white_space, *number_table, *symbol_slashify_table, *char_0;
static int32_t *digits;

static void init_ctables(void)
{
  int32_t i;

  exponent_table = (bool *)calloc(CTABLE_SIZE, sizeof(bool));
  slashify_table = (bool *)calloc(CTABLE_SIZE, sizeof(bool));
  symbol_slashify_table = (bool *)calloc(CTABLE_SIZE, sizeof(bool));
  char_ok_in_a_name = (bool *)calloc(CTABLE_SIZE, sizeof(bool));
  char_0 = (bool *)calloc(CTABLE_SIZE, sizeof(bool));
  white_space = (bool *)calloc(CTABLE_SIZE + 1, sizeof(bool));
  white_space++;      /* leave white_space[-1] false for white_space[EOF] */
  number_table = (bool *)calloc(CTABLE_SIZE, sizeof(bool));
  digits = (int32_t *)calloc(CTABLE_SIZE, sizeof(int32_t));

  for (i = 0; i < CTABLE_SIZE; i++)
    {
      char_ok_in_a_name[i] = true;
      white_space[i] = false;
      digits[i] = 256;
      number_table[i] = false;
    }

  char_ok_in_a_name[0] = false;
  char_ok_in_a_name[(uint8_t)'('] = false;  /* cast for C++ */
  char_ok_in_a_name[(uint8_t)')'] = false;
  char_ok_in_a_name[(uint8_t)';'] = false;
  char_ok_in_a_name[(uint8_t)'\t'] = false;
  char_ok_in_a_name[(uint8_t)'\n'] = false;
  char_ok_in_a_name[(uint8_t)'\r'] = false;
  char_ok_in_a_name[(uint8_t)' '] = false;
  char_ok_in_a_name[(uint8_t)'"'] = false;
  memcpy((void *)char_0, (void *)char_ok_in_a_name, CTABLE_SIZE * sizeof(bool));
  /* what about stuff like vertical tab?  or comma? */

  char_0[(uint8_t)'#'] = false;
  char_0[(uint8_t)'\''] = false;
  char_0[(uint8_t)'`'] = false;
  char_0[(uint8_t)','] = false;
  char_0[(uint8_t)';'] = false;

  white_space[(uint8_t)'\t'] = true;
  white_space[(uint8_t)'\n'] = true;
  white_space[(uint8_t)'\r'] = true;
  white_space[(uint8_t)'\f'] = true;
  white_space[(uint8_t)'\v'] = true;
  white_space[(uint8_t)' '] = true;
  white_space[(uint8_t)'\205'] = true; /* 133 */
  white_space[(uint8_t)'\240'] = true; /* 160 */

  /* surely only 'e' is needed... */
  exponent_table[(uint8_t)'e'] = true; exponent_table[(uint8_t)'E'] = true;
  exponent_table[(uint8_t)'@'] = true;
#if WITH_EXTRA_EXPONENT_MARKERS
  exponent_table[(uint8_t)'s'] = true; exponent_table[(uint8_t)'S'] = true;
  exponent_table[(uint8_t)'f'] = true; exponent_table[(uint8_t)'F'] = true;
  exponent_table[(uint8_t)'d'] = true; exponent_table[(uint8_t)'D'] = true;
  exponent_table[(uint8_t)'l'] = true; exponent_table[(uint8_t)'L'] = true;
#endif

  for (i = 0; i < 32; i++)
    slashify_table[i] = true;
  for (i = 127; i < 160; i++)
    slashify_table[i] = true;
  slashify_table[(uint8_t)'\\'] = true;
  slashify_table[(uint8_t)'"'] = true;
  slashify_table[(uint8_t)'\n'] = false;

  for (i = 0; i < CTABLE_SIZE; i++)
    symbol_slashify_table[i] = ((slashify_table[i]) || (!char_ok_in_a_name[i])); /* force use of (symbol ...) for cases like '(ab) as symbol */

  digits[(uint8_t)'0'] = 0; digits[(uint8_t)'1'] = 1; digits[(uint8_t)'2'] = 2; digits[(uint8_t)'3'] = 3; digits[(uint8_t)'4'] = 4;
  digits[(uint8_t)'5'] = 5; digits[(uint8_t)'6'] = 6; digits[(uint8_t)'7'] = 7; digits[(uint8_t)'8'] = 8; digits[(uint8_t)'9'] = 9;
  digits[(uint8_t)'a'] = 10; digits[(uint8_t)'A'] = 10;
  digits[(uint8_t)'b'] = 11; digits[(uint8_t)'B'] = 11;
  digits[(uint8_t)'c'] = 12; digits[(uint8_t)'C'] = 12;
  digits[(uint8_t)'d'] = 13; digits[(uint8_t)'D'] = 13;
  digits[(uint8_t)'e'] = 14; digits[(uint8_t)'E'] = 14;
  digits[(uint8_t)'f'] = 15; digits[(uint8_t)'F'] = 15;

  number_table[(uint8_t)'0'] = true; number_table[(uint8_t)'1'] = true; number_table[(uint8_t)'2'] = true; number_table[(uint8_t)'3'] = true;
  number_table[(uint8_t)'4'] = true; number_table[(uint8_t)'5'] = true; number_table[(uint8_t)'6'] = true; number_table[(uint8_t)'7'] = true;
  number_table[(uint8_t)'8'] = true; number_table[(uint8_t)'9'] = true; number_table[(uint8_t)'.'] = true;
  number_table[(uint8_t)'+'] = true;
  number_table[(uint8_t)'-'] = true;
  number_table[(uint8_t)'#'] = true;
}

#define is_white_space(C) white_space[C]
  /* this is much faster than C's isspace, and does not depend on the current locale.
   * if c == EOF (-1), it indexes into the empty (0) slot we preallocated below white_space
   */

/* -------------------------------- *#readers* -------------------------------- */
static s7_pointer check_sharp_readers(s7_scheme *sc, const char *name)
{
  s7_pointer reader, value, args;
  bool need_loader_port;
  value = sc->F;
  args = sc->F;

  /* *#reader* is assumed to be an alist of (char . proc)
   *    where each proc takes one argument, the string from just beyond the "#" to the next delimiter.
   *    The procedure can call read-char to read ahead in the current-input-port.
   *    If it returns anything other than #f, that is the value of the sharp expression.
   *    Since #f means "nothing found", it is tricky to handle #F:
   *       (cons #\F (lambda (str) (and (string=? str "F") (list 'not #t))))
   * This search happens after #|, #t, and #f (and #nD for multivectors?). #! has a fallback.
   */

  need_loader_port = is_loader_port(sc->input_port);
  if (need_loader_port)
    clear_loader_port(sc->input_port);

  /* normally read* can't read from sc->input_port if it is in use by the loader, but here we are deliberately making that possible. */
  for (reader = slot_value(sc->sharp_readers); is_not_null(reader); reader = cdr(reader))
    {
      if (name[0] == s7_character(caar(reader)))
	{
	  if (args == sc->F)
	    args = set_plist_1(sc, s7_make_string_wrapper(sc, name)); /* was list_1(sc, make_string(sc, name)) 16-Nov-18 */
	  /* args is GC protected by s7_apply_function?? (placed on the stack) */
	  value = s7_apply_function(sc, cdar(reader), args); /* this is much less error-safe than s7_call */
	  if (value != sc->F)
	    break;
	}
    }
  if (need_loader_port)
    set_loader_port(sc->input_port);
  return(value);
}

static s7_pointer g_sharp_readers_set(s7_scheme *sc, s7_pointer args)
{
  /* new value must be either () or a proper list of conses (char . func) */
  if (is_null(cadr(args))) return(cadr(args));
  if (is_pair(cadr(args)))
    {
      s7_pointer x;
      for (x = cadr(args); is_pair(x); x = cdr(x))
	{
	  if ((!is_pair(car(x))) ||
	      (!s7_is_character(caar(x))) ||
	      (!s7_is_procedure(cdar(x))))
	    return(s7_error(sc, sc->error_symbol, set_elist_2(sc, wrap_string(sc, "can't set *#readers* to ~S", 26), cadr(args))));
	}
      if (is_null(x))
	return(cadr(args));
    }
  return(s7_error(sc, sc->error_symbol, set_elist_2(sc, wrap_string(sc, "can't set *#readers* to ~S", 26), cadr(args))));
}


static bool is_abnormal(s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER:
    case T_RATIO:
      return(false);

    case T_REAL:
      return(is_inf(real(x)) ||
	     is_NaN(real(x)));

    case T_COMPLEX:
      return(((is_inf(s7_real_part(x)))  ||
	      (is_inf(s7_imag_part(x)))  ||
	      (is_NaN(s7_real_part(x))) ||
	      (is_NaN(s7_imag_part(x)))));

#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:
      return(false);

    case T_BIG_REAL:
      return((is_inf(s7_real_part(x))) ||
	     (is_NaN(s7_real_part(x))));

    case T_BIG_COMPLEX:
      return((is_inf(s7_real_part(x))) ||
	     (is_inf(s7_imag_part(x))) ||
	     (is_NaN(s7_real_part(x))) ||
	     (is_NaN(s7_imag_part(x))));
#endif

    default:
      return(true);
    }
}

static s7_pointer make_unknown(s7_scheme *sc, const char* name)
{
  s7_pointer p;
  char *newstr;
  s7_int len;
  new_cell(sc, p, T_UNDEFINED | T_IMMUTABLE);
  len = safe_strlen(name);
  newstr = (char *)malloc((len + 2) * sizeof(char)); /* this is a non-permanent unknown */
  newstr[0] = '#';
  if (len > 0)
    memcpy((void *)(newstr + 1), (void *)name, len);
  newstr[len + 1] = '\0';
  unique_name_length(p) = len + 1;
  unknown_name(p) = newstr;
  add_unknown(sc, p);
  return(p);
}

static s7_pointer unknown_sharp_constant(s7_scheme *sc, char *name)
{
  /* check *read-error-hook* */
  if (hook_has_functions(sc->read_error_hook))
    {
      s7_pointer result;
      /* see sc->error_hook for a more robust way to handle this */
      result = s7_call(sc, sc->read_error_hook, list_2(sc, sc->T, s7_make_string_wrapper(sc, name)));
      if (result != sc->unspecified)
	return(result);
    }
  /* old: return(sc->nil); */
  return(make_unknown(sc, name));
}

#define SYMBOL_OK true
#define NO_SYMBOLS false
static s7_pointer *chars;

static s7_pointer make_sharp_constant(s7_scheme *sc, char *name, bool with_error)
{
  /* name is the stuff after the '#', return sc->nil if not a recognized #... entity */
  s7_pointer x;

  if ((name[0] == 't') &&
      ((name[1] == '\0') || (strings_are_equal(name, "true"))))
    return(sc->T);

  if ((name[0] == 'f') &&
      ((name[1] == '\0') || (strings_are_equal(name, "false"))))
    return(sc->F);

  if (is_not_null(slot_value(sc->sharp_readers)))
    {
      x = check_sharp_readers(sc, name);
      if (x != sc->F)
	return(x);
    }

  if ((name[0] == '\0') || name[1] == '\0')
    return(unknown_sharp_constant(sc, name));

  switch (name[0])
    {
      /* -------- #< ... > -------- */
    case '<':
      if (strings_are_equal(name, "<unspecified>"))
	return(sc->unspecified);

      if (strings_are_equal(name, "<undefined>"))
	return(sc->undefined);

      if (strings_are_equal(name, "<eof>"))
	return(eof_object);

      return(unknown_sharp_constant(sc, name));

      /* -------- #o #x #b -------- */
    case 'o':   /* #o (octal) */
    case 'x':   /* #x (hex) */
    case 'b':   /* #b (binary) */
      {
	int32_t num_at = 1;
	/* the #b or whatever overrides any radix passed in earlier */
	x = make_atom(sc, (char *)(name + num_at), (name[0] == 'o') ? 8 : ((name[0] == 'x') ? 16 : 2), NO_SYMBOLS, with_error);
	if (is_abnormal(x))
	  return(unknown_sharp_constant(sc, name));
	return(x);
      }

      /* -------- #_... -------- */
    case '_':
      {
	s7_pointer sym;
	sym = make_symbol(sc, (char *)(name + 1));
	if ((!is_gensym(sym)) && (is_slot(initial_slot(sym))))
	  return(slot_value(initial_slot(sym)));

	/* here we should not necessarily raise an error that *_... is undefined.  reader-cond, for example, needs to
	 *    read undefined #_ vals that it will eventually discard.
	 */
	return(make_unknown(sc, name));    /* (define x (with-input-from-string "(#_asdf 1 2)" read)) (type-of (car x)) -> undefined? */
      }

      /* -------- #\... -------- */
    case '\\':
      if (name[2] == 0)                             /* the most common case: #\a */
	return(chars[(uint8_t)(name[1])]);
      /* not uint32_t here!  (uint32_t)255 (as a char) returns -1!! */
      switch (name[1])
	{
	case 'n':
	  if ((strings_are_equal(name + 1, "null")) ||
	      (strings_are_equal(name + 1, "nul")))
	    return(chars[0]);

	  if (strings_are_equal(name + 1, "newline"))
	    return(chars[(uint8_t)'\n']);
	  break;

	case 's':
	  if (strings_are_equal(name + 1, "space"))
	    return(chars[(uint8_t)' ']);
	  break;

	case 'r':
	  if (strings_are_equal(name + 1, "return"))
	    return(chars[(uint8_t)'\r']);
	  break;

	case 'l':
	  if (strings_are_equal(name + 1, "linefeed"))
	    return(chars[(uint8_t)'\n']);
	  break;

	case 't':
	  if (strings_are_equal(name + 1, "tab"))
	    return(chars[(uint8_t)'\t']);
	  break;

	case 'a':
	  /* the next 4 are for r7rs */
	  if (strings_are_equal(name + 1, "alarm"))
	    return(chars[7]);
	  break;

	case 'b':
	  if (strings_are_equal(name + 1, "backspace"))
	    return(chars[8]);
	  break;

	case 'e':
	  if (strings_are_equal(name + 1, "escape"))
	    return(chars[0x1b]);
	  break;

	case 'd':
	  if (strings_are_equal(name + 1, "delete"))
	    return(chars[0x7f]);
	  break;

	case 'x':
	  /* #\x is just x, but apparently #\x<num> is int->char? #\x65 -> #\e, and #\xcebb is lambda? */
	  {
	    /* sscanf here misses errors like #\x1.4, but make_atom misses #\x6/3,
	     *   #\x#b0, #\x#e0.0, #\x-0, #\x#e0e100 etc, so we have to do it at an even lower level.
	     * another problem: #\xbdca2cbec overflows so lval is -593310740 -> segfault unless caught
	     */
	    bool happy = true;
	    char *tmp;
	    int32_t lval = 0;

	    tmp = (char *)(name + 2);
	    while ((*tmp) && (happy) && (lval >= 0) && (lval < 256))
	      {
		int32_t dig;
		dig = digits[(int32_t)(*tmp++)];
		if (dig < 16)
		  lval = dig + (lval * 16);
		else happy = false;
	      }
	    if ((happy) &&
		(lval < 256) &&
		(lval >= 0))
	      return(chars[lval]);
	  }
	  break;
	}
    }
  return(unknown_sharp_constant(sc, name));
}


static s7_int string_to_integer(const char *str, s7_int radix, bool *overflow)
{
  bool negative = false;
  s7_int lval = 0;
  int32_t dig;
  char *tmp = (char *)str;
#if WITH_GMP
  char *tmp1;
#endif

  if (str[0] == '+')
    tmp++;
  else
    {
      if (str[0] == '-')
	{
	  negative = true;
	  tmp++;
	}
    }
  while (*tmp == '0') {tmp++;};
#if WITH_GMP
  tmp1 = tmp;
#endif

 if (radix == 10)
    {
      while (true)
	{
	  dig = digits[(uint8_t)(*tmp++)];
	  if (dig > 9) break;
#if HAVE_OVERFLOW_CHECKS
	  if ((multiply_overflow(lval, (s7_int)10, &lval)) ||
	      (add_overflow(lval, (s7_int)dig, &lval)))
	    {
	      if ((lval == s7_int_min) && (digits[(uint8_t)(*tmp++)] > 9))
		return(lval);
	      *overflow = true;
	      break;
	    }
#else
	  lval = dig + (lval * 10);
	  dig = digits[(uint8_t)(*tmp++)];
	  if (dig > 9) break;
	  lval = dig + (lval * 10);
#endif
	}
    }
  else
    {
      while (true)
	{
	  dig = digits[(uint8_t)(*tmp++)];
	  if (dig >= radix) break;
#if HAVE_OVERFLOW_CHECKS && (!WITH_GMP)
	  {
	    s7_int oval = 0;
	    if (multiply_overflow(lval, (s7_int)radix, &oval))
	      {
		/* maybe a bad idea!  #xffffffffffffffff -> -1??? this is needed for 64-bit number hacks (see s7test.scm bit-reverse) */
		if ((radix == 16) &&
		    (digits[(uint8_t)(*tmp)] >= radix))
		  {
		    lval -= 576460752303423488LL; /* turn off sign bit */
		    lval *= radix;
		    lval += dig;
		    lval -= 9223372036854775807LL;
		    return(lval - 1);
		  }
		else lval = oval; /* old case */
		if ((lval == s7_int_min)  && (digits[(uint8_t)(*tmp++)] > 9))
		  return(lval);
		*overflow = true;
		break;
	      }
	    else lval = oval;
	    if (add_overflow(lval, (s7_int)dig, &lval))
	      {
		if (lval == s7_int_min) return(lval);
		*overflow = true;
		break;
	      }
	  }
#else
	  lval = dig + (lval * radix);
	  dig = digits[(uint8_t)(*tmp++)];
	  if (dig >= radix) break;
	  lval = dig + (lval * radix);
#endif
	}
    }

#if WITH_GMP
  (*overflow) = ((lval > s7_int32_max) ||
		 ((tmp - tmp1) > s7_int_digits_by_radix[radix]));
  /* this tells the string->number readers to create a bignum.  We need to be very
   *    conservative here to catch contexts such as (/ 1/524288 19073486328125)
   */
#endif

  if (negative) return(-lval);
  return(lval);
}

/*  9223372036854775807                9223372036854775807
 * -9223372036854775808               -9223372036854775808
 * 0000000000000000000000000001.0     1.0
 * 1.0000000000000000000000000000     1.0
 * 1000000000000000000000000000.0e-40 1.0e-12
 * 0.0000000000000000000000000001e40  1.0e12
 * 1.0e00000000000000000001           10.0
 */

#if WITH_GMP
static s7_double string_to_double_with_radix(const char *ur_str, s7_int rad, bool *overflow)
#else
#define string_to_double_with_radix(Str, Rad, Over) string_to_double_with_radix_1(Str, Rad)
static s7_double string_to_double_with_radix_1(const char *ur_str, s7_int rad)
#endif
{
  /* strtod follows LANG which is not what we want (only "." is decimal point in Scheme).
   *   To overcome LANG in strtod would require screwing around with setlocale which never works.
   *   So we use our own code -- according to valgrind, this function is much faster than strtod.
   *
   * comma as decimal point causes ambiguities: `(+ ,1 2) etc
   */

  int32_t i, sign = 1, frac_len, int_len, dig, max_len, exponent = 0, radix;
  int64_t int_part = 0, frac_part = 0;
  char *str;
  char *ipart, *fpart;
  s7_double dval = 0.0;
  radix = (int32_t)rad;

  /* there's an ambiguity in number notation here if we allow "1e1" or "1.e1" in base 16 (or 15) -- is e a digit or an exponent marker?
   *   but 1e+1, for example disambiguates it -- kind of messy! -- the scheme spec says "e" can only occur in base 10.
   *   mpfr says "e" as exponent only in bases <= 10 -- else use '@' which works in any base.  This can only cause confusion
   *   in scheme, unfortunately, due to the idiotic scheme polar notation.  But we accept "s" and "l" as exponent markers
   *   so, perhaps for radix > 10, the exponent, if any, has to use one of S s L l?  Not "l"!  And "s" originally meant "short".
   *
   * '@' can now be used as the exponent marker (26-Mar-12).
   * Another slight ambiguity: 1+1/2i is parsed as 1 + 0.5i, not 1+1/(2i), or (1+1)/(2i) or (1+1/2)i etc
   */

  max_len = s7_int_digits_by_radix[radix];
  str = (char *)ur_str;

  if (*str == '+')
    str++;
  else
    {
      if (*str == '-')
	{
	  str++;
	  sign = -1;
	}
    }
  while (*str == '0') {str++;};

  ipart = str;
  while (digits[(int32_t)(*str)] < radix) str++;
  int_len = str - ipart;

  if (*str == '.') str++;
  fpart = str;
  while (digits[(int32_t)(*str)] < radix) str++;
  frac_len = str - fpart;

  if ((*str) && (exponent_table[(uint8_t)(*str)]))
    {
      int32_t exp_negative = false;
      str++;
      if (*str == '+')
	str++;
      else
	{
	  if (*str == '-')
	    {
	      str++;
	      exp_negative = true;
	    }
	}
      while ((dig = digits[(int32_t)(*str++)]) < 10) /* exponent itself is always base 10 */
	{
#if HAVE_OVERFLOW_CHECKS
	  if ((int32_multiply_overflow(exponent, 10, &exponent)) ||
	      (int32_add_overflow(exponent, dig, &exponent)))
	    {
	      exponent = 1000000; /* see below */
	      break;
	    }
#else
	  exponent = dig + (exponent * 10);
#endif
	}
#if (!defined(__GNUC__)) || (__GNUC__ < 5)
      if (exponent < 0)         /* we overflowed, so make sure we notice it below (need to check for 0.0e... first) (Brian Damgaard) */
	exponent = 1000000;     /*   see below for examples -- this number needs to be very big but not too big for add */
#endif
      if (exp_negative)
	exponent = -exponent;

      /*           2e12341234123123123123213123123123 -> 0.0
       * but exp len is not the decider: 2e00000000000000000000000000000000000000001 -> 20.0
       * first zero: 2e123412341231231231231
       * then:     2e12341234123123123123123123 -> inf
       * then:     2e123412341231231231231231231231231231 -> 0.0
       *           2e-123412341231231231231 -> inf
       * but:      0e123412341231231231231231231231231231
       */
    }

#if WITH_GMP
  /* 9007199254740995.0 */
  if (int_len + frac_len >= max_len)
    {
      (*overflow) = true;
      return(0.0);
    }
#endif

  str = ipart;
  if ((int_len + exponent) > max_len)
    {
      /*  12341234.56789e12                   12341234567889999872.0              1.234123456789e+19
       * -1234567890123456789.0              -1234567890123456768.0              -1.2345678901235e+18
       *  12345678901234567890.0              12345678901234567168.0              1.2345678901235e+19
       *  123.456e30                          123456000000000012741097792995328.0 1.23456e+32
       *  12345678901234567890.0e12           12345678901234569054409354903552.0  1.2345678901235e+31
       *  1.234567890123456789012e30          1234567890123456849145940148224.0   1.2345678901235e+30
       *  1e20                                100000000000000000000.0             1e+20
       *  1234567890123456789.0               1234567890123456768.0               1.2345678901235e+18
       *  123.456e16                          1234560000000000000.0               1.23456e+18
       *  98765432101234567890987654321.0e-5  987654321012345728401408.0          9.8765432101235e+23
       *  98765432101234567890987654321.0e-10 9876543210123456512.0               9.8765432101235e+18
       *  0.00000000000000001234e20           1234.0
       *  0.000000000000000000000000001234e30 1234.0
       *  0.0000000000000000000000000000000000001234e40 1234.0
       *  0.000000000012345678909876543210e15 12345.678909877
       *  0e1000                              0.0
       */

      for (i = 0; i < max_len; i++)
	{
	  dig = digits[(int32_t)(*str++)];
	  if (dig < radix)
	    int_part = dig + (int_part * radix);
	  else break;
	}

      /* if the exponent is huge, check for 0 int_part and frac_part before complaining (0e1000 or 0.0e1000)
       */
      if ((int_part == 0) &&
	  (exponent > max_len))
	{
	  /* if frac_part is also 0, return 0.0 */
	  if (frac_len == 0)
	    return(0.0);

	  str = fpart;
	  while ((dig = digits[(int32_t)(*str++)]) < radix)
	    frac_part = dig + (frac_part * radix);
	  if (frac_part == 0)
	    return(0.0);

#if WITH_GMP
	  (*overflow) = true;
#endif
	}

#if WITH_GMP
      (*overflow) = ((int_part > 0) || (exponent > 20));    /* .1e310 is a tricky case */
#endif

      if (int_part != 0) /* 0.<310 zeros here>1e310 for example --
			  *   pow (via ipow) thinks it has to be too big, returns Nan,
			  *   then Nan * 0 -> Nan and the NaN propagates
			  */
	{
	  if (int_len <= max_len)
	    dval = int_part * ipow(radix, exponent);
	  else dval = int_part * ipow(radix, exponent + int_len - max_len);
	}
      else dval = 0.0;

      /* shift by exponent, but if int_len > max_len then we assumed (see below) int_len - max_len 0's on the left */
      /*   using int_to_int or table lookups here instead of pow did not make any difference in speed */

      if (int_len < max_len)
	{
	  int32_t k, flen;
	  str = fpart;

	  for (k = 0; (frac_len > 0) && (k < exponent); k += max_len)
	    {
	      if (frac_len > max_len) flen = max_len; else flen = frac_len;
	      frac_len -= max_len;

	      frac_part = 0;
	      for (i = 0; i < flen; i++)
		frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);

	      if (frac_part != 0)                                /* same pow->NaN problem as above can occur here */
		dval += frac_part * ipow(radix, exponent - flen - k);
	    }
	}
      else
	{
	  /* some of the fraction is in the integer part before the negative exponent shifts it over */
	  if (int_len > max_len)
	    {
	      int32_t ilen;
	      /* str should be at the last digit we read */
	      ilen = int_len - max_len;                          /* we read these above */
	      if (ilen > max_len)
		ilen = max_len;

	      for (i = 0; i < ilen; i++)
		frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);

	      dval += frac_part * ipow(radix, exponent - ilen);
	    }
	}

      return(sign * dval);
    }

  /* int_len + exponent <= max_len */

  if (int_len <= max_len)
    {
      int32_t int_exponent;

      /* a better algorithm (since the inaccuracies are in the radix^exponent portion):
       *   strip off leading zeros and possible sign,
       *   strip off digits beyond max_len, then remove any trailing zeros.
       *     (maybe fiddle with the lowest order digit here for rounding, but I doubt it matters)
       *   read digits until end of number or max_len reached, ignoring the decimal point
       *   get exponent and use it and decimal point location to position the current result integer
       * this always combines the same integer and the same exponent no matter how the number is expressed.
       */

      int_exponent = exponent;
      if (int_len > 0)
	{
	  char *iend;
	  iend = (char *)(str + int_len - 1);
	  while ((*iend == '0') && (iend != str)) {iend--; int_exponent++;}

	  while (str <= iend)
	    int_part = digits[(int32_t)(*str++)] + (int_part * radix);
	}
      if (int_exponent != 0)
	dval = int_part * ipow(radix, int_exponent);
      else dval = (s7_double)int_part;
    }
  else
    {
      int32_t len, flen;
      int64_t frpart = 0;

      /* 98765432101234567890987654321.0e-20    987654321.012346
       * 98765432101234567890987654321.0e-29    0.98765432101235
       * 98765432101234567890987654321.0e-30    0.098765432101235
       * 98765432101234567890987654321.0e-28    9.8765432101235
       */

      len = int_len + exponent;
      for (i = 0; i < len; i++)
	int_part = digits[(int32_t)(*str++)] + (int_part * radix);

      flen = -exponent;
      if (flen > max_len)
	flen = max_len;

      for (i = 0; i < flen; i++)
	frpart = digits[(int32_t)(*str++)] + (frpart * radix);

      if (len <= 0)
	dval = int_part + frpart * ipow(radix, len - flen);
      else dval = int_part + frpart * ipow(radix, -flen);
    }

  if (frac_len > 0)
    {
      str = fpart;
      if (frac_len <= max_len)
	{
	  /* splitting out base 10 case saves very little here */
	  /* this ignores trailing zeros, so that 0.3 equals 0.300 */
	  char *fend;

	  fend = (char *)(str + frac_len - 1);
	  while ((*fend == '0') && (fend != str)) {fend--; frac_len--;} /* (= .6 0.6000) */

	  while (str <= fend)
	    frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);
	  dval += frac_part * ipow(radix, exponent - frac_len);

	  /* 0.6:    frac:    6, exp: 0.10000000000000000555, val: 0.60000000000000008882
	   * 0.60:   frac:   60, exp: 0.01000000000000000021, val: 0.59999999999999997780
	   * 0.6000: frac: 6000, exp: 0.00010000000000000000, val: 0.59999999999999997780
	   * :(= 0.6 0.60)
	   * #f
	   * :(= #i3/5 0.6)
	   * #f
	   * so (string->number (number->string num)) == num only if both num's are the same text (or you get lucky)
	   * :(= 0.6 6e-1) ; but not 60e-2
	   * #t
	   *
	   * to fix the 0.60 case, we need to ignore trailing post-dot zeros.
	   */
	}
      else
	{
	  if (exponent <= 0)
	    {
	      for (i = 0; i < max_len; i++)
		frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);

	      dval += frac_part * ipow(radix, exponent - max_len);
	    }
	  else
	    {
	      /* 1.0123456789876543210e1         10.12345678987654373771
	       * 1.0123456789876543210e10        10123456789.87654304504394531250
	       * 0.000000010000000000000000e10   100.0
	       * 0.000000010000000000000000000000000000000000000e10 100.0
	       * 0.000000012222222222222222222222222222222222222e10 122.22222222222222
	       * 0.000000012222222222222222222222222222222222222e17 1222222222.222222
	       */

	      int_part = 0;
	      for (i = 0; i < exponent; i++)
		int_part = digits[(int32_t)(*str++)] + (int_part * radix);

	      frac_len -= exponent;
	      if (frac_len > max_len)
		frac_len = max_len;

	      for (i = 0; i < frac_len; i++)
		frac_part = digits[(int32_t)(*str++)] + (frac_part * radix);

	      dval += int_part + frac_part * ipow(radix, -frac_len);
	    }
	}
    }

#if WITH_GMP
  if ((int_part == 0) &&
      (frac_part == 0))
    return(0.0);
  (*overflow) = ((frac_len - exponent) > max_len);
#endif

  return(sign * dval);
}


static s7_pointer make_atom(s7_scheme *sc, char *q, s7_int radix, bool want_symbol, bool with_error)
{
  /* make symbol or number from string */
  #define IS_DIGIT(Chr, Rad) (digits[(uint8_t)Chr] < Rad)

  char c, *p;
  bool has_dec_point1 = false;

  p = q;
  c = *p++;

  /* a number starts with + - . or digit, but so does 1+ for example (and there's also nan.0 and inf.0) */

  switch (c)
    {
    case '#':
      return(make_sharp_constant(sc, p, with_error)); /* make_sharp_constant expects the '#' to be removed */

    case '+':
    case '-':
      c = *p++;
      if (c == '.')
	{
	  has_dec_point1 = true;
	  c = *p++;
	}
      if (!c)
	return((want_symbol) ? make_symbol(sc, q) : sc->F);
      if (!IS_DIGIT(c, radix))
	{
	  if (c == 'n')
	    {
	      if (local_strcmp(p, "an.0"))
		return(real_NaN);
	    }
	  else
	    {
	      if (c == 'i')
		{
		  if (local_strcmp(p, "nf.0"))
		    return((q[0] == '+') ? real_infinity : real_minus_infinity);
		}
	    }
	  return((want_symbol) ? make_symbol(sc, q) : sc->F);
	}
      break;

    case '.':
      has_dec_point1 = true;
      c = *p++;

      if ((!c) || (!IS_DIGIT(c, radix)))
	return((want_symbol) ? make_symbol(sc, q) : sc->F);
      break;

    case '0':        /* these two are always digits */
    case '1':
      break;

    default:
      if (!IS_DIGIT(c, radix))
	return((want_symbol) ? make_symbol(sc, q) : sc->F);
      break;
    }

  /* now it's possibly a number -- the first character(s) could be part of a number in the current radix */
  {
    char *slash1 = NULL, *slash2 = NULL, *plus = NULL, *ex1 = NULL, *ex2 = NULL;
    bool has_i = false, has_dec_point2 = false;
    int32_t has_plus_or_minus = 0, current_radix;
#if (!WITH_GMP)
    bool overflow = false; /* for string_to_integer */
#endif
    current_radix = radix;  /* current_radix is 10 for the exponent portions, but radix for all the rest */

    for ( ; (c = *p) != 0; ++p)
      {
	/* what about embedded null? (string->number (string #\1 (integer->char 0) #\0))
	 *   currently we stop and return 1, but Guile returns #f
	 */
	if (!IS_DIGIT(c, current_radix))         /* moving this inside the switch statement was much slower */
	  {
	    current_radix = radix;

	    switch (c)
	      {
		/* -------- decimal point -------- */
	      case '.':
		if ((!IS_DIGIT(p[1], current_radix)) &&
		    (!IS_DIGIT(p[-1], current_radix)))
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if (has_plus_or_minus == 0)
		  {
		    if ((has_dec_point1) || (slash1))
		      return((want_symbol) ? make_symbol(sc, q) : sc->F);
		    has_dec_point1 = true;
		  }
		else
		  {
		    if ((has_dec_point2) || (slash2))
		      return((want_symbol) ? make_symbol(sc, q) : sc->F);
		    has_dec_point2 = true;
		  }
		continue;


		/* -------- exponent marker -------- */
#if WITH_EXTRA_EXPONENT_MARKERS
		/* 1st 3d-perspective 0.0f 128.0f 3d 1s -- in 2 million lines of public scheme code, not one actual use! */
	      case 's': case 'S':
	      case 'd': case 'D':
	      case 'f': case 'F':
	      case 'l': case 'L':
#endif
	      case 'e': case 'E':
		if (current_radix > 10)
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);
		/* see note above */
		/* fall through -- if '@' used, radices>10 are ok */

	      case '@':
		current_radix = 10;

		if (((ex1) ||
		     (slash1)) &&
		    (has_plus_or_minus == 0)) /* ee */
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if (((ex2) ||
		     (slash2)) &&
		    (has_plus_or_minus != 0)) /* 1+1.0ee */
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if ((!IS_DIGIT(p[-1], radix)) && /* was current_radix but that's always 10! */
		    (p[-1] != '.'))
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if (has_plus_or_minus == 0)
		  {
		    ex1 = p;
		    has_dec_point1 = true; /* decimal point illegal from now on */
		  }
		else
		  {
		    ex2 = p;
		    has_dec_point2 = true;
		  }
		p++;
		if ((*p == '-') || (*p == '+')) p++;
		if (IS_DIGIT(*p, current_radix))
		  continue;
		break;


		/* -------- internal + or - -------- */
	      case '+':
	      case '-':
		if (has_plus_or_minus != 0) /* already have the separator */
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if (c == '+') has_plus_or_minus = 1; else has_plus_or_minus = -1;
		plus = (char *)(p + 1);
		continue;

		/* ratio marker */
	      case '/':
		if ((has_plus_or_minus == 0) &&
		    ((ex1) ||
		     (slash1) ||
		     (has_dec_point1)))
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if ((has_plus_or_minus != 0) &&
		    ((ex2) ||
		     (slash2) ||
		     (has_dec_point2)))
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		if (has_plus_or_minus == 0)
		  slash1 = (char *)(p + 1);
		else slash2 = (char *)(p + 1);

		if ((!IS_DIGIT(p[1], current_radix)) ||
		    (!IS_DIGIT(p[-1], current_radix)))
		  return((want_symbol) ? make_symbol(sc, q) : sc->F);

		continue;


		/* -------- i for the imaginary part -------- */
	      case 'i':
		if ((has_plus_or_minus != 0) &&
		    (!has_i))
		  {
		    has_i = true;
		    continue;
		  }
		break;

	      default:
		break;
	      }
	    return((want_symbol) ? make_symbol(sc, q) : sc->F);
	  }
      }

    if ((has_plus_or_minus != 0) &&        /* that is, we have an internal + or - */
	(!has_i))                          /*   but no i for the imaginary part */
      return((want_symbol) ? make_symbol(sc, q) : sc->F);

    if (has_i)
      {
#if (!WITH_GMP)
	s7_double rl = 0.0, im = 0.0;
#else
	char e1 = 0, e2 = 0;
#endif
	s7_pointer result;
	s7_int len;
	char ql1, pl1;

	len = safe_strlen(q);

	if (q[len - 1] != 'i')
	  return((want_symbol) ? make_symbol(sc, q) : sc->F);

	/* save original string */
	ql1 = q[len - 1];
	pl1 = (*(plus - 1));
#if WITH_GMP
	if (ex1) {e1 = *ex1; (*ex1) = '@';} /* for mpfr */
	if (ex2) {e2 = *ex2; (*ex2) = '@';}
#endif

	/* look for cases like 1+i */
	if ((q[len - 2] == '+') || (q[len - 2] == '-'))
	  q[len - 1] = '1';
	else q[len - 1] = '\0'; /* remove 'i' */

	(*((char *)(plus - 1))) = '\0';

	/* there is a slight inconsistency here:
	   1/0      -> +nan.0
           1/0+0i   -> +inf.0 (0/1+0i is 0.0)
	   #i1/0+0i -> +inf.0
	   0/0      -> +nan.0
	   0/0+0i   -> -nan.0
	*/

#if (!WITH_GMP)
	if ((has_dec_point1) ||
	    (ex1))
	  {
	    /* (string->number "1100.1+0.11i" 2) -- need to split into 2 honest reals before passing to non-base-10 str->dbl */
	    rl = string_to_double_with_radix(q, radix, ignored);
	  }
	else
	  {
	    if (slash1)
	      {
		/* here the overflow could be innocuous if it's in the denominator and the numerator is 0
		 *    0/100000000000000000000000000000000000000-0i
		 */
		s7_int num, den;
		num = string_to_integer(q, radix, &overflow);
		den = string_to_integer(slash1, radix, &overflow);
		if (den == 0)
		  rl = NAN;
		else
		  {
		    if (num == 0)
		      {
			rl = 0.0;
			overflow = false;
		      }
		    else rl = (s7_double)num / (s7_double)den;
		  }
	      }
	    else rl = (s7_double)string_to_integer(q, radix, &overflow);
	    if (overflow) return(real_NaN);
	  }
	if (rl == -0.0) rl = 0.0;

	if ((has_dec_point2) ||
	    (ex2))
	  im = string_to_double_with_radix(plus, radix, ignored);
	else
	  {
	    if (slash2)
	      {
		/* same as above: 0-0/100000000000000000000000000000000000000i
		 */
		s7_int num, den;
		num = string_to_integer(plus, radix, &overflow);
		den = string_to_integer(slash2, radix, &overflow);
		if (den == 0)
		  im = NAN;
		else
		  {
		    if (num == 0)
		      {
			im = 0.0;
			overflow = false;
		      }
		    else im = (s7_double)num / (s7_double)den;
		  }
	      }
	    else im = (s7_double)string_to_integer(plus, radix, &overflow);
	    if (overflow) return(real_NaN);
	  }
	if ((has_plus_or_minus == -1) &&
	    (im != 0.0))
	  im = -im;
	result = s7_make_complex(sc, rl, im);
#else
	result = string_to_either_complex(sc, q, slash1, ex1, has_dec_point1, plus, slash2, ex2, has_dec_point2, radix, has_plus_or_minus);
#endif

	/* restore original string */
	q[len - 1] = ql1;
	(*((char *)(plus - 1))) = pl1;
#if WITH_GMP
	if (ex1) (*ex1) = e1;
	if (ex2) (*ex2) = e2;
#endif

	return(result);
      }

    /* not complex */
    if ((has_dec_point1) ||
	(ex1))
      {
	s7_pointer result;

	if (slash1)  /* not complex, so slash and "." is not a number */
	  return((want_symbol) ? make_symbol(sc, q) : sc->F);

#if (!WITH_GMP)
	result = make_real(sc, string_to_double_with_radix(q, radix, ignored));
#else
	{
	  char old_e = 0;
	  if (ex1)
	    {
	      old_e = (*ex1);
	      (*ex1) = '@';
	    }
	  result = string_to_either_real(sc, q, radix);
	  if (ex1)
	    (*ex1) = old_e;
	}
#endif
	return(result);
      }

    /* not real */
    if (slash1)
#if (!WITH_GMP)
      {
	s7_int n, d;

	n = string_to_integer(q, radix, &overflow);
	d = string_to_integer(slash1, radix, &overflow);

	if ((n == 0) && (d != 0))                        /* 0/100000000000000000000000000000000000000 */
	  return(small_int(0));
	if ((d == 0) || (overflow))
	  return(real_NaN);
	/* it would be neat to return 1 from 10000000000000000000000000000/10000000000000000000000000000
	 *   but q is the entire number ('/' included) and slash1 is the stuff after the '/', and every
	 *   big number comes through here, so there's no clean and safe way to check that q == slash1.
	 */
	return(s7_make_ratio(sc, n, d));
      }
#else
    return(string_to_either_ratio(sc, q, slash1, radix));
#endif

    /* integer */
#if (!WITH_GMP)
    {
      s7_int x;
      x = string_to_integer(q, radix, &overflow);
      if (overflow)
	return((q[0] == '-') ? real_minus_infinity : real_infinity);
      return(make_integer(sc, x));
    }
#else
    return(string_to_either_integer(sc, q, radix));
#endif
  }
}


/* -------------------------------- string->number -------------------------------- */

static s7_pointer s7_string_to_number(s7_scheme *sc, char *str, s7_int radix)
{
  s7_pointer x;
  x = make_atom(sc, str, radix, NO_SYMBOLS, WITHOUT_OVERFLOW_ERROR);
  if (s7_is_number(x))  /* only needed because str might start with '#' and not be a number (#t for example) */
    return(x);
  return(sc->F);
}

static s7_pointer g_string_to_number_1(s7_scheme *sc, s7_pointer args, s7_pointer caller)
{
  #define H_string_to_number "(string->number str (radix 10)) converts str into a number. \
If str does not represent a number, string->number returns #f.  If 'str' has an embedded radix, \
the optional 'radix' argument is ignored: (string->number \"#x11\" 2) -> 17 not 3."
  #define Q_string_to_number s7_make_signature(sc, 3, s7_make_signature(sc, 2, sc->is_number_symbol, sc->not_symbol), sc->is_string_symbol, sc->is_integer_symbol)

  s7_int radix = 0;
  char *str;

  if (!is_string(car(args)))
    return(method_or_bust(sc, car(args), caller, args, T_STRING, 1));

  if (is_pair(cdr(args)))
    {
      s7_pointer rad;
      rad = cadr(args);
      if (!s7_is_integer(rad))
	{
	  rad = check_value_slot(sc, rad);
	  if (!s7_is_integer(rad))
	    return(wrong_type_argument(sc, caller, 2, cadr(args), T_INTEGER));
	}
      radix = s7_integer(rad);
      if ((radix < 2) ||              /* what about negative int32_t as base (Knuth), reals such as phi, and some complex like -1+i */
	  (radix > 16))               /* the only problem here is printing the number; perhaps put each digit in "()" in base 10: (123)(0)(34) */
	return(out_of_range(sc, caller, small_int(2), rad, a_valid_radix_string));
    }
  else radix = 10;

  str = (char *)string_value(car(args));
  if ((!str) || (!(*str)))
    return(sc->F);

  switch (str[0])
    {
    case 'n':
      if (safe_strcmp(str, "nan.0"))
	return(real_NaN);
      break;

    case 'i':
      if (safe_strcmp(str, "inf.0"))
	return(real_infinity);
      break;

    case '-':
      if ((str[1] == 'i') && (safe_strcmp((const char *)(str + 1), "inf.0")))
	return(real_minus_infinity);
      break;

    case '+':
      if ((str[1] == 'i') && (safe_strcmp((const char *)(str + 1), "inf.0")))
	return(real_infinity);
      break;
    }
  return(s7_string_to_number(sc, str, radix));
}

static s7_pointer g_string_to_number(s7_scheme *sc, s7_pointer args)
{
  return(g_string_to_number_1(sc, args, sc->string_to_number_symbol));
}


static bool numbers_are_eqv(s7_pointer a, s7_pointer b)
{
  if (type(a) != type(b)) /* (eqv? 1 1.0) -> #f! */
    return(false);

  switch (type(a))
    {
    case T_INTEGER:
      return((integer(a) == integer(b)));

    case T_RATIO:
      return((numerator(a) == numerator(b)) &&
	     (denominator(a) == denominator(b)));

    case T_REAL:
      if (is_NaN(real(a)))
	return(false);
      return(real(a) == real(b));

    case T_COMPLEX:
      if ((is_NaN(real_part(a))) ||
	  (is_NaN(imag_part(a))))
	return(false);
      return((real_part(a) == real_part(b)) &&
	     (imag_part(a) == imag_part(b)));

    default:
#if WITH_GMP
      if ((is_big_number(a)) || (is_big_number(b))) /* this can happen if (member bignum ...) -> memv */
	return(big_numbers_are_eqv(a, b));
#endif
      break;
    }
  return(false);
}


static bool is_rational_via_method(s7_scheme *sc, s7_pointer p)
{
  if (s7_is_rational(p))
    return(true);
  if (has_methods(p))
    {
      s7_pointer f;
      f = find_method(sc, find_let(sc, p), sc->is_rational_symbol);
      if (f != sc->undefined)
	return(is_true(sc, s7_apply_function(sc, f, cons(sc, p, sc->nil))));
    }
  return(false);
}


/* -------------------------------- abs -------------------------------- */
#if (!WITH_GMP)
static s7_pointer g_abs(s7_scheme *sc, s7_pointer args)
{
  #define H_abs "(abs x) returns the absolute value of the real number x"
  #define Q_abs s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_real_symbol)

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) < 0)
	{
	  if (integer(x) == s7_int_min)
	    return(make_integer(sc, s7_int_max));
	  return(make_integer(sc, -integer(x)));
	}
      return(x);

    case T_RATIO:
      if (numerator(x) < 0)
	{
	  if (numerator(x) == s7_int_min)
	    return(s7_make_ratio(sc, s7_int_max, denominator(x)));
	  return(make_simple_ratio(sc, -numerator(x), denominator(x)));
	}
      return(x);

    case T_REAL:
      if (is_NaN(real(x)))                  /* (abs -nan.0) -> +nan.0, not -nan.0 */
	return(real_NaN);
      if (real(x) < 0.0)
	return(make_real(sc, -real(x)));
      return(x);

    default:
      return(method_or_bust_one_arg(sc, x, sc->abs_symbol, args, T_REAL));
    }
}

static s7_double abs_d_d(s7_double x) {return((x < 0.0) ? (-x) : x);}
static s7_int abs_i_i(s7_int x) {return((x < 0.0) ? (-x) : x);}


/* -------------------------------- magnitude -------------------------------- */

static double my_hypot(double x, double y)
{
  /* according to callgrind, this is much faster than libc's hypot */
  if (x == 0.0) return(fabs(y));
  if (y == 0.0) return(fabs(x));
  if (x == y) return(1.414213562373095 * fabs(x));
  if ((is_NaN(x)) || (is_NaN(y))) return(NAN);
  return(sqrt(x * x + y * y));
}

static s7_pointer g_magnitude(s7_scheme *sc, s7_pointer args)
{
  #define H_magnitude "(magnitude z) returns the magnitude of z"
  #define Q_magnitude s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_number_symbol)
  s7_pointer x;
  x = car(args);

  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == s7_int_min)
	return(make_integer(sc, s7_int_max));
      /* (magnitude -9223372036854775808) -> -9223372036854775808
       *   same thing happens in abs, lcm and gcd: (gcd -9223372036854775808) -> -9223372036854775808
       */
      if (integer(x) < 0)
        return(make_integer(sc, -integer(x)));
      return(x);

    case T_RATIO:
      if (numerator(x) < 0)
	return(make_simple_ratio(sc, -numerator(x), denominator(x)));
      return(x);

    case T_REAL:
      if (is_NaN(real(x)))                 /* (magnitude -nan.0) -> +nan.0, not -nan.0 */
	return(real_NaN);
      if (real(x) < 0.0)
	return(make_real(sc, -real(x)));
      return(x);

    case T_COMPLEX:
      return(make_real(sc, my_hypot(imag_part(x), real_part(x))));

    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->magnitude_symbol, args, a_number_string));
    }
}

static s7_double magnitude_d_7p(s7_scheme *sc, s7_pointer p)
{
  return(s7_number_to_real_with_caller(sc, g_magnitude(sc, set_plist_1(sc, p)), "magnitude"));
}


/* -------------------------------- rationalize -------------------------------- */
static s7_pointer g_rationalize(s7_scheme *sc, s7_pointer args)
{
  #define H_rationalize "(rationalize x err) returns the ratio with lowest denominator within err of x"
  #define Q_rationalize s7_make_signature(sc, 3, sc->is_real_symbol, sc->is_real_symbol, sc->is_real_symbol)
  /* result is real (not rational) if argument is NaN */
  s7_double err;
  s7_pointer x;

  x = car(args);
  if (!s7_is_real(x))
    return(method_or_bust(sc, x, sc->rationalize_symbol, args, T_REAL, 1));

  if (is_not_null(cdr(args)))
    {
      s7_pointer ex;
      ex = cadr(args);
      if (!s7_is_real(ex))
	return(method_or_bust(sc, ex, sc->rationalize_symbol, args, T_REAL, 2));

      err = real_to_double(sc, ex, "rationalize");
      if (is_NaN(err))
	return(out_of_range(sc, sc->rationalize_symbol, small_int(2), cadr(args), its_nan_string));
      if (err < 0.0) err = -err;
    }
  else err = sc->default_rationalize_error;

  switch (type(x))
    {
    case T_INTEGER:
      {
	s7_int a, b, pa;
	if (err < 1.0) return(x);
	a = s7_integer(x);
	if (a < 0) pa = -a; else pa = a;
	if (err >= pa) return(small_int(0));
	b = (s7_int)err;
	pa -= b;
	if (a < 0)
	  return(make_integer(sc, -pa));
	return(make_integer(sc, pa));
      }

    case T_RATIO:
      if (err == 0.0)
	return(x);

    case T_REAL:
      {
	s7_double rat;
	s7_int numer = 0, denom = 1;

	rat = s7_real(x); /* possible fall through from above */
	if ((is_NaN(rat)) || (is_inf(rat)))
	  return(out_of_range(sc, sc->rationalize_symbol, small_int(1), x, a_normal_real_string));

	if (err >= fabs(rat))
	  return(small_int(0));

	if ((rat > 9.2233720368548e+18) || (rat < -9.2233720368548e+18))
	  return(out_of_range(sc, sc->rationalize_symbol, small_int(1), x, its_too_large_string));

	if ((fabs(rat) + fabs(err)) < 1.0e-18)
	  err = 1.0e-18;
	/* (/ 1.0 most-positive-fixnum) is 1.0842021e-19, so if we let err be less than that,
	 * (rationalize 1e-19 1e-20) hangs, but this only affects the initial ceiling, I believe.
	 */

	if (fabs(rat) < fabs(err))
	  return(small_int(0));

	if (c_rationalize(rat, err, &numer, &denom))
	  return(s7_make_ratio(sc, numer, denom));

	return(sc->F);
      }
    }
  return(sc->F); /* make compiler happy */
}


/* -------------------------------- angle -------------------------------- */
static s7_pointer g_angle(s7_scheme *sc, s7_pointer args)
{
  #define H_angle "(angle z) returns the angle of z"
  #define Q_angle s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_number_symbol)
  s7_pointer x;
  /* (angle inf+infi) -> 0.78539816339745 ?
   *   I think this should be -pi < ang <= pi
   */

  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) < 0)
	return(real_pi);
      return(small_int(0));

    case T_RATIO:
      if (numerator(x) < 0)
	return(real_pi);
      return(small_int(0));

    case T_REAL:
      if (is_NaN(real(x))) return(x);
      if (real(x) < 0.0)
	return(real_pi);
      return(real_zero);

    case T_COMPLEX:
      return(make_real(sc, atan2(imag_part(x), real_part(x))));

    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->angle_symbol, args, a_number_string));
    }
}

static s7_double angle_d_7p(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
    case T_INTEGER: if (integer(x) < 0) return(M_PI); return(0.0);
    case T_RATIO:   if (numerator(x) < 0) return(M_PI); return(0.0);
    case T_REAL:    if (is_NaN(real(x))) return(NAN); if (real(x) < 0.0) return(M_PI); return(0.0);
    case T_COMPLEX: return(atan2(imag_part(x), real_part(x)));
    default: simple_wrong_type_argument_with_type(sc, sc->angle_symbol, x, a_number_string); break;
    }
  return(0.0);
}


/* -------------------------------- make-polar -------------------------------- */
#if (!WITH_PURE_S7)
static s7_pointer g_make_polar(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;
  s7_double ang, mag;
  #define H_make_polar "(make-polar mag ang) returns a complex number with magnitude mag and angle ang"
  #define Q_make_polar s7_make_signature(sc, 3, sc->is_number_symbol, sc->is_real_symbol, sc->is_real_symbol)

  x = car(args);
  y = cadr(args);

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(x) == 0) return(x);            /* (make-polar 0 1) -> 0 */
	  if (integer(y) == 0) return(x);            /* (make-polar 1 0) -> 1 */
	  mag = (s7_double)integer(x);
	  ang = (s7_double)integer(y);
	  break;

	case T_RATIO:
	  if (integer(x) == 0) return(x);
	  mag = (s7_double)integer(x);
	  ang = (s7_double)fraction(y);
	  break;

	case T_REAL:
	  ang = real(y);
	  if (ang == 0.0) return(x);
	  if (is_NaN(ang)) return(y);
	  if (is_inf(ang)) return(real_NaN);
	  if ((ang == M_PI) || (ang == -M_PI)) return(make_integer(sc, -integer(x)));
	  mag = (s7_double)integer(x);
	  break;

	default:
	  return(method_or_bust(sc, y, sc->make_polar_symbol, args, T_REAL, 2));
	}
      break;

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0) return(x);
	  mag = (s7_double)fraction(x);
	  ang = (s7_double)integer(y);
	  break;

	case T_RATIO:
	  mag = (s7_double)fraction(x);
	  ang = (s7_double)fraction(y);
	  break;

	case T_REAL:
	  ang = real(y);
	  if (ang == 0.0) return(x);
	  if (is_NaN(ang)) return(y);
	  if (is_inf(ang)) return(real_NaN);
	  if ((ang == M_PI) || (ang == -M_PI)) return(make_simple_ratio(sc, -numerator(x), denominator(x)));
	  mag = (s7_double)fraction(x);
	  break;

	default:
	  return(method_or_bust(sc, y, sc->make_polar_symbol, args, T_REAL, 2));
	}
      break;

    case T_REAL:
      mag = real(x);
      switch (type(y))
	{
	case T_INTEGER:
	  if (is_NaN(mag)) return(x);
	  if (integer(y) == 0) return(x);
	  ang = (s7_double)integer(y);
	  break;

	case T_RATIO:
	  if (is_NaN(mag)) return(x);
	  ang = (s7_double)fraction(y);
	  break;

	case T_REAL:
	  if (is_NaN(mag)) return(x);
	  ang = real(y);
	  if (ang == 0.0) return(x);
	  if (is_NaN(ang)) return(y);
	  if (is_inf(ang)) return(real_NaN);
	  break;

	default:
	  return(method_or_bust(sc, y, sc->make_polar_symbol, args, T_REAL, 2));
	}
      break;

    default:
      return(method_or_bust(sc, x, sc->make_polar_symbol, args, T_REAL, 1));
    }

  return(s7_make_complex(sc, mag * cos(ang), mag * sin(ang)));

  /* since sin is inaccurate for large arguments, so is make-polar:
   *    (make-polar 1.0 1e40) -> -0.76267273202438+0.64678458842683i, not 8.218988919070239214448025364432557517335E-1-5.696334009536363273080341815735687231337E-1i
   */
}
#endif


/* -------------------------------- complex -------------------------------- */

static s7_pointer c_complex(s7_scheme *sc, s7_double rl, s7_double im)
{
  /* same as s7_make_complex, but assumes im is not 0.0 */
  s7_pointer x;
  new_cell(sc, x, T_COMPLEX);
  set_real_part(x, rl);
  set_imag_part(x, im);
  return(x);
}

static s7_pointer g_complex(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;
  #define H_complex "(complex x1 x2) returns a complex number with real-part x1 and imaginary-part x2"
  #define Q_complex s7_make_signature(sc, 3, sc->is_number_symbol, sc->is_real_symbol, sc->is_real_symbol)

  x = car(args);
  y = cadr(args);

  switch (type(y))
    {
    case T_INTEGER:
      switch (type(x))
	{
	case T_INTEGER:
	  if (integer(y) == 0) return(x);
	  return(c_complex(sc, (s7_double)integer(x), (s7_double)integer(y)));

	case T_RATIO:
	  if (integer(y) == 0) return(x);
	  return(c_complex(sc, (s7_double)fraction(x), (s7_double)integer(y)));

	case T_REAL:
	  if (integer(y) == 0) return(x);
	  return(c_complex(sc, real(x), (s7_double)integer(y)));

	default:
	  return(method_or_bust(sc, x, sc->complex_symbol, args, T_REAL, 1));
	}

    case T_RATIO:
      switch (type(x))
	{
	case T_INTEGER: return(c_complex(sc, (s7_double)integer(x), (s7_double)fraction(y)));
	case T_RATIO:   return(c_complex(sc, (s7_double)fraction(x), (s7_double)fraction(y)));
	case T_REAL:    return(c_complex(sc, real(x), (s7_double)fraction(y)));
	default:
	  return(method_or_bust(sc, x, sc->complex_symbol, args, T_REAL, 1));
	}

    case T_REAL:
      switch (type(x))
	{
	case T_INTEGER:
	  if (real(y) == 0.0) return(x);
	  return(c_complex(sc, (s7_double)integer(x), real(y)));

	case T_RATIO:
	  if (real(y) == 0.0) return(x);
	  return(c_complex(sc, (s7_double)fraction(x), real(y)));

	case T_REAL:
	  if (real(y) == 0.0) return(x);
	  return(c_complex(sc, real(x), real(y)));

	default:
	  return(method_or_bust(sc, x, sc->complex_symbol, args, T_REAL, 1));
	}

    default:
      return(method_or_bust(sc, (is_let(x)) ? x : y, sc->complex_symbol, args, T_REAL, 2));
    }
}

static s7_pointer complex_p_ii(s7_scheme *sc, s7_int x, s7_int y)
{
  if (y == 0)
    return(make_real(sc, (s7_double)x));
  return(c_complex(sc, (s7_double)x, (s7_double)y));
}

static s7_pointer complex_p_dd(s7_scheme *sc, s7_double x, s7_double y)
{
  if (y == 0)
    return(make_real(sc, x));
  return(c_complex(sc, x, y));
}


/* -------------------------------- exp -------------------------------- */
static s7_pointer g_exp(s7_scheme *sc, s7_pointer args)
{
  #define H_exp "(exp z) returns e^z, (exp 1) is 2.718281828459"
  #define Q_exp sc->pcl_n

  s7_pointer x;

  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(small_int(1));                       /* (exp 0) -> 1 */
      return(make_real(sc, exp((s7_double)(integer(x)))));

    case T_RATIO:
      return(make_real(sc, exp((s7_double)fraction(x))));

    case T_REAL:
      return(make_real(sc, exp(real(x))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      return(s7_from_c_complex(sc, cexp(as_c_complex(x))));
      /* this is inaccurate for large arguments:
       *   (exp 0+1e20i) -> -0.66491178990701-0.74692189125949i, not 7.639704044417283004001468027378811228331E-1-6.45251285265780844205811711312523007406E-1i
       */
#else
      return(out_of_range(sc, sc->exp_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->exp_symbol, args, a_number_string));
    }
}

static s7_double exp_d_d(s7_double x) {return(exp(x));}


/* -------------------------------- log -------------------------------- */

#if __cplusplus
#define LOG_2 1.4426950408889634074
#else
#define LOG_2 1.4426950408889634073599246810018921L /* (/ (log 2.0)) */
#endif

static s7_pointer g_log(s7_scheme *sc, s7_pointer args)
{
  #define H_log "(log z1 (z2 e)) returns log(z1) / log(z2) where z2 (the base) defaults to e: (log 8 2) = 3"
  #define Q_log sc->pcl_n

  s7_pointer x;
  x = car(args);
  if (!s7_is_number(x))
    return(method_or_bust_with_type(sc, x, sc->log_symbol, args, a_number_string, 1));

  if (is_pair(cdr(args)))
    {
      s7_pointer y;

      y = cadr(args);
      if (!(s7_is_number(y)))
	return(method_or_bust_with_type(sc, y, sc->log_symbol, args, a_number_string, 2));

      if (y == small_int(2))
	{
	  /* (define (2^n? x) (and (not (zero? x)) (zero? (logand x (- x 1))))) */
	  if (is_integer(x))
	    {
	      s7_int ix;
	      ix = s7_integer(x);
	      if (ix > 0)
		{
		  s7_double fx;
#if (__ANDROID__) || (MS_WINDOWS) || ((__GNUC__) && ((__GNUC__ < 4) || ((__GNUC__ == 4) && (__GNUC_MINOR__ <= 4))))
		  /* just a guess -- log2 gets a warning in gcc 4.3.2, but not in 4.4.4 */
		  fx = log((double)ix) / log(2.0);
#else
		  fx = log2((double)ix);
#endif
		  /* (s7_int)fx rounds (log 8 2) to 2 in FreeBSD! */
#if ((__GNUC__) && ((__GNUC__ < 4) || ((__GNUC__ == 4) && (__GNUC_MINOR__ < 4))))
		  return(make_real(sc, fx));
#else
		  if ((ix & (ix - 1)) == 0)
		    return(make_integer(sc, (s7_int)s7_round(fx)));
		  return(make_real(sc, fx));
#endif
		}
	    }
	  if ((s7_is_real(x)) &&
	      (s7_is_positive(x)))
	    return(make_real(sc, log(s7_real(x)) * LOG_2));
	  return(s7_from_c_complex(sc, clog(s7_to_c_complex(x)) * LOG_2));
	}

      if ((x == small_int(1)) && (y == small_int(1)))  /* (log 1 1) -> 0 (this is NaN in the bignum case) */
	return(small_int(0));

      /* (log 1 0) must be 0 since everyone says (expt 0 0) is 1 */
      if (s7_is_zero(y))
	{
	  if ((y == small_int(0)) &&
	      (x == small_int(1)))
	    return(y);
	  return(out_of_range(sc, sc->log_symbol, small_int(2), y, wrap_string(sc, "can't be 0", 10)));
	}

      if (s7_is_one(y))          /* this used to raise an error, but the bignum case is simpler if we return inf */
	{
	  if (s7_is_one(x))      /* but (log 1.0 1.0) -> 0.0 */
	    return(real_zero);
	  return(real_infinity); /* currently (log 1/0 1) is inf? */
	}

      if ((s7_is_real(x)) &&
	  (s7_is_real(y)) &&
	  (s7_is_positive(x)) &&
	  (s7_is_positive(y)))
	{
	  if ((s7_is_rational(x)) &&
	      (s7_is_rational(y)))
	    {
	      s7_double res;
	      s7_int ires;
	      res = log(rational_to_double(sc, x)) / log(rational_to_double(sc, y));
	      ires = (s7_int)res;
	      if (res - ires == 0.0)
		return(make_integer(sc, ires));   /* (log 8 2) -> 3 or (log 1/8 2) -> -3 */
	      return(make_real(sc, res));         /* perhaps use rationalize here? (log 2 8) -> 1/3 */
	    }
	  return(make_real(sc, log(s7_real(x)) / log(s7_real(y))));
	}
      return(s7_from_c_complex(sc, clog(s7_to_c_complex(x)) / clog(s7_to_c_complex(y))));
    }

  if (s7_is_real(x))
    {
      if (s7_is_positive(x))
	return(make_real(sc, log(s7_real(x))));
      return(s7_make_complex(sc, log(-s7_real(x)), M_PI));
    }
  return(s7_from_c_complex(sc, clog(s7_to_c_complex(x))));
}


/* -------------------------------- sin -------------------------------- */
static s7_pointer g_sin(s7_scheme *sc, s7_pointer args)
{
  #define H_sin "(sin z) returns sin(z)"
  #define Q_sin sc->pcl_n

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_REAL:
      return(make_real(sc, sin(real(x))));

    case T_INTEGER:
      if (integer(x) == 0) return(small_int(0));                      /* (sin 0) -> 0 */
      return(make_real(sc, sin((s7_double)integer(x))));

    case T_RATIO:
      return(make_real(sc, sin((s7_double)(fraction(x)))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      return(s7_from_c_complex(sc, csin(as_c_complex(x))));
#else
      return(out_of_range(sc, sc->sin_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->sin_symbol, args, a_number_string));
    }

  /* sin is totally inaccurate over about 1e18.  There's a way to get true results,
   *   but it involves fancy "range reduction" techniques.
   *   This means that lots of things are inaccurate:
   * (sin (remainder 1e22 (* 2 pi)))
   * -0.57876806033477
   * but it should be -8.522008497671888065747423101326159661908E-1
   * ---
   * (remainder 1e22 (* 2 pi)) -> 1.0057952155665e+22 !!
   *   it should be 5.263007914620499494429139986095833592117E0
   */
}

static s7_double sin_d_d(s7_double x) {return(sin(x));}


/* -------------------------------- cos -------------------------------- */
static s7_pointer g_cos(s7_scheme *sc, s7_pointer args)
{
  #define H_cos "(cos z) returns cos(z)"
  #define Q_cos sc->pcl_n

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_REAL:
      return(make_real(sc, cos(real(x))));

    case T_INTEGER:
      if (integer(x) == 0) return(small_int(1));                     /* (cos 0) -> 1 */
      return(make_real(sc, cos((s7_double)integer(x))));

    case T_RATIO:
      return(make_real(sc, cos((s7_double)(fraction(x)))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      return(s7_from_c_complex(sc, ccos(as_c_complex(x))));
#else
      return(out_of_range(sc, sc->cos_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->cos_symbol, args, a_number_string));
    }
}

static s7_double cos_d_d(s7_double x) {return(cos(x));}


/* -------------------------------- tan -------------------------------- */
static s7_pointer g_tan(s7_scheme *sc, s7_pointer args)
{
  #define H_tan "(tan z) returns tan(z)"
  #define Q_tan sc->pcl_n

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_REAL:
      return(make_real(sc, tan(real(x))));

    case T_INTEGER:
      if (integer(x) == 0) return(small_int(0));                      /* (tan 0) -> 0 */
      return(make_real(sc, tan((s7_double)(integer(x)))));

    case T_RATIO:
      return(make_real(sc, tan((s7_double)(fraction(x)))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      if (imag_part(x) > 350.0)
	return(c_complex(sc, 0.0, 1.0));
      if (imag_part(x) < -350.0)
	return(c_complex(sc, 0.0, -1.0));
      return(s7_from_c_complex(sc, ctan(as_c_complex(x))));
#else
      return(out_of_range(sc, sc->tan_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->tan_symbol, args, a_number_string));
    }
}

static s7_double tan_d_d(s7_double x) {return(tan(x));}


/* -------------------------------- asin -------------------------------- */
static s7_pointer c_asin(s7_scheme *sc, s7_double x)
{
  s7_double absx, recip;
  s7_complex result;

  absx = fabs(x);
  if (absx <= 1.0)
    return(make_real(sc, asin(x)));

  /* otherwise use maxima code: */
  recip = 1.0 / absx;
  result = (M_PI / 2.0) - (_Complex_I * clog(absx * (1.0 + (sqrt(1.0 + recip) * csqrt(1.0 - recip)))));
  if (x < 0.0)
    return(s7_from_c_complex(sc, -result));
  return(s7_from_c_complex(sc, result));
}


static s7_pointer g_asin(s7_scheme *sc, s7_pointer args)
{
  #define H_asin "(asin z) returns asin(z); (sin (asin x)) = x"
  #define Q_asin sc->pcl_n
  s7_pointer n;

  n = car(args);
  switch (type(n))
    {
    case T_INTEGER:
      if (integer(n) == 0) return(small_int(0));                    /* (asin 0) -> 0 */
      /* in netBSD, (asin 2) returns 0.25383842987008+0.25383842987008i according to Peter Bex */
      return(c_asin(sc, (s7_double)integer(n)));

    case T_RATIO:
      return(c_asin(sc, (s7_double)numerator(n) / (s7_double)denominator(n)));

    case T_REAL:
      return(c_asin(sc, real(n)));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      /* if either real or imag part is very large, use explicit formula, not casin */
      /*   this code taken from sbcl's src/code/irrat.lisp */
      /* break is around x+70000000i */

      if ((fabs(real_part(n)) > 1.0e7) ||
	  (fabs(imag_part(n)) > 1.0e7))
	{
	  s7_complex sq1mz, sq1pz, z;
	  z = as_c_complex(n);
	  sq1mz = csqrt(1.0 - z);
	  sq1pz = csqrt(1.0 + z);
	  return(s7_make_complex(sc, atan(real_part(n) / creal(sq1mz * sq1pz)), asinh(cimag(sq1pz * conj(sq1mz)))));
	}
      return(s7_from_c_complex(sc, casin(as_c_complex(n))));
#else
      return(out_of_range(sc, sc->asin_symbol, small_int(1), n, no_complex_numbers_string));
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, n, sc->asin_symbol, list_1(sc, n), a_number_string));
    }
}


/* -------------------------------- acos -------------------------------- */
static s7_pointer c_acos(s7_scheme *sc, s7_double x)
{
  s7_double absx, recip;
  s7_complex result;

  absx = fabs(x);
  if (absx <= 1.0)
    return(make_real(sc, acos(x)));

  /* else follow maxima again: */
  recip = 1.0 / absx;
  if (x > 0.0)
    result = _Complex_I * clog(absx * (1.0 + (sqrt(1.0 + recip) * csqrt(1.0 - recip))));
  else result = M_PI - _Complex_I * clog(absx * (1.0 + (sqrt(1.0 + recip) * csqrt(1.0 - recip))));
  return(s7_from_c_complex(sc, result));
}

static s7_pointer g_acos(s7_scheme *sc, s7_pointer args)
{
  #define H_acos "(acos z) returns acos(z); (cos (acos 1)) = 1"
  #define Q_acos sc->pcl_n
  s7_pointer n;

  n = car(args);
  switch (type(n))
    {
    case T_INTEGER:
      if (integer(n) == 1) return(small_int(0));
      return(c_acos(sc, (s7_double)integer(n)));

    case T_RATIO:
      return(c_acos(sc, (s7_double)numerator(n) / (s7_double)denominator(n)));

    case T_REAL:
      return(c_acos(sc, real(n)));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      /* if either real or imag part is very large, use explicit formula, not cacos */
      /*   this code taken from sbcl's src/code/irrat.lisp */

      if ((fabs(real_part(n)) > 1.0e7) ||
	  (fabs(imag_part(n)) > 1.0e7))
	{
	  s7_complex sq1mz, sq1pz, z;
	  z = as_c_complex(n);
	  sq1mz = csqrt(1.0 - z);
	  sq1pz = csqrt(1.0 + z);	  /* creal(sq1pz) can be 0.0 */
	  if (creal(sq1pz) == 0.0)        /* so the atan arg will be inf, so the real part will be pi/2(?) */
	    return(s7_make_complex(sc, M_PI / 2.0, asinh(cimag(sq1mz * conj(sq1pz)))));
	  return(s7_make_complex(sc, 2.0 * atan(creal(sq1mz) / creal(sq1pz)), asinh(cimag(sq1mz * conj(sq1pz)))));
	}
      return(s7_from_c_complex(sc, cacos(s7_to_c_complex(n))));
#else
      return(out_of_range(sc, sc->acos_symbol, small_int(1), n, no_complex_numbers_string));
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, n, sc->acos_symbol, list_1(sc, n), a_number_string));
    }
}


/* -------------------------------- atan -------------------------------- */

static s7_pointer g_atan(s7_scheme *sc, s7_pointer args)
{
  #define H_atan "(atan z) returns atan(z), (atan y x) returns atan(y/x)"
  #define Q_atan s7_make_signature(sc, 3, sc->is_number_symbol, sc->is_number_symbol, sc->is_real_symbol)
  /* actually if there are two args, both should be real, but how to express that in the signature? */
  s7_pointer x, y;
  s7_double x1, x2;

  /* currently (atan inf.0 inf.0) -> 0.78539816339745, and (atan inf.0 -inf.0) -> 2.3561944901923 (etc) */

  x = car(args);
  if (!is_pair(cdr(args)))
    {
      switch (type(x))
	{
	case T_INTEGER:
	  if (integer(x) == 0) return(small_int(0));                /* (atan 0) -> 0 */

	case T_RATIO:
	case T_REAL:
	  return(make_real(sc, atan(s7_real(x))));

	case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
	  return(s7_from_c_complex(sc, catan(as_c_complex(x))));
#else
	  return(out_of_range(sc, sc->atan_symbol, small_int(1), x, no_complex_numbers_string));
#endif

	default:
	  return(method_or_bust_with_type_one_arg(sc, x, sc->atan_symbol, args, a_number_string));
	}
    }

  if (!s7_is_real(x))
    return(method_or_bust(sc, x, sc->atan_symbol, args, T_REAL, 1));

  y = cadr(args);
  if (!s7_is_real(y))
    return(method_or_bust(sc, y, sc->atan_symbol, args, T_REAL, 2));

  x1 = s7_real(x);
  x2 = s7_real(y);
  return(make_real(sc, atan2(x1, x2)));
}

static s7_double atan_d_dd(s7_double x, s7_double y) {return(atan2(x, y));}


/* -------------------------------- sinh -------------------------------- */
static s7_pointer g_sinh(s7_scheme *sc, s7_pointer args)
{
  #define H_sinh "(sinh z) returns sinh(z)"
  #define Q_sinh sc->pcl_n

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(small_int(0));                    /* (sinh 0) -> 0 */

    case T_REAL:
    case T_RATIO:
      return(make_real(sc, sinh(s7_real(x))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      return(s7_from_c_complex(sc, csinh(as_c_complex(x))));
#else
      return(out_of_range(sc, sc->sinh_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->sinh_symbol, args, a_number_string));
    }
}

static s7_double sinh_d_d(s7_double x) {return(sinh(x));}


/* -------------------------------- cosh -------------------------------- */
static s7_pointer g_cosh(s7_scheme *sc, s7_pointer args)
{
  #define H_cosh "(cosh z) returns cosh(z)"
  #define Q_cosh sc->pcl_n

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(small_int(1));                   /* (cosh 0) -> 1 */

    case T_REAL:
    case T_RATIO:
      /* this is not completely correct when optimization kicks in.
       * :(define (hi) (do ((i 0 (+ i 1))) ((= i 1)) (display (cosh i))))
       * hi
       * :(hi)
       * 1.0()
       * :(cosh 0)
       * 1
       */
      return(make_real(sc, cosh(s7_real(x))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      return(s7_from_c_complex(sc, ccosh(as_c_complex(x))));
#else
      return(out_of_range(sc, sc->cosh_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->cosh_symbol, args, a_number_string));
    }
}

static s7_double cosh_d_d(s7_double x) {return(cosh(x));}


/* -------------------------------- tanh -------------------------------- */
static s7_pointer g_tanh(s7_scheme *sc, s7_pointer args)
{
  #define H_tanh "(tanh z) returns tanh(z)"
  #define Q_tanh sc->pcl_n

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(small_int(0));  /* (tanh 0) -> 0 */

    case T_REAL:
    case T_RATIO:
      return(make_real(sc, tanh(s7_real(x))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
      if (real_part(x) > 350.0)
	return(real_one);                         /* closer than 0.0 which is what ctanh is about to return! */
      if (real_part(x) < -350.0)
	return(make_real(sc, -1.0));              /* closer than ctanh's -0.0 */
      return(s7_from_c_complex(sc, ctanh(as_c_complex(x))));
#else
      return(out_of_range(sc, sc->tanh_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->tanh_symbol, args, a_number_string));
    }
}

static s7_double tanh_d_d(s7_double x) {return(tanh(x));}


/* -------------------------------- asinh -------------------------------- */

static s7_pointer g_asinh(s7_scheme *sc, s7_pointer args)
{
  #define H_asinh "(asinh z) returns asinh(z)"
  #define Q_asinh sc->pcl_n
  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(small_int(0));
      return(make_real(sc, asinh((s7_double)integer(x))));

    case T_RATIO:
      return(make_real(sc, asinh((s7_double)numerator(x) / (s7_double)denominator(x))));

    case T_REAL:
      return(make_real(sc, asinh(real(x))));

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
  #if (defined(__OpenBSD__)) || (defined(__NetBSD__))
      return(s7_from_c_complex(sc, casinh_1(as_c_complex(x))));
  #else
      return(s7_from_c_complex(sc, casinh(as_c_complex(x))));
  #endif
#else
      return(out_of_range(sc, sc->asinh_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->asinh_symbol, list_1(sc, x), a_number_string));
    }
}


/* -------------------------------- acosh -------------------------------- */

static s7_pointer g_acosh(s7_scheme *sc, s7_pointer args)
{
  #define H_acosh "(acosh z) returns acosh(z)"
  #define Q_acosh sc->pcl_n
  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 1) return(small_int(0));

    case T_REAL:
    case T_RATIO:
      {
	double x1;
	x1 = s7_real(x);
	if (x1 >= 1.0)
	  return(make_real(sc, acosh(x1)));
      }

    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
  #ifdef __OpenBSD__
      return(s7_from_c_complex(sc, cacosh_1(s7_to_c_complex(x))));
  #else
      return(s7_from_c_complex(sc, cacosh(s7_to_c_complex(x)))); /* not as_c_complex because x might not be complex */
  #endif
#else
      /* since we can fall through to this branch, we need a better error message than "must be a number, not 0.0" */
      return(out_of_range(sc, sc->acosh_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->acosh_symbol, list_1(sc, x), a_number_string));
    }
}


/* -------------------------------- atanh -------------------------------- */

static s7_pointer g_atanh(s7_scheme *sc, s7_pointer args)
{
  #define H_atanh "(atanh z) returns atanh(z)"
  #define Q_atanh sc->pcl_n
  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      if (integer(x) == 0) return(small_int(0));                    /* (atanh 0) -> 0 */

    case T_REAL:
    case T_RATIO:
      {
	double x1;
	x1 = s7_real(x);
	if (fabs(x1) < 1.0)
	  return(make_real(sc, atanh(x1)));
      }

      /* if we can't distinguish x from 1.0 even with long doubles, we'll get inf.0:
       *    (atanh 9223372036854775/9223372036854776) -> 18.714973875119
       *    (atanh 92233720368547758/92233720368547757) -> inf.0
       */
    case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
  #if (defined(__OpenBSD__)) || (defined(__NetBSD__))
      return(s7_from_c_complex(sc, catanh_1(s7_to_c_complex(x))));
  #else
      return(s7_from_c_complex(sc, catanh(s7_to_c_complex(x))));
  #endif
#else
      return(out_of_range(sc, sc->atanh_symbol, small_int(1), x, no_complex_numbers_string));
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->atanh_symbol, list_1(sc, x), a_number_string));
    }
}


/* -------------------------------- sqrt -------------------------------- */
static s7_pointer g_sqrt(s7_scheme *sc, s7_pointer args)
{
  #define H_sqrt "(sqrt z) returns the square root of z"
  #define Q_sqrt sc->pcl_n

  s7_pointer n;
  s7_double sqx;

  n = car(args);
  switch (type(n))
    {
    case T_INTEGER:
      if (integer(n) >= 0)
	{
	  s7_int ix;
	  sqx = sqrt((s7_double)integer(n));
	  ix = (s7_int)sqx;
	  if ((ix * ix) == integer(n))
	    return(make_integer(sc, ix));
	  return(make_real(sc, sqx));
	  /* Mark Weaver notes that
	   *     (zero? (- (sqrt 9007199136250226) 94906265.0)) -> #t
	   * but (* 94906265 94906265) -> 9007199136250225 -- oops
	   * at least we return a real here, not an incorrect integer and
	   *     (sqrt 9007199136250225) -> 94906265
	   */
	}
      sqx = (s7_double)integer(n); /* we're trying to protect against (sqrt -9223372036854775808) where we can't negate the integer argument */
      return(s7_make_complex(sc, 0.0, sqrt((s7_double)(-sqx))));

    case T_RATIO:
      sqx = (s7_double)fraction(n);
      if (sqx > 0.0) /* else it's complex, so it can't be a ratio */
	{
	  s7_int nm = 0, dn = 1;
	  if (c_rationalize(sqx, 1.0e-16, &nm, &dn)) /* 1e-16 so that (sqrt 1/1099511627776) returns 1/1048576 */
	    {
#if HAVE_OVERFLOW_CHECKS
	      s7_int nm2, dn2;
	      if ((multiply_overflow(nm, nm, &nm2)) ||
		  (multiply_overflow(dn, dn, &dn2)))
		return(make_real(sc, sqrt(sqx)));
	      if ((nm2 == numerator(n)) &&
		  (dn2 == denominator(n)))
		return(s7_make_ratio(sc, nm, dn));
#else
	      if ((nm * nm == numerator(n)) &&
		  (dn * dn == denominator(n)))
		return(s7_make_ratio(sc, nm, dn));
#endif
	    }
	  return(make_real(sc, sqrt(sqx)));
	}
      return(s7_make_complex(sc, 0.0, sqrt(-sqx)));

    case T_REAL:
      if (is_NaN(real(n)))
	return(real_NaN);
      if (real(n) >= 0.0)
	return(make_real(sc, sqrt(real(n))));
      return(s7_make_complex(sc, 0.0, sqrt(-real(n))));

    case T_COMPLEX:
      /* (* inf.0 (sqrt -1)) -> -nan+infi, but (sqrt -inf.0) -> 0+infi */
#if HAVE_COMPLEX_NUMBERS
      return(s7_from_c_complex(sc, csqrt(as_c_complex(n))));
#else
      return(out_of_range(sc, sc->sqrt_symbol, small_int(1), n, no_complex_numbers_string));
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, n, sc->sqrt_symbol, args, a_number_string));
    }
}


/* -------------------------------- expt -------------------------------- */

static s7_int int_to_int(s7_int x, s7_int n)
{
  /* from GSL */
  s7_int value = 1;
  do {
    if (n & 1) value *= x;
    n >>= 1;
#if HAVE_OVERFLOW_CHECKS
    if (multiply_overflow(x, x, &x))
      break;
#else
    x *= x;
#endif
  } while (n);
  return(value);
}


static const int64_t nth_roots[63] = {
  S7_LLONG_MAX, S7_LLONG_MAX, 3037000499LL, 2097151, 55108, 6208, 1448, 511, 234, 127, 78, 52, 38, 28, 22,
  18, 15, 13, 11, 9, 8, 7, 7, 6, 6, 5, 5, 5, 4, 4, 4, 4, 3, 3, 3, 3, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2,
  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2};

static const int64_t int_nth_roots[31] = {
  S7_LONG_MAX, S7_LONG_MAX, 46340, 1290, 215, 73, 35, 21, 14, 10, 8, 7, 5, 5, 4, 4, 3, 3, 3, 3, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2};

static bool int_pow_ok(s7_int x, s7_int y)
{
  if (s7_int_bits > 31)
    return((y < 63) &&
	   (nth_roots[y] >= s7_int_abs(x)));
  return((y < 31) &&
	 (int_nth_roots[y] >= s7_int_abs(x)));
}


static s7_pointer g_expt(s7_scheme *sc, s7_pointer args)
{
  #define H_expt "(expt z1 z2) returns z1^z2"
  #define Q_expt sc->pcl_n
  s7_pointer n, pw;

  n = car(args);
  if (!s7_is_number(n))
    return(method_or_bust_with_type(sc, n, sc->expt_symbol, args, a_number_string, 1));

  pw = cadr(args);
  if (!s7_is_number(pw))
    return(method_or_bust_with_type(sc, pw, sc->expt_symbol, args, a_number_string, 2));

  /* this provides more than 2 args to expt:
   *  if (is_not_null(cddr(args)))
   *    return(g_expt(sc, list_2(sc, car(args), g_expt(sc, cdr(args)))));
   *
   * but it's unusual in scheme to process args in reverse order, and the
   * syntax by itself is ambiguous (does (expt 2 2 3) = 256 or 64?)
   */

  if (s7_is_zero(n))
    {
      if (s7_is_zero(pw))
	{
	  if ((s7_is_integer(n)) && (s7_is_integer(pw)))       /* (expt 0 0) -> 1 */
	    return(small_int(1));
	  return(real_zero);                                   /* (expt 0.0 0) -> 0.0 */
	}

      if (s7_is_real(pw))
	{
	  if (s7_is_negative(pw))                              /* (expt 0 -1) */
	    return(division_by_zero_error(sc, sc->expt_symbol, args));
	  /* (Clisp gives divide-by-zero error here, Guile returns inf.0) */

	  if ((!s7_is_rational(pw)) &&                         /* (expt 0 most-positive-fixnum) */
	      (is_NaN(s7_real(pw))))                           /* (expt 0 +nan.0) */
	    return(pw);
	}
      else
	{                                                      /* (expt 0 a+bi) */
	  if (real_part(pw) < 0.0)                             /* (expt 0 -1+i) */
	    return(division_by_zero_error(sc, sc->expt_symbol, args));
	  if ((is_NaN(real_part(pw))) ||                       /* (expt 0 0+1/0i) */
	      (is_NaN(imag_part(pw))))
	    return(real_NaN);
	}

      if ((s7_is_integer(n)) && (s7_is_integer(pw)))           /* pw != 0, (expt 0 2312) */
	return(small_int(0));
      return(real_zero);                                       /* (expt 0.0 123123) */
    }

  if (s7_is_one(pw))
    {
      if (s7_is_integer(pw))                                   /* (expt x 1) */
	return(n);
      if (is_rational(n))                                      /* (expt ratio 1.0) */
	return(make_real(sc, rational_to_double(sc, n)));
      return(n);
    }

  if (is_t_integer(pw))
    {
      s7_int y;
      y = integer(pw);
      if (y == 0)
	{
	  if (is_rational(n))                                 /* (expt 3 0) */
	    return(small_int(1));
	  if ((is_NaN(s7_real_part(n))) ||                    /* (expt 1/0 0) -> NaN */
	      (is_NaN(s7_imag_part(n))))                      /* (expt (complex 0 1/0) 0) -> NaN */
	    return(n);
	  return(real_one);                                   /* (expt 3.0 0) */
	}

      switch (type(n))
	{
	case T_INTEGER:
	  {
	    s7_int x;
	    x = s7_integer(n);
	    if (x == 1)                                       /* (expt 1 y) */
	      return(n);

	    if (x == -1)
	      {
		if (y == s7_int_min)                        /* (expt -1 most-negative-fixnum) */
		  return(small_int(1));

		if (s7_int_abs(y) & 1)                        /* (expt -1 odd-int) */
		  return(n);
		return(small_int(1));                         /* (expt -1 even-int) */
	      }

	    if (y == s7_int_min)                            /* (expt x most-negative-fixnum) */
	      return(small_int(0));
	    if (x == s7_int_min)                            /* (expt most-negative-fixnum y) */
	      return(make_real(sc, pow((double)x, (double)y)));

	    if (int_pow_ok(x, s7_int_abs(y)))
	      {
		if (y > 0)
		  return(make_integer(sc, int_to_int(x, y)));
		return(s7_make_ratio(sc, 1, int_to_int(x, -y)));
	      }
	  }
	  break;

	case T_RATIO:
	  {
	    s7_int nm, dn;

	    nm = numerator(n);
	    dn = denominator(n);

	    if (y == s7_int_min)
	      {
		if (s7_int_abs(nm) > dn)
		  return(small_int(0));              /* (expt 4/3 most-negative-fixnum) -> 0? */
		return(real_infinity);               /* (expt 3/4 most-negative-fixnum) -> inf? */
	      }

	    if ((int_pow_ok(nm, s7_int_abs(y))) &&
		(int_pow_ok(dn, s7_int_abs(y))))
	      {
		if (y > 0)
		  return(s7_make_ratio(sc, int_to_int(nm, y), int_to_int(dn, y)));
		return(s7_make_ratio(sc, int_to_int(dn, -y), int_to_int(nm, -y)));
	      }
	  }
	  break;
	  /* occasionally int^rat can be int32_t but it happens so infrequently it's probably not worth checking
	   *  one possibly easy case: (expt 1 1/2) -> 1 (-1?) etc
	   */

	case T_REAL:
	  /* (expt -1.0 most-positive-fixnum) should be -1.0
	   * (expt -1.0 (+ (expt 2 53) 1)) -> -1.0
	   * (expt -1.0 (- 1 (expt 2 54))) -> -1.0
	   */
	  if (real(n) == -1.0)
	    {
	      if (y == s7_int_min)
		return(real_one);

	      if (s7_int_abs(y) & 1)
		return(n);
	      return(real_one);
	    }
	  break;

	case T_COMPLEX:
#if HAVE_COMPLEX_NUMBERS
	  if ((s7_real_part(n) == 0.0) &&
	      ((s7_imag_part(n) == 1.0) ||
	       (s7_imag_part(n) == -1.0)))
	    {
	      bool yp, np;
	      yp = (y > 0);
	      np = (s7_imag_part(n) > 0.0);
	      switch (s7_int_abs(y) % 4)
		{
		case 0: return(real_one);
		case 1: return(s7_make_complex(sc, 0.0, (yp == np) ? 1.0 : -1.0));
		case 2: return(make_real(sc, -1.0));
		case 3: return(s7_make_complex(sc, 0.0, (yp == np) ? -1.0 : 1.0));
		}
	    }
#else
	  return(out_of_range(sc, sc->expt_symbol, small_int(2), n, no_complex_numbers_string));
#endif
	  break;
	}
    }

  if ((s7_is_real(n)) &&
      (s7_is_real(pw)))
    {
      s7_double x, y;

      if ((is_t_ratio(pw)) &&
	  (numerator(pw) == 1))
	{
	  if (denominator(pw) == 2)
	    return(g_sqrt(sc, args));
	  if (denominator(pw) == 3)
	    return(make_real(sc, cbrt(s7_real(n)))); /* (expt 27 1/3) should be 3, not 3.0... */

	  /* but: (expt 512/729 1/3) -> 0.88888888888889 */
	  /* and 4 -> sqrt(sqrt...) etc? */
	}

      x = s7_real(n);
      y = s7_real(pw);

      if (is_NaN(x)) return(n);
      if (is_NaN(y)) return(pw);
      if (y == 0.0) return(real_one);

      if (x > 0.0)
	return(make_real(sc, pow(x, y)));
      /* tricky cases abound here: (expt -1 1/9223372036854775807) */
    }

  /* (expt 0+i 1e+16) = 0.98156860153485-0.19111012657867i ?
   * (expt 0+i 1+1/0i) = 0.0 ??
   */
  return(s7_from_c_complex(sc, cpow(s7_to_c_complex(n), s7_to_c_complex(pw))));
}


/* -------------------------------- lcm -------------------------------- */
static s7_pointer g_lcm(s7_scheme *sc, s7_pointer args)
{
  /* (/ (* m n) (gcd m n)), (lcm a b c) -> (lcm a (lcm b c)) */
  #define H_lcm "(lcm ...) returns the least common multiple of its rational arguments"
  #define Q_lcm sc->pcl_f

  s7_int n = 1, d = 0;
  s7_pointer p;

  if (!is_pair(args))
    return(small_int(1));

  if (!is_pair(cdr(args)))
    {
      if (!is_rational(car(args)))
	return(method_or_bust_with_type(sc, car(args), sc->lcm_symbol, args, a_rational_string, 1));
      return(g_abs(sc, args));
    }

  for (p = args; is_pair(p); p = cdr(p))
    {
      s7_pointer x;
      s7_int b;
      x = car(p);
      switch (type(x))
	{
	case T_INTEGER:
	  if (integer(x) == 0)
	    n = 0;
	  else
	    {
	      b = integer(x);
	      if (b < 0) b = -b;
#if HAVE_OVERFLOW_CHECKS
	      if (multiply_overflow(n / c_gcd(n, b), b, &n))
		return(simple_out_of_range(sc, sc->lcm_symbol, args, result_is_too_large_string));
#else
	      n = (n / c_gcd(n, b)) * b;
#endif
	    }
	  if (d != 0) d = 1;
	  break;

	case T_RATIO:
	  b = numerator(x);
	  if (b < 0) b = -b;
	  n = (n / c_gcd(n, b)) * b;
	  if (d == 0)
	    {
	      if (p == args)
		d = s7_denominator(x);
	      else d = 1;
	    }
	  else d = c_gcd(d, s7_denominator(x));
	  break;

	default:
	  return(method_or_bust_with_type(sc, x, sc->lcm_symbol, cons(sc, (d <= 1) ? make_integer(sc, n) : s7_make_ratio(sc, n, d), p),
					  a_rational_string, position_of(p, args)));
	}
      if (n < 0) return(simple_out_of_range(sc, sc->lcm_symbol, args, result_is_too_large_string));
      if (n == 0)
	{
	  for (p = cdr(p); is_pair(p); p = cdr(p))
	    if (!is_rational_via_method(sc, car(p)))
	      return(wrong_type_argument_with_type(sc, sc->lcm_symbol, position_of(p, args), x, a_rational_string));
	  return(small_int(0));
	}
    }

  if (d <= 1)
    return(make_integer(sc, n));
  return(make_simple_ratio(sc, n, d));
}


/* -------------------------------- gcd -------------------------------- */
static s7_pointer g_gcd(s7_scheme *sc, s7_pointer args)
{
  #define H_gcd "(gcd ...) returns the greatest common divisor of its rational arguments"
  #define Q_gcd sc->pcl_f
  s7_int n = 0, d = 1;
  s7_pointer p;

  if (!is_pair(args))
    return(small_int(0));

  if (!is_pair(cdr(args)))
    {
      if (!is_rational(car(args)))
	return(method_or_bust_with_type(sc, car(args), sc->gcd_symbol, args, a_rational_string, 1));
      return(g_abs(sc, args));
    }

  for (p = args; is_pair(p); p = cdr(p))
    {
      s7_pointer x;
      s7_int b;
      x = car(p);
      switch (type(x))
	{
	case T_INTEGER:
	  n = c_gcd(n, integer(x));
	  break;

	case T_RATIO:
	  n = c_gcd(n, s7_numerator(x));
	  b = s7_denominator(x);
	  if (b < 0) b = -b;
	  d = (d / c_gcd(d, b)) * b;
	  if (d < 0) return(simple_out_of_range(sc, sc->gcd_symbol, args, result_is_too_large_string));
	  break;

	default:
	  return(method_or_bust_with_type(sc, x, sc->gcd_symbol, cons(sc, (d <= 1) ? make_integer(sc, n) : s7_make_ratio(sc, n, d), p),
					  a_rational_string, position_of(p, args)));
	}
      if (n < 0) return(simple_out_of_range(sc, sc->gcd_symbol, args, result_is_too_large_string));
    }

  if (d <= 1)
    return(make_integer(sc, n));
  return(make_simple_ratio(sc, n, d));
}


static s7_pointer s7_truncate(s7_scheme *sc, s7_pointer caller, s7_double xf)   /* can't use "truncate" -- it's in unistd.h */
{
  if ((xf > s7_int_max) ||
      (xf < s7_int_min))
    return(simple_out_of_range(sc, caller, wrap_real(sc, xf), its_too_large_string));

  if (xf > 0.0)
    return(make_integer(sc, (s7_int)floor(xf)));
  return(make_integer(sc, (s7_int)ceil(xf)));
}

static inline s7_int c_quo_int(s7_scheme *sc, s7_int x, s7_int y)
{
  if (y == 0)
    division_by_zero_error(sc, sc->quotient_symbol, set_elist_2(sc, wrap_integer1(sc, x), wrap_integer2(sc, y)));
  if ((y == -1) && (x == s7_int_min))   /* (quotient most-negative-fixnum -1) */
    simple_out_of_range(sc, sc->quotient_symbol, set_elist_2(sc, wrap_integer1(sc, x), wrap_integer2(sc, y)), its_too_large_string);
  return(x / y);
}

static s7_double c_quo_dbl(s7_scheme *sc, s7_double x, s7_double y)
{
  s7_double xf;

  if (y == 0.0)
    division_by_zero_error(sc, sc->quotient_symbol, set_elist_2(sc, wrap_real(sc, x), wrap_real2(sc, y)));
  if ((is_inf(y)) || (is_NaN(y)))
    wrong_type_argument_with_type(sc, sc->quotient_symbol, 2, wrap_real(sc, y), a_normal_real_string);

  xf = x / y;
  if ((xf > s7_int_max) ||
      (xf < s7_int_min))
    simple_out_of_range(sc, sc->quotient_symbol, wrap_real(sc, xf), its_too_large_string);

  if (xf > 0.0)
    return(floor(xf));
  return(ceil(xf));
}

static s7_int quotient_i_7ii(s7_scheme *sc, s7_int i1, s7_int i2) {return(c_quo_int(sc, i1, i2));}
static s7_int quotient_i_ii_direct(s7_int i1, s7_int i2) {return(i1 / i2);} /* i2 > 0 */

static s7_double quotient_d_7dd(s7_scheme *sc, s7_double x1, s7_double x2)
{
  if ((is_inf(x1)) || (is_NaN(x1)))
    wrong_type_argument_with_type(sc, sc->quotient_symbol, 1, wrap_real(sc, x1), a_normal_real_string);
  return(c_quo_dbl(sc, x1, x2));
}

static s7_pointer g_quotient(s7_scheme *sc, s7_pointer args)
{
  #define H_quotient "(quotient x1 x2) returns the integer quotient of x1 and x2; (quotient 4 3) = 1"
  #define Q_quotient sc->pcl_r
  /* (define (quo x1 x2) (truncate (/ x1 x2))) ; slib
   */
  s7_pointer x, y;
  s7_int d1, d2, n1, n2;

  x = car(args);
  y = cadr(args);

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_integer(sc, c_quo_int(sc, integer(x), integer(y))));

	case T_RATIO:
	  n1 = integer(x);
	  d1 = 1;
	  n2 = numerator(y);
	  d2 = denominator(y);
	  /* (quotient -9223372036854775808 -1/9223372036854775807): arithmetic exception in the no-overflow-checks case */
	  goto RATIO_QUO_RATIO;

	case T_REAL:
	  if (real(y) == 0.0)
	    return(division_by_zero_error(sc, sc->quotient_symbol, args));
	  if ((is_inf(real(y))) || (is_NaN(real(y))))
	    return(wrong_type_argument_with_type(sc, sc->quotient_symbol, 2, y, a_normal_real_string));
	  return(s7_truncate(sc, sc->quotient_symbol, (s7_double)integer(x) / real(y)));

	default:
	  return(method_or_bust(sc, y, sc->quotient_symbol, args, T_REAL, 2));
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    return(division_by_zero_error(sc, sc->quotient_symbol, args));
	  n1 = numerator(x);
	  d1 = denominator(x);
	  n2 = integer(y);
	  d2 = 1;
	  goto RATIO_QUO_RATIO;
	  /* this can lose:
	   *   (quotient 1 2305843009213693952/4611686018427387903) -> 2, not 1
	   *   (quotient 21053343141/6701487259 3587785776203/1142027682075) -> 1, not 0
	   */

	case T_RATIO:
	  n1 = numerator(x);
	  d1 = denominator(x);
	  n2 = numerator(y);
	  d2 = denominator(y);
	RATIO_QUO_RATIO:
	  if (d1 == d2)
	    return(make_integer(sc, n1 / n2));              /* (quotient 3/9223372036854775807 1/9223372036854775807) */
	  if (n1 == n2)
	    return(make_integer(sc, d2 / d1));              /* (quotient 9223372036854775807/2 9223372036854775807/8) */
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int n1d2, n2d1;
	    if ((multiply_overflow(n1, d2, &n1d2)) ||
		(multiply_overflow(n2, d1, &n2d1)))
	      return(s7_truncate(sc, sc->quotient_symbol, ((long_double)n1 / (long_double)n2) * ((long_double)d2 / (long_double)d1)));
	    return(make_integer(sc, n1d2 / n2d1));
	  }
#else
	  return(make_integer(sc, (n1 * d2) / (n2 * d1)));
#endif

	case T_REAL:
	  if (real(y) == 0.0)
	    return(division_by_zero_error(sc, sc->quotient_symbol, args));
	  if ((is_inf(real(y))) || (is_NaN(real(y))))
	    return(wrong_type_argument_with_type(sc, sc->quotient_symbol, 2, y, a_normal_real_string));
	  return(s7_truncate(sc, sc->quotient_symbol, (s7_double)fraction(x) / real(y)));

	default:
	  return(method_or_bust(sc, y, sc->quotient_symbol, args, T_REAL, 2));
	}

    case T_REAL:
      if ((is_inf(real(x))) || (is_NaN(real(x))))
	return(wrong_type_argument_with_type(sc, sc->quotient_symbol, 1, x, a_normal_real_string));

      /* if infs allowed we need to return infs/nans, else:
       *    (quotient inf.0 1e-309) -> -9223372036854775808
       *    (quotient inf.0 inf.0) -> -9223372036854775808
       */

      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    return(division_by_zero_error(sc, sc->quotient_symbol, args));
	  return(s7_truncate(sc, sc->quotient_symbol, real(x) / (s7_double)integer(y)));

	case T_RATIO:
	  return(s7_truncate(sc, sc->quotient_symbol, real(x) / (s7_double)fraction(y)));

	case T_REAL:
	  return(make_real(sc, c_quo_dbl(sc, real(x), real(y))));

	default:
	  return(method_or_bust(sc, y, sc->quotient_symbol, args, T_REAL, 2));
	}

    default:
      return(method_or_bust(sc, x, sc->quotient_symbol, args, T_REAL, 2));
    }
}


static inline s7_int c_rem_int(s7_scheme *sc, s7_int x, s7_int y)
{
  if (y == 0)
    division_by_zero_error(sc, sc->remainder_symbol, set_elist_2(sc, wrap_integer1(sc, x), wrap_integer2(sc, y)));
  if ((y == 1) || (y == -1))   /* (remainder most-negative-fixnum -1) will segfault with arithmetic exception */
    return(0);
  return(x % y);
}

static s7_double c_rem_dbl(s7_scheme *sc, s7_double x, s7_double y)
{
  s7_int quo;
  s7_double pre_quo;
  if (y == 0.0)
    division_by_zero_error(sc, sc->remainder_symbol, set_elist_2(sc, wrap_real(sc, x), wrap_real2(sc, y)));
  if ((is_inf(y)) || (is_NaN(y)))
    wrong_type_argument_with_type(sc, sc->remainder_symbol, 2, set_elist_1(sc, wrap_real(sc, y)), a_normal_real_string);

  pre_quo = x / y;
  if ((pre_quo > s7_int_max) || (pre_quo < s7_int_min))
    simple_out_of_range(sc, sc->remainder_symbol, set_elist_2(sc, make_real(sc, x), wrap_real(sc, y)), its_too_large_string);
  if (pre_quo > 0.0)
    quo = (s7_int)floor(pre_quo);
  else quo = (s7_int)ceil(pre_quo);
  return(x - (y * quo));
}

static s7_int remainder_i_7ii(s7_scheme *sc, s7_int i1, s7_int i2) {return(c_rem_int(sc, i1, i2));}
static s7_int remainder_i_ii_direct(s7_int i1, s7_int i2) {return(i1 % i2);} /* i2 > 1 */

static s7_double remainder_d_7dd(s7_scheme *sc, s7_double x1, s7_double x2)
{
  if ((is_inf(x1)) || (is_NaN(x1)))
    wrong_type_argument_with_type(sc, sc->remainder_symbol, 1, set_elist_1(sc, wrap_real(sc, x1)), a_normal_real_string);
  return(c_rem_dbl(sc, x1, x2));
}

static s7_pointer g_remainder(s7_scheme *sc, s7_pointer args)
{
  #define H_remainder "(remainder x1 x2) returns the remainder of x1/x2; (remainder 10 3) = 1"
  #define Q_remainder sc->pcl_r
  /* (define (rem x1 x2) (- x1 (* x2 (quo x1 x2)))) ; slib, if x2 is an integer (- x1 (truncate x1 x2)), fractional part: (remainder x 1) */

  s7_pointer x, y;
  s7_int quo, d1, d2, n1, n2;
  s7_double pre_quo;

  x = car(args);
  y = cadr(args);

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_integer(sc, c_rem_int(sc, integer(x), integer(y))));

	case T_RATIO:
	  n1 = integer(x);
	  d1 = 1;
	  n2 = numerator(y);
	  d2 = denominator(y);
	  goto RATIO_REM_RATIO;

	case T_REAL:
	  if (real(y) == 0.0)
	    return(division_by_zero_error(sc, sc->remainder_symbol, args));
	  if ((is_inf(real(y))) || (is_NaN(real(y))))
	    return(wrong_type_argument_with_type(sc, sc->remainder_symbol, 2, y, a_normal_real_string));

	  pre_quo = (s7_double)integer(x) / real(y);
	  if ((pre_quo > s7_int_max) || (pre_quo < s7_int_min))
	    return(simple_out_of_range(sc, sc->remainder_symbol, args, its_too_large_string));
	  if (pre_quo > 0.0) quo = (s7_int)floor(pre_quo); else quo = (s7_int)ceil(pre_quo);
	  return(make_real(sc, integer(x) - real(y) * quo));

	default:
	  return(method_or_bust(sc, y, sc->remainder_symbol, args, T_REAL, 2));
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  n2 = integer(y);
 	  if (n2 == 0)
 	    return(division_by_zero_error(sc, sc->remainder_symbol, args));
	  n1 = numerator(x);
	  d1 = denominator(x);
	  d2 = 1;
	  goto RATIO_REM_RATIO;

	case T_RATIO:
	  n1 = numerator(x);
	  d1 = denominator(x);
	  n2 = numerator(y);
	  d2 = denominator(y);
	RATIO_REM_RATIO:
	  if (d1 == d2)
	    quo = (s7_int)(n1 / n2);
	  else
	    {
	      if (n1 == n2)
		quo = (s7_int)(d2 / d1);
	      else
		{
#if HAVE_OVERFLOW_CHECKS
		  s7_int n1d2, n2d1;
		  if ((multiply_overflow(n1, d2, &n1d2)) ||
		      (multiply_overflow(n2, d1, &n2d1)))
		    {
		      pre_quo = ((long_double)n1 / (long_double)n2) * ((long_double)d2 / (long_double)d1);
		      if ((pre_quo > s7_int_max) || (pre_quo < s7_int_min))
			return(simple_out_of_range(sc, sc->remainder_symbol, args, its_too_large_string));
		      if (pre_quo > 0.0) quo = (s7_int)floor(pre_quo); else quo = (s7_int)ceil(pre_quo);
		    }
		  else quo = n1d2 / n2d1;
#else
		  quo = (n1 * d2) / (n2 * d1);
#endif
		}
	    }
	  if (quo == 0)
	    return(x);

#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int dn, nq;
	    if (!multiply_overflow(n2, quo, &nq))
	      {
		if ((d1 == d2) &&
		    (!subtract_overflow(n1, nq, &dn)))
		  return(s7_make_ratio(sc, dn, d1));

		if ((!multiply_overflow(n1, d2, &dn)) &&
		    (!multiply_overflow(nq, d1, &nq)) &&
		    (!subtract_overflow(dn, nq, &nq)) &&
		    (!multiply_overflow(d1, d2, &d1)))
		  return(s7_make_ratio(sc, nq, d1));
	      }
	  }
#else
	  if (d1 == d2)
	    return(s7_make_ratio(sc, n1 - n2 * quo, d1));

	  return(s7_make_ratio(sc, n1 * d2 - n2 * d1 * quo, d1 * d2));
#endif
	  return(simple_out_of_range(sc, sc->remainder_symbol, args, wrap_string(sc, "intermediate (a/b) is too large", 31)));

	case T_REAL:
	  {
	    s7_double frac;
	    if (real(y) == 0.0)
	      return(division_by_zero_error(sc, sc->remainder_symbol, args));
	    if ((is_inf(real(y))) || (is_NaN(real(y))))
	      return(wrong_type_argument_with_type(sc, sc->remainder_symbol, 2, y, a_normal_real_string));
	    frac = (s7_double)fraction(x);
	    pre_quo = frac / real(y);
	    if ((pre_quo > s7_int_max) || (pre_quo < s7_int_min))
	      return(simple_out_of_range(sc, sc->remainder_symbol, args, its_too_large_string));
	    if (pre_quo > 0.0) quo = (s7_int)floor(pre_quo); else quo = (s7_int)ceil(pre_quo);
	    return(make_real(sc, frac - real(y) * quo));
	  }

	default:
	  return(method_or_bust(sc, y, sc->remainder_symbol, args, T_REAL, 2));
	}

    case T_REAL:
      if ((is_inf(real(x))) || (is_NaN(real(x))))
	return(wrong_type_argument_with_type(sc, sc->remainder_symbol, 1, x, a_normal_real_string));

      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    return(division_by_zero_error(sc, sc->remainder_symbol, args));
	  pre_quo = real(x) / (s7_double)integer(y);
	  if ((pre_quo > s7_int_max) || (pre_quo < s7_int_min))
	    return(simple_out_of_range(sc, sc->remainder_symbol, args, its_too_large_string));
	  if (pre_quo > 0.0) quo = (s7_int)floor(pre_quo); else quo = (s7_int)ceil(pre_quo);
	  return(make_real(sc, real(x) - integer(y) * quo));
	  /* but... (remainder 1e+18 9223372036854775807) -> 1e+18 */

	case T_RATIO:
	  {
	    /* bad cases here start around 1e16: (remainder 1e15 3/13) -> 0.0 with loss of digits earlier
	     *   would long double help?
	     */
	    s7_double frac;
	    frac = (s7_double)fraction(y);
	    pre_quo = real(x) / frac;
	    if ((pre_quo > s7_int_max) || (pre_quo < s7_int_min))
	      return(simple_out_of_range(sc, sc->remainder_symbol, args, its_too_large_string));
	    if (pre_quo > 0.0) quo = (s7_int)floor(pre_quo); else quo = (s7_int)ceil(pre_quo);
	    return(make_real(sc, real(x) - frac * quo));
	  }

	case T_REAL:
	  return(make_real(sc, c_rem_dbl(sc, real(x), real(y))));

	  /* see under sin -- this calculation is completely bogus if "a" is large
	   * (quotient 1e22 (* 2 pi)) -> -9223372036854775808 -- should this return arithmetic-overflow?
	   *          but it should be 1591549430918953357688,
	   * (remainder 1e22 (* 2 pi)) -> 1.0057952155665e+22
	   * -- the "remainder" is greater than the original argument!
	   * Clisp gives 0.0 here, as does sbcl
	   * currently s7 throws an error (out-of-range).
	   */

	default:
	  return(method_or_bust(sc, y, sc->remainder_symbol, args, T_REAL, 2));
	}

    default:
      return(method_or_bust(sc, x, sc->remainder_symbol, args, T_REAL, 1));
    }
}


/* -------------------------------- floor -------------------------------- */

#define REAL_TO_INT_LIMIT 9.2233727815085e+18
/* unfortunately, this limit is only a max in a sense: (ceiling 9223372036854770.9) => 9223372036854770
 *    see s7test for more examples
 */

static s7_pointer g_floor(s7_scheme *sc, s7_pointer args)
{
  #define H_floor "(floor x) returns the integer closest to x toward -inf"
  #define Q_floor s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)

  s7_pointer x;

  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      return(x);

    case T_RATIO:
      {
	s7_int val;
	val = numerator(x) / denominator(x);
	/* C "/" truncates? -- C spec says "truncation toward 0" */
	/* we're avoiding "floor" here because the int->double conversion introduces inaccuracies for big numbers */
	if (numerator(x) < 0) /* not "val" because it might be truncated to 0 */
	  return(make_integer(sc, val - 1));
	return(make_integer(sc, val));
      }

    case T_REAL:
      {
	s7_double z;
	z = real(x);
	if (is_NaN(z))
	  return(simple_out_of_range(sc, sc->floor_symbol, x, its_nan_string));
	if (fabs(z) > REAL_TO_INT_LIMIT)
	  return(simple_out_of_range(sc, sc->floor_symbol, x, its_too_large_string));
	return(make_integer(sc, (s7_int)floor(z)));
	/* floor here rounds down, whereas a straight int<=real coercion apparently rounds towards 0 */
      }

    case T_COMPLEX:
    default:
      return(method_or_bust_one_arg(sc, x, sc->floor_symbol, args, T_REAL));
    }
}

static s7_int floor_i_i(s7_int i) {return(i);}

static s7_int floor_i_7d(s7_scheme *sc, s7_double x)
{

  if (is_NaN(x))
    simple_out_of_range(sc, sc->floor_symbol, wrap_real(sc, x), its_nan_string);
  if (fabs(x) > REAL_TO_INT_LIMIT)
    simple_out_of_range(sc, sc->floor_symbol, wrap_real(sc, x), its_too_large_string);
  return((s7_int)floor(x));
}

static s7_int floor_i_7p(s7_scheme *sc, s7_pointer p)
{
  if (is_t_integer(p)) return(s7_integer(p));
  if (is_t_real(p)) return(floor_i_7d(sc, real(p)));
  if (is_t_ratio(p)) return((s7_int)(floor(fraction(p))));
  s7_wrong_type_arg_error(sc, "floor", 0, p, "a real number");
  return(0);
}


/* -------------------------------- ceiling -------------------------------- */
static s7_pointer g_ceiling(s7_scheme *sc, s7_pointer args)
{
  #define H_ceiling "(ceiling x) returns the integer closest to x toward inf"
  #define Q_ceiling s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)

  s7_pointer x;

  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      return(x);

    case T_RATIO:
      {
	s7_int val;
	val = numerator(x) / denominator(x);
	if (numerator(x) < 0)
	  return(make_integer(sc, val));
	return(make_integer(sc, val + 1));
      }

    case T_REAL:
      {
	s7_double z;
	z = real(x);
	if (is_NaN(z))
	  return(simple_out_of_range(sc, sc->ceiling_symbol, x, its_nan_string));
	if ((is_inf(z)) ||
	    (z > REAL_TO_INT_LIMIT) ||
	    (z < -REAL_TO_INT_LIMIT))
	  return(simple_out_of_range(sc, sc->ceiling_symbol, x, its_too_large_string));
	return(make_integer(sc, (s7_int)ceil(real(x))));
      }

    case T_COMPLEX:
    default:
      return(method_or_bust_one_arg(sc, x, sc->ceiling_symbol, args, T_REAL));
    }
}

static s7_int ceiling_i_i(s7_int i) {return(i);}

static s7_int ceiling_i_7d(s7_scheme *sc, s7_double x)
{
  if (is_NaN(x))
    simple_out_of_range(sc, sc->ceiling_symbol, wrap_real(sc, x), its_nan_string);
  if ((is_inf(x)) ||
      (x > REAL_TO_INT_LIMIT) ||
      (x < -REAL_TO_INT_LIMIT))
    simple_out_of_range(sc, sc->ceiling_symbol, wrap_real(sc, x), its_too_large_string);
  return((s7_int)ceil(x));
}

static s7_int ceiling_i_7p(s7_scheme *sc, s7_pointer p)
{
  if (is_t_integer(p)) return(s7_integer(p));
  if (is_t_real(p)) return(ceiling_i_7d(sc, real(p)));
  if (is_t_ratio(p)) return((s7_int)(ceil(fraction(p))));
  s7_wrong_type_arg_error(sc, "ceiling", 0, p, "a real number");
  return(0);
}


/* -------------------------------- truncate -------------------------------- */
static s7_pointer g_truncate(s7_scheme *sc, s7_pointer args)
{
  #define H_truncate "(truncate x) returns the integer closest to x toward 0"
  #define Q_truncate s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      return(x);

    case T_RATIO:
      return(make_integer(sc, (s7_int)(numerator(x) / denominator(x)))); /* C "/" already truncates */

    case T_REAL:
      {
	s7_double z;
	z = real(x);
	if (is_NaN(z))
	  return(simple_out_of_range(sc, sc->truncate_symbol, x, its_nan_string));
	if (is_inf(z))
	  return(simple_out_of_range(sc, sc->truncate_symbol, x, its_infinite_string));
	return(s7_truncate(sc, sc->truncate_symbol, real(x)));
      }

    case T_COMPLEX:
    default:
      return(method_or_bust_one_arg(sc, x, sc->truncate_symbol, args, T_REAL));
    }
}

static s7_int truncate_i_i(s7_int i) {return(i);}

static s7_int truncate_i_7d(s7_scheme *sc, s7_double x)
{
  if (is_NaN(x))
    simple_out_of_range(sc, sc->truncate_symbol, wrap_real(sc, x), its_nan_string);
  if (is_inf(x))
    simple_out_of_range(sc, sc->truncate_symbol, wrap_real(sc, x), its_infinite_string);
  if (x > 0.0)
    {
      if (x > s7_int_max)
	simple_out_of_range(sc, sc->truncate_symbol, wrap_real(sc, x), its_too_large_string);
      return((s7_int)floor(x));
    }
  if (x < s7_int_min)
    simple_out_of_range(sc, sc->truncate_symbol, wrap_real(sc, x), its_too_large_string);
  return((s7_int)ceil(x));
}


/* -------------------------------- round -------------------------------- */
static s7_double round_per_R5RS(s7_double x)
{
  s7_double fl, ce, dfl, dce;

  fl = floor(x);
  ce = ceil(x);
  dfl = x - fl;
  dce = ce - x;

  if (dfl > dce) return(ce);
  if (dfl < dce) return(fl);
  if (fmod(fl, 2.0) == 0.0) return(fl);
  return(ce);
}

static s7_pointer g_round(s7_scheme *sc, s7_pointer args)
{
  #define H_round "(round x) returns the integer closest to x"
  #define Q_round s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_real_symbol)

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
      return(x);

    case T_RATIO:
      {
	s7_int truncated, remains;
	long double frac;

	truncated = numerator(x) / denominator(x);
	remains = numerator(x) % denominator(x);
	frac = s7_fabsl((long_double)remains / (long_double)denominator(x));

	if ((frac > 0.5) ||
	    ((frac == 0.5) &&
	     (truncated % 2 != 0)))
	  {
	    if (numerator(x) < 0)
	      return(make_integer(sc, truncated - 1));
	    return(make_integer(sc, truncated + 1));
	  }
	return(make_integer(sc, truncated));
      }

    case T_REAL:
      {
	s7_double z;
	z = real(x);
	if (is_NaN(z))
	  return(simple_out_of_range(sc, sc->round_symbol, x, its_nan_string));
	if ((is_inf(z)) ||
	    (z > REAL_TO_INT_LIMIT) ||
	    (z < -REAL_TO_INT_LIMIT))
	  return(simple_out_of_range(sc, sc->round_symbol, x, its_too_large_string));
	return(make_integer(sc, (s7_int)round_per_R5RS(z)));
      }

    case T_COMPLEX:
    default:
      return(method_or_bust_one_arg(sc, x, sc->round_symbol, args, T_REAL));
    }
}

static s7_int round_i_i(s7_int i) {return(i);}

static s7_int round_i_7d(s7_scheme *sc, s7_double z)

{
  if (is_NaN(z))
    simple_out_of_range(sc, sc->round_symbol, wrap_real(sc, z), its_nan_string);
  if ((is_inf(z)) ||
      (z > REAL_TO_INT_LIMIT) ||
      (z < -REAL_TO_INT_LIMIT))
    simple_out_of_range(sc, sc->round_symbol, wrap_real(sc, z), its_too_large_string);
  return((s7_int)round_per_R5RS(z));
}


static s7_int c_mod(s7_int x, s7_int y)
{
  s7_int z;
  if (y == 0) return(x);     /* else arithmetic exception */
  if ((y == 1) || (y == -1)) /* else (modulo most-negative-fixnum -1) will segfault with arithmetic exception */
    return(0);
  z = x % y;
  if (((y < 0) && (z > 0)) ||
      ((y > 0) && (z < 0)))
    return(z + y);
  return(z);
}

static s7_int modulo_i_ii(s7_int i1, s7_int i2) {return(c_mod(i1, i2));}
static s7_int modulo_i_ii_direct(s7_int i1, s7_int i2)
{
  /* i2 > 1 */
  s7_int z;
  z = i1 % i2;
  if (z < 0)
    return(z + i2);
  return(z);
}
static s7_double modulo_d_dd(s7_double x1, s7_double x2) {return(x1 - x2 * (s7_int)floor(x1 / x2));}

static s7_pointer g_modulo(s7_scheme *sc, s7_pointer args)
{
  #define H_modulo "(modulo x1 x2) returns x1 mod x2; (modulo 4 3) = 1.  The arguments can be real numbers."
  #define Q_modulo sc->pcl_r
  /* (define (mod x1 x2) (- x1 (* x2 (floor (/ x1 x2))))) from slib
   * (mod x 0) = x according to "Concrete Mathematics"
   */
  s7_pointer x, y;
  s7_double a, b;
  s7_int n1, n2, d1, d2;

  x = car(args);
  y = cadr(args);

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(make_integer(sc, c_mod(integer(x), integer(y))));

	case T_RATIO:
	  n1 = integer(x);
	  d1 = 1;
	  n2 = numerator(y);
	  d2 = denominator(y);
	  goto RATIO_MOD_RATIO;

	case T_REAL:
	  {
	    double c;
	    b = real(y);
	    if (b == 0.0) return(x);
	    if (is_NaN(b)) return(y);
	    if (is_inf(b)) return(real_NaN);
	    a = (s7_double)integer(x);
	    c = a / b;
	    if ((c > 1e19) || (c < -1e19))
	      return(simple_out_of_range(sc, sc->modulo_symbol, y, wrap_string(sc, "intermediate (a/b) is too large", 31)));
	    return(make_real(sc, a - b * (s7_int)floor(c)));
	  }

	default:
	  return(method_or_bust(sc, y, sc->modulo_symbol, args, T_REAL, 2));
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0) return(x);
	  n1 = numerator(x);
	  d1 = denominator(x);
	  n2 = integer(y);

	  if ((n2 > 0) && (n1 > 0) && (n2 > n1)) return(x);
	  if ((n2 < 0) && (n1 < 0) && (n2 < n1)) return(x);

	  if (n2 == s7_int_min)
	    return(simple_out_of_range(sc, sc->modulo_symbol, y, wrap_string(sc, "intermediate (a/b) is too large", 31)));
	  /* the problem here is that (modulo 3/2 most-negative-fixnum)
	   * will segfault with signal SIGFPE, Arithmetic exception, so try to trap it.
	   */

	  d2 = 1;
	  goto RATIO_MOD_RATIO;

	case T_RATIO:
	  n1 = numerator(x);
	  d1 = denominator(x);
	  n2 = numerator(y); /* can't be 0 */
	  d2 = denominator(y);
	  if (d1 == d2)
	    return(s7_make_ratio(sc, c_mod(n1, n2), d1));

	RATIO_MOD_RATIO:

	  if ((n1 == n2) &&
	      (d1 > d2))
	    return(x);                 /* signs match so this should be ok */
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int n2d1, n1d2, d1d2, fl;
	    if (!multiply_overflow(n2, d1, &n2d1))
	      {
		if (n2d1 == 1)
		  return(small_int(0));

		if (!multiply_overflow(n1, d2, &n1d2))
		  {
		    /* can't use "floor" here (int->float ruins everything) */
		    fl = (s7_int)(n1d2 / n2d1);
		    if (((n1 < 0) && (n2 > 0)) ||
			((n1 > 0) && (n2 < 0)))
		      fl -= 1;

		    if (fl == 0)
		      return(x);

		    if ((!multiply_overflow(d1, d2, &d1d2)) &&
			(!multiply_overflow(fl, n2d1, &fl)) &&
			(!subtract_overflow(n1d2, fl, &fl)))
		      return(s7_make_ratio(sc, fl, d1d2));
		  }
	      }
	  }
#else
	  {
	    s7_int n1d2, n2d1, fl;
	    n1d2 = n1 * d2;
	    n2d1 = n2 * d1;

	    if (n2d1 == 1)
	      return(small_int(0));

	    /* can't use "floor" here (int->float ruins everything) */
	    fl = (s7_int)(n1d2 / n2d1);
	    if (((n1 < 0) && (n2 > 0)) ||
		((n1 > 0) && (n2 < 0)))
	      fl -= 1;

	    if (fl == 0)
	      return(x);

	    return(s7_make_ratio(sc, n1d2 - (n2d1 * fl), d1 * d2));
	  }
#endif

	  /* there are cases here we might want to catch:
	   *    (modulo 9223372036 1/9223372036) -> error, not 0?
	   *    (modulo 1 1/9223372036854775807) -> error, not 0?
	   */
	  return(simple_out_of_range(sc, sc->modulo_symbol, x, wrap_string(sc, "intermediate (a/b) is too large", 31)));

	case T_REAL:
	  b = real(y);
	  if (b == 0.0) return(x);
	  if (is_NaN(b)) return(y);
	  if (is_inf(b)) return(real_NaN);
	  a = fraction(x);
	  return(make_real(sc, a - b * (s7_int)floor(a / b)));

	default:
	  return(method_or_bust(sc, y, sc->modulo_symbol, args, T_REAL, 2));
	}

    case T_REAL:
      a = real(x);

      switch (type(y))
	{
	case T_INTEGER:
	  if (is_NaN(a)) return(x);
	  if (is_inf(a)) return(real_NaN);
	  if (integer(y) == 0) return(x);
	  b = (s7_double)integer(y);
	  return(make_real(sc, a - b * (s7_int)floor(a / b)));

	case T_RATIO:
	  if (is_NaN(a)) return(x);
	  if (is_inf(a)) return(real_NaN);
	  b = fraction(y);
	  return(make_real(sc, a - b * (s7_int)floor(a / b)));

	case T_REAL:
	  {
	    s7_double c;
	    if (is_NaN(a)) return(x);
	    if (is_inf(a)) return(real_NaN);
	    b = real(y);
	    if (b == 0.0) return(x);
	    if (is_NaN(b)) return(y);
	    if (is_inf(b)) return(real_NaN);
	    c = a / b;
	    if ((c > 1e19) || (c < -1e19))
	      return(simple_out_of_range(sc, sc->modulo_symbol, y, wrap_string(sc, "intermediate (a/b) is too large", 31)));
	    return(make_real(sc, a - b * (s7_int)floor(c)));
	  }

	default:
	  return(method_or_bust(sc, y, sc->modulo_symbol, args, T_REAL, 2));
	}

    default:
      return(method_or_bust(sc, x, sc->modulo_symbol, args, T_REAL, 1));
    }
}

static s7_pointer g_mod_si(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_int y;

  x = symbol_to_value_unchecked(sc, car(args));
  y = integer(cadr(args));

  if (is_integer(x))
    {
      s7_int z;
      /* here we know y is positive */
      z = integer(x) % y;
      if (z < 0)
	return(make_integer(sc, z + y));
      return(make_integer(sc, z));
    }

  if (is_t_real(x))
    {
      s7_double a, b;
      a = real(x);
      if (is_NaN(a)) return(x);
      if (is_inf(a)) return(real_NaN);
      b = (s7_double)y;
      return(make_real(sc, a - b * (s7_int)floor(a / b)));
    }

  if (s7_is_ratio(x))
    return(g_modulo(sc, set_plist_2(sc, x, cadr(args))));

  return(method_or_bust(sc, x, sc->modulo_symbol, list_2(sc, x, cadr(args)), T_REAL, 1));
}
#endif
/* !WITH_GMP */


static int32_t reduce_fraction(s7_int *numer, s7_int *denom)
{
  /* we're assuming in several places that we have a normal s7 rational after returning,
   *    so the denominator needs to be positive.
   */
  s7_int divisor;

  if (*numer == 0)
    {
      *denom = 1;
      return(T_INTEGER);
    }
  if (*denom < 0)
    {
      if (*denom == *numer)
	{
	  *denom = 1;
	  *numer = 1;
	  return(T_INTEGER);
	}
      if (*denom == s7_int_min)
	{
	  if (*numer & 1)
	    return(T_RATIO);
	  *denom /= 2;
	  *numer /= 2;
	}
      else
	{
	  if (*numer == s7_int_min)
	    {
	      if (*denom & 1)
		return(T_RATIO);
	      *denom /= 2;
	      *numer /= 2;
	    }
	}
      *denom = -*denom;
      *numer = -*numer;
    }
  divisor = c_gcd(*numer, *denom);
  if (divisor != 1)
    {
      *numer /= divisor;
      *denom /= divisor;
    }
  if (*denom == 1)
    return(T_INTEGER);
  return(T_RATIO);
}


/* ---------------------------------------- add ---------------------------------------- */

static s7_pointer g_add(s7_scheme *sc, s7_pointer args)
{
  #define H_add "(+ ...) adds its arguments"
  #define Q_add sc->pcl_n
  s7_pointer x, p;
  s7_int num_a, den_a, dn;
  s7_double rl_a, im_a;

#if (!WITH_GMP)
  if (is_null(args))
    return(small_int(0));
#endif

  x = car(args);
  p = cdr(args);
  if (is_null(p))
    {
      if (!is_number(x))
	return(method_or_bust_with_type_one_arg(sc, x, sc->add_symbol, args, a_number_string));
      return(x);
    }

  switch (type(x))
    {
    case T_INTEGER:
      num_a = integer(x);

    ADD_INTEGERS:
#if WITH_GMP
      if ((num_a > s7_int32_max) ||
	  (num_a < s7_int32_min))
	return(big_add(sc, cons(sc, s7_int_to_big_integer(sc, num_a), p)));
#endif
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
#if HAVE_OVERFLOW_CHECKS
	  if (add_overflow(num_a, integer(x), &den_a))
	    {
	      rl_a = (s7_double)num_a + (s7_double)integer(x);
	      if (is_null(p)) return(make_real(sc, rl_a));
	      goto ADD_REALS;
	    }
#else
	  den_a = num_a + integer(x);
	  if (den_a < 0)
	    {
	      if ((num_a > 0) && (integer(x) > 0))
		{
		  rl_a = (s7_double)num_a + (s7_double)integer(x);
		  if (is_null(p)) return(make_real(sc, rl_a));
		  goto ADD_REALS;
		}
	    }
	  else
	    {
	      if ((num_a < 0) && (integer(x) < 0))
		{
		  rl_a = (s7_double)num_a + (s7_double)integer(x);
		  if (is_null(p)) return(make_real(sc, rl_a));

		  /* this is not ideal!  piano.scm has its own noise generator that wants integer
		   *    arithmetic to overflow as an integer.  Perhaps 'safety==0 would not check
		   *    anywhere?
		   */
		  goto ADD_REALS;
		}
	    }
#endif
	  if (is_null(p)) return(make_integer(sc, den_a));
	  num_a = den_a;
	  /* (+ 4611686018427387904 4611686018427387904) -> -9223372036854775808
	   * (+ most-positive-fixnum most-positive-fixnum) -> -2
	   * (+ most-negative-fixnum most-negative-fixnum) -> 0
	   * can't check result - arg: (- 0 most-negative-fixnum) -> most-negative-fixnum
	   */
	  goto ADD_INTEGERS;

	case T_RATIO:
	  den_a = denominator(x);
#if HAVE_OVERFLOW_CHECKS
	  if ((multiply_overflow(den_a, num_a, &dn)) ||
	      (add_overflow(dn, numerator(x), &dn)))
	    {
	      if (is_null(p))
		{
		  if (num_a == 0)                /* (+ 0 1/9223372036854775807) */
		    return(x);
		  return(make_real(sc, num_a + fraction(x)));
		}
	      rl_a = (s7_double)num_a + fraction(x);
	      goto ADD_REALS;
	    }
#else
	  dn = numerator(x) + (num_a * den_a);
#endif
	  if (is_null(p)) return(s7_make_ratio(sc, dn, den_a));
	  num_a = dn;

	  /* overflow examples:
	   *   (+ 100000 1/142857142857140) -> -832205957599110323/28571428571428
	   *   (+ 4611686018427387904 3/4) -> 3/4
	   * see s7test for more
	   */
	  goto ADD_RATIOS;

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, num_a + real(x)));
	  rl_a = (s7_double)num_a + real(x);
	  goto ADD_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, num_a + real_part(x), imag_part(x)));
	  rl_a = (s7_double)num_a + real_part(x);
	  im_a = imag_part(x);
	  goto ADD_COMPLEX;

	default:
	  return(method_or_bust_with_type(sc, x, sc->add_symbol, cons_unchecked(sc, s7_make_integer(sc, num_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	}
      break;

    case T_RATIO:
      num_a = numerator(x);
      den_a = denominator(x);
    ADD_RATIOS:
#if WITH_GMP
      if ((num_a > s7_int32_max) ||
	  (den_a > s7_int32_max) ||
	  (num_a < s7_int32_min))
	return(big_add(sc, cons(sc, s7_ratio_to_big_ratio(sc, num_a, den_a), p)));
#endif
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
#if HAVE_OVERFLOW_CHECKS
	  if ((multiply_overflow(den_a, integer(x), &dn)) ||
	      (add_overflow(dn, num_a, &dn)))
	    {
	      /* (+ 3/4 4611686018427387904) -> 3/4
	       * (+ 1/17179869184 1073741824) -> 1/17179869184
	       * (+ 1/8589934592 1073741824) -> -9223372036854775807/8589934592
	       */
	      if (is_null(p))
		return(make_real(sc, (s7_double)integer(x) + ((long_double)num_a / (long_double)den_a)));
	      rl_a = (s7_double)integer(x) + ((long_double)num_a / (long_double)den_a);
	      goto ADD_REALS;
	    }
#else
	  dn = num_a + (integer(x) * den_a);
#endif
	  if (is_null(p)) return(s7_make_ratio(sc, dn, den_a));
	  num_a = dn;
	  if (reduce_fraction(&num_a, &den_a) == T_INTEGER)
	    goto ADD_INTEGERS;
	  goto ADD_RATIOS;

	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    d1 = den_a;
	    n1 = num_a;
	    d2 = denominator(x);
	    n2 = numerator(x);
	    if (d1 == d2)                                     /* the easy case -- if overflow here, it matches the int32_t case */
	      {
		if (is_null(p))
		  return(s7_make_ratio(sc, n1 + n2, d1));
		num_a += n2;                  /* d1 can't be zero */
	      }
	    else
	      {
#if (!WITH_GMP)
#if HAVE_OVERFLOW_CHECKS
		s7_int n1d2, n2d1;
		if ((multiply_overflow(d1, d2, &den_a)) ||
		    (multiply_overflow(n1, d2, &n1d2)) ||
		    (multiply_overflow(n2, d1, &n2d1)) ||
		    (add_overflow(n1d2, n2d1, &num_a)))
		  {
		    if (is_null(p))
		      return(make_real(sc, ((long_double)n1 / (long_double)d1) + ((long_double)n2 / (long_double)d2)));
		    rl_a = ((long_double)n1 / (long_double)d1) + ((long_double)n2 / (long_double)d2);
		    goto ADD_REALS;
		  }
#else
		num_a = n1 * d2 + n2 * d1;
		den_a = d1 * d2;
#endif
#else
		num_a = n1 * d2 + n2 * d1;
		den_a = d1 * d2;
#endif
		if (is_null(p))
		  return(s7_make_ratio(sc, num_a, den_a));
	      }
	    /* (+ 1/100 99/100 (- most-positive-fixnum 2)) should not be converted to real
	     */
	    if (reduce_fraction(&num_a, &den_a) == T_INTEGER)
	    goto ADD_INTEGERS;
	  goto ADD_RATIOS;
	  }

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, ((long_double)num_a / (long_double)den_a) + real(x)));
	  rl_a = ((long_double)num_a / (long_double)den_a) + real(x);
	  goto ADD_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, ((long_double)num_a / (long_double)den_a) + real_part(x), imag_part(x)));
	  rl_a = ((long_double)num_a / (long_double)den_a) + real_part(x);
	  im_a = imag_part(x);
	  goto ADD_COMPLEX;

	default:
	  return(method_or_bust_with_type(sc, x, sc->add_symbol, cons_unchecked(sc, s7_make_ratio(sc, num_a, den_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	}
      break;

    case T_REAL:
      rl_a = real(x);

    ADD_REALS:
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  if (is_null(p)) return(make_real(sc, rl_a + integer(x)));
	  rl_a += (s7_double)integer(x);
	  goto ADD_REALS;

	case T_RATIO:
	  if (is_null(p)) return(make_real(sc, rl_a + fraction(x)));
	  rl_a += (s7_double)fraction(x);
	  goto ADD_REALS;

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, rl_a + real(x)));
	  rl_a += real(x);
	  goto ADD_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a + real_part(x), imag_part(x)));
	  rl_a += real_part(x);
	  im_a = imag_part(x);
	  goto ADD_COMPLEX;

	default:
	  return(method_or_bust_with_type(sc, x, sc->add_symbol, cons_unchecked(sc, make_real(sc, rl_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	}
      break;

    case T_COMPLEX:
      rl_a = real_part(x);
      im_a = imag_part(x);

    ADD_COMPLEX:
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a + integer(x), im_a));
	  rl_a += (s7_double)integer(x);
	  goto ADD_COMPLEX;

	case T_RATIO:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a + fraction(x), im_a));
	  rl_a += (s7_double)fraction(x);
	  goto ADD_COMPLEX;

	case T_REAL:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a + real(x), im_a));
	  rl_a += real(x);
	  goto ADD_COMPLEX;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a + real_part(x), im_a + imag_part(x)));
	  rl_a += real_part(x);
	  im_a += imag_part(x);
	  if (im_a == 0.0)
	    goto ADD_REALS;
	  goto ADD_COMPLEX;

	default:
	  return(method_or_bust_with_type(sc, x, sc->add_symbol, cons_unchecked(sc, s7_make_complex(sc, rl_a, im_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	}
      break;

    default:
      return(method_or_bust_with_type(sc, x, sc->add_symbol, args, a_number_string, 1));
    }
  return(NULL); /* make the compiler happy */
}

static s7_pointer add_ratios_1(s7_scheme *sc, s7_int n1, s7_int d1, s7_int n2, s7_int d2)
{
  if (d1 == d2)                                     /* the easy case -- if overflow here, it matches the int case */
    return(s7_make_ratio(sc, n1 + n2, d1));

#if HAVE_OVERFLOW_CHECKS
  {
    s7_int n1d2, n2d1, d1d2, dn;
    if ((multiply_overflow(d1, d2, &d1d2)) ||
	(multiply_overflow(n1, d2, &n1d2)) ||
	(multiply_overflow(n2, d1, &n2d1)) ||
	(add_overflow(n1d2, n2d1, &dn)))
      return(make_real(sc, ((long_double)n1 / (long_double)d1) + ((long_double)n2 / (long_double)d2)));
    return(s7_make_ratio(sc, dn, d1d2));
  }
#else
  return(s7_make_ratio(sc, n1 * d2 + n2 * d1, d1 * d2));
#endif
}

static s7_pointer add_ratios(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  return(add_ratios_1(sc, number_to_numerator(x), number_to_denominator(x), number_to_numerator(y), number_to_denominator(y)));
}

static s7_pointer add_out_x(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_methods(x))
    return(find_and_apply_method(sc, find_let(sc, x), sc->add_symbol, list_2(sc, x, y)));
  return(wrong_type_argument_with_type(sc, sc->add_symbol, 1, x, a_number_string));
}

static s7_pointer add_out_y(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_methods(y))
    return(find_and_apply_method(sc, find_let(sc, y), sc->add_symbol, list_2(sc, x, y)));
  return(wrong_type_argument_with_type(sc, sc->add_symbol, 2, y, a_number_string));
}

static inline s7_pointer add_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (type(x) == type(y))
    {
      if (is_t_real(x))
	return(make_real(sc, real(x) + real(y)));
      else
	{
	  switch (type(x))
	    {
#if HAVE_OVERFLOW_CHECKS
	    case T_INTEGER:
	      {
		s7_int val;
		if (add_overflow(integer(x), integer(y), &val))
		  return(make_real(sc, (double)integer(x) + (double)integer(y)));
		return(make_integer(sc, val));
	      }
#else
	    case T_INTEGER: return(make_integer(sc, integer(x) + integer(y)));
#endif
	    case T_RATIO:   return(add_ratios(sc, x, y));
	    case T_REAL:    return(make_real(sc, real(x) + real(y)));
	    case T_COMPLEX: return(make_complex(sc, real_part(x) + real_part(y), imag_part(x) + imag_part(y)));
	    default:
	      if (!is_number(x))
		return(add_out_x(sc, x, y));
	      return(add_out_y(sc, x, y));
	    }
	}
    }

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER: return(make_integer(sc, integer(x) + integer(y)));
	case T_RATIO:   return(add_ratios(sc, x, y));
	case T_REAL:    return(make_real(sc, integer(x) + real(y)));
	case T_COMPLEX: return(make_complex(sc, integer(x) + real_part(y), imag_part(y)));
	default:
	  return(add_out_y(sc, x, y));
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	case T_RATIO:   return(add_ratios(sc, x, y));
	case T_REAL:    return(make_real(sc, fraction(x) + real(y)));
	case T_COMPLEX: return(s7_make_complex(sc, fraction(x) + real_part(y), imag_part(y)));
	default:
	  return(add_out_y(sc, x, y));
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: return(make_real(sc, real(x) + integer(y)));
	case T_RATIO:   return(make_real(sc, real(x) + fraction(y)));
	case T_REAL:    return(make_real(sc, real(x) + real(y)));
	case T_COMPLEX: return(make_complex(sc, real(x) + real_part(y), imag_part(y)));
	default:
	  return(add_out_y(sc, x, y));
	}

    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER: return(s7_make_complex(sc, real_part(x) + integer(y), imag_part(x)));
	case T_RATIO:   return(s7_make_complex(sc, real_part(x) + fraction(y), imag_part(x)));
	case T_REAL:    return(s7_make_complex(sc, real_part(x) + real(y), imag_part(x)));
	case T_COMPLEX: return(make_complex(sc, real_part(x) + real_part(y), imag_part(x) + imag_part(y)));
	default:
	  return(add_out_y(sc, x, y));
	}

    default:
      return(add_out_x(sc, x, y));
    }
  return(x);
}

static s7_pointer g_add_2(s7_scheme *sc, s7_pointer args) {return(add_p_pp(sc, car(args), cadr(args)));}

static s7_pointer g_add_s1_1(s7_scheme *sc, s7_pointer x, s7_pointer args)
{
  switch (type(x))
    {
#if HAVE_OVERFLOW_CHECKS
    case T_INTEGER:
      {
	s7_int val;
	if (add_overflow(integer(x), 1, &val))
	  return(make_real(sc, (double)integer(x) + 1.0));
	return(make_integer(sc, val));
      }
#else
    case T_INTEGER: return(make_integer(sc, integer(x) + 1));
#endif
    case T_RATIO:   return(add_ratios(sc, x, small_int(1)));
    case T_REAL:    return(make_real(sc, real(x) + 1.0));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) + 1.0, imag_part(x)));
    default:
      return(method_or_bust_with_type(sc, x, sc->add_symbol, cons(sc, x, cdr(args)), a_number_string, 1));
    }
  return(x);
}

static s7_pointer g_add_s1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  x = car(args);
  if (is_t_integer(x))
    return(make_integer(sc, integer(x) + 1));
  return(g_add_s1_1(sc, x, args));
}

static s7_pointer g_add_cs1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
#if 0
  fprintf(stderr, "%s\n", DISPLAY_80(args));
  fprintf(stderr, "    %s\n", DISPLAY_80(sc->cur_code));
#endif
  x = symbol_to_value_unchecked(sc, car(args));
  if (is_integer(x))
    return(make_integer(sc, integer(x) + 1));
  return(g_add_s1_1(sc, x, args));
}

static s7_pointer g_add_1s(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;

  x = cadr(args);
  if (is_integer(x))
    return(make_integer(sc, integer(x) + 1));

  switch (type(x))
    {
    case T_INTEGER: return(make_integer(sc, integer(x) + 1));
    case T_RATIO:   return(add_ratios(sc, x, small_int(1)));
    case T_REAL:    return(make_real(sc, real(x) + 1.0));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) + 1.0, imag_part(x)));
    default:
      return(method_or_bust_with_type(sc, x, sc->add_symbol, args, a_number_string, 2));
    }
  return(x);
}

static s7_pointer g_add_sis(s7_scheme *sc, s7_pointer x, s7_int y)
{
  switch (type(x))
    {
    case T_INTEGER:
#if HAVE_OVERFLOW_CHECKS
      {
	s7_int val;
	if (add_overflow(integer(x), y, &val))
	  return(make_real(sc, (double)integer(x) + (double)y));
	return(make_integer(sc, val));
      }
#else
      return(make_integer(sc, integer(x) + y));
#endif
    case T_RATIO:   return(add_ratios_1(sc, numerator(x), denominator(x), y, 1));
    case T_REAL:    return(make_real(sc, real(x) + y));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) + y, imag_part(x)));
    default:
      return(method_or_bust_with_type(sc, x, sc->add_symbol, list_2(sc, x, make_integer(sc, y)), a_number_string, 1));
    }
  return(x);

}

static s7_pointer g_add_si(s7_scheme *sc, s7_pointer args) {return(g_add_sis(sc, symbol_to_value_unchecked(sc, car(args)), integer(cadr(args))));}

static s7_pointer g_add_sfs(s7_scheme *sc, s7_pointer x, s7_double y)
{
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) + y));
    case T_RATIO:   return(make_real(sc, fraction(x) + y));
    case T_REAL:    return(make_real(sc, real(x) + y));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) + y, imag_part(x)));
    default:
      return(method_or_bust_with_type(sc, x, sc->add_symbol, list_2(sc, x, make_real(sc, y)), a_number_string, 1));
    }
  return(x);
}

static s7_pointer g_add_sf(s7_scheme *sc, s7_pointer args) {return(g_add_sfs(sc, symbol_to_value_unchecked(sc, car(args)), real(cadr(args))));}
static s7_pointer g_add_fs(s7_scheme *sc, s7_pointer args) {return(g_add_sfs(sc, symbol_to_value_unchecked(sc, cadr(args)), real(car(args))));}

static s7_pointer g_add_f_sf(s7_scheme *sc, s7_pointer args)
{
  /* (+ x (* s y)) */
  s7_pointer vargs, s;
  s7_double x, y;

  x = real(car(args));
  vargs = cdadr(args);
  s = symbol_to_value_unchecked(sc, car(vargs));
  y = real(cadr(vargs));

  if (is_t_real(s))
    return(make_real(sc, x + (real(s) * y)));

  switch (type(s))
    {
    case T_INTEGER: return(make_real(sc, x + (integer(s) * y)));
    case T_RATIO:   return(make_real(sc, x + (fraction(s) * y)));
    case T_REAL:    return(make_real(sc, x + real(s) * y));
    case T_COMPLEX: return(s7_make_complex(sc, x + (real_part(s) * y), imag_part(s) * y));
    default:
      {
	s7_pointer func;
	if ((func = find_method(sc, find_let(sc, s), sc->multiply_symbol)) != sc->undefined)
	  return(add_p_pp(sc, car(args), s7_apply_function(sc, func, list_2(sc, s, cadr(vargs)))));
	return(wrong_type_argument_with_type(sc, sc->multiply_symbol, 1, s, a_number_string));
      }
    }
  return(s);
}

#if (!WITH_GMP)
static s7_pointer g_add_2_ff(s7_scheme *sc, s7_pointer args) {return(make_real(sc, real(car(args)) + real(cadr(args))));}
static s7_pointer g_add_2_ii(s7_scheme *sc, s7_pointer args)
{
#if HAVE_OVERFLOW_CHECKS
  s7_int val, x, y;
  x = integer(car(args));
  y = integer(cadr(args));
  if (add_overflow(x, y, &val))
    return(make_real(sc, (double)x + (double)y));
  return(make_integer(sc, val));
#else
  return(make_integer(sc, integer(car(args)) + integer(cadr(args))));
#endif
}

static s7_pointer g_add_2_if(s7_scheme *sc, s7_pointer args) {return(make_real(sc, integer(car(args)) + real(cadr(args))));}
static s7_pointer g_add_2_fi(s7_scheme *sc, s7_pointer args) {return(make_real(sc, real(car(args)) + integer(cadr(args))));}

static s7_pointer g_add_2_xi(s7_scheme *sc, s7_pointer args) {return(g_add_sis(sc, car(args), integer(cadr(args))));}
static s7_pointer g_add_2_ix(s7_scheme *sc, s7_pointer args) {return(g_add_sis(sc, cadr(args), integer(car(args))));}
static s7_pointer g_add_2_xf(s7_scheme *sc, s7_pointer args) {return(g_add_sfs(sc, car(args), real(cadr(args))));}
static s7_pointer g_add_2_fx(s7_scheme *sc, s7_pointer args) {return(g_add_sfs(sc, cadr(args), real(car(args))));}

static s7_pointer add_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_real(sc, x1 + x2));}
/* add_p_ii and add_d_id unhittable apparently */
#endif

static s7_double add_d_d(s7_double x) {return(x);}
static s7_double add_d_dd(s7_double x1, s7_double x2) {return(x1 + x2);}
static s7_double add_d_ddd(s7_double x1, s7_double x2, s7_double x3) {return(x1 + x2 + x3);}
static s7_double add_d_dddd(s7_double x1, s7_double x2, s7_double x3, s7_double x4) {return(x1 + x2 + x3 + x4);}

static s7_int add_i_ii(s7_int i1, s7_int i2) {return(i1 + i2);}
static s7_int add_i_iii(s7_int i1, s7_int i2, s7_int i3) {return(i1 + i2 + i3);}


/* ---------------------------------------- subtract ---------------------------------------- */

static s7_pointer g_subtract(s7_scheme *sc, s7_pointer args)
{
  #define H_subtract "(- x1 ...) subtracts its trailing arguments from the first, or negates the first if only one it is given"
  #define Q_subtract sc->pcl_n

  s7_pointer x, p;
  s7_int num_a, den_a;
  s7_double rl_a, im_a;

  x = car(args);
  p = cdr(args);

#if (!WITH_GMP)
  if (is_null(p))
    {
      if (!is_number(x))
	return(method_or_bust_with_type_one_arg(sc, x, sc->subtract_symbol, args, a_number_string));
      return(s7_negate(sc, x));
    }
#endif

  switch (type(x))
    {
    case T_INTEGER:
      num_a = integer(x);

    SUBTRACT_INTEGERS:
#if WITH_GMP
      if ((num_a > s7_int32_max) ||
	  (num_a < s7_int32_min))
	return(big_subtract(sc, cons(sc, s7_int_to_big_integer(sc, num_a), p)));
#endif
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
#if HAVE_OVERFLOW_CHECKS
	  if (subtract_overflow(num_a, integer(x), &den_a))
	    {
	      rl_a = (s7_double)num_a - (s7_double)integer(x);
	      if (is_null(p)) return(make_real(sc, rl_a));
	      goto SUBTRACT_REALS;
	    }
#else
	  den_a = num_a - integer(x);
	  if (den_a < 0)
	    {
	      if ((num_a > 0) && (integer(x) < 0))
		{
		  rl_a = (s7_double)num_a - (s7_double)integer(x);
		  if (is_null(p)) return(make_real(sc, rl_a));
		  goto SUBTRACT_REALS;
		}
	      /* (- most-positive-fixnum most-negative-fixnum) -> -1 (1.8446744073709551615E19)
	       */
	    }
	  else
	    {
	      if ((num_a < 0) && (integer(x) > 0))
		{
		  rl_a = (s7_double)num_a - (s7_double)integer(x);
		  if (is_null(p)) return(make_real(sc, rl_a));
		  goto SUBTRACT_REALS;
		}
	      /* (- most-negative-fixnum most-positive-fixnum) -> 1 (-1.8446744073709551615E19)
	       */
	    }
#endif
	  if (is_null(p)) return(make_integer(sc, den_a));
	  num_a = den_a;
	  goto SUBTRACT_INTEGERS;

	case T_RATIO:
	  {
	    s7_int dn;
	    den_a = denominator(x);
#if HAVE_OVERFLOW_CHECKS
	    if ((multiply_overflow(num_a, den_a, &dn)) ||
		(subtract_overflow(dn, numerator(x), &dn)))
	      {
		if (is_null(p)) return(make_real(sc, num_a - fraction(x)));
		rl_a = (s7_double)num_a - fraction(x);
		goto SUBTRACT_REALS;
	      }
#else
	    dn = (num_a * den_a) - numerator(x);
#endif
	    if (is_null(p)) return(s7_make_ratio(sc, dn, den_a));
	    num_a = dn;
	    if (reduce_fraction(&num_a, &den_a) == T_INTEGER)
	      goto SUBTRACT_INTEGERS;
	    goto SUBTRACT_RATIOS;
	  }

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, num_a - real(x)));
	  rl_a = (s7_double)num_a - real(x);
	  goto SUBTRACT_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, num_a - real_part(x), -imag_part(x)));
	  rl_a = (s7_double)num_a - real_part(x);
	  im_a = -imag_part(x);
	  goto SUBTRACT_COMPLEX;

	default:
	  return(method_or_bust_with_type(sc, x, sc->subtract_symbol, cons_unchecked(sc, s7_make_integer(sc, num_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	}
      break;

    case T_RATIO:
      num_a = numerator(x);
      den_a = denominator(x);
    SUBTRACT_RATIOS:
#if WITH_GMP
      if ((num_a > s7_int32_max) ||
	  (den_a > s7_int32_max) ||
	  (num_a < s7_int32_min))
	return(big_subtract(sc, cons(sc, s7_ratio_to_big_ratio(sc, num_a, den_a), p)));
#endif
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int di;
	    if ((multiply_overflow(den_a, integer(x), &di)) ||
		(subtract_overflow(num_a, di, &di)))
	      {
		if (is_null(p)) return(make_real(sc, ((long_double)num_a / (long_double)den_a) - integer(x)));
		rl_a = ((long_double)num_a / (long_double)den_a) - integer(x);
		goto SUBTRACT_REALS;
	      }
	    if (is_null(p)) return(s7_make_ratio(sc, di, den_a));
	    num_a = di;
	  }
#else
	  if (is_null(p)) return(s7_make_ratio(sc, num_a - (den_a * integer(x)), den_a));
	  num_a -= (den_a * integer(x));
#endif
	  if (reduce_fraction(&num_a, &den_a) == T_INTEGER)
	    goto SUBTRACT_INTEGERS;
	  goto SUBTRACT_RATIOS;

	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    d1 = den_a;
	    n1 = num_a;
	    d2 = denominator(x);
	    n2 = numerator(x);
	    if (d1 == d2)                                     /* the easy case -- if overflow here, it matches the int32_t case */
	      {
		if (is_null(p))
		  return(s7_make_ratio(sc, n1 - n2, d1));
		num_a -= n2;                  /* d1 can't be zero */
	      }
	    else
	      {
#if (!WITH_GMP) && HAVE_OVERFLOW_CHECKS
		s7_int n1d2, n2d1;
		if ((multiply_overflow(d1, d2, &den_a)) ||
		    (multiply_overflow(n1, d2, &n1d2)) ||
		    (multiply_overflow(n2, d1, &n2d1)) ||
		    (subtract_overflow(n1d2, n2d1, &num_a)))
		  {
		    if (is_null(p))
		      return(make_real(sc, ((long_double)n1 / (long_double)d1) - ((long_double)n2 / (long_double)d2)));
		    rl_a = ((long_double)n1 / (long_double)d1) - ((long_double)n2 / (long_double)d2);
		    goto SUBTRACT_REALS;
		  }
#else
		num_a = n1 * d2 - n2 * d1;
		den_a = d1 * d2;
#endif
		if (is_null(p))
		  return(s7_make_ratio(sc, num_a, den_a));
	      }
	    if (reduce_fraction(&num_a, &den_a) == T_INTEGER)
	    goto SUBTRACT_INTEGERS;
	  goto SUBTRACT_RATIOS;
	  }

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, ((long_double)num_a / (long_double)den_a) - real(x)));
	  rl_a = ((long_double)num_a / (long_double)den_a) - real(x);
	  goto SUBTRACT_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, ((long_double)num_a / (long_double)den_a) - real_part(x), -imag_part(x)));
	  rl_a = ((long_double)num_a / (long_double)den_a) - real_part(x);
	  im_a = -imag_part(x);
	  goto SUBTRACT_COMPLEX;

	default:
	  return(method_or_bust_with_type(sc, x, sc->subtract_symbol, cons_unchecked(sc, s7_make_ratio(sc, num_a, den_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	}
      break;

    case T_REAL:
      rl_a = real(x);

    SUBTRACT_REALS:
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  if (is_null(p)) return(make_real(sc, rl_a - integer(x)));
	  rl_a -= (s7_double)integer(x);
	  goto SUBTRACT_REALS;

	case T_RATIO:
	  if (is_null(p)) return(make_real(sc, rl_a - fraction(x)));
	  rl_a -= (s7_double)fraction(x);
	  goto SUBTRACT_REALS;

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, rl_a - real(x)));
	  rl_a -= real(x);
	  goto SUBTRACT_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a - real_part(x), -imag_part(x)));
	  rl_a -= real_part(x);
	  im_a = -imag_part(x);
	  goto SUBTRACT_COMPLEX;

	default:
	  return(method_or_bust_with_type(sc, x, sc->subtract_symbol, cons_unchecked(sc, make_real(sc, rl_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	}
      break;

    case T_COMPLEX:
      rl_a = real_part(x);
      im_a = imag_part(x);

    SUBTRACT_COMPLEX:
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a - integer(x), im_a));
	  rl_a -= (s7_double)integer(x);
	  goto SUBTRACT_COMPLEX;

	case T_RATIO:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a - fraction(x), im_a));
	  rl_a -= (s7_double)fraction(x);
	  goto SUBTRACT_COMPLEX;

	case T_REAL:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a - real(x), im_a));
	  rl_a -= real(x);
	  goto SUBTRACT_COMPLEX;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a - real_part(x), im_a - imag_part(x)));
	  rl_a -= real_part(x);
	  im_a -= imag_part(x);
	  if (im_a == 0.0)
	    goto SUBTRACT_REALS;
	  goto SUBTRACT_COMPLEX;

	default:
	  return(method_or_bust_with_type(sc, x, sc->subtract_symbol, cons_unchecked(sc, s7_make_complex(sc, rl_a, im_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	}
      break;

    default:
      return(method_or_bust_with_type(sc, x, sc->subtract_symbol, args, a_number_string, 1));
    }
  return(NULL); /* make the compiler happy */
}


static s7_pointer g_subtract_1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer p;

  p = car(args);
  switch (type(p))
    {
    case T_INTEGER:
      if (integer(p) == s7_int_min)
#if WITH_GMP
	return(big_negate(sc, set_plist_1(sc, promote_number(sc, T_BIG_INTEGER, p))));
#else
      return(simple_out_of_range(sc, sc->subtract_symbol, p, wrap_string(sc, "most-negative-fixnum can't be negated", 37)));
#endif
      return(make_integer(sc, -integer(p)));

    case T_RATIO:
      return(make_simple_ratio(sc, -numerator(p), denominator(p)));

    case T_REAL:
      return(make_real(sc, -real(p)));

    case T_COMPLEX:
      return(s7_make_complex(sc, -real_part(p), -imag_part(p)));

    default:
      return(method_or_bust_with_type(sc, p, sc->subtract_symbol, args, a_number_string, 1));
    }
}

static inline s7_pointer subtract_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y) /* inline here simply trades overheads with subtract_2 et al */
{
  if (type(x) == type(y))
    {
      if (is_t_real(x))
	return(make_real(sc, real(x) - real(y)));
      else
	{
	  switch (type(x))
	    {
#if HAVE_OVERFLOW_CHECKS
	    case T_INTEGER:
	      {
		s7_int val;
		if (subtract_overflow(integer(x), integer(y), &val))
		  return(make_real(sc, (double)integer(x) - (double)integer(y)));
		return(make_integer(sc, val));
	      }
#else
	    case T_INTEGER: return(make_integer(sc, integer(x) - integer(y)));
#endif
	    case T_RATIO:   return(g_subtract(sc, set_plist_2(sc, x, y)));
	    case T_REAL:    return(make_real(sc, real(x) - real(y)));
	    case T_COMPLEX: return(make_complex(sc, real_part(x) - real_part(y), imag_part(x) - imag_part(y)));
	    default:
	      if (!is_number(x))
		return(method_or_bust_with_type(sc, x, sc->subtract_symbol, list_2(sc, x, y), a_number_string, 1));
	      return(method_or_bust_with_type(sc, y, sc->subtract_symbol, list_2(sc, x, y), a_number_string, 2));
	    }
	}
    }

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER: return(make_integer(sc, integer(x) - integer(y)));
	case T_RATIO:   return(g_subtract(sc, set_plist_2(sc, x, y)));
	case T_REAL:    return(make_real(sc, integer(x) - real(y)));
	case T_COMPLEX: return(make_complex(sc, integer(x) - real_part(y), -imag_part(y)));
	default:
	  return(method_or_bust_with_type(sc, y, sc->subtract_symbol, list_2(sc, x, y), a_number_string, 2));
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	case T_RATIO:   return(g_subtract(sc, set_plist_2(sc, x, y)));
	case T_REAL:    return(make_real(sc, fraction(x) - real(y)));
	case T_COMPLEX: return(s7_make_complex(sc, fraction(x) - real_part(y), -imag_part(y)));
	default:
	  return(method_or_bust_with_type(sc, y, sc->subtract_symbol, list_2(sc, x, y), a_number_string, 2));
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: return(make_real(sc, real(x) - integer(y)));
	case T_RATIO:   return(make_real(sc, real(x) - fraction(y)));
	case T_REAL:    return(make_real(sc, real(x) - real(y)));
	case T_COMPLEX: return(make_complex(sc, real(x) - real_part(y), -imag_part(y)));
	default:
	  return(method_or_bust_with_type(sc, y, sc->subtract_symbol, list_2(sc, x, y), a_number_string, 2));
	}

    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER: return(s7_make_complex(sc, real_part(x) - integer(y), imag_part(x)));
	case T_RATIO:   return(s7_make_complex(sc, real_part(x) - fraction(y), imag_part(x)));
	case T_REAL:    return(s7_make_complex(sc, real_part(x) - real(y), imag_part(x)));
	case T_COMPLEX: return(make_complex(sc, real_part(x) - real_part(y), imag_part(x) - imag_part(y)));
	default:
	  return(method_or_bust_with_type(sc, y, sc->subtract_symbol, list_2(sc, x, y), a_number_string, 2));
	}

    default:
      return(method_or_bust_with_type(sc, x, sc->subtract_symbol, list_2(sc, x, y), a_number_string, 1));
    }
  return(x);
}

static s7_pointer g_subtract_2(s7_scheme *sc, s7_pointer args)
{
  return(subtract_p_pp(sc, car(args), cadr(args)));
}

static s7_pointer minus_c1(s7_scheme *sc, s7_pointer x)
{
  switch (type(x))
    {
#if HAVE_OVERFLOW_CHECKS
    case T_INTEGER:
      {
	s7_int val;
	if (subtract_overflow(integer(x), 1, &val))
	  return(make_real(sc, (double)integer(x) - 1.0));
	return(make_integer(sc, val));
      }
#else
    case T_INTEGER: return(make_integer(sc, integer(x) - 1));
#endif
    case T_RATIO:   return(subtract_ratios(sc, x, small_int(1)));
    case T_REAL:    return(make_real(sc, real(x) - 1.0));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) - 1.0, imag_part(x)));
    default:
      return(method_or_bust_with_type(sc, x, sc->subtract_symbol, list_2(sc, x, small_int(1)), a_number_string, 1));
    }
  return(x);
}

static s7_pointer g_subtract_cs1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  x = symbol_to_value_unchecked(sc, car(args));
  if (is_integer(x))
    return(make_integer(sc, integer(x) - 1));
  return(minus_c1(sc, x));
}

static s7_pointer g_subtract_s1(s7_scheme *sc, s7_pointer args)
{
  return(minus_c1(sc, car(args)));
}

static s7_pointer g_subtract_csn(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_int n;

  x = symbol_to_value_unchecked(sc, car(args));
  n = s7_integer(cadr(args));
  if (is_integer(x))
    return(make_integer(sc, integer(x) - n));

  switch (type(x))
    {
#if HAVE_OVERFLOW_CHECKS
    case T_INTEGER:
      {
	s7_int val;
	if (subtract_overflow(integer(x), n, &val))
	  return(make_real(sc, (double)integer(x) - (double)n));
	return(make_integer(sc, val));
      }
#else
    case T_INTEGER: return(make_integer(sc, integer(x) - n));
#endif
    case T_RATIO:   return(subtract_ratios(sc, x, cadr(args)));
    case T_REAL:    return(make_real(sc, real(x) - n));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) - n, imag_part(x)));
    default:
      return(method_or_bust_with_type(sc, x, sc->subtract_symbol, list_2(sc, x, cadr(args)), a_number_string, 1));
    }
  return(x);
}

static s7_pointer g_subtract_sf(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_double n;

  x = symbol_to_value_unchecked(sc, car(args));
  n = real(cadr(args));
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) - n));
    case T_RATIO:   return(make_real(sc, fraction(x) - n));
    case T_REAL:    return(make_real(sc, real(x) - n));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) - n, imag_part(x)));
    default:
      return(method_or_bust_with_type(sc, x, sc->subtract_symbol, list_2(sc, x, cadr(args)), a_number_string, 1));
    }
  return(x);
}

static s7_pointer g_subtract_2f(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_double n;

  x = car(args);
  n = real(cadr(args));
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) - n));
    case T_RATIO:   return(make_real(sc, fraction(x) - n));
    case T_REAL:    return(make_real(sc, real(x) - n));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) - n, imag_part(x)));
    default:
      return(method_or_bust_with_type(sc, x, sc->subtract_symbol, args, a_number_string, 1));
    }
  return(x);
}

static s7_pointer g_subtract_fs(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  s7_double n;

  x = symbol_to_value_unchecked(sc, cadr(args));
  n = real(car(args));
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, n - integer(x)));
    case T_RATIO:   return(make_real(sc, n - fraction(x)));
    case T_REAL:    return(make_real(sc, n - real(x)));
    case T_COMPLEX: return(s7_make_complex(sc, n - real_part(x), -imag_part(x)));
    default:
      return(method_or_bust_with_type(sc, x, sc->subtract_symbol, list_2(sc, car(args), x), a_number_string, 2));
    }
  return(x);
}

static s7_int subtract_i_ii(s7_int i1, s7_int i2) {return(i1 - i2);}
static s7_int subtract_i_i(s7_int x) {return(-x);}
static s7_int subtract_i_iii(s7_int i1, s7_int i2, s7_int i3) {return(i1 - i2 - i3);}

static s7_double subtract_d_d(s7_double x) {return(-x);}
static s7_double subtract_d_dd(s7_double x1, s7_double x2) {return(x1 - x2);}
static s7_double subtract_d_ddd(s7_double x1, s7_double x2, s7_double x3) {return(x1 - x2 - x3);}
static s7_double subtract_d_dddd(s7_double x1, s7_double x2, s7_double x3, s7_double x4) {return(x1 - x2 - x3 - x4);}

#if (!WITH_GMP)
static s7_pointer sub_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_real(sc, x1 - x2));}
#endif


/* ---------------------------------------- multiply ---------------------------------------- */

static s7_pointer multiply_method_or_bust(s7_scheme *sc, s7_pointer obj, s7_pointer caller, s7_pointer args, s7_pointer typ, int32_t num)
{
  if (has_methods(obj))
    return(find_and_apply_method(sc, find_let(sc, obj), sc->multiply_symbol, args));
  if (num == 0)
    return(simple_wrong_type_argument_with_type(sc, caller, obj, typ));
  return(wrong_type_argument_with_type(sc, caller, num, obj, typ));
}

static s7_pointer g_multiply_1(s7_scheme *sc, s7_pointer args, s7_pointer caller)
{
  #define H_multiply "(* ...) multiplies its arguments"
  #define Q_multiply sc->pcl_n

  s7_pointer x, p;
  s7_int num_a, den_a;
  s7_double rl_a, im_a;

#if (!WITH_GMP)
  if (is_null(args))
    return(small_int(1));
#endif

  x = car(args);
  p = cdr(args);
  if (is_null(p))
    {
      if (!is_number(x))
	return(multiply_method_or_bust(sc, x, caller, args, a_number_string, 0));
      return(x);
    }

  switch (type(x))
    {
    case T_INTEGER:
      num_a = integer(x);

    MULTIPLY_INTEGERS:
#if WITH_GMP
      if ((num_a > s7_int32_max) ||
	  (num_a < s7_int32_min))
	return(big_multiply(sc, cons(sc, s7_int_to_big_integer(sc, num_a), p)));
#endif
      x = car(p);
      p = cdr(p);
      switch (type(x))
	{
	case T_INTEGER:
#if WITH_GMP
	  if ((integer(x) > s7_int32_max) ||
	      (integer(x) < s7_int32_min))
	    return(big_multiply(sc, cons(sc, s7_int_to_big_integer(sc, num_a), cons(sc, x, p))));
#endif

#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int dn;
	    if (multiply_overflow(num_a, integer(x), &dn))
	      {
		if (is_null(p)) return(make_real(sc, (s7_double)num_a * (s7_double)integer(x)));
		rl_a = (s7_double)num_a * (s7_double)integer(x);
		goto MULTIPLY_REALS;
	      }
	    num_a = dn;
	  }
#else
	  num_a *= integer(x);
#endif
	  if (is_null(p)) return(make_integer(sc, num_a));
	  goto MULTIPLY_INTEGERS;

	case T_RATIO:
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int dn;
	    if (multiply_overflow(numerator(x), num_a, &dn))
	      {
		if (is_null(p))
		  return(make_real(sc, (s7_double)num_a * fraction(x)));
		rl_a = (s7_double)num_a * fraction(x);
		goto MULTIPLY_REALS;
	      }
	    num_a = dn;
	  }
#else
	  num_a *= numerator(x);
#endif
	  den_a = denominator(x);
	  if (is_null(p)) return(s7_make_ratio(sc, num_a, den_a));
	  if (reduce_fraction(&num_a, &den_a) == T_INTEGER)
	    goto MULTIPLY_INTEGERS;
	  goto MULTIPLY_RATIOS;

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, num_a * real(x)));
	  rl_a = num_a * real(x);
	  goto MULTIPLY_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, num_a * real_part(x), num_a * imag_part(x)));
	  rl_a = num_a * real_part(x);
	  im_a = num_a * imag_part(x);
	  goto MULTIPLY_COMPLEX;

	default:
	  return(multiply_method_or_bust(sc, x, caller, cons_unchecked(sc, s7_make_integer(sc, num_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	}
      break;

    case T_RATIO:
      num_a = numerator(x);
      den_a = denominator(x);
    MULTIPLY_RATIOS:
#if WITH_GMP
      if ((num_a > s7_int32_max) ||
	  (den_a > s7_int32_max) ||
	  (num_a < s7_int32_min))
	return(big_multiply(sc, cons(sc, s7_ratio_to_big_ratio(sc, num_a, den_a), p)));
#endif
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  /* as in +, this can overflow:
	   *   (* 8 -9223372036854775807 8) -> 64
	   *   (* 3/4 -9223372036854775807 8) -> 6
	   *   (* 8 -9223372036854775808 8) -> 0
	   *   (* -1 9223372036854775806 8) -> 16
	   *   (* -9223372036854775808 8 1e+308) -> 0.0
	   */
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int dn;
	    if (multiply_overflow(integer(x), num_a, &dn))
	      {
		if (is_null(p))
		  return(make_real(sc, ((s7_double)integer(x) / (s7_double)den_a) * (s7_double)num_a));
		rl_a = ((s7_double)integer(x) / (s7_double)den_a) * (s7_double)num_a;
		goto MULTIPLY_REALS;
	      }
	    num_a = dn;
	  }
#else
	  num_a *= integer(x);
#endif
	  if (is_null(p)) return(s7_make_ratio(sc, num_a, den_a));
	  if (reduce_fraction(&num_a, &den_a) == T_INTEGER)
	    goto MULTIPLY_INTEGERS;
	  goto MULTIPLY_RATIOS;

	case T_RATIO:
	  {
#if (!WITH_GMP)
#if HAVE_OVERFLOW_CHECKS
	    s7_int d1, n1;
#endif
#endif
	    s7_int d2, n2;
	    d2 = denominator(x);
	    n2 = numerator(x);
#if (!WITH_GMP)
#if HAVE_OVERFLOW_CHECKS
	    d1 = den_a;
	    n1 = num_a;
	    if ((multiply_overflow(n1, n2, &num_a)) ||
		(multiply_overflow(d1, d2, &den_a)))
	      {
		if (is_null(p))
		  return(make_real(sc, ((long_double)n1 / (long_double)d1) * ((long_double)n2 / (long_double)d2)));
		rl_a = ((long_double)n1 / (long_double)d1) * ((long_double)n2 / (long_double)d2);
		goto MULTIPLY_REALS;
	      }
#else
	    num_a *= n2;
	    den_a *= d2;
#endif
#else
	    num_a *= n2;
	    den_a *= d2;
#endif
	    if (is_null(p)) return(s7_make_ratio(sc, num_a, den_a));
	    if (reduce_fraction(&num_a, &den_a) == T_INTEGER)
	      goto MULTIPLY_INTEGERS;
	    goto MULTIPLY_RATIOS;
	  }

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, ((long_double)num_a / (long_double)den_a) * real(x)));
	  rl_a = ((long_double)num_a / (long_double)den_a) * real(x);
	  goto MULTIPLY_REALS;

	case T_COMPLEX:
	  {
	    s7_double frac;
	    frac = ((long_double)num_a / (long_double)den_a);
	    if (is_null(p)) return(s7_make_complex(sc, frac * real_part(x), frac * imag_part(x)));
	    rl_a = frac * real_part(x);
	    im_a = frac * imag_part(x);
	    goto MULTIPLY_COMPLEX;
	  }

	default:
	  return(multiply_method_or_bust(sc, x, caller, cons_unchecked(sc, s7_make_ratio(sc, num_a, den_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	}
      break;

    case T_REAL:
      rl_a = real(x);

    MULTIPLY_REALS:
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  if (is_null(p)) return(make_real(sc, rl_a * integer(x)));
	  rl_a *= integer(x);
	  goto MULTIPLY_REALS;

	case T_RATIO:
	  if (is_null(p)) return(make_real(sc, rl_a * fraction(x)));
	  rl_a *= (s7_double)fraction(x);
	  goto MULTIPLY_REALS;

	case T_REAL:
	  if (is_null(p)) return(make_real(sc, rl_a * real(x)));
	  rl_a *= real(x);
	  goto MULTIPLY_REALS;

	case T_COMPLEX:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a * real_part(x), rl_a * imag_part(x)));
	  im_a = rl_a * imag_part(x);
	  rl_a *= real_part(x);
	  goto MULTIPLY_COMPLEX;

	default:
	  return(multiply_method_or_bust(sc, x, caller, cons_unchecked(sc, make_real(sc, rl_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	}
      break;

    case T_COMPLEX:
      rl_a = real_part(x);
      im_a = imag_part(x);

    MULTIPLY_COMPLEX:
      x = car(p);
      p = cdr(p);

      switch (type(x))
	{
	case T_INTEGER:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a * integer(x), im_a * integer(x)));
	  rl_a *= integer(x);
	  im_a *= integer(x);
	  goto MULTIPLY_COMPLEX;

	case T_RATIO:
	  {
	    s7_double frac;
	    frac = fraction(x);
	    if (is_null(p)) return(s7_make_complex(sc, rl_a * frac, im_a * frac));
	    rl_a *= frac;
	    im_a *= frac;
	    goto MULTIPLY_COMPLEX;
	  }

	case T_REAL:
	  if (is_null(p)) return(s7_make_complex(sc, rl_a * real(x), im_a * real(x)));
	  rl_a *= real(x);
	  im_a *= real(x);
	  goto MULTIPLY_COMPLEX;

	case T_COMPLEX:
	  {
	    s7_double r1, r2, i1, i2;
	    r1 = rl_a;
	    i1 = im_a;
	    r2 = real_part(x);
	    i2 = imag_part(x);
	    if (is_null(p))
	      return(s7_make_complex(sc, r1 * r2 - i1 * i2, r1 * i2 + r2 * i1));
	    rl_a = r1 * r2 - i1 * i2;
	    im_a = r1 * i2 + r2 * i1;
	    if (im_a == 0.0)
	      goto MULTIPLY_REALS;
	    goto MULTIPLY_COMPLEX;
	  }

	default:
	  return(multiply_method_or_bust(sc, x, caller, cons_unchecked(sc, s7_make_complex(sc, rl_a, im_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	}
      break;

    default:
      return(multiply_method_or_bust(sc, x, caller, args, a_number_string, 1));
    }
  return(NULL); /* make the compiler happy */
}

static s7_pointer g_multiply(s7_scheme *sc, s7_pointer args)
{
  return(g_multiply_1(sc, args, sc->multiply_symbol));
}

#if (!WITH_GMP)
static inline s7_pointer multiply_p_pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (type(x) == type(y))
    {
      if (is_t_real(x))
	return(make_real(sc, real(x) * real(y)));
      else
	{
	  switch (type(x))
	    {
#if HAVE_OVERFLOW_CHECKS
	    case T_INTEGER:
	      {
		s7_int n;
		if (multiply_overflow(integer(x), integer(y), &n))
		  return(make_real(sc, ((s7_double)integer(x)) * ((s7_double)integer(y))));
		return(make_integer(sc, n));
	      }
#else
	    case T_INTEGER: return(make_integer(sc, integer(x) * integer(y)));
#endif
	    case T_RATIO:   return(g_multiply(sc, list_2(sc, x, y)));
	    case T_REAL:    return(make_real(sc, real(x) * real(y)));
	    case T_COMPLEX:
	      {
		s7_double r1, r2, i1, i2;
		r1 = real_part(x);
		r2 = real_part(y);
		i1 = imag_part(x);
		i2 = imag_part(y);
		return(s7_make_complex(sc, r1 * r2 - i1 * i2, r1 * i2 + r2 * i1));
	      }
	    default:
	      if (!is_number(x))
		return(method_or_bust_with_type(sc, x, sc->multiply_symbol, list_2(sc, x, y), a_number_string, 1));
	      return(method_or_bust_with_type(sc, y, sc->multiply_symbol, list_2(sc, x, y), a_number_string, 2));
	    }
	}
    }

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER: return(make_integer(sc, integer(x) * integer(y)));
	case T_RATIO:   return(g_multiply(sc, list_2(sc, x, y)));
	case T_REAL:    return(make_real(sc, integer(x) * real(y)));
	case T_COMPLEX: return(s7_make_complex(sc, integer(x) * real_part(y), integer(x) * imag_part(y)));
	default:
	  return(method_or_bust_with_type(sc, y, sc->multiply_symbol, list_2(sc, x, y), a_number_string, 2));
	}

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	case T_RATIO:    return(g_multiply(sc, list_2(sc, x, y)));
	case T_REAL:     return(make_real(sc, fraction(x) * real(y)));
	case T_COMPLEX:
	  {
	    s7_double frac;
	    frac = fraction(x);
	    return(s7_make_complex(sc, frac * real_part(y), frac * imag_part(y)));
	  }
	default:
	  return(method_or_bust_with_type(sc, y, sc->multiply_symbol, list_2(sc, x, y), a_number_string, 2));
	}

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: return(make_real(sc, real(x) * integer(y)));
	case T_RATIO:   return(make_real(sc, real(x) * fraction(y)));
	case T_REAL:    return(make_real(sc, real(x) * real(y)));
	case T_COMPLEX: return(s7_make_complex(sc, real(x) * real_part(y), real(x) * imag_part(y)));
	default:
	  return(method_or_bust_with_type(sc, y, sc->multiply_symbol, list_2(sc, x, y), a_number_string, 2));
	}

    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER: return(s7_make_complex(sc, real_part(x) * integer(y), imag_part(x) * integer(y)));
	case T_RATIO:
	  {
	    s7_double frac;
	    frac = fraction(y);
	    return(s7_make_complex(sc, real_part(x) * frac, imag_part(x) * frac));
	  }
	case T_REAL:    return(s7_make_complex(sc, real_part(x) * real(y), imag_part(x) * real(y)));
	case T_COMPLEX:
	  {
	    s7_double r1, r2, i1, i2;
	    r1 = real_part(x);
	    r2 = real_part(y);
	    i1 = imag_part(x);
	    i2 = imag_part(y);
	    return(s7_make_complex(sc, r1 * r2 - i1 * i2, r1 * i2 + r2 * i1));
	  }
	default:
	  return(method_or_bust_with_type(sc, y, sc->multiply_symbol, list_2(sc, x, y), a_number_string, 2));
	}

    default:
      return(method_or_bust_with_type(sc, x, sc->multiply_symbol, list_2(sc, x, y), a_number_string, 1));
    }
  return(x);
}

static s7_pointer g_multiply_2(s7_scheme *sc, s7_pointer args) {return(multiply_p_pp(sc, car(args), cadr(args)));}

/* all of these mess up if overflows occur
 *  (let () (define (f x) (* x 9223372036854775806)) (f -63)) -> -9223372036854775682, but (* -63 9223372036854775806) -> -5.810724383218509e+20
 *  how to catch this?  (affects * - +)
 */

static s7_pointer g_mul_sis(s7_scheme *sc, s7_pointer x, s7_int n, s7_pointer args)
{
  switch (type(x))
    {
#if HAVE_OVERFLOW_CHECKS
    case T_INTEGER:
      {
	s7_int val;
	if (multiply_overflow(integer(x), n, &val))
	  return(make_real(sc, (double)integer(x) * (double)n));
	return(make_integer(sc, val));
      }
    case T_RATIO:
      {
	s7_int val;
	if (multiply_overflow(numerator(x), n, &val))
	  return(make_real(sc, fraction(x) * (double)n));
	return(s7_make_ratio(sc, val, denominator(x)));
      }
#else
    case T_INTEGER: return(make_integer(sc, integer(x) * n));
    case T_RATIO:   return(s7_make_ratio(sc, numerator(x) * n, denominator(x)));
#endif
    case T_REAL:    return(make_real(sc, real(x) * n));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) * n, imag_part(x) * n));
    default:
      return(method_or_bust_with_type(sc, x, sc->multiply_symbol,
				      (is_symbol(car(args))) ? list_2(sc, x, cadr(args)) : list_2(sc, car(args), x),
				      a_number_string, 1));
    }
  return(x);
}

static s7_pointer g_multiply_si(s7_scheme *sc, s7_pointer args) {return(g_mul_sis(sc, symbol_to_value_unchecked(sc, car(args)), integer(cadr(args)), args));}
static s7_pointer g_multiply_is(s7_scheme *sc, s7_pointer args) {return(g_mul_sis(sc, symbol_to_value_unchecked(sc, cadr(args)), integer(car(args)), args));}

static s7_pointer g_mul_sfs(s7_scheme *sc, s7_pointer x, s7_double y)
{
  switch (type(x))
    {
    case T_INTEGER: return(make_real(sc, integer(x) * y));
    case T_RATIO:   return(make_real(sc, numerator(x) * y / denominator(x)));
    case T_REAL:    return(make_real(sc, real(x) * y));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) * y, imag_part(x) * y));
    default:
      return(method_or_bust_with_type(sc, x, sc->multiply_symbol, list_2(sc, make_real(sc, y), x), a_number_string, 1));
    }
  return(x);
}

static s7_pointer g_multiply_fs(s7_scheme *sc, s7_pointer args) {return(g_mul_sfs(sc, symbol_to_value_unchecked(sc, cadr(args)), real(car(args))));}
static s7_pointer g_multiply_sf(s7_scheme *sc, s7_pointer args) {return(g_mul_sfs(sc, symbol_to_value_unchecked(sc, car(args)), real(cadr(args))));}

static s7_pointer g_mul_2_ff(s7_scheme *sc, s7_pointer args) {return(make_real(sc, real(car(args)) * real(cadr(args))));}
static s7_pointer g_mul_2_ii(s7_scheme *sc, s7_pointer args)
{
#if HAVE_OVERFLOW_CHECKS
  s7_int val, x, y;
  x = integer(car(args));
  y = integer(cadr(args));
  if (multiply_overflow(x, y, &val))
    return(make_real(sc, (double)x * (double)y));
  return(make_integer(sc, val));
#else
  return(make_integer(sc, integer(car(args)) * integer(cadr(args))));
#endif
}
static s7_pointer g_mul_2_if(s7_scheme *sc, s7_pointer args) {return(make_real(sc, integer(car(args)) * real(cadr(args))));}
static s7_pointer g_mul_2_fi(s7_scheme *sc, s7_pointer args) {return(make_real(sc, real(car(args)) * integer(cadr(args))));}
static s7_pointer g_mul_2_xi(s7_scheme *sc, s7_pointer args) {return(g_mul_sis(sc, car(args), integer(cadr(args)), args));}
static s7_pointer g_mul_2_ix(s7_scheme *sc, s7_pointer args) {return(g_mul_sis(sc, cadr(args), integer(car(args)), args));}
static s7_pointer g_mul_2_xf(s7_scheme *sc, s7_pointer args) {return(g_mul_sfs(sc, car(args), real(cadr(args))));}
static s7_pointer g_mul_2_fx(s7_scheme *sc, s7_pointer args) {return(g_mul_sfs(sc, cadr(args), real(car(args))));}

static s7_pointer g_sqr_ss(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  x = symbol_to_value_unchecked(sc, car(args));

  switch (type(x))
    {
#if HAVE_OVERFLOW_CHECKS
    case T_INTEGER:
      {
	s7_int val;
	if (multiply_overflow(integer(x), integer(x), &val))
	  return(make_real(sc, (double)integer(x) * (double)integer(x)));
	return(make_integer(sc, val));
      }
    case T_RATIO:
      {
	s7_int num, den;
	if ((multiply_overflow(numerator(x), numerator(x), &num)) ||
	    (multiply_overflow(denominator(x), denominator(x), &den)))
	  return(make_real(sc, fraction(x) * fraction(x)));
	return(s7_make_ratio(sc, num, den));
      }
#else
    case T_INTEGER: return(s7_make_integer(sc, integer(x) * integer(x)));
    case T_RATIO:   return(s7_make_ratio(sc, numerator(x) * numerator(x), denominator(x) * denominator(x)));
#endif
    case T_REAL:    return(make_real(sc, real(x) * real(x)));
    case T_COMPLEX: return(s7_make_complex(sc, real_part(x) * real_part(x) - imag_part(x) * imag_part(x), 2.0 * real_part(x) * imag_part(x)));
    default:
      return(method_or_bust_with_type(sc, x, sc->multiply_symbol, list_2(sc, x, x), a_number_string, 1));
    }
  return(x);
}
#endif /* with-gmp */

static s7_int multiply_i_ii(s7_int i1, s7_int i2)
{
#if HAVE_OVERFLOW_CHECKS
  s7_int val;
  if (multiply_overflow(i1, i2, &val))
    return(s7_int_max); /* this is inconsistent with other unopt cases where an overflow -> double result */
  /* (let () (define (func) (do ((i 0 (+ i 1))) ((= i 1)) (do ((j 0 (+ j 1))) ((= j 1)) (even? (* (ash 1 43) (ash 1 43)))))) (define (hi) (func)) (hi)) */
  return(val);
#else
  return(i1 * i2);
#endif
}

static s7_int multiply_i_iii(s7_int i1, s7_int i2, s7_int i3)
{
#if HAVE_OVERFLOW_CHECKS
  s7_int val1, val2;
  if (multiply_overflow(i1, i2, &val1))
    return(s7_int_max);
  if (multiply_overflow(val1, i3, &val2))
    return(s7_int_max);
  return(val2);
#else
  return(i1 * i2 * i3);
#endif
}

static s7_double multiply_d_d(s7_double x) {return(x);}
static s7_double multiply_d_dd(s7_double x1, s7_double x2) {return(x1 * x2);}
static s7_double multiply_d_ddd(s7_double x1, s7_double x2, s7_double x3) {return(x1 * x2 * x3);}
static s7_double multiply_d_dddd(s7_double x1, s7_double x2, s7_double x3, s7_double x4) {return(x1 * x2 * x3 * x4);}
#if (!WITH_GMP)
static s7_pointer mul_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_real(sc, x1 * x2));}
#endif


/* ---------------------------------------- divide ---------------------------------------- */

static bool is_number_via_method(s7_scheme *sc, s7_pointer p)
{
  if (s7_is_number(p))
    return(true);
  if (has_methods(p))
    {
      s7_pointer f;
      f = find_method(sc, find_let(sc, p), sc->is_number_symbol);
      if (f != sc->undefined)
	return(is_true(sc, s7_apply_function(sc, f, cons(sc, p, sc->nil))));
    }
  return(false);
}

static s7_pointer g_divide(s7_scheme *sc, s7_pointer args)
{
  #define H_divide "(/ x1 ...) divides its first argument by the rest, or inverts the first if there is only one argument"
  #define Q_divide sc->pcl_n

  s7_pointer x, y, p;

  x = car(args);
  p = cdr(args);

  if (is_null(p))            /* (/ x) */
    {
      if (!is_number(x))
	return(method_or_bust_with_type_one_arg(sc, x, sc->divide_symbol, args, a_number_string));
      if (s7_is_zero(x))     /* (/ 0) */
	return(division_by_zero_error(sc, sc->divide_symbol, args));
      return(s7_invert(sc, x));
    }
  if (is_null(cdr(p)))
    y = cadr(args);
  else
    {
      y = g_multiply_1(sc, p, sc->divide_symbol); /* in some schemes (/ 1 0 +nan.0) is not equal to (/ 1 (* 0 +nan.0)), in s7 they're both +nan.0 */
#if WITH_GMP
      if (s7_is_bignum(y))
	return(big_divide(sc, set_plist_2(sc, x, y)));
#endif
    }

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	  /* -------- integer x -------- */
	case T_INTEGER:
	  if (integer(y) == 0)
	    return(division_by_zero_error(sc, sc->divide_symbol, set_elist_2(sc, x, y)));
	  return(s7_make_ratio(sc, integer(x), integer(y)));

	case T_RATIO:
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int dn;
	    if (multiply_overflow(integer(x), denominator(y), &dn))
	      return(make_real(sc, integer(x) * inverted_fraction(y)));
	    return(s7_make_ratio(sc, dn, numerator(y)));
	  }
#else
	  return(s7_make_ratio(sc, integer(x) * denominator(y), numerator(y)));
#endif

	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  if (is_inf(real(y))) return(real_zero);
	  if (real(y) == 0.0)
	    return(division_by_zero_error(sc, sc->divide_symbol, args));
	  return(make_real(sc, (s7_double)(integer(x)) / real(y)));

	case T_COMPLEX:
	  {
	    s7_double r1, i2, r2, den;
	    r1 = (s7_double)integer(x);
	    r2 = real_part(y);
	    i2 = imag_part(y);
	    den = 1.0 / (r2 * r2 + i2 * i2);
	    /* we could avoid the squaring (see Knuth II p613 16), not a big deal: (/ 1.0e308+1.0e308i 2.0e308+2.0e308i) => nan, (gmp case is ok here) */
	    return(s7_make_complex(sc, r1 * r2 * den, -(r1 * i2 * den)));
	  }

	default:
#if WITH_GMP
	  if (s7_is_bignum(y))
	    return(big_divide(sc, list_2(sc, x, y)));
#endif
	  return(method_or_bust_with_type(sc, y, sc->divide_symbol, list_2(sc, x, y), a_number_string, 2));
	}
      break;

      /* -------- ratio x -------- */
    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    return(division_by_zero_error(sc, sc->divide_symbol, set_elist_2(sc, x, y)));
#if HAVE_OVERFLOW_CHECKS
	  {
	    s7_int dn;
	    if (multiply_overflow(denominator(x), integer(y), &dn))
	      return(make_real(sc, (long_double)numerator(x) / ((long_double)denominator(x) * (s7_double)integer(y))));
	    return(s7_make_ratio(sc, numerator(x), dn));
	  }
#else
	  return(s7_make_ratio(sc, numerator(x), denominator(x) * integer(y)));
#endif

	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    d1 = denominator(x);
	    n1 = numerator(x);
	    d2 = denominator(y);
	    n2 = numerator(y);
	    if (d1 == d2)
	      return(s7_make_ratio(sc, n1, n2));
#if (!WITH_GMP) && HAVE_OVERFLOW_CHECKS
	    if ((multiply_overflow(n1, d2, &n1)) ||
		(multiply_overflow(n2, d1, &d1)))
	      {
		s7_double r1, r2;
		r1 = fraction(x);
		r2 = inverted_fraction(y);
		return(make_real(sc, r1 * r2));
	      }
	    return(s7_make_ratio(sc, n1, d1));
#else
	    return(s7_make_ratio(sc, n1 * d2, n2 * d1));
#endif
	  }

	case T_REAL:
	  if (real(y) == 0.0)
	    return(division_by_zero_error(sc, sc->divide_symbol, args));
	  return(make_real(sc, fraction(x) / real(y)));

	case T_COMPLEX:
	  {
	    s7_double rx, r2, i2, den;
	    rx = fraction(x);
	    r2 = real_part(y);
	    i2 = imag_part(y);
	    den = 1.0 / (r2 * r2 + i2 * i2);
	    return(s7_make_complex(sc, rx * r2 * den, -rx * i2 * den));
	  }

	default:
	  return(method_or_bust_with_type(sc, y, sc->divide_symbol, list_2(sc, x, y), a_number_string, 2));
	}

      /* -------- real x -------- */
    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(y) == 0)
	    return(division_by_zero_error(sc, sc->divide_symbol, args));
	  if (is_NaN(real(x))) return(real_NaN); /* what is (/ +nan.0 0)? */
	  if (is_inf(real(x))) return((real(x) > 0.0) ? ((integer(y) > 0) ? real_infinity : real_minus_infinity) : ((integer(y) > 0) ? real_minus_infinity : real_infinity));
	  return(make_real(sc, real(x) / (s7_double)integer(y)));

	case T_RATIO:
	  if (is_NaN(real(x))) return(real_NaN);
	  if (is_inf(real(x))) return((real(x) > 0) ? ((numerator(y) > 0) ? real_infinity : real_minus_infinity) : ((numerator(y) > 0) ? real_minus_infinity : real_infinity));
	  return(make_real(sc, real(x) * inverted_fraction(y)));

	case T_REAL:
	  if (is_NaN(real(y))) return(real_NaN);
	  if (real(y) == 0.0)
	    return(division_by_zero_error(sc, sc->divide_symbol, args));
	  if (is_NaN(real(x))) return(real_NaN);
	  if (is_inf(real(y)))
	    {
	      if (is_inf(real(x))) return(real_NaN);
	      return(real_zero);
	    }
	  return(make_real(sc, real(x) / real(y)));

	case T_COMPLEX:
	  {
	    s7_double den, r2, i2;
	    if (is_NaN(real(x))) return(real_NaN);
	    if (is_NaN(real_part(y))) return(real_NaN);
	    if (is_NaN(imag_part(y))) return(real_NaN);
	    if (is_inf(real_part(y))) return(real_zero);
	    r2 = real_part(y);
	    i2 = imag_part(y);
	    den = 1.0 / (r2 * r2 + i2 * i2);
	    return(s7_make_complex(sc, real(x) * r2 * den, -real(x) * i2 * den));
	  }

	default:
	  return(method_or_bust_with_type(sc, y, sc->divide_symbol, list_2(sc, x, y), a_number_string, 2));
	}

      /* -------- complex x -------- */
    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	  {
	    s7_double r1;
	    if (integer(y) == 0)
	      return(division_by_zero_error(sc, sc->divide_symbol, args));
	    r1 = 1.0 / (s7_double)integer(y);
	    return(s7_make_complex(sc, real_part(x) * r1, imag_part(x) * r1));
	  }

	case T_RATIO:
	  {
	    s7_double frac;
	    frac = inverted_fraction(y);
	    return(s7_make_complex(sc, real_part(x) * frac, imag_part(x) * frac));
	  }

	case T_REAL:
	  {
	    s7_double r1;
	    if (real(y) == 0.0)
	      return(division_by_zero_error(sc, sc->divide_symbol, args));
	    r1 = 1.0 / real(y);
	    return(s7_make_complex(sc, real_part(x) * r1, imag_part(x) * r1));
	  }

	case T_COMPLEX:
	  {
	    s7_double r1, r2, i1, i2, den;
	    r1 = real_part(x);
	    if (is_NaN(r1)) return(real_NaN);
	    i1 = imag_part(x);
	    if (is_NaN(i1)) return(real_NaN);
	    r2 = real_part(y);
	    if (is_NaN(r2)) return(real_NaN);
	    i2 = imag_part(y);
	    if (is_NaN(i2)) return(real_NaN);
	    den = 1.0 / (r2 * r2 + i2 * i2);
	    return(s7_make_complex(sc, (r1 * r2 + i1 * i2) * den, (r2 * i1 - r1 * i2) * den));
	  }

	default:
	  return(method_or_bust_with_type(sc, y, sc->divide_symbol, list_2(sc, x, y), a_number_string, 2));
	}

      /* -------- any other type x -------- */
    default:
      return(method_or_bust_with_type(sc, x, sc->divide_symbol, args, a_number_string, 1));
      break;
    }

  return(NULL); /* make the compiler happy */
}

#if (!WITH_GMP)
static s7_pointer g_invert_1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer p;
  p = car(args);
  switch (type(p))
    {
    case T_INTEGER:
      if (integer(p) != 0)
	return(s7_make_ratio(sc, 1, integer(p)));      /* a already checked, not 0 */
      return(division_by_zero_error(sc, sc->divide_symbol, args));

    case T_RATIO:
      return(make_simple_ratio(sc, denominator(p), numerator(p)));

    case T_REAL:
      if (real(p) != 0.0)
	return(make_real(sc, 1.0 / real(p)));
      return(division_by_zero_error(sc, sc->divide_symbol, args));

    case T_COMPLEX:
      {
	s7_double r2, i2, den;
	r2 = real_part(p);
	i2 = imag_part(p);
	den = (r2 * r2 + i2 * i2);
	return(s7_make_complex(sc, r2 / den, -i2 / den));
      }

    default:
      return(method_or_bust_with_type(sc, p, sc->divide_symbol, args, a_number_string, 1));
    }
}

static s7_pointer g_divide_1r(s7_scheme *sc, s7_pointer args)
{
  if (s7_is_real(cadr(args)))
    {
      s7_double rl;
      rl = s7_real(cadr(args));
      if (rl == 0.0)
	return(division_by_zero_error(sc, sc->divide_symbol, args));
      return(make_real(sc, 1.0 / rl));
    }
  return(g_divide(sc, args));
}
#endif


static s7_double divide_d_7d(s7_scheme *sc, s7_double x)
{
  if (x == 0.0) division_by_zero_error(sc, sc->divide_symbol, set_elist_1(sc, real_zero));
  return(1.0 / x);
}

static s7_double divide_d_7dd(s7_scheme *sc, s7_double x1, s7_double x2)
{
  if (x2 == 0.0) division_by_zero_error(sc, sc->divide_symbol, set_elist_1(sc, real_zero));
  return(x1 / x2);
}

static s7_pointer divide_p_ii(s7_scheme *sc, s7_int x, s7_int y) {return(s7_make_ratio(sc, x, y));} /* make-ratio checks for y==0 */


/* ---------------------------------------- max/min ---------------------------------------- */

static bool is_real_via_method_1(s7_scheme *sc, s7_pointer p)
{
  s7_pointer f;
  f = find_method(sc, find_let(sc, p), sc->is_real_symbol);
  if (f != sc->undefined)
    return(is_true(sc, s7_apply_function(sc, f, cons(sc, p, sc->nil))));
  return(false);
}

#define is_real_via_method(sc, p) ((s7_is_real(p)) || ((has_methods(p)) && (is_real_via_method_1(sc, p))))

static s7_pointer g_max(s7_scheme *sc, s7_pointer args)
{
  #define H_max "(max ...) returns the maximum of its arguments"
  #define Q_max sc->pcl_r

  s7_pointer x, y, p;
  s7_int num_a, num_b, den_a, den_b;

  x = car(args);
  p = cdr(args);

  switch (type(x))
    {
    case T_INTEGER:
    MAX_INTEGERS:
      if (is_null(p)) return(x);
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(x) < integer(y)) x = y;
	  goto MAX_INTEGERS;

	case T_RATIO:
	  num_a = integer(x);
	  den_a = 1;
	  num_b = numerator(y);
	  den_b = denominator(y);
	  goto RATIO_MAX_RATIO;

	case T_REAL:
	  if (is_NaN(real(y)))
	    {
	      for (; is_not_null(p); p = cdr(p))
		if (!is_real_via_method(sc, car(p)))
		  return(wrong_type_argument(sc, sc->max_symbol, position_of(p, args), car(p), T_REAL));
	      return(y);
	    }
	  if (integer(x) < real(y))
	    {
	      x = y;
	      goto MAX_REALS;
	    }
	  goto MAX_INTEGERS;

	default:
	  return(method_or_bust(sc, y, sc->max_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}


    case T_RATIO:
    MAX_RATIOS:
      if (is_null(p)) return(x);
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  num_a = numerator(x);
	  den_a = denominator(x);
	  num_b = integer(y);
	  den_b = 1;
	  goto RATIO_MAX_RATIO;

	case T_RATIO:
	  num_a = numerator(x);
	  den_a = denominator(x);
	  num_b = numerator(y);
	  den_b = denominator(y);

	RATIO_MAX_RATIO:
	  /* there are tricky cases here where long ints outrun doubles:
	   *   (max 92233720368547758/9223372036854775807 92233720368547757/9223372036854775807)
	   * which should be 92233720368547758/9223372036854775807) but first the fraction gets reduced
	   * to 13176245766935394/1317624576693539401, so we fall into the double comparison, and
	   * there we should be comparing
	   *    9.999999999999999992410584792601468961145E-3 and
	   *    9.999999999999999883990367544051025548645E-3
	   * but if using doubles we get
	   *    0.010000000000000000208166817117 and
	   *    0.010000000000000000208166817117
	   * that is, we can't distinguish these two fractions once they're coerced to doubles.
	   *
	   * Even long doubles fail in innocuous-looking cases:
	   *     (min 21053343141/6701487259 3587785776203/1142027682075) -> 3587785776203/1142027682075
	   *     (max 21053343141/6701487259 3587785776203/1142027682075) -> 3587785776203/1142027682075
	   *
	   * Another consequence: outside gmp, we can't handle cases like
	   *    (max 9223372036854776/9223372036854775807 #i9223372036854775/9223372036854775000)
	   *    (max #i9223372036854776/9223372036854775807 9223372036854775/9223372036854775000)
	   * I guess if the user is using "inexact" numbers (#i...), he accepts their inexactness.
	   */

	  if ((num_a < 0) && (num_b >= 0)) /* x < 0, y >= 0 -> y */
	    x = y;
	  else
	    {
	      if ((num_a < 0) || (num_b >= 0))
		{
		  if (den_a == den_b)
		    {
		      if (num_a < num_b)
			x = y;
		    }
		  else
		    {
		      if (num_a == num_b)
			{
			  if (((num_a >= 0) &&
			       (den_a > den_b)) ||
			      ((num_a < 0) &&
			       (den_a < den_b)))
			    x = y;
			}
		      else
			{
			  s7_int vala, valb;
			  vala = num_a / den_a;
			  valb = num_b / den_b;
			  if (!((vala > valb) ||
				((vala == valb) && (is_t_integer(y)))))
			    {
			      if ((valb > vala) ||
				  ((vala == valb) && (is_t_integer(x))) ||
				  /* sigh -- both are ratios and the int32_t parts are equal */
				  (((long_double)(num_a % den_a) / (long_double)den_a) <= ((long_double)(num_b % den_b) / (long_double)den_b)))
				x = y;
			    }}}}
	    }
	  if (is_t_ratio(x))
	    goto MAX_RATIOS;
	  goto MAX_INTEGERS;

	case T_REAL:
	  /* (max 3/4 +nan.0) should probably return NaN */
	  if (is_NaN(real(y)))
	    {
	      for (; is_not_null(p); p = cdr(p))
		if (!is_real_via_method(sc, car(p)))
		  return(wrong_type_argument(sc, sc->max_symbol, position_of(p, args), car(p), T_REAL));
	      return(y);
	    }

	  if (fraction(x) < real(y))
	    {
	      x = y;
	      goto MAX_REALS;
	    }
	  goto MAX_RATIOS;

	default:
	  return(method_or_bust(sc, y, sc->max_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}


    case T_REAL:
      if (is_NaN(real(x)))
	{
	  for (; is_not_null(p); p = cdr(p))
	    if (!is_real_via_method(sc, car(p)))
	      return(wrong_type_argument(sc, sc->max_symbol, position_of(p, args), car(p), T_REAL));
	  return(x);
	}

    MAX_REALS:
      if (is_null(p)) return(x);
      y = car(p);
      p = cdr(p);

      switch (type(y))
	{
	case T_INTEGER:
	  if (real(x) < integer(y))
	    {
	      x = y;
	      goto MAX_INTEGERS;
	    }
	  goto MAX_REALS;

	case T_RATIO:
	  if (real(x) < fraction(y))
	    {
	      x = y;
	      goto MAX_RATIOS;
	    }
	  goto MAX_REALS;

	case T_REAL:
	  if (is_NaN(real(y)))
	    {
	      for (; is_not_null(p); p = cdr(p))
		if (!is_real_via_method(sc, car(p)))
		  return(wrong_type_argument(sc, sc->max_symbol, position_of(p, args), car(p), T_REAL));
	      return(y);
	    }
	  if (real(x) < real(y)) x = y;
	  goto MAX_REALS;

	default:
	  return(method_or_bust(sc, y, sc->max_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}

    default:
      return(method_or_bust(sc, x, sc->max_symbol, cons(sc, x, p), T_REAL, 1));
    }
}

static s7_int max_i_ii(s7_int i1, s7_int i2) {return((i1 > i2) ? i1 : i2);}
static s7_int max_i_iii(s7_int i1, s7_int i2, s7_int i3) {return((i1 > i2) ? ((i1 > i3) ? i1 : i3) : ((i2 > i3) ? i2 : i3));}
static s7_double max_d_dd(s7_double x1, s7_double x2) {if (is_NaN(x1)) return(x1); return((x1 > x2) ? x1 : x2);}
static s7_double max_d_ddd(s7_double x1, s7_double x2, s7_double x3) {return(max_d_dd(x1, max_d_dd(x2, x3)));}
static s7_double max_d_dddd(s7_double x1, s7_double x2, s7_double x3, s7_double x4) {return(max_d_dd(x1, max_d_ddd(x2, x3, x4)));}

static s7_pointer g_min(s7_scheme *sc, s7_pointer args)
{
  #define H_min "(min ...) returns the minimum of its arguments"
  #define Q_min sc->pcl_r

  s7_pointer x, y, p;
  s7_int num_a, num_b, den_a, den_b;

  x = car(args);
  p = cdr(args);

  switch (type(x))
    {
    case T_INTEGER:
    MIN_INTEGERS:
      if (is_null(p)) return(x);
      y = car(p);
      p = cdr(p);

      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(x) > integer(y)) x = y;
	  goto MIN_INTEGERS;

	case T_RATIO:
	  num_a = integer(x);
	  den_a = 1;
	  num_b = numerator(y);
	  den_b = denominator(y);
	  goto RATIO_MIN_RATIO;

	case T_REAL:
	  if (is_NaN(real(y)))
	    {
	      for (; is_not_null(p); p = cdr(p))
		if (!is_real_via_method(sc, car(p)))
		  return(wrong_type_argument(sc, sc->min_symbol, position_of(p, args), car(p), T_REAL));
	      return(y);
	    }
	  if (integer(x) > real(y))
	    {
	      x = y;
	      goto MIN_REALS;
	    }
	  goto MIN_INTEGERS;

	default:
	  return(method_or_bust(sc, y, sc->min_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}


    case T_RATIO:
    MIN_RATIOS:
      if (is_null(p)) return(x);
      y = car(p);
      p = cdr(p);

      switch (type(y))
	{
	case T_INTEGER:
	  num_a = numerator(x);
	  den_a = denominator(x);
	  num_b = integer(y);
	  den_b = 1;
	  goto RATIO_MIN_RATIO;

	case T_RATIO:
	  num_a = numerator(x);
	  den_a = denominator(x);
	  num_b = numerator(y);
	  den_b = denominator(y);

	RATIO_MIN_RATIO:
	  if ((num_a >= 0) && (num_b < 0))
	    x = y;
	  else
	    {
	      if ((num_a >= 0) || (num_b < 0))
		{
		  if (den_a == den_b)
		    {
		      if (num_a > num_b)
			x = y;
		    }
		  else
		    {
		      if (num_a == num_b)
			{
			  if (((num_a >= 0) &&
			       (den_a < den_b)) ||
			      ((num_a < 0) &&
			       (den_a > den_b)))
			    x = y;
			}
		      else
			{
			  s7_int vala, valb;
			  vala = num_a / den_a;
			  valb = num_b / den_b;

			  if (!((vala < valb) ||
				((vala == valb) && (is_t_integer(x)))))
			    {
			      if ((valb < vala) ||
				  ((vala == valb) && (is_t_integer(y))) ||
				  (((long_double)(num_a % den_a) / (long_double)den_a) >= ((long_double)(num_b % den_b) / (long_double)den_b)))
				x = y;
			    }}}}
	    }
	  if (is_t_ratio(x))
	    goto MIN_RATIOS;
	  goto MIN_INTEGERS;

	case T_REAL:
	  /* (min 3/4 +nan.0) should probably return NaN */
	  if (is_NaN(real(y)))
	    {
	      for (; is_not_null(p); p = cdr(p))
		if (!is_real_via_method(sc, car(p)))
		  return(wrong_type_argument(sc, sc->min_symbol, position_of(p, args), car(p), T_REAL));
	      return(y);
	    }
	  if (fraction(x) > real(y))
	    {
	      x = y;
	      goto MIN_REALS;
	    }
	  goto MIN_RATIOS;

	default:
	  return(method_or_bust(sc, y, sc->min_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}


    case T_REAL:
      if (is_NaN(real(x)))
	{
	  for (; is_not_null(p); p = cdr(p))
	    if (!is_real_via_method(sc, car(p)))
	      return(wrong_type_argument(sc, sc->min_symbol, position_of(p, args), car(p), T_REAL));
	  return(x);
	}

    MIN_REALS:
      if (is_null(p)) return(x);
      y = car(p);
      p = cdr(p);

      switch (type(y))
	{
	case T_INTEGER:
	  if (real(x) > integer(y))
	    {
	      x = y;
	      goto MIN_INTEGERS;
	    }
	  goto MIN_REALS;

	case T_RATIO:
	  if (real(x) > fraction(y))
	    {
	      x = y;
	      goto MIN_RATIOS;
	    }
	  goto MIN_REALS;

	case T_REAL:
	  if (is_NaN(real(y)))
	    {
	      for (; is_not_null(p); p = cdr(p))
		if (!is_real_via_method(sc, car(p)))
		  return(wrong_type_argument(sc, sc->min_symbol, position_of(p, args), car(p), T_REAL));
	      return(y);
	    }
	  if (real(x) > real(y)) x = y;
	  goto MIN_REALS;

	default:
	  return(method_or_bust(sc, y, sc->min_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}

    default:
      return(method_or_bust(sc, x, sc->min_symbol, cons(sc, x, p), T_REAL, 1));
    }
}

static s7_int min_i_ii(s7_int i1, s7_int i2) {return((i1 < i2) ? i1 : i2);}
static s7_int min_i_iii(s7_int i1, s7_int i2, s7_int i3) {return((i1 < i2) ? ((i1 < i3) ? i1 : i3) : ((i2 < i3) ? i2 : i3));}
static s7_double min_d_dd(s7_double x1, s7_double x2) {if (is_NaN(x1)) return(x1); return((x1 < x2) ? x1 : x2);}
static s7_double min_d_ddd(s7_double x1, s7_double x2, s7_double x3) {return(min_d_dd(x1, min_d_dd(x2, x3)));}
static s7_double min_d_dddd(s7_double x1, s7_double x2, s7_double x3, s7_double x4) {return(min_d_dd(x1, min_d_ddd(x2, x3, x4)));}


/* ---------------------------------------- = > < >= <= ---------------------------------------- */

static s7_pointer g_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_equal "(= z1 ...) returns #t if all its arguments are equal"
  #define Q_equal s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_number_symbol)
  s7_pointer x, p;
  s7_int num_a, den_a;
  s7_double rl_a, im_a;

  x = car(args);
  p = cdr(args);

  switch (type(x))
    {
    case T_INTEGER:
      num_a = integer(x);
      while (true)
	{
	  x = car(p);
	  p = cdr(p);
	  switch (type(x))
	    {
	    case T_INTEGER:
	      if (num_a != integer(x)) goto NOT_EQUAL;
	      break;

	    case T_RATIO:
	    case T_COMPLEX:
	      goto NOT_EQUAL;

	    case T_REAL:
	      if (num_a != real(x)) goto NOT_EQUAL;
	      break;

	    default:
	      return(method_or_bust_with_type(sc, x, sc->eq_symbol, cons_unchecked(sc, make_integer(sc, num_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	    }
	  if (is_null(p))
	    return(sc->T);
	}

    case T_RATIO:
      num_a = numerator(x);
      den_a = denominator(x);
      rl_a = 0.0;
      while (true)
	{
	  x = car(p);
	  p = cdr(p);
	  switch (type(x))
	    {
	    case T_INTEGER:
	    case T_COMPLEX:
	      goto NOT_EQUAL;

	    case T_RATIO:
	      if ((num_a != numerator(x)) || (den_a != denominator(x)))	goto NOT_EQUAL; /* hidden cast here */
	      break;

	    case T_REAL:
	      if (rl_a == 0.0)
		rl_a = ((long_double)num_a) / ((long_double)den_a);
	      if (rl_a != real(x)) goto NOT_EQUAL;
	      break;

	    default:
	      return(method_or_bust_with_type(sc, x, sc->eq_symbol, cons_unchecked(sc, s7_make_ratio(sc, num_a, den_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	    }
	  if (is_null(p))
	    return(sc->T);
	}

    case T_REAL:
      rl_a = real(x);
      while (true)
	{
	  x = car(p);
	  p = cdr(p);
	  switch (type(x))
	    {
	    case T_INTEGER:
	      if (rl_a != integer(x)) goto NOT_EQUAL;
	      break;

	    case T_RATIO:
	      if (rl_a != (double)fraction(x)) goto NOT_EQUAL;
	      /* the cast to double is needed because rl_a is s7_double and we want (= ratio real) to be the same as (= real ratio):
	       *   (= 1.0 9223372036854775807/9223372036854775806)
	       *   (= 9223372036854775807/9223372036854775806 1.0)
	       */
	      break;

	    case T_REAL:
	      if (rl_a != real(x)) goto NOT_EQUAL;
	      break;

	    case T_COMPLEX:
	      goto NOT_EQUAL;

	    default:
	      return(method_or_bust_with_type(sc, x, sc->eq_symbol, cons_unchecked(sc, make_real(sc, rl_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	    }
	  if (is_null(p))
	    return(sc->T);
	}

    case T_COMPLEX:
      rl_a = real_part(x);
      im_a = imag_part(x);
      while (true)
	{
	  x = car(p);
	  p = cdr(p);
	  switch (type(x))
	    {
	    case T_INTEGER:
	    case T_RATIO:
	    case T_REAL:
	      goto NOT_EQUAL;

	    case T_COMPLEX:
	      if ((rl_a != real_part(x)) || (im_a != imag_part(x)))
		goto NOT_EQUAL;
	      break;

	    default:
	      return(method_or_bust_with_type(sc, x, sc->eq_symbol, cons_unchecked(sc, s7_make_complex(sc, rl_a, im_a), cons(sc, x, p)), a_number_string, position_of(p, args) - 1));
	    }
	  if (is_null(p))
	    return(sc->T);
	}

    default:
      return(method_or_bust_with_type(sc, x, sc->eq_symbol, args, a_number_string, 1));
    }

 NOT_EQUAL:
  for (; is_pair(p); p = cdr(p))
    if (!is_number_via_method(sc, car(p)))
      return(wrong_type_argument_with_type(sc, sc->eq_symbol, position_of(p, args), car(p), a_number_string));

  return(sc->F);
}


static s7_int c_object_length_to_int(s7_scheme *sc, s7_pointer obj);

#if (!WITH_GMP)
static s7_pointer g_equal_s_ic(s7_scheme *sc, s7_pointer args)
{
  s7_int y;
  s7_pointer val;

  val = symbol_to_value_unchecked(sc, car(args));
  y = integer(cadr(args));
  if (is_integer(val))
    return(make_boolean(sc, integer(val) == y));

  switch (type(val))
    {
    case T_INTEGER: return(make_boolean(sc, integer(val) == y));
    case T_RATIO:   return(sc->F);
    case T_REAL:    return(make_boolean(sc, real(val) == y));
    case T_COMPLEX: return(sc->F);
    default: return(method_or_bust_with_type(sc, val, sc->eq_symbol, list_2(sc, val, cadr(args)), a_number_string, 1)); /* not just args here -- need symbol lookup -> val */
    }
  return(sc->T);
}

static s7_pointer g_equal_length_ic(s7_scheme *sc, s7_pointer args)
{
  /* avoid make_integer (and telescope opts), we get here with car=length expr, cadr=int */
  s7_int ilen;
  s7_pointer val;

  val = symbol_to_value_unchecked(sc, cadar(args));
  ilen = integer(cadr(args));

  switch (type(val))
    {
    case T_PAIR:         return(make_boolean(sc, s7_list_length(sc, val) == ilen));
    case T_NIL:          return(make_boolean(sc, ilen == 0));
    case T_STRING:       return(make_boolean(sc, string_length(val) == ilen));
    case T_HASH_TABLE:   return(make_boolean(sc, (hash_table_mask(val) + 1) == ilen));
    case T_C_OBJECT:     return(make_boolean(sc, c_object_length_to_int(sc, val) == ilen));
    case T_LET:          return(make_boolean(sc, let_length(sc, val) == ilen));
    case T_BYTE_VECTOR:
    case T_INT_VECTOR:
    case T_FLOAT_VECTOR:
    case T_VECTOR:       return(make_boolean(sc, vector_length(val) == ilen));
    case T_CLOSURE:
    case T_CLOSURE_STAR: if (has_methods(val)) return(make_boolean(sc, closure_length(sc, val) == ilen));
    case T_ITERATOR:
      {
	s7_pointer len;
	len = s7_length(sc, iterator_sequence(val));
	return(make_boolean(sc, (is_integer(len)) && (integer(len) == ilen)));
      }
    default:
      return(simple_wrong_type_argument_with_type(sc, sc->length_symbol, val, a_sequence_string));
      /* here we already lost because we checked for the length above */
    }
  return(sc->F);
}


static bool equal_b_pi(s7_scheme *sc, s7_pointer x, s7_int y)
{
  if (is_integer(x))
    return(integer(x) == y);

  switch (type(x))
    {
    case T_INTEGER: return(integer(x) == y);
    case T_RATIO:   return(false);
    case T_REAL:    return(real(x) == y);
    case T_COMPLEX: return(false);
    default:
      simple_wrong_type_argument_with_type(sc, sc->eq_symbol, x, a_number_string);
    }
  return(false);
}

static s7_pointer eq_out_x(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_methods(x))
    return(find_and_apply_method(sc, find_let(sc, x), sc->eq_symbol, list_2(sc, x, y)));
  return(wrong_type_argument_with_type(sc, sc->eq_symbol, 1, x, a_number_string));
}

static s7_pointer eq_out_y(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_methods(y))
    return(find_and_apply_method(sc, find_let(sc, y), sc->eq_symbol, list_2(sc, x, y)));
  return(wrong_type_argument_with_type(sc, sc->eq_symbol, 2, y, a_number_string));
}

static s7_pointer c_equal_2(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
#if (!MS_WINDOWS)
  if (type(x) == type(y))
    {
      if (type(x) == T_INTEGER)
	return(make_boolean(sc, integer(x) == integer(y)));
      if (type(x) == T_REAL)
	return(make_boolean(sc, real(x) == real(y)));
      if (type(x) == T_COMPLEX)
	return(make_boolean(sc, (real_part(x) == real_part(y)) && (imag_part(x) == imag_part(y))));
      if (type(x) == T_RATIO)
	return(make_boolean(sc, (numerator(x) == numerator(y)) && (denominator(x) == denominator(y))));
    }
#endif

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER: return(make_boolean(sc, integer(x) == integer(y)));
	case T_RATIO:   return(sc->F);
	case T_REAL:    return(make_boolean(sc, integer(x) == real(y)));
	case T_COMPLEX: return(sc->F);
	default:
	  return(eq_out_y(sc, x, y));
	}
      break;

    case T_RATIO:
      switch (type(y))
	{
	case T_INTEGER: return(sc->F);
	case T_RATIO:   return(make_boolean(sc, (numerator(x) == numerator(y)) && (denominator(x) == denominator(y))));
	case T_REAL:    return(make_boolean(sc, fraction(x) == real(y)));            /* this could avoid the divide via numerator == denominator * x */
	case T_COMPLEX: return(sc->F);
	default:
	  return(eq_out_y(sc, x, y));
	}
      break;

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: return(make_boolean(sc, real(x) == integer(y)));
	case T_RATIO:   return(make_boolean(sc, real(x) == fraction(y)));
	case T_REAL:    return(make_boolean(sc, real(x) == real(y)));
	case T_COMPLEX: return(sc->F);
	default:
	  return(eq_out_y(sc, x, y));
	}
      break;

    case T_COMPLEX:
      switch (type(y))
	{
	case T_INTEGER:
	case T_RATIO:
	case T_REAL:
	  return(sc->F);

#if (!MS_WINDOWS)
	case T_COMPLEX:
	  return(make_boolean(sc, (real_part(x) == real_part(y)) && (imag_part(x) == imag_part(y))));
#else
	case T_COMPLEX:
	  if ((real_part(x) == real_part(y)) && (imag_part(x) == imag_part(y))) return(sc->T); else return(sc->F);
#endif
	default:
	  return(eq_out_y(sc, x, y));
	}
      break;

    default:
      return(eq_out_x(sc, x, y));
    }
  return(sc->F);
}

static s7_pointer equal_p_pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2) {return(c_equal_2(sc, p1, p2));}
static s7_pointer equal_p_pi(s7_scheme *sc, s7_pointer p1, s7_int p2)
{
  if (is_integer(p1))
    return((integer(p1) == p2) ? sc->T : sc->F);
  if (is_t_real(p1))
    return((real(p1) == p2) ? sc->T : sc->F);
  if (is_number(p1))
    return(sc->F);
  return(wrong_type_argument_with_type(sc, sc->eq_symbol, 1, p1, a_number_string));
}

static s7_pointer equal_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_boolean(sc, x1 == x2));}
static s7_pointer gt_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_boolean(sc, x1 > x2));}
static s7_pointer geq_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_boolean(sc, x1 >= x2));}
static s7_pointer lt_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_boolean(sc, x1 < x2));}
static s7_pointer leq_p_dd(s7_scheme *sc, s7_double x1, s7_double x2) {return(make_boolean(sc, x1 <= x2));}
static s7_pointer equal_p_ii(s7_scheme *sc, s7_int x1, s7_int x2) {return(make_boolean(sc, x1 == x2));}
static s7_pointer gt_p_ii(s7_scheme *sc, s7_int x1, s7_int x2) {return(make_boolean(sc, x1 > x2));}
static s7_pointer geq_p_ii(s7_scheme *sc, s7_int x1, s7_int x2) {return(make_boolean(sc, x1 >= x2));}
static s7_pointer lt_p_ii(s7_scheme *sc, s7_int x1, s7_int x2) {return(make_boolean(sc, x1 < x2));}
static s7_pointer leq_p_ii(s7_scheme *sc, s7_int x1, s7_int x2) {return(make_boolean(sc, x1 <= x2));}

static s7_pointer g_equal_2(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;
  x = car(args);
  y = cadr(args);
  if ((type(x) == T_INTEGER) && (type(y) == T_INTEGER)) /* this is by far the most common case (ratios aren't used much, and = with floats is frowned upon) */
    return(make_boolean(sc, integer(x) == integer(y)));
  return(c_equal_2(sc, x, y));
}

static s7_pointer g_equal_2i(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x, y;
  x = car(args);
  y = cadr(args);
  if (type(x) == T_INTEGER)
    return(make_boolean(sc, integer(x) == integer(y)));
  if (type(x) == T_REAL)
    return(make_boolean(sc, real(x) == integer(y)));
  if (!is_number(x))
    return(eq_out_x(sc, x, y));
  return(sc->F);
}

static s7_pointer g_less(s7_scheme *sc, s7_pointer args)
{
  #define H_less "(< x1 ...) returns #t if its arguments are in increasing order"
  #define Q_less s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_real_symbol)

  s7_pointer x, y, p;

  x = car(args);
  p = cdr(args);

  switch (type(x))
    {
    case T_INTEGER:
    INTEGER_LESS:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(x) >= integer(y)) goto NOT_LESS;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_LESS;

	case T_RATIO:
	  /* no gmp here, but this can overflow: (< 9223372036 1/9223372036), but conversion to real is also problematic
	   */
	  if ((integer(x) >= 0) && (numerator(y) < 0)) goto NOT_LESS;  /* (< 1 -1/2), ratio numerator can't be 0 */
	  if ((integer(x) <= 0) && (numerator(y) > 0))                 /* (< 0 1/2) */
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto RATIO_LESS;
	    }
	  if ((integer(x) < s7_int32_max) &&
	      (integer(x) > s7_int32_min) &&
	      (denominator(y) < s7_int32_max))
	    {
	      if ((integer(x) * denominator(y)) >= numerator(y)) goto NOT_LESS;
	    }
	  else
	    {
	      if (integer(x) >= fraction(y)) goto NOT_LESS;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_LESS;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_LESS;
	  if (integer(x) >= real(y)) goto NOT_LESS;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_LESS;

	default:
	  return(method_or_bust(sc, y, sc->lt_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}

    case T_RATIO:
    RATIO_LESS:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if ((numerator(x) > 0) && (integer(y) <= 0)) goto NOT_LESS;
	  if ((numerator(x) < 0) && (integer(y) >= 0))
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto INTEGER_LESS;
	    }
	  if ((integer(y) < s7_int32_max) &&
	      (integer(y) > s7_int32_min) &&
	      (denominator(x) < s7_int32_max))
	    {
	      if (numerator(x) >= (integer(y) * denominator(x))) goto NOT_LESS;
	    }
	  else
	    {
	      if (fraction(x) >= integer(y)) goto NOT_LESS;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_LESS;

	case T_RATIO:
	  /* conversion to real and >= is not safe here (see comment under g_greater) */
	  {
	    s7_int d1, d2, n1, n2;
	    d1 = denominator(x);
	    n1 = numerator(x);
	    d2 = denominator(y);
	    n2 = numerator(y);
	    if (d1 == d2)
	      {
		if (n1 >= n2) goto NOT_LESS;
	      }
	    else
	      {
#if HAVE_OVERFLOW_CHECKS
		if ((multiply_overflow(n1, d2, &n1)) ||
		    (multiply_overflow(n2, d1, &n2)))
		  {
		    if (fraction(x) >= fraction(y)) goto NOT_LESS;
		  }
		else
		  {
		    if (n1 >= n2) goto NOT_LESS;
		  }
#else
		if ((n1 * d2) >=  (n2 * d1)) goto NOT_LESS;
#endif
	      }
	  }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_LESS;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_LESS;
	  if (fraction(x) >= real(y)) goto NOT_LESS;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_LESS;

	default:
	  return(method_or_bust(sc, y, sc->lt_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}

    case T_REAL:
      if (is_NaN(real(x))) goto NOT_LESS;

    REAL_LESS:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (real(x) >= integer(y)) goto NOT_LESS;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_LESS;

	case T_RATIO:
	  if (real(x) >= fraction(y)) goto NOT_LESS;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_LESS;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_LESS;
	  if (real(x) >= real(y)) goto NOT_LESS;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_LESS;

	default:
	  return(method_or_bust(sc, y, sc->lt_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}

    default:
      return(method_or_bust(sc, x, sc->lt_symbol, args, T_REAL, 1));
    }

 NOT_LESS:
  for (; is_pair(p); p = cdr(p))
    if (!is_real_via_method(sc, car(p)))
      return(wrong_type_argument(sc, sc->lt_symbol, position_of(p, args), car(p), T_REAL));

  return(sc->F);
}


static s7_pointer g_less_or_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_less_or_equal "(<= x1 ...) returns #t if its arguments are in increasing order"
  #define Q_less_or_equal s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_real_symbol)

  s7_pointer x, y, p;

  x = car(args);
  p = cdr(args);

  switch (type(x))
    {
    case T_INTEGER:
    INTEGER_LEQ:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(x) > integer(y)) goto NOT_LEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_LEQ;

	case T_RATIO:
	  /* no gmp here, but this can overflow: (< 9223372036 1/9223372036), but conversion to real is also problematic
	   */
	  if ((integer(x) >= 0) && (numerator(y) < 0)) goto NOT_LEQ;  /* (< 1 -1/2), ratio numerator can't be 0 */
	  if ((integer(x) <= 0) && (numerator(y) > 0))                 /* (< 0 1/2) */
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto RATIO_LEQ;
	    }
	  if ((integer(x) < s7_int32_max) &&
	      (integer(x) > s7_int32_min) &&
	      (denominator(y) < s7_int32_max))
	    {
	      if ((integer(x) * denominator(y)) > numerator(y)) goto NOT_LEQ;
	    }
	  else
	    {
	      if (integer(x) > fraction(y)) goto NOT_LEQ;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_LEQ;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_LEQ;
	  if (integer(x) > real(y)) goto NOT_LEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_LEQ;

	default:
	  return(method_or_bust(sc, y, sc->leq_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}

    case T_RATIO:
    RATIO_LEQ:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if ((numerator(x) > 0) && (integer(y) <= 0)) goto NOT_LEQ;
	  if ((numerator(x) < 0) && (integer(y) >= 0))
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto INTEGER_LEQ;
	    }
	  if ((integer(y) < s7_int32_max) &&
	      (integer(y) > s7_int32_min) &&
	      (denominator(x) < s7_int32_max))
	    {
	      if (numerator(x) > (integer(y) * denominator(x))) goto NOT_LEQ;
	    }
	  else
	    {
	      if (fraction(x) > integer(y)) goto NOT_LEQ;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_LEQ;

	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    d1 = denominator(x);
	    n1 = numerator(x);
	    d2 = denominator(y);
	    n2 = numerator(y);
	    if (d1 == d2)
	      {
		if (n1 > n2) goto NOT_LEQ;
	      }
	    else
	      {
#if HAVE_OVERFLOW_CHECKS
		if ((multiply_overflow(n1, d2, &n1)) ||
		    (multiply_overflow(n2, d1, &n2)))
		  {
		    if (fraction(x) > fraction(y)) goto NOT_LEQ;
		  }
		else
		  {
		    if (n1 > n2) goto NOT_LEQ;
		  }
#else
		if ((n1 * d2) >  (n2 * d1)) goto NOT_LEQ;
#endif
	      }
	  }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_LEQ;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_LEQ;
	  if (fraction(x) > real(y)) goto NOT_LEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_LEQ;

	default:
	  return(method_or_bust(sc, y, sc->leq_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}

    case T_REAL:
      if (is_NaN(real(x))) goto NOT_LEQ;

    REAL_LEQ:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (real(x) > integer(y)) goto NOT_LEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_LEQ;

	case T_RATIO:
	  if (real(x) > fraction(y)) goto NOT_LEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_LEQ;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_LEQ;
	  if (real(x) > real(y)) goto NOT_LEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_LEQ;

	default:
	  return(method_or_bust(sc, y, sc->leq_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}

    default:
      return(method_or_bust(sc, x, sc->leq_symbol, args, T_REAL, 1));
    }

 NOT_LEQ:
  for (; is_pair(p); p = cdr(p))
    if (!is_real_via_method(sc, car(p)))
      return(wrong_type_argument(sc, sc->leq_symbol, position_of(p, args), car(p), T_REAL));

  return(sc->F);
}


static s7_pointer g_greater(s7_scheme *sc, s7_pointer args)
{
  #define H_greater "(> x1 ...) returns #t if its arguments are in decreasing order"
  #define Q_greater s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_real_symbol)

  s7_pointer x, y, p;
  x = car(args);
  p = cdr(args);

  switch (type(x))
    {
    case T_INTEGER:
    INTEGER_GREATER:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(x) <= integer(y)) goto NOT_GREATER;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_GREATER;

	case T_RATIO:
	  /* no gmp here, but this can overflow: (< 9223372036 1/9223372036), but conversion to real is also problematic
	   */
	  if ((integer(x) <= 0) && (numerator(y) > 0)) goto NOT_GREATER;
	  if ((integer(x) >= 0) && (numerator(y) < 0))
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto RATIO_GREATER;
	    }
	  if ((integer(x) < s7_int32_max) &&
	      (integer(x) > s7_int32_min) &&
	      (denominator(y) < s7_int32_max))
	    {
	      if ((integer(x) * denominator(y)) <= numerator(y)) goto NOT_GREATER;
	    }
	  else
	    {
	      if (integer(x) <= fraction(y)) goto NOT_GREATER;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_GREATER;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_GREATER;
	  if (integer(x) <= real(y)) goto NOT_GREATER;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_GREATER;

	default:
	  return(method_or_bust(sc, y, sc->gt_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}

    case T_RATIO:
    RATIO_GREATER:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if ((numerator(x) < 0) && (integer(y) >= 0)) goto NOT_GREATER;
	  if ((numerator(x) > 0) && (integer(y) <= 0))
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto INTEGER_GREATER;
	    }
	  if ((integer(y) < s7_int32_max) &&
	      (integer(y) > s7_int32_min) &&
	      (denominator(x) < s7_int32_max))
	    {
	      if (numerator(x) <= (integer(y) * denominator(x))) goto NOT_GREATER;
	    }
	  else
	    {
	      if (fraction(x) <= integer(y)) goto NOT_GREATER;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_GREATER;

	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    d1 = denominator(x);
	    n1 = numerator(x);
	    d2 = denominator(y);
	    n2 = numerator(y);
	    if (d1 == d2)
	      {
		if (n1 <= n2) goto NOT_GREATER;
	      }
	    else
	      {
#if HAVE_OVERFLOW_CHECKS
		if ((multiply_overflow(n1, d2, &n1)) ||
		    (multiply_overflow(n2, d1, &n2)))
		  {
		    if (fraction(x) <= fraction(y)) goto NOT_GREATER;
		  }
		else
		  {
		    if (n1 <= n2) goto NOT_GREATER;
		  }
#else
		if ((n1 * d2) <=  (n2 * d1)) goto NOT_GREATER;
#endif
	      }
	  }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_GREATER;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_GREATER;
	  if (fraction(x) <= real(y)) goto NOT_GREATER;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_GREATER;

	default:
	  return(method_or_bust(sc, y, sc->gt_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}

    case T_REAL:
      if (is_NaN(real(x))) goto NOT_GREATER;

    REAL_GREATER:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (real(x) <= integer(y)) goto NOT_GREATER;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_GREATER;

	case T_RATIO:
	  if (real(x) <= fraction(y)) goto NOT_GREATER;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_GREATER;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_GREATER;
	  if (real(x) <= real(y)) goto NOT_GREATER;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_GREATER;

	default:
	  return(method_or_bust(sc, y, sc->gt_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}

    default:
      return(method_or_bust(sc, x, sc->gt_symbol, args, T_REAL, 1));
    }

 NOT_GREATER:
  for (; is_pair(p); p = cdr(p))
    if (!is_real_via_method(sc, car(p)))
      return(wrong_type_argument(sc, sc->gt_symbol, position_of(p, args), car(p), T_REAL));

  return(sc->F);
}


static s7_pointer g_greater_or_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_greater_or_equal "(>= x1 ...) returns #t if its arguments are in decreasing order"
  #define Q_greater_or_equal s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_real_symbol)
  /* (>= 1+i 1+i) is an error which seems unfortunate */
  s7_pointer x, y, p;

  x = car(args);
  p = cdr(args);

  switch (type(x))
    {
    case T_INTEGER:
    INTEGER_GEQ:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (integer(x) < integer(y)) goto NOT_GEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_GEQ;

	case T_RATIO:
	  /* no gmp here, but this can overflow: (< 9223372036 1/9223372036), but conversion to real is also problematic
	   */
	  if ((integer(x) <= 0) && (numerator(y) > 0)) goto NOT_GEQ;
	  if ((integer(x) >= 0) && (numerator(y) < 0))
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto RATIO_GEQ;
	    }
	  if ((integer(x) < s7_int32_max) &&
	      (integer(x) > s7_int32_min) &&
	      (denominator(y) < s7_int32_max))
	    {
	      if ((integer(x) * denominator(y)) < numerator(y)) goto NOT_GEQ;
	    }
	  else
	    {
	      if (integer(x) < fraction(y)) goto NOT_GEQ;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_GEQ;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_GEQ;
	  if (integer(x) < real(y)) goto NOT_GEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_GEQ;

	default:
	  return(method_or_bust(sc, y, sc->geq_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}

    case T_RATIO:
    RATIO_GEQ:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if ((numerator(x) < 0) && (integer(y) >= 0)) goto NOT_GEQ;
	  if ((numerator(x) > 0) && (integer(y) <= 0))
	    {
	      if (is_null(p)) return(sc->T);
	      x = y;
	      goto INTEGER_GEQ;
	    }
	  if ((integer(y) < s7_int32_max) &&
	      (integer(y) > s7_int32_min) &&
	      (denominator(x) < s7_int32_max))
	    {
	      if (numerator(x) < (integer(y) * denominator(x))) goto NOT_GEQ;
	    }
	  else
	    {
	      if (fraction(x) < integer(y)) goto NOT_GEQ;
	    }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_GEQ;

	case T_RATIO:
	  {
	    s7_int d1, d2, n1, n2;
	    d1 = denominator(x);
	    n1 = numerator(x);
	    d2 = denominator(y);
	    n2 = numerator(y);
	    if (d1 == d2)
	      {
		if (n1 < n2) goto NOT_GEQ;
	      }
	    else
	      {
#if HAVE_OVERFLOW_CHECKS
		if ((multiply_overflow(n1, d2, &n1)) ||
		    (multiply_overflow(n2, d1, &n2)))
		  {
		    if (fraction(x) < fraction(y)) goto NOT_GEQ;
		  }
		else
		  {
		    if (n1 < n2) goto NOT_GEQ;
		  }
#else
		if ((n1 * d2) <  (n2 * d1)) goto NOT_GEQ;
#endif
	      }
	  }
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_GEQ;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_GEQ;
	  if (fraction(x) < real(y)) goto NOT_GEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_GEQ;

	default:
	  return(method_or_bust(sc, y, sc->geq_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}

    case T_REAL:
      if (is_NaN(real(x))) goto NOT_GEQ;

    REAL_GEQ:
      y = car(p);
      p = cdr(p);
      switch (type(y))
	{
	case T_INTEGER:
	  if (real(x) < integer(y)) goto NOT_GEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto INTEGER_GEQ;

	case T_RATIO:
	  if (real(x) < fraction(y)) goto NOT_GEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto RATIO_GEQ;

	case T_REAL:
	  if (is_NaN(real(y))) goto NOT_GEQ;
	  if (real(x) < real(y)) goto NOT_GEQ;
	  if (is_null(p)) return(sc->T);
	  x = y;
	  goto REAL_GEQ;

	default:
	  return(method_or_bust(sc, y, sc->geq_symbol, cons_unchecked(sc, x, cons(sc, y, p)), T_REAL, position_of(p, args) - 1));
	}

    default:
      return(method_or_bust(sc, x, sc->geq_symbol, args, T_REAL, 1));
    }

 NOT_GEQ:
  for (; is_pair(p); p = cdr(p))
    if (!is_real_via_method(sc, car(p)))
      return(wrong_type_argument(sc, sc->geq_symbol, position_of(p, args), car(p), T_REAL));

  return(sc->F);

}

static s7_pointer g_less_s0(s7_scheme *sc, s7_pointer args)
{
  s7_pointer x;
  x = car(args);
  if (is_integer(x))
    return(make_boolean(sc, integer(x) < 0));
  if (is_real(x))
    return(make_boolean(sc, s7_is_negative(x)));
  return(method_or_bust(sc, x, sc->lt_symbol, args, T_REAL, 1));
}

static bool ratio_lt_pi(s7_pointer x, s7_int y)
{
  if ((y >= 0) && (numerator(x) < 0))
    return(true);
  if ((y <= 0) && (numerator(x) > 0))
    return(false);
  if (denominator(x) < s7_int32_max)
    return(numerator(x) < (y * denominator(x)));
  return(fraction(x) < y);
}

static s7_pointer g_less_s_ic(s7_scheme *sc, s7_pointer args)
{
  s7_int y;
  s7_pointer x;

  x = car(args);
  y = integer(cadr(args));

  if (type(x) == T_INTEGER)
    return(make_boolean(sc, integer(x) < y));
  if (type(x) == T_REAL)
    return(make_boolean(sc, real(x) < y));
  if (type(x) == T_RATIO)
    return(make_boolean(sc, ratio_lt_pi(x, y)));
  return(method_or_bust(sc, x, sc->lt_symbol, args, T_REAL, 1));
}

static s7_pointer g_less_length_ic(s7_scheme *sc, s7_pointer args)
{
  s7_int ilen;
  s7_pointer val;

  val = symbol_to_value_unchecked(sc, cadar(args));
  ilen = integer(cadr(args));

  switch (type(val))
    {
    case T_PAIR:         return(make_boolean(sc, s7_list_length(sc, val) < ilen));
    case T_NIL:          return(make_boolean(sc, ilen > 0));
    case T_STRING:       return(make_boolean(sc, string_length(val) < ilen));
    case T_HASH_TABLE:   return(make_boolean(sc, (hash_table_mask(val) + 1) < ilen)); /* was <=? -- changed 15-Dec-15, then again 6-Jan-17: mask is len-1 */
    case T_C_OBJECT:     return(make_boolean(sc, c_object_length_to_int(sc, val) < ilen));
    case T_LET:          return(make_boolean(sc, let_length(sc, val) < ilen));  /* this works because let_length handles the length method itself! */
    case T_BYTE_VECTOR:
    case T_INT_VECTOR:
    case T_FLOAT_VECTOR:
    case T_VECTOR:       return(make_boolean(sc, vector_length(val) < ilen));
    case T_CLOSURE:
    case T_CLOSURE_STAR: if (has_methods(val)) return(make_boolean(sc, closure_length(sc, val) < ilen));
    case T_ITERATOR:
      {
	s7_pointer len;
	len = s7_length(sc, iterator_sequence(val));
	return(make_boolean(sc, (is_integer(len)) && (integer(len) < ilen)));
      }
    default:
      return(simple_wrong_type_argument_with_type(sc, sc->length_symbol, val, a_sequence_string)); /* no check method here because we checked above */
    }
  return(sc->F);
}

static bool lt_out_x(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_methods(x))
    return(find_and_apply_method(sc, find_let(sc, x), sc->lt_symbol, list_2(sc, x, y)) != sc->F);
  wrong_type_argument(sc, sc->lt_symbol, 1, x, T_REAL);
  return(false);
}

static bool lt_out_y(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_methods(y))
    return(find_and_apply_method(sc, find_let(sc, y), sc->lt_symbol, list_2(sc, x, y)) != sc->F);
  wrong_type_argument(sc, sc->lt_symbol, 2, y, T_REAL);
  return(false);
}

static inline bool lt_b_7pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (type(x) == type(y))
    {
      if (type(x) == T_INTEGER)
	return(integer(x) < integer(y));
      if (type(x) == T_REAL)
	return(real(x) < real(y));
      if (type(x) == T_RATIO)
	return(fraction(x) < fraction(y));
    }
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(integer(x) < integer(y));

	case T_RATIO:
	  return(g_less(sc, set_plist_2(sc, x, y)) != sc->F);

	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  return(integer(x) < real(y));

	default:
	  return(lt_out_y(sc, x, y));
	}
      break;

    case T_RATIO:
      return(g_less(sc, set_plist_2(sc, x, y)) != sc->F);

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (is_NaN(real(x))) return(false);
	  return(real(x) < integer(y));

	case T_RATIO:
	  if (is_NaN(real(x))) return(false);
	  return(real(x) < fraction(y));

	case T_REAL:
	  if (is_NaN(real(x))) return(false);
	  return(real(x) < real(y));

	default:
	  return(lt_out_y(sc, x, y));
	}
      break;

    default:
      return(lt_out_x(sc, x, y));
    }
  return(true);
}

static s7_pointer lt_p_pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2) {return(make_boolean(sc, lt_b_7pp(sc, p1, p2)));}
static s7_pointer g_less_2(s7_scheme *sc, s7_pointer args) {return(lt_p_pp(sc, car(args), cadr(args)));}

static bool ratio_leq_pi(s7_pointer x, s7_int y)
{
  if ((y >= 0) && (numerator(x) <= 0))
    return(true);
  if ((y <= 0) && (numerator(x) > 0))
    return(false);
  if (denominator(x) < s7_int32_max)
    return(numerator(x) <= (y * denominator(x)));
  return(fraction(x) <= y);
}

static s7_pointer g_leq_s_ic(s7_scheme *sc, s7_pointer args)
{
  s7_int y;
  s7_pointer x;

  x = car(args);
  y = integer(cadr(args));

  if (type(x) == T_INTEGER)
    return(make_boolean(sc, integer(x) <= y));
  if (type(x) == T_REAL)
    return(make_boolean(sc, real(x) <= y));
  if (type(x) == T_RATIO)
    return(make_boolean(sc, ratio_leq_pi(x, y)));
  return(method_or_bust(sc, x, sc->leq_symbol, args, T_REAL, 1));
}


static inline bool leq_b_7pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (type(x) == type(y))
    {
      if (type(x) == T_INTEGER)
	return(integer(x) <= integer(y));
      if (type(x) == T_REAL)
	return(real(x) <= real(y));
      if (type(x) == T_RATIO)
	return(fraction(x) <= fraction(y));
    }

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(integer(x) <= integer(y));

	case T_RATIO:
	  return(g_less_or_equal(sc, set_plist_2(sc, x, y)) != sc->F);

	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  return(integer(x) <= real(y));

	default:
	  return(method_or_bust(sc, y, sc->leq_symbol, list_2(sc, x, y), T_REAL, 2) != sc->F);
	}
      break;

    case T_RATIO:
      return(g_less_or_equal(sc, set_plist_2(sc, x, y)) != sc->F);

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (is_NaN(real(x))) return(false);
	  return(real(x) <= integer(y));

	case T_RATIO:
	  if (is_NaN(real(x))) return(false);
	  return(real(x) <= fraction(y));

	case T_REAL:
	  if (is_NaN(real(x))) return(false);
	  return(real(x) <= real(y));

	default:
	  return(method_or_bust(sc, y, sc->leq_symbol, list_2(sc, x, y), T_REAL, 2) != sc->F);
	}
      break;

    default:
      return(method_or_bust(sc, x, sc->leq_symbol, list_2(sc, x, y), T_REAL, 1) != sc->F);
    }
  return(true);
}

static s7_pointer leq_p_pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2) {return(make_boolean(sc, leq_b_7pp(sc, p1, p2)));}
static s7_pointer g_leq_2(s7_scheme *sc, s7_pointer args) {return(make_boolean(sc, leq_b_7pp(sc, car(args), cadr(args))));}

static s7_pointer g_greater_s_ic(s7_scheme *sc, s7_pointer args)
{
  s7_int y;
  s7_pointer x;

  x = car(args);
  y = integer(cadr(args));

  if (type(x) == T_INTEGER)
    return(make_boolean(sc, integer(x) > y));
  if (type(x) == T_REAL)
    return(make_boolean(sc, real(x) > y));
  if (type(x) == T_RATIO)
    return(make_boolean(sc, !ratio_leq_pi(x, y)));
  return(method_or_bust_with_type(sc, x, sc->gt_symbol, args, a_number_string, 1));
}

static s7_pointer g_greater_s_fc(s7_scheme *sc, s7_pointer args)
{
  s7_double y;
  s7_pointer x;

  x = car(args);
  y = real(cadr(args));

  if (is_t_real(x))
    return(make_boolean(sc, real(x) > y));

  switch (type(x))
    {
    case T_INTEGER:
      return(make_boolean(sc, integer(x) > y));

    case T_RATIO:
      /* (> 9223372036854775807/9223372036854775806 1.0) */
      if (denominator(x) < s7_int32_max) /* y range check was handled in greater_chooser */
	return(make_boolean(sc, (numerator(x) > (y * denominator(x)))));
      return(make_boolean(sc, fraction(x) > y));

    case T_REAL:
      return(make_boolean(sc, real(x) > y));

    default:
      return(method_or_bust_with_type(sc, x, sc->gt_symbol, args, a_number_string, 1));
    }
  return(sc->T);
}

static bool gt_out_x(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_methods(x))
    return(find_and_apply_method(sc, find_let(sc, x), sc->gt_symbol, list_2(sc, x, y)) != sc->F);
  wrong_type_argument(sc, sc->gt_symbol, 1, x, T_REAL);
  return(false);
}

static bool gt_out_y(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_methods(y))
    return(find_and_apply_method(sc, find_let(sc, y), sc->gt_symbol, list_2(sc, x, y)) != sc->F);
  wrong_type_argument(sc, sc->gt_symbol, 2, y, T_REAL);
  return(false);
}

static inline bool gt_b_7pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (type(x) == type(y))
    {
      if (type(x) == T_INTEGER)
	return(integer(x) > integer(y));
      if (type(x) == T_REAL)
	return(real(x) > real(y));
      if (type(x) == T_RATIO)
	return(fraction(x) > fraction(y));
    }

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(integer(x) > integer(y));

	case T_RATIO:
	  return(g_greater(sc, set_plist_2(sc, x, y)) != sc->F);

	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  return(integer(x) > real(y));

	default:
	  return(gt_out_y(sc, x, y));
	}
      break;

    case T_RATIO:
      return(g_greater(sc, set_plist_2(sc, x, y)) != sc->F);

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (is_NaN(real(x))) return(false);
	  return(real(x) > integer(y));

	case T_RATIO:
	  if (is_NaN(real(x))) return(false);
	  return(real(x) > fraction(y));

	case T_REAL:
	  if (is_NaN(real(x))) return(false);
	  return(real(x) > real(y));

	default:
	  return(gt_out_y(sc, x, y));
	}
      break;

    default:
      return(gt_out_x(sc, x, y));
    }
  return(true);
}

static s7_pointer gt_p_pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2) {return(make_boolean(sc, gt_b_7pp(sc, p1, p2)));}

static s7_pointer g_greater_2(s7_scheme *sc, s7_pointer args)
{
#if 0
  return(gt_p_pp(sc, car(args), cadr(args)));
#else
  /* ridiculous repetition, but overheads are killing this poor thing */
  s7_pointer x, y;
  x = car(args);
  y = cadr(args);
#if (!MS_WINDOWS)
  if (type(x) == type(y))
    {
      if (type(x) == T_INTEGER) return(make_boolean(sc, integer(x) > integer(y)));
      if (type(x) == T_REAL)	return(make_boolean(sc, real(x) > real(y)));
      if (type(x) == T_RATIO)	return(make_boolean(sc, fraction(x) > fraction(y)));
    }
#endif
  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER: return(make_boolean(sc, integer(x) > integer(y)));
	case T_RATIO:   return(g_greater(sc, set_plist_2(sc, x, y)));
	case T_REAL:    if (is_NaN(real(y))) return(sc->F); return(make_boolean(sc, integer(x) > real(y)));
	default:        return(make_boolean(sc, gt_out_y(sc, x, y)));
	}
      break;

    case T_RATIO:       return(g_greater(sc, set_plist_2(sc, x, y)));

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER: if (is_NaN(real(x))) return(sc->F); return(make_boolean(sc, real(x) > integer(y)));
	case T_RATIO:   if (is_NaN(real(x))) return(sc->F); return(make_boolean(sc, real(x) > fraction(y)));
	case T_REAL:    if (is_NaN(real(x))) return(sc->F); return(make_boolean(sc, real(x) > real(y)));
	default:        return(make_boolean(sc, gt_out_y(sc, x, y)));
	}
      break;

    default:            return(make_boolean(sc, gt_out_x(sc, x, y)));
    }
  return(sc->T);
#endif
}

static bool geq_out_x(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_methods(x))
    return(find_and_apply_method(sc, find_let(sc, x), sc->geq_symbol, list_2(sc, x, y)) != sc->F);
  wrong_type_argument(sc, sc->geq_symbol, 1, x, T_REAL);
  return(false);
}

static bool geq_out_y(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (has_methods(y))
    return(find_and_apply_method(sc, find_let(sc, y), sc->geq_symbol, list_2(sc, x, y)) != sc->F);
  wrong_type_argument(sc, sc->geq_symbol, 2, y, T_REAL);
  return(false);
}

static inline bool geq_b_7pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  if (type(x) == type(y))
    {
      if (type(x) == T_INTEGER)
	return(integer(x) >= integer(y));
      if (type(x) == T_REAL)
	return(real(x) >= real(y));
      if (type(x) == T_RATIO)
	return(fraction(x) >= fraction(y));
    }

  switch (type(x))
    {
    case T_INTEGER:
      switch (type(y))
	{
	case T_INTEGER:
	  return(integer(x) >= integer(y));

	case T_RATIO:
	  return(g_greater_or_equal(sc, set_plist_2(sc, x, y)) != sc->F);

	case T_REAL:
	  if (is_NaN(real(y))) return(false);
	  return(integer(x) >= real(y));

	default:
	  return(geq_out_y(sc, x, y));
	}
      break;

    case T_RATIO:
      return(g_greater_or_equal(sc, set_plist_2(sc, x, y)) != sc->F);

    case T_REAL:
      switch (type(y))
	{
	case T_INTEGER:
	  if (is_NaN(real(x))) return(false);
	  return(real(x) >= integer(y));

	case T_RATIO:
	  if (is_NaN(real(x))) return(false);
	  return(real(x) >= fraction(y));

	case T_REAL:
	  if (is_NaN(real(x))) return(false);
	  return(real(x) >= real(y));

	default:
	  return(geq_out_y(sc, x, y));
	}
      break;

    default:
      return(geq_out_x(sc, x, y));
    }
  return(true);
}

static s7_pointer geq_p_pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2) {return(make_boolean(sc, geq_b_7pp(sc, p1, p2)));}
static s7_pointer g_geq_2(s7_scheme *sc, s7_pointer args) {return(make_boolean(sc, geq_b_7pp(sc, car(args), cadr(args))));}

static s7_pointer g_geq_s_fc(s7_scheme *sc, s7_pointer args)
{
  s7_double y;
  s7_pointer x;

  x = car(args);
  y = real(cadr(args));

  if (is_t_real(x))
    return(make_boolean(sc, real(x) >= y));
  return(g_geq_2(sc, args));
}

static s7_pointer g_geq_s_ic(s7_scheme *sc, s7_pointer args)
{
  s7_int y;
  s7_pointer x;

  x = car(args);
  y = integer(cadr(args));

  if (type(x) == T_INTEGER)
    return(make_boolean(sc, integer(x) >= y));
  if (type(x) == T_REAL)
    return(make_boolean(sc, real(x) >= y));
  if (type(x) == T_RATIO)
    return(make_boolean(sc, !ratio_lt_pi(x, y)));
  return(method_or_bust(sc, x, sc->geq_symbol, args, T_REAL, 1));
}


static bool req_b_7pp(s7_scheme *sc, s7_pointer x, s7_pointer y) {return(c_equal_2(sc, x, y) != sc->F);}

static bool lt_b_pi(s7_scheme *sc, s7_pointer p1, s7_int p2)
{
  if (is_integer(p1)) return(integer(p1) < p2);
  if (is_t_real(p1))  return(real(p1) < p2);
  if (is_t_ratio(p1)) return(ratio_lt_pi(p1, p2));
  simple_wrong_type_argument(sc, sc->lt_symbol, p1, T_REAL);
  return(false);
}

static s7_pointer lt_p_pi(s7_scheme *sc, s7_pointer p1, s7_int p2) {return(make_boolean(sc, lt_b_pi(sc, p1, p2)));}

static bool leq_b_pi(s7_scheme *sc, s7_pointer p1, s7_int p2)
{
  if (is_integer(p1)) return(integer(p1) <= p2);
  if (is_t_real(p1))  return(real(p1) <= p2);
  if (is_t_ratio(p1)) return(ratio_leq_pi(p1, p2));
  simple_wrong_type_argument(sc, sc->leq_symbol, p1, T_REAL);
  return(false);
}

static s7_pointer leq_p_pi(s7_scheme *sc, s7_pointer p1, s7_int p2) {return(make_boolean(sc, leq_b_pi(sc, p1, p2)));}

static bool gt_b_pi(s7_scheme *sc, s7_pointer p1, s7_int p2)
{
  if (is_integer(p1)) return(integer(p1) > p2);
  if (is_t_real(p1))  return(real(p1) > p2);
  if (is_t_ratio(p1)) return(!ratio_leq_pi(p1, p2));
  simple_wrong_type_argument(sc, sc->gt_symbol, p1, T_REAL);
  return(false);
}

static s7_pointer gt_p_pi(s7_scheme *sc, s7_pointer p1, s7_int p2) {return(make_boolean(sc, gt_b_pi(sc, p1, p2)));}

static bool geq_b_pi(s7_scheme *sc, s7_pointer p1, s7_int p2)
{
  if (is_integer(p1)) return(integer(p1) >= p2);
  if (is_t_real(p1))  return(real(p1) >= p2);
  if (is_t_ratio(p1)) return(!ratio_lt_pi(p1, p2));
  simple_wrong_type_argument(sc, sc->geq_symbol, p1, T_REAL);
  return(false);
}

static s7_pointer geq_p_pi(s7_scheme *sc, s7_pointer p1, s7_int p2) {return(make_boolean(sc, geq_b_pi(sc, p1, p2)));}
#else
static s7_pointer big_less(s7_scheme *sc, s7_pointer args);
static bool lt_b_7pp(s7_scheme *sc, s7_pointer x, s7_pointer y)
{
  return(big_less(sc, set_plist_2(sc, x, y)) != sc->F);
}
#endif
/* end (!WITH_GMP) */

static bool req_b_ii(s7_int i1, s7_int i2) {return(i1 == i2);}
static bool lt_b_ii(s7_int i1, s7_int i2) {return(i1 < i2);}
static bool leq_b_ii(s7_int i1, s7_int i2) {return(i1 <= i2);}
static bool gt_b_ii(s7_int i1, s7_int i2) {return(i1 > i2);}
static bool geq_b_ii(s7_int i1, s7_int i2) {return(i1 >= i2);}
static bool req_b_dd(s7_double i1, s7_double i2) {return(i1 == i2);}
static bool lt_b_dd(s7_double i1, s7_double i2) {return(i1 < i2);}
static bool leq_b_dd(s7_double i1, s7_double i2) {return(i1 <= i2);}
static bool gt_b_dd(s7_double i1, s7_double i2) {return(i1 > i2);}
static bool geq_b_dd(s7_double i1, s7_double i2) {return(i1 >= i2);}


/* ---------------------------------------- real-part imag-part ---------------------------------------- */

s7_double s7_real_part(s7_pointer x)
{
  switch(type(x))
    {
    case T_INTEGER:     return((s7_double)integer(x));
    case T_RATIO:       return(fraction(x));
    case T_REAL:        return(real(x));
    case T_COMPLEX:     return(real_part(x));
#if WITH_GMP
    case T_BIG_INTEGER: return((s7_double)big_integer_to_s7_int(big_integer(x)));
    case T_BIG_RATIO:   return((s7_double)((long_double)big_integer_to_s7_int(mpq_numref(big_ratio(x))) / (long_double)big_integer_to_s7_int(mpq_denref(big_ratio(x)))));
    case T_BIG_REAL:    return((s7_double)mpfr_get_d(big_real(x), GMP_RNDN));
    case T_BIG_COMPLEX: return((s7_double)mpfr_get_d(mpc_realref(big_complex(x)), GMP_RNDN));
#endif
    }
  return(0.0);
}


s7_double s7_imag_part(s7_pointer x)
{
  switch (type(x))
    {
    case T_COMPLEX:     return(imag_part(x));
#if WITH_GMP
    case T_BIG_COMPLEX: return((s7_double)mpfr_get_d(mpc_imagref(big_complex(x)), GMP_RNDN));
#endif
    }
  return(0.0);
}

static s7_pointer g_real_part(s7_scheme *sc, s7_pointer args)
{
  #define H_real_part "(real-part num) returns the real part of num"
  #define Q_real_part s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_number_symbol)

  s7_pointer p;
  p = car(args);
  switch (type(p))
    {
    case T_INTEGER:
    case T_RATIO:
    case T_REAL:
      return(p);

    case T_COMPLEX:
      return(make_real(sc, real_part(p)));

#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:
    case T_BIG_REAL:
      return(p);

    case T_BIG_COMPLEX:
      {
	s7_pointer x;

	new_cell(sc, x, T_BIG_REAL);
	add_bigreal(sc, x);
	mpfr_init(big_real(x));
	mpc_real(big_real(x), big_complex(p), GMP_RNDN);

	return(x);
      }
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, p, sc->real_part_symbol, args, a_number_string));
    }
}


static s7_pointer g_imag_part(s7_scheme *sc, s7_pointer args)
{
  #define H_imag_part "(imag-part num) returns the imaginary part of num"
  #define Q_imag_part s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_number_symbol)
  s7_pointer p;
  /* currently (imag-part +nan.0) -> 0.0 ? it's true but maybe confusing */

  p = car(args);
  switch (type(p))
    {
    case T_INTEGER:
    case T_RATIO:
      return(small_int(0));

    case T_REAL:
      return(real_zero);

    case T_COMPLEX:
      return(make_real(sc, imag_part(p)));

#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:
      return(small_int(0));

    case T_BIG_REAL:
      return(real_zero);

    case T_BIG_COMPLEX:
      {
	s7_pointer x;
	new_cell(sc, x, T_BIG_REAL);
	add_bigreal(sc, x);
	mpfr_init(big_real(x));
	mpc_imag(big_real(x), big_complex(p), GMP_RNDN);

	return(x);
      }
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, p, sc->imag_part_symbol, args, a_number_string));
    }
}

#if (!WITH_GMP)
static s7_double real_part_d_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_number(p))
    simple_wrong_type_argument_with_type(sc, sc->real_part_symbol, p, a_number_string);
  return(s7_real_part(p));
}

static s7_double imag_part_d_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_number(p))
    simple_wrong_type_argument_with_type(sc, sc->imag_part_symbol, p, a_number_string);
  return(s7_imag_part(p));
}
#endif


/* ---------------------------------------- numerator denominator ---------------------------------------- */

static s7_pointer g_numerator(s7_scheme *sc, s7_pointer args)
{
  #define H_numerator "(numerator rat) returns the numerator of the rational number rat"
  #define Q_numerator s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_rational_symbol)

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_RATIO:       return(make_integer(sc, numerator(x)));
    case T_INTEGER:     return(x);
#if WITH_GMP
    case T_BIG_INTEGER: return(x);
    case T_BIG_RATIO:   return(mpz_to_big_integer(sc, mpq_numref(big_ratio(x))));
#endif
    default:            return(method_or_bust_with_type_one_arg(sc, x, sc->numerator_symbol, args, a_rational_string));
    }
}

static s7_int numerator_i_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_rational(p))
    simple_wrong_type_argument(sc, sc->numerator_symbol, p, T_RATIO);
  return(numerator(p));
}


static s7_pointer g_denominator(s7_scheme *sc, s7_pointer args)
{
  #define H_denominator "(denominator rat) returns the denominator of the rational number rat"
  #define Q_denominator s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_rational_symbol)

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_RATIO:       return(make_integer(sc, denominator(x)));
    case T_INTEGER:     return(small_int(1));
#if WITH_GMP
    case T_BIG_INTEGER: return(small_int(1));
    case T_BIG_RATIO:   return(mpz_to_big_integer(sc, mpq_denref(big_ratio(x))));
#endif
    default:            return(method_or_bust_with_type_one_arg(sc, x, sc->denominator_symbol, args, a_rational_string));
    }
}

static s7_int denominator_i_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_rational(p))
    simple_wrong_type_argument(sc, sc->denominator_symbol, p, T_RATIO);
  if (is_integer(p))
    return(1);
  return(denominator(p));
}


/* ---------------------------------------- nan? infinite? ---------------------------------------- */

static s7_pointer g_is_nan(s7_scheme *sc, s7_pointer args)
{
  #define H_is_nan "(nan? obj) returns #t if obj is a NaN"
  #define Q_is_nan sc->pl_bn

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
    case T_RATIO:
      return(sc->F);

    case T_REAL:
      return(make_boolean(sc, is_NaN(real(x))));

    case T_COMPLEX:
      return(make_boolean(sc, (is_NaN(real_part(x))) || (is_NaN(imag_part(x)))));

#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:
      return(sc->F);

    case T_BIG_REAL:
      return(make_boolean(sc, is_NaN(s7_real_part(x))));

    case T_BIG_COMPLEX:
      return(make_boolean(sc, (is_NaN(s7_real_part(x))) || (is_NaN(s7_imag_part(x)))));
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->is_nan_symbol, list_1(sc, x), a_number_string));
    }
}

static bool is_nan_b_7p(s7_scheme *sc, s7_pointer p) {return(g_is_nan(sc, set_plist_1(sc, p)) != sc->F);}


static s7_pointer g_is_infinite(s7_scheme *sc, s7_pointer args)
{
  #define H_is_infinite "(infinite? obj) returns #t if obj is an infinite real"
  #define Q_is_infinite sc->pl_bn

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
    case T_RATIO:
      return(sc->F);

    case T_REAL:
      return(make_boolean(sc, is_inf(real(x))));

    case T_COMPLEX:
      return(make_boolean(sc, (is_inf(real_part(x))) || (is_inf(imag_part(x)))));

#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:
      return(sc->F);

    case T_BIG_REAL:
      return(make_boolean(sc, mpfr_inf_p(big_real(x)) != 0));

    case T_BIG_COMPLEX:
      return(make_boolean(sc,
			  (mpfr_inf_p(big_real(g_real_part(sc, list_1(sc, x)))) != 0) ||
			  (mpfr_inf_p(big_real(g_imag_part(sc, list_1(sc, x)))) != 0)));
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->is_infinite_symbol, list_1(sc, x), a_number_string));
    }
}

static bool is_infinite_b_7p(s7_scheme *sc, s7_pointer p) {return(g_is_infinite(sc, set_plist_1(sc, p)) != sc->F);}


/* ---------------------------------------- number? complex? integer? byte? rational? real?  ---------------------------------------- */

static s7_pointer g_is_number(s7_scheme *sc, s7_pointer args)
{
  #define H_is_number "(number? obj) returns #t if obj is a number"
  #define Q_is_number sc->pl_bt
  check_boolean_method(sc, s7_is_number, sc->is_number_symbol, args); /* we need the s7_* versions here for the GMP case */
}

static s7_pointer g_is_integer(s7_scheme *sc, s7_pointer args)
{
  #define H_is_integer "(integer? obj) returns #t if obj is an integer"
  #define Q_is_integer sc->pl_bt
  check_boolean_method(sc, s7_is_integer, sc->is_integer_symbol, args);
}

static bool is_byte(s7_pointer p) {return((s7_is_integer(p)) && (s7_integer(p) >= 0) && (s7_integer(p) < 256));}
static s7_pointer g_is_byte(s7_scheme *sc, s7_pointer args)
{
  #define H_is_byte "(byte? obj) returns #t if obj is a byte (an integer between 0 and 255)"
  #define Q_is_byte sc->pl_bt
  check_boolean_method(sc, is_byte, sc->is_byte_symbol, args);
}

static s7_pointer g_is_real(s7_scheme *sc, s7_pointer args)
{
  #define H_is_real "(real? obj) returns #t if obj is a real number"
  #define Q_is_real sc->pl_bt
  check_boolean_method(sc, s7_is_real, sc->is_real_symbol, args);
}

static s7_pointer g_is_complex(s7_scheme *sc, s7_pointer args)
{
  #define H_is_complex "(complex? obj) returns #t if obj is a number"
  #define Q_is_complex sc->pl_bt
  check_boolean_method(sc, s7_is_number, sc->is_complex_symbol, args);
}

static s7_pointer g_is_rational(s7_scheme *sc, s7_pointer args)
{
  #define H_is_rational "(rational? obj) returns #t if obj is a rational number (either an integer or a ratio)"
  #define Q_is_rational sc->pl_bt
  check_boolean_method(sc, s7_is_rational, sc->is_rational_symbol, args);
  /* in the non-gmp case, (rational? 455702434782048082459/86885567283849955830) -> #f, not #t
   *  and similarly for exact? etc.
   */
}

static s7_pointer g_is_float(s7_scheme *sc, s7_pointer args)
{
  #define H_is_float "(float? x) returns #t is x is real and not rational."
  #define Q_is_float sc->pl_bt
  s7_pointer p;
  p = car(args);
  return(make_boolean(sc, is_float(p)));
}

static bool is_float_b(s7_pointer p) {return(is_float(p));}


/* ---------------------------------------- even? odd?---------------------------------------- */

static s7_pointer g_is_even(s7_scheme *sc, s7_pointer args)
{
  #define H_is_even "(even? int) returns #t if the integer int32_t is even"
  #define Q_is_even s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_integer_symbol)

  s7_pointer p;
  p = car(args);
  switch (type(p))
    {
    case T_INTEGER:     return(make_boolean(sc, ((integer(p) & 1) == 0)));
#if WITH_GMP
    case T_BIG_INTEGER: return(make_boolean(sc, mpz_even_p(big_integer(p))));
#endif
    default:            return(method_or_bust_one_arg(sc, p, sc->is_even_symbol, list_1(sc, p), T_INTEGER));
    }
}

static bool is_even_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (!s7_is_integer(p))
    simple_wrong_type_argument(sc, sc->is_even_symbol, p, T_INTEGER);
  return((integer(p) & 1) == 0);
}

static bool is_even_i(s7_int i1) {return((i1 & 1) == 0);}

static s7_pointer g_is_odd(s7_scheme *sc, s7_pointer args)
{
  #define H_is_odd "(odd? int) returns #t if the integer int32_t is odd"
  #define Q_is_odd s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_integer_symbol)

  s7_pointer p;
  p = car(args);
  switch (type(p))
    {
    case T_INTEGER:     return(make_boolean(sc, ((integer(p) & 1) == 1)));
#if WITH_GMP
    case T_BIG_INTEGER: return(make_boolean(sc, mpz_odd_p(big_integer(p))));
#endif
    default:            return(method_or_bust_one_arg(sc, p, sc->is_odd_symbol, list_1(sc, p), T_INTEGER));
    }
}

static bool is_odd_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (!s7_is_integer(p))
    simple_wrong_type_argument(sc, sc->is_odd_symbol, p, T_INTEGER);
  return((integer(p) & 1) == 1);
}

static bool is_odd_i(s7_int i1) {return((i1 & 1) == 1);}


/* ---------------------------------------- zero? ---------------------------------------- */

static s7_pointer g_is_zero(s7_scheme *sc, s7_pointer args)
{
  #define H_is_zero "(zero? num) returns #t if the number num is zero"
  #define Q_is_zero sc->pl_bn
  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:     return(make_boolean(sc, integer(x) == 0));
    case T_REAL:        return(make_boolean(sc, real(x) == 0.0));
    case T_RATIO:
    case T_COMPLEX:     return(sc->F);      /* ratios and complex numbers are already collapsed into integers and reals */
#if WITH_GMP
    case T_BIG_INTEGER: return(make_boolean(sc, mpz_cmp_ui(big_integer(x), 0) == 0));
    case T_BIG_REAL:    return(make_boolean(sc, mpfr_zero_p(big_real(x))));
    case T_BIG_RATIO:
    case T_BIG_COMPLEX: return(sc->F);
#endif
    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->is_zero_symbol, list_1(sc, x), a_number_string));
    }
}

static bool is_zero_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_number(p))
    simple_wrong_type_argument_with_type(sc, sc->is_zero_symbol, p, a_number_string);
  if (is_t_integer(p))
    return(integer(p) == 0);
  if (is_t_real(p))
    return(real(p) == 0.0);
  return(false);
}

static bool is_zero_i(s7_int p) {return(p == 0);}
static bool is_zero_d(s7_double p) {return(p == 0.0);}


/* -------------------------------- positive? -------------------------------- */

static s7_pointer g_is_positive(s7_scheme *sc, s7_pointer args)
{
  #define H_is_positive "(positive? num) returns #t if the real number num is positive (greater than 0)"
  #define Q_is_positive s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_real_symbol)
  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:     return(make_boolean(sc, integer(x) > 0));
    case T_RATIO:       return(make_boolean(sc, numerator(x) > 0));
    case T_REAL:        return(make_boolean(sc, real(x) > 0.0));
#if WITH_GMP
    case T_BIG_INTEGER: return(make_boolean(sc, (mpz_cmp_ui(big_integer(x), 0) > 0)));
    case T_BIG_RATIO:   return(make_boolean(sc, (mpq_cmp_ui(big_ratio(x), 0, 1) > 0)));
    case T_BIG_REAL:    return(make_boolean(sc, (mpfr_cmp_ui(big_real(x), 0) > 0)));
#endif
    default:
      return(method_or_bust_one_arg(sc, x, sc->is_positive_symbol, list_1(sc, x), T_REAL));
    }
}

static bool is_positive_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_real(p))
    simple_wrong_type_argument(sc, sc->is_positive_symbol, p, T_REAL);
  if (is_t_integer(p))
    return(integer(p) > 0);
  if (is_t_real(p))
    return(real(p) > 0.0);
  return(numerator(p) > 0);
}

static bool is_positive_i(s7_int p) {return(p > 0);}
static bool is_positive_d(s7_double p) {return(p > 0.0);}


/* -------------------------------- negative? -------------------------------- */

static s7_pointer g_is_negative(s7_scheme *sc, s7_pointer args)
{
  #define H_is_negative "(negative? num) returns #t if the real number num is negative (less than 0)"
  #define Q_is_negative s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_real_symbol)
  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:     return(make_boolean(sc, integer(x) < 0));
    case T_RATIO:       return(make_boolean(sc, numerator(x) < 0));
    case T_REAL:        return(make_boolean(sc, real(x) < 0.0));
#if WITH_GMP
    case T_BIG_INTEGER: return(make_boolean(sc, (mpz_cmp_ui(big_integer(x), 0) < 0)));
    case T_BIG_RATIO:   return(make_boolean(sc, (mpq_cmp_ui(big_ratio(x), 0, 1) < 0)));
    case T_BIG_REAL:    return(make_boolean(sc, (mpfr_cmp_ui(big_real(x), 0) < 0)));
#endif
    default:
      return(method_or_bust_one_arg(sc, x, sc->is_negative_symbol, list_1(sc, x), T_REAL));
    }
}

static bool is_negative_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_real(p))
    simple_wrong_type_argument(sc, sc->is_negative_symbol, p, T_REAL);
  if (is_t_integer(p))
    return(integer(p) < 0);
  if (is_t_real(p))
    return(real(p) < 0.0);
  return(numerator(p) < 0);
}

static bool is_negative_i(s7_int p) {return(p < 0);}
static bool is_negative_d(s7_double p) {return(p < 0.0);}


#if (!WITH_PURE_S7)
#if (!WITH_GMP)
/* ---------------------------------------- exact<->inexact exact? inexact? ---------------------------------------- */

static s7_pointer g_exact_to_inexact(s7_scheme *sc, s7_pointer args)
{
  #define H_exact_to_inexact "(exact->inexact num) converts num to an inexact number; (exact->inexact 3/2) = 1.5"
  #define Q_exact_to_inexact s7_make_signature(sc, 2, sc->is_number_symbol, sc->is_number_symbol)
  /* arg can be complex -> itself! */
  return(exact_to_inexact(sc, car(args)));
}

static s7_pointer g_inexact_to_exact(s7_scheme *sc, s7_pointer args)
{
  #define H_inexact_to_exact "(inexact->exact num) converts num to an exact number; (inexact->exact 1.5) = 3/2"
  #define Q_inexact_to_exact s7_make_signature(sc, 2, sc->is_real_symbol, sc->is_real_symbol)
  return(inexact_to_exact(sc, car(args), WITH_OVERFLOW_ERROR));
}
#endif
/* (!WITH_GMP) */


static s7_pointer g_is_exact(s7_scheme *sc, s7_pointer args)
{
  #define H_is_exact "(exact? num) returns #t if num is exact (an integer or a ratio)"
  #define Q_is_exact sc->pl_bn

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
    case T_RATIO:       return(sc->T);
    case T_REAL:
    case T_COMPLEX:     return(sc->F);
#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:   return(sc->T);
    case T_BIG_REAL:
    case T_BIG_COMPLEX: return(sc->F);
#endif
    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->is_exact_symbol, args, a_number_string));
    }
}

static bool is_exact_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_number(p))
    simple_wrong_type_argument_with_type(sc, sc->is_exact_symbol, p, a_number_string);
  return(is_rational(p));
}


static s7_pointer g_is_inexact(s7_scheme *sc, s7_pointer args)
{
  #define H_is_inexact "(inexact? num) returns #t if num is inexact (neither an integer nor a ratio)"
  #define Q_is_inexact sc->pl_bn

  s7_pointer x;
  x = car(args);
  switch (type(x))
    {
    case T_INTEGER:
    case T_RATIO:       return(sc->F);
    case T_REAL:
    case T_COMPLEX:     return(sc->T);
#if WITH_GMP
    case T_BIG_INTEGER:
    case T_BIG_RATIO:   return(sc->F);
    case T_BIG_REAL:
    case T_BIG_COMPLEX: return(sc->T);
#endif
    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->is_inexact_symbol, args, a_number_string));
    }
}

static bool is_inexact_b_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_number(p))
    simple_wrong_type_argument_with_type(sc, sc->is_inexact_symbol, p, a_number_string);
  return(!is_rational(p));
}


/* ---------------------------------------- integer-length, integer-decode-float ---------------------------------------- */

static int32_t integer_length(s7_int a)
{
  #define I_8 256LL
  #define I_16 65536LL
  #define I_24 16777216LL
  #define I_32 4294967296LL
  #define I_40 1099511627776LL
  #define I_48 281474976710656LL
  #define I_56 72057594037927936LL

  /* a might be most-negative-fixnum! in Clisp: (integer-length -9223372036854775808) -> 63
   */
  if (a < 0)
    {
      if (a == s7_int_min) return(63);
      a = -a;
    }
  if (a < I_8) return(bits[a]);
  if (a < I_16) return(8 + bits[a >> 8]);
  if (a < I_24) return(16 + bits[a >> 16]);
  if (a < I_32) return(24 + bits[a >> 24]);
  if (a < I_40) return(32 + bits[a >> 32]);
  if (a < I_48) return(40 + bits[a >> 40]);
  if (a < I_56) return(48 + bits[a >> 48]);
  return(56 + bits[a >> 56]);
}

static s7_pointer g_integer_length(s7_scheme *sc, s7_pointer args)
{
  #define H_integer_length "(integer-length arg) returns the number of bits required to represent the integer 'arg': (ceiling (log (abs arg) 2))"
  #define Q_integer_length sc->pcl_i

  s7_int x;
  s7_pointer p;

  p = car(args);
  if (!s7_is_integer(p))
    return(method_or_bust_one_arg(sc, p, sc->integer_length_symbol, args, T_INTEGER));


  x = s7_integer(p);
  if (x < 0)
    return(make_integer(sc, integer_length(-(x + 1))));
  return(make_integer(sc, integer_length(x)));
}

static s7_int integer_length_i_i(s7_int x)
{
  if (x < 0)
    return(integer_length(-(x + 1)));
  return(integer_length(x));
}
#endif /* !pure s7 */


static s7_pointer g_integer_decode_float(s7_scheme *sc, s7_pointer args)
{
  #define H_integer_decode_float "(integer-decode-float x) returns a list containing the significand, exponent, and \
sign of 'x' (1 = positive, -1 = negative).  (integer-decode-float 0.0): (0 0 1)"
  #define Q_integer_decode_float s7_make_signature(sc, 2, sc->is_pair_symbol, sc->is_float_symbol)

  /* no matter what s7_double is, integer-decode-float acts as if x is a C double */

  typedef union {
    int64_t ix;
    double fx;
  } decode_float_t;

  decode_float_t num;
  s7_pointer x;
  x = car(args);

  switch (type(x))
    {
    case T_REAL:
      num.fx = (double)real(x);
      break;

#if WITH_GMP
    case T_BIG_REAL:
      num.fx = (double)s7_real(x);
      break;
#endif

    default:
      return(method_or_bust_with_type_one_arg(sc, x, sc->integer_decode_float_symbol, args, wrap_string(sc, "a non-rational real", 19)));
    }

  if (num.fx == 0.0)
    return(list_3(sc, small_int(0), small_int(0), small_int(1)));

  return(list_3(sc,
		make_integer(sc, (s7_int)((num.ix & 0xfffffffffffffLL) | 0x10000000000000LL)),
		make_integer(sc, (s7_int)(((num.ix & 0x7fffffffffffffffLL) >> 52) - 1023 - 52)),
		make_integer(sc, ((num.ix & 0x8000000000000000LL) != 0) ? -1 : 1)));
}


/* -------------------------------- logior -------------------------------- */
static s7_pointer g_logior(s7_scheme *sc, s7_pointer args)
{
  #define H_logior "(logior int32_t ...) returns the OR of its integer arguments (the bits that are on in any of the arguments)"
  #define Q_logior sc->pcl_i
  s7_int result = 0;
  s7_pointer x;

  for (x = args; is_not_null(x); x = cdr(x))
    {
      if (!s7_is_integer(car(x)))
	return(method_or_bust(sc, car(x), sc->logior_symbol, cons(sc, make_integer(sc, result), x), T_INTEGER, position_of(x, args)));
      result |= s7_integer(car(x));
    }
  return(make_integer(sc, result));
}

static s7_int logior_i_ii(s7_int i1, s7_int i2) {return(i1 | i2);}
static s7_int logior_i_iii(s7_int i1, s7_int i2, s7_int i3) {return(i1 | i2 | i3);}


/* -------------------------------- logxor -------------------------------- */
static s7_pointer g_logxor(s7_scheme *sc, s7_pointer args)
{
  #define H_logxor "(logxor int32_t ...) returns the XOR of its integer arguments (the bits that are on in an odd number of the arguments)"
  #define Q_logxor sc->pcl_i
  s7_int result = 0;
  s7_pointer x;

  for (x = args; is_not_null(x); x = cdr(x))
    {
      if (!s7_is_integer(car(x)))
	return(method_or_bust(sc, car(x), sc->logxor_symbol, cons(sc, make_integer(sc, result), x), T_INTEGER, position_of(x, args)));
      result ^= s7_integer(car(x));
    }
  return(make_integer(sc, result));
}

static s7_int logxor_i_ii(s7_int i1, s7_int i2) {return(i1 ^ i2);}
static s7_int logxor_i_iii(s7_int i1, s7_int i2, s7_int i3) {return(i1 ^ i2 ^ i3);}


/* -------------------------------- logand -------------------------------- */
static s7_pointer g_logand(s7_scheme *sc, s7_pointer args)
{
  #define H_logand "(logand int32_t ...) returns the AND of its integer arguments (the bits that are on in every argument)"
  #define Q_logand sc->pcl_i
  s7_int result = -1;
  s7_pointer x;

  for (x = args; is_not_null(x); x = cdr(x))
    {
      if (!s7_is_integer(car(x)))
	return(method_or_bust(sc, car(x), sc->logand_symbol, cons(sc, make_integer(sc, result), x), T_INTEGER, position_of(x, args)));
      result &= s7_integer(car(x));
    }
  return(make_integer(sc, result));
}

static s7_int logand_i_ii(s7_int i1, s7_int i2) {return(i1 & i2);}
static s7_int logand_i_iii(s7_int i1, s7_int i2, s7_int i3) {return(i1 & i2 & i3);}


/* -------------------------------- lognot -------------------------------- */

static s7_pointer g_lognot(s7_scheme *sc, s7_pointer args)
{
  #define H_lognot "(lognot num) returns the negation of num (its complement, the bits that are not on): (lognot 0) -> -1"
  #define Q_lognot sc->pcl_i
  if (!s7_is_integer(car(args)))
    return(method_or_bust_one_arg(sc, car(args), sc->lognot_symbol, args, T_INTEGER));
  return(make_integer(sc, ~s7_integer(car(args))));
}

static s7_int lognot_i_i(s7_int i1) {return(~i1);}


/* -------------------------------- logbit? -------------------------------- */
/* logbit?  CL is (logbitp index int) using 2^index, but that order strikes me as backwards
 *   at least gmp got the arg order right!
 */

static s7_pointer g_logbit(s7_scheme *sc, s7_pointer args)
{
  #define H_logbit "(logbit? int32_t index) returns #t if the index-th bit is on in int, otherwise #f. The argument \
order here follows gmp, and is the opposite of the CL convention.  (logbit? int32_t bit) is the same as (not (zero? (logand int32_t (ash 1 bit))))."
  #define Q_logbit s7_make_circular_signature(sc, 1, 2, sc->is_boolean_symbol, sc->is_integer_symbol)

  s7_pointer x, y;
  s7_int index;      /* index in gmp is mp_bitcnt which is an unsigned long int */

  x = car(args);
  y = cadr(args);

  if (!s7_is_integer(x))
    return(method_or_bust(sc, x, sc->logbit_symbol, args, T_INTEGER, 1));
  if (!s7_is_integer(y))
    return(method_or_bust(sc, y, sc->logbit_symbol, args, T_INTEGER, 2));

  index = s7_integer(y);
  if (index < 0)
    return(out_of_range(sc, sc->logbit_symbol, small_int(2), y, its_negative_string));

#if WITH_GMP
  if (is_t_big_integer(x))
    return(make_boolean(sc, (mpz_tstbit(big_integer(x), index) != 0)));
#endif

  if (index >= s7_int_bits)           /* not sure about the >: (logbit? -1 64) ?? */
    return(make_boolean(sc, integer(x) < 0));

  /* :(zero? (logand most-positive-fixnum (ash 1 63)))
   *   -> ash argument 2, 63, is out of range (shift is too large)
   *   so logbit? has a wider range than the logand/ash shuffle above.
   */

  /* all these int64_ts are necessary, else C turns it into an int, gets confused about signs etc */
  return(make_boolean(sc, ((((int64_t)(1LL << (int64_t)index)) & (int64_t)integer(x)) != 0)));
}

static bool logbit_b_ii(s7_int i1, s7_int i2)
{
  if (i2 < 0)
    return(false); /* no b_7ii_t apparently */
  if (i2 >= s7_int_bits)
    return(i1 < 0);
  return((((int64_t)(1LL << (int64_t)i2)) & (int64_t)i1) != 0);
}

#if (!WITH_GMP)
static bool logbit_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  if (is_integer(p1))
    {
      if (is_integer(p2))
	return(logbit_b_ii(integer(p1), integer(p2)));
      simple_wrong_type_argument(sc, sc->logbit_symbol, p2, T_INTEGER);
    }
  simple_wrong_type_argument(sc, sc->logbit_symbol, p1, T_INTEGER);
  return(false);
}
#endif


/* -------------------------------- ash -------------------------------- */
static s7_int c_ash(s7_scheme *sc, s7_int arg1, s7_int arg2)
{
  if (arg1 == 0) return(0);

  if (arg2 >= s7_int_bits)
    {
      if ((arg1 == -1) && (arg2 == 63))   /* (ash -1 63): most-negative-fixnum */
	return(s7_int_min);
      out_of_range(sc, sc->ash_symbol, small_int(2), wrap_integer1(sc, arg2), its_too_large_string);
    }

  if (arg2 < -s7_int_bits)
    {
      if (arg1 < 0)                      /* (ash -31 -100) */
	return(-1);
      return(0);
    }

  /* I can't see any point in protecting this: (ash 9223372036854775807 1) -> -2, but anyone using ash must know something about bits */
  if (arg2 >= 0)
    {
      if (arg1 < 0)
	{
	  uint64_t z;
	  z = (uint64_t)arg1;
	  return((s7_int)(z << arg2));
	}
      return(arg1 << arg2);
    }
  return(arg1 >> -arg2);
}

static s7_pointer g_ash(s7_scheme *sc, s7_pointer args)
{
  #define H_ash "(ash i1 i2) returns i1 shifted right or left i2 times, i1 << i2, (ash 1 3) -> 8, (ash 8 -3) -> 1"
  #define Q_ash sc->pcl_i
  s7_pointer x, y;

  x = car(args);
  if (!s7_is_integer(x))
    return(method_or_bust(sc, x, sc->ash_symbol, args, T_INTEGER, 1));

  y = cadr(args);
  if (!s7_is_integer(y))
    return(method_or_bust(sc, y, sc->ash_symbol, args, T_INTEGER, 2));

  return(make_integer(sc, c_ash(sc, s7_integer(x), s7_integer(y))));
}

static s7_int ash_i_7ii(s7_scheme *sc, s7_int i1, s7_int i2) {return(c_ash(sc, i1, i2));}
#if (!WITH_GMP)
static s7_int rsh_i_ii_direct(s7_int i1, s7_int i2) {return(i1 >> (-i2));}
static s7_int lsh_i_ii_direct(s7_int i1, s7_int i2) {return(i1 << i2);}
static s7_int rsh_i_i2_direct(s7_int i1, s7_int i2) {return(i1 >> 1);}
#endif


/* ---------------------------------------- random ---------------------------------------- */

/* random numbers.  The simple version used in clm.c is probably adequate,
 *   but here I'll use Marsaglia's MWC algorithm.
 *     (random num) -> a number (0..num), if num == 0 return 0, use global default state
 *     (random num state) -> same but use this state
 *     (random-state seed) -> make a new state
 *   to save the current seed, use copy
 *   to save it across load, random-state->list and list->random-state.
 *   random-state? returns #t if its arg is one of these guys
 */

#if (!WITH_GMP)
s7_pointer s7_random_state(s7_scheme *sc, s7_pointer args)
{
  #define H_random_state "(random-state seed (carry plausible-default)) returns a new random number state initialized with 'seed'. \
Pass this as the second argument to 'random' to get a repeatable random number sequence:\n\
    (let ((seed (random-state 1234))) (random 1.0 seed))"
  #define Q_random_state s7_make_circular_signature(sc, 1, 2, sc->is_random_state_symbol, sc->is_integer_symbol)

  s7_pointer r1, r2, p;
  s7_int i1, i2;

  r1 = car(args);
  if (!s7_is_integer(r1))
    return(method_or_bust(sc, r1, sc->random_state_symbol, args, T_INTEGER, 1));
  i1 = integer(r1);
  if (i1 < 0)
    return(out_of_range(sc, sc->random_state_symbol, small_int(1), r1, its_negative_string));

  if (is_null(cdr(args)))
    {
      new_cell(sc, p, T_RANDOM_STATE);
      random_seed(p) = (uint64_t)i1;
      random_carry(p) = 1675393560;                          /* should this be dependent on the seed? */
      return(p);
    }

  r2 = cadr(args);
  if (!s7_is_integer(r2))
    return(method_or_bust(sc, r2, sc->random_state_symbol, args, T_INTEGER, 2));
  i2 = integer(r2);
  if (i2 < 0)
    return(out_of_range(sc, sc->random_state_symbol, small_int(2), r2, its_negative_string));

  new_cell(sc, p, T_RANDOM_STATE);
  random_seed(p) = (uint64_t)i1;
  random_carry(p) = (uint64_t)i2;
  return(p);
}

#define g_random_state s7_random_state
#endif

static s7_pointer rng_copy(s7_scheme *sc, s7_pointer args)
{
#if WITH_GMP
  return(sc->F); /* I can't find a way to copy a gmp random generator */
#else
  s7_pointer obj;
  obj = car(args);
  if (is_random_state(obj))
    {
      s7_pointer new_r;
      new_cell(sc, new_r, T_RANDOM_STATE);
      random_seed(new_r) = random_seed(obj);
      random_carry(new_r) = random_carry(obj);
      return(new_r);
    }
  return(sc->F);
#endif
}


static s7_pointer g_is_random_state(s7_scheme *sc, s7_pointer args)
{
  #define H_is_random_state "(random-state? obj) returns #t if obj is a random-state object (from random-state)."
  #define Q_is_random_state sc->pl_bt
  check_boolean_method(sc, is_random_state, sc->is_random_state_symbol, args);
}

static bool is_random_state_b(s7_pointer p) {return(type(p) == T_RANDOM_STATE);}

s7_pointer s7_random_state_to_list(s7_scheme *sc, s7_pointer args)
{
  #define H_random_state_to_list "(random-state->list r) returns the random state object as a list.\
You can later apply random-state to this list to continue a random number sequence from any point."
  #define Q_random_state_to_list s7_make_signature(sc, 2, sc->is_pair_symbol, sc->is_random_state_symbol)

#if WITH_GMP
  if ((is_pair(args)) &&
      (!is_random_state(car(args))))
    return(method_or_bust_with_type(sc, car(args), sc->random_state_to_list_symbol, args, a_random_state_object_string, 1));
  return(sc->nil);
#else
  s7_pointer r;
  if (is_null(args))
    r = sc->default_rng;
  else
    {
      r = car(args);
      if (!is_random_state(r))
	return(method_or_bust_with_type(sc, r, sc->random_state_to_list_symbol, args, a_random_state_object_string, 1));
    }
  return(list_2(sc, make_integer(sc, random_seed(r)), make_integer(sc, random_carry(r))));
#endif
}

#define g_random_state_to_list s7_random_state_to_list

void s7_set_default_random_state(s7_scheme *sc, s7_int seed, s7_int carry)
{
#if (!WITH_GMP)
  s7_pointer p;
  new_cell(sc, p, T_RANDOM_STATE);
  random_seed(p) = (uint64_t)seed;
  random_carry(p) = (uint64_t)carry;
  sc->default_rng = p;
#endif
}

#if (!WITH_GMP)
/* -------------------------------- random -------------------------------- */

static double next_random(s7_pointer r)
{
  /* The multiply-with-carry generator for 32-bit integers:
   *        x(n)=a*x(n-1) + carry mod 2^32
   * Choose multiplier a from this list:
   *   1791398085 1929682203 1683268614 1965537969 1675393560
   *   1967773755 1517746329 1447497129 1655692410 1606218150
   *   2051013963 1075433238 1557985959 1781943330 1893513180
   *   1631296680 2131995753 2083801278 1873196400 1554115554
   * ( or any 'a' for which both a*2^32-1 and a*2^31-1 are prime)
   */
  double result;
  uint64_t temp;
  #define RAN_MULT 2131995753UL

  temp = random_seed(r) * RAN_MULT + random_carry(r);
  random_seed(r) = (temp & 0xffffffffUL);
  random_carry(r) = (temp >> 32);
  result = (double)((uint32_t)(random_seed(r))) / 4294967295.5;
  /* divisor was 2^32-1 = 4294967295.0, but somehow this can round up once in a billion tries?
   *   do we want the double just less than 2^32?
   */

  /* (let ((mx 0) (mn 1000)) (do ((i 0 (+ i 1))) ((= i 10000)) (let ((val (random 123))) (set! mx (max mx val)) (set! mn (min mn val)))) (list mn mx)) */
  return(result);
}


s7_double s7_random(s7_scheme *sc, s7_pointer state)
{
  if (!state)
    return(next_random(sc->default_rng));
  return(next_random(state));
}


static s7_pointer g_random(s7_scheme *sc, s7_pointer args)
{
  #define H_random "(random num (state #f)) returns a random number between 0 and num (0 if num=0)."
  #define Q_random s7_make_signature(sc, 3, sc->is_number_symbol, sc->is_number_symbol, sc->is_random_state_symbol)
  s7_pointer r, num;

  if (is_not_null(cdr(args)))
    {
      r = cadr(args);
      if (!is_random_state(r))
	return(method_or_bust_with_type(sc, r, sc->random_symbol, args, a_random_state_object_string, 2));
    }
  else r = sc->default_rng;
  num = car(args);

  switch (type(num))
    {
    case T_INTEGER:
      return(make_integer(sc, (s7_int)(integer(num) * next_random(r))));

    case T_RATIO:
      {
	s7_double x, error;
	s7_int numer = 0, denom = 1;
	/* the error here needs to take the size of the fraction into account.  Otherwise, if
	 *    error is (say) 1e-6 and the fraction is (say) 9000000/9223372036854775807,
	 *    c_rationalize will always return 0.  But even that isn't foolproof:
	 *    (random 1/562949953421312) -> 1/376367230475000
	 */
	x = fraction(num);
	if ((x < 1.0e-10) && (x > -1.0e-10))
	  {
	    /* 1e-12 is not tight enough:
	     *    (random 1/2251799813685248) -> 1/2250240579436280
	     *    (random -1/4503599627370496) -> -1/4492889778435526
	     *    (random 1/140737488355328) -> 1/140730223985746
	     *    (random -1/35184372088832) -> -1/35183145492420
	     *    (random -1/70368744177664) -> -1/70366866392738
	     *    (random 1/4398046511104) -> 1/4398033095756
	     *    (random 1/137438953472) -> 1/137438941127
	     */
	    if (numerator(num) < -10)
	      numer = -(s7_int)(floor(-numerator(num) * next_random(r)));
	    else
	      {
		if (numerator(num) > 10)
		  numer = (s7_int)floor(numerator(num) * next_random(r));
		else
		  {
		    int64_t diff;
		    numer = numerator(num);
		    diff = s7_int_max - denominator(num);
		    if (diff < 100)
		      return(s7_make_ratio(sc, numer, denominator(num)));
		    denom = denominator(num) + (s7_int)floor(diff * next_random(r));
		    return(s7_make_ratio(sc, numer, denom));
		  }
	      }
	    return(s7_make_ratio(sc, numer, denominator(num)));
	  }
	if ((x < 1e-6) && (x > -1e-6))
	  error = 1e-18;
	else error = 1e-12;
	c_rationalize(x * next_random(r), error, &numer, &denom);
	return(s7_make_ratio(sc, numer, denom));
      }

    case T_REAL:
      return(make_real(sc, real(num) * next_random(r)));

    case T_COMPLEX:
      return(s7_make_complex(sc, real_part(num) * next_random(r), imag_part(num) * next_random(r)));

    default:
      return(method_or_bust_with_type(sc, num, sc->random_symbol, args, a_number_string, 1));
    }
  return(sc->F);
}

static s7_double random_d_7d(s7_scheme *sc, s7_double x)
{
  return(x * next_random(sc->default_rng));
}

static s7_int random_i_7i(s7_scheme *sc, s7_int i)
{
  return((s7_int)(i * next_random(sc->default_rng)));
}

static s7_pointer random_p_p(s7_scheme *sc, s7_pointer p)
{
  return(g_random(sc, set_plist_1(sc, p)));
}

static s7_pointer g_random_ic(s7_scheme *sc, s7_pointer args)
{
  return(make_integer(sc, (s7_int)(integer(car(args)) * next_random(sc->default_rng))));
}

static s7_pointer g_random_rc(s7_scheme *sc, s7_pointer args)
{
  return(make_real(sc, real(car(args)) * next_random(sc->default_rng)));
}

static s7_pointer g_random_1(s7_scheme *sc, s7_pointer args)
{
  s7_pointer r, num;
  num = car(args);
  r = sc->default_rng;
  if (is_integer(num))
    return(make_integer(sc, (s7_int)(integer(num) * next_random(r))));
  if (is_t_real(num))
    return(make_real(sc, real(num) * next_random(r)));
  return(g_random(sc, args));
}

static s7_pointer random_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  if (!ops) return(f);
  if (args == 1)
    {
      s7_pointer arg1;
      arg1 = cadr(expr);
      if (s7_is_integer(arg1))
	{
	  set_optimize_op(expr, HOP_SAFE_C_D);
	  return(sc->random_ic);
	}
      if (is_float(arg1))
	{
	  set_optimize_op(expr, HOP_SAFE_C_D);
	  return(sc->random_rc);
	}
      return(sc->random_1);
    }
  return(f);
}
#endif /* gmp */


/* -------------------------------- characters -------------------------------- */

#define NUM_CHARS 256

static s7_pointer g_char_to_integer(s7_scheme *sc, s7_pointer args)
{
  #define H_char_to_integer "(char->integer c) converts the character c to an integer"
  #define Q_char_to_integer s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_char_symbol)

  if (!s7_is_character(car(args)))
    return(method_or_bust_one_arg(sc, car(args), sc->char_to_integer_symbol, args, T_CHARACTER));
  return(small_int(character(car(args))));
}

static s7_int char_to_integer_i_7p(s7_scheme *sc, s7_pointer p)
{
  if (!s7_is_character(p))
    simple_wrong_type_argument(sc, sc->char_to_integer_symbol, p, T_CHARACTER);
  return(character(p));
}


static s7_pointer g_integer_to_char(s7_scheme *sc, s7_pointer args)
{
  #define H_integer_to_char "(integer->char i) converts the non-negative integer i to a character"
  #define Q_integer_to_char s7_make_signature(sc, 2, sc->is_char_symbol, sc->is_integer_symbol)

  s7_pointer x;
  s7_int ind;

  x = car(args);
  if (!s7_is_integer(x))
    return(method_or_bust_one_arg(sc, x, sc->integer_to_char_symbol, list_1(sc, x), T_INTEGER));
  ind = s7_integer(x);
  if ((ind < 0) || (ind >= NUM_CHARS))
    return(s7_out_of_range_error(sc, "integer->char", 1, x, "it doen't fit in an unsigned byte"));
  return(s7_make_character(sc, (uint8_t)ind));
}

static s7_pointer integer_to_char_p_p(s7_scheme *sc, s7_pointer x)
{
  s7_int ind;
  if (!s7_is_integer(x))
    simple_wrong_type_argument(sc, sc->integer_to_char_symbol, x, T_INTEGER);
  ind = s7_integer(x);
  if ((ind >= 0) && (ind < NUM_CHARS))
    return(s7_make_character(sc, (uint8_t)ind));
  return(s7_out_of_range_error(sc, "integer->char", 1, x, "it doen't fit in an unsigned byte"));
}


static uint8_t uppers[256], lowers[256];
static void init_uppers(void)
{
  int32_t i;
  for (i = 0; i < 256; i++)
    {
      uppers[i] = (uint8_t)toupper(i);
      lowers[i] = (uint8_t)tolower(i);
    }
}

static void init_chars(void)
{
  s7_cell *cells;
  int32_t i;

  chars = (s7_pointer *)malloc((NUM_CHARS + 1) * sizeof(s7_pointer)); /* chars is declared far above */
  cells = (s7_cell *)calloc(NUM_CHARS + 1, sizeof(s7_cell));

  chars[0] = &cells[0];
  eof_object = chars[0];
  set_type(eof_object, T_EOF_OBJECT | T_IMMUTABLE | T_UNHEAP);
  unique_name_length(eof_object) = 6;
  unique_name(eof_object) = "#<eof>";
  chars++;                    /* now chars[EOF] == chars[-1] == eof_object */
  cells++;

  for (i = 0; i < NUM_CHARS; i++)
    {
      s7_pointer cp;
      uint8_t c;

      c = (uint8_t)i;
      cp = &cells[i];
      set_type_bit(cp, T_IMMUTABLE | T_CHARACTER | T_UNHEAP);
      character(cp) = c;
      upper_character(cp) = (uint8_t)toupper(i);
      is_char_alphabetic(cp) = (bool)isalpha(i);
      is_char_numeric(cp) = (bool)isdigit(i);
      is_char_whitespace(cp) = white_space[i];
      is_char_uppercase(cp) = (((bool)isupper(i)) || ((i >= 192) && (i < 208)));
      is_char_lowercase(cp) = (bool)islower(i);
      chars[i] = cp;

      #define make_character_name(S) memcpy((void *)(&(character_name(cp))), (const void *)(S), character_name_length(cp) = strlen(S))
      switch (c)
	{
	case ' ':	 make_character_name("#\\space");     break;
	case '\n':       make_character_name("#\\newline");   break;
	case '\r':       make_character_name("#\\return");    break;
	case '\t':       make_character_name("#\\tab");       break;
	case '\0':       make_character_name("#\\null");      break;
	case (char)0x1b: make_character_name("#\\escape");    break;
	case (char)0x7f: make_character_name("#\\delete");    break;
	case (char)7:    make_character_name("#\\alarm");     break;
	case (char)8:    make_character_name("#\\backspace"); break;
	default:
	  {
            #define P_SIZE 12
	    int32_t len;
	    if ((c < 32) || (c >= 127))
	      len = snprintf((char *)(&(character_name(cp))), P_SIZE, "#\\x%x", c);
	    else len = snprintf((char *)(&(character_name(cp))), P_SIZE, "#\\%c", c);
	    character_name_length(cp) = len;
	    break;
          }
       }
    }
}

/* -------------------------------- char-upcase, char-downcase ----------------------- */
static s7_pointer g_char_upcase(s7_scheme *sc, s7_pointer args)
{
  #define H_char_upcase "(char-upcase c) converts the character c to upper case"
  #define Q_char_upcase sc->pcl_c
  if (!s7_is_character(car(args)))
    return(method_or_bust_one_arg(sc, car(args), sc->char_upcase_symbol, args, T_CHARACTER));
  return(s7_make_character(sc, upper_character(car(args))));
}

static s7_pointer g_char_downcase(s7_scheme *sc, s7_pointer args)
{
  #define H_char_downcase "(char-downcase c) converts the character c to lower case"
  #define Q_char_downcase sc->pcl_c
  if (!s7_is_character(car(args)))
    return(method_or_bust_one_arg(sc, car(args), sc->char_downcase_symbol, args, T_CHARACTER));
  return(s7_make_character(sc, lowers[character(car(args))]));
}

/* -------------------------------- char-alphabetic? char-numeric? char-whitespace? -------------------------------- */
static s7_pointer g_is_char_alphabetic(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char_alphabetic "(char-alphabetic? c) returns #t if the character c is alphabetic"
  #define Q_is_char_alphabetic sc->pl_bc
  if (!s7_is_character(car(args)))
    return(method_or_bust_one_arg(sc, car(args), sc->is_char_alphabetic_symbol, args, T_CHARACTER));
  return(make_boolean(sc, is_char_alphabetic(car(args))));

  /* isalpha returns #t for (integer->char 226) and others in that range */
}

static bool is_char_alphabetic_b_7p(s7_scheme *sc, s7_pointer c)
{
  if (!s7_is_character(c))
    simple_wrong_type_argument(sc, sc->is_char_alphabetic_symbol, c, T_CHARACTER);
  return(is_char_alphabetic(c));
}
static bool is_char_alphabetic_c(s7_pointer c) {return(is_char_alphabetic(c));}


static s7_pointer g_is_char_numeric(s7_scheme *sc, s7_pointer args)
{
  s7_pointer arg;
  #define H_is_char_numeric "(char-numeric? c) returns #t if the character c is a digit"
  #define Q_is_char_numeric sc->pl_bc

  arg = car(args);
  if (!s7_is_character(arg))
    return(method_or_bust_one_arg(sc, arg, sc->is_char_numeric_symbol, args, T_CHARACTER));
  return(make_boolean(sc, is_char_numeric(arg)));
}

static bool is_char_numeric_b_7p(s7_scheme *sc, s7_pointer c)
{
  if (!s7_is_character(c))
    simple_wrong_type_argument(sc, sc->is_char_numeric_symbol, c, T_CHARACTER);
  return(is_char_numeric(c));
}
static bool is_char_numeric_c(s7_pointer c) {return(is_char_numeric(c));}


static s7_pointer g_is_char_whitespace(s7_scheme *sc, s7_pointer args)
{
  s7_pointer arg;
  #define H_is_char_whitespace "(char-whitespace? c) returns #t if the character c is non-printing character"
  #define Q_is_char_whitespace sc->pl_bc

  arg = car(args);
  if (!s7_is_character(arg))
    return(method_or_bust_one_arg(sc, arg, sc->is_char_whitespace_symbol, args, T_CHARACTER));
  return(make_boolean(sc, is_char_whitespace(arg)));
}

static bool is_char_whitespace_b_7p(s7_scheme *sc, s7_pointer c)
{
  if (!s7_is_character(c))
    simple_wrong_type_argument(sc, sc->is_char_whitespace_symbol, c, T_CHARACTER);
  return(is_char_whitespace(c));
}
static bool is_char_whitespace_c(s7_pointer c) {return(is_char_whitespace(c));}

/* -------------------------------- char-upper-case? char-lower-case? -------------------------------- */
static s7_pointer g_is_char_upper_case(s7_scheme *sc, s7_pointer args)
{
  s7_pointer arg;
  #define H_is_char_upper_case "(char-upper-case? c) returns #t if the character c is in upper case"
  #define Q_is_char_upper_case sc->pl_bc

  arg = car(args);
  if (!s7_is_character(arg))
    return(method_or_bust_one_arg(sc, arg, sc->is_char_upper_case_symbol, args, T_CHARACTER));
  return(make_boolean(sc, is_char_uppercase(arg)));
}

static bool is_char_upper_case_b_7p(s7_scheme *sc, s7_pointer c)
{
  if (!s7_is_character(c))
    simple_wrong_type_argument(sc, sc->is_char_upper_case_symbol, c, T_CHARACTER);
  return(is_char_uppercase(c));
}

static bool is_char_upper_case_c(s7_pointer c) {return(is_char_uppercase(c));}

static s7_pointer g_is_char_lower_case(s7_scheme *sc, s7_pointer args)
{
  s7_pointer arg;
  #define H_is_char_lower_case "(char-lower-case? c) returns #t if the character c is in lower case"
  #define Q_is_char_lower_case sc->pl_bc

  arg = car(args);
  if (!s7_is_character(arg))
    return(method_or_bust_one_arg(sc, arg, sc->is_char_lower_case_symbol, args, T_CHARACTER));
  return(make_boolean(sc, is_char_lowercase(arg)));
}

static bool is_char_lower_case_b_7p(s7_scheme *sc, s7_pointer c)
{
  if (!s7_is_character(c))
    simple_wrong_type_argument(sc, sc->is_char_lower_case_symbol, c, T_CHARACTER);
  return(is_char_lowercase(c));
}

static bool is_char_lower_case_c(s7_pointer c) {return(is_char_lowercase(c));}

/* -------------------------------- char? -------------------------------- */
static s7_pointer g_is_char(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char "(char? obj) returns #t if obj is a character"
  #define Q_is_char sc->pl_bt
  check_boolean_method(sc, s7_is_character, sc->is_char_symbol, args);
}


s7_pointer s7_make_character(s7_scheme *sc, uint8_t c)
{
  return(chars[c]);
}

bool s7_is_character(s7_pointer p)
{
  return(type(p) == T_CHARACTER);
}

uint8_t s7_character(s7_pointer p)
{
  return(character(p));
}


/* -------------------------------- char<? char<=? char>? char>=? char=? -------------------------------- */
static int32_t charcmp(uint8_t c1, uint8_t c2)
{
  return((c1 == c2) ? 0 : (c1 < c2) ? -1 : 1);
  /* not tolower here -- the single case is apparently supposed to be upper case
   *   this matters in a case like (char-ci<? #\_ #\e) which Guile and Gauche say is #f
   *   although (char<? #\_ #\e) is #t -- the spec does not say how to interpret this!
   */
}

static bool is_character_via_method(s7_scheme *sc, s7_pointer p)
{
  if (s7_is_character(p))
    return(true);
  if (has_methods(p))
    {
      s7_pointer f;
      f = find_method(sc, find_let(sc, p), sc->is_char_symbol);
      if (f != sc->undefined)
	return(is_true(sc, s7_apply_function(sc, f, cons(sc, p, sc->nil))));
    }
  return(false);
}

static s7_pointer g_char_cmp(s7_scheme *sc, s7_pointer args, int32_t val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!s7_is_character(y))
    return(method_or_bust(sc, y, sym, args, T_CHARACTER, 1));

  for (x = cdr(args); is_pair(x); x = cdr(x))
    {
      if (!s7_is_character(car(x)))
	return(method_or_bust(sc, car(x), sym, cons(sc, y, x), T_CHARACTER, position_of(x, args)));

      if (charcmp(character(y), character(car(x))) != val)
	{
	  for (y = cdr(x); is_pair(y); y = cdr(y)) /* before returning #f, check for bad trailing arguments */
	    if (!is_character_via_method(sc, car(y)))
	      return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_CHARACTER));
	  return(sc->F);
	}
      y = car(x);
    }
  return(sc->T);
}

static s7_pointer g_char_cmp_not(s7_scheme *sc, s7_pointer args, int32_t val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!s7_is_character(y))
    return(method_or_bust(sc, y, sym, args, T_CHARACTER, 1));

  for (x = cdr(args); is_pair(x); x = cdr(x))
    {
      if (!s7_is_character(car(x)))
	return(method_or_bust(sc, car(x), sym, cons(sc, y, x), T_CHARACTER, position_of(x, args)));

      if (charcmp(character(y), character(car(x))) == val)
	{
	  for (y = cdr(x); is_pair(y); y = cdr(y))
	    if (!is_character_via_method(sc, car(y)))
	      return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_CHARACTER));
	  return(sc->F);
	}
      y = car(x);
    }
  return(sc->T);
}

static s7_pointer g_chars_are_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_equal "(char=? char ...) returns #t if all the character arguments are equal"
  #define Q_chars_are_equal sc->pcl_bc

  s7_pointer x, y;

  y = car(args);
  if (!s7_is_character(y))
    return(method_or_bust(sc, y, sc->char_eq_symbol, args, T_CHARACTER, 1));

  for (x = cdr(args); is_pair(x); x = cdr(x))
    {
      if (!s7_is_character(car(x)))
	return(method_or_bust(sc, car(x), sc->char_eq_symbol, cons(sc, y, x), T_CHARACTER, position_of(x, args)));

      if (car(x) != y)
	{
	  for (y = cdr(x); is_pair(y); y = cdr(y))
	    if (!is_character_via_method(sc, car(y)))
	      return(wrong_type_argument(sc, sc->char_eq_symbol, position_of(y, args), car(y), T_CHARACTER));
	  return(sc->F);
	}
    }
  return(sc->T);
}


static s7_pointer g_chars_are_less(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_less "(char<? char ...) returns #t if all the character arguments are increasing"
  #define Q_chars_are_less sc->pcl_bc

  return(g_char_cmp(sc, args, -1, sc->char_lt_symbol));
}

static s7_pointer g_chars_are_greater(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_greater "(char>? char ...) returns #t if all the character arguments are decreasing"
  #define Q_chars_are_greater sc->pcl_bc

  return(g_char_cmp(sc, args, 1, sc->char_gt_symbol));
}

static s7_pointer g_chars_are_geq(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_geq "(char>=? char ...) returns #t if all the character arguments are equal or decreasing"
  #define Q_chars_are_geq sc->pcl_bc

  return(g_char_cmp_not(sc, args, -1, sc->char_geq_symbol));
}

static s7_pointer g_chars_are_leq(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_leq "(char<=? char ...) returns #t if all the character arguments are equal or increasing"
  #define Q_chars_are_leq sc->pcl_bc

  return(g_char_cmp_not(sc, args, 1, sc->char_leq_symbol));
}

static s7_pointer g_simple_char_eq(s7_scheme *sc, s7_pointer args)
{
  return(make_boolean(sc, character(car(args)) == character(cadr(args))));
}


static inline void check_char2_args(s7_scheme *sc, s7_pointer caller, s7_pointer p1, s7_pointer p2)
{
  if (!s7_is_character(p1))
    simple_wrong_type_argument(sc, caller, p1, T_CHARACTER);
  if (!s7_is_character(p2))
    simple_wrong_type_argument(sc, caller, p2, T_CHARACTER);
}

static bool char_lt_b_direct(s7_pointer p1, s7_pointer p2) {return(character(p1) < character(p2));}
static bool char_lt_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_lt_symbol, p1, p2);
  return(character(p1) < character(p2));
}

static bool char_leq_b_direct(s7_pointer p1, s7_pointer p2) {return(character(p1) <= character(p2));}
static bool char_leq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_leq_symbol, p1, p2);
  return(character(p1) <= character(p2));
}

static bool char_gt_b_direct(s7_pointer p1, s7_pointer p2) {return(character(p1) > character(p2));}
static bool char_gt_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_gt_symbol, p1, p2);
  return(character(p1) > character(p2));
}

static bool char_geq_b_direct(s7_pointer p1, s7_pointer p2) {return(character(p1) >= character(p2));}
static bool char_geq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_geq_symbol, p1, p2);
  return(character(p1) >= character(p2));
}

static bool char_eq_b_direct(s7_pointer p1, s7_pointer p2) {return(character(p1) == character(p2));}
static bool char_eq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_eq_symbol, p1, p2);
  return(character(p1) == character(p2));
}

/* -------------------------------- char-ci<? char-ci<=? char-ci>? char-ci>=? char-ci=? -------------------------------- */
static s7_pointer g_char_equal_s_ic(s7_scheme *sc, s7_pointer args)
{
  s7_pointer c;
  c = symbol_to_value_unchecked(sc, car(args));
  if (c == cadr(args))
    return(sc->T);
  if (s7_is_character(c))
    return(sc->F);
  return(method_or_bust(sc, c, sc->char_eq_symbol, list_2(sc, c, cadr(args)), T_CHARACTER, 1));
}

static s7_pointer g_char_equal_2(s7_scheme *sc, s7_pointer args)
{
  if (!s7_is_character(car(args)))
    return(method_or_bust(sc, car(args), sc->char_eq_symbol, args, T_CHARACTER, 1));
  if (car(args) == cadr(args))
    return(sc->T);
  if (!s7_is_character(cadr(args)))
    return(method_or_bust(sc, cadr(args), sc->char_eq_symbol, args, T_CHARACTER, 2));
  return(sc->F);
}

static s7_pointer g_char_less_2(s7_scheme *sc, s7_pointer args)
{
  if (!s7_is_character(car(args)))
    return(method_or_bust(sc, car(args), sc->char_lt_symbol, args, T_CHARACTER, 1));
  if (!s7_is_character(cadr(args)))
    return(method_or_bust(sc, cadr(args), sc->char_lt_symbol, args, T_CHARACTER, 2));
  return(make_boolean(sc, character(car(args)) < character(cadr(args))));
}


static s7_pointer g_char_greater_2(s7_scheme *sc, s7_pointer args)
{
  if (!s7_is_character(car(args)))
    return(method_or_bust(sc, car(args), sc->char_gt_symbol, args, T_CHARACTER, 1));
  if (!s7_is_character(cadr(args)))
    return(method_or_bust(sc, cadr(args), sc->char_gt_symbol, args, T_CHARACTER, 2));
  return(make_boolean(sc, character(car(args)) > character(cadr(args))));
}

#if (!WITH_PURE_S7)
static s7_pointer g_char_cmp_ci(s7_scheme *sc, s7_pointer args, int32_t val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!s7_is_character(y))
    return(method_or_bust(sc, y, sym, args, T_CHARACTER, 1));

  for (x = cdr(args); is_pair(x); x = cdr(x))
    {
      if (!s7_is_character(car(x)))
	return(method_or_bust(sc, car(x), sym, cons(sc, y, x), T_CHARACTER, position_of(x, args)));
      if (charcmp(upper_character(y), upper_character(car(x))) != val)
	{
	  for (y = cdr(x); is_pair(y); y = cdr(y))
	    if (!is_character_via_method(sc, car(y)))
	      return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_CHARACTER));
	  return(sc->F);
	}
      y = car(x);
    }
  return(sc->T);
}

static s7_pointer g_char_cmp_ci_not(s7_scheme *sc, s7_pointer args, int32_t val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!s7_is_character(y))
    return(method_or_bust(sc, y, sym, args, T_CHARACTER, 1));
  for (x = cdr(args); is_pair(x); x = cdr(x))
    {
      if (!s7_is_character(car(x)))
	return(method_or_bust(sc, car(x), sym, cons(sc, y, x), T_CHARACTER, position_of(x, args)));
      if (charcmp(upper_character(y), upper_character(car(x))) == val)
	{
	  for (y = cdr(x); is_pair(y); y = cdr(y))
	    if (!is_character_via_method(sc, car(y)))
	      return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_CHARACTER));
	  return(sc->F);
	}
      y = car(x);
    }
  return(sc->T);
}

static s7_pointer g_chars_are_ci_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_ci_equal "(char-ci=? char ...) returns #t if all the character arguments are equal, ignoring case"
  #define Q_chars_are_ci_equal sc->pcl_bc

  return(g_char_cmp_ci(sc, args, 0, sc->char_ci_eq_symbol));
}

static s7_pointer g_chars_are_ci_less(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_ci_less "(char-ci<? char ...) returns #t if all the character arguments are increasing, ignoring case"
  #define Q_chars_are_ci_less sc->pcl_bc

  return(g_char_cmp_ci(sc, args, -1, sc->char_ci_lt_symbol));
}

static s7_pointer g_chars_are_ci_greater(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_ci_greater "(char-ci>? char ...) returns #t if all the character arguments are decreasing, ignoring case"
  #define Q_chars_are_ci_greater sc->pcl_bc

  return(g_char_cmp_ci(sc, args, 1, sc->char_ci_gt_symbol));
}

static s7_pointer g_chars_are_ci_geq(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_ci_geq "(char-ci>=? char ...) returns #t if all the character arguments are equal or decreasing, ignoring case"
  #define Q_chars_are_ci_geq sc->pcl_bc

  return(g_char_cmp_ci_not(sc, args, -1, sc->char_ci_geq_symbol));
}

static s7_pointer g_chars_are_ci_leq(s7_scheme *sc, s7_pointer args)
{
  #define H_chars_are_ci_leq "(char-ci<=? char ...) returns #t if all the character arguments are equal or increasing, ignoring case"
  #define Q_chars_are_ci_leq sc->pcl_bc

  return(g_char_cmp_ci_not(sc, args, 1, sc->char_ci_leq_symbol));
}


static bool char_ci_lt_b_direct(s7_pointer p1, s7_pointer p2) {return(upper_character(p1) < upper_character(p2));}
static bool char_ci_lt_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_ci_lt_symbol, p1, p2);
  return(upper_character(p1) < upper_character(p2));
}

static bool char_ci_leq_b_direct(s7_pointer p1, s7_pointer p2) {return(upper_character(p1) <= upper_character(p2));}
static bool char_ci_leq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_ci_leq_symbol, p1, p2);
  return(upper_character(p1) <= upper_character(p2));
}

static bool char_ci_gt_b_direct(s7_pointer p1, s7_pointer p2) {return(upper_character(p1) > upper_character(p2));}
static bool char_ci_gt_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_ci_gt_symbol, p1, p2);
  return(upper_character(p1) > upper_character(p2));
}

static bool char_ci_geq_b_direct(s7_pointer p1, s7_pointer p2) {return(upper_character(p1) >= upper_character(p2));}
static bool char_ci_geq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_ci_geq_symbol, p1, p2);
  return(upper_character(p1) >= upper_character(p2));
}

static bool char_ci_eq_b_direct(s7_pointer p1, s7_pointer p2) {return(upper_character(p1) == upper_character(p2));}
static bool char_ci_eq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_char2_args(sc, sc->char_ci_eq_symbol, p1, p2);
  return(upper_character(p1) == upper_character(p2));
}

#endif /* not pure s7 */

/* -------------------------------- char-position -------------------------------- */
static s7_pointer g_char_position(s7_scheme *sc, s7_pointer args)
{
  #define H_char_position "(char-position char-or-str str (start 0)) returns the position of the first occurrence of char in str, or #f"
  #define Q_char_position s7_make_signature(sc, 4, s7_make_signature(sc, 2, sc->is_integer_symbol, sc->not_symbol), s7_make_signature(sc, 2, sc->is_char_symbol, sc->is_string_symbol), sc->is_string_symbol, sc->is_integer_symbol)

  const char *porig, *pset;
  s7_int start, pos, len; /* not "int" because start arg might be most-negative-fixnum */
  s7_pointer arg1, arg2;

  arg1 = car(args);
  if ((!s7_is_character(arg1)) &&
      (!is_string(arg1)))
    return(method_or_bust(sc, arg1, sc->char_position_symbol, args, T_CHARACTER, 1));

  arg2 = cadr(args);
  if (!is_string(arg2))
    return(method_or_bust(sc, arg2, sc->char_position_symbol, args, T_STRING, 2));

  porig = string_value(arg2);
  len = string_length(arg2);

  if (is_pair(cddr(args)))
    {
      s7_pointer arg3;
      arg3 = caddr(args);
      if (!s7_is_integer(arg3))
	{
	  arg3 = check_value_slot(sc, arg3);
	  if (!s7_is_integer(arg3))
	    return(wrong_type_argument(sc, sc->char_position_symbol, 3, caddr(args), T_INTEGER));
	}
      start = s7_integer(arg3);
      if (start < 0)
	return(wrong_type_argument_with_type(sc, sc->char_position_symbol, 3, arg3, a_non_negative_integer_string));
    }
  else start = 0;
  if (start >= len) return(sc->F);

  if (s7_is_character(arg1))
    {
      char c;
      const char *p;
      c = character(arg1);
      p = strchr((const char *)(porig + start), (int)c); /* use strchrnul in Gnu C to catch embedded null case */
      if (p)
	return(make_integer(sc, p - porig));
      return(sc->F);
    }

  if (string_length(arg1) == 0)
    return(sc->F);
  pset = string_value(arg1);

  pos = strcspn((const char *)(porig + start), (const char *)pset);
  if ((pos + start) < len)
    return(make_integer(sc, pos + start));

  /* but if the string has an embedded null, we can get erroneous results here --
   *   perhaps check for null at pos+start?  What about a searched-for string that
   *   also has embedded nulls?
   */
  return(sc->F);
}

static s7_pointer char_position_p_ppi(s7_scheme *sc, s7_pointer p1, s7_pointer p2, s7_int start)
{
  /* p1 is char, p2 is string, p3 is int32_t */
  if (is_string(p2))
    {
      if (start >= 0)
	{
	  const char *porig, *p;
	  s7_int len;
	  char c;
	  c = character(p1);
	  len = string_length(p2);
	  porig = string_value(p2);
	  if (start >= len) return(sc->F);
	  p = strchr((const char *)(porig + start), (int)c);
	  if (p) return(make_integer(sc, p - porig));
	}
      else wrong_type_argument_with_type(sc, sc->char_position_symbol, 3, s7_make_integer(sc, start), a_non_negative_integer_string);
    }
  else simple_wrong_type_argument(sc, sc->char_position_symbol, p2, T_STRING);
  return(sc->F);
}

static s7_pointer g_char_position_csi(s7_scheme *sc, s7_pointer args)
{
  /* assume char arg1, no end */
  const char *porig, *p;
  char c;
  s7_pointer arg2;
  s7_int start, len;

  c = character(car(args));
  arg2 = cadr(args);

  if (!is_string(arg2))
    return(g_char_position(sc, args));

  len = string_length(arg2); /* can't return #f here if len==0 -- need start error check first */
  porig = string_value(arg2);

  if (is_pair(cddr(args)))
    {
      s7_pointer arg3;
      arg3 = caddr(args);
      if (!s7_is_integer(arg3))
	return(g_char_position(sc, args));
      start = s7_integer(arg3);
      if (start < 0)
	return(wrong_type_argument_with_type(sc, sc->char_position_symbol, 3, arg3, a_non_negative_integer_string));
      if (start >= len) return(sc->F);
    }
  else start = 0;

  if (len == 0) return(sc->F);
  p = strchr((const char *)(porig + start), (int)c);
  if (p)
    return(make_integer(sc, p - porig));
  return(sc->F);
}

/* -------------------------------- string-position -------------------------------- */
static s7_pointer g_string_position(s7_scheme *sc, s7_pointer args)
{
  #define H_string_position "(string-position str1 str2 (start 0)) returns the starting position of str1 in str2 or #f"
  #define Q_string_position s7_make_signature(sc, 4, s7_make_signature(sc, 2, sc->is_integer_symbol, sc->not_symbol), sc->is_string_symbol, sc->is_string_symbol, sc->is_integer_symbol)
  const char *s1, *s2, *p2;
  s7_int start = 0;
  s7_pointer s1p, s2p;

  s1p = car(args);
  if (!is_string(s1p))
    return(method_or_bust(sc, s1p, sc->string_position_symbol, args, T_STRING, 1));

  s2p = cadr(args);
  if (!is_string(s2p))
    return(method_or_bust(sc, s2p, sc->string_position_symbol, args, T_STRING, 2));

  if (is_pair(cddr(args)))
    {
      s7_pointer arg3;
      arg3 = caddr(args);
      if (!s7_is_integer(arg3))
	{
	  arg3 = check_value_slot(sc, arg3);
	  if (!s7_is_integer(arg3))
	    return(wrong_type_argument(sc, sc->string_position_symbol, 3, caddr(args), T_INTEGER));
	}
      start = s7_integer(arg3);
      if (start < 0)
	return(wrong_type_argument_with_type(sc, sc->string_position_symbol, 3, caddr(args), a_non_negative_integer_string));
    }

  if (string_length(s1p) == 0)
    return(sc->F);
  s1 = string_value(s1p);
  s2 = string_value(s2p);
  if (start >= string_length(s2p))
    return(sc->F);

  p2 = strstr((const char *)(s2 + start), s1);
  if (!p2) return(sc->F);
  return(make_integer(sc, p2 - s2));
}


/* -------------------------------- strings -------------------------------- */

/* prebuilding sc->empty_string and using it wherever len==0 did not produce more than about %.2 speedup
 *   (in index.scm where 11% of the strings are empty).  s7test max 4% empty, elsewhere much less.
 */
s7_pointer s7_make_string_with_length(s7_scheme *sc, const char *str, s7_int len)
{
  return(make_string_with_length(sc, str, len));
}

#define NUM_STRING_WRAPPERS 8 /* should be a power of 2 */

#if S7_DEBUGGING
/* #define wrap_string(Sc, Str, Len) wrap_string_1(Sc, Str, Len, __func__, __LINE__) */
static s7_pointer wrap_string_1(s7_scheme *sc, const char *str, s7_int len, const char *func, int line)
#else
static s7_pointer wrap_string(s7_scheme *sc, const char *str, s7_int len)
#endif
{
  s7_pointer x;
  x = sc->string_wrappers[sc->string_wrapper_pos];
  sc->string_wrapper_pos = (sc->string_wrapper_pos + 1) & (NUM_STRING_WRAPPERS - 1);
  string_value(x) = (char *)str;
  string_length(x) = len;
#if S7_DEBUGGING
  if ((strcmp(func, "g_substring_to_temp") != 0) &&
      (safe_strlen(str) != len))
    fprintf(stderr, "%s[%d]: %s len is not %ld but %ld\n", func, line, str, len, safe_strlen(str));
#endif
  return(x);
}

s7_pointer s7_make_string_wrapper(s7_scheme *sc, const char *str)
{
  return(wrap_string(sc, str, safe_strlen(str)));
}

static s7_pointer block_to_string(s7_scheme *sc, block_t *block, s7_int len)
{
  s7_pointer x;
  new_cell(sc, x, T_STRING | T_SAFE_PROCEDURE);
  string_block(x) = block;
  string_value(x) = (char *)block_data(block);
  string_length(x) = len;
  string_value(x)[len] = '\0';
  string_hash(x) = 0;
  add_string(sc, x);
  return(x);
}

static s7_pointer make_empty_string(s7_scheme *sc, s7_int len, char fill)
{
  s7_pointer x;
  block_t *b;
  new_cell(sc, x, T_STRING);
  b = mallocate(sc, len + 1);                   /* terminated_string_read_white_space needs the second #\null (is this still the case?) */
  string_block(x) = b;
  string_value(x) = (char *)block_data(b);
  if ((fill != '\0') && (len > 0))
    local_memset((void *)(string_value(x)), fill, len);
  string_value(x)[len] = 0;
  string_hash(x) = 0;
  string_length(x) = len;
  add_string(sc, x);
  return(x);
}

s7_pointer s7_make_string(s7_scheme *sc, const char *str)
{
  if (str)
    return(make_string_with_length(sc, str, safe_strlen(str)));
  return(make_empty_string(sc, 0, 0));
}

static char *make_permanent_c_string(s7_scheme *sc, const char *str)
{
  char *x;
  s7_int len;
  len = safe_strlen(str);
  x = (char *)alloc_permanent_string(sc, (len + 1) * sizeof(char));
  memcpy((void *)x, (void *)str, len);
  x[len] = 0;
  return(x);
}

s7_pointer s7_make_permanent_string(s7_scheme *sc, const char *str)
{
  /* for the symbol table which is never GC'd */
  s7_pointer x;
  x = alloc_pointer(sc);
  set_type(x, T_STRING | T_IMMUTABLE | T_UNHEAP);
  if (str)
    {
      s7_int len;
      len = safe_strlen(str);
      string_length(x) = len;
      /* string_block(x) = mallocate_block(); */
      string_block(x) = NULL;
      string_value(x) = (char *)alloc_permanent_string(sc, (len + 1) * sizeof(char));
      memcpy((void *)string_value(x), (void *)str, len);
      string_value(x)[len] = 0;
    }
  else
    {
      string_value(x) = NULL;
      string_length(x) = 0;
    }
  string_hash(x) = 0;
  return(x);
}

bool s7_is_string(s7_pointer p)
{
  return(is_string(p));
}


const char *s7_string(s7_pointer p)
{
  return(string_value(p));
}


static s7_pointer g_is_string(s7_scheme *sc, s7_pointer args)
{
  #define H_is_string "(string? obj) returns #t if obj is a string"
  #define Q_is_string sc->pl_bt

  check_boolean_method(sc, is_string, sc->is_string_symbol, args);
}

static s7_pointer make_permanent_string(const char *str)
{
  s7_pointer x;
  s7_int len;

  x = (s7_pointer)calloc(1, sizeof(s7_cell));
  set_type(x, T_STRING | T_IMMUTABLE | T_UNHEAP);

  len = safe_strlen(str);
  string_length(x) = len;
  string_block(x) = NULL;
  string_value(x) = (char *)str;
  string_hash(x) = 0;
  return(x);
}

static void init_strings(void)
{
  car_a_list_string = make_permanent_string("a list whose car is also a list");
  cdr_a_list_string = make_permanent_string("a list whose cdr is also a list");

  caar_a_list_string = make_permanent_string("a list whose caar is also a list");
  cadr_a_list_string = make_permanent_string("a list whose cadr is also a list");
  cdar_a_list_string = make_permanent_string("a list whose cdar is also a list");
  cddr_a_list_string = make_permanent_string("a list whose cddr is also a list");

  caaar_a_list_string = make_permanent_string("a list whose caaar is also a list");
  caadr_a_list_string = make_permanent_string("a list whose caadr is also a list");
  cadar_a_list_string = make_permanent_string("a list whose cadar is also a list");
  caddr_a_list_string = make_permanent_string("a list whose caddr is also a list");
  cdaar_a_list_string = make_permanent_string("a list whose cdaar is also a list");
  cdadr_a_list_string = make_permanent_string("a list whose cdadr is also a list");
  cddar_a_list_string = make_permanent_string("a list whose cddar is also a list");
  cdddr_a_list_string = make_permanent_string("a list whose cdddr is also a list");

  a_list_string =                 make_permanent_string("a list");
  an_eq_func_string =             make_permanent_string("a procedure that can take 2 arguments");
  an_association_list_string =    make_permanent_string("an association list");
  a_normal_real_string =          make_permanent_string("a normal real");
  a_rational_string =             make_permanent_string("an integer or a ratio");
  a_number_string =               make_permanent_string("a number");
  a_procedure_string =            make_permanent_string("a procedure");
  a_normal_procedure_string =     make_permanent_string("a normal procedure (not a continuation)");
  a_let_string =                  make_permanent_string("a let (environment)");
  a_proper_list_string =          make_permanent_string("a proper list");
  a_boolean_string =              make_permanent_string("a boolean");
  a_byte_vector_string =          make_permanent_string("a byte-vector");
  an_input_port_string =          make_permanent_string("an input port");
  an_open_port_string =           make_permanent_string("an open port");
  an_output_port_string =         make_permanent_string("an output port");
  an_input_string_port_string =   make_permanent_string("an input string port");
  an_input_file_port_string =     make_permanent_string("an input file port");
  an_output_string_port_string =  make_permanent_string("an output string port");
  an_output_file_port_string =    make_permanent_string("an output file port");
  a_thunk_string =                make_permanent_string("a thunk");
  a_symbol_string =               make_permanent_string("a symbol");
  a_non_negative_integer_string = make_permanent_string("a non-negative integer");
  an_unsigned_byte_string =       make_permanent_string("an unsigned byte");
  something_applicable_string =   make_permanent_string("a procedure or something applicable");
  a_random_state_object_string =  make_permanent_string("a random-state object");
  a_format_port_string =          make_permanent_string("#f, #t, (), or an open output port");
  a_non_constant_symbol_string =  make_permanent_string("a non-constant symbol");
  a_sequence_string =             make_permanent_string("a sequence");
  a_valid_radix_string =          make_permanent_string("should be between 2 and 16");
  result_is_too_large_string =    make_permanent_string("result is too large");
  its_too_large_string =          make_permanent_string("it is too large");
  its_too_small_string =          make_permanent_string("it is less than the start position");
  its_negative_string =           make_permanent_string("it is negative");
  its_nan_string =                make_permanent_string("NaN usually indicates a numerical error");
  its_infinite_string =           make_permanent_string("it is infinite");
  too_many_indices_string =       make_permanent_string("too many indices");
  value_is_missing_string =       make_permanent_string("~A argument '~A's value is missing");
  parameter_set_twice_string =    make_permanent_string("parameter set twice, ~S in ~S");
  immutable_error_string =        make_permanent_string("can't ~S ~S (it is immutable)");
#if (!HAVE_COMPLEX_NUMBERS)
  no_complex_numbers_string =     make_permanent_string("this version of s7 does not support complex numbers");
#endif

  format_string_1 = make_permanent_string("format: ~S ~{~S~^ ~}: ~A");
  format_string_2 = make_permanent_string("format: ~S: ~A");
  format_string_3 = make_permanent_string("format: ~S ~{~S~^ ~}~&~NT^: ~A");
  format_string_4 = make_permanent_string("format: ~S~&~NT^: ~A");

  too_many_arguments_string = make_permanent_string("~A: too many arguments: ~A");
  not_enough_arguments_string = make_permanent_string("~A: not enough arguments: ~A");
  missing_method_string = make_permanent_string("missing ~S method in ~S");
}


/* -------------------------------- make-string -------------------------------- */
static s7_pointer g_make_string(s7_scheme *sc, s7_pointer args)
{
  #define H_make_string "(make-string len (val #\\space)) makes a string of length len filled with the character val (default: space)"
  #define Q_make_string s7_make_signature(sc, 3, sc->is_string_symbol, sc->is_integer_symbol, sc->is_char_symbol)

  s7_pointer n;
  s7_int len;

  n = car(args);
  if (!s7_is_integer(n))
    {
      check_method(sc, n, sc->make_string_symbol, args);
      return(wrong_type_argument(sc, sc->make_string_symbol, 1, n, T_INTEGER));
    }

  len = s7_integer(n);
#if WITH_GMP
  if ((len == 0) && (!s7_is_zero(n)))
    return(s7_out_of_range_error(sc, "make-string", 1, n, "big integer is too big for s7_int"));
#endif
  if ((len < 0) || (len > sc->max_string_length))
    return(out_of_range(sc, sc->make_string_symbol, small_int(1), n, (len < 0) ? its_negative_string : its_too_large_string));

  if (is_not_null(cdr(args)))
    {
      char fill;
      if (!s7_is_character(cadr(args)))
	return(method_or_bust(sc, cadr(args), sc->make_string_symbol, args, T_CHARACTER, 2));
      fill = s7_character(cadr(args));
      n = make_empty_string(sc, len, fill);
      if ((fill == '\0') && (len > 0))
	memclr((void *)string_value(n), (size_t)len);
      return(n);
    }
  return(make_empty_string(sc, len, '\0')); /* #\null here means "don't fill/clear" */
}


#if (!WITH_PURE_S7)
static s7_pointer g_string_length(s7_scheme *sc, s7_pointer args)
{
  #define H_string_length "(string-length str) returns the length of the string str"
  #define Q_string_length s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_string_symbol)
  s7_pointer p;
  p = car(args);
  if (!is_string(p))
    return(method_or_bust_one_arg(sc, p, sc->string_length_symbol, args, T_STRING));
  return(make_integer(sc, string_length(p)));
}

static s7_int string_length_i_7p(s7_scheme *sc, s7_pointer p)
{
  if (!is_string(p))
    simple_wrong_type_argument(sc, sc->string_length_symbol, p, T_STRING);
  return(string_length(p));
}
#endif


/* -------------------------------- string-up|downcase -------------------------------- */
static s7_pointer g_string_downcase(s7_scheme *sc, s7_pointer args)
{
  #define H_string_downcase "(string-downcase str) returns the lower case version of str."
  #define Q_string_downcase sc->pcl_s

  s7_pointer p, newstr;
  s7_int i, len;
  uint8_t *nstr, *ostr;

  p = car(args);
  if (!is_string(p))
    /* perhaps gc-protect p here and in upcase below */
    return(method_or_bust_one_arg(sc, p, sc->string_downcase_symbol, list_1(sc, p), T_STRING));
  len = string_length(p);
  newstr = make_empty_string(sc, len, 0);

  ostr = (uint8_t *)string_value(p);
  nstr = (uint8_t *)string_value(newstr);

  if (len >= 128)
    {
      i = len - 1;
      while (i >= 8)
	LOOP_8(nstr[i] = lowers[(uint8_t)ostr[i]]; i--);
      while (i >= 0) {nstr[i] = lowers[(uint8_t)ostr[i]]; i--;}
    }
  else
    {
      for (i = 0; i < len; i++)
	nstr[i] = lowers[(uint8_t)ostr[i]];
    }
  return(newstr);
}

static s7_pointer g_string_upcase(s7_scheme *sc, s7_pointer args)
{
  #define H_string_upcase "(string-upcase str) returns the upper case version of str."
  #define Q_string_upcase sc->pcl_s

  s7_pointer p, newstr;
  s7_int i, len;
  uint8_t *nstr, *ostr;

  p = car(args);
  if (!is_string(p))
    return(method_or_bust_one_arg(sc, p, sc->string_upcase_symbol, list_1(sc, p), T_STRING));
  len = string_length(p);
  newstr = make_empty_string(sc, len, 0);

  ostr = (uint8_t *)string_value(p);
  nstr = (uint8_t *)string_value(newstr);

  if (len >= 128)
    {
      i = len - 1;
      while (i >= 8)
	LOOP_8(nstr[i] = uppers[(uint8_t)ostr[i]]; i--);
      while (i >= 0) {nstr[i] = uppers[(uint8_t)ostr[i]]; i--;}
    }
  else
    {
      for (i = 0; i < len; i++)
	nstr[i] = uppers[(uint8_t)ostr[i]];
    }
  return(newstr);
}

s7_int s7_string_length(s7_pointer str)
{
  return(string_length(str));
}


/* -------------------------------- string-ref -------------------------------- */
static s7_pointer string_ref_1(s7_scheme *sc, s7_pointer strng, s7_pointer index)
{
  char *str;
  s7_int ind;

  if (!s7_is_integer(index))
    {
      s7_pointer p;
      p = check_value_slot(sc, index);
      if (!s7_is_integer(p))
	return(wrong_type_argument(sc, sc->string_ref_symbol, 2, index, T_INTEGER));
      ind = s7_integer(p);
    }
  else ind = s7_integer(index);
  if (ind < 0)
    return(wrong_type_argument_with_type(sc, sc->string_ref_symbol, 2, index, a_non_negative_integer_string));
  if (ind >= string_length(strng))
    return(out_of_range(sc, sc->string_ref_symbol, small_int(2), index, its_too_large_string));

  str = string_value(strng);
  return(s7_make_character(sc, ((uint8_t *)str)[ind]));
}

static s7_pointer g_string_ref(s7_scheme *sc, s7_pointer args)
{
  #define H_string_ref "(string-ref str index) returns the character at the index-th element of the string str"
  #define Q_string_ref s7_make_signature(sc, 3, sc->is_char_symbol, sc->is_string_symbol, sc->is_integer_symbol)

  s7_pointer strng;
  strng = car(args);
  if (!is_string(strng))
    return(method_or_bust(sc, strng, sc->string_ref_symbol, args, T_STRING, 1));
  return(string_ref_1(sc, strng, cadr(args)));
}

static s7_pointer string_ref_p_pi(s7_scheme *sc, s7_pointer p1, s7_int i1)
{
  if (!is_string(p1))
    simple_wrong_type_argument(sc, sc->string_ref_symbol, p1, T_STRING);
  if ((i1 >= 0) && (i1 < string_length(p1)))
    return(chars[((uint8_t *)string_value(p1))[i1]]);
  out_of_range(sc, sc->string_ref_symbol, small_int(2), wrap_integer1(sc, i1), (i1 < 0) ? its_negative_string : its_too_large_string);
  return(p1);
}

static s7_pointer string_ref_p_pi_direct(s7_scheme *sc, s7_pointer p1, s7_int i1)
{
  if ((i1 >= 0) && (i1 < string_length(p1)))
    return(chars[((uint8_t *)string_value(p1))[i1]]);
  out_of_range(sc, sc->string_ref_symbol, small_int(2), wrap_integer1(sc, i1), (i1 < 0) ? its_negative_string : its_too_large_string);
  return(p1);
}

static s7_pointer string_ref_unchecked(s7_scheme *sc, s7_pointer p1, s7_int i1) {return(chars[((uint8_t *)string_value(p1))[i1]]);}


/* -------------------------------- string-set! -------------------------------- */
static s7_pointer g_string_set(s7_scheme *sc, s7_pointer args)
{
  #define H_string_set "(string-set! str index chr) sets the index-th element of the string str to the character chr"
  #define Q_string_set s7_make_signature(sc, 4, sc->is_char_symbol, sc->is_string_symbol, sc->is_integer_symbol, sc->is_char_symbol)

  s7_pointer strng, c, index;
  char *str;
  s7_int ind;

  strng = car(args);
  if (!is_mutable_string(strng))
    return(mutable_method_or_bust(sc, strng, sc->string_set_symbol, args, T_STRING, 1));

  index = cadr(args);
  if (!s7_is_integer(index))
    {
      s7_pointer p;
      p = check_value_slot(sc, index);
      if (!s7_is_integer(p))
	return(wrong_type_argument(sc, sc->string_set_symbol, 2, index, T_INTEGER));
      ind = s7_integer(p);
    }
  else ind = s7_integer(index);
  if (ind < 0)
    return(wrong_type_argument_with_type(sc, sc->string_set_symbol, 2, index, a_non_negative_integer_string));
  if (ind >= string_length(strng))
    return(out_of_range(sc, sc->string_set_symbol, small_int(2), index, its_too_large_string));

  str = string_value(strng);
  c = caddr(args);
  if (!s7_is_character(c))
    return(method_or_bust(sc, c, sc->string_set_symbol, args, T_CHARACTER, 3));

  str[ind] = (char)s7_character(c);
  return(c);
}

static s7_pointer string_set_p_pip(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_pointer p2)
{
  if (!is_string(p1))
    simple_wrong_type_argument(sc, sc->string_set_symbol, p1, T_STRING);
  if (!s7_is_character(p2))
    simple_wrong_type_argument(sc, sc->string_set_symbol, p2, T_CHARACTER);
  if ((i1 >= 0) && (i1 < string_length(p1)))
    string_value(p1)[i1] = s7_character(p2);
  else out_of_range(sc, sc->string_set_symbol, small_int(2), wrap_integer1(sc, i1), (i1 < 0) ? its_negative_string : its_too_large_string);
  return(p2);
}

static s7_pointer string_set_p_pip_direct(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_pointer p2)
{
  if ((i1 >= 0) && (i1 < string_length(p1)))
    string_value(p1)[i1] = s7_character(p2);
  else out_of_range(sc, sc->string_set_symbol, small_int(2),make_integer(sc, i1), (i1 < 0) ? its_negative_string : its_too_large_string);
  return(p2);
}

static s7_pointer string_set_unchecked(s7_scheme *sc, s7_pointer p1, s7_int i1, s7_pointer p2) {string_value(p1)[i1] = s7_character(p2); return(p2);}


/* -------------------------------- string-append -------------------------------- */
static s7_pointer g_string_append_1(s7_scheme *sc, s7_pointer args, s7_pointer caller)
{
  #define H_string_append "(string-append str1 ...) appends all its string arguments into one string"
  #define Q_string_append sc->pcl_s

  s7_int len = 0;
  s7_pointer x, newstr;
  char *pos;

  if (is_null(args))
    return(make_string_with_length(sc, "", 0));

  /* get length for new string */
  for (x = args; is_not_null(x); x = cdr(x))
    {
      s7_pointer p;
      p = car(x);
      if (!is_string(p))
	{
	  /* look for string-append and if found, cobble up a plausible intermediate call */
	  if (has_methods(p))
	    {
	      s7_pointer func;
	      func = find_method(sc, find_let(sc, p), caller);
	      if (func != sc->undefined)
		{
		  s7_pointer y;
		  if (len == 0)
		    return(s7_apply_function(sc, func, x)); /* not args (string-append "" "" ...) */
		  newstr = make_empty_string(sc, len, 0);
		  for (pos = string_value(newstr), y = args; y != x; pos += string_length(car(y)), y = cdr(y))
		    memcpy(pos, string_value(car(y)), string_length(car(y)));
		  return(s7_apply_function(sc, func, cons(sc, newstr, x)));
		}
	    }
	  return(wrong_type_argument(sc, caller, position_of(x, args), p, T_STRING));
	}
      len += string_length(p);
    }

  if (len == 0) return(car(args));

  if (len > sc->max_string_length)
    return(s7_error(sc, sc->out_of_range_symbol,
		    set_elist_4(sc, wrap_string(sc, "~S new string length, ~D, is larger than (*s7* 'max-string-length): ~D", 70),
				caller,
				wrap_integer1(sc, len),
				wrap_integer2(sc, sc->max_string_length))));

  newstr = make_empty_string(sc, len, 0);

  for (pos = string_value(newstr), x = args; is_not_null(x); pos += string_length(car(x)), x = cdr(x))
    if (string_length(car(x)) > 0)
      memcpy(pos, string_value(car(x)), string_length(car(x)));

  return(newstr);
}

static s7_pointer g_string_append(s7_scheme *sc, s7_pointer args)
{
  return(g_string_append_1(sc, args, sc->string_append_symbol));
}


#if (!WITH_PURE_S7)
static s7_pointer g_string_copy(s7_scheme *sc, s7_pointer args)
{
  #define H_string_copy "(string-copy str) returns a copy of its string argument"
  #define Q_string_copy s7_make_signature(sc, 2, sc->is_string_symbol, sc->is_string_symbol)
  s7_pointer p;
  p = car(args);
  if (!is_string(p))
    return(method_or_bust(sc, p, sc->string_copy_symbol, args, T_STRING, 1));
  return(make_string_with_length(sc, string_value(p), string_length(p)));
}
#endif


/* -------------------------------- substring -------------------------------- */
static s7_pointer start_and_end(s7_scheme *sc, s7_pointer caller, s7_pointer start_and_end_args, int32_t position, s7_int *start, s7_int *end)
{
  /* we assume that *start=0 and *end=length, that end is "exclusive"
   *   return true if the start/end points are not changed.
   */
  s7_pointer pstart, pend;
  s7_int index;

  pstart = car(start_and_end_args);
  if (!s7_is_integer(pstart))
    {
      s7_pointer p;
      p = check_value_slot(sc, pstart);
      if (!s7_is_integer(p))
	return(wrong_type_argument(sc, caller, position, pstart, T_INTEGER));
      index = s7_integer(p);
    }
  else index = s7_integer(pstart);
  if ((index < 0) ||
      (index > *end)) /* *end == length here */
    return(out_of_range(sc, caller, small_int(position), pstart, (index < 0) ? its_negative_string : its_too_large_string));
  *start = index;

  if (is_null(cdr(start_and_end_args)))
    return(sc->gc_nil);

  pend = cadr(start_and_end_args);
  if (!s7_is_integer(pend))
    {
      s7_pointer p;
      p = check_value_slot(sc, pend);
      if (!s7_is_integer(p))
	return(wrong_type_argument(sc, caller, position + 1, pend, T_INTEGER));
      index = s7_integer(p);
    }
  else index = s7_integer(pend);
  if ((index < *start) ||
      (index > *end))
    return(out_of_range(sc, caller, small_int(position + 1), pend, (index < *start) ? its_too_small_string : its_too_large_string));
  *end = index;
  return(sc->gc_nil);
}


static s7_pointer g_substring(s7_scheme *sc, s7_pointer args)
{
  #define H_substring "(substring str start (end (length str))) returns the portion of the string str between start and \
end: (substring \"01234\" 1 2) -> \"1\""
  #define Q_substring s7_make_circular_signature(sc, 2, 3, sc->is_string_symbol, sc->is_string_symbol, sc->is_integer_symbol)

  s7_pointer x, str;
  s7_int start = 0, end, len;
  char *s;

  str = car(args);
  if (!is_string(str))
    return(method_or_bust(sc, str, sc->substring_symbol, args, T_STRING, 1));

  end = string_length(str);
  if (!is_null(cdr(args)))
    {
      x = start_and_end(sc, sc->substring_symbol, cdr(args), 2, &start, &end);
      if (x != sc->gc_nil) return(x);
    }
  s = string_value(str);
  len = end - start;
  x = make_string_with_length(sc, (char *)(s + start), len);
  string_value(x)[len] = 0;
  return(x);
}


static s7_pointer g_substring_to_temp(s7_scheme *sc, s7_pointer args)
{
  s7_pointer str;
  s7_int start = 0, end;

  str = car(args);
  if (!is_string(str))
    return(method_or_bust(sc, str, sc->substring_symbol, args, T_STRING, 1));

  end = string_length(str);
  if (!is_null(cdr(args)))
    {
      s7_pointer x;
      x = start_and_end(sc, sc->substring_symbol, cdr(args), 2, &start, &end);
      if (x != sc->gc_nil) return(x);
    }
  return(wrap_string(sc, (char *)(string_value(str) + start), end - start));
}


/* -------------------------------- string comparisons -------------------------------- */
static int32_t scheme_strcmp(s7_pointer s1, s7_pointer s2)
{
  /* tricky here because str[i] must be treated as unsigned: (string<? (string (integer->char #xf0)) (string (integer->char #x70)))
   *   and null or lack thereof does not say anything about the string end
   */
  size_t i, len, len1, len2;
  char *str1, *str2;

  len1 = (size_t)string_length(s1);
  len2 = (size_t)string_length(s2);
  if (len1 > len2)
    len = len2;
  else len = len1;

  str1 = string_value(s1);
  str2 = string_value(s2);

  if (len < sizeof(size_t))
    {
      for (i = 0; i < len; i++)
	if ((uint8_t)(str1[i]) < (uint8_t )(str2[i]))
	  return(-1);
	else
	  {
	    if ((uint8_t)(str1[i]) > (uint8_t)(str2[i]))
	      return(1);
	  }
    }
  else
    {
      /* this algorithm from stackoverflow(?), with various changes (original did not work for large strings, etc) */
      size_t last, pos;
      size_t *ptr1, *ptr2;

      last = len / sizeof(size_t);
      for (ptr1 = (size_t *)str1, ptr2 = (size_t *)str2, i = 0; i < last; i++)
	if (ptr1[i] ^ ptr2[i])
	  break;

      for (pos = i * sizeof(size_t); pos < len; pos++)
	{
	  if ((uint8_t)str1[pos] < (uint8_t)str2[pos])
	    return(-1);
	  if ((uint8_t)str1[pos] > (uint8_t)str2[pos])
	    return(1);
	}
    }

  if (len1 < len2)
    return(-1);
  if (len1 > len2)
    return(1);
  return(0);
}

static bool is_string_via_method(s7_scheme *sc, s7_pointer p)
{
  if (s7_is_string(p))
    return(true);
  if (has_methods(p))
    {
      s7_pointer f;
      f = find_method(sc, find_let(sc, p), sc->is_string_symbol);
      if (f != sc->undefined)
	return(is_true(sc, s7_apply_function(sc, f, cons(sc, p, sc->nil))));
    }
  return(false);
}

static s7_pointer g_string_cmp(s7_scheme *sc, s7_pointer args, int32_t val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!is_string(y))
    return(method_or_bust(sc, y, sym, args, T_STRING, 1));

  for (x = cdr(args); is_not_null(x); x = cdr(x))
    {
      if (!is_string(car(x)))
	return(method_or_bust(sc, car(x), sym, cons(sc, y, x), T_STRING, position_of(x, args)));
      if (scheme_strcmp(y, car(x)) != val)
	{
	  for (y = cdr(x); is_pair(y); y = cdr(y))
	    if (!is_string_via_method(sc, car(y)))
	      return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_STRING));
	  return(sc->F);
	}
      y = car(x);
    }
  return(sc->T);
}


static s7_pointer g_string_cmp_not(s7_scheme *sc, s7_pointer args, int32_t val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!is_string(y))
    return(method_or_bust(sc, y, sym, args, T_STRING, 1));

  for (x = cdr(args); is_not_null(x); x = cdr(x))
    {
      if (!is_string(car(x)))
	return(method_or_bust(sc, car(x), sym, cons(sc, y, x), T_STRING, position_of(x, args)));
      if (scheme_strcmp(y, car(x)) == val)
	{
	  for (y = cdr(x); is_pair(y); y = cdr(y))
	    if (!is_string_via_method(sc, car(y)))
	      return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_STRING));
	  return(sc->F);
	}
      y = car(x);
    }
  return(sc->T);
}


static bool scheme_strings_are_equal(s7_pointer x, s7_pointer y)
{
  return((string_length(x) == string_length(y)) &&
	 (strings_are_equal_with_length(string_value(x), string_value(y), string_length(x))));
}

static s7_pointer g_strings_are_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_equal "(string=? str ...) returns #t if all the string arguments are equal"
  #define Q_strings_are_equal sc->pcl_bs

  /* C-based check stops at null, but we can have embedded nulls.
   *   (let ((s1 "1234") (s2 "1245")) (string-set! s1 1 #\null) (string-set! s2 1 #\null) (string=? s1 s2))
   */
  s7_pointer x, y;
  bool happy = true;

  y = car(args);
  if (!is_string(y))
    return(method_or_bust(sc, y, sc->string_eq_symbol, args, T_STRING, 1));

  for (x = cdr(args); is_pair(x); x = cdr(x))
    {
      s7_pointer p;
      p = car(x);
      if (y != p)
	{
	  if (!is_string(p))
	    return(method_or_bust(sc, p, sc->string_eq_symbol, cons(sc, y, x), T_STRING, position_of(x, args)));
	  if (happy)
	    happy = scheme_strings_are_equal(p, y);
	}
    }
  if (!happy)
    return(sc->F);
  return(sc->T);
}


static s7_pointer g_strings_are_less(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_less "(string<? str ...) returns #t if all the string arguments are increasing"
  #define Q_strings_are_less sc->pcl_bs

  return(g_string_cmp(sc, args, -1, sc->string_lt_symbol));
}


static s7_pointer g_strings_are_greater(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_greater "(string>? str ...) returns #t if all the string arguments are decreasing"
  #define Q_strings_are_greater sc->pcl_bs

  return(g_string_cmp(sc, args, 1, sc->string_gt_symbol));
}


static s7_pointer g_strings_are_geq(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_geq "(string>=? str ...) returns #t if all the string arguments are equal or decreasing"
  #define Q_strings_are_geq sc->pcl_bs

  return(g_string_cmp_not(sc, args, -1, sc->string_geq_symbol));
}


static s7_pointer g_strings_are_leq(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_leq "(string<=? str ...) returns #t if all the string arguments are equal or increasing"
  #define Q_strings_are_leq sc->pcl_bs

  return(g_string_cmp_not(sc, args, 1, sc->string_leq_symbol));
}

static s7_pointer g_string_equal_2(s7_scheme *sc, s7_pointer args)
{
  if (!is_string(car(args)))
    return(method_or_bust(sc, car(args), sc->string_eq_symbol, args, T_STRING, 1));
  if (!is_string(cadr(args)))
    return(method_or_bust(sc, cadr(args), sc->string_eq_symbol, args, T_STRING, 2));
  return(make_boolean(sc, scheme_strings_are_equal(car(args), cadr(args))));
}


static s7_pointer g_string_less_2(s7_scheme *sc, s7_pointer args)
{
  if (!is_string(car(args)))
    return(method_or_bust(sc, car(args), sc->string_lt_symbol, args, T_STRING, 1));
  if (!is_string(cadr(args)))
    return(method_or_bust(sc, cadr(args), sc->string_lt_symbol, args, T_STRING, 2));
  return(make_boolean(sc, scheme_strcmp(car(args), cadr(args)) == -1));
}


static s7_pointer g_string_greater_2(s7_scheme *sc, s7_pointer args)
{
  if (!is_string(car(args)))
    return(method_or_bust(sc, car(args), sc->string_gt_symbol, args, T_STRING, 1));
  if (!is_string(cadr(args)))
    return(method_or_bust(sc, cadr(args), sc->string_gt_symbol, args, T_STRING, 2));
  return(make_boolean(sc, scheme_strcmp(car(args), cadr(args)) == 1));
}

static inline void check_string2_args(s7_scheme *sc, s7_pointer caller, s7_pointer p1, s7_pointer p2)
{
  if (!is_string(p1))
    simple_wrong_type_argument(sc, caller, p1, T_STRING);
  if (!s7_is_string(p2))
    simple_wrong_type_argument(sc, caller, p2, T_STRING);
}

static bool string_lt_b_direct(s7_pointer p1, s7_pointer p2) {return(scheme_strcmp(p1, p2) == -1);}
static bool string_lt_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_string2_args(sc, sc->string_lt_symbol, p1, p2);
  return(scheme_strcmp(p1, p2) == -1);
}

static bool string_leq_b_direct(s7_pointer p1, s7_pointer p2) {return(scheme_strcmp(p1, p2) != 1);}
static bool string_leq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_string2_args(sc, sc->string_leq_symbol, p1, p2);
  return(scheme_strcmp(p1, p2) != 1);
}

static bool string_gt_b_direct(s7_pointer p1, s7_pointer p2) {return(scheme_strcmp(p1, p2) == 1);}
static bool string_gt_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_string2_args(sc, sc->string_gt_symbol, p1, p2);
  return(scheme_strcmp(p1, p2) == 1);
}

static bool string_geq_b_direct(s7_pointer p1, s7_pointer p2) {return(scheme_strcmp(p1, p2) != -1);}
static bool string_geq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_string2_args(sc, sc->string_geq_symbol, p1, p2);
  return(scheme_strcmp(p1, p2) != -1);
}

static bool string_eq_b_direct(s7_pointer p1, s7_pointer p2) {return(scheme_strings_are_equal(p1, p2));}
static bool string_eq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_string2_args(sc, sc->string_eq_symbol, p1, p2);
  return(scheme_strings_are_equal(p1, p2));
}


#if (!WITH_PURE_S7)

static int32_t scheme_strcasecmp(s7_pointer s1, s7_pointer s2)
{
  /* same as scheme_strcmp -- watch out for unwanted sign! and lack of trailing null (length sets string end).
   */
  s7_int i, len, len1, len2;
  uint8_t *str1, *str2;

  len1 = string_length(s1);
  len2 = string_length(s2);
  if (len1 > len2)
    len = len2;
  else len = len1;

  str1 = (uint8_t *)string_value(s1);
  str2 = (uint8_t *)string_value(s2);

  for (i = 0; i < len; i++)
    if (uppers[(int32_t)str1[i]] < uppers[(int32_t)str2[i]])
      return(-1);
    else
      {
	if (uppers[(int32_t)str1[i]] > uppers[(int32_t)str2[i]])
	  return(1);
      }

  if (len1 < len2)
    return(-1);
  if (len1 > len2)
    return(1);
  return(0);
}


static bool scheme_strequal_ci(s7_pointer s1, s7_pointer s2)
{
  /* same as scheme_strcmp -- watch out for unwanted sign! */
  s7_int i, len, len2;
  uint8_t *str1, *str2;

  len = string_length(s1);
  len2 = string_length(s2);
  if (len != len2)
    return(false);

  str1 = (uint8_t *)string_value(s1);
  str2 = (uint8_t *)string_value(s2);

  for (i = 0; i < len; i++)
    if (uppers[(int32_t)str1[i]] != uppers[(int32_t)str2[i]])
      return(false);
  return(true);
}

static s7_pointer string_check_method(s7_scheme *sc, s7_pointer sym, s7_pointer x, s7_pointer y, s7_pointer args)
{
  for (y = cdr(x); is_pair(y); y = cdr(y))
    if (!is_string_via_method(sc, car(y)))
      return(wrong_type_argument(sc, sym, position_of(y, args), car(y), T_STRING));
  return(sc->F);
}

static s7_pointer g_string_ci_cmp(s7_scheme *sc, s7_pointer args, int32_t val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!is_string(y))
    return(method_or_bust(sc, y, sym, args, T_STRING, 1));

  for (x = cdr(args); is_not_null(x); x = cdr(x))
    {
      if (!is_string(car(x)))
	return(method_or_bust(sc, car(x), sym, cons(sc, y, x), T_STRING, position_of(x, args)));
      if (val == 0)
	{
	  if (!scheme_strequal_ci(y, car(x)))
	    return(string_check_method(sc, sym, x, y, args));
	}
      else
	{
	  if (scheme_strcasecmp(y, car(x)) != val)
	    return(string_check_method(sc, sym, x, y, args));
	}
      y = car(x);
    }
  return(sc->T);
}


static s7_pointer g_string_ci_cmp_not(s7_scheme *sc, s7_pointer args, int32_t val, s7_pointer sym)
{
  s7_pointer x, y;

  y = car(args);
  if (!is_string(y))
    return(method_or_bust(sc, y, sym, args, T_STRING, 1));

  for (x = cdr(args); is_not_null(x); x = cdr(x))
    {
      if (!is_string(car(x)))
	return(method_or_bust(sc, car(x), sym, cons(sc, y, x), T_STRING, position_of(x, args)));
      if (scheme_strcasecmp(y, car(x)) == val)
	return(string_check_method(sc, sym, x, y, args));
      y = car(x);
    }
  return(sc->T);
}


static s7_pointer g_strings_are_ci_equal(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_ci_equal "(string-ci=? str ...) returns #t if all the string arguments are equal, ignoring case"
  #define Q_strings_are_ci_equal sc->pcl_bs
  return(g_string_ci_cmp(sc, args, 0, sc->string_ci_eq_symbol));
}


static s7_pointer g_strings_are_ci_less(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_ci_less "(string-ci<? str ...) returns #t if all the string arguments are increasing, ignoring case"
  #define Q_strings_are_ci_less sc->pcl_bs
  return(g_string_ci_cmp(sc, args, -1, sc->string_ci_lt_symbol));
}


static s7_pointer g_strings_are_ci_greater(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_ci_greater "(string-ci>? str ...) returns #t if all the string arguments are decreasing, ignoring case"
  #define Q_strings_are_ci_greater sc->pcl_bs
  return(g_string_ci_cmp(sc, args, 1, sc->string_ci_gt_symbol));
}


static s7_pointer g_strings_are_ci_geq(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_ci_geq "(string-ci>=? str ...) returns #t if all the string arguments are equal or decreasing, ignoring case"
  #define Q_strings_are_ci_geq sc->pcl_bs
  return(g_string_ci_cmp_not(sc, args, -1, sc->string_ci_geq_symbol));
}


static s7_pointer g_strings_are_ci_leq(s7_scheme *sc, s7_pointer args)
{
  #define H_strings_are_ci_leq "(string-ci<=? str ...) returns #t if all the string arguments are equal or increasing, ignoring case"
  #define Q_strings_are_ci_leq sc->pcl_bs
  return(g_string_ci_cmp_not(sc, args, 1, sc->string_ci_leq_symbol));
}


static bool string_ci_lt_b_direct(s7_pointer p1, s7_pointer p2) {return(scheme_strcasecmp(p1, p2) == -1);}
static bool string_ci_lt_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_string2_args(sc, sc->string_ci_lt_symbol, p1, p2);
  return(scheme_strcasecmp(p1, p2) == -1);
}

static bool string_ci_leq_b_direct(s7_pointer p1, s7_pointer p2) {return(scheme_strcasecmp(p1, p2) != 1);}
static bool string_ci_leq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_string2_args(sc, sc->string_ci_leq_symbol, p1, p2);
  return(scheme_strcasecmp(p1, p2) != 1);
}

static bool string_ci_gt_b_direct(s7_pointer p1, s7_pointer p2) {return(scheme_strcasecmp(p1, p2) == 1);}
static bool string_ci_gt_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_string2_args(sc, sc->string_ci_gt_symbol, p1, p2);
  return(scheme_strcasecmp(p1, p2) == 1);
}

static bool string_ci_geq_b_direct(s7_pointer p1, s7_pointer p2) {return(scheme_strcasecmp(p1, p2) != -1);}
static bool string_ci_geq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_string2_args(sc, sc->string_ci_geq_symbol, p1, p2);
  return(scheme_strcasecmp(p1, p2) != -1);
}

static bool string_ci_eq_b_direct(s7_pointer p1, s7_pointer p2) {return(scheme_strcasecmp(p1, p2) == 0);}
static bool string_ci_eq_b_7pp(s7_scheme *sc, s7_pointer p1, s7_pointer p2)
{
  check_string2_args(sc, sc->string_ci_eq_symbol, p1, p2);
  return(scheme_strcasecmp(p1, p2) == 0);
}

#endif /* pure s7 */


static s7_pointer g_string_fill_1(s7_scheme *sc, s7_pointer caller, s7_pointer args)
{
  s7_pointer x, chr;
  s7_int start = 0, end;
  x = car(args);

  if (!is_string(x))
    return(method_or_bust(sc, x, sc->string_fill_symbol, args, T_STRING, 1)); /* not two methods here */
  if (is_immutable_string(x))
    return(immutable_object_error(sc, set_elist_3(sc, immutable_error_string, caller, x)));

  chr = cadr(args);
  if (!s7_is_character(chr))
    {
      s7_pointer p;
      p = check_value_slot(sc, chr);
      if (!s7_is_character(p))
	return(wrong_type_argument(sc, caller, 2, chr, T_CHARACTER));
      chr = p;
    }

  end = string_length(x);
  if (!is_null(cddr(args)))
    {
      s7_pointer p;
      p = start_and_end(sc, caller, cddr(args), 3, &start, &end);
      if (p != sc->gc_nil)
	return(p);
      if (start == end) return(chr);
    }
  if (end == 0) return(chr);

  if ((int)(character(chr)) == 0)
    memclr((void *)(string_value(x) + start), end - start);
  else local_memset((void *)(string_value(x) + start), (int32_t)character(chr), end - start);

  return(chr);
}

#if (!WITH_PURE_S7)
static s7_pointer g_string_fill(s7_scheme *sc, s7_pointer args)
{
  #define H_string_fill "(string-fill! str chr start end) fills the string str with the character chr"
  #define Q_string_fill s7_make_signature(sc, 5, s7_make_signature(sc, 2, sc->is_char_symbol, sc->is_integer_symbol), sc->is_string_symbol, sc->is_char_symbol, sc->is_integer_symbol, sc->is_integer_symbol)
  return(g_string_fill_1(sc, sc->string_fill_symbol, args));
}
#endif

static s7_pointer g_string_1(s7_scheme *sc, s7_pointer args, s7_pointer sym)
{
  int32_t i, len;
  s7_pointer x, newstr;
  char *str;

  /* get length for new string and check arg types */
  for (len = 0, x = args; is_not_null(x); len++, x = cdr(x))
    {
      s7_pointer p;
      p = car(x);
      if (!s7_is_character(p))
	{
	  if (has_methods(p))
	    {
	      s7_pointer func;
	      func = find_method(sc, find_let(sc, p), sym);
	      if (func != sc->undefined)
		{
		  s7_pointer y;
		  if (len == 0)
		    return(s7_apply_function(sc, func, args));
		  newstr = make_empty_string(sc, len, 0);
		  str = string_value(newstr);
		  for (i = 0, y = args; y != x; i++, y = cdr(y))
		    str[i] = character(car(y));
		  return(g_string_append_1(sc, set_plist_2(sc, newstr, s7_apply_function(sc, func, x)), sym));
		}
	    }
	  return(wrong_type_argument(sc, sym, len + 1, car(x), T_CHARACTER));
	}
    }
  newstr = make_empty_string(sc, len, 0);
  str = string_value(newstr);
  for (i = 0, x = args; is_not_null(x); i++, x = cdr(x))
    str[i] = character(car(x));

  return(newstr);
}

/* -------------------------------- string -------------------------------- */
static s7_pointer g_string(s7_scheme *sc, s7_pointer args)
{
  #define H_string "(string chr...) appends all its character arguments into one string"
  #define Q_string s7_make_circular_signature(sc, 1, 2, sc->is_string_symbol, sc->is_char_symbol)

  if (is_null(args))
    return(make_string_with_length(sc, "", 0));
  return(g_string_1(sc, args, sc->string_symbol));
}


#if (!WITH_PURE_S7)
static s7_pointer g_list_to_string(s7_scheme *sc, s7_pointer args)
{
  #define H_list_to_string "(list->string lst) appends all the list's characters into one string; (apply string lst)"
  #define Q_list_to_string s7_make_signature(sc, 2, sc->is_string_symbol, sc->is_proper_list_symbol)

  if (is_null(car(args)))
    return(make_string_with_length(sc, "", 0));

  if (!s7_is_proper_list(sc, car(args)))
    return(method_or_bust_with_type_one_arg(sc, car(args), sc->list_to_string_symbol, args,
				     wrap_string(sc, "a (proper, non-circular) list of characters", 43)));
  return(g_string_1(sc, car(args), sc->list_to_string_symbol));
}
#endif

static s7_pointer s7_string_to_list(s7_scheme *sc, const char *str, s7_int len)
{
  s7_int i;
  s7_pointer result;

  if (len == 0)
    return(sc->nil);
  if (len >= (sc->free_heap_top - sc->free_heap))
    {
      gc(sc);
      while (len >= (sc->free_heap_top - sc->free_heap))
	resize_heap(sc);
    }

  sc->v = sc->nil;
  for (i = len - 1; i >= 0; i--)
    sc->v = cons_unchecked(sc, s7_make_character(sc, ((uint8_t)str[i])), sc->v);
  result = sc->v;
  sc->v = sc->nil;
  return(result);
}

#if (!WITH_PURE_S7)
static s7_pointer g_string_to_list(s7_scheme *sc, s7_pointer args)
{
  #define H_string_to_list "(string->list str start end) returns the elements of the string str in a list; (map values str)"
  #define Q_string_to_list s7_make_circular_signature(sc, 2, 3, sc->is_proper_list_symbol, sc->is_string_symbol, sc->is_integer_symbol)

  s7_int i, start = 0, end;
  s7_pointer p, str;

  str = car(args);
  if (!is_string(str))
    return(method_or_bust_one_arg(sc, str, sc->string_to_list_symbol, args, T_STRING));

  end = string_length(str);
  if (!is_null(cdr(args)))
    {
      p = start_and_end(sc, sc->string_to_list_symbol, cdr(args), 2, &start, &end);
      if (p != sc->gc_nil) return(p);
      if (start == end) return(sc->nil);
    }
  else
    {
      if (end == 0) return(sc->nil);
    }
  if ((end - start) > sc->max_list_length)
    return(out_of_range(sc, sc->string_to_list_symbol, small_int(1), car(args), its_too_large_string));

  if ((start == 0) && (end == string_length(str)))
    return(s7_string_to_list(sc, string_value(str), string_length(str)));

  sc->w = sc->nil;
  for (i = end - 1; i >= start; i--)
    sc->w = cons(sc, s7_make_character(sc, ((uint8_t)string_value(str)[i])), sc->w);

  p = sc->w;
  sc->w = sc->nil;
  return(p);
}
#endif


/* -------------------------------- port-closed? -------------------------------- */
static s7_pointer g_is_port_closed(s7_scheme *sc, s7_pointer args)
{
  #define H_is_port_closed "(port-closed? p) returns #t if the port p is closed."
  #define Q_is_port_closed s7_make_signature(sc, 2, sc->is_boolean_symbol, s7_make_signature(sc, 3, sc->is_input_port_symbol, sc->is_output_port_symbol, sc->not_symbol))
  s7_pointer x;

  x = car(args);
  if ((is_input_port(x)) || (is_output_port(x)))
    return(make_boolean(sc, port_is_closed(x)));
  if ((x == sc->output_port) && (x == sc->F))
    return(sc->F);
  return(method_or_bust_with_type_one_arg(sc, x, sc->is_port_closed_symbol, args, wrap_string(sc, "a port", 6)));
}

static bool is_port_closed_b_7p(s7_scheme *sc, s7_pointer x)
{
  if ((is_input_port(x)) || (is_output_port(x)))
    return(port_is_closed(x));
  if ((x == sc->output_port) && (x == sc->F))
    return(false);
  simple_wrong_type_argument_with_type(sc, sc->is_port_closed_symbol, x, wrap_string(sc, "a port", 6));
  return(false);
}

/* -------------------------------- port-line-number -------------------------------- */
static s7_pointer c_port_line_number(s7_scheme *sc, s7_pointer x)
{
  if ((!(is_input_port(x))) ||
      (port_is_closed(x)))
    return(method_or_bust_with_type_one_arg(sc, x, sc->port_line_number_symbol, list_1(sc, x), an_input_port_string));
  return(make_integer(sc, port_line_number(x)));
}

static s7_pointer g_port_line_number(s7_scheme *sc, s7_pointer args)
{
  #define H_port_line_number "(port-line-number input-file-port) returns the current read line number of port"
  #define Q_port_line_number s7_make_signature(sc, 2, sc->is_integer_symbol, sc->is_input_port_symbol)
  return(c_port_line_number(sc, (is_null(args)) ? sc->input_port : car(args)));
}

s7_int s7_port_line_number(s7_scheme *sc, s7_pointer p)
{
  if (!(is_input_port(p)))
    simple_wrong_type_argument(sc, sc->port_line_number_symbol, p, T_INPUT_PORT);
  return(port_line_number(p));
}

static s7_int port_line_number_i_7p(s7_scheme *sc, s7_pointer p)
{
  return(s7_port_line_number(sc, p));
}

static s7_pointer g_set_port_line_number(s7_scheme *sc, s7_pointer args)
{
  s7_pointer p, line;

  if ((is_null(car(args))) ||
      ((is_null(cdr(args))) && (is_integer(car(args)))))
    p = sc->input_port;
  else
    {
      p = car(args);
      if (!(is_input_port(p)))
	return(s7_wrong_type_arg_error(sc, "set! port-line-number", 1, p, "an input port"));
    }

  line = (is_null(cdr(args)) ? car(args) : cadr(args));
  if (!is_integer(line))
    return(s7_wrong_type_arg_error(sc, "set! port-line-number", 2, line, "an integer"));
  port_line_number(p) = integer(line);
  return(line);
}

/* -------------------------------- port-filename -------------------------------- */
const char *s7_port_filename(s7_scheme *sc, s7_pointer x)
{
  if (((is_input_port(x)) ||
       (is_output_port(x))) &&
      (!port_is_closed(x)))
    return(port_filename(x));
  return(NULL);
}

static s7_pointer c_port_filename(s7_scheme *sc, s7_pointer x)
{
  if (((is_input_port(x)) ||
       (is_output_port(x))) &&
      (!port_is_closed(x)))
    {
      if (port_filename(x))
	return(make_string_with_length(sc, port_filename(x), port_filename_length(x))); /* not wrapper here! */
      return(make_string_with_length(sc, "", 0));
      /* otherwise (eval-string (port-filename)) and (string->symbol (port-filename)) segfault */
    }
  return(method_or_bust_with_type_one_arg(sc, x, sc->port_filename_symbol, list_1(sc, x), an_open_port_string));
}

static s7_pointer g_port_filename(s7_scheme *sc, s7_pointer args)
{
  #define H_port_filename "(port-filename file-port) returns the filename associated with port"
  #define Q_port_filename s7_make_signature(sc, 2, sc->is_string_symbol, s7_make_signature(sc, 2, sc->is_input_port_symbol, sc->is_output_port_symbol))
  return(c_port_filename(sc, (is_null(args)) ? sc->input_port : car(args)));
}

/* -------------------------------- input-port? -------------------------------- */
bool s7_is_input_port(s7_scheme *sc, s7_pointer p) {return(is_input_port(p));}
static bool is_input_port_b(s7_pointer p) {return(is_input_port(p));}

static s7_pointer g_is_input_port(s7_scheme *sc, s7_pointer args)
{
  #define H_is_input_port "(input-port? p) returns #t if p is an input port"
  #define Q_is_input_port sc->pl_bt
  check_boolean_method(sc, is_input_port, sc->is_input_port_symbol, args);
}

/* -------------------------------- output-port? -------------------------------- */
bool s7_is_output_port(s7_scheme *sc, s7_pointer p) {return(is_output_port(p));}
static bool is_output_port_b(s7_pointer p) {return(is_output_port(p));}

static s7_pointer g_is_output_port(s7_scheme *sc, s7_pointer args)
{
  #define H_is_output_port "(output-port? p) returns #t if p is an output port"
  #define Q_is_output_port sc->pl_bt
  check_boolean_method(sc, is_output_port, sc->is_output_port_symbol, args);
}

/* -------------------------------- current-input-port -------------------------------- */
s7_pointer s7_current_input_port(s7_scheme *sc) {return(sc->input_port);}

static s7_pointer g_current_input_port(s7_scheme *sc, s7_pointer args)
{
  #define H_current_input_port "(current-input-port) returns the current input port"
  #define Q_current_input_port s7_make_signature(sc, 1, sc->is_input_port_symbol)
  return(sc->input_port);
}

static s7_pointer g_set_current_input_port(s7_scheme *sc, s7_pointer args)
{
  #define H_set_current_input_port "(set-current-input-port port) sets the current-input port to port and returns the previous value of the input port"
  #define Q_set_current_input_port s7_make_signature(sc, 2, sc->is_input_port_symbol, sc->is_input_port_symbol)

  s7_pointer old_port, port;
  old_port = sc->input_port;
  port = car(args);
  if ((is_input_port(port)) &&
      (!port_is_closed(port)))
    sc->input_port = port;
  else
    {
      check_method(sc, port, sc->set_current_input_port_symbol, args);
      return(s7_wrong_type_arg_error(sc, "set-current-input-port", 0, port, "an open input port"));
    }
  return(old_port);
}

s7_pointer s7_set_current_input_port(s7_scheme *sc, s7_pointer port)
{
  s7_pointer old_port;
  old_port = sc->input_port;
  sc->input_port = port;
  return(old_port);
}

/* -------------------------------- current-output-port -------------------------------- */
s7_pointer s7_current_output_port(s7_scheme *sc) {return(sc->output_port);}

s7_pointer s7_set_current_output_port(s7_scheme *sc, s7_pointer port)
{
  s7_pointer old_port;
  old_port = sc->output_port;
  sc->output_port = port;
  return(old_port);
}

static s7_pointer g_current_output_port(s7_scheme *sc, s7_pointer args)
{
  #define H_current_output_port "(current-output-port) returns the current output port"
  #define Q_current_output_port s7_make_signature(sc, 1, sc->is_output_port_symbol)
  return(sc->output_port);
}

static s7_pointer g_set_current_output_port(s7_scheme *sc, s7_pointer args)
{
  #define H_set_current_output_port "(set-current-output-port port) sets the current-output port to port and returns the previous value of the output port"
  #define Q_set_current_output_port s7_make_signature(sc, 2, sc->is_output_port_symbol, s7_make_signature(sc, 2, sc->is_output_port_symbol, sc->not_symbol))

  s7_pointer old_port, port;
  old_port = sc->output_port;
  port = car(args);
  if (((is_output_port(port)) &&
       (!port_is_closed(port))) ||
      (port == sc->F))
    sc->output_port = port;
  else
    {
      check_method(sc, port, sc->set_current_output_port_symbol, args);
      return(s7_wrong_type_arg_error(sc, "set-current-output-port", 0, port, "an open output port"));
    }
  return(old_port);
}

/* -------------------------------- current-error-port -------------------------------- */
s7_pointer s7_current_error_port(s7_scheme *sc) {return(sc->error_port);}

s7_pointer s7_set_current_error_port(s7_scheme *sc, s7_pointer port)
{
  s7_pointer old_port;
  old_port = sc->error_port;
  sc->error_port = port;
  return(old_port);
}

static s7_pointer g_current_error_port(s7_scheme *sc, s7_pointer args)
{
  #define H_current_error_port "(current-error-port) returns the current error port"
  #define Q_current_error_port s7_make_signature(sc, 1, sc->is_output_port_symbol)
  return(sc->error_port);
}

static s7_pointer g_set_current_error_port(s7_scheme *sc, s7_pointer args)
{
  #define H_set_current_error_port "(set-current-error-port port) sets the current-error port to port and returns the previous value of the error port"
  #define Q_set_current_error_port s7_make_signature(sc, 2, sc->is_output_port_symbol, s7_make_signature(sc, 2, sc->is_output_port_symbol, sc->not_symbol))
  s7_pointer old_port, port;

  old_port = sc->error_port;
  port = car(args);
  if (((is_output_port(port)) &&
       (!port_is_closed(port))) ||
      (port == sc->F))
    sc->error_port = port;
  else
    {
      check_method(sc, port, sc->set_current_error_port_symbol, args);
      return(s7_wrong_type_arg_error(sc, "set-current-error-port", 0, port, "an open output port"));
    }
  return(old_port);
}

/* -------------------------------- char-ready? -------------------------------- */
#if (!WITH_PURE_S7)
static s7_pointer g_is_char_ready(s7_scheme *sc, s7_pointer args)
{
  #define H_is_char_ready "(char-ready? (port (current-input-port))) returns #t if a character is ready for input on the given port"
  #define Q_is_char_ready s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_input_port_symbol)
  if (is_not_null(args))
    {
      s7_pointer pt = car(args);
      if (!is_input_port(pt))
	return(method_or_bust_with_type_one_arg(sc, pt, sc->is_char_ready_symbol, args, an_input_port_string));
      if (port_is_closed(pt))
	return(simple_wrong_type_argument_with_type(sc, sc->is_char_ready_symbol, pt, an_open_port_string));

      if (is_function_port(pt))
	return((*(port_input_function(pt)))(sc, S7_IS_CHAR_READY, pt));
      return(make_boolean(sc, is_string_port(pt)));
    }
  return(make_boolean(sc, (is_input_port(sc->input_port)) && (is_string_port(sc->input_port))));
}

#endif

/* -------- ports -------- */
static int32_t closed_port_read_char(s7_scheme *sc, s7_pointer port);
static s7_pointer closed_port_read_line(s7_scheme *sc, s7_pointer port, bool with_eol, bool copied);
static void closed_port_write_char(s7_scheme *sc, uint8_t c, s7_pointer port);
static void closed_port_write_string(s7_scheme *sc, const char *str, s7_int len, s7_pointer port);
static void closed_port_display(s7_scheme *sc, const char *s, s7_pointer port);

void s7_close_input_port(s7_scheme *sc, s7_pointer p)
{
  if ((is_immutable_port(p)) ||
      ((is_input_port(p)) && (port_is_closed(p))))
    {
#if S7_DEBUGGING
      if (port_needs_free(p))
	fprintf(stderr, "closed input needs free\n");
#endif
      return;
    }
  if (port_filename(p))
    {
      /* for string ports, this is the original input file name */
      liberate(sc, port_filename_block(p));
      port_filename(p) = NULL;
    }

  if (is_string_port(p))
    {
      if (port_needs_unprotect(p))
	{
	  s7_gc_unprotect_at(sc, port_gc_loc(p));
	  port_needs_unprotect(p) = false;
	}
    }
  else
    {
      if (is_file_port(p))
	{
	  if (port_file(p))
	    {
	      fclose(port_file(p));
	      port_file(p) = NULL;
	    }
	}
    }
  if (port_needs_free(p))
    {
      if (port_data(p))
	{
	  liberate(sc, port_data_block(p));
	  port_data_block(p) = NULL;
	  port_data(p) = NULL;
	  port_data_size(p) = 0;
	}
      port_needs_free(p) = false;
    }

  port_read_character(p) = closed_port_read_char;
  port_read_line(p) = closed_port_read_line;
  port_write_character(p) = closed_port_write_char;
  port_write_string(p) = closed_port_write_string;
  port_display(p) = closed_port_display;
  port_set_closed(p, true);
  port_position(p) = 0;
}

/* -------------------------------- close-input-port -------------------------------- */
static s7_pointer g_close_input_port(s7_scheme *sc, s7_pointer args)
{
  s7_pointer pt;
  #define H_close_input_port "(close-input-port port) closes the port"
  #define Q_close_input_port s7_make_signature(sc, 2, sc->is_unspecified_symbol, sc->is_input_port_symbol)

  pt = car(args);
  if (!is_input_port(pt))
    return(method_or_bust_with_type_one_arg(sc, pt, sc->close_input_port_symbol, set_plist_1(sc, pt), an_input_port_string));
  if ((!is_immutable_port(pt)) &&  /* (close-input-port *stdin*) */
      (!is_loader_port(pt)))       /* top-level unmatched (close-input-port (current-input-port)) should not clobber the loader's input port */
    s7_close_input_port(sc, pt);
  return(sc->unspecified);
}

/* -------------------------------- flush-output-port -------------------------------- */
void s7_flush_output_port(s7_scheme *sc, s7_pointer p)
{
  if ((!is_output_port(p)) ||
      (!is_file_port(p)) ||
      (port_is_closed(p)) ||
      (p == sc->F))
    return;

  if (port_file(p))
    {
      if (port_position(p) > 0)
	{
	  if (fwrite((void *)(port_data(p)), 1, port_position(p), port_file(p)) != (size_t)port_position(p))
	    s7_warn(sc, 64, "fwrite trouble in flush-output-port\n");
	  port_position(p) = 0;
	}
      fflush(port_file(p));
    }
}

static s7_pointer g_flush_output_port(s7_scheme *sc, s7_pointer args)
{
  #define H_flush_output_port "(flush-output-port port) flushes the port"
  #define Q_flush_output_port s7_make_signature(sc, 2, sc->T, s7_make_signature(sc, 2, sc->is_output_port_symbol, sc->not_symbol))
  s7_pointer pt;

  if (is_null(args))
    pt = sc->output_port;
  else pt = car(args);

  if (!is_output_port(pt))
    {
      if (pt == sc->F) return(pt);
      return(method_or_bust_with_type_one_arg(sc, pt, sc->flush_output_port_symbol, args, an_output_port_string));
    }
  s7_flush_output_port(sc, pt);
  return(pt);
}

/* -------------------------------- close-output-port -------------------------------- */
static void close_output_port(s7_scheme *sc, s7_pointer p)
{
  if (is_file_port(p))
    {
      if (port_filename(p)) /* only a file output port has a filename(?) */
	{
	  liberate(sc, port_filename_block(p));
	  port_filename(p) = NULL;
	  port_filename_length(p) = 0;
	}
      if (port_file(p))
	{
	  if (port_position(p) > 0)
	    {
	      if (fwrite((void *)(port_data(p)), 1, port_position(p), port_file(p)) != (size_t)port_position(p))
		s7_warn(sc, 64, "fwrite trouble in close-output-port\n");
	    }
	  fflush(port_file(p));
	  fclose(port_file(p));
	  port_file(p) = NULL;
	}
    }
  else
    {
      if (is_string_port(p))
	{
	  if (port_data(p))
	    {
	      port_data(p) = NULL;
	      port_data_size(p) = 0;
	    }
	}
    }
  port_read_character(p) = closed_port_read_char;
  port_read_line(p) = closed_port_read_line;
  port_write_character(p) = closed_port_write_char;
  port_write_string(p) = closed_port_write_string;
  port_display(p) = closed_port_display;
  port_set_closed(p, true);
  port_position(p) = 0;
}

void s7_close_output_port(s7_scheme *sc, s7_pointer p)
{
  if ((is_immutable_port(p)) ||
      ((is_output_port(p)) && (port_is_closed(p))) ||
      (p == sc->F))
    return;
  close_output_port(sc, p);
}

static s7_pointer g_close_output_port(s7_scheme *sc, s7_pointer args)
{
  s7_pointer pt;
  #define H_close_output_port "(close-output-port port) closes the port"
  #define Q_close_output_port s7_make_signature(sc, 2, sc->is_unspecified_symbol, s7_make_signature(sc, 2, sc->is_output_port_symbol, sc->not_symbol))

  pt = car(args);
  if (!is_output_port(pt))
    {
      if (pt == sc->F) return(sc->unspecified);
      return(method_or_bust_with_type_one_arg(sc, pt, sc->close_output_port_symbol, set_plist_1(sc, pt), an_output_port_string));
    }
  if (!(is_immutable_port(pt)))
    s7_close_output_port(sc, pt);
  return(sc->unspecified);
}


/* -------- read character functions -------- */

static int32_t file_read_char(s7_scheme *sc, s7_pointer port)
{
  return(fgetc(port_file(port)));
}


static int32_t function_read_char(s7_scheme *sc, s7_pointer port)
{
  return(character((*(port_input_function(port)))(sc, S7_READ_CHAR, port)));
}


static int32_t string_read_char(s7_scheme *sc, s7_pointer port)
{
  if (port_data_size(port) <= port_position(port)) /* port_string_length is 0 if no port string */
    return(EOF);
  return((uint8_t)port_data(port)[port_position(port)++]);
}


static int32_t output_read_char(s7_scheme *sc, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->read_char_symbol, port, an_input_port_string);
  return(0);
}


static int32_t closed_port_read_char(s7_scheme *sc, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->read_char_symbol, port, an_open_port_string);
  return(0);
}


/* -------- read line functions -------- */

static s7_pointer output_read_line(s7_scheme *sc, s7_pointer port, bool with_eol, bool copied)
{
  return(simple_wrong_type_argument_with_type(sc, sc->read_line_symbol, port, an_input_port_string));
}


static s7_pointer closed_port_read_line(s7_scheme *sc, s7_pointer port, bool with_eol, bool copied)
{
  return(simple_wrong_type_argument_with_type(sc, sc->read_line_symbol, port, an_open_port_string));
}


static s7_pointer function_read_line(s7_scheme *sc, s7_pointer port, bool with_eol, bool copied)
{
  return((*(port_input_function(port)))(sc, S7_READ_LINE, port));
}


static s7_pointer stdin_read_line(s7_scheme *sc, s7_pointer port, bool with_eol, bool copied)
{
  if (!sc->read_line_buf)
    {
      sc->read_line_buf_size = 1024;
      sc->read_line_buf = (char *)malloc(sc->read_line_buf_size * sizeof(char));
    }

  if (fgets(sc->read_line_buf, sc->read_line_buf_size, stdin))
    return(s7_make_string(sc, sc->read_line_buf)); /* fgets adds the trailing '\0' */
  return(make_string_with_length(sc, NULL, 0));
}


static s7_pointer file_read_line(s7_scheme *sc, s7_pointer port, bool with_eol, bool copied)
{
  char *buf;
  s7_int read_size, previous_size = 0;

  if (!sc->read_line_buf)
    {
      sc->read_line_buf_size = 1024;
      sc->read_line_buf = (char *)malloc(sc->read_line_buf_size * sizeof(char));
    }

  buf = sc->read_line_buf;
  read_size = sc->read_line_buf_size;

  while (true)
    {
      char *p, *rtn;
      size_t len;

      p = fgets(buf, read_size, port_file(port));
      if (!p)
	return(eof_object);

      rtn = strchr(buf, (int)'\n');
      if (rtn)
	{
	  port_line_number(port)++;
	  return(make_string_with_length(sc, sc->read_line_buf, (with_eol) ? (previous_size + rtn - p + 1) : (previous_size + rtn - p)));
	}
      /* if no newline, then either at eof or need bigger buffer */
      len = strlen(sc->read_line_buf);

      if ((len + 1) < (size_t)sc->read_line_buf_size)
	return(make_string_with_length(sc, sc->read_line_buf, len));

      previous_size = sc->read_line_buf_size;
      sc->read_line_buf_size *= 2;
      sc->read_line_buf = (char *)realloc(sc->read_line_buf, sc->read_line_buf_size * sizeof(char));
      read_size = previous_size;
      previous_size -= 1;
      buf = (char *)(sc->read_line_buf + previous_size);
    }
  return(eof_object);
}


static s7_pointer string_read_line(s7_scheme *sc, s7_pointer port, bool with_eol, bool copied)
{
  s7_int i, port_start;
  uint8_t *port_str, *cur, *start;

  port_start = port_position(port);
  port_str = port_data(port);
  start = (uint8_t *)(port_str + port_start);

  cur = (uint8_t *)strchr((const char *)start, (int)'\n'); /* this can run off the end making valgrind unhappy, but I think it's innocuous */
  if (cur)
      {
	port_line_number(port)++;
	i = cur - port_str;
	port_position(port) = i + 1;
	if (copied)
	  return(make_string_with_length(sc, (const char *)start, ((with_eol) ? i + 1 : i) - port_start));
	return(wrap_string(sc, (char *)start, ((with_eol) ? i + 1 : i) - port_start));
      }
  i = port_data_size(port);
  port_position(port) = i;
  if (i <= port_start)         /* the < part can happen -- if not caught we try to create a string of length -1 -> segfault */
    return(eof_object);

  if (copied)
    return(make_string_with_length(sc, (const char *)start, i - port_start));
  return(wrap_string(sc, (char *)start, i - port_start));
}


/* -------- write character functions -------- */

static void resize_port_data(s7_scheme *sc, s7_pointer pt, s7_int new_size)
{
  s7_int loc;
  block_t *nb;

  loc = port_data_size(pt);
  if (new_size < loc) return;

  nb = reallocate(sc, port_data_block(pt), new_size);
  port_data_block(pt) = nb;
  port_data(pt) = (uint8_t *)(block_data(nb));
  port_data_size(pt) = new_size;
}

static void string_write_char_resized(s7_scheme *sc, uint8_t c, s7_pointer pt)
{
  /* this division looks repetitive, but it is much faster */
  resize_port_data(sc, pt, port_data_size(pt) * 2);
  port_data(pt)[port_position(pt)++] = c;
}

static void string_write_char(s7_scheme *sc, uint8_t c, s7_pointer pt)
{
  if (port_position(pt) < port_data_size(pt))
    port_data(pt)[port_position(pt)++] = c;
  else string_write_char_resized(sc, c, pt);
}

static void stdout_write_char(s7_scheme *sc, uint8_t c, s7_pointer port)
{
  fputc(c, stdout);
}

static void stderr_write_char(s7_scheme *sc, uint8_t c, s7_pointer port)
{
  fputc(c, stderr);
}

static void function_write_char(s7_scheme *sc, uint8_t c, s7_pointer port)
{
  (*(port_output_function(port)))(sc, c, port);
}


#define PORT_DATA_SIZE 256
static void file_write_char(s7_scheme *sc, uint8_t c, s7_pointer port)
{
  if (port_position(port) == PORT_DATA_SIZE)
    {
      if (fwrite((void *)(port_data(port)), 1, PORT_DATA_SIZE, port_file(port)) != PORT_DATA_SIZE)
	s7_warn(sc, 64, "fwrite trouble during write-char\n");
      port_position(port) = 0;
    }
  port_data(port)[port_position(port)++] = c;
}


static void input_write_char(s7_scheme *sc, uint8_t c, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->write_char_symbol, port, an_output_port_string);
}


static void closed_port_write_char(s7_scheme *sc, uint8_t c, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->write_char_symbol, port, an_open_port_string);
}


/* -------- write string functions -------- */

static void input_write_string(s7_scheme *sc, const char *str, s7_int len, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->write_symbol, port, an_output_port_string);
}

static void closed_port_write_string(s7_scheme *sc, const char *str, s7_int len, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->write_symbol, port, an_open_port_string);
}


static void input_display(s7_scheme *sc, const char *s, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->display_symbol, port, an_output_port_string);
}

static void closed_port_display(s7_scheme *sc, const char *s, s7_pointer port)
{
  simple_wrong_type_argument_with_type(sc, sc->display_symbol, port, an_open_port_string);
}


static void stdout_write_string(s7_scheme *sc, const char *str, s7_int len, s7_pointer port)
{
  if (str[len] == '\0')
    fputs(str, stdout);
  else
    {
      s7_int i;
      for (i = 0; i < len; i++)
	fputc(str[i], stdout);
    }
}

static void stderr_write_string(s7_scheme *sc, const char *str, s7_int len, s7_pointer port)
{
  if (str[len] == '\0')
    fputs(str, stderr);
  else
    {
      s7_int i;
      for (i = 0; i < len; i++)
	fputc(str[i], stderr);
    }
}


static void string_write_string_resized(s7_scheme *sc, const char *str, s7_int len, s7_pointer pt)
{
  s7_int new_len;  /* len is known to be non-zero, str may not be 0-terminated */
  new_len = port_position(pt) + len;
  resize_port_data(sc, pt, new_len * 2);
  memcpy((void *)(port_data(pt) + port_position(pt)), (void *)str, len);
  port_position(pt) = new_len;
}

static void string_write_string(s7_scheme *sc, const char *str, s7_int len, s7_pointer pt)
{
#if S7_DEBUGGING
  if (len == 0) {fprintf(stderr, "string_write_string len == 0\n"); abort();}
#endif
  if (port_position(pt) + len < port_data_size(pt))
    {
      memcpy((void *)(port_data(pt) + port_position(pt)), (void *)str, len);
      /* memcpy is much faster than the equivalent while loop, and faster than using the 4-bytes-at-a-time shuffle */
      port_position(pt) += len;
    }
  else string_write_string_resized(sc, str, len, pt);
}


static void file_write_string(s7_scheme *sc, const char *str, s7_int len, s7_pointer pt)
{
  s7_int new_len;
  new_len = port_position(pt) + len;
  if (new_len >= PORT_DATA_SIZE)
    {
      if (port_position(pt) > 0)
	{
	  if (fwrite((void *)(port_data(pt)), 1, port_position(pt), port_file(pt)) != (size_t)port_position(pt))
	    s7_warn(sc, 64, "fwrite trouble in write-string\n");
	  port_position(pt) = 0;
	}
      if (fwrite((void *)str, 1, len, port_file(pt)) != (size_t)len)
	s7_warn(sc, 64, "fwrite trouble in write-string\n");
    }
  else
    {
      memcpy((void *)(port_data(pt) + port_position(pt)), (void *)str, len);
      port_position(pt) = new_len;
    }
}


static void string_display(s7_scheme *sc, const char *s, s7_pointer port)
{
  if (s)
    string_write_string(sc, s, safe_strlen(s), port);
}

static void file_display(s7_scheme *sc, const char *s, s7_pointer port)
{
  if (s)
    {
      if (port_position(port) > 0)
	{
	  if (fwrite((void *)(port_data(port)), 1, port_position(port), port_file(port)) != (size_t)port_position(port))
	    s7_warn(sc, 64, "fwrite trouble in display\n");
	  port_position(port) = 0;
	}
      if (fputs(s, port_file(port)) == EOF)
	s7_warn(sc, 64, "write to %s: %s\n", port_filename(port), strerror(errno));
    }
}

static void function_display(s7_scheme *sc, const char *s, s7_pointer port)
{
  if (s)
    {
      for (; *s; s++)
	(*(port_output_function(port)))(sc, *s, port);
    }
}

static void function_write_string(s7_scheme *sc, const char *str, s7_int len, s7_pointer pt)
{
  s7_int i;
  for (i = 0; i < len; i++)
    (*(port_output_function(pt)))(sc, str[i], pt);
}


static void stdout_display(s7_scheme *sc, const char *s, s7_pointer port)
{
  if (s) fputs(s, stdout);
}

static void stderr_display(s7_scheme *sc, const char *s, s7_pointer port)
{
  if (s) fputs(s, stderr);
}

/* -------------------------------- write-string -------------------------------- */
static s7_pointer g_write_string(s7_scheme *sc, s7_pointer args)
{
  #define H_write_string "(write-string str port start end) writes str to port."
  #define Q_write_string s7_make_circular_signature(sc, 3, 4, sc->is_string_symbol, sc->is_string_symbol, s7_make_signature(sc, 2, sc->is_output_port_symbol, sc->not_symbol), sc->is_integer_symbol)
  s7_pointer str, port;
  s7_int start = 0, end;

  str = car(args);
  if (!is_string(str))
    return(method_or_bust(sc, str, sc->write_string_symbol, args, T_STRING, 1));

  end = string_length(str);
  if (!is_null(cdr(args)))
    {
      s7_pointer inds;
      port = cadr(args);
      inds = cddr(args);
      if (!is_null(inds))
	{
	  s7_pointer p;
	  p = start_and_end(sc, sc->write_string_symbol, inds, 3, &start, &end);
	  if (p != sc->gc_nil) return(p);
	}
    }
  else port = sc->output_port;
  if (!is_output_port(port))
    {
      if (port == sc->F)
	{
	  s7_int len;
	  if ((start == 0) && (end == string_length(str)))
	    return(str);
	  len = (s7_int)(end - start);
	  return(make_string_with_length(sc, (char *)(string_value(str) + start), len));
	}
      return(method_or_bust_with_type(sc, port, sc->write_string_symbol, args, an_output_port_string, 2));
    }
  if (start == end)
    return(str);
  port_write_string(port)(sc, (char *)(string_value(str) + start), (end - start), port);
  return(str);
}


/* -------- skip to newline readers -------- */

static token_t file_read_semicolon(s7_scheme *sc, s7_pointer pt)
{
  int32_t c;
  do (c = fgetc(port_file(pt))); while ((c != '\n') && (c != EOF));
  port_line_number(pt)++;
  if (c == EOF)
    return(TOKEN_EOF);
  return(token(sc));
}

static token_t string_read_semicolon(s7_scheme *sc, s7_pointer pt)
{
  const char *orig_str, *str;
  str = (const char *)(port_data(pt) + port_position(pt));
  orig_str = strchr(str, (int)'\n');
  if (!orig_str)
    {
      port_position(pt) = port_data_size(pt);
      return(TOKEN_EOF);
    }
  port_position(pt) += (orig_str - str + 1); /* + 1 because strchr leaves orig_str pointing at the newline */
  port_line_number(pt)++;
  return(token(sc));
}


/* -------- white space readers -------- */

static int32_t file_read_white_space(s7_scheme *sc, s7_pointer port)
{
  int32_t c;
  while (is_white_space(c = fgetc(port_file(port))))
    if (c == '\n')
      port_line_number(port)++;
  return(c);
}

static int32_t terminated_string_read_white_space(s7_scheme *sc, s7_pointer pt)
{
  const uint8_t *str;
  uint8_t c;
  /* here we know we have null termination and white_space[#\null] is false. */

  str = (const uint8_t *)(port_data(pt) + port_position(pt));

  while (white_space[c = *str++]) /* (let ((a 1)) a) -- 255 is not -1 = EOF */
    if (c == '\n')
      port_line_number(pt)++;
  if (c)
    port_position(pt) = str - port_data(pt);
  else port_position(pt) = port_data_size(pt);
  return((int32_t)c);
}


/* name (alphanumeric token) readers */

static void resize_strbuf(s7_scheme *sc, s7_int needed_size)
{
  s7_int i, old_size;
  old_size = sc->strbuf_size;
  while (sc->strbuf_size <= needed_size) sc->strbuf_size *= 2;
  sc->strbuf = (char *)realloc(sc->strbuf, sc->strbuf_size * sizeof(char));
  for (i = old_size; i < sc->strbuf_size; i++) sc->strbuf[i] = '\0';
}


static s7_pointer file_read_name_or_sharp(s7_scheme *sc, s7_pointer pt, bool atom_case)
{
  int32_t c;
  s7_int i = 1;
  /* sc->strbuf[0] has the first char of the string we're reading */

  do {
    c = fgetc(port_file(pt)); /* might return EOF */
    if (c == '\n')
      port_line_number(pt)++;

    sc->strbuf[i++] = c;
    if (i >= sc->strbuf_size)
      resize_strbuf(sc, i);
  } while ((c != EOF) && (char_ok_in_a_name[c]));

  if ((i == 2) &&
      (sc->strbuf[0] == '\\'))
    sc->strbuf[2] = '\0';
  else
    {
      if (c != EOF)
	{
	  if (c == '\n')
	    port_line_number(pt)--;
	  ungetc(c, port_file(pt));
	}
      sc->strbuf[i - 1] = '\0';
    }

  if (atom_case)
    return(make_atom(sc, sc->strbuf, BASE_10, SYMBOL_OK, WITH_OVERFLOW_ERROR));

  return(make_sharp_constant(sc, sc->strbuf, WITH_OVERFLOW_ERROR));
}

static s7_pointer file_read_name(s7_scheme *sc, s7_pointer pt)
{
  return(file_read_name_or_sharp(sc, pt, true));
}

static s7_pointer file_read_sharp(s7_scheme *sc, s7_pointer pt)
{
  return(file_read_name_or_sharp(sc, pt, false));
}


static s7_pointer string_read_name_no_free(s7_scheme *sc, s7_pointer pt)
{
  /* sc->strbuf[0] has the first char of the string we're reading */
  s7_pointer result;
  char *str;

  str = (char *)(port_data(pt) + port_position(pt));

  if (char_ok_in_a_name[(uint8_t)*str])
    {
      s7_int k;
      char *orig_str;

      orig_str = (char *)(str - 1);
      str++;
      while (char_ok_in_a_name[(uint8_t)(*str)]) {str++;}
      k = str - orig_str;
      if (*str != 0)
	port_position(pt) += (k - 1);
      else port_position(pt) = port_data_size(pt);

      /* this is equivalent to:
       *    str = strpbrk(str, "(); \"\t\r\n");
       *    if (!str)
       *      {
       *        k = strlen(orig_str);
       *        str = (char *)(orig_str + k);
       *      }
       *    else k = str - orig_str;
       * but slightly faster.
       */

      if (!number_table[(uint8_t)(*orig_str)])
	return(make_symbol_with_length(sc, orig_str, k));

      /* eval_c_string string is a constant so we can't set and unset the token's end char */
      if ((k + 1) >= sc->strbuf_size)
	resize_strbuf(sc, k + 1);

      memcpy((void *)(sc->strbuf), (void *)orig_str, k);
      sc->strbuf[k] = '\0';
      return(make_atom(sc, sc->strbuf, BASE_10, SYMBOL_OK, WITH_OVERFLOW_ERROR));
    }

  result = sc->singletons[(uint8_t)(sc->strbuf[0])];
  if (!result)
    {
      sc->strbuf[1] = '\0';
      result = make_symbol_with_length(sc, sc->strbuf, 1);
      sc->singletons[(uint8_t)(sc->strbuf[0])] = result;
    }
  return(result);
}


static s7_pointer string_read_sharp(s7_scheme *sc, s7_pointer pt)
{
  /* sc->strbuf[0] has the first char of the string we're reading.
   *   since a *#readers* function might want to get further input, we can't mess with the input even when it is otherwise safe
   */
  char *str;

  str = (char *)(port_data(pt) + port_position(pt));

  if (char_ok_in_a_name[(uint8_t)*str])
    {
      s7_int k;
      char * orig_str;
      orig_str = (char *)(str - 1);
      str++;
      while (char_ok_in_a_name[(uint8_t)(*str)]) {str++;}
      k = str - orig_str;
      if (*str != 0)
	port_position(pt) += (k - 1);
      else port_position(pt) += k;

      if ((k + 1) >= sc->strbuf_size)
	resize_strbuf(sc, k + 1);

      memcpy((void *)(sc->strbuf), (void *)orig_str, k);
      sc->strbuf[k] = '\0';
      return(make_sharp_constant(sc, sc->strbuf, WITH_OVERFLOW_ERROR));
    }
  if (sc->strbuf[0] == 'f')
    return(sc->F);
  if (sc->strbuf[0] == 't')
    return(sc->T);
  if (sc->strbuf[0] == '\\')
    {
      /* must be from #\( and friends -- a character that happens to be not ok-in-a-name */
      sc->strbuf[1] = str[0];
      sc->strbuf[2] = '\0';
      port_position(pt)++;
    }
  else sc->strbuf[1] = '\0';
  return(make_sharp_constant(sc, sc->strbuf, WITH_OVERFLOW_ERROR));
}


static s7_pointer string_read_name(s7_scheme *sc, s7_pointer pt)
{
  /* port_string was allocated (and read from a file) so we can mess with it directly */
  s7_pointer result;
  char *str;

  str = (char *)(port_data(pt) + port_position(pt));
  if (char_ok_in_a_name[(uint8_t)*str])
    {
      s7_int k;
      char endc;
      char *orig_str;
      orig_str = (char *)(str - 1);
      str++;
      while (char_ok_in_a_name[(uint8_t)(*str)]) {str++;}
      k = str - orig_str;
      if (*str != 0)
	port_position(pt) += (k - 1);
      else port_position(pt) = port_data_size(pt);

      if (!number_table[(uint8_t)(*orig_str)])
	return(make_symbol_with_length(sc, orig_str, k));

      endc = (*str);
      (*str) = '\0';
      result = make_atom(sc, orig_str, BASE_10, SYMBOL_OK, WITH_OVERFLOW_ERROR);
      (*str) = endc;
      return(result);
    }
  result = sc->singletons[(uint8_t)(sc->strbuf[0])];
  if (!result)
    {
      sc->strbuf[1] = '\0';
      result = make_symbol_with_length(sc, sc->strbuf, 1);
      sc->singletons[(uint8_t)(sc->strbuf[0])] = result;
    }
  return(result);
}

static inline void port_set_filename(s7_scheme *sc, s7_pointer p, const char *name, size_t len)
{
  block_t *b;
  b = mallocate(sc, len + 1);
  port_filename_block(p) = b;
  port_filename(p) = (char *)block_data(b);
  memcpy((void *)block_data(b), (void *)name, len);
  port_filename(p)[len] = '\0';
}

static block_t *mallocate_port(s7_scheme *sc)
{
  #define PORT_LIST 8 /* sizeof(port_t): 160 */
  block_t *p;
  p = sc->block_lists[PORT_LIST];
  if (p)
    {
      sc->block_lists[PORT_LIST] = (block_t *)block_next(p);
      block_next(p) = NULL;
    }
  else
    {
      p = mallocate_block(sc);
      block_data(p) = (void *)alloc_permanent_string(sc, (size_t)(1 << PORT_LIST));
      block_set_index(p, PORT_LIST);
    }
  block_set_size(p, sizeof(port_t));
  return(p);
}

static s7_pointer read_file(s7_scheme *sc, FILE *fp, const char *name, s7_int max_size, const char *caller)
{
  s7_pointer port;
#if (!MS_WINDOWS)
  s7_int size;
#endif
  s7_int port_loc;
  block_t *b;

  new_cell(sc, port, T_INPUT_PORT);
  port_loc = s7_gc_protect_1(sc, port);
  b = mallocate_port(sc);
  port_block(port) = b;
  port_port(port) = (port_t *)block_data(b);
  port_set_closed(port, false);
  port_original_input_string(port) = sc->nil;
  port_write_character(port) = input_write_char;
  port_write_string(port) = input_write_string;

  /* if we're constantly opening files, and each open saves the file name in permanent
   *   memory, we gradually core-up.
   */
  port_filename_length(port) = safe_strlen(name);
  port_set_filename(sc, port, name, port_filename_length(port));
  port_line_number(port) = 1;  /* first line is numbered 1 */
  port_file_number(port) = 0;
  add_input_port(sc, port);

#if (!MS_WINDOWS)
  /* this doesn't work in MS C */
  fseek(fp, 0, SEEK_END);
  size = ftell(fp);
  rewind(fp);

  /* pseudo files (under /proc for example) have size=0, but we can read them, so don't assume a 0 length file is empty */

  if ((size > 0) &&                          /* if (size != 0) we get (open-input-file "/dev/tty") -> (open "/dev/tty") read 0 bytes of an expected -1? */
      ((max_size < 0) || (size < max_size))) /* load uses max_size = -1 */
    {
      size_t bytes;
      block_t *block;
      uint8_t *content;

      block = mallocate(sc, size + 2);
      content = (uint8_t *)(block_data(block));
      bytes = fread(content, sizeof(uint8_t), size, fp);
      if (bytes != (size_t)size)
	{
	  if (sc->output_port != sc->F)
	    {
	      char tmp[256];
	      int32_t len;
	      len = snprintf(tmp, 256, "(%s \"%s\") read %ld bytes of an expected %" print_s7_int "?", caller, name, (long)bytes, size);
	      port_write_string(sc->output_port)(sc, tmp, len, sc->output_port);
	    }
	  size = bytes;
	}
      content[size] = '\0';
      content[size + 1] = '\0';
      fclose(fp);

      port_file(port) = NULL; /* make valgrind happy */
      port_type(port) = STRING_PORT;
      port_data(port) = content;
      port_data_block(port) = block;
      port_data_size(port) = size;
      port_position(port) = 0;
      port_needs_free(port) = true;
      port_needs_unprotect(port) = false;
      port_read_character(port) = string_read_char;
      port_read_line(port) = string_read_line;
      port_display(port) = input_display;
      port_read_semicolon(port) = string_read_semicolon;
      port_read_white_space(port) = terminated_string_read_white_space;
      port_read_name(port) = string_read_name;
      port_read_sharp(port) = string_read_sharp;
    }
  else
    {
      port_file(port) = fp;
      port_type(port) = FILE_PORT;
      port_data(port) = NULL;
      port_data_block(port) = NULL;
      port_data_size(port) = 0;
      port_position(port) = 0;
      port_needs_free(port) = false;
      port_read_character(port) = file_read_char;
      port_read_line(port) = file_read_line;
      port_display(port) = input_display;
      port_read_semicolon(port) = file_read_semicolon;
      port_read_white_space(port) = file_read_white_space;
      port_read_name(port) = file_read_name;
      port_read_sharp(port) = file_read_sharp; /* was string_read_sharp?? */
    }
#else
  /* _stat64 is no better than the fseek/ftell route, and
   *    GetFileSizeEx and friends requires Windows.h which makes hash of everything else.
   *    fread until done takes too long on big files, so use a file port
   */
  port_file(port) = fp;
  port_type(port) = FILE_PORT;
  port_needs_free(port) = false;
  port_data(port) = NULL;
  port_data_block(port) = NULL;
  port_data_size(port) = 0;
  port_position(port) = 0;
  port_read_character(port) = file_read_char;
  port_read_line(port) = file_read_line;
  port_display(port) = input_display;
  port_read_semicolon(port) = file_read_semicolon;
  port_read_white_space(port) = file_read_white_space;
  port_read_name(port) = file_read_name;
  port_read_sharp(port) = file_read_sharp;
#endif

  s7_gc_unprotect_at(sc, port_loc);
  return(port);
}

/* -------------------------------- open-input-file -------------------------------- */
static int32_t remember_file_name(s7_scheme *sc, const char *file)
{
  int32_t i;

  for (i = 0; i <= sc->file_names_top; i++)
    if (safe_strcmp(file, string_value(sc->file_names[i])))
      return(i);

  sc->file_names_top++;
  if (sc->file_names_top >= sc->file_names_size)
    {
      int32_t old_size = 0;
      if (sc->file_names_size == 0)
	{
	  sc->file_names_size = INITIAL_FILE_NAMES_SIZE;
	  sc->file_names = (s7_pointer *)calloc(sc->file_names_size, sizeof(s7_pointer));
	}
      else
	{
	  old_size = sc->file_names_size;
	  sc->file_names_size *= 2;
	  sc->file_names = (s7_pointer *)realloc(sc->file_names, sc->file_names_size * sizeof(s7_pointer));
	}
      for (i = old_size; i < sc->file_names_size; i++)
	sc->file_names[i] = sc->F;
    }
  sc->file_names[sc->file_names_top] = s7_make_permanent_string(sc, file);
  return(sc->file_names_top);
}

static s7_pointer make_input_file(s7_scheme *sc, const char *name, FILE *fp)
{
  #define MAX_SIZE_FOR_STRING_PORT 5000000
  return(read_file(sc, fp, name, MAX_SIZE_FOR_STRING_PORT, "open"));
}

#if (!MS_WINDOWS)
#include <sys/stat.h>
#endif

static bool is_directory(const char *filename)
{
#if (!MS_WINDOWS)
  #ifdef S_ISDIR
    struct stat statbuf;
    return((stat(filename, &statbuf) >= 0) &&
	   (S_ISDIR(statbuf.st_mode)));
  #endif
#endif
  return(false);
}


static s7_pointer open_input_file_1(s7_scheme *sc, const char *name, const char *mode, const char *caller)
{
  FILE *fp;
  /* see if we can open this file before allocating a port */

  if (is_directory(name))
    return(file_error(sc, caller, "file is a directory:", name));

  errno = 0;
  fp = fopen(name, mode);
  if (!fp)
    {
#if (!MS_WINDOWS)
      if (errno == EINVAL)
	return(file_error(sc, caller, "invalid mode", mode));
  #if WITH_GCC
      /* catch one special case, "~/..." */
      if ((name[0] == '~') &&
	  (name[1] == '/'))
	{
	  char *home;
	  home = getenv("HOME");
	  if (home)
	    {
	      block_t *b;
	      char *filename;
	      s7_int len;
	      len = safe_strlen(name) + safe_strlen(home) + 1;
	      b = mallocate(sc, len);
	      filename = (char *)block_data(b);
	      filename[0] = '\0';
	      catstrs(filename, len, home, (char *)(name + 1), NULL);
	      fp = fopen(filename, "r");
	      liberate(sc, b);
	      if (fp)
		return(make_input_file(sc, name, fp));
	    }
	}
  #endif
#endif
      return(file_error(sc, caller, strerror(errno), name));
    }
  return(make_input_file(sc, name, fp));
}


s7_pointer s7_open_input_file(s7_scheme *sc, const char *name, const char *mode)
{
  return(open_input_file_1(sc, name, mode, "open-input-file"));
}

static s7_pointer g_open_input_file(s7_scheme *sc, s7_pointer args)
{
  #define H_open_input_file "(open-input-file filename (mode \"r\")) opens filename for reading"
  #define Q_open_input_file s7_make_signature(sc, 3, sc->is_input_port_symbol, sc->is_string_symbol, sc->is_string_symbol)
  s7_pointer name = car(args);
  /* open-input-file can create a new output file if the file to be opened does not exist, and the "a" mode is given */

  if (!is_string(name))
    return(method_or_bust(sc, name, sc->open_input_file_symbol, args, T_STRING, 1));
  /* what if the file name is a byte-vector? currently we accept it */

  if (is_pair(cdr(args)))
    {
      s7_pointer mode;
      mode = cadr(args);
      if (!is_string(mode))
	return(method_or_bust_with_type(sc, mode, sc->open_input_file_symbol, args,
				 wrap_string(sc, "a string (a mode such as \"r\")", 29), 2));
      /* since scheme allows embedded nulls, dumb stuff is accepted here: (open-input-file file "a\x00b") -- should this be an error? */
      return(open_input_file_1(sc, string_value(name), string_value(mode), "open-input-file"));
    }
  return(open_input_file_1(sc, string_value(name), "r", "open-input-file"));
}

static void make_standard_ports(s7_scheme *sc)
{
  s7_pointer x;

  /* standard output */
  x = alloc_pointer(sc);
  set_type(x, T_OUTPUT_PORT | T_IMMUTABLE | T_UNHEAP);
  port_port(x) = (port_t *)calloc(1, sizeof(port_t));
  port_type(x) = FILE_PORT;
  port_data(x) = NULL;
  port_data_block(x) = NULL;
  port_set_closed(x, false);
  port_filename_length(x) = 8;
  port_set_filename(sc, x, "*stdout*", 8);
  port_file_number(x) = remember_file_name(sc, port_filename(x)); /* these numbers need to be correct for the evaluator (__FUNC__ data) */
  port_line_number(x) = 0;
  port_file(x) = stdout;
  port_needs_free(x) = false;
  port_read_character(x) = output_read_char;
  port_read_line(x) = output_read_line;
  port_display(x) = stdout_display;
  port_write_character(x) = stdout_write_char;
  port_write_string(x) = stdout_write_string;
  sc->standard_output = x;

  /* standard error */
  x = alloc_pointer(sc);
  set_type(x, T_OUTPUT_PORT | T_IMMUTABLE | T_UNHEAP);
  port_port(x) = (port_t *)calloc(1, sizeof(port_t));
  port_type(x) = FILE_PORT;
  port_data(x) = NULL;
  port_data_block(x) = NULL;
  port_set_closed(x, false);
  port_filename_length(x) = 8;
  port_set_filename(sc, x, "*stderr*", 8);
  port_file_number(x) = remember_file_name(sc, port_filename(x));
  port_line_number(x) = 0;
  port_file(x) = stderr;
  port_needs_free(x) = false;
  port_read_character(x) = output_read_char;
  port_read_line(x) = output_read_line;
  port_display(x) = stderr_display;
  port_write_character(x) = stderr_write_char;
  port_write_string(x) = stderr_write_string;
  sc->standard_error = x;

  /* standard input */
  x = alloc_pointer(sc);
  set_type(x, T_INPUT_PORT | T_IMMUTABLE | T_UNHEAP);
  port_port(x) = (port_t *)calloc(1, sizeof(port_t));
  port_type(x) = FILE_PORT;
  port_set_closed(x, false);
  port_original_input_string(x) = sc->nil;
  port_filename_length(x) = 7;
  port_set_filename(sc, x, "*stdin*", 7);
  port_file_number(x) = remember_file_name(sc, port_filename(x));
  port_line_number(x) = 0;
  port_file(x) = stdin;
  port_data_block(x) = NULL;
  port_needs_free(x) = false;
  port_read_character(x) = file_read_char;
  port_read_line(x) = stdin_read_line;
  port_display(x) = input_display;
  port_read_semicolon(x) = file_read_semicolon;
  port_read_white_space(x) = file_read_white_space;
  port_read_name(x) = file_read_name;
  port_read_sharp(x) = file_read_sharp;
  port_write_character(x) = input_write_char;
  port_write_string(x) = input_write_string;
  sc->standard_input = x;

  s7_define_constant_with_documentation(sc, "*stdin*", sc->standard_input, "*stdin* is the built-in input port, C's stdin");
  s7_define_constant_with_documentation(sc, "*stdout*", sc->standard_output, "*stdout* is the built-in buffered output port, C's stdout");
  s7_define_constant_with_documentation(sc, "*stderr*", sc->standard_error, "*stderr* is the built-in unbuffered output port, C's stderr");

  sc->input_port = sc->standard_input;
  sc->output_port = sc->standard_output;
  sc->error_port = sc->standard_error;
  sc->current_file = NULL;
  sc->current_line = -1;
}

/* -------------------------------- open-output-file -------------------------------- */
s7_pointer s7_open_output_file(s7_scheme *sc, const char *name, const char *mode)
{
  FILE *fp;
  s7_pointer x;
  block_t *block, *b;
  /* see if we can open this file before allocating a port */

  errno = 0;
  fp = fopen(name, mode);
  if (!fp)
    {
#if (!MS_WINDOWS)
      if (errno == EINVAL)
	return(file_error(sc, "open-output-file", "invalid mode", mode));
#endif
      return(file_error(sc, "open-output-file", strerror(errno), name));
    }

  new_cell(sc, x, T_OUTPUT_PORT);
  b = mallocate_port(sc);
  port_block(x) = b;
  port_port(x) = (port_t *)block_data(b);
  port_type(x) = FILE_PORT;
  port_set_closed(x, false);
  port_filename_length(x) = safe_strlen(name);
  port_set_filename(sc, x, name, port_filename_length(x));
  port_line_number(x) = 1;
  port_file_number(x) = 0;
  port_file(x) = fp;
  port_needs_free(x) = true;  /* hmm -- I think these are freed via s7_close_output_port -> close_output_port */
  port_read_character(x) = output_read_char;
  port_read_line(x) = output_read_line;
  port_display(x) = file_display;
  port_write_character(x) = file_write_char;
  port_write_string(x) = file_write_string;
  port_position(x) = 0;
  port_data_size(x) = PORT_DATA_SIZE;
  block = mallocate(sc, PORT_DATA_SIZE);
  port_data_block(x) = block;
  port_data(x) = (uint8_t *)(block_data(block));
  add_output_port(sc, x);
  return(x);
}

static s7_pointer g_open_output_file(s7_scheme *sc, s7_pointer args)
{
  #define H_open_output_file "(open-output-file filename (mode \"w\")) opens filename for writing"
  #define Q_open_output_file s7_make_signature(sc, 3, sc->is_output_port_symbol, sc->is_string_symbol, sc->is_string_symbol)
  s7_pointer name = car(args);

  if (!is_string(name))
    return(method_or_bust(sc, name, sc->open_output_file_symbol, args, T_STRING, 1));

  if (is_pair(cdr(args)))
    {
      if (!is_string(cadr(args)))
	return(method_or_bust_with_type(sc, cadr(args), sc->open_output_file_symbol, args,
				 wrap_string(sc, "a string (a mode such as \"w\")", 29), 2));
      return(s7_open_output_file(sc, string_value(name), string_value(cadr(args))));
    }
  return(s7_open_output_file(sc, string_value(name), "w"));
}

/* -------------------------------- open-input-string -------------------------------- */
static s7_pointer open_input_string(s7_scheme *sc, const char *input_string, s7_int len)
{
  s7_pointer x;
  block_t *b;
  new_cell(sc, x, T_INPUT_PORT);
  b = mallocate_port(sc);
  port_block(x) = b;
  port_port(x) = (port_t *)block_data(b);
  port_type(x) = STRING_PORT;
  port_set_closed(x, false);
  port_original_input_string(x) = sc->nil;
  port_data(x) = (uint8_t *)input_string;
  port_data_block(x) = NULL;
  port_data_size(x) = len;
  port_position(x) = 0;
  port_filename_block(x) = NULL;
  port_filename_length(x) = 0;
  port_filename(x) = NULL;
  port_file_number(x) = 0;
  port_line_number(x) = 0;
  port_needs_free(x) = false;
  port_needs_unprotect(x) = false;
  port_read_character(x) = string_read_char;
  port_read_line(x) = string_read_line;
  port_display(x) = input_display;
  port_read_semicolon(x) = string_read_semicolon;
#if S7_DEBUGGING
  if (input_string[len] != '\0')
    {
      fprintf(stderr, "read_white_space string is not terminated: %s", input_string);
      abort();
    }
#endif
  port_read_white_space(x) = terminated_string_read_white_space;
  port_read_name(x) = string_read_name_no_free;
  port_read_sharp(x) = string_read_sharp;
  port_write_character(x) = input_write_char;
  port_write_string(x) = input_write_string;
  add_input_port(sc, x);
  return(x);
}

static s7_pointer open_and_protect_input_string(s7_scheme *sc, s7_pointer str)
{
  s7_pointer p;
  p = open_input_string(sc, string_value(str), string_length(str));
  port_gc_loc(p) = s7_gc_protect_1(sc, str);
  port_needs_unprotect(p) = true;
  return(p);
}

s7_pointer s7_open_input_string(s7_scheme *sc, const char *input_string)
{
  return(open_input_string(sc, input_string, safe_strlen(input_string)));
}

/* -------------------------------- open-output-string -------------------------------- */
static s7_pointer g_open_input_string(s7_scheme *sc, s7_pointer args)
{
  #define H_open_input_string "(open-input-string str) opens an input port reading str"
  #define Q_open_input_string s7_make_signature(sc, 2, sc->is_input_port_symbol, sc->is_string_symbol)
  s7_pointer input_string, port;

  input_string = car(args);
  if (!is_string(input_string))
    return(method_or_bust_one_arg(sc, input_string, sc->open_input_string_symbol, args, T_STRING));
  port = open_and_protect_input_string(sc, input_string);
  return(port);
}

#define FORMAT_PORT_LENGTH 128
/* the large majority (> 99% in my tests) of the output strings have less than 128 chars when the port is finally closed
 *   256 is slightly slower (the calloc time below dominates the realloc time in string_write_string)
 *   64 is much slower (realloc dominates)
 */

static s7_pointer open_output_string(s7_scheme *sc, s7_int len)
{
  s7_pointer x;
  block_t *block, *b;
  new_cell(sc, x, T_OUTPUT_PORT);
  b = mallocate_port(sc);
  port_block(x) = b;
  port_port(x) = (port_t *)block_data(b);
  port_type(x) = STRING_PORT;
  port_set_closed(x, false);
  port_data_size(x) = len;
  block = mallocate(sc, len);
  port_data_block(x) = block;
  port_data(x) = (uint8_t *)(block_data(block));
  port_data(x)[0] = '\0';        /* in case s7_get_output_string before any output */
  port_position(x) = 0;
  port_needs_free(x) = true;
  port_filename_block(x) = NULL;
  port_filename_length(x) = 0;   /* protect against (port-filename (open-output-string)) */
  port_filename(x) = NULL;
  port_read_character(x) = output_read_char;
  port_read_line(x) = output_read_line;
  port_display(x) = string_display;
  port_write_character(x) = string_write_char;
  port_write_string(x) = string_write_string;
  add_output_port(sc, x);
  return(x);
}

s7_pointer s7_open_output_string(s7_scheme *sc) {return(open_output_string(sc, sc->initial_string_port_length));}

static s7_pointer open_output_string_p(s7_scheme *sc) {return(s7_open_output_string(sc));}

static s7_pointer g_open_output_string(s7_scheme *sc, s7_pointer args)
{
  #define H_open_output_string "(open-output-string) opens an output string port"
  #define Q_open_output_string s7_make_signature(sc, 1, sc->is_output_port_symbol)
  return(s7_open_output_string(sc));
}

/* -------------------------------- get-output-string -------------------------------- */
const char *s7_get_output_string(s7_scheme *sc, s7_pointer p)
{
  port_data(p)[port_position(p)] = '\0';
  return((const char *)port_data(p));
}

static s7_pointer g_get_output_string(s7_scheme *sc, s7_pointer args)
{
  #define H_get_output_string "(get-output-string port clear-port) returns the output accumulated in port.  \
If the optional 'clear-port' is #t, the current string is flushed."
  #define Q_get_output_string s7_make_signature(sc, 3, sc->is_string_symbol, s7_make_signature(sc, 2, sc->is_output_port_symbol, sc->not_symbol), sc->is_boolean_symbol)

  s7_pointer p;
  bool clear_port = false;

  if (is_pair(cdr(args)))
    {
      p = cadr(args);
      if (!s7_is_boolean(p))
	return(wrong_type_argument(sc, sc->get_output_string_symbol, 2, p, T_BOOLEAN));
      clear_port = (p == sc->T);
    }
  p = car(args);
  if ((!is_output_port(p)) ||
      (!is_string_port(p)))
    {
      if (p == sc->F) return(make_empty_string(sc, 0, 0));
      return(method_or_bust_with_type_one_arg(sc, p, sc->get_output_string_symbol, args, wrap_string(sc, "an output string port", 21)));
    }
  if (port_is_closed(p))
    return(simple_wrong_type_argument_with_type(sc, sc->get_output_string_symbol, p, wrap_string(sc, "an active (open) string port", 28)));

  if (port_position(p) > sc->max_string_length)
    return(s7_out_of_range_error(sc, "get-output-string", 0, s7_make_integer(sc, port_position(p)),
				 "output string length is greater than (*s7* 'max-string-length)"));
  /* this includes (port_position(p) >= 2147483648) which will be a negative length in make_string_with_length */

  if ((clear_port) &&
      (port_position(p) < port_data_size(p)))
    {
      block_t *block;
      s7_pointer result;
      result = block_to_string(sc, port_data_block(p), port_position(p));
      port_data_size(p) = 64;
      block = mallocate(sc, 64);
      port_data_block(p) = block;
      port_data(p) = (uint8_t *)(block_data(block));
      port_position(p) = 0;
      port_data(p)[0] = '\0';
      return(result);
    }
  return(make_string_with_length(sc, (const char *)port_data(p), port_position(p)));
}

static s7_pointer op_get_output_string(s7_scheme *sc)
{
  s7_pointer port;
  port = sc->code;
  if ((!is_output_port(port)) ||
      (port_is_closed(port)))
    simple_wrong_type_argument_with_type(sc, sc->with_output_to_string_symbol, port,
					 wrap_string(sc, "an open string output port", 26));
  if (port_position(port) >= port_data_size(port))
    sc->value = block_to_string(sc, reallocate(sc, port_data_block(port), port_position(port) + 1), port_position(port));
  else sc->value = block_to_string(sc, port_data_block(port), port_position(port));
  port_data(port) = NULL;
  port_data_size(port) = 0;
  port_data_block(port) = NULL;
  port_needs_free(port) = false;
  return(NULL);
}

/* -------------------------------- open-input-function -------------------------------- */
s7_pointer s7_open_input_function(s7_scheme *sc, s7_pointer (*function)(s7_scheme *sc, s7_read_t read_choice, s7_pointer port))
{
  s7_pointer x;
  block_t *b;
  new_cell(sc, x, T_INPUT_PORT);
  b = mallocate_port(sc);
  port_block(x) = b;
  port_port(x) = (port_t *)block_data(b);
  port_type(x) = FUNCTION_PORT;
  port_set_closed(x, false);
  port_original_input_string(x) = sc->nil;
  port_data_block(x) = NULL;
  port_needs_free(x) = false;
  port_filename_block(x) = NULL;
  port_filename(x) = NULL;
  port_filename_length(x) = 0;
  port_file_number(x) = 0;
  port_line_number(x) = 0;
  port_input_function(x) = function;
  port_read_character(x) = function_read_char;
  port_read_line(x) = function_read_line;
  port_display(x) = input_display;
  port_write_character(x) = input_write_char;
  port_write_string(x) = input_write_string;
  add_input_port(sc, x);
  return(x);
}

/* -------------------------------- open-output-function -------------------------------- */
s7_pointer s7_open_output_function(s7_scheme *sc, void (*function)(s7_scheme *sc, uint8_t c, s7_pointer port))
{
  s7_pointer x;
  block_t *b;
  new_cell(sc, x, T_OUTPUT_PORT);
  b = mallocate_port(sc);
  port_block(x) = b;
  port_port(x) = (port_t *)block_data(b);
  port_type(x) = FUNCTION_PORT;
  port_data(x) = NULL;
  port_data_block(x) = NULL;
  port_set_closed(x, false);
  port_needs_free(x) = false;
  port_output_function(x) = function;
  port_read_character(x) = output_read_char;
  port_read_line(x) = output_read_line;
  port_display(x) = function_display;
  port_write_character(x) = function_write_char;
  port_write_string(x) = function_write_string;
  add_output_port(sc, x);
  return(x);
}


static void push_input_port(s7_scheme *sc, s7_pointer new_port)
{
#if S7_DEBUGGING
  if (!is_input_port(new_port)) fprintf(stderr, "push %s\n", DISPLAY(new_port));
#endif
  sc->input_port_stack = cons(sc, sc->input_port, sc->input_port_stack);
  sc->input_port = new_port;
}


static void pop_input_port(s7_scheme *sc)
{
  if (is_pair(sc->input_port_stack))
    {
      s7_pointer nxt;
      sc->input_port = car(sc->input_port_stack);
      nxt = cdr(sc->input_port_stack);
      /* is this safe? */
      free_cell(sc, sc->input_port_stack);
      sc->input_port_stack = nxt;
    }
  else sc->input_port = sc->standard_input;
}


static int32_t inchar(s7_pointer pt)
{
  int32_t c;
  if (is_file_port(pt))
    c = fgetc(port_file(pt)); /* not uint8_t! -- could be EOF */
  else
    {
      if (port_data_size(pt) <= port_position(pt))
	return(EOF);
      c = (uint8_t)port_data(pt)[port_position(pt)++];
    }

  if (c == '\n')
    port_line_number(pt)++;

  return(c);
}


static void backchar(char c, s7_pointer pt)
{
  if (c == '\n')
    port_line_number(pt)--;

  if (is_file_port(pt))
    ungetc(c, port_file(pt));
  else
    {
      if (port_position(pt) > 0)
	port_position(pt)--;
    }
}

static s7_pointer input_port_if_not_loading(s7_scheme *sc)
{
  s7_pointer port;
  port = sc->input_port;
  if (is_loader_port(port)) /* this flag is turned off by the reader macros, so we aren't in that context */
    {
      int32_t c;
      c = port_read_white_space(port)(sc, port);
      if (c > 0)            /* we can get either EOF or NULL at the end */
	{
	  backchar(c, port);
	  return(NULL);
	}
      return(sc->standard_input);
    }
  return(port);
}

/* -------------------------------- read-char -------------------------------- */
s7_pointer s7_read_char(s7_scheme *sc, s7_pointer port)
{
  int32_t c;
  c = port_read_character(port)(sc, port);
  if  (c == EOF) return(eof_object);
  return(chars[c]);
}

static s7_pointer g_read_char(s7_scheme *sc, s7_pointer args)
{
  #define H_read_char "(read-char (port (current-input-port))) returns the next character in the input port"
  #define Q_read_char s7_make_signature(sc, 2, s7_make_signature(sc, 2, sc->is_char_symbol, sc->is_eof_object_symbol), sc->is_input_port_symbol)
  s7_pointer port;

  if (is_not_null(args))
    port = car(args);
  else
    {
      port = input_port_if_not_loading(sc);
      if (!port) return(eof_object);
    }
  if (!is_input_port(port))
    return(method_or_bust_with_type_one_arg(sc, port, sc->read_char_symbol, args, an_input_port_string));
  return(chars[port_read_character(port)(sc, port)]);
}

/* -------------------------------- write-char -------------------------------- */
s7_pointer s7_write_char(s7_scheme *sc, s7_pointer c, s7_pointer pt)
{
  if (pt != sc->F)
    port_write_character(pt)(sc, s7_character(c), pt);
  return(c);
}

static s7_pointer g_write_char(s7_scheme *sc, s7_pointer args)
{
  #define H_write_char "(write-char char (port (current-output-port))) writes char to the output port"
  #define Q_write_char s7_make_signature(sc, 3, sc->is_char_symbol, sc->is_char_symbol, s7_make_signature(sc, 2, sc->is_output_port_symbol, sc->not_symbol))
  s7_pointer port, chr;

  chr = car(args);
  if (!s7_is_character(chr))
    return(method_or_bust(sc, chr, sc->write_char_symbol, args, T_CHARACTER, 1));

  if (is_pair(cdr(args)))
    port = cadr(args);
  else port = sc->output_port;
  if (port == sc->F) return(chr);
  if (!is_output_port(port))
    return(method_or_bust_with_type(sc, port, sc->write_char_symbol, args, an_output_port_string, 2));

  port_write_character(port)(sc, s7_character(chr), port);
  return(chr);
}

static s7_pointer write_char_p_p(s7_scheme *sc, s7_pointer c)
{
  if (!s7_is_character(c))
    simple_wrong_type_argument(sc, sc->write_char_symbol, c, T_CHARACTER);
  if (sc->output_port == sc->F) return(c);
  port_write_character(sc->output_port)(sc, s7_character(c), sc->output_port);
  return(c);
}

static s7_pointer write_char_p_pp(s7_scheme *sc, s7_pointer c, s7_pointer port)
{
  if (!s7_is_character(c))
    simple_wrong_type_argument(sc, sc->write_char_symbol, c, T_CHARACTER);
  if (port == sc->F) return(c);
  if (!is_output_port(port))
    simple_wrong_type_argument_with_type(sc, sc->write_char_symbol, port, an_output_port_string);
  port_write_character(port)(sc, s7_character(c), port);
  return(c);
}

/* (with-output-to-string (lambda () (write-char #\space))) -> " "
 * (with-output-to-string (lambda () (write #\space))) -> "#\\space"
 * (with-output-to-string (lambda () (display #\space))) -> " "
 * is this correct?  It's what Guile does.  write-char is actually display-char.
 */

/* -------------------------------- peek-char -------------------------------- */
s7_pointer s7_peek_char(s7_scheme *sc, s7_pointer port)
{
  int32_t c;              /* needs to be an int32_t so EOF=-1, but not 255 */
  if (is_string_port(port))
    {
      if (port_data_size(port) <= port_position(port))
	return(chars[EOF]);
      return(chars[(uint8_t)port_data(port)[port_position(port)]]);
    }
  c = port_read_character(port)(sc, port);
  if (c == EOF) return(eof_object);
  backchar(c, port);
  return(chars[c]);
}

static s7_pointer g_peek_char(s7_scheme *sc, s7_pointer args)
{
  #define H_peek_char "(peek-char (port (current-input-port))) returns the next character in the input port, but does not remove it from the input stream"
  #define Q_peek_char s7_make_signature(sc, 2, s7_make_signature(sc, 2, sc->is_char_symbol, sc->is_eof_object_symbol), sc->is_input_port_symbol)
  s7_pointer port;

  if (is_not_null(args))
    port = car(args);
  else port = sc->input_port;

  if (!is_input_port(port))
    return(method_or_bust_with_type_one_arg(sc, port, sc->peek_char_symbol, args, an_input_port_string));
  if (port_is_closed(port))
    return(simple_wrong_type_argument_with_type(sc, sc->peek_char_symbol, port, an_open_port_string));

  if (is_function_port(port))
    return((*(port_input_function(port)))(sc, S7_PEEK_CHAR, port));
  return(s7_peek_char(sc, port));
}

/* -------------------------------- read-byte -------------------------------- */
static s7_pointer g_read_byte(s7_scheme *sc, s7_pointer args)
{
  #define H_read_byte "(read-byte (port (current-input-port))): reads a byte from the input port"
  #define Q_read_byte s7_make_signature(sc, 2, s7_make_signature(sc, 2, sc->is_byte_symbol, sc->is_eof_object_symbol), sc->is_input_port_symbol)
  s7_pointer port;
  int32_t c;

  if (is_not_null(args))
    port = car(args);
  else
    {
      port = input_port_if_not_loading(sc);
      if (!port) return(eof_object);
    }
  if (!is_input_port(port))
    return(method_or_bust_with_type_one_arg(sc, port, sc->read_byte_symbol, args, an_input_port_string));

  c = port_read_character(port)(sc, port);
  if (c == EOF)
    return(eof_object);
  return(small_int(c));
}

/* -------------------------------- write-byte -------------------------------- */
static s7_pointer g_write_byte(s7_scheme *sc, s7_pointer args)
{
  #define H_write_byte "(write-byte byte (port (current-output-port))): writes byte to the output port"
  #define Q_write_byte s7_make_signature(sc, 3, sc->is_byte_symbol, sc->is_byte_symbol, s7_make_signature(sc, 2, sc->is_output_port_symbol, sc->not_symbol))
  s7_pointer port, b;
  s7_int val;

  b = car(args);
  if (!s7_is_integer(b))
    return(method_or_bust(sc, car(args), sc->write_byte_symbol, args, T_INTEGER, 1));

  val = s7_integer(b);
  if ((val < 0) || (val > 255)) /* need to check this before port==#f, else (write-byte most-positive-fixnum #f) is not an error */
    return(wrong_type_argument_with_type(sc, sc->write_byte_symbol, 1, b, an_unsigned_byte_string));

  if (is_pair(cdr(args)))
    port = cadr(args);
  else port = sc->output_port;

  if (!is_output_port(port))
    {
      if (port == sc->F) return(car(args));
      return(method_or_bust_with_type_one_arg(sc, port, sc->write_byte_symbol, args, an_output_port_string));
    }

  port_write_character(port)(sc, (uint8_t)val, port);
  return(b);
}

/* -------------------------------- read-line -------------------------------- */
static s7_pointer g_read_line(s7_scheme *sc, s7_pointer args)
{
  #define H_read_line "(read-line port (with-eol #f)) returns the next line from port, or #<eof>. \
If 'with-eol' is not #f, read-line includes the trailing end-of-line character."
  #define Q_read_line s7_make_signature(sc, 3, s7_make_signature(sc, 2, sc->is_string_symbol, sc->is_eof_object_symbol), sc->is_input_port_symbol, sc->is_boolean_symbol)

  s7_pointer port;
  bool with_eol = false;

  if (is_not_null(args))
    {
      port = car(args);
      if (!is_input_port(port))
	return(method_or_bust_with_type(sc, port, sc->read_line_symbol, args, an_input_port_string, 1));

      if (is_not_null(cdr(args)))
	with_eol = (cadr(args) != sc->F);
    }
  else
    {
      port = input_port_if_not_loading(sc);
      if (!port) return(eof_object);
    }
  return(port_read_line(port)(sc, port, with_eol, true));
}

static s7_pointer g_read_line_uncopied(s7_scheme *sc, s7_pointer args)
{
  s7_pointer port;
  bool with_eol = false;
  port = car(args);
  if (!is_input_port(port))
    return(g_read_line(sc, args));
  if (is_not_null(cdr(args)))
    with_eol = (cadr(args) != sc->F);
  return(port_read_line(port)(sc, port, with_eol, false));
}

static s7_pointer read_line_p_pp(s7_scheme *sc, s7_pointer port, s7_pointer with_eol)
{
  if (!is_input_port(port))
    return(method_or_bust_with_type(sc, port, sc->read_line_symbol, list_2(sc, port, with_eol), an_input_port_string, 1));
  return(port_read_line(port)(sc, port, with_eol != sc->F, true));
}


/* -------------------------------- read-string -------------------------------- */
static s7_pointer g_read_string(s7_scheme *sc, s7_pointer args)
{
  /* read-chars would be a better name -- read-string could mean CL-style read-from-string (like eval-string)
   *   similarly read-bytes could return a byte-vector (rather than r7rs's read-bytevector)
   *   and write-string -> write-chars, write-bytevector -> write-bytes
   */
  #define H_read_string "(read-string k port) reads k characters from port into a new string and returns it."
  #define Q_read_string s7_make_signature(sc, 3, s7_make_signature(sc, 2, sc->is_string_symbol, sc->is_eof_object_symbol), sc->is_integer_symbol, sc->is_input_port_symbol)
  s7_pointer k, port, s;
  s7_int i, chars;
  uint8_t *str;

  k = car(args);
  if (!s7_is_integer(k))
    return(method_or_bust(sc, k, sc->read_string_symbol, args, T_INTEGER, 1));
  chars = s7_integer(k);

  if (!is_null(cdr(args)))
    port = cadr(args);
  else port = input_port_if_not_loading(sc);

  if (chars < 0)
    return(wrong_type_argument_with_type(sc, sc->read_string_symbol, 1, wrap_integer1(sc, chars), a_non_negative_integer_string));
  if (chars > sc->max_string_length)
    return(out_of_range(sc, sc->read_string_symbol, small_int(1), wrap_integer1(sc, chars), its_too_large_string));

  if (!port) return(eof_object);
  if (!is_input_port(port))
    return(method_or_bust_with_type(sc, port, sc->read_string_symbol, list_2(sc, make_integer(sc, chars), port), an_input_port_string, 2));

  if (chars == 0)
    return(make_empty_string(sc, 0, 0));

  s = make_empty_string(sc, chars, 0);
  str = (uint8_t *)string_value(s);
  if (is_string_port(port))
    {
      s7_int pos, end, len;
      pos = port_position(port);
      end = port_data_size(port);
      len = end - pos;
      if (len > chars) len = chars;
      if (len <= 0) return(eof_object);
      memcpy((void *)str, (void *)(port_data(port) + pos), len);
      string_length(s) = len;
      str[len] = '\0';
      port_position(port) += len;
      return(s);
    }
  for (i = 0; i < chars; i++)
    {
      int32_t c;
      c = port_read_character(port)(sc, port);
      if (c == EOF)
	{
	  if (i == 0)
	    return(eof_object);
	  string_length(s) = i;
	  return(s);
	}
      str[i] = (uint8_t)c;
    }
  return(s);
}


#define declare_jump_info() bool old_longjmp; int32_t old_jump_loc, jump_loc; jmp_buf old_goto_start

#define store_jump_info(Sc)						\
  do {									\
      old_longjmp = Sc->longjmp_ok;					\
      old_jump_loc = Sc->setjmp_loc;					\
      memcpy((void *)old_goto_start, (void *)(Sc->goto_start), sizeof(jmp_buf)); \
  } while (0)

#define restore_jump_info(Sc)						\
  do {									\
      Sc->longjmp_ok = old_longjmp;					\
      Sc->setjmp_loc = old_jump_loc;					\
      memcpy((void *)(Sc->goto_start), (void *)old_goto_start, sizeof(jmp_buf)); \
      if ((jump_loc == ERROR_JUMP) &&					\
          (sc->longjmp_ok))						\
        longjmp(sc->goto_start, ERROR_JUMP);				\
  } while (0)

#define set_jump_info(Sc, Tag)				\
  do {							\
      sc->longjmp_ok = true;				\
      sc->setjmp_loc = Tag;				\
      jump_loc = setjmp(sc->goto_start);		\
  } while (0)

/* -------------------------------- read -------------------------------- */
s7_pointer s7_read(s7_scheme *sc, s7_pointer port)
{
  if (is_input_port(port))
    {
      s7_pointer old_envir;
      declare_jump_info();

      old_envir = sc->envir;
      sc->envir = sc->nil;
      push_input_port(sc, port);

      store_jump_info(sc);
      set_jump_info(sc, READ_SET_JUMP);
      if (jump_loc != NO_JUMP)
	{
	  if (jump_loc != ERROR_JUMP)
	    eval(sc, sc->cur_op);
	}
      else
	{
	  push_stack_no_let_no_code(sc, OP_BARRIER, port);
	  push_stack(sc, OP_EVAL_DONE, sc->args, sc->code);

	  eval(sc, OP_READ_INTERNAL);

	  if (sc->tok == TOKEN_EOF)
	    sc->value = eof_object;

	  if ((sc->cur_op == OP_EVAL_DONE) &&
	      (stack_op(sc->stack, s7_stack_top(sc) - 1) == OP_BARRIER))
	    pop_stack(sc);
	}
      pop_input_port(sc);
      sc->envir = old_envir;

      restore_jump_info(sc);
      return(sc->value);
    }
  return(simple_wrong_type_argument_with_type(sc, sc->read_symbol, port, an_input_port_string));
}

static s7_pointer g_read(s7_scheme *sc, s7_pointer args)
{
  /* would it be useful to add an environment arg here?  (just set sc->envir at the end?)
   *    except for expansions, nothing is evaluated at read time, unless...
   *    say we set up a dot reader:
   *        (set! *#readers* (cons (cons #\. (lambda (str) (if (string=? str ".") (eval (read)) #f))) *#readers*))
   *    then
   *        (call-with-input-string "(+ 1 #.(+ 1 hiho))" (lambda (p) (read p)))
   *    evaluates hiho in the rootlet, but how to pass the env to the inner eval or read?
   * (eval, eval-string and load already have an env arg)
   */
  #define H_read "(read (port (current-input-port))) returns the next object in the input port, or #<eof> at the end"
  #define Q_read s7_make_signature(sc, 2, sc->T, sc->is_input_port_symbol)
  s7_pointer port;

  if (is_not_null(args))
    port = car(args);
  else
    {
      port = input_port_if_not_loading(sc);
      if (!port) return(eof_object);
    }

  if (!is_input_port(port))
    return(method_or_bust_with_type_one_arg(sc, port, sc->read_symbol, args, an_input_port_string));

  if (is_function_port(port))
    return((*(port_input_function(port)))(sc, S7_READ, port));

  if ((is_string_port(port)) &&
      (port_data_size(port) <= port_position(port)))
    return(eof_object);

  push_input_port(sc, port);
  push_stack_op_let(sc, OP_READ_DONE); /* this stops the internal read process so we only get one form */
  push_stack_op_let(sc, OP_READ_INTERNAL);

  return(port);
}


/* -------------------------------- load -------------------------------- */

#if WITH_MULTITHREAD_CHECKS
typedef struct {
  s7_scheme* sc;
  const int32_t lock_count; /* Remember lock count in case we have skipped calls to leave_track_scope by longjmp-ing. */
} lock_scope_t;

static lock_scope_t enter_lock_scope(s7_scheme *sc)
{
  int result = pthread_mutex_trylock(&sc->lock);
  if (result != 0)
    {
      fprintf(stderr, "pthread_mutex_trylock failed: %d (EBUSY: %d)", result, EBUSY);
      abort();
    }

  sc->lock_count++;
  {
    lock_scope_t st = {.sc = sc, .lock_count = sc->lock_count};
    return(st);
  }
}

static void leave_lock_scope(lock_scope_t *st)
{
  while (st->sc->lock_count > st->lock_count)
    {
      st->sc->lock_count--;
      pthread_mutex_unlock(&st->sc->lock);
    }
}
#define TRACK(Sc) lock_scope_t lock_scope __attribute__ ((__cleanup__(leave_lock_scope))) = enter_lock_scope(Sc)
#else
#define TRACK(Sc)
#endif

static block_t *search_load_path(s7_scheme *sc, const char *name)
{
  s7_int len;
  s7_pointer lst;

  lst = s7_load_path(sc);
  len = s7_list_length(sc, lst);

  if (len > 0)
    {
      block_t *b;
      s7_int i;
      char *filename;
      b = mallocate(sc, 1024);
      filename = (char *)block_data(b);

      for (i = 0; i < len; i++)
	{
	  const char *new_dir;
	  new_dir = string_value(s7_list_ref(sc, lst, i));
	  if (new_dir)
	    {
	      FILE *fp;
	      filename[0] = '\0';
	      catstrs(filename, 1024, new_dir, "/", name, NULL);
	      fp = fopen(filename, "r");
	      if (fp)
		{
		  block_info(b) = (void *)fp;
		  return(b);
		}
	    }
	}
      liberate(sc, b);
    }
  return(NULL);
}

s7_pointer s7_load_with_environment(s7_scheme *sc, const char *filename, s7_pointer e)
{
  s7_pointer port;
  FILE *fp;
  char *new_filename = NULL;
  declare_jump_info();
  TRACK(sc);

  fp = fopen(filename, "r");
  if (!fp)
    {
      block_t *b;
      b = search_load_path(sc, filename);
      if (b)
	{
	  new_filename = copy_string((const char *)block_data(b)); /* (require libc.scm) for example needs the directory for cload in some cases */
	  fp = (FILE *)block_info(b);
	  liberate(sc, b);
	}
    }
  if (!fp)
    return(file_error(sc, "load", "can't open", filename));

  if (hook_has_functions(sc->load_hook))
    s7_call(sc, sc->load_hook, list_1(sc, sc->temp4 = s7_make_string(sc, filename)));

  port = read_file(sc, fp, (new_filename) ? (const char *)new_filename : filename, -1, "load");   /* -1 means always read its contents into a local string */
  port_file_number(port) = remember_file_name(sc, filename);
  if (new_filename) free(new_filename);
  set_loader_port(port);
  sc->temp6 = port;
  push_input_port(sc, port);
  sc->temp6 = sc->nil;

  /* it's possible to call this recursively (s7_load is Xen_load_file which can be invoked via s7_call)
   *   but in that case, we actually want it to behave like g_load and continue the evaluation upon completion
   */
  sc->envir = e;
  push_stack(sc, OP_LOAD_RETURN_IF_EOF, port, sc->code);

  store_jump_info(sc);
  set_jump_info(sc, LOAD_SET_JUMP);
  if (jump_loc != NO_JUMP)
    {
      if (jump_loc != ERROR_JUMP)
	eval(sc, sc->cur_op);
    }
  else eval(sc, OP_READ_INTERNAL);

  pop_input_port(sc);
  if (is_input_port(port))
    s7_close_input_port(sc, port);

  restore_jump_info(sc);
  if (is_multiple_value(sc->value))
    sc->value = splice_in_values(sc, multiple_value(sc->value));
  return(sc->value);
}

s7_pointer s7_load(s7_scheme *sc, const char *filename)
{
  return(s7_load_with_environment(sc, filename, sc->nil));
}


#if WITH_C_LOADER
#include <dlfcn.h>

static block_t *full_filename(s7_scheme *sc, const char *filename)
{
  s7_int len;
  char *pwd, *rtn;
  block_t *block;
  pwd = getcwd(NULL, 0); /* docs say this means it will return a new string of the right size */
  len = safe_strlen(pwd) + safe_strlen(filename) + 8;
  block = mallocate(sc, len * sizeof(char));
  rtn = (char *)block_data(block);
  if (pwd)
    {
      rtn[0] = '\0';
      catstrs(rtn, len, pwd, "/", filename, NULL);
      free(pwd);
    }
  else
    {
      memcpy((void *)rtn, (void *)filename, len);
      rtn[len] = '\0';
    }
  return(block);
}
#endif


static s7_pointer g_load(s7_scheme *sc, s7_pointer args)
{
  #define H_load "(load file (env (rootlet))) loads the scheme file 'file'. The 'env' argument \
defaults to the rootlet.  To load into the current environment instead, pass (curlet)."
  #define Q_load s7_make_signature(sc, 3, sc->values_symbol, sc->is_string_symbol, sc->is_let_symbol)

  FILE *fp = NULL;
  s7_pointer name, port;
  const char *fname;

  name = car(args);
  if (!is_string(name))
    return(method_or_bust(sc, name, sc->load_symbol, args, T_STRING, 1));

  if (is_not_null(cdr(args)))
    {
      s7_pointer e;
      e = cadr(args);
      if (!is_let(e))
	return(wrong_type_argument_with_type(sc, sc->load_symbol, 2, e, a_let_string));
      if (e == sc->rootlet)
	sc->envir = sc->nil;
      else sc->envir = e;
    }
  else sc->envir = sc->nil;

  fname = string_value(name);
  if ((!fname) || (!(*fname)))                 /* fopen("", "r") returns a file pointer?? */
    return(s7_error(sc, sc->out_of_range_symbol, set_elist_2(sc, wrap_string(sc, "load's first argument, ~S, should be a filename", 47), name)));

  if (is_directory(fname))
    return(s7_error(sc, sc->wrong_type_arg_symbol, set_elist_2(sc, wrap_string(sc, "load argument, ~S, is a directory", 33), name)));

#if WITH_C_LOADER
  /* if fname ends in .so, try loading it as a c shared object
   *   (load "/home/bil/cl/m_j0.so" (inlet (cons 'init_func 'init_m_j0)))
   */
  {
    s7_int fname_len;

    fname_len = safe_strlen(fname);
    if ((fname_len > 3) &&
	(is_pair(cdr(args))) &&
	(local_strcmp((const char *)(fname + (fname_len - 3)), ".so")))
      {
	s7_pointer init;

	init = s7_let_ref(sc, (is_null(sc->envir)) ? sc->rootlet : sc->envir, s7_make_symbol(sc, "init_func"));
	if (is_symbol(init))
	  {
	    void *library;
	    char *pwd_name;
	    block_t *pname = NULL;

	    if (fname[0] != '/')
	      {
		pname = full_filename(sc, fname); /* this is necessary, at least in Linux -- we can't blithely dlopen whatever is passed to us */
		pwd_name = (char *)block_data(pname);
	      }
	    library = dlopen((pname) ? pwd_name : fname, RTLD_NOW);
	    if (library)
	      {
		const char *init_name;
		void *init_func;

		init_name = symbol_name(init);
		init_func = dlsym(library, init_name);
		if (init_func)
		  {
		    typedef void *(*dl_func)(s7_scheme *sc);
		    ((dl_func)init_func)(sc);
		    if (pname) liberate(sc, pname);
		    return(sc->T);
		  }
		else
		  {
		    s7_warn(sc, 512, "loaded %s, but can't find %s (%s)?\n", fname, init_name, dlerror());
		    dlclose(library);
		  }
	      }
	    else s7_warn(sc, 512, "load %s failed: %s\n", (pname) ? pwd_name : fname, dlerror());
	    if (pname) liberate(sc, pname);
	  }
	else s7_warn(sc, 512, "can't load %s: no init function\n", fname);
	return(sc->F);
      }
  }
#endif

  fp = fopen(fname, "r");

#if WITH_GCC
  if (!fp)
    {
      /* catch one special case, "~/..." since it causes 99.9% of the "can't load ..." errors */
      if ((fname[0] == '~') &&
	  (fname[1] == '/'))
	{
	  char *home;
	  home = getenv("HOME");
	  if (home)
	    {
	      block_t *b;
	      char *filename;
	      s7_int len;
	      len = safe_strlen(fname) + safe_strlen(home) + 1;
	      b = mallocate(sc, len);
	      filename = (char *)block_data(b);
	      filename[0] = '\0';
	      catstrs(filename, len, home, (char *)(fname + 1), NULL);
	      fp = fopen(filename, "r");
	      liberate(sc, b);
	    }
	}
    }
#endif

  if (!fp)
    {
      block_t *b;
      b = search_load_path(sc, fname);
      if (!b)
	return(file_error(sc, "load", "can't open", fname));
      fp = (FILE *)block_info(b);
      liberate(sc, b);
    }

  port = read_file(sc, fp, fname, -1, "load");
  port_file_number(port) = remember_file_name(sc, fname);
  set_loader_port(port);
  sc->temp6 = port;
  push_input_port(sc, port);
  sc->temp6 = sc->nil;

  push_stack_op_let(sc, OP_LOAD_CLOSE_AND_POP_IF_EOF);  /* was pushing args and code, but I don't think they're used later */
  push_stack_op_let(sc, OP_READ_INTERNAL);

  /* now we've opened and moved to the file to be loaded, and set up the stack to return
   *   to where we were.  Call *load-hook* if it is a procedure.
   */

  if (hook_has_functions(sc->load_hook))
    s7_apply_function(sc, sc->load_hook, list_1(sc, sc->temp4 = s7_make_string(sc, fname)));

  return(sc->unspecified);
}


s7_pointer s7_load_path(s7_scheme *sc)
{
  return(s7_symbol_value(sc, sc->load_path_symbol));
}


s7_pointer s7_add_to_load_path(s7_scheme *sc, const char *dir)
{
  s7_symbol_set_value(sc,
		      sc->load_path_symbol,
		      cons(sc,
			   s7_make_string(sc, dir),
			   s7_symbol_value(sc, sc->load_path_symbol)));
  return(s7_symbol_value(sc, sc->load_path_symbol));
}


static s7_pointer g_load_path_set(s7_scheme *sc, s7_pointer args)
{
  /* new value must be either () or a proper list of strings */
  if (is_null(cadr(args))) return(cadr(args));
  if (is_pair(cadr(args)))
    {
      s7_pointer x;
      for (x = cadr(args); is_pair(x); x = cdr(x))
	if (!is_string(car(x)))
	  return(s7_error(sc, sc->error_symbol, set_elist_2(sc, wrap_string(sc, "can't set *load-path* to ~S", 27), cadr(args))));
      if (is_null(x))
	return(cadr(args));
    }
  return(s7_error(sc, sc->error_symbol, set_elist_2(sc, wrap_string(sc, "can't set *load-path* to ~S", 27), cadr(args))));
}

static s7_pointer g_cload_directory_set(s7_scheme *sc, s7_pointer args)
{
  s7_pointer cl_dir;
  cl_dir = cadr(args);
  if (!is_string(cl_dir))
    return(s7_error(sc, sc->error_symbol, set_elist_2(sc, wrap_string(sc, "can't set *cload-directory* to ~S", 33), cadr(args))));
  s7_symbol_set_value(sc, sc->cload_directory_symbol, cl_dir);
  if (safe_strlen(string_value(cl_dir)) > 0)
    s7_add_to_load_path(sc, (const char *)(string_value(cl_dir)));
  return(cl_dir);
}


/* ---------------- autoload ---------------- */
void s7_autoload_set_names(s7_scheme *sc, const char **names, s7_int size)
{
  /* the idea here is that by sticking to string constants we can handle 90% of the work at compile-time,
   *   with less start-up memory.  Then eventually we'll add C libraries a la xg (gtk) as environments
   *   and every name in that library will come as an import once dlopen has picked up the library.
   *   So, hopefully, we can pre-declare as many names as we want from as many libraries as we want,
   *   without a bloated mess of a run-time image.  And new libraries are easy to accommodate --
   *   add the names to be auto-exported to this list with the name of the scheme file that cloads
   *   the library and exports the given name. So, we'll need a separate such file for each library?
   *
   * the environment variable could use the library base name in *: *libm* or *libgtk*
   *   (*libm* 'j0)
   * why not just predeclare these libraries?  The caller could import what he wants via require.
   * So the autoloader need only know which libraries, but this doesn't fit the current use of gtk in xg
   * In fact, we only need to see *libm* -> libm.so etc, but we still need the arg/return types of each function, etc
   * And libgtk is enormous -- seems too bad to tie-in everything via the FFI when we need less than 1% of it.
   * Perhaps each module as an environment within the main one: ((*libgtk* *gtkwidget*) 'gtk_widget_new)?
   * But that requires inside knowlege of the library, and changes without notice.
   *
   * Also we need to decide how to handle name collisions (by order of autoload lib setup)
   * And (lastly?) how to handle different library versions?
   *
   *
   * so autoload known libs here in s7 so we're indepentdent of snd
   *   (currently these are included in make-index.scm[line 575] -> snd-xref.c)
   * for each module, include an env in the lib env (*libgtk* 'gtkwidget.h) or whatever that has the names in that header
   * in autoload below, don't sort! -- just build a list of autoload tables and check each in order at autoload time (we want startup to be fast)
   * for versions, include wrapper macro at end of each c-define choice
   * in the xg case, there's no savings in delaying the defines
   *
   */

  if (!sc->autoload_names)
    {
      sc->autoload_names = (const char ***)calloc(INITIAL_AUTOLOAD_NAMES_SIZE, sizeof(const char **));
      sc->autoload_names_sizes = (s7_int *)calloc(INITIAL_AUTOLOAD_NAMES_SIZE, sizeof(s7_int));
      sc->autoloaded_already = (bool **)calloc(INITIAL_AUTOLOAD_NAMES_SIZE, sizeof(bool *));
      sc->autoload_names_top = INITIAL_AUTOLOAD_NAMES_SIZE;
      sc->autoload_names_loc = 0;
    }
  else
    {
      if (sc->autoload_names_loc >= sc->autoload_names_top)
	{
	  s7_int i;
	  sc->autoload_names_top *= 2;
	  sc->autoload_names = (const char ***)realloc(sc->autoload_names, sc->autoload_names_top * sizeof(const char **));
	  sc->autoload_names_sizes = (s7_int *)realloc(sc->autoload_names_sizes, sc->autoload_names_top * sizeof(s7_int));
	  sc->autoloaded_already = (bool **)realloc(sc->autoloaded_already, sc->autoload_names_top * sizeof(bool *));
	  for (i = sc->autoload_names_loc; i < sc->autoload_names_top; i++)
	    {
	      sc->autoload_names[i] = NULL;
	      sc->autoload_names_sizes[i] = 0;
	      sc->autoloaded_already[i] = NULL;
	    }
	}
    }

  sc->autoload_names[sc->autoload_names_loc] = names;
  sc->autoload_names_sizes[sc->autoload_names_loc] = size;
  sc->autoloaded_already[sc->autoload_names_loc] = (bool *)calloc(size, sizeof(bool));
  sc->autoload_names_loc++;
}

static const char *find_autoload_name(s7_scheme *sc, s7_pointer symbol, bool *already_loaded, bool loading)
{
  s7_int l = 0, pos = -1, lib, libs;
  const char *name, *this_name;

  name = symbol_name(symbol);
  libs = sc->autoload_names_loc;

  for (lib = 0; lib < libs; lib++)
    {
      const char **names;
      s7_int u;
      u = sc->autoload_names_sizes[lib] - 1;
      names = sc->autoload_names[lib];

      while (true)
	{
	  s7_int comp;
	  if (u < l) break;
	  pos = (l + u) / 2;
	  this_name = names[pos * 2];
	  comp = strcmp(this_name, name);
	  if (comp == 0)
	    {
	      *already_loaded = sc->autoloaded_already[lib][pos];
	      if (loading) sc->autoloaded_already[lib][pos] = true;
	      return(names[pos * 2 + 1]);             /* file name given func name */
	    }
	  if (comp < 0)
	    l = pos + 1;
	  else u = pos - 1;
	}
    }
  return(NULL);
}

s7_pointer s7_autoload(s7_scheme *sc, s7_pointer symbol, s7_pointer file_or_function)
{
  /* add '(symbol . file) to s7's autoload table */
  if (is_null(sc->autoload_table))
    sc->autoload_table = s7_make_hash_table(sc, sc->default_hash_table_length);
  s7_hash_table_set(sc, sc->autoload_table, symbol, file_or_function);
  return(file_or_function);
}

static s7_pointer g_autoload(s7_scheme *sc, s7_pointer args)
{
  #define H_autoload "(autoload symbol file-or-function) adds the symbol to its table of autoloadable symbols. \
If that symbol is encountered as an unbound variable, s7 either loads the file (following *load-path*), or calls \
the function.  The function takes one argument, the calling environment.  Presumably the symbol is defined \
in the file, or by the function."
  #define Q_autoload s7_make_signature(sc, 3, sc->T, sc->is_symbol_symbol, sc->T)

  s7_pointer sym, value;

  sym = car(args);
  if (is_string(sym))
    {
      if (string_length(sym) == 0)                   /* (autoload "" ...) */
	return(s7_wrong_type_arg_error(sc, "autoload", 1, sym, "a symbol-name or a symbol"));
      sym = make_symbol_with_length(sc, string_value(sym), string_length(sym));
    }
  if (!is_symbol(sym))
    {
      check_method(sc, sym, sc->autoload_symbol, args);
      return(s7_wrong_type_arg_error(sc, "autoload", 1, sym, "a string (symbol-name) or a symbol"));
    }
  if (is_keyword(sym))
    return(s7_wrong_type_arg_error(sc, "autoload", 1, sym, "a normal symbol (a keyword is never unbound)"));

  value = cadr(args);
  if (is_string(value))
    return(s7_autoload(sc, sym, s7_immutable(make_string_with_length(sc, string_value(value), string_length(value)))));
  if (((is_closure(value)) || (is_closure_star(value))) &&
      (s7_is_aritable(sc, value, 1)))
    return(s7_autoload(sc, sym, value));

  check_method(sc, value, sc->autoload_symbol, args);
  return(s7_wrong_type_arg_error(sc, "autoload", 2, value, "a string (file-name) or a thunk"));
}

static s7_pointer g_autoloader(s7_scheme *sc, s7_pointer args)
{
  #define H_autoloader "(*autoload* sym) returns the autoload info for the symbol sym, or #f."
  #define Q_autoloader s7_make_signature(sc, 2, sc->T, sc->is_symbol_symbol)
  s7_pointer sym;

  sym = car(args);
  if (!is_symbol(sym))
    {
      check_method(sc, sym, sc->autoloader_symbol, args);
      return(s7_wrong_type_arg_error(sc, "*autoload*", 1, sym, "a symbol"));
    }
  if (sc->autoload_names)
    {
      const char *file;
      bool loaded = false;
      file = find_autoload_name(sc, sym, &loaded, false);
      if (file)
	return(s7_make_string(sc, file));
    }
  if (is_hash_table(sc->autoload_table))
    return(s7_hash_table_ref(sc, sc->autoload_table, sym));

  return(sc->F);
}


/* ---------------- require ---------------- */
#if S7_DEBUGGING
#define unstack(sc) unstack_1(sc, __func__, __LINE__)
static void unstack_1(s7_scheme *sc, const char *func, int line)
{
  sc->stack_end -= 4;
  if (((opcode_t)sc->stack_end[3]) != OP_GC_PROTECT) fprintf(stderr, "%s[%d]: popped %s?\n", func, line, op_names[(opcode_t)sc->stack_end[3]]);
}
#else
#define unstack(sc) sc->stack_end -= 4
#endif

static s7_pointer g_require(s7_scheme *sc, s7_pointer args)
{
  #define H_require "(require symbol . symbols) loads each file associated with each symbol if it has not been loaded already.\
The symbols refer to the argument to \"provide\".  (require lint.scm)"
  /* #define Q_require s7_make_circular_signature(sc, 1, 2, sc->T, sc->is_symbol_symbol) */

  s7_pointer p;
  push_stack_no_let_no_code(sc, OP_GC_PROTECT, args);
  for (p = args; is_pair(p); p = cdr(p))
    {
      s7_pointer sym;
      if (is_symbol(car(p)))
	sym = car(p);
      else
	{
	  if ((is_proper_quote(sc, car(p))) &&
	      (is_symbol(cadar(p))))
	    sym = cadar(p);
	  else return(s7_error(sc, sc->wrong_type_arg_symbol, set_elist_2(sc, wrap_string(sc, "require: ~S is not a symbol", 27), car(p))));
	}

      if ((!is_slot(symbol_to_slot(sc, sym))) &&
	  (sc->is_autoloading))
	{
	  s7_pointer f;
	  f = g_autoloader(sc, list_1(sc, sym));
	  if (is_string(f))
	    s7_load_with_environment(sc, string_value(f), sc->envir);
	  else return(s7_error(sc, sc->autoload_error_symbol, set_elist_2(sc, wrap_string(sc, "require: no autoload info for ~S", 32), sym)));
	}
    }
  unstack(sc);
  return(sc->T);
}


static bool is_memq(s7_pointer sym, s7_pointer lst)
{
  s7_pointer x;
  for (x = lst; is_pair(x); x = cdr(x))
    if (sym == car(x))
      return(true);
  return(false);
}

/* ---------------- provided? ---------------- */
static s7_pointer g_is_provided(s7_scheme *sc, s7_pointer args)
{
  #define H_is_provided "(provided? symbol) returns #t if symbol is a member of the *features* list"
  #define Q_is_provided s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_symbol_symbol)
  s7_pointer sym, topf, x;

  sym = car(args);
  if (!is_symbol(sym))
    return(method_or_bust_one_arg(sc, sym, sc->is_provided_symbol, list_1(sc, sym), T_SYMBOL));

  /* here the *features* list is spread out (or can be anyway) along the curlet chain,
   *   so we need to travel back all the way to the top level checking each *features* list in turn.
   *   Since *features* grows via cons (newest first), we can stop the scan if we hit the shared
   *   top-level at least.
   */
  topf = slot_value(global_slot(sc->features_symbol));
  if (is_memq(sym, topf))
    return(sc->T);

  if (is_global(sc->features_symbol))
    return(sc->F);
  for (x = sc->envir; symbol_id(sc->features_symbol) < let_id(x); x = outlet(x));
  for (; is_let(x); x = outlet(x))
    {
      s7_pointer y;
      for (y = let_slots(x); is_slot(y); y = next_slot(y))
	if (slot_symbol(y) == sc->features_symbol)
	  {
	    if ((slot_value(y) != topf) &&
		(is_memq(sym, slot_value(y))))
	      return(sc->T);
	  }
    }
  return(sc->F);
}

bool s7_is_provided(s7_scheme *sc, const char *feature)
{
  return(is_memq(s7_make_symbol(sc, feature), s7_symbol_value(sc, sc->features_symbol))); /* this goes from local outward */
}

static bool is_provided_b_7p(s7_scheme *sc, s7_pointer sym)
{
  if (!is_symbol(sym))
    simple_wrong_type_argument(sc, sc->is_provided_symbol, sym, T_SYMBOL);
  return(is_memq(sym, s7_symbol_value(sc, sc->features_symbol)));
}


/* ---------------- provide ---------------- */
static s7_pointer c_provide(s7_scheme *sc, s7_pointer sym)
{
  /* this has to be relative to the curlet: (load file env)
   *   the things loaded are only present in env, and go away with it, so should not be in the global *features* list
   */
  s7_pointer p;
  if (!is_symbol(sym))
    return(method_or_bust_one_arg(sc, sym, sc->provide_symbol, list_1(sc, sym), T_SYMBOL));

  p = symbol_to_local_slot(sc, sc->features_symbol, sc->envir); /* if sc->envir is nil, this returns the global slot, else local slot */
  if ((is_slot(p)) && (is_immutable(p)))
    s7_warn(sc, 256, "provide: *features* is immutable!\n");
  else
    {
      s7_pointer lst;
      lst = slot_value(symbol_to_slot(sc, sc->features_symbol));    /* in either case, we want the current *features* list */
      if (p == sc->undefined)
	make_slot_1(sc, sc->envir, sc->features_symbol, cons(sc, sym, lst));
      else
	{
	  if (!is_memq(sym, lst))
	    slot_set_value(p, cons(sc, sym, lst));
	  /* if two different provide statements provide the same symbol, is that an error?
	   * Should we warn about it if safety>0? similarly should autoload warn about an overwrite?
	   */
	}
    }

  /* require looks up its symbol argument to see if the associated code (dsp.scm etc) has been autoloaded,
   *   so here we're defining the provided symbol for that possibility.  Perhaps better would be to forgo
   *   the definition here, and in require check *features* -- it is local to the current env.
   */
  if (!is_slot(symbol_to_slot(sc, sym))) /* *features* name might be the same as an existing function */
    s7_define(sc, sc->envir, sym, sym);

  return(sym);
}

static s7_pointer g_provide(s7_scheme *sc, s7_pointer args)
{
  #define H_provide "(provide symbol) adds symbol to the *features* list"
  #define Q_provide s7_make_signature(sc, 2, sc->is_symbol_symbol, sc->is_symbol_symbol)

  if ((is_immutable(sc->envir)) &&
      (sc->envir != sc->nil))
    s7_error(sc, sc->error_symbol,
	     set_elist_2(sc, wrap_string(sc, "can't provide '~S (current environment is immutable)", 52), car(args)));

  return(c_provide(sc, car(args)));
}

void s7_provide(s7_scheme *sc, const char *feature) {c_provide(sc, s7_make_symbol(sc, feature));}

static s7_pointer g_features_set(s7_scheme *sc, s7_pointer args)
{
  /* setter for set/let of *features* which can only be changed via provide */
  if (s7_is_list(sc, cadr(args)))
    return(cadr(args));
  return(s7_error(sc, sc->error_symbol, set_elist_2(sc, wrap_string(sc, "can't set *features* to ~S", 26), cadr(args))));
}


/* -------------------------------- eval-string -------------------------------- */

s7_pointer s7_eval_c_string_with_environment(s7_scheme *sc, const char *str, s7_pointer e)
{
  s7_pointer code, port;
  TRACK(sc);
  port = s7_open_input_string(sc, str);
  code = s7_read(sc, port);
  s7_close_input_port(sc, port);
  return(s7_eval(sc, T_Pos(code), e));
}

s7_pointer s7_eval_c_string(s7_scheme *sc, const char *str)
{
  return(s7_eval_c_string_with_environment(sc, str, sc->nil));
}

static s7_pointer g_eval_string(s7_scheme *sc, s7_pointer args)
{
  #define H_eval_string "(eval-string str (env (curlet))) returns the result of evaluating the string str as Scheme code"
  #define Q_eval_string s7_make_signature(sc, 3, sc->values_symbol, sc->is_string_symbol, sc->is_let_symbol)
  s7_pointer port, str;

  str = car(args);
  if (!is_string(str))
    return(method_or_bust(sc, str, sc->eval_string_symbol, args, T_STRING, 1));

  if (is_not_null(cdr(args)))
    {
      s7_pointer e;
      e = cadr(args);
      if (!is_let(e))
 	return(wrong_type_argument_with_type(sc, sc->eval_string_symbol, 2, e, a_let_string));
      if (e == sc->rootlet)
	sc->envir = sc->nil;
      else sc->envir = e;
    }

  port = open_and_protect_input_string(sc, str);
  push_input_port(sc, port);

  sc->temp3 = sc->args;
  push_stack(sc, OP_EVAL_STRING, args, sc->code);
  push_stack_op_let(sc, OP_READ_INTERNAL);

  return(sc->F);
}

static s7_pointer eval_string_chooser(s7_scheme *sc, s7_pointer f, int32_t args, s7_pointer expr, bool ops)
{
  check_for_substring_temp(sc, expr);
  return(f);
}

static s7_pointer op_eval_string(s7_scheme *sc)
{
  while (s7_peek_char(sc, sc->input_port) != eof_object) /* (eval-string "(+ 1 2) this is a mistake") */
    {
      int32_t tk;
      tk = token(sc);                             /* (eval-string "(+ 1 2) ; a comment (not a mistake)") */
      if (tk != TOKEN_EOF)
	{
	  s7_int trail_len;
	  s7_pointer trail_data;
	  trail_len = port_data_size(sc->input_port) - port_position(sc->input_port) + 1;
	  if (trail_len > 32) trail_len = 32;
	  trail_data = make_string_with_length(sc, (const char *)(port_data(sc->input_port) + port_position(sc->input_port) - 1), trail_len);
	  s7_close_input_port(sc, sc->input_port);
	  pop_input_port(sc);
	  s7_error(sc, sc->read_error_symbol,
		   set_elist_2(sc, wrap_string(sc, "eval-string trailing junk: ~S", 29), trail_data));
	}
    }
  s7_close_input_port(sc, sc->input_port);
  pop_input_port(sc);
  sc->code = sc->value;
  return(NULL);
}


/* -------------------------------- call-with-input-string -------------------------------- */

static s7_pointer call_with_input(s7_scheme *sc, s7_pointer port, s7_pointer args)
{
  s7_pointer p;
  p = cadr(args);
  port_original_input_string(port) = car(args);
  push_stack(sc, OP_UNWIND_INPUT, sc->input_port, port);
  push_stack(sc, OP_APPLY, list_1(sc, port), p);
  return(sc->F);
}

static s7_pointer g_call_with_input_string(s7_scheme *sc, s7_pointer args)
{
  s7_pointer str, proc;
  #define H_call_with_input_string "(call-with-input-string str proc) opens a string port for str and applies proc to it"
  #define Q_call_with_input_string sc->pl_sf
  /* (call-with-input-string "44" (lambda (p) (+ 1 (read p)))) -> 45 */

  str = car(args);
  if (!is_string(str))
    return(method_or_bust(sc, str, sc->call_with_input_string_symbol, args, T_STRING, 1));

  proc = cadr(args);
  if (is_let(proc))
    check_method(sc, proc, sc->call_with_input_string_symbol, args);

  if (!s7_is_aritable(sc, proc, 1))
    return(wrong_type_argument_with_type(sc, sc->call_with_input_string_symbol, 2, proc,
					 wrap_string(sc, "a procedure of one argument (the port)", 38)));

  if ((is_continuation(proc)) || (is_goto(proc)))
    return(wrong_type_argument_with_type(sc, sc->call_with_input_string_symbol, 2, proc, a_normal_procedure_string));

  return(call_with_input(sc, open_and_protect_input_string(sc, str), args));
}


/* -------------------------------- call-with-input-file -------------------------------- */

static s7_pointer g_call_with_input_file(s7_scheme *sc, s7_pointer args)
{
  #define H_call_with_input_file "(call-with-input-file filename proc) opens filename and calls proc with the input port as its argument"
  #define Q_call_with_input_file sc->pl_sf
  s7_pointer str, proc;

  str = car(args);
  if (!is_string(str))
    return(method_or_bust(sc, str, sc->call_with_input_file_symbol, args, T_STRING, 1));

  proc = cadr(args);
  if (!s7_is_aritable(sc, proc, 1))
    return(wrong_type_argument_with_type(sc, sc->call_with_input_file_symbol, 2, proc,
					 wrap_string(sc, "a procedure of one argument (the port)", 38)));
  if ((is_continuation(proc)) || (is_goto(proc)))
    return(wrong_type_argument_with_type(sc, sc->call_with_input_file_symbol, 2, proc, a_normal_procedure_string));

  return(call_with_input(sc, open_input_file_1(sc, string_value(str), "r", "call-with-input-file"), args));
}


/* -------------------------------- with-input-from-string -------------------------------- */

static s7_pointer with_input(s7_scheme *sc, s7_pointer port, s7_pointer args)
{
  s7_pointer old_input_port, p;
  old_input_port = sc->input_port;
  sc->input_port = port;
  port_original_input_string(port) = car(args);
  push_stack(sc, OP_UNWIND_INPUT, old_input_port, port);
  p = cadr(args);
  push_stack(sc, OP_APPLY, sc->nil, p);
  return(sc->F);
}

static s7_pointer g_with_input_from_string(s7_scheme *sc, s7_pointer args)
{
  #define H_with_input_from_string "(with-input-from-string str thunk) opens str as the temporary current-input-port and calls thunk"
  #define Q_with_input_from_string sc->pl_sf
  s7_pointer str;

  str = car(args);
  if (!is_string(str))
    return(method_or_bust(sc, str, sc->with_input_from_string_symbol, args, T_STRING, 1));

  if (cadr(args) == slot_value(global_slot(sc->read_symbol)))
    {
      if (string_length(str) == 0)
	return(eof_object);
      if (char_0[(uint8_t)(string_value(str)[0])])
	return(make_atom(sc, string_value(str), BASE_10, SYMBOL_OK, WITHOUT_OVERFLOW_ERROR));
    }
  if (!is_thunk(sc, cadr(args)))
    return(method_or_bust_with_type(sc, cadr(args), sc->with_input_from_string_symbol, args, a_thunk_string, 2));

  /* since the arguments are evaluated before we get here, we can get some confusing situations:
   *   (with-input-from-string "#x2.1" (read))
   *     (read) -> whatever it can get from the current input port!
   *     ";with-input-from-string argument 2, #<eof>, is untyped but should be a thunk"
   *   (with-input-from-string "" (read-line)) -> hangs awaiting stdin input
   */
  return(with_input(sc, open_and_protect_input_string(sc, str), args));
}



/* -------------------------------- with-input-from-file -------------------------------- */

static s7_pointer g_with_input_from_file(s7_scheme *sc, s7_pointer args)
{
  #define H_with_input_from_file "(with-input-from-file filename thunk) opens filename as the temporary current-input-port and calls thunk"
  #define Q_with_input_from_file sc->pl_sf

  if (!is_string(car(args)))
    return(method_or_bust(sc, car(args), sc->with_input_from_file_symbol, args, T_STRING, 1));

  if (!is_thunk(sc, cadr(args)))
    return(method_or_bust_with_type(sc, cadr(args), sc->with_input_from_file_symbol, args, a_thunk_string, 2));

  return(with_input(sc, open_input_file_1(sc, string_value(car(args)), "r", "with-input-from-file"), args));
}


/* -------------------------------- iterators -------------------------------- */

#if S7_DEBUGGING
static s7_pointer titr_let(s7_pointer p, const char *func, int32_t line)
{
  if (!is_let(iterator_sequence(p)))
    {
      fprintf(stderr, "%s%s[%d]: let iterator sequence is %s%s\n", BOLD_TEXT, func, line, check_name(unchecked_type(iterator_sequence(p))), UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(p);
}

static s7_pointer titr_pair(s7_pointer p, const char *func, int32_t line)
{
  if (!is_pair(iterator_sequence(p)))
    {
      fprintf(stderr, "%s%s[%d]: pair iterator sequence is %s%s\n", BOLD_TEXT, func, line, check_name(unchecked_type(iterator_sequence(p))), UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(p);
}

static s7_pointer titr_hash(s7_pointer p, const char *func, int32_t line)
{
  if (!is_hash_table(iterator_sequence(p)))
    {
      fprintf(stderr, "%s%s[%d]: hash iterator sequence is %s%s\n", BOLD_TEXT, func, line, check_name(unchecked_type(iterator_sequence(p))), UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(p);
}

static s7_pointer titr_len(s7_pointer p, const char *func, int32_t line)
{
  if ((is_hash_table(iterator_sequence(p))) ||
      (is_pair(iterator_sequence(p))))
    {
      fprintf(stderr, "%s%s[%d]: iterator length sequence is %s%s\n", BOLD_TEXT, func, line, check_name(unchecked_type(iterator_sequence(p))), UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(p);
}

static s7_pointer titr_pos(s7_scheme *sc, s7_pointer p, const char *func, int32_t line)
{
  if (((is_let(iterator_sequence(p))) &&
       (iterator_sequence(p) != sc->rootlet)) ||
      (is_pair(iterator_sequence(p))))
    {
      fprintf(stderr, "%s%s[%d]: iterator position sequence is %s%s\n", BOLD_TEXT, func, line, check_name(unchecked_type(iterator_sequence(p))), UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  return(p);
}
#endif

/* -------------------------------- iterator? -------------------------------- */
static s7_pointer g_is_iterator(s7_scheme *sc, s7_pointer args)
{
  #define H_is_iterator "(iterator? obj) returns #t if obj is an iterator."
  #define Q_is_iterator sc->pl_bt
  s7_pointer x;

  x = car(args);
  if (is_iterator(x)) return(sc->T);
  check_closure_for(sc, x, sc->local_iterator_symbol);
  check_boolean_method(sc, is_iterator, sc->is_iterator_symbol, args);
  return(sc->F);
}

bool s7_is_iterator(s7_pointer obj) {return(is_iterator(obj));}
static bool is_iterator_b_7p(s7_scheme *sc, s7_pointer obj) {return(g_is_iterator(sc, set_plist_1(sc, obj)) != sc->F);}


static s7_pointer iterator_copy(s7_scheme *sc, s7_pointer p)
{
  /* fields are obj cur [loc|lcur] [len|slow|hcur] next, but untangling them in debugging case is a pain */
  s7_pointer iter;
  new_cell(sc, iter, T_ITERATOR | T_SAFE_PROCEDURE);
  memcpy((void *)iter, (void *)p, sizeof(s7_cell));
  return(iter);
}

static s7_pointer iterator_finished(s7_scheme *sc, s7_pointer iterator)
{
  return(ITERATOR_END);
}

static s7_pointer iterator_quit(s7_pointer iterator)
{
  iterator_next(iterator) = iterator_finished;
  clear_iter_ok(iterator);
  return(ITERATOR_END);
}

static s7_pointer let_iterate(s7_scheme *sc, s7_pointer iterator)
{
  s7_pointer slot;
  slot = iterator_current_slot(iterator);
  if (is_slot(slot))
    {
      iterator_set_current_slot(iterator, next_slot(slot));
      if (iterator_let_cons(iterator))
	{
	  s7_pointer p;
	  p = iterator_let_cons(iterator);
	  set_car(p, slot_symbol(slot));
	  set_cdr(p, slot_value(slot));
	  return(p);
	}
      return(cons(sc, slot_symbol(slot), slot_value(slot)));
    }
  return(iterator_quit(iterator));
}

static s7_pointer rootlet_iterate(s7_scheme *sc, s7_pointer iterator)
{
  s7_pointer slot;
  slot = iterator_current(iterator);
  if (is_slot(slot))
    {
      if (iterator_position(iterator) < sc->rootlet_entries)
	{
	  iterator_position(iterator)++;
	  iterator_current(iterator) = rootlet_element(sc->rootlet, iterator_position(iterator));
	}
      else iterator_current(iterator) = sc->nil;
      return(cons(sc, slot_symbol(slot), slot_value(slot)));
    }
  return(iterator_quit(iterator));
}

static s7_pointer hash_table_iterate(s7_scheme *sc, s7_pointer iterator)
{
  s7_pointer table;
  s7_int loc, len;
  hash_entry_t **elements;
  hash_entry_t *lst;

  lst = iterator_hash_current(iterator);
  if (lst)
    {
      iterator_hash_current(iterator) = hash_entry_next(lst);
      if (iterator_current(iterator))
	{
	  s7_pointer p;
	  p = iterator_current(iterator);
	  set_car(p, hash_entry_key(lst));
	  set_cdr(p, hash_entry_value(lst));
	  return(p);
	}
      return(cons(sc, hash_entry_key(lst), hash_entry_value(lst)));
    }

  table = iterator_sequence(iterator); /* using iterator_length and hash_table_entries here was slightly slower */
  len = hash_table_mask(table) + 1;
  elements = hash_table_elements(table);

  for (loc = iterator_position(iterator) + 1; loc < len;  loc++)
    {
      hash_entry_t *x;
      x = elements[loc];
      if (x)
	{
	  iterator_position(iterator) = loc;
	  iterator_hash_current(iterator) = hash_entry_next(x);
	  if (iterator_current(iterator))
	    {
	      s7_pointer p;
	      p = iterator_current(iterator);
	      set_car(p, hash_entry_key(x));
	      set_cdr(p, hash_entry_value(x));
	      return(p);
	    }
	  return(cons(sc, hash_entry_key(x), hash_entry_value(x)));
	}
    }
  return(iterator_quit(iterator));
}

static s7_pointer string_iterate(s7_scheme *sc, s7_pointer obj)
{
  if (iterator_position(obj) < iterator_length(obj))
    return(s7_make_character(sc, (uint8_t)(string_value(iterator_sequence(obj))[iterator_position(obj)++])));
  return(iterator_quit(obj));
}

static s7_pointer byte_vector_iterate(s7_scheme *sc, s7_pointer obj)
{
  if (iterator_position(obj) < iterator_length(obj))
    return(small_int(byte_vector_bytes(iterator_sequence(obj))[iterator_position(obj)++]));
  return(iterator_quit(obj));
}

static s7_pointer float_vector_iterate(s7_scheme *sc, s7_pointer obj)
{
  if (iterator_position(obj) < iterator_length(obj))
    return(make_real(sc, float_vector(iterator_sequence(obj), iterator_position(obj)++)));
  return(iterator_quit(obj));
}

static s7_pointer int_vector_iterate(s7_scheme *sc, s7_pointer obj)
{
  if (iterator_position(obj) < iterator_length(obj))
    return(make_integer(sc, int_vector(iterator_sequence(obj), iterator_position(obj)++)));
  return(iterator_quit(obj));
}

static s7_pointer vector_iterate(s7_scheme *sc, s7_pointer obj)
{
  if (iterator_position(obj) < iterator_length(obj))
    return(vector_element(iterator_sequence(obj), iterator_position(obj)++));
  return(iterator_quit(obj));
}

static s7_pointer closure_iterate(s7_scheme *sc, s7_pointer obj)
{
  s7_pointer result;
  result = s7_call(sc, iterator_sequence(obj), sc->nil);
  /* this can't use s7_apply_function -- we need to catch the error handler's longjmp here */
  if (result == ITERATOR_END)
    {
      iterator_next(obj) = iterator_finished;
      clear_iter_ok(obj);
    }
  return(result);
}

static s7_pointer c_object_iterate(s7_scheme *sc, s7_pointer obj)
{
  if (iterator_position(obj) < iterator_length(obj))
    {
      s7_pointer result, p, cur;
      p = iterator_sequence(obj);
      cur = iterator_current(obj);
      set_car(sc->z2_1, sc->x);
      set_car(sc->z2_2, sc->z); /* is this necessary? */
      set_car(cur, p);
      set_car(cdr(cur), make_integer(sc, iterator_position(obj)));
      result = (*(c_object_ref(sc, p)))(sc, cur);
      sc->x = car(sc->z2_1);
      sc->z = car(sc->z2_2);
      iterator_position(obj)++;
      if (result == ITERATOR_END)
	{
	  iterator_next(obj) = iterator_finished;
	  clear_iter_ok(obj);
	}
      return(result);
    }
  return(iterator_quit(obj));
}


static s7_pointer pair_iterate_1(s7_scheme *sc, s7_pointer obj);
static s7_pointer pair_iterate(s7_scheme *sc, s7_pointer obj)
{
  if (is_pair(iterator_current(obj)))
    {
      s7_pointer result;
      result = car(iterator_current(obj));
      iterator_current(obj) = cdr(iterator_current(obj));
      if (iterator_current(obj) == iterator_slow(obj))
	iterator_current(obj) = sc->nil;
      iterator_next(obj) = pair_iterate_1;
      return(result);
    }
  return(iterator_quit(obj));
}

static s7_pointer pair_iterate_1(s7_scheme *sc, s7_pointer obj)
{
  if (is_pair(iterator_current(obj)))
    {
      s7_pointer result;
      result = car(iterator_current(obj));
      iterator_current(obj) = cdr(iterator_current(obj));
      if (iterator_current(obj) == iterator_slow(obj))
	iterator_current(obj) = sc->nil;
      else iterator_set_slow(obj, cdr(iterator_slow(obj)));
      iterator_next(obj) = pair_iterate;
      return(result);
    }
  return(iterator_quit(obj));
}

static s7_pointer iterator_method(s7_scheme *sc, s7_pointer e)
{
  s7_pointer func;
  if ((has_methods(e)) &&
      ((func = find_method(sc, find_let(sc, e), sc->make_iterator_symbol)) != sc->undefined))
    {
      s7_pointer it;
      it = s7_apply_function(sc, func, list_1(sc, e));
      if (!is_iterator(it))
	return(s7_error(sc, sc->error_symbol, set_elist_2(sc, wrap_string(sc, "make-iterator method must return an interator: ~S", 49), it)));
      return(it);
    }
  return(NULL);
}

/* -------------------------------- make-iterator -------------------------------- */
s7_pointer s7_make_iterator(s7_scheme *sc, s7_pointer e)
{
  s7_pointer iter;

  new_cell(sc, iter, T_ITERATOR | T_SAFE_PROCEDURE | T_ITER_OK);
  iterator_sequence(iter) = e;
  if ((!is_let(e)) && (!is_pair(e)))
    iterator_position(iter) = 0;

  switch (type(e))
    {
    case T_LET:
      if (e == sc->rootlet)
	{
	  iterator_current(iter) = rootlet_element(e, 0); /* unfortunately tricky -- let_iterate uses different fields */
	  iterator_position(iter) = 0;
	  iterator_next(iter) = rootlet_iterate;
	}
      else
	{
	  s7_pointer f;
	  sc->temp6 = iter;
	  f = iterator_method(sc, e);
	  sc->temp6 = sc->nil;
	  if (f) {free_cell(sc, iter); return(f);}
	  iterator_set_current_slot(iter, let_slots(e));
	  iterator_next(iter) = let_iterate;
	  iterator_let_cons(iter) = NULL;
	}
      break;

    case T_HASH_TABLE:
      iterator_hash_current(iter) = NULL;
      iterator_current(iter) = NULL;
      iterator_position(iter) = -1;
      iterator_next(iter) = hash_table_iterate;
      break;

    case T_STRING:
      iterator_length(iter) = string_length(e);
      iterator_next(iter) = string_iterate;
      break;

    case T_BYTE_VECTOR:
      iterator_length(iter) = byte_vector_length(e);
      iterator_next(iter) = byte_vector_iterate;
      break;

    case T_VECTOR:
      iterator_length(iter) = vector_length(e);
      iterator_next(iter) = vector_iterate;
      break;

    case T_INT_VECTOR:
      iterator_length(iter) = vector_length(e);
      iterator_next(iter) = int_vector_iterate;
      break;

    case T_FLOAT_VECTOR:
      iterator_length(iter) = vector_length(e);
      iterator_next(iter) = float_vector_iterate;
      break;

    case T_PAIR:
      iterator_current(iter) = e;
      iterator_next(iter) = pair_iterate;
      iterator_set_slow(iter, e);
      break;

    case T_NIL: /* (make-iterator #()) -> #<iterator: vector>, so I guess () should also work */
      iterator_length(iter) = 0;
      iterator_next(iter) = iterator_finished;
      clear_iter_ok(iter);
      break;

    case T_MACRO:   case T_MACRO_STAR:
    case T_BACRO:   case T_BACRO_STAR:
    case T_CLOSURE: case T_CLOSURE_STAR:
      {
	s7_pointer p;
	p = cons(sc, e, sc->nil);
	if (g_is_iterator(sc, p) != sc->F) /* this checks sc->local_iterator_symbol: +iterator+ */
	  {
	    set_car(p, small_int(0));
	    iterator_current(iter) = p;
	    set_mark_seq(iter);
	    iterator_next(iter) = closure_iterate;
	    if (has_methods(e))
	      iterator_length(iter) = closure_length(sc, e);
	    else iterator_length(iter) = s7_int_max;
	  }
	else
	  {
	    free_cell(sc, iter);
	    return(simple_wrong_type_argument_with_type(sc, sc->make_iterator_symbol, e,
							wrap_string(sc, "a closure/macro with a '+iterator+ local that is not #f", 55)));
	  }
      }
      break;

    case T_C_OBJECT:
      {
	s7_pointer f;
	iterator_length(iter) = c_object_length_to_int(sc, e);
	sc->temp6 = iter;
	f = iterator_method(sc, e);
	sc->temp6 = sc->nil;
	if (f) {free_cell(sc, iter); return(f);}
	iterator_current(iter) = list_2(sc, e, small_int(0));
	set_mark_seq(iter);
	iterator_next(iter) = c_object_iterate;
      }
      break;

    default:
      return(simple_wrong_type_argument_with_type(sc, sc->make_iterator_symbol, e, a_sequence_string));
    }
  return(iter);
}

static s7_pointer g_make_iterator(s7_scheme *sc, s7_pointer args)
{
  #define H_make_iterator "(make-iterator sequence) returns an iterator object that returns the next value \
in the sequence each time it is called.  When it reaches the end, it returns " ITERATOR_END_NAME "."
  #define Q_make_iterator s7_make_signature(sc, 3, sc->is_iterator_symbol, sc->is_sequence_symbol, sc->is_pair_symbol)

  s7_pointer iter;
  /* we need to call s7_make_iterator before fixing up the optional second arg in case let->method */
  iter = s7_make_iterator(sc, car(args));

  if (is_pair(cdr(args)))
    {
      s7_pointer ip;
      ip = cadr(args);
      if (is_pair(ip))
	{
	  if (is_immutable_pair(ip))
	    return(immutable_object_error(sc, set_elist_3(sc, immutable_error_string, sc->make_iterator_symbol, ip)));
	  if (is_hash_table(iterator_sequence(iter)))
	    {
	      iterator_current(iter) = ip;
	      set_mark_seq(iter);
	    }
	  else
	    {
	      if ((is_let(iterator_sequence(iter))) &&
		  (iterator_sequence(iter) != sc->rootlet))
		{
		  iterator_let_cons(iter) = ip;
		  set_mark_seq(iter);
		}
	    }
	}
      else return(simple_wrong_type_argument(sc, sc->make_iterator_symbol, ip, T_PAIR));
    }
  return(iter);
}

/* -------------------------------- iterate -------------------------------- */
static s7_pointer g_iterate(s7_scheme *sc, s7_pointer args)
{
  #define H_iterate "(iterate obj) returns the next element from the iterator obj, or " ITERATOR_END_NAME "."
  #define Q_iterate s7_make_signature(sc, 2, sc->T, sc->is_iterator_symbol)

  s7_pointer iter;
  iter = car(args);
  if (!is_iterator(iter))
    return(method_or_bust_one_arg(sc, iter, sc->iterate_symbol, args, T_ITERATOR));
  return((iterator_next(iter))(sc, iter));
}

static s7_pointer iterate_p_p(s7_scheme *sc, s7_pointer iter)
{
  if (!is_iterator(iter))
    return(method_or_bust_one_arg(sc, iter, sc->iterate_symbol, set_plist_1(sc, iter), T_ITERATOR));
  return((iterator_next(iter))(sc, iter));
}

s7_pointer s7_iterate(s7_scheme *sc, s7_pointer obj)
{
  return((iterator_next(obj))(sc, obj));
}

bool s7_iterator_is_at_end(s7_scheme *sc, s7_pointer obj)
{
  if (iter_ok(obj))
    return(false);
  if (!is_iterator(obj))
    simple_wrong_type_argument(sc, sc->iterator_is_at_end_symbol, obj, T_ITERATOR);
  return(true);
}

static bool op_iterate(s7_scheme *sc)
{
  s7_pointer s;
  s = symbol_to_value_checked(sc, car(sc->code));
  if (!is_iterator(s)) {sc->last_function = s; return(false);}
  sc->value = (iterator_next(s))(sc, s);
  return(true);
}


/* -------------------------------- iterator-at-end? -------------------------------- */
static bool iterator_is_at_end_b_7p(s7_scheme *sc, s7_pointer obj)
{
  if (iter_ok(obj))
    return(false);
  if (!is_iterator(obj))
    simple_wrong_type_argument(sc, sc->iterator_is_at_end_symbol, obj, T_ITERATOR);
  return(true);
}

static s7_pointer g_iterator_is_at_end(s7_scheme *sc, s7_pointer args)
{
  #define H_iterator_is_at_end "(iterator-at-end? iter) returns #t if the iterator has reached the end of its sequence."
  #define Q_iterator_is_at_end s7_make_signature(sc, 2, sc->is_boolean_symbol, sc->is_iterator_symbol)
  s7_pointer iter;

  iter = car(args);
  if (iter_ok(iter))
    return(sc->F);
  if (!is_iterator(iter))
    return(simple_wrong_type_argument(sc, sc->iterator_is_at_end_symbol, iter, T_ITERATOR));
  return(sc->T);
}

/* -------------------------------- iterator-sequence -------------------------------- */
static s7_pointer g_iterator_sequence(s7_scheme *sc, s7_pointer args)
{
  #define H_iterator_sequence "(iterator-sequence iterator) returns the sequence that iterator is traversing."
  #define Q_iterator_sequence s7_make_signature(sc, 2, sc->is_sequence_symbol, sc->is_iterator_symbol)

  s7_pointer iter;

  iter = car(args);
  if (!is_iterator(iter))
    return(simple_wrong_type_argument(sc, sc->iterator_sequence_symbol, iter, T_ITERATOR));
  return(iterator_sequence(iter));
}


/* -------- cycles -------- */

#define INITIAL_SHARED_INFO_SIZE 8

static int32_t shared_ref(shared_info *ci, s7_pointer p)
{
  /* from print after collecting refs, not called by equality check */
  int32_t i;
  s7_pointer *objs;

  if (!is_collected(p)) return(0);

  objs = ci->objs;
  for (i = 0; i < ci->top; i++)
    if (objs[i] == p)
      {
	int32_t val;
	val = ci->refs[i];
	if (val > 0)
	  ci->refs[i] = -ci->refs[i];
	return(val);
      }
  return(0);
}

static void flip_ref(shared_info *ci, s7_pointer p)
{
  int32_t i;
  s7_pointer *objs;
  objs = ci->objs;
  for (i = 0; i < ci->top; i++)
    if (objs[i] == p)
      {
	ci->refs[i] = -ci->refs[i];
	break;
      }
}

static int32_t peek_shared_ref(shared_info *ci, s7_pointer p)
{
  /* returns 0 if not found, otherwise the ref value for p */
  int32_t i;
  s7_pointer *objs;

  objs = ci->objs;
  for (i = 0; i < ci->top; i++)
    if (objs[i] == p) return(ci->refs[i]);

  return(0);
}

static void enlarge_shared_info(shared_info *ci)
{
  int32_t i;
  ci->size *= 2;
  ci->size2 = ci->size - 2;
  ci->objs = (s7_pointer *)realloc(ci->objs, ci->size * sizeof(s7_pointer));
  ci->refs = (int32_t *)realloc(ci->refs, ci->size * sizeof(int32_t));
  ci->defined = (bool *)realloc(ci->defined, ci->size * sizeof(bool));
  for (i = ci->top; i < ci->size; i++)
    {
      ci->refs[i] = 0;
      ci->objs[i] = NULL;
    }
}

static void add_shared_ref(shared_info *ci, s7_pointer x, int32_t ref_x)
{
  /* called only in equality check, not printer */

  if (ci->top == ci->size)
    enlarge_shared_info(ci);

  set_collected(x);

  ci->objs[ci->top] = x;
  ci->refs[ci->top++] = ref_x;
}

static bool collect_shared_info(s7_scheme *sc, shared_info *ci, s7_pointer top, bool stop_at_print_length);
static hash_entry_t *hash_equal(s7_scheme *sc, s7_pointer table, s7_pointer key);

static bool check_collected(s7_pointer top, shared_info *ci)
{
  s7_pointer *p, *objs_end;
  int32_t i;
  objs_end = (s7_pointer *)(ci->objs + ci->top);
  for (p = ci->objs; p < objs_end; p++)
    if ((*p) == top)
      {
	i = (int32_t)(p - ci->objs);
	if (ci->refs[i] == 0)
	  {
	    ci->has_hits = true;
	    ci->refs[i] = ++ci->ref;  /* if found, set the ref number */
	  }
	break;
      }
  set_cyclic(top);
  return(true);
}

static bool collect_vector_info(s7_scheme *sc, shared_info *ci, s7_pointer top, bool stop_at_print_length)
{
  s7_int i, plen;
  bool cyclic = false;

  if (stop_at_print_length)
    {
      plen = sc->print_length;
      if (plen > vector_length(top))
	plen = vector_length(top);
    }
  else plen = vector_length(top);

  for (i = 0; i < plen; i++)
    {
      s7_pointer vel;
      vel = unchecked_vector_element(top, i);   /* "unchecked" because top might be rootlet, I think */
      if ((has_structure(vel)) &&
	  (collect_shared_info(sc, ci, vel, stop_at_print_length)))
	{
	  set_cyclic(vel);
	  cyclic = true;
	  if ((is_c_pointer(vel)) ||
	      (is_iterator(vel)) ||
	      (is_c_object(vel)))
	    check_collected(top, ci);
	}
    }
  if (cyclic) set_cyclic(top);
  return(cyclic);
}

static bool collect_shared_info(s7_scheme *sc, shared_info *ci, s7_pointer top, bool stop_at_print_length)
{
  /* look for top in current list.
   *
   * As we collect objects (guaranteed to have structure) we set the collected bit.  If we ever
   *   encounter an object with that bit on, we've seen it before so we have a possible cycle.
   *   Once the collection pass is done, we run through our list, and clear all these bits.
   */
  bool top_cyclic;

  if (is_collected_or_shared(top))
    {
      if (is_shared(top))
	return(false);
      return(check_collected(top, ci));
    }

  /* top not seen before -- add it to the list */
  set_collected(top);

  if (ci->top == ci->size)
    enlarge_shared_info(ci);
  ci->objs[ci->top++] = top;

  top_cyclic = false;
  /* now search the rest of this structure */
  if (is_pair(top))
    {
      s7_pointer p, cp;
      if ((has_structure(car(top))) &&
	  (collect_shared_info(sc, ci, car(top), stop_at_print_length)))
	top_cyclic = true;

      for (p = cdr(top); is_pair(p); p = cdr(p))
	{
	  if (is_collected_or_shared(p))
	    {
	      set_cyclic(top);
	      set_cyclic(p);
	      if (is_shared(p))
		{
		  if (!top_cyclic)
		    for (cp = top; cp != p; cp = cdr(cp)) set_shared(cp);
		  return(top_cyclic);
		}
	      return(check_collected(p, ci));
	    }
	  set_collected(p);
	  if (ci->top == ci->size)
	    enlarge_shared_info(ci);
	  ci->objs[ci->top++] = p;
	  if ((has_structure(car(p))) &&
	      (collect_shared_info(sc, ci, car(p), stop_at_print_length)))
	    top_cyclic = true;
	}
      if ((has_structure(p)) &&
	  (collect_shared_info(sc, ci, p, stop_at_print_length)))
	{
	  set_cyclic(top);
	  return(true);
	}
      if (!top_cyclic)
	for (cp = top; is_pair(cp); cp = cdr(cp)) set_shared(cp);
      else set_cyclic(top);
      return(top_cyclic);
    }

  switch (type(top))
    {
    case T_VECTOR:
      if (collect_vector_info(sc, ci, top, stop_at_print_length))
	top_cyclic = true;
      break;

    case T_ITERATOR:
      if (collect_shared_info(sc, ci, iterator_sequence(top), stop_at_print_length))
	{
	  if (peek_shared_ref(ci, iterator_sequence(top)) == 0)
	    check_collected(iterator_sequence(top), ci);
	  top_cyclic = true;
	}
      break;

    case T_HASH_TABLE:
      if (hash_table_entries(top) > 0)
	{
	  s7_int i, len;
	  hash_entry_t **entries;
	  bool keys_safe;

	  keys_safe = ((hash_table_checker(top) != hash_equal) && (!hash_table_checker_locked(top)));
	  entries = hash_table_elements(top);
	  len = hash_table_mask(top) + 1;
	  for (i = 0; i < len; i++)
	    {
	      hash_entry_t *p;
	      for (p = entries[i]; p; p = hash_entry_next(p))
		{
		  if ((!keys_safe) &&
		      (has_structure(hash_entry_key(p))) &&
		      (collect_shared_info(sc, ci, hash_entry_key(p), stop_at_print_length)))
		    top_cyclic = true;
		  if ((has_structure(hash_entry_value(p))) &&
		      (collect_shared_info(sc, ci, hash_entry_value(p), stop_at_print_length)))
		    {
		      if ((is_c_pointer(hash_entry_value(p))) ||
			  (is_iterator(hash_entry_value(p))) ||
			  (is_c_object(hash_entry_value(p))))
			check_collected(top, ci);
		      top_cyclic = true;
		    }
		}
	    }
	}
      break;

    case T_SLOT: /* this can be hit if we somehow collect_shared_info on sc->rootlet via collect_vector_info (see the let case below) */
      if ((has_structure(slot_value(top))) &&
	  (collect_shared_info(sc, ci, slot_value(top), stop_at_print_length)))
	top_cyclic = true;
      break;

    case T_LET:
      if (top == sc->rootlet)
	{
	  if (collect_vector_info(sc, ci, top, stop_at_print_length))
	    top_cyclic = true;
	}
      else
	{
	  s7_pointer p, q;
	  for (q = top; is_let(q) && (q != sc->rootlet); q = outlet(q))
	    {
	      for (p = let_slots(q); is_slot(p); p = next_slot(p))
		if ((has_structure(slot_value(p))) &&
		    (collect_shared_info(sc, ci, slot_value(p), stop_at_print_length)))
		  {
		    top_cyclic = true;
		    if ((is_c_pointer(slot_value(p))) ||
			(is_iterator(slot_value(p))) ||
			(is_c_object(slot_value(p))))
		      check_collected(top, ci);
		  }
	    }
	}
      break;

    case T_CLOSURE:
    case T_CLOSURE_STAR:
      if (collect_shared_info(sc, ci, closure_body(top), stop_at_print_length))
	{
	  if (peek_shared_ref(ci, top) == 0)
	    check_collected(top, ci);
	  top_cyclic = true;
	}
      break;

    case T_C_POINTER:
      if ((has_structure(c_pointer_type(top))) &&
	  (collect_shared_info(sc, ci, c_pointer_type(top), stop_at_print_length)))
	{
	  if (peek_shared_ref(ci, c_pointer_type(top)) == 0)
	    check_collected(c_pointer_type(top), ci);
	  top_cyclic = true;
	}
      if ((has_structure(c_pointer_info(top))) &&
	  (collect_shared_info(sc, ci, c_pointer_info(top), stop_at_print_length)))
	{
	  if (peek_shared_ref(ci, c_pointer_info(top)) == 0)
	    check_collected(c_pointer_info(top), ci);
	  top_cyclic = true;
	}
      break;

    case T_C_OBJECT:
      if ((c_object_to_list(sc, top)) &&
	  (c_object_set(sc, top)) &&
	  (collect_shared_info(sc, ci, (*(c_object_to_list(sc, top)))(sc, set_plist_1(sc, top)), stop_at_print_length)))
	{
	  if (peek_shared_ref(ci, top) == 0)
	    check_collected(top, ci);
	  top_cyclic = true;
	}
      break;
    }

  if (!top_cyclic)
    set_shared(top);
  else set_cyclic(top);
  return(top_cyclic);
}

static shared_info *init_circle_info(s7_scheme *sc)
{
  shared_info *ci;
  ci = (shared_info *)calloc(1, sizeof(shared_info));
  ci->size = INITIAL_SHARED_INFO_SIZE;
  ci->size2 = ci->size - 2;
  ci->objs = (s7_pointer *)malloc(ci->size * sizeof(s7_pointer));
  ci->refs = (int32_t *)calloc(ci->size, sizeof(int32_t));   /* finder expects 0 = unseen previously */
  ci->defined = (bool *)calloc(ci->size, sizeof(bool));
  ci->cycle_port = sc->F;
  ci->init_port = sc->F;
  return(ci);
}

static inline shared_info *new_shared_info(s7_scheme *sc)
{
  shared_info *ci;
  ci = sc->circle_info;
  if (ci->top > 0)
    {
      int32_t i;
      memclr((void *)(ci->refs), ci->top * sizeof(int32_t));
      memclr((void *)(ci->defined), ci->top * sizeof(bool));
      for (i = 0; i < ci->top; i++)
	clear_cyclic_bits(ci->objs[i]); /* LOOP_4 is not faster */
      ci->top = 0;
    }
  ci->ref = 0;
  ci->has_hits = false;
  return(ci);
}

static shared_info *make_shared_info(s7_scheme *sc, s7_pointer top, bool stop_at_print_length)
{
  /* for the printer */
  shared_info *ci;
  int32_t i, refs;
  s7_pointer *ci_objs;
  int32_t *ci_refs;
  bool no_problem = true, cyclic = false;
  s7_int k, stop_len;

  /* check for simple cases first */
  if (is_pair(top))
    {
      s7_pointer x;
      x = top;
      if (stop_at_print_length)
	{
	  s7_pointer slow;
	  stop_len = sc->print_length;
	  slow = top;
	  for (k = 0; k < stop_len; k += 2)
	    {
	      if (!is_pair(x))
		break;
	      if (has_structure(car(x)))
		{
		  no_problem = false;
		  break;
		}
	      x = cdr(x);
	      if (!is_pair(x))
		break;
	      if (has_structure(car(x)))
		{
		  no_problem = false;
		  break;
		}
	      x = cdr(x);
	      slow = cdr(slow);
	      if (x == slow)
		{
		  no_problem = false;
		  break;
		}
	    }
	}
      else
	{
	  if (s7_list_length(sc, top) == 0) /* it is circular at the top level (following cdr) */
	    no_problem = false;
	  else
	    {
	      for (; is_pair(x); x = cdr(x))
		if (has_structure(car(x)))
		  {
		    /* it can help a little in some cases to scan vectors here (and slots):
		     *   if no element has structure, it's ok (maybe also hash_table_entries == 0)
		     */
		    no_problem = false;
		    break;
		  }
	    }
	}
      if ((no_problem) &&
	  (!is_null(x)) &&
	  (has_structure(x)))
	no_problem = false;

      if (no_problem)
	return(NULL);
    }
  else
    {
      if (is_any_vector(top))
	{
	  if (!is_normal_vector(top))
	    return(NULL);

	  stop_len = vector_length(top);
	  if ((stop_at_print_length) &&
	      (stop_len > sc->print_length))
	    stop_len = sc->print_length;

	  for (k = 0; k < stop_len; k++)
	    if (has_structure(vector_element(top, k)))
	      {
		no_problem = false;
		break;
	      }
	  if (no_problem)
	    return(NULL);
	}
    }

  ci = new_shared_info(sc);

  /* collect all pointers associated with top */
  cyclic = collect_shared_info(sc, ci, top, stop_at_print_length);

  ci_objs = ci->objs;
  for (i = 0; i < ci->top; i++)
    clear_collected_and_shared(ci_objs[i]);

  if (!cyclic)
    return(NULL);

  if (!(ci->has_hits))
    return(NULL);

  ci_refs = ci->refs;
  /* find if any were referenced twice (once for just being there, so twice=shared)
   *   we know there's at least one such reference because has_hits is true.
   */
  for (i = 0, refs = 0; i < ci->top; i++)
    if (ci_refs[i] > 0)
      {
	set_collected(ci_objs[i]);
	if (i == refs)
	  refs++;
	else
	  {
	    ci_objs[refs] = ci_objs[i];
	    ci_refs[refs++] = ci_refs[i];
	    ci_refs[i] = 0;
	    ci_objs[i] = NULL;
	  }
      }
  ci->top = refs;
  return(ci);
}


/* -------------------------------- cyclic-sequences -------------------------------- */

static s7_pointer cyclic_sequences(s7_scheme *sc, s7_pointer obj, bool return_list)
{
  if (has_structure(obj))
    {
      shared_info *ci;
      if (sc->object_out_locked)
	ci = sc->circle_info;
      else ci = make_shared_info(sc, obj, false); /* false=don't stop at print length (vectors etc) */
      if (ci)
	{
	  if (return_list)
	    {
	      int32_t i;
	      s7_pointer lst;
	      sc->w = sc->nil;
	      for (i = 0; i < ci->top; i++)
		sc->w = cons(sc, ci->objs[i], sc->w);
	      lst = sc->w;
	      sc->w = sc->nil;
	      return(lst);
	    }
	  else return(sc->T);
	}
    }
  return(sc->nil);
}

static s7_pointer g_cyclic_sequences(s7_scheme *sc, s7_pointer args)
{
  #define H_cyclic_sequences "(cyclic-sequences obj) returns a list of elements that are cyclic."
  #define Q_cyclic_sequences s7_make_signature(sc, 2, sc->is_proper_list_symbol, sc->T)
  return(cyclic_sequences(sc, car(args), true));
}


static int32_t circular_list_entries(s7_pointer lst)
{
  int32_t i;
  s7_pointer x;
  for (i = 1, x = cdr(lst); ; i++, x = cdr(x))
    {
      int32_t j;
      s7_pointer y;
      for (y = lst, j = 0; j < i; y = cdr(y), j++)
	if (x == y)
	  return(i);
    }
}

static void object_to_port_with_circle_check_1(s7_scheme *sc, s7_pointer vr, s7_pointer port, use_write_t use_write, shared_info *ci);
#define object_to_port_with_circle_check(Sc, Vr, Port, Use_Write, Ci) \
  do { \
      s7_pointer _V_ = Vr; \
      if ((Ci) && (has_structure(_V_))) \
        object_to_port_with_circle_check_1(Sc, _V_, Port, Use_Write, Ci); \
      else object_to_port(Sc, _V_, Port, Use_Write, Ci);\
     } while (0)

static void (*display_functions[256])(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write, shared_info *ci);
#define object_to_port(Sc, Obj, Port, Use_Write, Ci) (*display_functions[unchecked_type(Obj)])(Sc, Obj, Port, Use_Write, Ci)

static bool string_needs_slashification(const char *str, s7_int len)
{
  /* we have to go by len (str len) not *s==0 because s7 strings can have embedded nulls */
  uint8_t *p, *pend;
  pend = (uint8_t *)(str + len);
  for (p = (uint8_t *)str; p < pend; p++)
    if (slashify_table[*p])
      return(true);
  return(false);
}

#define IN_QUOTES true
#define NOT_IN_QUOTES false

static void slashify_string_to_port(s7_scheme *sc, s7_pointer port, const char *p, s7_int len, bool quoted)
{
  uint8_t *pcur, *pend, *pstart = NULL;

  if (len == 0)
    {
      if (quoted)
	port_write_string(port)(sc, "\"\"", 2, port);
      return;
    }
  pend = (uint8_t *)(p + len);

  /* what about the trailing nulls? Guile writes them out (as does s7 currently)
   *    but that is not ideal.  I'd like to use ~S for error messages, so that
   *    strings are clearly identified via the double-quotes, but this way of
   *    writing them is ugly:
   *    :(let ((str (make-string 8 #\null))) (set! (str 0) #\a) str)
   *    "a\x00\x00\x00\x00\x00\x00\x00"
   * but it would be misleading to omit them because:
   *    :(let ((str (make-string 8 #\null))) (set! (str 0) #\a) (string-append str "bc"))
   *    "a\x00\x00\x00\x00\x00\x00\x00bc"
   *
   * also it is problematic to use sc->print_length here because
   *    it is normally (say) 8 which truncates just about every string.  In CL, *print-length*
   *    does not affect strings, symbols, or bit-vectors.  But if the string is enormous,
   *    this function can bring us to a complete halt, which is annoying if it is the error
   *    handler that is trying to write the string.  I think if port is sc->error_port, I'll
   *    truncate strings to some reasonable size -- (*s7* 'max-format-length) maybe.
   */
  if ((len > sc->max_format_length) &&
      (port == sc->error_port))
    len = sc->max_format_length;
  if (quoted) port_write_character(port)(sc, '"', port);
  for (pcur = (uint8_t *)p; pcur < pend; pcur++)
    {
      if (slashify_table[*pcur])
	{
	  if (pstart) pstart++; else pstart = (uint8_t *)p;
	  if (pstart != pcur)
	    {
	      port_write_string(port)(sc, (char *)pstart, pcur - pstart, port);
	      pstart = pcur;
	    }
	  port_write_character(port)(sc, '\\', port);
	  switch (*pcur)
	    {
	    case '"':   port_write_character(port)(sc, '"', port);   break;
	    case '\\':  port_write_character(port)(sc, '\\', port);  break;
	    case '\'':  port_write_character(port)(sc, '\'', port);  break;
	    case '\t':  port_write_character(port)(sc, 't', port);   break;
	    case '\r':  port_write_character(port)(sc, 'r', port);   break;
	    case '\b':  port_write_character(port)(sc, 'b', port);   break;
	    case '\f':  port_write_character(port)(sc, 'f', port);   break;
	    case '\?':  port_write_character(port)(sc, '?', port);   break;
	    case 'x':   port_write_character(port)(sc, 'x', port);   break;
	    default:
	      {
		s7_int n;
		port_write_character(port)(sc, 'x', port);
		n = (s7_int)(*pcur);
		if (n < 16)
		  port_write_character(port)(sc, '0', port);
		else port_write_character(port)(sc, dignum[(n / 16) % 16], port);
		port_write_character(port)(sc, dignum[n % 16], port);
		port_write_character(port)(sc, ';', port);
	      }
	      break;
	    }
	}
    }
  if (!pstart)
    port_write_string(port)(sc, (char *)p, len, port);
  else
    {
      pstart++;
      if (pstart != pcur)
	port_write_string(port)(sc, (char *)pstart, pcur - pstart, port);
    }
  if (quoted) port_write_character(port)(sc, '"', port);
}

static void output_port_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write, shared_info *ci)
{
  if ((obj == sc->standard_output) ||
      (obj == sc->standard_error))
    port_write_string(port)(sc, port_filename(obj), port_filename_length(obj), port);
  else
    {
      if (use_write == P_READABLE)
	{
	  if (port_is_closed(obj))
	    port_write_string(port)(sc, "(let ((p (open-output-string))) (close-output-port p) p)", 56, port);
	  else
	    {
	      if (is_string_port(obj))
		{
		  port_write_string(port)(sc, "(let ((p (open-output-string)))", 31, port);
		  if (port_position(obj) > 0)
		    {
		      port_write_string(port)(sc, " (display ", 10, port);
		      slashify_string_to_port(sc, port, (const char *)port_data(obj), port_position(obj), IN_QUOTES);
		      port_write_string(port)(sc, " p)", 3, port);
		    }
		  port_write_string(port)(sc, " p)", 3, port);
		}
	      else
		{
		  char str[256];
		  int32_t nlen;
		  str[0] = '\0';
		  nlen = catstrs(str, 256, "(open-output-file \"", port_filename(obj), "\" \"a\")", NULL);
		  port_write_string(port)(sc, str, nlen, port);
		}
	    }
	}
      else
	{
	  if (is_string_port(obj))
	    port_write_string(port)(sc, "<output-string-port", 19, port);
	  else
	    {
	      if (is_file_port(obj))
		port_write_string(port)(sc, "<output-file-port", 17, port);
	      else port_write_string(port)(sc, "<output-function-port", 21, port);
	    }
	  if (port_is_closed(obj))
	    port_write_string(port)(sc, " (closed)>", 10, port);
	  else port_write_character(port)(sc, '>', port);
	}
    }
}

static void input_port_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write, shared_info *ci)
{
  if (obj == sc->standard_input)
    port_write_string(port)(sc, port_filename(obj), port_filename_length(obj), port);
  else
    {
      if (use_write == P_READABLE)
	{
	  if (port_is_closed(obj))
	    port_write_string(port)(sc, "(call-with-input-string \"\" (lambda (p) p))", 42, port);
	  else
	    {
	      if (is_function_port(obj))
		port_write_string(port)(sc, "#<input-function-port>", 22, port);
	      else
		{
		  if (is_file_port(obj))
		    {
		      char str[256];
		      int32_t nlen;
		      str[0] = '\0';
		      nlen = catstrs(str, 256, "(open-input-file \"", port_filename(obj), "\")", NULL);
		      port_write_string(port)(sc, str, nlen, port);
		    }
		  else
		    {
		      s7_int data_len;
		      data_len = port_data_size(obj) - port_position(obj);
		      if (data_len > 100)
			{
			  const char *filename;
			  filename = (const char *)s7_port_filename(sc, obj);
			  if (filename)
			    {
			      #define DO_STR_LEN 1024
			      char do_str[DO_STR_LEN];
			      int32_t len;
			      do_str[0] = '\0';
			      if (port_position(obj) > 0)
				{
				  len = catstrs(do_str, DO_STR_LEN, "(let ((port (open-input-file \"", filename, "\")))", NULL);
				  port_write_string(port)(sc, do_str, len, port);
				  do_str[0] = '\0';
				  len = catstrs(do_str, DO_STR_LEN, " (do ((i 0 (+ i 1)) (c (read-char port) (read-char port))) ((= i ",
						pos_int_to_str_direct(sc, port_position(obj) - 1),
						") port)))", NULL);
				}
			      else len = catstrs(do_str, DO_STR_LEN, "(open-input-file \"", filename, "\")", NULL);
			      port_write_string(port)(sc, do_str, len, port);
			      return;
			    }
			}
		      port_write_string(port)(sc, "(open-input-string ", 19, port);
		      /* not port_write_string here because there might be embedded double-quotes */
		      slashify_string_to_port(sc, port, (const char *)(port_data(obj) + port_position(obj)), port_data_size(obj) - port_position(obj), IN_QUOTES);
		      port_write_character(port)(sc, ')', port);
		    }
		}
	    }
	}
      else
	{
	  if (is_string_port(obj))
	    port_write_string(port)(sc, "<input-string-port", 18, port);
	  else
	    {
	      if (is_file_port(obj))
		port_write_string(port)(sc, "<input-file-port", 16, port);
	      else port_write_string(port)(sc, "<input-function-port", 20, port);
	    }
	  if (port_is_closed(obj))
	    port_write_string(port)(sc, " (closed)>", 10, port);
	  else port_write_character(port)(sc, '>', port);
	}
    }
}

static bool symbol_needs_slashification(s7_scheme *sc, s7_pointer obj)
{
  uint8_t *p, *pend;
  const char *str;
  s7_int len;

  str = symbol_name(obj);
  if ((str[0] == '#') || (str[0] == '\'') || (str[0] == ','))
    return(true);
  if (s7_string_to_number(sc, (char *)str, 10) != sc->F)
    return(true);

  len = symbol_name_length(obj);
  pend = (uint8_t *)(str + len);
  for (p = (uint8_t *)str; p < pend; p++)
    if (symbol_slashify_table[*p])
      return(true);

  set_clean_symbol(obj);
  return(false);
}

static inline void symbol_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write, shared_info *ci)
{
  /* I think this is the only place we print a symbol's name */
  if ((!is_clean_symbol(obj)) &&
      (symbol_needs_slashification(sc, obj)))
    {
      port_write_string(port)(sc, "(symbol \"", 9, port);
      slashify_string_to_port(sc, port, symbol_name(obj), symbol_name_length(obj), NOT_IN_QUOTES);
      port_write_string(port)(sc, "\")", 2, port);
    }
  else
    {
      if (!is_keyword(obj))
	{
	  if (use_write == P_READABLE)
	    port_write_character(port)(sc, '\'', port);
	  else
	    {
	      if (use_write == P_KEY)
		port_write_character(port)(sc, ':', port);
	    }
	}
      if (is_string_port(port))
	{
	  s7_int new_len;
	  new_len = port_position(port) + symbol_name_length(obj);
	  if (new_len >= port_data_size(port))
	    resize_port_data(sc, port, new_len * 2);
	  memcpy((void *)(port_data(port) + port_position(port)), (void *)symbol_name(obj), symbol_name_length(obj));
	  port_position(port) = new_len;
	}
      else port_write_string(port)(sc, symbol_name(obj), symbol_name_length(obj), port);
    }
}

static char *multivector_indices_to_string(s7_scheme *sc, s7_int index, s7_pointer vect, char *str, int32_t cur_dim)
{
  s7_int size, ind;

  size = vector_dimension(vect, cur_dim);
  ind = index % size;
  if (cur_dim > 0)
    multivector_indices_to_string(sc, (index - ind) / size, vect, str, cur_dim - 1);
  catstrs(str, 128, " ", pos_int_to_str_direct(sc, ind), NULL);
  return(str);
}

#define NOT_P_DISPLAY(Choice) ((Choice == P_DISPLAY) ? P_WRITE : Choice)

static int32_t multivector_to_port(s7_scheme *sc, s7_pointer vec, s7_pointer port,
				   int32_t out_len, int32_t flat_ref, int32_t dimension, int32_t dimensions, bool *last,
				   use_write_t use_write, shared_info *ci)
{
  int32_t i;

  if (use_write != P_READABLE)
    {
      if (*last)
	port_write_string(port)(sc, " (", 2, port);
      else port_write_character(port)(sc, '(', port);
      (*last) = false;
    }

  for (i = 0; i < vector_dimension(vec, dimension); i++)
    {
      if (dimension == (dimensions - 1))
	{
	  if (flat_ref < out_len)
	    {
	      object_to_port_with_circle_check(sc, vector_getter(vec)(sc, vec, flat_ref), port, NOT_P_DISPLAY(use_write), ci);

	      if (use_write == P_READABLE)
		port_write_string(port)(sc, ") ", 2, port);
	      flat_ref++;
	    }
	  else
	    {
	      port_write_string(port)(sc, "...)", 4, port);
	      return(flat_ref);
	    }
	  if ((use_write != P_READABLE) &&
	      (i < (vector_dimension(vec, dimension) - 1)))
	    port_write_character(port)(sc, ' ', port);
	}
      else
	{
	  if (flat_ref < out_len)
	    flat_ref = multivector_to_port(sc, vec, port, out_len, flat_ref, dimension + 1, dimensions, last, NOT_P_DISPLAY(use_write), ci);
	  else
	    {
	      port_write_string(port)(sc, "...)", 4, port);
	      return(flat_ref);
	    }
	}
    }
  if (use_write != P_READABLE)
    port_write_character(port)(sc, ')', port);
  (*last) = true;
  return(flat_ref);
}


static void make_vector_to_port(s7_scheme *sc, s7_pointer vect, s7_pointer port)
{
  s7_int vlen;
  int32_t plen;
  char buf[128];
  const char* vtyp = "";

  if (is_float_vector(vect))
    vtyp = "float-";
  else
    {
      if (is_int_vector(vect))
	vtyp = "int-";
      else
	{
	  if (is_byte_vector(vect))
	    vtyp = "byte-";
	}
    }

  vlen = vector_length(vect);
  if (vector_rank(vect) == 1)
    {
      plen = catstrs_direct(buf, "(make-", vtyp, "vector ", integer_to_string_no_length(sc, vlen), " ", NULL);
      port_write_string(port)(sc, buf, plen, port);
    }
  else
    {
      s7_int dim;
      plen = catstrs_direct(buf, "(make-", vtyp, "vector '(", NULL);
      port_write_string(port)(sc, buf, plen, port);
      for (dim = 0; dim < vector_ndims(vect) - 1; dim++)
	{
	  plen = catstrs_direct(buf, integer_to_string_no_length(sc, vector_dimension(vect, dim)), " ", NULL);
	  port_write_string(port)(sc, buf, plen, port);
	}
      plen = catstrs_direct(buf, integer_to_string_no_length(sc, vector_dimension(vect, dim)), ") ", NULL);
      port_write_string(port)(sc, buf, plen, port);
    }
}

static void write_vector_dimensions(s7_scheme *sc, s7_pointer vect, s7_pointer port)
{
  char buf[128];
  s7_int dim, plen;
  port_write_string(port)(sc, " '(", 3, port);
  for (dim = 0; dim < vector_ndims(vect) - 1; dim++)
    {
      plen = catstrs_direct(buf, integer_to_string_no_length(sc, vector_dimension(vect, dim)), " ", NULL);
      port_write_string(port)(sc, buf, plen, port);
    }
  plen = catstrs_direct(buf, integer_to_string_no_length(sc, vector_dimension(vect, dim)), "))", NULL);
  port_write_string(port)(sc, buf, plen, port);
}

static void vector_to_port(s7_scheme *sc, s7_pointer vect, s7_pointer port, use_write_t use_write, shared_info *ci)
{
  s7_int i, len, plen;
  bool too_long = false;
  char buf[128];

  len = vector_length(vect);
  if (len == 0)
    {
      if (vector_rank(vect) > 1)
	{
	  plen = catstrs_direct(buf, "#", pos_int_to_str_direct(sc, vector_ndims(vect)), "d()", NULL);
	  port_write_string(port)(sc, buf, plen, port);
	}
      else port_write_string(port)(sc, "#()", 3, port);
      return;
    }

  if (use_write != P_READABLE)
    {
      if (sc->print_length == 0)
	{
	  if (vector_rank(vect) > 1)
	    {
	      plen = catstrs_direct(buf, "#", pos_int_to_str_direct(sc, vector_ndims(vect)), "d(...)", NULL);
	      port_write_string(port)(sc, buf, plen, port);
	    }
	  else port_write_string(port)(sc, "#(...)", 6, port);
	  return;
	}
      if (len > sc->print_length)
	{
	  too_long = true;
	  len = sc->print_length;
	}
    }
  if ((!ci) &&
      (len > 1000))
    {
      s7_int vlen;
      s7_pointer p0;
      s7_pointer *els;
      vlen = vector_length(vect);
      els = vector_elements(vect);
      p0 = els[0];
      for (i = 1; i < vlen; i++)
	if (els[i] != p0)
	  break;
      if (i == vlen)
	{
	  make_vector_to_port(sc, vect, port);
	  object_to_port(sc, p0, port, use_write, NULL);
	  port_write_character(port)(sc, ')', port);
	  return;
	}
    }

  if (use_write == P_READABLE)
    {
      if ((ci) &&
	  (is_cyclic(vect)) &&
	  (peek_shared_ref(ci, vect) != 0))
	{
	  s7_pointer *els;
	  int32_t vref;
	  vref = peek_shared_ref(ci, vect);
	  if (vref < 0) vref = -vref;

	  els = vector_elements(vect);

	  if ((ci->defined[vref]) || (port == ci->cycle_port))
	    {
	      plen = catstrs_direct(buf, "<", pos_int_to_str_direct(sc, vref), ">", NULL);
	      port_write_string(port)(sc, buf, plen, port);
	      return;
	    }

	  if (vector_rank(vect) > 1)
	    port_write_string(port)(sc, "(subvector ", 11, port);

	  port_write_string(port)(sc, "(vector", 7, port); /* top level let */
	  for (i = 0; i < len; i++)
	    {
	      if (has_structure(els[i]))
		{
		  size_t len;
		  char *indices;
		  int32_t eref;
		  port_write_string(port)(sc, " #f", 3, port);
		  eref = peek_shared_ref(ci, els[i]);

		  if (eref != 0)
		    {
		      if (eref < 0) eref = -eref;
		      if (vector_rank(vect) > 1)
			{
			  s7_int dimension;
			  block_t *b;
			  dimension = vector_rank(vect) - 1;
			  b = callocate(sc, 128);
			  indices = (char *)block_data(b);
			  plen = catstrs_direct(buf, "(set! (<", pos_int_to_str_direct(sc, vref), ">",
						multivector_indices_to_string(sc, i, vect, indices, dimension),
						") <", pos_int_to_str_direct_1(sc, eref), ">)\n ", NULL);
			  port_write_string(ci->cycle_port)(sc, buf, plen, ci->cycle_port);
			  liberate(sc, b);
			}
		      else
			{
			  len = catstrs_direct(buf, "  (set! (<", pos_int_to_str_direct(sc, vref), "> ",
					       integer_to_string(sc, i, &plen), ") <",
					       pos_int_to_str_direct_1(sc, eref), ">)\n", NULL);
			  port_write_string(ci->cycle_port)(sc, buf, len, ci->cycle_port);
			}

		    }
		  else
		    {
		      if (vector_rank(vect) > 1)
			{
			  s7_int dimension;
			  block_t *b;
			  dimension = vector_rank(vect) - 1;
			  b = callocate(sc, 128);
			  indices = (char *)block_data(b);
			  buf[0] = '\0';
			  multivector_indices_to_string(sc, i, vect, indices, dimension); /* writes to indices */
			  plen = catstrs(buf, 128, "(set! (<", pos_int_to_str_direct(sc, vref), ">", indices, ") ", NULL);
			  port_write_string(ci->cycle_port)(sc, buf, plen, ci->cycle_port);
			  liberate(sc, b);
			}
		      else
			{
			  len = catstrs_direct(buf, "  (set! (<", pos_int_to_str_direct(sc, vref), "> ", integer_to_string_no_length(sc, i), ") ", NULL);
			  port_write_string(ci->cycle_port)(sc, buf, len, ci->cycle_port);
			}

		      object_to_port_with_circle_check(sc, els[i], ci->cycle_port, P_READABLE, ci);
		      port_write_string(ci->cycle_port)(sc, ")\n", 2, ci->cycle_port);
		    }
		}
	      else
		{
		  port_write_character(port)(sc, ' ', port);
		  object_to_port_with_circle_check(sc, els[i], port, P_READABLE, ci);
		}
	    }
	  port_write_character(port)(sc, ')', port);
	  if (vector_rank(vect) > 1)
	    write_vector_dimensions(sc, vect, port);
	}
      else
	{
	  if (vector_rank(vect) > 1)
	    port_write_string(port)(sc, "(subvector ", 11, port);

	  if (is_immutable_vector(vect))
	    port_write_string(port)(sc, "(immutable! ", 12, port);

	  port_write_string(port)(sc, "(vector", 7, port);
	  for (i = 0; i < len; i++)
	    {
	      port_write_character(port)(sc, ' ', port);
	      object_to_port_with_circle_check(sc, vector_element(vect, i), port, P_READABLE, ci);
	    }
	  port_write_character(port)(sc, ')', port);
	  if (is_immutable_vector(vect))
	    port_write_character(port)(sc, ')', port);

	  if (vector_rank(vect) > 1)
	    write_vector_dimensions(sc, vect, port);
	}
    }
  else /* not readable write */
    {
      if (vector_rank(vect) > 1)
	{
	  bool last = false;
	  if (vector_ndims(vect) > 1)
	    {
	      plen = catstrs_direct(buf, "#", pos_int_to_str_direct(sc, vector_ndims(vect)), "d", NULL);
	      port_write_string(port)(sc, buf, plen, port);
	    }
	  else port_write_character(port)(sc, '#', port);
	  multivector_to_port(sc, vect, port, len, 0, 0, vector_ndims(vect), &last, use_write, ci);
	}
      else
	{
	  port_write_string(port)(sc, "#(", 2, port);
	  for (i = 0; i < len - 1; i++)
	    {
	      object_to_port_with_circle_check(sc, vector_element(vect, i), port, NOT_P_DISPLAY(use_write), ci);
	      port_write_character(port)(sc, ' ', port);
	    }
	  object_to_port_with_circle_check(sc, vector_element(vect, i), port, NOT_P_DISPLAY(use_write), ci);

	  if (too_long)
	    port_write_string(port)(sc, " ...)", 5, port);
	  else port_write_character(port)(sc, ')', port);
	}
    }
}

static int32_t print_vector_length(s7_scheme *sc, s7_pointer vect, s7_pointer port, use_write_t use_write)
{
  int32_t len, plen;
  char buf[128];
  const char *vtype = "r";

  if (is_int_vector(vect))
    vtype = "i";
  else
    {
      if (is_byte_vector(vect))
	vtype = "u";
    }
  len = vector_length(vect);
  if (len == 0)
    {
      if (vector_rank(vect) > 1)
	plen = catstrs_direct(buf, "#", vtype, pos_int_to_str_direct(sc, vector_ndims(vect)), "d()", NULL);
      else plen = catstrs_direct(buf, "#", vtype, "()", NULL);
      port_write_string(port)(sc, buf, plen, port);
      return(-1);
    }

  if (use_write == P_READABLE)
    return(len);

  if (sc->print_length == 0)
    {
      if (vector_rank(vect) > 1)
	{
	  plen = catstrs_direct(buf, "#", vtype, pos_int_to_str_direct(sc, vector_ndims(vect)), "d(...)", NULL);
	  port_write_string(port)(sc, buf, plen, port);
	}
      else
	{
	  if (is_int_vector(vect))
	    port_write_string(port)(sc, "#i(...)", 7, port);
	  else
	    {
	      if (is_float_vector(vect))
		port_write_string(port)(sc, "#r(...)", 7, port);
	      else port_write_string(port)(sc, "#u(...)", 7, port);
	    }
	}
      return(-1);
    }

  if (len > sc->print_length)
    return(sc->print_length);
  return(len);
}

static void int_vector_to_port(s7_scheme *sc, s7_pointer vect, s7_pointer port, use_write_t use_write, shared_info *ignored)
{
  s7_int i, len, plen;
  bool too_long;
  char buf[128];
  char *p;

  len = print_vector_length(sc, vect, port, use_write);
  if (len < 0) return;
  too_long = (len < vector_length(vect));

  if ((use_write == P_READABLE) &&
      (is_immutable_vector(vect)))
    port_write_string(port)(sc, "(immutable! ", 12, port);

  if (len > 1000)
    {
      s7_int vlen;
      s7_int first;
      s7_int *els;
      vlen = vector_length(vect);
      els = int_vector_ints(vect);
      first = els[0];
      for (i = 1; i < vlen; i++)
	if (els[i] != first)
	  break;
      if (i == vlen)
	{
	  make_vector_to_port(sc, vect, port);
	  p = integer_to_string(sc, int_vector(vect, 0), &plen);
	  port_write_string(port)(sc, p, plen, port);
	  port_write_character(port)(sc, ')', port);
	  if ((use_write == P_READABLE) &&
	      (is_immutable_vector(vect)))
	    port_write_character(port)(sc, ')', port);
	  return;
	}
    }

  if (vector_rank(vect) == 1)
    {
      port_write_string(port)(sc, "#i(", 3, port);
      if (!is_string_port(port))
	{
	  p = integer_to_string(sc, int_vector(vect, 0), &plen);
	  port_write_string(port)(sc, p, plen, port);
	  for (i = 1; i < len; i++)
	    {
	      plen = catstrs_direct(buf, " ", integer_to_string_no_length(sc, int_vector(vect, i)), NULL);
	      port_write_string(port)(sc, buf, plen, port);
	    }
	}
      else
	{
	  /* an experiment */
	  s7_int new_len, next_len;
	  uint8_t *dbuf;
	  new_len = port_position(port);
	  next_len = port_data_size(port) - 128;
	  dbuf = port_data(port);

	  if (new_len >= next_len)
	    {
	      resize_port_data(sc, port, port_data_size(port) * 2);
	      next_len = port_data_size(port) - 128;
	      dbuf = port_data(port);
	    }
	  p = integer_to_string(sc, int_vector(vect, 0), &plen);
	  memcpy((void *)(dbuf + new_len), (void *)p, plen);
	  new_len += plen;
	  for (i = 1; i < len; i++)
	    {
	      if (new_len >= next_len)
		{
		  resize_port_data(sc, port, port_data_size(port) * 2);
		  next_len = port_data_size(port) - 128;
		  dbuf = port_data(port);
		}
	      plen = catstrs_direct(buf, " ", integer_to_string_no_length(sc, int_vector(vect, i)), NULL);
	      memcpy((void *)(dbuf + new_len), (void *)buf, plen);
	      new_len += plen;
	    }
	  port_position(port) = new_len;
	}

      if (too_long)
	port_write_string(port)(sc, " ...)", 5, port);
      else port_write_character(port)(sc, ')', port);
    }
  else
    {
      bool last = false;
      plen = catstrs_direct(buf, "#i", pos_int_to_str_direct(sc, vector_ndims(vect)), "d", NULL);
      port_write_string(port)(sc, buf, plen, port);
      multivector_to_port(sc, vect, port, len, 0, 0, vector_ndims(vect), &last, P_DISPLAY, NULL);
    }

  if ((use_write == P_READABLE) &&
      (is_immutable_vector(vect)))
    port_write_character(port)(sc, ')', port);
}

static void float_vector_to_port(s7_scheme *sc, s7_pointer vect, s7_pointer port, use_write_t use_write, shared_info *ignored)
{
  s7_int i, len, plen;
  bool too_long;
  char buf[128];
  s7_double *els;

  len = print_vector_length(sc, vect, port, use_write);
  if (len < 0) return;
  too_long = (len < vector_length(vect));
  els = float_vector_floats(vect);

  if ((use_write == P_READABLE) &&
      (is_immutable_vector(vect)))
    port_write_string(port)(sc, "(immutable! ", 12, port);

  if (len > 1000)
    {
      s7_int vlen;
      s7_double first;
      vlen = vector_length(vect);
      first = els[0];
      for (i = 1; i < vlen; i++)
	if (els[i] != first)
	  break;
      if (i == vlen)
	{
	  make_vector_to_port(sc, vect, port);
	  plen = snprintf(buf, 128, "%.*g)", sc->float_format_precision, first);
	  port_write_string(port)(sc, buf, plen, port);
	  if ((use_write == P_READABLE) &&
	      (is_immutable_vector(vect)))
	    port_write_character(port)(sc, ')', port);
	  return;
	}
    }

  if (vector_rank(vect) == 1)
    {
      port_write_string(port)(sc, "#r(", 3, port);
      plen = snprintf(buf, 124, "%.*g", sc->float_format_precision, els[0]); /* 124 so floatify has room */
      floatify(buf, &plen);
      port_write_string(port)(sc, buf, plen, port);
      for (i = 1; i < len; i++)
	{
	  plen = snprintf(buf, 124, " %.*g", sc->float_format_precision, els[i]);
	  floatify(buf, &plen);
	  port_write_string(port)(sc, buf, plen, port);
	}
      if (too_long)
	port_write_string(port)(sc, " ...)", 5, port);
      else port_write_character(port)(sc, ')', port);
    }
  else
    {
      bool last = false;
      plen = catstrs_direct(buf, "#r", pos_int_to_str_direct(sc, vector_ndims(vect)), "d", NULL);
      port_write_string(port)(sc, buf, plen, port);
      multivector_to_port(sc, vect, port, len, 0, 0, vector_ndims(vect), &last, P_DISPLAY, NULL);
    }

  if ((use_write == P_READABLE) &&
      (is_immutable_vector(vect)))
    port_write_character(port)(sc, ')', port);
}

static void byte_vector_to_port(s7_scheme *sc, s7_pointer vect, s7_pointer port, use_write_t use_write, shared_info *ignored)
{
  s7_int i, len, plen;
  bool too_long;
  char buf[128];
  char *p;

  len = print_vector_length(sc, vect, port, use_write);
  if (len < 0) return;
  too_long = (len < vector_length(vect));

  if ((use_write == P_READABLE) &&
      (is_immutable_vector(vect)))
    port_write_string(port)(sc, "(immutable! ", 12, port);

  if (len > 1000)
    {
      s7_int vlen;
      uint8_t first;
      uint8_t *els;
      vlen = vector_length(vect);
      els = byte_vector_bytes(vect);
      first = els[0];
      for (i = 1; i < vlen; i++)
	if (els[i] != first)
	  break;
      if (i == vlen)
	{
	  make_vector_to_port(sc, vect, port);
	  p = integer_to_string(sc, byte_vector(vect, 0), &plen);
	  port_write_string(port)(sc, p, plen, port);
	  port_write_character(port)(sc, ')', port);
	  if ((use_write == P_READABLE) &&
	      (is_immutable_vector(vect)))
	    port_write_character(port)(sc, ')', port);
	  return;
	}
    }

  if (vector_rank(vect) == 1)
    {
      port_write_string(port)(sc, "#u(", 3, port);
      p = integer_to_string(sc, byte_vector(vect, 0), &plen);
      port_write_string(port)(sc, p, plen, port);
      for (i = 1; i < len; i++)
	{
	  plen = catstrs_direct(buf, " ", integer_to_string_no_length(sc, byte_vector(vect, i)), NULL);
	  port_write_string(port)(sc, buf, plen, port);
	}
      if (too_long)
	port_write_string(port)(sc, " ...)", 5, port);
      else port_write_character(port)(sc, ')', port);
    }
  else
    {
      bool last = false;
      plen = catstrs_direct(buf, "#u", pos_int_to_str_direct(sc, vector_ndims(vect)), "d", NULL);
      port_write_string(port)(sc, buf, plen, port);
      multivector_to_port(sc, vect, port, len, 0, 0, vector_ndims(vect), &last, P_DISPLAY, NULL);
    }

  if ((use_write == P_READABLE) &&
      (is_immutable_vector(vect)))
    port_write_character(port)(sc, ')', port);
}


static void string_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write, shared_info *ignored)
{
  if ((use_write == P_READABLE) &&
      (is_immutable_string(obj)))
    port_write_string(port)(sc, "(immutable! ", 12, port);

  if (string_length(obj) > 0)
    {
      /* since string_length is a scheme length, not C, this write can embed nulls from C's point of view */
      if (string_length(obj) > 1000) /* was 10000 28-Feb-18 */
	{
	  size_t size;
	  char buf[128];
	  buf[0] = string_value(obj)[0];
	  buf[1] = '\0';
	  size = strspn((const char *)(string_value(obj) + 1), buf); /* if all #\null, this won't work */
	  if (size == (size_t)(string_length(obj) - 1))
	    {
	      int32_t nlen;
	      s7_pointer c;
	      c = chars[(int32_t)((uint8_t)(buf[0]))];
	      nlen = catstrs_direct(buf, "(make-string ", pos_int_to_str_direct(sc, string_length(obj)), " ", NULL);
	      port_write_string(port)(sc, buf, nlen, port);
	      port_write_string(port)(sc, character_name(c), character_name_length(c), port);
	      /* (string-ref (eval-string (object->string (make-string 14766 (integer->char 255)) :readable)) 0) */
	      port_write_character(port)(sc, ')', port);
	      return;
	    }
	}
      if (use_write == P_DISPLAY)
	port_write_string(port)(sc, string_value(obj), string_length(obj), port);
      else
	{
	  if (!string_needs_slashification(string_value(obj), string_length(obj)))
	    {
	      port_write_character(port)(sc, '"', port);
	      port_write_string(port)(sc, string_value(obj), string_length(obj), port);
	      port_write_character(port)(sc, '"', port);
	    }
	  else slashify_string_to_port(sc, port, string_value(obj), string_length(obj), IN_QUOTES);
	}
    }
  else
    {
      if (use_write != P_DISPLAY)
	port_write_string(port)(sc, "\"\"", 2, port);
    }

  if ((use_write == P_READABLE) &&
      (is_immutable_string(obj)))
    port_write_character(port)(sc, ')', port);
}

static void simple_list_readable_display(s7_scheme *sc, s7_pointer lst, s7_int true_len, s7_int len, s7_pointer port, shared_info *ci)
{
  /* the easier cases: no circles or shared refs to patch up */
  s7_pointer x;

  if (is_immutable(lst))
    port_write_string(port)(sc, "immutable! (", 12, port);

  if (true_len > 0)
    {
      port_write_string(port)(sc, "list", 4, port);
      for (x = lst; is_pair(x); x = cdr(x))
	{
	  port_write_character(port)(sc, ' ', port);
	  object_to_port_with_circle_check(sc, car(x), port, P_READABLE, ci);
	}
      port_write_character(port)(sc, ')', port);
    }
  else
    {
      s7_int i;
      port_write_string(port)(sc, "cons ", 5, port);
      object_to_port_with_circle_check(sc, car(lst), port, P_READABLE, ci);
      for (x = cdr(lst); is_pair(x); x = cdr(x))
	{
	  port_write_string(port)(sc, " (cons ", 7, port);
	  object_to_port_with_circle_check(sc, car(x), port, P_READABLE, ci);
	}
      port_write_character(port)(sc, ' ', port);
      object_to_port_with_circle_check(sc, x, port, P_READABLE, ci);
      for (i = 1; i < len; i++)
	port_write_character(port)(sc, ')', port);
    }
  if (is_immutable(lst))
    port_write_character(port)(sc, ')', port);
}

static void pair_to_port(s7_scheme *sc, s7_pointer lst, s7_pointer port, use_write_t use_write, shared_info *ci)
{
  /* we need list_to_starboard... (making port_write_string|character local was noticeable slower) */
  s7_pointer x;
  s7_int i, len, true_len;

  true_len = s7_list_length(sc, lst);
  if (true_len < 0)                    /* a dotted list -- handle cars, then final cdr */
    len = (-true_len + 1);
  else
    {
      if (true_len == 0)               /* circular list (nil is handled by unique_to_port) */
	len = circular_list_entries(lst);
      else len = true_len;
    }

  if ((use_write == P_READABLE) &&
      (ci) &&
      (peek_shared_ref(ci, lst) != 0))
    {
      int32_t href;
      href = peek_shared_ref(ci, lst);
      if (href < 0) href = -href;
      if ((ci->defined[href]) || (port == ci->cycle_port))
	{
	  char buf[128];
	  int32_t plen;
	  plen = catstrs_direct(buf, "<", pos_int_to_str_direct(sc, href), ">", NULL);
	  port_write_string(port)(sc, buf, plen, port);
	  return;
	}
    }

  if ((car(lst) == sc->quote_symbol) &&
      (true_len == 2))
    {
      /* len == 1 is important, otherwise (list 'quote 1 2) -> '1 2 which looks weird
       *   or (object->string (apply . `''1)) -> "'quote 1"
       * so (quote x) = 'x but (quote x y z) should be left alone (if evaluated, it's an error)
       */
      if (use_write == P_READABLE)
	{
	  if (is_immutable_pair(lst))
	    port_write_string(port)(sc, "(immutable! ", 12, port);
	  port_write_string(port)(sc, "''", 2, port);
	  object_to_port_with_circle_check(sc, cadr(lst), port, P_WRITE, ci);
	  if (is_immutable_pair(lst))
	    port_write_character(port)(sc, ')', port);
	}
      else
	{
	  port_write_character(port)(sc, '\'', port);
	  object_to_port_with_circle_check(sc, cadr(lst), port, P_WRITE, ci);
	}
      return;
    }
  else port_write_character(port)(sc, '(', port);

  if (is_multiple_value(lst))
    port_write_string(port)(sc, "values ", 7, port);

  if (use_write == P_READABLE)
    {
      if (!is_cyclic(lst))
	{
	  simple_list_readable_display(sc, lst, true_len, len, port, ci);
	  return;
	}
      if (ci)
	{
	  int32_t plen;
	  char buf[128], lst_name[128];
	  int32_t lst_ref;
	  bool lst_local = false;
	  s7_pointer local_port;

	  lst_ref = peek_shared_ref(ci, lst);
	  if (lst_ref == 0)
	    {
	      s7_pointer p;
	      for (p = lst; is_pair(p); p = cdr(p))
		if ((has_structure(car(p))) ||
		    ((is_pair(cdr(p))) &&
		     (peek_shared_ref(ci, cdr(p)) != 0)))
		  {
		    lst_name[0] = '<'; lst_name[1] = 'L'; lst_name[2] = '>'; lst_name[3] = '\0';
		    lst_local = true;
		    port_write_string(port)(sc, "let ((<L> (list", 15, port); /* '(' above */
		    break;
		  }
	      if ((!lst_local) && (!is_null(p)))
		{
		  if (has_structure(p))
		    {
		      lst_name[0] = '<'; lst_name[1] = 'L'; lst_name[2] = '>'; lst_name[3] = '\0';
		      lst_local = true;
		      port_write_string(port)(sc, "let ((<L> (list", 15, port); /* '(' above */
		    }
		}
	      if (!lst_local)
		{
		  simple_list_readable_display(sc, lst, true_len, len, port, ci);
		  return;
		}
	    }
	  else
	    {
	      if (lst_ref < 0) lst_ref = -lst_ref;
	      catstrs_direct(lst_name, "<", pos_int_to_str_direct(sc, lst_ref), ">", NULL);
	      port_write_string(port)(sc, "list", 4, port); /* '(' above */
	    }

	  for (i = 0, x = lst; (i < len) && (is_pair(x)); x = cdr(x), i++)
	    {
	      if ((has_structure(car(x))) &&
		  (is_cyclic(car(x))))
		port_write_string(port)(sc, " #f", 3, port);
	      else
		{
		  port_write_character(port)(sc, ' ', port);
		  object_to_port_with_circle_check(sc, car(x), port, use_write, ci);
		}
	      if ((is_pair(cdr(x))) &&
		  (peek_shared_ref(ci, cdr(x)) != 0))
		break;
	    }

	  if (lst_local)
	    port_write_string(port)(sc, ")))\n", 4, port);
	  else port_write_character(port)(sc, ')', port);

	  local_port = ((lst_local) || (ci->cycle_port == sc->F)) ? port : ci->cycle_port; /* (object->string (list-values `(x . 1) (signature (int-vector))) :readable) */
	  for (x = lst, i = 0; (i < len) && (is_pair(x)); x = cdr(x), i++)
	    {
	      if ((has_structure(car(x))) &&
		  (is_cyclic(car(x))))
		{
		  int32_t lref;
		  if (i == 0)
		    plen = catstrs_direct(buf, "  (set-car! ", lst_name, " ", NULL);
		  else plen = catstrs_direct(buf, "  (set! (", lst_name, " ", pos_int_to_str_direct(sc, i), ") ", NULL);
		  port_write_string(local_port)(sc, buf, plen, local_port);
		  lref = peek_shared_ref(ci, car(x));
		  if (lref == 0)
		    object_to_port_with_circle_check(sc, car(x), local_port, use_write, ci);
		  else
		    {
		      if (lref < 0) lref = -lref;
		      plen = catstrs_direct(buf, "<", pos_int_to_str_direct(sc, lref), ">", NULL);
		      port_write_string(local_port)(sc, buf, plen, local_port);
		    }
		  port_write_string(local_port)(sc, ")\n", 2, local_port);
		}
	      if ((is_pair(cdr(x))) &&
		  (peek_shared_ref(ci, cdr(x)) != 0))
		{
		  int32_t ref;
		  ref = peek_shared_ref(ci, cdr(x));
		  if (ref < 0) ref = -ref;
		  if (i == 0)
		    plen = catstrs_direct(buf, (lst_local) ? "    " : "  ", "(set-cdr! ", lst_name, " <", pos_int_to_str_direct(sc, ref), ">)\n", NULL);
		  else
		    {
		      if (i == 1)
			plen = catstrs_direct(buf, (lst_local) ? "    " : "  ", "(set-cdr! (cdr ", lst_name, ") <", pos_int_to_str_direct(sc, ref), ">)\n", NULL);
		      else plen = catstrs_direct(buf, (lst_local) ? "    " : "  ",
						 "(set-cdr! (list-tail ", lst_name, " ", pos_int_to_str_direct_1(sc, i),
						 ") <", pos_int_to_str_direct(sc, ref), ">)\n", NULL);
		    }
		  port_write_string(local_port)(sc, buf, plen, local_port);
		  break;
		}
	    }
	  if (true_len < 0) /* dotted list */
	    {
	      if (true_len == -1) /* cons cell */
		plen = catstrs_direct(buf, (lst_local) ? "    " : "  ", "(set-cdr! ", lst_name, " ", NULL);
	      else
		{
		  if (true_len == -2)
		    plen = catstrs_direct(buf, (lst_local) ? "    " : "  ", "(set-cdr! (cdr ", lst_name, ") ", NULL);
		  else plen = catstrs_direct(buf, "(set-cdr! (list-tail ", lst_name, " ", pos_int_to_str_direct(sc, len - 2), ") ", NULL);
		}
	      port_write_string(local_port)(sc, buf, plen, local_port);
	      object_to_port_with_circle_check(sc, x, local_port, use_write, ci);
	      port_write_string(local_port)(sc, ")\n", 2, local_port);
	    }

	  if (lst_local)
	    port_write_string(local_port)(sc, "    <L>)", 8, local_port);
	}
      else simple_list_readable_display(sc, lst, true_len, len, port, ci);
    }
  else /* not :readable */
    {
      s7_int plen;
      plen = (len > sc->print_length) ? sc->print_length : len;
      if (plen <= 0)
	{
	  port_write_string(port)(sc, "(list ...)", 10, port);
	  return;
	}

      if (ci)
	{
	  for (x = lst, i = 0; (is_pair(x)) && (i < plen) && ((i == 0) || (peek_shared_ref(ci, x) == 0)); i++, x = cdr(x))
	    {
	      object_to_port_with_circle_check(sc, car(x), port, NOT_P_DISPLAY(use_write), ci);
	      if (i < (len - 1))
		port_write_character(port)(sc, ' ', port);
	    }
	  if (is_not_null(x))
	    {
	      if (plen < len)
		port_write_string(port)(sc, " ...", 4, port);
	      else
		{
		  if ((true_len == 0) &&
		      (i == len))
		    port_write_string(port)(sc, " . ", 3, port);
		  else port_write_string(port)(sc, ". ", 2, port);
		  object_to_port_with_circle_check(sc, x, port, NOT_P_DISPLAY(use_write), ci);
		}
	    }
	  port_write_character(port)(sc, ')', port);
	}
      else
	{
	  s7_int len1;
	  len1 = plen - 1;
	  if (is_string_port(port))
	    {
	      for (x = lst, i = 0; (is_pair(x)) && (i < len1); i++, x = cdr(x))
		{
		  object_to_port(sc, car(x), port, NOT_P_DISPLAY(use_write), ci);
		  if (port_position(port) >= sc->objstr_max_len)
		    return;
		  if (port_position(port) >= port_data_size(port))
		    resize_port_data(sc, port, port_data_size(port) * 2);
		  port_data(port)[port_position(port)++] = (uint8_t)' ';
		}
	    }
	  else
	    {
	      for (x = lst, i = 0; (is_pair(x)) && (i < len1); i++, x = cdr(x))
		{
		  object_to_port(sc, car(x), port, NOT_P_DISPLAY(use_write), ci);
		  port_write_character(port)(sc, ' ', port);
		}
	    }
	  if (is_pair(x))
	    {
	      object_to_port(sc, car(x), port, NOT_P_DISPLAY(use_write), ci);
	      x = cdr(x);
	    }
	  if (is_not_null(x))
	    {
	      if (plen < len)
		port_write_string(port)(sc, " ...", 4, port);
	      else
		{
		  port_write_string(port)(sc, ". ", 2, port);
		  object_to_port(sc, x, port, NOT_P_DISPLAY(use_write), ci);
		}
	    }
	  port_write_character(port)(sc, ')', port);
	}
    }
}

static void hash_table_to_port(s7_scheme *sc, s7_pointer hash, s7_pointer port, use_write_t use_write, shared_info *ci)
{
  s7_int i, len, gc_iter;
  bool too_long = false;
  s7_pointer iterator, p;

  /* if hash is a member of ci, just print its number
   * (let ((ht (hash-table '(a . 1)))) (hash-table-set! ht 'b ht))
   *
   * since equal? does not care about the hash-table lengths, we can ignore that complication in the :readable case
   * there's no way to make a truly :readable version of a weak hash-table (or a normal hash-table that uses eq? with pairs, for example)
   */
  len = hash_table_entries(hash);
  if (len == 0)
    {
      if ((is_weak_hash_table(hash)) && (use_write == P_READABLE))
	port_write_string(port)(sc, "(make-weak-hash-table)", 22, port);
      else port_write_string(port)(sc, "(hash-table)", 12, port);
      return;
    }

  if (use_write != P_READABLE)
    {
      s7_int plen;
      plen = sc->print_length;
      if (plen <= 0)
	{
	  port_write_string(port)(sc, "(hash-table ...)", 16, port);
	  return;
	}
      if (len > plen)
	{
	  too_long = true;
	  len = plen;
	}
    }

  if ((use_write == P_READABLE) &&
      (ci) &&
      (peek_shared_ref(ci, hash) != 0))
    {
      int32_t href;
      href = peek_shared_ref(ci, hash);
      if (href < 0) href = -href;
      if ((ci->defined[href]) || (port == ci->cycle_port))
	{
	  char buf[128];
	  int32_t plen;
	  plen = catstrs_direct(buf, "<", pos_int_to_str_direct(sc, href), ">", NULL);
	  port_write_string(port)(sc, buf, plen, port);
	  return;
	}
    }

  iterator = s7_make_iterator(sc, hash);
  gc_iter = s7_gc_protect_1(sc, iterator);
  p = cons(sc, sc->F, sc->F);
  iterator_current(iterator) = p;
  set_mark_seq(iterator);

  if ((use_write == P_READABLE) &&
      (is_immutable(hash)))
    port_write_string(port)(sc, "(immutable! ", 12, port);

  if ((use_write == P_READABLE) &&
      (ci) &&
      (is_cyclic(hash)) &&
      (peek_shared_ref(ci, hash) != 0))
    {
      /* TODO: eq-func if user-set */
      /* (let ((<1> (make-hash-table <default-size> morally-equal?)))
       *   and then fill all fields in cycle_port
       *   hash_table_checker_locked?
       * also weak hash (let ((h (make-weak-hash-table))) then (set! (h key) val)?
       */

      int32_t href;
      href = peek_shared_ref(ci, hash);
      if (href < 0) href = -href;

      port_write_string(port)(sc, "(hash-table*", 12, port); /* top level let */
      for (i = 0; i < len; i++)
	{
	  s7_pointer key_val, key, val;

	  key_val = hash_table_iterate(sc, iterator);
	  key = car(key_val);
	  val = cdr(key_val);
	  if ((has_structure(val)) ||
	      (has_structure(key)))
	    {
	      char buf[128];
	      int32_t eref, kref, plen;
	      eref = peek_shared_ref(ci, val);
	      kref = peek_shared_ref(ci, key);
	      plen = catstrs_direct(buf, "(set! (<", pos_int_to_str_direct(sc, href), "> ", NULL);
	      port_write_string(ci->cycle_port)(sc, buf, plen, ci->cycle_port);

	      if (kref != 0)
		{
		  if (kref < 0) kref = -kref;
		  plen = catstrs_direct(buf, "<", pos_int_to_str_direct(sc, kref), ">", NULL);
		  port_write_string(ci->cycle_port)(sc, buf, plen, ci->cycle_port);
		}
	      else object_to_port(sc, key, ci->cycle_port, P_READABLE, ci);

	      if (eref != 0)
		{
		  if (eref < 0) eref = -eref;
		  plen = catstrs_direct(buf, ") <", pos_int_to_str_direct(sc, eref), ">)\n", NULL);
		  port_write_string(ci->cycle_port)(sc, buf, plen, ci->cycle_port);
		}
	      else
		{
		  port_write_string(ci->cycle_port)(sc, ") ", 2, ci->cycle_port);
		  object_to_port_with_circle_check(sc, val, ci->cycle_port, P_READABLE, ci);
		  port_write_string(ci->cycle_port)(sc, ")\n", 2, ci->cycle_port);
		}
	    }
	  else
	    {
	      port_write_character(port)(sc, ' ', port);
	      object_to_port_with_circle_check(sc, key, port, P_READABLE, ci);
	      port_write_character(port)(sc, ' ', port);
	      object_to_port_with_circle_check(sc, val, port, P_READABLE, ci);
	    }
	}
      port_write_character(port)(sc, ')', port);
    }
  else
    {
      port_write_string(port)(sc, "(hash-table*", 12, port);
      for (i = 0; i < len; i++)
	{
	  s7_pointer key_val;
	  port_write_character(port)(sc, ' ', port);
	  key_val = hash_table_iterate(sc, iterator);
	  if (use_write != P_READABLE)
	    {
	      if ((is_symbol(car(key_val))) &&
		  (!is_keyword(car(key_val))))
		port_write_character(port)(sc, '\'', port);
	    }
	  object_to_port_with_circle_check(sc, car(key_val), port, NOT_P_DISPLAY(use_write), ci);
	  port_write_character(port)(sc, ' ', port);
	  object_to_port_with_circle_check(sc, cdr(key_val), port, NOT_P_DISPLAY(use_write), ci);
	}
      if (too_long)
	port_write_string(port)(sc, " ...)", 5, port);
      else port_write_character(port)(sc, ')', port);
    }

  if ((use_write == P_READABLE) &&
      (is_immutable(hash)))
    port_write_character(port)(sc, ')', port);

  s7_gc_unprotect_at(sc, gc_iter);
  iterator_current(iterator) = sc->nil;
  free_cell(sc, p);
  free_cell(sc, iterator);
}

static int32_t slot_to_port_1(s7_scheme *sc, s7_pointer x, s7_pointer port, use_write_t use_write, shared_info *ci, int32_t n)
{
  if (is_slot(x))
    {
      n = slot_to_port_1(sc, next_slot(x), port, use_write, ci, n);
      if (n <= sc->print_length)
	{
	  port_write_character(port)(sc, ' ', port);
	  object_to_port_with_circle_check(sc, x, port, use_write, ci);
	}
      if (n == (sc->print_length + 1))
	port_write_string(port)(sc, " ...", 4, port);
    }
  return(n + 1);
}

static void slot_list_to_port(s7_scheme *sc, s7_pointer slot, s7_pointer port, shared_info *ci)
{
  if (is_slot(slot))
    {
      slot_list_to_port(sc, next_slot(slot), port, ci);
      port_write_character(port)(sc, ' ', port);
      symbol_to_port(sc, slot_symbol(slot), port, P_KEY, ci);  /* (object->string (inlet (symbol "(\")") 1) :readable) */
      port_write_character(port)(sc, ' ', port);
      object_to_port_with_circle_check(sc, slot_value(slot), port, P_READABLE, ci);
    }
}

static void slot_list_to_port_with_cycle(s7_scheme *sc, s7_pointer obj, s7_pointer slot, s7_pointer port, shared_info *ci)
{
  if (is_slot(slot))
    {
      s7_pointer sym, val;
      slot_list_to_port_with_cycle(sc, obj, next_slot(slot), port, ci);
      sym = slot_symbol(slot);
      val = slot_value(slot);

      port_write_character(port)(sc, ' ', port);
      symbol_to_port(sc, sym, port, P_KEY, ci);
      if (has_structure(val))
	{
	  char buf[128];
	  int32_t symref, len;
	  port_write_string(port)(sc, " #f", 3, port);

	  len = catstrs_direct(buf, "  (set! (<", pos_int_to_str_direct(sc, -peek_shared_ref(ci, obj)), "> ", NULL);
	  port_write_string(ci->cycle_port)(sc, buf, len, ci->cycle_port);
	  symbol_to_port(sc, sym, ci->cycle_port, P_KEY, ci);

	  symref = peek_shared_ref(ci, val);
	  if (symref != 0)
	    {
	      if (symref < 0) symref = -symref;
	      len = catstrs_direct(buf, ") <", pos_int_to_str_direct(sc, symref), ">)\n", NULL);
	      port_write_string(ci->cycle_port)(sc, buf, len, ci->cycle_port);
	    }
	  else
	    {
	      port_write_string(ci->cycle_port)(sc, ") ", 2, ci->cycle_port);
	      object_to_port_with_circle_check(sc, val, ci->cycle_port, P_READABLE, ci);
	      port_write_string(ci->cycle_port)(sc, ")\n", 2, ci->cycle_port);
	    }
	}
      else
	{
	  port_write_character(port)(sc, ' ', port);
	  object_to_port_with_circle_check(sc, val, port, P_READABLE, ci);
	}
    }
}

static void let_to_port(s7_scheme *sc, s7_pointer obj, s7_pointer port, use_write_t use_write, shared_info *ci)
{
  /* if outer env points to (say) method list, the object needs to specialize object->string itself */
  if (has_methods(obj))
    {
      s7_pointer print_func;
      print_func = find_method(sc, obj, sc->object_to_string_symbol);
      if (print_func != sc->undefined)
	{
	  s7_pointer p;
	  /* what needs to be protected here? for one, the function might not return a string! */

	  clear_has_methods(obj);
	  if (use_write == P_WRITE)
	    p = s7_apply_function(sc, print_func, list_1(sc, obj));
	  else p = s7_apply_function(sc, print_func, list_2(sc, obj, (use_write == P_DISPLAY) ? sc->F : sc->key_readable_symbol));
	  set_has_methods(obj);

	  if ((is_string(p)) &&
	      (string_length(p) > 0))
	    port_write_string(port)(sc, string_value(p), string_length(p), port);
	  return;
	}
    }

  if (obj == sc->rootlet)
    port_write_string(port)(sc, "(rootlet)", 9, port);
  else
    {
      if (sc->short_print)
	port_write_string(port)(sc, "#<let>", 6, port);
      else
	{
	  /* circles can happen here:
	   *    (let () (let ((b (curlet))) (curlet)))
	   *    #<let 'b #<let>>
	   * or (let ((b #f)) (set! b (curlet)) (curlet))
	   *    #1=#<let 'b #1#>
	   */
	  if (use_write == P_READABLE)
	    {
	      if (has_methods(obj))
		port_write_string(port)(sc, "(openlet ", 9, port);

	      if ((ci) &&
		  (is_cyclic(obj)) &&
		  (peek_shared_ref(ci, obj) != 0))
		{
		  {
		    int32_t lref;
		    lref = peek_shared_ref(ci, obj);
		    if (lref < 0) lref = -lref;
		    if ((ci->defined[lref]) || (port == ci->cycle_port))
		      {
			char buf[128];
			int32_t len;
			len = catstrs_direct(buf, "<", pos_int_to_str_direct(sc, lref), ">", NULL);
			port_write_string(ci->cycle_port)(sc, buf, len, ci->cycle_port);
			return;
		      }
		  }
		  if ((outlet(obj) != sc->nil) &&
		      (outlet(obj) != sc->rootlet))
		    {
		      char buf[128];
		      int32_t len;
		      len = catstrs_direct(buf, "  (set! (outlet <", pos_int_to_str_direct(sc, -peek_shared_ref(ci, obj)), ">) ", NULL);
		      port_write_string(ci->cycle_port)(sc, buf, len, ci->cycle_port);
		      let_to_port(sc, outlet(obj), ci->cycle_port, use_write, ci);
		      port_write_string(ci->cycle_port)(sc, ")\n", 2, ci->cycle_port);
		    }
		  port_write_string(port)(sc, "(inlet", 6, port);
		  slot_list_to_port_with_cycle(sc, obj, let_slots(obj), port, ci);
		  port_write_character(port)(sc, ')', port);
		}
	      else
		{
		  if (is_immutable(obj))
		    port_write_string(port)(sc, "(immutable! ", 12, port);
		  if ((outlet(obj) != sc->nil) &&
		      (outlet(obj) != sc->rootlet))
		    {
		      port_write_string(port)(sc, "(sublet ", 8, port);
		      if ((ci) && (peek_shared_ref(ci, outlet(obj)) < 0))
			{
			  char buf[128];
			  int32_t len;
			  len = catstrs_direct(buf, "<", pos_int_to_str_direct(sc, -peek_shared_ref(ci, outlet(obj))), ">", NULL);
			  port_write_string(port)(sc, buf, len, port);
			}
		      else let_to_port(sc, outlet(obj), port, use_write, ci);
		    }
		  else port_write_string(port)(sc, "(inlet", 6, port);
		  slot_list_to_port(sc, let_slots(obj), port, ci);
		  port_write_character(port)(sc, ')', port);
		  if (is_immutable(obj))
		    port_write_character(port)(sc, ')', port);
		}
	      if (has_methods(obj))
		port_write_character(port)(sc, ')', port);
	    }
	  else /* not readable write */
	    {
	      port_write_string(port)(sc, "(inlet", 6, port);
	      slot_to_port_1(sc, let_slots(obj), port, use_write, ci, 0);
	      port_write_character(port)(sc, ')', port);
	    }
	}
    }
}

static void write_macro_readably(s7_scheme *sc, s7_pointer obj, s7_pointer port)
{
  s7_pointer arglist, body, expr;

  body = closure_body(obj);
  arglist = closure_args(obj);

  port_write_string(port)(sc, "(define-", 8, port);
  port_write_string(port)(sc, ((is_macro(obj)) || (is_macro_star(obj))) ? "macro" : "bacro", 5, port);
  if ((is_macro_star(obj)) || (is_bacro_star(obj)))
    port_write_character(port)(sc, '*', port);
  port_write_string(port)(sc, " (_m_", 5, port);
  if (is_symbol(arglist))
    {
      port_write_string(port)(sc, " . ", 3, port);
      port_write_string(port)(sc, symbol_name(arglist), symbol_name_length(arglist), port);
    }
  else
    {
      if (is_pair(arglist))
	{
	  for (expr = arglist; is_pair(expr); expr = cdr(expr))
	    {
	      port_write_character(port)(sc, ' ', port);
	      object_to_port(sc, car(expr), port, P_WRITE, NULL);
	    }
	  if (!is_null(expr))
	    {
	      port_write_string(port)(sc, " . ", 3, port);
	      object_to_port(sc, expr, port, P_WRITE, NULL);
	    }
	}
    }
  port_write_string(port)(sc, ") ", 2, port);
  for (expr = body; is_pair(expr); expr = cdr(expr))
    object_to_port(sc, car(expr), port, P_WRITE, NULL);
  port_write_character(port)(sc, ')', port);
}


static s7_pointer match_symbol(s7_scheme *sc, s7_pointer symbol, s7_pointer e)
{
  s7_pointer y, le;
  for (le = e; is_let(le) && (le != sc->rootlet); le = outlet(le))
    for (y = let_slots(le); is_slot(y); y = next_slot(y))
      if (slot_symbol(y) == symbol)
	return(y);
  return(NULL);
}

static bool slot_memq(s7_pointer symbol, s7_pointer symbols)
{
  s7_pointer x;
  for (x = symbols; is_pair(x); x = cdr(x))
    if (slot_symbol(car(x)) == symbol)
      return(true);
  return(false);
}

static bool arg_memq(s7_pointer symbol, s7_pointer args)
{
  s7_pointer x;
  for (x = args; is_pair(x); x = cdr(x))
    if ((car(x) == symbol) ||
	((is_pair(car(x))) &&
	 (caar(x) == symbol)))
      return(true);
  return(false);
}

static void collect_symbol(s7_scheme *sc, s7_pointer sym, s7_pointer e, s7_pointer args, s7_int gc_loc)
{
  if ((!arg_memq(T_Sym(sym), args)) &&
      (!slot_memq(sym, gc_protected_at(sc, gc_loc))))
    {
      s7_pointer slot;
      slot = match_symbol(sc, sym, e);
      if (slot)
	gc_protected_at(sc, gc_loc) = cons(sc, slot, gc_protected_at(sc, gc_loc));
    }
}

static void collect_locals(s7_scheme *sc, s7_pointer body, s7_pointer e, s7_pointer args, s7_int gc_loc)
{ /* currently called only in write_closure_readably */
  if (is_pair(body))
    {
      collect_locals(sc, car(body), e, args, gc_loc);
      collect_locals(sc, cdr(body), e, args, gc_loc);
    }
  else
    {
      if (is_symbol(body))
	collect_symbol(sc, body, e, args, gc_loc);
    }
}

static void collect_specials(s7_scheme *sc, s7_pointer e, s7_pointer args, s7_int gc_loc)
{
  collect_symbol(sc, sc->local_signature_symbol, e, args, gc_loc);
  collect_symbol(sc, sc->local_setter_symbol, e, args, gc_loc);
  collect_symbol(sc, sc->local_documentation_symbol, e, args, gc_loc);
  collect_symbol(sc, sc->local_iterator_symbol, e, args, gc_loc);
}

static s7_pointer find_closure(s7_scheme *sc, s7_pointer closure, s7_pointer cur_env)
{
  s7_pointer e, y;
  for (e = cur_env; is_let(e); e = outlet(e))
    {
      if (is_funclet(e))
	{
	  s7_pointer sym, f;
	  sym = funclet_function(e);
	  f = s7_symbol_local_value(sc, sym, e);
	  if (f == closure)
	    return(sym);
	}

      for (y = let_slots(e); is_slot(y); y = next_slot(y))
	if (slot_value(y) == closure)
	  return(slot_symbol(y));
    }
  return(sc->nil);
}

static void write_closure_name(s7_scheme *sc, s7_pointer closure, s7_pointer port)
{
  s7_pointer x;
  x = find_closure(sc, closure, closure_let(closure));
  /* this can be confusing!
   * (let ((a (lambda () 1))) a)
   * #<lambda ()>
   * (letrec ((a (lambda () 1))) a)
   * a
   * (let () (define (a) 1) a)
   * a
   * (let () (define a (lambda () 1)))
   * a
   * (let () (define a (lambda () 1)))
   * a
   * (let () (define (a) (lambda () 1)) (a))
   * #<lambda ()>
   */
  if (is_symbol(x)) /* after find_closure */
    {
      port_write_string(port)(sc, symbol_name(x), symbol_name_length(x), port);
      return;
    }

  /* names like #<closure> and #<macro> are useless -- try to be a bit more informative */
  switch (type(closure))
    {
    case T_CLOSURE:
      port_write_string(port)(sc, "#<lambda ", 9, port);
      break;

    case T_CLOSURE_STAR:
      port_write_string(port)(sc, "#<lambda* ", 10, port);
      break;

    case T_MACRO:
      if (is_expansion(closure))
	port_write_string(port)(sc, "#<expansion ", 12, port);
      else port_write_string(port)(sc, "#<macro ", 8, port);
      break;

    case T_MACRO_STAR:
      port_write_string(port)(sc, "#<macro* ", 9, port);
      break;

    case T_BACRO:
      port_write_string(port)(sc, "#<bacro ", 8, port);
      break;

    case T_BACRO_STAR:
      port_write_string(port)(sc, "#<bacro* ", 9, port);
      break;
    }

  if (is_null(closure_args(closure)))
    port_write_string(port)(sc, "()>", 3, port);
  else
    {
      s7_pointer args;
      args = closure_args(closure);
      if (is_symbol(args))
	{
	  port_write_string(port)(sc, symbol_name(args), symbol_name_length(args), port);
	  port_write_character(port)(sc, '>', port);    /* (lambda a a) -> #<lambda a> */
	}
      else
	{
	  port_write_character(port)(sc, '(', port);
	  x = car(args);
	  if (is_pair(x)) x = car(x);
	  port_write_string(port)(sc, symbol_name(x), symbol_name_length(x), port);
	  if (!is_null(cdr(args)))
	    {
	      s7_pointer y;
	      port_write_character(port)(sc, ' ', port);
	      if (is_pair(cdr(args)))
		{
		  y = cadr(args);
		  if (is_pair(y))
		    y = car(y);
		  else
		    {
		      if (y == sc->key_rest_symbol)
			{
			  port_write_string(port)(sc, ":rest ", 6, port);
			  args = cdr(args);
			  y = cadr(args);
			  if (is_pair(y)) y = car(y);
			}
		    }
		}
	      else
		{
		  port_write_string(port)(sc, ". ", 2, port);
		  y = cdr(args);
		}
	      port_write_string(port)(sc, symbol_name(y), symbol_name_length(y), port);
	      if ((is_pair(cdr(args))) &&
		  (!is_null(cddr(args))))
		port_write_string(port)(sc, " ...", 4, port);
	    }
	  port_write_string(port)(sc, ")>", 2, port);
	}
    }
}

static s7_pointer closure_name(s7_scheme *sc, s7_pointer closure)
{
  /* this is used by the error handlers to get the current function name */
  s7_pointer x;

  x = find_closure(sc, closure, sc->envir);
  if (is_symbol(x))
    return(x);

  if (is_pair(current_code(sc)))
    return(current_code(sc));

  return(closure); /* desperation -- the parameter list (caar here) will cause endless confusion in OP_APPLY errors! */
}

static void write_closure_readably_1(s7_scheme *sc, s7_pointer obj, s7_pointer arglist, s7_pointer body, s7_pointer port)
{
  s7_int old_print_length;
  s7_pointer p;

  if (type(obj) == T_CLOSURE_STAR)
    port_write_string(port)(sc, "(lambda* ", 9, port);
  else port_write_string(port)(sc, "(lambda ", 8, port);

  if ((is_pair(arglist)) &&
      (allows_other_keys(arglist)))
    {
      sc->temp9 = s7_append(sc, arglist, cons(sc, sc->key_allow_other_keys_symbol, sc->nil));
      object_to_port(sc, sc->temp9, port, P_WRITE, NULL);
      sc->temp9 = sc->nil;
    }
  else object_to_port(sc, arglist, port, P_WRITE, NULL); /* here we just want the straight output (a b) not (list 'a 'b) */

  old_print_length = sc->print_length;
  sc->print_length = 1048576;
  for (p = body; is_pair(p); p = cdr(p))
    {
      port_write_character(port)(sc, ' ', port);
      object_to_port(sc, car(p), port, P_WRITE, NULL);
    }
  port_write_character(port)(sc, ')', port);
  sc->print_length = old_print_length;
}

static void write_closure_readably(s7_scheme *sc, s7_pointer obj, s7_pointer port)
{
  s7_pointer body, arglist, pe, local_slots, setter = NULL;
  s7_int gc_loc;

  body = closure_body(obj);
  if (sc->safety > NO_SAFETY)
    {
      if (tree_is_cyclic(sc, body))
	s7_error(sc, sc->wrong_type_arg_symbol, wrap_string(sc, "write_closure: body is cyclic", 29));
      /* TODO: if any sequence in the closure_body is cyclic, complain, but how to check without clobbering ci?
       *   perhaps pass ci, and use make_shared_info if ci=null else continue_shared_info?
       *   this can happen only if (apply lambda ... cyclic-seq ...) I think
       *   long-term we need to include closure_body(obj) in the top object_out make_shared_info
       */
    }

  arglist = closure_args(obj);
  pe = closure_let(obj);

  gc_loc = s7_gc_protect_1(sc, sc->nil);
  collect_locals(sc, body, pe, (is_list(arglist)) ? arglist : sc->nil, gc_loc);   /* collect locals used only here (and below) */
  collect_specials(sc, pe, (is_list(arglist)) ? arglist : sc->nil, gc_loc);
  if (s7_is_dilambda(obj))
    {
      setter = closure_setter(obj);
      if ((!(has_closure_let(setter))) ||
	  (closure_let(setter) != pe))
	setter = NULL;
    }
  if (setter)
    collect_locals(sc, closure_body(setter), pe, closure_args(setter), gc_loc);

  local_slots = T_Lst(gc_protected_at(sc, gc_loc)); /* possibly a list of slots */
  if (!is_null(local_slots))
    {
      s7_pointer x;
      port_write_string(port)(sc, "(let (", 6, port);
      for (x = local_slots; is_pair(x); x = cdr(x))
	{
	  s7_pointer slot;
	  slot = car(x);
	  if ((!is_any_closure(slot_value(slot))) &&    /* mutually referencing closures? ./snd -l snd-test 24 hits this in the effects dialogs */
	      ((!has_structure(slot_value(slot))) ||    /* see s7test example, vector has closure that refers to vector */
	       (slot_symbol(slot) == sc->local_signature_symbol)))
	    {
	      port_write_character(port)(sc, '(', port);
	      port_write_string(port)(sc, symbol_name(slot_symbol(slot)), symbol_name_length(slot_symbol(slot)), port);
	      port_write_character(port)(sc, ' ', port);
	      /* (object->string (list (let ((local 1)) (lambda (x) (+ x local)))) :readable) */
	      object_to_port(sc, slot_value(slot), port, P_READABLE, NULL);
	      if (is_null(cdr(x)))
		port_write_character(port)(sc, ')', port);
	      else port_write_string(port)(sc, ") ", 2, port);
	    }
	}
      port_write_string(port)(sc, ") ", 2, port);
    }

  if (setter)
    port_write_string(port)(sc, "(dilambda ", 10, port);

  write_closure_readably_1(sc, obj, arglist, body, port);

  if (setter)
    {
      port_write_character(port)(sc, ' ', port);
      write_closure_readably_1(sc, setter, closure_args(setter), closure_body(setter), port);
      port_write_character(port)(sc, ')', port);
    }

  if (!is_null(local_slots))
    port_write_character(port)(sc, ')', port);
  s7_gc_unprotect_at(sc, gc_loc);
}


#if TRAP_SEGFAULT
#include <signal.h>
static sigjmp_buf senv; /* global here is not a problem -- it is used only to protect s7_is_valid */
static volatile sig_atomic_t can_jump = 0;
static void segv(int32_t ignored) {if (can_jump) siglongjmp(senv, 1);}
#endif

bool s7_is_valid(s7_scheme *sc, s7_pointer arg)
{
  bool result = false;
  if (!arg) return(false);

#if TRAP_SEGFAULT
  if (sigsetjmp(senv, 1) == 0)
    {
      void (*old_segv)(int32_t sig);
      can_jump = 1;
      old_segv = signal(SIGSEGV, segv);
#endif
      if ((unchecked_type(arg) > T_FREE) &&
	  (unchecked_type(arg) < NUM_TYPES))
	{
	  if (not_in_heap(arg))
	    result = true;
	  else
	    {
	      int64_t loc;
	      loc = heap_location(sc, arg);
	      if ((loc >= 0) && (loc < sc->heap_size))
		result = (sc->heap[loc] == arg);
#if S7_DEBUGGING
	      if (!result)
		check_heap_location(sc, arg, loc, __func__, __LINE__);
#endif
	    }
	}
#if TRAP_SEGFAULT
      signal(SIGSEGV, old_segv);
    }
  else result = false;
  can_jump = 0;
#endif
  return(result);
}

enum {NO_ARTICLE, INDEFINITE_ARTICLE};

static char *describe_type_bits(s7_scheme *sc, s7_pointer obj) /* used outside S7_DEBUGGING in display_any (fallback for display_functions) */
{
  uint64_t full_typ;
  uint8_t typ;
  char *buf;

  buf = (char *)malloc(1024 * sizeof(char));
  typ = unchecked_type(obj);
  full_typ = typeflag(obj);

  /* if debugging all of these bits are being watched, so we need to access them directly */
  snprintf(buf, 1024, "type: %d (%s), opt_op: %d, flags: #x%" PRIx64 "%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s",
	   typ,
	   type_name(sc, obj, NO_ARTICLE),
	   optimize_op(obj),
	   full_typ,
	   /* bit 0 (the first 8 bits are easy...) */
	   ((full_typ & T_MULTIFORM) != 0) ?      ((is_any_closure(obj)) ? (((full_typ & T_ONE_FORM) != 0) ? " clo-has-fx" : " multiform") : " ?0?") : "",
	   /* bit 1 */
	   ((full_typ & T_SYNTACTIC) != 0) ?      (((is_pair(obj)) || (is_syntax(obj)) || (is_symbol(obj))) ? " syntactic" : " ?1?") : "",
	   /* bit 2 */
	   ((full_typ & T_SIMPLE_ARG_DEFAULTS) != 0) ? ((is_pair(obj)) ? " simple-args|in-use" :
							((is_any_closure(obj)) ? " one-form" :
							 " ?2?")) : "",
	   /* bit 3 */
	   ((full_typ & T_OPTIMIZED) != 0) ?      ((is_c_function(obj)) ? " scope-safe" :
						   ((is_pair(obj)) ? " optimized" :
						    " ?3?")) : "",
	   /* bit 4 */
	   ((full_typ & T_SAFE_CLOSURE) != 0) ?   " safe-closure" : "",
	   /* bit 5 */
	   ((full_typ & T_DONT_EVAL_ARGS) != 0) ? " dont-eval-args" : "",
	   /* bit 6 */
	   ((full_typ & T_EXPANSION) != 0) ?      (((is_symbol(obj)) || (is_macro(obj))) ? " expansion" :
						   " ?6?") : "",
	   /* bit 7 */
	   ((full_typ & T_MULTIPLE_VALUE) != 0) ? ((is_symbol(obj)) ? " matched" :
						   ((is_pair(obj)) ? " values|matched" :
						    " ?7?")) : "",
	   /* bit 8 */
	   ((full_typ & T_GLOBAL) != 0) ?         ((is_pair(obj)) ? " unsafe-do" :
						   (((is_symbol(obj)) || (is_syntax(obj))) ? " global" :
						    " ?8?")) : "",
	   /* bit 9 */
	   ((full_typ & T_COLLECTED) != 0) ?      " collected" : "",
	   /* bit 10 */
	   ((full_typ & T_LINE_NUMBER) != 0) ?    ((is_pair(obj)) ? " line-number" :
						   ((is_input_port(obj)) ? " loader-port" :
						    ((is_let(obj)) ? " with-let" :
						     ((is_any_procedure(obj)) ? " simple-defaults" :
						      (((is_symbol(obj)) || (is_slot(obj))) ? " has-setter" :
						       " ?10?"))))) : "",
	   /* bit 11 */
	   ((full_typ & T_SHARED) != 0) ?         " shared" : "",
	   /* bit 12 */
	   ((full_typ & T_LOCAL) != 0) ?          ((is_symbol(obj)) ? " local" :
						   " ?12?") : "",
	   /* bit 13 */
	   ((full_typ & T_SAFE_PROCEDURE) != 0) ? " safe-procedure" : "",
	   /* bit 14 */
	   ((full_typ & T_CHECKED) != 0) ?        " checked" : "",
	   /* bit 15 */
	   ((full_typ & T_UNSAFE) != 0) ?         ((is_symbol(obj)) ? " clean-symbol" :
						   ((is_slot(obj)) ? " has-stepper" :
						    ((is_pair(obj)) ? " unsafely-opt|no-float-opt" :
						     " ?15?"))) : "",
	   /* bit 16 */
	   ((full_typ & T_IMMUTABLE) != 0) ?      " immutable" : "",
	   /* bit 17 */
	   ((full_typ & T_SETTER) != 0) ?         ((is_symbol(obj)) ? " setter" :
						   ((is_pair(obj)) ? " allow-other-keys|no-int-opt" :
						    (((is_hash_table(obj)) || (is_let(obj))) ? " removed" :
						     ((is_slot(obj)) ? " has-expression" :
						      " ?17?")))) : "",
	   /* bit 18 */
	   ((full_typ & T_MUTABLE) != 0) ?        ((is_number(obj)) ? " mutable" :
						   ((is_symbol(obj)) ? " has-keyword" :
						    ((is_let(obj)) ? " let-ref-fallback" :
						     ((is_iterator(obj)) ? " mark-sequence" :
						      ((is_slot(obj)) ? " step-end" :
						       ((is_let(obj)) ? " ref-fallback" :
							((is_pair(obj)) ? " no-opt" :
							 " ?18?"))))))) : "",
	   /* bit 19 */
	   ((full_typ & T_SAFE_STEPPER) != 0) ?   ((is_let(obj)) ? " set-fallback" :
						   ((is_slot(obj)) ? " safe-stepper" :
						    ((is_c_function(obj)) ? " maybe-safe" :
						     ((is_number(obj)) ? " print-name" :
						      ((is_pair(obj)) ? " direct_x_opt" :
						       ((is_hash_table(obj)) ? " weak-hash" :
							" ?19?")))))) : "",
	   /* bit 20, for c_function case see sc->apply */
	   ((full_typ & T_COPY_ARGS) != 0) ?      (((is_any_macro(obj)) || (is_any_closure(obj)) || (is_c_function(obj))) ? " copy-args" :
						    " ?20?") : "",
	   /* bit 21 */
	   ((full_typ & T_GENSYM) != 0) ?         ((is_let(obj)) ? " funclet" :
						   ((is_symbol(obj)) ? " gensym" :
						    ((is_string(obj)) ? " documented-symbol" :
						     ((is_hash_table(obj)) ? " hash-chosen" :
						      ((is_pair(obj)) ? " dotted" :
						       ((is_any_vector(obj)) ? " subvector" :
							((is_slot(obj)) ? " has-pending-value" :
							 " ?21?"))))))) : "",
	   /* bit 22 */
	   ((full_typ & T_HAS_METHODS) != 0) ?    " has-methods" : "",
	   /* bit 23 */
	   ((full_typ & T_ITER_OK) != 0) ?        ((is_iterator(obj)) ? " iter-ok" : " ?23?") : "",
	   /* bit 24+16 */
	   ((full_typ & T_FULL_SYMCONS) != 0) ?   ((is_symbol(obj)) ? " possibly-constant" :
						   ((is_procedure(obj)) ? " has-let-arg" :
						    " ?24?")) : "",
	   /* bit 25+16 */
	   ((full_typ & T_FULL_S7_LET_FIELD) != 0) ?   ((is_symbol(obj)) ? " s7-let-field" :
							((is_let(obj)) ? " has-let-file" :
							 ((is_pair(obj)) ? " has-oplist" :
							  ((is_any_vector(obj)) ? " typed-vector" :
							   ((is_hash_table(obj)) ? " typed-hash-table" :
							    ((is_c_function(obj)) ? " has-bool-setter" :
							     " ?25?")))))) : "",
	   /* bit 26+16 */
	   ((full_typ & T_FULL_DEFINER) != 0) ?   ((is_symbol(obj)) ? " definer" :
						   ((is_pair(obj)) ? " has-fx" :
						    ((is_slot(obj)) ? " slot-defaults" :
						     " ?26?"))) : "",
	   /* bit 27+16 */
	   ((full_typ & T_TREE_COLLECTED) != 0) ?  ((is_pair(obj)) ? " tree-collected" :
						    ((is_hash_table(obj)) ? " simple-values" :
						     ((is_c_function(obj)) ? " type-info" :
						      ((is_symbol(obj)) ? " binder" :
						       " ?27?")))) : "",
	   /* bit 28+16 */
	   ((full_typ & T_VERY_SAFE_CLOSURE) != 0) ? " very-safe-closure" : "",
	   /* bit 29+16 */
	   ((full_typ & T_CYCLIC) != 0) ?         " cyclic" : "",
	   /* bit 30+16 */
	   ((full_typ & T_CYCLIC_SET) != 0) ?     " cyclic-set" : "",
	   /* bit 31+16 */
	   ((full_typ & T_KEYWORD) != 0) ?        ((is_symbol(obj)) ? " keyword" : " ?31?") : "",
	   /* bit 32+16 */
	   ((full_typ & T_FULL_SIMPLE_ELEMENTS) != 0) ? ((is_normal_vector(obj)) ? " simple-elements" :
							 ((is_hash_table(obj)) ? " simple-keys" :
							  ((is_pair(obj)) ? " ctr3-set" :
							   " 32?"))) : "",

	   ((full_typ & UNUSED_BITS) != 0) ?      " unused bits set?" : "",

	   /* bit 54 */
	   ((full_typ & T_UNHEAP) != 0) ?         " unheap" : "",
	   /* bit 55 */
	   (((full_typ & T_GC_MARK) != 0) && (in_heap(obj))) ? " gc-marked" : "");
  return(buf);
}

#if S7_DEBUGGING
static bool has_odd_bits(s7_pointer obj)
{
  uint64_t full_typ;
  full_typ = typeflag(obj);

  if ((full_typ & UNUSED_BITS) != 0) return(true);
  if (((full_typ & T_MULTIFORM) != 0) && (!is_any_closure(obj))) return(true);
  if (((full_typ & T_KEYWORD) != 0) && (!is_symbol(obj))) return(true);
  if (((full_typ & T_TREE_COLLECTED) != 0) && ((!is_pair(obj)) && (!is_hash_table(obj)) && (!is_c_function(obj)) && (!is_symbol(obj)))) return(true);
  if (((full_typ & T_SYNTACTIC) != 0) && (!is_syntax(obj)) && (!is_pair(obj)) && (!is_symbol(obj))) return(true);
  if (((full_typ & T_SIMPLE_ARG_DEFAULTS) != 0) && (!is_pair(obj)) && (!is_any_closure(obj))) return(true);
  if (((full_typ & T_OPTIMIZED) != 0) && (!is_c_function(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_SAFE_CLOSURE) != 0) && (!is_any_closure(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_EXPANSION) != 0) && (!is_symbol(obj)) && (!is_macro(obj))) return(true);
  if (((full_typ & T_MULTIPLE_VALUE) != 0) && (!is_symbol(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_GLOBAL) != 0) && (!is_pair(obj)) && (!is_symbol(obj)) && (!is_syntax(obj))) return(true);
  if (((full_typ & T_ITER_OK) != 0) && (!is_iterator(obj))) return(true);
  if (((full_typ & T_FULL_DEFINER) != 0) && (!is_symbol(obj)) && (!is_pair(obj)) && (!is_slot(obj))) return(true);
  if (((full_typ & T_FULL_SYMCONS) != 0) && (!is_symbol(obj)) && (!is_procedure(obj))) return(true);
  if (((full_typ & T_LOCAL) != 0) && (!is_symbol(obj))) return(true);
  if (((full_typ & T_COPY_ARGS) != 0) && (!is_any_macro(obj)) && (!is_any_closure(obj)) && (!is_c_function(obj))) return(true);
  if (((full_typ & T_UNSAFE) != 0) && (!is_symbol(obj)) && (!is_slot(obj)) && (!is_pair(obj))) return(true);
  if (((full_typ & T_VERY_SAFE_CLOSURE) != 0) && (!is_pair(obj)) && (!is_any_closure(obj))) return(true);
  if (((full_typ & T_FULL_SIMPLE_ELEMENTS) != 0) && ((!is_normal_vector(obj)) && (!is_hash_table(obj)) && (!is_pair(obj)))) return(true);
  if (((full_typ & T_FULL_S7_LET_FIELD) != 0) &&
      (!is_symbol(obj)) && (!is_let(obj)) && (!is_pair(obj)) && (!is_any_vector(obj)) && (!is_hash_table(obj)) && (!is_c_function(obj)))
    return(true);
  if (((full_typ & T_SAFE_STEPPER) != 0) &&
      (!is_let(obj)) && (!is_slot(obj)) && (!is_c_function(obj)) && (!is_number(obj)) && (!is_pair(obj)) && (!is_hash_table(obj)))
    return(true);
  if (((full_typ & T_SETTER) != 0) &&
      (!is_slot(obj)) && (!is_symbol(obj)) && (!is_pair(obj)) && (!is_closure(obj)) && (!is_hash_table(obj)) && (!is_let(obj)))
    return(true);
  if (((full_typ & T_LINE_NUMBER) != 0) &&
      (!is_pair(obj)) && (!is_input_port(obj)) && (!is_let(obj)) && (!is_any_procedure(obj)) && (!is_symbol(obj)) && (!is_slot(obj)))
    return(true);
  if (((full_typ & T_MUTABLE) != 0) &&
      (!is_number(obj)) && (!is_symbol(obj)) && (!is_let(obj)) && (!is_iterator(obj)) &&
      (!is_slot(obj)) && (!is_let(obj)) && (!is_pair(obj)))
    return(true);
  if (((full_typ & T_GENSYM) != 0) && (!is_slot(obj)) &&
      (!is_let(obj)) && (!is_symbol(obj)) && (!is_string(obj)) && (!is_hash_table(obj)) && (!is_pair(obj)) && (!is_any_vector(obj)))
    return(true);
  return(false);
}
#endif

void s7_show_let(s7_scheme *sc) /* debugging convenience */
{
  s7_pointer olet;
  for (olet = sc->envir; (is_let(olet)) && (olet != sc->rootlet); olet = outlet(olet))
    {
      if (olet == sc->owlet)
	fprintf(stderr, "(owlet): ");
      else
	{
	  if (is_funclet(olet))
	    fprintf(stderr, "(%s funclet): ", DISPLAY(funclet_function(olet)));
	  else
	    {
	      if (olet == sc->shadow_rootlet)
		fprintf(stderr, "(shadow rootlet): ");
	    }
	}
      fprintf(stderr, "%s\n", DISPLAY(olet));
    }
}

void s7_show_history(s7_scheme *sc)
{
#if WITH_HISTORY
  s7_pointer p;
  for (p = cdr(sc->cur_code); p != sc->cur_code; p = cdr(p))
    fprintf(stderr, "%s\n", DISPLAY_80(car(p)));
#endif
  fprintf(stderr, "%s\n", DISPLAY_80(sc->cur_code));
}

void s7_show_stack(s7_scheme *sc)
{
#if S7_DEBUGGING || OP_NAMES
  int64_t i;
  fprintf(stderr, "stack:\n");
  for (i = s7_stack_top(sc) - 1; i >= 3; i -= 4)
    fprintf(stderr, "  %s\n", op_names[stack_op(sc->stack, i)]);
#else
  fprintf(stderr, "can't show stack ops (OP_NAMES: 0)\n");
#endif
}

#if S7_DEBUGGING
static const char *check_name(int32_t typ)
{
  if ((typ >= 0) && (typ < NUM_TYPES))
    {
      s7_pointer p;
      p = prepackaged_type_names[typ];
      if (is_string(p)) return(string_value(p));

      switch (typ)
	{
	case T_C_OBJECT:    return("a c-object");
	case T_INPUT_PORT:  return("an input port");
	case T_OUTPUT_PORT: return("an output port");
	}
    }
  return("unknown type!");
}

static char *safe_object_to_string(s7_pointer p)
{
  uint8_t typ;
  char *buf;
  typ = unchecked_type(p);
  buf = (char *)malloc(128 * sizeof(char));
  snprintf(buf, 128, "type: %d", typ);
  return(buf);
}

static void complain(const char* complaint, s7_pointer p, const char *func, int line, uint8_t typ)
{
  fprintf(stderr, complaint,
	  BOLD_TEXT,
	  func, line, check_name(typ), safe_object_to_string(p),
	  UNBOLD_TEXT);
  if (stop_at_error) abort();
}

static s7_pointer check_ref(s7_pointer p, uint8_t expected_type, const char *func, int32_t line, const char *func1, const char *func2)
{
  if (!p)
    fprintf(stderr, "%s[%d]: null pointer passed to check_ref\n", func, line);
  else
    {
      uint8_t typ;
      typ = unchecked_type(p);
      if (typ != expected_type)
	{
	  if ((!func1) || (typ != T_FREE))
	    {
	      fprintf(stderr, "%s%s[%d]: not %s, but %s (%s)%s\n",
		      BOLD_TEXT,
		      func, line, check_name(expected_type), check_name(typ), safe_object_to_string(p),
		      UNBOLD_TEXT);
	      if ((typ != T_FREE) && (is_syntactic_pair(p)) && (optimize_op(p) == 0))
		fprintf(stderr, "syn 0: %s[%d]\n", func, line);
	    }
	  else
	    {
	      if ((strcmp(func, func1) != 0) &&
		  ((!func2) || (strcmp(func, func2) != 0)))
		{
		  fprintf(stderr, "%s%s[%d]: free cell, not %s%s\n", BOLD_TEXT, func, line, check_name(expected_type), UNBOLD_TEXT);
		  if (stop_at_error) abort();
		}
	    }
	}
    }
  return(p);
}

static s7_pointer check_let_ref(s7_pointer p, uint64_t role, const char *func, int32_t line)
{
  check_ref(p, T_LET, func, line, NULL, NULL);
  if ((p->debugger_bits & L_HIT) == 0) fprintf(stderr, "%s[%d]: let not set\n", func, line);
  if ((p->debugger_bits & L_MASK) != role) fprintf(stderr, "%s[%d]: let bad role\n", func, line);
  return(p);
}

static s7_pointer check_let_set(s7_pointer p, uint64_t role, const char *func, int32_t line)
{
  check_ref(p, T_LET, func, line, NULL, NULL);
  p->debugger_bits &= (~L_MASK);
  p->debugger_bits |= (L_HIT | role);
  return(p);
}

static s7_pointer check_ref2(s7_pointer p, uint8_t expected_type, int32_t other_type, const char *func, int32_t line, const char *func1, const char *func2)
{
  if (!p)
    fprintf(stderr, "%s[%d]: null pointer passed to check_ref2\n", func, line);
  else
    {
      uint8_t typ;
      typ = unchecked_type(p);
      if ((typ != expected_type) && (typ != other_type))
	return(check_ref(p, expected_type, func, line, func1, func2));
    }
  return(p);
}

static s7_pointer check_ref3(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  typ = unchecked_type(p);
  if ((typ != T_INPUT_PORT) && (typ != T_OUTPUT_PORT) && (typ != T_FREE))
    complain("%s%s[%d]: not a port, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref4(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  typ = unchecked_type(p);
  if ((!t_vector_p[typ]) && (typ != T_FREE))
    complain("%s%s[%d]: not a vector, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref5(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  typ = unchecked_type(p);
  if (!t_has_closure_let[typ])
    complain("%s%s[%d]: not a closure, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref6(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  typ = unchecked_type(p);
  if ((typ < T_C_FUNCTION_STAR) && (typ != T_C_MACRO))
    complain("%s%s[%d]: not a c function, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref7(s7_pointer p, const char *func, int32_t line)
{
  if ((!func) || (strcmp(func, "decribe_type_bits") != 0))
    {
      uint8_t typ;
      typ = unchecked_type(p);
      if ((typ < T_INTEGER) || (typ > T_COMPLEX))
	complain("%s%s[%d]: not a number, but %s (%s)%s\n", p, func, line, typ);
    }
  return(p);
}

static s7_pointer check_ref8(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  typ = unchecked_type(p);
  if ((!t_sequence_p[typ]) && (!t_structure_p[typ]) && (!is_any_closure(p))) /* closure calling itself an iterator?? */
    complain("%s%s[%d]: not a sequence or structure, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref9(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  typ = unchecked_type(p);
  if ((typ != T_LET) && (typ != T_C_OBJECT) && (!is_any_closure(p)) && (!is_any_macro(p)) && (typ != T_C_POINTER))
    complain("%s%s[%d]: not a possible method holder, but %s (%s)%s\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref10(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  typ = unchecked_type(p);
  if ((typ != T_PAIR) && (typ != T_NIL) && (typ != T_SYMBOL))
    complain("%s%s[%d]: arglist is %s (%s)%s?\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref11(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  typ = unchecked_type(p);
  if ((!t_applicable_p[typ]) && (typ != T_BOOLEAN))
    complain("%s%s[%d]: applicable object is %s (%s)%s?\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_ref12(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  typ = unchecked_type(p);
  if ((!t_has_closure_let[typ]) && (typ != T_PAIR) && (typ != T_BOOLEAN)) /* #f actually */
    complain("%s%s[%d]: safe-closure is %s (%s)%s?\n", p, func, line, typ);
  return(p);
}

static s7_pointer check_cell(s7_pointer p, const char *func, int32_t line)
{
  if (!p)
    {
      fprintf(stderr, "%s%s[%d]: null pointer!%s\n", BOLD_TEXT, func, line, UNBOLD_TEXT);
      if (stop_at_error) abort();
    }
  else
    {
      uint8_t typ;
      typ = unchecked_type(p);
      if (typ >= NUM_TYPES)
	{
	  fprintf(stderr, "%s%s[%d]: attempt to use messed up cell (type: %d)%s\n", BOLD_TEXT, func, line, typ, UNBOLD_TEXT);
	  if (stop_at_error) abort();
	}
    }
  return(p);
}

static s7_pointer check_nref(s7_pointer p, const char *func, int32_t line)
{
  uint8_t typ;
  check_cell(p, func, line);
  typ = unchecked_type(p);
  if (typ == T_FREE)
    {
      char *s;
      fprintf(stderr, "%s%s[%d]: attempt to use free cell%s\n", BOLD_TEXT, func, line, UNBOLD_TEXT);
      typeflag(p) = p->current_alloc_type;
      fprintf(stderr, "  free cell %p alloc: %s[%d], alloc type: #x%x %s\n",
	      p, p->current_alloc_func, p->current_alloc_line,
	      (unsigned int)(p->current_alloc_type), s = describe_type_bits(cur_sc, p));
      free(s);
      typeflag(p) = 0;
      if (stop_at_error) abort();
    }
  return(p);
}

static void print_gc_info(s7_pointer obj, int32_t line)
{
  if (!obj)
    fprintf(stderr, "[%d]: obj is %p\n", line, obj);
  else
    {
      if (unchecked_type(obj) != T_FREE)
	fprintf(stderr, "[%d]: %p type is %d?\n", line, obj, unchecked_type(obj));
      else
	{
	  fprintf(stderr, "%s%p is free (line %d), current: %s[%d], previous: %s[%d]%s\n",
		  BOLD_TEXT,
		  obj, line,
		  obj->current_alloc_func, obj->current_alloc_line,
		  obj->previous_alloc_func, obj->previous_alloc_line,
		  UNBOLD_TEXT);
	}
    }
  abort();
}

static const char *opt1_role_name(uint32_t role)
{
  if (role == E_FAST) return("opt1_fast");
  if (role == E_CFUNC) return("opt1_cfunc");
  if (role == E_LAMBDA) return("opt_lambda");
  if (role == E_CLAUSE) return("opt1_clause");
  if (role == E_GOTO) return("opt1_goto");
  if (role == E_SYM) return("opt1_sym");
  if (role == E_PAIR) return("opt1_pair");
  if (role == E_CON) return("opt1_con");
  if (role == E_ANY) return("opt1_any");
  if (role == E_SLOT) return("opt1_slot");
  return("unknown");
}

static const char *opt2_role_name(uint32_t role)
{
  if (role == F_CALL) return("c_call(ee)");
  if (role == F_KEY) return("opt2_any");
  if (role == F_SLOW) return("opt2_slow");
  if (role == F_SYM) return("opt2_sym");
  if (role == F_PAIR) return("opt2_pair");
  if (role == F_CON) return("opt2_con");
  if (role == F_LAMBDA) return("opt2_lambda");
  return("unknown");
}

static const char *opt3_role_name(uint32_t role)
{
  if (role == G_ARGLEN) return("opt3_arglen");
  if (role == G_SYM) return("opt3_sym");
  if (role == G_AND) return("opt3_pair");
  if (role == G_ANY) return("opt3_any");
  if (role == G_CTR) return("opt3_ctr");
  if (role == G_CON) return("opt3_con");
  if (role == G_DIRECT) return("direct_opt3");
  if (role == S_LEN) return("s_len");
  if (role == S_LINE) return("s_line");
  if (role == S_HASH) return("s_hash");
  return("unknown");
}

static char* show_debugger_bits(int64_t bits)
{
  char *bits_str;
  bits_str = (char *)malloc(512 * sizeof(char));
  snprintf(bits_str, 512, " %s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s",
	  ((bits & E_SET) != 0) ? " e-set" : "",
	  ((bits & E_FAST) != 0) ? " opt1_fast" : "",
	  ((bits & E_CFUNC) != 0) ? " opt1_cfunc" : "",
	  ((bits & E_CLAUSE) != 0) ? " opt1_clause" : "",
	  ((bits & E_LAMBDA) != 0) ? " opt_lambda" : "",
	  ((bits & E_SYM) != 0) ? " opt1_sym" : "",
	  ((bits & E_PAIR) != 0) ? " opt1_pair" : "",
	  ((bits & E_CON) != 0) ? " opt1_con" : "",
	  ((bits & E_GOTO) != 0) ? " opt1_goto" : "",
	  ((bits & E_ANY) != 0) ? " opt1_any" : "",
	  ((bits & E_SLOT) != 0) ? " opt1_slot" : "",
	  ((bits & F_SET) != 0) ? " f-set" : "",
	  ((bits & F_KEY) != 0) ? " opt2_any" : "",
	  ((bits & F_SLOW) != 0) ? " opt2_slow" : "",
	  ((bits & F_SYM) != 0) ? " opt2_sym" : "",
	  ((bits & F_PAIR) != 0) ? " opt2_pair" : "",
	  ((bits & F_CON) != 0) ? " opt2_con" : "",
	  ((bits & F_CALL) != 0) ? " c_call(ee)" : "",
	  ((bits & F_LAMBDA) != 0) ? " opt2_lambda" : "",
	  ((bits & G_SET) != 0) ? " g-set" : "",
	  ((bits & G_ARGLEN) != 0) ? " opt3_arglen" : "",
	  ((bits & G_SYM) != 0) ? " opt3_sym" : "",
	  ((bits & G_AND) != 0) ? " opt3_pair " : "",
	  ((bits & G_ANY) != 0) ? " opt3_any " : "",
	  ((bits & G_CTR) != 0) ? " opt3_ctr " : "",
	  ((bits & G_CON) != 0) ? " opt3_con " : "",
	  ((bits & G_DIRECT) != 0) ? " opt3_direct_x" : "",
	  ((bits & S_NAME) != 0) ? " raw-name" : "",
	  ((bits & S_HASH) != 0) ? " raw-hash" : "",
	  ((bits & S_LINE) != 0) ? " line" : "",
	  ((bits & S_LEN) != 0) ? " len" : "");
  return(bits_str);
}

static void show_opt1_bits(s7_pointer p, const char *func, int32_t line, uint32_t role)
{
  char *bits;
  bits = show_debugger_bits(p->debugger_bits);
  fprintf(stderr, "%s%s[%d]: opt1: %p->%p wants %s, debugger bits are %" PRIx64 "%s but expects %x%s\n",
	  BOLD_TEXT,
	  func, line, p, p->object.cons.opt1,
	  opt1_role_name(role),
	  p->debugger_bits, bits, role,
	  UNBOLD_TEXT);
  free(bits);
}

static s7_pointer opt1_1(s7_pointer p, uint32_t role, const char *func, int32_t line)
{
  if ((!opt1_is_set(p)) ||
      ((!opt1_role_matches(p, role)) &&
       (role != E_ANY)))
    {
      show_opt1_bits(p, func, line, role);
      if (stop_at_error) abort();
    }
  return(p->object.cons.opt1);
}

static s7_pointer set_opt1_1(s7_pointer p, s7_pointer x, uint32_t role, const char *func, int32_t line)
{
  p->object.cons.opt1 = x;
  set_opt1_role(p, role);
  set_opt1_is_set(p);
  return(x);
}

static uint64_t s_hash_1(s7_pointer p, const char *func, int32_t line)
{
  if ((!opt1_is_set(p)) ||
      (!opt1_role_matches(p, S_HASH)))
    {
      show_opt1_bits(p, func, line, (uint32_t)S_HASH);
      if (stop_at_error) abort();
    }
  return(p->object.sym_cons.hash);
}

static void set_s_hash_1(s7_pointer p, uint64_t x, const char *func, int32_t line)
{
  p->object.sym_cons.hash = x;
  set_opt1_role(p, S_HASH);
  set_opt1_is_set(p);
}

static void show_opt2_bits(s7_pointer p, const char *func, int32_t line, uint32_t role)
{
  char *bits;
  bits = show_debugger_bits(p->debugger_bits);
  fprintf(stderr, "%s%s[%d]: opt2: %p->%p wants %s, debugger bits are %" PRIx64 "%s but expects %x%s%s%s%s%s%s%s%s%s%s\n",
	  BOLD_TEXT,
	  func, line, p, p->object.cons.opt2,
	  opt2_role_name(role),
	  p->debugger_bits, bits, role,
	  ((role & F_SET) != 0) ? " f-set" : "",
	  ((role & F_KEY) != 0) ? " any" : "",
	  ((role & F_SLOW) != 0) ? " slow" : "",
	  ((role & F_SYM) != 0) ? " sym" : "",
	  ((role & F_PAIR) != 0) ? " pair" : "",
	  ((role & F_CON) != 0) ? " con" : "",
	  ((role & F_CALL) != 0) ? " call" : "",
	  ((role & F_LAMBDA) != 0) ? " lambda" : "",
	  ((role & S_NAME) != 0) ? " raw-name" : "",
	  UNBOLD_TEXT);
  free(bits);
}

static s7_pointer opt2_1(s7_scheme *sc, s7_pointer p, uint32_t role, const char *func, int32_t line)
{
  if ((!opt2_is_set(p)) ||
      (!opt2_role_matches(p, role)))
    {
      show_opt2_bits(p, func, line, role);
      fprintf(stderr, "p: %s\n", string_value(s7_object_to_string(sc, p, false)));
      if (stop_at_error) abort();
    }
  return(p->object.cons.opt2);
}

static void set_opt2_1(s7_scheme *sc, s7_pointer p, s7_pointer x, uint32_t role, const char *func, int32_t line)
{
  if ((role == F_CALL) &&
      (x == NULL))  /* this happens apparently innocuously in check_and|or */
    {
      if ((safe_strcmp(func, "check_and") != 0) &&
	  (safe_strcmp(func, "check_or") != 0))
	fprintf(stderr, "%s[%d]: set c_call for %s to null\n", func, line, string_value(object_to_truncated_string(sc, p, 80)));
    }
  if ((role != F_CALL) &&
      (opt2_role_matches(p, F_CALL)) &&
      (has_fx(p)))
    fprintf(stderr, "%s[%d]: %s clobbers fx, p: %s\n", func, line, opt2_role_name(role), string_value(s7_object_to_string(sc, p, false)));
  p->object.cons.opt2 = x;
  set_opt2_role(p, role);
  set_opt2_is_set(p);
}

static const char *s_name_1(s7_pointer p, const char *func, int32_t line)
{
  if ((!opt2_is_set(p)) ||
      (!opt2_role_matches(p, S_NAME)))
    {
      show_opt2_bits(p, func, line, (uint32_t)S_NAME);
      if (stop_at_error) abort();
    }
  return(p->object.sym_cons.fstr);
}

static void set_s_name_1(s7_pointer p, const char *str, const char *func, int32_t line)
{
  p->object.sym_cons.fstr = str;
  set_opt2_role(p, S_NAME);
  set_opt2_is_set(p);
}

static void show_opt3_bits(s7_pointer p, const char *func, int32_t line, int32_t role)
{
  char *bits;
  bits = show_debugger_bits(p->debugger_bits);
  fprintf(stderr, "%s%s[%d]: opt3: %s %" PRIx64 "%s%s\n",
	  BOLD_TEXT,
	  func, line, opt3_role_name(role), p->debugger_bits, bits,
	  UNBOLD_TEXT);
  free(bits);
}

static s7_pointer opt3_1(s7_pointer p, uint32_t role, const char *func, int32_t line)
{
  if ((!opt3_is_set(p)) ||
      (!opt3_role_matches(p, role)))
    {
      show_opt3_bits(p, func, line, role);
      if (stop_at_error) abort();
    }
  return(p->object.cons.opt3);
}

static void set_opt3_1(s7_pointer p, s7_pointer x, uint32_t role, const char *func, int32_t line)
{
  clear_type_bit(p, T_LINE_NUMBER);
  p->object.cons.opt3 = x;
  set_opt3_is_set(p);
  set_opt3_role(p, role);
}

static uint8_t opt3_con_1(s7_pointer p, uint32_t role, const char *func, int32_t line)
{
  if ((!opt3_is_set(p)) ||
      (!opt3_role_matches(p, role)))
    {
      show_opt3_bits(p, func, line, role);
      if (stop_at_error) abort();
    }
  return(p->object.cons_ext.ce.opt_type);
}

static void set_opt3_con_1(s7_pointer p, uint8_t x, uint32_t role, const char *func, int32_t line)
{
  clear_type_bit(p, T_LINE_NUMBER);
  p->object.cons_ext.ce.opt_type = x;
  set_opt3_is_set(p);
  set_opt3_role(p, role);
}

static int32_t opt3_ctr_1(s7_pointer p, int32_t role, const char *func, int32_t line)
{
  if ((!opt3_is_set(p)) ||
      (!opt3_role_matches(p, role)))
    {
      show_opt3_bits(p, func, line, role);
      if (stop_at_error) abort();
    }
  return(p->object.cons_ext.ce.ctr);
}

static void set_opt3_ctr_1(s7_pointer p, int32_t x, uint32_t role, const char *func, int32_t line)
{
  p->object.cons_ext.ce.ctr = x;
  set_ctr3_is_set(p);
  set_opt3_is_set(p);
  set_opt3_role(p, role);
}

static void increment_opt3_ctr_1(s7_pointer p, uint32_t role, const char *func, int32_t line)
{
  if (ctr3_is_set(p)) 
    p->object.cons_ext.ce.ctr++;
  else p->object.cons_ext.ce.ctr = 0;
  set_ctr3_is_set(p);
  set_opt3_is_set(p);
  set_opt3_role(p, role);
}

/* S_LINE */
static uint32_t s_line_1(s7_pointer p, const char *func, int32_t line)
{
  if ((!opt3_is_set(p)) ||
      ((p->debugger_bits & S_LINE) == 0) ||
      (!has_line_number(p)))
    {
      show_opt3_bits(p, func, line, (uint32_t)S_LINE);
      if (stop_at_error) abort();
    }
  return(p->object.sym_cons.line);
}

static void set_s_line_1(s7_pointer p, uint32_t x, const char *func, int32_t line)
{
  p->object.sym_cons.line = x;
  (p)->debugger_bits = (S_LINE | (p->debugger_bits & ~S_LEN)); /* turn on line, cancel len */
  set_opt3_is_set(p);
}

static void set_s_file_1(s7_scheme *sc, s7_pointer p, uint32_t x, const char *func, int32_t line)
{
  p->object.sym_cons.file = x;
  if ((int32_t)x > sc->file_names_top)
    {
      fprintf(stderr, "%s[%d]: pair_set_file_name to %u?\n", func, line, x);
      if (stop_at_error) abort();
    }
}

/* S_LEN (collides with S_LINE) */
static uint32_t s_len_1(s7_pointer p, const char *func, int32_t line)
{
  if ((!opt3_is_set(p)) ||
      ((p->debugger_bits & S_LEN) == 0) ||
      (has_line_number(p)))
    {
      show_opt3_bits(p, func, line, (uint32_t)S_LEN);
      if (stop_at_error) abort();
    }
  return(p->object.sym_cons.line);
}

static void set_s_len_1(s7_pointer p, uint32_t x, const char *func, int32_t line)
{
  clear_type_bit(p, T_LINE_NUMBER);
  p->object.sym_cons.line = x;
  (p)->debugger_bits = (S_LEN | (p->debugger_bits & ~(S_LINE)));
  set_opt3_is_set(p);
}

static void print_debugging_state(s7_scheme *sc, s7_pointer obj, s7_pointer port)
{
  /* show current state, current allocated state, and previous allocated state */
  char *current_bits, *allocated_bits, *previous_bits, *str;
  int64_t save_typeflag;
  s7_int len, nlen;
  const char *excl_name;
  block_t *b;

  if (is_free(obj))
    excl_name = "free cell!";
  else excl_name = "unknown object!";

  