use core:lang;
use core:asm;

/**
 * Lambda expressions. There are two types of them: one requires specifying types in the parameter
 * list, but can be used anywhere. The other does not require types in the parameter list, but needs
 * to be used in a context where the types of the parameters can be inferred (not implemented yet).
 *
 * We should also support captures in some manner.
 */
class LambdaExpr extends Expr {
	// Parameters to the function. Parameters where the type is 'void' were not specified at the
	// declaration site and need to be inferred.
	private Array<ValParam> fnParams;

	// Scope for the lambda function.
	private Scope scope;

	// Body of the lambda function, used whenever we need to deduce parameters.
	private SExpr rawBody;

	// Transformed body of the function, assuming we know the type of all parameters.
	private LambdaBody? tfmBody;

	// Parent block.
	private Block parent;

	// Type of the function object, if known.
	private Value fnType;

	// Create a lambda expression where the types of all parameters are known.
	init(SrcPos pos, Block parent, Array<NameParam> params, SExpr body) {
		self(pos, parent, params.resolve(parent.scope), body);
	}

	// Create a lambda expression where the types of all parameters are known.
	init(SrcPos pos, Block parent, Array<ValParam> params, SExpr body) {
		self(pos, parent, params, body, LambdaCaptureStrategy());
	}

	// Create a lambda expression where the types of all parameters are known, providing a custom storage strategy.
	init(SrcPos pos, Block parent, Array<ValParam> params, SExpr body, LambdaCaptureStrategy storage) {
		Scope scope = lambdaScope(parent);

		LambdaBody tfmBody(scope, parent, params, body, storage);

		SimpleName name;
		name.add("core");
		name.add("Fn", { var z = params.values(); z.insert(0, tfmBody.result.type); z; });

		init(pos) {
			fnParams = params;
			scope = scope;
			rawBody = body;
			tfmBody = tfmBody;
			parent = parent;
			fnType = rootScope.value(name, pos);
		}
	}

	// Create a lambda expression where the type of the parameters are unknown and have to be deduced.
	init(SrcPos pos, Block parent, Array<SStr> paramNames, SExpr body) {
		ValParam[] params;
		for (i in paramNames)
			params << ValParam(Value(), i);

		init(pos) {
			fnParams = params;
			scope = lambdaScope(parent);
			rawBody = body;
			tfmBody = null;
			fnType = Value();
			parent = parent;
		}
	}

	ExprResult result() {
		ExprResult(fnType);
	}

	// Attempt automatic type deduction if we need to.
	Int castPenalty(Value to) {
		if (to == fnType) {
			0;
		} else if (acceptable(to)) {
			10;
		} else {
			-1;
		}
	}

	void code(CodeGen to, CodeResult r) {
		unless (fnType = r.type.type) {
			throw SyntaxError(pos, "Unable to deduce parameter types for the lambda function.");
			return;
		}

		if (!acceptable(r.type))
			throw SyntaxError(pos, "Unable to deduce parameter types for the lambda function.");

		if (!r.needed())
			return;

		// Note: "acceptable" only accepts exact parameter matches if the parameter types were
		// specified (i.e. if tfmBody is non-null). The return type might, however, be different.
		ValParam[] params = fnParams;
		LambdaBody body = if (tfmBody) {
			tfmBody;
		} else {
			params = ValParam[];
			for (i, p in fnParams)
				params << ValParam(fnType.params[i + 1], p.name);

			LambdaBody(scope, parent, params, rawBody, LambdaCaptureStrategy());
		};

		if (body.anyCaptures) {
			captureCode(to, r, fnType.params[0], params, body);
		} else {
			plainCode(to, r, fnType.params[0], params, body);
		}
	}

	private void plainCode(CodeGen to, CodeResult r, Value result, ValParam[] params, LambdaBody body) {
		BSTreeFn fn(result, SStr("<lambda>", pos), params, to.runOn.thread);
		fn.parentLookup = lambdaParent(parent);
		fn.body = body.createFnBody(fn, scope);

		var v = r.location(to);
		to.l << mov(v, objPtr(pointer(fn)));
		r.created(to);
	}

	private void captureCode(CodeGen to, CodeResult r, Value result, ValParam[] params, LambdaBody body) {
		Type type("<lambda>", TypeFlags:typeClass);
		type.parentLookup = lambdaParent(parent);
		// TODO: Maybe we want to support 'known, but not statically' as well. I don't think it will
		// make much of a difference in this case though.
		if (t = to.runOn.thread)
			type.setThread(t);

		var create = body.createType(type);

		params.insert(0, thisParam(type));
		BSTreeFn fn(result, SStr("<run>", pos), params, null);
		type.add(fn);
		fn.body = body.createFnBody(fn, scope);

		FnPtr p(create, fn, pos);
		p.code(to, r);
	}

	protected void toS(StrBuf to) {
		to << "(";
		for (i, p in fnParams) {
			if (i > 0)
				to << ", ";
			if (p.type != Value())
				to << p.type << " ";
			to << p.name;
		}
		to << ") => ";
		if (tfmBody) {
			to << tfmBody;
		} else {
			// TODO: It would be nice to output a nice formatted body here, but at the moment that
			// is not easily done since we need to transform the body in a suitable context for that
			// to work. As it is now, we don't necessarily know that context here.
			to << "<lambda body>";
		}
	}

	// Is 'type' an acceptable return type given what we currently know?
	private Bool acceptable(Value type) {
		unless (fn = type.type as FnType)
			return false;

		// Proper number of parameters?
		if (fn.params.count != fnParams.count + 1)
			return false;

		// Make sure that all parameter types that are specified are properly set.
		for (i, p in fnParams) {
			if (p.type == Value()) {
				// Not specified. Anything goes!
			} else if (p.type != fn.params[i + 1]) {
				// Non-compatible type. Not possible!
				return false;
			}
		}

		// All is well!
		true;
	}
}


/**
 * Root block of a lambda function. Intercepts variable lookups and creates suitable replacement
 * variables.
 *
 * Inherits from ReturnPoint (which inherits from Block) to support "return" statements properly.
 */
private class LambdaBody extends ReturnPoint {
	// Create.
	init(Scope scope, Block context, Array<ValParam> params, SExpr body, LambdaCaptureStrategy capture) {
		DeductiveReturnInfo info;

		init(body.pos, scope, info) {
			context = context;
			capture = capture;
		}

		for (p in params) {
			LocalVar created(p.name, p.type, body.pos, true);
			parameters << created;
			add(created);
		}

		this.body = info.deduceResult(body.transform(this), scope);
	}

	// Context from where the lambda function can capture variables.
	private Block context;

	// Variable capture strategy.
	private LambdaCaptureStrategy capture;

	// Variables captured from the surrounding context.
	private LocalVar[] captured;

	// Corresponding variables created here.
	private LocalVar[] shadows;

	// Variables created to capture parameters. These need to be replaced before using in the real
	// function.
	private LocalVar[] parameters;

	// The contents of the block.
	private Expr? body;

	// The function body we are located inside. Used to update parameters etc.
	private FnBody? parentBody;

	// Anything captured?
	Bool anyCaptures() {
		captured.any;
	}

	// Intercept variable lookup.
	LocalVar? variableHere(SimplePart name) : override {
		// Look here first:
		if (here = super:variableHere(name))
			return here;

		// Otherwise, look inside the context.
		if (inContext = context.variable(name)) {
			var type = capture.shadowAs(inContext.result);
			LocalVar shadow(inContext.name, type, inContext.pos, false);
			captured << inContext;
			shadows << shadow;
			add(shadow);
			return shadow;
		}

		return null;
	}

	// Create a FnBody, taking care of the parameters in here.
	FnBody createFnBody(BSRawFn fn, Scope scope) {
		FnBody body(fn, scope);
		parentBody = body;
		body.add(this);
		body;
	}

	// Generate code.
	void code(CodeGen gen, CodeResult to) : override {
		unless (parentBody)
			throw InternalError("Trying to compile a lambda without creating a surrounding function body!");
		unless (body)
			throw InternalError("The body should be set.");

		CodeGen childGen = gen.child();

		gen.l << begin(childGen.block);

		// Initialize parameter variables. In this case, we can just "steal" the parameter from the
		// function block.
		Nat offset = parentBody.parameters.count - parameters.count;
		for (i, p in parameters) {
			p.var = parentBody.parameters[i + offset].var;
		}

		// Call the super class, will create local variables for the references we created as well.
		initVariables(gen);

		// Initialize captured variables. This assumes that the first parameter is the "this" parameter.
		if (captured.any) {
			if (offset == 0)
				throw InternalError("Captured variables require the function to be a member function.");
			var thisVar = parentBody.parameters[0];

			unless (thisType = thisVar.result.type)
				throw InternalError("Invalid this variable.");

			// Initialize other variables by getting a pointer from the type.
			Bool resetPtrA = true;
			for (s in shadows) {
				unless (member = thisType.find(SimplePart(s.name, thisType), Scope()) as MemberVar)
					throw InternalError("Failed to find member variable of the captured object.");

				if (resetPtrA)
					gen.l << mov(ptrA, thisVar.var.v);
				resetPtrA = capture.restore(gen, s, member, ptrA);
			}
		}

		// Generate code for the body. We must take care of the return information as well to make
		// 'return' statements work.
		info.inlineResult = to;
		info.inlineLabel = gen.l.label();
		info.inlineBlock = childGen.block;

		body.code(childGen, to.split(gen));

		gen.l << end(childGen.block);
		gen.l << info.inlineLabel;
		to.created(gen);
	}

	// Populate a type with any captured variables. The returned expression will instantiate the
	// type with the proper parameters.
	Expr createType(Type type) {
		MemberVar[] vars;
		ValParam[] formals;
		formals << thisParam(type);

		// Compute formal parameters.
		Actuals actuals;
		for (i, c in captured) {
			MemberVar member(c.name, capture.storeAs(c.result), type);
			type.add(member);
			vars << member;

			formals << ValParam(c.result, i.toS);

			actuals.add(LocalVarAccess(pos, c));
		}

		// Create a constructor.
		BSTreeCtor ctor(formals, pos);
		ctor.body = {
			CtorBody b(ctor, rootScope); // We don't need a proper scope.
			Initializer[] init;
			for (i, v in vars) {
				var formal = b.parameters[i + 1];
				var initExpr = capture.storeInitializer(v.type, LocalVarAccess(pos, formal));
				init << Initializer(SStr(v.name), initExpr);
			}
			InitBlock s(pos, b, Actuals(), init);
			b.add(s);
			b;
		};
		type.add(ctor);

		CtorCall(pos, rootScope, ctor, actuals);
	}

	// Print, mostly for debugging.
	void toS(StrBuf to) : override {
		if (body) {
			to << body;
		}
	}
}


/**
 * Class that allows customizing how lambda functions store captured variables.
 *
 * This is useful to store certain types of variables in a special way, for example. Introduced to
 * handle nonmovable variables in the effects implementation specially.
 *
 * The default class implements the default strategy.
 */
class LambdaCaptureStrategy {
	// Compute the type of how a captured variable should be shadowed.
	//
	// This allows determining how different types should be shadowed. In most situations, this is
	// typically used to determine which variables should be values and which should be references.
	Value shadowAs(Value original) {
		original.asRef;
	}

	// Compute the storage type of a captured variable.
	//
	// Called by LambdaBody when it has determined that it needs to store a captured variable inside
	// an object. The function returns the type that should be used to store the variable inside the
	// lambda object.
	Value storeAs(Value original) {
		// Note: Never store things as references. That won't work.
		original.asRef(false);
	}

	// Customize how to initialize a variable.
	//
	// If the 'storeAs' was overriden to return an incompatible type for some data type, this
	// function allows customizing the expression that is used to initialize the new variable inside
	// the type.
	Expr storeInitializer(Value type, Expr original) {
		original;
	}

	// Customize how to load a captured variable from the object.
	//
	// Called to customize the initialization of the local variable that is used in the lambda
	// function. That is, the expression produced here should produce code that initializes the
	// variable "init" from the member "member" that was previously stored in the object. This
	// should be done from the object that resides in "baseReg". Should return 'true' if 'baseReg'
	// was clobbered.
	Bool restore(CodeGen gen, LocalVar init, MemberVar member, Reg baseReg) {
		gen.l << lea(init.var.v, ptrRel(baseReg, member.offset));
		false;
	}
}


/**
 * ReturnInfo that deduces the return type based on what is actually used with 'return'.
 */
class DeductiveReturnInfo extends ReturnInfo {
	init() {
		init(Value()) {}
	}

	// All captured return expressions.
	private Wrap[] returnExprs;

	// Called for all 'return' statements.
	Expr? checkResult(SrcPos pos, Scope scope, Expr? expr) : override {
		if (expr) {
			Wrap wrap(expr);
			returnExprs << wrap;
			return wrap;
		} else {
			return null;
		}
	}

	// Call when the tree has been transformed. Deduces a single return type for the entire block
	// and inserts suitable type-casts.
	Expr deduceResult(Expr body, Scope scope) {
		ExprResult bodyResult = body.result();
		ExprResult acc = bodyResult;
		for (e in returnExprs) {
			acc = e.findType(acc);
		}

		// If 'acc' is "never returns", then all paths throws. This means that we can just skip
		// type-conversions.
		if (acc.nothing)
			return body;

		Value finalType = acc.type;
		this.type = finalType;

		// Now, we have a type. Try to cast everything to that type.
		for (e in returnExprs) {
			e.castTo(finalType, scope);
		}

		if (!bodyResult.nothing())
			return expectCastTo(body, finalType, scope);
		else
			return body;
	}

	// Wrap expressions that are returned, so that we can change their type when we have all type
	// information.
	private class Wrap extends Expr {
		// Create.
		init(Expr inside) {
			init(inside.pos) {
				inside = inside;
			}
		}

		// Contained expression.
		private Expr inside;

		// Result.
		ExprResult result() : override { inside.result(); }

		// Generate code.
		void code(CodeGen code, CodeResult result) : override { inside.code(code, result); }

		// Note: This only needs to be enough for the "return" expr. As such, we don't need to
		// implement the rest of the interface.

		// Help out in type deduction.
		ExprResult findType(ExprResult acc) {
			var insideResult = inside.result();

			if (acc.nothing)
				return insideResult;
			if (insideResult.nothing)
				return acc;

			// Use regular inheritance check.
			var c = common(acc.type, insideResult.type);

			// If it failed, then just return 'acc'. This makes us try a type conversion later,
			// which might fail (which is fine).
			if (c == Value())
				return acc;
			return ExprResult(c);
		}

		// Cast the contained expression.
		void castTo(Value type, Scope scope) {
			inside = expectCastTo(inside, type, scope);
		}
	}
}


/**
 * Find the proper scope for a lambda function.
 */
private Scope lambdaScope(Block block) on Compiler {
	NameLookup at = block.lookup;
	while (true) {
		if (at as NameSet) {
			return block.scope.child(at);
		} else if (parent = at.parent) {
			at = parent;
		} else {
			return rootScope;
		}
	}

	// Crude fallback. Should not happen.
	return rootScope();
}

/**
 * Find the proper parent to a lambda function.
 */
private NameLookup lambdaParent(Block block) on Compiler {
	NameLookup at = block.lookup;
	while (true) {
		// print(at.toS);
		if (at as Package) {
			return at;
		} else if (at as Type) {
			return at;
		} else if (parent = at.parent) {
			at = parent;
		} else {
			return rootPkg;
		}
	}

	return rootPkg;
}
