/*
 * @(#)JeksDisplay.java   01/04/98
 *
 * Copyright (c) 2000 Emmanuel PUYBARET / eTeks <info@eteks.com>. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Visit eTeks web site for up-to-date versions of this file and other
 * Java tools and tutorials : http://www.eteks.com/
 */
import java.applet.Applet;
import java.awt.Component;
import java.awt.TextField;
import java.awt.TextComponent;
import java.awt.CheckboxGroup;
import java.awt.Checkbox;
import java.awt.Button;
import java.awt.Panel;
import java.awt.Label;
import java.awt.GridLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.BorderLayout;
import java.awt.Event;
import java.awt.Insets;
import java.util.StringTokenizer;
import java.util.NoSuchElementException;

import com.eteks.parser.FunctionParser;
import com.eteks.parser.CompiledFunction;
import com.eteks.parser.CompilationException;

import com.eteks.tools.awt.Curve2DDisplay;
import com.eteks.tools.awt.Curve3DDisplay;

/**
 * An applet able to display 2D or 3D curves defined by their function <i>f(x)=...</i>
 * or <i>f(x,y)=...</i>.
 *
 * @version 1.0
 * @author  Emmanuel Puybaret
 * @since   Jeks 1.0
 */
public class JeksDisplay extends Applet
{
  private TextField      functionField;
  private TextField      parameterField;
  private Label          resultLabel;

  private TextField      minimumX;
  private TextField      maximumX;

  private CheckboxGroup  displayModeGroup;
  private Checkbox       ligthingCheckbox;
  private Checkbox       meshCheckbox;
  private Checkbox       allMeshCheckbox;
  private Button         defaultButton;

  private Panel          containerDisplayParam;
  private Panel          containerParam3D;

  private Component      curveDisplay;
  private Curve2DDisplay displayCanvas2D = new Curve2DDisplay ();
  private Curve3DDisplay displayCanvas3D = new Curve3DDisplay ();

  private FunctionParser   parser = new FunctionParser ();
  private CompiledFunction lastFunction;

  private int            meshSize = 20;

  public void init ()
  {
    // Build GUI
    setLayout (new BorderLayout (4, 4));

    GridBagLayout      gridbag = new GridBagLayout();
    GridBagConstraints c = new GridBagConstraints();

    Panel containerInput = new Panel ();
    containerInput.setLayout (gridbag);

    Label label = new Label ("Function :");
    c.fill   = GridBagConstraints.BOTH;
    c.insets = new Insets (2, 1, 2, 1);
    gridbag.setConstraints (label, c);
    containerInput.add (label);

    functionField  = new TextField ();
    c.gridwidth = GridBagConstraints.REMAINDER;
    c.weightx   = 1.;
    gridbag.setConstraints (functionField, c);
    containerInput.add (functionField);

    label = new Label ("Compute :");
    c.gridwidth = 1;
    c.weightx   = 0.;
    gridbag.setConstraints (label, c);
    containerInput.add (label);

    parameterField = new TextField ();
    c.gridwidth = 1;
    c.weightx   = 0.2;
    gridbag.setConstraints (parameterField, c);
    containerInput.add (parameterField);

    resultLabel = new Label ();
    c.gridwidth = GridBagConstraints.REMAINDER;
    c.weightx   = 0.8;
    gridbag.setConstraints(resultLabel, c);
    containerInput.add (resultLabel);

    add ("North", containerInput);

    // Minimum / Maximum
    Panel containerMinMax  = new Panel ();
    containerMinMax.setLayout (new GridLayout (4, 1, 1, 1));

    containerMinMax.add (new Label ("Minimum :"));
    minimumX = new TextField ("-2");
    containerMinMax.add (minimumX);

    containerMinMax.add (new Label ("Maximum :"));
    maximumX = new TextField ("2");
    containerMinMax.add (maximumX);

    // 3D parameters
    containerParam3D = new Panel ();
    containerParam3D.setLayout (new GridLayout (5, 1, 1, 1));

    containerParam3D.add (new Label ("Mesh mode :"));

    displayModeGroup = new CheckboxGroup ();
    allMeshCheckbox = new Checkbox ("Complete mesh", displayModeGroup, true);
    containerParam3D.add (allMeshCheckbox);

    meshCheckbox = new Checkbox ("White mesh", displayModeGroup, false);
    containerParam3D.add (meshCheckbox);

    ligthingCheckbox = new Checkbox ("Lighted mesh", displayModeGroup, false);
    containerParam3D.add (ligthingCheckbox);

    defaultButton = new Button ("Default position");
    containerParam3D.add (defaultButton);

    // Display parameters
    gridbag = new GridBagLayout();
    c = new GridBagConstraints();

    containerDisplayParam = new Panel ();
    containerDisplayParam.setLayout (gridbag);

    c.insets    = new Insets (2, 1, 2, 1);
    c.gridwidth = GridBagConstraints.REMAINDER;
    c.anchor    = GridBagConstraints.NORTHWEST;
    gridbag.setConstraints (containerMinMax, c);
    gridbag.setConstraints (containerParam3D, c);

    containerDisplayParam.add (containerMinMax);
    // containerParam3D added when needed
    add ("East", containerDisplayParam);

    // Retrieve parameters
    String param = getParameter ("meshSize");
    if (param != null)
      meshSize = Integer.parseInt (param);

    param = getParameter ("defaultFunction");
    if (param != null)
    {
      ((TextComponent)functionField).setText (param);
      postEvent (new Event (functionField, Event.ACTION_EVENT, param));
    }
  }

  public boolean action (Event event, Object eventArg)
  {
    if (event.target == functionField)
      try
      {
        // Compile the user function
        CompiledFunction function = parser.compileFunction (functionField.getText ());

        if (   function.getParameterCount () == 1
            || function.getParameterCount () == 2)
        {
          resultLabel.setText ("Compilation of " + function.getName () + " OK.");
          displayFunction (function);
        }
        else
          resultLabel.setText ("Error : Can display only functions with 1 or 2 parameters.");
      }
      catch (CompilationException e)
      {
        StringBuffer pointer = new StringBuffer ("^");
        for (int i = 0; i < e.getCharacterIndex (); i++)
          pointer.insert (0, ' ');
        resultLabel.setText ("Error : " + e.getMessage () + ".");
      }
    else
      if (lastFunction != null)
        if (   event.target == minimumX
            || event.target == maximumX)
          displayFunction (lastFunction);
        else
        if (event.target == parameterField)
          try
          {
            StringTokenizer rechStr = new StringTokenizer (parameterField.getText (), " (,)");
            if (!lastFunction.getName ().equals (rechStr.nextToken ()))
              throw new IllegalArgumentException ("Unkonwn function");

            double [] parameters = new double [lastFunction.getParameterCount ()];
            for (int i = 0; i < lastFunction.getParameterCount (); i++)
              parameters [i] = Double.valueOf (rechStr.nextToken ()).doubleValue ();

            resultLabel.setText (  "Result of " + parameterField.getText () + " = "
                                 + lastFunction.computeFunction (parameters));
          }
          catch (NumberFormatException e)
          {
            resultLabel.setText ("Error : " + e + ".");
          }
          catch (IllegalArgumentException e)
          {
            resultLabel.setText ("Error : " + e + ".");
          }
          catch (NoSuchElementException e)
          {
            resultLabel.setText ("Error : Missing function or parameters");
          }
        else if (lastFunction.getParameterCount () == 2)
          if (event.target == defaultButton)
            displayCanvas3D.setDefaultPosition ();
          else if (event.target == ligthingCheckbox)
            displayCanvas3D.setDisplayMode (Curve3DDisplay.DISPLAY_WITH_LIGHTING);
          else if (event.target == meshCheckbox)
            displayCanvas3D.setDisplayMode (Curve3DDisplay.DISPLAY_MESH);
          else if (event.target == allMeshCheckbox)
            displayCanvas3D.setDisplayMode (Curve3DDisplay.DISPLAY_ALL_MESH);

    return true;
  }

  private void displayFunction (CompiledFunction function)
  {
    lastFunction = function;
    if (curveDisplay != null)
      remove (curveDisplay);
    if (lastFunction.getParameterCount () == 1)
    {
      curveDisplay = displayCanvas2D;
      containerDisplayParam.remove (containerParam3D);
      displayCanvas2D.computeCurve (lastFunction,
                                    Double.valueOf (minimumX.getText ()).doubleValue (),
                                    Double.valueOf (maximumX.getText ()).doubleValue ());
    }
    else
    {
      curveDisplay = displayCanvas3D;
      containerDisplayParam.add (containerParam3D);
      displayCanvas3D.computeCurve (lastFunction,
                                     Double.valueOf (minimumX.getText ()).doubleValue (),
                                     Double.valueOf (maximumX.getText ()).doubleValue (),
                                     meshSize);
    }

    add ("Center", curveDisplay);
    validate ();
  }
}

