//===-- CoreFoundation/URL/CFURLSessionInterface.h - Very brief description -----------*- C++ -*-===//
//
// This source file is part of the Swift.org open source project
//
// Copyright (c) 2014 - 2016 Apple Inc. and the Swift project authors
// Licensed under Apache License v2.0 with Runtime Library Exception
//
// See http://swift.org/LICENSE.txt for license information
// See http://swift.org/CONTRIBUTORS.txt for the list of Swift project authors
//
//===----------------------------------------------------------------------===//
///
/// \file
/// This file contains wrappes / helpers to import libcurl into Swift.
/// It is used to implement the NSURLSession API.
///
/// In most cases each `curl_…` API is mapped 1-to-1 to a corresponding
/// `CFURLSession_…` API.
///
/// This approach lets us keep most of the logic inside Swift code as opposed
/// to more C code.
///
/// - SeeAlso: https://curl.haxx.se/libcurl/c/
///
//===----------------------------------------------------------------------===//

#if !defined(__COREFOUNDATION_URLSESSIONINTERFACE__)
#define __COREFOUNDATION_URLSESSIONINTERFACE__ 1

#include <CoreFoundation/CoreFoundation.h>
#include <stdio.h>
#if defined(_WIN32)
#include <winsock2.h>
#endif

CF_IMPLICIT_BRIDGING_ENABLED
CF_EXTERN_C_BEGIN


/// CURL
typedef void * CFURLSessionEasyHandle;

/// CURLM
typedef void * CFURLSessionMultiHandle;

// This must match libcurl's curl_socket_t
#if defined(_WIN32)
typedef SOCKET CFURLSession_socket_t;
#else
typedef int CFURLSession_socket_t;
#endif



typedef struct CFURLSessionEasyCode {
    int value;
} CFURLSessionEasyCode;

CF_EXPORT const char * _Nullable CFURLSessionEasyCodeDescription(CFURLSessionEasyCode code);

CF_EXPORT int const CFURLSessionEasyErrorSize;

/// CURLcode
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeOK; // CURLE_OK
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeUNSUPPORTED_PROTOCOL; // CURLE_UNSUPPORTED_PROTOCOL
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFAILED_INIT; // CURLE_FAILED_INIT
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeURL_MALFORMAT; // CURLE_URL_MALFORMAT
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeNOT_BUILT_IN; // CURLE_NOT_BUILT_IN
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeCOULDNT_RESOLVE_PROXY; // CURLE_COULDNT_RESOLVE_PROXY
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeCOULDNT_RESOLVE_HOST; // CURLE_COULDNT_RESOLVE_HOST
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeCOULDNT_CONNECT; // CURLE_COULDNT_CONNECT
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFTP_WEIRD_SERVER_REPLY; // CURLE_FTP_WEIRD_SERVER_REPLY
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeREMOTE_ACCESS_DENIED; // CURLE_REMOTE_ACCESS_DENIED
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFTP_ACCEPT_FAILED; // CURLE_FTP_ACCEPT_FAILED
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFTP_WEIRD_PASS_REPLY; // CURLE_FTP_WEIRD_PASS_REPLY
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFTP_ACCEPT_TIMEOUT; // CURLE_FTP_ACCEPT_TIMEOUT
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFTP_WEIRD_PASV_REPLY; // CURLE_FTP_WEIRD_PASV_REPLY
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFTP_WEIRD_227_FORMAT; // CURLE_FTP_WEIRD_227_FORMAT
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFTP_CANT_GET_HOST; // CURLE_FTP_CANT_GET_HOST
//CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeHTTP2; // CURLE_HTTP2
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFTP_COULDNT_SET_TYPE; // CURLE_FTP_COULDNT_SET_TYPE
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodePARTIAL_FILE; // CURLE_PARTIAL_FILE
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFTP_COULDNT_RETR_FILE; // CURLE_FTP_COULDNT_RETR_FILE
#if LIBCURL_VERSION_MAJOR > 7 || (LIBCURL_VERSION_MAJOR == 7 && LIBCURL_VERSION_MINOR > 67) || (LIBCURL_VERSION_MAJOR == 7 && LIBCURL_VERSION_MINOR == 67 && LIBCURL_VERSION_PATCH >= 1)
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeHTTP3; // CURLE_HTTP3
#else
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeOBSOLETE20; // CURLE_OBSOLETE20
#endif
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeQUOTE_ERROR; // CURLE_QUOTE_ERROR
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeHTTP_RETURNED_ERROR; // CURLE_HTTP_RETURNED_ERROR
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeWRITE_ERROR; // CURLE_WRITE_ERROR
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeOBSOLETE24; // CURLE_OBSOLETE24
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeUPLOAD_FAILED; // CURLE_UPLOAD_FAILED
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeREAD_ERROR; // CURLE_READ_ERROR
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeOUT_OF_MEMORY; // CURLE_OUT_OF_MEMORY
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeOPERATION_TIMEDOUT; // CURLE_OPERATION_TIMEDOUT
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeOBSOLETE29; // CURLE_OBSOLETE29
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFTP_PORT_FAILED; // CURLE_FTP_PORT_FAILED
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFTP_COULDNT_USE_REST; // CURLE_FTP_COULDNT_USE_REST
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeOBSOLETE32; // CURLE_OBSOLETE32
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeRANGE_ERROR; // CURLE_RANGE_ERROR
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeHTTP_POST_ERROR; // CURLE_HTTP_POST_ERROR
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSSL_CONNECT_ERROR; // CURLE_SSL_CONNECT_ERROR
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeBAD_DOWNLOAD_RESUME; // CURLE_BAD_DOWNLOAD_RESUME
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFILE_COULDNT_READ_FILE; // CURLE_FILE_COULDNT_READ_FILE
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeLDAP_CANNOT_BIND; // CURLE_LDAP_CANNOT_BIND
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeLDAP_SEARCH_FAILED; // CURLE_LDAP_SEARCH_FAILED
//CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeOBSOLETE40; // CURLE_OBSOLETE40
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFUNCTION_NOT_FOUND; // CURLE_FUNCTION_NOT_FOUND
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeABORTED_BY_CALLBACK; // CURLE_ABORTED_BY_CALLBACK
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeBAD_FUNCTION_ARGUMENT; // CURLE_BAD_FUNCTION_ARGUMENT
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeOBSOLETE44; // CURLE_OBSOLETE44
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeINTERFACE_FAILED; // CURLE_INTERFACE_FAILED
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeOBSOLETE46; // CURLE_OBSOLETE46
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeTOO_MANY_REDIRECTS; // CURLE_TOO_MANY_REDIRECTS
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeUNKNOWN_OPTION; // CURLE_UNKNOWN_OPTION
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeTELNET_OPTION_SYNTAX; // CURLE_TELNET_OPTION_SYNTAX
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeOBSOLETE50; // CURLE_OBSOLETE50
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodePEER_FAILED_VERIFICATION; // CURLE_PEER_FAILED_VERIFICATION
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeGOT_NOTHING; // CURLE_GOT_NOTHING
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSSL_ENGINE_NOTFOUND; // CURLE_SSL_ENGINE_NOTFOUND
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSSL_ENGINE_SETFAILED; // CURLE_SSL_ENGINE_SETFAILED
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSEND_ERROR; // CURLE_SEND_ERROR
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeRECV_ERROR; // CURLE_RECV_ERROR
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeOBSOLETE57; // CURLE_OBSOLETE57
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSSL_CERTPROBLEM; // CURLE_SSL_CERTPROBLEM
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSSL_CIPHER; // CURLE_SSL_CIPHER
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSSL_CACERT; // CURLE_SSL_CACERT
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeBAD_CONTENT_ENCODING; // CURLE_BAD_CONTENT_ENCODING
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeLDAP_INVALID_URL; // CURLE_LDAP_INVALID_URL
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFILESIZE_EXCEEDED; // CURLE_FILESIZE_EXCEEDED
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeUSE_SSL_FAILED; // CURLE_USE_SSL_FAILED
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSEND_FAIL_REWIND; // CURLE_SEND_FAIL_REWIND
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSSL_ENGINE_INITFAILED; // CURLE_SSL_ENGINE_INITFAILED
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeLOGIN_DENIED; // CURLE_LOGIN_DENIED
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeTFTP_NOTFOUND; // CURLE_TFTP_NOTFOUND
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeTFTP_PERM; // CURLE_TFTP_PERM
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeREMOTE_DISK_FULL; // CURLE_REMOTE_DISK_FULL
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeTFTP_ILLEGAL; // CURLE_TFTP_ILLEGAL
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeTFTP_UNKNOWNID; // CURLE_TFTP_UNKNOWNID
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeREMOTE_FILE_EXISTS; // CURLE_REMOTE_FILE_EXISTS
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeTFTP_NOSUCHUSER; // CURLE_TFTP_NOSUCHUSER
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeCONV_FAILED; // CURLE_CONV_FAILED
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeCONV_REQD; // CURLE_CONV_REQD
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSSL_CACERT_BADFILE; // CURLE_SSL_CACERT_BADFILE
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeREMOTE_FILE_NOT_FOUND; // CURLE_REMOTE_FILE_NOT_FOUND
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSSH; // CURLE_SSH
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSSL_SHUTDOWN_FAILED; // CURLE_SSL_SHUTDOWN_FAILED
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeAGAIN; // CURLE_AGAIN
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSSL_CRL_BADFILE; // CURLE_SSL_CRL_BADFILE
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSSL_ISSUER_ERROR; // CURLE_SSL_ISSUER_ERROR
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFTP_PRET_FAILED; // CURLE_FTP_PRET_FAILED
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeRTSP_CSEQ_ERROR; // CURLE_RTSP_CSEQ_ERROR
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeRTSP_SESSION_ERROR; // CURLE_RTSP_SESSION_ERROR
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeFTP_BAD_FILE_LIST; // CURLE_FTP_BAD_FILE_LIST
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeCHUNK_FAILED; // CURLE_CHUNK_FAILED
CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeNO_CONNECTION_AVAILABLE; // CURLE_NO_CONNECTION_AVAILABLE
//CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSSL_PINNEDPUBKEYNOTMATCH; // CURLE_SSL_PINNEDPUBKEYNOTMATCH
//CF_EXPORT CFURLSessionEasyCode const CFURLSessionEasyCodeSSL_INVALIDCERTSTATUS; // CURLE_SSL_INVALIDCERTSTATUS



/// CURLOPTTYPE
typedef enum {
    CFURLSessionOptTypeLONG          = 0, // CURLOPTTYPE_LONG
    CFURLSessionOptTypeOBJECTPOINT   = 10000, // CURLOPTTYPE_OBJECTPOINT
    CFURLSessionOptTypeFUNCTIONPOINT = 20000, // CURLOPTTYPE_FUNCTIONPOINT
    CFURLSessionOptTypeOFF_T         = 30000, // CURLOPTTYPE_OFF_T
} CFURLSessionOptType;



typedef struct CFURLSessionOption {
    int value;
} CFURLSessionOption;

/// CURLoption
CF_EXPORT CFURLSessionOption const CFURLSessionOptionWRITEDATA; // CURLOPT_WRITEDATA
CF_EXPORT CFURLSessionOption const CFURLSessionOptionURL; // CURLOPT_URL
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPORT; // CURLOPT_PORT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPROXY; // CURLOPT_PROXY
CF_EXPORT CFURLSessionOption const CFURLSessionOptionUSERPWD; // CURLOPT_USERPWD
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPROXYUSERPWD; // CURLOPT_PROXYUSERPWD
CF_EXPORT CFURLSessionOption const CFURLSessionOptionRANGE; // CURLOPT_RANGE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionREADDATA; // CURLOPT_READDATA
CF_EXPORT CFURLSessionOption const CFURLSessionOptionERRORBUFFER; // CURLOPT_ERRORBUFFER
CF_EXPORT CFURLSessionOption const CFURLSessionOptionWRITEFUNCTION; // CURLOPT_WRITEFUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionREADFUNCTION; // CURLOPT_READFUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionTIMEOUT; // CURLOPT_TIMEOUT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionINFILESIZE; // CURLOPT_INFILESIZE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPOSTFIELDS; // CURLOPT_POSTFIELDS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionREFERER; // CURLOPT_REFERER
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFTPPORT; // CURLOPT_FTPPORT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionUSERAGENT; // CURLOPT_USERAGENT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionLOW_SPEED_LIMIT; // CURLOPT_LOW_SPEED_LIMIT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionLOW_SPEED_TIME; // CURLOPT_LOW_SPEED_TIME
CF_EXPORT CFURLSessionOption const CFURLSessionOptionRESUME_FROM; // CURLOPT_RESUME_FROM
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCOOKIE; // CURLOPT_COOKIE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionHTTPHEADER; // CURLOPT_HTTPHEADER
CF_EXPORT CFURLSessionOption const CFURLSessionOptionHTTPPOST; // CURLOPT_HTTPPOST
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSLCERT; // CURLOPT_SSLCERT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionKEYPASSWD; // CURLOPT_KEYPASSWD
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCRLF; // CURLOPT_CRLF
CF_EXPORT CFURLSessionOption const CFURLSessionOptionQUOTE; // CURLOPT_QUOTE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionHEADERDATA; // CURLOPT_HEADERDATA
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCOOKIEFILE; // CURLOPT_COOKIEFILE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSLVERSION; // CURLOPT_SSLVERSION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionTIMECONDITION; // CURLOPT_TIMECONDITION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionTIMEVALUE; // CURLOPT_TIMEVALUE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCUSTOMREQUEST; // CURLOPT_CUSTOMREQUEST
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSTDERR; // CURLOPT_STDERR
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPOSTQUOTE; // CURLOPT_POSTQUOTE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionOBSOLETE40; // CURLOPT_OBSOLETE40
CF_EXPORT CFURLSessionOption const CFURLSessionOptionVERBOSE; // CURLOPT_VERBOSE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionHEADER; // CURLOPT_HEADER
CF_EXPORT CFURLSessionOption const CFURLSessionOptionNOPROGRESS; // CURLOPT_NOPROGRESS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionNOBODY; // CURLOPT_NOBODY
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFAILONERROR; // CURLOPT_FAILONERROR
CF_EXPORT CFURLSessionOption const CFURLSessionOptionUPLOAD; // CURLOPT_UPLOAD
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPOST; // CURLOPT_POST
CF_EXPORT CFURLSessionOption const CFURLSessionOptionDIRLISTONLY; // CURLOPT_DIRLISTONLY
CF_EXPORT CFURLSessionOption const CFURLSessionOptionAPPEND; // CURLOPT_APPEND
CF_EXPORT CFURLSessionOption const CFURLSessionOptionNETRC; // CURLOPT_NETRC
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFOLLOWLOCATION; // CURLOPT_FOLLOWLOCATION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionTRANSFERTEXT; // CURLOPT_TRANSFERTEXT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPUT; // CURLOPT_PUT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPROGRESSFUNCTION; // CURLOPT_PROGRESSFUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPROGRESSDATA; // CURLOPT_PROGRESSDATA
CF_EXPORT CFURLSessionOption const CFURLSessionOptionAUTOREFERER; // CURLOPT_AUTOREFERER
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPROXYPORT; // CURLOPT_PROXYPORT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPOSTFIELDSIZE; // CURLOPT_POSTFIELDSIZE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionHTTPPROXYTUNNEL; // CURLOPT_HTTPPROXYTUNNEL
CF_EXPORT CFURLSessionOption const CFURLSessionOptionINTERFACE; // CURLOPT_INTERFACE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionKRBLEVEL; // CURLOPT_KRBLEVEL
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSL_VERIFYPEER; // CURLOPT_SSL_VERIFYPEER
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCAINFO; // CURLOPT_CAINFO
CF_EXPORT CFURLSessionOption const CFURLSessionOptionMAXREDIRS; // CURLOPT_MAXREDIRS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFILETIME; // CURLOPT_FILETIME
CF_EXPORT CFURLSessionOption const CFURLSessionOptionTELNETOPTIONS; // CURLOPT_TELNETOPTIONS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionMAXCONNECTS; // CURLOPT_MAXCONNECTS
//CF_EXPORT CFURLSessionOption const CFURLSessionOptionOBSOLETE72; // CURLOPT_OBSOLETE72
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFRESH_CONNECT; // CURLOPT_FRESH_CONNECT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFORBID_REUSE; // CURLOPT_FORBID_REUSE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionRANDOM_FILE; // CURLOPT_RANDOM_FILE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionEGDSOCKET; // CURLOPT_EGDSOCKET
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCONNECTTIMEOUT; // CURLOPT_CONNECTTIMEOUT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionHEADERFUNCTION; // CURLOPT_HEADERFUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionHTTPGET; // CURLOPT_HTTPGET
//CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSL_VERIFYHOST; // CURLOPT_SSL_VERIFYHOST
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCOOKIEJAR; // CURLOPT_COOKIEJAR
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSL_CIPHER_LIST; // CURLOPT_SSL_CIPHER_LIST
CF_EXPORT CFURLSessionOption const CFURLSessionOptionHTTP_VERSION; // CURLOPT_HTTP_VERSION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFTP_USE_EPSV; // CURLOPT_FTP_USE_EPSV
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSLCERTTYPE; // CURLOPT_SSLCERTTYPE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSLKEY; // CURLOPT_SSLKEY
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSLKEYTYPE; // CURLOPT_SSLKEYTYPE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSLENGINE; // CURLOPT_SSLENGINE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSLENGINE_DEFAULT; // CURLOPT_SSLENGINE_DEFAULT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionDNS_USE_GLOBAL_CACHE; // CURLOPT_DNS_USE_GLOBAL_CACHE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionDNS_CACHE_TIMEOUT; // CURLOPT_DNS_CACHE_TIMEOUT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPREQUOTE; // CURLOPT_PREQUOTE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionDEBUGFUNCTION; // CURLOPT_DEBUGFUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionDEBUGDATA; // CURLOPT_DEBUGDATA
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCOOKIESESSION; // CURLOPT_COOKIESESSION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCAPATH; // CURLOPT_CAPATH
CF_EXPORT CFURLSessionOption const CFURLSessionOptionBUFFERSIZE; // CURLOPT_BUFFERSIZE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionNOSIGNAL; // CURLOPT_NOSIGNAL
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSHARE; // CURLOPT_SHARE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPROXYTYPE; // CURLOPT_PROXYTYPE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionACCEPT_ENCODING; // CURLOPT_ACCEPT_ENCODING
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPRIVATE; // CURLOPT_PRIVATE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionHTTP200ALIASES; // CURLOPT_HTTP200ALIASES
CF_EXPORT CFURLSessionOption const CFURLSessionOptionUNRESTRICTED_AUTH; // CURLOPT_UNRESTRICTED_AUTH
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFTP_USE_EPRT; // CURLOPT_FTP_USE_EPRT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionHTTPAUTH; // CURLOPT_HTTPAUTH
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSL_CTX_FUNCTION; // CURLOPT_SSL_CTX_FUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSL_CTX_DATA; // CURLOPT_SSL_CTX_DATA
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFTP_CREATE_MISSING_DIRS; // CURLOPT_FTP_CREATE_MISSING_DIRS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPROXYAUTH; // CURLOPT_PROXYAUTH
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFTP_RESPONSE_TIMEOUT; // CURLOPT_FTP_RESPONSE_TIMEOUT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionIPRESOLVE; // CURLOPT_IPRESOLVE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionMAXFILESIZE; // CURLOPT_MAXFILESIZE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionINFILESIZE_LARGE; // CURLOPT_INFILESIZE_LARGE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionRESUME_FROM_LARGE; // CURLOPT_RESUME_FROM_LARGE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionMAXFILESIZE_LARGE; // CURLOPT_MAXFILESIZE_LARGE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionNETRC_FILE; // CURLOPT_NETRC_FILE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionUSE_SSL; // CURLOPT_USE_SSL
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPOSTFIELDSIZE_LARGE; // CURLOPT_POSTFIELDSIZE_LARGE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionTCP_NODELAY; // CURLOPT_TCP_NODELAY
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFTPSSLAUTH; // CURLOPT_FTPSSLAUTH
CF_EXPORT CFURLSessionOption const CFURLSessionOptionIOCTLFUNCTION; // CURLOPT_IOCTLFUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionIOCTLDATA; // CURLOPT_IOCTLDATA
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFTP_ACCOUNT; // CURLOPT_FTP_ACCOUNT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCOOKIELIST; // CURLOPT_COOKIELIST
CF_EXPORT CFURLSessionOption const CFURLSessionOptionIGNORE_CONTENT_LENGTH; // CURLOPT_IGNORE_CONTENT_LENGTH
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFTP_SKIP_PASV_IP; // CURLOPT_FTP_SKIP_PASV_IP
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFTP_FILEMETHOD; // CURLOPT_FTP_FILEMETHOD
CF_EXPORT CFURLSessionOption const CFURLSessionOptionLOCALPORT; // CURLOPT_LOCALPORT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionLOCALPORTRANGE; // CURLOPT_LOCALPORTRANGE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCONNECT_ONLY; // CURLOPT_CONNECT_ONLY
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCONV_FROM_NETWORK_FUNCTION; // CURLOPT_CONV_FROM_NETWORK_FUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCONV_TO_NETWORK_FUNCTION; // CURLOPT_CONV_TO_NETWORK_FUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCONV_FROM_UTF8_FUNCTION; // CURLOPT_CONV_FROM_UTF8_FUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionMAX_SEND_SPEED_LARGE; // CURLOPT_MAX_SEND_SPEED_LARGE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionMAX_RECV_SPEED_LARGE; // CURLOPT_MAX_RECV_SPEED_LARGE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFTP_ALTERNATIVE_TO_USER; // CURLOPT_FTP_ALTERNATIVE_TO_USER
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSOCKOPTFUNCTION; // CURLOPT_SOCKOPTFUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSOCKOPTDATA; // CURLOPT_SOCKOPTDATA
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSL_SESSIONID_CACHE; // CURLOPT_SSL_SESSIONID_CACHE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSH_AUTH_TYPES; // CURLOPT_SSH_AUTH_TYPES
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSH_PUBLIC_KEYFILE; // CURLOPT_SSH_PUBLIC_KEYFILE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSH_PRIVATE_KEYFILE; // CURLOPT_SSH_PRIVATE_KEYFILE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFTP_SSL_CCC; // CURLOPT_FTP_SSL_CCC
CF_EXPORT CFURLSessionOption const CFURLSessionOptionTIMEOUT_MS; // CURLOPT_TIMEOUT_MS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCONNECTTIMEOUT_MS; // CURLOPT_CONNECTTIMEOUT_MS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionHTTP_TRANSFER_DECODING; // CURLOPT_HTTP_TRANSFER_DECODING
CF_EXPORT CFURLSessionOption const CFURLSessionOptionHTTP_CONTENT_DECODING; // CURLOPT_HTTP_CONTENT_DECODING
CF_EXPORT CFURLSessionOption const CFURLSessionOptionNEW_FILE_PERMS; // CURLOPT_NEW_FILE_PERMS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionNEW_DIRECTORY_PERMS; // CURLOPT_NEW_DIRECTORY_PERMS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPOSTREDIR; // CURLOPT_POSTREDIR
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSH_HOST_PUBLIC_KEY_MD5; // CURLOPT_SSH_HOST_PUBLIC_KEY_MD5
CF_EXPORT CFURLSessionOption const CFURLSessionOptionOPENSOCKETFUNCTION; // CURLOPT_OPENSOCKETFUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionOPENSOCKETDATA; // CURLOPT_OPENSOCKETDATA
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCOPYPOSTFIELDS; // CURLOPT_COPYPOSTFIELDS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPROXY_TRANSFER_MODE; // CURLOPT_PROXY_TRANSFER_MODE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSEEKFUNCTION; // CURLOPT_SEEKFUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSEEKDATA; // CURLOPT_SEEKDATA
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCRLFILE; // CURLOPT_CRLFILE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionISSUERCERT; // CURLOPT_ISSUERCERT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionADDRESS_SCOPE; // CURLOPT_ADDRESS_SCOPE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCERTINFO; // CURLOPT_CERTINFO
CF_EXPORT CFURLSessionOption const CFURLSessionOptionUSERNAME; // CURLOPT_USERNAME
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPASSWORD; // CURLOPT_PASSWORD
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPROXYUSERNAME; // CURLOPT_PROXYUSERNAME
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPROXYPASSWORD; // CURLOPT_PROXYPASSWORD
CF_EXPORT CFURLSessionOption const CFURLSessionOptionNOPROXY; // CURLOPT_NOPROXY
CF_EXPORT CFURLSessionOption const CFURLSessionOptionTFTP_BLKSIZE; // CURLOPT_TFTP_BLKSIZE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSOCKS5_GSSAPI_SERVICE; // CURLOPT_SOCKS5_GSSAPI_SERVICE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSOCKS5_GSSAPI_NEC; // CURLOPT_SOCKS5_GSSAPI_NEC
CF_EXPORT CFURLSessionOption const CFURLSessionOptionPROTOCOLS; // CURLOPT_PROTOCOLS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionREDIR_PROTOCOLS; // CURLOPT_REDIR_PROTOCOLS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSH_KNOWNHOSTS; // CURLOPT_SSH_KNOWNHOSTS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSH_KEYFUNCTION; // CURLOPT_SSH_KEYFUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSH_KEYDATA; // CURLOPT_SSH_KEYDATA
CF_EXPORT CFURLSessionOption const CFURLSessionOptionMAIL_FROM; // CURLOPT_MAIL_FROM
CF_EXPORT CFURLSessionOption const CFURLSessionOptionMAIL_RCPT; // CURLOPT_MAIL_RCPT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFTP_USE_PRET; // CURLOPT_FTP_USE_PRET
CF_EXPORT CFURLSessionOption const CFURLSessionOptionRTSP_REQUEST; // CURLOPT_RTSP_REQUEST
CF_EXPORT CFURLSessionOption const CFURLSessionOptionRTSP_SESSION_ID; // CURLOPT_RTSP_SESSION_ID
CF_EXPORT CFURLSessionOption const CFURLSessionOptionRTSP_STREAM_URI; // CURLOPT_RTSP_STREAM_URI
CF_EXPORT CFURLSessionOption const CFURLSessionOptionRTSP_TRANSPORT; // CURLOPT_RTSP_TRANSPORT
CF_EXPORT CFURLSessionOption const CFURLSessionOptionRTSP_CLIENT_CSEQ; // CURLOPT_RTSP_CLIENT_CSEQ
CF_EXPORT CFURLSessionOption const CFURLSessionOptionRTSP_SERVER_CSEQ; // CURLOPT_RTSP_SERVER_CSEQ
CF_EXPORT CFURLSessionOption const CFURLSessionOptionINTERLEAVEDATA; // CURLOPT_INTERLEAVEDATA
CF_EXPORT CFURLSessionOption const CFURLSessionOptionINTERLEAVEFUNCTION; // CURLOPT_INTERLEAVEFUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionWILDCARDMATCH; // CURLOPT_WILDCARDMATCH
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCHUNK_BGN_FUNCTION; // CURLOPT_CHUNK_BGN_FUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCHUNK_END_FUNCTION; // CURLOPT_CHUNK_END_FUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFNMATCH_FUNCTION; // CURLOPT_FNMATCH_FUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCHUNK_DATA; // CURLOPT_CHUNK_DATA
CF_EXPORT CFURLSessionOption const CFURLSessionOptionFNMATCH_DATA; // CURLOPT_FNMATCH_DATA
CF_EXPORT CFURLSessionOption const CFURLSessionOptionRESOLVE; // CURLOPT_RESOLVE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionTLSAUTH_USERNAME; // CURLOPT_TLSAUTH_USERNAME
CF_EXPORT CFURLSessionOption const CFURLSessionOptionTLSAUTH_PASSWORD; // CURLOPT_TLSAUTH_PASSWORD
CF_EXPORT CFURLSessionOption const CFURLSessionOptionTLSAUTH_TYPE; // CURLOPT_TLSAUTH_TYPE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionTRANSFER_ENCODING; // CURLOPT_TRANSFER_ENCODING
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCLOSESOCKETFUNCTION; // CURLOPT_CLOSESOCKETFUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionCLOSESOCKETDATA; // CURLOPT_CLOSESOCKETDATA
CF_EXPORT CFURLSessionOption const CFURLSessionOptionGSSAPI_DELEGATION; // CURLOPT_GSSAPI_DELEGATION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionDNS_SERVERS; // CURLOPT_DNS_SERVERS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionACCEPTTIMEOUT_MS; // CURLOPT_ACCEPTTIMEOUT_MS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionTCP_KEEPALIVE; // CURLOPT_TCP_KEEPALIVE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionTCP_KEEPIDLE; // CURLOPT_TCP_KEEPIDLE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionTCP_KEEPINTVL; // CURLOPT_TCP_KEEPINTVL
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSL_OPTIONS; // CURLOPT_SSL_OPTIONS
CF_EXPORT CFURLSessionOption const CFURLSessionOptionMAIL_AUTH; // CURLOPT_MAIL_AUTH
CF_EXPORT CFURLSessionOption const CFURLSessionOptionSASL_IR; // CURLOPT_SASL_IR
CF_EXPORT CFURLSessionOption const CFURLSessionOptionXFERINFOFUNCTION; // CURLOPT_XFERINFOFUNCTION
CF_EXPORT CFURLSessionOption const CFURLSessionOptionXFERINFODATA;
CF_EXPORT CFURLSessionOption const CFURLSessionOptionXOAUTH2_BEARER; // CURLOPT_XOAUTH2_BEARER
CF_EXPORT CFURLSessionOption const CFURLSessionOptionDNS_INTERFACE; // CURLOPT_DNS_INTERFACE
CF_EXPORT CFURLSessionOption const CFURLSessionOptionDNS_LOCAL_IP4; // CURLOPT_DNS_LOCAL_IP4
CF_EXPORT CFURLSessionOption const CFURLSessionOptionDNS_LOCAL_IP6; // CURLOPT_DNS_LOCAL_IP6
CF_EXPORT CFURLSessionOption const CFURLSessionOptionLOGIN_OPTIONS; // CURLOPT_LOGIN_OPTIONS
//CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSL_ENABLE_NPN; // CURLOPT_SSL_ENABLE_NPN
//CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSL_ENABLE_ALPN; // CURLOPT_SSL_ENABLE_ALPN
//CF_EXPORT CFURLSessionOption const CFURLSessionOptionEXPECT_100_TIMEOUT_MS; // CURLOPT_EXPECT_100_TIMEOUT_MS
//CF_EXPORT CFURLSessionOption const CFURLSessionOptionPROXYHEADER; // CURLOPT_PROXYHEADER
//CF_EXPORT CFURLSessionOption const CFURLSessionOptionHEADEROPT; // CURLOPT_HEADEROPT
//CF_EXPORT CFURLSessionOption const CFURLSessionOptionPINNEDPUBLICKEY; // CURLOPT_PINNEDPUBLICKEY
//CF_EXPORT CFURLSessionOption const CFURLSessionOptionUNIX_SOCKET_PATH; // CURLOPT_UNIX_SOCKET_PATH
//CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSL_VERIFYSTATUS; // CURLOPT_SSL_VERIFYSTATUS
//CF_EXPORT CFURLSessionOption const CFURLSessionOptionSSL_FALSESTART; // CURLOPT_SSL_FALSESTART
//CF_EXPORT CFURLSessionOption const CFURLSessionOptionPATH_AS_IS; // CURLOPT_PATH_AS_IS
//CF_EXPORT CFURLSessionOption const CFURLSessionOptionPROXY_SERVICE_NAME; // CURLOPT_PROXY_SERVICE_NAME
//CF_EXPORT CFURLSessionOption const CFURLSessionOptionSERVICE_NAME; // CURLOPT_SERVICE_NAME
//CF_EXPORT CFURLSessionOption const CFURLSessionOptionPIPEWAIT; // CURLOPT_PIPEWAIT


/// This is a mash-up of these two types:
/// curl_infotype & CURLoption
typedef struct CFURLSessionInfo {
    int value;
} CFURLSessionInfo;

CF_EXPORT CFURLSessionInfo const CFURLSessionInfoTEXT; // CURLINFO_TEXT
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoHEADER_IN; // CURLINFO_HEADER_IN
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoHEADER_OUT; // CURLINFO_HEADER_OUT
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoDATA_IN; // CURLINFO_DATA_IN
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoDATA_OUT; // CURLINFO_DATA_OUT
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoSSL_DATA_IN; // CURLINFO_SSL_DATA_IN
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoSSL_DATA_OUT; // CURLINFO_SSL_DATA_OUT
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoEND; // CURLINFO_END
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoNONE; // CURLINFO_NONE
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoEFFECTIVE_URL; // CURLINFO_EFFECTIVE_URL
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoRESPONSE_CODE; // CURLINFO_RESPONSE_CODE
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoTOTAL_TIME; // CURLINFO_TOTAL_TIME
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoNAMELOOKUP_TIME; // CURLINFO_NAMELOOKUP_TIME
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoCONNECT_TIME; // CURLINFO_CONNECT_TIME
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoPRETRANSFER_TIME; // CURLINFO_PRETRANSFER_TIME
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoSIZE_UPLOAD; // CURLINFO_SIZE_UPLOAD
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoSIZE_DOWNLOAD; // CURLINFO_SIZE_DOWNLOAD
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoSPEED_DOWNLOAD; // CURLINFO_SPEED_DOWNLOAD
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoSPEED_UPLOAD; // CURLINFO_SPEED_UPLOAD
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoHEADER_SIZE; // CURLINFO_HEADER_SIZE
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoREQUEST_SIZE; // CURLINFO_REQUEST_SIZE
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoSSL_VERIFYRESULT; // CURLINFO_SSL_VERIFYRESULT
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoFILETIME; // CURLINFO_FILETIME
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoCONTENT_LENGTH_DOWNLOAD; // CURLINFO_CONTENT_LENGTH_DOWNLOAD
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoCONTENT_LENGTH_UPLOAD; // CURLINFO_CONTENT_LENGTH_UPLOAD
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoSTARTTRANSFER_TIME; // CURLINFO_STARTTRANSFER_TIME
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoCONTENT_TYPE; // CURLINFO_CONTENT_TYPE
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoREDIRECT_TIME; // CURLINFO_REDIRECT_TIME
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoREDIRECT_COUNT; // CURLINFO_REDIRECT_COUNT
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoPRIVATE; // CURLINFO_PRIVATE
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoHTTP_CONNECTCODE; // CURLINFO_HTTP_CONNECTCODE
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoHTTPAUTH_AVAIL; // CURLINFO_HTTPAUTH_AVAIL
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoPROXYAUTH_AVAIL; // CURLINFO_PROXYAUTH_AVAIL
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoOS_ERRNO; // CURLINFO_OS_ERRNO
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoNUM_CONNECTS; // CURLINFO_NUM_CONNECTS
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoSSL_ENGINES; // CURLINFO_SSL_ENGINES
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoCOOKIELIST; // CURLINFO_COOKIELIST
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoLASTSOCKET; // CURLINFO_LASTSOCKET
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoFTP_ENTRY_PATH; // CURLINFO_FTP_ENTRY_PATH
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoREDIRECT_URL; // CURLINFO_REDIRECT_URL
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoPRIMARY_IP; // CURLINFO_PRIMARY_IP
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoAPPCONNECT_TIME; // CURLINFO_APPCONNECT_TIME
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoCERTINFO; // CURLINFO_CERTINFO
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoCONDITION_UNMET; // CURLINFO_CONDITION_UNMET
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoRTSP_SESSION_ID; // CURLINFO_RTSP_SESSION_ID
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoRTSP_CLIENT_CSEQ; // CURLINFO_RTSP_CLIENT_CSEQ
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoRTSP_SERVER_CSEQ; // CURLINFO_RTSP_SERVER_CSEQ
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoRTSP_CSEQ_RECV; // CURLINFO_RTSP_CSEQ_RECV
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoPRIMARY_PORT; // CURLINFO_PRIMARY_PORT
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoLOCAL_IP; // CURLINFO_LOCAL_IP
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoLOCAL_PORT; // CURLINFO_LOCAL_PORT
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoTLS_SESSION; // CURLINFO_TLS_SESSION
CF_EXPORT CFURLSessionInfo const CFURLSessionInfoLASTONE; // CURLINFO_LASTONE

typedef struct CFURLSessionMultiOption {
    int value;
} CFURLSessionMultiOption;


/// CURLMoption
CF_EXPORT CFURLSessionMultiOption const CFURLSessionMultiOptionSOCKETFUNCTION; // CURLMOPT_SOCKETFUNCTION
CF_EXPORT CFURLSessionMultiOption const CFURLSessionMultiOptionSOCKETDATA; // CURLMOPT_SOCKETDATA
CF_EXPORT CFURLSessionMultiOption const CFURLSessionMultiOptionPIPELINING; // CURLMOPT_PIPELINING
CF_EXPORT CFURLSessionMultiOption const CFURLSessionMultiOptionTIMERFUNCTION; // CURLMOPT_TIMERFUNCTION
CF_EXPORT CFURLSessionMultiOption const CFURLSessionMultiOptionTIMERDATA; // CURLMOPT_TIMERDATA
CF_EXPORT CFURLSessionMultiOption const CFURLSessionMultiOptionMAXCONNECTS; // CURLMOPT_MAXCONNECTS
CF_EXPORT CFURLSessionMultiOption const CFURLSessionMultiOptionMAX_HOST_CONNECTIONS; // CURLMOPT_MAX_HOST_CONNECTIONS
CF_EXPORT CFURLSessionMultiOption const CFURLSessionMultiOptionMAX_PIPELINE_LENGTH; // CURLMOPT_MAX_PIPELINE_LENGTH
CF_EXPORT CFURLSessionMultiOption const CFURLSessionMultiOptionCONTENT_LENGTH_PENALTY_SIZE; // CURLMOPT_CONTENT_LENGTH_PENALTY_SIZE
CF_EXPORT CFURLSessionMultiOption const CFURLSessionMultiOptionCHUNK_LENGTH_PENALTY_SIZE; // CURLMOPT_CHUNK_LENGTH_PENALTY_SIZE
CF_EXPORT CFURLSessionMultiOption const CFURLSessionMultiOptionPIPELINING_SITE_BL; // CURLMOPT_PIPELINING_SITE_BL
CF_EXPORT CFURLSessionMultiOption const CFURLSessionMultiOptionPIPELINING_SERVER_BL; // CURLMOPT_PIPELINING_SERVER_BL
CF_EXPORT CFURLSessionMultiOption const CFURLSessionMultiOptionMAX_TOTAL_CONNECTIONS; // CURLMOPT_MAX_TOTAL_CONNECTIONS



typedef struct CFURLSessionMultiCode {
    int value;
} CFURLSessionMultiCode;


/// CURLMcode
CF_EXPORT CFURLSessionMultiCode const CFURLSessionMultiCodeCALL_MULTI_PERFORM; // CURLM_CALL_MULTI_PERFORM
CF_EXPORT CFURLSessionMultiCode const CFURLSessionMultiCodeOK; // CURLM_OK
CF_EXPORT CFURLSessionMultiCode const CFURLSessionMultiCodeBAD_HANDLE; // CURLM_BAD_HANDLE
CF_EXPORT CFURLSessionMultiCode const CFURLSessionMultiCodeBAD_EASY_HANDLE; // CURLM_BAD_EASY_HANDLE
CF_EXPORT CFURLSessionMultiCode const CFURLSessionMultiCodeOUT_OF_MEMORY; // CURLM_OUT_OF_MEMORY
CF_EXPORT CFURLSessionMultiCode const CFURLSessionMultiCodeINTERNAL_ERROR; // CURLM_INTERNAL_ERROR
CF_EXPORT CFURLSessionMultiCode const CFURLSessionMultiCodeBAD_SOCKET; // CURLM_BAD_SOCKET
CF_EXPORT CFURLSessionMultiCode const CFURLSessionMultiCodeUNKNOWN_OPTION; // CURLM_UNKNOWN_OPTION
CF_EXPORT CFURLSessionMultiCode const CFURLSessionMultiCodeADDED_ALREADY; // CURLM_ADDED_ALREADY
CF_EXPORT CFURLSessionMultiCode const CFURLSessionMultiCodeLAST; // CURLM_LAST


typedef struct CFURLSessionPoll {
    int value;
} CFURLSessionPoll;
CF_EXPORT CFURLSessionPoll const CFURLSessionPollNone; // CURL_POLL_NONE
CF_EXPORT CFURLSessionPoll const CFURLSessionPollIn; // CURL_POLL_IN
CF_EXPORT CFURLSessionPoll const CFURLSessionPollOut; // CURL_POLL_OUT
CF_EXPORT CFURLSessionPoll const CFURLSessionPollInOut; // CURL_POLL_INOUT
CF_EXPORT CFURLSessionPoll const CFURLSessionPollRemove; // CURL_POLL_REMOVE


typedef long CFURLSessionProtocol;


CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolHTTP; // CURLPROTO_HTTP
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolHTTPS; // CURLPROTO_HTTPS
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolFTP; // CURLPROTO_FTP
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolFTPS; // CURLPROTO_FTPS
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolSCP; // CURLPROTO_SCP
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolSFTP; // CURLPROTO_SFTP
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolTELNET; // CURLPROTO_TELNET
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolLDAP; // CURLPROTO_LDAP
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolLDAPS; // CURLPROTO_LDAPS
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolDICT; // CURLPROTO_DICT
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolFILE; // CURLPROTO_FILE
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolTFTP; // CURLPROTO_TFTP
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolIMAP; // CURLPROTO_IMAP
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolIMAPS; // CURLPROTO_IMAPS
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolPOP3; // CURLPROTO_POP3
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolPOP3S; // CURLPROTO_POP3S
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolSMTP; // CURLPROTO_SMTP
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolSMTPS; // CURLPROTO_SMTPS
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolRTSP; // CURLPROTO_RTSP
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolRTMP; // CURLPROTO_RTMP
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolRTMPT; // CURLPROTO_RTMPT
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolRTMPE; // CURLPROTO_RTMPE
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolRTMPTE; // CURLPROTO_RTMPTE
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolRTMPS; // CURLPROTO_RTMPS
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolRTMPTS; // CURLPROTO_RTMPTS
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolGOPHER; // CURLPROTO_GOPHER
//CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolSMB; // CURLPROTO_SMB
//CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolSMBS; // CURLPROTO_SMBS
CF_EXPORT CFURLSessionProtocol const CFURLSessionProtocolALL; // CURLPROTO_ALL


// The following WebSockets symbols are -1 on libcurl < 7.86.0, or when WebSockets are disabled

typedef unsigned int CFURLSessionWebSocketsMessageFlag;

CF_EXPORT CFURLSessionWebSocketsMessageFlag const CFURLSessionWebSocketsText; // CURLWS_TEXT
CF_EXPORT CFURLSessionWebSocketsMessageFlag const CFURLSessionWebSocketsBinary; // CURLWS_BINARY
CF_EXPORT CFURLSessionWebSocketsMessageFlag const CFURLSessionWebSocketsCont; // CURLWS_CONT
CF_EXPORT CFURLSessionWebSocketsMessageFlag const CFURLSessionWebSocketsClose; // CURLWS_CLOSE
CF_EXPORT CFURLSessionWebSocketsMessageFlag const CFURLSessionWebSocketsPing; // CURLWS_PING
CF_EXPORT CFURLSessionWebSocketsMessageFlag const CFURLSessionWebSocketsPong; // CURLWS_PONG

CF_EXPORT CFURLSessionOption const CFURLSessionWebSocketsRawMode; // CURLWS_RAW_MODE

// The following WebSockets functions are functional with libcurl 7.86.0 or later, when WebSockets support is enabled.  On libcurl versions without WebSockets support, they'll trap on use.  Consult CFURLSessionWebSocketsSupported() to get a runtime signal whether they're functional.
CF_EXPORT Boolean CFURLSessionWebSocketsSupported(void);

typedef struct CFURLSessionWebSocketsFrame {
    int age; /* always zero */
    CFURLSessionWebSocketsMessageFlag flags;
    long long offset;
    long long bytesLeft;
} CFURLSessionWebSocketsFrame;

CF_EXPORT CFURLSessionEasyCode CFURLSessionEasyHandleWebSocketsReceive(CFURLSessionEasyHandle _Nonnull handle, char *_Nonnull data, size_t dataLen, size_t * _Nonnull receivedDataLen, CFURLSessionWebSocketsFrame * _Nullable receivedFrame);
CF_EXPORT CFURLSessionEasyCode CFURLSessionEasyHandleWebSocketsSend(CFURLSessionEasyHandle _Nonnull handle, const char *_Nonnull data, size_t dataLen, size_t * _Nonnull writtenDataLen, long long frameSize, CFURLSessionWebSocketsMessageFlag messageFlags);

CF_EXPORT CFURLSessionWebSocketsFrame * _Nonnull CFURLSessionEasyHandleWebSocketsMetadata(CFURLSessionEasyHandle _Nonnull handle);


CF_EXPORT size_t const CFURLSessionMaxWriteSize; // CURL_MAX_WRITE_SIZE

CF_EXPORT char * _Nonnull CFURLSessionCurlVersionString(void);
typedef struct CFURLSessionCurlVersion {
    int major;
    int minor;
    int patch;
} CFURLSessionCurlVersion;
CF_EXPORT CFURLSessionCurlVersion CFURLSessionCurlVersionInfo(void);


CF_EXPORT int const CFURLSessionWriteFuncPause;
CF_EXPORT int const CFURLSessionReadFuncPause;
CF_EXPORT int const CFURLSessionReadFuncAbort;

CF_EXPORT CFURLSession_socket_t const CFURLSessionSocketTimeout;

CF_EXPORT int const CFURLSessionSeekOk;
CF_EXPORT int const CFURLSessionSeekCantSeek;
CF_EXPORT int const CFURLSessionSeekFail;

CF_EXPORT CFURLSessionEasyHandle _Nonnull CFURLSessionEasyHandleInit(void);
CF_EXPORT void CFURLSessionEasyHandleDeinit(CFURLSessionEasyHandle _Nonnull handle);
CF_EXPORT CFURLSessionEasyCode CFURLSessionEasyHandleSetPauseState(CFURLSessionEasyHandle _Nonnull handle, int send, int receive);

CF_EXPORT CFURLSessionMultiHandle _Nonnull CFURLSessionMultiHandleInit(void);
CF_EXPORT CFURLSessionMultiCode CFURLSessionMultiHandleDeinit(CFURLSessionMultiHandle _Nonnull handle);
CF_EXPORT CFURLSessionMultiCode CFURLSessionMultiHandleAddHandle(CFURLSessionMultiHandle _Nonnull handle, CFURLSessionEasyHandle _Nonnull curl);
CF_EXPORT CFURLSessionMultiCode CFURLSessionMultiHandleRemoveHandle(CFURLSessionMultiHandle _Nonnull handle, CFURLSessionEasyHandle _Nonnull curl);
CF_EXPORT CFURLSessionMultiCode CFURLSessionMultiHandleAssign(CFURLSessionMultiHandle _Nonnull handle, CFURLSession_socket_t socket, void * _Nullable sockp);
CF_EXPORT CFURLSessionMultiCode CFURLSessionMultiHandleAction(CFURLSessionMultiHandle _Nonnull handle, CFURLSession_socket_t socket, int bitmask, int * _Nonnull running_handles);
typedef struct CFURLSessionMultiHandleInfo {
    CFURLSessionEasyHandle _Nullable easyHandle;
    CFURLSessionEasyCode resultCode;
} CFURLSessionMultiHandleInfo;
CF_EXPORT CFURLSessionMultiHandleInfo CFURLSessionMultiHandleInfoRead(CFURLSessionMultiHandle _Nonnull handle, int * _Nonnull msgs_in_queue);

CF_EXPORT CFURLSessionEasyCode CFURLSession_easy_setopt_fptr(CFURLSessionEasyHandle _Nonnull curl, CFURLSessionOption option, void *_Nullable a);
CF_EXPORT CFURLSessionEasyCode CFURLSession_easy_setopt_ptr(CFURLSessionEasyHandle _Nonnull curl, CFURLSessionOption option, void *_Nullable a);
CF_EXPORT CFURLSessionEasyCode CFURLSession_easy_setopt_int(CFURLSessionEasyHandle _Nonnull curl, CFURLSessionOption option, int a);
CF_EXPORT CFURLSessionEasyCode CFURLSession_easy_setopt_long(CFURLSessionEasyHandle _Nonnull curl, CFURLSessionOption option, long a);
CF_EXPORT CFURLSessionEasyCode CFURLSession_easy_setopt_int64(CFURLSessionEasyHandle _Nonnull curl, CFURLSessionOption option, long long a);
CF_EXPORT CFURLSessionEasyCode CFURLSession_easy_setopt_wc(CFURLSessionEasyHandle _Nonnull curl, CFURLSessionOption option, size_t(*_Nonnull a)(char *_Nonnull, size_t, size_t, void *_Nullable));
CF_EXPORT CFURLSessionEasyCode CFURLSession_easy_setopt_fwc(CFURLSessionEasyHandle _Nonnull curl, CFURLSessionOption option, size_t(*_Nonnull a)(char *_Nonnull, size_t, size_t, void *_Nullable));
CF_EXPORT CFURLSessionEasyCode CFURLSession_easy_setopt_dc(CFURLSessionEasyHandle _Nonnull curl, CFURLSessionOption option, int(*_Nonnull a)(CFURLSessionEasyHandle _Nonnull handle, int type, char *_Nonnull data, size_t size, void *_Nullable userptr));
typedef enum {
    CFURLSessionSocketTypeIPCXN,  // socket created for a specific IP connection
    CFURLSessionSocketTypeAccept, // socket created by accept() call
} CFURLSessionSocketType;
typedef int (CFURLSessionSocketOptionCallback)(void *_Nullable clientp, int fd, CFURLSessionSocketType purpose);
CF_EXPORT CFURLSessionEasyCode CFURLSession_easy_setopt_sc(CFURLSessionEasyHandle _Nonnull curl, CFURLSessionOption option, CFURLSessionSocketOptionCallback * _Nullable a);
typedef int (CFURLSessionSeekCallback)(void *_Nullable userp, long long offset, int origin);
CF_EXPORT CFURLSessionEasyCode CFURLSession_easy_setopt_seek(CFURLSessionEasyHandle _Nonnull curl, CFURLSessionOption option, CFURLSessionSeekCallback * _Nullable a);
typedef int (CFURLSessionTransferInfoCallback)(void *_Nullable userp, long long dltotal, long long dlnow, long long ultotal, long long ulnow);
CF_EXPORT CFURLSessionEasyCode CFURLSession_easy_setopt_tc(CFURLSessionEasyHandle _Nonnull curl, CFURLSessionOption option, CFURLSessionTransferInfoCallback * _Nullable a);

CF_EXPORT CFURLSessionEasyCode CFURLSession_easy_getinfo_long(CFURLSessionEasyHandle _Nonnull curl, CFURLSessionInfo info, long *_Nonnull a);
CF_EXPORT CFURLSessionEasyCode CFURLSession_easy_getinfo_double(CFURLSessionEasyHandle _Nonnull curl, CFURLSessionInfo info, double *_Nonnull a);
CF_EXPORT CFURLSessionEasyCode CFURLSession_easy_getinfo_charp(CFURLSessionEasyHandle _Nonnull curl, CFURLSessionInfo info, char *_Nullable*_Nonnull a);

CF_EXPORT CFURLSessionMultiCode CFURLSession_multi_setopt_ptr(CFURLSessionMultiHandle _Nonnull multi_handle, CFURLSessionMultiOption option, void *_Nullable a);
CF_EXPORT CFURLSessionMultiCode CFURLSession_multi_setopt_l(CFURLSessionMultiHandle _Nonnull multi_handle, CFURLSessionMultiOption option, long a);
CF_EXPORT CFURLSessionMultiCode CFURLSession_multi_setopt_sf(CFURLSessionMultiHandle _Nonnull multi_handle, CFURLSessionMultiOption option, int (*_Nonnull a)(CFURLSessionEasyHandle _Nonnull, CFURLSession_socket_t, int, void *_Nullable, void *_Nullable));
CF_EXPORT CFURLSessionMultiCode CFURLSession_multi_setopt_tf(CFURLSessionMultiHandle _Nonnull multi_handle, CFURLSessionMultiOption option, int (*_Nonnull a)(CFURLSessionMultiHandle _Nonnull, long, void *_Nullable));

CF_EXPORT CFURLSessionEasyCode CFURLSessionInit(void);


typedef struct CFURLSessionSList CFURLSessionSList;
CF_EXPORT CFURLSessionSList *_Nullable CFURLSessionSListAppend(CFURLSessionSList *_Nullable list, const char * _Nullable string);
CF_EXPORT void CFURLSessionSListFreeAll(CFURLSessionSList *_Nullable list);



CF_EXTERN_C_END
CF_IMPLICIT_BRIDGING_DISABLED

#endif /* __COREFOUNDATION_URLSESSIONINTERFACE__ */
