/*
 This source file is part of the Swift.org open source project

 Copyright (c) 2021 Apple Inc. and the Swift project authors
 Licensed under Apache License v2.0 with Runtime Library Exception

 See https://swift.org/LICENSE.txt for license information
 See https://swift.org/CONTRIBUTORS.txt for Swift project authors
*/

import SwiftDocC
import Foundation

/// The result of executing an action.
public struct ActionResult {
    /// Problems found while trying to perform the action.
    public var problems: [Problem]
    
    /// Outputs generated by the action, if any.
    public var outputs: [URL]

    /// If an error was encountered during the action.
    public var didEncounterError: Bool
    
    /// Creates a new action results with a flag indicating if an error occured during the action and any URLs
    /// the action generated.
    ///
    /// - Parameters:
    ///   - didEncounterError: A Boolean value indicating if an error occurred during the action.
    ///   - outputs: The URLs generated by the action this result represents.
    public init(didEncounterError: Bool = false, outputs: [URL] = []) {
        self.problems = []
        self.outputs = outputs
        self.didEncounterError = didEncounterError
    }
}

extension Problem {
    /// Creates a new problem with the given description and documentation source location.
    ///
    /// Use this to provide a user-friendly description of an error,
    /// along with a direct reference to the source file and line number where you call this initializer.
    ///
    /// - Parameters:
    ///   - description: A brief description of the problem.
    ///   - source: The URL for the documentation file that caused this problem, if any.
    ///   - file: The source file where you call this initializer.
    public init(description: String, source: URL?, file: String = #file) {
        let fileName = URL(fileURLWithPath: file).deletingPathExtension().lastPathComponent

        let singleDiagnostic = Diagnostic(source: source, severity: .error, range: nil, identifier: "org.swift.docc.\(fileName)", summary: description)
        self.init(diagnostic: singleDiagnostic, possibleSolutions: [])
    }
}
