//===----------------------------------------------------------------------===//
//
// This source file is part of the Swift open source project
//
// Copyright (c) 2021 Apple Inc. and the Swift project authors
// Licensed under Apache License v2.0 with Runtime Library Exception
//
// See http://swift.org/LICENSE.txt for license information
// See http://swift.org/CONTRIBUTORS.txt for the list of Swift project authors
//
//===----------------------------------------------------------------------===//

import struct Foundation.URL

import PackageCollectionsSigning
import TSCBasic

/// Configuration in this file is intended for package collection sources to define certificate policies
/// that are more restrictive. For example, a source may want to require that all their package
/// collections be signed using certificate that belongs to certain subject user ID.
internal struct PackageCollectionSourceCertificatePolicy {
    private static let defaultSourceCertPolicies: [String: [CertificatePolicyConfig]] = [
        "developer.apple.com": [
            CertificatePolicyConfig(
                certPolicyKey: .appleSwiftPackageCollection(subjectUserID: "XLVRDL8TZV"),
                base64EncodedRootCerts: ["MIIEuzCCA6OgAwIBAgIBAjANBgkqhkiG9w0BAQUFADBiMQswCQYDVQQGEwJVUzETMBEGA1UEChMKQXBwbGUgSW5jLjEmMCQGA1UECxMdQXBwbGUgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkxFjAUBgNVBAMTDUFwcGxlIFJvb3QgQ0EwHhcNMDYwNDI1MjE0MDM2WhcNMzUwMjA5MjE0MDM2WjBiMQswCQYDVQQGEwJVUzETMBEGA1UEChMKQXBwbGUgSW5jLjEmMCQGA1UECxMdQXBwbGUgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkxFjAUBgNVBAMTDUFwcGxlIFJvb3QgQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDkkakJH5HbHkdQ6wXtXnmELes2oldMVeyLGYne+Uts9QerIjAC6Bg++FAJ039BqJj50cpmnCRrEdCju+QbKsMflZ56DKRHi1vUFjczy8QPTc4UadHJGXL1XQ7Vf1+b8iUDulWPTV0N8WQ1IxVLFVkds5T39pyez1C6wVhQZ48ItCD3y6wsIG9wtj8BMIy3Q88PnT3zK0koGsj+zrW5DtleHNbLPbU6rfQPDgCSC7EhFi501TwN22IWq6NxkkdTVcGvL0Gz+PvjcM3mo0xFfh9Ma1CWQYnEdGILEINBhzOKgbEwWOxaBDKMaLOPHd5lc/9nXmW8Sdh2nzMUZaF3lMktAgMBAAGjggF6MIIBdjAOBgNVHQ8BAf8EBAMCAQYwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQUK9BpR5R2Cf70a40uQKb3R01/CF4wHwYDVR0jBBgwFoAUK9BpR5R2Cf70a40uQKb3R01/CF4wggERBgNVHSAEggEIMIIBBDCCAQAGCSqGSIb3Y2QFATCB8jAqBggrBgEFBQcCARYeaHR0cHM6Ly93d3cuYXBwbGUuY29tL2FwcGxlY2EvMIHDBggrBgEFBQcCAjCBthqBs1JlbGlhbmNlIG9uIHRoaXMgY2VydGlmaWNhdGUgYnkgYW55IHBhcnR5IGFzc3VtZXMgYWNjZXB0YW5jZSBvZiB0aGUgdGhlbiBhcHBsaWNhYmxlIHN0YW5kYXJkIHRlcm1zIGFuZCBjb25kaXRpb25zIG9mIHVzZSwgY2VydGlmaWNhdGUgcG9saWN5IGFuZCBjZXJ0aWZpY2F0aW9uIHByYWN0aWNlIHN0YXRlbWVudHMuMA0GCSqGSIb3DQEBBQUAA4IBAQBcNplMLXi37Yyb3PN3m/J20ncwT8EfhYOFG5k9RzfyqZtAjizUsZAS2L70c5vu0mQPy3lPNNiiPvl4/2vIB+x9OYOLUyDTOMSxv5pPCmv/K/xZpwUJfBdAVhEedNO3iyM7R6PVbyTi69G3cN8PReEnyvFteO3ntRcXqNx+IjXKJdXZD9Zr1KIkIxH3oayPc4FgxhtbCS+SsvhESPBgOJ4V9T0mZyCKM2r3DYLP3uujL/lTaltkwGMzd/c6ByxW69oPIQ7aunMZT7XZNn/Bh1XZp5m5MkL72NVxnn6hUrcbvZNCJBIqxw8dtk2cXmPIS4AXUKqK1drk/NAJBzewdXUh"]
            ),
        ],
    ]

    private let sourceCertPolicies: [String: [CertificatePolicyConfig]]

    var allRootCerts: Set<String>? {
        let allRootCerts = self.sourceCertPolicies.values
            .flatMap { configs in configs.compactMap { $0.base64EncodedRootCerts } }
            .flatMap { $0 }
        return allRootCerts.isEmpty ? nil : Set(allRootCerts)
    }

    init(sourceCertPolicies: [String: [CertificatePolicyConfig]]? = nil) {
        guard sourceCertPolicies?.values.first(where: { $0.isEmpty }) == nil else {
            preconditionFailure("CertificatePolicyConfig array must not be empty")
        }
        self.sourceCertPolicies = sourceCertPolicies ?? Self.defaultSourceCertPolicies
    }

    func mustBeSigned(source: Model.CollectionSource) -> Bool {
        source.certPolicyConfigKey.map { self.sourceCertPolicies[$0] != nil } ?? false
    }

    func certificatePolicyKeys(for source: Model.CollectionSource) -> [CertificatePolicyKey]? {
        // Certificate policy is associated to a collection host
        source.certPolicyConfigKey.flatMap { self.sourceCertPolicies[$0]?.map { $0.certPolicyKey } }
    }

    struct CertificatePolicyConfig {
        let certPolicyKey: CertificatePolicyKey

        /// Root CAs of the signing certificates. Each item is the base64-encoded string
        /// of the DER representation of a root CA.
        let base64EncodedRootCerts: [String]?
    }
}

private extension Model.CollectionSource {
    var certPolicyConfigKey: String? {
        self.url.host
    }
}
