#! /bin/sh
# check script for Tarlz - Archiver with multimember lzip compression
# Copyright (C) 2013-2019 Antonio Diaz Diaz.
#
# This script is free software: you have unlimited permission
# to copy, distribute and modify it.

LC_ALL=C
export LC_ALL
objdir=`pwd`
testdir=`cd "$1" ; pwd`
TARLZ="${objdir}"/tarlz
framework_failure() { echo "failure in testing framework" ; exit 1 ; }

if [ ! -f "${TARLZ}" ] || [ ! -x "${TARLZ}" ] ; then
	echo "${TARLZ}: cannot execute"
	exit 1
fi

[ -e "${TARLZ}" ] 2> /dev/null ||
	{
	echo "$0: a POSIX shell is required to run the tests"
	echo "Try bash -c \"$0 $1 $2\""
	exit 1
	}

if [ -d tmp ] ; then rm -rf tmp ; fi
mkdir tmp
cd "${objdir}"/tmp || framework_failure

in="${testdir}"/test.txt
in_lz="${testdir}"/test.txt.lz
in_tar="${testdir}"/test.txt.tar
in_tar_lz="${testdir}"/test.txt.tar.lz
inbad1="${testdir}"/test_bad1.txt
inbad2="${testdir}"/test_bad2.txt
test3="${testdir}"/test3.tar
test3_lz="${testdir}"/test3.tar.lz
test3dir_lz="${testdir}"/test3_dir.tar.lz
test3dot_lz="${testdir}"/test3_dot.tar.lz
tarint1_lz="${testdir}"/tar_in_tlz1.tar.lz
tarint2_lz="${testdir}"/tar_in_tlz2.tar.lz
t155="${testdir}"/t155.tar
t155_lz="${testdir}"/t155.tar.lz
tlzit1="${testdir}"/tlz_in_tar1.tar
tlzit2="${testdir}"/tlz_in_tar2.tar
bad1="${testdir}"/test3_bad1.tar
bad2="${testdir}"/test3_bad2.tar
bad3="${testdir}"/test3_bad3.tar
bad4="${testdir}"/test3_bad4.tar
bad5="${testdir}"/test3_bad5.tar
bad1_lz="${testdir}"/test3_bad1.tar.lz
bad2_lz="${testdir}"/test3_bad2.tar.lz
bad3_lz="${testdir}"/test3_bad3.tar.lz
bad4_lz="${testdir}"/test3_bad4.tar.lz
bad5_lz="${testdir}"/test3_bad5.tar.lz
bad6_lz="${testdir}"/test3_bad6.tar.lz
eof_lz="${testdir}"/eof.tar.lz
fail=0
lwarn=0
test_failed() { fail=1 ; printf " $1" ; [ -z "$2" ] || printf "($2)" ; }
lzlib_1_11() { [ ${lwarn} = 0 ] &&
	printf "\nwarning: testing --keep-damaged requires lzlib-1.11 or newer\n$1"
	lwarn=1 ; }

# Description of test files for tarlz:
# test.txt.tar.lz:   1 member (test.txt).
# t155.tar[.lz]:     directory + links + file + eof, all with 155 char names
# tar_in_tlz1.tar.lz 2 members (test.txt.tar test3.tar) 3 lzip members
# tar_in_tlz2.tar.lz 2 members (test.txt.tar test3.tar) 5 lzip members
# test_bad1.tar.lz:  truncated at offset 6000 (of 7495)
# test_bad2.tar.lz:  byte at offset 6000 changed from 0x56 to 0x46
# test3.tar:         3 members (foo bar baz) + 2 zeroed 512-byte blocks
# test3_dir.tar.lz:  like test3.tar.lz but members /dir/foo /dir/bar /dir/baz
# test3_dot.tar.lz:  3 times 3 members ./foo ././bar ./././baz
#                    the 3 central members with filename in extended header
# test3_bad1.tar:    byte at offset  259 changed from 't' to '0' (magic)
# test3_bad2.tar:    byte at offset 1283 changed from 't' to '0' (magic)
# test3_bad3.tar:    byte at offset 2559 changed from 0x00 to 0x20 (padding)
# test3_bad4.tar:    byte at offset 1283 changed from 't' to '0' (magic)
#                    byte at offset 2307 changed from 't' to '0' (magic)
# test3_bad5.tar:    510 zeros + "LZ" prepended to test3.tar (bogus lz header)
# test3_bad1.tar.lz: byte at offset    2 changed from 'I' to 'i' (magic)
# test3_bad2.tar.lz: byte at offset   49 changed from 0x49 to 0x69 (mid stream)
# test3_bad3.tar.lz: byte at offset  176 changed from 0x7D to 0x6D (mid stream)
# test3_bad4.tar.lz: combined damage of test3_bad2.tar.lz and test3_bad3.tar.lz
# test3_bad5.tar.lz: [71-134] --> zeroed (first trailer + seconf header)
# test3_bad6.tar.lz: 510 zeros prepended to test3.tar.lz (header in two blocks)
# test3_eof1.tar.lz: test3.tar.lz without eof blocks
# test3_eof2.tar.lz: test3.tar.lz with only one eof block
# test3_eof3.tar.lz: test3.tar.lz with one zeroed block between foo and bar
# tlz_in_tar1.tar:   1 member (test3.tar.lz) first magic damaged
# tlz_in_tar2.tar:   2 members (foo test3.tar.lz) first magic damaged
# ug32chars.tar.lz:  1 member (foo) with 32-character owner and group names

printf "testing tarlz-%s..." "$2"

"${TARLZ}" -q -tf "${in}"
[ $? = 2 ] || test_failed $LINENO
"${TARLZ}" -q -tf "${in_lz}"
[ $? = 2 ] || test_failed $LINENO
"${TARLZ}" -q -tf nx_file
[ $? = 1 ] || test_failed $LINENO
"${TARLZ}" -tf 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${TARLZ}" -q -cf out.tar.lz
[ $? = 1 ] || test_failed $LINENO
[ ! -e out.tar.lz ] || test_failed $LINENO
"${TARLZ}" -rf out.tar.lz || test_failed $LINENO
[ ! -e out.tar.lz ] || test_failed $LINENO
"${TARLZ}" -q -rf - "${in}"
[ $? = 1 ] || test_failed $LINENO
[ ! -e - ] || test_failed $LINENO
"${TARLZ}" -q -r "${in}"
[ $? = 1 ] || test_failed $LINENO
"${TARLZ}" --uncompressed -q -rf out.tar "${in}"
[ $? = 1 ] || test_failed $LINENO
[ ! -e out.tar ] || test_failed $LINENO
cat "${test3_lz}" > test.tar.lz || framework_failure
"${TARLZ}" --uncompressed -q -rf test.tar.lz "${in}"
[ $? = 1 ] || test_failed $LINENO
cmp "${test3_lz}" test.tar.lz || test_failed $LINENO
rm -f test.tar.lz || framework_failure
cat "${test3}" > test.tar || framework_failure
"${TARLZ}" -q -rf test.tar "${in}"
[ $? = 2 ] || test_failed $LINENO
cmp "${test3}" test.tar || test_failed $LINENO
rm -f test.tar || framework_failure
"${TARLZ}" -q -c "${in}" nx_file > /dev/null
[ $? = 1 ] || test_failed $LINENO
"${TARLZ}" -q -c -C nx_dir "${in}"
[ $? = 1 ] || test_failed $LINENO
"${TARLZ}" -q -x -C nx_dir "${test3_lz}"
[ $? = 1 ] || test_failed $LINENO
touch empty.tar.lz empty.tlz			# list an empty lz file
"${TARLZ}" -q -tf empty.tar.lz
[ $? = 2 ] || test_failed $LINENO
"${TARLZ}" -q -tf empty.tlz
[ $? = 2 ] || test_failed $LINENO
rm -f empty.tar.lz empty.tlz || framework_failure
"${TARLZ}" -q -cr
[ $? = 1 ] || test_failed $LINENO
"${TARLZ}" -q -ct
[ $? = 1 ] || test_failed $LINENO
"${TARLZ}" -q -cx
[ $? = 1 ] || test_failed $LINENO
"${TARLZ}" -q -tx
[ $? = 1 ] || test_failed $LINENO
"${TARLZ}" -q -ctx
[ $? = 1 ] || test_failed $LINENO
"${TARLZ}" --help > /dev/null || test_failed $LINENO
"${TARLZ}" -V > /dev/null || test_failed $LINENO
"${TARLZ}" --bad_option -tf "${test3_lz}" 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${TARLZ}" -tf 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${TARLZ}" --owner=invalid_oner_name -tf "${test3_lz}" 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${TARLZ}" --group=invalid_goup_name -tf "${test3_lz}" 2> /dev/null
[ $? = 1 ] || test_failed $LINENO

# test --list and --extract
"${TARLZ}" -tf "${eof_lz}" --missing-crc || test_failed $LINENO
"${TARLZ}" -xf "${eof_lz}" --missing-crc || test_failed $LINENO
"${TARLZ}" -tf "${in_tar_lz}" --missing-crc > /dev/null || test_failed $LINENO
"${TARLZ}" -xf "${in_tar_lz}" --missing-crc || test_failed $LINENO
cmp "${in}" test.txt || test_failed $LINENO
rm -f test.txt || framework_failure
"${TARLZ}" -C nx_dir -tf "${in_tar}" > /dev/null || test_failed $LINENO
"${TARLZ}" -xf "${in_tar}" --missing-crc || test_failed $LINENO
cmp "${in}" test.txt || test_failed $LINENO
rm -f test.txt || framework_failure

# reference files for cmp
cat "${testdir}"/rfoo > cfoo || framework_failure
cat "${testdir}"/rbar > cbar || framework_failure
cat "${testdir}"/rbaz > cbaz || framework_failure
rm -f foo bar baz || framework_failure
"${TARLZ}" -xf "${test3_lz}" --missing-crc || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -tf "${test3_lz}" ./foo ./bar ./baz || test_failed $LINENO
"${TARLZ}" -q -xf "${test3_lz}" ./foo ./bar ./baz || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -xf "${test3_lz}" foo/ bar// baz/// || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -xf "${test3}" --missing-crc || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${test3dot_lz}" --missing-crc || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -tf "${test3dot_lz}" foo bar baz || test_failed $LINENO
"${TARLZ}" -q -xf "${test3dot_lz}" foo bar baz || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${test3dir_lz}" --missing-crc || test_failed $LINENO
cmp cfoo dir/foo || test_failed $LINENO
cmp cbar dir/bar || test_failed $LINENO
cmp cbaz dir/baz || test_failed $LINENO
rm -rf dir || framework_failure
"${TARLZ}" -q -tf "${test3dir_lz}" dir/foo dir/bar dir/baz || test_failed $LINENO
"${TARLZ}" -q -xf "${test3dir_lz}" dir/foo dir/bar dir/baz || test_failed $LINENO
cmp cfoo dir/foo || test_failed $LINENO
cmp cbar dir/bar || test_failed $LINENO
cmp cbaz dir/baz || test_failed $LINENO
rm -rf dir || framework_failure
#
"${TARLZ}" -q -tf "${testdir}"/test3_eof1.tar.lz
[ $? = 2 ] || test_failed $LINENO
"${TARLZ}" -q -tf "${testdir}"/test3_eof2.tar.lz || test_failed $LINENO
"${TARLZ}" -q -tf "${testdir}"/test3_eof3.tar.lz || test_failed $LINENO
"${TARLZ}" -q -n0 -tf "${testdir}"/test3_eof1.tar.lz
[ $? = 2 ] || test_failed $LINENO
"${TARLZ}" -q -n0 -tf "${testdir}"/test3_eof2.tar.lz || test_failed $LINENO
"${TARLZ}" -q -n0 -tf "${testdir}"/test3_eof3.tar.lz || test_failed $LINENO
#
"${TARLZ}" -q -xf "${testdir}"/test3_eof1.tar.lz
[ $? = 2 ] || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -xf "${testdir}"/test3_eof2.tar.lz || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -xf "${testdir}"/test3_eof3.tar.lz || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
[ ! -e bar ] || test_failed $LINENO
[ ! -e baz ] || test_failed $LINENO
rm -f foo bar baz || framework_failure
#
"${TARLZ}" -q -n0 -xf "${testdir}"/test3_eof1.tar.lz
[ $? = 2 ] || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -n0 -xf "${testdir}"/test3_eof2.tar.lz || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -n0 -xf "${testdir}"/test3_eof3.tar.lz || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
[ ! -e bar ] || test_failed $LINENO
[ ! -e baz ] || test_failed $LINENO
rm -f foo bar baz || framework_failure
#
for i in "${tarint1_lz}" "${tarint2_lz}" ; do
	for j in 0 2 6 ; do
		"${TARLZ}" -tf "$i" --threads=$j > out$j ||
			test_failed $LINENO "$i $j"
		"${TARLZ}" -tvf "$i" --threads=$j > outv$j ||
			test_failed $LINENO "$i $j"
	done
	cmp out0 out2 || test_failed $LINENO
	cmp out0 out6 || test_failed $LINENO
	cmp out2 out6 || test_failed $LINENO
	cmp outv0 outv2 || test_failed $LINENO
	cmp outv0 outv6 || test_failed $LINENO
	cmp outv2 outv6 || test_failed $LINENO
	rm -f out0 out2 out6 outv0 outv2 outv6 || framework_failure
	"${TARLZ}" -xf "$i" || test_failed $LINENO
	cmp "${in_tar}" test.txt.tar || test_failed $LINENO
	cmp "${test3}" test3.tar || test_failed $LINENO
	rm -f test.txt.tar test3.tar || framework_failure
done

# test --concatenate
cat "${in_tar_lz}" > out.tar.lz || framework_failure
"${TARLZ}" -Af out.tar.lz "${test3_lz}" || test_failed $LINENO
"${TARLZ}" -xf out.tar.lz || test_failed $LINENO
cmp "${in}" test.txt || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
touch aout.tar.lz || framework_failure		# concatenate to empty file
"${TARLZ}" -Af aout.tar.lz "${in_tar_lz}" "${test3_lz}" || test_failed $LINENO
cmp out.tar.lz aout.tar.lz || test_failed $LINENO
cat "${in_tar_lz}" > aout.tar.lz || framework_failure
"${TARLZ}" -Aqf aout.tar.lz "${test3_lz}" "${test3}"
[ $? = 2 ] || test_failed $LINENO
cmp out.tar.lz aout.tar.lz || test_failed $LINENO
rm -f test.txt foo bar baz out.tar.lz aout.tar.lz || framework_failure

# test --create
cat "${in}" > test.txt || framework_failure
"${TARLZ}" -0 -cf out.tar.lz test.txt || test_failed $LINENO
rm -f test.txt || framework_failure
"${TARLZ}" -xf out.tar.lz --missing-crc || test_failed $LINENO
cmp "${in}" test.txt || test_failed $LINENO
cat "${in}" > test.txt || framework_failure
"${TARLZ}" --uncompressed -cf out.tar test.txt || test_failed $LINENO
rm -f test.txt || framework_failure
"${TARLZ}" -xf out.tar --missing-crc || test_failed $LINENO
cmp "${in}" test.txt || test_failed $LINENO
rm -f test.txt out.tar out.tar.lz || framework_failure

cat cfoo > foo || framework_failure
rm -f bar || framework_failure
cat cbaz > baz || framework_failure
"${TARLZ}" -q -cf out.tar.lz foo bar baz
[ $? = 1 ] || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -xf out.tar.lz --missing-crc || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
[ ! -e bar ] || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf out.tar.lz bar
[ $? = 1 ] || test_failed $LINENO
[ ! -e foo ] || test_failed $LINENO
[ ! -e bar ] || test_failed $LINENO
[ ! -e baz ] || test_failed $LINENO
rm -f out.tar.lz || framework_failure

cat cfoo > foo || framework_failure
cat cbar > bar || framework_failure
cat cbaz > baz || framework_failure
"${TARLZ}" -0 -cf out.tar.lz foo bar baz || test_failed $LINENO
"${TARLZ}" -0 -q -cf aout.tar.lz foo bar aout.tar.lz baz || test_failed $LINENO
cmp out.tar.lz aout.tar.lz || test_failed $LINENO
"${TARLZ}" -q -Af aout.tar.lz aout.tar.lz || test_failed $LINENO
cmp out.tar.lz aout.tar.lz || test_failed $LINENO
"${TARLZ}" -q -rf aout.tar.lz aout.tar.lz || test_failed $LINENO
cmp out.tar.lz aout.tar.lz || test_failed $LINENO
rm -f aout.tar.lz || framework_failure
#
"${TARLZ}" -0 -cf aout.tar.lz foo bar baz -C / || test_failed $LINENO
cmp out.tar.lz aout.tar.lz || test_failed $LINENO
rm -f aout.tar.lz || framework_failure
"${TARLZ}" -0 -C / -cf aout.tar.lz -C "${objdir}"/tmp foo bar baz ||
	test_failed $LINENO
cmp out.tar.lz aout.tar.lz || test_failed $LINENO
rm -f aout.tar.lz || framework_failure
"${TARLZ}" --asolid -0 -cf aout.tar.lz foo bar baz || test_failed $LINENO
cmp out.tar.lz aout.tar.lz || test_failed $LINENO
rm -f aout.tar.lz || framework_failure
"${TARLZ}" -0 -q -cf aout.tar.lz foo/ ./bar ./baz/ || test_failed $LINENO
cmp out.tar.lz aout.tar.lz || test_failed $LINENO
rm -f aout.tar.lz || framework_failure
mkdir dir1 || framework_failure
"${TARLZ}" -C dir1 -xf out.tar.lz || test_failed $LINENO
cmp cfoo dir1/foo || test_failed $LINENO
cmp cbar dir1/bar || test_failed $LINENO
cmp cbaz dir1/baz || test_failed $LINENO
rm -f aout.tar.lz foo bar baz || framework_failure
"${TARLZ}" -C dir1 -0 -cf aout.tar.lz foo bar baz || test_failed $LINENO
"${TARLZ}" -xf aout.tar.lz || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f aout.tar.lz foo bar baz || framework_failure
"${TARLZ}" -C dir1 -0 -c foo bar baz | "${TARLZ}" -x || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f dir1/foo dir1/bar dir1/baz || framework_failure
"${TARLZ}" -0 -c foo bar baz | "${TARLZ}" -C dir1 -x || test_failed $LINENO
cmp cfoo dir1/foo || test_failed $LINENO
cmp cbar dir1/bar || test_failed $LINENO
cmp cbaz dir1/baz || test_failed $LINENO
rm -f dir1/foo dir1/bar dir1/baz || framework_failure
"${TARLZ}" -0 -c foo bar baz | "${TARLZ}" -x foo bar baz -C dir1 ||
	test_failed $LINENO
cmp cfoo dir1/foo || test_failed $LINENO
cmp cbar dir1/bar || test_failed $LINENO
cmp cbaz dir1/baz || test_failed $LINENO
rm -f foo dir1/bar baz || framework_failure
"${TARLZ}" -0 -cf aout.tar.lz -C dir1 foo -C .. bar -C dir1 baz ||
	test_failed $LINENO
cmp out.tar.lz aout.tar.lz || test_failed $LINENO
"${TARLZ}" -0 -cf aout.tar.lz dir1/foo dir1/baz || test_failed $LINENO
rm -rf dir1 bar || framework_failure
"${TARLZ}" -xf aout.tar.lz dir1 || test_failed $LINENO
cmp cfoo dir1/foo || test_failed $LINENO
cmp cbaz dir1/baz || test_failed $LINENO
rm -rf dir1 || framework_failure
rm -f out.tar.lz aout.tar.lz || framework_failure

# test --append
cat cfoo > foo || framework_failure
cat cbar > bar || framework_failure
cat cbaz > baz || framework_failure
"${TARLZ}" -0 -cf out.tar.lz foo bar baz || test_failed $LINENO
"${TARLZ}" -0 -cf nout.tar.lz foo bar baz --no-solid || test_failed $LINENO
"${TARLZ}" -0 -cf aout.tar.lz foo || test_failed $LINENO
"${TARLZ}" -0 -rf aout.tar.lz bar baz --no-solid || test_failed $LINENO
cmp nout.tar.lz aout.tar.lz || test_failed $LINENO
rm -f nout.tar.lz aout.tar.lz || framework_failure
touch aout.tar.lz || framework_failure		# append to empty file
"${TARLZ}" -0 -rf aout.tar.lz foo bar baz || test_failed $LINENO
cmp out.tar.lz aout.tar.lz || test_failed $LINENO
"${TARLZ}" -0 -rf aout.tar.lz || test_failed $LINENO	# append nothing
cmp out.tar.lz aout.tar.lz || test_failed $LINENO
"${TARLZ}" -0 -rf aout.tar.lz -C nx_dir || test_failed $LINENO
cmp out.tar.lz aout.tar.lz || test_failed $LINENO
"${TARLZ}" -0 -q -rf aout.tar.lz nx_file
[ $? = 1 ] || test_failed $LINENO
cmp out.tar.lz aout.tar.lz || test_failed $LINENO
cat "${eof_lz}" > aout.tar.lz || framework_failure	# append to empty archive
"${TARLZ}" -0 -rf aout.tar.lz foo bar baz || test_failed $LINENO
cmp out.tar.lz aout.tar.lz || test_failed $LINENO
rm -f out.tar.lz aout.tar.lz || framework_failure

# append to solid archive
"${TARLZ}" --solid -0 -cf out.tar.lz foo || test_failed $LINENO
cat out.tar.lz > aout.tar.lz || framework_failure
for i in --asolid --bsolid --dsolid --solid -0 ; do
	"${TARLZ}" $i -q -rf out.tar.lz bar baz
	[ $? = 2 ] || test_failed $LINENO $i
	cmp out.tar.lz aout.tar.lz || test_failed $LINENO $i
done
rm -f out.tar.lz aout.tar.lz || framework_failure
for i in --asolid --bsolid --dsolid -0 ; do
	for j in --asolid --bsolid --dsolid --solid -0 ; do
		"${TARLZ}" $i -0 -cf out.tar.lz foo ||
			test_failed $LINENO "$i $j"
		"${TARLZ}" $j -0 -rf out.tar.lz bar baz ||
			test_failed $LINENO "$i $j"
		rm -f foo bar baz || framework_failure
		"${TARLZ}" -xf out.tar.lz || test_failed $LINENO "$i $j"
		cmp cfoo foo || test_failed $LINENO "$i $j"
		cmp cbar bar || test_failed $LINENO "$i $j"
		cmp cbaz baz || test_failed $LINENO "$i $j"
		rm -f out.tar.lz || framework_failure
	done
done
rm -f foo bar baz || framework_failure

# test directories and links
mkdir dir1 || framework_failure
"${TARLZ}" -0 -cf out.tar.lz dir1 || test_failed $LINENO
rmdir dir1 || framework_failure
"${TARLZ}" -xf out.tar.lz || test_failed $LINENO
[ -d dir1 ] || test_failed $LINENO
rmdir dir1
rm -f out.tar.lz || framework_failure
mkdir dir1 || framework_failure
"${TARLZ}" --uncompressed -cf out.tar dir1 || test_failed $LINENO
rmdir dir1 || framework_failure
"${TARLZ}" -xf out.tar || test_failed $LINENO
[ -d dir1 ] || test_failed $LINENO
rmdir dir1
rm -f out.tar || framework_failure

touch dummy_file || framework_failure
if ln dummy_file dummy_link 2> /dev/null &&
   ln -s dummy_file dummy_slink 2> /dev/null ; then
	ln_works=yes
	name_100=name_100_bytes_long_nnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnn
	path_100=dir1/dir2/dir3/path_100_bytes_long_nnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnn
	path_106=dir1/dir2/dir3/path_longer_than_100_bytes_nnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnnn
	mkdir dir1 || framework_failure
	mkdir dir1/dir2 || framework_failure
	mkdir dir1/dir2/dir3 || framework_failure
	cat "${in}" > dir1/dir2/dir3/in || framework_failure
	ln dir1/dir2/dir3/in dir1/dir2/dir3/"${name_100}" || framework_failure
	ln dir1/dir2/dir3/in "${path_100}" || framework_failure
	ln dir1/dir2/dir3/in "${path_106}" || framework_failure
	ln -s in dir1/dir2/dir3/link || framework_failure
	ln -s "${name_100}" dir1/dir2/dir3/link_100 || framework_failure
	"${TARLZ}" -0 -cf out.tar.lz dir1 || test_failed $LINENO
	rm -rf dir1 || framework_failure
	"${TARLZ}" -xf out.tar.lz || test_failed $LINENO
	cmp "${in}" dir1/dir2/dir3/in || test_failed $LINENO
	cmp "${in}" dir1/dir2/dir3/"${name_100}" || test_failed $LINENO
	cmp "${in}" "${path_100}" || test_failed $LINENO
	cmp "${in}" "${path_106}" || test_failed $LINENO
	cmp "${in}" dir1/dir2/dir3/link || test_failed $LINENO
	cmp "${in}" dir1/dir2/dir3/link_100 || test_failed $LINENO
	rm -f dir1/dir2/dir3/in || framework_failure
	cmp "${in}" dir1/dir2/dir3/link 2> /dev/null && test_failed $LINENO
	cmp "${in}" dir1/dir2/dir3/link_100 || test_failed $LINENO
	"${TARLZ}" -xf out.tar.lz || test_failed $LINENO
	rm -f out.tar.lz || framework_failure
	cmp "${in}" dir1/dir2/dir3/in || test_failed $LINENO
	cmp "${in}" dir1/dir2/dir3/link || test_failed $LINENO
	"${TARLZ}" -0 -q -c ../tmp/dir1 | "${TARLZ}" -x || test_failed $LINENO
	diff -r tmp/dir1 dir1 || test_failed $LINENO
	rm -rf tmp/dir1 dir1 || framework_failure
else
	printf "\nwarning: skipping link test: 'ln' does not work on your system."
fi
rm -f dummy_slink dummy_link dummy_file || framework_failure

printf "\ntesting long names..."

"${TARLZ}" -q -tf "${t155}" || test_failed $LINENO
"${TARLZ}" -q -tf "${t155_lz}" || test_failed $LINENO
if [ "${ln_works}" = yes ] ; then
	mkdir dir1 || framework_failure
	"${TARLZ}" -C dir1 -xf "${t155}" || test_failed $LINENO
	mkdir dir2 || framework_failure
	"${TARLZ}" -C dir2 -xf "${t155_lz}" || test_failed $LINENO
	diff -r dir1 dir2 || test_failed $LINENO
	"${TARLZ}" -cf out.tar.lz dir2 || test_failed $LINENO
	rm -rf dir2 || framework_failure
	"${TARLZ}" -xf out.tar.lz || test_failed $LINENO
	diff -r dir1 dir2 || test_failed $LINENO
	rmdir dir2 2> /dev/null && test_failed $LINENO
	rmdir dir1 2> /dev/null && test_failed $LINENO
	rm -rf out.tar.lz dir2 dir1 || framework_failure
fi

"${TARLZ}" -tvf "${testdir}"/ug32chars.tar.lz | grep -q \
	-e very_long_owner_name_of_32_chars/very_long_group_name_of_32_chars ||
	test_failed $LINENO
"${TARLZ}" -tvf "${testdir}"/ug32chars.tar.lz | grep -q \
	-e very_long_owner_name_of_32_charsvery_long_group_name_of_32_chars &&
	test_failed $LINENO
"${TARLZ}" -tvf "${testdir}"/ug32chars.tar.lz | grep -q \
	-e very_long_group_name_of_32_chars/very_long_group_name_of_32_chars &&
	test_failed $LINENO
"${TARLZ}" -xf "${testdir}"/ug32chars.tar.lz || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
rm -f foo || framework_failure

printf "\ntesting bad input..."

"${TARLZ}" -q -xf "${testdir}"/dotdot1.tar.lz || test_failed $LINENO
[ ! -e ../dir ] || test_failed $LINENO
"${TARLZ}" -q -xf "${testdir}"/dotdot2.tar.lz || test_failed $LINENO
[ ! -e ../dir ] || test_failed $LINENO
"${TARLZ}" -q -xf "${testdir}"/dotdot3.tar.lz || test_failed $LINENO
[ ! -e dir ] || test_failed $LINENO
"${TARLZ}" -q -xf "${testdir}"/dotdot4.tar.lz || test_failed $LINENO
[ ! -e dir ] || test_failed $LINENO
"${TARLZ}" -q -xf "${testdir}"/dotdot5.tar.lz || test_failed $LINENO
[ ! -e dir ] || test_failed $LINENO

dd if="${in_tar}" of=truncated.tar bs=1000 count=1 2> /dev/null
"${TARLZ}" -q -tf truncated.tar > /dev/null
[ $? = 2 ] || test_failed $LINENO
"${TARLZ}" -q -xf truncated.tar
[ $? = 2 ] || test_failed $LINENO
[ ! -e test.txt ] || test_failed $LINENO
rm -f truncated.tar || framework_failure

# test compressed and --keep-damaged
rm -f test.txt || framework_failure
for i in "${inbad1}" "${inbad2}" ; do
	"${TARLZ}" -q -xf "${i}.tar.lz"
	[ $? = 2 ] || test_failed $LINENO "$i"
	[ ! -e test.txt ] || test_failed $LINENO "$i"
	rm -f test.txt || framework_failure
	"${TARLZ}" -q -xf "${i}.tar.lz" --keep-damaged
	[ $? = 2 ] || test_failed $LINENO "$i"
	[ -e test.txt ] || test_failed $LINENO "$i"
	cmp "$i" test.txt 2> /dev/null || lzlib_1_11 "$LINENO $i"
	rm -f test.txt || framework_failure
done
#
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${bad1_lz}"
[ $? = 2 ] || test_failed $LINENO
[ ! -e foo ] || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${bad2_lz}"
[ $? = 2 ] || test_failed $LINENO
[ ! -e foo ] || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${bad3_lz}"
[ $? = 2 ] || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
[ ! -e bar ] || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${bad3_lz}" --keep-damaged
[ $? = 2 ] || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar 2> /dev/null || lzlib_1_11 $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${bad4_lz}"
[ $? = 2 ] || test_failed $LINENO
[ ! -e foo ] || test_failed $LINENO
[ ! -e bar ] || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${bad4_lz}" --keep-damaged
[ $? = 2 ] || test_failed $LINENO
[ ! -e foo ] || test_failed $LINENO
cmp cbar bar 2> /dev/null || lzlib_1_11 $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${bad5_lz}"
[ $? = 2 ] || test_failed $LINENO
[ ! -e foo ] || test_failed $LINENO
[ ! -e bar ] || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${bad5_lz}" --keep-damaged
[ $? = 2 ] || test_failed $LINENO
cmp cfoo foo 2> /dev/null || lzlib_1_11 $LINENO
[ ! -e bar ] || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${bad6_lz}"
[ $? = 2 ] || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO

# test uncompressed and --keep-damaged
rm -f test.txt || framework_failure
"${TARLZ}" -q -xf "${inbad1}.tar"
[ $? = 2 ] || test_failed $LINENO
[ ! -e test.txt ] || test_failed $LINENO
rm -f test.txt || framework_failure
"${TARLZ}" -q -xf "${inbad1}.tar" --keep-damaged
[ $? = 2 ] || test_failed $LINENO
[ -e test.txt ] || test_failed $LINENO
cmp "${inbad1}" test.txt 2> /dev/null || test_failed $LINENO
rm -f test.txt || framework_failure
#
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${bad1}"
[ $? = 2 ] || test_failed $LINENO
[ ! -e foo ] || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${bad2}"
[ $? = 2 ] || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
[ ! -e bar ] || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${bad3}"
[ $? = 2 ] || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
[ ! -e baz ] || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${bad4}"
[ $? = 2 ] || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
[ ! -e bar ] || test_failed $LINENO
[ ! -e baz ] || test_failed $LINENO
rm -f foo bar baz || framework_failure
"${TARLZ}" -q -xf "${bad5}"
[ $? = 2 ] || test_failed $LINENO
cmp cfoo foo || test_failed $LINENO
cmp cbar bar || test_failed $LINENO
cmp cbaz baz || test_failed $LINENO
rm -f cfoo cbar cbaz foo bar baz || framework_failure
#
rm -f test3.tar.lz || framework_failure
"${TARLZ}" -q -xf "${tlzit1}"
[ $? = 2 ] || test_failed $LINENO
[ ! -e foo ] || test_failed $LINENO
[ ! -e bar ] || test_failed $LINENO
[ ! -e baz ] || test_failed $LINENO
[ ! -e test3.tar.lz ] || test_failed $LINENO
rm -f foo bar baz test3.tar.lz || framework_failure
"${TARLZ}" -q -xf "${tlzit2}"
[ $? = 2 ] || test_failed $LINENO
[ ! -e foo ] || test_failed $LINENO
[ ! -e bar ] || test_failed $LINENO
[ ! -e baz ] || test_failed $LINENO
cmp "${test3_lz}" test3.tar.lz || test_failed $LINENO
rm -f foo bar baz test3.tar.lz || framework_failure

echo
if [ ${fail} = 0 ] ; then
	echo "tests completed successfully."
	cd "${objdir}" && rm -r tmp
else
	echo "tests failed."
fi
exit ${fail}
