# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

from __future__ import absolute_import, print_function, unicode_literals

import os

import attr
from mozilla_version.gecko import FirefoxVersion

from ..cli import BaseTryParser
from ..push import push_to_try, vcs


def read_file(path):
    with open(path) as fh:
        return fh.read()


class ReleaseParser(BaseTryParser):
    name = 'release'
    arguments = [
        [['-v', '--version'],
         {'metavar': 'STR',
          'required': True,
          'action': 'store',
          'type': FirefoxVersion.parse,
          'help': "The version number to use for the staging release.",
          }],
        [['--migration'],
         {'metavar': 'STR',
          'action': 'append',
          'dest': 'migrations',
          'choices': ['central-to-beta', 'beta-to-release'],
          'help': "Migration to run for the release (can be specified multiple times).",
          }],
        [['--no-limit-locales'],
         {'action': 'store_false',
          'dest': 'limit_locales',
          'help': "Don't build a limited number of locales in the staging release.",
          }],
    ]
    common_groups = ['push']


def run_try_release(
    version, migrations=(), push=True, message='{msg}', limit_locales=True, **kwargs
):

    if version.is_beta:
        app_version = attr.evolve(version, beta_number=None)
    else:
        app_version = version

    files_to_change = {
        'browser/config/version.txt': '{}\n'.format(app_version),
        'browser/config/version_display.txt': '{}\n'.format(version),
    }

    release_type = version.version_type.name.lower()
    if release_type not in ('beta', 'release', 'esr'):
        raise Exception(
            "Can't do staging release for version: {} type: {}".format(
                version, version.version_type))
    task_config = {
        'version': 2,
        'parameters': {
            'target_tasks_method': 'staging_release_builds',
            'optimize_target_tasks': True,
            'include_nightly': True,
            'release_type': release_type,
        },
    }

    for migration in migrations:
        migration_path = os.path.join(
            vcs.path,
            'testing/mozharness/configs/merge_day',
            '{}.py'.format(migration.replace('-', '_')),
        )
        migration_config = {}
        execfile(migration_path, migration_config, migration_config)
        for (path, from_, to) in migration_config['config']['replacements']:
            if path in files_to_change:
                contents = files_to_change[path]
            else:
                contents = read_file(path)
            files_to_change[path] = contents.replace(from_, to)

    if limit_locales:
        files_to_change['browser/locales/l10n-changesets.json'] = read_file(
            os.path.join(vcs.path, 'browser/locales/l10n-onchange-changesets.json'))
        files_to_change['browser/locales/shipped-locales'] = "en-US\n" + read_file(
            os.path.join(vcs.path, 'browser/locales/onchange-locales'))

    msg = 'staging release: {}'.format(version)
    return push_to_try(
        'release', message.format(msg=msg),
        push=push, closed_tree=kwargs["closed_tree"],
        try_task_config=task_config,
        files_to_change=files_to_change,
    )
