package NGCP::Panel::Controller::API::Vouchers;
use NGCP::Panel::Utils::Generic qw(:all);

use Sipwise::Base;

use boolean qw(true);
use Data::HAL qw();
use Data::HAL::Link qw();
use HTTP::Headers qw();
use HTTP::Status qw(:constants);
use MIME::Base64;

use NGCP::Panel::Utils::DateTime;
use NGCP::Panel::Utils::Voucher;

sub allowed_methods{
    return [qw/GET POST OPTIONS HEAD/];
}

sub api_description {
    return 'Defines vouchers to top-up subscriber balances.';
};

sub query_params {
    return [
        {
            param => 'reseller_id',
            description => 'Filter for vouchers belonging to a specific reseller',
            query => {
                first => sub {
                    my $q = shift;
                    { reseller_id => $q };
                },
                second => sub {},
            },
        },
        {
            param => 'package_id',
            description => 'Filter for vouchers belonging to a specific profile package',
            query => {
                first => sub {
                    my $q = shift;
                    { package_id => $q };
                },
                second => sub {},
            },
        },
        {
            param => 'base64_code',
            description => 'Filter for a voucher with the code, the value is expected as an encoded base64 string',
            new_rs => sub {
                my ($c, $q, $rs) = @_;
                $q .= '=' unless $q =~ /=$/;
                my $code = decode_base64($q);
                my $enc_code = NGCP::Panel::Utils::Voucher::encrypt_code($c, $code);
                return $rs->search_rs({
                    code => $enc_code,
                });
            },
        },
    ],
}

use parent qw/NGCP::Panel::Role::Entities NGCP::Panel::Role::API::Vouchers/;

sub resource_name{
    return 'vouchers';
}

sub dispatch_path{
    return '/api/vouchers/';
}

sub relation{
    return 'http://purl.org/sipwise/ngcp-api/#rel-vouchers';
}

__PACKAGE__->set_config({
    allowed_roles => [qw/admin reseller/],
    required_licenses => [qw/billing/],
});

sub GET :Allow {
    my ($self, $c) = @_;
    my $page = $c->request->params->{page} // 1;
    my $rows = $c->request->params->{rows} // 10;
    {
        my $items = $self->item_rs($c);
        #todo - is it really necessary? move to item_rs?
        $items = $items->search_rs({}, {prefetch => ['reseller']});
        (my $total_count, $items, my $items_rows) = $self->paginate_order_collection($c, $items);
        my (@embedded, @links);
        my $form = $self->get_form($c);
        $self->expand_prepare_collection($c);
        for my $item (@$items_rows) {
            push @embedded, $self->hal_from_item($c, $item, $form);
            push @links, Data::HAL::Link->new(
                relation => 'ngcp:'.$self->resource_name,
                href     => sprintf('/%s%d', $c->request->path, $item->id),
            );
        }
        $self->expand_collection_fields($c, \@embedded);
        push @links,
            Data::HAL::Link->new(
                relation => 'curies',
                href => 'http://purl.org/sipwise/ngcp-api/#rel-{rel}',
                name => 'ngcp',
                templated => true,
            ),
            Data::HAL::Link->new(relation => 'profile', href => 'http://purl.org/sipwise/ngcp-api/'),
            $self->collection_nav_links($c, $page, $rows, $total_count, $c->request->path, $c->request->query_params);

        my $hal = Data::HAL->new(
            embedded => [@embedded],
            links => [@links],
        );
        $hal->resource({
            total_count => $total_count,
        });
        my $response = HTTP::Response->new(HTTP_OK, undef, 
            HTTP::Headers->new($hal->http_headers(skip_links => 1)), $hal->as_json);
        $c->response->headers($response->headers);
        $c->response->body($response->content);
        return;
    }
    return;
}

sub POST :Allow {
    my ($self, $c) = @_;

    unless($c->user->billing_data) {
        $self->error($c, HTTP_FORBIDDEN, "Insufficient rights to create voucher",
                     "user does not have billing data rights");
        return;
    }

    my $guard = $c->model('DB')->txn_scope_guard;
    {
        my $resource = $self->get_valid_post_data(
            c => $c, 
            media_type => 'application/json',
        );
        last unless $resource;

        my $form = $self->get_form($c);
        last unless $self->validate_form(
            c => $c,
            resource => $resource,
            form => $form,
        );
        if($c->user->roles eq "admin") {
        } elsif($c->user->roles eq "reseller") {
            $resource->{reseller_id} = $c->user->reseller_id;
        }

        my $item;
        my $code = NGCP::Panel::Utils::Voucher::encrypt_code($c, $resource->{code});
        $item = $c->model('DB')->resultset('vouchers')->find({
            reseller_id => $resource->{reseller_id},
            code => $code,
        });
        if($item) {
            $self->error($c, HTTP_UNPROCESSABLE_ENTITY, "Voucher with this code already exists for this reseller",
                         "voucher with code '$$resource{code}' already exists for reseller_id '$$resource{reseller_id}'");
            last;
        }
        $resource->{code} = $code;
        $resource->{created_at} = NGCP::Panel::Utils::DateTime::current_local;


        try {
            $item = $c->model('DB')->resultset('vouchers')->create($resource);
        } catch($e) {
            $self->error($c, HTTP_INTERNAL_SERVER_ERROR, "Failed to create voucher.", $e);
            last;
        }

        $guard->commit;

        $c->response->status(HTTP_CREATED);
        $c->response->header(Location => sprintf('/%s%d', $c->request->path, $item->id));
        $c->response->body(q());
    }
    return;
}

1;

# vim: set tabstop=4 expandtab:
