#!/usr/bin/env python3

import copy
import os
import subprocess
import sys
import unittest
from yaml import load, dump
try:
    from yaml import CLoader as Loader, CDumper as Dumper
except ImportError:
    from yaml import Loader, Dumper


def executeAndReturnOutput(command, env):
    p = subprocess.Popen(command, encoding="utf-8",
                         stdout=subprocess.PIPE,
                         stderr=subprocess.PIPE, env=env)
    stdoutdata, stderrdata = p.communicate()
    print(stdoutdata, file=sys.stdout)
    print(stderrdata, file=sys.stderr)
    return p.returncode, stdoutdata, stderrdata


command = ["./ngcp-service", '--verbose', '--debug']
ROLES = {
    'spce': ['proxy', 'lb', 'rtp', 'db', 'mgmt'],
    'sp1': ['proxy', 'lb', 'rtp', 'db', 'mgmt'],
    'slb01a': ['lb', 'rtp'],
    'prx01a': ['proxy', ],
}


def create_prog(filename, ok=True):
    os.system('echo "#!/bin/sh" > %s' % filename)
    if ok:
        os.system('echo /bin/true >> %s' % filename)
    else:
        os.system('echo /bin/false >> %s' % filename)
    os.system('chmod +x %s' % filename)


def mkdir_p(pathname):
    if not os.path.exists(pathname):
        os.makedirs(pathname)


def rmdir_p(pathname):
    if os.path.exists(pathname):
        os.rmdir(pathname)


class TestCommandLine(unittest.TestCase):

    def _set_config(self):
        stream = open(self.env['NGCP_CONFIG'], 'w')
        dump(self.config, stream)
        stream.close()

    def _set_platform(self, value):
        self.config['host']['platform'] = value

    def _set_ngcp_type(self, value):
        self.config['host']['ngcp_type'] = value

    def _set_status(self):
        create_prog('tests/usr/sbin/ngcp-check-active', self.active)

    def _set_handler(self, hostname, service, ok=True, process_handler=None):
        if process_handler is None:
            p = self.config['host']['process_handler']
        else:
            self.config['host']['process_handler'] = process_handler
            p = process_handler

        if p == 'none' and self.config['host']['init_system'] == 'systemd':
            p = 'systemd'

        if p == 'none':
            filename = 'tests/usr/sbin/service'
        elif p == 'systemd':
            filename = 'tests/bin/systemctl'
        elif p == 'monit':
            filename = 'tests/usr/bin/monit'
        else:
            raise Exception("unknown process_handler[%s]" % p)

        if self.config['host']['init_system'] == 'systemd':
            mkdir_p('tests/run/systemd/system')
        else:
            rmdir_p('tests/run/systemd/system')

        self.config['host']['roles'] = ROLES[hostname]
        self._set_config()
        create_prog(filename, ok)

    def setUp(self):
        self.command = copy.deepcopy(command)
        self.active = False
        self.env = {
            'PERL5LIB': 'lib',
            'NGCP_SERVICES': 'tests/fixtures/nsservices.yml',
            'NGCP_SERVICES_DIR': 'tests/fixtures/nsservices.d',
            'NGCP_CONFIG': 'tests/etc/ngcp-service/nsconfig.yml',
            'NGCP_ROOT_DIR': 'tests',
        }
        self.config = {'host': {'init_system': 'systemd',
                                'process_handler': 'none',
                                'platform': 'ngcp',
                                'ngcp_type': 'spce',
                                },
                       }
        self._set_config()

    def tearDown(self):
        os.system('make clean > /dev/null 2>&1')

    def testWrongParam(self):
        self.command.append('--fake-param')
        self.assertCountEqual(self.command, command + ['--fake-param'])
        res = executeAndReturnOutput(self.command, self.env)
        self.assertNotEqual(res[0], 0)
        self.assertRegex(
            res[2], "ngcp-service: error: Unknown option: fake-param")

    def testWrongArgsOnlyService(self):
        self.command = self.command + \
            ['--service', 'asterisk', '--service', 'ngcp-panel']
        res = executeAndReturnOutput(self.command, self.env)
        self.assertNotEqual(res[0], 0)
        self.assertRegex(
            res[2],
            "ngcp-service: error: missing action")

    def testWrongArgsOnlyAction(self):
        self.command = self.command + \
            ['--action', 'restart', '--action', 'reload']
        res = executeAndReturnOutput(self.command, self.env)
        self.assertNotEqual(res[0], 0)
        self.assertRegex(
            res[2],
            r"ngcp-service: error: missing service\(s\) or group")

    def testArgsServiceFirst(self):
        self.command = self.command + \
            ['asterisk', 'start']
        self._set_status()
        self._set_handler('spce', 'asterisk')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/bin/systemctl start asterisk.service")

    def testArgsActionFirst(self):
        self.command = self.command + \
            ['start', 'asterisk']
        self._set_status()
        self._set_handler('spce', 'asterisk')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/bin/systemctl start asterisk.service")

    def testWrongArgsActionLastMulti(self):
        self.command = self.command + \
            ['asterisk', 'ssh', 'ngcp-panel', 'start']
        self.active = True
        self._set_status()
        self._set_handler('spce', 'asterisk')
        self._set_handler('spce', 'ngcp-panel')
        self._set_handler('spce', 'ssh')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertNotEqual(res[0], 0)
        self.assertRegex(
            res[2],
            "ngcp-service: error: unknown action: asterisk")

    def testArgsActionFirstMulti(self):
        self.command = self.command + \
            ['start', 'asterisk', 'ssh', 'ngcp-panel']
        self.active = True
        self._set_status()
        self._set_handler('spce', 'asterisk')
        self._set_handler('spce', 'ngcp-panel')
        self._set_handler('spce', 'ssh')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/bin/systemctl start "
                    "asterisk.service ssh.service ngcp-panel.service")

    def testHelp(self):
        self.command.append('--help')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[1],
            "Usage: ngcp-service .*<command>.*")

    def testIsManagedServiceOk(self):
        self.command = self.command + \
            ['--action', 'is-managed', '--service', 'ngcp-panel']
        self._set_status()
        self._set_platform('ngcp')
        self._set_ngcp_type('sppro')
        self._set_handler('sp1', 'ngcp-panel', process_handler='none')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)

    def testIsManagedServiceNotPlatformKo(self):
        self.command = self.command + \
            ['--action', 'is-managed', '--service', 'ngcp-panel']
        self._set_status()
        self._set_platform('alien')
        self._set_ngcp_type('sppro')
        self._set_handler('spce', 'ngcp-panel', process_handler='none')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 1)
        self.assertRegex(
            res[2],
            "is-managed skipped, service ngcp-panel is not for this platform")

    def testIsManagedServiceNotTypeKo(self):
        self.command = self.command + \
            ['--action', 'is-managed', '--service', 'captagent']
        self._set_status()
        self._set_ngcp_type('spce')
        self._set_handler('spce', 'captagent', process_handler='none')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 1)
        self.assertRegex(
            res[2],
            "is-managed skipped, service captagent is not for ngcp_type spce")

    def testIsManagedServiceNotRoleKo(self):
        self.command = self.command + \
            ['--action', 'is-managed', '--service', 'ngcp-panel']
        self._set_status()
        self._set_ngcp_type('sppro')
        self._set_handler('prx01a', 'ngcp-panel', process_handler='none')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 1)
        self.assertRegex(
            res[2],
            "is-managed skipped, service ngcp-panel is not for this role")

    def testIsManagedServiceNotEnabledKo(self):
        self.command = self.command + \
            ['--action', 'is-managed', '--service', 'proxy']
        self.active = True
        self._set_status()
        self._set_ngcp_type('sppro')
        self._set_handler('prx01a', 'kamailio-proxy', process_handler='none')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 1)
        self.assertRegex(
            res[2], "is-managed skipped, service proxy is not enabled")

    def testNonDefinedService(self):
        self.command = self.command + ['--action', 'start', '--service', 'foo']
        res = executeAndReturnOutput(self.command, self.env)
        self.assertNotEqual(res[0], 0)
        self.assertRegex(res[2], 'No foo defined')

    def testNonDefinedNGCP_ROOT_DIR(self):
        self.command = self.command + ['--action', 'start',
                                       '--service', 'asterisk']
        del self.env['NGCP_ROOT_DIR']
        self._set_status()
        self._set_handler('spce', 'asterisk')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertNotEqual(res[0], 0)
        self.assertRegex(
            res[2], 'No such file or directory')

    def testShortServiceOk(self):
        self.command = self.command + \
            ['asterisk', 'start']
        self._set_status()
        self._set_handler('spce', 'asterisk')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/bin/systemctl start asterisk.service")

    def testServiceOk(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'asterisk']
        self._set_status()
        self._set_handler('spce', 'asterisk')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/bin/systemctl start asterisk.service")

    def testServiceKo(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'asterisk']
        self._set_status()
        self._set_handler('spce', 'asterisk', ok=False)
        res = executeAndReturnOutput(self.command, self.env)
        self.assertNotEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/bin/systemctl start asterisk.service")

    def testServiceAsyncOk(self):
        self.command = self.command + \
            ['--async', '--action', 'start', '--service', 'asterisk']
        self._set_status()
        self._set_handler('spce', 'asterisk')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2],
            "execute: tests/bin/systemctl --no-block start asterisk.service")

    def testServiceAliases(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'ngcp-captagent']
        self._set_ngcp_type('sppro')
        self.active = True
        self._set_status()
        self._set_handler('sp1', 'captagent', process_handler='none')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2],
            "execute: tests/bin/systemctl start captagent.service")

    def testSystemdServiceOk(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'asterisk']
        self._set_status()
        self._set_handler('spce', 'asterisk')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/bin/systemctl start asterisk.service")

    def testSysvinitServiceOk(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'asterisk']
        self.config['host']['init_system'] = 'sysv'
        self._set_status()
        self._set_handler('spce', 'asterisk', process_handler='none')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/usr/sbin/service asterisk start")

    def testSysvinitFallbackSystemdServiceOk(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'asterisk']
        self.config['host']['init_system'] = 'sysv'
        self._set_status()
        self._set_handler('spce', 'asterisk', process_handler='none')
        self._set_handler('spce', 'asterisk', process_handler='systemd')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/usr/sbin/service asterisk start")

    def testServiceMonitOk(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'asterisk']
        self._set_status()
        self._set_handler('sp1', 'asterisk', process_handler='systemd')
        self._set_handler('sp1', 'asterisk', process_handler='monit')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/bin/systemctl start asterisk.service")

    def testServiceMonitOkMonitored(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'faxserver']
        self.active = True
        self._set_status()
        self._set_handler('sp1', 'faxserver', process_handler='systemd')
        self._set_handler('sp1', 'faxserver', process_handler='monit')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/usr/bin/monit unmonitor faxserver")
        self.assertRegex(
            res[2],
            "execute: tests/bin/systemctl start ngcp-faxserver.service")
        self.assertRegex(
            res[2], "execute: tests/usr/bin/monit monitor faxserver")

    def testServiceSysvinitMonitOk(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'asterisk']
        self.config['host']['init_system'] = 'sysv'
        self._set_status()
        self._set_handler('sp1', 'asterisk', process_handler='none')
        self._set_handler('sp1', 'asterisk', process_handler='monit')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/usr/sbin/service asterisk start")

    def testServiceSysvinitMonitOkMonitored(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'faxserver']
        self.active = True
        self.config['host']['init_system'] = 'sysv'
        self._set_status()
        self._set_handler('sp1', 'faxserver', process_handler='none')
        self._set_handler('sp1', 'faxserver', process_handler='monit')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/usr/bin/monit unmonitor faxserver")
        self.assertRegex(
            res[2], "execute: tests/usr/sbin/service ngcp-faxserver start")
        self.assertRegex(
            res[2], "execute: tests/usr/bin/monit monitor faxserver")

    def testServiceMultiMonitOk(self):
        self.command = self.command + \
            ['--action', 'start',
             '--service', 'asterisk',
             '--service', 'ngcp-panel',
             '--service', 'faxserver']
        self._set_status()
        self._set_handler('sp1', 'asterisk', process_handler='systemd')
        self._set_handler('sp1', 'asterisk', process_handler='monit')
        self._set_handler('sp1', 'ngcp-panel', process_handler='systemd')
        self._set_handler('sp1', 'ngcp-panel', process_handler='monit')
        self._set_handler('sp1', 'faxserver', process_handler='systemd')
        self._set_handler('sp1', 'faxserver', process_handler='monit')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/bin/systemctl start "
                    "asterisk.service ngcp-panel.service")
        self.assertNotRegex(
            res[2], "execute: tests/bin/systemctl start faxserver.service")

    def testServiceMultiMonitOkMonitored(self):
        self.command = self.command + \
            ['--action', 'start',
             '--service', 'asterisk',
             '--service', 'ngcp-panel',
             '--service', 'faxserver']
        self.active = True
        self._set_status()
        self._set_handler('sp1', 'asterisk', process_handler='systemd')
        self._set_handler('sp1', 'asterisk', process_handler='monit')
        self._set_handler('sp1', 'ngcp-panel', process_handler='systemd')
        self._set_handler('sp1', 'ngcp-panel', process_handler='monit')
        self._set_handler('sp1', 'faxserver', process_handler='systemd')
        self._set_handler('sp1', 'faxserver', process_handler='monit')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/usr/bin/monit unmonitor asterisk")
        self.assertRegex(
            res[2], "execute: tests/usr/bin/monit unmonitor ngcp-panel")
        self.assertRegex(
            res[2], "execute: tests/usr/bin/monit unmonitor faxserver")
        self.assertRegex(
            res[2], "execute: tests/bin/systemctl start"
                    " asterisk.service"
                    " ngcp-panel.service"
                    " ngcp-faxserver.service")
        self.assertRegex(
            res[2], "execute: tests/usr/bin/monit monitor faxserver")
        self.assertRegex(
            res[2], "execute: tests/usr/bin/monit monitor ngcp-panel")
        self.assertRegex(
            res[2], "execute: tests/usr/bin/monit monitor asterisk")

    def testServiceNotActiveOk(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'captagent']
        self._set_ngcp_type('sppro')
        self._set_status()
        self._set_handler('sp1', 'captagent', process_handler='systemd')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertNotRegex(
            res[2],
            "execute: tests/bin/systemctl start captagent")

    def testServiceActiveOk(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'captagent']
        self._set_ngcp_type('sppro')
        self.active = True
        self._set_status()
        self._set_handler('sp1', 'captagent', process_handler='systemd')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2],
            "execute: tests/bin/systemctl start captagent")

    def testServiceGroupOk(self):
        self.command = self.command + \
            ['--action', 'start', '--group', 'ngcp-upgrade-ha']
        self.active = True
        self._set_status()
        self._set_ngcp_type('sppro')
        self._set_handler('sp1', 'monit')
        self._set_handler('sp1', 'pacemaker')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2],
            "execute: tests/bin/systemctl start"
            " monit.service"
            " pacemaker.service")

    def testServiceGroupOkStop(self):
        self.command = self.command + \
            ['--action', 'stop', '--group', 'ngcp-upgrade-ha']
        self.active = True
        self._set_status()
        self._set_ngcp_type('sppro')
        self._set_handler('sp1', 'monit')
        self._set_handler('sp1', 'pacemaker')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2],
            "execute: tests/bin/systemctl stop"
            " monit.service"
            " pacemaker.service")

    def testServiceNotRoleKo(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'captagent']
        self.active = True
        self._set_status()
        self.config['host']['process_handler'] = 'systemd'
        self._set_handler('prx01a', 'captagent')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertNotRegex(
            res[2],
            "execute: tests/bin/systemctl start captagent")

    def testServiceRoleOk(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'captagent']
        self.active = True
        self._set_ngcp_type('sppro')
        self._set_status()
        self._set_handler('slb01a', 'captagent', process_handler='systemd')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2],
            "execute: tests/bin/systemctl start captagent")

    def testNgcpTypeKo(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'captagent']
        self.active = True
        self._set_ngcp_type('spce')
        self._set_status()
        self._set_handler('spce', 'captagent', process_handler='systemd')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2],
            "start skipped, service captagent is not for ngcp_type spce")

    def testNgcpTypeOk(self):
        self.command = self.command + \
            ['asterisk', 'start']
        self._set_ngcp_type('carrier')
        self._set_status()
        self._set_handler('spce', 'asterisk')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "execute: tests/bin/systemctl start asterisk.service")

    def testServiceSPCERoleOk(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'ssh']
        self.active = True
        self._set_status()
        self._set_handler('spce', 'ssh')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2],
            "execute: tests/bin/systemctl start ssh.service")

    def testServiceSPCEOk(self):
        self.command = self.command + \
            ['--action', 'start', '--service', 'ngcp-panel']
        self.active = True
        self._set_status()
        self._set_handler('spce', 'ngcp-panel')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2],
            "execute: tests/bin/systemctl start ngcp-panel.service")

    def testServiceSelf(self):
        self.command = self.command + \
            ['--action', 'restart', '--service', 'monit']
        self.active = True
        self._set_ngcp_type('sppro')
        self._set_status()
        # generates the systemctl command
        self._set_handler('slb01a', 'monit', process_handler='systemd')
        # test self call in monit
        self._set_handler('slb01a', 'monit', process_handler='monit')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2],
            "execute: tests/bin/systemctl restart monit.service")

    def testServiceFallbackOk(self):
        self.command = self.command + \
            ['--action', 'restart', '--service', 'pacemaker']
        self.active = True
        self._set_ngcp_type('sppro')
        self._set_status()
        # generate service prog
        self._set_handler('slb01a', 'pacemaker', process_handler='systemd')
        self._set_handler('slb01a', 'pacemaker', process_handler='monit')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2],
            "DEBUG: process_handler is monit, using systemd")
        self.assertRegex(
            res[2],
            "execute: tests/bin/systemctl restart pacemaker.service")

    def testServiceFallbackKo(self):
        self.command = self.command + \
            ['--action', 'restart', '--service', 'pacemaker']
        self.active = True
        self.config['host']['init_system'] = 'sysv'
        self._set_ngcp_type('sppro')
        self._set_status()
        self._set_handler('slb01a', 'pacemaker', process_handler='none')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertNotEqual(res[0], 0)
        self.assertRegex(
            res[2],
            r"pacemaker\[sysv\] not defined")

    def testServiceRestart_notEnable(self):
        self.command = self.command + \
            ['--action', 'restart', '--service', 'proxy']
        self.active = True
        self._set_status()
        self._set_ngcp_type('sppro')
        # generates the service command
        self._set_handler('prx01a', 'kamailio-proxy', process_handler='none')
        self._set_handler('prx01a', 'kamailio-proxy', process_handler='monit')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2],
            "restart skipped, service proxy is not enabled")
        self.assertNotRegex(
            res[2],
            "execute: tests/bin/monit restart proxy")
        self.assertNotRegex(
            res[2],
            "execute: tests/bin/systemctl stop ngcp-kamailio-proxy.service")
        self.assertNotRegex(
            res[2],
            "execute: tests/usr/sbin/service kamailio-proxy stop")

    def testServiceRestart_notEnable_sysv(self):
        self.command = self.command + \
            ['--action', 'restart', '--service', 'pacemaker-disabled']
        self.active = True
        self._set_status()
        self._set_ngcp_type('sppro')
        self._set_handler('prx01a', 'pacemaker-disabled',
                          process_handler='none')
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2],
            "restart skipped, service pacemaker-disabled is not enabled")
        self.assertNotRegex(
            res[2],
            "execute: tests/usr/bin/service pacemaker restart")
        self.assertNotRegex(
            res[2],
            "execute: tests/usr/bin/service pacemaker stop")

    def testServiceQueue(self):
        self.active = True
        self._set_status()
        self._set_ngcp_type('sppro')
        # generates the service command
        self._set_handler('sp1', 'asterisk', process_handler='systemd')
        self._set_handler('sp1', 'captagent', process_handler='systemd')
        self._set_handler('sp1', 'faxserver', process_handler='systemd')
        self._set_handler('sp1', 'pacemaker', process_handler='systemd')
        self._set_handler('sp1', 'monit', process_handler='systemd')
        self._set_handler('sp1', 'ngcp-panel', process_handler='systemd')
        self.command = command + \
            ['queue-start']
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertEqual(res[2], '')
        self.command = command + \
            ['--enqueue', '--action', 'restart',
                          '--service', 'asterisk',
                          '--service', 'ngcp-panel',
                          '--service', 'faxserver']
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertEqual(res[2], '')
        self.command = command + \
            ['--enqueue', '--action', 'stop',
                          '--service', 'captagent',
                          '--service', 'faxserver']
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertEqual(res[2], '')
        self.command = command + \
            ['--enqueue', '--action', 'start',
                          '--service', 'captagent',
                          '--service', 'pacemaker',
                          '--service', 'monit',
                          '--service', 'asterisk']
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertEqual(res[2], '')
        self.command = command + \
            ['--enqueue', '--action', 'reload',
                          '--service', 'pacemaker',
                          '--service', 'ngcp-panel']
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertEqual(res[2], '')
        self.command = command + \
            ['--enqueue', '--action', 'stop',
                          '--service', 'monit']
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertEqual(res[2], '')
        self.command = command + \
            ['queue-run']
        res = executeAndReturnOutput(self.command, self.env)
        self.assertEqual(res[0], 0)
        self.assertRegex(
            res[2], "synthesized action=restart for service=asterisk")
        self.assertRegex(
            res[2], "synthesized action=restart for service=ngcp-panel")
        self.assertRegex(
            res[2], "synthesized action=stop for service=faxserver")
        self.assertRegex(
            res[2], "synthesized action=restart for service=captagent")
        self.assertRegex(
            res[2], "synthesized action=restart for service=pacemaker")
        self.assertRegex(
            res[2], "synthesized action=stop for service=monit")
        self.assertRegex(
            res[2], "execute: tests/bin/systemctl stop"
                    " ngcp-faxserver.service"
                    " monit.service")
        self.assertRegex(
            res[2], "execute: tests/bin/systemctl restart"
                    " asterisk.service"
                    " captagent.service"
                    " ngcp-panel.service"
                    " pacemaker.service")
