"""BaseModel definitions for dialog."""

from __future__ import annotations

from pydantic import BaseModel, Field


class Dialog(BaseModel):
    """Dialog definition.

    Dialog: A dialog is a peer-to-peer SIP relationship between two
    UAs that persists for some time. A dialog is established by
    SIP messages, such as a 2xx response to an INVITE request. A
    dialog is identified by a call identifier, local tag, and a
    remote tag. A dialog was formerly known as a call leg in RFC
    2543.
    Call-ID contains a globally unique identifier for this call,
    generated by the combination of a random string and the softphone's
    host name or IP address. The combination of the To tag, From tag,
    and Call-ID completely defines a peer-to-peer SIP relationship
    between Alice and Bob and is referred to as a dialog.
    """

    callid: str
    ftag: str = Field(default="")
    ttag: str = Field(default="")


class DialogParams(BaseModel):
    """Dialog parameters definition."""

    dialog: Dialog


class CounterDialogParams(DialogParams):
    """Counter dialog request body parameter constraints."""

    counter: str
    id: str | None = Field(default=None)


class PeerDialogParams(DialogParams):
    """Peer dialog request body parameter constraints."""

    peer_id: str


class CallerDialogParams(DialogParams):
    """Caller dialog request body parameter constraints."""

    user_id: str
    account_id: str
    reseller_id: str
    location_id: str = Field(default="")
    p_to_group: int


class CalleeDialogParams(DialogParams):
    """Callee dialog request body parameter constraints."""

    user_id: str
    account_id: str
    reseller_id: str
    location_id: str = Field(default="")
    p_to_group: int
    callee_ip: str = Field(default="")


class ActiveUserDialogParams(DialogParams):
    """Active User dialog request body parameter constraints."""

    user_id: str | None = Field(default=None)


class DialogKeys(BaseModel):
    """Dialog keys.

    Each group contains a list of redis keys
    that for which an action will be performed.
    """

    user: list[str] | None = Field(default=None)
    account: list[str] | None = Field(default=None)
    location: list[str] | None = Field(default=None)
    reseller: list[str] | None = Field(default=None)
    peer: list[str] | None = Field(default=None)
    general: list[str] | None = Field(default=None)


class DialogKeyIds(BaseModel):
    """
    Dialog key configuration.

    Contains the identification parameter for
    the different key types.
    """


class UserDialogKeyIds(DialogKeyIds):
    """
    User Dialog key configuration.

    Contains a user id.
    """

    user: str


class AccountDialogKeyIds(DialogKeyIds):
    """
    Account dialog key configuration.

    Contains an identification parameter for
    the keys inside the account key group.
    """

    account: str


class LocationDialogKeyIds(DialogKeyIds):
    """
    Location dialog key configuration.

    Contains an identification parameter for
    the keys inside the location key group.
    """

    location: str


class ResellerDialogKeyIds(DialogKeyIds):
    """
    Reseller dialog key configuration.

    Contains an identification parameter for
    the keys inside the reseller key group.
    """

    reseller: str


class PeerDialogKeyIds(DialogKeyIds):
    """
    Peer dialog key configuration.

    Contains an identification parameter for
    the keys inside the peer key group.
    """

    peer: str


class CalleeDialogKeyIds(DialogKeyIds):
    """
    Local dialog key configuration.

    Contains an identification parameter for
    the keys related to a specific callee.
    """

    user: str
    account: str
    location: str | None = Field(default=None)


class CallerDialogKeyIds(DialogKeyIds):
    """
    Caller dialog key configuration.

    Contains an identification parameter for
    the keys related to a specific caller.
    """

    user: str
    account: str
    location: str | None = Field(default=None)
    reseller: str


class HuntgroupDialogKeyIds(DialogKeyIds):
    """
    Huntgroup dialog key configuration.

    Contains an identification parameter for
    the keys related to a specific huntgroup.
    """

    user: str
    account: str
    location: str | None = Field(default=None)


class DialogBundle(BaseModel):
    """
    Dialog bundle.

    Just for compability on how dialog info is added.
    """

    dialog: Dialog


class DialogIdBundle(BaseModel):
    """
    Dialog id bundle.

    Bundles together theDialog object for the current
    dialog and a set of dialog keys ids.
    """

    dialog: Dialog
    dialog_key_ids: DialogKeyIds


class UserDialogIdBundle(DialogIdBundle):
    """
    User dialog id bundle.

    Holds the dialog key id for a user.
    """

    dialog_key_ids: UserDialogKeyIds


class AccountDialogIdBundle(DialogIdBundle):
    """
    Account dialog id bundle.

    Holds the dialog key id for an account.
    """

    dialog_key_ids: AccountDialogKeyIds


class ResellerDialogIdBundle(DialogIdBundle):
    """
    Reseller dialog id bundle.

    Holds the dialog key id for a reseller.
    """

    dialog_key_ids: ResellerDialogKeyIds


class LocationDialogIdBundle(DialogIdBundle):
    """
    Location dialog id bundle.

    Holds the dialog key id for a location.
    """

    dialog_key_ids: LocationDialogKeyIds


class PeerDialogIdBundle(DialogIdBundle):
    """
    Peer Dialog id bundle.

    Holds the dialog key id for a peer.
    """

    dialog_key_ids: PeerDialogKeyIds


class CalleeDialogIdBundle(DialogIdBundle):
    """
    Callee dialog id bundle.

    Holds the dialog key ids for a callee.
    """

    dialog_key_ids: CalleeDialogKeyIds


class CallerDialogIdBundle(DialogIdBundle):
    """
    Caller dialog id bundle.

    Holds the dialog key ids for a caller.
    """

    dialog_key_ids: CallerDialogKeyIds


class HuntgroupDialogIdBundle(DialogIdBundle):
    """
    Huntgroup dialog id bundle.

    Holds the dialog key ids for a huntgroup.
    """

    dialog_key_ids: HuntgroupDialogKeyIds
