import {Body, Controller, Delete, Get, Param, ParseIntPipe, Patch, Post, Put, Req, ValidationPipe} from '@nestjs/common'
import {ApiBody, ApiConsumes, ApiOkResponse, ApiQuery, ApiTags} from '@nestjs/swagger'
import {Request} from 'express'
import {Transactional} from 'typeorm-transactional'
import {number} from 'yargs'

import {PeeringRuleRequestParamDto} from './dto/peering-rule-request.param.dto'
import {PeeringRuleResponseDto} from './dto/peering-rule-response.dto'
import {PeeringRuleService} from './peering-rule.service'

import {JournalResponseDto} from '~/api/journals/dto/journal-response.dto'
import {JournalService} from '~/api/journals/journal.service'
import {PeeringRuleRequestDto} from '~/api/peerings/groups/rules/dto/peering-rule-request.dto'
import {RbacRole} from '~/config/constants.config'
import {CrudController} from '~/controllers/crud.controller'
import {ApiCreatedResponse} from '~/decorators/api-created-response.decorator'
import {ApiPaginatedResponse} from '~/decorators/api-paginated-response.decorator'
import {ApiPutBody} from '~/decorators/api-put-body.decorator'
import {Auth} from '~/decorators/auth.decorator'
import {ParamOrBody} from '~/decorators/param-or-body.decorator'
import {PatchDto} from '~/dto/patch.dto'
import {internal} from '~/entities'
import {Dictionary} from '~/helpers/dictionary.helper'
import {Operation} from '~/helpers/patch.helper'
import {Operation as PatchOperation,patchToEntity} from '~/helpers/patch.helper'
import {SearchLogic} from '~/helpers/search-logic.helper'
import {ServiceRequest} from '~/interfaces/service-request.interface'
import {LoggerService} from '~/logger/logger.service'
import {ParseIdDictionary} from '~/pipes/parse-id-dictionary.pipe'
import {ParseIntIdArrayPipe} from '~/pipes/parse-int-id-array.pipe'
import {ParseOneOrManyPipe} from '~/pipes/parse-one-or-many.pipe'
import {ParsePatchPipe} from '~/pipes/parse-patch.pipe'

const resourceName = 'peerings/groups'

@Auth(
    RbacRole.system,
    RbacRole.admin,
)
@ApiTags('Peering')
@Controller(resourceName)
export class PeeringRuleController extends CrudController<PeeringRuleRequestDto, PeeringRuleResponseDto> {
    private readonly log = new LoggerService(PeeringRuleController.name)

    constructor(
        private readonly ruleService: PeeringRuleService,
        private readonly journalService: JournalService,
    ) {
        super(resourceName, ruleService)
    }

    @Post(':groupId?/rules')
    @ApiCreatedResponse(PeeringRuleResponseDto)
    @ApiBody({
        type: PeeringRuleRequestDto,
        isArray: true,
    })
    @Transactional()
    async create(
        @Body(new ParseOneOrManyPipe({items: PeeringRuleRequestDto})) createDto: PeeringRuleRequestDto[],
        @Req() req: Request,
    ): Promise<PeeringRuleResponseDto[]> {
        this.log.debug({
            message: 'create peering rule bulk',
            func: this.create.name,
            url: req.url,
            method: req.method,
        })
        const sr = new ServiceRequest(req)
        const rules = await Promise.all(createDto.map(async set => set.toInternal()))
        const created = await this.ruleService.create(rules, sr)
        return await Promise.all(created.map(async server => new PeeringRuleResponseDto(
            server,
            {url: req.url},
        )))
    }

    @Get(':groupId?/rules')
    @ApiQuery({type: SearchLogic})
    @ApiPaginatedResponse(PeeringRuleResponseDto)
    async readAll(
        @Req() req: Request,
        @Param(new ValidationPipe()) _reqParams: PeeringRuleRequestParamDto): Promise<[PeeringRuleResponseDto[], number]> {
        this.log.debug({
            message: 'read all peering rules across all groups',
            func: this.readAll.name,
            url: req.url,
            method: req.method,
        })
        const sr = new ServiceRequest(req)
        const [entity, totalCount] =
            await this.ruleService.readAll(sr)
        const responseList = entity.map(e => new PeeringRuleResponseDto(
            e,
            {url: req.url},
        ))
        return [responseList, totalCount]
    }

    @Get(':groupId?/rules/:id')
    @ApiOkResponse({
        type: PeeringRuleResponseDto,
    })
    async read(
        @Param('id', ParseIntPipe) id: number,
        @Req() req: Request,
        // TODO: _Prefix does not work here, fix?
        // eslint-disable-next-line @typescript-eslint/no-unused-vars
        @Param(new ValidationPipe()) {groupId}: PeeringRuleRequestParamDto = new PeeringRuleRequestParamDto(),
    ): Promise<PeeringRuleResponseDto> {
        this.log.debug({
            message: 'read peering rule by id',
            id: id,
            func: this.read.name,
            url: req.url,
            method: req.method,
        })
        return new PeeringRuleResponseDto(
            await this.ruleService.read(id, new ServiceRequest(req)),
            {url: req.url, containsResourceId: true},
        )
    }

    @Put(':groupId?/rules/:id')
    @ApiOkResponse({
        type: PeeringRuleResponseDto,
    })
    @Transactional()
    async update(@Param('id', ParseIntPipe) id: number,
        dto: PeeringRuleRequestDto,
        @Req() req: Request,
        // TODO: _Prefix does not work here, fix?
        // eslint-disable-next-line @typescript-eslint/no-unused-vars
        @Param(new ValidationPipe()) {groupId}: PeeringRuleRequestParamDto = new PeeringRuleRequestParamDto(),
    ): Promise<PeeringRuleResponseDto> {
        this.log.debug({
            message: 'update peering rule by id',
            id: id,
            func: this.update.name,
            url: req.url,
            method: req.method,
        })
        const sr = new ServiceRequest(req)
        const updates = new Dictionary<internal.VoipPeeringRule >()
        updates[id] = Object.assign(new PeeringRuleRequestDto(), dto).toInternal({id: id, assignNulls: true})
        const ids = await this.ruleService.update(updates, sr)
        const entity = await this.ruleService.read(ids[0], sr)
        const response = new PeeringRuleResponseDto(
            entity,
            {url: req.url, containsResourceId: true},
        )
        await this.journalService.writeJournal(sr, id, response)
        return response
    }

    @Put(':groupId?/rules')
    @ApiPutBody(PeeringRuleRequestDto)
    @Transactional()
    async updateMany(
        @Body(new ParseIdDictionary({items: PeeringRuleRequestDto})) updates: Dictionary<PeeringRuleRequestDto>,
        @Req() req: Request,
    ): Promise<number[]> {
        this.log.debug({message: 'update peering rule Sets bulk', func: this.updateMany.name, url: req.url, method: req.method})
        const sr = new ServiceRequest(req)
        const sets = new Dictionary<internal.VoipPeeringRule    >()
        for (const id of Object.keys(updates)) {
            const dto: PeeringRuleRequestDto = updates[id]
            sets[id] = dto.toInternal({id: parseInt(id), assignNulls: true})
        }
        return await this.ruleService.update(sets, sr)
    }

    @Patch(':groupId?/rules/:id')
    @ApiConsumes('application/json-patch+json')
    @ApiBody({
        type: [PatchDto],
    })
    @Transactional()
    async adjust(
        @Param('id', ParseIntPipe) id: number,
        @Body(new ParsePatchPipe()) patch: Operation[],
        @Req() req: Request,
    ): Promise<PeeringRuleResponseDto> {
        this.log.debug({
            message: 'patch peering rule set by id',
            id: id,
            func: this.adjust.name,
            url: req.url,
            method: req.method,
        })
        const sr = new ServiceRequest(req)

        const oldEntity = await this.ruleService.read(id, sr)
        const entity = await patchToEntity<internal.VoipPeeringRule , PeeringRuleRequestDto>(oldEntity, patch, PeeringRuleRequestDto)
        const update = new Dictionary<internal.VoipPeeringRule  >(id.toString(), entity)

        const ids = await this.ruleService.update(update, sr)
        const updatedEntity = await this.ruleService.read(ids[0], sr)
        const response = new PeeringRuleResponseDto(
            updatedEntity,
            {url: req.url, containsResourceId: true},
        )
        await this.journalService.writeJournal(sr, id, response)

        return response
    }

    @Patch(':groupId?/rules')
    @ApiConsumes('application/json-patch+json')
    @ApiPutBody(PatchDto)
    @Transactional()
    async adjustMany(
        @Body(new ParseIdDictionary({items: PatchDto, valueIsArray: true})) patches: Dictionary<PatchOperation[]>,
        @Req() req: Request,
    ): Promise<number[]> {
        const sr = new ServiceRequest(req)

        const updates = new Dictionary<internal.VoipPeeringRule >()
        for (const id of Object.keys(patches)) {
            const oldEntity = await this.ruleService.read(+id, sr)
            const entity = await patchToEntity<internal.VoipPeeringRule , PeeringRuleRequestDto>(oldEntity, patches[id], PeeringRuleRequestDto)
            updates[id] = entity
        }

        return await this.ruleService.update(updates, sr)
    }

    @Delete(':groupId?/rules/:id?')
    @ApiOkResponse({
        type: [number],
    })
    @Transactional()
    async delete(
        @ParamOrBody('id', new ParseIntIdArrayPipe()) ids: number[],
        @Req() req: Request,
    ): Promise<number[]> {
        this.log.debug({
            message: 'delete peering rule set by id',
            id: ids,
            func: this.delete.name,
            url: req.url,
            method: req.method,
        })
        const sr = new ServiceRequest(req)
        const deletedIds = await this.ruleService.delete(ids, sr)
        for (const deletedId of deletedIds) {
            await this.journalService.writeJournal(sr, deletedId, {})
        }
        return deletedIds
    }

    @Get(':groupId?/rules/:id/journal')
    @ApiOkResponse({
        type: [JournalResponseDto],
    })
    async journal(@Param('id') id: number | string, @Req() req: Request): Promise<[JournalResponseDto[], number]> {
        this.log.debug({
            message: 'read peering rule journal by id',
            id: id,
            func: this.journal.name,
            url: req.url,
            method: req.method,
        })
        return super.journal(id, req)
    }
}
