#!/usr/bin/perl

use strict;
use warnings;

use Test::More tests => 33;
use File::Path qw(make_path rmtree);

BEGIN {
    use_ok('NGCP::Config');
}

my $tmpdir = $ENV{RESULTS} // 't.tmp';
my $testlog = "$tmpdir/cfg.log";

# Make sure we are running on a clean temporary tree.
rmtree($tmpdir);
make_path($tmpdir);

my $log;

$log = NGCP::Log->new(
    progname => 'test-cfg',
    output => $testlog,
);

my $cfg;

my $cfg_schema = {
    bool_def => {
        type => NGCP::Config::CONF_BOOL,
        default => 'no',
    },
    bool_true => {
        type => NGCP::Config::CONF_BOOL,
    },
    bool_false => {
        type => NGCP::Config::CONF_BOOL,
    },
    bool_yes => {
        type => NGCP::Config::CONF_BOOL,
    },
    bool_no => {
        type => NGCP::Config::CONF_BOOL,
    },
    bool_on => {
        type => NGCP::Config::CONF_BOOL,
    },
    bool_off => {
        type => NGCP::Config::CONF_BOOL,
    },
    int_def => {
        type => NGCP::Config::CONF_INT,
        default => 10,
    },
    int => {
        type => NGCP::Config::CONF_INT,
    },
    text_def => {
        type => NGCP::Config::CONF_TEXT,
        default => 'text',
    },
    text_quoted => {
        type => NGCP::Config::CONF_TEXT,
    },
    text => {
        type => NGCP::Config::CONF_TEXT,
    },
    list_def => {
        type => NGCP::Config::CONF_LIST,
        default => [ qw(aa bb) ],
    },
    list_sep_def => {
        type => NGCP::Config::CONF_LIST,
        default => [ 'aa,  bb, cc', 'xx,yy', 'zz' ],
        sep => qr{, *},
    },
    list => {
        type => NGCP::Config::CONF_LIST,
    },
    list_sep => {
        type => NGCP::Config::CONF_LIST,
        sep => qr{:},
    },
    file_def =>  {
        type => NGCP::Config::CONF_FILE,
        default => '/run/nonexistent',
    },
    file =>  {
        type => NGCP::Config::CONF_FILE,
    },
    port_def => {
        type => NGCP::Config::CONF_PORT,
        default => 30000,
    },
    port => {
        type => NGCP::Config::CONF_PORT,
    },
    host_def => {
        type => NGCP::Config::CONF_HOST,
        default => 'example.org',
    },
    host => {
        type => NGCP::Config::CONF_HOST,
    },
    logs_def => {
        type => NGCP::Config::CONF_LOGS,
        default => 'console',
    },
    logs_console => {
        type => NGCP::Config::CONF_LOGS,
    },
    logs_syslog => {
        type => NGCP::Config::CONF_LOGS,
    },
    logs_filename => {
        type => NGCP::Config::CONF_LOGS,
    },
    override => {
        type => NGCP::Config::CONF_INT,
    },
};

my $cfg_alias = {
    bool_default => 'bool_def',
    boolean_false => 'bool_false',
};

$cfg = NGCP::Config->new(
    log => $log,
    schema => $cfg_schema,
    aliases => $cfg_alias,
    filenames => [ 't/NGCP_Config/cfg-aa', 't/NGCP_Config/cfg-bb' ],
);

$cfg->init();
$cfg->parse();

# Getting values.

ok(ref $cfg->get() eq 'HASH', 'parameter less get returns a HASH');

is($cfg->get('bool_def'), 0, 'bool default');
is($cfg->get('bool_default'), 0, 'bool default (via alias)');
is($cfg->get('bool_true'), 1, 'bool true');
is($cfg->get('bool_false'), 0, 'bool false');
is($cfg->get('boolean_false'), 0, 'bool false (via alias)');
is($cfg->get('bool_yes'), 1, 'bool yes');
is($cfg->get('bool_no'), 0, 'bool no');
is($cfg->get('bool_on'), 1, 'bool on');
is($cfg->get('bool_off'), 0, 'bool off');
is($cfg->get('int_def'), 10, 'int default');
is($cfg->get('int'), 100, 'int 100');
is($cfg->get('text_def'), 'text' , 'text default');
is($cfg->get('text_quoted'), 'Some quoted text', 'text quoted');
is($cfg->get('text'), 'Some unquoted text' , 'text');
is_deeply($cfg->get('list_def'), [ qw(aa bb) ], 'list default');
is_deeply($cfg->get('list_sep_def'), [ qw(aa bb cc xx yy zz) ], 'list separator default');
is_deeply($cfg->get('list'), [ qw(foo bar) ], 'list');
is_deeply($cfg->get('list_sep'), [ qw(aa bb xx yy zz) ], 'list separator');
is($cfg->get('file_def'), '/run/nonexistent', 'file default');
is($cfg->get('file'), '/nonexistent', 'file');
is($cfg->get('port_def'), 30000, 'port default');
is($cfg->get('port'), 45000, 'port');
is($cfg->get('host_def'), 'example.org', 'host default');
is($cfg->get('host'), 'example.net', 'host');
is($cfg->get('logs_def'), 'console', 'logs default');
is($cfg->get('logs_console'), 'console', 'logs console');
is($cfg->get('logs_syslog'), 'syslog', 'logs syslog');
is($cfg->get('logs_filename'), '/some-log-file', 'logs filename');
is($cfg->get('override'), 500, 'override value');

# Setting values.

is($cfg->set('var_set', 'value'), 'value', 'setting value');
is($cfg->get('var_set'), 'value', 'getting set value');

1;
