from datetime import datetime
from time import time
from typing import Any, AnyStr, Dict, Generic, List
from uuid import uuid4

from .typing import Primitive


class Request():
    """Request class.

    Attributes:
    uuid (str): unique id of the request (UUID),
        must be provided by the client
    task (str): task name, if non-existent,
        rejected by the server
    src (str): source host name
        (or another known to server identifier)
    dst (str): destination host name
        (or can be a wildcard match) that in case of multiple task agent
        instances to only process the request on some of them
        e.g:
        dst='local'
            accept the request on the local node
        dst='localhost'
            accept the request on the local node
        dst='prx01a'
            accept the request on prx01a node
        dst='prx*'
            accept the request on all proxy nodes
        dst='*|state=active'
            accept the request on all active nodes
        dst='*|status=online'
            accept the request on all nodes with online status
        dst='*|state=active;role=proxy'
            accept the request on all active proxy nodes
        dst='*|state=active;role=proxy+lb'
            accept the request on all active proxy and lb nodes
        dst='prx01a,lb*,db01?|state=active'
            accept the request on prx01a, all lb nodes
              and the active db01 node
    timestamp (float): unixtime of the message,
        generated by the server upon receiving the request
    datetime (str): datetime of the message,
        generated by the server upon receiving the request
    options (Dict): task related options,
        usually something that is expected
        by the backend, like 'feedback_channel'
    data (Any): additional data to provide to the server
        (ignored by the server if contains unexpected data)
    originator (Dict): for a task created from the "follow" option,
        contains information about the original request

    """
    uuid: str = ''
    task: str = ''
    src: str = ''
    dst: str = ''
    timestamp: float = 0
    datetime: str = ''
    options: Dict[str, Primitive] = {}
    data: Any = ''
    originator: Dict[str, Any] = {}

    def __init__(self) -> None:
        """Constructor for Request class.

        Args:
            None

        Returns:
            None

        """
        self.uuid = str(uuid4())
        self.timestamp = time()
        self.datetime = datetime.fromtimestamp(self.timestamp).strftime(
            '%Y-%m-%d %H:%M:%S.f'
        )[:-3]
