from datetime import datetime
from time import time
from typing import Any, Dict
from uuid import uuid4


class Response:
    """Response class.

    Attributes:
    uuid (str): unique id of the response
    ref (str): unique id of the original request,
        used to relate responses to the request
    task (str): task name that is used in the response
        (usually the same that is provided by the request)
    src (str): server host name
    dst (str): client host name
    timestamp (float): unixtime of the message,
        generated by the server when dispatching the response
    datetime (str): datetime of the message,
        generated by the server when dispatching the response
    chunk (int): chunk number of the response (default: 1),
        in cases when a response is split into multiple chunks
        each next response's value is increased by 1
    chunks (int): (default: 1) in case if a response is split by the server
        into multiple chunks, contains the total amount of chunks
    status (str): [accepted|rejected|done|error]
        indication of the server state related to the response,
        (accepted): response is sent by the server if the request is accepted
            and just before processing it by the broker
        (rejected):  response is issued if something went wrong,
            and usually the 'reason' field contains relevant info about it
        (done): when a task is completed and in this case the response
            contains the final data
            (in case of chunked responses they all have the same 'status')
        (error): when an error occurred on the server side
    reason (str): reason string,
        empty in case of 'accepted' or 'done'
        and used for 'rejected' and 'error' responses
    data (Any): - resulting data that is provided by the server

    """
    uuid: str = ''
    ref: str = ''
    task: str = ''
    src: str = ''
    dst: str = ''
    timestamp: float = 0
    datetime: str = ''
    chunk: int = 1
    chunks: int = 1
    status: str = ''
    reason: str = ''
    data: Any = {}

    def __init__(self) -> None:
        """Constructor for Response class.

        Args:
            None

        Returns:
            None

        """
        self.uuid = str(uuid4())
        self.timestamp = time()
        self.datetime = datetime.fromtimestamp(self.timestamp).strftime(
            '%Y-%m-%d %H:%M:%S.%f'
        )[:-3]
        self.chunk = 1
        self.chunks = 1
        self.status = 'unknown'
